//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\AuxComPorts.h
// Date: 2023-12-20
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Header file for the Remote CW Keyer's "Additional (ex:Auxiliary) COM Ports".
//          Details only in the implementation, Remote_CW_Keyer/AuxComPorts.h .
//---------------------------------------------------------------------------

#ifndef  AUX_COM_PORTS_H
# define AUX_COM_PORTS_H

#ifndef  CIRCULAR_FIFO_INCLUDED
# include "CircularFifo.h"
#endif
#ifndef  _YHF_TIMERS_H
# include "Timers.h"  // T_TIM_Stopwatch ..
#endif
#ifndef RigControlH
# include "RigControl.h"  // T_RigCtrl_PortInstance used in the definition of T_AuxComPortInstance (not vice versa)
#endif


#ifndef HANDLE
// No-No: # include <windows.h> // 'HANDLE' (a Windows thing for almost anything) defined HERE.
  typedef void* HANDLE;
#endif // ndef HANDLE

// Macro to determine the size of a struct member in BYTES, in analogy to C's sizeof(type) :
#ifndef  sizeof_member  // were in good old C, not C11 or C++, so:
# define sizeof_member(struct_t, member_t) (sizeof(((struct_t*)0)->member_t))
#endif


#define AUX_COM_PORT_FIFO_SIZE 1024
typedef struct
{ T_CFIFO fifo;
  BYTE bExtraBytes[ AUX_COM_PORT_FIFO_SIZE - SWI_CIRCULAR_FIFO_SIZE ];
  //                 '--> e.g. 1024 bytes     '--> e.g. 256 bytes
} T_AuxComPortFifo;

typedef enum
{ AUXCOM_IO_TOKEN_CW = 0, // "drive an output with the current CW key up/key down state"
  AUXCOM_IO_TOKEN_PTT,    // "drive an output with the current CW key up/key down state",
                          // or "activate the PTT-switching sequence if this INPUT is active"
  AUXCOM_IO_TOKEN_DOT,    // "secondary DOT (aka "dit"-) input"
  AUXCOM_IO_TOKEN_DASH,   // "secondary DASH (aka "dah"-) input"
  AUXCOM_IO_TOKEN_SKEY,   // "separate input for a STRAIGHT MORSE KEY" (e.g. for tuning and generating QRM ;o)
  AUXCOM_IO_TOKEN_TOGGLE, // "evil command to TOGGLE an output pin as fast as the thread, and the COM port driver permits"
  AUXCOM_IO_TOKEN_RXACK,  // "generate a short pulse on this OUTPUT after RECEPTION on RXD"
  AUXCOM_IO_TOKEN_TXACK,  // "generate a short pulse on this OUTPUT after TRANSMISSION on TXD"
  AUXCOM_IO_TOKEN_RXBUSY, // "set this output HIGH as long as ReadFile() is "pending" (fReadPending) in AuxComThread()"
  AUXCOM_IO_TOKEN_TXBUSY, // "set this output HIGH as long as WriteFile() is "pending" (fWritePending) in AuxComThread()"

  AUXCOM_IO_NUM_TOKENS
} enAuxComIoTokens;

// To allow MULTIPLE "auxiliary/additional COM ports", everything that
// AuxComThread() and other functions need (besides local variables)
// must be instantiable. Keep it simple (and use C rather than C++) :
//  Put everything the thread needs inside the following struct,
//  and use an ARRAY of them:
typedef struct tAuxComPortInstance  // -> AuxComPorts[SWI_NUM_AUX_COM_PORTS]
{

  // "Configuration parameters". Some set via argument in AuxCom_Start(),
  // some may be set by the application between AuxCom_InitStruct() and AuxCom_Start():
  int iComPortNumber; // "COM port number" (not a HANDLE) from AuxCom_Start().
      // The application may compare this against CwKeyer_Config.iAuxComPortNumber[]
      // do decide if "something has been modified by the user" (on the fly),
      // and call AuxCom_Start() with the NEW (different) setting.
  T_RigCtrl_PortInstance *pRigctrlPort; // allows accessing a "Rig-Control PORT INSTANCE",
      // with e.g. pRigtrlPort->iPortUsage: RIGCTRL_PORT_USAGE_NONE / .._RADIO_CONTROL / .._SERIAL_TUNNEL / .._VIRTUAL_RIG / etc
      //          (ex: T_AuxComPortInstance.iComPortUsage = "purpose" of this serial port, set in AuxCom_Start() )
  int iBitsPerSecond;  // serial baudrate in the unit implied by the member's name :)
  int iNumDatabits; // 5..8, almost always EIGHT (as in the serial format "8-N-1")
  int iParity;      // SER_PORT_PARITY_NONE/ODD/EVEN/MARK/SPACE, defined in CwKeyer.h (!)
  int iNumStopbits; // SER_PORT_STOPBITS_1 / 2 / 1_5, defined in CwKeyer.h (!)
  int iTunnelIndex; // ZERO-BASED "Tunnel Index"; see purpose / details
      // in the user manual ( Remote_CW_Keyer.htm#Serial_Port_Tunnels ).
      // This allows basic routing between serial ports, and maybe other
      // forms of communicating in future. For example data RECEIVED FROM  TCP/IP
      //   with iTunnelIndex = 1 may be SENT TO multiple(!) serial ports.
      //   What matters for that is the TunnelIndex, not the array index
      //   into AuxComPorts[i]. See AuxCom_OnReceptionFromNetworkPortTunnel() !
#     define AUXCOM_TUNNEL_INDEX_ANY -1 // dummy for a "tunnel" that shall...
      //   RECEIVE DATA FROM, and possibly also SEND DATA TO, any/all serial
      //   ports configured with iComPortUsage = RIGCTRL_PORT_USAGE_SERIAL_TUNNEL.
      //   For multiplexing in the TCP/IP bytestream (in module CwNet.c),
      //   data received on an additional COM port with AUXCOM_TUNNEL_INDEX_ANY
      //   will be multiplexed with CWNET_CMD_SERIAL_PORT_TUNNEL_ANY,
      //   while data for "normal" (usually "point-to-point") tunnels
      //   are multiplexed with CWNET_CMD_SERIAL_PORT_TUNNEL_1/2/3 .

  char sz40ExpressionForDTR[40]; // periodically evaluated expression to drive "DTR". Leave empty for the default setting.
  char sz40ExpressionForRTS[40]; // periodically evaluated expression to drive "RTS". Leave empty for the default setting.

  DWORD dwParams; // bitwise combineable "parameters", entered as a STRING in the GUI,
        // parsed by KeyerGUI_ParseAuxComParams(), listed in KeyerGUI_AuxComParams[] :
#   define AUX_COM_PARAMS_DEFAULT   0x00000000  // "don't try to DECODE anything, don't LOG this, don't SHOW DEBUG INFO", etc
#   define AUX_COM_PARAMS_DEBUG     0x00000001  // if the amount of data permits,
        // try to show the port's traffic ON THE DEBUG TAB
        // (controlled by the "hints" to make this human-readable
        //  in AUX_COM_PARAMS_MASK_PROTOCOL, defined further below.
        // Since 2025-08, if AUX_COM_PARAMS_DEBUG is NOT set in pAuxComPort->dwParams,
        // CI-V message received from / sent to an additional COM port
        // are NOT passed on to RigControl.c : RigCtrl_AddToTrafficLog().
        // To simplify that, T_AuxComPortInstance.dwParams.AUX_COM_PARAMS_DEBUG is kind of 'mirrored'
        // in RigControl.h : T_RigCtrl_PortInstance.fTrafficMonitorEnabled .
        //  If we haven't got a clue, AuxCom_DecodeOrDumpBlock()
        //  will guess if the to-be-displayed block is ASCII ("-> show as TEXT")
        //  or "something binary" (-> show as hex-dump) .
        // Similar also for AUX_COM_PARAMS_DUMP_TO_TERMINAL ...
#   define AUX_COM_PARAMS_DUMP_TO_TERMINAL 0x00000002  // pass anything traveling over this port
        // on to the built-in Text Terminal, even if 'Port used as..' is set
        // to e.g. Serial Port Tunnel.
        // In contrast to AUX_COM_PARAMS_DEBUG, if the received characters
        // are PLAIN ASCII, the display doesn't use ONE LINE PER DATA BLOCK
        // but emits the data like RIGCTRL_PORT_USAGE_TEXT_TERMINAL .
#   define AUX_COM_PARAMS_MASK_PROTOCOL     0x000F0000
#   define AUX_COM_PARAMS_PROTOCOL_NONE     0x00000000  // "there's no PROTOCOL to decode"
#   define AUX_COM_PARAMS_PROTOCOL_ICOM     0x00010000  // "most likely this is ICOM's CI-V"
#   define AUX_COM_PARAMS_PROTOCOL_KENWOOD  0x00020000  // (not sure how Kenwood named their protocol, and if it's only ONE.
#   define AUX_COM_PARAMS_PROTOCOL_YAESU_5_BYTE 0x00030000 // (not sure how Yaesu named THIS protocol .. "CAT" isn't enough to specify it ! )
#   define AUX_COM_PARAMS_PROTOCOL_YAESU_ASCII  0x00040000 // the semicolon-delimited stuff with TWO-ASCII-CHARACTER-COMMANDS, used in e.g. FT-891
    // (See WB's notes about those protocols in RigControl.c : RigCtrl_SetRadioControlProtocol() )
  int iExtraDelay_ms; // optional 'extra delay' in milliseconds, to slow down RESPONSES for very chatty remote clients / throttle down their traffic when e.g. polling dozens of parameters every few milliseconds.
                      // Can be activated through the 'parameter string' in a dialog window (delay=<number of milliseconds>) .

  // -------------- end of "Configuration parameters" ----------------


  int iThreadStatus; // may contain one of the following:
#   define AUX_COM_THREAD_STATUS_NOT_CREATED 0
#   define AUX_COM_THREAD_STATUS_LAUNCHED    1
#   define AUX_COM_THREAD_STATUS_RUNNING     2 // <- only set ONCE(!), immediately when entering the thread function
#   define AUX_COM_THREAD_STATUS_TERMINATE   3 // <- just a KLUDGE to "request POLITE termination", but takes a few dozen milliseconds to actually TERMINATE !
#   define AUX_COM_THREAD_STATUS_TERMINATED -1 // <- set by the worker thread if it TERMINATED ITSELF

  DWORD dwThreadLoops;      // number of thread loops, counted in AuxComThread()
  DWORD dwThreadErrors;     // number of errors (mostly TIMING ERRORS) counted in AuxComThread()
  DWORD dwSerialPortErrors; // number of errors related with SERIAL PORTS counted in AuxComThread()
  int   iLastErrorLine;     // __LINE__ of the last error detection, to avoid reporting the same error twice without anything else in between  
  DWORD dw8ThreadIntervals_us[8]; // history of number of MICROSECONDS per loop in AuxComThread()
  DWORD dwNumBytesSent, dwNumBytesRcvd; // byte counters for diagnostics ("which of those ports is alive"?)
  DWORD dwNumSamplesGenerated;    // sample- or byte counter for TEST PATTERNS generated in AuxComThread()

  T_AuxComPortFifo sRxFifo, sTxFifo; // wrappers for T_CFIFO (thread-safe circular FIFOs
     //  accessable by e.g. module RigControl.c, see T_RigCtrl_PortInstance;
     //  but also by CwNet.c for the "TCP/IP side" in mode RIGCTRL_PORT_USAGE_SERIAL_TUNNEL )


  HANDLE hComPort; // INVALID_HANDLE_VALUE if the COM port is not open (a Windows thing!)
       // Note: The RCWKeyer GUI occasionally checks for INVALID_HANDLE_VALUE,
       //       because when some worker thread decides to CLOSE the port
       //       after serious errors, and -after closing a port- sets its handle
       //       to INVALID_HANDLE_VALUE again.
       // Beware: For THE ABOVE HANDLE (hComPort), some lunatic in Richmond(?) decided
       //         that zero (alias NULL, as a pointer) is a perfectly valid 'handle value'.
       //         Thus the need for INVALID_HANDLE_VALUE for THESE kinds of handles.
       //         The "invalid handle value" for OTHER kinds of handle
       //         isn't INVALID_HANDLE_VALUE but zero (or NULL).
  HANDLE hThread; // also a HANDLE, but for a THREAD HANDLE, "NULL" means invalid. Yucc.
  DWORD  dwThreadId; // the joys of Win32 API programming.. not only do we need a
              //  "thread handle", but also a "thread ID"...
              //  we may need it for polite thread termination,
              //  or for waking up the thread via event one day.
  HANDLE hEventToWakeUpThread; // also a HANDLE. This one is returned by CreateEvent().
              //  CreateEvent() returns NULL when the handle (or event) could
              //  NOT be created, so do NOT use INVALID_HANDLE_VALUE for THIS kind of HANDLE
              //  as an indicator for 'not being created or opened yet' .
  BOOL   fWritePending, fReadPending; // flags for the 'overlapped I/O' in AuxComThread().
              // Not local variables anymore, to see these states in 'real time' on DTR or RTS.
  BOOL   fSendImmediately; // prevents WAITING for hEventToWakeUpThread, but only for a single thread loop.
              //  Set, for example, in AuxCom_RunEchoTest() .
  T_TIM_Stopwatch swWriteFilePending; // stopwatch to measure how long WriteFile()
              // is chewing on the last block of data. Added 2025-08-25, after
              // experiencing the trouble with "com0com", when the REMOTE END
              // of the Virtual Null-Modem Cable (realized via "com0com") shuts
              // down. Unlike a REAL NULL MODEM CABLE without hardware flow control
              // or handshake, unplugging one end of the cable never stops the
              // sender from sending... very strange. Grep for the date (2025-08-25) !

  DWORD dwLastSentOutputStates; // snapshot of last 'digital outputs',
       // to minimize the number of calls to SetCommState() or EscapeCommFunctions().
       // Also a bitwise combination of AUX_COM_PORT_OUTPUT_DTR + .._RTS .

  DWORD dwDigitalSignalStates;  // bitwise combination of AUX_COM_DIG_SIGNAL_..
       // used as a kind of 'process image' for combinatorial logic in AuxComThread()
       // and to display all these states somewhere in the GUI :
#  define AUX_COM_DIG_SIGNAL_DCD  (1<<0) // digital INPUT from a serial port's point of view
#  define AUX_COM_DIG_SIGNAL_DSR  (1<<1) // digital INPUT
#  define AUX_COM_DIG_SIGNAL_CTS  (1<<2) // digital INPUT
#  define AUX_COM_DIG_SIGNAL_RI   (1<<3) // digital INPUT
#  define AUX_COM_DIG_SIGNAL_DTR  (1<<4) // digital OUTPUT from a serial port's point of view
#  define AUX_COM_DIG_SIGNAL_RTS  (1<<5) // digital OUTPUT (bits 6 and 7 reserved for more OUTPUTS...)
#  define AUX_COM_DIG_SIGNAL_DOT   (1<<8) // "PLC"-driven signal for the KEYER THREAD (!) / secondary input from a PADDLE (dot contact)
#  define AUX_COM_DIG_SIGNAL_DASH  (1<<9) // "PLC"-driven signal for the KEYER THREAD (!) / secondary input from a PADDLE (dash contact)
#  define AUX_COM_DIG_SIGNAL_SKEY (1<<10) // "PLC"-driven signal for the KEYER THREAD (!) / input from a STRAIGHT KEY
#  define AUX_COM_DIG_SIGNAL_PTT  (1<<11) // "PLC"-driven signal for the KEYER THREAD (!) / secondary input for a MANUAL PTT SWITCH



  char sz255LastError[256];


  BYTE bPrevRcvdChar, bPrevPrevRcvdChar; // "previous" received character, and the character pulled from sRxFifo BEFORE that
    // (required for the messy interpretation of LINE ENDINGS; the old annoyance
    //  with "\r\n" versus "\r only" and "\n only".. hope for the best, expect the worst)

  // "Serial Port Tunnel"-specific members of T_AuxComPortInstance:
  int iRxBufferTailForSerialPortTunnel; // <- used in AuxCom_RunSerialPortTunnel()
        // to read from sRxFifo without draining it for the TCP/IP transmission.


  // Winkeyer specific members of T_AuxComPortInstance:
  // - - - - - - - - -
  BYTE bWKStatus, bReportedWKStatus; // bitwise combination of the following:
# define C_WINKEYER_WK_STATUS_TAG_MSK  0xE0 // bits 7..5 fixed to 0b110 (same for WK1, WK2, WK3)
# define C_WINKEYER_WK_STATUS_TAG_VAL  0xC0
# define C_WINKEYER_WK_STATUS_WAITING  0x10 // bit 4: "WK is waiting for an internally timed event to finish"
# define C_WINKEYER1_WK_STATUS_KEYDOWN 0x08 // bit 3: "Key down status (Tune) 1 = key down" (WK1 only, different meanings for WK2 and WK3. Sigh..)
# define C_WINKEYER2_WK_STATUS_PUSHBTN 0x08 // bit 3: 0 = "This is a WK Status Byte". 1 = "This is a pushbutton status byte"
# define C_WINKEYER_WK_STATUS_BUSY     0x04 // bit 2: "Keyer is busy sending Morse when = 1"
# define C_WINKEYER_WK_STATUS_BREAKIN  0x02 // bit 1: "Paddle break-in active when = 1"
# define C_WINKEYER_WK_STATUS_XOFF     0x01 // bit 0: "Buffer is more than 2/3 full when = 1"
         // In WK2 + WK3, with bit 3 = C_WINKEYER2_WK_STATUS_PUSHBTN *set*,
         // bits 4(!), 2, 1, and 0 get a different meaning: "PUSHBUTTON STATUS":
  BYTE bPBStatus, bReportedPBStatus; // current and "last reported" PUSHBUTTON status (a la Winkeyer2, not 1)
# define C_WINKEYER_PB_STATUS_BUTTON4  0x10 // bit 4: "1 when PB4 pressed, 0 when PB4 unpressed"
# define C_WINKEYER_PB_STATUS_BUTTON3  0x04 // bit 2: "1 when PB3 pressed, 0 when PB3 unpressed"
# define C_WINKEYER_PB_STATUS_BUTTON2  0x02 // bit 1: "1 when PB2 pressed, 0 when PB2 unpressed"
# define C_WINKEYER_PB_STATUS_BUTTON1  0x01 // bit 0: "1 when PB1 pressed, 0 when PB1 unpressed"
         // > A Pushbutton status byte is sent whenever the PB state changes,
         // > in other words a byte will be sent when a pushbutton is pressed
         // > and a byte will be sent when the pushbutton is released.
         // > Only one press will be detected at a time.
         // (see also: "Unsolicited Status Transmission" in AuxCom_Winkeyer.c)
  int iWKImmediateCmdParserState; // State machine for the streaming parser,
  int iWKBufferedCmdParserState;  // used in AuxCom_Winkeyer.c . Details only THERE.
  int iByteOfImmediateCommand;    // zero-based byte index within a received 'IMMEDIATE command'.
                      // Used in the streaming parser in AuxCom_Winkeyer.c .
  int nBytesOfImmediateCommand; // number of bytes of the currently received 'IMMEDIATE command'.
                      // Also used in the streaming parser in AuxCom_Winkeyer.c .
  BYTE bImmediateSubcommand;    // e.g. 0x00="Calibrate", 0x01="Reset", 0x02="Host Open" after the ADMIN command, etc...
  int  iByteOfBufferedCommand;  // zero-based byte index within a received 'BUFFERED command'.
  int  nBytesOfBufferedCommand; // number of bytes of the currently received 'BUFFERED command'.
  char c4CharsForProsign[4];    // ASCII codes (?) of TWO characters merged into a 'prosign',
       // in the same "Icom notation" (e.g. ^AR) as used throughout this app
  BYTE bCharacterCurrentlySent; // ASCII code of the character CURRENTLY BEEN SENT (for WK's 'Echo Morse in ASCII' feature)


# define C_WINKEYER_INPUT_BUFFER_FIFO_SIZE 128 // 128 bytes for Winkeyer2; later models seem to have more
  BYTE b128Fifo[C_WINKEYER_INPUT_BUFFER_FIFO_SIZE]; // 128(?)-byte FIFO from Figure 3 in WinkeyUSBman.pdf
  BYTE bFifoHead, bFifoTail; // head- and tail index for the above CIRCULAR FIFO aka Winkeyer "Input Buffer"
  BYTE bFifoOverwritePos;    // Extra, esoteric index into b128Fifo[] for K1EL's strange "Input Buffer Command Set"
                             // (commands 0x16 0x00    = "Reset input buffer pointers",
                             //           0x16 0x01 NN = "Move input pointer to new position in OVERWRITE mode",
                             //           0x16 0x02 NN = "Move input pointer to new position in APPEND mode",
                             //           0x16 0x03 NN = "Add multiple nulls to the buffer" (NN=number of nulls).
  BOOL fWKOverwriteMode; // FALSE = normal "buffer APPEND mode", writing to b128Fifo[ bFifoHead++ ],
                         // TRUE  = special "buffer OVERWRITE mode", writing to b128Fifo[ bFifoOverwritePos++ ],
                         //         until bFifoOverwritePos reaches the 'normal' bFifoHead (?)
                      // (implemented only to allow N1MM Logger+ to start sending a contest report
                      //  BEFORE having entered the callsign completely. See musings in
                      //  C:\cbproj\Remote_CW_Keyer\Bugs\2025_08_16_N1MM and RCWK with winkeyer emulation.txt .. )
  BOOL fWKHostMode; // FALSE=not in "Host Mode", TRUE = "Host Mode" (received cmd "Host Open" but not "Host Close").
                    // (not being in "Host Mode" seems to DISABLE unsolicited reports, etc)
  BOOL fWKImmediatePauseState; // FALSE=not paused, TRUE=paused from immediate command "Set Pause State" <06><nn>
  // ex: T_CwGen *pCwGenerator; // associated "CW generator" (struct defined in CwGen.h)
  // To avoid circular header-inclusion / dependencies, use these ugly kludges:
  void *pCwGenerator;       // associated "CW generator" (actually a 'T_CwGen pointer', struct defined in CwGen.h)
  void *pYaesu5ByteControl; // associated "Yaesu-5-Byte-CAT" decoder instance, actually a 'T_Yaesu5ByteControl pointer')
  void *pRigControl;        // optional associated T_RigCtrlInstance*, to store eavesdropped rig parameters (may be NULL)

  // Internal stuff for debugging / benchmarking some APU function calls ...
  // Possible values for 'iTestItem' : function argument for e.g. AuxCom_UpdateSpeedTestResult,
  // and actually a ARRAY INDICES into the following (microcontroller-friendly):
# define AUXCOM_SPEEDTEST_SERIAL_IO 0  // time spent "doing serial I/O" in each thread loop
# define AUXCOM_SPEEDTEST_NUM_ITEMS 1
  long i32SpeedTestSums_us[AUXCOM_SPEEDTEST_NUM_ITEMS];
  long i32SpeedTestCounts_us[AUXCOM_SPEEDTEST_NUM_ITEMS];
  int  iSpeedTestPeaks_us[AUXCOM_SPEEDTEST_NUM_ITEMS];
  T_TIM_Stopwatch swRxAckPulse, swTxAckPulse;

} T_AuxComPortInstance;

extern T_AuxComPortInstance AuxComPorts[SWI_NUM_AUX_COM_PORTS];

extern int AuxCom_iDiagnosticFlags; // bitwise combination of the following:
#   define AUX_COM_DIAG_FLAGS_NONE                0x0000
#   define AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC 0x0001
#   define AUX_COM_DIAG_FLAGS_SHOW_OTHER_TRAFFIC  0x0002 // e.g. from 'Serial Port Tunnels', etc

//----------------------------------------------------------------------------
// Prototypes (for functions callable from C and C++)
//----------------------------------------------------------------------------

#ifndef CPROT  // For peaceful co-existence of "C" and "C++" ....
# ifdef __cplusplus
#   define CPROT extern "C"
# else
#   define CPROT
# endif  // ! "cplusplus"
#endif  // ndef CPROT


CPROT void AuxCom_InitStruct( T_AuxComPortInstance *pInstance,
              T_RigCtrlInstance      *pRigControl,  // [in]
              T_RigCtrl_PortInstance *pRigctrlPort, // [in]
              void *pCwGenerator); // [in] associated "CW generator" (actually a 'T_CwGen pointer', struct defined in CwGen.h)

CPROT BOOL AuxCom_Start( T_AuxComPortInstance *pInstance,
              int iComPortNumber, // [in] "COM port number", e.g. 8 for "COM8"
              int iComPortUsage,  // [in] function assigned to this port, e.g. RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR
              int iBitsPerSecond); // [in] serial baudrate in the unit implied by the argument's name :)
              // Note: Additional parameters like the number of databits per frame,
              //       the transmission and checking of a PARITY bit,
              //       and the number of stopbits for a 'real' serial interface
              //       must be configured BY THE CALLER,
              // between AuxCom_InitStruct() and AuxCom_Start() !

CPROT BOOL AuxCom_IsInUse( T_AuxComPortInstance *pInstance );
CPROT const char* AuxCom_GetCurrentStatusAsString( T_AuxComPortInstance *pServer );
CPROT const char* AuxCom_SignalStatesToString( T_AuxComPortInstance *pInstance, char *szBuffer, int iMaxLen );
CPROT void AuxCom_UpdateSpeedTestResult( T_AuxComPortInstance *pInstance, int iTestItem, int nMicroseconds );
CPROT int  AuxCom_GetSpeedTestAverage_us( T_AuxComPortInstance *pInstance,int iTestItem );
CPROT void AuxCom_ResetSpeedTestResults(T_AuxComPortInstance *pInstance);

CPROT void AuxComPorts_MultiMediaTimerCallback_2ms(void);  // Called every 2 ms(!) from the "Multi-Media-Timer".
      //     '---> Prefix (with plural) shall imply this is used for ALL INSTANCES.
      // Must be sufficiently robust for being called FROM ANYWHERE, AT ANY TIME,
      // even when none of the <SWI_NUM_AUX_COM_PORTS> Aux-COM-port-instances
      // has been initialized, or has been stopped already !

CPROT BOOL AuxCom_OnReceptionFromNetworkPortTunnel( void/*ex:T_CwNet*/ *pvCwNet, int iTunnelIndex, BYTE bRcvdChar );

CPROT void AuxCom_Stop( T_AuxComPortInstance *pInstance );


//---------------------------------------------------------------------------
// API functions invoked from the Remote CW Keyer GUI :
//---------------------------------------------------------------------------

CPROT BOOL AuxCom_OnKeystrokeInTextTerminal( char cASCII );


//---------------------------------------------------------------------------
// Stuff for debugging / development / troubleshooting / protocol analysis :
//---------------------------------------------------------------------------

CPROT void AuxCom_Debug_FlushRcvdCharHexDump( void ); // Flush characters from AuxCom_AppendRcvdCharToHexDumpForDebug()
CPROT int  AuxCom_Debug_GetTimeElapsedSinceLastHexDump_ms(void);
CPROT void AuxCom_Debug_AppendRcvdCharToHexDump( BYTE bRcvdChar ); // queue up another received character for a later single-line HEX DUMP



//---------------------------------------------------------------------------
// "Winkeyer"-specific subroutines, implemented in AuxCom_Winkeyer.c :
//---------------------------------------------------------------------------

CPROT void AuxCom_RunWinkeyerHost( T_AuxComPortInstance *pInstance );
CPROT void AuxCom_RunWinkeyerEmulator( T_AuxComPortInstance *pInstance );

#endif // AUX_COM_PORTS_H ?


/* end < Remote_CW_Keyer/AuxComPorts.h > */
