//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\AuxComPorts.c
// Date: 2025-06-21
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Worker thread(s) for a few "Additional (ex:Auxiliary) COM Ports"
//          in DL4YHF's 'Remote CW Keyer', short : 'RCW'.
//      Not to be confused with RCW's ELECTRONIC KEYER THREAD, which uses a
//      *different* port (also a "COM port") - see Remote_CW_Keyer\KeyerThread.c .
//
//---------------------------------------------------------------------------


  // Details / about this implementation:
  //
  // 2025-09-04 : Bugfix for the Winkeyer emulation. From the RCWKeyer group:
  //               > I've been able to characterize this a little better
  //               > winkeyer works, but once you send with paddles, it never
  //               > comes back to winkeyer unless the program is restarted
  //               > paddles continue to work, but one did with the paddles
  //               > and winkeyer emulator is dead until next program restart .
  //              Possibly related to a bug reported 2025-08-16 . Test config in
  //               C:\CwKeyerTest\N1MM_winkeyer_test_see_bugs_2025_08_16.INI .
  //              Details still in
  //               C:\cbproj\Remote_CW_Keyer\Bugs\2025_08_16_N1MM and RCWK with winkeyer emulation.txt
  //
  // 2025-07-04 : When configured as 'Serial Port Tunnel', data travelling over
  //              the code may be DECODED (or at least tried to..) by the
  //              Rig-Control module. Details in the implementation of
  //              AuxCom_PassRxOrTxDataToProtocolSpecificDecoder() .
  // 2025-06-06 : The new module 'AuxComPorts.c' started as a quick hack
  //              to play with N1MM Logger+ ("Logger Plus"), as suggested
  //              by DF2RQ in the Remote CW Keyer User's group at groups.ic :
  // > Occasionally I have fun participating in contests, I therefore would like
  // > to use N1MM+ together with Remote CW Keyer at the computer next to
  // > my station in order to have sidetone with N1MM+.
  // >
  // > A dedicated keyer input through a to be defined COM port (in fact
  // > it would be one end of a virtual com port cable) supporting the
  // > N1MM+ CW signal (as far as I know ASCII signs on a to be defined com port
  // > at DTR (this would be the other end of the virtual cable com pair))
  // > would therefore be very useful.  This does not require a dedicated
  // > Winkeyer setup, not at N1MM+ and not at Remote CW Keyer.
  // > It just requires reading the ASCII signs as provided by N1MM+
  // > and using the Remote CW Keyer audio engine.
  // For that purpose, set one of the RCW 'Auxiliary COM Ports'
  // to "Winkey EMULATOR" (CwKeyer_Config.iAuxComPortUsage[]=RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR).
  //

#include "switches.h"  // project specific compiler switches ("options"),
                       // must be included before anything else !
#include "yhf_type.h"  // classic types like BYTE, WORD, DWORD, BOOL, ..

#include <windows.h>  // try NOT to use any Borland-specific stuff here (no VCL)
#include <string.h>
#include <stdio.h>    // no "standard I/O" but e.g. sprintf() used here

#if( SWI_USE_DSOUND ) // use "Direct Sound" / dsound_wrapper.c ?
# include "dsound_wrapper.h"
#endif // SWI_USE_DSOUND ?


#include "Utilities.h" // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "Timers.h"    // high-resolution 'current time'-function, T_TIM_Stopwatch(), etc.
#include "Debouncer.h" // 'debouncer' for paddle inputs (T_Debouncer in T_ElbugInstance)
#include "Elbug.h"     // old 'Elbug' functions, converted from PIC-assembler to "C"
#include "CwGen.h"     // CW generator (converts text to Morse code)
#include "CwDSP.h"     // CW-'Digital Signal Processor' / sidetone generator
#include "CwNet.h"     // Socket-based 'Client or Server' for the Remote CW Keyer
#include "StraightKeyDecoder.h" // Morse decoder for input from a STRAIGHT KEY
#include "CwKeyer.h"   // CW-keyer ("from paddle contacts to Morse code") .
          // '--> Contains an AWFUL lot of 'external' references, e.g.:
          //      CwKeyer_Config, CwKeyer_Elbug, CwKeyer_StraightKeyDecoder,
          //      CwKeyer_Gen,    CwKeyer_DSP,   CwKeyer_ClientOrServer,
          //      CwKeyer_TimingScope,           CwKeyer_DecoderFifo,
          //      Keyer_RadioControlPortRxFifo, Keyer_RadioControlPortTxFifo, ..
          // (depending on its configuration, CwKeyer_Config.iAuxComPortUsage[],
          //  the thread servicing a certain 'Auxiliary COM Port' may read from,
          //  or even write to, some of the above objects)

#include "AuxComPorts.h" // structs and API functions for the "Auxiliary" (later: "Additional") COM ports

#include "Yaesu5Byte.h"

const T_SL_TokenList AuxComPortIOLineFunctions[] = // .. for AuxComThread_ProcessDigitalInput() + AuxComThread_ControlDigitalOutput()
{ { "CW",  AUXCOM_IO_TOKEN_CW  }, // "drive an output with the current CW key up/key down state"
  { "PTT", AUXCOM_IO_TOKEN_PTT }, // "drive an output with the current CW key up/key down state",
                                  // or "activate the PTT-switching sequence if this INPUT is active"
  { "DOT", AUXCOM_IO_TOKEN_DOT }, // "secondary DOT (aka "dit"-) input"
  { "DASH",AUXCOM_IO_TOKEN_DASH}, // "secondary DASH (aka "dah"-) input"
  { "SKEY",AUXCOM_IO_TOKEN_SKEY}, // "optional STRAIGHT MORSE KEY input" (e.g. for tuning at QRO)

  { "TOGGLE",AUXCOM_IO_TOKEN_TOGGLE}, // "toggle the state of this OUTPUT" (evil stress test, see implementation and manual)
  { "RXACK", AUXCOM_IO_TOKEN_RXACK},  // "generate a short pulse on this OUTPUT after RECEPTION on RXD"
  { "TXACK", AUXCOM_IO_TOKEN_TXACK},  // "generate a short pulse on this OUTPUT after TRANSMISSION on TXD"
  { "RXBUSY",AUXCOM_IO_TOKEN_RXBUSY}, // "set this output HIGH as long as ReadFile() is "pending" (fReadPending) in AuxComThread()"
  { "TXBUSY",AUXCOM_IO_TOKEN_TXBUSY}, // "set this output HIGH as long as WriteFile() is "pending" (fWritePending) in AuxComThread()"

  { NULL, 0 } // "all zeros" mark the end of the list
  // All these "functions" are explained in the manual, see
  //  file:///C:/cbproj/Remote_CW_Keyer/manual/Remote_CW_Keyer.htm#AuxCom_IO_Lines

}; // end AuxComPortIOLineFunctions[]


// To allow MULTIPLE "auxiliary/additional COM ports", put everything that
// a 'auxiliary COM port worker thread' needs in a struct, used in this ARRAY:
T_AuxComPortInstance AuxComPorts[SWI_NUM_AUX_COM_PORTS];
int AuxCom_iDiagnosticFlags = AUX_COM_DIAG_FLAGS_NONE;




//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int AuxCom_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM__AUX_COM()  AuxCom_iLastSourceLine=__LINE__
     // (see complete list of other XYZ_iLastSourceLine variables to watch
     //  in C:\cbproj\Remote_CW_Keyer\Keyer_Main.cpp, near GUI_iLastSourceLine)
#else
# define HERE_I_AM__AUX_COM()
#endif // SWI_HARDCORE_DEBUGGING ?


//----------------------------------------------------------------------------
// "Internal function prototypes"
//  (forward references for functions called before their implementation)
//----------------------------------------------------------------------------

DWORD WINAPI AuxComThread( LPVOID lpParam );

static BOOL AuxCom_OpenAndConfigureSerialPort( T_AuxComPortInstance *pInstance,
              HANDLE *pHandle, char *psz255Error );

static void AuxComThread_RunProgrammableLogic( T_AuxComPortInstance *pInstance,
              int iAuxComInstance );

static void AuxCom_DumpMessageToLog( T_AuxComPortInstance *pInstance,
              int iRigCtrlOrigin,        // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX
              BYTE *pbData, int nBytes,  // [in] chunk of data received from, or sent to the port (not necessarily a complete, nor a SINGLE 'Message')
              char *pszComment );        // [in] optional comment generated by a protocol-decoder, may be NULL

static void AuxCom_PassRxOrTxDataToProtocolSpecificDecoder(
              T_AuxComPortInstance *pInstance,
              int iRigCtrlOrigin,        // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX
              BYTE *pbData, int nBytes); // [in] chunk of data received from, or sent to the port (not necessarily a complete, nor a SINGLE 'Message')
static BOOL AuxCom_MustPassDataToDecoder( T_AuxComPortInstance *pInstance ); // -> TRUE=yes, FALSE=no.


static int  AuxCom_InstancePtrToIndex( T_AuxComPortInstance *pInstance );
static T_AuxComPortInstance* AuxCom_IndexToInstancePtr( int iAuxComInstance );



//----------------------------------------------------------------------------
// "Application Interface" (functions called e.g. from the test application)
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
void AuxCom_InitStruct( T_AuxComPortInstance *pInstance, // [out]
        T_RigCtrlInstance      *pRigControl,  // [in]
        T_RigCtrl_PortInstance *pRigctrlPort, // [in]
        void *pCwGenerator) // [in] associated "CW generator" (actually a 'T_CwGen pointer', struct defined in CwGen.h)
{
  memset( pInstance, 0, sizeof(T_AuxComPortInstance) );
  pInstance->pRigControl = pRigControl;
  pInstance->pRigctrlPort= pRigctrlPort;
  pInstance->pCwGenerator= pCwGenerator;

  // Initialize the foolish incompatible 'HANDLE' types a la Windows,
  // to make sure they are NOT OPEN YET. Remember, a HANDLE for a "COM port"
  // has INVALID_HANDLE_VALUE when invalid, while a HANDLE (same data type)
  // for a THREAD is NULL (alias ZERO) when invalid .  Thank you, thank you.
  pInstance->hComPort = INVALID_HANDLE_VALUE;
  // No-No: pInstance->hThread = INVALID_HANDLE_VALUE; <--  WRONG ! Leave it ZERO=NULL.


  // reset the Winkeyer PARSER STATE MACHINE ..
  pInstance->iWKImmediateCmdParserState = -1;
  pInstance->iWKBufferedCmdParserState  = -1;
  // .. and initialize those variables that MUST NOT BE ZERO when periodically
  //    calling AuxCom_RunWinkeyerEmulator() or AuxCom_RunWinkeyerHost() :
  pInstance->bWKStatus = pInstance->bReportedWKStatus = C_WINKEYER_WK_STATUS_TAG_VAL; // "Winkeyer Status Byte" (with the "PUSHBUTTON"-message-indicator-bit CLEARED)
  pInstance->bPBStatus = pInstance->bReportedPBStatus = C_WINKEYER_WK_STATUS_TAG_VAL | C_WINKEYER2_WK_STATUS_PUSHBTN; // "Pushbutton Status Byte"

  // Most other members are ok with ZERO from the "memset" above.
  // Those NOT passed as function argument to AuxCom_InitStruct()
  // should be modified by the caller BEFORE calling AuxCom_Start() !

} // end AuxCom_InitStruct()


//----------------------------------------------------------------------------
static int AuxCom_InstancePtrToIndex( T_AuxComPortInstance *pInstance )
  //  [return] : iAuxComInstance = an INDEX into CwKeyer_Config.AuxComXYZ[iAuxComInstance].
  // Kludge to access "configuration data" that -for simplicity-
  // are NOT copied from the global configuration struct (CwKeyer_Config)
  // into a T_AuxComPortInstance between AuxCom_InitStruct() and AuxCom_Start().
  // For example, iAuxComInstance is used to access...
  //  * CwKeyer_Config.sz40AuxComDTR[iAuxComInstance],
  //  * CwKeyer_Config.sz40AuxComRTS[iAuxComInstance],
  //  * CwKeyer_Config.sz40AuxComDCD[iAuxComInstance],
  //  * CwKeyer_Config.sz40AuxComDSR[iAuxComInstance],
  //  * CwKeyer_Config.sz40AuxComCTS[iAuxComInstance],
  //  * CwKeyer_Config.sz40AuxComRI [iAuxComInstance], and who-knows-what ...
{
  int iAuxComInstance;
  for(iAuxComInstance=0; iAuxComInstance<SWI_NUM_AUX_COM_PORTS; ++iAuxComInstance )
   { if( pInstance == &AuxComPorts[iAuxComInstance] )
      { return iAuxComInstance;
      }
   }
  return 0; // if the "pInstance" isn't ours, use the configuration of the FIRST instance
} // end AuxCom_InstancePtrToIndex()

//----------------------------------------------------------------------------
static T_AuxComPortInstance* AuxCom_IndexToInstancePtr( int iAuxComInstance )
  // Inverse to AuxCom_InstancePtrToIndex() .
{ if( (iAuxComInstance>=0) && (iAuxComInstance<SWI_NUM_AUX_COM_PORTS) )
   { return &AuxComPorts[iAuxComInstance];
   }
  else
   { return NULL;
   }
} // end AuxCom_IndexToInstancePtr()


//----------------------------------------------------------------------------
void AuxCom_Stop( T_AuxComPortInstance *pInstance )
     // Stops this port's worker thread (politely if possible),
     // closes this instance's SERIAL PORT ("COMx"),
     // and if we ever need that frees additional resources used by it.
     // Note: It doesn't hurt to call AuxCom_Stop() when it hasn't been STARTED.
     //   But the struct MUST have been initialized via AuxCom_InitStruct() !
{
  int i;

  if( ( pInstance->iThreadStatus == AUX_COM_THREAD_STATUS_LAUNCHED )
    ||( pInstance->iThreadStatus == AUX_COM_THREAD_STATUS_RUNNING  ) )
   { pInstance->iThreadStatus = AUX_COM_THREAD_STATUS_TERMINATE;
     for(i=0; i<20; ++i )
      { Sleep(10);
        if( pInstance->iThreadStatus == AUX_COM_THREAD_STATUS_TERMINATED ) // bingo..
         { break;  // .. the thread has terminated itself, so no need to kill it
         }
      }
   }
  pInstance->iThreadStatus = AUX_COM_THREAD_STATUS_NOT_CREATED;
  if( pInstance->hThread != NULL )  // handle for this instance's WORKER THREAD valid ?
   { CloseHandle( pInstance->hThread );
     pInstance->hThread  = NULL; // Remember: an invalid THREAD HANDLE value isn't INVALID_HANDLE_VALUE but NULL.
                                 // Thanks to the crazy folks in Richmond.
   }
  if( pInstance->hComPort != INVALID_HANDLE_VALUE ) // an invalid SERIAL PORT HANDLE value isn't NULL but INVALID_HANDLE_VALUE. Madness takes it's toll.
   { CloseHandle( pInstance->hComPort );
     pInstance->hComPort = INVALID_HANDLE_VALUE; // Remember: INVALID_HANDLE_VALUE is not NULL (but 0xFFFFFFFF).
     // Thanks to the crazy folks in Richmond.
     // (sigh.. why couldn't the simply exclude 'zero' from being a valid FILE or SERIAL PORT handle ?
     //  Maybe, in days long gone by, this particular flavour of 'HANDLE' was indeed an array index.)
   }

} // end AuxCom_Stop()


//----------------------------------------------------------------------------
BOOL AuxCom_Start( T_AuxComPortInstance *pInstance, // Opens ONE of the ports, and starts its 'worker thread'
        int iComPortNumber, // [in] "COM port number", e.g. 8 for "COM8". Negative for "test WITHOUT a real port".
        int iComPortUsage,  // [in] function assigned to this port, e.g. RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR
        int iBitsPerSecond) // [in] serial baudrate in the unit implied by the argument's name :)
        // Note: Additional parameters like the number of databits per frame,
        //       the transmission and checking of a PARITY bit,
        //       and the number of stopbits for a 'real' serial interface
        //       must be set in *pInstance BY THE CALLER,
        // between AuxCom_InitStruct() and AuxCom_Start() !

  // Returns TRUE when successful,
  //      or FALSE when something went wrong. In that case,
  //               pInstance->sz255LastError[] contains the 'reason' for failing.
{
  pInstance->iComPortNumber = iComPortNumber; // save this for later ...
  pInstance->pRigctrlPort->iPortUsage = iComPortUsage;  // because sub-modules like AuxCom_Winkeyer.c need this info, too
  pInstance->iBitsPerSecond = iBitsPerSecond;

  // Just in case the application didn't STOP an already running instance before,
  // make sure the old one (that possibly drove the PTT and/or CW keying output)
  // doesn't drive an 'auxiliary input' for the built-in keyer / elbug:
  pInstance->dwDigitalSignalStates  = 0; // neither AUX_COM_DIG_SIGNAL_DOT/DASH/SKEY/PTT (etc) !
  pInstance->dwLastSentOutputStates = pInstance->dwDigitalSignalStates;

  switch( pInstance->pRigctrlPort->iPortUsage ) // Decide for the INITIAL STATES of DTR and DSR :
   { case RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR :
     case RIGCTRL_PORT_USAGE_WINKEYER_HOST     :
        // To drive a K1EL 'Winkeyer' as recommended by PSEUDO- ("psuedo") code in the
        //      "WK_USB Application Interface Guide v1.1 5/8/2006", see page 7:
        // > // Set new state 1200 baud, one stop bit, no parity, 8 bits, enable DTR, disable RTS
        // > dcb.BaudRate = CBR_1200;    // <- DL4YHF: We'll leave this AS CONFIGURED IN THE GUI
        // > dcb.StopBits = ONESTOPBIT;
        // > dcb.Parity   = NOPARITY;
        // > dcb.ByteSize = 8; // Microsofts thinks they can change the size of a BYTE. Well, it's an 8-bit FRAME.
        // > dcb.fDtrControl = DTR_CONTROL_ENABLE;
        // > dcb.fDsrSensitivity = FALSE;
        // > dcb.fOutX = FALSE;
        // > dcb.fInX = FALSE;
        // > dcb.fNull = FALSE;
        // > dcb.fRtsControl = RTS_CONTROL_DISABLE;
        pInstance->dwDigitalSignalStates |=  AUX_COM_DIG_SIGNAL_DTR;  // -> DTR_CONTROL_ENABLE further below
        pInstance->dwDigitalSignalStates &= ~AUX_COM_DIG_SIGNAL_RTS;  // -> RTS_CONTROL_DISABLE further below
                // '--> Initial state in the PLC's 'process image'.
                //      Can be observed in the 'Addition COM Port' dialog window.
        break;
     case RIGCTRL_PORT_USAGE_TEXT_TERMINAL : // control other 'station equipment' via commands entered on the 'Debug' tab ?
     case RIGCTRL_PORT_USAGE_SERIAL_TUNNEL : // two or more serial ports "communicating with each other" ?
     case RIGCTRL_PORT_USAGE_VIRTUAL_RIG   : // RCW Keyer "emulates" a radio (a 'virtual rig') on this port
     case RIGCTRL_PORT_USAGE_ECHO_TEST: // send ANYTHING received on this port back AS FAST AS POSSIBLE ?
     case RIGCTRL_PORT_USAGE_NONE :
     default :
        // When using the same DTR / DSR settings as for the Winkeyer (further above)
        // on the USB port of an IC-7300 (configured to be keyed in CW via its internal VCP,
        //   "COM5 (Silicon Labs CP210x USB to UART Bridge)",
        // the rig immediately started to TRANSMIT. Not good. So use these 'safer' settings,
        // which can be overridden by entering an EXPRESSION (!) or a CONSTANT
        // in the edit fields in the 'Additional COM Port' **dialog** :
        pInstance->dwDigitalSignalStates &= ~AUX_COM_DIG_SIGNAL_DTR; // -> DTR_CONTROL_ENABLE further below
        pInstance->dwDigitalSignalStates &= ~AUX_COM_DIG_SIGNAL_RTS; // -> RTS_CONTROL_DISABLE further below
        break;
   } // end switch < pInstance->pRigctrlPort->iPortUsage for the INITIAL STATES of DTR and DSR >

  if( (pInstance->hThread  != NULL ) // oops .. there is still an OLD "thread handle" ?!
   || (pInstance->hComPort != INVALID_HANDLE_VALUE) ) // oops.. the SERIAL PORT was still open ?!
   { AuxCom_Stop( pInstance ); // stop and close before re-opening with different parameters
   }

  // (re-)initialize the lock-free, thread-safe FIFOs between AuxComThread() and RigControl.c :
  CFIFO_Init( &pInstance->sRxFifo.fifo, AUX_COM_PORT_FIFO_SIZE, sizeof(BYTE), 0.0/*dblSecondsPerSample*/ );
  CFIFO_Init( &pInstance->sTxFifo.fifo, AUX_COM_PORT_FIFO_SIZE, sizeof(BYTE), 0.0/*dblSecondsPerSample*/ );


  // TRY TO (!) open the serial port (before starting its worker thread) :
  if( iComPortNumber >= 0 )
   {
     if( ! AuxCom_OpenAndConfigureSerialPort(
              pInstance, // [in] :  COM port number, bitrate, startbits,parity,stopbits, DTR control, RTS control, etc ...
              &pInstance->hComPort, pInstance->sz255LastError ) )
      { return FALSE;  // THE APPLICATION should display pInstance->sz255LastError somewhere.
        // When tested with a deliberately wrong COM port number, the result
        // was pInstance->sz255LastError = "Could not open serial port 'COM235'" .
      }
     else
      { snprintf( pInstance->sz255LastError, 255, "Opened port COM%d.", (int)iComPortNumber );
      }
   }

  if( pInstance->hEventToWakeUpThread == NULL ) // no HANDLE to an "event" object yet ? Create...
   { pInstance->hEventToWakeUpThread = CreateEvent( NULL, FALSE,  FALSE,  NULL );
     // Pointer to security attributes ---------------'    |       |       |
     // Flag for manual-reset event.  <--------------------'       |       |
     //    If FALSE, Windows automatically resets                  |       |
     //    the state to nonsignaled after a                        |       |
     //    single waiting thread has been released.                |       |
     // Flag for initial state: FALSE = not signalled <------------'       |
     // Pointer to event-object name  <------------------------------------'
     // Note: If "CreateEvent" fails, it returns NULL (not INVALID_HANDLE_VALUE).
     //       But the return value is type HANDLE. Thank you, Microsoft.
   }



  // Create the serial port's worker thread (using WIN32 API functions only,
  //                           no "dot net" framework, no VCL, no Qt, etc)
  //
  // From "Microsoft Win32 Programmer's Reference" ("Creating Threads").
  // > The CreateThread function creates a new thread for a process.
  // > The creating thread must specify the starting address of the code
  // > that the new thread is to execute. Typically, the starting address
  // > is the name of a function defined in the program code.
  // > This function takes a single parameter and returns a DWORD value.
  // > A process can have multiple threads simultaneously
  // > executing the same function.
  // > The following example demonstrates how to create a new thread
  // > that executes the locally defined function, ThreadFunc.
  if( pInstance->hThread==NULL )
   {  pInstance->iThreadStatus = AUX_COM_THREAD_STATUS_LAUNCHED;
      pInstance->hThread = CreateThread(
         NULL,    // LPSECURITY_ATTRIBUTES lpThreadAttributes = pointer to thread security attributes
         65536,   // DWORD dwStackSize  = initial thread stack size, in bytes
         AuxComThread,     // LPTHREAD_START_ROUTINE lpStartAddress = pointer to thread function
         (void*)pInstance, // LPVOID lpParameter = argument for new thread. Here: Address of our instance
         0,       // DWORD dwCreationFlags = creation flags
                  // zero -> the thread runs immediately after creation
         &pInstance->dwThreadId // LPDWORD lpThreadId = pointer to returned thread id
       );
      // > The thread object remains in the system until the thread has terminated
      // > and all handles to it have been closed through a call to CloseHandle.
   }
  if( pInstance->hThread==NULL ) // Check the return value for success.
   {
     strcpy( pInstance->sz255LastError, "CreateThread failed." );
     pInstance->iThreadStatus = AUX_COM_THREAD_STATUS_NOT_CREATED;
     AuxCom_Stop( pInstance ); // here: called to close the already opened serial port
     return FALSE;
   }

  // > Define the Thread's priority as required.
  SetThreadPriority( pInstance->hThread, // handle to the thread
                     THREAD_PRIORITY_NORMAL); // thread priority level

  return TRUE;   // even when returning TRUE, 'minor problems' may be reported via pInstance->sz255LastError !

} // end AuxCom_Start()

//---------------------------------------------------------------------------
void AuxCom_WakeUp( T_AuxComPortInstance *pInstance ) // wakes up THE WORKER THREAD.
  // May be called from any thread/task to speed up e.g. the transmission
  // of data, after being fed into T_AuxComPortInstance.sTxFifo .
{
  if( pInstance->hEventToWakeUpThread != NULL ) // shouldn't be NULL, but PLAY SAFE..
   { SetEvent( pInstance->hEventToWakeUpThread ); // CQ worker thread ! WaitForSingleObject() please RETURN !
        // '-- not really sure if THIS CALL is always allowed (from "anywhere").
        //     In old forum threads, someone speculated, from the "multimedia
        //     timer" it's not. But we just "dare to send an event"
        //     from anywhere (except from the worker thread itself; that's why
        //     we have the flag pInstance->fSendImmediately: It prevents WAITING,
        //     but only for a single thread loop ) .
     // In Microsoft geek speak:
     // > Sets the specified event object to the signaled state.
     //
   }
} // end AuxCom_WakeUp()


//----------------------------------------------------------------------------
void AuxComPorts_MultiMediaTimerCallback_2ms(void)  // Called every 2 ms(!) from the "Multi-Media-Timer".
   // [in] AuxComPorts[0..SWI_NUM_AUX_COM_PORTS-1]
   // Must be sufficiently robust for being called FROM ANYWHERE, AT ANY TIME,
   // even when none of the <SWI_NUM_AUX_COM_PORTS> Aux-COM-port-instances
   // has been initialized, or has been stopped already !
{
  T_AuxComPortInstance *pInstance;
  int iAuxComInstance;  // 0 .. SWI_NUM_AUX_COM_PORTS-1 (ZERO-based index)
  for(iAuxComInstance=0; iAuxComInstance<SWI_NUM_AUX_COM_PORTS; ++iAuxComInstance )
   { pInstance = &AuxComPorts[iAuxComInstance];
     if( (pInstance->iThreadStatus == AUX_COM_THREAD_STATUS_RUNNING) // worker thread has NOT been asked to terminate itself ?
      && (pInstance->hThread != NULL )  // got a valid handle for this instance's WORKER THREAD valid ?
      && (pInstance->hEventToWakeUpThread != NULL ) // got a valid handle for an 'EVENT' to wake up the thread ?
       )
      { // Theoretically, may call SetEvent(pInstance->hEventToWakeUpThread) now.
        // But is it really necessary ? Does the worker thread have such
        // time-critical actions to do (like the 'Elbug Keyer Thread' itself) ?
        // (note: if other threads want to SEND serial data with a low latency,
        //        they may wake up ONE of the AuxComThread()-instances selectively,
        //        by calling AuxCom_WakeUp() immediately after e.g. CFIFO_Write() )
        if( (pInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR)
          ||(pInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_WINKEYER_HOST)
          ||(pInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_SERIAL_TUNNEL)
          ||(pInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_VIRTUAL_RIG  )
          ||(pInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_ECHO_TEST    )
          ||(pInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_TX_STRESS_TEST)
          )
         {  // the above 'usages' for THIS auxiliary COM port may be timing-critical, so:
           SetEvent( pInstance->hEventToWakeUpThread ); // CQ worker thread !
           // '--> In Microsoft geek speak:
           //    > Sets the specified event object to the signaled state.
           //
           // Here, to keep the number of Win32 API calls low,
           // pInstance->hEventToWakeUpThread is "self-resetting" which means,
           // as soon as the thread waiting for it (in e.g. WaitForSingleObject)
           // has been woken up, pInstance->hEventToWakeUpThread is RESET,
           // without the need for ResetEvent().
           // If the event is "set" more often than the event receiver
           //   - in this case AuxComThread() -
           // can handle, the 'event' simply remains signaled, but there is no
           // fancy queue or anything that "counts". Just signaled or not signaled.
         } // end if < time-critical port ? >
      }   // end if < thread running, and valid handles ? >
   }     // end for < iAuxComInstance >
}       // end AuxComPorts_MultiMediaTimerCallback_2ms()


//----------------------------------------------------------------------------
BOOL AuxCom_OnReceptionFromNetworkPortTunnel( // Called from a worker thread in CwNet.c / CwNet_OnReceive() !
        void *pvCwNet, // [in] "CW Network instance" that received the byte via TCP/IP
                       //      (actually a T_CwNet *, but couldn't use that
                       //       in the prototype in AuxComPorts.h due to circular header inclusions)
        int iTunnelIndex, // [in] ZERO-BASED "Serial Tunnel Index"; purpose / details
                         // in the user manual ( Remote_CW_Keyer.htm#Serial_Port_Tunnels )
        BYTE bRcvdChar ) // [in] a single received data byte, demultiplexed from the TCP/IP stream
  // Returns  TRUE  when bRcdChar has been passed on to one or multiple serial ports,
  //          FALSE otherwise.
  // Call stacks (seen by Borland C++ Builder):
  //    "ntdll.dll" -> SysWOW64\KERNEL32.DLL -> ClientThread(!) -> CwNet_OnReceive()
  //                    -> AuxCom_OnReceptionFromNetworkPortTunnel()
{
  BOOL fResult = FALSE;
  T_CwNet *pCwNet = (T_CwNet *)pvCwNet;

  // For some basic 'serial port routing', if the Remote CW Keyer instance has
  // MULTIPLE 'Additional COM Ports' configured as 'Client/Server Tunnel',
  // ON RECEPTION, the freely configurable 'Serial Tunnel Index' is used as follows:
  // * If, in the RCW Keyer instance on this side of the tunnel,
  //   there is only ONE of the 'Additional COM Ports' configured as 'Tunnel',
  //   the received byte goes to exactly THAT PORT (regardless of the SENDER's
  //   "TunnelIndex" for routing).
  // * If, in the RCW Keyer instance on this side of the tunnel,
  //   there are MULTIPLE 'Additional COM Ports' configured as 'Tunnel',
  //   only the ones with a matching 'Serial Tunnel Index' will get the data.
  //   NOTE: THAT MAY BE MORE THAN ONE SERIAL PORT, which actually was the
  //         reason for implementing it on the TCP/IP side that way.
  // * If, for what it's worth (TESTING?), a certain Additional COM Port
  //   shall emit bytes received on ANY of the 'serial port tunnels'.
  //   T_AuxComPortInstance.iTunnelIndex can be set to -1 for "any".
  //        ( btw that's the DEFAULT set in AuxCom_InitStruct() )
  // * If none of the RCW Keyer instances on this side of the tunnel
  //   is configured with T_AuxComPortInstance.pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_SERIAL_TUNNEL,
  //   the received byte is simply thrown away, without an error message.
  T_AuxComPortInstance *pInstance;
  int iAuxComInstance;
  for(iAuxComInstance=0; iAuxComInstance<SWI_NUM_AUX_COM_PORTS; ++iAuxComInstance )
   { pInstance = &AuxComPorts[iAuxComInstance];
     if( pInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_SERIAL_TUNNEL )
      { if((pInstance->iTunnelIndex < 0 ) // this instance wants to "receive all" ?
         ||(pInstance->iTunnelIndex==iTunnelIndex) ) // matching 'tunnel index' ?
         { // bingo, found a matching "serial tunnel", or a joker that receives EVERYTHING :
           CFIFO_WriteForMultipleReaders( &pInstance->sTxFifo.fifo, &bRcvdChar, 1/*# bytes*/ );
           // '--> The data written into the TX FIFO ("transmit" from the SERIAL PORT's point of view)
           //      will be pulled out again in AuxComThread(), after "if( ! fWritePending )" .
           AuxCom_WakeUp( pInstance );
           fResult = TRUE;
         } // end if < matching tunnel index ? >
      }   // end if < port used as CLIENT/SERVER TUNNEL ? >
   }     // end for < iAuxComInstance >

  (void)pCwNet; // ... assigned a value that is never used ..

  return fResult;  // -> TRUE = "delivered to someone", FALSE = didn't ...
} // end AuxCom_OnReceptionFromNetworkPortTunnel()



//---------------------------------------------------------------------------
// API functions invoked from the Remote CW Keyer GUI :
//---------------------------------------------------------------------------

//----------------------------------------------------------------------------
BOOL AuxCom_OnKeystrokeInTextTerminal( char cASCII )
        // [in] cKey : 8-bit ASCII, including control codes like '\r'
        // Caller: TKeyerMainForm::Ed_ErrorHistoryKeyPress() [main thread]
        //    -> KeyerGUI_OnKeyInErrorHistory() -> AuxCom_OnKeystrokeInTextTerminal() .
        // Note: For the opposite direction (data received on a serial port
        //       and 'printed' into RCW Keyer's primitive text terminal),
        //       characters are PULLED from the addiotional COM port's RX-FIFOs
        //       in KeyerGUI_DumpCharsFromTerminalToRichText() .
{
  BOOL fResult = FALSE;  // returns TRUE if the key (ASCII or control code) could be processed here (e.g. SENT via serial port)
  T_AuxComPortInstance *pInstance;
  int iAuxComInstance;
  for(iAuxComInstance=0; iAuxComInstance<SWI_NUM_AUX_COM_PORTS; ++iAuxComInstance )
   { pInstance = &AuxComPorts[iAuxComInstance];
     if( pInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_TEXT_TERMINAL )
      { CFIFO_WriteForMultipleReaders( &pInstance->sTxFifo.fifo, &cASCII, 1/*# bytes*/ );
      }   // end if < port used as a TEXT TERMINAL ? >
   }     // end for < iAuxComInstance >
  return fResult;  // -> TRUE = "sent to someone", FALSE = didn't ...
} // end AuxCom_OnKeystrokeInTextTerminal()

//----------------------------------------------------------------------------
void AuxCom_RunEchoTest( T_AuxComPortInstance *pInstance )
  // Moves anything from the RX BUFFER (pInstance->sRxFifo)
  //                into the RX BUFFER (pInstance->sTxFifo) .
  // Called periodically (once per thread loop) from AuxComThread(),
  // with RIGCTRL_PORT_USAGE_ECHO_TEST. Test results below.
{ BYTE buffer[AUX_COM_PORT_FIFO_SIZE];  // <- no malloc/free; just a speedy STACK VARIABLE !
  int nBytesWriteable, nBytesRead;
  nBytesWriteable = CFIFO_GetNumBytesWriteable( &pInstance->sTxFifo.fifo );
  if( nBytesWriteable > 0 )
   { if( nBytesWriteable > sizeof(buffer) )
      {  nBytesWriteable = sizeof(buffer);
      }
     nBytesRead = CFIFO_Read( &pInstance->sRxFifo.fifo, buffer/*dest*/, nBytesWriteable , NULL/*pdblTimestamp_s*/ );
     if( nBytesRead > 0 )
      { CFIFO_Write( &pInstance->sTxFifo.fifo, buffer/*source*/, nBytesRead, 0.0/*dblTimestamp_s*/ );
        //   '--> Got a FIFO overflow here (reported on the DEBUG tab) when Tera Term
        //        sent a huge file, and the 'Echo test' had GAPS in the echoed stream.
        //      This happened at pInstance->dwNumBytesRcvd = 1792,
        //                       pInstance->dwNumBytesSent =  768,
        //              pInstance->sTxFifo.fifo.iTailIndex =  768, \ head+1 == tail
        //              pInstance->sTxFifo.fifo.head.index =  767. /  -> indeed FULL !
        //              pInstance->sTxFifo.fifo.iSize      = 1024.
        //      With WriteTotalTimeoutConstant = 5 : 100 ms of TXD activity,
        //                                           followed by a 200 ms gap, etc.
        //      With WriteTotalTimeoutConstant = 0 :     ms of TXD activity,
        //                                           followed by a     ms gap, etc.
        pInstance->fSendImmediately = TRUE;
        //          '--> prevents WAITING for hEventToWakeUpThread, but only for a single thread loop.
      }
   }
  // Tested 2025-06-21, using COM8 ("Digitus" RS-232 adapter with FTDI chip), at 115 kBit/second,
  //            with a DSO (MSO5104) hooked to TXD and RXD. Results:
  //   * sometimes, the "echo" time (between RECEPTION and TRANSMISSION) was 10 ms (=the thread loop interval);
  //   * sometimes, the "echo" time was as large as 40 ms (!)
  //   * Tried to reduce the serial port's LATENCY SETTING in Windows 10,
  //       which -for this particular OS- was in "Device Manager -> Ports (COM & LPT") ->
  //            [find YOUR COM port there] -> Properties -> Port Settings
  //              -> "Advanced..." -> "Advanced Settings for COM8" (or whatever) :
  //            "BM Options": "Select lower settings to correct response problems"..
  //             > "Latency Timer (msec):"  change from 16 to 1
  //       ->    > "Your hardware settings have changed. You must restart
  //             >  your computer for these changes to take effect.
  //             >  Do you want to restart your computer now?     [ YES ]
  //      -> Bingo.. times between a single byte on RXD
  //                         and the byte echoed on TXD
  //                 were now mostly around 5 ms, and never above 10 ms.

} // end AuxCom_RunEchoTest()


//----------------------------------------------------------------------------
void AuxCom_RunTxStressTest( T_AuxComPortInstance *pInstance )
  // Fills the TX FIFO up to the maximum with a test pattern (e.g. "ramp").
  // While this test runs, we want to see an absolutely GAP-LESS stream of bytes
  // with an oscilloscope hooked up to the serial port's TXD pin.
  // This is easier than having to start 'File Send' on Tera Term over and over again.
{ BYTE buffer[AUX_COM_PORT_FIFO_SIZE];  // <- no malloc/free; just a speedy STACK VARIABLE !
  int i,nBytes = CFIFO_GetNumBytesWriteable( &pInstance->sTxFifo.fifo );
  if( nBytes > 0 )
   { for(i=0; i<nBytes; ++i )
      { buffer[i] = (BYTE)(pInstance->dwNumSamplesGenerated++);
      }
     CFIFO_Write( &pInstance->sTxFifo.fifo, buffer/*source*/, nBytes, 0.0/*dblTimestamp_s*/ );
     pInstance->fSendImmediately = TRUE;
     //          '--> prevents WAITING for hEventToWakeUpThread, but only for a single thread loop.
   }
  // Tested 2025-06-21, using COM8 ("Digitus" RS-232 adapter with FTDI chip), at 115 kBit/second,
  //            with a DSO (MSO5104) hooked to TXD. Results:
  //  * Most bytes were successfully decoded by the scope's RS-232 decoder.
  //    But every umpteenth byte was displayed with a red 'S' at the end,
  //    which possibly means corrupted STOP BIT (?)
  //  * But the reason was in fact the GAPLESS TRANSMISSION, causing the scope
  //    to misinterpret the start bits ! Could be "fixed" by slowly scrolling
  //    the 'frozen' image vertically.. with a STOPBIT on the left side of
  //    the scope, the test pattern generated above was ok.
  //  * Signals "RXBUSY" on DTR (fWritePending) and "TXBUSY" on RTS (fWritePending)
  //    sometimes toggled like crazy (high or low for only ~300 us),
  //    and the DEBUG TAB showed crazy thread loop intervals during the test:
  //     > ADDITIONAL COM PORT #2 (on COM8)..
  //     >  Status: TX Stress Test, tx=12874240, rx=0 bytes, 100600 loops
  //     >  Serial I/O times: av= 9745 us, pk= 25787 us
  //     >  Loop intervals: 220 21979 318 21772 203 22075 299 21670  us
  //   -> Which of Serial I/O API functions was BLOCKING, when the port
  //      was opened with CreateFile( .. FILE_FLAG_OVERLAPPED ) ?!
  //  * Still the CPU load from the entire Remote CW Keyer was below 1 %,
  //    according to Windows 'Task Manager'. Nothing wrong with the thread.
  //    But when 'heavily loaded' with DATA BYTES shuffled around,
  //    the USB <-> RS-232 adapter seems to queue up requests to switch DTR and RTS,
  //    so their transition times on the oscilloscope are no good indicators
  //    for the times where pInstance->fWritePending or fReadPending change.

} // end AuxCom_RunTxStressTest()


//----------------------------------------------------------------------------
static void AuxCom_RunSerialPortTunnel( T_AuxComPortInstance *pInstance )
       // [in]  pInstance->sRxFifo.fifo = thread-safe RX FIFO for THIS port.
       // [out] AuxComPorts[0..SWI_NUM_AUX_COM_PORTS-1].sTxFifo.fifo = thread-safe
       //         TX FIFO for OTHER ports (configured as 'Serial Port Tunnels'
       //         with the same "tunnel"-number.
  // In contrast to AuxCom_OnReceptionFromNetworkPortTunnel(), shares
  // data received on a SERIAL PORT (here: pInstance) with
  // OTHER SERIAL PORTS on the same "tunnel" (here: AuxComPorts[], an ARRAY).
{

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // "Distributes" data from THIS Aux COM Port's
  //              OTHER Aux COM port's TRANSMIT BUFFERS.
  //   Additional complexity:  Optionally DECODE(!) what travels
  //              through the tunnel (controlled by the "Params" string in the
  //              'Additional COM Port #X'-dialog, here: controlled by
  //              AuxCom_MustPassDataToDecoder().
  //
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Most important (to let the CLIENT / SERVER Tunnels do their job
  //                 over the TCP/IP connection) :
  // Don't drain pInstance->sTxFifo.fifo HERE ! Instead,
  //    let CwNet.c : drain pInstance->sRxFifo.fifo,
  //                  and send ALL IT CAN as fast as possible;
  //    and CwNet.c : CwNet_OnReceive() fill pInstance->sTxFifo.fifo,
  //                  and WAKE UP AuxComThread() as fast as possible.
  // Reason for doing this "as fast as possible", despite the large FIFOs on all sides:
  //   The old Yaesu '5-byte-block' protocol uses FIVE-BYTE-BLOCKS from
  //    PC to the radio but the number of RESPONSE BYTES depends on
  //    the command. There is no delimiter or similar between blocks.
  //    Yaesu's own "specification" is poor, or even wrong:
  //      > All commands sent from the computer to the transceiver
  //      > consist of five-byte blocks, with up to 200 ms between each
  //      > byte.
  //    From KA7OEI:
  //      > It is specified in the manual that these 5 bytes are to be
  //      > sent in quick succession - within 200 milliseconds of each other.
  //      > Experimentation reveals, however, that this would be too slow
  //      > - it is more likely that all 5 bytes must be sent
  //      >   within a 200 millisecond period.
  //    From DL4YHF:
  //      Cannot wait for 'gaps' in bytestream between these X-byte-blocks,
  //      last not least because we don't know which protocol is used here.
  //      If we're lucky, the actual transfer is kind of HALF DUPLEX,
  //      e.g. when the host sends a command, then waits for a response,
  //      and in such ultra-primitive protocols, the radio cannot send
  //      unsolicited reports (e.g. when rotating the VFO knob) anyway.
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Timing seen when observing the old "Yaesu-5-byte-CAT" (KA7OEI:"meow")
  // when "FT817 Commander" was talking to an FT-817ND *directly*,
  // with 4800 bit/second (seems to be the FT-817's default baudrate):
  //
  //  TX (PC to radio) --[00][00][00][00][03]-------------------------[00]...
  //                     : "Read frequency  :  Response:              :
  //                     :        and mode" :   144.48799 MHz CW-N    :
  //  RX (radio to PC) ------------------------[14][44][87][99][82]------
  //                     :      11 ms (*)   :  :        11 ms     :   :
  //  "delta t"          :<---------------->:<>:<---------------->:<->:
  //                                       0.5 ms !               5.8 ms
  //  (*) With 5 bytes * (1+8+2) bits/frame / 4800 bits/second,
  //           expect the 5-byte block to be 11.45 milliseconds long.
  //      The scope doesn't see the last byte's TWO STOPBITS -> 11.0 ms .
  //      Perfect. Thus, the PC as well as the radio leave NO GAPS
  //      between bytes of a frame at all.
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Timing seen when tunneling the old "Yaesu-5-byte-CAT" protocol
  // between "FT817 Commander" and an FT-817ND via RCW Keyer,
  // using "com0com" between Commander and RCW Keyer :
  //
  //  The delay between the last byte of the RESPONSE and the
  //  first byte of the NEXT 5-byte-block wasn't 5.8 ms now,
  //  but a surprising 80(!) millisecond. Tried to figure out why:
  //      * The built-in "network ping" (OVER THE TCP/IP CONNECTION),
  //        reported by RCW Keyer on the DEBUG tab, was only 3 (!) ms.
  //      * Loop intervals in the TCP/IP thread (when *not* woken up
  //        due to received data) were ~~20 ms (ok - see CwNet.c) :
  //          > Loop intervals: 19977 20939 20953 66 19997 21096 20792 20307  us
  //                   xyzThread()  "woken up" ----'
  //      * Times doing 'serial I/O' in AuxComThread() were harmless (peak = 55 us).
  //      * Loop intervals in AuxComThread() :
  //          >
  // See also: CONFIGURABLE SERIAL PORT LATENCIES (managed by Windows,
  //           under "Device Manager->Com Port->Properties->Advanced")
  //           in AuxCom_RunEchoTest() !
  //
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //
  // Summary: To successfully "tunnel" UNKNOWN PROTOCOLS between serial ports,
  //          do all we can (as much as Windows permits) to ...
  //     * keep the latency as low as possible here (TCP/IP will add XX ms),
  //     * keep the timing (including GAPS) between subsequent bytes intact.
  //  But even with the original (measured) 12-millisecond loop interval
  //   of AuxComThread(), hooking two instances of RCW Keyer (Client+Server)
  //   between "FT817 Commander" and an FT-817ND, the transfer was unreliable,
  //   and often caused the 'keyboard lock' symbol appear on the FT-817 screen
  //    .. same problem as reported by KA7OEI, quoted in
  //        C:\cbproj\Remote_CW_Keyer\Yaesu5Byte.c, which occurs when
  //       the FT-817 firmware "misinterprets" a received 5-byte-command !
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  BYTE bBuffer[AUX_COM_PORT_FIFO_SIZE];  // <- no malloc/free; just a speedy STACK VARIABLE !
  int nBytesRead = CFIFO_ReadBytesForTailIndex( &pInstance->sRxFifo.fifo, bBuffer, sizeof(bBuffer),
                &pInstance->iRxBufferTailForSerialPortTunnel ); // [in,out] int *piTailIndex (incremented)
      // (again, see the long explanation above why we cannot use CFIFO_Read() here:
      //  pInstance->sRxFifo.fifo.iTailIndex is used to SEND VIA TCP/IP, in CwNet.c)
  int nBytesWritten;
  T_AuxComPortInstance *pOtherInstance;
  int iAuxComInstance;
       // Decide if bytes received from, or sent to, an Additional COM Port
       // must be passed on to a protocol-specific decoder :
  BOOL fPassToDecoder = AuxCom_MustPassDataToDecoder( pInstance ); // -> TRUE=yes, FALSE=no.
       // '--> If streams passing through this tunnel shall be DECODED, the result it TRUE.
       //      If they shall only be sent from one end to the others, the result is FALSE.

  if( nBytesRead > 0 )  // got something to "distribute" to OTHER LOCAL SERIAL PORTS on the same tunnel ?
   { for(iAuxComInstance=0; iAuxComInstance<SWI_NUM_AUX_COM_PORTS; ++iAuxComInstance )
      { pOtherInstance = &AuxComPorts[iAuxComInstance];
        if( (pOtherInstance->pRigctrlPort->iPortUsage == RIGCTRL_PORT_USAGE_SERIAL_TUNNEL )
          &&(pOtherInstance != pInstance) )  // don't send on the serial port that RECEIVED the data (because this isn't the "Echo Test")
         { if((pOtherInstance->iTunnelIndex < 0 ) // the other instance wants to "receive all" ?
            ||(pOtherInstance->iTunnelIndex==pInstance->iTunnelIndex) ) // matching 'tunnel index' ?
            { // bingo, found a matching "LOCAL serial tunnel", so stuff in the data into THE OTHER INSTANCE'S TX-FIFO:
              nBytesWritten = CFIFO_Write( &pOtherInstance->sTxFifo.fifo, bBuffer/*source*/, nBytesRead, 0.0/*dblTimestamp_s*/ );
              if( nBytesWritten != nBytesRead )
               { // Oops.. CFIFO_Write() could not push in as many bytes into the buffer as we wanted !
                 nBytesWritten = nBytesWritten; // <- set a breakpoint HERE !
                 // 2025-08-25: RCW Keyer ran without buffer overflows for hours,
                 //   with WFView <-> RCW Keyer <-> IC-9700 connected
                 //   via 'Serial Port Tunnel'. But as soon as CLOSING
                 //   WFView, a breakpoint above fired with the following call stack:
                 //   AuxComThread() -> AuxCom_RunSerialPortTunnel(),
                 //    nBytesRead = 60 but nBytesWritten = 44 .
                 //   Obviously, in contrast to a REAL serial port,
                 //   com0com seems to block TRANSMISSION on one side of the
                 //   'Virtual Null-Modem Cable' if the other end is 'not connected',
                 //   despite NO flow-control (where one would expect that
                 //   data passed to WriteFile() in AuxComThread() would simply
                 //   escape into "nowhere", just like the end of a open RS-232-cable).
                 //   To check for this problem 'where it happens' (when calling WriteFile()),
                 //   AuxComThread() now checks the time WriteFile() is "pending",
                 //   and when pending for longer than 1000(?) ms, cries for help
                 //   in the ERROR HISTORY. This happened in 2025-08-25:
                 // > 2448 19:50:16.9 r1 009 FE FE E1 94 14 07 01 28 FD ; PassbandT1=50 %
                 // > 2449 19:50:16.9 t2 009 FE FE E1 94 14 07 01 28 FD ; PassbandT1=50 %
                 //   (at THIS point, WFView was terminated, RCW Keyer kept running,
                 //    and the IC-7300 kept sending SPECTRA that RCW tried to 'tunnel' to WFview)
                 // > AuxComThread: WriteFile() pending endlessly on COM24. 167377 bytes sent, 12333 bytes rcvd.
                 // ,------------------------------------------------|___|
                 // '--> That was ONE END of a "com0com" Virtual Null Modem port pair,
                 //      configured as follows in the "Setup for com0com" (control panel):
                 //  Virtual Port Pair 3 :
                 //   [+] COM24    [v] use Ports class (all other options UNCHECKED)
                 //   [+] COM25    [v] use Ports class (all other options UNCHECKED)
                 //
               } // end if( nBytesWritten != nBytesRead )
              pOtherInstance->fSendImmediately = TRUE;
              // '--> The data written into the TX FIFO ("transmit" from the SERIAL PORT's point of view)
              //      will be pulled out again in ANOTHER AuxComThread() instance very soon
              AuxCom_WakeUp( pInstance );

            } // end if < matching tunnel index ? >
         }   // end if < OTHER serial port ALSO used as CLIENT/SERVER TUNNEL ? >
      }     // end for < iAuxComInstance >
     // Regardless of having distributed the data received from the "tunnel"
     //    to other serial ports or not, check for 'what else to do'
     //    with the data already pulled from pInstance->sRxFifo.fifo :
     if( fPassToDecoder )
      { AuxCom_PassRxOrTxDataToProtocolSpecificDecoder( pInstance, RIGCTRL_ORIGIN_COM_PORT_RX, // here: called after RECEPTION from AuxCom_RunSerialPortTunnel()
           bBuffer, nBytesRead ); // [in] chunk received data (not necessarily a complete, nor a SINGLE 'Message')
      } // end if < pass the received chunk to a protocol-specific decoder (also for the purpose of COMMENTING it for the log)
   }   // end if( nBytesRead > 0 )
}     // end AuxCom_RunSerialPortTunnel()


//----------------------------------------------------------------------------
static void AuxCom_RunVirtualRig( T_AuxComPortInstance *pInstance )
  // Tries to "emulate" a radio on a serial port, instead of "tunneling"
  // data DIRECTLY between a 3rd party application and the real radio.
  // Because module RigControl.c has its own cache for data read from, or sent to,
  // the real radio, READ COMMANDS received on a 'Virtual Radio' port often don't
  // need to be passed on to the radio. WRITE COMMANDS may have passed on to the
  // radio. So most of the work doesn't happen in AuxComPorts.c but RigControl.c .
{
  BYTE buffer[AUX_COM_PORT_FIFO_SIZE];  // <- no malloc/free here; just a speedy STACK VARIABLE !

  int nBytes = CFIFO_Read( &pInstance->sRxFifo.fifo, buffer, sizeof(buffer), NULL/*no timestamp required*/ );
  if( nBytes > 0 )  // got something to pass to the associate RigControl PORT INSTANCE ..
   { AuxCom_PassRxOrTxDataToProtocolSpecificDecoder( // here: called after RECEPTION from AuxCom_RunVirtualRig()
           pInstance, RIGCTRL_ORIGIN_COM_PORT_RX,
           buffer, nBytes ); // [in] chunk received data (not necessarily a complete, nor a SINGLE 'Message')
     // Problem: The 3rd party application may have sent MULTIPLE commands in a single block.
     //          So MULTIPLE of those commands must be responded to .
     //          Solved by a 'callback' from RigControl.c to AuxComPorts.c,
     //          which may be called EVEN LATER, depeding on whether a
     //          certain message must be forwarded to the 'real' radio or not.
     //  -> AuxCom_RigControlCallback() will to the rest, incl. sending responses.
   }   // end if( nBytes > 0 )
} // end AuxCom_RunVirtualRig()


//----------------------------------------------------------------------------
// Tons of bulk / junk to access/control SERIAL PORTS under Windows ..
//    .. some borrowed from the older KeyerThread.c
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
static BOOL AuxCom_OpenAndConfigureSerialPort(
             T_AuxComPortInstance *pInstance, // [in] :  COM port number, bitrate, startbits,parity,stopbits, DTR control, RTS control, etc ...
             HANDLE *pHandle,  // [out] handle to the serial port
             char *sz255Error) // [out] error details as text
  // Not an API function - only called internally, from AuxCom_Start() !
  // You will have a lot of fun to adapt THIS for non-Windows targets.
{
  #define L_TX_BUFFER_SIZE 4096 // write to SERIAL PORT in small chunks (no problem here, in this "fast-running thread loop")
    // e.g. 20 ms * 921600 bits/sec = 18432 bits = 2304 bytes per thread loop

  char szPort[80];
  HANDLE hComPort;
  COMMTIMEOUTS timeouts;
  DCB dcb;       // this funny-named fellow is a "Device Contol Block".
                 // The only type of device that it can control is a SERIAL PORT.
  COMMPROP props; // this funny-named fellow may(!) tell us a bit more about
                 // the serial port driver's current configuration. We are
                 // especially interested about prop.dwMaxTxQueue, the
                 //  > "size of the driver's internal output buffer, in bytes"
  int iDtrControl, iRtsControl;


  if( pInstance->iComPortNumber < 0 )  // "COM port number" set to "NONE" in the configuration ?
   { // COM-port set to "NONE" (in the GUI's combo boxe) -> Don't try to open..
     *pHandle = INVALID_HANDLE_VALUE; // .. and set the caller's handle-variable to "invalid handle"
     return TRUE;  // COM port not open, but not an error when set to "NONE"
     // (the 'keyer thread' may run with less than two COM ports,
     //  especially when used as REMOTE KEYER CLIENT or -SERVER,
     //  with a network connection between them. Or, maybe, in 'code practice'
     //  mode with only a SIDETONE output)
   }


  TIM_StartStopwatch( &pInstance->swWriteFilePending ); // start the stopwatch that LATER checks for 'sudden death' of WriteFile() on e.g. "com0com"

  iDtrControl = ( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_DTR )
              ? DTR_CONTROL_ENABLE : DTR_CONTROL_DISABLE;
  iRtsControl = ( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_RTS )
              ? RTS_CONTROL_ENABLE : RTS_CONTROL_DISABLE;


  wsprintf( szPort, "\\\\.\\COM%d", pInstance->iComPortNumber );
  hComPort = CreateFile( szPort, // [in] pointer to name of the file
         GENERIC_READ | GENERIC_WRITE, // [in] access (read-write) mode
                                    0, // [in] share mode
                                 NULL, // [in] pointer to security attributes
                        OPEN_EXISTING, // [in] "how to create"
                 FILE_FLAG_OVERLAPPED, // [in] file attribute "overlapped" (geek speak for non-blocking)
                                 NULL  // [in] handle to file with attributes to copy
                             );
  if( hComPort != INVALID_HANDLE_VALUE ) // successfully "created" (opened) the "file" (serial port)
   { // Next step (using bare Win32 API to configure the port) :
     // Configure serial data format and baudrate.
     dcb.DCBlength = sizeof( DCB );
     GetCommState( hComPort, &dcb );
     if( pInstance->iBitsPerSecond > 0 )
      { dcb.BaudRate = pInstance->iBitsPerSecond; // override the current baud rate
      }
     dcb.fBinary  = TRUE;             // binary mode, no EOF check
     dcb.fOutxCtsFlow = FALSE;        // no CTS output flow control (CTS INPUT STATE ignored by the serial port driver)
     dcb.fOutxDsrFlow = FALSE;        // no DSR output flow control (DSR INPUT STATE ignored by the serial port driver)
     // Many "simple" level shifting interfaces draw the positive supply from the
     // DTR line (pin 4 of 9pins), so :
     dcb.fDtrControl = iDtrControl;   // DTR [flow] control type :
     // ,--------------'
     // '--> Possible values and their meanings:
     // > DTR_CONTROL_DISABLE : Disables the DTR line when the device is opened
     // >                       and leaves it disabled.
     // > DTR_CONTROL_ENABLE  : Enables the DTR line when the device is opened
     // >                       and leaves it on.
     // > DTR_CONTROL_HANDSHAKE: Enables DTR handshaking. If handshaking is enabled,
     // >                       it is an error for the application to adjust
     // >                       the line by using the EscapeCommFunction function.
     // Wikipedia said:
     //  > When a serial connection is made between two computers using a
     //  > null modem adapter, the DTR and the Data Carrier Detect (DCD)
     //  > lines are typically paired. This allows both ends of the
     //  > connection to sense when the connection is active.
     //  > On many operating systems, including Windows, the DTR line is held
     //  > low while the serial port is unused and not being controlled
     //  > by any applications.
     //       Thus, via 'DTR_CONTROL_ENABLE', we let the other guy know
     //       that WE HAVE OPENED THE INTERFACE from our side.
     dcb.fDsrSensitivity = FALSE;   // DSR sensitivity (FALSE=ignore state of DSR)
     dcb.fTXContinueOnXoff = FALSE; // XOFF continues Tx ?
     dcb.fOutX      = FALSE; // XON/XOFF out flow control
     dcb.fInX       = FALSE; // XON/XOFF in flow control
     dcb.fErrorChar = FALSE; // enable error replacement ? (no)
     dcb.fNull      = FALSE; // enable null stripping ? (no)
     dcb.fRtsControl= iRtsControl; // RTS [flow] control a la Microsoft:
     // ,--------------'
     // '--> RTS_CONTROL_DISABLE: Disables the RTS line when the device is opened
     //    >                      and leaves it disabled.
     //    > RTS_CONTROL_ENABLE : Enables the RTS line when the device is opened
     //    >                      and leaves it on.
     //    > RTS_CONTROL_HANDSHAKE: Enables RTS handshaking. The driver raises
     //    >                      the RTS line when the "type-ahead" (input) buffer
     //    >                      is less than one-half full and lowers the RTS line
     //    >                      when the buffer is more than three-quarters full.
     //    >                      If handshaking is enabled, it is an error for the
     //    >                      application to adjust the line by using the
     //    >                      EscapeCommFunction function.
     //    > RTS_CONTROL_TOGGLE:  Specifies that the RTS line will be high if bytes
     //    >                      are available for transmission. After all buffered
     //    >                      bytes have been sent, the RTS line will be low.
     dcb.fAbortOnError = FALSE; // abort reads/writes on error
     if( (pInstance->iNumDatabits >= 5) && (pInstance->iNumDatabits <= 9) )
      { // Even though SOME UARTS may support 9 data bits per frame, not supported here      {
        // due to the problem of how to pack 9-bit-data into a BYTE-array :
        dcb.ByteSize = (BYTE)pInstance->iNumDatabits;
      }
     else
      { dcb.ByteSize = (BYTE)8; // "number of bits/byte" (what they mean is SERIAL FRAME)
      }
     dcb.fParity  = FALSE;        // disable parity checking ON RECEIVE ...
     switch( pInstance->iParity ) // .. but on TRANSMIT, the parity FORMAT must be set properly:
      { // Translate our own (OS-independent) codes for the PARITY FORMAT into Microsoft's:
        case SER_PORT_PARITY_NONE : // None (N) means that no parity bit is sent and the transmission is shortened.
             dcb.Parity = NOPARITY;
             break;
        case SER_PORT_PARITY_ODD  : // Odd (O) means that the parity bit is set so that the number of 1 bits is odd.
             dcb.Parity = ODDPARITY;
             break;
        case SER_PORT_PARITY_EVEN : // Even (E) means that the parity bit is set so that the number of 1 bits is even.
             dcb.Parity = EVENPARITY;
             break;
        case SER_PORT_PARITY_MARK : // Mark (M) parity means that the parity bit is always set to the mark signal condition (1 bit value).
             dcb.Parity = MARKPARITY;
             break;
        case SER_PORT_PARITY_SPACE: // Space (S) parity always sends the parity bit in the space signal condition (0 bit value).
             dcb.Parity = SPACEPARITY;
             break;
        default:
             dcb.Parity = NOPARITY;
             break;
      }
     switch( pInstance->iNumStopbits ) // again, translate the codes from CwKeyer.h to Microsoft's ..
      { case SER_PORT_STOPBITS_1   :
             dcb.StopBits = ONESTOPBIT;
             break;
        case SER_PORT_STOPBITS_2   :   // a stoneage FT-817 insisted on this
             dcb.StopBits = TWOSTOPBITS;
             break;
        case SER_PORT_STOPBITS_1_5 :  // omg.. 1.5 stopbits (one and a half) !
             dcb.StopBits = ONE5STOPBITS;
             break;
        default:
             dcb.StopBits = ONESTOPBIT;
             break;
      }
     // dcb.XonChar = 17;    // Tx and Rx XON character
     // dcb.XoffChar= 19;    // Tx and Rx XOFF character
     // dcb.ErrorChar= 0;    // error replacement character
     // dcb.EofChar = 0;     // end of input character
     // dcb.EvtChar = 0;     // received event character
     SetCommState( hComPort, &dcb );

     // Next step (optional, and not "guaranteed to work" [*] ) :
     // Try to reduce the TRANSMIT BUFFER (output buffer), but first,
     // find out the COM port driver's current setting:
     memset( &props, 0, sizeof(props) );
     props.wPacketLength = sizeof(props); // "PacketLength" is a funny name for 'size of this struct' !
     props.dwProvSpec1 = COMMPROP_INITIALIZED;
     if( GetCommProperties( hComPort, &props ) )
      { // > "If the function succeeds, the return value is nonzero" ....
        // With an USB<->RS232 adapter with FTDI chip, got here with
        // props.wPacketVersion = 2
        // props.dwMaxBaud      = 0x10000000 = "BAUD_USER". A-ha.
        // props.dwMaxTxQueue = 0 = "no maximum value is imposed by the serial provider"
        // props.dwMaxRxQueue = 0 = "no maximum value is imposed by the serial provider"
        // props.dwCurrentTxQueue = 0 = "the value is unavailable" (Nnngrrr...)
        // Next step (added 2025-06-21 when experiencing TX(!)-buffer-overflows in AuxCom_RunEchoTest():
        SetupComm( hComPort, // <- this funny-named fellow may be able to configure BUFFER SIZES (or not..)
           L_TX_BUFFER_SIZE,  // [in] dwInQueue  (number of bytes)
           L_TX_BUFFER_SIZE); // [in] dwOutQueue (number of bytes): e.g. 20 ms * 921600 bits/sec = 18432 bits = 2304 bytes per thread loop
                  // '--> With an FTDI interface, this had NO EFFECT AT ALL.
      } // end if <GetCommProperties() successful >

     // Next step (using bare Win32 API to configure the port) :
     // Configure the COM port "timeouts", so that ReadFile() / WriteFile()
     //  won't block for more than a few dozen milliseconds:
     GetCommTimeouts( hComPort, &timeouts );
     timeouts.ReadIntervalTimeout       = 10; // (1)  [before 2025-06-21: 0]
     // About COMMTIMEOUTS.ReadIntervalTimeout   (1) :
     // > "maximum time, in ms, allowed to elapse between the arrival of two characters" . (..)
     // > A value of MAXDWORD, combined with zero values for both the
     // > ReadTotalTimeoutConstant and ReadTotalTimeoutMultiplier members,
     // > specifies that the read operation is to return immediately
     // > with the characters that have already been received, even if
     // > no characters have been received.
     // The above WOULD BE used to make "ReadFile" non-blocking.
     //
     timeouts.ReadTotalTimeoutMultiplier = 0; // (2)
     // About   *ReadTotalTimeoutMultiplier*     (2) :
     // > The multiplier used to calculate the total time-out period
     // > for read operations, in milliseconds. For each read operation,
     // > this value (2) is multiplied by the requested number of bytes to be read.
     // (In this application, ReadFile() is always called with the MAXIMUM
     //  size of our own receive-buffer to drain as many bytes bytes as
     //  possible in a single call, and of course we never wait
     //  until the COMPLETE buffer is filled.
     //  Thus, a 'real' ReadTotalTimeoutMultiplier is useless for this purpose. )
     //
     timeouts.ReadTotalTimeoutConstant = 5;   // (3)  [before 2025-06-21: 20, see AuxCom_RunEchoTest() ]
     // About * ReadTotalTimeoutConstant *       (3)  :
     // > A constant used to calculate the total time-out period for read operations,
     // > in milliseconds. For each read operation, this value (3) is added
     // > to the product of the ReadTotalTimeoutMultiplier member (2)
     // > and the requested number of bytes.
     // > Remarks
     // > If an application sets ReadIntervalTimeout (1) to MAXDWORD
     // >         **and** ReadTotalTimeoutMultiplier (2) to MAXDWORD
     // >      **and** sets ReadTotalTimeoutConstant (3) to a value greater
     // > than zero and less than MAXDWORD, one of the following occurs
     // > when the ReadFile function is called:
     // >   * If there are any bytes in the input buffer,
     // >      ReadFile returns immediately with the bytes in the buffer.
     // >   * If there are no bytes in the input buffer, ReadFile waits until
     // >      a byte arrives and then returns immediately.
     // >   * If no bytes arrive within the time specified by
     // >      ReadTotalTimeoutConstant, ReadFile times out.
     //
     timeouts.WriteTotalTimeoutMultiplier= 0;  // (4)
     // About * WriteTotalTimeoutMultiplier *  // (4) :
     // > The multiplier used to calculate the total time-out period
     // > for write operations, in milliseconds. For each write operation,
     // > this value is multiplied by the number of bytes to be written.
     timeouts.WriteTotalTimeoutConstant  = 0;  // (5) [before 2025-06-21: 0, see AuxCom_RunEchoTest() ]
     // About * WriteTotalTimeoutConstant   *  // (5) :
     // > A constant used to calculate the total time-out period for
     // > write operations, in milliseconds. For each write operation,
     // > this value is added to the product of the
     // > WriteTotalTimeoutMultiplier member and the number of bytes
     // > to be written.
     // >     A value of zero for both the WriteTotalTimeoutMultiplier
     // >     **and** WriteTotalTimeoutConstant members indicates that
     // >     total time-outs are not used for write operations.
     //    (the latter appeared ok for overlapped I/O :
     //         ReadIntervalTimeout        = MAXDWORD,
     //         ReadTotalTimeoutMultiplier = MAXDWORD,
     //         ReadTotalTimeoutConstant   = 5 .. 20 [ms, but see TESTS further above]
     SetCommTimeouts( hComPort, &timeouts );

     // For Winkeyer1, K1EL suggests to wait for 400 milliseconds after the above,
     // and then purge (discard) any garbage in the serial port's receive buffer:
     // From the "WK_USB Application Interface Guide v1.1" page 3:
     //  > If you plan to support both WK1 and WK2 you need a single 400 msec
     //  > delay after asserting the RTS and DTR lines to allow WK1 to finish
     //  > its power up init.
     // In Windows, they use yet another funny-named COM port API function for this:
     PurgeComm( hComPort, PURGE_RXCLEAR );
       // (Wont block the caller for 400 milliseconds here !
       //  If a "PurgeComm()" is also required LATER, simply DISCARD received bytes
       //  for the first XXX milliseconds after start,
       //  somewhere in AuxCom_Winkeyer.c : AuxCom_RunWinkeyerEmulator()  ).


     *pHandle = hComPort;    // store the serial port's handle to access it
     return TRUE;
   }
  else // could't open the serial port, so don't try to CONFIGURE it :
   { snprintf( sz255Error, 255, "Could not open serial port 'COM%d'",(int)pInstance->iComPortNumber );
     return FALSE;
   }
} // end AuxCom_OpenAndConfigureSerialPort()



//---------------------------------------------------------------------------
static void AuxCom_CheckLastErrorAfterTroubleWithSerialPort(
             T_AuxComPortInstance *pInstance, // [in,out] pInstance->hSerialPort, possibly closed and set to INVALID_HANDLE_VALUE
             char *pszTriedWhat )    // [in] : e.g. "ReadFile"
  // If the check of the 'Last Error Code' indicates a real error,
  // the serial port's handle is closed, and the handle value set to "illegal".
  // The application(?) / worker thread may decide to re-open the port a few times,
  //     after waiting for a considerable time to allow the OPERATOR
  //     to re-insert the USB device or similar.
{
  DWORD dwLastError = GetLastError(); // <- MS claims this is 'thread aware' but.. :
    // > You should call the GetLastError function immediately when a function's
    // > return value indicates that such a call will return useful data.
    // > That is because some functions call SetLastError with a zero when
    // > they succeed, wiping out the error code set by the most recently
    // > failed function.
  char sz127[128];
  BOOL fCloseSerialPort = FALSE;

  // Is it a REAL error (not one of those 'false alerts' caused by overlapped I/O) ?
  switch( dwLastError )
   { case ERROR_IO_PENDING   : // the infamous '997' : "Overlapped I/O operation is in progress."
     case ERROR_IO_INCOMPLETE: // the infamous '996' : "Overlapped I/O event is not in a signaled state."
        break;
     case ERROR_OPERATION_ABORTED: // '995' : "The I/O operation has been aborted because of either a thread exit or an application request."
        fCloseSerialPort = TRUE;
        if( (pInstance->dwSerialPortErrors < 10 )   // Cry for help in the 'error history' ..
         && (pInstance->iLastErrorLine != (__LINE__+1) ) ) // .. but don't show the same error TWICE (without other errors in between)
         { pInstance->iLastErrorLine = __LINE__;
           ++pInstance->dwSerialPortErrors;
           // "The I/O operation has been aborted because of either a thread exit or an application request."
           // Whow. Now go and explain that to your customer. Good luck asking for mercy.
           ShowError( ERROR_CLASS_ERROR, "AuxComThread: I/O on the RADIO PORT failed (%s)", pszTriedWhat );
         }
        break;
     default: // guess it's another REAL error, related with a "COM" port :
        fCloseSerialPort = TRUE;
        if( (pInstance->dwSerialPortErrors < 10 )   // Cry for help in the 'error history' ..
         && (pInstance->iLastErrorLine != (__LINE__+1) ) ) // .. but don't show the same error TWICE (without other errors in between)
         { pInstance->iLastErrorLine = __LINE__;
           ++pInstance->dwSerialPortErrors;
           // When pulling out the USB cable connecting the keyer to an IC-7300,
           // the result (shown in the 'Debug' tab) sometimes was:
           //  "Access is denied".
           //    '--> OH MY GOD.
           // In other cases, the result in sz127 was a long, chatty story like this (#995) :
           // > "The I/O operation has been aborted because of either a thread exit or an application request."
           // Whow. Now go and explain that to your customer. Good luck asking for mercy.
           // > "The communication with the remotely controlled radio stopped because your cat has pulled out the USB cable."
           // Well, maybe, after Microsoft forced the introduction of AI in Windows.
           UTL_LastWindowsErrorCodeToString( dwLastError, sz127, 127/*iMaxDestLen*/ );
           ShowError( ERROR_CLASS_ERROR, "AuxComThread: RADIO PORT failed (%s -> %s)", pszTriedWhat, sz127 );
         }
        break; // end default < dwLastError considered to be a REAL error with the "COM" / "Comm" port API >
   } // end switch( dwLastError )
  if( fCloseSerialPort )
   { if( pInstance->hComPort != INVALID_HANDLE_VALUE )
      { CloseHandle( pInstance->hComPort );
        pInstance->hComPort = INVALID_HANDLE_VALUE;
      }
   }
} // end AuxCom_CheckLastErrorAfterTroubleWithSerialPort()


//---------------------------------------------------------------------------
DWORD WINAPI AuxComThread(   // Implements the serial port 'worker thread'...
                LPVOID lpParam ) // [in] : type dictated by Windows,
                // but it's actually a pointer to our T_AuxComPortInstance
  // 'Serial port worker thread' : Polls the paddle- or straight key contacts,
  //   and TRIES TO (CW-)"key" the radio via USB / Virtual COM port.
  //   Later versions could also play text from memory, using CwGen.c .
  //   Even later versions also produced a sidetone on a selectable
  //                'audio' device (speaker, soundcard) using CwDSP.c .
  //   Also, this thread may interact with the 'CW Network',  CwNet.c .
  //
  // Like applications in a PLC (Programmable Logic Controller) ...
  //   * this thread ideally runs with a constant loop time (2 ms),
  //   * samples all inputs for the 'process image' (PLC term) only ONCE per loop,
  //   * and drives all outputs from the 'process image' only ONCE per loop,
  //   * must not call any subroutine that consumes more than say ONE LOOP TIME.
  //
{
  T_AuxComPortInstance *pInstance = (T_AuxComPortInstance *)lpParam;
  int iAuxComInstance = AuxCom_InstancePtrToIndex( pInstance );
      // '--> Array index into CwKeyer_Config.AuxComXYZ[iAuxComInstance]

  BYTE  bMyTxBuffer[L_TX_BUFFER_SIZE];
  #define L_THREAD_MAX_MS_PER_TX_LOOP 5
    // If it wasn't WINDOWS, we'd only wait for 5 ms per thread loop
    // to keep the 'bursts' small but sent as frequently as possible.
    // But "expect the worst" (WaitForSingleObject() waiting for DOZENS OF MILLISECONDS,
    //                         or even worse, Sleep() ),
    // so make L_TX_BUFFER_SIZE much larger so the next WriteFile()
    // can send larger blocks to 'catch up' with the buffer-filling-rate .
  DWORD dwNumBytesToWrite, dwNumBytesWritten, dwNumBytesRead;
  DWORD dwModemStatus;
  DCB   sCommState; // this funny-named fellow is a "Device Contol Block".
       // The only type of device that it can control is a SERIAL PORT.
  BOOL  fDigitalInputState, fMorseOutput, fMorseOutput_sent, fResult;
  T_CwNet *pCwNet = CwKeyer_Config.pCwNet; // optional instance for CW keying "via network"


  // ex: BOOL  fWaitedSomewhere;  // Useless. We MEASURE the time spent in a loop,
  // to find out if Windows has "waited" in any of the API functions called below.
  T_TIM_Stopwatch sw_BeginOfThreadLoop, sw_SpeedTest, sw_MorseTxFifo, sw_TxBusy,
                  sw_DotInputWatchdog, sw_DashInputWatchdog;
  long nMicrosecondsSinceLastCall;
  int  t_us, t_ms, iResult;
  WORD wCwChar;
  char *cp;
  char sz127[128];


 // INT i,iMax,iChn;
 // INT error_code;

  // Also here (similar as for CSound_SerialPortReaderThread) : See Microsoft's
  //     "Serial Communications in Win32" by Allen Denver,
  // saved on WB's workhorse as file:///C:/downloads/Windows_Programming/msdn_serial.htm
  // or (slightly vamped up) C:\downloads\Windows_Programming\Serial Communications in Win32.pdf .
  int   iThreadExitCode = 1;
         // Note: The code STILL_ACTIVE must not be used as parameter for
         // ExitThread(), but what else ? See Win32 API help !
         // in winbase.h:
         //    #define STILL_ACTIVE STATUS_PENDING
        // in winnt.h:
        //    #define STATUS_PENDING ((DWORD   )0x00000103L)

  DWORD dwWaitResult, dwRes, dwNewSignalStates;
  BOOL  fSignalStatesModified;
  OVERLAPPED osWrite = { 0 };
  OVERLAPPED osRead  = { 0 };
  char  sz4Temp[4];
  BYTE  b256RxBuffer[256], b256TxBuffer[256];

  pInstance->fReadPending  = FALSE;
  pInstance->fWritePending = FALSE;
  pInstance->iThreadStatus = AUX_COM_THREAD_STATUS_RUNNING;
  pInstance->dwThreadLoops = pInstance->dwThreadErrors = pInstance->dwSerialPortErrors = 0;
  pCwNet->fCurrentRemotelyReceivedKeyDownState = pCwNet->fNextRemotelyReceivedKeyDownState = FALSE;


  // Create the overlapped event for READING (only used within this function, thus LOCAL).
  // Must be closed before exiting to avoid a handle leak.
  osWrite.hEvent = CreateEvent(               NULL, FALSE, FALSE,  NULL);
      // Pointer to security attributes ---------'    |     |       |
      // Flag for manual-reset event.  <--------------'     |       |
      //    If FALSE, Windows automatically resets          |       |
      //    the state to nonsignaled after a                |       |
      //    single waiting thread has been released.        |       |
      // Flag for initial state: FALSE = "not signalled" <--'       |
      // Pointer to event-object name  <----------------------------'
  // Similar as above for READING from the serial port:
  // Because we opened the serial port with FILE_FLAG_OVERLAPPED, this is mandatory:
  osRead.hEvent  = CreateEvent(NULL, FALSE, FALSE,  NULL);

  if( (osWrite.hEvent == NULL) || (osRead.hEvent  == NULL) )
   { pInstance->iThreadStatus = AUX_COM_THREAD_STATUS_TERMINATED;
     return -1;  // Error creating overlapped event; abort.
     // (if NOT ALL 'creations' failed, there's a leak now. Oh well.)
   }

  CwKeyer_fUpdateAllOutputs = TRUE; // send all outputs (with dozens of EscapeCommFunction()-calls) at least ONCE


  //- - - - - - - - - No 'lazy return' after this point ! - - - - - - - - - - -

  TIM_StartStopwatch( &sw_BeginOfThreadLoop ); // don't leave this un-initialized..
  TIM_StartStopwatch( &sw_MorseTxFifo );
  TIM_StopStopwatch( &sw_TxBusy );  // not TRANSMITTING yet so keep THIS stopwatch stopped :o)
  TIM_StopStopwatch( &CwKeyer_swMorseActivityTimer );
  TIM_StartStopwatch( &sw_DotInputWatchdog );
  TIM_StartStopwatch( &sw_DashInputWatchdog );

  fMorseOutput_sent = FALSE;
  Keyer_fDotInputWasPassive = Keyer_fDashInputWasPassive = FALSE;

  // Almost endless SERIAL PORT thread loop begins HERE...................
  while( pInstance->iThreadStatus == AUX_COM_THREAD_STATUS_RUNNING )
   {
     ++pInstance->dwThreadLoops;
     // Measure the number of microseconds elapsed since getting HERE in the previous thread loop:
     pInstance->dw8ThreadIntervals_us[pInstance->dwThreadLoops&7] = TIM_ReadAndRestartStopwatch_us( &sw_BeginOfThreadLoop );
       //  As for other worker threads of this application,
       //  test results can be displayed on the RCW Keyer's DEBUG tab. Example:
       // > ADDITIONAL COM PORT #2 (on COM8)..
       // >  Status: Client/Server Tunnel, tx=4885, rx=1952 bytes, 14808 loops
       // >  Serial I/O times: av= 3 us, pk= 23 us
       // >  Loop intervals: 10052 10185 10024 10995 10969 10022 10999 10934  us



     // WAIT for an event from another thread; e.g. if we've got something TO SEND a.s.a.p. :
     if( pInstance->hEventToWakeUpThread != NULL ) // shouldn't be NULL, but PLAY SAFE..
      { HERE_I_AM__AUX_COM(); // -> AuxCom_iLastSourceLine here ? Died in 'WaitForSingleObject()' !
        if( pInstance->fSendImmediately && ( ! pInstance->fWritePending ) )
         {  pInstance->fSendImmediately = FALSE; // here: cleared when suppressing the call of WaitForSingleObject()
         }
        else // no data waiting to be sent "immediately", so avoid a busy-spinning thread loop:
        if( (dwWaitResult=WaitForSingleObject( pInstance->hEventToWakeUpThread, 10/*ms plus a LOT (*) */ ) )
             == WAIT_OBJECT_0 ) // the event is in a "signaled" state so RESET it (for the next time)
         { HERE_I_AM__AUX_COM();
         }
        else
         { HERE_I_AM__AUX_COM(); // -> AuxCom_iLastSourceLine here ? failed to wait for the "thread-wake-up-event" !
           // After resuming from the PC's "sleep mode" (?), WaitForSingleObject()
           // didn't wait for circa 2000 us anymore, but for only about 10 us
           // per loop .. in other words, AuxComThread() turned into a CPU-hogging
           // monster (busy-spinning loop). Tried to find out why.
           // From the documentation about WaitForSingleObject() :
           // > If the function succeeds, the return value indicates the event
           // > that caused the function to return. (...)
           // > WAIT_OBJECT_0  : This is what we EXPECT when all works as planned.
           // > WAIT_ABANDONED :
           // >   The specified object is a mutex object that was not released
           // >   by the thread that owned the mutex object
           // >   before the owning thread terminated.
           // > WAIT_TIMEOUT : The time-out interval elapsed, and the object's
           // >   state is nonsignaled.
           //    (will happen here regularly, unless the MultiMedia-Timer wakes up
           //     this thread periodically. It MAY have to do that;
           //     see AuxComPorts_
           // > WAIT_FAILED : The function has failed. To get extended error
           // >   information, call GetLastError.
           switch( dwWaitResult )
            { case WAIT_ABANDONED :
                   ++pInstance->dwThreadErrors;
                   break;
              case WAIT_TIMEOUT   :  // hopefully just a 'temporary hiccup' from Mr. Multimedia-Timer ..
                   // ex: ++pInstance->dwThreadErrors; // not sure if Mr. Multimedia-Timer is involved,
                   //     so getting a TIMEOUT from WaitForSingleObject() isn't an error .
                   break;
              case WAIT_FAILED    :
                   ++pInstance->dwThreadErrors;
                   if( (pInstance->dwThreadErrors < 10 )  // Cry for help in the 'error history' ?
                    && (pInstance->iLastErrorLine != (__LINE__+1) ) ) // .. but don't show the same error TWICE (without other errors in between)
                    { pInstance->iLastErrorLine = __LINE__;
                      UTL_LastWindowsErrorCodeToString( GetLastError(), sz127, 127/*iMaxDestLen*/ );
                      ShowError( ERROR_CLASS_ERROR, "AuxComThread: Failed to wait for something (%s)", sz127 );
                    }
                   break;
            }
         } // end < WaitForSingleObject() returned something 'unusual' >
      }
     else // it there's no "event" (handle) to wait for, Sleep() instead ...
      { HERE_I_AM__AUX_COM();
        Sleep(5); // <- This completely sucks for periodic events,
           // because windows will let this thread "sleep"
           // for MUCH LONGER than 5 milliseconds (often 16 ms),
           // and the thread cannot wake up earlier (without a valid hEventToWakeUpThread).
           // But SLEEPING here (when there's no "multimedia timer")
           // is better than burning CPU time in a busy-spinning loop.
      } // end else < invalid pInstance->hEventToWakeUpThread >
     HERE_I_AM__AUX_COM();


     // Poll the current "digital inputs" (Modem status lines)
     // by calling GetCommModemStatus() as a BLOCKING CALL, because ..
     // (from Microsoft about "Communications Events")  :
     // > When monitoring an event that occurs when a signal (CTS, DSR, and so on)
     // > changes state, WaitCommEvent reports the change, but not the current state.
     // > To query the current state of the CTS (clear-to-send), DSR (data-set-ready),
     // > RLSD (receive-line-signal-detect), and ring indicator signals,
     // > a process can use the GetCommModemStatus function.
     dwNewSignalStates = pInstance->dwDigitalSignalStates;
     TIM_StartStopwatch( &sw_SpeedTest );  // measure time spent in the following calls:
     if( pInstance->hComPort != INVALID_HANDLE_VALUE ) // is the serial port still open ?
      { // > The GetCommModemStatus function retrieves modem control-register values.
        // > If the function succeeds, the return value is nonzero.
        // > If the function fails, the return value is zero.
        if( GetCommModemStatus( pInstance->hComPort, &dwModemStatus ) )
         { if( dwModemStatus & MS_CTS_ON )
            { // > "The CTS (clear-to-send) signal is on." ->
              dwNewSignalStates |= AUX_COM_DIG_SIGNAL_CTS;
            }
           else
            { dwNewSignalStates &= ~AUX_COM_DIG_SIGNAL_CTS;
            }
           if( dwModemStatus & MS_DSR_ON )
            { // > "The DSR (data-set-ready) signal is on." ->
              dwNewSignalStates |= AUX_COM_DIG_SIGNAL_DSR;
            }
           else
            { dwNewSignalStates &= ~AUX_COM_DIG_SIGNAL_DSR;
            }
           if( dwModemStatus & MS_RING_ON )
            { // > "The ring indicator signal is on." ->
              dwNewSignalStates |= AUX_COM_DIG_SIGNAL_RI;
            }
           else
            { dwNewSignalStates &= ~AUX_COM_DIG_SIGNAL_RI;
            }
           if( dwModemStatus & MS_RLSD_ON )
            { //
              // Microsoft: > "The RLSD (receive-line-signal-detect) signal is on."
              // (a more common name for this funny thing is "DCD" = "Data Carrier Detect")
              dwNewSignalStates |= AUX_COM_DIG_SIGNAL_DCD;
            }
           else
            { dwNewSignalStates &= ~AUX_COM_DIG_SIGNAL_DCD;
            }
         } // end if < GetCommModemStatus() successful > ?
      }   // end if( pInstance->hComPort != INVALID_HANDLE_VALUE ) ?
     if( dwNewSignalStates != pInstance->dwDigitalSignalStates )
      { fSignalStatesModified = TRUE;  // <- may use this flag to wake up OTHER threads to take notice !
      }
     pInstance->dwDigitalSignalStates = dwNewSignalStates;
     HERE_I_AM__AUX_COM();


     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // Run the 'Programmable logic' to drive OUTPUTS on this serial port
     //    (and flags polled somewhere else, e.g. AUX_COM_DIG_SIGNAL_DOT/DASH, etc)
     // [in] pAuxCom->dwDigitalSignalStates : Kind of 'process image' as in a PLC.
     //      Bitwise combination of AUX_COM_DIG_SIGNAL_DCD/DSR/CTS/RI/DTR/RTS .
     // [in] CwKeyer_Config.sz40AuxComDTR/RTS/DCD/DSR/CTS/RI[iAuxComInstance] .
     // [out] pAuxCom->dwDigitalSignalStates : AUX_COM_DIG_SIGNAL_DTR + _RTS
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     AuxComThread_RunProgrammableLogic( pInstance, iAuxComInstance );


     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // Switch any of the DIGITAL OUTPUTS (modem control signals) ?
     //    [in]  <various GLOBAL variables, e.g. Keyer_dwCurrentSignalStates>
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // One may be tempted to use the bulky "SetCommState()" for this purpose, but:
     // > The EscapeCommFunction function directs a specified
     // > communications device to perform an extended function.
     // > BOOL EscapeCommFunction(  HANDLE hFile,  DWORD dwFunc )
     // > hFile  [in] Handle to the communications device.
     // >             The CreateFile function returns this handle.
     // > dwFunc [in] Extended function to be performed.
     // >             This parameter can be one of the following values.
     // > CLRBREAK Restores character transmission and places
     // >   the transmission line in a nonbreak state. The CLRBREAK
     // >   extended function code is identical to the ClearCommBreak function.
     // > SETBREAK Suspends character transmission
     // >        and places the transmission line in a break state
     // >        until the ClearCommBreak function is called
     // >        (or EscapeCommFunction is called with the CLRBREAK extended
     // >        function code). The SETBREAK extended function code
     // >        is identical to the SetCommBreak function.
     // >        Note that this extended function does not flush data
     // >        that has not been transmitted.
     //  WB: BREAK on  = continuous high pegel on TXD
     //      BREAK off = TXD controlled by serial output, or low pegel (negative voltage = IDLE STATE)
     // > CLRDTR Clears the DTR (data-terminal-ready) signal.
     // > SETDTR  Sends the DTR (data-terminal-ready) signal.
     // > CLRRTS Clears the RTS (request-to-send) signal.
     // > SETRTS  Sends the RTS (request-to-send) signal.
     // > SETXOFF Causes transmission to act as if an XOFF character has been received.
     // > SETXON  Causes transmission to act as if an XON character has been received.
     // Note: With certain crappy USB<->RS-232 adapters, or crappy drivers,
     //       the "EscapeCommFunction" can be INCREDIBLY SLOW .
     TIM_StartStopwatch( &sw_SpeedTest );  // "benchmark" the Windows API functions called below
     if( pInstance->hComPort != INVALID_HANDLE_VALUE ) // is the serial port still open ?
      {
        fResult = TRUE;  // so far, no error from EscapeCommFunction() ..

        // [in]  pInstance->dwLastSentOutputStates, pInstance->dwDigitalSignalStates;
        // [out] pInstance->dwLastSentOutputStates : set to dwDigitalSignalStates below.
        if(  ( pInstance->dwDigitalSignalStates  & AUX_COM_DIG_SIGNAL_DTR )
           > ( pInstance->dwLastSentOutputStates & AUX_COM_DIG_SIGNAL_DTR ) )
         { // must turn DTR on this serial port ON ("enable") :
           fResult &= EscapeCommFunction( pInstance->hComPort, SETDTR );  // DTR high (positive voltage on RS-232)
           // > "If the function succeeds, the return value is nonzero."
           pInstance->dwLastSentOutputStates |= AUX_COM_DIG_SIGNAL_DTR;
         }
        if(  ( pInstance->dwDigitalSignalStates  & AUX_COM_DIG_SIGNAL_DTR )
           < ( pInstance->dwLastSentOutputStates & AUX_COM_DIG_SIGNAL_DTR ) )
         { // must turn DTR on this serial port OFF ("disable") :
           fResult &= EscapeCommFunction( pInstance->hComPort, CLRDTR );  // DTR low (negative voltage on RS-232)
           pInstance->dwLastSentOutputStates &= ~AUX_COM_DIG_SIGNAL_DTR;
         }
        if(  ( pInstance->dwDigitalSignalStates  & AUX_COM_DIG_SIGNAL_RTS )
           > ( pInstance->dwLastSentOutputStates & AUX_COM_DIG_SIGNAL_RTS ) )
         { // must turn RTS on this serial port ON ("enable") :
           fResult &= EscapeCommFunction( pInstance->hComPort, SETRTS );  // RTS high (positive voltage on RS-232)
           // > "If the function succeeds, the return value is nonzero."
           pInstance->dwLastSentOutputStates |= AUX_COM_DIG_SIGNAL_RTS;
         }
        if(  ( pInstance->dwDigitalSignalStates  & AUX_COM_DIG_SIGNAL_RTS )
           < ( pInstance->dwLastSentOutputStates & AUX_COM_DIG_SIGNAL_RTS ) )
         { // must turn RTS on this serial port OFF ("disable") :
           fResult &= EscapeCommFunction( pInstance->hComPort, CLRRTS );  // RTS low (negative voltage on RS-232)
           pInstance->dwLastSentOutputStates &= ~AUX_COM_DIG_SIGNAL_RTS;
         }
        if( (!fResult) && ( pInstance->sz255LastError[0] == '\0' ) )
         { sprintf(pInstance->sz255LastError, "EscapeCommFunction() failed on COM%d !",
                      (int)CwKeyer_Config.iComPortNumber_IN );
         }

        // Regardless if the serial port is really used for SERIAL COMMUNICATION
        // or not, drain the port's RECEIVE buffer here, and place the result
        // in a thread-safe, lock-free, circular FIFO:
        dwNumBytesRead = 0;
        if( ! pInstance->fReadPending )
         { // About Windows's "ReadFile()" :
           // > If hFile is opened with FILE_FLAG_OVERLAPPED, the lpOverlapped
           // > parameter must point to a valid and unique OVERLAPPED structure,
           // > otherwise the function can incorrectly report that       |
           // > the read operation is complete.                          |
           if( /*BOOL*/ReadFile( pInstance->hComPort,                //  |
                                 b256RxBuffer, sizeof(b256RxBuffer), //  |
                                 &dwNumBytesRead,                    //  |
                                 &osRead ) )  // <-----------------------'
            { // Arrived here ? Surprise surprise; despite the OVERLAPPED thingy,
              // ReadFile seems to have completed IMMEDIATELY,
              // so take dwNumBytesRead for granted, and PROCESS those bytes
              // further below (after also checking for an overlapped I/O completion)
              dwNumBytesRead = dwNumBytesRead; // <-- place for a BREAKPOINT
            }
           else // ReadFile() returned FALSE : We're optimistic and guess the operation
            { // is PENDING .. which is not an error.
              pInstance->fReadPending = TRUE;
              AuxCom_CheckLastErrorAfterTroubleWithSerialPort( pInstance, "ReadFile" );
            }
         }
        else // a ReadFile() on the serial port is ALREADY PENDING ->
         { if( /*BOOL*/GetOverlappedResult( pInstance->hComPort,
                &osRead, &dwNumBytesRead, FALSE/* do NOT wait*/ ) )
            { // GetOverlappedResult() claimed success for the previous ReadFile(),
              // which doesn't necessarily mean we really RECEIVED something !
              pInstance->fReadPending = FALSE; // all we know it ReadFile() isn't PENDING anymore !
              // Regardless of dwNumBytesRead was set HERE or immediately
              // in ReadFile(), whatever was received will be processed further below.
              // Due to the "commtimeouts", often got here with the overlapped I/O
              // "completed", but NOTHING received. That's a feature, not a bug.
              // Guess what happened with ReadFile() / GetOverlappedResult()
              //            after pulling out the USB cable ?
              // NOTHING !
            }
           else // GetOverlappedResult( pInstance->hComPort ..) failed...
            { // Maybe THIS is a way to reliably detect a broken-down USB connection ?
              AuxCom_CheckLastErrorAfterTroubleWithSerialPort( pInstance, "WaitForRead" );
            }
         } // end if( fReadFromRadioPortPending ) ?
        if( dwNumBytesRead > 0 ) // received something on this serial port ? Pass it to our FIFO.
         { // The rest ("processing") happens in e.g. AuxCom_RunWinkeyerHost(),
           //    AuxCom_RunWinkeyerEmulator(), AuxCom_RunSerialPortTunnel(),
           //    AuxCom_RunEchoTest(),  AuxCom_RunTxStressTest(), etc .
           pInstance->dwNumBytesRcvd += dwNumBytesRead;
           CFIFO_WriteForMultipleReaders(
                &pInstance->sRxFifo.fifo, b256RxBuffer, (int)dwNumBytesRead );
           // ,-|______________________|
           // '--> Drained in subroutines called from AuxComThread(),
           //      depending on pInstance->pRigctrlPort->iPortUsage,
           //      e.g. AuxCom_Winkeyer.c : AuxCom_RunWinkeyerEmulator().
           //   There may be no "primary reader" of this FIFO that drains it,
           //   so use CFIFO_WriteForMultipleReaders(), not CFIFO_Write() .
           TIM_StartStopwatch( &pInstance->swRxAckPulse );
         }

        // Use ANOTHER thread-safe, circular FIFO to TRANSMIT DATA from this thread's point of view:
        if( ! pInstance->fWritePending )
         { double dblTimestampFromFIO_s;
           int nBytesToSend = CFIFO_GetNumBytesReadable( &pInstance->sTxFifo.fifo );
           if( nBytesToSend > sizeof(b256TxBuffer) )
            {  nBytesToSend = sizeof(b256TxBuffer);
            }
           if( nBytesToSend > 0 )
            {  nBytesToSend = CFIFO_Read( &pInstance->sTxFifo.fifo, b256TxBuffer, nBytesToSend, &dblTimestampFromFIO_s );
            } // end if( nBytesToSend > 0 )
           if( nBytesToSend > 0 )  // ok, REALLY got something to send on the serial port .. so START writing (overlapped):
            {
              if( (pInstance->pRigctrlPort->iPortUsage != RIGCTRL_PORT_USAGE_VIRTUAL_RIG ) // don't add TRANSMITTED MESSAGES to the traffic log twice (*)
               && AuxCom_MustPassDataToDecoder( pInstance ) ) // -> TRUE=yes, FALSE=no ...
               { // (*) When SENDING A RESPONSE to a remote client, the message has already been added
                 //     to the traffic log in RigCtrl_SendServerResponse() -> RigCtrl_SendAndLogMessage() !
                 AuxCom_PassRxOrTxDataToProtocolSpecificDecoder( pInstance, RIGCTRL_ORIGIN_COM_PORT_TX, // here: called immediately before WriteFile() on a COM port
                    b256TxBuffer, nBytesToSend); // [in] chunk of data to process (here: SENT TO someone out there)
                    // Note: If the 'protocol decoder module', e.g. Yeaesu5Byte.c, still has ....
                    //   ((T_Yaesu5ByteControl*)pInstance->pYaesu5ByteControl)->nBytesInRxBuffer
                    // ,-|_____________________________________________________________________|
                    // '---> Copy this precious expression into a WATCH WINDOW.
                    //       Even the stoneage Borland C++ Builder was able to evaluate it.
                    // .... in T_Yaesu5ByteControl.bRxBuffer[YAESU5B_RX_BUFFER_SIZE],
                    // the 'protocol decoder module' invokes AuxCom_Y5B_OnDecodeCallback() MULTIPLE TIMES here:
                    //  * first for the OLD data (still waiting in the decoder's internal buffer mentioned above,
                    //      often with ((T_Yaesu5ByteControl*)pInstance->pYaesu5ByteControl)->nBytesInRxBuffer = 2
                    //      because the not-yet-emitted bytes for the HEX DUMP were a two-byte response
                    //      from the 'READ EEPROM' command shelled out by "FT817 Commander" at a deadly pace;
                    //  * then for the NEW data ( b256TxBuffer , nBytesToSend ),
                    //    which in many cases (with nBytesToSend=5) are a complete "Yaesu 5-Byte COMMAND"
                    //    that Remote CW Keyer sends to the real radio on behalf of e.g. "FT817 Commander".
                    //    Phew !
                    // How to retrieve TIMESTAMPS for the display on the 'DEBUG'-tab remains to be solved yet.
               } // end if < must pass the TO-BE-TRANSMITTED show serial port traffic to a protocol-specific DECODER ? >
              if( WriteFile( pInstance->hComPort,
                       b256TxBuffer, // pointer to data to write to file
                       nBytesToSend, // number of bytes to write
                 &dwNumBytesWritten, // pointer to number of bytes written
                 &osWrite) ) // pointer to the windows 'overlapped' thing
               { // Arrived here ? Surprise surprise; despite the OVERLAPPED thingy,
                 // WriteFile seems to have completed IMMEDIATELY !
                 pInstance->dwNumBytesSent += dwNumBytesWritten;
               }
              else // no "immediate success", so guess the WriteFile() is pending..
               { pInstance->fWritePending = TRUE;
                 TIM_StartStopwatch( &pInstance->swWriteFilePending );
               }
              // Let the "TX_ACK" pulse begin on DTR or RTS, because we're optimistic, and WriteFile() HAS BEEN CALLED now:
              TIM_StartStopwatch( &pInstance->swTxAckPulse );
            }
         }
        else // fWriteToRadioPortPending -> is the above WriteFile() STILL pending ?
         { if( /*BOOL*/GetOverlappedResult( Keyer_hComPortRadioKeyingAndControl,
                &osWrite, &dwNumBytesWritten, FALSE/* do NOT wait*/ ) )
            { // GetOverlappedResult() claimed success for the previous WriteFile(),
              // which means we can fill in more data if we have (in the next thread loop).
              pInstance->fWritePending = FALSE;
              pInstance->dwNumBytesSent += dwNumBytesWritten;
            }
           if( TIM_ReadStopwatch_ms( &pInstance->swWriteFilePending ) > 1000/*ms*/ )
            { // VERY suspicious : The other instance's WriteFile() [to SEND on a serial port]
              // has been "pending" (chewing on the overlapped I/O) for over a second !
              if( (pInstance->dwSerialPortErrors < 10 )   // Cry for help in the 'error history' ..
               && (pInstance->iLastErrorLine != (__LINE__+1) ) ) // .. but don't show the same error TWICE (without other errors in between)
               { pInstance->iLastErrorLine = __LINE__;
                 ++pInstance->dwSerialPortErrors;
                 ShowError( ERROR_CLASS_ERROR, "AuxComThread: WriteFile() pending endlessly on COM%d. %ld bytes sent, %ld bytes rcvd.",
                    (int)pInstance->iComPortNumber, (long)pInstance->dwNumBytesSent, (long)pInstance->dwNumBytesRcvd );
                 // 2025-08-25 : Got here when 'the other end' of a com0com "Virtual Null Modem pair"
                 //              was closed, and trying to forward SPECTRUM DATA from an IC-7300
                 //              over a "Serial Port Tunnel".
               } // end if < ok to report the "looooong pending WriteFile()" in the error history >
            } // end if < WriteFile() has been pending for over a SECOND ! >
         } // end if < previous call of WriteFile() still pending > ?
      } // end if( pInstance->hComPort != INVALID_HANDLE_VALUE)
     AuxCom_UpdateSpeedTestResult( pInstance, AUXCOM_SPEEDTEST_SERIAL_IO, TIM_ReadStopwatch_us( &sw_SpeedTest ) );


     // Immediately PROCESS the received data, and maybe SEND data from here ?
     // Depends on the purpose, i.e. pInstance->pRigctrlPort->iPortUsage :
     switch( pInstance->pRigctrlPort->iPortUsage )
      {
        case RIGCTRL_PORT_USAGE_WINKEYER_HOST     : // be a HOST using K1EL's "CW Keyer IC for Windows", aka "Winkeyer"/"Winkeyer2"
           AuxCom_RunWinkeyerHost( pInstance );
           break; // end case RIGCTRL_PORT_USAGE_WINKEYER_HOST
        case RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR : // try to EMULATE  K1EL's "CW Keyer IC for Windows", used e.g. to fool the N1MM Logger
           AuxCom_RunWinkeyerEmulator( pInstance );
           break; // end case RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR
        case RIGCTRL_PORT_USAGE_TEXT_TERMINAL : // control other 'station equipment' via commands entered on the 'Debug' tab
           // don't do anything here; let the GUI drain pInstance->sRxFifo.fifo,
           // and emit received characters into the not-thread-safe "Rich Text" edit control.
           break;
        case RIGCTRL_PORT_USAGE_SERIAL_TUNNEL:
           AuxCom_RunSerialPortTunnel( pInstance );
           break;
        case RIGCTRL_PORT_USAGE_VIRTUAL_RIG  :
           AuxCom_RunVirtualRig( pInstance );
           break;
        case RIGCTRL_PORT_USAGE_ECHO_TEST: // send ANYTHING received on this port back AS FAST AS POSSIBLE ?
           AuxCom_RunEchoTest( pInstance );
           break;
        case RIGCTRL_PORT_USAGE_TX_STRESS_TEST: // <- trying to flood TXD with data ('ramp')
           AuxCom_RunTxStressTest( pInstance );
           break;

        default: // case RIGCTRL_PORT_USAGE_NONE
           // also don't do anything here.
           break;
      } // end switch( pInstance->pRigctrlPort->iPortUsage ) [here: in AuxComThread()]

   } // end while < thread loop >

  HERE_I_AM__AUX_COM();

  // Close the overlapped event to avoid handle leaks.
  CloseHandle(osWrite.hEvent);
  CloseHandle(osRead.hEvent);

  (void)fSignalStatesModified; // ... assigned a value that is never used .. so what
  (void)fMorseOutput_sent;

  // If the program ever gets here, the thread has "politely" terminated itself,
  //                    or someone has pulled the emergency brake .
  pInstance->iThreadStatus = AUX_COM_THREAD_STATUS_TERMINATED;
  HERE_I_AM__AUX_COM();
  ExitThread( iThreadExitCode ); // exit code for this thread
  HERE_I_AM__AUX_COM();
  return iThreadExitCode; // will this ever be reached after "ExitThread" ?
} // end AuxComThread()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
static void AuxComThread_InterpretIOFunction(  // called THOUSANDS OF TIMES per second ! !
   T_AuxComPortInstance *pInstance, // [in,out] "process image" for THIS PORT:
                                    //      pInstance->dwDigitalSignalStates
          DWORD dwAuxComSignalMask, // [in] ONE of the bits AUX_COM_DIG_SIGNAL_DCD/DSR/CTS/RI/DTR/RTS
             BOOL fIsDigitalOutput, // [in] FALSE when called to 'poll an input' (DCD,DSR,CTS,RI);
                                    //      TRUE when called to 'drive a digital output' (DTR,RTS).
                                    //      This flag is necessary because SOME
                char *pszFunction ) // [in] any of the tokens in AuxComPortIOLineFunctions[] .
  // [in] Keyer_dwCurrentSignalStates bits KEYER_SIGNAL_INDEX_CW
  //                                   and KEYER_SIGNAL_INDEX_PTT :
  //          ONLY SET IN THE KEYER THREAD ( KeyerThread() ),
  //          READ-ONLY here (and in anything called from AuxComThread() ) !

{
  int  iToken, iValue;
  BOOL fInverted = FALSE;
  const char *pszSrc;
  if( pszFunction==NULL ) // nothing to do; don't waste time !
   { return;
   }
  // Preset iValue with the pin's CURRENT STATE from the PLC-like 'process image':
  iValue = ( pInstance->dwDigitalSignalStates & dwAuxComSignalMask ) != 0;
    // '--> result = e.g. TRUE if dwAuxComSignalMask is AUX_COM_DIG_SIGNAL_CTS
    //              and the current state of "CTS" ('Clear To Send') is TRUE (high).
    //              Also works with DIGITAL OUTPUTS of this instance's COM port,
    //              allowing the "TOGGLE" command to.. guess what !

  pszSrc = pszFunction;
  SL_SkipSpaces( &pszSrc );
  if( *pszSrc=='\0' ) // nothing to do; don't waste time !
   { return;
   }
  if( SL_SkipChar( &pszSrc, '!' ) ) // boolean inversion of the expression that FOLLOWS,
   { fInverted = TRUE;              // like the unary operator in "C"
   }
  SL_SkipSpaces( &pszSrc );
  iToken = SL_SkipOneOfNTokens( &pszSrc, AuxComPortIOLineFunctions );
  // ,------'
  // '--> Returns the token value from the list (if successfull),
  //      otherwise a ***NEGATIVE*** error value .
  //      CASE-INSENSITIVE (for "historic" reasons, because WB used it for HTML) !
  switch( iToken )
   {
     case AUXCOM_IO_TOKEN_CW :  // "drive an output with the current CW key up/key down state"
        // Read this flag (iValue) from the KEYER THREAD'S(!) process image, see KeyerThread.c :
        iValue = ( Keyer_dwCurrentSignalStates & (1<<KEYER_SIGNAL_INDEX_CW) ) != 0;
        break; // end case AUXCOM_IO_TOKEN_CW
     case AUXCOM_IO_TOKEN_PTT:  // "drive an output with the current CW key up/key down state",
                                // or "activate the PTT-switching sequence if this INPUT is active"
        // Read this flag (iValue) from the KEYER THREAD'S(!) process image, see KeyerThread.c :
        iValue = ( Keyer_dwCurrentSignalStates & (1<<KEYER_SIGNAL_INDEX_PTT) ) != 0;
        break; // end case AUXCOM_IO_TOKEN_PTT
     case AUXCOM_IO_TOKEN_DOT:  // "secondary DOT (aka "dit"-) input"
        if( ! fIsDigitalOutput ) // "DOT" assigned to a digital INPUT ?
         {
           if( fInverted )
            { iValue = !iValue;   // invert FROM INPUT
              fInverted = FALSE;  // "done" (don't "invert" a second time, for clarity)
            }
           if( iValue != 0 )
            { pInstance->dwDigitalSignalStates |= AUX_COM_DIG_SIGNAL_DOT;
            }
           else // don't SET but RESET this flag (written in AuxComThread(), polled in KeyerThread()):
            { pInstance->dwDigitalSignalStates &= ~AUX_COM_DIG_SIGNAL_DOT;
            }
         }
        break; // end case AUXCOM_IO_TOKEN_DOT
     case AUXCOM_IO_TOKEN_DASH: // "secondary DASH (aka "dah"-) input"
        if( ! fIsDigitalOutput )  // "DOT" assigned to a digital INPUT ?
         {
           if( fInverted )
            { iValue = !iValue;   // invert FROM INPUT
              fInverted = FALSE;  // "done" (don't "invert" a second time, for clarity)
            }
           if( iValue != 0 )
            { pInstance->dwDigitalSignalStates |=AUX_COM_DIG_SIGNAL_DASH;
            }
           else // don't SET but RESET this flag (written in AuxComThread(), polled in KeyerThread()):
            { pInstance->dwDigitalSignalStates &= ~AUX_COM_DIG_SIGNAL_DASH;
            }
         }
        break; // end case AUXCOM_IO_TOKEN_DASH
     case AUXCOM_IO_TOKEN_SKEY: // "input connected to a STRAIGHT MORSE KEY" (e.g. for tuning at QRO)
        if( ! fIsDigitalOutput )  // "STRAIGHT KEY" assigned to a digital INPUT ?
         {
           if( fInverted )
            { iValue = !iValue;   // invert FROM INPUT
              fInverted = FALSE;  // "done" (don't "invert" a second time, for clarity)
            }
           if( iValue != 0 )
            { pInstance->dwDigitalSignalStates |=AUX_COM_DIG_SIGNAL_SKEY;
            }
           else // don't SET but RESET this flag (written in AuxComThread(), polled in KeyerThread()):
            { pInstance->dwDigitalSignalStates &= ~AUX_COM_DIG_SIGNAL_SKEY;
            }
         }
        break; // end case AUXCOM_IO_TOKEN_SKEY
     case AUXCOM_IO_TOKEN_TOGGLE: // "evil command to TOGGLE an output pin as fast as the thread, and the COM port driver permits"
        iValue = !iValue;
        break;
     case AUXCOM_IO_TOKEN_RXACK: // "generate a short pulse on this OUTPUT after RECEPTION (on RXD)"
        iValue = TIM_ReadStopwatch_ms( &pInstance->swRxAckPulse );
        if( (iValue<=0) || (iValue > 10/*ms*/ ) )
         { TIM_StopStopwatch( &pInstance->swRxAckPulse );
           iValue = 0;
         }
        else
         { iValue = 1;
         }
        break;
     case AUXCOM_IO_TOKEN_TXACK: // "generate a short pulse on this OUTPUT after TRANSMISSION (on TXD)"
        iValue = TIM_ReadStopwatch_ms( &pInstance->swTxAckPulse );
        if( (iValue<=0) || (iValue > 10/*ms*/ ) )
         { TIM_StopStopwatch( &pInstance->swTxAckPulse );
           iValue = 0;
         }
        else
         { iValue = 1;
         }
        break;
     case AUXCOM_IO_TOKEN_RXBUSY :
        iValue = pInstance->fReadPending;
        break;
     case AUXCOM_IO_TOKEN_TXBUSY :
        iValue = pInstance->fWritePending;
        break;
     
     default:  // maybe it's not a TOKEN but a simple NUMERIC CONSTANT like '0' or '1' ?
        if( SL_IsDigit(*pszSrc) )
         { iValue = SL_ParseInteger( &pszSrc );
           if( fInverted )
            { iValue = (iValue==0); // invert FROM EVALUATED EXPRESSION
              fInverted = FALSE;    // "done" (don't "invert" a second time, for clarity)
            }
         }
        break;
   } // end switch( iToken )

  // Write the NEW value (iValue) with the pin's NEW STATE back into the PLC-like 'process image' ?
  if( fIsDigitalOutput )
   {
     if( fInverted )
      { iValue = !iValue; // invert FOR OUTPUT (if not already inverted in the switch(iToken) list)
      }
     if( iValue != 0 )
      { pInstance->dwDigitalSignalStates |= dwAuxComSignalMask;
      }
     else // don't SET but RESET the output pin (only works with DTR and RTS):
      { pInstance->dwDigitalSignalStates &= ~dwAuxComSignalMask;
      }
   }

} // end AuxComThread_InterpretIOFunction()




//---------------------------------------------------------------------------
static void AuxComThread_RunProgrammableLogic( T_AuxComPortInstance *pInstance,
             int iAuxComInstance ) // [in] array index to access CwKeyer_Config.xyz[]
  // This function performs the 'Combinational logic' to drive OUTPUTS
  //   on this serial port (and -maybe- routes signals between THIS PORT
  //                        and the RCW Keyer's own keyer, or RX/TX control),
  //   and pass on INPUTS to other parts (worker threads) of the application.
  // [in] pAuxCom->dwDigitalSignalStates : Kind of 'process image' as in a PLC.
  //      Bitwise combination of AUX_COM_DIG_SIGNAL_DCD/DSR/CTS/RI/DTR/RTS .
  // [in] CwKeyer_Config.sAuxCom[iAuxComInstance].sz40DTR/RTS/DCD/DSR/CTS/RI .
  // [out] pAuxCom->dwDigitalSignalStates : AUX_COM_DIG_SIGNAL_DTR + _RTS
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
{
  T_AuxComConfig *pAuxComCfg = &CwKeyer_Config.sAuxCom[iAuxComInstance];
  // As in classic PLCs (Programmable Logic Controllers),
  // first "poll the inputs", then "drive the outputs", once in a PLC cycle (here: thread loop)
   AuxComThread_InterpretIOFunction( pInstance,
       AUX_COM_DIG_SIGNAL_DCD, // [in] dwAuxComSignalMask : ONE of the bits /DSR/CTS/RI/DTR/RTS
       FALSE, // [in] fIsDigitalOutput : FALSE when called to 'poll an input' (DCD,DSR,CTS,RI);
              //      TRUE when called to 'drive a digital output' (DTR,RTS).
       pAuxComCfg->sz40DCD ); // [in] any of the tokens in AuxComPortIOLineFunctions[],
              // decorated with some very basic BOOLEAN OPERATORS like "!" to invert something
   AuxComThread_InterpretIOFunction( pInstance, AUX_COM_DIG_SIGNAL_DSR, FALSE, pAuxComCfg->sz40DSR );
   AuxComThread_InterpretIOFunction( pInstance, AUX_COM_DIG_SIGNAL_CTS, FALSE, pAuxComCfg->sz40CTS );
   AuxComThread_InterpretIOFunction( pInstance, AUX_COM_DIG_SIGNAL_RI,  FALSE, pAuxComCfg->sz40RI  );
   // Next: Run the simplistic "I/O function interpreter" for the COM PORT OUTPUTS, DTR and RTS:
   AuxComThread_InterpretIOFunction( pInstance, AUX_COM_DIG_SIGNAL_DTR, TRUE, pAuxComCfg->sz40DTR );
   AuxComThread_InterpretIOFunction( pInstance, AUX_COM_DIG_SIGNAL_RTS, TRUE, pAuxComCfg->sz40RTS );

   // At this point, the Additional COM Port Worker Thread's(!) own process image,
   // pInstance->pInstance->dwDigitalSignalStates may have been modified.
   // If those modifications have affected the two DIGITAL OUTPUTS, DTR+RTS,
   // AuxComThread() will notice the CHANGE(!) and call the appropriate
   // function (in Microsoft's bizarre "COM port API") to actually switch the
   // outputs - and as in a PLC, only ONCE per cycle.
   // That's why the 'TOGGLE' command (mentioned in the manual) is so incredibly
   // evil, because it will cause HUNDREDS of those COM port API function calls
   // per second. With a real(!) digital oscilloscope connected to DTR (in the
   // 'Simple Paddle Adapter', used as the CW KEYING OUTPUT), the achievable
   // "toggling frequency" can be measured, including the THREAD LOOP TIMING JITTER.
   // Test results (2025-06-14) :
   //   * when the AuxComThread() that busily TOGGLES its DTR output was not
   //     'woken up' via thread event, the DTR output toggled every 10 to 12 ms
   //     under Windows 10 (with the "Digitus" [FTDI] USB<->RS-232 adapter).
   //   * After "toggling" for ~20 minutes, the "Test Report" on the DEBUG tab showed:
   //      > ADDITIONAL COM PORT #3 (on COM8)..
   //      > Status: Client/Server Tunnel, tx=0, rx=0 bytes, 98130 loops
   //      > Serial I/O times: av= 143 us, pk= 1992 us
   //        (98130 loops * 12 ms = circa 20 minutes; ok.
   //         Spent 98130 * 143 us = 14 seconds of 20 min busy from "serial I/O"; ok.)
   //   *

} // end AuxComThread_RunProgrammableLogic()


//---------------------------------------------------------------------------
// Stuff for debugging / development / troubleshooting / protocol analysis
//---------------------------------------------------------------------------


static T_TIM_Stopwatch AuxCom_swTimeSinceLastHexDumpFlush = -1;
static int  AuxCom_nBytesBufferedForHexDump = 0;
static BYTE AuxCom_b16BufferForHexDump[16+4];

//----------------------------------------------------------------------------
void AuxCom_Debug_FlushRcvdCharHexDump( void ) // Flush characters from AuxCom_AppendRcvdCharToHexDumpForDebug()
  // into the DISPLAY BUFFER so another hex-dump line appears on the 'DEBUG' tab.
  // Must be sufficiently thread-safe, but only used by ONE 'Aux COM Port' instance.
  // Called from e.g. the streaming parser for the 'Winkeyer emulator',
  // if the received characters is the begin of a NEW command sequence.
{ char sz80HexDump[84];
  char *pszDest    = sz80HexDump;
  char *pszEndstop = sz80HexDump+79;
  int  i;

  if( AuxCom_nBytesBufferedForHexDump > 16 ) // oops..
   {  AuxCom_nBytesBufferedForHexDump = 16;
   }
  if( AuxCom_nBytesBufferedForHexDump > 0 )
   { for(i=0; i<AuxCom_nBytesBufferedForHexDump; ++i )
      { SL_AppendPrintf(&pszDest,pszEndstop, "%02X ", (unsigned int)AuxCom_b16BufferForHexDump[i] );
      }
     ShowError( ERROR_CLASS_RX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "AuxCom: RX[%02d] 0x%s",
                (int)AuxCom_nBytesBufferedForHexDump, sz80HexDump );
     AuxCom_nBytesBufferedForHexDump = 0;

     TIM_StartStopwatch( &AuxCom_swTimeSinceLastHexDumpFlush );
   }
} // end AuxCom_Debug_FlushRcvdCharHexDump()

//----------------------------------------------------------------------------
int AuxCom_Debug_GetTimeElapsedSinceLastHexDump_ms(void)
{
  return TIM_ReadStopwatch_ms(&AuxCom_swTimeSinceLastHexDumpFlush);
}


//----------------------------------------------------------------------------
void AuxCom_Debug_AppendRcvdCharToHexDump( BYTE bRcvdChar )
{
  if( bRcvdChar==0x07 )
   {  bRcvdChar = bRcvdChar; // <- place for a 'hard-coded conditional breakpoint' (Winkeyer "Get Speed Pot")
   }
  if( AuxCom_swTimeSinceLastHexDumpFlush < 0 )
   { TIM_StartStopwatch( &AuxCom_swTimeSinceLastHexDumpFlush );
   }

  AuxCom_b16BufferForHexDump[ (AuxCom_nBytesBufferedForHexDump++) & 15 ] = bRcvdChar;
  if(  (AuxCom_nBytesBufferedForHexDump >= 16)
    || (TIM_ReadStopwatch_ms(&AuxCom_swTimeSinceLastHexDumpFlush) >= 1000 ) )
   { // Time to emit (flush) the last received bytes as Hex dump for the display:
     AuxCom_Debug_FlushRcvdCharHexDump();
   }

  // Together with the few 'decoded' commands emitted in the streaming parsers
  // further below, the initial communication between N1MM Logger+ ("Winkeyer HOST")
  // and the Remote CW Keyer (as "Winkeyer EMULATOR") looked as follows:
  // > 20:51:17.2 Winkey: Rcvd Admin:Host Open
  //                              ,--|_______|
  //                              |     ,--- "Set Winkeyer2 Mode Byte":
  //                              |     |    0x05 = "Serial Echoback" (bit 2)
  //                              |     |         + "CT Spacing" (bit 0 set) ? ?
  //                              |     |     ,-- "Set Speed Pot Range" <05><m><wr><pr>
  //                              |     |     |           ,-- "Set Sidetone Frequency" <01><freq>
  //                              |     |     |           |   "Set Output Pin Config" <09><config>
  //                            ,-'-, ,-'-, ,-'-------, ,-'-, ,-'-,
  // > 20:51:18.2 AuxCom: Rcvd  00 02 0E 05 05 0A 2D FF 01 88 09 07
  //
  // > 20:51:18.3 Winkey: Rcvd  Set PTT Delays <04><leadin><tail>
  //                              |        ,-- "Set First Element Correction"  <10><msec>
  //                              |        |     ,-- "Set Keying Compensation" <11><comp>
  //                              |        |     |     ,-- "Set Key Weighting" <03><weight>
  //                              |        |     |     |    ,-- "Request Speed Pot Value" <07>
  //                              |        |     |     |    |   ,-- "Set Morse Sending Speed" <02><WPM>
  //                            ,-'----, ,-'-, ,-'-, ,-'-, /| ,-'-, ,---,--- same command again ?!
  // > 20:51:19.2 AuxCom: Rcvd  04 00 00 10 00 11 00 03 32 07 02 1E 02 1E
} // end AuxCom_AppendRcvdCharToHexDumpForDebug()


//---------------------------------------------------------------------------
static void AuxCom_DumpMessageToLog( T_AuxComPortInstance *pInstance,
              int iRigCtrlOrigin,        // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX
              BYTE *pbData, int nBytes,  // [in] chunk of data received from, or sent to the port (not necessarily a complete, nor a SINGLE 'Message')
              char* pszComment )         // [in] optional comment generated by a protocol-decoder, may be NULL
   // [in] pInstance->dwParams: bitwise combination of flags
   //      like AUX_COM_PARAMS_DEBUG, AUX_COM_PARAMS_DUMP_TO_TERMINAL, etc.
   //      (see specification in Remote_CW_Keyer/AuxComPorts.h) .
   // [out] a line of text displayed on the application's "Debug" tab, via ShowError()
   //
   // Callers (example when using the 'Yaesu 5-byte CAT protocol') :
   //  * for RECEPTION:    AuxComThread() -> AuxCom_PassRxOrTxData() -> Yaesu5Byte_ProcessData()
   //                        -> AuxCom_Y5B_OnDecodeCallback() -> AuxCom_DumpMessageToLog() .
   //  * for TRANSMISSION: AuxComThread() [when calling WriteFile()]
   //        [note: The calls may be 'delayed' for reasons explained
   //               in

   //
   // Example: To study the traffic between 'FT817 Commander' and an FT-817,
   //          configure one of the "Client/Server Tunnels", or the port
   //          that connects to the radio as follows:
   //        > Params       :  dump, protocol=Yaesu5B
   //      (shouldn't matter if that's on the real port with the FT-817 CAT adapter,
   //       or e.g. "COM7 (com0com - serial port emulator)" on the "Virtual Null-Modem Pair"
   //       with the 'radio controlling application' (e.g. "FT817 Commander")
   //       on the other side. Only "TX" and "TX" will be from a different point of view.)
   //
{
  char sz80HexDump[84];
  char *pszDest = sz80HexDump;
  char *pszEndstop = sz80HexDump+79;
  char *pszRxTx = "RX";
  char sz15Module[16]; // e.g. "COMx" or just "AuxCom"
  int  i, nCharsToPad;
  int  iErrorClass = SHOW_ERROR_TIMESTAMP;

  switch( iRigCtrlOrigin ) // "sent" or "received" from this port's point of view ?
   { case RIGCTRL_ORIGIN_COM_PORT_RX :  // data RECEIVED FROM a 'local COM port'
     default:
         iErrorClass |= ERROR_CLASS_RX_TRAFFIC;
         break;
     case RIGCTRL_ORIGIN_COM_PORT_TX :  // data SENT TO a 'local COM port'
         iErrorClass |= ERROR_CLASS_TX_TRAFFIC;
         pszRxTx = "TX";
   } // end switch( iRigCtrlOrigin )
  if( pInstance->iComPortNumber > 0 )
   { sprintf(sz15Module, "COM%d", (int)pInstance->iComPortNumber );
   }
  else
   { sprintf(sz15Module, "AuxCom%d", (int)(AuxCom_InstancePtrToIndex(pInstance)+1) );
   }

  if( pInstance->dwParams & AUX_COM_PARAMS_DEBUG ) // optional output in ONE LINE on RCWK's "DEBUG" tab ?
   { // If the amount of data permits, try to show the port's traffic ON THE DEBUG TAB
     // (controlled by the "hints" to make this human-readable
     //  in AUX_COM_PARAMS_MASK_PROTOCOL, defined further below.
     //  If we haven't got a clue, AuxCom_DumpMessageToLog()
     //  will guess if the to-be-displayed block is ASCII ("-> show as TEXT")
     //  or "something binary" (-> show as hex-dump) .
     pszDest    = sz80HexDump;
     pszEndstop = sz80HexDump+79;
     if( pbData != NULL )
      { for(i=0; (i<nBytes) && ((pszDest+4)<pszEndstop); ++i )
         { SL_AppendPrintf(&pszDest,pszEndstop, "%02X ", (unsigned int)pbData[i] );
         }
        if( (pszComment!=NULL) && (pszComment[0] != '\0' ) ) // pad with SPACES before appending a COMMENT ?
         { nCharsToPad = 30 - (pszDest-sz80HexDump);  // for column alignment ..
           //             |   |___________________|
           //             |     '--> Number of chars already emitted to sz80HexDump .
           //             '--> Worst-case guesstimate of the column with the comment.
           // Example:                    ,---HexDump---,,--padding----,,---------- sz40Comment ------------,
           // > 20:41:56.7 COM8: TX[05] 0x00 55 00 00 BB                ; Read EEPROM(0x0055)=Memo/VFO Select
           SL_PadSpaces( &pszDest, pszEndstop, nCharsToPad );
         }
      } // end if < emit HEX DATA before the comment > ?
     if( (pszComment!=NULL) && (pszComment[0] != '\0' ) ) // <- note the evaluation sequence: LEFT TO RIGHT
      { SL_AppendPrintf(&pszDest,pszEndstop, "; %s", pszComment );
      }
     ShowError( iErrorClass, "%s: %s[%02d] 0x%s", sz15Module, pszRxTx, (int)nBytes, sz80HexDump );
   } // end if < AUX_COM_PARAMS_DEBUG > ?

} // end AuxCom_DumpMessageToLog()


//----------------------------------------------------------------------------
int AuxCom_RigControlCallback( // may be invoked ONCE or even MULTIPLE TIMES from RigCtrl_ProcessData() ..
  //     '--> return value depends on iRigCtrlCbkEvent, usually ZERO for 'no error'.
  struct tRigCtrl_PortInstance *pPortInstance, // [in] RigControl port instance that 'decoded' the message, with ...
                      // [in] pPortInstance->iMsgTypeFromParser, iUnifiedParameterNumberFromParser, etc
  int iRigCtrlOrigin, // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, ..
  int iRigCtrlCbkEvent, // [in] e.g. RIGCTRL_CBK_SEND_MSG(1), RIGCTRL_CBK_DECODED(2), ..
  BYTE *pbMessage, int iMsgLength, // [in] raw message data and length in bytes
  char* pszComment )  // [in] human readable "comment" for the log
  // Call stacks:
  // (1) AuxComThread() -> AuxCom_RunVirtualRig() -> AuxCom_PassRxOrTxDataToProtocolSpecificDecoder()
  //      -> RigCtrl_ProcessData() -> RigCtrl_ParseCIV()
  //           -> AuxCom_RigControlCallback( iRigCtrlCbkEvent = RIGCTRL_CBK_DECODED )
  //         [e.g. when an external client sent a CI-V read- or write-COMMAND]
  // (2) AuxComThread() -> AuxCom_RunVirtualRig() -> AuxCom_PassRxOrTxDataToProtocolSpecificDecoder()
  //      -> RigCtrl_ProcessData() -> RigCtrl_SendServerResponse() -> RigCtrl_SendAndLogMessage()
  //           -> AuxCom_RigControlCallback( iRigCtrlCbkEvent = RIGCTRL_CBK_SEND_MSG )
  //         [e.g. when a CI-V sent a CI-V read- or write-RESPONSE must be sent]
  //
{
  int iResult = 0;
  int iRigCtrlPort = RigCtrl_PortInstancePtrToIndex(pPortInstance);
  char sz255[256];
  T_AuxComPortInstance *pAuxComInstance = AuxCom_IndexToInstancePtr( iRigCtrlPort - RIGCTRL_PORT_AUX_COM_1 );
  if( pAuxComInstance == NULL ) // oops.. not a valid AUX-COM-INSTANCE-INDEX !
   { return -1;
   }

  switch( iRigCtrlCbkEvent )
   { case RIGCTRL_CBK_SEND_MSG: // RigControl.c (or whoever) is asking us to SEND ANOTHER MESSAGE on this port.
        // (RIGCTRL_CBK_SEND_MSG does NOT occurr in 'Serial Port Tunnel' mode:
        //   With pPortInstance->fActAsServer==FALSE, the "TX-path" is described in AuxCom_RunSerialPortTunnel() )
        switch( pPortInstance->iServerState )
         { case RIGCTRL_SSTATE_RESPONSE_READY :
              RigCtrl_SetServerState(pPortInstance, RIGCTRL_SSTATE_PASSIVE); // server port passive again; available for the next COMMAND
              // (see server state transitions in RigControl_CIV_Server.c )
              // Callers seen with a breakpoint above:
              //  AuxComThread() -> AuxCom_RunVirtualRig() -> AuxCom_PassRxOrTxDataToProtocolSpecificDecoder()
              //   -> RigCtrl_ProcessData() -> RigCtrl_SendServerResponse()
              //       -> RigCtrl_SendAndLogMessage(!) -> AuxCom_RigControlCallback(RIGCTRL_CBK_SEND_MSG) .
              break;
           case RIGCTRL_SSTATE_FWD2RADIO_DONE : // similar as ..RESPONSE_READY,
              // but the command / response has been FORWARDED to / from the RADIO CONTROL PORT.
              RigCtrl_SetServerState(pPortInstance, RIGCTRL_SSTATE_PASSIVE); // server port passive again; available for the next COMMAND
              // Callers seen with a breakpoint above:
              //  RigCtrl_Handler() -> RigCtrl_ProcessData() [for the RADIO PORT]
              //    -> RigCtrl_ParseCIV() -> RigCtrl_CheckAndForwardResponseToClients()
              //       -> RigCtrl_SendAndLogMessage(!) -> AuxCom_RigControlCallback(RIGCTRL_CBK_SEND_MSG) .
              break;
           default:
              break;
         } // end switch( pServerPortInstance->iServerState ) in AuxCom_RigControlCallback()


        // Because this call MAY(!) be made from any thread (not necessarily
        // from our AuxComThread()), so use the lock-free circular buffer again:
        CFIFO_Write( &pAuxComInstance->sTxFifo.fifo,
            pbMessage/*source*/, iMsgLength/*nBytes*/, 0.0/*dblTimestamp_s*/ );
        // Note: Unless we ask it not to do so (via callback return value),
        //       the message (CI-V or whatever) has already been added to the
        //       traffic log by RigControl.c or RigControl_CIV_Server.c  .
        // If the message just SENT on this 'Aux COM Port' was a response
        // from our server to a remote client, it's time for another SERVER STATE TRANSITION:
        AuxCom_WakeUp( pAuxComInstance ); // wake up OUR OWN WORKER THREAD.
        // Fortunately, Windows doesn't mind if a thread tries to wake up himself,
        // because SOMETIMES (if the response could be assembled immediately)
        // the call stack looked like this:
        //  AuxComThread() -> AuxCom_RunVirtualRig() -> AuxCom_PassRxOrTxDataToProtocolSpecificDecoder()
        //   -> RigCtrl_ProcessData() [somewhere after RigCtrl_ParseCIV():]
        //    -> RigCtrl_SendServerResponse() -> RigCtrl_SendAndLogMessage()
        //     -> AuxCom_RigControlCallback() [via callback function pointer]
        //          with iRigCtrlCbkEvent = RIGCTRL_CBK_SEND_MSG .
        break;  // end case RIGCTRL_CBK_SEND_MSG
     case RIGCTRL_CBK_DECODED :  // RigControl.c (or whoever) has *DECODED* a message ...
        // Try to find out if e.g. WFView's "initialisation phase" is over :
        //  (similar also for "Serial Port Tunnels", see AuxComPort.c .
        //   For that purpose, RigCongrol_CIV_Server.c COUNTS e.g. S-meter-reports).
        if( !pPortInstance->fActAsServer ) // NOT "acting as a server" usually means "acting as a SERIAL PORT TUNNEL"...
         { // To properly comment messages for the traffic log, store e.g. the
           // CI-V "command"-byte, and optional "sub-command", or even the
           // dreadful (rig-mode-specific) FOUR-DIGIT HEX (or BCD?-) number
           // in the PORT INSTANCE DATA. The scary RigCtrl_ParseCIV() monster
           // has already 'classified' the CI-V message type (at this point,
           // in pPortInstance->iMsgTypeFromParser) :
           switch( pPortInstance->iMsgTypeFromParser & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )
            {
              case RIGCTRL_MSGTYPE_FLAG_READ_CMD   : // RigCtrl_ParseCIV() just decoded a READ  COMMAND :
              case RIGCTRL_MSGTYPE_FLAG_WRITE_CMD  : // RigCtrl_ParseCIV() just decoded a WRITE COMMAND :
                   // Only the above CI-V *COMMAND* types contain
                   // at least the unique command, sometimes a sub-command,
                   // and in rare cases (the dreadful 0x1A 0x05 group) even more.
                   pPortInstance->dwExpectedResponse_CmdAndSubcode =
                     RigCtrl_GetCombinedCmdAndSubcodeForCIV( pbMessage, iMsgLength, pPortInstance->iMsgTypeFromParser );
                   // Example: pbMessage = { 0xFE, 0xFE, 0x94, 0xE0, 0x15, 0x02, 0xFD } = "Read S-meter level" (at least in an IC-9700..)
                   //          iMsgLength= 7
                   //          pPortInstance->iMsgTypeFromParser = 0x1801F = RIGCTRL_MSGTYPE_FLAG_READ_CMD(1<<16) | RIGCTRL_MSGTYPE_OTHER(31)
                   // -> pPortInstance->dwExpectedResponse_CmdAndSubcode = 0x1502FFFF = "cmd 0x15 sub 0x02, no sub-sub-codes"
                   //     ,-------------|______________________________|
                   //     '--> May be CLEARED LATER, when on the same "Aux COM port",
                   //          a RESPONSE (Read- or Write- or Error-response)
                   //          passes through in the OPPOSITE direction .
                   //
                   break; // end case < "Read-Command","Write-Command" on a SERIAL PORT TUNNEL >

              case RIGCTRL_MSGTYPE_FLAG_READ_RESP  : // RigCtrl_ParseCIV() just decoded a READ RESPONSE :
                   // A CI-V READ RESPONSE also contains the unique command/sub/etc,
                   // but we don't always need that here because there is nothing
                   // that 'follows' after THIS message that need to be commented
                   // for the TRAFFIC LOG !
                   break; // end case < "Read-Reponse" on a SERIAL PORT TUNNEL  >

              case RIGCTRL_MSGTYPE_FLAG_WRITE_RESP : // RigCtrl_ParseCIV() just decoded a WRITE RESPONSE:
                   // A CI-V WRITE RESPONSE only says '0xFB' (="OK"),
                   // but it doesn't give a clue about WHAT was "ok".
                   // Thus the need for all the hassle with pPortInstance->dwExpectedResponse_CmdAndSubcode !
                   break; // end case < "Write-Reponse" on a SERIAL PORT TUNNEL  >

              case RIGCTRL_MSGTYPE_FLAG_OTHER_RESP : // RigCtrl_ParseCIV() just decoded a WRITE COMMAND :
              case RIGCTRL_MSGTYPE_FLAG_UNSOLICITED: // RigCtrl_ParseCIV() thinks this was an UNSOLICITED message ("transceived" in Icom jargon, but forget that)
              case RIGCTRL_MSGMASK_FLAGS_RW_UNKNOWN: // RigCtrl_ParseCIV() didn't have a clue about WHAT it was.. (shit happens!)
              default:
                   break;
            } // end switch( iMsgType & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )

           // For the decision if e.g. WFView's "initialisation phase" is over :
           //  (when ACTING AS SERVER, S-Meter-Level-reports are counted in RigControl_CIV_Server.c)
           if( pPortInstance->iUnifiedParameterNumberFromParser != 0 ) // <- only to set a BREAKPOINT on the next executable statement..
            { switch( pPortInstance->iUnifiedParameterNumberFromParser ) // Is this an "indicator" for having reached the "INIT PHASE" ?
               { case RIGCTRL_PN_S_METER_LEVEL_DB :  // <- periodically polled by WFView, but not by N1MM Logger ..
                    if( pPortInstance->iNumSMeterReports < 32767 )
                     { ++pPortInstance->iNumSMeterReports;
                     }
                    break;
                 case RIGCTRL_PN_FREQUENCY :
                    if( pPortInstance->iNumVfoFreqReports < 32767 )
                     { ++pPortInstance->iNumVfoFreqReports;
                     }
                    break;
                 default:
                    break;
               }
            } // end if( pPortInstance->iUnifiedParameterNumberFromParser != 0 )

         } // end if < aux COM port used as SERIAL PORT TUNNEL, with decoder > ?

        if( (RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE ) // here: for a SERVER PORT or SERIAL PORT TUNNEL WITH PROTOCOL SPECIFIED (e.g. protocol=icom) ...
         && (!pPortInstance->fTrafficMonitorPausedOnTrigger) )
         { // In contrast to the RADIO CONTROL PORT, it's difficult to find out
           // when the REMOTE CLIENT has finished the 'initialisation phase', i.e.
           // after all parameters have been read ONCE. For exampl, if the remote client
           // is WFVIEW, one of the FIRST parameters read during the init-phase
           // is the Icom radio's 'Default CI-V Address' (to detect the RIG TYPE).
           // C:\cbproj\Remote_CW_Keyer\Extra_Literature\Trouble_with_WFVIEW_on_CLIENT_PORT.txt
           // contains a few examples (traffic log extracts) with the MANY parameters
           // read by WFView before beginning the "normal polling cycle".
           // Indicators for the begin of a remote client's "normal polling cycle" were:
           //  * repeated "read S-Meter" (RIGCTRL_PN_S_METER_LEVEL_DB)
           //  * repeated "read OpMode"  (RIGCTRL_PN_OP_MODE)
           //  * repeated "read VFO frequency"  (especially for N1MM Logger+)
           // Thus:
           if( ( (pPortInstance->iNumSMeterReports >= 2 )   // <- also counted in SERIAL PORT TUNNEL MODE..
              || (pPortInstance->iNumVfoFreqReports>= 2) )
             &&(pPortInstance->dwNumMessagesSent > 10 )   // see explanation further below (*) ...
             &&(pPortInstance->dwNumMessagesRcvd > 10 ) ) // ... why we need these EXTRA CRITERIA !
            {
              // (*) When the REMOTE CLIENT was WFView, terminating and restarting RCWKeyer
              //     without stopping ("[Disconnect from Radio]") or terminating WFview,
              //     a LOT of read-requests filled the serial port buffer, long before
              //     RCW-Keyer had finished its own initialisation (read dozens of parameters
              //     from the radio). Amongst the CI-V messages that were processed
              //     in a single burst (because THEY ALL HAD THE SAME TIMESTAMP in milliseconds)
              //     there were SEVERAL DOZENS of "SMeterLevel ?" requests,
              //     even though not a single response had been sent from server to client yet:
              // > ; Nr time/ms tr len pream to fm CIV-cmd, params..  postamble=FD
              // > 1 0000884 r2 007 FE FE 94 E1 1C 00 FD                          ; Transmitting ?
              // >   0000884  Illegal transition from server state wait_radio_non_busy to parsing_cmd
              // > 2 0000884 r2 007 FE FE 94 E1 15 02 FD                           ; SMeterLevel ?
              // >   0000884  Illegal transition from server state wait_radio_non_busy to parsing_cmd
              // > 3 0000884 r2 007 FE FE 94 E1 15 02 FD                           ; SMeterLevel ?
              // >   0000884  Illegal transition from server state wait_radio_non_busy to parsing_cmd
              // > 4 0000884 r2 007 FE FE 94 E1 15 07 FD                          ; OverflowStat ?
              // >   0000884  Illegal transition from server state wait_radio_non_busy to parsing_cmd
              // > 5 0000884 r2 007 FE FE 94 E1 15 02 FD                           ; SMeterLevel ?
              // >   0000884  Illegal transition from server state wait_radio_non_busy to parsing_cmd
              //       (etc, etc, etc).
              //     The EXTRA CRITERIA (dwNumMessagesSent, dwNumMessagesRvcd)
              //     now prevents a too-early stop, when the 'initialisation' hasn't even BEGUN.
              //
              // 2025-09-05: Similar problem also with N1MM Logger+ as the remote client,
              //             which surprisingly seems to START with a WRITE COMMAND:
              // >   0007468  Traffic Log paused on RadioPort/IC-7300 after initialisation.
              // >   2726 0053854 r2 007 FE FE 94 E0 0F 00 FD                             ; SplitMode:off  (N1MM -> RCWK)
              // >   2730 0053896 TX 007 FE FE 94 E0 0F 00 FD                           ; write SplitMode  (RCWK -> IC-7300)
              // >   2731 0053960 RX 03C FE FE E0 94 27 00 00 04 11 09 09 04 0F 0B 15 ..  ; fragment 4/11  (unsolicited interference)
              // >   2732 0053960 RX 006 FE FE E0 94 FB FD                        ; OK for cmd 0xFFFFFFFF  (IC-7600 -> RCWK)
              // >   2733 0053960 t2 006 FE FE E0 94 FB FD                        ; OK for cmd 0xFFFFFFFF  (RCWK -> N1MM. Ok so far.)
              // >   2819 0055712 r2 007 FE FE 94 E0 07 00 FD                              ; SelectVFO:00  (N1MM -> RCWK)
              // >   2823 0055725 TX 007 FE FE 94 E0 07 00 FD                              ; SelectVFO:00  (RCWK -> IC-7300)
              // >        0055762  Illegal transition from server state forward_busy to parsing_cmd
              // >   2824 0055762 r2 007 FE FE 94 E0 0F 00 FD                             ; SplitMode:off  (oops.. only 50 ms after the PREVIOUS command?!)
              // >   2825 0055787 RX 006 FE FE E0 94 FB FD                                         ; 'OK'  (IC-7600 -> RCWK, for the "SelectVFO")
              // >   2826 0055787 RX 03C FE FE E0 94 27 00 00 10 11 17 0F 05 0E 0F 0E .. ; fragment 10/11
              // >   2827 0055787 RX 023 FE FE E0 94 27 00 00 11 11 00 00 0C 16 12 12 .. ; fragment 11/11
              // >   2828 0055787 TX 007 FE FE 94 E0 0F 00 FD                           ; write SplitMode
              // >        0055842  Illegal transition from server state forward_busy to parsing_cmd
              // >   2829 0055842 r2 007 FE FE 94 E0 07 00 FD                              ; SelectVFO:00
              // >   2830 0055850 RX 006 FE FE E0 94 FB FD                        ; OK for cmd 0xFFFFFFFF
              // >   2831 0055850 RX 016 FE FE E0 94 27 00 00 01 11 02 00 10 49 03 00 .. ; 3491..3591 kHz
              // >   2832 0055850 RX 03C FE FE E0 94 27 00 00 02 11 0A 00 12 00 15 0F ..  ; fragment 2/11
              // >   2833 0055850 RX 03C FE FE E0 94 27 00 00 03 11 15 12 0C 04 0B 12 ..  ; fragment 3/11
              // >   2834 0055850 TX 007 FE FE 94 E0 07 00 FD                              ; SelectVFO:00
              // >        0055898  Illegal transition from server state forward_busy to parsing_cmd
              // >   2835 0055898 r2 007 FE FE 94 E0 25 00 FD                              ; SelVFOFreq ?
              // >   2836 0055898 t2 00C FE FE E0 94 25 00 00 10 54 03 00 FD    ; SelVFOFreq=3541000.0 Hz
              // >   2837 0055912 RX 006 FE FE E0 94 FB FD                                         ; 'OK'
              // (etc etc. Not sure about the cause for this "impatient transmission of another command",
              //           when the response for the PREVIOUS command had not been sent yet...)
              //
              // Before pausing the log FOR THIS PORT, emit a note explaining WHY the traffic monitor is paused:
              sprintf( sz255, "Traffic Log paused on %s after initialisation.",
                   RigCtrl_GetDescriptivePortName(pPortInstance) ); // <- delivers e.g. "Radio Control Port", "Virtual Radio Port #2", etc
              RigCtrl_AddToTrafficLog( pPortInstance->pRC, iRigCtrlPort, RIGCTRL_ORIGIN_CONTROLLER, NULL/*pbMessage*/, 0/*iMsgLength*/,
                   RIGCTRL_MSGTYPE_INFO, sz255 );
              RigCtrl_fUpdateTrafficLog = TRUE; // now "ready for being UPDATED on-screen"
              pPortInstance->fTrafficMonitorPausedOnTrigger = TRUE;
            }
         } // end RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE
        break;  // end case RIGCTRL_CBK_DECODED
     default:   // ignore any of the other events from RigControl.h
        break;
   } // end switch( iRigCtrlCbkEvent )
  return iResult;
} // end AuxCom_RigControlCallback()


//----------------------------------------------------------------------------
static void AuxCom_PassRxOrTxDataToProtocolSpecificDecoder(
              T_AuxComPortInstance *pInstance,
              int iRigCtrlOrigin,       // [in] e.g. RIGCTRL_ORIGIN_COM_PORT_RX(2) or ..TX(3) (*)
              BYTE *pbData, int nBytes) // [in] chunk of received data received from, or sent to the port (not necessarily a complete, nor a SINGLE 'Message')
  // (*) This function invokes protocol-specific parsers for..
  //  * RECEPTION (when draining pInstance->sRxFifo.fifo) in AuxCom_RunSerialPortTunnel() or AuxCom_RunVirtualRig()
  //  * TRANSMISSION (send from pInstance->sTxFifo.fifo) directly in AuxComThread() .
  // The iRigCtrlOrigin argument tells the two "directions" from each other,
  //     including if the block of data wasn't received from / sent to the RADIO PORT
  //               but one of the SERIAL PORT TUNNELS (important for not interfering,
  //               see implementation of XYZ_ProcessData() in other modules).
  //     When called with a DIFFERENT value (e.g. RIGCTRL_ORIGIN_COM_PORT_RX now,
  //                            but RIGCTRL_ORIGIN_COM_PORT_RX in the last call),
  //     iRigCtrlOrigin causes the protocol decoder to FLUSH / PROCESS or even DISCARD
  //     previously received data, because a call with a different iRigCtrlOrigin
  //     usually marks switching between e.g. COMMAND and RESPONSE
  //      - at least for such simple half-duplex protocols like Yaesu's stoneage
  //        "Five-Byte-Command protocol", where a RESPONSE can only be interpreted
  //        if you know the COMMAND that it applies to.
  //     Simple example of a "half-duplex" protocol: See Yaesu5Byte.c
  //
{
  int iAuxComInstance = AuxCom_InstancePtrToIndex( pInstance );
  int iRigCtrlPort = RIGCTRL_PORT_AUX_COM_1 + iAuxComInstance; // [in] iRigCtrlPort, to tell this PORT from others for the 'Rig Control' unit (RigControl.c)
      // '--> Note: The "Rig Control Port number" cannot indicate the DIRECTION of data (RX,TX).
      // That's why we also pass 'iRigCtrlOrigin' to the protocol-specific decoder,
      // last not least to have a human friendly output in the traffic log.

  switch( pInstance->dwParams & AUX_COM_PARAMS_MASK_PROTOCOL )
   { case AUX_COM_PARAMS_PROTOCOL_NONE : // "there's no PROTOCOL to decode"
     case AUX_COM_PARAMS_PROTOCOL_KENWOOD : // ASCII-based with ';' as delimiter (aka "Terminator"), e.g. for TS-480, TS-2000 .
     case AUX_COM_PARAMS_PROTOCOL_YAESU_ASCII: // possibly "inspired by Kenwood", e.g. for FT-891, FT-991[A], .. (?)
     default:
        AuxCom_DumpMessageToLog( pInstance, iRigCtrlOrigin, pbData, nBytes, NULL/*pszComment*/ );
        // Note: AuxCom_DumpMessageToLog() above cannot tell the end of a COMMAND or RESPONSE,
        //       thus for certain protocols, it was replaced by AuxCom_RigControlCallback().
        break;
     case AUX_COM_PARAMS_PROTOCOL_ICOM : // "most likely this is ICOM's CI-V"
     case AUX_COM_PARAMS_PROTOCOL_YAESU_5_BYTE: // FIVE-BYTE commands, e.g. for FT-817, FT-897: "experimental" support in RigControl, too
        if( pInstance->pRigControl != NULL ) // <- optional... there may be no 'Rig-Control' instance at all !
         { RigCtrl_ProcessData(  // .. for SERIAL PORT TUNNEL with fListenOnlyMode=TRUE, or VIRTUAL RIG with fListenOnlyMode=FALSE...
                 (T_RigCtrlInstance*)pInstance->pRigControl, // [in,out] Rig-Control instance, with the prime focus on ICOM radios
                 iRigCtrlPort, iRigCtrlOrigin/*RIGCTRL_ORIGIN_COM_PORT_RX/TX*/,
                 AuxCom_RigControlCallback, pbData, nBytes );
           // ,--|_______________________|
           // '-> May be called MULTIPLE TIMES because pbData[] may contain more than one CI-V message.
           //     May be called NOT AT ALL if pbData[] only contains A PART of a CI-V message.
           //     For the same ("Aux-COM"-)port, RigCtrl_ProcessData() uses the following:
           //      > pMsgBuf = &pPortInstance->sRxMsg[iRigCtrlOrigin & 1];
           //          '--> TWO internal message buffers for de-packetizing e.g. CI-V:
           //               One de-packetizing buffer for RIGCTRL_ORIGIN_COM_PORT_RX,
           //               one de-packetizing buffer for RIGCTRL_ORIGIN_COM_PORT_TX!
         }
        break;
   } // end switch( pInstance->dwParams & AUX_COM_PARAMS_MASK_PROTOCOL )
} // end AuxCom_PassRxOrTxDataToProtocolSpecificDecoder()

//---------------------------------------------------------------------------
BOOL AuxCom_MustPassDataToDecoder( T_AuxComPortInstance *pInstance ) // -> TRUE=yes, FALSE=no.
   // Decide if bytes received from, or sent to, an Additional COM Port
   // must be passed on to a protocol-specific decoder.
   // Return value:   TRUE=yes (must be passed to a 'decoder'), FALSE=no.

{
  BOOL fPassToDecoder = FALSE;
  if( pInstance->dwParams & (AUX_COM_PARAMS_DEBUG | AUX_COM_PARAMS_DUMP_TO_TERMINAL) )
   { // '--> this is what the operator may configure in the 'Additional COM Port Details'
     //      dialog, edit field "Params", parsed in KeyerGUI_ParseAuxComParams().
     fPassToDecoder = TRUE;
     // Note: Similar passing of data to protocol-specific parsers happens for..
     //  * RECEPTION (when draining pInstance->sRxFifo.fifo) in AuxCom_RunSerialPortTunnel()
     //  * TRANSMISSION (send from pInstance->sTxFifo.fifo) directly in AuxComThread() .
   }
  if( (pInstance->dwParams & AUX_COM_PARAMS_MASK_PROTOCOL) != 0 )
   { // '--> this is what the operator may configure in the 'Additional COM Port Details'
     //      dialog, edit field "Params", "protocol=XYZ", also parsed in KeyerGUI_ParseAuxComParams().
     fPassToDecoder = TRUE;
   }
  if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_OTHER_TRAFFIC )
   { // '--> this is what the operator may turn on / turn off in the
     //      context menu on the 'DEBUG' tab (related to the traffic log),
     //  > "Show traffic from 'Additional COM Ports' with other purposes" .
     fPassToDecoder = TRUE;
   }
  return fPassToDecoder;
} // end AuxCom_MustPassDataToDecoder() [answers a "simple question", nothing else]


//----------------------------------------------------------------------------
BOOL AuxCom_IsInUse( T_AuxComPortInstance *pInstance ) // -> TRUE=yes, FALSE=no
  // Added for the report (summary as text) in Keyer_Main.cpp :
{ if( pInstance != NULL )
   { if( pInstance->hThread != NULL )  // got a valid handle for this instance's WORKER THREAD valid ?
      { return TRUE;   // .. then consider this instance of being "in use"
      }
     if( pInstance->hComPort != INVALID_HANDLE_VALUE ) // handle for this instance's SERIAL PORT valid ? (Remember the madness from Richmond for THIS kind of HANDLE..)
      { return TRUE;   // .. then also consider this instance of being "in use"
      }
   }
  return FALSE;  // Arrived here ? The instance is completely invalid or NOT in use !
} // end AuxCom_IsInUse()

//---------------------------------------------------------------------------
const char* AuxCom_ProtocolToString( DWORD dwParams )   // only for the GUI..
{ switch( dwParams & AUX_COM_PARAMS_MASK_PROTOCOL )
   { case AUX_COM_PARAMS_PROTOCOL_NONE : return "No Protocol";
     case AUX_COM_PARAMS_PROTOCOL_ICOM : return "CI-V";
     case AUX_COM_PARAMS_PROTOCOL_KENWOOD: return "Kenwood";
     case AUX_COM_PARAMS_PROTOCOL_YAESU_5_BYTE: return "Yaesu 5-Byte";
     case AUX_COM_PARAMS_PROTOCOL_YAESU_ASCII : return "Yaesu ASCII";
     default: return "Protocol??";
   }
} // end AuxCom_ProtocolToString()

//---------------------------------------------------------------------------
const char* AuxCom_GetCurrentStatusAsString( T_AuxComPortInstance *pInstance )   // API (for the GUI)
{
  static char sz127Result[128]; // only called by the GUI, no need to be thread safe,
                                // so a simple static buffer is ok here
  char *cp         = sz127Result;
  char *pszEndstop = sz127Result + 120;

  SL_AppendString( &cp, pszEndstop, RigCtrl_PortUsageToString(pInstance->pRigctrlPort->iPortUsage) );
  switch( pInstance->pRigctrlPort->iPortUsage )
   { case RIGCTRL_PORT_USAGE_SERIAL_TUNNEL : // also show the TUNNEL NUMBER:
        if( pInstance->iTunnelIndex < 0 )
         { SL_AppendString( &cp, pszEndstop, " (any)" );
         }
        else  // even though the array index (iTunnelIndex) is zero-based, start numbering at ONE in the GUI (as in the Aux-COM-Port-Dialog's combo boxes):
         { SL_AppendPrintf( &cp, pszEndstop, " %d", (int)(pInstance->iTunnelIndex+1) );
         }
        break;
     case RIGCTRL_PORT_USAGE_VIRTUAL_RIG : // also show the PROTOCOL use on this 'Virtual Rig'-emulating port:
        SL_AppendPrintf( &cp, pszEndstop, " (%s)", AuxCom_ProtocolToString( pInstance->dwParams ) );
        break;
     default:
        break;
   } // end switch( pInstance->pRigctrlPort->iPortUsage )
  SL_AppendPrintf( &cp, pszEndstop, ", tx=%ld, rx=%ld bytes, %ld loops",
                   (long)pInstance->dwNumBytesSent,
                   (long)pInstance->dwNumBytesRcvd,
                   (long)pInstance->dwThreadLoops );
      // (the string should be short enough to fit in the little
      //  "COM Port Details" dialog, AuxComPortDetails_Dialog.cpp .
      //  The full-length result can only be seen in the keyer GUI's
      //  "Debug"-tab, but that's not a periodically updated 'life' display)

  if( pInstance->dwThreadErrors > 0 )
   { SL_AppendPrintf( &cp, pszEndstop, ", %ld thread errors",
                   (long)pInstance->dwThreadErrors );
   }
  if( pInstance->dwSerialPortErrors > 0 )
   { SL_AppendPrintf( &cp, pszEndstop, ", %ld serial port errors",
                   (long)pInstance->dwSerialPortErrors );
   }
  if( pInstance->sRxFifo.fifo.nOverflows > 0 )
   { SL_AppendPrintf( &cp, pszEndstop, ", %ld RxFIFO overflows",
                   (long)pInstance->sRxFifo.fifo.nOverflows );
   }
  if( pInstance->sTxFifo.fifo.nOverflows > 0 )
   { SL_AppendPrintf( &cp, pszEndstop, ", %ld TxFIFO overflows",
                   (long)pInstance->sTxFifo.fifo.nOverflows );
   }

  return sz127Result;
} // end AuxCom_GetCurrentStatusAsString()

//---------------------------------------------------------------------------
const char* AuxCom_SignalStatesToString( T_AuxComPortInstance *pInstance, char *szBuffer, int iMaxLen )
  // [in] pInstance->dwDigitalSignalStates, with a bit combination of
  //          AUX_COM_DIG_SIGNAL_DCD/DSR/CTS/RI/RTS ("fixed")
  //       +  AUX_COM_DIG_SIGNAL_DOT/DASH/SKEY/PTT  ("programmable") .
  // [return, sz15Buffer] : short string (must fit on the 'TEST' tab)
  //          where only the CURRENTLY ACTIVE (logic HIGH) bits are listed.
  //          Worst case: "DCD DSR CTS RI DTR RTS DOT DASH SKEY PTT" (all bits set)
  //                       |______________________________________|
  //                           '--> circa 40 characters,
{
  char *pszDst = szBuffer;
  char *pszEnd = szBuffer+iMaxLen-1;

  *szBuffer = '\0';
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_DCD )
   { SL_AppendString( &pszDst, pszEnd, "DCD ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_DSR )
   { SL_AppendString( &pszDst, pszEnd, "DSR ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_CTS )
   { SL_AppendString( &pszDst, pszEnd, "CTS ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_RI )
   { SL_AppendString( &pszDst, pszEnd, "RI ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_DTR )
   { SL_AppendString( &pszDst, pszEnd, "DTR ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_RTS )
   { SL_AppendString( &pszDst, pszEnd, "RTS ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_DOT )
   { SL_AppendString( &pszDst, pszEnd, "DOT ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_DASH )
   { SL_AppendString( &pszDst, pszEnd, "DASH ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_SKEY )
   { SL_AppendString( &pszDst, pszEnd, "SKEY ");
   }
  if( pInstance->dwDigitalSignalStates & AUX_COM_DIG_SIGNAL_PTT )
   { SL_AppendString( &pszDst, pszEnd, "PTT ");
   }
  if( pszDst > szBuffer )
   {  pszDst[-1] = '\0';  // remove the trailing space
   }
  return szBuffer;
} // end AuxCom_SignalStatesToString()


//---------------------------------------------------------------------------
void AuxCom_UpdateSpeedTestResult( // .. for development and software test ..
             T_AuxComPortInstance *pInstance,
             int iTestItem,     // [in] e.g. KEYER_SPEEDTEST_POLL_KEYBOARD
             int nMicroseconds ) // [in] new test result in microseconds
  // Note: Similar _UpdateSpeedTestResult() incarnations exist in
  //        AuxComPorts.c, CwText.c, and KeyerThread.c. The latter is even
  //        instantiatable for an ARRAY of 'worker threads' using serial ports.
{
  if( nMicroseconds >= pInstance->iSpeedTestPeaks_us[iTestItem] )
   {  pInstance->iSpeedTestPeaks_us[iTestItem] = nMicroseconds;
      // Note: A new "peak- and average detection" begins
      //   when clicking "Report Test Results (on the 'debug' tab)" in the GUI
      //  -> AuxCom_ResetSpeedTestResults()
      // Typical test results using VIRTUAL NULL-MODEMS ("com0com"):
      // > ADDITIONAL COM PORT #1 (on COM7)..
      // >  Status: Winkeyer EMULATOR, tx=4, rx=57 bytes, 70769 thread loops
      // >  Serial I/O times: av= 0 us, pk= 329 us
      // > ADDITIONAL COM PORT #2 (on COM21)..
      // >  Status: Client/Server Tunnel, tx=0, rx=0 bytes, 1026 thread loops
      // >  Serial I/O times: av= 0 us, pk= 50 us
   }
  pInstance->i32SpeedTestSums_us[iTestItem] += nMicroseconds;
  pInstance->i32SpeedTestCounts_us[iTestItem]++;

} // end AuxCom_UpdateSpeedTestResult()

//---------------------------------------------------------------------------
int  AuxCom_GetSpeedTestAverage_us( T_AuxComPortInstance *pInstance,
              int iTestItem ) // [in] e.g. KEYER_SPEEDTEST_POLL_KEYBOARD
  // Important also for the USER, to find out if "his" USB <-> serial port's
  //  adapter / driver is fast enough for being called several hundred times
  //  per second. So far, FTDI interfaces were ok, some non-name stuff was not.
{ long i32Divisor = pInstance->i32SpeedTestCounts_us[ iTestItem ];
  if( i32Divisor > 0 )
   { return (int)( pInstance->i32SpeedTestSums_us[iTestItem] / i32Divisor );
   }
  else // no valid data for this 'item' -> say the average time spent for it was ZERO
   { return 0;
   }
} // end AuxCom_GetSpeedTestAverage_us()

//---------------------------------------------------------------------------
void AuxCom_ResetSpeedTestResults(T_AuxComPortInstance *pInstance) // begins a new peak detection
{
  memset( (void*)pInstance->iSpeedTestPeaks_us,   0, sizeof_member(T_AuxComPortInstance,iSpeedTestPeaks_us) );
  memset( (void*)pInstance->i32SpeedTestSums_us,  0, sizeof_member(T_AuxComPortInstance,i32SpeedTestSums_us)  );
  memset( (void*)pInstance->i32SpeedTestCounts_us,0, sizeof_member(T_AuxComPortInstance,i32SpeedTestCounts_us));
} // end AuxCom_ResetSpeedTestResults()





/* end < Remote_CW_Keyer/AuxComPorts.c > */



