// File: C:\cbproj\Remote_CW_Keyer\ALawCompression.c
// Purpose: A-Law compression for low-bandwidth audio streams
//          with 8 bits per sample but a better dynamic range,
//          as explained below (copied from an article by Ben Joan; 
//          but the actual IMPLEMENTATION in nice plain C was found at
//          hazelware.luggle.com/tutorials/mulawcompression.html )
// > A-law vs u-Law
// > A-law and u-law are two algorithms that are used in modifying an input signal for digitization. 
// > These algorithms are implemented in telephony systems all over the world. The two algorithms 
// > have a fairly minimal difference and most people would not know the difference. 
// > The first difference between the two is the dynamic range of the ouput; U-law has a larger 
// > dynamic range than a-law. Dynamic range is basically the ratio between the quietest 
// > and loudest sound that can be represented in the signal. The downside of having a higher 
// > dynamic range is greater distortion of small signals. This simply means that a-law would 
// > sound better than u-law when the sound input is very soft.
// > As mentioned earlier, A-Law compression is extremely similar to Mu-Law compression. 
//
// We expect narrow-band filtered CW signals to be "very soft", 
//    thus use A-law compression rather than u-law ..

#include "ALawCompression.h"  // header for this module


//---------------------------------------------------------------------------
// A-Law Compression Code:
//---------------------------------------------------------------------------

static char ALawCompressTable[128] =
{
  1,1,2,2,3,3,3,3,
  4,4,4,4,4,4,4,4,
  5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,
  6,6,6,6,6,6,6,6,
  6,6,6,6,6,6,6,6,
  6,6,6,6,6,6,6,6,
  6,6,6,6,6,6,6,6,
  7,7,7,7,7,7,7,7,
  7,7,7,7,7,7,7,7,
  7,7,7,7,7,7,7,7,
  7,7,7,7,7,7,7,7,
  7,7,7,7,7,7,7,7,
  7,7,7,7,7,7,7,7,
  7,7,7,7,7,7,7,7,
  7,7,7,7,7,7,7,7
};

BYTE LinearToALawSample(short sample)
{
  int sign;
  int exponent;
  int mantissa;
  BYTE compressedByte;

  sign = ((~sample) >> 8) & 0x80;
  if( !sign)
   { sample = (short)-sample;
   }
  // if( sample > 32767) // BCB: "Condition is always false". Indeed.
  //  {  sample = 32767;
  //  }
  if( sample >= 256)
   {
     exponent = (int)ALawCompressTable[(sample >> 8) & 0x7F];
     mantissa = (sample >> (exponent + 3) ) & 0x0F;
     compressedByte = ((exponent << 4) | mantissa);
   }
  else
   {
     compressedByte = (unsigned char)(sample >> 4);
   }
  compressedByte ^= (sign ^ 0x55);
  return compressedByte;
}



//---------------------------------------------------------------------------
// A-law Decompression:  Use a look-up table; no function call at all !
// > Now, the most obvious way to decompress a compressed Mu-Law or A-Law
// > sample would be to reverse the algorithm. But a more efficient method 
// > exists. Consider for a moment the fact that A-Law and Mu-Law both take
// > a 16-bit value and crunch it down to an 8-bit value. 
// > The reverse of that is to take an 8-bit value and turn it into a sixteen
// > bit value.  So, why not take a page from the world of graphics 
// > and use palettes for the Mu-Law and A-Law compression look up ? 
// > To convert from your compressed sample back to the raw 16-bit sample, 
// > just use your compressed sample as the index into the table, and the 
// > corresponding value in the table is your decompressed 16-bit sample. 
//---------------------------------------------------------------------------


const short ALawDecompressTable[256] =  // look-up table for A-Law decompression
{
     -5504, -5248, -6016, -5760, -4480, -4224, -4992, -4736,
     -7552, -7296, -8064, -7808, -6528, -6272, -7040, -6784,
     -2752, -2624, -3008, -2880, -2240, -2112, -2496, -2368,
     -3776, -3648, -4032, -3904, -3264, -3136, -3520, -3392,
     -22016,-20992,-24064,-23040,-17920,-16896,-19968,-18944,
     -30208,-29184,-32256,-31232,-26112,-25088,-28160,-27136,
     -11008,-10496,-12032,-11520,-8960, -8448, -9984, -9472,
     -15104,-14592,-16128,-15616,-13056,-12544,-14080,-13568,
     -344,  -328,  -376,  -360,  -280,  -264,  -312,  -296,
     -472,  -456,  -504,  -488,  -408,  -392,  -440,  -424,
     -88,   -72,   -120,  -104,  -24,   -8,    -56,   -40,
     -216,  -200,  -248,  -232,  -152,  -136,  -184,  -168,
     -1376, -1312, -1504, -1440, -1120, -1056, -1248, -1184,
     -1888, -1824, -2016, -1952, -1632, -1568, -1760, -1696,
     -688,  -656,  -752,  -720,  -560,  -528,  -624,  -592,
     -944,  -912,  -1008, -976,  -816,  -784,  -880,  -848,
      5504,  5248,  6016,  5760,  4480,  4224,  4992,  4736,
      7552,  7296,  8064,  7808,  6528,  6272,  7040,  6784,
      2752,  2624,  3008,  2880,  2240,  2112,  2496,  2368,
      3776,  3648,  4032,  3904,  3264,  3136,  3520,  3392,
      22016, 20992, 24064, 23040, 17920, 16896, 19968, 18944,
      30208, 29184, 32256, 31232, 26112, 25088, 28160, 27136,
      11008, 10496, 12032, 11520, 8960,  8448,  9984,  9472,
      15104, 14592, 16128, 15616, 13056, 12544, 14080, 13568,
      344,   328,   376,   360,   280,   264,   312,   296,
      472,   456,   504,   488,   408,   392,   440,   424,
      88,    72,   120,   104,    24,     8,    56,    40,
      216,   200,   248,   232,   152,   136,   184,   168,
      1376,  1312,  1504,  1440,  1120,  1056,  1248,  1184,
      1888,  1824,  2016,  1952,  1632,  1568,  1760,  1696,
      688,   656,   752,   720,   560,   528,   624,   592,
      944,   912,  1008,   976,   816,   784,   880,   848
};

