/* 
   vbidecode.cc - videotext, intercast, VPS  and videocrypt VBI data decoder
   
   V1.04  29.3.1999

   Improved version by Rolf Bleher <dk7in@qsl.net>, see README.rb
   - only videotext and intercast data format and PAL for now...

   Based on vbidecode 1.1.2
     Copyright (C) 1997,1998 by Ralph Metzler (rjkm@thp.uni-koeln.de)
     All rights reserved.

   Parts of this program based on alevt 1.4.2 
     by Edgar Toernig (froese@gmx.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
*/

/*
TODO:

 - support NTSC Teletext, CC and intercast
 - RS error correction for intercast
   (did some experimenting, but nothing included here)
 - handling of data with bad CRC or hamming code
 - make everything byte order independent for Linux on big endian machines
   (pointers in intercast data are little endian)
 - support other data formats besides Bt848 raw data (e.g. YUV)
 - handling of interleaved intercast data 
 - make sense of strange directory structure transmission in DSF
 - lots more ... 

 -rb- cleanup the fragments I left behind me

 Brooktree datasheet:
 PAL Video Frame:
     1-  6  Vert Sync  \
     7- 23  VBI         > odd field
    24-310  Video      /
   311-318  Vert Sync  \
   319-335  VBI         > even field
   336-625  Video      /

 NTSC Video Frame:
     1-  9  Vert Sync  \
    10- 20  VBI         > odd field
    21-263  Video      /
   263-272  Vert Sync  \
   272-283  VBI         > even field
   283-525  Video      /

  PAL:
    vdelay 0x20   offset    ?? 32 ?? Offset fr Bild?
    vbispl 2044   samples per line, buffer: 2048 per line
    vbinum  16    vbi lines 0..15, odd: 0..15, even: 19..34
 
 VBI Frame Output Mode
   includes hor blank/sync
   8 packets of 256 DWORDs per VBI line = 2048 bytes

   Reg:                  p.130
     0x0E0 VBI_PACK_SIZE  7..0 VBI_PKT_LO
     0x0E4 VBI_PACK_SEL   7..2 VBI_HDELAY
     0x0E4 VBI_PACK_SEL      1 EXT_FRAME
     0x0E4 VBI_PACK_SEL      0 VBI_PKT_HI
     0x100 INT_STAT      31.28 RISCS    status only
                            27 RISC_EN
                            25 RACK
                            24 FIELD
                            19 SCERR
                            18 OCERR
                            17 PABORT
                          ...
     0x104 INT_MASK

*/

#include <sys/types.h>
#include <string.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strstream.h>
#include <errno.h>
#include <fstream.h>
#include <iostream.h>
#include <iomanip.h>
#include <dirent.h>
#include <sched.h>
#include <sys/stat.h>

#include "vbidec-rb.h"
#include "tables-rb.h"
#include "../../driver/bttv.h"

#define RBTIMING            // better timing (borrowed from alevt)
#define TEST3PIX            // bit detection with 2 out of 3 pixels

#define BPL 2048                       // bytes per vbi line
#define VBI_LINENUM VBI_MAXLINES       // 19 lines per frame
#define VBI_BPF 2*VBI_LINENUM*BPL      // 2 * 19 * 2048 byte (raw vbi)
#define VT_PAGESIZE 25*40              // 25 lines, 40 columns
#define VTPAGE_DIRTY 1                 //
#define VBI_VT  1                      // Videotext
#define VBI_VPS 2                      // VPS
#define VBI_VC  4                      // VideoCrypt
#define VBI_VD  8                      // VideoDat
#define FPSHIFT 16                     //
#define FPFAC (1<<FPSHIFT)             // FAC
#define FAC FPFAC
#define STEP vtstep

int verbosity=0;
int debug = 0;
//int lcnt=0;                          // for testing
//int mcnt=0;

#define LOG(v,l) {if (verbosity&(v)) {l;}}

#define HEX(N) hex << setw(N) << setfill(int('0')) 
#define DEC(N) dec << setw(N) << setfill(int('0')) 

//----------------------------------------------------------------------------

void hdump(u8 *data, int len, u8 xor=0)
{
  int i;
  for (i=0; i<len; i++)
    cout << HEX(2) << int(xor^data[i]) << " ";
}

//----------------------------------------------------------------------------

void hdump10(u8 *data, int len, int width=16)
{
  int i,j;
  for (i=0; i<len; i+=width) {
    for (j=0; j<width; j++)
      if (i+j < len) 
	cout << HEX(2) << int(data[i+j]) << " ";
      else
	cout << "   ";
    for (j = 0; j < width && i + j < len; j++)
      cout << char((data[i+j]>31 && data[i+j]<127) ? data[i+j] : '.');
    cout << "\n";
  }
}

//----------------------------------------------------------------------------

void adump(u8 *data, int len, u8 xor=0)
{
  int i;
  u8 c;

  for (i=0; i<len; i++) {
    c=0x7f&(xor^data[i]);
    if (c<0x20)
      c='.';
    cout << char(c);
  }
}

//----------------------------------------------------------------------------

static int
hamm4(u8 *p, int *err)
{
    int a = hammtab[p[0]];
    *err += a;
    return a & 15;
}

//----------------------------------------------------------------------------

static int
hamm8(u8 *p, int *err)
{
    int a = hammtab[p[0]];
    int b = hammtab[p[1]];
    *err += a;
    *err += b;
    return (a & 15) | (b & 15) * 16;
}

//----------------------------------------------------------------------------

static int
chk_parity0(u8 *p, int n)
{
    int err;

    for (err = 0; n--; p++)
      if (! odd_parity[*p])
        *p = BAD_CHAR, err++;
//    else
//      *p &= 0x7f;    // should be stored with parity...
    return err;
}

//----------------------------------------------------------------------------

static int
chk_parity(u8 *p, int n)
{
    int err;

    for (err = 0; n--; p++)
      if (! odd_parity[*p])
        *p = BAD_CHAR, err++;
    else
      *p &= 0x7f;    // should be stored with parity...
    return err;
}

//----------------------------------------------------------------------------

struct vpsinfo {
  char lastname[9];
  char chname[9];
  int namep;
  u8 *info;
  
  getname(char *name) {
    memcpy(name,lastname,9);
  }
  vpsinfo() {
    namep=0;
    chname[8]=0;
  }
  void decode(u8 *data) {
    info=data;
    LOG(2,cout << "VPS: ";
    hdump(info+2,13);
    adump(info+2,13);
    )
    if ((info[3]&0x80)) {
      chname[namep]=0;
      if (namep==8) {
	memcpy(lastname,chname,9);
	LOG(2,cout << chname;)
      }
      namep=0;
    }
    chname[namep++]=info[3]&0x7f;
    if (namep==9) 
      namep=0;

    LOG(2,cout << "\n";)
  }

  /* Who wants to add other VPS stuff (start/stop of VCR, etc. )??? */
  
};

//----------------------------------------------------------------------------

struct chan_name {
  chan_name *next;
  char *name;
};
static chan_name *channel_names=0;

//----------------------------------------------------------------------------

void getcnames(void)
{
  DIR *dp;
  struct dirent *dirp;
  struct stat statbuf;

  dp=opendir(".");
  LOG(1, cout << "Data directories: ";)
  while ((dirp=readdir(dp)) !=NULL) {
    if (strcmp(dirp->d_name, ".") == 0 ||
	strcmp(dirp->d_name, "..") == 0)
      continue;
    lstat(dirp->d_name, &statbuf);
    if (S_ISDIR(statbuf.st_mode)) {
      chan_name *cn=new chan_name;
      cn->name=new char[strlen(dirp->d_name)+1];
      strcpy(cn->name, dirp->d_name);
      cn->next=channel_names;
      channel_names=cn;
      LOG(1,cout << cn->name << " ";)
    }
  }
  LOG(1,cout << endl;)
}

//----------------------------------------------------------------------------

int nmatch(char *str, char *sub, int len) { // str: Sender, sub: Dirs
  int i, n=0; 
  for (i=0; i<len; i++) 
    if (str[i]=='\1' || str[i]==sub[i])
      n++;
  return n;
}

//----------------------------------------------------------------------------

void filter(u8 *name,int len) {   // filter chars in channel names
  int i;                                     // 24 char
  u8 c;
  
  for (i=0; i< len; i++){
    c=name[i];
//  if (!(unhamtab[c] &0x80)) // -> hamm4( )
//    c='\1';
    c&=0x7f;
    if (c < ' ') 
      c=' ';
    switch (c) {
    case '/':
    case 0x23:        // #
    case 0x24:        // $
    case 0x40:        // @
    case 0x5b:        // [
    case 0x5c:        // \
    case 0x5d:        // ]
    case 0x5e:        // ^
    case 0x5f:        // _
    case 0x60:        // ' 
    case 0x7b:        // {
    case 0x7c:        // |
    case 0x7d:        // }
    case 0x7e:        // ~
      c = '\1';       // matched jeden char
      break;
    }
    name[i]=c;
  }
};
  
//----------------------------------------------------------------------------

struct VTpage {
  VTpage *next;
  unsigned int number; 
  unsigned int linepage[25];
  u8 page[VT_PAGESIZE];                     // eine VTX Seite
  ushort flags;

  /* try to match a sub-directory name to part of the header line */
  char *match(char *cname) {
    u8 c;
    int i, len,n;
    chan_name *names=channel_names;
    
    while (names) {
      len=strlen(names->name);
      for (i=0; i<24-len; i++) {
	n=nmatch(cname+i,names->name,len);
	if (n>len-1) {
	  //if (n!=len)
	  //cerr << "Partial match on " << names->name << "\n";
	  return names->name;
	}
      }
      names=names->next;
    }
    return (char *) 0;
  };


  void write(void) {                    // write one page
    char name[80];
    u8 cname[25];
    ushort pn;
    ostrstream oname(name,80);
    char *cmatch;

    memcpy(cname,page+8,24);
    filter(cname,24);
    cname[24]=0;
    cmatch=match((char *)cname);

    if (!cmatch) {
      LOG(1, cout << "No channel match for: ";
        adump(page+8,24);
        cout << "\n";)
      return;
    }
    for (int i=0;i<25;i++) {             // fehlende Zeilen lschen
//      lcnt++;        // test
      if (linepage[i]!=number) { 
        memset(page+40*i, ' ', 40);
//        mcnt++;      // test
      }
    }
//    if (lcnt > 10000) {
//      cout << DEC(3) << (int)(1000*mcnt/lcnt) << "\n";
//      lcnt = mcnt = 0;
//      cout << "---\n" ;
//    }

//    for (int j=0;j<10;j++) {
//      cout << HEX(2) << page[40*10+j];
//    }
//    cout << "\n";

    pn=number&0xfff;
    if (!(pn&0xf00))                    // 0xx -> 8xx
      pn+=0x800;

    oname                               //<< "/var/spool/vtx/" 
      << cmatch << "/"                  //"VTX/" 
	<< HEX(3) << pn << "_" << HEX(2) << int(number>>16) 
	  << ".vtx" << char(0);
    ofstream of(name);
    of << "VTXV4" << (u8)(number&0xff) << (u8)((pn>>8)&0xf)
      << (u8)(number>>24) << (u8)(number>>16);

    // hardcode language code to 4 until I figure out how it works:
//  of << (u8)(flags>>5)
    flags = 0x04;
    of << (u8)(flags)
      << (u8)(0);

    of << (u8)(0);
    of.write(page,40*24);
    of.close(); 
  };


//  Flags (alevt):
// PG_SUPPHEADER 0x01  C7  b4 & 0x01  row 0 is not to be displayed
// PG_UPDATE     0x02  C8  b4 & 0x02  row 1-28 has modified (editors flag)
// PG_OUTOFSEQ   0x04  C9  b4 & 0x04  page out of numerical order
// PG_NODISPLAY  0x08  C10 b4 & 0x08  rows 1-24 is not to be displayed
// PG_MAGSERIAL  0x10  C11 b4 & 0x10  serial trans. (any pkt0 terminates page)
// PG_ERASE      0x20  C4  b2 & 0x80  clear previously stored lines
// PG_NEWSFLASH  0x40  C5  b3 & 0x40  box it and insert into normal video pict
// PG_SUBTITLE   0x80  C6  b3 & 0x80  box it and insert into normal video pict

// flags (vbidecode):
// 0x01  b4 & 0x10  C11
// 0x02  b4 & 0x08  C10
// 0x04  b4 & 0x04  C9
// 0x08  b4 & 0x02  C8
// 0x10  b4 & 0x01  C7
// 0x20  b3 & 0x80  C6
// 0x40  b3 & 0x40  C5
// 0x80  b2 & 0x80  C4

  void setline(u8 *data, int rbflags, int line) {
  int err=0;

    if (!line) {                            // 0: init flags from line 0
      u8 c= hamm8(data+4, &err);            // b3
      flags=hamm8(data+2, &err)&0x80;       // b2 & 0x80
      flags|=(c&0x40)| ((c>>2)&0x20);
      c=hamm8(data+6, &err);                // b4
      /* Hmm, this is probably not completely right, or is it? */
      flags|=((c<<4)&0x10)|((c<<2)&0x08)|(c&0x04)|((c>>2)&0x02)|((c>>4)&0x01);
//a   flags = rbflags & 0xff;  // ? anderes Format!
    }

    if (line<25) {                                     // 0 .. 24
      linepage[line]=number;                           // tag
      memcpy(page+40*line, data, 40);                  // store line
    }

    if (line==23) {                                    // ??
      write();                                         // write page
      memset(page, ' ',24*40);       // ? 25*40 ?      // clear memory
    }
    
    // line==24 gibt es gar nicht...  nur ab und zu bei n-tv, VOX, BR
  
/*
    if (line==23) {
      cout << " PAGE " << number << "\n";
      for (int i=0; i<24; i++) {
	cout << "DP "; adump(page+40*i,40);
	cout << "\n";
      }
    }
*/

  }


  VTpage(unsigned int pnum) : number(pnum) {           // ?? new
    next=0;
    flags=0;
    memset(linepage,0,sizeof(linepage));               // clear page
  };

};

//----------------------------------------------------------------------------

struct VTmagazine {
  VTpage *pages;
  VTpage *current;

  VTmagazine(void) {                        // constructor
    pages=current=NULL;                     // init: keine Seiten
  };

  ~VTmagazine(void) {                       // destructor
    VTpage *p=pages, *pn;
    while (p) {
      pn=p->next;
      delete p;                             // alle Seiten lschen
      p=pn;
    }
  }

  void selectpage(unsigned int pagenum, int flags, u8 *header) {
    VTpage *p=pages;

    if ((pagenum&0xff)==0xff)
      return;
    while (p) {                             // search page
      if (p->number==pagenum) {             // page is in memory
	current=p;                          // set current page
	p->setline(header,flags,0);         // line 0: store 40 bytes header
	return;
      }
      p=p->next;
    } 

    /* Comment this out, if you want to keep all pages in memory */
    /* actually, if you do NOT want all pages in memory (default) you can
       delete a lot of other stuff */
    if ((p=pages)) {
      pages=p->next;
      delete p;
    }

    p=new VTpage(pagenum);                  // insert new page
    p->next=pages;
    pages=current=p;
    setline(header,flags,0);                // line 0: store 40 bytes header
    // current->

//    if ((pagenum&0xfff)==0x303)     // search page
//      cout << "Created page " << HEX(5) << pagenum << "\n";
    // 10556       subpage 1, page 556
  }

  void setline(u8 *data, int flags, int line) {
    if (current)                            // line 0 mu gesetzt sein
      current->setline(data,flags,line);    // line: store 40 bytes
  };
};

//----------------------------------------------------------------------------

struct VTchannel {
  char name[20];
  VTmagazine mag[8];                             /* Hunderter */

  // line 0
  void selectpage(unsigned int pagenum, int flags, u8 *header) {
    mag[(pagenum>>8)&0x07].selectpage(pagenum, flags, header);
  };

  // line 1..24
  void setline(u8 *data, int line, int magnum) {

    mag[magnum].setline(data,0,line);
  };

};

VTchannel vtch;

//----------------------------------------------------------------------------

struct VDdeco {
  int Vflag;

  write(u8 *data, int n) {
    int i;
    for (i=0; i<n; i++)
      decode(data[i]);
  }

  decode(u8 dat) {
    if (!Vflag) {
      if (dat==0x56)
	Vflag=1;
    } else {
      if (dat==0xae)
      LOG(32,cerr << "P";);
      Vflag=0;
    }

  }

  VDdeco(void) {
    Vflag=0;
  }
};

//----------------------------------------------------------------------------

/* Intercast decoder */
/* only handles the strange (and nowhere documented) format used on
   the German channels ZDF and DSF for now 
*/

struct ICdeco {
  u8 blocks[28*16];
  u8 data[0x100];
  u8 *fbuf;
  unsigned int packnum, packtot, packlen, total;
  int datap, ok, esc, ciok;
  u8 ci,lastci;  /* ci = continuity index */
  ushort dat;
  unsigned int length;
  unsigned int done, flength;
  int pnum;
  char *name;

  /* This is where the actual data arrives from the lower level 
     bit reading routines
     */
  void setblock(int nr, u8 *block) {
    memcpy(blocks+28*nr,block,28);
    /* XXX: also check if all 15 blocks actually arrived */
    if (nr==0x0f)
      procblocks();
  }

  /* These "inner loops" can probably be optimized a lot */
  /* They handle the SLIP-like escape codes (according to the internet draft it
     should be EXACTLY like SLIP!?!?) which mark the separate packages
     */
  void adata(u8 c) {
    if (ok) {
      if (datap>=0x100)
	ok=0;
      else
	data[datap++]=c;
    }
  }
  void procbyte(u8 c) {
    if (!esc) {
      // 0x10 escapes 0x02, 0x03 and itself
      if (c==0x10) {
	esc=1;
	return;
      }
      // unescaped 0x02 starts a package
      if (c==0x02)
	newpack();
      adata(c);
      // unescaped 0x03 ends a package
      if (c==0x03)
	procpack();
    } else {
      esc=0;
      adata(c);
    }
  }

  /* put working error correction code here .. */
  void fecblocks() {
    u8 rsblock[256];
    memset(rsblock,0,256);
    memcpy(rsblock,blocks,26);
    //encode_rs(rsblock,rsblock+253);
    hdump10(blocks,28);
    hdump10(rsblock+253,2);
  }

  void procblocks(void) {
    //fecblocks();
    int i,j;
    u8 *block;
    for (i=0;i<14;i++) {
      block=blocks+28*i;
      for (j=0;j<26;j++)
	procbyte(block[j]);
    }
    memset(blocks,0,16*28);
  }
  ICdeco(void) {
    reset();
  }
  void reset(void) {
    fbuf=0;
    esc=0;
    done=0;
    pnum=0;
    ok=0;
    datap=0;
    lastci=0xff;
  }
  void newpack() {
    datap=0;
    ok=1;
    
  }

  /* this handles a single data package 
     no time to explain this
     if you understand this and find something new plesae tell me! :-)
     */
  void procpack() {
    if (!ok)
      return;
    ok=0;
    length=datap-10;
    if (length>datap || length!=data[6]) {
      //ciok=0;
      LOG(1,cout << "packet length mismatch!" << "\n";);
      return;
    }
    //    hdump10 (data,7);
    //   hdump10 (data+7,datap-10); cout << endl;
    ci=data[4];
    if (data[2]) {
      u8 *pack=data+7;
      if (!ci) {
	/* XXX: this works on little endian only!!! */
	packnum=*((unsigned int *)(pack+0x0c));
	packtot=*((unsigned int *)(pack+0x10));
	packlen=*((unsigned int *)(pack+0x14));
	total = *((unsigned int *)(pack+0x18));
	
	LOG(4,
	    cout << "\nICP: packet " << HEX(2) << packnum << " of "
	    << HEX(2) << packtot << ", ";
	    cout << "length: " << HEX(4) << packlen << ", ";
	    cout << "total: " << HEX(8) << total << "\n";
	    );
	if (packnum==1) {
	  lastci=0x0f;
	  done=0;
	  /* I keep the whole file in memory for now */
          /* go ahead and change this but note that you will need to buffer
             the last 2 packages since name and other information might
             be on a package boundary! 
	     */
	  fbuf=(u8 *)realloc((void *)fbuf,total);
	  if (!fbuf)
	    cerr << "Realloc failed!\n";
	}
	if (lastci==0x0f)
	  ciok=1;
	lastci=0;
	if (ciok && (length>0x38)) {
	  memcpy(fbuf+done,pack+0x38,length-0x38);
	  done+=length-0x38;
	}    
      } else {
	if (lastci+1!=ci)
	  ciok=0;
	lastci=ci;
	if (ciok) {
	  LOG(4,cout << ".";cout.flush());
	  memcpy(fbuf+done,pack,length);
	  done+=length;
	} else
 	  LOG(4,cout << "x";cout.flush(););
      }
      if (total && (done==total)) {
	unsigned int npos, ipos;
	int nlen;

	/* At fbuf+total-6 is a pointer to an info structure */
	ipos=*((unsigned int *)(fbuf+total-6));
	flength=*((unsigned int *)(fbuf+ipos+2));

	nlen=(int) fbuf[ipos+0x34];
	char name[nlen+3];
	if (fbuf[ipos+0x32]&0x80) 
	  npos=ipos+0x35;
	else
	  npos=*((unsigned int *)(fbuf+ipos+0x38));
	/* Only save the file if pointers pass some sanity checks */
	if (npos+nlen<total && ipos<total && flength<=total) {
	  strcpy(name,"IC/");
	  strncpy(name+3, (char *)fbuf+npos, nlen);
	  name[nlen+2]=0; // just to be sure ...
	  ofstream icfile(name);
	  icfile.write(fbuf,flength);
	  icfile.close();
	  cout << "\nFile: "  << name 
	    << ", length: " << HEX(8) << flength 
	      << "\n";
	} 
	done=0;
	total=0;
	ciok=0;
	lastci=0xff;
      }

    }
  }
};


/* XXX: Make this and possibly other (interleaved) intercast transmissions
   be allocated dynamically */
ICdeco icd;

//----------------------------------------------------------------------------

static int
local_init(void)
{
    static int inited = 0;
    int i;

    if (inited)
        return 0;

    memset(lang_char, 0, sizeof(lang_char));
    for (i = 1; i <= 13; i++)
        lang_char[lang_chars[0][i]] = i;

    inited = 1;
    return 0;
}

//----------------------------------------------------------------------------

static void
conv2latin1(u8 *p, int n, u8 *lang)
{
    int c, gfx = 0;

    while (n--)
    {
        if (lang_char[c = *p])
        {
            if (! gfx || (c & 0xa0) != 0x20)
                *p = lang[lang_char[c]];
        }
        else if ((c & 0xe8) == 0)
            gfx = c & 0x10;
        p++;
    }
}

//----------------------------------------------------------------------------

static void
out_of_sync(struct vbi *vbi)
{
    int i;
    if (vbi->seq == 0) {
      LOG(1,cout << "noSync Frame=0" << endl;)
    }
    else {
      LOG(1,cout << "noSync" << endl;)
    }
    // discard all in progress pages
    for (i = 0; i < 8; ++i)
      vbi->page[i].flags &= ~PG_ACTIVE;
}

//----------------------------------------------------------------------------

// fine tune pll
// this routines tries to adjust the sampling point of the decoder.
// it collects parity and hamming errors and moves the sampling point
// a 10th of a bitlength left or right.

static void
pll_add(struct vbi *vbi, int n, int err)
{
    if (vbi->pll_fixed)
        return;

    if (err > PLL_ERROR*2/3)    // limit burst errors
        err = PLL_ERROR*2/3;

    vbi->pll_err += err;
    vbi->pll_cnt += n;
    if (vbi->pll_cnt < PLL_SAMPLES)
        return;

    if (vbi->pll_err > PLL_ERROR)
    {
        if (vbi->pll_err > vbi->pll_lerr)
            vbi->pll_dir = -vbi->pll_dir;
        vbi->pll_lerr = vbi->pll_err;

        vbi->pll_adj += vbi->pll_dir;
        if (vbi->pll_adj < -PLL_ADJUST || vbi->pll_adj > PLL_ADJUST)
        {
            vbi->pll_adj  =  0;
            vbi->pll_dir  = -1;
            vbi->pll_lerr =  0;
        }

        if (debug)
            printf("pll_adj = %2d\n", vbi->pll_adj);
    }
    vbi->pll_cnt = 0;
    vbi->pll_err = 0;
}


//----------------------------------------------------------------------------

void
vbi_pll_reset(struct vbi *vbi, int fine_tune)
{
    vbi->pll_fixed = fine_tune >= -PLL_ADJUST && fine_tune <= PLL_ADJUST;

    vbi->pll_err  =  0;
    vbi->pll_lerr =  0;
    vbi->pll_cnt  =  0;
    vbi->pll_dir  = -1;
    vbi->pll_adj  =  0;
    if (vbi->pll_fixed)
        vbi->pll_adj = fine_tune;
    if (debug)
        if (vbi->pll_fixed)
            printf("pll_reset (fixed@%d)\n", vbi->pll_adj);
        else
            printf("pll_reset (auto)\n");
}

//----------------------------------------------------------------------------

// decode data in videotext-like packages, like videotext itself, intercast, ...
static int
vt_line(struct vbi *vbi, u8 *p)             // 0 .. 41
{
  int FL,NR;
  u8 pack, mpag, flags, ftal, ft, al, page;
  unsigned int addr;
  static unsigned int pnum=0;
  ushort sub;
  char c;

  struct vt_page *cvtp;
  int hdr, mag, mag8, pkt, i;
  int err = 0;

  hdr = hamm8(p, &err);                     // = mpag
  if (err & 0xf000)
    return -4;

  mag  = hdr & 7;                           // Magazin 0..7
  mag8 = mag?: 8;                           // Hunderter der Seitennummer
  pkt  = (hdr >> 3) & 0x1f;                 // Art des Datenpakets = pack
  p += 2;

  cvtp = vbi->page + mag;                   // ptr in vbi

  switch (pkt)
  {
    case 0:                                 // setzt die Seitenzahl
    {
      int b1, b2, b3, b4;

      b1 = hamm8(p,   &err);                // page number    -nn
      b2 = hamm8(p+2, &err);                // subpage number + flags
      b3 = hamm8(p+4, &err);                // subpage number + flags
      b4 = hamm8(p+6, &err);                // language code + more flags

      //if (vbi->ppage->flags & PG_MAGSERIAL)   // ????
      //  vbi_send_page(vbi, vbi->ppage, b1);
      //vbi_send_page(vbi, cvtp, b1);

      if (err & 0xf000)
        return 4;

      cvtp->errors = (err >> 8) + chk_parity0(p + 8, 32);
      cvtp->pgno   = 256 * mag8 + b1;                      // nnn
      cvtp->subno  = (256 * b3 + b2) & 0x3f7f;
      cvtp->lang   = "\0\4\2\6\1\5\3\7"[b4 >> 5];
      cvtp->flags  = b4 & 0x1f;                  // 5 bits C7-C11
      cvtp->flags |= b3 & 0xc0;                  // 2 bits C5-C6
      cvtp->flags |= (b2 & 0x80) >> 2;           // 1 bit  C4
      cvtp->lines  = 1;
      cvtp->flof   = 0;
      vbi->ppage   = cvtp;

      pll_add(vbi, 1, cvtp->errors);

//a   conv2latin1(p + 8, 32, lang_chars[cvtp->lang + 1]);
      //vbi_send(vbi, EV_HEADER, mag8, cvtp->flags, cvtp->errors, p);

      if (b1 == 0xff)                       // ? see pkt = 30
        return 0;

      cvtp->flags |= PG_ACTIVE;             // currently fetching this page
//a   memcpy(cvtp->data[0]+0, p, 40);       // store data
//a   memset(cvtp->data[0]+40, ' ', sizeof(cvtp->data)-40); // blank rest
      pnum = (cvtp->subno<<16) | cvtp->pgno;
      vtch.selectpage(pnum, cvtp->flags, p);  // store line 0 + Flags
      return 0;
    }

    case 1 ... 24:                          // line 1 .. 24
    {
      pll_add(vbi, 1, err = chk_parity0(p, 40));

      if (~cvtp->flags & PG_ACTIVE)
        return 0;

      cvtp->errors += err;
      cvtp->lines |= 1 << pkt;
//a   conv2latin1(p, 40, lang_chars[cvtp->lang + 1]);
//a   memcpy(cvtp->data[pkt], p, 40);       // store data
      vtch.setline(p, pkt, mag);
      return 0;
    }

    case 25:      // VOX/Pro7/BR/BBC/DSF/Kabel/tVB
      /*    ?? Laufschrift in Header ?????
      pnum = (cvtp->subno<<16) | cvtp->pgno;
      cout << "page " << HEX(4) << int(pnum)  << " ";
      cout << "AltHeader:"; adump(p,40); cout << "\n";
      */
      return 0;

    case 26:      // VOX/Pro7/N3/WDR/BR/arte/DSF/Kabel/tVB/Prem
      // PDC      ????
      /*
      cout << "Pkt-26 p:";
      for (i=0;i<20;i++) {
        //cout << " " << HEX(2) << (int)p[i];
        cout << " " << HEX(2) << (int)hamm8(p+1+2*i,&err);
        //cout << " " << HEX(2) << (int)hamm8(p+2*i,&err);
      }
      cout << "\n";
      */
      return 0;

#ifdef HUHU
    case 27:       // BR/n-tv/CNN/BBC/tVB/OKB/TRT
    {
      // FLOF data
      int b1,b2,b3,x;

      if (~cvtp->flags & PG_ACTIVE)
        return 0; // -1 flushes all pages.  we may never resync again :(

      b1 = hamm4(p,    &err);
      b2 = hamm4(p+37, &err);
      if (err & 0xf000)
        return 4;
      if (b1 != 0 || !(b2 & 8))
        return 0;

      for (i = 0; i < 6; ++i)
      {
        err = 0;
        b1 = hamm8(p+1+6*i, &err);
        b2 = hamm8(p+3+6*i, &err);
        b3 = hamm8(p+5+6*i, &err);
        if (err & 0xf000)
          return 1;
        x = (b2 >> 7) | ((b3 >> 5) & 0x06);
        cvtp->link[i].pgno = ((mag ^ x) ?: 8) * 256 + b1;
        cvtp->link[i].subno = (b2 + b3 * 256) & 0x3f7f;
      }
      cvtp->flof = 1;
      return 0;
    }
#endif

    case 28:  // ZDF/Pro7/WDR/BR/arte/tVB/Prem
      /*
      cout << "Pkt-28 p:";
      for (i=0;i<20;i++) {
        //cout << " " << HEX(2) << (int)p[i];
        //cout << " " << HEX(2) << (int)hamm8(p+1+2*i,&err);
        cout << " " << HEX(2) << (int)hamm8(p+2*i,&err);
      }
      cout << "\n";
      */
      return 0;

    case 29:  // ZDF
      return 0;

    case 30: // B1/3sat/RTL2/VOX/N3/WDR/CNN/BBC/arte/MTV/Kabel/tVB/TV5/TRT/Prem
    {
      if (mag8 != 8)
        return 0;

      p[0] = hamm4(p,   &err);            // designation code
      p[1] = hamm8(p+1, &err);            // initial page
      p[3] = hamm8(p+3, &err);            // initial subpage + mag
      p[5] = hamm8(p+5, &err);            // initial subpage + mag
      // 0 00 ff 3f    bei arte zustzlich 2 00 ff 3f
      // bei MTV 0 ff 7f 3f

      if (err & 0xf000)
        return 4;

      err += chk_parity(p+20, 20);
      //conv2latin1(p+20, 20, lang_chars[1]);
      /*
      cout << "Pkt-30/8:";
      cout << " " << HEX(1) << (int)p[0];
      cout << " " << HEX(2) << (int)p[1];
      cout << " " << HEX(2) << (int)p[3];
      cout << " " << HEX(2) << (int)p[5];
      cout << ": ";
      adump(p+20,20);
      cout << "\n";
      */
      /*
      B1:    SFB-3               
      3sat:  3SAT / SWISS TXT    
      RTL2:  RTL2                
      VOX:   VOX                 
      N3:    NORDDEUTSCHES FS N3 
      WDR:   WDR                 
      CNN:   abwechselnd: CNN Interactive / CNN International   
      BBC:   BBC WORLD           
      arte:  DER TAG DER HEUSCHR.
      MTV:                     
      Kabel: INFOTEXT            
      tVB:   TV.BERLIN           
      TV5:   TV5 / SWISS TXT     
      TRT:   TRT TELEGUN         
      Prem:     PREMIERE      
      */
      //vbi_send(vbi, EV_XPACKET, mag8, pkt, err, p);
      return 0;
    }


    case 31:                                // Intercast
    {
      ftal=hamm8(p, &err);
      al = ftal>>4;                         // address length
      ft = ftal&0x0f;
      p += 2;
      for (addr=0,i=0; i<al; i++)
        addr = (addr<<4) | hamm4(p+i, &err);
      p += al;
      LOG(2,                                       // n-tv ZDF DSF
        cout << "FT:"    << HEX(1) << (int)ft;     //  4    0  div
        cout << " AL:"   << HEX(1) << (int)al;     //  1    6  div
        cout << " ADDR:" << HEX(8) << addr;        //  1   500 div
        if (ft&4)
          cout << " CI:" << HEX(2) << (int)p[0];  // 00..ff
        if (ft&8)
          cout << " RI:" << HEX(2) << (int)p[1];
          // byte count for text ?
        cout << "\n";
      )
      /*
        RTL:  FT:4 AL:0 ADDR:00000000 CI:nn        CI zhlt hoch (je 5)
        3sat: FT:c AL:6 ADDR:00000000 CI:nn RI:19  schnell! CI(1)++
        3sat: FT:4 AL:1 ADDR:00000001 CI:nn        schnell! CI(2)++
        n-tv: FT:4 AL:1 ADDR:00000001 CI:nn        schnell! CI++
        EuSp: FT:4 AL:1 ADDR:00000001 CI:nn        schnell! CI++
        TV5:  FT:c AL:2 ADDR:0000001a CI:nn RI:1a  schnell, CI++
        ZDF:  FT:0 AL:6 ADDR:00000500
        DSF:  FT:c AL:6 ADDR:00000500 CI:nn RI:nn  schnell
        ZDF:  FT:0 AL:6 ADDR:00000f00              seltener
        ARD:  FT:c AL:6 ADDR:00777777 CI:nn RI:07  alle 2 sec, CI++
        ZDF:  FT:c AL:6 ADDR:00777777 CI:nn RI:07  alle 2 sec, CI++
        B1:   FT:c AL:6 ADDR:00777777 CI:nn RI:08  alle 2 sec, CI durcheinander
        Phoe: FT:c AL:6 ADDR:00777777 CI:nn RI:07  alle 2 sec, CI++
        3sat: FT:  AL:  ADDR:00ffffff              jede sec
        DSF:  FT:f AL:f ADDR:ffffffff CI:20 RI:20  schnell
        VOX:  FT:f AL:f ADDR:ffffffff CI:20 RI:20  schnell! konstant
        Pro7: FT:f AL:f ADDR:ffffffff CI:20 RI:20  schnell! konstant
        TvPo: FT:f AL:f ADDR:ffffffff CI:20 RI:20  schnell! konstant
        BR:   FT:f AL:f ADDR:ffffffff CI:20 RI:20  schnell! konstant
        Kab:  FT:f AL:f ADDR:ffffffff CI:20 RI:20  schnell! konstant
        tVB:  FT:f AL:f ADDR:ffffffff CI:20 RI:20  schnell! konstant
        Pre:  FT:f AL:f ADDR:ffffffff CI:20 RI:20  schnell! konstant
      */
      switch (addr)
      {
        case 0x00:                          // RTL/3sat
          /*
          cout << "ADDR=0 p:";
          for (i=0;i<20;i++) {
            cout << " " << HEX(2) << (int)p[i];
            //cout << " " << HEX(2) << (int)hamm8(p+1+2*i,&err);
          }
          cout << "\n";
          */
          /* RTL:  jede sec 5 gleiche Bursts, CI zhlt hoch    FT:4
          ADDR=0 p: 82 9b b6 c7 49 49 5e ea 02 9b 5e 64 c7 b6 15 15 15 15 15 15
          ADDR=0 p: 83 9b b6 73 b6 49 5e 15 49 9b 5e 64 c7 b6 15 15 15 15 15 15
          ADDR=0 p: 84 9b b6 02 b6 49 5e 02 49 9b 5e 64 c7 b6 15 15 15 15 15 15
          ADDR=0 p: 85 9b b6 b6 b6 49 5e 49 49 9b 5e 64 c7 b6 15 15 15 15 15 15
          ADDR=0 p: 86 9b b6 c7 b6 49 5e 5e 49 9b 5e 64 c7 b6 15 15 15 15 15 15
          ADDR=0 p: 87 9b b6 73 a1 49 5e 64 49 9b 5e 64 c7 b6 15 15 15 15 15 15
          ADDR=0 p: 88 9b b6 02 a1 49 5e 73 49 9b 5e 64 c7 b6 15 15 15 15 15 15
          ADDR=0 p: 89 9b b6 b6 a1 49 5e 38 49 9b 5e 64 c7 b6 15 15 15 15 15 15
          ???
          */
          /* 3sat   CI zhlt hoch    FT:c
          ADDR=0 p: a8 19 0a 0d 53 4f 46 54 45 4c 20 53 79 73 74 65 6d 73 20 54
          cnt=19, Text= lf, cr, "SOFTEL_Systems_Teletext"
          */
          break;

        case 0x01:             // 3sat/n-tv/EuroSport  FT:4  CI zhlt hoch
          /*
          cout << "ADDR=1 p:";
          for (i=0;i<20;i++) {
            cout << " " << HEX(2) << (int)p[i];
            //cout << " " << HEX(2) << (int)hamm8(p+1+2*i,&err);
          }
          cout << "\n";
          */
          /* 3sat
          ADDR=1 p: 8b 07 55 93 1b 08 5c 0e 0e 72 e6 4f 99 8e 3c 81 ca 76 b7 be
          ADDR=1 p: 8c 2e d7 8c f3 83 f2 c0 15 10 23 f6 22 1f 18 2a 24 a2 e5 5d
          ADDR=1 p: 8d cb 2c 18 83 74 0c 2c ae 4e 86 4a 06 13 68 de 31 62 65 85
          ADDR=1 p: 8e 3e eb 57 b2 9a df 6d d9 9e 02 b0 c0 a0 c5 25 76 b7 23 91
          */
          break;

        case 0x07:                          // ???
          //cout << " ??:" << HEX(2) << int(hamm4(p+2, &err));
          break;

        case 0x0000001a:                    // TV5
          break;

        case 0x0500:
          /* here we usually have the following structure: (03-0e are hammed)
           dat: 00 01 02  03 04  05 06  07-0c    0d  0e  0f-2a 2b-2c
                  SYNC    MPAG    FTAL   ADDR    NR  FL   DATA   CRC
                55 55 27   31    00 06  000500   1-f 08  28bytes
                                                 ^p
          */
          NR = int(hamm4(p,   &err));       // line number
          FL = int(hamm4(p+1, &err));       // flags?
          p += 2;
          LOG(2,cout << " NR:" << HEX(1) << NR;)
          LOG(2,cout << " FL:" << HEX(1) << FL << "\n";)
          icd.setblock(NR,p);            // 28 bytes data to intercast decoder
          // 2 bytes CRC should be checked!
          if (!(FL&4)) {
            //cout << "\nICH: ";  hdump(p,26);
            //cout << "\nICA:";   adump(p,26);
            //cout << endl;
          }
          break;

        case 0x0f00: /* also used by ZDF and DSF, data format unknown */
          break;

        case 0x00777777:               // ARD/ZDF/B1/Phoenix
          //cout << "ADDR=777777 " << HEX(2) << (int)p[0] << " ";
          //adump(p+2,(int)p[1]-2);
          //cout << "\n";
          //      p+ 00  01 02 03 04 05 06 07 08 09 0a 0b ...
          //         nn cnt Text...
          // ARD     nn 07  T  I  C  X  X  cr lf  U  U  U  U ...
          // ZDF     nn 07  T  I  D  X  X  cr lf  U  U  U  U ...
          // B1      nn 08  T  I  D  X  X  A  cr lf  U  U  U ...
          // Phoenix nn 07  T  I  X  X  X  cr lf  U  U  U  U ...
          // I don't know what this is...
          break;

        case 0x00ffffff:                    // 3sat jede sec
          break;

        case 0xffffffff:                    // DSF/Vox/Pro7/...
          break;

        default:
          cout << "ADDR= " << HEX(8) << addr << "\n";
          break;
      };
      return 0;
    }


    default:
      //cout << "Packet " << dec << int(pkt) << "\n";
      // unused at the moment...
      //vbi_send(vbi, EV_XPACKET, mag8, pkt, err, p);
      return 0;

  }
  return 0;
}

//----------------------------------------------------------------------------

/* Low level decoder of raw VBI data 
   It calls the higher level decoders as needed 
*/

struct VBIdecoder {
  u8 *vbibuf;
  u8 *lbuf;
  int flags;
#define VBI_VT  1
#define VBI_VPS 2
#define VBI_VC  4
#define VBI_VD  8
  int lpf;   // lines per field
  int field;  
  int line;
  int bpl;   // bytes per line
  int bpf;   // bytes per frame 
  u8 vcbuf[20];
  u8 vc2buf[20];
  u8 vdat[10];
  u8 off, thresh;
  uint spos;
  vpsinfo vpsi;
  ICdeco icd;
  VDdeco vdd;
  VTchannel vtch;
  double freq;
  unsigned int vtstep, vcstep, vpsstep, vdatstep;
/* use fixpoint arithmetic for scanning steps */
#define FPSHIFT 16
#define FPFAC (1<<FPSHIFT)
  int norm;
  ofstream vdout;
  struct vbi rbvbi;
  struct vbi *vbi;

  struct vbi *
  getvbi(void)
  {
    return vbi;
  }

  int decode_line(struct vbi *vbi, u8 *data, int line);

  void decode_frame(struct vbi *vbi, u8 *data) {  // alle Zeilen einer Seite dekodieren
    vbibuf=data;
    for (line=0; line<lpf*2; line++) {
      lbuf=line*bpl+vbibuf;
      decode_line(vbi,lbuf, line);
    }
  }

  void setfreq(double f, int n=0) {
    double vtfreq=norm ? 5.72725 : 6.9375;            // 144ns
    double vpsfreq=5.0; // does NTSC have VPS???
    double vdatfreq=2.0;                              // 500ns
    double vcfreq=0.77;
    
    norm=n;
    /* if no frequency given, use standard ones for Bt848 and PAL/NTSC */
    if (f==0.0) 
      freq=norm ? 28.636363 : 35.468950;       // PAL: 35.4MHz, 28.2ns
    else
      freq=f;

    vtstep=int((freq/vtfreq)*FPFAC+0.5);       // 335062
    /* VPS is shift encoded, so just sample first "state" */
    vpsstep =2*int((freq/vpsfreq)*FPFAC+0.5); 
    vdatstep=int((freq/vdatfreq)*FPFAC+0.5); 
    vcstep  =int((freq/vcfreq)*FPFAC+0.5);
  }

  void init(int lines, int bytes, int flag, int n=0, double f=0.0) {
    vbi = &rbvbi;
    flags=flag;
    lpf=lines;
    bpl=bytes;
    freq=f;
    bpf=lpf*bpl*2;
    setfreq(f,n);
    vbi->seq = 0;
    vbi->bufsize = VBI_BPF;
    out_of_sync(vbi);
    vbi->ppage = vbi->page;
    vbi_pll_reset(vbi, 0);
    local_init();                         // char set
  }

  VBIdecoder(int lines, int bytes, int flag=VBI_VT, int n=0, double f=0.0) {
    init(lines,bytes,flag,n,f);
    if (flags&VBI_VD)
      vdout.open("vdat.cap");
  }; 

  ~VBIdecoder() {
    //
    delete [] vbibuf;
  }

  /* primitive automatic gain control to determine the right slicing offset
     XXX: it should suffice to do this once per channel change
     XXX: handle channel changes :-)
   */
  void AGC(int start, int stop, int step) {
    int i, min=255, max=0;
    
    for (i=start; i<stop; i+=step) {
      if (lbuf[i]<min) 
	min=lbuf[i];
      if (lbuf[i]>max) 
	max=lbuf[i];
    }
    thresh=(max+min)/2;
    off=128-thresh;
#ifdef DEBUGAGC
    cout << HEX(2) << (int)min << " "    << HEX(2) << (int)max << endl;
#endif
  }

  
  inline u8 scan(unsigned int step) {     // ein Byte, LSB first
    int j;
    u8 dat;
    for (j=7, dat=0; j>=0; j--, spos+=step)
#ifdef TEST3PIX                      // test at least two out of three pixel
      if ((lbuf[(spos>>FPSHIFT)-1]+off)&0x80)
        dat|=(((lbuf[spos>>FPSHIFT]+off)&0x80)|((lbuf[(spos>>FPSHIFT)+1]+off)&0x80))>>j;
      else
        dat|=(((lbuf[spos>>FPSHIFT]+off)&0x80)&((lbuf[(spos>>FPSHIFT)+1]+off)&0x80))>>j;
#else
      dat|=((lbuf[spos>>FPSHIFT]+off)&0x80)>>j;
#endif
    return dat;
  }


  // read one serial byte (10 bits)  
  inline int scan10(unsigned int step, u8 &dat) { 
    int j;
  
    // check for start bit, spos: middle of start bit
    if (!((lbuf[spos>>FPSHIFT]+off)&0x80)) { // must be 1
      LOG(32,cerr << "S";);
      spos+=10*step;
      return -1;
    }
    spos+=step;                             // 1st data bit

    for (j=7, dat=0; j>=0; j--, spos+=step)  // bit 7..0
      dat|=((lbuf[spos>>FPSHIFT]+off)&0x80)>>j;

    /* check for stop bit */
    if (((lbuf[spos>>FPSHIFT]+off)&0x80))  // must be 0
      LOG(32,cerr << "s";);
    spos+=step;                            // middle of next start bit
    dat^=0xff;
    return 0;
  }

  
  inline u8 vtscan(void) {
    return scan(vtstep);
  }

  
  inline u8 vpsscan(void) {
    return scan(vpsstep);
  }

  
  inline u8 vdatscan(u8 &dat) {
    return scan10(vdatstep,dat);            // get 1 serial byte
    // horizontal whrend 96.3% des sichtbaren Bildes
    // 3.7% schwarz, dann 10 bytes (100 bits) Daten bis rechte Bildkante
  }

  
  u8 vcscan(void) {
    int j;
    u8 dat;

    for (dat=0,j=7; j>=0; j--, spos+=vcstep) 
      dat|=((lbuf[spos>>FPSHIFT]+off)&0x80)>>j;
    return dat;
  }
};


//----------------------------------------------------------------------------

int VBIdecoder::decode_line(struct vbi *vbi, u8 *p, int line)
//static int vbi_line(struct vbi *vbi, u8 *p)
{
  u8 data[43], min, max;
  int dt[256], hi[6], lo[6];
  int i, n, sync, thr;
  int aline=line % lpf;

/* only videotext data format and PAL for now... */

// process one raw vbi line    2048 bytes   timing from alevt
  if (flags&VBI_VT && aline<16) {

    /* remove DC. edge-detector */
    for (i = 40; i < 240; ++i)
      dt[i] = p[i+STEP/FAC] - p[i];	    // amplifies the edges best.

    /* set barrier */
    for (i = 240; i < 256; i += 2)
      dt[i] = 100, dt[i+1] = -100;

    /* find 6 rising and falling edges */
    for (i = 40, n = 0; n < 6; ++n)
    {
      while (dt[i] < 32)
        i++;
      hi[n] = i;
      while (dt[i] > -32)
        i++;
      lo[n] = i;
    }
    if (i >= 240)
      return -1;	                    // not enough periods found

    i = hi[5] - hi[1];	   // length of 4 periods (8 bits), normally 40.85
    if (i < 39 || i > 42)  // ...variabel: 8*freq/vtfreq +/-1
      return -1;	                    // bad frequency

    /* AGC and sync-reference */
    min = 255, max = 0, sync = 0;
    for (i = hi[4]; i < hi[5]; ++i)
      if (p[i] > max)
        max = p[i], sync = i;
    for (i = lo[4]; i < lo[5]; ++i)
      if (p[i] < min)
        min = p[i];
    thr = (min + max) / 2;

    p += sync;

    /* search start-byte 11100100 */
    for (i = 4*STEP + vbi->pll_adj*STEP/10; i < 16*STEP; i += STEP)
      if (p[i/FAC] > thr && p[(i+STEP)/FAC] > thr) // two ones is enough...
      {
        /* got it... */
        memset(data, 0, sizeof(data));

        for (n = 0; n < 43*8; ++n, i += STEP)
          if (p[i/FAC] > thr)
       	    data[n/8] |= 1 << (n%8);

        if (data[0] != 0x27)          // really 11100100? (rev order!)
          return -1;

        if (i = vt_line(vbi, data+1))
          if (i < 0)
            pll_add(vbi, 2, -i);
          else
            pll_add(vbi, 1, i);
        return 0;
      }
    return -1;
  }


}
  
//----------------------------------------------------------------------------

#ifndef RBTIMING

// old version
void VBIdecoder::decode_line(u8 *d, int line) 
{
  u8 data[45];
  int i,j,p;
  int rbeg,rpos,rneg,rmax;
  u8 c;
  int aline=line % lpf;
  lbuf=d;                   // 2048 bytes per line
  
  /* all kinds of data with videotext data format: videotext, intercast, ... */

#ifdef DEBUGAGC
  cout << DEC(2) << (int)line << "  ";    // show line nr for AGC values
#endif
  AGC(120,450,1);                 // Schwelle festlegen auf PixelBasis

  // ...
  
  /* VPS information with channel name, time, VCR programming info, etc. */
  if (flags&VBI_VPS && line==9) {
    uint p;
    int i;
    u8 off=128-thresh;
    p=150;                                 /* 150 .. 260*/
    while ((lbuf[p]<thresh)&&(p<260))
      p++;
    p+=2;
    spos=p<<FPSHIFT;
    if ((data[0]=vpsscan())!=0xff)        /* 0xff 0x5d */
      return 0;
    if ((data[1]=vpsscan())!=0x5d)
      return 0;
    for (i=2; i<16; i++)                  /* 2 .. 15, 14 bytes VPS */
      data[i]=vpsscan();
    vpsi.decode(data);
  }

  
  // Videodat
  if ( (flags&VBI_VD) && (aline==17 || aline==18)) {
    uint p;
    int i;
    
    //AGC(200,450,1);
    p=150;
    thresh=128;
    off=128-thresh;
    while ((lbuf[p]<100)&&(p<200))       /* 150 .. 200 */
      p++;                               // search start bit of 1st byte
    if (p<200) {                         // found
      spos=p<<FPSHIFT;
      spos+=vdatstep/2;                  // middle of start bit
      for (i=0; i<10; i++)               // 10 bytes / line
	if (vdatscan(vdat[i]))           // get 1 byte
	  break;
      vdd.write(vdat,i);
      vdout.write(vdat,i);
      //hdump10(vdat,i);
      LOG(32,cerr << ".";);
    } else
//      cout << "p: " << p << endl;
      LOG(32,cerr << "X";);
  }



  /* Videocrypt 1/2 data which includes data like on screen messages (usually
     the channel name), the field number (so that the decoder decrypts the
     right fields) and the datagram which is sent to the smartcard 
     */
  if (flags&VBI_VC) {
    /* Videocrypt stuff removed due to uncertain legal situation */
  }
}
#endif RBTIMING

//----------------------------------------------------------------------------

void usage(void)
{
  cout << "vbidecode -? -h -vc -vd -vp -vx devicename\n\n";
  cout << "-? -h  this help screen\n";
  cout << "-vc    decode Videocrypt data\n";
  cout << "-vd    extract VideoDat stream to vdat.cap\n";
  cout << "-vp    decode VPS information\n";
  cout << "-vx    x=integer, select verbosity level\n";
}


//----------------------------------------------------------------------------


main(int argc,char **argv)
{

  char *c;
  int i, n, norm=0, flags=VBI_VT;
  static u8 data[VBI_BPF];                  // one frame of raw vbi data
  int seq, fh;
  struct vbi *vbi;
  
  char *devname=0;

  while (--argc) {
    c=(*++argv);
    switch (c[0]) {
    case '-' :
      switch (c[1]) {
      case '?':
      case 'h':
	usage();
	exit(0);
      case 'v':
	switch(c[2]) {
	case 'c':
	  flags|=VBI_VC;
	  break;
	case 'd':
	  flags|=VBI_VD;
	  break;
	case 'p':
	  flags|=VBI_VPS;
	  break;
	case '\0':
	  verbosity++;
	  break;
	case '1' ... '9':
	  if (c[2])
	    verbosity=atoi(c+2);
	  break;
	default:
	  cerr << "-v" << c[2] << " ???" << "\n"; 
	  break;
	}
	break;
      case 'n':
	norm=1;
	break;
      default:
	cerr << "vbidecode : Unknown switch: " << c << "\n"; 
	break;
      }
      break;
    default:
      devname=c; break;
    }
  }

  if (!devname)
    devname="/dev/vbi0";
  //ifstream fin(devname);
  if ((fh = open(devname, O_RDONLY)) == -1) {
  //if (!fin) {
    cerr << "Could not open VBI device\n";
    exit(1);
  }

  cout << "vbidecode 1.1.2 (C) 1997,98 Ralph Metzler (rjkm@thp.uni-koeln.de)\n";
  cout << "-- with improvements from Rolf Bleher <dk7in@qsl.net> 19.3.99 --\n";
  
  getcnames();                              // read in channel names
  VBIdecoder vbid(VBI_LINENUM,BPL,flags,norm);
  vbi = vbid.getvbi();                      // get pointer
  vbi->fd = fh;
  

  /* give vbidecode higher priority,
     this seems to decrease the number of decoding errors when other programs
     are running at the same time
     (it might cause trouble on very slow computers!)
     */

// increasing the priority crashes my computer (K6-2/300) really fast...
// without it works 24h a day...      -rb-
// there now may be some decoding errors but no crashing computer...
/*
  sched_param p;
  p.sched_priority=1;
  sched_setscheduler(0,SCHED_FIFO,&p);
*/

//  while (fin) {
  while (1) {
    n = read(vbi->fd, data, vbi->bufsize);
//    if (n != vbi->bufsize) {                  // wrong byte count ?
//      cout << "wrong count\n";                // gab es noch nicht
//    } 
//    else {

//      fin.read(data, VBI_BPF);             // read raw vbi data
      seq = *(int *)(data+VBI_BPF - 4);
      if (++vbi->seq != seq) {                // frames out of sync
        vbi->seq = seq;
        out_of_sync(vbi);
        n = read(vbi->fd, data, vbi->bufsize);
        seq = *(int *)(data+VBI_BPF - 4);
        if (++vbi->seq != seq) {                // frames out of sync
          // sometimes it does not resync automatically!
          // so we have to help
          // I think the problem lies in bttv
          // this is just trial and error, but seems to work
          close(fh);
          if ((vbi->fd = open(devname, O_RDONLY)) == -1) {
            cerr << "Could not reopen VBI device\n";
            exit(1);
          }
          LOG(1,cout << "reopen VBI\n";)
          n = read(vbi->fd, data, vbi->bufsize);
          seq = *(int *)(data+VBI_BPF - 4);
        }
        // n = read(vbi->fd, data, 1234);        // ??
      }
      else {
        vbid.decode_frame(vbi,data);
        LOG(16,cout << "Frame: " << HEX(8) << *(int *)(data+VBI_BPF-4) << endl;)
//        cout << HEX(8) << *(int *)(data+VBI_BPF-4) << endl;
      }
      vbi->seq = seq;
//    }
  }
}
