/***************************************************************

  DVMS            Digital Voice Mailbox System
  VTGEN
  SERV

  --------------------------------------------------
  non-preemptive multitasking functions
  --------------------------------------------------


  Copyright (c)92-95 Detlef Fliegl DG9MHZ
          Guardinistr. 47
          D-81375 Muenchen

  Alle Rechte vorbehalten / All Rights reserved

 ***************************************************************/


#ifdef DVMS
#include "dvms.h"
#include "dvmsfwd.h"
#endif
#ifdef VTGEN
#include "vtgen.h"
#endif

#include "tnclow.h"
#include "util.h"

long global_msg;

static tcb_t *tcb[_MAXTASKS+1],dummy;
static tcb_t *currtask=&dummy;
static tcb_t *next;
static int tasksrun=0,i;
static long syst=0L,sysbeg;
static char *kernel="Kernel",*dumpstr="segmentation fault task#%d %s IP%x:%x";
static jmp_buf local;

void scheduler(void)
{
  if(!tasksrun || (currtask==NULL)||(currtask==&dummy))
    return;

#ifdef TDEBUG
  if(currtask->state&_TSUSPEND)
    screen[currtask->pid*2]=' ';

  if(currtask->state&254)
    screen[currtask->pid*2]='E';
#endif

  if(((timer()-currtask->lastcall)<currtask->prio)&&!currtask->state)
    return;

  if(setjmp(currtask->jb))
    return;

  sysbeg=timer();
  currtask->cputime+=(timer()-currtask->lastcall);

#ifdef TDEBUG
  if(!currtask->state)
    screen[currtask->pid*2]='W';
#endif

  disable();
  currtask=currtask->next;
  if(currtask==NULL)
    currtask=tcb[get_firsttask()];
  enable();

#ifdef TDEBUG
  if(!currtask->state)
    screen[currtask->pid*2]='R';//currtask+0x30;//
#endif

  syst+=(timer()-sysbeg);
  currtask->lastcall=timer();

  #ifdef __MSDOS__
  if(farheapcheck()<0)
  {
    write_log(0,"heapcheck",dumpstr,currtask,currtask->name,currtask->jb->j_ss,currtask->jb->j_sp );
    longjmp(local,1);
  }

  if(currtask->jb->j_cs<0x1000)
  {
    write_log(0,"reschedule",dumpstr,currtask,currtask->name,currtask->jb->j_ss,currtask->jb->j_sp );
    longjmp(local,1);
  }
  else
  #endif
    longjmp(currtask->jb,1);

  return;
}


void suspend(unsigned long t)
{ unsigned long ti=timer();

  currtask->state|=_TSUSPEND;

  do{ scheduler();
    }while(((timer()-ti)<t) && tasksrun);

  currtask->state&=(0xffffffff-_TSUSPEND);
#ifdef TDEBUG
  screen[currtask->pid*2]='R';
#endif
}

int waitfor(long what)
{  int ret=1;

  currtask->state|=what;
  while((currtask->state&what))
  {
    if(what&_KBHIT)
    { if(global_msg&_SHUTDOWN)
      {  ret=-1;
        break;
      }
      if(kbhit())
      {  ret=getch();
        break;
      }
    }
    if(what&_TNC_TXFREE)
    {  if((tget_lstate()<info_transfer))
      {  ret=-1;
        break;
      }
      if(tnc_putfree())
        break;
    }
    if(what&_TNC_RXVAL)
    {  if((tget_lstate()<info_transfer)||(global_msg&_SHUTDOWN))
      {  ret=-1;
        break;
      }
      if((ret=tnc_get())!=-1)
        break;
    }
    if(what&_MSG)
      if(currtask->msgwrcnt)
        break;
    scheduler();
  }

  currtask->state&=(0xffffffff-what);
  return ret;
}

static void free_rsc(tcb_t *t);

void killtask(void)
{
  disable();
  _SS=local->j_ss;
  _SP=local->j_sp;
  enable();

  //set_win(defaultwin);
  //cprintf("Kill called %d\r\n",currtask->pid);

#ifdef TDEBUG
  screen[currtask->pid*2]='.';
#endif

  tasksrun--;

  if(!tasksrun||(currtask==NULL)||(currtask==&dummy))
    longjmp(local,1);

  disable();
  if(currtask->prev!=NULL)
    currtask->prev->next=currtask->next;

  if(currtask->next!=NULL)
  {
    currtask->next->prev=currtask->prev;
    next=currtask->next;
  }
  else
  {
    next=tcb[get_firsttask()];

    if(next==currtask)
      next=next->next;
  }

  tcb[currtask->pid]=NULL;
  free_rsc(currtask);
  currtask=next;
  enable();
  #ifdef __MSDOS__
  if(currtask->jb->j_cs<0x1000)
  {
    write_log(0,"kill()",dumpstr,currtask,currtask->name,currtask->jb->j_ss,currtask->jb->j_sp );
    longjmp(local,1);
  }
  else
  #endif
    longjmp(currtask->jb,1);
}

int kill (int task)
{
  if(tcb[task]->type!=-1)
  { tcb_t *next;

#ifdef TDEBUG
    screen[task*2]='K';
#endif
    tasksrun--;

    if(!tasksrun|| (currtask==NULL)||(currtask==&dummy))
      longjmp(local,1);

    disable();

    if((next=tcb[task]->next)!=NULL)
      tcb[task]->next->prev=tcb[task]->prev;
    else
    {
      next=tcb[get_firsttask()];
      if(next==tcb[task])
        next=next->next;
    }

    if(tcb[task]->prev!=NULL)
      tcb[task]->prev->next=tcb[task]->next;

    i=currtask->pid;
    free_rsc(tcb[task]);
    tcb[task]=NULL;
    enable();
    if(task==i)
    {
      currtask=next;
      #ifdef __MSDOS__
      if(currtask->jb->j_cs<0x1000)
      {
        write_log(0,"kill(task)",dumpstr,currtask,currtask->name,currtask->jb->j_ss,currtask->jb->j_sp );
        longjmp(local,1);
      }
      else
      #endif
        longjmp(currtask->jb,1);
    }
    return 1;
  }
  return 0;
}

int get_firsttask()
{
  int i;
  char *r="get_firsttask";
  tcb_t *t;

  if(tasksrun)
  for(i=0;i<_MAXTASKS;i++)
  { t=tcb[i];
    if((t!=NULL)&&(t->prev==NULL))
      return i;
  }

  write_log(_serious,r,"return -1");
  return -1;
}

static int get_lasttask()
{
  int i;
  char *r="get_lasttask";
  tcb_t *t;
  if(tasksrun)
  for(i=0;i<_MAXTASKS;i++)
  { t=tcb[i];
    if((t!=NULL)&&(t->next==NULL))
      return i;
  }
  write_log(_serious,r,"return -1");
  return -1;
}
/*
static void trace(char *s)
{ set_win(defaultwin);
  cprintf("%s\r\n",s);
}
*/
int fork(char *name,void *task,unsigned long prio,unsigned long type,int stream,unsigned stacksize)
{
  char *r="fork";
  unsigned *stack;
  tcb_t *l_tcb;

  char str[30];
  int pid=currtask->pid;
  strcpy(str,currtask->name);


  for(i=0;i<_MAXTASKS;i++)
    if(tcb[i]==NULL)
      break;

  if(i==_MAXTASKS)
  {
    write_log(_serious,r,"no more tcb entries");
    return -1;
  }

  strcpy(currtask->name,name);
  currtask->pid=i;

  l_tcb=(tcb_t *)tmalloc(sizeof(tcb_t));
  if(l_tcb==NULL)
  {
    write_log(_serious,r,"no more memory for tcb");
    strcpy(currtask->name,str);
    currtask->pid=pid;
    return -1;
  }

  memset(l_tcb,0,sizeof(tcb_t));

  l_tcb->stack=(unsigned *)tmalloc(stacksize+20);

  if(l_tcb->stack==NULL)
  {
    write_log(_serious,r,"no more memory for taskstack");
    strcpy(currtask->name,str);
    currtask->pid=pid;
    tfree(l_tcb);
    return -1;
  }

  memset(l_tcb->stack,0x55,stacksize+20);

  stack=(unsigned *)MK_FP(FP_SEG(l_tcb->stack)+1,0);

  stack[(stacksize/2)-1]=FP_SEG(killtask);
  stack[(stacksize/2)-2]=FP_OFF(killtask);

  setjmp(l_tcb->jb);
  #ifdef __MSDOS__
  l_tcb->jb->j_ip=FP_OFF(task);
  l_tcb->jb->j_cs=FP_SEG(task);
  l_tcb->jb->j_ss=FP_SEG(stack);
  l_tcb->jb->j_sp=stacksize-4;
  #else
  #error "scheduler needs to be updated"
  #endif
  memccpy(l_tcb->name,name,0,30);
  l_tcb->lastcall=timer();
  l_tcb->lastinput=-1;
  l_tcb->lastoutput=-1;
  l_tcb->prio=prio;
  l_tcb->type=type;
  l_tcb->tty=stream;
  l_tcb->stacksize=stacksize;
  l_tcb->create=time(NULL);
  l_tcb->pid=i;

  if(tasksrun)
    l_tcb->parentid=currtask->pid;
  else
    l_tcb->parentid=-1;

  //falls schon andere Prozesse laufen
  if(tasksrun)
  //nach letztem tcb suchen
  //und einklinken
  { i=get_lasttask();
    l_tcb->prev = tcb[i];
    tcb[i]->next=l_tcb;
  }
  //tcb adresse zuweisen;
  tcb[l_tcb->pid]=l_tcb;

  tasksrun++;
#ifdef TDEBUG
  screen[l_tcb->pid*2]='N';
#endif
  strcpy(currtask->name,str);
  currtask->pid=pid;
  return l_tcb->pid;
}

void init_rsc(void);

int initsched(void)
{ init_rsc();
  //tcb pointer komplett lschen
  memset(tcb,0,_MAXTASKS*sizeof(tcb_t*));
  //dummy proze erzeugen
  memset(&dummy,0,sizeof(tcb_t));
  memccpy(dummy.name,"Kernel",0,10);
  dummy.pid=-1;
  currtask=&dummy;
  return 1;
}

static int idlehist[]={100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100},idlecount=0;

int getidle(int mode)
{
  static long lastcall=0L;
  static int idle=100;
  int momidle;

  tcb_t *next=tcb[get_firsttask()];

  if(!tasksrun|| (currtask==NULL)||(currtask==&dummy))
    return idle;

  if(!mode && ((ad_time(NULL)-lastcall)<5))
   return idle;

  idle=0;

  do
    if(next->state)
      idle++;
  while((next=next->next)!=NULL);

  momidle=(100*idle)/tasksrun;

  if(((ad_time(NULL)-lastcall)<5))
    return momidle;

  idlehist[idlecount]=momidle;
  idlecount++;
  idlecount%=16;

  idle=0;

  for(i=0;i<16;i++)
    idle+=idlehist[i];

  idle>>=4;

  lastcall=ad_time(NULL);

  if(mode)
    return momidle;
  else
    return idle;
}

static void show_idle(void)
{ int i;
  tcb_t *next=get_tcb(get_firsttask());
  long totalcpu=0;

  putf("CPU load in 5sec-intervals:\rSample:  now -5s -10s ...\rRunning:");
  for(i=1;i<17;i++)
    putf("% 4d",100-idlehist[abs((idlecount-i)%16)]);
  putf("\rIdle:   ");
  for(i=1;i<17;i++)
    putf("% 4d",idlehist[abs((idlecount-i)%16)]);
  do
    totalcpu+=next->cputime;
  while((next=next->next)!=NULL);

  putf("\rTotal CPU time: %s ",zeitspanne(totalcpu+syst,2));
  putf("(Task %s + ",zeitspanne(totalcpu,2));
  putf("%s %s)\r",kernel,zeitspanne(syst,2));
}

void startsched(void)
{
  if(setjmp(local))
  {
    free_rsc(currtask);
    tfree(tcb);
    return;
  }

  currtask=tcb[get_firsttask()];
  currtask->lastcall=timer();
  longjmp(currtask->jb,1);
}

void taskend(void)
{
  longjmp(local,1);
}

long get_global_msg(void)
{
  return global_msg;
}

void put_global_msg(long msg)
{
  char *buffer="ALL \x07 *** system shutdown ***"
       #ifndef DVMS
       ,
       *ani="......................."
       #endif
       ;
  if(msg==_SHUTDOWN)
  {
    put_msg("talkd",(long)buffer);
     #ifndef DVMS
     int i;
     for(i=0;ani[i];i++)
     {
       set_win(defaultwin);
       putch(ani[i]);
       suspend(1);
     }
     #endif
  }
  global_msg=msg;
}

int put_msg(char *name,long msg)
{ return put_msg(get_tasknum(name),msg);
}

int put_msg(int task,long msg)
{ if((task>=_MAXTASKS) || (task <0) || (tcb[task]->type==-1) )
    return -1;

  if(tcb[task]->msgwrcnt==_MAXMSG)
    return -1;
  tcb_t *t=tcb[task];
  t->msg[t->msgwrcnt].msg=msg;
  t->msg[t->msgwrcnt].pid=currtask->pid;
  return t->msgwrcnt++;
}

msg_t *get_msg(void)
{ int c=currtask->msgrdcnt;

  if(!currtask->msgrdcnt && !currtask->msgwrcnt)
    return NULL;

  currtask->msgrdcnt++;

  if(currtask->msgrdcnt==currtask->msgwrcnt)
  { currtask->msgrdcnt=0;
    currtask->msgwrcnt=0;
  }

  return &currtask->msg[c];
}

int get_tasknum(void)
{ return currtask->pid;
}

int get_tasknum(char *name)
{ tcb_t *next=tcb[get_firsttask()];
  if(!tasksrun || (currtask==NULL)||(currtask==&dummy))
    return -1;
  do
    if(!strncmp(next->name,name,strlen(name)))
      return next->pid;
  while((next=next->next)!=NULL);
  return -1;
}


tcb_t *get_tcb(int task)
{ if((task==-1) || (task>_MAXTASKS))
    return &dummy;
  return tcb[task];
}

tcb_t *get_tcb(char *name)
{ int i=get_tasknum(name);

  if(i==-1)
    return &dummy;

  return tcb[i];
}

tcb_t *get_tcb(void)
{
  /*if(!tasksrun || (currtask==NULL)||(currtask==&dummy))
    return &dummy;
  */
  return currtask;
}

int get_tasksrun(void)
{ return tasksrun;
}


time_t ad_time(time_t *t)
{ static long lastcall=-1000L;
  static time_t ti;

  if((timer()-lastcall)>9)
  {  time(&ti);
    lastcall=timer();
  }

  if(t!=NULL)
    *t=ti;
  return ti;
}

/*----------------------------------------------------------------*/
/* Resourcenverwaltung                                            */
/*----------------------------------------------------------------*/
#define RSC 0

#define _MAXFOPENRSC 40
#define _MAXOPENRSC  40
#define _MAXMEMRSC   200

static fopenrsc_t *fopenrsc[_MAXFOPENRSC+1];
static openrsc_t  *openrsc [_MAXOPENRSC+1];
static memrsc_t   *memrsc  [_MAXMEMRSC+1];

static void init_rsc(void)
{
  #ifdef RSC
  memset(fopenrsc,0,sizeof(fopenrsc_t *)*_MAXFOPENRSC);
  memset(openrsc,0,sizeof(openrsc_t *)*_MAXOPENRSC);
  memset(memrsc,0,sizeof(memrsc_t *)*_MAXMEMRSC);
  #endif
}

static memrsc_t *alloc_memrsc(void)
{ int i;
  for(i=0;i<_MAXMEMRSC;i++)
   if(memrsc[i]==NULL)
     return memrsc[i]=(memrsc_t *)malloc(sizeof(memrsc_t));
  return NULL;
}

static memrsc_t *find_memrsc(int pid)
{ int i;
  for(i=0;i<_MAXMEMRSC;i++)
   if((memrsc[i]!=NULL)&&(memrsc[i]->pid==pid))
     return memrsc[i];
  return 0;
}

static int free_memrsc(void *mem)
{ int i;
  for(i=0;i<_MAXMEMRSC;i++)
   if((memrsc[i]!=NULL)&&(memrsc[i]->mem==mem))
   { free(memrsc[i]);
     memrsc[i]=NULL;
     return 1;
   }
  return 0;
}

static openrsc_t *alloc_openrsc(void)
{ int i;
  for(i=0;i<_MAXOPENRSC;i++)
   if(openrsc[i]==NULL)
     return openrsc[i]=(openrsc_t *)malloc(sizeof(openrsc_t));
  return NULL;
}

static openrsc_t *find_openrsc(int pid)
{ int i;
  for(i=0;i<_MAXOPENRSC;i++)
   if((openrsc[i]!=NULL)&&(openrsc[i]->pid==pid))
     return openrsc[i];
  return 0;
}

static int free_openrsc(int handle)
{ int i;
  for(i=0;i<_MAXOPENRSC;i++)
   if((openrsc[i]!=NULL)&&(openrsc[i]->handle==handle))
   { free(openrsc[i]);
     openrsc[i]=NULL;
     return 1;
   }
  return 0;
}

static fopenrsc_t *alloc_fopenrsc(void)
{ int i;
  for(i=0;i<_MAXFOPENRSC;i++)
   if(fopenrsc[i]==NULL)
     return fopenrsc[i]=(fopenrsc_t *)malloc(sizeof(fopenrsc_t));
  return NULL;
}

static fopenrsc_t *find_fopenrsc(int pid)
{ int i;
  for(i=0;i<_MAXFOPENRSC;i++)
   if((fopenrsc[i]!=NULL)&&(fopenrsc[i]->pid==pid))
     return fopenrsc[i];
  return 0;
}

static int free_fopenrsc(FILE *f)
{ int i;
  for(i=0;i<_MAXFOPENRSC;i++)
   if((fopenrsc[i]!=NULL)&&(fopenrsc[i]->f==f))
   { free(fopenrsc[i]);
     fopenrsc[i]=NULL;
     return 1;
   }
  return 0;
}
#ifdef RSC
void *tmalloc(size_t bytes)
{ void *mem;
  memrsc_t *rsc;

  if((rsc=alloc_memrsc())==NULL)
    return NULL;

  memset(rsc,0,sizeof(memrsc_t));

  if((mem=malloc(bytes))==NULL)
  { free_memrsc(mem);
    return NULL;
  }

  rsc->size=bytes;
  rsc->mem=mem;
  rsc->pid=currtask->pid;
  write_log(_rsc,"malloc","%p -> %u",mem,bytes);
  //trace("malloc %p -> %u",mem,bytes");
  return mem;
}
#else
void *tmalloc(size_t bytes)
{ return malloc(bytes);
}
#endif
void tfree(void *block)
{
  #ifdef RSC
  write_log(_rsc,"free","%p",block);
  //trace("malloc %p -> %u",mem,bytes");
  free_memrsc(block);
  #endif
  free(block);
}

#ifdef RSC
FILE *tfopen(const char *filename, const char *mode)
{ FILE *f;
  fopenrsc_t *rsc;

  if((rsc=alloc_fopenrsc())==NULL)
    return NULL;

  memset(rsc,0,sizeof(fopenrsc_t));

  if((f=fopen(filename,mode))==NULL)
  { free_fopenrsc(f);
    return NULL;
  }
  //putf("Filename %s %s",filename,mode);
  memccpy(rsc->file,filename,0,_MAX_PATH);
  memccpy(rsc->mode,mode,0,9);
  rsc->f=f;
  rsc->pid=currtask->pid;
  write_log(_rsc,"fopen","%s %s -> %p",filename,mode,f);
  return f;
}
#else
FILE *tfopen(const char *filename, const char *mode)
{ return fopen(filename,mode);
}
#endif

int tfclose(FILE *stream)
{
  #ifdef RSC
  write_log(_rsc,"fclose","%p",stream);
  free_fopenrsc(stream);
  #endif
  return fclose(stream);
}
#ifdef RSC
int topen(const char *path, int access, unsigned mode )
{
  int handle;
  openrsc_t *rsc;

  if((rsc=alloc_openrsc())==NULL)
    return NULL;

  memset(rsc,0,sizeof(openrsc_t));

  if((handle=open(path,access,mode))<1)
  { free_openrsc(0);
    return handle;
  }

  memccpy(rsc->file,path,0,_MAX_PATH);
  rsc->access=access;
  rsc->mode=mode;
  rsc->handle=handle;
  rsc->pid=currtask->pid;

  write_log(_rsc,"open","%s %d %u -> %d",path,access,mode,handle);

  return handle;
}
#else
int topen(const char *path, int access, unsigned mode )
{  return open(path,access,mode);
}
#endif

int topen(const char *path, int access)
{
  return topen(path,access,0);
}

int tclose(int handle)
{
  #ifdef RSC
  write_log(_rsc,"close","%d",handle);
  free_openrsc(handle);
  #endif
  return close(handle);
}

static void free_rsc(tcb_t *t)
{
  #ifdef RSC
  memrsc_t *m;
  openrsc_t *o;
  fopenrsc_t *f;

  if(t->type&_TNCPROC)
    tdiscport(t->tty,0);

  while((o=find_openrsc(t->pid))!=NULL)
    tclose(o->handle);
  while((f=find_fopenrsc(t->pid))!=NULL)
    tfclose(f->f);
  while((m=find_memrsc(t->pid))!=NULL)
    tfree(m->mem);
  #endif
}

void rscstat(char *s)
{ int i;
  memrsc_t *m;
  fopenrsc_t *f;
  openrsc_t *o;
  long count=0L;

  switch(s[0]&0x5f)
  { case 'M':
    putf("Memory  Size     Pid  Owner\r");
    for(i=0;i<_MAXMEMRSC;i++)
      if((m=memrsc[i])!=NULL)
      { count+=m->size;
        putf("%04lX  % 6u  % 6u  ",(unsigned long)(m->mem)>>16L,m->size,m->pid);
        if(m->pid!=0xffff)
          putf("%-10.10s\r",tcb[m->pid]->name);
        else
          putf("%s\r",kernel);
      }
    putf("Total: %ld KByte\r",count>>10L);
    break;
    case 'C':
    show_idle();
    break;
    #ifdef DVMS
    case 'F':
    fwdstat();
    break;
    #endif
    case 'O':
    putf("\rFile                            Access  Mode     Pid  Owner \r");
     for(i=0;i<_MAXFOPENRSC;i++)
      if((f=fopenrsc[i])!=NULL)
      {
         putf("%-30.30s  %-5.5s   %-5.5s % 6u  ",f->file,f->mode,"shared",f->pid);
        if(f->pid!=0xffff)
          putf("%-10.10s\r",tcb[f->pid]->name);
        else
          putf("%s\r",kernel);

      }
    for(i=0;i<_MAXFOPENRSC;i++)
      if((o=openrsc[i])!=NULL)
      { putf("%-30.30s  %04X    %04X  % 6u  ",o->file,o->access,o->mode,o->pid);

        if(o->pid!=0xffff)
          putf("%-10.10s\r",tcb[o->pid]->name);
        else
          putf("%s\r",kernel);

      }

    break;
    #ifdef DVMS
    case 'P': putf("PTT is %s\r",ptt_state?"on":"off");break;
    #endif
    case 'T':
    tcb_t *next=get_tcb(get_firsttask());
    putf("  Pid  Stack  left  Name\r");
    do
    {
      putf("%c%  4X  ",next==get_tcb()?'*':' ',next->pid);
      //putf("% 4lX  % 5u  ",(unsigned long)next->stack>>16L,next->jb->j_sp);
      int left=0;
      for(i=0;i<next->stacksize;i++,left++)
        if(((char*)next->stack)[i]!=0x55)
           break;

      putf("% 4lX  % 5u  ",(unsigned long)next->stack>>16L,left);

      putf("%s \r",next->name);
    }
    while((next=next->next)!=NULL);
    break;

    default: putf("syntax: STATUS cpu|(forward)|memory|openfiles|(ptt)|(semaphores)|taskstack\r");

  }
}

void show_ps(char *s)
{
  tcb_t *next=get_tcb(get_firsttask());
  char *buffer,*stnc="",*styp="",*sout="",help[20];
  int all=!!strchr(s+1,'b');
  int typ=!!strchr(s+1,'a');
  int tnc=!!strchr(s+1,'t');
  //int stk=!!strchr(s+1,'s');
  if(strchr(s+1,'?'))
  {
    putf("syntax: PS -abt\r");
    return;
  }

  if(typ)
  {
   styp="State Type ";
   sout="Output ";
  }
  if(tnc)
   stnc="  TTY";

  putf("   Pid %s %s CPUt Create Input %sName       Command\r",stnc,styp,sout);
  do
  {
    if(all||(next->type&_TNCPROC)||(next->type&_CONSOLE))
    {
      putf("%c% 5u  ",get_tcb()==next?'*':' ',next->pid);
      if(tnc)
        putf("% 4d ",next->tty);

      if(typ)
      {
      switch(next->state)
      {
        case 0:
          if(get_tcb()==next)
            buffer="run";
          else
            buffer="slice";
          break;
        case _KBHIT|_TSUSPEND|_MSG:
        case _KBHIT|_TSUSPEND:
            buffer="kbhit";
          break;
        case _TNC_TXFREE|_TSUSPEND:
            buffer="txf";
          break;
        case _TNC_RXVAL|_TSUSPEND:
            buffer="rxval";
          break;
        case _TNC_RXVAL|_TSUSPEND|_MSG:
            buffer="rxmsg";
          break;
        case _TSUSPEND:
            buffer="idle";
          break;
        default:
            sprintf(help,"% 5lX",next->state);
            buffer=help;
      }
      putf("%-5s ",buffer);
      switch(next->type&0x7)
      {
        case _NPROC:
            buffer="sys";
            break;
        case _TNCPROC:
            buffer="user";
            break;
        case _TNCPROC|_FWDPROC:
            buffer="fwd";
            break;
        default:
            sprintf(help,"% 5lX",next->type);
            buffer=help;
      }
      putf("%-5s",buffer);
      }
      putf("% 5s",zeitspanne(next->cputime,2));
      putf("% 6s",zeitspanne((ad_time(NULL)-next->create),1));
      putf("% 6s",zeitspanne((ad_time(NULL)-next->lastinput),1));

      if(typ)
        putf(" % 6s",zeitspanne((ad_time(NULL)-next->lastoutput),1));

      putf("  %-10.10s ",next->name);

      if(typ)
        putf("%-18.18s\r",next->cmd);
      else
        putf("%-24.24s\r",next->cmd);
    }
  }
  while((next=next->next)!=NULL);
}

