/***************************************************************

  DVMS            Digital Voice Mailbox System

  ----------------------------------------------------
  sound blaster 16 and DTMF signal processing routines
  ----------------------------------------------------


  Copyright (c)92-99 Detlef Fliegl DG9MHZ
		     Guardinistr. 47
		     D-81375 Muenchen

  Alle Rechte vorbehalten / All Rights reserved

  This sub module was developed by Steffen Koehler DH1DM
				   Brentanostr. 14
				   D-01157 Dresden

  Acknoledgements

  The DTMF parts of this source code were inspired by the
  work of Thomas Sailer HB9JNX (see his MULTIMON program).

 ***************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <math.h>
#include "sbdefs.h"
#include "costab.h"
#include "dvmssb.h"
#include "dvms.h"
#include "prozca.h"
#include "dvmsirq.h"

// Structure for storing data associated with the SB card.
typedef struct {
  int base;                           // SB card base I/O address
  int irq;                            // IRQ number
  int dma8;                           // 8-bit DMA channel
  int dma16;                          // 16-bit DMA channel
  int intrnum;                        // mapped interrupt number
  void interrupt (*intvecsave)(...);  // previous interrupt vector
  char init_complete;                 // SB setup is now complete
  char action;                        // current I/O action
  int samp_rate;                      // current sample rate
  int new_rate;                       // sample rate for next I/O action
  int dma8_cnt;                       // 8-bit DMA cycle end IRQ counter
  int dma16_cnt;                      // 16-bit DMA cycle end IRQ counter
} cardinfo;

// Structure for mixer value storage
typedef struct
{
  char InSrcL;
  char InSrcR;
  char OutSrc;
  char LineVolL;
  char LineVolR;
  char SynthVolL;
  char SynthVolR;
  char VocVolL;
  char VocVolR;
  char MasterVolL;
  char MasterVolR;
  char MicVol;
} mixerinfo;

static cardinfo sb16_card;
static mixerinfo sb16_mixer;

// Structure for retrieving DMA controller register port addresses.
typedef struct {
  unsigned int addr, count, page, mask, mode, FF;
} dmaportstruct;

// Constant array that defines port addresses for DMA controller registers.
const dmaportstruct dmaportarray[8] = {
  {0x0,  0x1,  0x87, 0xA,  0xB,  0xC},   // chan 0
  {0x2,  0x3,  0x83, 0xA,  0xB,  0xC},   // chan 1
  {0x4,  0x5,  0x81, 0xA,  0xB,  0xC},   // chan 2  DON'T USE
  {0x6,  0x7,  0x82, 0xA,  0xB,  0xC},   // chan 3
  {0x0,  0x0,  0x00, 0xD4, 0xD6, 0xD8},  // chan 4  DON'T USE
  {0xC4, 0xC6, 0x8B, 0xD4, 0xD6, 0xD8},  // chan 5
  {0xC8, 0xCA, 0x89, 0xD4, 0xD6, 0xD8},  // chan 6
  {0xCC, 0xCE, 0x8A, 0xD4, 0xD6, 0xD8}   // chan 7
};

#define BUFSIZE  1024

// DTMF frequencies
//
//      1209 1336 1477 1633
//  697   1    2    3    A
//  770   4    5    6    B
//  852   7    8    9    C
//  941   *    0    #    D

#define SRATE_32000	31250
#define PHINC_32(x)	((x)*0x10000/SRATE_32000)

unsigned int dtmf_phinc_32[] = {
  PHINC_32(1209), PHINC_32(1336), PHINC_32(1477), PHINC_32(1633),
  PHINC_32( 697), PHINC_32( 770), PHINC_32( 852), PHINC_32( 941),
  PHINC_32(1750)
};

#define SRATE_16000	15625
#define PHINC_16(x)	((x)*0x10000/SRATE_16000)

unsigned int dtmf_phinc_16[] = {
  PHINC_16(1209), PHINC_16(1336), PHINC_16(1477), PHINC_16(1633),
  PHINC_16( 697), PHINC_16( 770), PHINC_16( 852), PHINC_16( 941),
  PHINC_16(1750)
};

// dtmf input translation table
char dtmf_transl[] = {1,2,3,13,4,5,6,14,7,8,9,15,11,10,12,0};

typedef struct {
  unsigned int ph[9];
  float tenergy[18];
  float energy[9];
  int newch;
  int dtmfch;
  int dtmf_cnt;
  int call_cnt;
} dtmf_struct;

dtmf_struct dtmf_state;
int af_level;

/* sample buffer pointer variables */
int far *dma16_inbuf, *dma16_outbuf;
char far *dma8_inbuf;


int find_max_idx(float threshold, float *f, int size)
// find maximum frequency energy index for DTMF decision
{
  float en = 1;
  int idx = -1;
  int i;

  for (i = 0; i < size; i++)
    if (f[i] > en) {
      en = f[i];
      idx = i;
    }

  if (idx < 0)
    return -1;

  en *= threshold;
  for (i = 0; i < size; i++)
    if (idx != i && f[i] > en)
      return -1;

  return idx;
}


void process_block(void)
/* Calculate tone frequency energies (square magnitude
 * from real/imaginary parts), compare each spectral
 * energy against the others and trigger DTMF and
 * CALL tone (1750 Hz) events
 */
{
  float max0, max1;
  int idx0, idx1;
  int i, ch;

  for (i = 0; i < 9; i++)
    dtmf_state.energy[i] = dtmf_state.tenergy[i]   * dtmf_state.tenergy[i] +
			   dtmf_state.tenergy[i+9] * dtmf_state.tenergy[i+9];

  memset(dtmf_state.tenergy, 0, sizeof(dtmf_state.tenergy));

  /* make call tone decision */
  if (find_max_idx(0.2,dtmf_state.energy,9) == 8)
  { if (dtmf_state.call_cnt++ > 20)
      dtmf_state.dtmfch &= 0xdf;
    return;
  }
  dtmf_state.call_cnt = 0;
  dtmf_state.dtmfch |= 0x20;

  /* make dual tone decision */
  if ((idx0 = find_max_idx(0.1,dtmf_state.energy,4)) < 0)
  { dtmf_state.dtmf_cnt = 0;
    dtmf_state.dtmfch &= 0xef;
    return;
  }

  if ((idx1 = find_max_idx(0.1,dtmf_state.energy+4,4)) < 0)
  { dtmf_state.dtmf_cnt = 0;
    dtmf_state.dtmfch &= 0xef;
    return;
  }

  max0 = dtmf_state.energy[idx0];
  max1 = dtmf_state.energy[idx1+4];
  if (((max0 * 100) < max1) || ((max1 * 100) < max0))
  { dtmf_state.dtmf_cnt = 0;
    dtmf_state.dtmfch &= 0xef;
    return;
  }

  ch = dtmf_transl[(idx0 & 3) | ((idx1 << 2) & 0x0C)];
  if (ch != dtmf_state.newch)
  { dtmf_state.dtmf_cnt = 0;
    dtmf_state.dtmfch &= 0xef;
    dtmf_state.newch = ch;
    return;
  }

  if (dtmf_state.dtmf_cnt++ > 2)
    dtmf_state.dtmfch = dtmf_state.newch | 0x30;
}


void dtmf_8(signed char *buffer, int length)
/* get 8-bit input samples, perform sine/cosine table lookup
 * calculations (depending on the current sample frequency)
 */
{
  float s_in;
  float s_sum = 0;
  float s_len = length;
  unsigned int *dtmf_phinc = (sb16_card.samp_rate==SRATE_16000) ? dtmf_phinc_16 : dtmf_phinc_32;

  while(length--) {
    s_in = *buffer++;
    s_sum += fabs(s_in);
    for (int i = 0; i < 9; i++) {
      dtmf_state.tenergy[i] += costab[dtmf_state.ph[i] >> 6] * s_in;
      dtmf_state.tenergy[i+9] += sintab[dtmf_state.ph[i] >> 6] * s_in;
      dtmf_state.ph[i] += dtmf_phinc[i];
    }
  }
  process_block();
  af_level = (s_sum / (s_len * 8));
}


void dtmf_16(int *buffer, int length)
/* get 16-bit input samples, perform sine/cosine table lookup
 * calculations (depending on the current sample frequency)
 */
{
  float s_in;
  float s_sum = 0;
  float s_len = length;
  unsigned int *dtmf_phinc = (sb16_card.samp_rate==SRATE_16000) ? dtmf_phinc_16 : dtmf_phinc_32;

  while(length--) {
    s_in = *(buffer++);
    s_sum += fabs(s_in);
    for (int i = 0; i < 9; i++) {
      dtmf_state.tenergy[i] += costab[dtmf_state.ph[i] >> 6] * s_in;
      dtmf_state.tenergy[i+9] += sintab[dtmf_state.ph[i] >> 6] * s_in;
      dtmf_state.ph[i] += dtmf_phinc[i];
    }
  }
  process_block();
  af_level = (s_sum / (s_len * 2048));
}


void far *dma_alloc(unsigned int size)
/* allocate page aligned DMA memory using malloc function
 * try to avoid trouble with the runtime memory manager
 */
{ unsigned int seg;
  unsigned long addr, page_end;
  void far *mem, *oldmem = NULL;

  while((mem = malloc(size + 16)) != NULL)
  {
    if (oldmem != NULL) free(oldmem);              /* free old memory */
    oldmem = mem;

    seg = FP_SEG(mem) + (FP_OFF(mem) >> 4L) + 1;
    addr = (unsigned long)seg << 4L;          /* 20bit linear address */
    page_end = addr + 0xFFFFL;                        /* add one page */
    page_end &= 0xFFFF0000L;   /* number of bytes in the current page */
    page_end -= addr;

    if (page_end >= size)  /* Does the new block cross a 64k border ? */
      return MK_FP(seg, 0);
  }
  if (oldmem != NULL) free(oldmem);                /* free old memory */

  return NULL;
}


void dspout(unsigned int val)
// Output a byte to the Digital Sound Processor.
{
   unsigned int cnt = 65535U;
   while (inp(sb16_card.base+dspoffsetWrBuf) & 0x80)
     if (!(--cnt)) return;
   outp(sb16_card.base+dspoffsetWrBuf, val);
}


unsigned int dspin(void)
// Read a byte from the Digital Sound Processor.
{
   unsigned int cnt = 65535U;
   while (!(inp(sb16_card.base+dspoffsetDataAvail) & 0x80))
     if (!(--cnt)) return 0;
   return inp(sb16_card.base+dspoffsetReadData);
}


void SetupDMA(int dmachan, void far *ptr, unsigned int count, char dmacmd)
// This programs the DMA controller.
{
   dmaportstruct dmaports;
   unsigned short page, ofs;
   unsigned long physaddr = (unsigned long)FP_SEG(ptr) << 4;

   page = physaddr >> 16;
   ofs = physaddr & 0xFFFF;

   dmaports = dmaportarray[dmachan];

   if (dmachan > 3) {
      dmachan -= 4;     // make dmachan local to DMAC
      ofs = (ofs >> 1) + ((page & 1) << 15);
   }

   outp(dmaports.mask, 4 | dmachan);            // mask off dma channel
   outp(dmaports.FF, 0);                        // clear flip-flop
/* This next value to DMAC is different between auto-init and non-auto-init.
 * (0x58 vs. 0x48) */
   outp(dmaports.mode, dmacmd | dmachan);       // set mode for DAC
   outp(dmaports.addr, (ofs & 0xFF));           // low byte base addr
   outp(dmaports.addr, (ofs >> 8));             // high byte base addr
   outp(dmaports.page, page);                   // physical page number
   outp(dmaports.count, ((count-1) & 0xFF));    // count low byte
   outp(dmaports.count, ((count-1) >> 8));      // count high byte
   outp(dmaports.mask, dmachan);                // enable dma channel
}


void SetupInput16(void far *ptr, unsigned int count, unsigned int samp_rate)
// Program the DMA controller and the sound processor for 16-bit input.
{
   SetupDMA(sb16_card.dma16, ptr, count, DMAMODEWRITE);

   // Program the A/D sample rate.
   dspout(dspcmdADSampRate);
   dspout(samp_rate >> 8);
   dspout(samp_rate & 0xFF);

   // Program card for 16-bit signed input.
   dspout(dspSB16DMA16 | dspSB16ADC | dspSB16AI | dspSB16FifoOn);
   dspout(dspSB16ModeMono | dspSB16ModeSigned);
   dspout((count/2-1) & 0xFF);
   dspout((count/2-1) >> 8);
}


void SetupInput8(void far *ptr, unsigned int count, unsigned int samp_rate)
// Program the DMA controller and the sound processor for 8-bit input.
{
   SetupDMA(sb16_card.dma8, ptr, count, DMAMODEWRITE);

   // Program the A/D sample rate.
   dspout(dspcmdADSampRate);
   dspout(samp_rate >> 8);
   dspout(samp_rate & 0xFF);

   // Program card for 8-bit signed input.
   dspout(dspSB16DMA8 | dspSB16ADC | dspSB16AI | dspSB16FifoOn);
   dspout(dspSB16ModeMono | dspSB16ModeSigned);
   dspout((count/2-1) & 0xFF);
   dspout((count/2-1) >> 8);
}


void SetupOutput16(void far *ptr, unsigned int count, unsigned int samp_rate)
// Program the DMA controller and the sound processor for 16-bit output.
{
   SetupDMA(sb16_card.dma16, ptr, count, DMAMODEREAD);

   // Program the D/A sample rate.
   dspout(dspcmdDASampRate);
   dspout(samp_rate >> 8);
   dspout(samp_rate & 0xFF);

   // Program card for 16-bit signed output.
   dspout(dspSB16DMA16 | dspSB16DAC | dspSB16AI | dspSB16FifoOn);
   dspout(dspSB16ModeMono | dspSB16ModeSigned);
   dspout((count/2-1) & 0xFF);
   dspout((count/2-1) >> 8);
}


void interrupt DMAend(...)
{
   static int irq_active;
   int intstat;

   irq_active++;
   outp(sb16_card.base + dspoffsetMixerAddr, INTSTATUS);
   intstat = inp(sb16_card.base + dspoffsetMixerData);

   if (intstat & DMA16IntStatBit) {
      inp(sb16_card.base + dspoffsetDMA16Ack);      // acknowledge interrupt
      sb16_card.dma16_cnt++;
      if (irq_active==1) {
	 if (sb16_card.action==1)
	    write_buffer((sb16_card.dma16_cnt & 1) ? dma16_inbuf : dma16_inbuf+BUFSIZE/2, BUFSIZE/2);
	 if (sb16_card.action==2)
	    read_buffer((sb16_card.dma16_cnt & 1) ? dma16_outbuf : dma16_outbuf+BUFSIZE/2, BUFSIZE/2);
	 else
	    dtmf_16((sb16_card.dma16_cnt & 1) ? dma16_inbuf : dma16_inbuf+BUFSIZE/2, BUFSIZE/2);
      }
   }
   if (intstat & DMA8IntStatBit) {
      inp(sb16_card.base + dspoffsetDataAvail);     // acknowledge interrupt
      sb16_card.dma8_cnt++;
      if (irq_active==1) {
	 dtmf_8((sb16_card.dma8_cnt & 1) ? dma8_inbuf : dma8_inbuf+BUFSIZE/2, BUFSIZE/2);
      }
   }

   if (sb16_card.irq & 8)                           // If irq 10, send EOI to second PIC.
      outp(PIC2MODE, PICEOI);
   outp(PIC1MODE, PICEOI);                          // Send EOI to first PIC.
   irq_active--;
}


int sb16_getconfig(char *env_name)
// Read a blaster environment string.
{
   char *env = getenv(env_name);

   if (!env)
   {  printf("- SB16: %s environment variable not set.\n", env_name);
      return 5;
   }

   while (*env) {
      switch(toupper( *(env++) )) {

	 case 'A': if (!(sb16_card.base = strtol(env, &env, 16))) {
		      printf("- SB16: Error in port address specification.\n");
		      return 1;
		   }
		   break;

	 case 'I': if (!(sb16_card.irq = strtol(env, &env, 10))) {
		      printf("- SB16: Error in IRQ number specification.\n");
		      return 2;
		   }
		   break;

	 case 'D': if (!(sb16_card.dma8 = strtol(env, &env, 10))) {
		      printf("- SB16: Error in 8-bit DMA channel specification.\n");
		      return 3;
		   }
		   break;

	 case 'H': if (!(sb16_card.dma16 = strtol(env, &env, 10))) {
		      printf("- SB16: Error in 16-bit DMA channel specification.\n");
		      return 4;
		   }
		   break;

	 default:
		   break;
      }
   }

   return 0;
}


void enable_interrupt(void interrupt (*newvect)(...))
{
   int intrmask;

   // calculate interrupt number for IRQ
   if (sb16_card.irq & 8)
      sb16_card.intrnum = sb16_card.irq + 0x68;       // IRQs 8-15 map to interrupts 70H-78H.
   else
      sb16_card.intrnum = sb16_card.irq + 0x08;       // IRQs 0-7 map to interrupts 08H-0FH.

   sb16_card.intvecsave = getvect(sb16_card.intrnum); // save previous interrupt vector
   setvect(sb16_card.intrnum, newvect);               // set new interrupt vector

   // enable interrupts at interrupt controllers
   intrmask = 1 << sb16_card.irq;
   outp(PIC1MASK, inp(PIC1MASK) & ~intrmask);
   intrmask >>= 8;
   outp(PIC2MASK, inp(PIC2MASK) & ~intrmask);
}


void disable_interrupt(void)
{
   int intrmask;

   // disable interrupts at interrupt controllers
   intrmask = 1 << sb16_card.irq;
   outp(PIC1MASK, inp(PIC1MASK) | intrmask);
   intrmask >>= 8;
   outp(PIC2MASK, inp(PIC2MASK) | intrmask);

   // Restore previous vector
   setvect(sb16_card.intrnum, sb16_card.intvecsave);
}


void mydelay(unsigned long clocks)
/*
 * "clocks" is clock pulses (at 1.193180 MHz) to elapse, but remember that
 * normally the system timer runs in mode 3, in which it counts down by twos,
 * so delay3(1193180) will only delay half a second.
 *
 *   clocks = time * 2386360
 *
 *     time = clocks / 2386360
 */
{
   unsigned long elapsed=0;
   unsigned int last,next;

   /* Read the counter value. */
   outp(0x43,0);				/* want to read timer 0 */
   last=inp(0x40);				/* low byte */
   last=~((inp(0x40) << 8) + last);		/* high byte */
   do {
      /* Read the counter value. */
      outp(0x43,0);				/* want to read timer 0 */
      next=inp(0x40);				/* low byte */
      next=~((inp(0x40) << 8) + next);		/* high byte */
      elapsed += next-last;			/* add to elapsed cycles */
      last=next;
   } while (elapsed < clocks);
}


void opl_poke(int reg, int val)
/* This function outputs a value to a specified FM register at the Sound
 * Blaster port address.
 */
{
   outp(sb16_card.base+8, reg);
   mydelay(8);          /* need to wait 3.3 microsec */
   outp(sb16_card.base+9, val);
   mydelay(55);         /* need to wait 23 microsec */
}


void sb16_setmixer(char addr, char val)
{
   outp(sb16_card.base+dspoffsetMixerAddr,addr);
   outp(sb16_card.base+dspoffsetMixerData,val);
}


char sb16_getmixer(char addr)
{
   outp(sb16_card.base+dspoffsetMixerAddr,addr);
   return inp(sb16_card.base+dspoffsetMixerData);
}


void sb16_init_opl(void)
{
   for (int i=0; i < 256; i++)
     opl_poke(i, 0);

   opl_poke(0xC0,1);     /* parallel connection */

   /***************************************
    * Set parameters for the carrier cell *
    ***************************************/
   opl_poke(0x23,0x21);  /* no amplitude modulation (D7=0), no vibrato (D6=0),
			  * sustained envelope type (D5=1), KSR=0 (D4=0),
			  * frequency multiplier=1 (D4-D0=1)
			  */
   opl_poke(0x43,0x0);   /* no volume decrease with pitch (D7-D6=0),
			  * no attenuation (D5-D0=0)
		          */
   opl_poke(0x63,0xff);  /* fast attack (D7-D4=0xF) and decay (D3-D0=0xF) */
   opl_poke(0x83,0x05);  /* high sustain level (D7-D4=0), slow release rate (D3-D0=5) */

   /*****************************************
    * Set parameters for the modulator cell *
    *****************************************/
   opl_poke(0x20,0x20);  /* sustained envelope type, frequency multiplier=0    */
   opl_poke(0x40,0x3f);  /* maximum attenuation, no volume decrease with pitch */
}


void sb16_init_mixer(void)
{
   /* save input/output enable flags */
   sb16_mixer.InSrcL    = sb16_getmixer(ADCSRC_L);
   sb16_mixer.InSrcR    = sb16_getmixer(ADCSRC_R);
   sb16_mixer.OutSrc    = sb16_getmixer(OUTSRC);

   /* save input/output volume */
   sb16_mixer.LineVolL   = sb16_getmixer(LINEVOL_L);
   sb16_mixer.LineVolR   = sb16_getmixer(LINEVOL_R);
   sb16_mixer.MasterVolL = sb16_getmixer(MASTERVOL_L);
   sb16_mixer.MasterVolR = sb16_getmixer(MASTERVOL_R);
   sb16_mixer.VocVolL    = sb16_getmixer(VOCVOL_L);
   sb16_mixer.VocVolR    = sb16_getmixer(VOCVOL_R);
   sb16_mixer.SynthVolL  = sb16_getmixer(FMVOL_L);
   sb16_mixer.SynthVolR  = sb16_getmixer(FMVOL_R);
   sb16_mixer.MicVol     = sb16_getmixer(MICVOL);

   /* select LINE IN and MIC input */
   sb16_setmixer(ADCSRC_L,    0x19);
   sb16_setmixer(ADCSRC_R,    0x19);

   /* set input/output volume */
   sb16_setmixer(LINEVOL_L,   cfg.level_in  << 4);
   sb16_setmixer(LINEVOL_R,   cfg.level_in  << 4);
   sb16_setmixer(MASTERVOL_L, cfg.level_out << 4);
   sb16_setmixer(MASTERVOL_R, cfg.level_out << 4);
   sb16_setmixer(VOCVOL_L,    cfg.level_voc << 4);
   sb16_setmixer(VOCVOL_R,    cfg.level_voc << 4);
   sb16_setmixer(FMVOL_L,     cfg.level_cw  << 4);
   sb16_setmixer(FMVOL_R,     cfg.level_cw  << 4);
   sb16_setmixer(MICVOL,      cfg.level_mic << 4);
}


void sb16_restore_mixer(void)
{
   /* restore input/output enable flags */
   sb16_setmixer(ADCSRC_L,    sb16_mixer.InSrcL);
   sb16_setmixer(ADCSRC_R,    sb16_mixer.InSrcR);
   sb16_setmixer(OUTSRC,      sb16_mixer.OutSrc);

   /* restore input/output volume */
   sb16_setmixer(LINEVOL_L,   sb16_mixer.LineVolL);
   sb16_setmixer(LINEVOL_R,   sb16_mixer.LineVolR);
   sb16_setmixer(MASTERVOL_L, sb16_mixer.MasterVolL);
   sb16_setmixer(MASTERVOL_R, sb16_mixer.MasterVolR);
   sb16_setmixer(VOCVOL_L,    sb16_mixer.VocVolL);
   sb16_setmixer(VOCVOL_R,    sb16_mixer.VocVolR);
   sb16_setmixer(FMVOL_L,     sb16_mixer.SynthVolL);
   sb16_setmixer(FMVOL_R,     sb16_mixer.SynthVolR);
   sb16_setmixer(MICVOL,      sb16_mixer.MicVol);
}


void sb16_synth(long freq)
{
   int block=0;
   long fn;

   if (!sb16_card.init_complete)
     return;

   do
     fn = freq * 20.9715 / (1 << block);
   while ((fn > 1024) && (block++ < 8));
   opl_poke(0xA0,(fn & 0xFF));
   opl_poke(0xB0,((fn >> 8) & 0x3) | (block << 2) | 0x20);
}


char sb16_reset(void)
{
  outp(sb16_card.base+dspoffsetReset,1);
  mydelay(20);
  outp(sb16_card.base+dspoffsetReset,0);
  mydelay(100);

  if (dspin() != 0xAA)
    return 0;

  return 1;
}


/* end up everything peacefully */
void sb16_atexit(void)
{
  /* get silence */
  sb16_synth(0);
  sb16_reset();
  disable_interrupt();
  sb16_restore_mixer();
}


int sb16_init(void)
// Make sure it's an SB16 with different 8- and 16-bit DMA channels.
{  unsigned int ver, ver_minor;

   /* detect sb card */
   if (!sb16_reset())
   { printf("- SB16: SoundBlaster card not found.");
     return 1;
   }

   /* get DSP version number */
   dspout(dspcmdGetVersion);
   ver = dspin();
   ver_minor = dspin();

   printf(" SB16: DSP version %d.%d found.\n",ver,ver_minor);
   if (ver!=4)
   { printf("- SB16: SoundBlaster 16 (DSP V4.XX) required.");
     return 2;
   }

   if (!getco()) {
     printf("- SB16: Numeric co-processor required.\n");
     return 3;
   }

   if (sb16_card.dma8 == sb16_card.dma16) {
     printf("- SB16: 8-bit and 16-bit DMA may not share the same channel.\n");
     return 4;
   }

   /* allocate in/out DMA buffer */
   if ((dma16_outbuf = (int *)dma_alloc(2*BUFSIZE)) == NULL) {
      printf("- SB16: Unable to allocate output DMA buffer\n");
      return 5;
   }
   if ((dma16_inbuf = (int *)dma_alloc(2*BUFSIZE)) == NULL) {
      printf("- SB16: Unable to allocate input DMA buffer\n");
      return 5;
   }
   dma8_inbuf = (char *)dma16_inbuf;

   /* init OPL3 synthesizer */
   sb16_init_opl();

   /* init mixer settings from ini file */
   sb16_init_mixer();

   enable_interrupt(DMAend);
   atexit(sb16_atexit);

   /* start DTMF receiver sample input */
   SetupInput16(dma16_inbuf,BUFSIZE,SRATE_32000);
   mydelay(100000);
   if(!sb16_card.dma16_cnt)
   { printf("- SB16: IRQ %d failure !\n",sb16_card.irq);
     return 6;
   }

   printf(" SB16: SoundBlaster 16 found at io=0x%X, irq=%d, dma8=%d, dma16=%d.\n",
	  sb16_card.base,sb16_card.irq,sb16_card.dma8,sb16_card.dma16);

   sb16_card.init_complete=1;
   return 0;
}


char sb16_getdtmfch(void)
{
   return dtmf_state.dtmfch;
}


char sb16_getlevel(void)
{
   return af_level;
}


void sb16_setrate(long srate)
{
   if (!srate) return;
   sb16_card.new_rate=(srate < 24000) ? SRATE_16000 : SRATE_32000;
}


void sb16_mute(int mute)
{
   if(mute)
     sb16_setmixer(OUTSRC,0x19);		// enable LINE IN + MIC -> OUT
   else
     sb16_setmixer(OUTSRC,0);			// disable LINE IN + MIC -> OUT
}


void sb16_action(int action)
{
   action &= 3;
   if (action!=sb16_card.action)
   { sb16_card.action=action;
     sb16_reset();
     sb16_card.dma8_cnt=0; sb16_card.dma16_cnt=0;
     sb16_card.samp_rate = sb16_card.new_rate;
     memset(dma16_outbuf,0,2*BUFSIZE);
     switch(action)
     { case 0: SetupInput16(dma16_inbuf,BUFSIZE,sb16_card.samp_rate);
	       break;
       case 1: SetupInput16(dma16_inbuf,BUFSIZE,sb16_card.samp_rate);
	       break;
       case 2: SetupInput8(dma8_inbuf,BUFSIZE,sb16_card.samp_rate);
	       SetupOutput16(dma16_outbuf,BUFSIZE,sb16_card.samp_rate);
	       break;
     }
   }
}
