/***************************************************************

  DVMS            Digital Voice Mailbox System


  ------------------------------------------------
  FX709 interrupt and INT 08 handling
  ------------------------------------------------


  Copyright (c)92-95 Detlef Fliegl DG9MHZ
          Guardinistr. 47
	  D-81375 Muenchen

  Alle Rechte vorbehalten / All Rights reserved

 ***************************************************************/
#ifdef __MSDOS__
#include "dvms.h"
#include "dvmscw.h"
#include "dvmsdcf.h"
#include "dvmsirq.h"
#include "dvmssuba.h"
#include "dvmssb.h"
#include "dvmscvsd.h"

static int volatile WD_flag=1,
	    WD_state=1,
	    WD_IRQ_state,
	    WD_IRQ_flag,
	    WD_IRQ_count=18,
	    IRQ_action=2,
	    IRQ_status,
	    IRQ_flag=1,
	    IRQ_Power,
	    IRQ_loop=0,
	    IRQ_audio,
	    irq_add,
	    irq_ctrl,
	    channel_b=0;

static long volatile WD_maxcount=600*18;
static long volatile IRQ_count;
static long volatile IRQ_length=1L;
static long volatile IRQ_start=0L;
static long volatile IRQ_pause;
static long volatile WD_counter;
static long volatile tick=0L;

char far *indos;
unsigned lpt=0;
static char cvsd_buf[256];

static void interrupt timerneu(__CPPARGS);    /* interrupt prototype */
static void interrupt (*timeralt)(__CPPARGS); /* interrupt function screen */
static void interrupt record_play_irq(__CPPARGS);    /* interrupt prototype */
static void interrupt (*old_irq)(__CPPARGS); /* interrupt function screen */
void (far *reset)(void)=(void (far *)())MK_FP(0xffff,0);

void set_wd(unsigned long t)
{
  WD_maxcount=(t*18);
  WD_counter=WD_maxcount;
  WD_state=!WD_state;
}

/****************************************************************************
 Entfernen der beiden Interruptroutinen rcksetzten des Interruptcontrollers
 und freigeben des Timerinterrupt-Stacks
****************************************************************************/
static void uninstall_irq(void)
{ _dos_setvect( 0x1c, timeralt );
  if (usesb16)
    return;
  //if(IRQ_PRIO)
  outp(irq_ctrl,0x0c7);
  if(cfg.irq>7)                                              //DL1GLH
    outp(irq_ctrl+1,inp(irq_ctrl+1)|(int)(1<<(cfg.irq-8)));  //DL1GLH
  else                                                       //DL1GLH
    outp(irq_ctrl+1,inp(irq_ctrl+1)|(int)(1<<cfg.irq));
  _dos_setvect( irq_add+cfg.irq, old_irq );
}

/****************************************************************************
 Einrichten des Timerinterrupt-Stacks, der beiden Interruptroutinen
 und Initialisierung der Variablen.
****************************************************************************/
void install_irq(void)
{  atexit(uninstall_irq);
  strcpy(cw,cfg.call);
  _AH=0x34;
  geninterrupt(0x21);
  indos=(char far*)MK_FP(_ES,_BX);

  if(cfg.lpt)
  { lpt= peek(0x40,0x08+(cfg.lpt-1)*2);
    outp(lpt,0xfc);
  }
  set_wd(_WDTIME);

  timeralt     = _dos_getvect( 0x1c );
  _dos_setvect( 0x1c, timerneu );

  if(usesb16)
  { IRQ_action=0;
    return;
  }
  if(cfg.irq>7)
  { irq_add=0x68;
    irq_ctrl=0xa0;
    outp(irq_ctrl+1,inp(irq_ctrl+1)&(255-(int)(1<<(cfg.irq-8))));
  }
  else
  { irq_add=8;
    irq_ctrl=0x20;                                            //DL1GLH
    outp(irq_ctrl+1,inp(irq_ctrl+1)&(255-(int)(1<<cfg.irq))); //DL1GLH
  }

  old_irq = _dos_getvect( cfg.irq+irq_add );
  _dos_setvect( cfg.irq+irq_add, record_play_irq );

  if(_IRQ_PRIO&&(cfg.irq<8))
    outp(irq_ctrl,0xc0+cfg.irq-1);
  if(_IRQ_PRIO&&(cfg.irq>7))
    outp(irq_ctrl,0xc0+cfg.irq-9);

  outp(0x020,0x20);
  if(cfg.irq&8)                           //DL1GLH
	  outp(0x0a0,0x020);                    //DL1GLH

  if(!skiphwcheck)
  {
    outp(cfg.iobase+1,144);
    outp(cfg.iobase+2,170);
    delay(100);

    for(int i=0;i<3;i++)
    {
      outp(cfg.iobase+2,0x55);
      delay(50);
      inp(cfg.iobase+2);
      delay(50);
      if(!IRQ_action)
	     break;
    }
  }
  IRQ_action=0;
  if(!skiphwcheck&&IRQ_action)
  {  printf("- FX709: INT %d signal missing!\a\n",cfg.irq);
    write_log(_fatal,"install_irq","FX709: INT %d signal missing!",cfg.irq);
    if(!ignore)
    exit(20);
  }

  printf(" Installed IRQ %d handler.\n",cfg.irq);
}

/****************************************************************************
 Mit dieser Routine wird der IRQ Task gestartet:
 Service: 1 Aufname, 2 Wiedergabe
 irq_start: Startadresse des Pufferspeichers
 _count: Lnge fr Aufname oder Wiedergabe
****************************************************************************/
int do_irq(int Service,unsigned long irq_start,unsigned long _count)
{
  time_t t=ad_time(NULL);

  IRQ_length=_count;
  IRQ_start=irq_start;
  IRQ_action=Service;
  IRQ_flag=1;
  IRQ_pause=cfg.max_pause;
  while(IRQ_flag)
  { if((ad_time(NULL)-t)>1)
    { if (usesb16)
	write_log(_fatal,"do_irq","SB16 IRQ died");
      else
	write_log(_fatal,"do_irq","FX709 IRQ died");
      return 0;
    }
    IRQ_action=Service;
    if (usesb16)
    { sb16_action(Service);
    }
    else
    { if(IRQ_action==2)outp(cfg.iobase+2,0x55);
      if(IRQ_action==1)inp(cfg.iobase+2);
      inp(cfg.iobase);
    }
  }
  scheduler();
  return 1;
}

/****************************************************************************
 Hier wird jeder IRQ Task sofort gestopt
****************************************************************************/
void stop_irq(void)
{ IRQ_action=0;
  IRQ_flag=1;
  IRQ_length=0L;
  IRQ_audio=32;
  if (usesb16)
    sb16_action(0);
}

int irq_state (void)
{ char skip=0;
  int timeout=10;
  static long count=-1L;

  if(IRQ_action&2)
  { while(timeout--)
    { if(IRQ_count!=count) break;
      suspend(1);
    }
    if(IRQ_count==count)
    { if(usesb16)
        write_log(_fatal,"irq_state","SB16 IRQ died");
      else
        write_log(_fatal,"irq_state","FX709 IRQ died");
      stop_irq();
      skip=1;
    }
  }
  count=IRQ_count;
  suspend(1);
  if(inp(0x60)==1)
    return 0;
  if(!IRQ_flag&&!skip)
    return 1;
  return 0;
}

/****************************************************************************
*   Hier wird der Interrupt des Fx709 verarbeitet: Abspielen, Aufnehmen,
*   Auslesen des Powerregisters    Variablen Satz: IRQ_...
*****************************************************************************/
#define STKLEN 1000
static void interrupt record_play_irq(__CPPARGS)
{
  static char stack[STKLEN+1];
  static unsigned ss,sp;
  static int in_irq=0,i;

  //Auslesen des Statusregisters
  IRQ_status=inp(cfg.iobase);
  screen[77*2]=WD_IRQ_flag*'F';

  if(!in_irq)
  {
    //beep(1000);
    //Rekursionsbremse gegen Reentrenz auf langsamen Rechnern
    in_irq=1;
    ss=_SS;
    sp=_SP;
    _SS=FP_SEG(stack+STKLEN);
    _SP=FP_OFF(stack+STKLEN);

    //Blinkendes F rechts oben im Bild / optische Kontrolle ob FX709 noch
    //IRQs produziert, sonst rcksetzten im Timerinterrupt
    //screen[77*2+1]=0x70;
    WD_IRQ_flag^=1;
    //Update der IRQ_Zustandsvariablen:bei Rufton, DTMF Ton oder
    //geschlossener Rauschsperre
    i=getdt(0);
    if((!(i&SQ_BIT)||(i&DTMF_STB)||!(i&CALL_BIT))&&((IRQ_action&1)||channel_b))
      IRQ_action|=4;
    else
      IRQ_action&=123;

    //Update der IRQ_Zustandsvariablen:
    //Rauschsperreninformation offen 128 / zu 0
    IRQ_action=(i&SQ_BIT)?IRQ_action|SQ_BIT:IRQ_action&~SQ_BIT;

    //Aufnahme wird gestoppt / Encoder mit 10101 gefttert
    //und Decoder leer augelesen
    if((IRQ_action&4)&&(!(IRQ_action&2)))
    {
      if(IRQ_status&9) inp(cfg.iobase+2);
      if(IRQ_status&18) outp(cfg.iobase+2,0xaa);
    }

    //Signalwegschaltung
    if(IRQ_status&36)
    {
      //Auslesen des Feldstrke- und Audiopegels
      IRQ_Power=inp(cfg.iobase+1);

      //Falls Aufnahme und Rufton, DTMF etc (siehe oben), dann
      //Signalwege A und B sperren, sonst durchschalten.
      if(IRQ_action&1)
      if(IRQ_action&4)
	      outp(cfg.iobase+1,32/*+7*/);
      else
	      outp(cfg.iobase+1,16/*+7*/);

      //Falls Wiedergabe und Rufton, DTMF etc (siehe oben), dann
      //Signalwege A und B sperren, sonst A durchschalten.
      //Signalweg B wird in Abhngigkeit von Rauschsperreninformation
      //und channel_b=1 durchgeschalten
      //Verwendung: CW_Rufzeichen wiedergeben, aber gleichzeitig
      //Betrieb ber Umsetzer.
      if(IRQ_action&2)
	if(IRQ_action&4)
	  outp(cfg.iobase+1,32/*+7*/);
	else
	  outp(cfg.iobase+1,((IRQ_action&128)>>7)*channel_b+32/*+7*/);

      //Falls weder Aufnahme, noch Wiedergabe,dann Signalwegschaltung,
      //wie in IRQ_audio angegeben.
      if((IRQ_action&3)==0)
	outp(cfg.iobase+1,IRQ_audio/*+7*/);

      //Sprechpausenkompensation bei Aufnahme durch Audiopegel-
      //Schwellwertermittlung
      if((IRQ_action&2)||((IRQ_Power&15)>_AUDIO_MIN))
      {
	IRQ_pause=cfg.max_pause;
	IRQ_action=IRQ_action&247;
      }
      else
      {
	if(IRQ_pause>0)
	  IRQ_pause--;
	else
	{
	  IRQ_action|=8;
	  inp(cfg.iobase+2);
	}
      }
    }

    //Start von Aufname oder Wiedergabe durch setzen von
    //IRQ_action=1|2, IRQ_start=Startadresse und IRQ_flag=1
    if((IRQ_action&3)&&IRQ_flag)
    {
      IRQ_count=IRQ_start;
      IRQ_flag=0;
    }
    //goto end;
    //Bei Aufname oder Wiedergabe wird hier entweder ein Byte
    //gelesen oder geschrieben, solange, bis IRQ_count==IRQ_length
    //falls Ende erreicht kann bei IRQ_loop nochmal gestartet werden.
    //cfg.anz trgt die Puffergre in Bytes
    if(IRQ_count<IRQ_length)
    {                             //neu
      if( (((IRQ_action&~SQ_BIT)==1)||( ((IRQ_action&~SQ_BIT)==5) && !(IRQ_action&SQ_BIT)))
	  &&(IRQ_status&9) )
	*(puffer+(IRQ_count++&(_MEM-1)))=inp(cfg.iobase+2);

      if((IRQ_action&2)&&(IRQ_status&18) )
	outp(cfg.iobase+2,*(puffer+(IRQ_count++&(_MEM-1))));
    }
    else
    {
      if(IRQ_loop==0)
      { if(IRQ_action&2)
	  outp(cfg.iobase+2,85);
	IRQ_action=0;
	IRQ_flag=1;
      }
      else
	IRQ_count=0L;
    }
    end:;
    //beep(0);
    //Wiederherstellen des alten Stacks.
    _SS=ss;
    _SP=sp;

    //Rekursionssperre aufheben
    in_irq=0;
    //screen[77*2+1]=0x07;
  }

  //EOI an den Interruptcontroller 1 senden
  outp(0x020,0x020);

  //EOI an den Interruptcontroller 2 senden, falls Irq >7
  if(cfg.irq&8)
    outp(0x0a0,0x020);
}

/****************************************************************************
 Timerinterrupt: Softwarewatchdog, Watchdog fr Fx709 Irq, Impulse fr
 Hardwarewatchdog, DCF-Synchronisation und Rufzeichengeber
****************************************************************************/
static void interrupt timerneu(__CPPARGS)
{ static char stack[STKLEN+1];
  static unsigned ss,sp;
  static in_timer;

  //Hardwareimpuls erzeugen
   if(cfg.lpt)
     outp(lpt,inp(lpt)^1);

  tick++;

  //Rekursionssperre
  if(in_timer==0)
  {
    in_timer=1;
    //eigenen Stack aufbauen
    ss=_SS;
    sp=_SP;
    _SS=FP_SEG(stack+STKLEN);
    _SP=FP_OFF(stack+STKLEN);

    if (!usesb16)
      outp(cfg.iobase+5,inp(cfg.iobase+5)^1);

    //Blinkendes T rechts oben im Bild / optische Kontrolle fr
    //Timerinterrupt
    screen[78*2]=WD_flag*'T';

    //Watchdog fr Fx709 Irq
    if(WD_IRQ_flag==WD_IRQ_state)
      WD_IRQ_count--;
    else
      WD_IRQ_count=18;

    //Neu-Initialisierung fr Fx709, damit er wieder irqs schickt
    if(!WD_IRQ_count && !usesb16)
    {
      inp(cfg.iobase);
      inp(cfg.iobase+1);
      inp(cfg.iobase+2);
      outp(cfg.iobase+2,85);
//      outp(0x20,0x20);
//      if(cfg.irq&8)
//        outp(0x0a0,0x020);
    }

    WD_IRQ_state=WD_IRQ_flag;

    //Softwarewatchdog fr Software: Wenn lngere Zeit die Variable
    //WD_flag nicht verndert wird, dann erfolgt ein Reset des
    //Rechners
    if(WD_flag==WD_state)
      WD_counter--;
    else
      WD_counter=WD_maxcount;

    if(WD_counter==0)
    {
      hw_set(6,0);
      hw_set(8,0);
      reset();
    }
    WD_flag=WD_state;

    //Einsprung fr DCF-77 Synchronisation
    if(cfg.dcf_used)
      dcf_update();

    call_generator();

    //Wiederherstellen des alten Stacks.
    _SS=ss;
    _SP=sp;
    //Rekursionssperre aufheben
    in_timer=0;
  }
  //mit DOS Timerinterruptroutine beginnen.
  timeralt();
}

void read_buffer(int *audio_buf, int len)
{ int i;

  if (IRQ_action & 2)
  {
    if (IRQ_flag)
    { IRQ_count=IRQ_start;
      IRQ_flag=0;
    }
    if (IRQ_count < (IRQ_length+(len>>2)))
    {
      memset(cvsd_buf, 0xAA, (len>>3));
      for(i=0; i < (len>>3); i++,IRQ_count++)
	if (IRQ_count < IRQ_length)
	  cvsd_buf[i]=puffer[IRQ_count%_MEM];
      cvsd_read(cvsd_buf, audio_buf, len);
    }
    else
    { IRQ_action=0;
      IRQ_flag=1;
    }
  }
}

void write_buffer(int *audio_buf, int len)
{ int i;

  if (IRQ_action & 1)
  {
    if (IRQ_flag)
    { IRQ_count=IRQ_start;
      IRQ_flag=0;
    }
    if (IRQ_count < IRQ_length)
    { cvsd_write(cvsd_buf, audio_buf, len);
      for(i=0; i < (len>>3); i++)
	if (IRQ_count<IRQ_length)
	  puffer[IRQ_count++%_MEM]=cvsd_buf[i];
    }
    else
    { IRQ_action=0;
      IRQ_flag=1;
    }

    i=getdt(0);
    if(!(i&SQ_BIT)||(i&DTMF_STB)||!(i&CALL_BIT))
      sb16_mute(0);
    else
      sb16_mute(1);

    IRQ_action = (i&SQ_BIT) ? IRQ_action|SQ_BIT:IRQ_action&~SQ_BIT;
  }
}

int getdt(int mode)
{ int mask=0xff;
  char ch;

  if(mode)
    mask=31;

  if(get_global_msg()&_SHUTDOWN)
    return CALL_BIT & mask;

  if(usesb16)
  { ch = sb16_getdtmfch();
    if(cfg.lpt) ch |= inp(lpt+1) & SQ_BIT;
  }
  else
    ch = inp(cfg.iobase+4);

  if(cfg.sq_inv) 
    ch ^= SQ_BIT;
  return ch & mask;
}


int getsq(void)
{ if(get_global_msg()&_SHUTDOWN)
    return 0;

  if(!cfg.repeater&&ptt_state)
    return 0;

  if(cfg.config&_CFGSUBSQ)
  { update_ctcss();
    if(ctcss)
      return 1;
  }

  if((cfg.config&_CFGSQDIS)&&(cfg.config&_CFGSUBSQ))
  { return 0;
  }

  return getdt(0) & SQ_BIT;
}

void setrate(long rate)
{ int x=0,y=0,z=0;

  srate=rate;
  if (usesb16 && (rate>11000))
  { sb16_setrate(rate);
    cvsd_setrate(rate);
    return;
  }
  if (rate>11000){ x=1;y=0;z=1;}
  if (rate>15000){ x=0;y=0;z=1;}
  if (rate>24000){ x=1;y=0;z=0;}
  if (rate>31000){ x=0;y=0;z=0;}
  outp(cfg.iobase+0,z*36+y*72+x*144);
}

void hw_set(int reg,int value)
{ if(!usesb16)
    outp(cfg.iobase+reg,value);
}

int hw_get(int reg)
{ if(usesb16) return 0;
  return inp(cfg.iobase+reg);
}

void toggle_wd(void)
{
  WD_state=!WD_state;
}

void set_chb(int b)
{ if(b)
    channel_b=16;
  else
    channel_b=0;
}

int get_chb(void)
{ return !!channel_b;
}

void set_audio(int val)
{
  if(usesb16) sb16_mute(val & 16);
  IRQ_audio=val;
}

int get_irqact(void)
{ return IRQ_action;
}

long get_irqcnt(void)
{ return IRQ_count;
}

long get_irqpse(void)
{ return IRQ_pause;
}

void set_irqloop(int val)
{
  IRQ_loop=val;
}

int get_irqpwr(void)
{
  if(usesb16) IRQ_Power = sb16_getlevel();
  if(cfg.sig_inv||(cfg.config&_CFGSIGINV))
    return ((15-(IRQ_Power>>4))<<4)|(IRQ_Power&15);
  else
    return IRQ_Power;
}

void set_irqact(int val)
{
  IRQ_action=val;
}

long timer(void)
{
  return tick;
}
#endif
