/***************************************************************

  DVMS            Digital Voice Mailbox System

  ------------------------------------------------
  CVSD signal processing codec routines
  ------------------------------------------------

  Copyright (c)92-95 Detlef Fliegl DG9MHZ
		     Guardinistr. 47
		     D-81375 Muenchen

  Alle Rechte vorbehalten / All Rights reserved

  and

  Copyright (c)96,97 Thomas Sailer HB9JNX/AE4WA
		     Swiss Federal Institute of Technology
		     E-Mail: sailer@ife.ee.ethz.ch

 ***************************************************************/

#include <math.h>
#include <string.h>
#include "dvmscvsd.h"

/* ---------------------------------------------------------------------- */

#define ENC_FILTERLEN 48
#define DEC_FILTERLEN 48

/* ---------------------------------------------------------------------- */

static const float dec_filter_16[48] = {
	       0.001102,       0.001159,       0.000187,      -0.000175,
	       0.002097,       0.006543,       0.009384,       0.008004,
	       0.006562,       0.013569,       0.030745,       0.047053,
	       0.050491,       0.047388,       0.062171,       0.109115,
	       0.167120,       0.197144,       0.195471,       0.222098,
	       0.354745,       0.599184,       0.849632,       0.956536,
	       0.849632,       0.599184,       0.354745,       0.222098,
	       0.195471,       0.197144,       0.167120,       0.109115,
	       0.062171,       0.047388,       0.050491,       0.047053,
	       0.030745,       0.013569,       0.006562,       0.008004,
	       0.009384,       0.006543,       0.002097,      -0.000175,
	       0.000187,       0.001159,       0.001102,       0.000000
};

/* ---------------------------------------------------------------------- */

static const float dec_filter_32[48] = {
	       0.001950,       0.004180,       0.006331,       0.007907,
	       0.008510,       0.008342,       0.008678,       0.011827,
	       0.020282,       0.035231,       0.055200,       0.075849,
	       0.091585,       0.098745,       0.099031,       0.101287,
	       0.120058,       0.170672,       0.262333,       0.392047,
	       0.542347,       0.684488,       0.786557,       0.823702,
	       0.786557,       0.684488,       0.542347,       0.392047,
	       0.262333,       0.170672,       0.120058,       0.101287,
	       0.099031,       0.098745,       0.091585,       0.075849,
	       0.055200,       0.035231,       0.020282,       0.011827,
	       0.008678,       0.008342,       0.008510,       0.007907,
	       0.006331,       0.004180,       0.001950,      -0.000000
};

/* ---------------------------------------------------------------------- */

#ifndef	GNUC
#define	__inline__
#endif

/* ---------------------------------------------------------------------- */
/*
 * private data structures
 */

struct cvsd_decode_state {
	float mla_int;
	float mla_tc0;
	float mla_tc1;
	float output_filter[DEC_FILTERLEN];
	unsigned ovrload;
	unsigned shreg;
	unsigned mask;
	unsigned cnt;
	unsigned cvsd_rate;
};

struct cvsd_encode_state {
	float mla_int;
	float mla_tc0;
	float mla_tc1;
	unsigned ovrload;
	unsigned shreg;
	unsigned mask;
	unsigned cnt;
	unsigned cvsd_rate;
	int recon_int;
};

struct cvsd_decode_state dec;
struct cvsd_encode_state enc;

/* ---------------------------------------------------------------------- */

static float __inline__ float_conv(const float *fp1, const float *fp2, int n)
{
	float res = 0;
	for(; n > 0; n--)
		res += (*fp1++) * (*fp2++);
	return res;
}

/* ---------------------------------------------------------------------- */
/*
 * some remarks about the implementation of the CVSD decoder
 * the principal integrator is integrated into the output filter
 * to achieve this, the coefficients of the output filter are multiplied
 * with (1/(1-1/z)) in the filter generation utility.
 * the output filter must have a sharp zero at f=0 (i.e. the sum of the
 * filter parameters must be zero). This prevents an accumulation of
 * DC voltage at the principal integration.
 */
/* ---------------------------------------------------------------------- */

void cvsd_setrate(long srate)
{
	float *fp1;
	int i;

	dec.cvsd_rate = (srate < 24000) ? 16000 : 32000;
	dec.ovrload = 0x5;
	dec.mla_int = 0;
	dec.mla_tc0 = exp((-200.0)/((float)(dec.cvsd_rate)));
	dec.mla_tc1 = 0.1 * (1 - dec.mla_tc0);
	dec.shreg = dec.cnt = 0;
	dec.mask = 1;
	/*
	 * zero the filter
	 */
	for(fp1 = dec.output_filter, i = DEC_FILTERLEN; i > 0; i--)
		*fp1++ = 0;

	enc.cvsd_rate = (srate < 24000) ? 16000 : 32000;
	enc.ovrload = 0x5;
	enc.mla_int = 0;
	enc.mla_tc0 = exp((-200.0)/((float)(enc.cvsd_rate)));
	enc.mla_tc1 = 0.1 * (1 - enc.mla_tc0);
	enc.shreg = enc.cnt = 0;
	enc.mask = 1;
	enc.recon_int = 0;
}

/* ---------------------------------------------------------------------- */

int cvsd_read(char *cvsd_buf, int *audio_buf, int nsamp)
{
	int bytes=0;

	while (nsamp--) {
		if (!dec.cnt) {
			dec.shreg = *(cvsd_buf++);
			dec.cnt = 8;
			dec.mask = 1;
			bytes++;
		}
		/*
		 * handle one bit
		 */
		dec.cnt--;
		dec.ovrload <<= 1;
		dec.ovrload |= (dec.shreg & dec.mask) ? 1 : 0;
		dec.ovrload &= 7;
		dec.mask <<= 1;

		dec.mla_int *= dec.mla_tc0;
		if ((dec.ovrload == 0) || (dec.ovrload == 7))
			dec.mla_int += dec.mla_tc1;

		memmove(dec.output_filter+1, dec.output_filter,
			sizeof(dec.output_filter)-sizeof(float));

		if (dec.ovrload & 1)
			dec.output_filter[0] = dec.mla_int;
		else
			dec.output_filter[0] = -dec.mla_int;
		/*
		 * check if the next output is due
		 */
		*(audio_buf++) = float_conv(dec.output_filter,
					    (dec.cvsd_rate < 24000) ?
					    dec_filter_16 : dec_filter_32,
					    DEC_FILTERLEN) * 32768.0;
	}
	return bytes;
}

/* ---------------------------------------------------------------------- */

int cvsd_write(char *cvsd_buf, int *audio_buf, int nsamp)
{
	int bytes=0;

	while(nsamp--) {
		/*
		 * encode one bit
		 */
		enc.ovrload <<= 1;
		enc.ovrload |= (*(audio_buf++) > enc.recon_int) ? 1 : 0;
		enc.ovrload &= 7;

		enc.mla_int *= enc.mla_tc0;
		if ((enc.ovrload == 0) || (enc.ovrload == 7))
			enc.mla_int += enc.mla_tc1;

		if (enc.ovrload & 1) {
			enc.recon_int += enc.mla_int * 32768;
			enc.shreg |= enc.mask;
		} else
			enc.recon_int -= enc.mla_int * 32768;

		if (++enc.cnt >= 8) {
			*(cvsd_buf++)=enc.shreg;
			enc.shreg = enc.cnt = 0;
			enc.mask = 1;
			bytes++;
		} else {
			enc.mask <<= 1;
		}
	}
	return bytes;
}

/* ---------------------------------------------------------------------- */
