//---------------------------------------------------------------------------
// File  :  PHW_intf_dll.h
// Date  :  2008-05-19   (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
// Location: <DL4YHF>C:\cbproj\WinPicPr\interface_dll_sources\PHW_intf_dll.h
//
// Description:
//   Header file used by all "hardware-interface"-plugins for WinPic .
//   Compile with Dev-Cpp or Borland C++; MS-C may work but never tested.

#ifndef _PHW_INTF_DLL_H_
#define _PHW_INTF_DLL_H_

#if BUILDING_DLL /* compilation switch, set to 1 (one) when BUILDING the DLL.*/
/* In their endless wisdom, C compiler authors seem to put a lot of effort */
/* in making their compilers as incompatible as possible. Some examples:   */
/*  Borland C++ :  int __stdcall __declspec(dllimport)  MyFunction(void);  */
/*  DevCpp/GNU  :  __stdcall __declspec(dllimport) int  MyFunction(void);  */
/* To avoid cluttering the code with dozens of #ifdefs, we use one macro   */
/* BEFORE and another macro AFTER the data type, so the prototype in the   */
/* DLL header file looks the same for all compilers:                       */
/*    API_EXPORT(int) MyNeatFunction(int something);                       */
# ifdef __BORLANDC__
#    define API_EXPORT(type) type __stdcall __declspec(dllexport)
# else /* not __BORLANDC__ but most likely DevCpp : */
#    define API_EXPORT(type) __stdcall __declspec(dllexport) type
# endif /* !__BORLANDC__ */
#else /* Not BUILDING_DLL (but "using" the DLL) : */
# ifdef __BORLANDC__
#    define API_EXPORT(type) type __stdcall __declspec(dllimport)
# else /* not __BORLANDC__ but most likely DevCpp : */
#    define API_EXPORT(type) __stdcall __declspec(dllimport) type
# endif /* !__BORLANDC__ */
#endif /* Not BUILDING_DLL */


//--------------------------------------------------------------------------
// Data Types and structures
//--------------------------------------------------------------------------

#ifndef _INC_WINDOWS
 // If WINDOWS.H was *NOT* included before THIS header,
 //   define some basic data types compatible to windows.h / wtypes.h :
 #ifndef  _WORD_DEFINED
   #define _WORD_DEFINED
   typedef unsigned short WORD;
 #endif // ! _WORD_DEFINED

 #ifndef _DWORD_DEFINED
   #define _DWORD_DEFINED
   typedef unsigned long DWORD;
 #endif // ! _DWORD_DEFINED
#endif// ! def _INC_WINDOWS


typedef struct // T_PHWInfo. A pointer to this structure
               //   is passed to the exported plugin functions.
{              // Last modified :  2008-05-06 (creation)
  int iSizeOfStruct;  // set to sizeof(T_PHWInfo) by the host .
      // Should be used by the plugin to check compatibility :
      // if (sizeof(T_PHWInfo) != pInfo->iSizeOfStruct ) then bail out !
      // Note: Make sure your compiler uses 32-bit "int" .

  //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Information about the programmer (interface; adapter) .
  char   sz255InfoText[256]; // info text, should be set in PHW_Init() .
  int    iInfoTextModifyCnt; // increment this after modifying sz255InfoText
  //  Info text, will be displayed on WinPic's interface control panel .
  //  Unlike the "error text" below, the "info text" is displayed on a static
  //  text control - not in a scrolling editor window.
  //  If the plugin wants a new 'info text' to be updated on the control panel,
  //  it must FIRST print the string into sz255InfoText[] ,
  //  and then increment iInfoTextModifyCnt to let the user interface know
  //  that a new text string is now ready for displaying.
  //  This reduces problems with multithreading.
  //  The same principle is also used for other strings in T_PHWInfo .

  char   sz255ErrorText[256]; // error string, appended to WinPic's "message" list
  // Error message, may be set by plugin if a function fails.
  //  Visible on WinPic's control panel in a small, scrolling, multiline editor window.
  //  (unlike the "info text" which is displayed as non-scrolling static text).
  //  Can be used for debugging too .
  int    iErrorTextModifyCnt; // increment this after modifying sz255ErrorText
  int    iErrorTextCommand;   // special command to control the "error" window:
    #define PHW_ETCMD_NO_ACTION    0  /* do nothing (set by GUI after execution)*/
    #define PHW_ETCMD_CLEAR_WINDOW 1  /* clear the contents of the error window */

  char   sz255PluginFileName[256]; // name of the plugin, set by WinPic (!)
    // when loading the file with the LoadLibrary()-function. READ-ONLY for the plugin !
    // The plugin (DLL code) may use this member to check if it has been
    // renamed (since the original compilation) for whatever reason .
    // Furthermore, sz255pszPluginFileName contains the FULL DIRECTORY PATH.

  //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Some information about the device currently selected for programming.
  // "Simple" (bit-banging) interfaces don't need this, but "smart" interfaces
  // need this because they will do all the serial output, timing, internally.
  char   sz63ProgrammedDeviceName[64]; // name of the device currently selected
    // for programming; for example "PIC16F628A"  .  The plugin may need this
    // info to program some "exotic" chips, which may not be PICs at all .
    // Also, depending on the device name, the plugin may decide to use special
    // timing parameters .
  int  iCodeMemorySize_words;  // size of the device's CODE MEMORY,
                               // measured in "Code memory WORDS" .
     // For PICs, this can be 12, 14, 16, 24, or even 32 bits per "WORD" )
  int  iBitsPerCodeMemoryWord; // number of bits per code-memory-word
  int  iDataEEPROMSize_bytes;  // size of the device's DATA EEPROM,
                               // measured in BYTES (! ! !) .
  int  iCodeMemWriteLatchSize; // number of *BYTES* (not "words") programmed at a time - only for PIC18F
  int  iCodeMemEraseLatchSize; // number of *BYTES* (not "words") ERASED at a time - only for PIC18F
  int  iConfMemBase; // configuration memory base address,  0x2000 for 16Fxxx, 0xF80000 for dsPIC
  int  iConfWordAdr; // address of configuration WORD,      0x2007 for 16Fxxx, 0xF80000 for dsPIC

  int  iCurrProgAddress; // current program address (seen from PIC device)
                         // READ-ONLY for the plugin .

  //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // More (32-bit) integer parameters ...
  int iProductionGradeProgrammer;
       // 0= this is NOT a production-grade programmer [ = default] .
       //    (which means cannot verify the chip at different supply voltages)
       // 1= this should be a production-grade programmer .
       //    (which can verify the chip at different supply voltages)
       // other values : future reserve

  int iProgrammerHardwareClass;
       // 0=unknown (this is the default value, set by WinPic)
       // 1= use LOW-LEVEL hardware access functions only
       // 2= use MID-LEVEL hardware access functions if possible (*)
       // other values : future reserve

  int iStateOfDataOutWhileReading; // LOGIC(!) state of data-output-line while reading (passive output)
       // usually '1' = HIGH, so the data line can be pulled low by PIC .
       // Hardware-specific; must be set by the plugin during PHW_Init() .

  int iSlowInterface;  // TEST option for hardware troubleshooting :
     // 0=no "slow" interface, may use fast clock pulses   (=default)
     // 1=must use SLOW clock pulses for "slow" interfaces
     // ( in fact, make MOST delay loops longer by a factor of ten ;
     //   but certain delays -like the "externally timed programming cycle"
     //   MUST NOT BE MADE LONGER just because someone designed a bad interface.
     // )

  int iPresentVddSelection;  // current Vdd selection for production grade programmers:
       // 0 = verify or read with the "lowest allowed" supply voltage
       // 1 = verify or read with the "normal" (nominal) supply voltage
       // 2 = verify or read with the "highest allowed" supply voltage

  int iReserved[20-4];  // "reserved" 32-bit integer parameters in the struct

  //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // (64-bit) Floating point parameters ...
  double dblNominalSupplyVoltage;       // usually 5.0 [Volts] for most PICs.  Set by WinPic .
  double dblNominalProgrammingVoltage;  // usually 13.0 [Volts] for most PICs. Set by WinPic .
  double dblReserved[20-2]; // "reserved" double-precision floats (future)


  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Additional information for plugins with an own control panel (window) .
  // Note that all windows-specific types like 'HINSTANCE', 'HWND', etc
  // are replaced by simple DWORDs here to avoid the necessity of including
  // dozens of windows-specific headers before including PHW_intf_dll.h :
  DWORD dwhAppInstance;  // HINSTANCE to application instance (WinPic), set by HOST
  DWORD dwhMainWindow;   // HWND to the application's main window (WinPic), set by HOST
  DWORD dwhPluginDll;    // HMODULE returned by LoadLibrary() when loading the plugin
  DWORD dwhPluginWindow; // handle to the plugin's own window, set by PLUGIN if used
  // Position and size of the plugin's window is initialized by the host,
  // and saved in WinPic's configuration file. But the plugin may overwrite this:
  int iPluginWindowX, iPluginWindowY, iPluginWindowWidth, iPluginWindowHeight;

  void *pInstData; // Pointer to a plugin-dependent "instance" data structure .
     // Simplifies the task of adding "own" data (see PHW_demo_jdm2.c) .
     // T_PHWInfo::pInstData is initialized with NULL before calling
     // PHW_Init().  If the plugin allocates memory for an own "extension"
     // structure, it is responsible to clean up this memory in PHW_Exit(),
     // and set pInstData=NULL to indicate it hasn't forgotten to free it !


} T_PHWInfo;


//--------------------------------------------------------------------------
// Error codes
//--------------------------------------------------------------------------

#define PHW_ERROR_ALL_OK 0         /* no error at all */
#define PHW_ERROR_FUNCTION_FAILED -1 /* general error which doesn't match anything below */
#define PHW_ERROR_ILLEGAL_PARAM   -2
#define PHW_ERROR_VERSION_INCOMPATIBLE -3
#define PHW_ERROR_OUT_OF_MEMORY   -4
#define PHW_ERROR_FILE_ERROR      -5 /* file not found, file not opened, etc */
#define PHW_ERROR_FUNCTION_MISSING -6 /* function not supported or missing (in DLL) */


//--------------------------------------------------------------------------
// Function prototypes
//--------------------------------------------------------------------------

#ifdef __cplusplus__
extern "C" {
#endif /* __cplusplus__ */


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_Init(T_PHWInfo *pInfo);
  /* Initializes the PIC Hardware plugin (module prefix PHW = Programmer HardWare).
   * Called ONCE(!) after loading the plugin . See also: PHW_Exit() .
   *
   *  Input: pInfo = pointer to an "info structure" initialized by the host.
   *         Some members are valid before calling PHW_Init(),
   *         others must/should be set by the plugin in PHW_Init() .
   *         See definition of T_PHWInfo in PHW_intf_dll.h  .
   *  Return value:  one of the PHW_ERROR_.. codes (hopefully PHW_ERROR_ALL_OK)
   */

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_Exit(T_PHWInfo *pInfo);
  /* Called once by the host (WinPic,..) to stop processing, or when terminating.
   * The plugin should free up the resources allocated in PHW_Init(),
   * and -if used- close its own graphic user interface.
   * Return value:  one of the PHW_ERROR_.. codes (hopefully PHW_ERROR_ALL_OK)
   */


//--------------------------------------------------------------------------
API_EXPORT(void) PHW_OnTimer(T_PHWInfo *pInfo);
  /* Periodically called from the main thread (not from a worker thread! ) .
   * In contrast to ProcessSamples(),  OnTimer() may call windows functions,
   * do graphic stuff, etc etc.
   *
   * The calling interval is approximately 200 ms, but DON'T RELY ON THAT :
   *  The source is a windows timer event, which may be slower if the CPU
   *  is heavily loaded.
   */

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_ExecCmd(T_PHWInfo *pInfo,
     char *pszCommand,     // [in]:  command (C string)
     char *psz255Response);// [out]: optional response (C string, up to 255 chars)
  /* OPTIONAL 'command handler' inside the plugin, usable for debugging etc.
   *  The input is a plain C string like "help", "test", "restore", etc .
   *     It can be sent from the interface control panel in WinPic .
   *
   *  Some commands (which act like "functions returning a string")
   *     may place a response string in pszResponse[0..254] .
   *     This response will then be printed into the MESSAGE WINDOW,
   *     so the message window (actually a multi-line edit control)
   *     can work like a tiny debugging console .
   *
   *     Unlike most other exported functions in the plugin, PHW_ExecCmd()
   *     is only optional, which means you don't have to implement it in your
   *     DLL .
   *
   *  Return value:  one of the PHW_ERROR_.. codes (hopefully PHW_ERROR_ALL_OK)
   *          A NEGATIVE VALUE INDICATES AN ERROR,
   *          zero or positive means "ok" .
   *          Some functions return a logic state: 0 = low,  1 = high .
   */



//==========================================================================
//  LOW-LEVEL hardware access functions
//==========================================================================
  // These functions require a fast "bit-banging" access to the programmer,
  // because every single signal transition is initiated by an individual
  // function call.   This is the way most "simple" interfaces for the
  // parallel or the serial port work .

//--------------------------------------------------------------------------
// Low-level OUTPUT functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetVpp(   // Set the programming voltage
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iVppLevel );  // [in]: 0 = turn Vpp off, 1 = set Vpp "high" (13 V) [*]
     // [*] some future devices may require a different programming voltage,
     //     for example 3.3 or 5.0 Volts only .   To be prepared for that, WinPic
     //     specifies the nominal supply voltage (as a floating point value)
     //     in pInfo->dblNominalProgrammingVoltage .


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetVdd(   // Set the "logic" supply voltage
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iVppLevel );  // [in]: 0 = turn Vdd off (0 V), 1 = turn Vpp on (5 V) [*]
     // [*] some future devices may require a different supply voltage,
     //     for example 3.3 Volts .   To be prepared for that, WinPic
     //     specifies the nominal supply voltage (as a floating point value)
     //     in pInfo->dblNominalSupplyVoltage .

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetClockAndData( // Sets the logic state for clock- and data line
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iClockState,  // [in]: 1=serial clock "high", 0=serial clock "low"
            int iDataState);  // [in]: 1=serial data  "high", 0=serial data  "low"
     // One of the most important low-level programming functions .
     //  Used to shift bits "out" (from the programmer to the PIC) .

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetClockEnable(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iNewState );  // [in]: 1=serial clock enabled, 0=serial clock disabled (high-Z)
     // Enables the clock-output-driver (from the programmer to the PIC) .
     // Only used by a few true "in-circuit" programmers.
     // WinPic will call this function to disconnect the programmer
     //  so the serial programming clock can be used as normal I/O-port
     //  from the PIC's point of view .
     // You don't need to implement this function in your DLL
     //   - if WinPic doesn't find it in the DLL, it won't complain .

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetDataEnable(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iNewState );  // [in]: 1=output enabled, 0=output disabled (high-Z)
     // Enables the data-output-driver (from the programmer to the PIC) .
     // Only used by a few true "in-circuit" programmers !
     //  This function is *not* called to clock in data bits while "reading" !
     //  Most "simple" PIC-programmers set the data line '1' = HIGH through
     //  a pull-up resistor, so the data line can be pulled low by the PIC
     //  after a "read-data" command is received .
     //  For this reason, there is no need for SetDataEnable() while programming
     //  and verifying the PIC .
     // WinPic will call this function to disconnect the programmer after
     //  finishing the erase/program/verify cycle, so the serial clock line
     //  can be used as normal I/O-port from the PIC's point of view (after
     //  resetting the target) .
     // You don't need to implement this function in your DLL
     //   - if WinPic doesn't find it in the DLL, it won't complain .


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_PullMclrToGnd(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iNewState );  // [in]: 1=tied to GND,  0=not tied to GND but hi-Z (open)
     // Pulls the PIC's RESET-line to ground, or releases it .
     // This function is usually called before the PIC is switched into
     // programming mode; quickly followed by PHW_SetVpp() .
     // For very exotic interfaces, you can alternatively use the
     // mid-level function

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_ConnectTarget(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iNewState );  // [in]: 1=connected,    0=disconnected,  -1=query state

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetLeds(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iGreenLedState, // [in]: 1=green LED on    0=green LED off
            int iRedLedState);  // [in]: 1=red LED on      0=red LED off


//--------------------------------------------------------------------------
// Low-level INPUT functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_GetDataInBit(
  T_PHWInfo *pInfo ); // [in]: pointer to structure with "additional" info
  // INPUT function, samples the state of serial data from PIC to PC.
  //  Returns 1=HIGH or 0=LOW, everything else indicates an error .

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_GetOkButton(
  T_PHWInfo *pInfo ); // [in]: pointer to structure with "additional" info
  // INPUT function, reads the state of the optional "OK"-button on the programmer.
  // Returns 1="button pressed" or 0="button not pressed".
  // Everything else indicates an error or "function not supported" .




//---------------------------------------------------------------------------
// Low-level "auxiliary" functions  (added 2008-05-19)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
API_EXPORT(int) PHW_FlushCommand(
  T_PHWInfo *pInfo ,  // [in]: pointer to structure with "additional" info
  DWORD dwCommand,    // [in]: command-code which has just been sent; usually 6 bits (for 14-bit core)
  DWORD dwNumCommandBits, // [in]: number of bits for the command-code, usually 6 (!)
  DWORD dwData,       // [in]: optional DATA which have been sent along with the command; usually 14 bits
  DWORD dwNumDataBits);// [in]: number of DATA-BITS which have been sent; usually 14 (for 14-bit core)
  // PHW_FlushCommand() may be required for some "intelligent" programmers
  //     like USB-interfaces, which collect all clock- and data transitions
  //     in a single packet (of a few dozen bytes),
  // for example using a single USB BULK TRANSFER instead of sending each
  // clock- or data-transition in a single transfer (which would be incredibly
  // slow, depending on the USB hardware).
  // When the ..._FlushCommand() is issued, the PIC hardware driver knows
  // that it's time to send the previously collected block to the programmer.
  // Added 2008-05-19 by WB .


//==========================================================================
//  MID-LEVEL hardware access functions
//==========================================================================
  // These functions require a "smart" programming interface, with an own
  // microcontroller. Such interfaces use the serial port (or USB) much more
  // efficient than the "simple" interfaces (which are entirely driven
  // through the "LOW-LEVEL" functions listed in the previous section) .
  // The downside of the MID-LEVEL functions is, they need to know a lot
  // about the device to be programmed, because the bit timing must be
  // realized by the microcontroller in the "smart" programmer.
  // For this purpose, WinPic copies the required timing parameters into
  // the PIC-hardware-info-structure (pInfo) .  So your hardware driver
  // and access them whenever necessary, and send them to the microcontroller
  // when necessary. For example, "a serial clock bit must be at least 5
  // microseconds long", etc etc .
  //
  //  To let WinPic know if the "low-level" or "mid-level" hardware access
  //  functions should be used, set pInfo->


#ifdef __cplusplus__
}
#endif /* __cplusplus__ */


#endif /* _PHW_INTF_DLL_H_ */
