//---------------------------------------------------------------------------
// File  :  PHW_demo_jdm2.c
// Date  :  2008-05-06   (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
// Location: <DL4YHF>C:\cbproj\WinPicPr\interface_dll_sources\PHW_demo_jdm2.h

//
// Description:
//   Sample hardware-interface plugin for WinPic /   "JDM2" .
//   Compile with Dev-Cpp or Borland C++ Builder V4;
//           MS-C may work but never tested.
//

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

#include "PHW_intf_dll.h"   /* header for all PIC-Hardware-Interface-DLLs */
#include "PHW_framework.h"  /* some framework for the FFT-filter plugin   */

//---------------------------------------------------------------------
// Compilation switches
//---------------------------------------------------------------------

#ifndef  SWI_PLUGIN_WITH_GUI  /* shall the plugin have its own control panel ? */
 #define SWI_PLUGIN_WITH_GUI 1   /* 0=no, 1=yes */
#endif

//---------------------------------------------------------------------
// Internal defines
//---------------------------------------------------------------------

#if(SWI_PLUGIN_WITH_GUI)  // If the plugin has it's own "GUI" (control panel):
 // Define a *unique*  class name for the plugin's own window !
 // C_PLUGIN_WINDOW_CLASS = name of the window class; max. 40 characters .
 // To avoid conflicts with the "window class name" when running two plugins
 // compiled with different compilers at the same time, we use different names
 // for the plugin, depending on the development environment.
 // Without this, RegisterClassEx() would fail (see PHW_frameworkX.c) .
 #ifdef __BORLANDC__     /* Borland C, Borland C++Builder : */
   #define C_PLUGIN_WINDOW_CLASS "PIC Hardware Interface for JDM2 BC"
 #endif
 #ifdef __LCC__          /* lcc C compiler system for Win32: */
   #define C_PLUGIN_WINDOW_CLASS "PIC Hardware Interface for JDM2 LCC"
 #endif
 #ifdef __GNUC__         /* GNU C compiler, including MinGW or DevCpp */
   #define C_PLUGIN_WINDOW_CLASS "PIC Hardware Interface for JDM2 GCC"
 #endif
 #ifndef C_PLUGIN_WINDOW_CLASS
  #define C_PLUGIN_WINDOW_CLASS "PIC Hardware Interface for JDM2" /* unique name, max. 40 chars */
 #endif
#endif // (SWI_PLUGIN_WITH_GUI)

//---------------------------------------------------------------------
// Internal data types
//---------------------------------------------------------------------

typedef struct // extra filter info / internal values for each instance of this plugin.
{ //  A pointer to this structure 'extends' the
  //  T_PHWInfo structure (.pExtra) for everything
  //  which we need in the processing callbacks,
  //  but don't want to have in a global variable .
  char   sz80WindowTitle[84]; // text which shall appear in the plugin's window title
} T_MyInstData;
#define THIS_INST ((T_MyInstData*)pInfo->pInstData)

//---------------------------------------------------------------------
// Internal variables
//---------------------------------------------------------------------

  /* none ! */



//---------------------------------------------------------------------
// Internal functions
//---------------------------------------------------------------------

/***************************************************************************/
void PHW_PrintErrorString( T_PHWInfo *pInfo, char *pszFormat, ... )
  /* Prints a string into the multiline message window
   *  on the filter Control Panel (inside WinPic).
   *  Caution: don't call too often, because WinPic only polls a flag
   *  every 100 milliseconds (approx).
   */
{
  va_list parameter;
  va_start( parameter, pszFormat/*lastfix*/ );
  wvsprintf( pInfo->sz255ErrorText, pszFormat, parameter );
  va_end(parameter);
  ++pInfo->iErrorTextModifyCnt; // acts as a "signal" for the filter c.p.
} // end PHW_PrintErrorString()




//---------------------------------------------------------------------
// Helper functions for the control panel
//---------------------------------------------------------------------

#if (SWI_PLUGIN_WITH_GUI)
//---------------------------------------------------------------------
void ClearScreen( T_PHWInfo *pInfo )  // clears the client area
{
 HDC  hdc;
 RECT rct;

  if( pInfo->dwhPluginWindow != 0 )
   { // If the plugin's own control window exists, clear the screen:
     hdc = GetDC( (HWND)pInfo->dwhPluginWindow );
     if( hdc != NULL )
      {
        GetClientRect( (HWND)pInfo->dwhPluginWindow, &rct );
        FillRect( hdc, &rct, GetStockObject(WHITE_BRUSH) );
        ReleaseDC( (HWND)pInfo->dwhPluginWindow, hdc ); // never forget !
      } // end if < valid device context for drawing >
   }
} // end ClearScreen()
#endif // SWI_PLUGIN_WITH_GUI

#if (SWI_PLUGIN_WITH_GUI)
//---------------------------------------------------------------------
void PrintToDC( HDC hdc, int x, int y, char *str )
  // Prints a null-terminated C-string to the graphic device context .
  // x or y < 0 means "don't change, use current position from previous call"
{
  static int cursx = 0;
  static int cursy = 0;
  SIZE tsize;
  int slen=strlen(str);
  if(x<0) x=cursx;
  if(y<0) y=cursy;
  TextOut( hdc, x, y, str, slen );
       // Note: if DevCpp throws an error like
       //   "[Linker error] undefined reference to 'TextOutA@20',
       //  add the "libgdi32.a" library to the project !
       // DevCpp doesn't include that lib automatically
       //  because the project started as a "Win32 DLL",
       //  not a "Win32 GUI Application" . This may apply to
       //  other linker errors too, so DON'T REMOVE THIS COMMENT .
  cursx = x;
  cursy = y;
  if( GetTextExtentPoint( hdc, str, slen, &tsize ) )
   { cursx += tsize.cx;
   }
} // end Print2DC()
#endif // SWI_PLUGIN_WITH_GUI



//---------------------------------------------------------------------
// Optional event handlers, called from PHW_framework :
//---------------------------------------------------------------------
#if (SWI_PLUGIN_WITH_GUI)
//---------------------------------------------------------------------
int PHW_OnKeyDown( T_PHWInfo *pInfo, int nVirtKey, long lKeyData )
  // Called when a key was pressed in the plugin's own window (WM_KEYDOWN) .
{

  pInfo=pInfo;          // to suppress 'unused'-warnings
  nVirtKey=nVirtKey;
  lKeyData=lKeyData;

  switch( nVirtKey )    // often upper-case ASCII, or VK_... from WINUSER.H
   {
     case 'C':  // clear the plot-screen
        ClearScreen( pInfo );
        break; // end case 'c'
     default:
        break;
   } // end switch(key)

  return 0;  // return zero if application processes the WM_KEYDOWN message .
} // end PHW_OnKeyDown()
#endif // SWI_PLUGIN_WITH_GUI



//---------------------------------------------------------------------
// Exported DLL functions
//---------------------------------------------------------------------

//---------------------------------------------------------------------
API_EXPORT(int) PHW_Init(T_PHWInfo *pInfo)
  /* Initializes the Filter Plugin (module prefix FFP = Fft Filter Plugin).
   * Called ONCE(!) after loading the plugin . See also: PHW_Exit() .
   *
   *  Input: pInfo = pointer to an "info structure" initialized by the host.
   *         Some members are valid before calling PHW_Init(),
   *         others must/should be set by the plugin in PHW_Init() .
   *         See definition of T_PHWInfo in PHW_intf_dll.h  .
   *  Return value:  one of the PHW_ERROR_.. codes (hopefully PHW_ERROR_ALL_OK)
   */
{
  BOOL fRestoreDefaults = FALSE;
  int  i;

  // Compatibility check.. see ..\fft_filter_plugins\readme.txt !
  i = sizeof(T_PHWInfo);  // place in a variable to see it in the debugger..
  if( pInfo->iSizeOfStruct != i )
   { // big problem: The plugin is INCOMPATIBLE with WinPic,
     //  because the size of the plugin's T_PHWInfo structure
     //  doesnt match what the host expects -> refuse to work !
     return PHW_ERROR_VERSION_INCOMPATIBLE;
   }

  // Fill some fields of the plugin info structure:
  strcpy(pInfo->sz255InfoText,"WinPic demo plugin for \"JDM2\"");
  pInfo->iProductionGradeProgrammer = 0; // 0: this is NOT a production-grade programmer
  pInfo->iProgrammerHardwareClass   = 1; // 1: use LOW-LEVEL hardware access functions only
  pInfo->iStateOfDataOutWhileReading = 1; // data output must be set HIGH while reading


  // Allocate an "extra" part which we need to pass to all subroutines
  //  (which would be METHODS in a C++ world; and pInfo would be 'this') :
  pInfo->pInstData = (void*)malloc( sizeof(T_MyInstData) );
  if( pInfo->pInstData == NULL )
   { return PHW_ERROR_OUT_OF_MEMORY; // can this really happen under windoze ?
   }

#if(SWI_PLUGIN_WITH_GUI)  // OPTIONAL : Plugin with it's own, small control panel.
  // The text which appears in the window title :
  sprintf( THIS_INST->sz80WindowTitle, "PIC-HW Plugin Demo" );
  PHW_OpenControlPanel( pInfo, THIS_INST->sz80WindowTitle );
#endif // SWI_PLUGIN_WITH_GUI


  return PHW_ERROR_ALL_OK;
} // end PHW_Init()



#ifdef __BORLANDC__
 #pragma argsused   // Borland gripes "never used" - so what  ;-)
#endif

//---------------------------------------------------------------------
API_EXPORT(int) PHW_Exit(T_PHWInfo *pInfo)
  /* Called once by the host (WinPic,..) to stop processing, or when terminating.
   * The plugin should free up the resources allocated in PHW_Init(),
   * and -if used- close its own graphic user interface.
   * Return value:  one of the PHW_ERROR_.. codes (hopefully PHW_ERROR_ALL_OK)
   */
{

  if( pInfo->iSizeOfStruct == sizeof(T_PHWInfo) )
   {
     PHW_DestroyControlPanel(pInfo); // destroy the plugin's control panel (if existing)
     if( pInfo->pInstData != NULL )
      {  free( pInfo->pInstData );   // free "extra" instance data if we allocated some
         pInfo->pInstData = NULL;
      }
     return PHW_ERROR_ALL_OK;
   }
  else
   { return PHW_ERROR_VERSION_INCOMPATIBLE;
   }

} // end PHW_Exit()




//--------------------------------------------------------------------------
API_EXPORT(void) PHW_OnTimer(T_PHWInfo *pInfo)
  /* Periodically called from the main thread (not from a worker thread! ) .
   * In contrast to other functions in this plugin, PHW_OnTimer() may 
   * call windows functions,  do graphic stuff, etc .
   *
   * The calling interval is approximately 200 ms, but DON'T RELY ON THAT :
   *  The source is a windows timer event, which may be slower if the CPU
   *  is heavily loaded.
   */
{

  // .. place your own code here ..
  //  (if your plugin needs to do something periodically,
  //   thread-safe called from WinPic's main thread)
  int  x,y, max_x, max_y;
  char sz80Temp[84];
  HDC  hdc;           // 'device context' for windows graphic functions
  RECT rct;

  if( pInfo->dwhPluginWindow != 0 )
   { // If the plugin's own control window is visible, show some info there:
    hdc = GetDC( (HWND)pInfo->dwhPluginWindow );
    if( hdc != NULL )
     {
       GetClientRect( (HWND)pInfo->dwhPluginWindow, &rct );
       max_x = rct.right-1;
       max_y = rct.bottom-1;
       // Clear old garbage ? Better not; the screen may flicker .
       // Use trailing spaces in this ultra-simple demo to remove old stuff
       // if the "new" string displayed in a line is shorter than the "old" .

       // Show something in the tiny control window ...
       sprintf(sz80Temp, "Hardware : JDM2" );
       PrintToDC( hdc, 0/*x*/, 0/*y*/, sz80Temp );
       sprintf(sz80Temp, "Compiled : %s", __DATE__ );
       PrintToDC( hdc, 0/*x*/, 16/*y*/, sz80Temp );
       sprintf(sz80Temp, "Dev : %s      ", pInfo->sz63ProgrammedDeviceName );
       PrintToDC( hdc, 0/*x*/, 32/*y*/, sz80Temp );
       sprintf(sz80Temp, "Addr: 0x%04lX ", (long)pInfo->iCurrProgAddress );
       PrintToDC( hdc, 0/*x*/, 48/*y*/, sz80Temp );
       PrintToDC( hdc, 0, 80, "[C]lear" );
       ReleaseDC( (HWND)pInfo->dwhPluginWindow, hdc ); // never forget !
     } // end if < got a valid device context >
   } // end if < plugin's own control window exists >

} // end  PHW_OnTimer()

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_ExecCmd(T_PHWInfo *pInfo,
     char *pszCommand,     // [in]:  command (C string)
     char *psz255Response) // [out]: optional response (C string, up to 255 chars)
  /* OPTIONAL 'command handler' inside the plugin, usable for debugging etc.
   *  Return value:  one of the PHW_ERROR_.. codes (hopefully PHW_ERROR_ALL_OK)
   *  Full documentation in fft_filter_dll.h  .
   */
{
  const char *pszCmds[] = { "about", "help", "restore", "show", "test", "\0" };
  int i;
  int iResult = PHW_ERROR_ALL_OK;
  for(i=0; pszCmds[i][0]>'\0'; ++i)
   { if(stricmp(pszCommand, pszCmds[i])==0) break;
   }
  switch(i)
   { case 0:  // "about"
        // wvsprintf(psz255Response, "FFT Filter Plugin demo, compiled %s", __DATE__ );
        // wvsprintf crashed here, but sprintf worked - problems with STRINGS ?
        sprintf(psz255Response, "FFT Filter Plugin demo, compiled %s", __DATE__ );
        break;
     case 1:  // "help":
        strcpy( psz255Response, "Commands:" );
        for(i=0; pszCmds[i][0]>'\0'; ++i)
         { if(stricmp(pszCommand, pszCmds[i])==0) break;
           strcat( psz255Response, " " );
           strcat( psz255Response, pszCmds[i] );
         }
        break;
     case 2:  // "restore"
        strcpy( psz255Response, "ok." );
        break;

     case 3:  // "show"  :  command to SHOW the window (if someone closed it)
#if(SWI_PLUGIN_WITH_GUI)  // OPTIONAL : Plugin with it's own, small control panel.
        PHW_OpenControlPanel( pInfo, THIS_INST->sz80WindowTitle );
#endif // SWI_PLUGIN_WITH_GUI
        break;

     case 4:  // "test"
        break;

     // ... add other 'interpreter commands' for your filter plugin here ...

     default:
        strcpy( psz255Response, "unknown command" );
        iResult = PHW_ERROR_FUNCTION_MISSING;
        break;
   }
  return iResult;
} // end PHW_ExecCmd()


//==========================================================================
//  LOW-LEVEL hardware access functions
//==========================================================================
  // These functions require a fast "bit-banging" access to the programmer,
  // because every single signal transition is initiated by an individual
  // function call.   This is the way most "simple" interfaces for the
  // parallel or the serial port work .

//--------------------------------------------------------------------------
// Low-level OUTPUT functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetVpp(   // Set the programming voltage
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iVppLevel )   // [in]: 0 = turn Vpp off, 1 = set Vpp "high" (13 V) [*]
                              //       2 = set Vpp to Vpp (5 V) if possible
     // [*] some future devices may require a different programming voltage,
     //     for example 3.3 or 5.0 Volts only .   To be prepared for that, WinPic
     //     specifies the nominal supply voltage (as a floating point value)
     //     in pInfo->dblNominalProgrammingVoltage .
{
  return PHW_ERROR_ALL_OK;
}


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetVdd(   // Set the "logic" supply voltage
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iVppLevel )   // [in]: 0 = turn Vdd off (0 V), 1 = turn Vpp on (5 V) [*]
     // [*] some future devices may require a different supply voltage,
     //     for example 3.3 Volts .   To be prepared for that, WinPic
     //     specifies the nominal supply voltage (as a floating point value)
     //     in pInfo->dblNominalSupplyVoltage .
{
  return PHW_ERROR_ALL_OK;
}


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetClockAndData( // Sets the logic state for clock- and data line
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iClockState,  // [in]: 1=serial clock "high", 0=serial clock "low"
            int iDataState)   // [in]: 1=serial clock "high", 0=serial clock "low"
     // One of the most important low-level programming functions .
     //  Used to shift bits "out" (from the programmer to the PIC) .
{
  return PHW_ERROR_ALL_OK;
}


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetClockEnable(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iNewState )   // [in]: 1=serial clock enabled, 0=serial clock disabled (high-Z)
     // Enables the clock-output-driver (from the programmer to the PIC) .
     // Only used by a few true "in-circuit" programmers.
     // WinPic will call this function to disconnect the programmer
     //  so the serial programming clock can be used as normal I/O-port
     //  from the PIC's point of view .
     // You don't need to implement this function in your DLL
     //   - if WinPic doesn't find it in the DLL, it won't complain .
{
  return PHW_ERROR_ALL_OK;
}


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetDataEnable(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iNewState )   // [in]: 1=output enabled, 0=output disabled (high-Z)
     // Enables the data-output-driver (from the programmer to the PIC) .
     // Only used by a few true "in-circuit" programmers !
     //  This function is *not* called to clock in data bits while "reading" !
     //  Most "simple" PIC-programmers set the data line '1' = HIGH through
     //  a pull-up resistor, so the data line can be pulled low by the PIC
     //  after a "read-data" command is received .
     //  For this reason, there is no need for SetDataEnable() while programming
     //  and verifying the PIC .
     // WinPic will call this function to disconnect the programmer after
     //  finishing the erase/program/verify cycle, so the serial clock line
     //  can be used as normal I/O-port from the PIC's point of view (after
     //  resetting the target) .
     // You don't need to implement this function in your DLL
     //   - if WinPic doesn't find it in the DLL, it won't complain .
{
  return PHW_ERROR_ALL_OK;
}


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_PullMclrToGnd(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iNewState )   // [in]: 1=tied to GND,  0=not tied to GND but hi-Z (open)
     // Pulls the PIC's RESET-line to ground, or releases it .
     // This function is usually called before the PIC is switched into
     // programming mode; quickly followed by PHW_SetVpp() .
{
  return PHW_ERROR_ALL_OK;
}


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_ConnectTarget(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iNewState )   // [in]: 1=connected,    0=disconnected,  -1=query state
{
  return PHW_ERROR_ALL_OK;
}


//--------------------------------------------------------------------------
API_EXPORT(int) PHW_SetLeds(
            T_PHWInfo *pInfo, // [in]: pointer to structure with "additional" info
            int iGreenLedState, // [in]: 1=green LED on    0=green LED off
            int iRedLedState)   // [in]: 1=red LED on      0=red LED off
{
  return PHW_ERROR_ALL_OK;
}


//--------------------------------------------------------------------------
// Low-level INPUT functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_GetDataInBit(
  T_PHWInfo *pInfo ) // [in]: pointer to structure with "additional" info
  // INPUT function, samples the state of serial data from PIC to PC.
  //  Returns 1=HIGH or 0=LOW, everything else indicates an error .
{
  return 1;
} // end PHW_GetDataInBit()

//--------------------------------------------------------------------------
API_EXPORT(int) PHW_GetOkButton(
  T_PHWInfo *pInfo ) // [in]: pointer to structure with "additional" info
  // INPUT function, reads the state of the optional "OK"-button on the programmer.
  // Returns 1="button pressed" or 0="button not pressed".
  // Everything else indicates an error or "function not supported" .
{
  return 0;
} // end  PHW_GetOkButton()


//---------------------------------------------------------------------------
API_EXPORT(int) PHW_FlushCommand(
  T_PHWInfo *pInfo ,  // [in]: pointer to structure with "additional" info
  DWORD dwCommand,    // [in]: command-code which has just been sent; usually 6 bits (for 14-bit core)
  DWORD dwNumCommandBits, // [in]: number of bits for the command-code, usually 6 (!)
  DWORD dwData,       // [in]: optional DATA which have been sent along with the command; usually 14 bits
  DWORD dwNumDataBits) // [in]: number of DATA-BITS which have been sent; usually 14 (for 14-bit core)
  // PHW_FlushCommand() may be required for some "intelligent" programmers
  //     like USB-interfaces, which collect all clock- and data transitions
  //     in a single packet (of a few dozen bytes),
  // for example using a single USB BULK TRANSFER instead of sending each
  // clock- or data-transition in a single transfer (which would be incredibly
  // slow, depending on the USB hardware).
  // When the ..._FlushCommand() is issued, the PIC hardware driver knows
  // that it's time to send the previously collected block to the programmer.
  // Added 2008-05-19 by WB .
{
  return 0;
} // end  PHW_FlushCommand()



#ifdef __BORLANDC__
 #pragma argsused   // Borland gripes "never used" - so what  ;-)
#endif

//---------------------------------------------------------------------
// DLL main entry point.
//   Not required by WinPic, but possibly important for your DLL.
//
// Borland C++Builder users:
//   Read notes about 'DllMain' versus 'DllEntryPoint'
//   in the automagically created 'main module' which unfortunately
//   cannot be removed from the Borland DLL project .
//---------------------------------------------------------------------

//---------------------------------------------------------------------
BOOL APIENTRY DllMain(
    HINSTANCE hInst,   // [in]: Library instance handle
    DWORD reason,      // [in]: Reason this function is being called
    LPVOID reserved )  // [in]: Not used.
{
  switch (reason)
    {
      case DLL_PROCESS_ATTACH:   // Host called 'LoadLibrary'
        ++FPHW_iDllInstanceCount;  // needs to be in SHARED MEMORY to make sense!
        break;

      case DLL_PROCESS_DETACH:   // Host called 'LoadLibrary'
        --FPHW_iDllInstanceCount;
        break;

      case DLL_THREAD_ATTACH: // called for each new THREAD the host creates,
        // even if that thread hasn't got anything to do with this DLL.
        // For that reason, DLL_THREAD_ATTACH is totally useless here .
        break;

      case DLL_THREAD_DETACH:
        break;
    }

    /* Returns TRUE on success, FALSE on failure */
    return TRUE;
}

/* EOF < PHW_demo_jdm2.c > */
