/*------------------------------------------------------------------*/
/* File: C:\cbproj\rsNTP\WSJTX_Sync.c                               */
/*                                                                  */
/* Purpose:                                                         */
/* DL4YHF's addition for the 'ridiculously simple NTP client'       */
/*          to synchronize the PC's system time based on the output */
/*          of K1JT/G4WJS/K9AN'S "WSJT-X" application, in FT8 mode. */
/*       Details and an explanation of the principle further below. */
/*       Doesn't work with "modern" Windows versions anymore,       */
/*       and doesn't work with recent version of WSJT-X anymore,    */
/*       but since there was so much effort put in this module,     */
/*       keep it for the future (to detect OTHER windows            */
/*       and retrieve text from them - if the paranoid OS allows.)  */
/*                                                                  */
/*------------------------------------------------------------------*/

/* FT8 decoder output sample, with a reasonably good system time,   */
/* copied and pasted from WSJT-X's "Band Activity" window :         */
/* UTC  dB  DT  Freq    Message

121130  15  0.2 1704 ~  CQ DM9EE JO41
121130  -1  0.1 1917 ~  SP6LTM DL5OAS JO42
121130 -21  0.0 1772 ~  CQ SP1O JO73
121145  -8  0.1 2121 ~  CQ DX PI4DR JO32
121145  -9 -0.9 2012 ~  <...> DL30SHZ
121200  19  0.2 1705 ~  CQ DM9EE JO41
121200   2  0.1 1920 ~  SP6LTM DL5OAS JO42
121200 -17 -1.3 1970 ~  PI4DR DM2CF JO60
121200 -18  0.0 1769 ~  CQ SP1O JO73
121215 -11 -0.9 2013 ~  <...> DL30SHZ
121230  15  0.2 1705 ~  CQ DM9EE JO41
121230  -7 -1.3 1788 ~  DG3TF DM2CF JO60
121230 -20  0.0 1765 ~  CQ SP1O JO73
121245 -11 -0.9 2010 ~  <...> DL30SHZ
121245 -14  0.1 1939 ~  CQ DX PI4DR JO32
121245 -19  0.1 1787 ~  CQ DG3TF JO53
121300   9  0.2 1706 ~  CQ DM9EE JO41
121300 -11 -1.3 1788 ~  DG3TF DM2CF JO60
121300 -23 -0.5 2010 ~  <DL30SHZ> SP6URZ -06
121315  -9 -0.9 2016 ~  SP6URZ <DL30SHZ> R-19
121330  18  0.2 1706 ~  CQ DM9EE JO41
121345 -14  0.1 1970 ~  CQ DX PI4DR JO32
121345 -15  0.5 1046 ~  SP1O DC5HB JO53
121400  13  0.2 1707 ~  CQ DM9EE JO41
121415 -13  0.1 1971 ~  CQ DX PI4DR JO32
121415 -10 -0.9 2012 ~  SP6URZ <DL30SHZ> R-19
121430  16  0.2 1707 ~  CQ DM9EE JO41
121445  -5 -0.9 2013 ~  SP6URZ <DL30SHZ> R-19
121445 -15  0.1 1970 ~  CQ DX PI4DR JO32
121445 -21 -0.5 1717 ~  YO9XHN IY6KLX R 7C KY
121500  16  0.2 1708 ~  CQ DM9EE JO41
121515  -3 -0.9 2013 ~  SP6URZ <DL30SHZ> R-19
121515 -15 -1.3 1710 ~  DM9EE DM2CF JO60
121515 -15  0.1 1970 ~  CQ PI4DR JO32
121515 -20  0.5 1046 ~  SP1O DC5HB JO53
121530  15  0.2 1708 ~  CQ DM9EE JO41
121545   0 -0.9 2012 ~  SP6URZ <DL30SHZ> R-19
121545 -15  0.1 1970 ~  CQ PI4DR JO32
121600  16  0.2 1708 ~  CQ DM9EE JO41
121600 -19 -0.7 2010 ~  SHZ LOCATOR?
121615  -5 -1.3 1709 ~  DM9EE DM2CF JO60
121615  -3 -0.9 2012 ~  SP6URZ <DL30SHZ> R-19
121615 -13  0.1 1970 ~  CQ PI4DR JO32
121615 -21  0.5 1046 ~  SP1O DC5HB JO53
121630  15  0.2 1708 ~  CQ DM9EE JO41
121645 -12 -0.9 2010 ~  SP6URZ <DL30SHZ> R-19
121645 -11  0.1 1970 ~  CQ PI4DR JO32
121645 -14 -1.3 1710 ~  DM9EE DM2CF JO60

The above is a typical example: MOST FT8 users have their system times
nicely set, only a few are 'quite far off'. We'll try to solve this problem
as follows (if there are a sufficient number of decodes in the RX text window):
 (1) First calculate the AVERAGE 'DT' value from all lines, e.g.:
     ( 0.2 + 0.1 + 0.0 + 0.1 - 0.9 + 0.2 + 0.1 - 1.3 + 0.0 - 0.9 + 0.2
     - 1.3 + 0.0 - 0.9 + 0.1 + 0.1 + 0.2 - 1.3 - 0.5 - 0.9 + 0.2 + 0.1
     + 0.5 + 0.2 + 0.1 - 0.9 + 0.2 - 0.9 + 0.1 - 0.5 + 0.2 - 0.9 - 1.3
     + 0.1 + 0.5 + 0.2 - 0.9 + 0.1 - 0.2 - 0.7 - 1.3 - 0.9 + 0.1 + 0.5
     + 0.2 - 0.9 + 0.1 - 1.3  ) / 48 = -0.291667  [seconds, AVERAGE]

 (2) Ignore 50 percent(!) of the available decodes (in this example, 48/2),
     and take only those (50 %) decodes into account that are closest
     to the AVERAGE. For this purpose, sort the array of 'DT'-values
     by their absolute (sign-less) difference to the average from step 1.
     Instead of doing this by hand here, circa 20 decodes were removed from
     the list, beginning with the 'most grossly off-time stations' :
     ( 0.2 + 0.1 + 0.0 + 0.1       + 0.2 + 0.1       + 0.0       + 0.2
           + 0.0       + 0.1 + 0.1 + 0.2                   + 0.2 + 0.1
           + 0.2 + 0.1       + 0.2       + 0.1       + 0.2
     + 0.1       + 0.2       + 0.1 - 0.2                   + 0.1
     + 0.2       + 0.1        ) / 26 = 0.115385  [seconds, REVISED AVERAGE]

     [When these sample data were collected, the PC's system time was less
      than 30 milliseconds off.  The 'DT' values reported by WSJT-X seem to
      be  POSITIVE if the decoding PC's system time is AHEAD of the correct time,
      and NEGATIVE if the decoding PC'S system time is LATE (too low).
      Thus :  'DT' = Decoding PC's system time MINUS 'correct' UTC .   ]

 As a first step, the entire text had to be copied and pasted
    from WSJT-X's 'Band Activity' window with a few clicks or keystrokes:
      (1) click into WSJT-X's "Band Activity" to set the keyboard focus there.
      (2) press CTRL-A to select "all"
      (3) press CTRL-C to copy the selected text into the windows clipboard
      (4) switch to rsNTP's preliminary "Sync from WSJT-X" window / editor
      (5) press CTRL-V to paste the text from the clipboard (with a few dozen FT8-"decodes")
      (6) click 'Synchronize' to let this program (module WSJTX_Sync.c)
          parse the decoder output, and calculate the statistics as mentioned
          further above.
 Not very user friendly. But how to eliminate steps 1 to 5 shown above ?
 If we knew the (window-) handle of WSJT-X'es "Band Activity" display, yes.
 Without wading through the WSJT-X sources (including the GUI written in QT):
 Using the tool 'inspect.exe' (part of the Windows SDK / "Windows Kits", installed
 under Windows 8.1 in C:\Program Files (x86)\Windows Kits\8.1\bin\x64\inspect.exe),
 the WSJT-X main window name was indeed "WSJT-X   v2.2.2   by K1JT, G4WJS, and K9AN"
  - note the lot of spaces, and the name that will be subject to change
    with each new version.
 See WSJTX_Sync_FindHandleOfDecoderOutputWindow() .
*/


#include "WSJTX_Sync.h" // header file for THIS module (DL4YHF's "WSJT-X Synchronizer")
#include <stdio.h>      // of course no "standard I/O" used here, but sprintf()

typedef struct t_GUITreeNode
{
  HWND hWnd;                // window handle of this thing
  char sz511Title[512];     // window title (aka 'window text' if we could retrieve it)
  char sz511ClassName[512]; // window class name (if we could retrieve one)
  struct t_GUITreeNode *pParent;      // pointers to other tree elements ("nodes")...
  struct t_GUITreeNode *pPrevSibling; // NULL when not existing / not enumerated yet
  struct t_GUITreeNode *pNextSibling;
  struct t_GUITreeNode *pFirstChild;
  // Instead of dynamically allocating T_GUITreeNode, a T_EnumWindowsHelper
  // contains an ARRAY of them. T_EnumWindowsHelper.nNodesUsed counts them.
} T_GUITreeNode;
#define WSJTX_SYNC_MAX_TREE_NODES 100

typedef struct t_EnumWindowsHelper
{ // Struct passed as a 'void pointer' to various Windows-Enumeration-callbacks
   int iOptions; // bit combination with e.g. WSJTX_SYNC_OPTION_VERBOSE_OUTPUT,
                 // passed on to the various windows-enumeration-callbacks
                 // from WSJTX_Sync_FindHandleOfDecoderOutputWindow(), etc.
   void (*pvInfoStringOuput)(char *cpInfo); // optional pointer to a procedure to 'print info strings'.
                 // NULL if the caller (e.g. 'rsNTP') doesn't want to print info.
   HWND hwndDecoderOutputWindow; // null as long as 'not found'

   // The following list of 'already enumerated window' is also used to avoid
   // crashing in an endless loop, for example if a certain window (-handle)
   // appears as its own SIBLING or CHILD, etc etc.
   T_GUITreeNode nodes[WSJTX_SYNC_MAX_TREE_NODES];
   int  nNodesUsed;             // index into nodes[] for pseudo-dynamic allocation.
   T_GUITreeNode *pCurrentNode;  // "where to append new CHILDREN"

} T_EnumWindowsHelper;

HWND WSJTX_Sync_hwndMainWindow = 0;   // handle to WSJT-X's *main window* (0 when invalid)
char WSJTX_Sync_sz511Title[512] = ""; // complete name (aka "title") of WSJT-X's *main window*
int  WSJTX_Sync_iRecursionDepth= 0;   // kludge to prevent endless recursion,
        // for example when trying to enumerate "windows" (windowed controls)
#define MAX_RECURSION_DEPTH 10   // maximum nesting level for all those ugly
                                 // EnumXYZWindows() calls .
T_EnumWindowsHelper WSJTX_EnumWindowsHelper; // <- global to inspect via debugger

//---------------------------------------------------------------------------
static BOOL IsKnownWindowHandle( T_EnumWindowsHelper *pEnumHelper, HWND hWnd )
   // Returns TRUE if a certain window (identified by its 'handle')
   // has already been enumerated (and stored in T_EnumWindowsHelper.node[]).
{ int i;
  for(i=0; (i<pEnumHelper->nNodesUsed) && (i<WSJTX_SYNC_MAX_TREE_NODES); ++i )
   { if( hWnd == pEnumHelper->nodes[i].hWnd )
      { return TRUE;
      }
   }
  // Arrived here ? Seems to be a window handle that hasn't been "enumerated" yet.
  return FALSE;

} // end IsKnownWindowHandle()


//---------------------------------------------------------------------------
static BOOL CALLBACK MyEnumWindowsProc( // our callback for EnumWindows() [not EnumChildWindows()]
       HWND hWnd,      // [in] handle to a top-level window
       LPARAM lParam)  // application-defined value ( actually a T_EnumWindowsHelper* )
{
  // Fasten seat belts .. we're diving deep into the ancient Win32 API again !
  // > The GetWindowTextLength function retrieves the length, in characters,
  // > of the specified window's title bar text (if the window has a title bar).
  // > If the specified window is a control, the function retrieves the length
  // > of the text within the control.
  int  iLength = GetWindowTextLength(hWnd);
  char sz511Title[512], sz1kOutput[1024], *cp;
  T_EnumWindowsHelper *pEnumHelper = (T_EnumWindowsHelper*)lParam;
  T_GUITreeNode *pTreeNode = NULL;

  if( IsKnownWindowHandle( pEnumHelper, hWnd ) )
   { // Oops.. already saw this window, so ignore this call !
     return TRUE;
   }

  if( (iLength>0) && (iLength<511) )
   { // > The GetWindowText function copies the text of the specified window's
     // > title bar (if it has one) into a buffer.
     // > If the specified window is a control, the text of the control is copied.
     // > This function cannot retrieve the text of an edit control
     // > in another application.
     // > To retrieve the text of a control in another process, send a
     // > WM_GETTEXT message directly instead of calling GetWindowText."
     GetWindowText(hWnd, (LPTSTR)sz511Title, iLength + 1);

     if( (pEnumHelper->pvInfoStringOuput != NULL) // show what's going on ?
      && (pEnumHelper->iOptions & WSJTX_SYNC_OPTION_VERBOSE_OUTPUT) )
      { sprintf(sz1kOutput, "0x%08lX:'%s'", (long)hWnd, sz511Title );
        pEnumHelper->pvInfoStringOuput( sz1kOutput );
      }

     // Regardless of what it is, store all we need later in a new 'tree node'.
     // In WSJTX_Sync_FindWSJTXMainWindowHandle() -> EnumWindows() -> MyEnumWindowsProc(),
     // we're only enumerating what Microsoft calls "Top Level Windows", not children:
     if( pEnumHelper->nNodesUsed < WSJTX_SYNC_MAX_TREE_NODES )
      { pTreeNode = &pEnumHelper->nodes[pEnumHelper->nNodesUsed++];
        pTreeNode->hWnd = hWnd;
        strncpy( pTreeNode->sz511Title, sz511Title, 511 );
        GetClassName( hWnd, pTreeNode->sz511ClassName, 511 );
        // With the stupid frameworks by Microsoft, 'ClassNames' are not what
        // they used to be (=descriptive) but crap like the following:
        //   sz511ClassName = "Afx:00007FF744EE0000:0", together with
        //   sz511Title     = "CBHOHostMainWindowMetro",  etc etc .
        pTreeNode->pParent = pEnumHelper->pCurrentNode;
        pTreeNode->pPrevSibling = NULL;
        pTreeNode->pNextSibling = NULL;
        pTreeNode->pFirstChild  = NULL;
      } // end if( pEnumHelper->nNodesUsed < WSJTX_SYNC_MAX_TREE_NODES )

     // Check if this is really the WSJT-X *main* window (not the "spectrum").
     // When tested under Windows 8.1, got here with abstruse names like
     //   "CBHOHostMainWindowMetro", "Programmumschaltung", "Suche", "Akkustand",
     //   "Network Flyout", "WSJT-X", "WSJT-X", "WSJT-X" (maybe several times more),
     //   "CBHOHostMainWindowDskTop", "Datenbank-Engine-Fehler",
     //   "C++Builder 6 - NtpClient [Ausfhren von]", "WSJTX_Sync.c" (Borland's EDITOR WINDOW),
     //   "Projektverwaltung", "Liste berwachter Ausdrcke..", "Klassenhierarchie" (that's Borland, too),
     //   "rsNTP (Ridiculously Simple NTP Client)" (that's us),
     //   "Translation-Manager",
     //   "WSJT-X  v2.2.2  by K1JT, G4WJS, and K9AN" (hooray.. but wait),
     //   "Total Commander (x64) ...",  "Total Commander" (guess what that is for),
     //   "WSJT-X - Wide Graph",  "Ham Radio Soft",
     //   "Realtek SpeakerTestManager", "watchdirwindow" (wtf.. who's watching?),
     //   "NvSvc" (Nvidia Control Panel bloat ?),
     //   "HP CoolSense" (more bloat .. this thing doesn't have a visible window),
     //   "GDI+ Window", "MsgWnd", "EmuUsbAudioCP" (hasn't been used for ages..),
     //   "CVenderSpecSetting" (wtf..), "ViennaSettingUI",
     //   "Realtek HD Audio New GUI" (bloat, bloat, BLOAT), "Animate Manager",
     //   "Audio Engine For Realtek HD Audio New GUI" (omg..),
     //   "RTK TRAYICON", ...
     // (The autor gave up at this point. Too many invisible windows out there.)
     cp = sz511Title;
     if( strncmp(cp, "WSJT-X", 6/*chars*/) == 0 )  // looks like one of WSJT-X's windows..
      { cp += 6; // skip the fixed (version-independent) part...
        while( (*cp==' ')  || (*cp=='\t') )
         { ++cp; // skip an unknown number of SPACES and TABS after the "WSJT-X"
         }
        // Hope the scheme of the WSJT-X main window doesn't change,
        // and the next fragment always remains the version, beginning with
        // a lower-case 'v' (as in "WSJT-X  v2.2.2  by K1JT, G4WJS, and K9AN").
        switch( *cp ) // what's following after "WSJT-X" ?
         { case 'v':  // hooray, it's a lower-case 'v' so guess we found the MAIN WINDOW !
              WSJTX_Sync_hwndMainWindow = hWnd;
              strncpy(WSJTX_Sync_sz511Title, sz511Title, 511 );
              return FALSE; // don't waste more time in EnumWindows() !
           case '-':  // possibly "WSJT-X - Wide Graph" . Not our business !
              break;
           default :  // anything else isn't our business, too
              break;
         }
      }
   }
  return TRUE; // TRUE = "give me the next window, please"
} // end MyEnumWindowsProc()


//---------------------------------------------------------------------------
HWND WSJTX_Sync_FindWSJTXMainWindowHandle(void)
  // Returns (HWND)0 if we couldn't find a running instance of WSJT-X .
  // The result (window handle) is also stored in WSJTX_Sync_hwndMainWindow .
{
  // > The EnumWindows function enumerates all top-level windows on the screen
  // > by passing the handle of each window, in turn, to an application-defined
  // > callback function. EnumWindows continues until the last top-level window
  // > is enumerated or the callback function returns FALSE.
  // > The EnumWindowsProc function is an application-defined callback function that receives top-level window handles as a result of a call to the EnumWindows or EnumDesktopWindows function.
  // > BOOL CALLBACK EnumWindowsProc(
  // >        HWND hwnd,      // handle to parent window ("parent" ? nope. Handle of a TOP-LEVEL window.)
  // >        LPARAM lParam); // application-defined value
  // > To continue enumeration, the callback function must return TRUE;
  // > to stop enumeration, it must return FALSE.
  // >

  WSJTX_Sync_hwndMainWindow = (HWND)0; // forget old WSJTX main window handle
  memset( WSJTX_Sync_sz511Title, 0, sizeof(WSJTX_Sync_sz511Title) ); // forget the old WSJT-X main window title
  WSJTX_Sync_iRecursionDepth = 0;
  EnumWindows( MyEnumWindowsProc, (LPARAM)&WSJTX_EnumWindowsHelper );

  return WSJTX_Sync_hwndMainWindow;

} // end WSJTX_Sync_FindWSJTXMainWindowHandle()

//---------------------------------------------------------------------------
static BOOL CALLBACK MyEnumChildWindowsProc( // our callback for EnumChildWindows() [not EnumWindows()]
       HWND hWnd,      // [in] handle to a (?child?-) window
       LPARAM lParam)  // [in] application-defined value  ( actually a T_EnumWindowsHelper* )
       // [out] (T_EnumWindowsHelper*)lParam->hWndDecoderOutput,
       //          only set if we found the WSJT-X-GUI's right child window
       //          (the edit control somewhere under "Band Activity",
       //           under the static headline that hopefully contains
       //           the WindowText " UTC   dB   DT Freq    Message",
       //           which would confirm WSJT-X currently runs FT8 or FT4).
  //  Since the WSJT-X GUI was written in Qt, finding that window handle
  //  turned out a tough job - maybe too touch, and this will never work...
  //
  // This function is called indirectly from WSJTX_Sync_FindHandleOfDecoderOutputWindow().
  //      Details THERE.
  // Call stack (seen when a breakpoint in MyEnumChildWindowsProc() fired
  //             for the FIRST time) :
  //    WSJTX_Sync_FindHandleOfDecoderOutputWindow()
  //     -> SysWO64/apphelp.dll [in fact EnumChildWindows()] -> ...
  //
  //
  //
{
  // Fasten seat belts .. we're diving deep into the ancient Win32 API again !
  //        For details about 'enumerating windows', see MyEnumWindowsProc().
  //        MyEnumChildWindowsProc() is similar, but as the name implies,
  //        it doesn't search for the fellow application's MAIN WINDOW,
  //        but for a certain CHILD WINDOW (aka "control").
  int  iClassNameLength, iWindowTextLength;
  char sz511ClassName[512], sz511WindowText[512], sz1kOutput[1024], *cp;
  BOOL fRecognized = FALSE;
  T_EnumWindowsHelper *pEnumHelper = (T_EnumWindowsHelper*)lParam;
  T_GUITreeNode *pTreeNode = NULL;

  if( IsKnownWindowHandle( pEnumHelper, hWnd ) )
   { // Oops.. already saw this window, so ignore this call, and don't try
     // to recursively walk through this tree-node's CHILDREN !
     return TRUE;
   }

  iClassNameLength = GetClassName( hWnd, sz511ClassName, 511 );
  // About GetClassName() [Win32 API, not Borland-specific] :
  // > If the function succeeds, the return value is the number of characters
  // > copied to the buffer, not including the terminating null character.

  iWindowTextLength = GetWindowText( hWnd, sz511WindowText, 511 );
  // About GetWindowText() [also Win32 API] :
  // > This function cannot retrieve the text of an edit control in another application.
  // Bwaaah. Would have been fine if you also told us what to use instead !
  sz511WindowText[511] = '\0';  // truncate for safety
  cp = sz511WindowText; // <- place for a BREAKPOINT

  if( (pEnumHelper->pvInfoStringOuput != NULL) // show what's going on ?
   && (pEnumHelper->iOptions & WSJTX_SYNC_OPTION_VERBOSE_OUTPUT) )
   { sprintf(sz1kOutput, "0x%08lX:class='%s', title='%s'", (long)hWnd, sz511ClassName, sz511WindowText );
     pEnumHelper->pvInfoStringOuput( sz1kOutput );
   }

  // Regardless of what it is, store all we need later in a new 'tree node'.
  // In WSJTX_Sync_FindWSJTXMainWindowHandle() -> EnumWindows() -> MyEnumWindowsProc(),
  // we're only enumerating what Microsoft calls "Top Level Windows", not children:
  if( pEnumHelper->nNodesUsed < WSJTX_SYNC_MAX_TREE_NODES )
   { pTreeNode = &pEnumHelper->nodes[pEnumHelper->nNodesUsed++];
     pTreeNode->hWnd = hWnd;
     strncpy( pTreeNode->sz511Title, sz511WindowText, 511 );
     GetClassName( hWnd, pTreeNode->sz511ClassName, 511 );
     pTreeNode->pParent = pEnumHelper->pCurrentNode; // <- may be NULL if this is a TOP-LEVEL window !
     pTreeNode->pPrevSibling = NULL;
     pTreeNode->pNextSibling = NULL;
     pTreeNode->pFirstChild  = NULL;
   } // end if( pEnumHelper->nNodesUsed < WSJTX_SYNC_MAX_TREE_NODES )

  // Check if we can make any sense out of this ("ClassName" and "WindowText"):
  // When tested with rsNTP's own main window (called from TrsNTP::SynchonizeNow()
  //      with the applications OWN 'main window handle'), got here with ....
  //  ClassName = "TBitBtn" with
  //     '--- WindowText : "&Options" (ok, that's one of our buttons,
  //                                   but neither the first nor the last.
  //                                   Where are THE OTHER buttons (siblings) ?
  //                                  )
  //
  //  ClassName = "TEdit" with
  //     '--- WindowText : "0",
  //
  //  ClassName = "TCheckBox" with
  //     '--- WindowText : "verbose output" .
  //
  //  ClassName = "TCheckBox" with
  //     '--- WindowText : "synchronize periodically" .
  //
  //  ClassName = "TComboBox" with
  //     '--- WindowText : "WSJT-X/FT8 'DT'" .
  //
  //  ClassName = "Edit" (not "TEdit") with
  //     '--- WindowText : "WSJT-X/FT8 'DT'" (same as above ?!)
  //
  //  ClassName = "TBitBtn" with
  //     '--- WindowText : "&Help"
  //
  //  ClassName = "TBitBtn" with
  //     '--- WindowText : "&Synchronize"
  //
  //  ClassName = "TMemo" with
  //     '--- WindowText : "Synchronising.."
  //
  // THAT WAS ALL (again, when operating on rsNTP's own GUI) !
  // Note the lack of calls for (VCL-)objects with class names like
  //      "TLabel" (non-selectable text), "TPopupMenu", "TMenuItem", etc .
  //      They don't seem to have their own WINDOW at all,
  //      and thus are unknown from EnumChildWindows' point of view .
  //
  //
  // When tested with WSJT-X v2.2.2 in August 2020, never got here when
  //      invoked from EnumChildWindows() at all !
  // Gave up on retrieving the text from WSJT-X's "decoder output" window
  //      for a while, but decided NOT to throw away what was already written.
  if( strcmp(sz511ClassName, "WSJT-X" ) == 0 )  // ok so far...
   {
     pEnumHelper->hwndDecoderOutputWindow = hWnd; // guess this is WSJT-X's "decoder output" window, somewhere UNDER "Band Activity"
     return FALSE; // BINGO - don't waste more time in EnumChildWindows() !
   }
  if( ! fRecognized ) // haven't got a clue yet ? recursively check for MORE children..
   { if( WSJTX_Sync_iRecursionDepth < MAX_RECURSION_DEPTH )
      { ++WSJTX_Sync_iRecursionDepth; // no 'lazy return' after this point !

        --WSJTX_Sync_iRecursionDepth; // no 'lazy return' before this point !
      }
   }
  return TRUE; // TRUE = "give me the next CHILD window, please"
} // end MyEnumChildWindowsProc()

//---------------------------------------------------------------------------
static BOOL CALLBACK MyEnumThreadWindowsProc(
       HWND hWnd,      // [in] handle to a top-level window
       LPARAM lParam)  // [in] application-defined value  ( actually a T_EnumWindowsHelper* )
  // Called from WSJTX_Sync_FindHandleOfDecoderOutputWindow(). Details THERE.
{
  // Fasten seat belts .. we're diving deep into the ancient Win32 API again !
  //        For details about 'enumerating windows', see MyEnumWindowsProc().
  //        MyEnumThreadWindowsProc() is similar, but as the name implies,
  //        it doesn't search for the fellow application's MAIN WINDOW,
  //        but for all (top-level?) windows created by THE SAME THREAD.
  int  iClassNameLength, iWindowTextLength;
  char sz511ClassName[512], sz511WindowText[512], *cp;
  BOOL fRecognized = FALSE;
  T_EnumWindowsHelper *pEnumHelper = (T_EnumWindowsHelper*)lParam;

  if( IsKnownWindowHandle( pEnumHelper, hWnd ) )
   { // Oops.. already saw this window, so ignore this call, and don't try
     // to recursively walk through this tree-node's CHILDREN !
     return TRUE;
   }


  iClassNameLength = GetClassName( hWnd, sz511ClassName, 511 );
  // About GetClassName() [Win32 API, not Borland-specific] :
  // > If the function succeeds, the return value is the number of characters
  // > copied to the buffer, not including the terminating null character.

  iWindowTextLength = GetWindowText( hWnd, sz511WindowText, 511 );
  // About GetWindowText() [also Win32 API] :
  // > This function cannot retrieve the text of an edit control in another application.
  // Bwaaah. Would have been fine if you also told us what to use instead !
  cp = sz511WindowText; // <- place for a BREAKPOINT

  // Check if we can make any sense out of this ("ClassName" and "WindowText"):
  // When tested with WSJT-X v2.2.2 in August 2020, only got here when
  // invoked from EnumThreadWindows() -but not from EnumChildWindows()-
  // with the following WINDOW CLASS NAMES :
  //  ClassName = "Qt5QWindowPopupDropShadowSaveBits" (several times) with
  //     '--- WindowText : "WSJT-X",   "WSJT-X",   "WSJT-X",   "WSJT-X",    "WSJT-X",  .. (not too helpful)
  //          hWnd       : 0x00A50334  0x00940596  0x004604AE  0x004705D0  0x000F05E4
  //
  //  ClassName = "Qt5WindowIcon" (two times) with ...
  //     '--- WindowText : "WSJT-X  v2.2.2  by K1JT, .."
  //     |      hWnd     : 0x0024059A == WSJTX_Sync_hwndMainWindow (!)
  //     '--- WindowText : "WSJT-X - Wide Graph"
  //            hWnd     : 0x00890508
  //
  //  ClassName = "MSCTFIME UI" with ...
  //     '--- WindowText : "MSCTFIME UI"  (same as the strange ClassName)
  //
  //  ClassName = "IME" with ...
  //     '--- WindowText : "Default IME"
  //

  // None of this really helps. One would expect at least a few (child-)windows
  //   with ClassName = "QWidget" (because "inspect.exe" showed them).
  //
  //
  //
  //
  //
  if( strcmp(sz511ClassName, "WSJT-X" ) == 0 )  // ok so far...
   {
     return FALSE; // BINGO - don't waste more time in EnumChildWindows() !
   }
  if( ! fRecognized ) // haven't got a clue yet ? Look at even more CHILDREN !
   { if( WSJTX_Sync_iRecursionDepth < MAX_RECURSION_DEPTH )
      { ++WSJTX_Sync_iRecursionDepth; // no 'lazy return' after this point !
        EnumChildWindows( // here: try to enumerate all children of another 'thread window', called from MyEnumThreadWindowProc()
          hWnd, // [in] HWND hWndParent; a handle to the parent window
                //      whose child windows are to be enumerated.
          MyEnumChildWindowsProc, // [in] WNDENUMPROC lpEnumFunc; callback for enumeration
          (LPARAM)pEnumHelper);   // [in] address of something user-defined passed to the callback
        // 2020-08-16: Also in THIS case, EnumChildWindows() NEVER invoked
        //             the callback (MyEnumChildWindowsProc) .
        // Maybe Windows is so paranoid now that it doesn't allow enumerating
        // 'child windows' in an EXTERNAL application ? Tried this with
        // OUR OWN application (rsNTP) for testing purposes - see TEST CALL
        // of WSJTX_Sync_FindHandleOfDecoderOutputWindow()
        // from C:\cbproj\rsNTP\NtpClientU1.cpp : TrsNTP::SynchronizeNow() .
        --WSJTX_Sync_iRecursionDepth; // no 'lazy return' before this point !
        // 2020-08-23: Seemed to crash shortly after THIS point.
        //             Try again STEPPING INTO subroutines after this ...
        //
      }
   } // end if( ! fRecognized )
  return TRUE; // TRUE = "give me the next CHILD window, please"
} // end MyEnumThreadWindowsProc()

//---------------------------------------------------------------------------
HWND WSJTX_Sync_FindHandleOfDecoderOutputWindow(
        HWND hWndMainWindow, // [in] handle of the WSJT-X *main* window (hopefully)
        int iOptions, // [in] bitwise combination of WSJTX_SYNC_OPTION_...
        void (*pvInfoStringOuput)(char *cpInfo) ) // [in, optional] procedure to 'print info strings'
  // Tries to find the WINDOW HANDLE of the WSJT-X GUI's "Band Activity"
  // output window, which in fact is an EDIT CONTROL (possibly a
  // RichEdit control for the coloured background, but that's a QT-thing
  // we really don't want to investigate any further here).
  //
  // When successfull, returns a valid handle; otherwise returns (HWND)0 .
{
  // How to find out the window handle of WJST-X's "Band Activity" editor ?
  // Using Microsoft's "inspect.exe" (from the Windows SDK), it became
  //  obvious that the following THREE windowed controls are SIBLINGS:
  //    (1) a text control with Name: "Band Activity"
  //    (2) next sibling: text control with Name: "  UTC   dB   DT Freq    Message" (note the spaces)
  //    (3) next sibling: edit control without a 'Name' but with
  //         AutomationId: "MainWindow.centralWidget.splitter.layoutWidget.decodedTextBrowser"
  //         ClassName:    "DisplayText"
  // The last part of the "automation ID" appeared ideal for the purpose,
  // but it seems impossible to retrieve without the 'Dot Net'-stuff .
  // Microsoft says
  //  > To retrieve an IUIAutomationElement from an HWND, use the
  //  > IUIAutomation::ElementFromHandle method.  (thus we're out of the game)
  // So instead, try to enumerate all CHILD WINDOWS of the WSJT-X GUI,
  // looking for the text control "Band Activity", and it's siblings :
  //  > EnumChildWindows enumerates the child windows that belong to the
  //  > specified parent window (here: hWndMainWindow) by passing the handle
  //  > to each child window, in turn, to an application-defined callback function.
  //  > EnumChildWindows continues until the last child window is enumerated
  //  > or the callback function returns FALSE.
  DWORD dwThreadId;
  // ex: T_EnumWindowsHelper enumWindowsHelper; // local variables are a nightmare for debugging...
  T_EnumWindowsHelper *pEnumHelper = &WSJTX_EnumWindowsHelper; // .. use this GLOBAL variable instead,
                // which can be inspected even after crashing with an exception

  memset( pEnumHelper, 0, sizeof(T_EnumWindowsHelper) );
  pEnumHelper->pvInfoStringOuput = pvInfoStringOuput; // <- may be NULL !


  EnumChildWindows( // here: try to enumerate all (direct) children of the MAIN WINDOW (failed!)
      hWndMainWindow, // [in] HWND hWndParent; a handle to the parent window
                      //      whose child windows are to be enumerated.
                      //      e.g. 0x24059A (same as inspect.exe showed as 'Window:'
                      //        when selecting "WSJT-X  v2.2.2  by K1JT.." from the tree.
                      //      According to inspect.ext, the main window had...
                      // > "WSJT-X   v2.2.2   by K1JT, G4WJS, and K9AN" (name shown LEFT, tree's root)
                      // >          ChildCount:7 (!),  Window: 0x24059A (not unique - same shown for children)
                      // >   |
                      // >  [+] "Systemmen" : Menleiste : markierbar
                      // >  [+] none : Titelleiste : markierbar
                      // >  [+] "WSJT-X   v2.2.2   by K1JT, G4WJS, and K9AN" : Fenster : markierbar
                      // >   |   '--- ChildCount:3 (!), Window: 0x24059A (same as for the parent)
                      // Strange. Why is "Window:" (==hWndMainWindow) the same for parent and all children ?
      MyEnumChildWindowsProc, // [in] WNDENUMPROC lpEnumFunc; callback for enumeration
      (LPARAM)pEnumHelper);   // [in] address of something user-defined passed to the callback
  // > The EnumChildWindows function does not enumerate top-level windows
  // > owned by the specified window, nor does it enumerate any other owned windows.
  // > If EnumChildWindows() fails, the return value is zero.
  // When tested with WSJT-X's "main window" (handle from WSJTX_Sync_FindWSJTXMainWindowHandle()),
  // EnumChildWindows() didn't call MyEnumChildWindowsProc() at all.
  // Surprise surprise. Use EnumThreadWindows() instead, which according to MS,
  // > enumerates all nonchild windows associated with a thread by passing
  // > the handle of each window, in turn, to an application-defined
  // > callback function.
  if( pEnumHelper->hwndDecoderOutputWindow == (HWND)0 ) // EnumChildWindows() -> MyEnumChildWindowsProc() NOT successfull ?
   { // For EnumThreadWindows(), we need a "thread identifier", not a WINDOW HANDLE.
     // So how to retrieve the other application's *THREAD ID*,
     //    when all we have is its *MAIN WINDOW HANDLE* ?
     // Possibly GetWindowThreadProcessId(), which according to MS ...
     //  > Retrieves the identifier of the thread that created the specified
     //  > window and, optionally, the identifier of the process
     //  > that created the window.
     //  > The return value is the identifier of the thread that created the window.
     dwThreadId = GetWindowThreadProcessId(hWndMainWindow,NULL/*lpdwProcessId*/);
     // Got here with WSJT-X's hWndMainWindow = 0x0024059A and dwThreadId = 7116 .
     // The THREAD ID was confirmed by Process Explorer (which also shows
     //     THREAD IDs in decimal form, in column 'TID') .
     // When called from EnumThreadWindows(!), a breakpoint in MyEnumChildWindowsProc()
     // fired a couple of times,
     if( dwThreadId != 0 )
      { EnumThreadWindows(  // <- 2020-08-23: seemed to crash HERE
         dwThreadId, // [in] thread identifier;
                     // identifies THE THREAD whose windows are to be enumerated.
          MyEnumThreadWindowsProc, // [in] WNDENUMPROC lpEnumFunc; callback for enumeration
      (LPARAM)pEnumHelper); // [in] address of something user-defined passed to the callback
      }
   } // end if <  EnumChildWindows() -> MyEnumChildWindowsProc() NOT successfull >


  return pEnumHelper->hwndDecoderOutputWindow; // <- non-NULL when successfull



} // end WSJTX_Sync_FindHandleOfDecoderOutputWindow()



//--------------------------------------------------------------------------
BOOL WSJTX_Sync_UpdateClockOffset( // 'twin' of NtpClient_Update(), but without NTP ...
     int   iOptions,             // [in] NTP_SYNC_SHORT_INFO, NTP_SYNC_VERBOSE_INFO, ..
     void (pvInfoStringOuput(char *cpInfo)), // [in, optional] procedure to 'print info strings'
     long double *pClockOffset ) // [out] offset, in seconds, which must be added to the 'local' system time to correct it
  // Determines the clock offset between the PC's system time
  // and as many remote FT8-transmitting stations, based on the output
  // in WSJT-X's "Band Activity" window.
  // Like the original NtpClient_Update(), this function does NOT modify
  // the PC's local time (that's done in TNtpClientForm::SynchronizeNow) .
  // It only tries to MEASURE the local clock error (number of seconds),
  // as far as the 15-second FT8 transmit cycle permits.
  //
  //
{
  HWND hwndWsjtMainWindow, hwndWsjtDecoderOutputWindow;

  // Check if WSJT-X is still running (on the same machine) .
  // On this occasion, also retrieve the HANDLE to WSJT-X's *main window*:
  hwndWsjtMainWindow = WSJTX_Sync_FindWSJTXMainWindowHandle();

  if( hwndWsjtMainWindow == (HWND)0 ) // oops..
   { if( pvInfoStringOuput != NULL )
      {  pvInfoStringOuput( "Couldn't find WSJT-X main window" );
      }
     return FALSE;  // Yes, shit happens. Especially when communicating with external apps ;)
   }

  // Retrieve the window handle of WSJT-X's "Band Activity" window:
  hwndWsjtDecoderOutputWindow = WSJTX_Sync_FindHandleOfDecoderOutputWindow( hwndWsjtMainWindow, WSJTX_SYNC_OPTION_NONE, pvInfoStringOuput );
  if( hwndWsjtDecoderOutputWindow == (HWND)0 ) // oops..
   { if( pvInfoStringOuput != NULL )
      {  pvInfoStringOuput( "Couldn't find WSJT-X decoder output window" );
      }
     return FALSE;  // Shit happens. More than once !
   }

  // Retrieve the entire text in WSJT-X's decoder output window (under "Band Activity"):

  return FALSE;

} // end WSJTX_Sync_UpdateClockOffset()


/* EOF < C:\cbproj\rsNTP\WSJTX_Sync.c > */
