/*
 * File: C:\cbproj\rsNTP\NtpClientU1.cpp
 * Purpose: Simple NTP client with a simple GUI, for windows.
 * Author:  Wolfgang Buescher (DL4YHF) .
 *
 * This code will query a ntp server for the local time and display
 * it.  it is intended to show how to use a NTP server as a time
 * source for a simple network connected device.
 * This is the C version.  The orignal was in Perl.
 *
 * For better clock management see the offical NTP info at:
 * http://www.eecis.udel.edu/~ntp/
 *
 * Initial code written by Tim Hogard (thogard@abnormal.com)
 *                Thu Sep 26 13:35:41 EAST 2002
 * Converted to C Fri Feb 21 21:42:49 EAST 2003
 *   This code is in the public domain.
 *   The original could be found here http://www.abnormal.com/~thogard/ntp/
 *
 * Converted into a simple GUI application by Wolfgang Buescher, 2014.
 *   Put those important and informative comments (from Tim's original source)
 *   back in. The compiled program, along with this sourcecode, can be found at
 *   http://www.qsl.net/dl4yhf/rsNTP/rsNTP.htm
 *
 * Latest modifications:
 *   2020-08-15 (by W.B.): Tried to add an option to adjust the system time
 *                by the average 'DT' value (3rd colum in the WSJT-X receive
 *                text display, at least at the time of this writing) in F8,
 *                so computers without internet access (or other accurate time
 *                sync sources) can easily be synchronized BY and FOR WSJT-X.
 *                Since this hasn't got ANYTHING to do with NTP, it was put
 *                inside a new module - see
 */

//---------------------------------------------------------------------------

#include <vcl.h>
#pragma hdrstop


#include <stdio.h>
#include <math.h>
#include <winsock2.h>       // a Berkeley-like socket API should be all we need to communicate

#ifndef  uint64_t
# define uint64_t unsigned __int64
#endif
#ifndef  int64_t
# define int64_t __int64
#endif
#ifndef  uint32_t
# define uint32_t DWORD
#endif
#ifndef  int32_t
# define int32_t int
#endif
#ifndef  socklen_t
# define socklen_t int  /* can we have an extra data type for storing grandma's age ? GRRRR ! */
#endif

#include "WSJTX_Sync.h"  // header file for DL4YHF's "WSJT-X Synchronizer"

#include "NtpClientU1.h" // header for THIS module (generated by Borland C++Builder)
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"

//---------------------------------------------------------------------------
// Non-Borland-specific stuff (at least, no Borland-VCL-dependencies below)
//---------------------------------------------------------------------------

typedef union { // 8/16/32-bit - adressable "long"-data type
    BYTE b[4];
    WORD w[2];
    long  i32;
    DWORD dw;
} T_BLONG;


char APPL_szIniFileName[] = "rsNTPclient.ini";
HINSTANCE APPL_hInstance; // handle to current instance. Lousy red tape for windoze.
BOOL g_VerboseOutput = FALSE;

char   NtpClient_sz255LastErrorInfo[256];
double NtpClient_dblClockOffset = 0.0;  // value ADDED to the PC's system time (to 'correct' it)
                 // Does NOT include the "deliberate", user-defineable offset !
double NtpClient_dblDeliberateOffset_sec = 0.0; // extra, user-adjustable "deliberate" offset, added before setting the PC's system time
double NtpClient_dblSpecialOffset1_sec=10.0; // special user-adjustable "deliberate" offset, added during a certain interval every hour (for WSPR)
BOOL   NtpClient_fSpecialExtraOffsetActiveNow = FALSE;
int    NtpClient_iSpecialOptions = 0;   // added 2016-05 for "deliberate offsets" for WSJT, see http://www.qsl.net/zl1rs/oceanfloater.html / PA3FYM
#         define NTP_SPECIAL_OPTION_NONE                      0
#         define NTP_SPECIAL_OPTION_OFFSET_AT_SECONDS_OF_HOUR 0x0001
       // 2016-05-29 : The above "deliberate offset" can be restricted
       //              to a certain timeslot, which occurs once per HOUR.
       // Details from an email by Remco, PA3FYM :
  // > Please read this: http://www.qsl.net/zl1rs/oceanfloater.html
  // > Having been one of / the only Northern hemisphere station(s)
  // > able to receive balloons etc from the Pacific I am asked for assistance.
  // > Due to a bug in the floater software its WSPR and JT9 clock are now 10 seconds off.
  // > Of course I can enter this in rsNTP but then other WSPR traces cannot be decoded.
  // > So, I thought there might be a way round to get the best of both worlds:
  // > Is it possible for you to create a version of rsNTP for me that
  // > (a)  deliberately sets the clock 10 secs ahead in time in the 45th minute every hour (say at xx:45:30)
  // > (b)  and restores the clock to the correct time just before the 50th minute, e.g. xx:49:55 ?
  // From DL4YHF : (a) is configurable as NtpClient_iBeginOffsetAtSecondOfHour (default 45*60 seconds),
  //               (b) is configurable as NtpClient_iBeginOffsetAtSecondOfHour (default 50*60-5 seconds).
  // This feature is activated via NtpClient_iSpecialOptions, bitmask NTP_SPECIAL_OPTION_OFFSET_AT_SECONDS_OF_HOUR:
int NtpClient_iBeginOffsettingAtSecondOfHour = 45*60;
int NtpClient_iEndOffsettingAtSecondOfHour   = 50*60-5;

// Added 2016-06 : buffer for 'commands' received via WM_COPYDATA from other applications:
char   NtpClient_sz255RcvdCmd[256];
BOOL   NtpClient_fRcvdCommand = FALSE;
HWND   NtpClient_hwndCmdSender;



//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Mother's little helpers ..

//---------------------------------------------------------------------------
// Note on windoze-"INI-files": If you don't specify a full path,
// your INI-file will be placed in some strange windoze-system-directory !
// Use ExpandFilenameToFullPath() if you want the INI file in the current dir.
// BUT: Placing the INI file where *YOU* want to have it doesn't work under
//      Windows "Vista" (yucc) and Windows 7 anymore !!
//   So forget about this, forget about ini files, forget about windoze... ?

//---------------------------------------------------------------------------
void WriteStringToIniFile(char *pszIniFileName, char *pszSection, char *pszKey, char *pszData)
{
  WritePrivateProfileString(pszSection,pszKey,pszData,pszIniFileName);
}

//---------------------------------------------------------------------------
void WriteIntegerToIniFile(char *pszIniFileName, char *pszSection, char *pszKey, long lData)
{
  char sz15Temp[16];
  wsprintf(sz15Temp,"%ld",(long)lData);
  WriteStringToIniFile(pszIniFileName,pszSection,pszKey,sz15Temp);
}

//---------------------------------------------------------------------------
void ReadStringFromIniFile(char *pszIniFileName, char *pszSection, char *pszKey,
                           char *pszDest, int iMaxLen, char *pszDefault)
{
  GetPrivateProfileString(
     pszSection,pszKey,pszDefault,   // section, key name, default string
     pszDest,iMaxLen,                // returned string, max length
     pszIniFileName );               // "initialization filename"
}

//---------------------------------------------------------------------------
long ReadIntegerFromIniFile(char *pszIniFileName, char *pszSection, char *pszKey,
                            long lDefault)
{
  return GetPrivateProfileInt(
     pszSection,pszKey,lDefault,   // section, key name, default value
     pszIniFileName );             // "initialization filename"
}

//---------------------------------------------------------------------------
double ReadDoubleFromIniFile(char *pszIniFileName, char *pszSection, char *pszKey,
                             double dblDefault)
{
  char   sz40Default[44], sz80Read[256];
  double dblResult;
  sprintf( sz40Default, "%lf", dblDefault );
  ReadStringFromIniFile( APPL_szIniFileName, pszSection, pszKey, sz80Read/*dest*/, 80/*maxlen*/, sz40Default );
  dblResult = atof( sz80Read );
  if( dblResult==HUGE_VAL )
   {  dblResult = dblDefault;
   }
  return dblResult;
} // end ReadDoubleFromIniFile()


//---------------------------------------------------------------------
double NtpClient_GetCurrentDeliberateOffset(void) // f( NtpClient_fSpecialExtraOffsetActiveNow, NtpClient_iSpecialOptions, NtpClient_dblSpecialOffset1_sec, NtpClient_dblDeliberateOffset_sec)
{ if( NtpClient_fSpecialExtraOffsetActiveNow && (NtpClient_iSpecialOptions & NTP_SPECIAL_OPTION_OFFSET_AT_SECONDS_OF_HOUR) )
   { return NtpClient_dblSpecialOffset1_sec;
   }
  else // not the "special" but the "normal" deliberate offset in effect now:
   { return NtpClient_dblDeliberateOffset_sec;
   }
} // end NtpClient_GetCurrentDeliberateOffset()

//---------------------------------------------------------------------
long double NtpClient_GetSystemDateAndTimeAsUnixSeconds(
       int options )
#  define WITHOUT_DELIBERATE_OFFSET  1
#  define WITH_DELIBERATE_OFFSET     0
  // Retrieves the PC's 'local system time' as number of seconds
  //           elapsed since 1970-01-01 00:00:00 (UTC).
  // Since the SI unit for time is SECONDS, we use seconds, not micro, not milli.. !
  // A 'long double' has a mantissa of approximately 64 bits which is precise enough.
{
  FILETIME ft; // date and time in a crazy windows specific format
  unsigned __int64 u64StrangeMsTime;
  long double ldblUnixSeconds;

  GetSystemTimeAsFileTime(&ft);  // this is a crazy Windows API function:
     // > The GetSystemTimeAsFileTime function obtains the current system date and time.
     // > The information is in Coordinated Universal Time (UTC) format.
     // > The FILETIME structure is a 64-bit value representing
     // > the number of 100-nanosecond intervals since January 1, 1601.
     // > typedef struct _FILETIME {
     // >     DWORD dwLowDateTime;   // low-order 32 bits of the 'file' time
     // >     DWORD dwHighDateTime;  // high-order 32 bits of the 'file' time
     // > } FILETIME;
  u64StrangeMsTime = ((unsigned __int64)ft.dwHighDateTime << 32)
                   | ((unsigned __int64)ft.dwLowDateTime) ;
  // Convert microsoft's strange "file time" to Unix seconds :
  ldblUnixSeconds = (long double)u64StrangeMsTime * (long double)100e-9 - 11644473600.0;

  // Retrieve the current time (UTC) with or without the "deliberate offset" ?
  if( options == WITHOUT_DELIBERATE_OFFSET )
   { // Retrieve the time "as if we had NOT fooled the PC's system time" :
     //   Because when last setting the PC's system time,
     //   the 'deliberate offset' (measured in seconds) had been ADDED,
     //   so the same amount of seconds must be SUBTRACTED here:
     ldblUnixSeconds -= NtpClient_GetCurrentDeliberateOffset(); // // f( NtpClient_fSpecialExtraOffsetActiveNow, ... )
   }

  return ldblUnixSeconds;
} // end NtpClient_GetSystemDateAndTimeAsUnixSeconds()


//---------------------------------------------------------------------------
void NtpClient_SplitUnixDateTimeToYMDhms( long double dblUnixDateTime,
         int *piYear,
         int *piMonth,
         int *piDay,
         int *piHour, int *piMinute, double *pdblSecond ) // [out] time of day
  // Splits a date-and-time in UNIX format (seconds since Jan 1st, 1970, 00:00:00 UTC)
  // into year, month, day, hour, minute, and (fractional) seconds .
  // Much better than the ugly, thread-unsafe, global-variable-dependent,
  //                      stupid, stonage function named 'gmtime()' !
{
  double dbl, dblUnixDay, dblSecondsOfDay, dblHour, dblMin;
  long   jdayno, j, d, m, y;

  // Split DAY and SECONDS-OF-DAY (here: still UNIX, not MJD, and not JD):
  dblUnixDay = floorl( dblUnixDateTime / 86400.0/*seconds per day*/ );
  dblSecondsOfDay = dblUnixDateTime - dblUnixDay * 86400.0;
  dbl = dblSecondsOfDay / 3600.0;      // -> hours of day, fractional
  dblHour = floor(dbl); dbl=60.0*(dbl-dblHour);  // -> hours of day
  dblMin  = floor(dbl); dbl=60.0*(dbl-dblMin);   // -> minutes of hour
  if( piHour != NULL )
   { *piHour = (int)dblHour;
   }
  if( piMinute!= NULL )
   { *piMinute = (int)dblMin;
   }
  if( pdblSecond!= NULL )
   { *pdblSecond = dbl;   // remaining "seconds of minute", fractional
   }

  // Now for the trickier part: Split year, month of year, and day of month.
  // From   http://en.wikipedia.org/wiki/Julian_day :
  // > Unix Time (seconds since January 1st, 1970, UTC)
  // >  calculated from Julian Day by : (JD - 2440587.5) * 86400
  // So, calculate 'jdayno' (Julian Day, JD) for the algo further below.
  //  "UnixDay" = JulianDay - 2440587.5, so :
  jdayno = (long)( dblUnixDay + 2440587.5 );  // -> Julian Day, INTEGER

  // Die folgende Berechnung stammt aus Google Code Search, Stichwort
  //  "jdater - sets day, month and year given jdayno" (haystack.edu .. date.c)
  //  "inverse of jday method" .
  // Was in der Beschreibung leider komplett fehlt, ist eine Spezifikation
  //  der ZAEHLWEISE, speziell fuer jdayno (0...x oder 1...x),
  //  Monat(0..11 oder 1..12) und Tag (0..30 oder 1..31) !
  // Von gmtime() ist man ja Elendes gewohnt (Tage von 1..31 aber Monate von 0..11).
  // Hier scheint aber alles AB EINS gezaehlt zu werden (Pascal-Freaks?),
  // auch der 'jdayno' beginnt die Zaehlung scheinbar bei EINS, nicht NULL, darum:
  ++jdayno;
  // Original formula from haystack.edu starts here:
  j = jdayno - 1721119;
  y = (4*j - 1)/146097;
  j  =  4*j - 1 - 146097*y;
  d = j/4;
  j = (4*d + 3)/1461;
  d = 4*d + 3 - 1461*j;
  d = (d + 4)/4;
  m = (5*d - 3)/153;
  d = 5*d - 3 - 153*m;
  d = (d + 5)/5;
  y = 100*y + j;
  if (m < 10)
   {  m = m + 3;
   }
  else
   {  m = m - 9;
      y = y + 1;
   }

  if( piYear != NULL )
   { *piYear = y;
   }
  if( piMonth != NULL )
   { *piMonth = m;
   }
  if( piDay != NULL )
   { *piDay = d;
   }
} // end NtpClient_SplitUnixDateTimeToYMDhms()

const char *NtpClient_months1[12]= // used by NtpClient_FormatDateAndTime()
   { "JAN","FEB","MAR","APR","MAY","JUN","JUL","AUG","SEP","OCT","NOV","DEC" };
const char *NtpClient_months2[12]= // used by NtpClient_FormatDateAndTime()
   { "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec" };

char *NtpClient_FormatDateAndTime( char *format, long double dblUnixDateTime, char *dest)
  // Example for format:  "YYYY-MM-DD hh:mm:ss"  (preferred ISO8601)
  //           ->result:  "2003-06-01 14:28:02"  (placed in dest)
  // The input parameter dblDateTime is similar to the UNIX time:
  //         "SECONDS elapsed since January 1st, 1970, 00:00:00" .
{
  static char sz255Result[256];  // not thread-safe; only used when caller doesn't provide 'dest'
  char   fmt,fmt2,fmt3;
  int    i;
  int   year,month,day,hour,min;
  double dblSec;
  char   *cpResult;
  if( dest==NULL )
   {  dest = sz255Result;
   }
  cpResult = dest;


  NtpClient_SplitUnixDateTimeToYMDhms( dblUnixDateTime,
                                 &year, &month, &day, &hour, &min, &dblSec );
  while( (fmt = *format++) != 0)
   {
    fmt2 = *format; // to look ahead at following letters in the format string
    fmt3 = *(format+1);
    switch(fmt)
     {
       case 'Y':  // year  (1 to 4 digits)
            if(fmt2=='Y')
             {
              ++format;  // skip 2nd letter in format string
              if(fmt3=='Y')
               {
                ++format;  // skip 3rd letter in format string
                if(*format=='Y')
                 {
                  ++format;  // skip 4th letter in format string
                  sprintf(dest,"%04d",(int)year );
                 }
                else // 3-letter format (??)
                 {
                  sprintf(dest,"%03d",(int)(year%1000) );
                 }
               }
              else // 2-letter format:
               {
                sprintf(dest,"%02d",(int)(year%100) );
               }
             } // end if (fmt2..)
            else // 1-letter format (used for automatic filenames somewhere)
             {
              sprintf(dest,"%01d",(int)(year%10) );
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;

       case 'M':  // month (2 digits or THREE LETTERS (Jan,Feb, etc) )
            if(fmt2=='M' || fmt2=='m')
             {
              ++format;  // skip 2nd letter in format string
              if(fmt3=='M') // "MMM" = { JAN, FEB, .. }
               {
                ++format;  // skip 3rd letter in format string
                if( month<1 ) month=1;
                strcpy(dest,NtpClient_months1[(month-1) % 12]);
               }
              else
              if(fmt3=='m') // "Mmm" = { Jan, Feb, .. }
               {
                ++format;  // skip 3rd letter in format string
                if( month<1 ) month=1;
                strcpy(dest,NtpClient_months2[(month-1) % 12]);
               }
              else // 2-letter format: numerical, 1..12
               {
                sprintf(dest,"%02d",(int)month );
               }
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)month );
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;

       case 'D':  // day (2 digits)
            if(fmt2=='D')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)day);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)day);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 'h':  // hour
            if(fmt2=='h')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)hour);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)hour);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 'm':  // minute (always 2 digits)
            if(fmt2=='m')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)min);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)min);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 's':  // second (at least 2 digits, possible 'fractional' !)
            if(fmt2=='s')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)dblSec );
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)dblSec );
             }
            dest += strlen(dest);
            if( format[0]=='.' && format[1]=='s')
             { // looks like tenths of a second after a decimal point !
               format+=2; // skip ".s" from the format string
               i = fmod( 10*dblSec, 10.0);
               if (i<0)   // ??
                   i=0;
               if (i>9)   // ????
                   i=9;   // set breakpoint here if fmod behaves strange...
               *dest++ = '.';
               *dest++ = (char)('0'+i);  // append 1/10 second(s)
               if(*format=='s')
                 { // gosh.. he wants to see hundreds of a second :
                   ++format; // skip another "s" from the format string
                   i = fmod( 100*dblSec, 10.0);
                   *dest++ = (char)('0'+i);  // append 1/100 second(s)
                 }
               if(*format=='s')
                 { // since 2003-07 even SINGLE MILLISECONDS can be displayed:
                   ++format; // skip another "s" from the format string (ss.sss)
                   i = fmod( 1000*dblSec, 10.0);
                   *dest++ = (char)('0'+i);  // append 1/1000 second(s)
                 }
               *dest   = '\0';
             }
            fmt = 0; // format letter has been handled
            break;

       default: // all other characters will be passed 1:1 to "dest"
            break;
     } // end switch(fmt)

    if (fmt) // character from format string could not be handled:
     {
      *dest++ = fmt;
     }
   } // end while(fmt)
  *dest = '\0';  // null-terminate the destination string
  return cpResult;

} // end NtpClient_FormatDateAndTime()


//--------------------------------------------------------------------------
char *NtpClient_SocketErrorToString( int iWSAErrorCode )
{
  static char sz80OtherError[88];
  switch( iWSAErrorCode )
   {
    case WSAEINTR          : return "Interrupted function call";
    case WSAEBADF          : return "Invalid socket descriptor";
    case WSAEACCES         : return "Permission denied";
    case WSAEFAULT         : return "Bad address";
    case WSAEINVAL         : return "Invalid argument";
    case WSAEMFILE         : return "Too many open files or sockets";
    case WSAEWOULDBLOCK    : return "Call of WINSOCK routine would block";
    case WSAEINPROGRESS    : return "Operation now in progress";
    case WSAEALREADY       : return "Operation already in progress";
    case WSAENOTSOCK       : return "Socket operation on non-socket";
    case WSAEDESTADDRREQ   : return "Destination address required";
    case WSAEMSGSIZE       : return "Message too long";
    case WSAEPROTOTYPE     : return "Protocol wrong type for socket";
    case WSAENOPROTOOPT    : return "Bad protocol option";
    case WSAEPROTONOSUPPORT: return "Socket type not supported";
    case WSAESOCKTNOSUPPORT: return "Socket not supported";
    case WSAEOPNOTSUPP     : return "Operation not supported";
    case WSAEPFNOSUPPORT   : return "Protocol family not supported";
    case WSAEAFNOSUPPORT   : return "Address family not supported by protocol family";
    case WSAEADDRINUSE     : return "Address already in use";
    case WSAEADDRNOTAVAIL  : return "Cannot assign requested address";
    case WSAENETDOWN       : return "Network is down";
    case WSAENETUNREACH    : return "Network is unreachable";
    case WSAENETRESET      : return "Network dropped connection on reset";
    case WSAECONNABORTED   : return "Software caused connection abort";
    case WSAECONNRESET     : return "Connection reset by peer";
    case WSAENOBUFS        : return "No buffer space available";
    case WSAEISCONN        : return "socket already connected";
    case WSAENOTCONN       : return "Socket is not connected";
    case WSAESHUTDOWN      : return "Cannot send after socket shutdown";
    case WSAETOOMANYREFS   : return "too many refs";
    case WSAETIMEDOUT      : return "Connection timed out";
    case WSAECONNREFUSED   : return "Connection refused";
    case WSAELOOP          : return "loop detected";
    case WSAENAMETOOLONG   : return "name too long";
    case WSAEHOSTDOWN      : return "Host is down";
    case WSAEHOSTUNREACH   : return "No route to host";
    case WSAENOTEMPTY      : return "not empty";
    case WSAEPROCLIM       : return "Too many processes";
    case WSAEUSERS         : return "users";
    case WSAEDQUOT         : return "quot";
    case WSAESTALE         : return "stale";
    case WSAEREMOTE        : return "remote";
    case WSAEDISCON        : return "Graceful shutdown in progress";
    case WSASYSNOTREADY    : return "Network subsystem is unavailable";
    case WSAVERNOTSUPPORTED: return "WINSOCK.DLL version out of range";
    case WSANOTINITIALISED : return "WSAStartup not yet performed";
    case WSAHOST_NOT_FOUND : return "Host not found";
    case WSATRY_AGAIN      : return "Non-authoritative host not found";
    case WSANO_RECOVERY    : return "non-recoverable error";
    case WSANO_DATA        : return "valid name but no data";
    default                :
         sprintf(sz80OtherError, "Other WINSOCK error, code %d",(int)iWSAErrorCode);
         return sz80OtherError;
   } // end switch( iWSAErrorCode )
} // end NtpClient_SocketErrorToString()



//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Core of the simple NTP client...

// Options for NtpClient_Update(), compatible with WSJTX_SYNC_OPTION_ (bitmasks)
#define NTP_SYNC_SHORT_INFO    0x0000
#define NTP_SYNC_VERBOSE_INFO  0x0001

// Defines for the NTP datagrams (UDP packets)
#define NTP_MODE_CLIENT        3
#define NTP_VERSION            3
// Byte offsets within the response (UDP packet) received from the remove server.
// Details about the UDP structure in RFC 1059 (Network Time Protocol July 1988),
// Appendix B.
// Note that all entries are in 32-bit groups, in annoying 'Motorola' byte order,
// thus (for example) the fields 'LI' (bits "0 to 1") and 'VN' (bits "2 to 4")
// of the first 32-bit group are NOT THE FIRST, BUT THE THIRD BYTE !
#define NTP_UDP_OFFS_LI_VN_STRAT_PREC 0  /* Leap Indicator,  Version number, 'Stratum', 'Poll', 'Precision' (bytes[0..3]) */
#define NTP_UDP_OFFS_SYNC_DISTANCE    4
#define NTP_UDP_OFFS_ESTD_DRIFT_RATE  8
#define NTP_UDP_OFFS_REF_CLOCK_ID     12
#define NTP_UDP_OFFS_REFERENCE_TIME 16 /* "Local time at which the local clock was last set or corrected" */
#define NTP_UDP_OFFS_ORIGINATE_TIME 24 /* "Local time at which the request departed the client host for the service host" */
#define NTP_UDP_OFFS_RECEIVE_TIME   32 /* "Local time at which the request arrived at the service host"   */
#define NTP_UDP_OFFS_TRANSMIT_TIME  40 /* "Local time at which the reply departed the service host for the client host" */
#define OFFSET_1900_TO_1970 0x83AA7E80 /* Number of seconds from 1900-01-01 to 1970-01-01, fits in a 32-bit UNSIGNED int */

BOOL NtpClient_Init(void)
{
  // Initialise the winsock stack
  WSADATA wsaData;

  // (try to) start winsock
  if( WSAStartup(MAKEWORD(1, 1)/*min required version*/, &wsaData) != 0)
   {
     strcpy(NtpClient_sz255LastErrorInfo, "Winsock won't cooperate.");
     WSACleanup();
     return FALSE;
   }

  return TRUE;

} // end NtpClient_Init()


DWORD NtpClient_ReadInt32_MSBFirst(BYTE* buffer, int offset)
{
  // Note: Don't use the stupid (signed) 'char' type as buffer element !
  buffer += offset;
  return ((DWORD)buffer[0]<<24)
       | ((DWORD)buffer[1]<<16)
       | ((DWORD)buffer[2]<<8)
       | ((DWORD)buffer[3]) ;
}

long double NtpClient_ReadTimestampAsUnixSeconds( BYTE *buffer, int offset)
  // Reads a 64-bit timestamp from an NTP datagram (buffer)
  // and converts it into 'unix seconds' (with fractional part).
{
  long double ldblUnixSeconds;

  // > NTP timestamps are represented as a 64-bit unsigned
  // > fixed-point number, in seconds relative to 0000 UT on 1 January 1900.
  // > The integer part is in the first 32 bits and the fraction part in the
  // > last 32 bits.
  //  (WB : if the fraction could reach 2^32, that would be ONE SECOND.
  //        Since we don't know if the compiler is smart enough to calculate
  //        1.0 / (1<<32) as a floating point constant (some are not) :
  DWORD seconds  = NtpClient_ReadInt32_MSBFirst(buffer, offset);
  DWORD fraction = NtpClient_ReadInt32_MSBFirst(buffer, offset + 4);
  DWORD dwOffset = OFFSET_1900_TO_1970;
  ldblUnixSeconds = (long double)seconds  - (long double)dwOffset
                  + (long double)fraction / (long double)4294967296.0/*1<<32*/;
  return ldblUnixSeconds;
} // end NtpClient_ReadTimestampAsUnixSeconds()


void NtpClient_WriteTimestampFromUnixSeconds( BYTE* buffer, int offset, long double ldblUnixSeconds)
  // Writes a 64-bit timestamp into an NTP datagram (buffer) .
  // Converts a timestamp from 'unix seconds' (with fractional part) into
  // the format used for NTP, and puts it as a 64-bit timestamp into a buffer.
{
  DWORD  seconds  = (DWORD)floorl( ldblUnixSeconds );
  long double ldblFract = fmodl(ldblUnixSeconds, 1.0); // can we trust fmodl() ?
  DWORD  fraction = (DWORD)( ldblFract * (long double)4294967296.0/*1<<32*/ );
  DWORD  dwOffset = OFFSET_1900_TO_1970;
  seconds += dwOffset; // result should not exceed 32 bits, at least not before year 2036:
   //
   // Test case : 2014-02-08 00:00:00 (UTC), the Unix timestamp was 1391817600 [seconds] = 0x52F57380 .
   //    Add 'OFFSET_1900_TO_1970' : 0x52F57380 + 0x83AA7E80 = 0xD69FF200 .
   //    Some spare, but not too much, until overflow (0xFFFFFFFF -> 0) :
   // > Should NTP be in use in 2036, some external means will be necessary
   // > to qualify time relative to 1900 and time relative to 2036
   // > (and other multiples of 136 years).


  // write seconds in big endian format
  buffer[offset++] = (seconds >> 24);
  buffer[offset++] = (seconds >> 16);
  buffer[offset++] = (seconds >> 8);
  buffer[offset++] = (seconds >> 0);

  // write fraction in big endian format
  buffer[offset++] = (fraction >> 24);
  buffer[offset++] = (fraction >> 16);
  buffer[offset++] = (fraction >> 8);
  // low order bits should be random data (??)
  buffer[offset++] = random(256);
} // end NtpClient_WriteTimestampFromUnixSeconds()


//--------------------------------------------------------------------------
BOOL NtpClient_Update(
     char *pszHostname,          // [in] something like "129.6.15.30" or "time-c.nist.gov"
     int   iOptions,             // [in] NTP_SYNC_SHORT_INFO, NTP_SYNC_VERBOSE_INFO, ..
     void (pvInfoStringOuput(char *cpInfo)), // [in, optional] procedure to 'print info strings'
     long double *pClockOffset ) // [out] offset, in seconds, which must be added to the 'local' system time to correct it
  // Determines the clock offset between the PC's system time
  // and the 'reference' time on a remote NTP server.
  // Does NOT modify the PC's local time (that's done in TNtpClientForm::SynchronizeNow) !
  //
  // Must not be called too often !
  // One day, NIST may have to block 'too frequent callers' because their server
  // is permanently flooded by requests from poorly written software. Quote from NIST:
  //
  // > All users should ensure that their software NEVER queries a server
  // > more frequently than once every 4 seconds. Systems that exceed this rate
  // > will be refused service. In extreme cases, systems that exceed this limit
  // > may be considered as attempting a denial-of-service attack.
  //
{
  int  portno=123; // NTP is port 123
  #define L_MAX_BUF_LEN 48 // check our buffers
  int   i,nBytesRcvd;
  BYTE  msg[L_MAX_BUF_LEN]; // the packet we send
  BYTE  buf[L_MAX_BUF_LEN]; // the buffer we get back
  char  sz255Info[256], *cpInfo;
  char  *cp;
  struct sockaddr_in server_addr;
  struct sockaddr saddr;
  struct hostent *hp;
  DWORD  dw;
  socklen_t saddr_l;
  SOCKET s;        // socket
  long double ldblTemp;
  long double ldblLocalTimeOfRequest;
  long double ldblLocalTimeOfResponse;
  // Note: 'local' time doesn't apply to the local TIMEZONE;
  //       in this case it means 'date and time of the LOCAL PC, but measured in Unix seconds.
  //       All THESE 'times' are actually date+time in factional Unix seconds,
  //       regardless of the operating system and the internal time format (in NTP) !
  long double ldblOriginateTimeFromNtp;
  long double ldblReceiveTimeFromNtp;
  long double ldblTransmitTimeFromNtp;
  long double ldblRoundTripTime;
  // long double ldblNtpTime;
  long double ldblNtpTimeReference;


  // Build an NTP request message.  Our message is all zeros except for a one in the
  // protocol version field, and our own 'request time' we send to the remote NTP server.
  // msg[] in binary is 00 001 000 00000000
  // it should be a total of 48 bytes long
  // Background: The remote NTP server needs to know OUR 'local' (but not localized) time
  //             to estimate the roundtrip delays, etc etc.
  memset( msg,0,sizeof(msg));
  msg[0]= NTP_MODE_CLIENT | (NTP_VERSION << 3);

  // Create a 'socket' to communicate with the internet (here: via UDP).
  s = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if( s==INVALID_SOCKET )
   { strcpy(NtpClient_sz255LastErrorInfo, "Failed to create a socket"  );
     return FALSE;  // Yes, shit happens. Especially with internet protocols.
   }
  // To avoid the complexity with multithreading, use a simple BLOCKING socket,
  // but to avoid getting stuck here forever, use TIMEOUTS for the socket ?
  // The question is HOW to set a maximum time spent in sendto() and especially in recvfrom(),
  //  without having to clutter this simple code with select() and co. From the BCB Help system:
  // > One major issue in porting applications from a Berkeley sockets environment
  // > to a Windows environment involves "blocking"; that is, invoking a function
  // > that does not return until the associated operation is completed.
  // > A problem arises when the operation takes an arbitrarily long time
  // > to complete: an example is a recv, which might block until data has been
  // > received from the peer system.
  // > The default behavior within the Berkeley sockets model is for a socket
  // > to operate in a blocking mode unless the programmer explicitly requests
  // > that operations be treated as nonblocking.
  // > Windows Sockets 1.1 environments could not assume pre-emptive scheduling.
  // > Therefore, it was strongly recommended that programmers use the nonblocking
  // > (asynchronous) operations if at all possible.
  // Etc etc.. the easiest workaround, e.g. using meaningful TIMEOUTS at least
  // for reception, was not mentioned anywhere.  WB spent some time to find THIS (at msdn):
  // > setsockopt(s,l, SO_RCVTIMEO, .., ..) :
  // >    Sets the timeout, in milliseconds, for blocking receive calls.
  // Sounds as if this is just what the doctor prescribed, or what ?
  // Beware, the incompatibilities between WINDOWS and LINUX/UNIX are shocking (as usual):
  // For setsockopt( .., SO_RCVTIMEO, address_of_something, size_of_something ),
  // that 'something' is a DWORD in milliseconds for WINDOWS,
  //                 and a struct timeval for LINUX/UNIX ! !
  dw = 5000;  // MILLISECONDS ! Must be longer than you would expect..
  if( SOCKET_ERROR==setsockopt( s, SOL_SOCKET, SO_RCVTIMEO,(const char*)&dw,sizeof(dw) ) )
   { strcpy(NtpClient_sz255LastErrorInfo, "Failed to set the socket's receive-timeout" );
     closesocket( s );
     return FALSE;  // Again, shit happens. Especially with internet protocols...
   }

  memset( &server_addr, 0, sizeof( server_addr ));
  server_addr.sin_family=AF_INET;
  // To keep it simple, if the 'host name' begins with a digit, assume it's a "dotted adress";
  //                    otherwise, assume it's a human-friendly name which needs a DNS lookup.
  // Because the NTP pool hostnames begin with digits, check the entire name:
  cp = pszHostname;
  while( *cp != '\0' )
   { if( (*cp>='0' && *cp<='9') || *cp=='.' )
      { ++cp;
      }
     else
      { break;
      }
   }
  if( *cp == '\0' ) // the entire string consists of digits and dots:
   { // > inet_addr() converts a string containing an Internet Protocol dotted address into an in_addr.
     server_addr.sin_addr.s_addr = inet_addr(pszHostname);
   }
  else // look up the remote server's IP address ("server_addr.sin_addr.s_addr"-YUCC..) :
   {
     if( pvInfoStringOuput && (iOptions & NTP_SYNC_VERBOSE_INFO) )
      { sprintf(sz255Info, "%s : Looking up %s",
           NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.ss",
               NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET), NULL ),
           pszHostname );
        pvInfoStringOuput( sz255Info );
      }

     // > gethostbyname() gets host information corresponding to a hostname .
     // As usual, it uses yet another funny type ("struct hostent FAR pointer")
     // to do that. Actually, server_addr.sin_addr is a 'struct  in_addr',
     // and (as usual) it's difficult to find the matching member
     // in the obfuscated 'hostent' structure (returned AS A POINTER below).
     hp = gethostbyname(pszHostname);  /* get HIS address info */
     if (hp == NULL)                   /* HE doen't exist !    */
      { sprintf(NtpClient_sz255LastErrorInfo, "gethostbyname failed (%s)",
                 NtpClient_SocketErrorToString( WSAGetLastError() )  );
        closesocket( s );
        return FALSE;  // Did you know shit happens, especially with internet protocols ?
      }
     else
      { // Not sure what hp->h_addr actually is ... just memcopy it.. eeeeek !
        memcpy((char *)&server_addr.sin_addr, hp->h_addr, hp->h_length);   /* copy address info */
        server_addr.sin_family = hp->h_addrtype; /* retrieve (IP-) address as string... */
      }
   } // end if < "numeric" or "symbolic" address (for the remote NTP server) ?
  server_addr.sin_port=htons(portno);

  if( pvInfoStringOuput && (iOptions & NTP_SYNC_VERBOSE_INFO) )
   { sprintf(sz255Info, "%s : Querying %s",
         NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.ss",
             NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET), NULL ),
         inet_ntoa( server_addr.sin_addr ) );
      pvInfoStringOuput( sz255Info );
   }

  // Fill in our own ('local') time in the sent datagram as short as possible before sending.
  // In this special case, EXCLUDE the 'deliberate' offset because
  //    ldblLocalTimeOfRequest shall be the 'real' time in UTC,
  //    *without* NtpClient_dblDeliberateOffset_sec which was *added* in TNtpClientForm::SynchronizeNow() .
  // (otherwise, the remote NTP server would suggest to correct our time
  //    by approximately NtpClient_dblDeliberateOffset_sec, which is NOT what we want !)
  ldblLocalTimeOfRequest = NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET); // -> UTC *without* the deliberate offset !
  NtpClient_WriteTimestampFromUnixSeconds(msg, NTP_UDP_OFFS_TRANSMIT_TIME, ldblLocalTimeOfRequest);
#if (0) // TEST for the Unix->NTP timestamp conversions:
  ldblTemp = NtpClient_ReadTimestampAsUnixSeconds(msg, NTP_UDP_OFFS_TRANSMIT_TIME );
  ldblTemp -= ldblLocalTimeOfRequest; // result should be way below 1 ms
  if( ldblTemp<-0.001 || ldblTemp>0.001 )
   { sprintf(NtpClient_sz255LastErrorInfo, "Bug in timestamp conversion (error=%lf seconds) !",(double)ldblTemp  );
     closesocket( s );
     return FALSE;  // If shit happens, it usually happens HERE, for any of many reasons !
   }
#endif // TEST ?

  // Send the request (UDP packet) to the remote NTP server:
  if( SOCKET_ERROR == sendto(s,msg,sizeof(msg),0,(struct sockaddr *)&server_addr,sizeof(server_addr)) )
   { // > If no error occurs, sendto returns the total number of bytes sent,
     // > which can be less than the number indicated by len. Otherwise, a value
     // > of SOCKET_ERROR is returned, and a specific error code can be retrieved
     // > by calling WSAGetLastError.
     sprintf(NtpClient_sz255LastErrorInfo, "sendto() failed (%s)",
             NtpClient_SocketErrorToString( WSAGetLastError() )  );
     closesocket( s );
     return FALSE;  // again and again and again ... shit happens, especially with internet protocols !
   } // end if < sendto() failed >

  // get the data back from the remote server, via UDP, thus use recvfrom() :
  saddr_l = sizeof (saddr);
  nBytesRcvd = recvfrom(s,buf,sizeof(buf),0,&saddr,&saddr_l);
  // Read our local clock again, as fast as possible after receiving the response:
  ldblLocalTimeOfResponse  = NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET); // -> UTC *without* the deliberate offset !
  closesocket( s );  // we don't need the socket anymore so close it

  // Remember that 'shit happens, especially with internet protocols' ?
  // > If no error occurs, recvfrom returns the number of bytes received.
  // > If the connection has been gracefully closed, the return value is zero.
  // > Otherwise, a value of SOCKET_ERROR is returned, ...
  if( nBytesRcvd != sizeof(buf) )
   { sprintf(NtpClient_sz255LastErrorInfo, "recvfrom() returned %d, expected %d .",
              (int)nBytesRcvd, (int)sizeof(buf)  );
     return FALSE;  // If shit happens, it usually happens HERE, because
                    // the remote server didn't send a response at all,
                    // or it took longer than our moderate timeout.
   }

  // If all worked as planned, we got 12 doublewords back in 'Network' byte order.
  // (geek speak, in this case it means 'MOST significant byte first') .
  // Example, copied from the hex dump created further below:
  // 2014-02-08 19:17:18.54 : Received response :
  //  Pr Po St Li  |syncDist.|  |DriftRate|  |RefClockId|
  //  1C 01 00 E3  00 00 00 00  00 00 00 00  41 43 54 53
  //  D6 A1 01 13  1E 76 BE F9  D6 A1 01 3E  5C 67 CF 65
  //  D6 A1 01 3C  20 DD 0C 1B  D6 A1 01 3C  20 DD 98 B7
  //  |___receive timestamp__|  |__transmit timestamp__|
  // Let's analyse the 'transmit timestamp' :
  //    Year: 1900 + 0xD6A1013C / 31556925(average seconds per year) = 2014.10731
  //    0.10731 years already elapsed, that's approx. 0.10731 * 365.2425 (average days per year)
  //    = 39.2 days passed in 2014, i.e. 2014-02-08 .  QED.
  //
  ldblOriginateTimeFromNtp = NtpClient_ReadTimestampAsUnixSeconds(buf, NTP_UDP_OFFS_ORIGINATE_TIME);
  ldblReceiveTimeFromNtp   = NtpClient_ReadTimestampAsUnixSeconds(buf, NTP_UDP_OFFS_RECEIVE_TIME);
  ldblTransmitTimeFromNtp  = NtpClient_ReadTimestampAsUnixSeconds(buf, NTP_UDP_OFFS_TRANSMIT_TIME);
  ldblRoundTripTime = (ldblLocalTimeOfResponse - ldblLocalTimeOfRequest)
                    - (ldblTransmitTimeFromNtp - ldblReceiveTimeFromNtp);
  NtpClient_dblClockOffset = (  (ldblReceiveTimeFromNtp  - ldblOriginateTimeFromNtp)
                 + (ldblTransmitTimeFromNtp - ldblLocalTimeOfResponse)) / 2.0;
  // Note: NtpClient_dblClockOffset is typically just a few ten milliseconds.
  //  It shall NOT include the 'deliberate' error (NtpClient_dblDeliberateOffset_sec) .
    //printf ("%lld + %lld = %ld %ld\n", (receiveTime - originateTime), (transmitTime - responseTime), (receiveTime - originateTime + transmitTime - responseTime)/2, clockOffset);
  // ldblNtpTime = ldblLocalTimeOfResponse + NtpClient_dblClockOffset;
  if( pvInfoStringOuput && (iOptions & NTP_SYNC_VERBOSE_INFO) )
   { sprintf( sz255Info, "%s : Received response :",
        NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.ss",
        NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET), NULL) );
     pvInfoStringOuput( sz255Info );
#  if(0) // show a hex dump of the entire NTP response ?
     cpInfo = sz255Info;
     for(i=0; i<nBytesRcvd; ++i)
      { sprintf( cpInfo, " %02X", (int)buf[i] ); // simple HEX dump
        if( (i & 15) == 15)
         { strcat( cpInfo, "\r\n" );
         }
        else if( (i & 3) == 3)
         { strcat( cpInfo, " " );
         }
        cpInfo += strlen(cpInfo);
      }
     pvInfoStringOuput( sz255Info );
#   endif // show NTP response as hex dump ?

     sprintf( sz255Info, " Originate time from NTP : %s",
        NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.sss", ldblOriginateTimeFromNtp, NULL) );
     pvInfoStringOuput( sz255Info );
     sprintf( sz255Info, " Receive time from NTP   : %s",
        NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.sss", ldblReceiveTimeFromNtp, NULL) );
     pvInfoStringOuput( sz255Info );
     sprintf( sz255Info, " Transmit time from NTP  : %s",
        NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.sss", ldblTransmitTimeFromNtp, NULL) );
     pvInfoStringOuput( sz255Info );
     sprintf( sz255Info, " Round trip     : %lf seconds", (double)ldblRoundTripTime );
     pvInfoStringOuput( sz255Info );
     sprintf( sz255Info, " Clock offset   : %lf seconds", (double)NtpClient_dblClockOffset );
     pvInfoStringOuput( sz255Info );
   } // end if( pvInfoStringOuput )

  if( pClockOffset != NULL )
   { *pClockOffset = NtpClient_dblClockOffset;  // Note: 'what to do' with the measured offset is up to the caller !
     // In the most simple case, add 'clockOffset' to the system time read out
     // by any suitable OS 'get current date and time' - function (with sufficient resolution).
   }

  return TRUE;

} // end NtpClient_Update()

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Above: "core" of the ridiculously simple NTP client.
// Below: special options, without Borland VCL dependencies, but used by the GUI.
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// Dialog button codes (not button IDs, but return codes for the modal dialogs)
#define DLG_BTN_NONE   0
#define DLG_BTN_OK     1  // .. or "YES" (same meaning)
#define DLG_BTN_YES    1
#define DLG_BTN_CANCEL 2  // also returned when instead of  'Yes' or 'No' the user CLOSED the dialog box !
#define DLG_BTN_HELP   3
#define DLG_BTN_NO     4  // sometimes required to distinguish between 'No' and 'Cancel'/'Abort'/'Close' !
#define DLG_BTN_USER1  11
#define DLG_BTN_USER2  12

#define DLG_ID_EDIT1   21 /* control ID for the edit field in the dialog */

// Edit field options (used in some simple dialogs)
#define  DLG_EDIT_NORMAL       0
#define  DLG_EDIT_DECIMAL_ONLY 0x0010
#define  DLG_EDIT_HEXADECIMAL  0x0020

//---------------------------------------------------------------------------
// internal variables for the "simple dialog window" (input dialog w/o resources)
//---------------------------------------------------------------------------
static int   Dlg_iEditOptions = 0;
static char *Dlg_pszTitle = "NoTitle";
static char *Dlg_pszLabel1= "NoLabel1";
static char *Dlg_pszLabel2= "NoLabel2";
static char  Dlg_sz80EditedString[84];
static HWND  Dlg_hwndModalDialog;
static HWND  Dlg_hwndEditField;
static long  Dlg_i32BackgndColor = 0x00F0E0E0; /*light bluish gray by default*/
static BOOL  Dlg_fDialogWndclassRegistered = FALSE;
static BOOL  Dlg_fDialogWindowClosed;
static int   Dlg_iModalResult;

//------------------------------------------------------------------------
LONG Dlg_OnWmPaint( HWND hWnd )
{ // reaction to a WM_PAINT message
  HDC hDC;
  HBRUSH hBrush,hOldBrush;
  PAINTSTRUCT ps;
  RECT rctClientArea;
  int  iScreenWidth,iScreenHeight;

  hDC = BeginPaint(hWnd,&ps);

  // Center a message in the "parameter display area"..
  GetClientRect(hWnd,&rctClientArea);
  iScreenHeight = rctClientArea.bottom-rctClientArea.top;
  iScreenWidth  = rctClientArea.right-rctClientArea.left;
  if(iScreenWidth>50 && iScreenHeight>50)
   {

    // Erase the entire background (that's why we don't want WM_ERASEBKGND)
    hBrush  = CreateSolidBrush( Dlg_i32BackgndColor );
    hOldBrush = SelectObject( hDC, hBrush );
    SetBkColor( hDC, Dlg_i32BackgndColor );
    FillRect( hDC, &rctClientArea, hBrush );
    if( Dlg_pszLabel1 != NULL )
     { TextOut( hDC, 16/*x*/, 8/*y*/, Dlg_pszLabel1, strlen(Dlg_pszLabel1) );
     }
    if( Dlg_pszLabel2 != NULL )
     { TextOut( hDC, 16/*x*/, 32/*y*/, Dlg_pszLabel2, strlen(Dlg_pszLabel2) );
     }
    SelectObject( hDC, hOldBrush );
    DeleteObject( hBrush );
   }

  EndPaint(hWnd,&ps);


  // An application should return zero if it processes this message...
  return 0L;
} // end Dlg_OnWmPaint()



/****************************************************************************/
static LRESULT CALLBACK Dlg_WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
  // "window procedure" (message handler) for a simple "dialog"(*) window
  // (*) it's modal, but in reality it's NOT a dialog window !
{
  switch(message)
   {
     // case WM_INITDIALOG:  // not handled here ! we look like a dialog but aren't !
     case WM_COMMAND:
        switch(LOWORD(wParam))
         {
           case DLG_BTN_OK:
                if( Dlg_hwndEditField != NULL )
                 { memset( Dlg_sz80EditedString, 0, sizeof(Dlg_sz80EditedString) );
                   GetWindowText( Dlg_hwndEditField, Dlg_sz80EditedString, 80 );
                 }
                Dlg_iModalResult = DLG_BTN_OK;
                SendMessage( hWnd, WM_CLOSE, 0, 0 ); // CloseWindow( hWnd ) didn't work
                return 0;
           case DLG_BTN_CANCEL:
                Dlg_iModalResult = DLG_BTN_CANCEL;
                SendMessage( hWnd, WM_CLOSE, 0, 0 );
                return 0;
         }
        break;

     case WM_ERASEBKGND: // sent when the window background must be erased
        // > If the application returns zero, the window will remain marked for erasing.
        return 0;

     case WM_PAINT:
        return Dlg_OnWmPaint( hWnd );

     case WM_CLOSE:  // someone closed the dialog window, somehow..
        // > The WM_CLOSE message is sent as a signal that a window
        // > (or an application, which is not the case here) should terminate.
        // > An application can prompt the user for confirmation, prior to
        // > destroying a window, by processing the WM_CLOSE message and calling
        // > the DestroyWindow function only if the user confirms the choice.
        // (a-ha. sounds like we should call DestroyWindow() in the window procedure)
        Dlg_fDialogWindowClosed = TRUE; // the window hasn't closed ITSELF yet
        DestroyWindow( hWnd );  // sends WM_DESTROY and cleans up..
        // > (WM_QUIT) If an application processes this message, it should return zero.
        // (not the stupid info from someone talking about Win32 API programming..
        //    > "in general you return FALSE for messages you don't process," NO-NO-NO !
        return 0;  // return ZERO because we DID process this message .
     case WM_DESTROY: // we're being destroyed;
        Dlg_fDialogWindowClosed = TRUE; // should already have been set in WM_CLOSE, anyway..
        break;     // let DefWindowProc() look at this, too

     default:
        break;  // let DefWindowProc() do whatever it needs to ... see below
        // DON'T ASSUME ANYTHING, ESPECIALLY NOT WHETER TO RETURN "TRUE" OR "FALSE" HERE !
   } // end switch(Message)

  // By order of the prophet (Micro$oft, in the Win32 programmer's ref or the MSDN):
  // > An application-defined window procedure should pass any messages
  // > that it does not process to the DefWindowProc function for default
  // > processing.
  // Note that -for good reason- they don't say anything about
  //      *THE VALUE* to return .
  // You'll sometimes find advice on the net like this:
  //    > in general you return FALSE for messages you don't process,
  //    > and TRUE for messages you do process
  //  THIS MAY BE TRUE "IN MANY CASES" BUT TAKEN STRICTLY, IT'S WRONG .
  //  Do you know which messages M$ will come up with in windoze 7,8,9 ?  ;o)
  return DefWindowProc (hWnd, message, wParam, lParam) ;
} // end Dlg_WndProc()

/****************************************************************************/
static int Dlg_RunModalDialog( HWND hwndOwner, // INTERNAL function !
              int iWidth, int iHeight )
{
  int  x,y, screen_width, screen_height, button_width;
  WNDCLASS wndclass ;
  HWND hwndOldFocus;
  MSG  Msg;
  static BOOL already_here = FALSE;

  if( APPL_hInstance==NULL || hwndOwner==NULL )
   { return DLG_BTN_CANCEL;  // say "not open for business"
   }
  if( already_here )   // avoid recursion ! this dialog mustn't call itself
   { return DLG_BTN_CANCEL;  // say "not open for business"
   }
  already_here = TRUE;

  // Forget about "CreateDialog" and similar - they all need RESOURCE FILES,
  // or at least RESOURCE TEMPLATES .  Googling around for a while showed this:
  // > I've been reading about that DialogBox function and tried to understand,
  // > why the window which is displayed using that function becomes modal,
  // > I was actually trying to read between the lines, there had to be
  // > something which triggered that effect, that's what I've been looking for.
  // > As I went deeper, I found out that dialog boxes simply disable
  // > the owner window , so what I actually wanted to hear in the reply,
  // > was this one line (two actually):
  // >  Code:
  // > // add this line before calling the ShowWindow to open a modal window
  // > wasEnabled = EnableWindow(hwndOwner, FALSE); // disable the owner window
  // > // before calling function to close the child (modal) window, add this:
  // > EnableWindow(hwndOwner, wasEnabled);
  // But that's not the full story. The application's endless message loop
  // must still be kept alive, so we cannot simply run in our own little loop
  // here (or can we .. until the OK- or CANCEL- button was clicked ? ? )
  // The catch is that a function like RunStringEditDialog() should not return
  // to the caller until the input is "finished". Everything else would make
  // things utterly complex (and one would be better off with a non-modal,
  //  i.e. a non-blocking window, which is not a "dialog"-window) .

  // set up window class (if not already registered in a previous call)
  if( ! Dlg_fDialogWndclassRegistered )
   { wndclass.style         = CS_HREDRAW | CS_VREDRAW ;
     wndclass.lpfnWndProc   = Dlg_WndProc ;
     wndclass.cbClsExtra    = 0 ;
     wndclass.cbWndExtra    = 0 ;
     wndclass.hInstance     = APPL_hInstance;
     wndclass.hIcon         = NULL;
     wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW);
     wndclass.hbrBackground = (HBRUSH)COLOR_WINDOW;
     wndclass.lpszMenuName  = 0 ;   // no menu !
     wndclass.lpszClassName = "YhfDlgWindow"; // window name
     RegisterClass (&wndclass) ;
     Dlg_fDialogWndclassRegistered = TRUE;
   }

  // Set up some internal (static) variables for the window procedure:
  Dlg_fDialogWindowClosed = FALSE; // the window hasn't closed ITSELF yet
  Dlg_iModalResult = DLG_BTN_NONE;
  screen_width  = GetSystemMetrics(SM_CXSCREEN);
  screen_height = GetSystemMetrics(SM_CYSCREEN);
  hwndOldFocus  = GetFocus();

  // now create the dialog window (which is a "normal" window so far..)
  x = (screen_width-iWidth)   / 2;
  y = (screen_height-iHeight) / 2;
  Dlg_hwndModalDialog = CreateWindow(
    "YhfDlgWindow",   // lpClassName,    pointer to registered class name
    Dlg_pszTitle, // "lpWindowName", pointer to window name, effectively the TITLE
    // DWORD dwStyle,  window style. Tried a lot ...
    // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX,  // non-sizeable window
    // WS_VSCROLL | WS_HSCROLL |  // scroll bars ? heavens, no !
    // WS_OVERLAPPEDWINDOW | WS_VISIBLE, // quite normal window, but SIZEABLE which we don't want
    // WS_POPUP | WS_DLGFRAME ,   // window with no title, quite unusual as main window
    // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_SIZEBOX, // sizeable
    // WS_CAPTION,   // with title, but no icon and no system menu nor close symbol
    // WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX, // suggested somewhere, but produced garbage here
    // WS_OVERLAPPEDWINDOW ^ WS_THICKFRAME) | WS_VISIBLE, // a thick frame is a sick frame, it makes our dialog SIZEABLE ?!
    // WS_OVERLAPPEDWINDOW = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_THICKFRAME | WS_MINIMIZEBOX | WS_MAXIMIZEBOX)
       WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_VISIBLE, // is THIS a good "dialog" window ? 
    //
    x, y, iWidth, iHeight,
    hwndOwner,      // HWND hWndParent, handle to parent or owner(!) window
    NULL,           // HMENU hMenu,     handle to menu or child-window identifier
    APPL_hInstance, // HANDLE hInstance, handle to application instance
    NULL);          // LPVOID lpParam,   pointer to window-creation data
  if( Dlg_hwndModalDialog != NULL )
   { // CreateWindow (for the "dialog") was successfull...
     EnableWindow(hwndOwner, FALSE); // disable the owner window
     button_width = 64;
     x = (iWidth-3*button_width) / 2;
     CreateWindow( "BUTTON"/*ClassName*/,  "Ok"/*window name*/,
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | BS_DEFPUSHBUTTON	 | BS_TEXT /*Style*/,
                  x, iHeight-70/*y*/,
                  button_width, 24/*height*/,
                  Dlg_hwndModalDialog/*hWndParent*/,
                  (HMENU)DLG_BTN_OK, /*hMenu abused for the CONTROL ID - gg "Assigning a control id to a win32 button" */
                  APPL_hInstance, NULL);
     CreateWindow( "BUTTON"/*ClassName*/,  "Cancel"/*window name*/,
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | BS_PUSHBUTTON | BS_TEXT /*Style*/,
                  iWidth-button_width-x/*x*/, iHeight-70/*y*/,
                  button_width, 24/*height*/,
                  Dlg_hwndModalDialog/*hWndParent*/,
                  (HMENU)DLG_BTN_CANCEL, APPL_hInstance, NULL);
     Dlg_hwndEditField = CreateWindow( "EDIT"/*ClassName*/,
                  Dlg_sz80EditedString, /* "window name" (here: initial text) */
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | WS_BORDER | ES_LEFT  /*Style...*/
                     | ( (Dlg_iEditOptions & DLG_EDIT_DECIMAL_ONLY )  // allow ONLY NUMERIC input ?
                          ? ES_NUMBER	: 0 )  ,
                  16/*x*/, iHeight-64-32/*y*/,
                  iWidth-34, 24/*height*/,
                  Dlg_hwndModalDialog/*hWndParent*/,
                  (HMENU)DLG_ID_EDIT1, APPL_hInstance, NULL);
     // Set the focus (with the blinking cursor) to the EDIT FIELD (not to the "ok"-button) :
     SetFocus( Dlg_hwndEditField );

     // Dialog loop: process all window messages (not just "our own"! ),
     //      SIMILAR to what you'll find in most win32 API application .
     // But: If GetMessage() receives the WM_QUIT message, the return value is zero.
     //      This doesn't help us to return from the modal "dialog" here, because
     //      the WM_QUIT message indicates a request to terminate an *application*
     //      and is generated when the application calls the PostQuitMessage
     //      function. It causes the GetMessage function to return zero.
     // Fix: If Dlg_WndProc() closes its window, it kindly sets a flag
     //      which we stupidly poll in the message loop below .
     while(GetMessage(&Msg, NULL, 0, 0))
      {
        // From MSDN (remember this if the TAB KEY doesn't switch the active control..)
        // > Although the IsDialogMessage function is intended for modeless dialog boxes,
        // > you can use it with any window that contains controls,
        // > enabling the windows to provide the same keyboard selection
        // > as is used in a dialog box.
        // > When IsDialogMessage processes a message, it checks for keyboard messages
        // > and converts them into selections for the corresponding dialog box.
        // > For example, the TAB key, when pressed, selects the next control
        // > or group of controls, and the DOWN ARROW key, when pressed,
        // > selects the next control in a group.
        // > Because the IsDialogMessage function performs all necessary translating
        // > and dispatching of messages, a message processed by IsDialogMessage
        // > must *not* be passed to the TranslateMessage or DispatchMessage function.
        //
        if (!IsDialogMessage( Dlg_hwndModalDialog, &Msg) ) // important for TAB-nav.
          {
            TranslateMessage(&Msg);
            DispatchMessage(&Msg);
          }
        if( Dlg_fDialogWindowClosed )
           break;  // workaround for the missing WM_QUIT in the dialog
      }

     EnableWindow(hwndOwner, TRUE );
     SetFocus( hwndOldFocus );  // ex: SetFocus(hwndOwner);
   } // end if < creation of the dialog window successful >

  already_here = FALSE;  // accept next call
  return Dlg_iModalResult; // DLG_BTN_OK, DLG_BTN_CANCEL; maybe more

} // end RunModalDialog()


/****************************************************************************/
int RunIntegerInputDialog(  // MODAL dialog; returns DLG_BTN_...
  HWND hwndOwner,  // handle to the window which will be blocked by the dialog
  char *pszTitle, char *pszLabel1,  char *pszLabel2,
  int  *piValue,  int iEditOptions, int  iHelpContext )
  // return value:  one of the DLG_BTN_ codes .
{
   int iButtonCode;

   Dlg_iEditOptions = iEditOptions;
   if( (iEditOptions & DLG_EDIT_HEXADECIMAL) == 0 )
    { Dlg_iEditOptions |= DLG_EDIT_DECIMAL_ONLY; // allow ONLY NUMERIC input
    }
   Dlg_pszTitle = pszTitle;
   if(pszLabel1)
           Dlg_pszLabel1 = pszLabel1;
     else  Dlg_pszLabel1 = "";
   if(pszLabel2)
           Dlg_pszLabel2 = pszLabel2;
     else  Dlg_pszLabel2 = "";
   sprintf(Dlg_sz80EditedString, "%d",  *piValue);
   iButtonCode = Dlg_RunModalDialog(hwndOwner, 320/*w*/, 160/*h*/ );
   if( iButtonCode==DLG_BTN_OK )
    {
      if(Dlg_sz80EditedString[0]=='$')
        { sscanf(Dlg_sz80EditedString+1,"%X", piValue );
          // beware of lousy error handling in some scanf implementations !
        }
      else
        { sscanf(Dlg_sz80EditedString,"%d", piValue );
        }
    }
   return iButtonCode;
} // end RunIntegerInputDialog()


/****************************************************************************/
int RunFloatInputDialog(  // MODAL dialog; returns DLG_BTN_...
  HWND hwndOwner,  // handle to the window which will be blocked by the dialog
  char *pszTitle, char *pszLabel1, char *pszLabel2,
  double *pdblValue,  int iEditOptions, int  iHelpContext )
  // return value:  one of the DLG_BTN_ codes .
{
  double dblTemp;
  int  iButtonCode;

  Dlg_iEditOptions = iEditOptions;
  Dlg_pszTitle = pszTitle;
  if(pszLabel1)
       Dlg_pszLabel1 = pszLabel1;
  else Dlg_pszLabel1 = "";
  if(pszLabel2)
       Dlg_pszLabel2 = pszLabel2;
  else Dlg_pszLabel2 = "";
  sprintf(Dlg_sz80EditedString,"%.3lf", *pdblValue);
  iButtonCode = Dlg_RunModalDialog(hwndOwner, 320/*w*/, 160/*h*/ );
  if( iButtonCode==DLG_BTN_OK )
   {
     if(Dlg_sz80EditedString[0]>0)
      { dblTemp = strtod( Dlg_sz80EditedString, NULL);
        if( dblTemp!=HUGE_VAL && dblTemp!=-HUGE_VAL )
         { *pdblValue = dblTemp;
           return DLG_BTN_OK;
         }
      }
   }
  return iButtonCode;
} // end RunFloatInputDialog()


//---------------------------------------------------------------------------
// BELOW: Borland-specific stuff ! You're invited to adapt this for Qt (etc)..
//---------------------------------------------------------------------------

TrsNTP *rsNTP;  // <<< Borland-specific stuff

//---------------------------------------------------------------------------
CPROT void MyInfoStringOuput(char *cpInfo) // callback procedure to 'print info strings'.
  // Must be callable from 'C' (not just 'C++'), thus the 'CPROT' .
{
  rsNTP->Mem_Output->Lines->Add( cpInfo );
  if( g_VerboseOutput ) // immediately update the display ...
   { rsNTP->Update();   // ... to find out where the program gets stuck (if it gets stuck)
     // ("rsNTP" is only the GUI but not the NTP client itself,
     //  and "Update" is just a VCL method (Visual Component Library) )
   }

} // end MyInfoStringOuput()

//---------------------------------------------------------------------------
__fastcall TrsNTP::TrsNTP(TComponent* Owner)
   : TForm(Owner)
{ WindowProc = MyWindowProc;
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TrsNTP::MyWindowProc(Messages::TMessage &Message)
  // Low level (windows-)message handler.
  // Installed by  the application's main window : "WindowProc = MyWindowProc" .
  // Receives both SENT and POSTED window messages
  //   (in contrast to the lousy TApplication.OnMessage-thingy...)
{
  T_BLONG blCommand; // byte-wise addressable 'long' parameter (1*32 bit, 2*16 bit, 4*8 bit)
  int     iCmdLength;
  char    *cpCmdLine;


  switch( Message.Msg )
   {
      case WM_COPYDATA :
        // The WM_COPYDATA message is sent when an application passes data
        //                 to another application.
        //
        // WM_COPYDATA
        // wParam = (WPARAM) (HWND) hwnd;            // handle of sending window
        // lParam = (LPARAM) (PCOPYDATASTRUCT) pcds; // pointer to structure with data
        //
        // Parameters
        //   hwnd   Identifies the window passing the data.
        //   pcds   Points to a COPYDATASTRUCT structure that contains
        //          the data to be passed.
        // Return Values
        //   If the receiving application processes this message,
        //   it should return TRUE; otherwise, it should return FALSE.
        //
        // Remarks
        //   An application must use the SendMessage function to send this message,
        //   not the PostMessage function.
        //   The data being passed must not contain pointers or other references
        //   to objects not accessible to the application receiving the data.
        //   While this message is being sent, the referenced data must not
        //   be changed by another thread of the sending process.
        //   The receiving application should consider the data read-only.
        //   The pcds parameter is valid only during the processing of the message.
        //   The receiving application should not free the memory referenced by pcds.
        // ! If the receiving application must access the data after SendMessage
        // ! returns, it must copy the data into a local buffer.
        if( Message.LParam != NULL)
         { // only if the pointer to a COPYDATASTRUCT struct exists:
           // typedef struct tagCOPYDATASTRUCT {  // cds
           //    DWORD dwData;    // Specifies up to 32 bits of data
           //                     // to be passed to the receiving application.
           //    DWORD cbData;    // Specifies the size, in bytes, of the data
           //                     // pointed to by the lpData member.
           //    PVOID lpData;    // Points to data to be passed to the receiving
           //                     // application. This member can be NULL.
           //  } COPYDATASTRUCT;
           // Shall the received WM_COPYDATA message be processed by our tiny
           // 'Command Line Interpreter' (module identifier 'CL', see details
           //         at www.qsl.net/dl4yhf/yhf_comm/yhf_comm_info.htm )  ?
           blCommand.dw = ((PCOPYDATASTRUCT)Message.LParam)->dwData;
           // Split the 32 bit "command" into 4 bytes (8-bit characters) :
           //   b[0]="Module1", b[1]="Module2", b[2]="Command1",  b[3]="Command2" )
           switch(blCommand.b[0])  // first 'Module' letter in command ...
            {
              case 'A':  // It's an "Audio Message" (a message containing AUDIO DATA)..
              default :  // .. or any of the other functions which we DO NOT support here
                 break;  // not processed here (in rsNTP)

              case 'C':  // It may be something for the 'Command Line Interpreter' (Module ID = "CL") ..
                 if( blCommand.b[1] == 'L' )
                  { // as specified in www.qsl.net/dl4yhf/yhf_comm/yhf_comm_info.htm,
                    // the command line interpreter ("CL") should support
                    //      'CF' = "Calculate Function", returns a result to the caller
                    // and  'EC' = "Execute Command", doesn't return anything to the caller
                    iCmdLength = ((PCOPYDATASTRUCT)Message.LParam)->cbData,  // size of datablock in bytes
                    cpCmdLine  = (char*)(((PCOPYDATASTRUCT)Message.LParam)->lpData); // pointer to datablock
                    if( (cpCmdLine!=NULL) && (iCmdLength>0) && (iCmdLength<255) )
                     { if( (blCommand.b[2] == 'E') && (blCommand.b[3] == 'C') )
                        { // "Execute Command": In this case, the command is a
                          // string of characters, passed through WM_COPYDATA.
                          strncpy( NtpClient_sz255RcvdCmd, cpCmdLine, 255 );
                          NtpClient_sz255RcvdCmd[ iCmdLength ] = '\0'; // turn this into a nice "C"-string
                          NtpClient_hwndCmdSender= (HWND)Message.WParam;
                          NtpClient_fRcvdCommand = TRUE;
                        } // end if < "CL EC" = Command Line Interpreter : Execute Command >
                     } // end if < WM_COPYDATA message contained a nonempty data block >
                  }  // end if < "CL" = "Command Line" (interpreter command sent via WM_COPYDATA) >
                 break; // end case WM_COPYDATA with 'C' in the 1st 'data' bytes (here used as 'Module ID')
            } // end switch <first MODULE letter in command parameter>
         } // end if Message.LParam != NULL
        break; // end case WM_COPYDATA

     default:
        WndProc(Message); // WndProc ist die "Original-Prozedur" fuers Message-Handling, allerdings nicht von Windows sondern von Borland's VCL !
        break;
   } // end switch( Message.Msg )
} // end TrsNTP::MyWindowProc()
//---------------------------------------------------------------------------


void __fastcall TrsNTP::FormCreate(TObject *Sender)
{
  char sz255[256], *cpInfo;
#define L_TEST 0
#if(L_TEST) // TEST : Convert 2014-02-08 00:00:00 to UNIX, NTP, and back to UNIX:
  long double ldblUnixSeconds, ldblUnixSeconds2, ldblDiff;
  BYTE buffer[8];
#endif

  APPL_hInstance = GetModuleHandle(NULL);

  // Load the settings from an old-fashioned INI file. AVOID THE REGISTRY.
  Chk_SyncPeriodically->Checked=ReadIntegerFromIniFile(APPL_szIniFileName, "NTP_Client", "PeriodicSync", 1 );
  Chk_VerboseOutput->Checked = ReadIntegerFromIniFile( APPL_szIniFileName, "NTP_Client", "VerboseOutput", 0 );
  NtpClient_iSpecialOptions  = ReadIntegerFromIniFile( APPL_szIniFileName, "NTP_Client", "SpecialOptions", 0 );
  NtpClient_iBeginOffsettingAtSecondOfHour = ReadIntegerFromIniFile( APPL_szIniFileName, "NTP_Client", "BeginOffsettingAtSecondsOfHour", NtpClient_iBeginOffsettingAtSecondOfHour );
  NtpClient_iEndOffsettingAtSecondOfHour   = ReadIntegerFromIniFile( APPL_szIniFileName, "NTP_Client", "EndOffsettingAtSecondsOfHour",   NtpClient_iEndOffsettingAtSecondOfHour );
  NtpClient_dblSpecialOffset1_sec = ReadDoubleFromIniFile( APPL_szIniFileName, "NTP_Client", "SpecialOffset1_Seconds", NtpClient_dblSpecialOffset1_sec );
  strncpy(sz255, CB_NtpServer->Text.c_str(), 255);
  // Caution ! The "Server" name may be a special item like "WSJT-X/FT8 'DT'",
  //           which at THIS point may, or may not, exist in CB_NtpServer->Items !
  //           No problem for a (Borland-)VCL TComboBox, because TComboBox.Text
  //           doesn't necessarily need to match any of the ITEMS. From Borland,
  //           with TComboBox.Style = csDropDown :
  // > Erzeugt eine Dropdown-Liste mit einem Eingabefeld zur manuellen Eingabe
  // > von Text. Alle Eintrge sind Strings mit derselben Hhe.
  //           Thus no problem if rsNTP is launched BEFORE WSJT-X
  //           with CB_NtpServer->Text = "WSJT-X/FT8 'DT'" (WSJTX_SYNC_DUMMY_SERVER_NAME).
  //
  ReadStringFromIniFile( APPL_szIniFileName, "NTP_Client", "Server",sz255/*dest*/, 255/*maxlen*/, sz255/*default*/ );
  CB_NtpServer->Text = sz255;
  ReadStringFromIniFile( APPL_szIniFileName, "NTP_Client", "SecondsAddedToNetworkTime",sz255/*dest*/, 255/*maxlen*/, "0.0"/*default*/ );
  Ed_AddSeconds->Text = sz255;
  NtpClient_dblDeliberateOffset_sec = atof( sz255 );
  if( NtpClient_dblDeliberateOffset_sec==HUGE_VAL )
   {  NtpClient_dblDeliberateOffset_sec = 0.0;
   }

  NtpClient_Init();   // init winsock, and do whatever else is required

  sprintf( sz255, "%s (UTC) : Program started",
     NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.ss",
     NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET), NULL) );
  Mem_Output->Text = sz255;

#if(L_TEST) // TEST : Convert 2014-02-08 00:00:00 to UNIX, NTP, and back to UNIX:
  // Test case : 2014-02-08 00:00:00 (UTC), the Unix timestamp was 1391817600 [seconds];
  //             plus one millisecond which must not get lost due to rounding:
  ldblUnixSeconds = 1391817600.001;
  // UNIX timestam = 1391817600 [seconds] = 0x52F57380 .
  // Add 'OFFSET_1900_TO_1970' : 0x52F57380 + 0x83AA7E80 = 0xD69FF200 .
  // That's what we expect in the first four bytes in 'network' (big endian) byte order.
  NtpClient_WriteTimestampFromUnixSeconds( buffer, 0/*offset*/, ldblUnixSeconds );
  if( (buffer[0]!=0xD6) || (buffer[1]!=0x9F) || (buffer[2]!=0xF2) || (buffer[3]!=0x00) )
   {  Mem_Output->Lines->Add("Bug in ..WriteTimestamp");
   }
  ldblUnixSeconds2 = NtpClient_ReadTimestampAsUnixSeconds( buffer, 0/*offset*/);
  ldblDiff = ldblUnixSeconds2 - ldblUnixSeconds; // should be WAY below 1 ms !
#endif      // TEST ?

   m_SyncDisableCountdown = 0;  // 50 * 100 ms
}

void __fastcall TrsNTP::FormClose(TObject *Sender,
      TCloseAction &Action)
{
  char sz255[256];

  // Save the settings in an old-fashioned INI file. AVOID THE REGISTRY.
  WriteIntegerToIniFile( APPL_szIniFileName, "NTP_Client", "PeriodicSync", Chk_SyncPeriodically->Checked );
  WriteIntegerToIniFile( APPL_szIniFileName, "NTP_Client", "VerboseOutput",Chk_VerboseOutput->Checked );
  WriteIntegerToIniFile( APPL_szIniFileName, "NTP_Client", "SpecialOptions",NtpClient_iSpecialOptions );
  WriteIntegerToIniFile( APPL_szIniFileName, "NTP_Client", "BeginOffsettingAtSecondsOfHour", NtpClient_iBeginOffsettingAtSecondOfHour );
  WriteIntegerToIniFile( APPL_szIniFileName, "NTP_Client", "EndOffsettingAtSecondsOfHour",   NtpClient_iEndOffsettingAtSecondOfHour );
  sprintf( sz255,"%.3lf", NtpClient_dblSpecialOffset1_sec );
  WriteStringToIniFile(  APPL_szIniFileName, "NTP_Client", "SpecialOffset1_Seconds", sz255  );

  strncpy(sz255, CB_NtpServer->Text.c_str(), 255);
  WriteStringToIniFile(  APPL_szIniFileName, "NTP_Client", "Server", sz255 );
  strncpy(sz255, Ed_AddSeconds->Text.c_str(), 255 );
  WriteStringToIniFile(  APPL_szIniFileName, "NTP_Client", "SecondsAddedToNetworkTime", sz255 );


}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TrsNTP::Btn_ExitClick(TObject *Sender)
{
   Close();  // bye-bye
}
//---------------------------------------------------------------------------

void __fastcall TrsNTP::Btn_HelpClick(TObject *Sender)
{
  Mem_Output->Text =
    "Select an NTP server close to your location.\r\n"
    "Click 'Synchronize' for a one-time sync with details.\r\n"
    "For periodic synchronisation, set the checkmark above.\r\n"
    "Disable other time-sync utilities, especially window's own.\r\n"
    "Under Win 7 (etc), run the program 'as administrator',\r\n"
    "otherwise the PC's system time cannot be set.\r\n"
    "\r\n"
    "For details and updates, visit\r\n"
    " http://www.qsl.net/dl4yhf/rsNTP/rsNTP.htm";
}

void __fastcall TrsNTP::MI_WSJTX_HelpClick(TObject *Sender)
{
  Mem_Output->Text =
    "Copy a few lines from the WSJT-X 'Band Activity' window\r\n"
    "with the decoded FT8 into this window.\r\n"
    "Then click 'Synchronize' to let the program evaluate the\r\n"
    "'DT' column (= system time MINUS other station's times),\r\n"
    "compute a weighed average, and adjust the system time\r\n"
    "to minimize the time offset for own transmissions.\r\n"
    "Unlike the NTP sync, this doesn't require internet access.\r\n"
    "\r\n"
    "For details and updates, visit\r\n"
    " http://www.qsl.net/dl4yhf/rsNTP/rsNTP.htm";
}
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
BOOL TrsNTP::SynchronizeNow(
        int iOptions ) // [in] NTP_SYNC_SHORT_INFO, NTP_SYNC_VERBOSE_INFO, ..
{

  char sz255[256], sz255Hostname[256], sz255Info[256];
  long double ldblUnixSeconds, clockOffset;
  int  iYear, iMonth, iDay, iHour, iMinute;
  double d, dblSecond, dblOldClockOffset;
  SYSTEMTIME  st;
  BOOL fOk, fResult = FALSE;


  // Added 2015-10-02 for a 'very special application' (WSPR) :
  //    Fool the PC by setting a deliberately false system time
  //    to compensate the 'external delay' of an audio signal entering the PC
  //    via network plus some digital preprocessing, while WSPR 'things'
  //    the signal was directly received (in real time) via soundcard.
  // Since 2016, the 'deliberate offset' from the input field may be superseded
  //    via command from external applications, for from an 'hourly schedule'.
  NtpClient_dblDeliberateOffset_sec = atof( Ed_AddSeconds->Text.c_str() );
  if( NtpClient_dblDeliberateOffset_sec==HUGE_VAL )
   {  NtpClient_dblDeliberateOffset_sec = 0.0;
   }


  strncpy(sz255Hostname, CB_NtpServer->Text.c_str(), 255);
  sz255Hostname[255] = '\0'; // just for safety...
  // To comply with the warning at NIST : Avoid flushing the server with requests
  m_SyncDisableCountdown = 40;  // 40 * 100 ms : limit the update frequency, according to the warning at NIST
  Btn_Synchronize->Enabled = FALSE;
  if( ! Chk_SyncPeriodically->Checked )
   { Mem_Output->Lines->Clear();
   }
  if( iOptions & NTP_SYNC_VERBOSE_INFO )
   { g_VerboseOutput = TRUE;
     Mem_Output->Text = "Synchronising.."; // clear old text to have room for 'verbose info'
   }
  else
   { g_VerboseOutput = FALSE;  // flag for 'MyInfoStringOuput'
   }
  Update();
  if( strcmp( sz255Hostname, WSJTX_SYNC_DUMMY_SERVER_NAME ) == 0 )
   { // special case: Don't synchronize our clock (system time) via NTP
     // but based on the 'cleaned up' average of "DT" (FT8 Time Differences):
     fOk = WSJTX_Sync_UpdateClockOffset( // 'twin' to NtpClient_Update() without NTP but using WSJT-X...
         iOptions,          // [in] bit combination like NTP_SYNC_SHORT_INFO, NTP_SYNC_VERBOSE_INFO
         MyInfoStringOuput, // [in, optional] procedure to 'print info strings'
         &clockOffset );    // [out] offset, in seconds, to add to the 'local' system time for correction
   }
  else // no synchronisation based on WSJT-X's "DT" column (Time Differences from decoded FT8)..
   { fOk = NtpClient_Update( // Measure the ERROR (without the "deliberate" offset) between network- and local clock
         sz255Hostname,     // [in] something like "129.6.15.30" (worked) or "time-c.nist.gov" (didn't work)
         iOptions,          // [in] bit combination like NTP_SYNC_SHORT_INFO, NTP_SYNC_VERBOSE_INFO
         MyInfoStringOuput, // [in, optional] procedure to 'print info strings'
         &clockOffset );    // [out] offset, in seconds, to add to the 'local' system time for correction
   }
  if( fOk )
   { // Successfully determined the offset between the 'local' clock and the ideal value.
     // To 'correct' the PC's system time, there is a pair of rarely known functions
     // which can modify the system time 'smoothly'; see
     //   SetSystemTimeAdjustment() + GetSystemTimeAdjustment() .
     // But since this is a 'ridiculously simple NTP client', we don't use
     // the above functions (not yet?..), and try to remove the clock offset
     // 'in a single over', using  SetSystemTime() :
     // > Sets the current system time and date.
     // > The system time is expressed in Coordinated Universal Time (UTC).
     // Unfortunately, this function uses (..you guessed it..) YET ANOTHER TIME STRUCTURE,
     // called 'SYSTEMTIME'...
     memset( &st, 0, sizeof(st) );
     ldblUnixSeconds = NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITH_DELIBERATE_OFFSET) + clockOffset;
     //
     NtpClient_SplitUnixDateTimeToYMDhms( ldblUnixSeconds,
              &iYear, &iMonth, &iDay, &iHour, &iMinute, &dblSecond );
     st.wYear  = iYear;
     st.wMonth = iMonth;
     // st.wDayOfWeek : ignored when SETTING the system time
     st.wDay   = iDay;
     st.wHour  = iHour;
     st.wMinute= iMinute;
     st.wSecond= (WORD)floor(dblSecond);
     st.wMilliseconds = (WORD)( 1000.0 * fmod( dblSecond, 1.0 ) );
     if( 0 == SetSystemTime( &st )  )  // something went wrong with 'SetSystemTime' !
      {  Mem_Output->Lines->Add( "SetSytemTime failed. Try runing 'as admin' !" );
      }
     else // SetSystemTime() returned 'success'
      {  dblOldClockOffset = NtpClient_dblClockOffset;
         NtpClient_dblClockOffset = 0.0; // assume the new unwanted offset is ZERO now
         sprintf(sz255Info, "%s : Clock adjusted by %lf sec",
            NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.ss",
             NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET), NULL ),
            (double)dblOldClockOffset );
         Mem_Output->Lines->Add( sz255Info );
         d = NtpClient_GetCurrentDeliberateOffset(); // f( NtpClient_fSpecialExtraOffsetActiveNow, ... )
         if( d != 0.0 )
          { sprintf(sz255Info, "                 plus %.3lf s 'deliberate offset'", (double)d );
            Mem_Output->Lines->Add( sz255Info );
          }
         fResult = TRUE;
      }
   }
  else // something went wrong.. be prepared, SHIT HAPPENS, over and over again..
   {
     sprintf(sz255Info, "%s : %s",
            NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.ss",
             NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET), NULL ),
            NtpClient_sz255LastErrorInfo );
     Mem_Output->Lines->Add( sz255Info );
   }
  if( fResult == TRUE )
   { m_ldblUnixSecondsAtLastSync = NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITH_DELIBERATE_OFFSET);
   }

  return fResult;
} // end TNtpClientForm::SynchronizeNow()

//---------------------------------------------------------------------------
void __fastcall TrsNTP::Btn_SynchronizeClick(TObject *Sender)
{
  SynchronizeNow( NTP_SYNC_VERBOSE_INFO );
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void TrsNTP::AdjustSystemTime(
        long double ldblUnixSeconds) // [in] last known time WITHOUT deliberate offset,
                                     //      read microseconds earlier *before* modifying
                                     //      the 'deliberate offset' itself
{
  SYSTEMTIME  st;
  int    iYear, iMonth, iDay, iHour, iMinute;
  double dblSecond;
  char   sz255Info[256], *cp;

  double dblDeliberateOffset = NtpClient_GetCurrentDeliberateOffset(); // // f( NtpClient_fSpecialExtraOffsetActiveNow, ... )

  NtpClient_SplitUnixDateTimeToYMDhms( ldblUnixSeconds+dblDeliberateOffset, &iYear, &iMonth, &iDay, &iHour, &iMinute, &dblSecond );
  st.wYear  = iYear;
  st.wMonth = iMonth;
  st.wDay   = iDay;
  st.wHour  = iHour;
  st.wMinute= iMinute;
  st.wSecond= (WORD)floor(dblSecond);
  st.wMilliseconds = (WORD)( 1000.0 * fmod( dblSecond, 1.0 ) );
  if( 0 == SetSystemTime( &st )  )  // something went wrong with 'SetSystemTime' !
   { Mem_Output->Lines->Add( "SetSytemTime failed. Try runing 'as admin' !" );
   }
  else // SetSystemTime() returned 'success'
   { NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.ss", ldblUnixSeconds, sz255Info );
     cp = sz255Info+strlen(sz255Info);
     sprintf(cp, " : New deliberate offset: %5.3lg s", (double)dblDeliberateOffset );
     Mem_Output->Lines->Add( sz255Info );
   }
} // end AdjustSystemTime()


//---------------------------------------------------------------------------
void TrsNTP::ExecuteCommand( char * pszCommand, HWND hwndSender )
  // Tiny "command line interpreter" built inside this NTP client.
  // Implemented 2016-06 to let other applications (like Spectrum Lab)
  // use *A FEW* of the NTP client features.
  //
  // From Spectrum Lab's command interpreter window,
  //      this function can be invoked as follows :
  //  >   send("rsNTP", "set offset=1.234" );
  //           |_____|
  //              |
  //      the real Window Class Name is "TrsNTP", 'T' dictated by Borland,
  //      but Spectrum Lab is smart enough to try "TrsNTP" if there's
  //      no window with class name = "rsNTP" .
{ double d;
  long double ldblUnixSeconds;
  char   sz255[256], *cp;

  ldblUnixSeconds = NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET);
  NtpClient_FormatDateAndTime("YYYY-MM-DD hh:mm:ss.ss", ldblUnixSeconds, sz255 );
  cp = sz255+strlen(sz255);
  sprintf(cp, " : ExecCmd( %s )", pszCommand );
  Mem_Output->Lines->Add( sz255 );

  if( strncmp( pszCommand, "set offset=", 11) == 0 )
   { pszCommand += 11; // skip the above keyword, parse its parameter:
     d = atof(pszCommand);
     if( (d != HUGE_VAL) && (d != -HUGE_VAL) )
      {
        // ex: Ed_AddSeconds->Text = FormatFloat( "0.0##", d );  // FormatFloat is evil.. it's affected by some stupid "Locale" settings, and may replace '.' by ',' !
        sprintf( sz255, "%5.3lg", d );
        Ed_AddSeconds->Text = sz255;
        if( d != NtpClient_dblDeliberateOffset_sec )
         {
           NtpClient_dblDeliberateOffset_sec = d;
           AdjustSystemTime( ldblUnixSeconds );
         }
      }
   } // end if < "set offset" >
} // end ExecuteCommand()


//---------------------------------------------------------------------------
void __fastcall TrsNTP::Timer1Timer(TObject *Sender)
{
  int  iOptions, iSecondsOfHour, iSecondsUntilNextSync;
  static int iPrevSecondsUntilNextSync = 0;
  BOOL fDeliberateOffsetActiveNow;
  long double ldblUnixSeconds = NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITH_DELIBERATE_OFFSET);
  int    iYear, iMonth, iDay, iHour, iMinute;
  double deltaT;
  char   sz255Info[256], *cp;


  Lab_DateAndTime->Caption = NtpClient_FormatDateAndTime( "YYYY-MM-DD hh:mm:ss.s", ldblUnixSeconds, NULL );

  if( Chk_SyncPeriodically->Checked )
   {
# define SYNC_INTERVAL_IN_MINUTES 30.0
     deltaT = SYNC_INTERVAL_IN_MINUTES * 60.0 - (ldblUnixSeconds - m_ldblUnixSecondsAtLastSync);
     iSecondsUntilNextSync = (int)(deltaT + 0.5);
     if( iSecondsUntilNextSync != iPrevSecondsUntilNextSync )
      { Chk_SyncPeriodically->Hint = "next sync in "
           + IntToStr(iSecondsUntilNextSync/60) + " min "
           + IntToStr(iSecondsUntilNextSync%60) + " sec";
      }
     if( (deltaT <- 0.5 )
      || (deltaT > (SYNC_INTERVAL_IN_MINUTES+1)*60.0) ) // <- also synchronize after unexpected "time warp" !
      { // resynchronize NOW .. approx. every 30 minutes (doesn't matter exactly)
        if( Chk_VerboseOutput->Checked )
         { iOptions = NTP_SYNC_VERBOSE_INFO; // .. with immediate update of the display
         }
        else
         { iOptions = NTP_SYNC_SHORT_INFO;
         }
        if( SynchronizeNow( iOptions ) )
         { // SUCCESSFULLY synchronized: Next sync in approximately 30 minutes
           m_ldblUnixSecondsAtLastSync = ldblUnixSeconds + 1.0;
         }
        else // NTP sync failed for some reason: try again in a minute
         {   // (hopefully the internet connection will be available then)
           m_ldblUnixSecondsAtLastSync = ldblUnixSeconds - (SYNC_INTERVAL_IN_MINUTES-1)*60.0;
         }
      }
   } // end if < synchronize periodically >
  else
   { Chk_SyncPeriodically->Hint = "";
   }

  if( m_SyncDisableCountdown>0 )
   { --m_SyncDisableCountdown;
     if( m_SyncDisableCountdown==0 )
      { Btn_Synchronize->Enabled = TRUE;
      }
   }


  // Added 2016-05-29 : Time to DISABLE or ENABLE the "deliberate time offset" now ?
  if( NtpClient_iSpecialOptions & NTP_SPECIAL_OPTION_OFFSET_AT_SECONDS_OF_HOUR )
   { // To decide when the "deliberate offset" begins and ends, don't use
     // the PC's system time, but the current NTP-based time (in UTC) WITHOUT the deliberate offset:
     ldblUnixSeconds = NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITHOUT_DELIBERATE_OFFSET);
     iSecondsOfHour  = fmodl( ldblUnixSeconds, 60.0 * 60.0 );
     fDeliberateOffsetActiveNow = ( iSecondsOfHour >= NtpClient_iBeginOffsettingAtSecondOfHour )
                              &&  ( iSecondsOfHour <  NtpClient_iEndOffsettingAtSecondOfHour   );
     if( fDeliberateOffsetActiveNow != NtpClient_fSpecialExtraOffsetActiveNow ) // "Let's do the time-warp again !"
      { NtpClient_fSpecialExtraOffsetActiveNow = fDeliberateOffsetActiveNow;
        Ed_AddSeconds->Color = fDeliberateOffsetActiveNow ? clYellow : clWindow;
        // Don't query the NTP server again ("out of the schedule").
        // Instead, set the PC's local date (and time) again, with or without the offset:
        AdjustSystemTime( ldblUnixSeconds );
      } // end if < do the time-warp again > ?
   } // end if(NtpClient_iSpecialOptions & NTP_SPECIAL_OPTION_OFFSET_AT_SECONDS_OF_HOUR ) ?

  // Added 2016-06-03 : The "deliberate time offset" between system time and (minus) NTP time
  //                    can now also be modified by external programs,
  //                    communicating with the rsNTP client via WM_COPYDATA or similar.
  //  To limit the number of time adjustments per second, the real work is done HERE,
  //  in the time method, in which we can also safely "print" into the edit control:
  if( NtpClient_fRcvdCommand )
   { ExecuteCommand( NtpClient_sz255RcvdCmd, NtpClient_hwndCmdSender );
     NtpClient_fRcvdCommand = FALSE;  // "done" (have processed the received command; don't process again)
   }

}
//---------------------------------------------------------------------------


void __fastcall TrsNTP::Chk_SyncPeriodicallyClick(TObject *Sender)
{
  if( Chk_SyncPeriodically->Checked )
   { // start cycle in 2 seconds (don't wait up to 30 minutes checking 'Sync Periodically')
     m_ldblUnixSecondsAtLastSync =
           NtpClient_GetSystemDateAndTimeAsUnixSeconds(WITH_DELIBERATE_OFFSET)
         - (SYNC_INTERVAL_IN_MINUTES*60.0 - 2.0/*sec*/);

   }
}
//---------------------------------------------------------------------------


void __fastcall TrsNTP::Btn_OptionsClick(TObject *Sender)
{
  Menu_Options->Popup( Mouse->CursorPos.x, Mouse->CursorPos.y );
}
//---------------------------------------------------------------------------

void __fastcall TrsNTP::Menu_OptionsPopup(TObject *Sender)
{
  char sz80[84];
  MI_OffsetAtMinutesOfHour->Caption = "Add the SPECIAL offset (below)"\
                    " between the following times, every hour:";
  MI_OffsetAtMinutesOfHour->Checked = (NtpClient_iSpecialOptions & NTP_SPECIAL_OPTION_OFFSET_AT_SECONDS_OF_HOUR) != 0;
  sprintf(sz80, "Special offset, added to system time during the intervals below : %.1lf seconds",
                (double)NtpClient_dblSpecialOffset1_sec );
  MI_SpecialOffset_s->Caption = sz80;
  sprintf(sz80, "Begin 'offsetting' the system time at xx:%02d:%02d (every hour)",
                (int)NtpClient_iBeginOffsettingAtSecondOfHour/60,
                (int)NtpClient_iBeginOffsettingAtSecondOfHour%60);
  MI_BeginAtTimeOfHour->Caption = sz80;
  sprintf(sz80, "End   'offsetting'  the system time at xx:%02d:%02d (every hour)",
                (int)NtpClient_iEndOffsettingAtSecondOfHour/60,
                (int)NtpClient_iEndOffsettingAtSecondOfHour%60);
  MI_EndAtTimeOfHour->Caption = sz80;
}
//---------------------------------------------------------------------------

void __fastcall TrsNTP::MI_OffsetAtMinutesOfHourClick(
      TObject *Sender)
{
  NtpClient_iSpecialOptions ^= NTP_SPECIAL_OPTION_OFFSET_AT_SECONDS_OF_HOUR;
}
//---------------------------------------------------------------------------

void __fastcall TrsNTP::MI_BeginAtTimeOfHourClick(TObject *Sender)
{
  RunIntegerInputDialog(
      Handle, // [in] handle to the window which will be blocked by the dialog
      "Begin 'offsetting' ..",                 // [in] window title
      " .. the PC's system time each hour at",   // [in] char *pszLabel1
      "  [0...3599 seconds into the hour]",      // [in] char *pszLabel2
      &NtpClient_iBeginOffsettingAtSecondOfHour, // [in,out] int  *piValue
      DLG_EDIT_DECIMAL_ONLY, // [in] int iEditOptions
      0);                    // [in] int  iHelpContext
}
//---------------------------------------------------------------------------

void __fastcall TrsNTP::MI_EndAtTimeOfHourClick(TObject *Sender)
{
  RunIntegerInputDialog(
      Handle, // [in] handle to the window which will be blocked by the dialog
      "End 'offsetting' ..",                 // [in] window title
      " .. the PC's system time each hour at", // [in] char *pszLabel1
      "  [1...3600 seconds into the hour]",    // [in] char *pszLabel2
      &NtpClient_iEndOffsettingAtSecondOfHour, // [in,out] int  *piValue
      DLG_EDIT_DECIMAL_ONLY, // [in] int iEditOptions
      0);                    // [in] int  iHelpContext
}
//---------------------------------------------------------------------------


void __fastcall TrsNTP::MI_SpecialOffset_sClick(TObject *Sender)
{
  RunFloatInputDialog(
      Handle, // [in] handle to the window which will be blocked by the dialog
      "Number of seconds ..",  // [in] window title
      " .. added to the PC's system time", // [in] char *pszLabel1
      "   during the 'special' interval",  // [in] char *pszLabel1
      &NtpClient_dblSpecialOffset1_sec,    // [in,out] double *pdblValue
      DLG_EDIT_DECIMAL_ONLY, // [in] int iEditOptions
      0);                    // [in] int  iHelpContext
}
//---------------------------------------------------------------------------

void __fastcall TrsNTP::CB_NtpServerDropDown(TObject *Sender)
{
  AnsiString sLastItemText = "";
  int index_of_wsjtx = CB_NtpServer->Items->IndexOf( WSJTX_SYNC_DUMMY_SERVER_NAME );
  // About Borland's TStrings::IndexOf :
  // > If the searched string doesn't exist, IndexOf() returns -1 .
  //
  // Just before "dropping down" the contents of the list of "Servers",
  // try to find WSJT-X's main window. If it exists, add an item with text=
  // WSJTX_SYNC_DUMMY_SERVER_NAME . More on that in C:\cbproj\rsNTP\WSJTX_Sync.c .
  if( WSJTX_Sync_FindWSJTXMainWindowHandle() != (HWND)0 )
   { if( index_of_wsjtx < 0 ) // APPEND this special item to the list of "Servers":
      { CB_NtpServer->Items->Add( WSJTX_SYNC_DUMMY_SERVER_NAME );
      }
   } // end if < WSJT-X's main window seems to exist >
  else  // WSJT-X doesn't seem to run AT THE MOMENT ...
   { if( index_of_wsjtx >= 0 ) // REMOVE this special item from the list of "Servers":
      { CB_NtpServer->Items->Delete( index_of_wsjtx );
      }
   }
}
//---------------------------------------------------------------------------


void __fastcall TrsNTP::MI_WSJTX_Compatibility_TestClick(TObject *Sender)
{
  HWND hwndMain, hwndWsjtDecoderOutputWindow;
  char sz511Title[512], sz1kOutput[1024];

  Mem_Output->Text = "Trying to find WSJT-X instance .."; // clear old text to have room for 'verbose info'

#if(0) // (1)=normal compilation (searching for WSJT-X's main window),
       // (0)=test compilation to enumerate rsNTP's *own* windows/windowed controls
       //     [by traversing the tree of 'children' in
  hwndMain = WSJTX_Sync_FindWSJTXMainWindowHandle();
#else
  // Because EnumChildWindows() didn't seem to work on WSJT-X's GUI
  //   (written in Qt), try the same enumeration of windowed controls
  // in rsNTP itself (written in Borland C++Builder as a 'VCL' application):
  hwndMain = Handle;
  // > Mit der Eigenschaft Handle knnen Sie auf das Fenster-Handle
  // > des Hauptformulars (Fensters) der Anwendung zugreifen.
#endif
  if( hwndMain == (HWND)0 )
   { Mem_Output->Lines->Add( "Didn't even find WSJT-X's main window !" );
     return;
   }
  GetWindowText(hwndMain, (LPTSTR)sz511Title, sizeof(sz511Title) );
  // > The GetWindowText function copies the text of the specified window's
  // > title bar (if it has one) into a buffer.
  sprintf( sz1kOutput, "Found '%s', handle=0x%08lX",sz511Title,(long)hwndMain );
  Mem_Output->Lines->Add( sz1kOutput );

  hwndWsjtDecoderOutputWindow = WSJTX_Sync_FindHandleOfDecoderOutputWindow(
        hwndMain, WSJTX_SYNC_OPTION_VERBOSE_OUTPUT, MyInfoStringOuput );
  // 2020-08-16: When trying to enumerate OUR OWN windowed controls,
  //    WSJTX_Sync_FindHandleOfDecoderOutputWindow() -> EnumChildWindows()
  //     -> MyEnumChildWindowsProc() was called ___ times
  //
  if( hwndWsjtDecoderOutputWindow == (HWND)0 ) // oops..
   { Mem_Output->Lines->Add( "Couldn't find WSJT-X decoder output window" );
   }
}
//---------------------------------------------------------------------------


