//---------------------------------------------------------------------------
// File  :  WinStuff.cpp
// Date  :  2017-12-03   (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//   "Windows Stuff" for a NON-VCL application.  Not application specific !
//   Includes:
//    - Some wrappers for certain Windoze API calls
//    - Multiline Edit Controls  (Micro$oft RichEdit)
//    - Status Windows (the little bar at the bottom)
//    - Simple dialog windows to edit strings, integer, and floating point.
//
// Revision history (YYYY-MM-DD):
//   2017-12-03  Modified the layout of some dialog boxes because the
//                FAT UGLY BORDERS of recent windoze versions wrecked
//                the layout completely.
//                Even the GUI of Borland C++Builder itself (!)
//                was wrecked by those FAT UGLY BORDERS : Some scrollbars
//                disappeared almost completely, due to some bug in the VCL .
//
//   2014-02-21  Added the string- and integer-edit dialog here,
//               copied from DL4YHF's  YHF_Tools\YHF_Dialogs.cpp,
//               with some dependencies removed (including the module prefix).
//
//   2014-02-15  Used in DL4YHF's extension of ZL2AFP's WSQ2 .
//
//   2003-04-05  Used in DL4YHF's Complex Calculating Editor without VCL.
//                 ( but some traces of the VCL can still be found
//                   in the skeleton where WINAPI WinMain() is :-(  )
//---------------------------------------------------------------------------

#include <windows.h>
#include <richedit.h>  // RichEdit, part of windoze, no 'special' DLL
#include <commctrl.h>  // Common Controls, part of windoze, no 'special' DLL
#include <string.h>
#include <stdio.h>     // no standard I/O in use, but sprintf defined here
#include <math.h>      // HUGE_VAL defined here, at least by Borland

#include "WinStuff.h"


#ifndef FT_MATCHCASE       // should be in richedit.h but isn't ?!?
 #define FT_MATCHCASE 0x4
#endif
#ifndef FT_WHOLEWORD
 #define FT_WHOLEWORD 0x2
#endif

#ifndef __BORLANDC__       // PellesC didn't know stricmp() .. sigh...
# define stricmp _stricmp  //  ... maybe it supports _stricmp instead
#endif

//---------------------------------------------------------------------------
// Variables (global for simplicity)
//---------------------------------------------------------------------------
HINSTANCE WS_hInstance; // must be set by the application, typically in WinMain()


//**************************************************************************
// Functions to make low-level windoze programming a bit easier ...
//    .. or even more obscure ?)
//**************************************************************************

int GetWindowHeight( HWND hWnd )
{
 RECT MyRect;
  if(! hWnd)
    return 0;
  // The GetWindowRect function retrieves the dimensions of the bounding
  // rectangle of the specified window. The dimensions are given in screen
  // coordinates that are relative to the upper-left corner of the screen.
  if(! GetWindowRect( hWnd, &MyRect) )
     return 0;
  return MyRect.bottom - MyRect.top;
} // end GetWindowHeight()

int GetWindowWidth( HWND hWnd )
{
 RECT MyRect;
  if(! hWnd)
    return 0;
  if(! GetWindowRect( hWnd, &MyRect) )  // See GetWindowHeight() for details
     return 0;
  return MyRect.right - MyRect.left;
} // end GetWindowWidth()


//-------------------------------------------------------------------------
void GetTextExtent( HDC hdc, char *pszString, int *piWidth, int *piHeight)
     // Computes the width and height of the specified string of text .
{
 SIZE size;
 if(GetTextExtentPoint32( hdc, pszString, strlen(pszString), &size) )
  { if(piWidth) *piWidth=size.cx;
    if(piHeight)*piHeight=size.cy;
  }
 else // function failed..
  { if(piWidth) *piWidth=1;
    if(piHeight)*piHeight=1;
  }
} // end GetTextExtent()

//-------------------------------------------------------------------------
void DrawLine(HDC hdc, int x1, int y1, int x2, int y2)
{
  MoveToEx(hdc,x1,y1,NULL);
  LineTo(hdc,x2,y2);   // unfortunately this does not include the end coord for odd reasons
}

//---------------------------------------------------------------------------
void LeftAlignedTextOut( HDC dc, int x, int y, char *pszText )
{
  if( dc!=NULL && pszText!=NULL)
   { SetTextAlign( dc, TA_LEFT|TA_TOP);
     TextOut( dc, x, y, pszText, strlen(pszText) );
   }
}

//-------------------------------------------------------------------------
BOOL WS_CheckMenuItem(  // an easier-to-use wrapper for CheckMenuItem()
                HMENU hmenu,   // [in] handle to menu
                UINT  uIDItem, // [in] menu item to modify
                BOOL  fCheck)  // [in] TRUE to check, FALSE to uncheck (boolean condition, not MF_CHECKED/UNCKECKED!)
{
  DWORD dw = CheckMenuItem( hmenu, uIDItem, fCheck ? MF_CHECKED : MF_UNCHECKED );
  // CheckMenuItem() returns the previous state of the menu item (either MF_CHECKED or MF_UNCHECKED).
  // If the menu item does not exist, the return value is 0xFFFFFFFF.
  return dw == MF_CHECKED;  // -> TRUE or FALSE, easier to use
} // end WS_CheckMenuItem()

//-------------------------------------------------------------------------
BOOL WS_EnableMenuItem( // an easier-to-use wrapper for EnableMenuItem()
                HMENU hmenu,   // [in] handle to menu
                UINT  uIDItem, // [in] menu item to modify
                BOOL fEnabled) // [in] FALSE->disbled+grayed (i.e. N/A), TRUE->ENABLED
{
  DWORD dw = EnableMenuItem( hmenu, uIDItem, fEnabled ? MF_ENABLED : MF_GRAYED );
  // CheckMenuItem() returns the previous state of the menu item (either MF_CHECKED or MF_UNCHECKED).
  // If the menu item does not exist, the return value is 0xFFFFFFFF.
  return dw == MF_ENABLED;  // -> TRUE or FALSE, easier to use
} // end WS_EnableMenuItem()

//-------------------------------------------------------------------------
BOOL WS_PrintMenuItem(
                HMENU hmenu,   // [in] handle to menu
                UINT  uIDItem, // [in] menu item to modify
                // format and argument list as for "sprintf" :
                char *pszFormat, ... )  // format string for wsprintf (no floats!)
{
 va_list parameter;              /* Parameter-Liste fr VA_... Macros */
 char szText[255];                  /* Puffer fr formatierten String */
 MENUITEMINFO mii;

 va_start( parameter, pszFormat );
 // doesn't support float: wvsprintf( szText, pszFormat, parameter );
 vsprintf( szText, pszFormat, parameter );
 va_end(parameter);

 ZeroMemory(&mii,sizeof(MENUITEMINFO));
 mii.cbSize= sizeof(MENUITEMINFO);
 mii.fMask = MIIM_TYPE; // MIIM_TYPE sets the fType and dwTypeData members
 mii.fType = MFT_STRING;
 mii.dwTypeData = (LPSTR)szText;
 return SetMenuItemInfo( hmenu, uIDItem, FALSE/*"by command, not by position"*/, &mii);
} // end WS_PrintMenuItem()

//-------------------------------------------------------------------------
CPROT BOOL WS_GetMenuText(
                HMENU hmenu,   // [in] handle to menu
                UINT  uIDItem, // [in] menu item to modify
                char *pszText, int iMaxLen ) // [out] string with a size limitation
  // Kind of inverse to WS_PrintMenuItem(). Retrieve the text in the specified menu item.
  // Used, for example, in WSQ_MainWin.c to retrieve the name of the clicked audio device.
{
 MENUITEMINFO mii;
 ZeroMemory(&mii,sizeof(MENUITEMINFO));
 mii.cbSize= sizeof(MENUITEMINFO);
 mii.fMask = MIIM_TYPE; // MIIM_TYPE sets the fType and dwTypeData members
 mii.fType = MFT_STRING;
 mii.dwTypeData = (LPSTR)pszText;
 mii.cch   = iMaxLen;
 // > To retrieve a menu item of type MFT_STRING, first find the size of the string
 // > by setting the dwTypeData member of MENUITEMINFO to NULL
 // > and then calling GetMenuItemInfo. The value of cch+1 is the size needed.
 // > Then allocate a buffer of this size, place the pointer to the buffer in dwTypeData,
 // > increment cch by one, and then call GetMenuItemInfo once again
 // > to fill the buffer with the string.
 // No, thanks. Keep it simple !  We assume the caller's buffer is long enough;
 // if not, the function will simply fail.
 pszText[0] = '\0';
 if( GetMenuItemInfo( hmenu, uIDItem, FALSE/*"by command, not by position"*/, &mii) )
  { if( /*mii.cch>=0 && */ (int)mii.cch<iMaxLen )
     { pszText[ mii.cch ] = '\0'; // make sure the result is properly terminated !
     }
    return TRUE;
  }
 else
  { return FALSE;
  }
} // end WS_GetMenuText()


//-------------------------------------------------------------------------
void InitLogFont(LOGFONT *pLogFont, char *pszFaceName, int iHeight)
{
 ZeroMemory(pLogFont, sizeof(LOGFONT));
 pLogFont->lfHeight = iHeight;
 strncpy(pLogFont->lfFaceName, pszFaceName, LF_FACESIZE);

 // Though the lousy documentation says about LOGFONT :
 // > A value of ZERO means 'use default values', ..
 // this is definitely not true in all cases. Setting the editor's font to
 // "Courier New" with everything ZERO (except lfFaceName and lfHeight)
 // caused windows to use a totally different font ---- not "Courier New" !
 pLogFont->lfWeight = 400;
 pLogFont->lfOutPrecision = '\x03';
 pLogFont->lfClipPrecision = '\x02';
 pLogFont->lfQuality = '\x01';
 pLogFont->lfPitchAndFamily = 1;
 pLogFont->lfQuality = '\x01';
 pLogFont->lfCharSet = DEFAULT_CHARSET;
}

//-------------------------------------------------------------------------
void SetFont(HWND hWnd, HFONT hFont)
  // if the hFont parameter is NULL,
  // the control uses the default system font to draw text.
{
  SendMessage(hWnd,WM_SETFONT,(WPARAM)hFont, (LPARAM)1 );
}

//-------------------------------------------------------------------------
CPROT void CharformatToLogfont( CHARFORMAT *pCf, LOGFONT *pLf )
{
   // The font "Height" parameter seems to have different units :
   //  (CHARFORMAT.yHeight=320)   =?=   (LOGFONT.lfHeight=16)   ?!?
   #define LFHEIGHT_TO_CFHEIGHT 20
   int ilfHeight = pCf->yHeight / LFHEIGHT_TO_CFHEIGHT;

   // A note on LOGFONT.lfHeight from WinProgRef :
   //  > For the MM_TEXT mapping mode, you can use the following formula
   //  > to specify a height for a font with a given point size:
   //  > lfHeight = -MulDiv(PointSize, GetDeviceCaps( hDC, LOGPIXELSY), 72);
   // Since we don't have a device-context here.. a short test:
   //  GetDeviceCaps( GetDC(Editor.hWnd) , LOGPIXELSY)  returned  "96" for my screen,
   //  so
   // lfHeight = -MulDiv(PointSize, GetDeviceCaps( GetDC(), LOGPIXELSY), 72)
   //          = -  (PointSize * 96 ) / 72 = Pointsize * 1.33 =  -(Pointsize+Pointsize/3);
   InitLogFont(pLf, pCf->szFaceName, -(ilfHeight+ilfHeight/3) );
   if(pCf->dwEffects & CFE_BOLD) pLf->lfWeight=FW_BOLD;
                        else     pLf->lfWeight=FW_NORMAL;
   // ex: pLf->lfItalic    =  ( pCf->dwEffects & CFE_ITALIC)    ? 1 : 0 ;
   //     "conversion may lose significant digits" .. ?!?!?
   if(pCf->dwEffects & CFE_ITALIC) pLf->lfItalic = 1;
                         else      pLf->lfItalic = 0;
   // ex: pLf->lfStrikeOut =  ( pCf->dwEffects & CFE_STRIKEOUT) ? 1 : 0 ;
   if(pCf->dwEffects & CFE_STRIKEOUT)  pLf->lfStrikeOut = 1;
                         else          pLf->lfStrikeOut = 0;
   // ex: pLf->lfUnderline =  ( pCf->dwEffects & CFE_UNDERLINE) ? 1 : 0 ;
   if(pCf->dwEffects & CFE_UNDERLINE)  pLf->lfUnderline = 1;
                         else          pLf->lfUnderline = 0;
} // end CharformatToLogfont()

//-------------------------------------------------------------------------
void LogfontToCharformat( LOGFONT *pLf, CHARFORMAT *pCf )
{
 int iHeight;
   pCf->cbSize=sizeof(CHARFORMAT);
   pCf->dwMask |= // what shall be modified - based on the infos from a LOGFONT
     CFM_BOLD | CFM_FACE | CFM_ITALIC | CFM_SIZE | CFM_UNDERLINE  | CFM_STRIKEOUT ;
           // | CFM_COLOR ; // NOT HERE !
   iHeight = pLf->lfHeight;
   if(iHeight<0)  // Negative value ?
    {             // >  The font mapper transforms this value into device units
                  // >  and matches its absolute value against
                  // >  the character height of the available fonts.
                  // Nothing understood ? see below..
      iHeight = -iHeight;              // found by trial-and-error
    }
   else           // Positive value ?
    {             // >  The font mapper transforms this value into device units
                  // >  and matches it against
                  // >  the cell height of the available fonts.
                  // What's the difference between CHARACTER- and CELL- height?
                  // Guess they call it "INTERNAL LEADING" (lead = metal)
                  // or whatever.
      iHeight = iHeight;
    }
   // lfHeight =  -  (PointSize * 96 ) / 72
   // =>  PointSize=(lfHeight*72)/96 = lfHeight - lfHeight/4 (!)
   iHeight -= iHeight/4;   // "font height" -> "point size" (?)
   pCf->yHeight = iHeight * LFHEIGHT_TO_CFHEIGHT; // which metrics used in "CharFormat" ?!
   pCf->dwEffects = 0
       // CFE_AUTOCOLOR
       | ((pLf->lfWeight>=FW_BOLD)? CFE_BOLD : 0)
       | ((pLf->lfItalic)   ? CFE_ITALIC     : 0)
       | ((pLf->lfStrikeOut)? CFE_STRIKEOUT  : 0)
       | ((pLf->lfUnderline)? CFE_UNDERLINE  : 0) ;
   strncpy( pCf->szFaceName,pLf->lfFaceName,LF_FACESIZE);
} // end LogfontToCharformat()

//-------------------------------------------------------------------------
void SetWindowTitle(HWND hWnd, char *pszCaption )
{
  SendMessage(hWnd,WM_SETTEXT,(WPARAM)0, (LPARAM)pszCaption );
}

//-------------------------------------------------------------------------
void InitParaFormatStruct( PARAFORMAT *pParaFormat )
{
  ZeroMemory(pParaFormat,sizeof(PARAFORMAT));
  pParaFormat->cbSize=sizeof(PARAFORMAT); // Size in bytes of this structure. Must be filled before passing to the rich edit control.
  pParaFormat->dwMask = 0;  // by default, NONE of the parameters shall be modified
} // end InitParaFormatStruct()


//-------------------------------------------------------------------------
BOOL IsRichEditControl( HWND hWndEditor )
  // Returns TRUE if the specified control is a *RICH* edit control,
  // otherwise FALSE (i.e. a normal 'poor' edit control, or whatever)
{
  char sz255ClassName[256];
  int  nChars = GetClassName( hWndEditor, sz255ClassName, 255 );
  if( nChars>=80 )  // at least "RICHEDIT"... ignore the version
   {
#   ifdef __BORLANDC__
     if( strnicmp( sz255ClassName, "richedit", 8) == 0 )
#   else  // PellesC didn't know strnicmp.. aaaaaargh...
     if( _strnicmp( sz255ClassName, "richedit", 8) == 0 )
#   endif
      { return TRUE;
      }
     // Under Windows XP with a normal multi-line edit control,
     //  got here with sz255ClassName = "Edit" .
   }
  return FALSE;
} // end IsRichEditControl()

// GetTextXXXXX + SetTextXXXXX routines: multi-line text edit controls.
//   Should work for normal EDIT controls as well as RichEdit controls.
//-------------------------------------------------------------------------
void GetTextSelection( HWND hWndEditor, DWORD *pdwStartPos, DWORD *pdwEndPos )
{ // Retrieves the start- and end-position of the SELECTED TEXT
  // in a multiline edit control.
  // More info in Win32 Programmer's reference:
  //   "About Edit Controls" + "Character and Line Operations" .
  CHARRANGE range;
  if( IsRichEditControl( hWndEditor ) )
   { SendMessage( hWndEditor, // handle of destination window
                  EM_EXGETSEL, // EM_EXGETSEL only works for RICH EDIT controls !
                  (WPARAM)0, (LPARAM)&range);
   }
  else  // if the editor is not a 'Rich' edit control, don't use EM_EXGETSEL on it..
   { range.cpMax = -1;  // .. but the ye olde EM_GETSEL further below
   }
  if(range.cpMax<0) // "range includes everything" .. holy shit, not very helpful !
   { SendMessage( hWndEditor, // handle of destination window
       EM_GETSEL, (WPARAM)&range.cpMin, (LPARAM)&range.cpMax);
   }
  if(pdwStartPos)
    *pdwStartPos = range.cpMin;
  if(pdwEndPos)
    *pdwEndPos = range.cpMax;
} // end GetTextSelection()

//-------------------------------------------------------------------------
void SetTextSelection( HWND hWndEditor, DWORD dwStartPos, DWORD dwEndPos )
{ // Select a range of characters in an edit control .
  SendMessage( hWndEditor, // handle of destination window
                EM_SETSEL, // message to send
       (WPARAM)dwStartPos, // starting position
       (LPARAM)dwEndPos ); // ending position
} // end SetTextSelection()

//-------------------------------------------------------------------------
void SetCaretToEndOfText( HWND hWndEditor )
{ // Sets the text caret (formerly known as cursor, but that's the 'mouse pointer' these days)
  // to the end of the editor text, with no character selected.
  // That way, the next character typed into the editor won't overwrite anything.
  DWORD dwTextLength = SendMessage(hWndEditor, WM_GETTEXTLENGTH, 0, 0);
  SendMessage( hWndEditor, // handle of destination window
                EM_SETSEL, // message to send
     (WPARAM)dwTextLength, // starting position
    (LPARAM)dwTextLength); // ending position  (-> LENGTH of the selection is zero)
} // end SetCaretToEndOfText()


//-------------------------------------------------------------------------
int GetSelectedText( HWND hWndEditor, char *pszDest, int iMaxLength )
  // Retrieves the currently selected text IN A RICH EDIT CONTROL(!) .
{
 int iSelTextSize;
 int iReturnedTextSize;

  // Because the EM_GETSELTEXT message doesn't care a damn
  // for the caller's buffer size, we must determine
  // the size of the selected text before actually 'getting' it :
  DWORD dwSelStartPos, dwSelEndPos;
  GetTextSelection( hWndEditor, &dwSelStartPos, &dwSelEndPos );
  iSelTextSize = dwSelEndPos - dwSelStartPos;
  if(iSelTextSize >= iMaxLength)
     return -1;  // shit, there WOULD HAVE been a big bang..

   // > The EM_GETSELTEXT message retrieves the currently selected text
   // > in a rich edit control.
  iReturnedTextSize = SendMessage(
               hWndEditor, // handle of destination window
            EM_GETSELTEXT, // message to send
                (WPARAM)0, // no information passed in this parameter ?!
        (LPARAM)pszDest ); // address of buffer for returned text
   // LPARAM points to a buffer that receives the selected text.
   // The calling application must ensure that the buffer is large
   // enough to hold the selected text.  (ain't this nice ... damn !)
   // Return Values
   // Returns the number of characters copied, not including
   // the terminating null character.
  return iReturnedTextSize;

} // end GetSelectedText()

//-------------------------------------------------------------------------
void ReplaceSelectedText( HWND hWndEditor, char *pszText, BOOL fCanUndo )
  // Replaces the currently selected text in a multiline edit control
  //      If there is no current selection, the replacement text is inserted
  //      at the current location of the caret.
  // fCanUndo specifies whether the replacement operation can be undone.
  //      If this is TRUE, the operation can be undone.
  //      If this is FALSE , the operation cannot be undone.
{ // Implementation :
  // Send an EM_REPLACESEL message to replace the current selection
  // in an edit control with the specified text.
  // More info in Win32 Programmer's reference.
  SendMessage( hWndEditor, // handle of destination window
            EM_REPLACESEL, // message to send
         (WPARAM)fCanUndo, // flag that specifies whether replacement can be undone
         (LPARAM)pszText ); // pointer to replacement text string
  // This message does not return a value !
} // end ReplaceSelectedText()

//-------------------------------------------------------------------------
DWORD GetTextLineCount( HWND hWndEditor )
{  // Converts one of the results from GetTextSelection() into the LINE INDEX.
  // An application sends an EM_GETLINECOUNT message to retrieve the number
  // of lines in a multiline edit control.
  // More info in Win32 Programmer's reference: "Character and Line Operations"

  return SendMessage( hWndEditor, // handle of destination window
      EM_GETLINECOUNT, // message to send
      (WPARAM)0,    // not used; must be zero
      (LPARAM)0 );  // not used; must be zero
  // The return value is an integer specifying the number of lines
  // in the multiline edit control. If no text is in the edit control,
  // the return value is 1.
} // end GetTextLineCount()

//-------------------------------------------------------------------------
DWORD GetTextLineIndexFromCharIndex( HWND hWndEditor, DWORD dwCharacterIndex )
{  // Converts one of the results from GetTextSelection() into the LINE INDEX.
  // More info in Win32 Programmer's reference: "Character and Line Operations"
  return SendMessage( hWndEditor, // handle of destination window
          EM_LINEFROMCHAR, // message to send
      (WPARAM)dwCharacterIndex, // character index
      (LPARAM)0        );  // not used; must be zero
  // An application sends an EM_LINEFROMCHAR message to retrieve
  // the index of the line that contains the specified character
  // index in a multiline edit control.
  // A character index is the number of characters from the beginning
  // of the edit control.
  // The return value is the zero-based line number of the line
  // containing the character index specified by <dwCharacterIndex> .
} // end GetTextLineIndexFromCharIndex()

//-------------------------------------------------------------------------
DWORD GetTextCharIndexFromLineIndex( HWND hWndEditor, int iLineIndex )
{ // Retrieves the character index of a line in a multiline edit control.
  // The character index is the number of characters from the beginning
  //     of the edit control to the specified line.
  // More info in Win32 Programmer's reference: "Character and Line Operations"
  return SendMessage( hWndEditor, // handle of destination window
          EM_LINEINDEX,    // message to send
      (WPARAM)iLineIndex,  // line number
      (LPARAM)0        );  // not used; must be zero
  // An application sends an EM_LINEINDEX message to retrieve
  // the character index of a line in a multiline edit control.
  // The character index is the number of characters from the beginning
  // of the edit control to the specified line.
  // wParam  specifies the zero-based line number. A value of -1 specifies
  //         the current line number (the line that contains the caret).
  // The return value is the character index of the line specified
  //     in the line parameter, or it is -1 if the specified line number
  //     is greater than the number of lines in the edit control.
} // end GetTextCharIndexFromLineIndex()

//-------------------------------------------------------------------------
void GetTextCaretLineAndColumn( HWND hWndEditor, DWORD *pdwLine, DWORD *pdwColumn)
     // both "result pointers" may be NULL if not needed.
     // Note:  'Caret' is the blinking text cursor in an edit window,
     //  while 'Cursor' is what the MS gurus call the mouse pointer !
{
  DWORD dwSelStart,dwSelEnd;
  DWORD dwLine,dwCharIndexOfLine;

  GetTextSelection( hWndEditor, &dwSelStart, &dwSelEnd );
  dwLine = GetTextLineIndexFromCharIndex( hWndEditor, dwSelStart );
  dwCharIndexOfLine  = GetTextCharIndexFromLineIndex( hWndEditor, dwLine );
  if(pdwLine)   *pdwLine  = dwLine;
  if(pdwColumn) *pdwColumn= dwSelStart - dwCharIndexOfLine;
} // end GetTextCaretLineAndColumn()

//-------------------------------------------------------------------------
void SetTextCaretLineAndColumn( HWND hWndEditor, DWORD dwLine, DWORD dwColumn)
{
 DWORD dwCharIndex;
  dwCharIndex = GetTextCharIndexFromLineIndex( hWndEditor, dwLine );
  dwCharIndex += dwColumn;
  SetTextSelection( hWndEditor, dwCharIndex, dwCharIndex );
} // end SetTextCaretLineAndColumn()

//---------------------------------------------------------------------
//  Modify text in a multiline edit control..
//---------------------------------------------------------------------

int GetTextLineAsCString( HWND hWndEditor, int iLineIndex, char *pszDest, int iMaxLength)
   // the result is the returned string length (whithout the trailing zero) .
{
 DWORD dwCharsCopied;

  if(iMaxLength<2)  // need at least two bytes because of windoze's funny buffer length indicator!
     return -1;

  // More info in Win32 Programmer's reference: "Character and Line Operations"
  *((WORD*)pszDest)=(WORD)iMaxLength;      // max length in 1st 2 bytes of buffer(!!)
  dwCharsCopied = SendMessage( hWndEditor, // handle of destination window
            EM_GETLINE,    // message to send
      (WPARAM)iLineIndex,  // line number to retrieve
      (LPARAM)pszDest  );  // address of buffer for line
  // An application sends an EM_GETLINE message to copy a line of text
  // from an edit control and place it in a specified buffer.
  // wParam specifies the zero-based index of the line to retrieve
  //        from a multiline edit control.
  //        A value of zero specifies the topmost line.
  // lParam points to the buffer that receives a copy of the line.
  //        The first word of the buffer specifies the maximum number
  //        of characters that can be copied to the buffer.
  // The return value is the number of characters copied.
  // The return value is zero if the line number specified
  // by the line parameter is greater than the number of lines
  // in the edit control.
  // The copied line does not contain a terminating null character.
  //  What the help system does not tell you:
  //       the \r\n characters are included, what we don't want !
  if((int)dwCharsCopied<iMaxLength)
   {
    if(dwCharsCopied >= 2 && pszDest[dwCharsCopied-2]=='\r')
     { dwCharsCopied-=2; pszDest[dwCharsCopied] = '\0';  }
    else
     { pszDest[dwCharsCopied+1] = '\0'; }
   }
  return dwCharsCopied;
} // end GetTextLineAsCString()

//-------------------------------------------------------------------------
BOOLEAN ReplaceTextLine( HWND hWndEditor, int iLineIndex, char *pszSource, BOOL fCanUndo)
    // Side effect: the SELECTION will be modified,
    //              so -if necessary- save and restore the caret position !
{
 int iCharIndex;
 int iOldLineLength;

  // More info in Win32 Programmer's reference: "Character and Line Operations"
  // There is NO 'EM_SETLINE'  message for this purpose. Instead, we must
  // modify the "current selection" to do the job, using a lot of messages :(

  // Send an EM_LINEINDEX message to retrieve the character index of the line
  iCharIndex=SendMessage(hWndEditor, EM_LINEINDEX, (WPARAM)iLineIndex, (LPARAM) 0);
  // The return value is the character index of the line specified in the line
  // parameter, or it is -1 if the specified line number is greater
  // than the number of lines in the edit control.
  if(iCharIndex<0) return FALSE; // line does not exist

  // Send an EM_LINELENGTH message to retrieve the length of the line :
  iOldLineLength=SendMessage(hWndEditor, EM_LINELENGTH, (WPARAM)iCharIndex, (LPARAM) 0);
  // (interestingly, this does not include CR+NL though EM_GETLINE would return them !)

  // Now SELECT -precisely- the LINE without CR so it can be replaced..
  SetTextSelection( hWndEditor, iCharIndex/*start*/, iCharIndex+iOldLineLength/*end*/ );

  // And finally, REPLACE the selected line..
  ReplaceSelectedText( hWndEditor, pszSource, fCanUndo );

  return TRUE; // hmmm

} // end ReplaceTextLine()

//-------------------------------------------------------------------------
BOOLEAN GetModifiedFlag( HWND hWndEditor )
{
  // An application sends an EM_GETMODIFY message to determine
  // whether the content of an edit control has been modified.
  // If the content of edit control has been modified,
  // the return value is TRUE; otherwise, it is FALSE.

  //ex: return SendMessage(hWndEditor, EM_GETMODIFY, (WPARAM)0,(LPARAM)0 );
  // ah, shut up with this "conversion may lose significant digits" stuff please..
  return (BOOLEAN)SendMessage(hWndEditor, EM_GETMODIFY, (WPARAM)0,(LPARAM)0 );
} // end GetModifiedFlag()

void SetModifiedFlag( HWND hWndEditor, BOOL fModified )
{
  SendMessage(hWndEditor, EM_SETMODIFY, (WPARAM)fModified, (LPARAM)0 );
} // end SetModifiedFlag()



//**************************************************************************
//  "Editor" routines...
//**************************************************************************

//-------------------------------------------------------------------------
void Editor_InitStruct( T_WinEditor *pEd )    // only call this ONCE(!)
{
  ZeroMemory( pEd, sizeof(T_WinEditor) );
  // all HANDLES are NULL now, etc.
  InitLogFont(&pEd->LogFont, "Courier New", 16);
  pEd->wpOldEditSubclassProc = NULL;  // there is definitely NO subclass proc yet..
  pEd->wpNewEditSubclassProc = NULL;
  InitParaFormatStruct( &pEd->ParaFormat );
  pEd->dwCreationStyle =  WS_CHILD | WS_VISIBLE | WS_HSCROLL | WS_VSCROLL | WS_BORDER
   //    | ES_AUTOHSCROLL  // scroll text to the right BUT TURNS WORD WRAP OFF
         | ES_AUTOVSCROLL | ES_LEFT  | ES_MULTILINE
         | ES_NOHIDESEL    // the selected text is inverted, even if the control does not have the focus
    //   | ES_WANTRETURN   // carriage return shall be inserted when the user presses the ENTER key
         ;
  strcpy(pEd->sz16ClassName,"RichEdit");
}

//-------------------------------------------------------------------------
void Editor_DestroyWindow( T_WinEditor *pEd )
{
  // Remove the subclass from the edit control, and free all(?) resources
  if(pEd->hWnd)
   { if(pEd->wpOldEditSubclassProc)
      { SetWindowLong(pEd->hWnd, GWL_WNDPROC,  (LONG) pEd->wpOldEditSubclassProc);
        pEd->wpOldEditSubclassProc = NULL;
      }

     if(pEd->hFont)
      { SetFont(pEd->hWnd, NULL);
        DeleteObject(pEd->hFont);
        pEd->hFont = NULL;
      }

     DestroyWindow(pEd->hWnd);
     pEd->hWnd = NULL;

   } // end if(pEd->hWnd)
} // end Editor_DestroyWindow()

//-------------------------------------------------------------------------
void Editor_Create( T_WinEditor *pEd, HWND hWndParent,
             int left, int top, int width, int height )
{
  static int siLoadedRichEd32 = 0;
  if( !siLoadedRichEd32 )
   { LoadLibrary("RICHED32.DLL");  // See Win32ProgRef, "About Rich Edit Controls"
     siLoadedRichEd32 = 1;
   }

  if(strcmp(pEd->sz16ClassName,"RichEdit") != 0)
   {
    Editor_InitStruct( pEd );    // forgotten to call this, SHAME ON YOU ! ;-)
   }

  if(pEd->hWnd != NULL)          // is there an old editor window ?
   {
     Editor_DestroyWindow(pEd);
   }

  pEd->hWnd  = CreateWindowEx(   // create a multiline edit control...
        WS_EX_LEFT,  pEd->sz16ClassName, // dwExStyle, lpClassName
        "",         // lpWindowName would be the default text IN(!) the editor
        pEd->dwCreationStyle, // dwStyle
        left, top, width, height,
        hWndParent, 0,        // hWndParent,hMenu
        (HINSTANCE)GetWindowLong(hWndParent,GWL_HINSTANCE), // hInstance
        NULL );  // lpParam

   pEd->hFont = CreateFontIndirect(&pEd->LogFont);
   if(pEd->hFont)
      SetFont(pEd->hWnd, pEd->hFont);

  // Increase the upper limit of the amount of text in the Rich Edit control.
  //  (the default value seems ridiculous, only 32 kByte !)  We want 2 MByte.
  SendMessage( pEd->hWnd, EM_EXLIMITTEXT, (WPARAM)0, (LPARAM)(DWORD)(2048L*1024) );
} // end Editor_Create()

//-------------------------------------------------------------------------
void Editor_RecreateWnd( T_WinEditor *pEd )
  // This is required after certain modifications, for example modifying the
  // border style, the word wrap mode  etc etc etc etc (who knows what else)
{
  HWND hWndParent;
  RECT MyRect;
  POINT pt;
  DWORD dwOldSelStart,dwOldSelEnd;
  WND_PROC wpPrevNewEditSubclassProc = NULL;
  BOOLEAN fOldModifiedFlag;

  if(pEd->hWnd != NULL)          // is there an old editor window ?
   {
     // Before destroying the "old" window, save some of its properties..
     fOldModifiedFlag = GetModifiedFlag(pEd->hWnd);
     GetTextSelection( pEd->hWnd, &dwOldSelStart, &dwOldSelEnd );
     hWndParent = (HWND)GetWindowLong( pEd->hWnd, GWL_HWNDPARENT );
     GetWindowRect(pEd->hWnd, &MyRect);  // -> absolute SCREEN coords
     pt.x=MyRect.left; pt.y=MyRect.top;
     ScreenToClient(hWndParent,&pt);     // -> CLIENT coords required below

     // Tricky not to lose all the text in the window ! Abuse the clipboard..
     SendMessage(pEd->hWnd,EM_SETSEL,0,-1);
     SendMessage(pEd->hWnd, WM_COPY, 0, 0);

     if(pEd->wpOldEditSubclassProc != NULL)
      { SetWindowLong(pEd->hWnd, GWL_WNDPROC, (LONG)pEd->wpOldEditSubclassProc);
        pEd->wpOldEditSubclassProc = NULL;  // forget 'old' stuff, but..
        // DON'T FORGET pEd->wpNewEditSubclassProc (it will be re-installed below)
        wpPrevNewEditSubclassProc = pEd->wpNewEditSubclassProc;
      }

     // Now "destroy" the window... only to create a new one very soon
     Editor_DestroyWindow(pEd);

     // Re-Create the window to make the new style parameters(etc) effective
     Editor_Create( pEd, hWndParent,
           pt.x, pt.y, MyRect.right-MyRect.left, MyRect.bottom-MyRect.top );

     if(pEd->hWnd != NULL)          // is there a new editor window ?
      {
        // if there has been a special "subclass" procedure for this editor,
        // re-install it (because it seems to get lost when destroying the window)
        if(wpPrevNewEditSubclassProc != NULL)
         { Editor_SetSubclassProc( pEd, wpPrevNewEditSubclassProc );
         }

        // paste the complete text here, using the new formatting options..
        SendMessage( pEd->hWnd,WM_PASTE, 0, 0);
        SetTextSelection( pEd->hWnd, dwOldSelStart, dwOldSelEnd );
        SetModifiedFlag(pEd->hWnd, fOldModifiedFlag );        
      } // end if(pEd->hWnd != NULL) (for NEW editor window)
   } // end if(pEd->hWnd != NULL)

} // end Editor_RecreateWnd()

//-------------------------------------------------------------------------
void Editor_SetSubclassProc( T_WinEditor *pEd, WND_PROC EditSubclassProc )
{
 WND_PROC prevSubclassProc;

   // Subclass the edit control, so we can handle its keyboard events ourselves
   // > To subclass an instance of a window, call the SetWindowLong function
   // > and specify the handle of the window to subclass the GWL_WNDPROC
   // > flag and a pointer to the subclass procedure.
   // > SetWindowLong returns a pointer to the original window procedure;
   // >  use this pointer to pass messages to the original procedure.
   // > The subclass window procedure must use the CallWindowProc function
   // > to call the original window procedure.
   // Here, SetWindowLong is called in the WM_CREATE handler directly after
   // creation of the edit control.
   prevSubclassProc =(WND_PROC)SetWindowLong(pEd->hWnd,
                   GWL_WNDPROC, (LONG) EditSubclassProc);
   if(prevSubclassProc!=NULL) // only if SetWindowLong() successful..
    {  pEd->wpNewEditSubclassProc = EditSubclassProc;
       if(pEd->wpOldEditSubclassProc==NULL)
          pEd->wpOldEditSubclassProc = prevSubclassProc;
    }
} // end Editor_SetSubclassProc()

//-------------------------------------------------------------------------
DWORD CALLBACK EditStreamInCallback(DWORD dwCookie, LPBYTE pbBuff, LONG cb, LONG FAR *pcb)
// Callback from a RichEdit control for loading streams (here: files).
//     See Win32 Programmer's Help on RichEdit control, EM_STREAMIN message .
//      (yeah, keep on streamin' baby.. but beware about the 64 kByte limit)
//     The control calls the callback function repeatedly,
//     transferring a portion of the data with each call.
// Parameters:
//  dwCookie = value of the dwCookie member of the EDITSTREAM structure. Yummy.
//  pbBuff   = Pointer to the buffer to read from or write to.
//  cb       = Count of bytes to read or write.
//  pcb      = Pointer to a variable that receives the number of bytes actually read or written.
// Return Value:
//  The return value is zero to continue to the stream operation, or nonzero to abort it.
{
 DWORD dwNumberOfBytesRead;

  if(!ReadFile( (HANDLE)dwCookie, // handle of file to read
             pbBuff,          // address of buffer that receives data
             cb,              // number of bytes to read
            &dwNumberOfBytesRead, // address of number of bytes read
             NULL) )          // lpOverlapped
   { *pcb = 0;
     return 0xFFFFFFFF;  // abort
   }
  else // success..
   { *pcb = dwNumberOfBytesRead;
     return 0/*zero to continue*/;
   }
} // end EditStreamInCallback()

//-------------------------------------------------------------------------
BOOL Editor_LoadFromFile( T_WinEditor *pEd, char *pszFileName )
   // Loads the editor control with a text file (or RTF) from disk.
   //   Note about loading ancient text files under Win98 (yet another bug):
   //      The RichEdit control -under Windows 98- will not load files properly
   //      if the lines are terminated with CR-characters only (0x0D),
   //      instead of CR+NL (0x0D 0x0A) which seems to be common practise
   //      under windows and DOS, but not on other platforms !
   //   To fix this, load such a file into WORDPAD, and save it again.
   //   Under Windows XP, this trick is not necessary.
{
 HANDLE hFile;
 EDITSTREAM eds;
 UINT uFormat;
 char *pszExtension = strrchr(pszFileName,'.');

  uFormat = SF_TEXT;
  if( pszExtension )
   { if(stricmp(pszExtension,".rtf")==0)
        uFormat = SF_RTF;
     // 2014-02: PellesC didn't know stricmp(). Baaah.
     // Some geek decided "it's not POSIX nor ANSI compatible so we don't support it".
     //
   }

  hFile = CreateFile( pszFileName, // pointer to name of the file
                     GENERIC_READ, // access (read-write) mode
                                0, // share mode
                             NULL, // pointer to security attributes
                    OPEN_EXISTING, // how to create
            FILE_ATTRIBUTE_NORMAL, // file attributes
                            NULL); // handle to file with attributes to copy
  if( hFile == INVALID_HANDLE_VALUE )
      return FALSE;

  // Fill an EDITSTREAM structure which must be passed to the EM_STREAMIN message..
  eds.dwCookie = (DWORD)hFile;
  eds.dwError  = 0;     // no error so far
  // Select the function that the control calls to transfer data :
  eds.pfnCallback = EditStreamInCallback;  // callback-fctn to 'stream' the file in

  // Clear the previous text because otherwise some "format info" remains.
  //  Added after a bug report from SM6LKM 2003-09-25 .
  SendMessage(pEd->hWnd,WM_SETTEXT,(WPARAM)0,(LPARAM)"");
  SetFont(pEd->hWnd,pEd->hFont);  // totally unknown why this is required under Win98

  // Send an EM_STREAMIN message to the control to load it from our stream..
  SendMessage(pEd->hWnd,EM_STREAMIN,(WPARAM)uFormat,(LPARAM)&eds);

  // Close the input file handle and return
  return CloseHandle(hFile);
} // end Editor_LoadFromFile()

//-------------------------------------------------------------------------
DWORD CALLBACK EditStreamOutCallback(DWORD dwCookie, LPBYTE pbBuff, LONG cb, LONG FAR *pcb)
{
 DWORD dwNumberOfBytesWritten;

  if(!WriteFile( (HANDLE)dwCookie,   // handle of file to write
             pbBuff,                 // pointer to data to write to file
             cb,                     // number of bytes to write
            &dwNumberOfBytesWritten, // pointer to number of bytes written
             NULL) )          // lpOverlapped (here: no overlapped I/O)
   { *pcb = 0;
     return 0xFFFFFFFF;  // abort
   }
  else // success..
   { *pcb = dwNumberOfBytesWritten;
     return 0/*zero to continue*/;
   }
} // end EditStreamOutCallback()

//-------------------------------------------------------------------------
BOOL Editor_SaveAsFile( T_WinEditor *pEd, char *pszFileName )
{
 HANDLE hFile;
 EDITSTREAM eds;
 UINT uFormat;
 char *pszExtension = strrchr(pszFileName,'.');

  uFormat = SF_TEXT;
  if( pszExtension )
   { if(stricmp(pszExtension,".rtf")==0)
        uFormat = SF_RTF;
   }

  hFile = CreateFile( pszFileName, // pointer to name of the file
                    GENERIC_WRITE, // access (read-write) mode
                                0, // share mode
                             NULL, // pointer to security attributes
                    CREATE_ALWAYS, // how to create
            FILE_ATTRIBUTE_NORMAL, // file attributes
                            NULL); // handle to file with attributes to copy
  if( hFile == INVALID_HANDLE_VALUE )
      return FALSE;

  // Fill an EDITSTREAM structure which must be passed to the EM_STREAMIN message..
  eds.dwCookie = (DWORD)hFile;
  eds.dwError  = 0;     // no error so far
  eds.pfnCallback = EditStreamOutCallback; // function that the control calls to transfer data

  // Send an EM_STREAMOUT message to the control to load it from our stream..
  SendMessage(pEd->hWnd,EM_STREAMOUT,(WPARAM)uFormat,(LPARAM)&eds);

  // Close the input file handle and return
  return CloseHandle(hFile);
} // end Editor_SaveAsFile()

//-------------------------------------------------------------------------
BOOL Editor_IsSomethingSelected( T_WinEditor *pEd )
{
  DWORD dwSelStart, dwSelEnd;
  GetTextSelection( pEd->hWnd, &dwSelStart, &dwSelEnd );
  return dwSelEnd > dwSelStart;
} // Editor_IsSomethingSelected()

//-------------------------------------------------------------------------
void Editor_SetWordWrap( T_WinEditor *pEd, BOOL fEnableWordWrap )
{
  BOOL fMustRecreate;

 // no-no: SendMessage(hwndEdit, EM_FMTLINES, (WPARAM)fWordBreakOn, (LPARAM) 0);
 // ES_AUTOHSCROLL automatically scrolls text to the right AND TURNS WORD WRAP OFF (!)
 if(fEnableWordWrap)
    { fMustRecreate = (pEd->dwCreationStyle & ES_AUTOHSCROLL) != 0;
      pEd->dwCreationStyle &= ~ES_AUTOHSCROLL;
    }
  else
    { fMustRecreate = (pEd->dwCreationStyle & ES_AUTOHSCROLL) == 0;
      pEd->dwCreationStyle |=  ES_AUTOHSCROLL;
    }
 // SetWindowLong(pEd->hWnd, GWL_STYLE, dwStyle);
 // What the lousy ProgRef won't tell you: Setting the window style with
 //    "SetWindowLong" has no effect on the window's appearance.
 //    The VCL calls an obscure function named 'RecreateWnd' in this case
 //    which destroys the old and creates a new control (!)
 if(fMustRecreate)
    Editor_RecreateWnd( pEd );
} // end Editor_SetWordWrap()

//-------------------------------------------------------------------------
BOOL Editor_GetWordWrap( T_WinEditor *pEd )
{
 return  ( pEd->dwCreationStyle & ES_AUTOHSCROLL) == 0;
}

//-------------------------------------------------------------------------
void Editor_GetParaFormat( T_WinEditor *pEd )
{
  InitParaFormatStruct( &pEd->ParaFormat );
  // The EM_SETPARAFORMAT message sets the paragraph formatting
  // for the current selection in a rich edit control.
  SendMessage(pEd->hWnd, EM_GETPARAFORMAT,
              (WPARAM)0,  // not used; must be zero
              (LPARAM)&pEd->ParaFormat );
} // end GetParaFormat()

//-------------------------------------------------------------------------
void Editor_SetParaFormat( T_WinEditor *pEd )
{
  // The EM_SETPARAFORMAT message sets the paragraph formatting
  // for the current selection in a rich edit control.
  SendMessage(pEd->hWnd, EM_SETPARAFORMAT,
              (WPARAM)0,  // not used; must be zero
              (LPARAM)&pEd->ParaFormat );
} // end SetParaFormat()

//-------------------------------------------------------------------------
void Editor_GetCharFormat( T_WinEditor *pEd, BOOL fFromSelection )
{
  ZeroMemory(&pEd->CharFormat,sizeof(CHARFORMAT));
  pEd->CharFormat.cbSize=sizeof(CHARFORMAT);

  // The EM_GETPARAFORMAT message retrieves the character formatting
  // for the current selection in a rich edit control.
  SendMessage(pEd->hWnd, EM_GETCHARFORMAT,
              (WPARAM)fFromSelection, // TRUE=from selection, FALSE=default format
              (LPARAM)&pEd->CharFormat );
} // end GetCharFormat()

//-------------------------------------------------------------------------
void Editor_SetCharFormat( T_WinEditor *pEd, BOOL fForSelection )
{
  SendMessage(pEd->hWnd, EM_SETCHARFORMAT,
              (WPARAM)( fForSelection?SCF_SELECTION:0) ,
              (LPARAM)&pEd->CharFormat );
} // end SetCharFormat()

//-------------------------------------------------------------------------
void Editor_OpenFindReplaceWindow( T_WinEditor *pEd, HWND hWndOwner, BOOL fReplaceToo )
{

 if(pEd->hWndFindReplaceDlg == NULL)  // only if the Find/Replace dialog is NOT open..
  {
   // Initialize the editor's FINDREPLACE structure.
   //  The FINDREPLACE structure contains information that the
   //  FindText and ReplaceText functions use to initialize
   //  the Find and Replace common dialog boxes.
   //  The FINDMSGSTRING registered message uses this structure
   //  to pass the user's search or replacement input to the owner window
   //  of a Find or Replace common dialog box.
   ZeroMemory(&pEd->FindReplace, sizeof(FINDREPLACE));
   pEd->FindReplace.lStructSize = sizeof(FINDREPLACE);
   pEd->FindReplace.hwndOwner   = hWndOwner;
   pEd->FindReplace.hInstance   = NULL;    // no template used !
   pEd->FindReplace.Flags = FR_HIDEUPDOWN; // RichEdit can only search "forward"
   pEd->FindReplace.lpstrFindWhat    = pEd->sz255FindWhat;
   pEd->FindReplace.lpstrReplaceWith = pEd->sz255ReplaceWith;
   pEd->FindReplace.wFindWhatLen     = 255;
   pEd->FindReplace.wReplaceWithLen  = 255;
   pEd->FindReplace.lCustData = 0;
   pEd->FindReplace.lpfnHook  = NULL;
   pEd->FindReplace.lpTemplateName = NULL;

   // get a message identifier for FINDMSGSTRING..
   if(pEd->uFindReplaceMsg==0)
      pEd->uFindReplaceMsg = RegisterWindowMessage(FINDMSGSTRING);

   if(fReplaceToo)
           pEd->hWndFindReplaceDlg = ReplaceText( &pEd->FindReplace );
     else  pEd->hWndFindReplaceDlg = FindText( &pEd->FindReplace );
  } // end if(pEd->hWndFindReplaceDlg == NULL)

} // end Editor_OpenFindReplaceWindow()

//-------------------------------------------------------------------------
void Editor_HandleFindReplaceDlgMsg( T_WinEditor *pEd, FINDREPLACE *pfr )
{  // Handles FINDMSGSTRING messages, sent by the find/replace dialog window

 UINT uFindFlags;
 FINDTEXT ft;
 long lFoundAt;
 DWORD dwSelStart,dwSelEnd;
 BOOL fReplaced = FALSE;
 BOOL fFound = FALSE;
 int  iAnswer;

  // If the FR_DIALOGTERM flag is set,
  // invalidate the handle identifying the dialog box.
  if (pfr->Flags & FR_DIALOGTERM)
   {
     pEd->hWndFindReplaceDlg = NULL;
     return;
   }

  GetTextSelection( pEd->hWnd, &dwSelStart, &dwSelEnd );   
  uFindFlags = 0;
  if(pfr->Flags & FR_MATCHCASE) uFindFlags |= FT_MATCHCASE;
  if(pfr->Flags & FR_WHOLEWORD) uFindFlags |= FT_WHOLEWORD;


  if (pfr->Flags & FR_REPLACE )
   { // FR_REPLACE set in a FINDMSGSTRING message, indicates that the user
     // clicked the Replace button in a Replace dialog box.
     // The lpstrFindWhat member specifies the string to be replaced
     // and the lpstrReplaceWith member specifies the replacement string.
     if( (dwSelEnd-dwSelStart) == strlen(pfr->lpstrFindWhat) )
      { // only if something selected :
       ReplaceSelectedText( pEd->hWnd, pfr->lpstrReplaceWith, TRUE/*fCanUndo*/ );
       // here: after replacing, automatically search for the next occurrence !
       fReplaced = TRUE;
     }
   } // end FR_REPLACE

  // If the FR_FINDNEXT flag is set,
  // call the application-defined search routine
  // to search for the requested string.
  if( (pfr->Flags & FR_FINDNEXT) || (fReplaced) )
   {
    ft.chrg.cpMin = dwSelStart+1;
    ft.chrg.cpMax = - 1;  // 0..-1 means 'full search range'
    ft.lpstrText  = pfr->lpstrFindWhat;
    lFoundAt = SendMessage(pEd->hWnd,EM_FINDTEXT,(WPARAM)uFindFlags,(LPARAM)&ft);
    // EM_FINDTEXT returns the zero-based character position of the next match,
    // or  - 1 if there are no more matches.
    if(lFoundAt >= 0)
     { // found the text : set the 'current selection' to it.
       SetTextSelection( pEd->hWnd, lFoundAt, lFoundAt+strlen(pfr->lpstrFindWhat) );
       fFound = TRUE;
     }
    else // not found: unselect text
     {
       SetTextSelection( pEd->hWnd, dwSelStart, dwSelStart );
       MessageBox( pEd->hWnd, "Text not found","CalcEd",MB_OK);
     }
   } // end FR_FINDNEXT

  if (pfr->Flags & FR_REPLACEALL )
   {
    do
     { // possibly more than once for FR_REPLACEALL
      fReplaced = FALSE;
      fFound = FALSE;
      GetTextSelection( pEd->hWnd, &dwSelStart, &dwSelEnd );
      ft.chrg.cpMin = dwSelStart+1;
      ft.chrg.cpMax = - 1;  // 0..-1 means 'full search range'
      ft.lpstrText  = pfr->lpstrFindWhat;
      lFoundAt = SendMessage(pEd->hWnd,EM_FINDTEXT,(WPARAM)uFindFlags,(LPARAM)&ft);
      if(lFoundAt>=0)
       { // found the text : set the 'current selection' to it, and ask what to do
        SetTextSelection( pEd->hWnd, lFoundAt, lFoundAt+strlen(pfr->lpstrFindWhat) );
        fFound = TRUE;
        iAnswer = MessageBox( pEd->hWnd, "Replace this occurrence ?","CalcEd",
                            MB_YESNOCANCEL | MB_DEFBUTTON2 );
        switch(iAnswer)
         {
           case IDYES:
                ReplaceSelectedText( pEd->hWnd, pfr->lpstrReplaceWith, TRUE);
                fReplaced = TRUE;
                break;
           case IDNO:
                break;
           case IDCANCEL:
           default:
                return;
         } // end switch(iAnswer)
       } // end if(lFoundAt>=0)
     } while( fFound && (pfr->Flags & FR_REPLACEALL)!=0 );
   } // end FR_REPLACEALL

} // end Editor_HandleFindReplaceDlgMsg()

//**************************************************************************
// General routines to play with Status Windows ...
//**************************************************************************

//-------------------------------------------------------------------------
HWND DoCreateStatusWindow(HWND hwndParent, int nStatusID,
    HINSTANCE hinst, int nParts, int *piPartWidths)
 // Creates a status window and divides it into the specified number of parts.
 // Returns the handle to the status window.
 // hwndParent - parent window for the status window
 // nStatusID - child window identifier
 // hinst - handle to the application instance
 // nParts - number of parts into which to divide the status window
{
    HWND hwndStatus;

    RECT rcClient;
    HLOCAL hloc;
    LPINT lpParts;
    int i,x; // ,nWidth;

    // Ensure that the common control DLL is loaded.
    InitCommonControls();

    // Create the status window.
    hwndStatus = CreateWindowEx(
        0,                       // no extended styles
        STATUSCLASSNAME,         // name of status window class
        (LPCTSTR) NULL,          // no text when first created
        SBARS_SIZEGRIP |         // includes a sizing grip
        WS_VISIBLE |             // make the status window visible immediately
        WS_CHILD,                // creates a child window
        0, 0, 0, 0,              // ignores size and position
        hwndParent,              // handle to parent window
        (HMENU) nStatusID,       // child window identifier
        hinst,                   // handle to application instance
        NULL);                   // no window creation data

    // Get the coordinates of the parent window's client area.
    GetClientRect(hwndParent, &rcClient);

    // Allocate an array for holding the right edge coordinates.
    hloc = LocalAlloc(LHND, sizeof(int) * nParts);
    lpParts = (LPINT)LocalLock(hloc);

    // Calculate the right edge coordinate for each part, and
    // copy the coordinates to the array.
    x = 0;
    for (i = 0; i < nParts; i++)
     {  x += piPartWidths[i];
        if( (i+1) < nParts) lpParts[i] = x;
        else                lpParts[i] = rcClient.right;
     }

    // Tell the status window to create the window parts.
    SendMessage(hwndStatus, SB_SETPARTS, (WPARAM) nParts, (LPARAM) lpParts);

    // Free the array, and return.
    LocalUnlock(hloc);
    LocalFree(hloc);
    return hwndStatus;
}

//-------------------------------------------------------------------------
void SetStatusWindowText(HWND hWndStatusWindow, int iPart, int iType, char *szText)
{
  if(hWndStatusWindow)
   SendMessage(hWndStatusWindow,  SB_SETTEXT,
                (WPARAM) iPart | iType,
                (LPARAM) szText );
  // Parameters for SB_SETTEXT message:
  // iPart
  //   Zero-based index of the part to set. If this value is 255,
  //   the status window is assumed to be a simple window having only one part.
  // uType
  //   Type of drawing operation. This parameter can be one of the following values:
  //    0 = text is drawn with a border to appear lower than the plane of the window.
  //    SBT_NOBORDERS = text is drawn without borders
  //    SBT_OWNERDRAW = the text is drawn by the parent window
  //    SBT_POPOUT    = ..with a border to appear higher than the plane of the window.
} // end SetStatusWindowText()



//**************************************************************************
// Common Dialog Boxes ...
//**************************************************************************

void InitStructOfn( OPENFILENAME *pOfnStruct,
          HWND hwndOwner, HINSTANCE hInstance,
          char *pszTitle,  char *pszFileAndPath, int iMaxLength,
          char *pszFilter,
                // Pointer to a buffer containing pairs of null-terminated filter strings.
                // The last string in the buffer must be terminated by two NULL characters.
                //    Example: "Text files\0*.TXT;*.DOC;*.BAK\0\0"

          char *pszDefaultExtension )
{
 int i, iFilterIndex;
 char *pszFilterScan;
 char *pszExtension;


  // Set the default filter index to the extension of the filename (!) .
  // This is especially important because the LOUSY EXPLORER-STYLE file selector
  // does not even show the extension after the filename !!!! (HOLY SHIT)
  pszExtension = strrchr(pszFileAndPath,'.');
  pszFilterScan= pszFilter;
  iFilterIndex = 0;
  if(pszExtension && pszFilterScan!=NULL )
   { // Example: pszExtension should point to ".txt" or ".rtf" now (hopefully)
     i = 0;
     do
      { ++i;
       if(*pszFilterScan=='\0') break;
       while(*pszFilterScan!=0) ++pszFilterScan; // skip 1st part, like "Text files"
       ++pszFilterScan;   // skip \0 after 1st part of filter pair
       if(*pszFilterScan=='\0') break;  // double zero byte means end of list
       while(*pszFilterScan != '.' && (BYTE)*pszFilterScan>0)
         ++pszFilterScan;  // skip "*" in most cases
       if(*pszFilterScan!='.')  break;  // unexpected filter format
       if(stricmp(pszExtension,pszFilterScan)==0)
        { // hooray, found the extension !
          iFilterIndex = i;     break;
        }
       while(*pszFilterScan!=0) ++pszFilterScan; // skip 1st part, like "*.txt"
       ++pszFilterScan;   // skip \0 after 2nd part of filter pair
      }while(   (iFilterIndex==0) && (i<10)/*emergency brake*/   );
   } // end if if(pszExtension)


  ZeroMemory(pOfnStruct, sizeof(OPENFILENAME) );
  pOfnStruct->lStructSize = sizeof(OPENFILENAME);
  pOfnStruct->hwndOwner = hwndOwner;
  pOfnStruct->hInstance = hInstance;
  pOfnStruct->lpstrFilter = pszFilter;
  pOfnStruct->lpstrCustomFilter = NULL;   // this is a difficult beast !
  // Quoted from some paper about OPENFILENAME.lpstrCustomFilter :
  // > Pointer to a static buffer that contains a pair of null-terminated
  // > filter strings for preserving the filter pattern chosen by the user.
  // > The first string is your display string that describes the custom filter,
  // > and the second string is the filter pattern selected by the user.
  // > The first time your application creates the dialog box, you specify
  // > the first string, which can be any nonempty string.
  // > When the user selects a file, the dialog box copies
  // > the current filter pattern to the second string.
  // > The preserved filter pattern can be one of the patterns specified
  // > in the lpstrFilter buffer, or it can be a filter pattern typed by the user.
  // > The system uses the strings to initialize the user-defined file filter
  // > the next time the dialog box is created.
  // > If the nFilterIndex member is zero, the dialog box uses the custom filter.
  // > If this member is NULL, the dialog box does not preserve user-defined filter patterns.
  // > If this member is not NULL, the value of the nMaxCustFilter member
  // >    must specify the size, in bytes (ANSI version) or characters (Unicode version),
  // >    of the lpstrCustomFilter buffer.
  // Heck, why do they have to turn simple tasks into big science ?
  pOfnStruct->nMaxCustFilter = 0;
  pOfnStruct->nFilterIndex   = iFilterIndex;
  pOfnStruct->lpstrFile = pszFileAndPath;
  // > lpstrFile : Pointer to a buffer that contains a filename used to initialize the File Name edit control.
  //
  pOfnStruct->nMaxFile  = iMaxLength;
  pOfnStruct->lpstrFileTitle = NULL;  // can receive the file name w/o path
  pOfnStruct->nMaxFileTitle  = 0;
  pOfnStruct->lpstrInitialDir = NULL; // NULL = use 'current' directory
  pOfnStruct->lpstrTitle = pszTitle;
  pOfnStruct->Flags = OFN_NOCHANGEDIR | OFN_FILEMUSTEXIST;
  pOfnStruct->lpstrDefExt = pszDefaultExtension;
}

//-------------------------------------------------------------------------
BOOL WS_SelectFileToOpen( HWND hwndOwner,HINSTANCE hInstance,
                char *pszTitle,
                char *pszFileAndPath, // default filename + buffer for result
                int iMaxLength, // the buffer should be at least 256 characters long.
                char *pszFilter,
                   // Pointer to a buffer containing pairs of null-terminated filter strings.
                   // The last string in the buffer must be terminated by two NULL characters.
                   //    Example: "Text files\0*.TXT;*.DOC;*.BAK\0\0"
                char *pszDefaultExtension )  // only needed if user fails to ENTER an extension
{
  OPENFILENAME OfnStruct;
  // If the caller PASSED IN a non-empty filename, he will have a reason for that:
  // Most likely, that name (and path) shall be PRESET in the damned dialog.
  // But it was quite tough to convince GetOpenFileName() to do that.
  // See notes
  InitStructOfn( &OfnStruct, hwndOwner, hInstance,
                pszTitle,  pszFileAndPath, iMaxLength,
                pszFilter, pszDefaultExtension );
  return GetOpenFileName( &OfnStruct );
}

//-------------------------------------------------------------------------
BOOL WS_SelectFileToSave( HWND hwndOwner,HINSTANCE hInstance,
                char *pszTitle,
                char *pszFileAndPath, // default filename + buffer for result
                int iMaxLength, // the buffer should be at least 256 characters long.
                char *pszFilter,
                   // Pointer to a buffer containing pairs of null-terminated filter strings.
                   // The last string in the buffer must be terminated by two NULL characters.
                   //    Example: "Text files\0*.TXT;*.DOC;*.BAK\0\0"
                char *pszDefaultExtension )  // only needed if user fails to ENTER an extension
{
 OPENFILENAME OfnStruct;
  InitStructOfn( &OfnStruct, hwndOwner, hInstance,
                pszTitle,  pszFileAndPath, iMaxLength,
                pszFilter, pszDefaultExtension );
  return GetSaveFileName( &OfnStruct );
}

//-------------------------------------------------------------------------
BOOL SelectFont( HWND hwndOwner,DWORD *pdwTextColorRGB,LOGFONT *pLogFont )
{
  CHOOSEFONT cf;            // common dialog box structure

  // Initialize CHOOSEFONT
  ZeroMemory(&cf, sizeof(CHOOSEFONT));
  cf.lStructSize = sizeof (CHOOSEFONT);
  cf.hwndOwner = hwndOwner;
  if(pLogFont)
     cf.lpLogFont = pLogFont;
  if(pdwTextColorRGB)
     cf.rgbColors = *pdwTextColorRGB;
  cf.Flags = CF_SCREENFONTS | CF_EFFECTS | CF_INITTOLOGFONTSTRUCT;

  if (ChooseFont(&cf)==TRUE)
   {
    if(pLogFont)        *pLogFont = *cf.lpLogFont;
    if(pdwTextColorRGB) *pdwTextColorRGB=cf.rgbColors;
    return TRUE;
   }
  return FALSE;
} // end SelectFont()


//--------------------------------------------------------------------------
double GetSecondsSinceMidnightUTC( void )
{
 // The GetSystemTime function retrieves the current system date and time.
 // The system time is expressed in Coordinated Universal Time (UTC).
 SYSTEMTIME t;

 GetSystemTime( &t );
 return (double)t.wHour*3600.0
       +(double)t.wMinute*60.0
       +(double)t.wSecond
       +(double)t.wMilliseconds * 1E-3;
}


//--------------------------------------------------------------------------
// Some helpers to deal with INI-files
//--------------------------------------------------------------------------
const char *IntToStr(long l)
{ static char sz15Result[16];
  sprintf(sz15Result,"%ld",(long)l );
  return sz15Result;
}

long ReadIntFromIniFile( const char *pszIniFileName, const char *pszSection, const char *pszKeyName, long lDefault)
{
 return GetPrivateProfileInt(  // what an ugly argument sequence in this WinAPI routine..
    // section, key name,   default value, ini-file name :
    pszSection, pszKeyName, lDefault,      pszIniFileName);
} // end ReadIntFromIniFile()

void WriteIntToIniFile( const char *pszIniFileName, const char *pszSection, const char *pszKeyName, long lValue)
{ char sz31Temp[32];
  sprintf(sz31Temp, "%ld", (long)lValue);
  WritePrivateProfileString( pszSection, pszKeyName, sz31Temp, pszIniFileName);
} // end WriteIntToIniFile()

double ReadDoubleFromIniFile( const char *pszIniFileName, const char *pszSection, const char *pszKeyName, double dblDefault)
{
  double d;
  char sz31Temp[32];
  sz31Temp[0] = '\0';
  GetPrivateProfileString(  // what an ugly argument sequence in this WinAPI routine..
    // section, key name,   default value,  returnedString, maxLen, ini-file name :
    pszSection, pszKeyName, "",             sz31Temp,       31,     pszIniFileName);
  if( sz31Temp[0] != '\0' )
   { d = atof( sz31Temp );  // supported format : [whitespace][sign][ddd][.][ddd][e|E[sign]ddd]
     if( (d != HUGE_VAL) && (d != -HUGE_VAL) )
      { return d;
      }
   }
  return dblDefault;
} // end ReadDoubleFromIniFile()

void WriteDoubleToIniFile( const char *pszIniFileName, const char *pszSection, const char *pszKeyName, double dblValue)
{ char sz31Temp[32];
  sprintf(sz31Temp, "%lf", dblValue);
  WritePrivateProfileString( pszSection, pszKeyName, sz31Temp, pszIniFileName);
} // end WriteDoubleToIniFile()

void ReadStringFromIniFile( const char *pszIniFileName, const char *pszSection, const char *pszKeyName,
            char *pszDefault,  char *pszResult, int iMaxLen )
{
  GetPrivateProfileString(  // what an ugly argument sequence in this WinAPI routine..
    // section, key name,   default value,  returnedString, maxLen, ini-file name :
    pszSection, pszKeyName, pszDefault,      pszResult,   iMaxLen,   pszIniFileName);
} // end ReadStringFromIniFile()

void WriteStringToIniFile( const char *pszIniFileName, const char *pszSection, const char *pszKeyName, char *pszValue )
{
  WritePrivateProfileString( pszSection, pszKeyName, pszValue, pszIniFileName);
} // end ReadStringFromIniFile()

//---------------------------------------------------------------------------
// Snipped from \cbproj\YHF_Tool\YHF_Dialog.cpp (but converted to "C" ) :
//   internal variables for the "simple dialog windows" (w/o resources)
//---------------------------------------------------------------------------
static int   YhfDlgs_iEditOptions = 0;
static char *YhfDlgs_pszTitle = "NoTitle";
static char *YhfDlgs_pszLabel1= "NoLabel1";
static char *YhfDlgs_pszLabel2= "NoLabel2";
static char  YhfDlgs_sz80EditedString[84];
static HWND  YhfDlgs_hwndModalDialog;
static HWND  YhfDlgs_hwndEditField;
static long  YhfDlgs_i32BackgndColor = 0x00F0E0E0; /*light bluish gray by default*/
static BOOL  YhfDlgs_fDialogWndclassRegistered = FALSE;
static BOOL  YhfDlgs_fDialogWindowClosed;
static int   YhfDlgs_iModalResult;

//------------------------------------------------------------------------
LONG YhfDlg1_OnWmPaint( HWND hWnd )
{ // reaction to a WM_PAINT message
  HDC hDC;
  // HPEN hPen,hOldPen;
  HBRUSH hBrush,hOldBrush;
  PAINTSTRUCT ps;
  RECT rctClientArea;
  int  iScreenWidth,iScreenHeight;

  hDC = BeginPaint(hWnd,&ps);

  // Center a message in the "parameter display area"..
  GetClientRect(hWnd,&rctClientArea);
  iScreenHeight = rctClientArea.bottom-rctClientArea.top;
  iScreenWidth  = rctClientArea.right-rctClientArea.left;
  if(iScreenWidth>50 && iScreenHeight>50)
   {

    // Erase the entire background (that's why we don't want WM_ERASEBKGND)
    hBrush  = CreateSolidBrush( YhfDlgs_i32BackgndColor );
    hOldBrush = SelectObject( hDC, hBrush );
    SetBkColor( hDC, YhfDlgs_i32BackgndColor );
    FillRect( hDC, &rctClientArea, hBrush );
    if( YhfDlgs_pszLabel1 != NULL )
     { LeftAlignedTextOut( hDC, 16/*x*/, 8/*y*/, YhfDlgs_pszLabel1 );
     }
    if( YhfDlgs_pszLabel2 != NULL )
     { LeftAlignedTextOut( hDC, 16/*x*/, 32/*y*/, YhfDlgs_pszLabel2 );
     }
    SelectObject( hDC, hOldBrush );
    DeleteObject( hBrush );
   }

  EndPaint(hWnd,&ps);


  // An application should return zero if it processes this message...
  return 0L;
} // end YhfDlg1_OnWmPaint()


/****************************************************************************/
static LRESULT CALLBACK YhfDlgs_WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
  // "window procedure" (message handler) for the simple "dialog"(*) window
  // (*) it's modal, but in reality it's NOT a dialog window !
{
  switch(message)
   {
     // case WM_INITDIALOG:  // not handled here ! we look like a dialog but aren't !
     case WM_COMMAND:
        switch(LOWORD(wParam))
         {
           case WS_DLG_BTN_OK:
                if( YhfDlgs_hwndEditField != NULL )
                 { memset( YhfDlgs_sz80EditedString, 0, sizeof(YhfDlgs_sz80EditedString) );
                   GetWindowText( YhfDlgs_hwndEditField, YhfDlgs_sz80EditedString, 80 );
                 }
                YhfDlgs_iModalResult = WS_DLG_BTN_OK;
                SendMessage( hWnd, WM_CLOSE, 0, 0 ); // CloseWindow( hWnd ) didn't work
                return 0;
           case WS_DLG_BTN_CANCEL:
                YhfDlgs_iModalResult = WS_DLG_BTN_CANCEL;
                SendMessage( hWnd, WM_CLOSE, 0, 0 );
                return 0;
         }
        break;

     case WM_ERASEBKGND: // sent when the window background must be erased
        // > If the application returns zero, the window will remain marked for erasing.
        return 0;

     case WM_PAINT:
        return YhfDlg1_OnWmPaint( hWnd );

     case WM_CLOSE:  // someone closed the dialog window, somehow..
        // > The WM_CLOSE message is sent as a signal that a window
        // > (or an application, which is not the case here) should terminate.
        // > An application can prompt the user for confirmation, prior to
        // > destroying a window, by processing the WM_CLOSE message and calling
        // > the DestroyWindow function only if the user confirms the choice.
        // (a-ha. sounds like we should call DestroyWindow() in the window procedure)
        YhfDlgs_fDialogWindowClosed = TRUE; // the window hasn't closed ITSELF yet
        DestroyWindow( hWnd );  // sends WM_DESTROY and cleans up..
        // > (WM_QUIT) If an application processes this message, it should return zero.
        // (not the stupid info from someone talking about Win32 API programming..
        //    > "in general you return FALSE for messages you don't process," NO-NO-NO !
        return 0;  // return ZERO because we DID process this message .
     case WM_DESTROY: // we're being destroyed;
        YhfDlgs_fDialogWindowClosed = TRUE; // should already have been set in WM_CLOSE, anyway..
        break;     // let DefWindowProc() look at this, too

     default:
        break;  // let DefWindowProc() do whatever it needs to ... see below
        // DON'T ASSUME ANYTHING, ESPECIALLY NOT WHETER TO RETURN "TRUE" OR "FALSE" HERE !
   } // end switch(Message)

  // By order of the prophet (Micro$oft, in the Win32 programmer's ref or the MSDN):
  // > An application-defined window procedure should pass any messages
  // > that it does not process to the DefWindowProc function for default
  // > processing.
  // Note that -for good reason- they don't say anything about
  //      *THE VALUE* to return .
  // You'll sometimes find advice on the net like this:
  //    > in general you return FALSE for messages you don't process,
  //    > and TRUE for messages you do process
  //  THIS MAY BE TRUE "IN MANY CASES" BUT TAKEN STRICTLY, IT'S WRONG .
  //  Do you know which messages M$ will come up with in windoze 7,8,9 ?  ;o)
  return DefWindowProc (hWnd, message, wParam, lParam) ;
} // end YhfDlgs_WndProc()

/****************************************************************************/
static int YhfDlgs_RunModalDialog( HWND hwndOwner, // INTERNAL function !
              int iWidth, int iHeight )
{
  int  x,y, screen_width, screen_height, button_width;
  WNDCLASS wndclass ;
  HWND hwndOldFocus;
  MSG  Msg;
  static BOOL already_here = FALSE;

  if( WS_hInstance==NULL || hwndOwner==NULL )
   { return WS_DLG_BTN_CANCEL;  // say "not open for business"
   }
  if( already_here )   // avoid recursion ! this dialog mustn't call itself
   { return WS_DLG_BTN_CANCEL;  // say "not open for business"
   }
  already_here = TRUE;

  // Forget about "CreateDialog" and similar - they all need RESOURCE FILES,
  // or at least RESOURCE TEMPLATES .  Googling around for a while showed this:
  // > I've been reading about that DialogBox function and tried to understand,
  // > why the window which is displayed using that function becomes modal,
  // > I was actually trying to read between the lines, there had to be
  // > something which triggered that effect, that's what I've been looking for.
  // > As I went deeper, I found out that dialog boxes simply disable
  // > the owner window , so what I actually wanted to hear in the reply,
  // > was this one line (two actually):
  // >  Code:
  // > // add this line before calling the ShowWindow to open a modal window
  // > wasEnabled = EnableWindow(hwndOwner, FALSE); // disable the owner window
  // > // before calling function to close the child (modal) window, add this:
  // > EnableWindow(hwndOwner, wasEnabled);
  // But that's not the full story. The application's endless message loop
  // must still be kept alive, so we cannot simply run in our own little loop
  // here (or can we .. until the OK- or CANCEL- button was clicked ? ? )
  // The catch is that a function like RunStringEditDialog() should not return
  // to the caller until the input is "finished". Everything else would make
  // things utterly complex (and one would be better off with a non-modal,
  //  i.e. a non-blocking window, which is not a "dialog"-window) .

  // set up window class (if not already registered in a previous call)
  if( ! YhfDlgs_fDialogWndclassRegistered )
   { wndclass.style         = CS_HREDRAW | CS_VREDRAW ;
     wndclass.lpfnWndProc   = YhfDlgs_WndProc ;
     wndclass.cbClsExtra    = 0 ;
     wndclass.cbWndExtra    = 0 ;
     wndclass.hInstance     = WS_hInstance;
     wndclass.hIcon         = NULL;
     wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW);
     wndclass.hbrBackground = (HBRUSH)COLOR_WINDOW;
     wndclass.lpszMenuName  = 0 ;   // no menu !
     wndclass.lpszClassName = "YhfDlgWindow"; // window name
     RegisterClass (&wndclass) ;
   }

  // Set up some internal (static) variables for the window procedure:
  YhfDlgs_fDialogWindowClosed = FALSE; // the window hasn't closed ITSELF yet
  YhfDlgs_iModalResult = WS_DLG_BTN_NONE;
  screen_width  = GetSystemMetrics(SM_CXSCREEN); // width of the entire "screen" (desktop of a single monitor?) in pixels
  screen_height = GetSystemMetrics(SM_CYSCREEN); // height of the entire "screen" in pixels
  hwndOldFocus  = GetFocus();

  // now create the dialog window (which is a "normal" window so far..)
  x = (screen_width-iWidth)   / 2;
  y = (screen_height-iHeight) / 2;
  YhfDlgs_hwndModalDialog = CreateWindow(
    "YhfDlgWindow",   // lpClassName,    pointer to registered class name
    YhfDlgs_pszTitle, // "lpWindowName", pointer to window name, effectively the TITLE
    // DWORD dwStyle,  window style. Tried a lot ...
    // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX,  // non-sizeable window
    // WS_VSCROLL | WS_HSCROLL |  // scroll bars ? heavens, no !
       WS_OVERLAPPEDWINDOW | WS_VISIBLE, // quite normal window
    // WS_POPUP | WS_DLGFRAME ,   // window with no title, quite unusual as main window
    // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_SIZEBOX, // sizeable
    // WS_CAPTION,   // with title, but no icon and no system menu nor close symbol
    x, y, iWidth, iHeight,
    hwndOwner,      // HWND hWndParent, handle to parent or owner(!) window
    NULL,           // HMENU hMenu,     handle to menu or child-window identifier
    WS_hInstance, // HANDLE hInstance, handle to application instance
    NULL);          // LPVOID lpParam,   pointer to window-creation data
  if( YhfDlgs_hwndModalDialog != NULL )
   { // CreateWindow (for the "dialog") was successfull...
     EnableWindow(hwndOwner, FALSE); // disable the owner window
     button_width = 64;
     x = (iWidth-3*button_width) / 2;
     CreateWindow( "BUTTON"/*ClassName*/,  "Ok"/*window name*/,
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | BS_DEFPUSHBUTTON	 | BS_TEXT /*Style*/,
                  x,
                  iHeight - 80,  // y, ex: iHeight-64, but the button was only "half visible" on some late Windoze with its UGLY FAT BORDERS
                  button_width, 24/*height*/,
                  YhfDlgs_hwndModalDialog/*hWndParent*/,
                  (HMENU)WS_DLG_BTN_OK, /*hMenu abused for the CONTROL ID - gg "Assigning a control id to a win32 button" */
                  WS_hInstance, NULL);
     CreateWindow( "BUTTON"/*ClassName*/,  "Cancel"/*window name*/,
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | BS_PUSHBUTTON | BS_TEXT /*Style*/,
                  iWidth-button_width-x/*x*/,
                  iHeight - 80,  // y, ex: iHeight-64, but the button was only "half visible" on some late Windoze
                  button_width, 24/*height*/,
                  YhfDlgs_hwndModalDialog/*hWndParent*/,
                  (HMENU)WS_DLG_BTN_CANCEL, WS_hInstance, NULL);
     YhfDlgs_hwndEditField = CreateWindow( "EDIT"/*ClassName*/,
                  YhfDlgs_sz80EditedString, /* "window name" (here: initial text) */
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | WS_BORDER | ES_LEFT  /*Style...*/
                     | ( (YhfDlgs_iEditOptions & WS_EDIT_DECIMAL_ONLY )  // allow ONLY NUMERIC input ? (forget it, if number may be NEGATIVE)
                          ? ES_NUMBER	: 0 )  ,
                  16/*x*/,
                  iHeight-80-32/*y*/,
                  iWidth-48,
                  24/*height*/,
                  YhfDlgs_hwndModalDialog/*hWndParent*/,
                  (HMENU)WS_DLG_ID_EDIT1, WS_hInstance, NULL);
     // Set the focus (with the blinking cursor) to the EDIT FIELD (not to the "ok"-button) :
     SetFocus( YhfDlgs_hwndEditField );

     // Dialog loop: process all window messages (not just "our own"! ),
     //      SIMILAR to what you'll find in most win32 API application .
     // But: If GetMessage() receives the WM_QUIT message, the return value is zero.
     //      This doesn't help us to return from the modal "dialog" here, because
     //      the WM_QUIT message indicates a request to terminate an *application*
     //      and is generated when the application calls the PostQuitMessage
     //      function. It causes the GetMessage function to return zero.
     // Fix: If YhfDlgs_WndProc() closes its window, it kindly sets a flag
     //      which we stupidly poll in the message loop below .
     while(GetMessage(&Msg, NULL, 0, 0))
      {
        // From MSDN (remember this if the TAB KEY doesn't switch the active control..)
        // > Although the IsDialogMessage function is intended for modeless dialog boxes,
        // > you can use it with any window that contains controls,
        // > enabling the windows to provide the same keyboard selection
        // > as is used in a dialog box.
        // > When IsDialogMessage processes a message, it checks for keyboard messages
        // > and converts them into selections for the corresponding dialog box.
        // > For example, the TAB key, when pressed, selects the next control
        // > or group of controls, and the DOWN ARROW key, when pressed,
        // > selects the next control in a group.
        // > Because the IsDialogMessage function performs all necessary translating
        // > and dispatching of messages, a message processed by IsDialogMessage
        // > must *not* be passed to the TranslateMessage or DispatchMessage function.
        //
        if (!IsDialogMessage( YhfDlgs_hwndModalDialog, &Msg) ) // important for TAB-nav.
          {
            TranslateMessage(&Msg);
            DispatchMessage(&Msg);
          }
        if( YhfDlgs_fDialogWindowClosed )
           break;  // workaround for the missing WM_QUIT in the dialog
      }

     EnableWindow(hwndOwner, TRUE );
     SetFocus( hwndOldFocus );  // ex: SetFocus(hwndOwner);
   } // end if < creation of the dialog window successful >

  already_here = FALSE;  // accept next call
  return YhfDlgs_iModalResult; // WS_DLG_BTN_OK, WS_DLG_BTN_CANCEL; maybe more

} // end YhfDlgs_RunModalDialog()

/****************************************************************************/
int WS_RunIntegerInputDialog(  // MODAL dialog; returns WS_DLG_BTN_...
  HWND hwndOwner,  // handle to the window which will be blocked by the dialog
  char *pszTitle, char *pszLabel1,  char *pszLabel2,
  int  *piValue,  int iEditOptions, int  iHelpContext )
  // return value:  one of the WS_DLG_BTN_ codes defined in WS_Dialogs.h .
{
   int iButtonCode;

   YhfDlgs_iEditOptions = iEditOptions;
  //ex:  if( (iEditOptions & WS_EDIT_HEXADECIMAL) == 0 )
  //      { YhfDlgs_iEditOptions |= WS_EDIT_DECIMAL_ONLY; // allow ONLY NUMERIC input
  //      }
  // leave it to the caller to specify this flag.
  // With the 'ES_DECIMAL' style, the windoze thing doesn't accept NEGATIVE values !
   YhfDlgs_pszTitle = pszTitle;
   if(pszLabel1)
           YhfDlgs_pszLabel1 = pszLabel1;
     else  YhfDlgs_pszLabel1 = "";
   if(pszLabel2)
           YhfDlgs_pszLabel2 = pszLabel2;
     else  YhfDlgs_pszLabel2 = "";
   if(iEditOptions & WS_EDIT_HEXADECIMAL)
           sprintf(YhfDlgs_sz80EditedString,"$%04X",*piValue);
     else  sprintf(YhfDlgs_sz80EditedString, "%d",  *piValue);
   // StringEditForm->m_iHelpContext = iHelpContext;  // not supported here
   iButtonCode = YhfDlgs_RunModalDialog(hwndOwner, 320/*w*/, 200/*h*/ );
   if( iButtonCode==WS_DLG_BTN_OK )
    {
      if(YhfDlgs_sz80EditedString[0]=='$')
        { sscanf(YhfDlgs_sz80EditedString+1,"%X", piValue );
          // beware of lousy error handling in some scanf implementations !
        }
      else
        { sscanf(YhfDlgs_sz80EditedString,"%d", piValue );
        }
    }
   return iButtonCode;
} // end WS_RunIntegerInputDialog()

/****************************************************************************/
int WS_RunFloatInputDialog(  // MODAL dialog; returns WS_DLG_BTN_...
  HWND hwndOwner,  // handle to the window which will be blocked by the dialog
  char *pszTitle, char *pszLabel1, char *pszLabel2,
  float *pfltValue,  int iEditOptions, int  iHelpContext )
  // return value:  one of the WS_DLG_BTN_ codes defined in WS_Dialogs.h .
{
  float fltTemp;
  int  iButtonCode;

   YhfDlgs_iEditOptions = iEditOptions;
   YhfDlgs_pszTitle = pszTitle;
   if(pszLabel1)
           YhfDlgs_pszLabel1 = pszLabel1;
     else  YhfDlgs_pszLabel1 = "";
   if(pszLabel2)
           YhfDlgs_pszLabel2 = pszLabel2;
     else  YhfDlgs_pszLabel2 = "";
   sprintf(YhfDlgs_sz80EditedString,"%.3f", *pfltValue);
   // StringEditForm->m_iHelpContext = iHelpContext;  // not supported here
   iButtonCode = YhfDlgs_RunModalDialog(hwndOwner, 320/*w*/, 200/*h*/ );
   if( iButtonCode==WS_DLG_BTN_OK )
    {
      if(YhfDlgs_sz80EditedString[0]>0)
       { fltTemp = strtod( YhfDlgs_sz80EditedString, NULL);
         if( fltTemp!=HUGE_VAL && fltTemp!=-HUGE_VAL )
          { *pfltValue = fltTemp;
            return WS_DLG_BTN_OK;
          }
       }
    }
   return iButtonCode;
} // end WS_RunFloatInputDialog()



/****************************************************************************/
int WS_RunStringEditDialog( // MODAL dialog; returns WS_DLG_BTN_...
  HWND hwndOwner,  // handle to the window which will be blocked by the dialog
  char *pszTitle,
  char *pszLabel1, char *pszLabel2,
  char *pszDefaultValue, int iEditOptions,
  char *pszDestination,  int iMaxLength,
  int  iHelpContext )
  // return value:  one of the WS_DLG_BTN_ codes defined in WS_Dialogs.h .
{
  int  iButtonCode;

   YhfDlgs_iEditOptions = iEditOptions;
   YhfDlgs_pszTitle = pszTitle;
   if(pszLabel1)
           YhfDlgs_pszLabel1 = pszLabel1;
     else  YhfDlgs_pszLabel1 = "";
   if(pszLabel2)
           YhfDlgs_pszLabel2 = pszLabel2;
     else  YhfDlgs_pszLabel2 = "";
   if( pszDefaultValue != NULL )
    { strncpy(YhfDlgs_sz80EditedString, pszDefaultValue, 80);
    }
   else
    { YhfDlgs_sz80EditedString[0] = '\0';
    }
   // StringEditForm->m_iHelpContext = iHelpContext;  // not supported here
   iButtonCode = YhfDlgs_RunModalDialog(hwndOwner, 320/*w*/, 200/*h*/ );
   if( iButtonCode==WS_DLG_BTN_OK )
    {
      strncpy( pszDestination, YhfDlgs_sz80EditedString, iMaxLength);
      pszDestination[iMaxLength-1] = 0;
    }
   return iButtonCode;
} // end WS_RunStringEditDialog()


//---------------------------------------------------------------------------
BOOL WS_GetMouseCursorPos( int *piScreenX, int *piScreenY )
  //  Replacement for Borland's 'Mouse' object .
  //  Retrieves the mouse cursor position (SCREEN coordinate),
  //  without the bugs in the VCL ...
  //    from http://qc.codegear.com/wc/qcmain.aspx?d=9344
  //  > When Mouse.GetCursorPos is called while Windows is switching
  //  > the active desktop it fails and returns:
  //  >  "System error.  Code: 6 (sometimes 5). The handle is invalid / "access denied"
  //  >  Switching the desktop can happen when the screensaver is activated
  //  > or when the computer is being locked using Ctrl+Alt+Delete.
  //  > THintWindow.ActivateHint uses Mouse.GetCursorPos to determine
  //  > the mouseposition, so this can lead to the problem described.
{
  POINT pt;
  BOOL  fOkToGetCursorPos;
  HDESK hDesk;  // added 2009-09-09 to cure the "handle leak"
 // DWORD dwLastError;
 // char  sz80[84];
  OSVERSIONINFO windoze_version_info;

  *piScreenX = 0; // return dummy coord if the call fails..
  *piScreenY = 0;


  // First check if Desktop is not LOCKED ...
  if(  (hDesk=OpenInputDesktop( // 2009-09-09: Calling this function EATS ONE MORE HANDLE !
     0 ,// DWORD dwFlags, flags to control interaction with other applications
     FALSE, // BOOL fInherit, specifies whether returned handle is inheritable
     DESKTOP_READOBJECTS)) // DWORD dwDesiredAccess, specifies access of returned handle
      != NULL )
   { // the desktop doesn't seem to be "locked",
     // it should be ok to trying to retrieve the mouse position:
     fOkToGetCursorPos = TRUE;
     // 2009-09-09 : Just a "feeling", even though the Win32 API reference doesn't mention this,
     //              a function beginning with "Open".. should have an associated "Close".
     //    This seemed to be missing in the (otherwise nice) example
     //    at http://qc.codegear.com/wc/qcmain.aspx?d=9344 ! !
     CloseDesktop( hDesk );  // close this desktop handle thing, whatever it is exactly.
     // At least it releases the handle (but doesn't close the visible desktop, sigh..)
   } // end if < Desktop NOT LOCKED >
  else // OpenInputDesktop() failed. There may be different reasons for this:
   { // - the desktop may be LOCKED (under Windows XP)
     // - the PC is running windows 98 (yes, some people still do !)
     //     [not clear if OpenInputDesktop() is available under Win98,
     //      since the lousy microsoft documentation never mentions it anymore]
     fOkToGetCursorPos = FALSE;
  // dwLastError = GetLastError();  // meaningless result under win98, removed
  // // Got here with dwLastError=120=0x00078 under Windows 98, when the desktop
  // //     was definitely "valid".
  // QFile_LastErrorCodeToString(dwLastError/*in*/, sz80/*out*/, 80 );
  // //  Win89 -> "Die Funktion ist nur im Win32-Modus gltig" ?!?!
  // //   (forget about this, it's utterly useless, don't rely on that junk)
  // //   Second attempt (2009-02-19) :
     // Which version of windoze are we running,  win98 or xp ?
     windoze_version_info.dwOSVersionInfoSize = sizeof( OSVERSIONINFO );
     if( GetVersionEx( &windoze_version_info ) )
      { if( windoze_version_info.dwMajorVersion >= 5 )
         { // Win2000, Win XP, etc..  , but definitely NOT win98:
           // Must not ignore the error from OpenInputDesktop(),
           //  otherwise the program would -most likely- crash !
         }
        else // Got here with windoze_version_info.dwMajorVersion=4 under Win98.
         { // -> We're running on a stoneage computer, using a stoneage OS,
           // most likely the unsuccessful call of OpenInputDesktop()
           // must be IGNORED in this case :
           fOkToGetCursorPos = TRUE; // Call GetCursorPos() even though OpenInputDesktop() failed !
         }
      }
   }
  if( fOkToGetCursorPos )
   { if( GetCursorPos( &pt ) )
      { *piScreenX = pt.x;
        *piScreenY = pt.y;
        return TRUE;
      }
   } // end if < Ok To GetCursorPos >


  // Arrived here ?
  // Bad luck; windoze doesn't allow reading the mouse-coord at the moment.
  return FALSE;
} // end WS_GetMouseCursorPos()


// EOF <WinStuff.c>
