/***************************************************************************/
/*  WaveIO.h :                                                             */
/*   Unit to import and export *.wav - files.                              */
/*   - by DL4YHF August '2000                                              */
/*   - based on the description of the WAVE-file format                    */
/*     found at http://www.wotsit.org  (excellent site, all file formats!  */
/*   - Converted from C++ to plain C for WSQ2, see C:\cbproj\WSQ2\WaveIO.h */
/*-------------------------------------------------------------------------*/

#ifndef _WAVE_IO_H_
#define _WAVE_IO_H_

/*----------- Constant Definitions ----------------------------------------*/
// Optional flags for the parameter 'channel_nr' in ReadSampleBlocks() :
#define AUDIO_FILE_TWO_CHANNELS_PER_BLOCK 0x0200  // read TWO channels per destination block

// Possible values for the parameter 'file_mode' in OutOpen() :
#define AUDIO_FILE_MODE_OVERWRITE 0
#define AUDIO_FILE_MODE_APPEND    1

#define AUDIO_FILE_MAX_SAMPLE_RATE 10000000 /* [Hz].. 10 MSample/sec for SETI */

// Bitflags for .iDataTypeFlags  ( in addition to .iSizeOfDataType ) :
#define AUDIO_FILE_DATA_TYPE_NORMAL   0    /* "normal" = none of the following: */
#define AUDIO_FILE_DATA_TYPE_SIGNED_I 0
#define AUDIO_FILE_DATA_TYPE_UNSIGNED 0x01 /* if NOT set, it's a SIGNED integer */
#define AUDIO_FILE_DATA_TYPE_FLOAT    0x02
#define AUDIO_FILE_DATA_TYPE_MOTOROLA 0x04

// Audio file options for OPENING / CREATING an audio file . Parameter iAudioFileOptions.
//  Added to write Ogg Vorbis files with a timestamp channel (2010-06-06) .
#define AUDIO_FILE_OPTION_NORMAL                      0x0000
#define AUDIO_FILE_OPTION_TIMESTAMPS                  0x0001
#define AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER 0x0002
#define AUDIO_FILE_OPTION_SAVE_EXTRA_INFO_IN_AUX_FILE 0x0004




/*----------- Data Types --------------------------------------------------*/

// A RIFF file has an 8-byte RIFF header, identifying the file,
// and giving the residual length after the header (i.e. file_length - 8):
typedef struct
{
  char  id_riff[4];   // identifier string = "RIFF"
  DWORD len;          // remaining length after this header
  // The riff_hdr is immediately followed by a 4-byte data type identifier.
  // For .WAV files this is "WAVE" as follows:
  char wave_id[4];    // WAVE file identifier = "WAVE"
} T_WAV_RIFF_HDR;


// RIFF Chunks
//   The entire remainder of the RIFF file is "chunks".
//   Each chunk has an 8-byte chunk header identifying the type of chunk,
//   and giving the length in bytes of the data following the chunk header,
//   as follows:

typedef struct {      // CHUNK 8-byte header
  char  id[4];        // identifier, e.g. "fmt " or "data" , sometimes "inf1" (proprietary)
  DWORD dwChunkSize;  // remaining chunk/junk:) length after header
                      // data bytes follow chunk header .
     // Note: 32 bit unsigned for the "chunk size" limit the size
     //       of a wave file to 4 GBytes (or GiB, if you are a wiki geek) !
} T_WAV_CHUNK_HDR;


// The WAVE form is defined as follows. Programs must expect(and ignore)
// any unknown chunks encountered, as with all RIFF forms.
// However, <fmt-ck> must always occur before <wave-data>, and both of these
// chunks are mandatory in a WAVE file.
//
// WAVE-form> ->
//	RIFF( 'WAVE'
//	<fmt-ck>          	// Format
//	[<fact-ck>]          // Fact chunk
//	[<cue-ck>]           // Cue points
//	[<playlist-ck>] 		// Playlist
//	[<assoc-data-list>] 	// Associated data list
//	<wave-data>   ) 		// Wave data
//

// WAVE Format Chunk
//  The WAVE format chunk <fmt-ck> specifies the format of the <wave-data>.
//  Since I only work with PCM format, this structure is as follows
//   (combination of <commond-fields> and <format-specific-fields>):
typedef struct {
  // The first 16 bytes form the classic 'WAVEFORMAT' structure :
  WORD  wFormatTag;         // Format category,    1=Microsoft PCM
  WORD  wChannels;          // Number of channels, 1=mono, 2=stereo
  DWORD dwSamplesPerSec;    // Sampling rate     , samples per second
  DWORD dwAvgBytesPerSec;   // For buffer estimation
  WORD  wBlockAlign;        // Data block size ?? Bytes per Sample ????
  WORD  wBitsPerSample;     // Sample size, usually 8 or 16 bits per sample,
              // some M$ article says "Rigidly Defined as Container Size" (?)

  // After the classic 'WAVEFORMAT', there is another 16-bit WORD called cbSize:
  WORD   cbSize;   // added 2010-05-27 (to read 24-bit wave audio files) .....
  // <<<<  this is the end of what microsoft calls 'WAVEFORMATEX'  .
  //       Note that 'WAVEFORMATEX' includes the 'cbSize' member .
  //   If wFormatTag is 1,     the format is the classic "Microsoft PCM".
  //   If wFormatTag is 65534, the format is 'WAVEFORMATEXTENSIBLE';
  //      in that case, the WAVEFORMATEX (above) is followed by these OPTIONAL
  //      fields, which we must know to read exotic files with 24 bits/sample,
  //      etc etc etc :
  union {     // specific to the WAVEFORMATEXTENSIBLE-thingy..
    WORD wValidBitsPerSample;  /* bits of precision          */
    WORD wSamplesPerBlock;     /* valid if wBitsPerSample==0 */
    WORD wReserved;            /* If neither applies, set to zero. */
   } Samples;
  DWORD dwChannelMask;     /* which channels are present in stream */
  /*GUID*/ BYTE b16GUID[16]; // you really don't want to fool around with this :)
     // > GUIDs identify objects such as interfaces,
     // > manager entry-point vectors (EPVs), and class objects.
     // > A GUID is a 128-bit value consisting of one group
     // > of 8 hexadecimal digits, followed by three groups of 4 hexadecimal
     // > digits each, followed by one group of 12 hexadecimal digits.
   // The total size of the "WAVE_FORMAT_EXTENSIBLE" structure should be 40
   //  - at least that's what the 'RIFF chunk header' indicated in a wave
   //    file with 24 bits/sample -> WAVEIO_SIZEOF_WAVE_FORMAT_EXTENSIBLE .
} T_WAV_CHUNK_FORMAT_EX;
#define WAVEIO_SIZEOF_STANDARD_WAVEFORMAT    16
#define WAVEIO_SIZEOF_WAVE_FORMAT_EXTENSIBLE 40 /* found in a 24-bit wave file */

// The wBitsPerSample field specifies the number of bits of data used
// to represent each sample of each channel. If there are multiple channels,
// the sample size is the same for each channel. (... less interesting details removed here...)
// Data Chunk
//  The Data chunk contains the actual sample frames (ie, all channels
//  of waveform data).

typedef struct {
    char   id[4];   // "data" (4 bytes)
    DWORD  chunkSize;
 // BYTE   waveformData[chunkSize-8];
} T_WAV_DATA_CHUNK;

// The ID is always data. chunkSize is the number of bytes in the chunk,
// not counting the 8 bytes used by ID and Size fields nor any possible
// pad byte needed to make the chunk an even size (ie, chunkSize is the
// number of remaining bytes in the chunk after the chunkSize field,
// not counting any trailing pad byte).
// The waveformData array contains the actual waveform data.
// The data is arranged into what are called sample frames.
// You can determine how many bytes of actual waveform data there is
// from the Data chunk's chunkSize field. The number of sample frames
// in waveformData is determined by dividing this chunkSize by the Format
// chunk's wBlockAlign.
// The Data Chunk is required. One, and only one, Data Chunk may appear
// in a WAVE.




/*----------- Definition of the T_WaveIO struct ---------------------------*/
typedef struct tWaveIO
{
  // ex: public:
   // Public properties for simplicity .. we know what we're doing, right ?
   T_WAV_RIFF_HDR     RiffHdr;
   T_WAV_CHUNK_HDR    ChunkHdr;
   T_WAV_CHUNK_FORMAT_EX ChunkFormat;  // a damned variable-sized monster since 2010-05-27 !

   double dblStartTime;  // seconds elapsed since 00:00:00 GMT, January 1, 1970.
   int    iFileFormat;
   int    iFileMode;
   int    iAudioFileOptions; // AUDIO_FILE_OPTION_NORMAL etc; see c:\cbproj\SoundUtl\AudioFileDefs.h
   char   cErrorString[84];
   BOOL   OpenedForReading;
   BOOL   OpenedForWriting;
   double dbl_SampleRate;   // precise sampling rate (possible with an additional header)

   // Some more properties for reading RAW FILES . Should be set before opening
   // a RAW file (extension *.raw or *.dat)
   int    iSizeOfDataType, iDataTypeFlags, iBitsPerSample, iNumChannels;
   WORD   wRawFileNumberOfChannels;
   double dblGainFactor;


  // ex private:
   INT   hFile;
   BOOL  fMustWriteHeader;
   DWORD dwFilePos_Data;  // file-internal position of 1st audio sample
   DWORD dwCurrFilePos;   // current file access position
   DWORD dwFileDataSize;  // netto size of the samples in BYTES. Sometimes read from "data" chunk
   char  sz255FileName[256];

} T_WaveIO;


BOOL WaveIO_InOpen( T_WaveIO *pWIO,
                char *pszFileName,     // [in] filename to open (for reading)
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength); // [in] size of pszExtraInfo
     /* Opens a wave file for READING audio samples.
      * Returns TRUE on success and FALSE on any error.
      */

   /************************************************************************/
DWORD WaveIO_GetTotalCountOfSamples(T_WaveIO *pWIO);
     /* Returns the total number of sampling points in the file .
      */


   /************************************************************************/
DWORD WaveIO_GetCurrentSampleIndex(T_WaveIO *pWIO);
     /* Reads the current sample index that will be used on the next
      * READ- or WRITE- access to the audio samples in the opened file.
      * Returns 0xFFFFFFFF on any error.
      */


   /************************************************************************/
BOOL WaveIO_WindToSampleIndex(T_WaveIO *pWIO, DWORD dwNewSampleIndex);
     /* Sets the current sample index that will be used on the next
      * READ- or WRITE- access to the audio samples an opened file.
      * Returns TRUE if ok ,  otherwise FALSE.
      * As long as you don't use this routine/method,
      * the audio file will be played in a strictly "sequential" way.
      */


   /************************************************************************/
LONG WaveIO_ReadSamples( T_WaveIO *pWIO, BYTE* pData, LONG Length );
     /* Reads some samples from a wave file which has been opened for READING.
      * "Length" is the size of the caller's buffer capacity in BYTES.
      * Return value: number of SAMPLE POINTS (intervals), not BYTES !!
      * More info: see WaveIO.cpp !
      */


   /************************************************************************/
LONG WaveIO_ReadSampleBlocks( T_WaveIO *pWIO,
         int channel_nr,    // channel_nr for 1st destination block
         int nr_samples,    // nr_samples (per destination block, length of destination arrays)
         float *pFltDest1, // 1st destination block (~"LEFT")
         float *pFltDest2, // 2nd destination block (~"RIGHT"), may be NULL
         char *pszExtraInfo, int iMaxStrlen ); // [out,optional] extra string with user data for wave.rd_info
     /* Reads some samples from a wave file which has been opened for READING
      * and converts them to double (floats) in the range of +- 32768.0 .
      * More info: see WaveIO.cpp !
      */



   /************************************************************************/
BOOL WaveIO_OutOpen( T_WaveIO *pWIO,
           char *file_name,
           int file_mode,        // AUDIO_FILE_MODE_OVERWRITE etc
           int iAudioFileOptions, // AUDIO_FILE_OPT_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
           int bits_per_sample,   // 8 or 16 (per SINGLE CHANNEL, not per 'frame')
           int channels,          // 1 (mono)  or  2 (stereo)
           double dblSampleRate);
     /* Creates and Opens an audio file for WRITING audio samples.
      *         Returns TRUE on success and FALSE on any error.
      *         Detailed description of arguments in WaveIO.c .
      */


   /************************************************************************/
   // ex: (not pivate) LONG WriteSamples( BYTE* pData, LONG Length); // low-level output, try to avoid
LONG WaveIO_WriteSamples_Float( T_WaveIO *pWIO, // app-layer output, using NORMALIZED floats
      float *pfltSrc, int nSamplePoints, int nSourceChannelsPerSample);


   /************************************************************************/
DWORD WaveIO_GetCurrentFileSize(T_WaveIO *pWIO);
     /* Returns the current file size (if there is an opened WAVE file).
      * Useful especially when logging audio samples  to keep track
      * of the used disk size (stop before Windows is getting into trouble).
      */

   /************************************************************************/
   // Some more 'Get' - and 'Set' - routines for the AUDIO FILE class ..
void   WaveIO_GetFileName(T_WaveIO *pWIO, char *pszDest, int iMaxLen);
double WaveIO_GetSampleRate(T_WaveIO *pWIO);
void   WaveIO_SetSampleRate(T_WaveIO *pWIO, double dblSampleRate);

int    WaveIO_GetNrChannels(T_WaveIO *pWIO);
void   WaveIO_SetNrChannels(T_WaveIO *pWIO, int iNrChannels);       // important to read RAW files

void   WaveIO_SetDataType(T_WaveIO *pWIO, int iSizeOfDataType,   int iDataTypeFlags); // important to read RAW files
void   WaveIO_GetDataType(T_WaveIO *pWIO, int *piSizeOfDataType, int *piDataTypeFlags);
int    WaveIO_GetBitsPerSample(T_WaveIO *pWIO);  // per "single" sample, regardless of number of channels
void   WaveIO_SetGain(T_WaveIO *pWIO,  double dblGainFactor );  // .. for raw files with FLOATING POINT numbers !
double WaveIO_GetGain(T_WaveIO *pWIO );

double WaveIO_GetCurrentRecordingTime(T_WaveIO *pWIO); // .. depending on current sample index
double WaveIO_GetRecordingStartTime(T_WaveIO *pWIO);   // seconds elapsed since 00:00:00 GMT, January 1, 1970.
void   WaveIO_SetRecordingStartTime(T_WaveIO *pWIO, double dblStartTime);

BOOL   WaveIO_AllParamsIncluded(T_WaveIO *pWIO);

BOOL   WaveIO_CloseFile(T_WaveIO *pWIO);


#endif // _WAVE_IO_H_

/* EOF <WaveIO.h> */