/***************************************************************************/
/*  WaveIO.c : A SIMPLIFIED VARIANT of WaveIO.cpp by DL4YHF .              */
/*   Unit to import and export *.wav - files and similar..                 */
/*  - by DL4YHF August '2000  ...  February 2014                           */
/*  - Converted from C++ to plain C for WSQ2 in February 2014 .            */
/*  - based on the description of the WAVE-file format                     */
/*    found at http://www.wotsit.org  (excellent site with file formats!)  */
/*  - does NOT use any fancy windows gimmics, just standard file I/O.      */
/*-------------------------------------------------------------------------*/

#ifdef __BORLANDC__
// prevent that bugger from bugging us with 'initialized data in header, cannot pre-compile..." :
# pragma hdrstop
#endif


#define __POCC__OLDNAMES /* to prevent having to use stupid underscores in many functions */

#include <windows.h> // definition of data types like LONGLONG (wtypes.h) etc
#include <stdlib.h>
#include <stdio.h>
#ifndef __BORLANDC__
# include <strings.h>  // stringS.h doesn't exist when compiling with Borland
#endif
#include <io.h>          /* _open (ex: _rtl_open) and other file functions */
#include <fcntl.h>       /* O_RDONLY  and other file flags     */
#include <sys/stat.h>    /* S_IREAD | S_IWRITE for _creat()    */
#include <time.h>

#ifdef __BORLANDC__
# define _tell tell // Borland compiler doesn't know '_tell', PellesLinker doesn't find 'tell'.  Hell.
# pragma warn -8004 // <var is a assigned a value that is never used> - so what
#endif

#include "WaveIO.h"  // header for this file with "single source"-variables

 // Found this GUID in an 'existing' wave file (pls55.wav), here as a BYTE array:
const BYTE WaveIO_b16GUID_PCM[16] = /* taken from an 'existing' wave file */
//  [0]  [1]  [2]  [3]  [4]  [5]  [6]  [7]  [8]  [9]  [10] [11] [12] [13] [14] [15]
  { 0x01,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x80,0x00,0x00,0xaa,0x00,0x38,0x9b,0x71 };
 // The MS site says it should be this "GUID" :
 //  00000001 - 0000 - 0010 - 8000 - 00aa00389b71
 // Note the strange different byte orders in bytes[6/7] and [8/9] !


// Helper functions (no class methods, to make them easy to use...)

/***************************************************************************/
static float ReadUns16Intel( short *sp )
{ return ((float)(*(unsigned short*)(sp)) - 32767.0)  * ( 1.0 / 32678.0 );
} // end ReadUns16Intel()

/***************************************************************************/
static float ReadSigned16Motorola( short *sp )
{ union { BYTE b[2];  short i16; } t16;
  t16.b[0] = ((BYTE*)sp)[1];
  t16.b[1] = ((BYTE*)sp)[0];
  return (float)t16.i16 * ( 1.0 / 32678.0 );
} // end ReadSigned16Motorola()

/***************************************************************************/
static float ReadUns16Motorola( short *sp )
{ union { BYTE b[2];  unsigned short u16; } t16;
  t16.b[0] = ((BYTE*)sp)[1];
  t16.b[1] = ((BYTE*)sp)[0];
  return ((float)t16.u16 - 32767.0) * ( 1.0 / 32678.0 );
} // end ReadUns16Motorola()


/***************************************************************************/
float Read24BitIntegerAsFloat( BYTE *pbSource )
{
  union { BYTE b[4];  long i32; } t;

  // We'll always run on a machine with 'INTEL BYTE ORDER', so : b[0] = LSB  .
  // The following worked properly with a test file [ M1F1-int24-AFsp.wav ]  !
  t.b[0] = *pbSource++;   // bits 7..0
  t.b[1] = *pbSource++;   // bits 15..8
  t.b[2] = *pbSource  ;   // bits 23..16
  if( t.b[2] & 0x80 )     // expand the sign-bit to bits 31..24 :
       t.b[3] = 0xFF;
  else t.b[3] = 0x00;
  return (float)t.i32 * ( 1.0 / (256.0*32768.0) );  // normalize to +/- 1.0 now

} // end Read24BitIntegerAsFloat()


/***************************************************************************/
void Conv24BitSamplesToFloat(
        BYTE *pbSource,          // [in] source data, 24-bit signed integers, possibly 'n' channels interleaved
        int  nSamplePoints,      // [in] number of SAMPLE POINTS (with possibly more than one channel each)
        int  nSrcChannelsPerSample, // [in] number of channels per sample point IN THE SOURCE
        int  nDstChannelsPerSample, // [in] number of channels per sample point IN THE DESTINATION
        float *pFltDest1,      // [out] 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
        float *pFltDest2)      // [out] 2nd destination block (~"RIGHT"), may be NULL (!)
{
  int i;
  DWORD dwAddressIncrementPerSample = 3 * nSrcChannelsPerSample;
  //ex: if(   ( (nChannelsPerSample==2) && (pFltDest2==NULL) )
  //       || ( (nChannelsPerSample==4) && (pFltDest2!=NULL) )  )
  if( nDstChannelsPerSample>1 )
   { // two channels (usually I+Q) per destination block,
     // which means TWO destination array indices (in pFltDest*) per sample point
     for(i=0; i<nSamplePoints; i++ )
      { // up to 8 bits per sample, TWO SEPARATE destination blocks:
        *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // 1st channel ("I")
        *pFltDest1++ = Read24BitIntegerAsFloat( pbSource+3 );  // 2nd channel ("Q")
        if( (nSrcChannelsPerSample==4) && (pFltDest2!=NULL) )
         { *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+6 ); // 3rd channel ("I")
           *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+9 ); // 4th channel ("Q")
         }
        pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .
      }
   }
  else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
   { // (in other words, ONE destination array index per sample point)
     if( (nDstChannelsPerSample<2) || (pFltDest2==NULL) )
      { // only ONE channel to be copied...
        for(i=0; i<nSamplePoints; i++ )
         { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
           *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // one DESTINATION channel
           pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .

         }
      }
     else // (ChunkFormat.wChannels>=2) && (pFltDest2!=NULL) :
      {  // TWO channels must be copied (and "split" into two destination blocks) :
        for(i=0; i<nSamplePoints; i++ )
         { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
           *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // 1st DESTINATION channel
           *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+3 );  // 2nd DESTINATION channel
           pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .
         }
      }
   } // end else < not TWO channels per destination block >
} // end Conv24BitSamplesToFloat()

/***************************************************************************/
void Conv32BitSamplesToFloat(
        BYTE *pbSource,          // [in] source data, 32-bit floating point, possibly 'n' channels interleaved
        int  nSamplePoints,      // [in] number of SAMPLE POINTS (with possibly more than one channel each)
        int  nSrcChannelsPerSample, // [in] number of channels per sample point IN THE SOURCE
        int  nDstChannelsPerSample, // [in] number of channels per sample point IN THE DESTINATION
        double dblGainFactor,    // [in] linear gain factor, often the reciprocal of the file's full-scale sample value
        float *pFltDest1,      // [out] 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
        float *pFltDest2)      // [out] 2nd destination block (~"RIGHT"), may be NULL (!)
{
  int i;
  float *pfltSource;
  // The source data must be aligned to 4-byte boundaries, thus:
  if( ((int)pbSource&3)==0)
   { pfltSource = (float*)pbSource;
     if( nDstChannelsPerSample>1 )
      { // two channels (usually I+Q) per destination block,
        // which means TWO destination array indices (in pFltDest*) per sample point
        for(i=0; i<nSamplePoints; i++ )
         { *pFltDest1++ = pfltSource[0] * dblGainFactor;    // 1st channel ("I")
           *pFltDest1++ = pfltSource[1] * dblGainFactor;    // 2nd channel ("Q")
           if( (nSrcChannelsPerSample==4) && (pFltDest2!=NULL) )
            { *pFltDest2++ = pfltSource[2] * dblGainFactor; // 3rd channel ("I")
              *pFltDest2++ = pfltSource[2] * dblGainFactor; // 4th channel ("Q")
            }
           pfltSource += nSrcChannelsPerSample;
         }
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      { // (in other words, ONE destination array index per sample point)
        if( (nDstChannelsPerSample<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<nSamplePoints; i++ )
            { *pFltDest1++ = pfltSource[0] * dblGainFactor;  // one DESTINATION channel
              pfltSource += nSrcChannelsPerSample;
            }
         }
        else // (ChunkFormat.wChannels>=2) && (pFltDest2!=NULL) :
         {  // TWO channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<nSamplePoints; i++ )
            { *pFltDest1++ = pfltSource[0] * dblGainFactor;  // 1st DESTINATION channel
              *pFltDest2++ = pfltSource[1] * dblGainFactor;  // 2nd DESTINATION channel
              pfltSource += nSrcChannelsPerSample;
            }
         }
      } // end else < not TWO channels per destination block >
   } // end if < source block properly aligned ? >
} // end Conv32BitSamplesToFloat()

/***************************************************************************/
void Conv64BitSamplesToFloat(
        BYTE *pbSource,          // [in] source data, 32-bit floating point, possibly 'n' channels interleaved
        int  nSamplePoints,      // [in] number of SAMPLE POINTS (with possibly more than one channel each)
        int  nSrcChannelsPerSample, // [in] number of channels per sample point IN THE SOURCE
        int  nDstChannelsPerSample, // [in] number of channels per sample point IN THE DESTINATION
        double dblGainFactor,    // [in] linear gain factor, often the reciprocal of the file's full-scale sample value
        float *pFltDest1,      // [out] 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
        float *pFltDest2)      // [out] 2nd destination block (~"RIGHT"), may be NULL (!)
{
  int i;
  double *pdblSource;
  // The source data must be aligned to 8-byte boundaries, thus:
  if( ((int)pbSource&3)==0)
   { pdblSource = (double*)pbSource;
     if( nDstChannelsPerSample>1 )
      { // two channels (usually I+Q) per destination block,
        // which means TWO destination array indices (in pFltDest*) per sample point
        for(i=0; i<nSamplePoints; i++ )
         { *pFltDest1++ = pdblSource[0] * dblGainFactor;    // 1st channel ("I")
           *pFltDest1++ = pdblSource[1] * dblGainFactor;    // 2nd channel ("Q")
           if( (nSrcChannelsPerSample==4) && (pFltDest2!=NULL) )
            { *pFltDest2++ = pdblSource[2] * dblGainFactor; // 3rd channel ("I")
              *pFltDest2++ = pdblSource[2] * dblGainFactor; // 4th channel ("Q")
            }
           pdblSource += nSrcChannelsPerSample;
         }
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      { // (in other words, ONE destination array index per sample point)
        if( (nDstChannelsPerSample<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<nSamplePoints; i++ )
            { *pFltDest1++ = pdblSource[0] * dblGainFactor;  // one DESTINATION channel
              pdblSource += nSrcChannelsPerSample;
            }
         }
        else // (ChunkFormat.wChannels>=2) && (pFltDest2!=NULL) :
         {  // TWO channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<nSamplePoints; i++ )
            { *pFltDest1++ = pdblSource[0] * dblGainFactor;  // 1st DESTINATION channel
              *pFltDest2++ = pdblSource[1] * dblGainFactor;  // 2nd DESTINATION channel
              pdblSource += nSrcChannelsPerSample;
            }
         }
      } // end else < not TWO channels per destination block >
   } // end if < source block properly aligned ? >
} // end Conv64BitSamplesToFloat()


/*------------- Implementation of the former class C_WaveIO -----------------*/
//  (in this "C" variant, the first argument is always a pointer
//   to the T_WaveIO structure, which resembles 'this' in C++)

/***************************************************************************/
void WaveIO_Init(T_WaveIO *pWIO) // formerly C_WaveIO::C_WaveIO()
  /* Constructor of the Wave-IO Class.
   * Initializes all properties etc.
   */
{
   memset( &pWIO->RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
   memset( &pWIO->ChunkHdr,    0 , sizeof(T_WAV_CHUNK_HDR)    );
   memset( &pWIO->ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX) );

   pWIO->hFile = -1;
   pWIO->OpenedForReading = FALSE;
   pWIO->OpenedForWriting = FALSE;
   pWIO->dblStartTime     = 0.0;
   pWIO->dwCurrFilePos    = 0;
   pWIO->dwFileDataSize   = 0;
   pWIO->iBitsPerSample   = 16;         // just a "meaningful default" !
   pWIO->cErrorString[0]  = '\0';

   pWIO->dbl_SampleRate = 11025.0;  // initial guess, only important for RAW files (w/o header)
   pWIO->iSizeOfDataType = 1;       // just a GUESS for reading RAW files,
   pWIO->wRawFileNumberOfChannels = 1;  // should be properly set before opening such files
   pWIO->iNumChannels    = 1;
   pWIO->fMustWriteHeader = FALSE;

} // end of C_WaveIO's constructor


/***************************************************************************/
BOOL WaveIO_InOpenInternal( T_WaveIO *pWIO,
                char * pszFileName,
                BOOL fReOpenForWriting,
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength ) // [in] size of pszExtraInfo
  /* Opens a wave file for READING, or an (existing) file for WRITING .
   * Returns TRUE on success and FALSE on any error .
   */
{
 int i;
 T_WAV_CHUNK_HDR  chunk_hdr;
 // struct ftime f_time;  // PellesC: "error, undefined size ..." ?!
 char *cp;
 BOOL fExtensionIsWav;
 // BOOL fSubtractRecLengthFromTime = FALSE;
 BOOL ok = TRUE;
 BOOL found_data_chunk;

  WaveIO_CloseFile( pWIO );   // Close "old" file if it was open

  // Look at the file extension..
  cp = strrchr(pszFileName, '.');
  if(cp)
#    ifdef __BORLANDC__
       fExtensionIsWav = (strnicmp(cp,".wav",4)==0);
#    else  // PellesC didn't know strnicmp.. sigh...
       fExtensionIsWav = (_strnicmp(cp,".wav",4)==0);
#    endif
  else fExtensionIsWav = FALSE;

  // Erase old chunks (but not the info prepared in WaveIO_OutOpen) :
  memset( &pWIO->RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
  memset( &pWIO->ChunkHdr,    0 , sizeof(T_WAV_CHUNK_HDR)    );
  // No-No: memset( &pWIO->ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX) );
  // No-No: memset( pWIO->sz255FileName,0 , 255);  // <-- pszFileName may be exactly THIS !

  if( fReOpenForWriting )
   {
      pWIO->hFile = _open( pszFileName, _O_RDWR | _O_BINARY );
      // note that 'zero' is a perfectly valid file handle (unlike other handles which are in fact POINTERS) !
      ok = (pWIO->hFile>=0);
   }
  else  // open for READING (only) :
   {
      pWIO->hFile = _open( pszFileName, _O_RDONLY  | _O_BINARY );
      ok = (pWIO->hFile>=0);
      // From Borland's help system, on _rtl_open() :
      //  > Hinweis:	Diese Funktion ersetzt die Funktion _open, die nicht mehr verwendet wird.
      // From Microsoft:
      //  > // Note: _open is deprecated; consider using _sopen_s instead.
      // Maybe Borland, or Micro$oft don't want us to use _open() anymore. BUT:
      // In PellesC, there are no funny-named functions like _rtl_open & Co anywhere.
      //
   }
  if(ok)
   {
    strncpy(pWIO->sz255FileName, pszFileName, 255);

    // analyse the header of the WAV-file (which if POSSIBLY A RIFF-format)
    i = _read(pWIO->hFile, &pWIO->RiffHdr, sizeof(T_WAV_RIFF_HDR) );
    if( i==sizeof(T_WAV_RIFF_HDR) )
     { // reading the RIFF header successful:
      if(strncmp(pWIO->RiffHdr.id_riff, "RIFF", 4) == 0)
        {
           // if the audio file has a RIFF header, the RIFF-ID should be "WAVE", nothing else:
           if(strncmp(pWIO->RiffHdr.wave_id, "WAVE", 4) != 0)
            { strcpy(pWIO->cErrorString,"no WAVE id");
              ok = FALSE;
            }
         }
     }
   } // end if <file opened successfully>
  else
   {
     if( fReOpenForWriting ) // this is NOT an error when 'trying to re-open',
      { // because when WaveIO_InOpenInternal() is used for append / check-if-already-exists,
        // a non-existing file just means "create a new one" which is NOT an error.
      }
     else // tried to open for READ-ONLY, so getting here is an error:
      {
        strcpy(pWIO->cErrorString,"_open failed");
      }
     // ex: ok = FALSE;
     return FALSE;
   }


  // For RIFF WAVE files (*.wav), analyse all CHUNKS, until the first "data" chunk has been found
  if( ok )
   {
     memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
     found_data_chunk = FALSE;
     while( (ok) && (! found_data_chunk) )
      {
        // Read the next chunk HEADER from the input file:
        i = _read(pWIO->hFile, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
        // 2008-04-26: Got here with chunk_hdr.id="fmt", chunk_hdr.dwChunkSize=18 ,
        //   with a 32kHz stereo wave file saved by "sndrec32" from windoze XP .
        //
        if( i!=sizeof(T_WAV_CHUNK_HDR) )
         { strcpy(pWIO->cErrorString,"bad header size");
           ok = FALSE;
         }
        else // reading the chunk header was ok. Analyse it:
         {
           i = 0; // number of bytes read from a "known" chunk.
           if(   (strncmp(chunk_hdr.id, "fmt ", 4) == 0)
              && (chunk_hdr.dwChunkSize >= WAVEIO_SIZEOF_STANDARD_WAVEFORMAT )
              && (chunk_hdr.dwChunkSize <= sizeof(T_WAV_CHUNK_FORMAT_EX) )
             )
            { // whow, it's *AT LEAST* a standard T_WAV_CHUNK_FORMAT - structure,
              //   read a T_WAV_CHUNK_FORMAT_xyz from the input file .
              // BUT: The 'fmt ' chunk may be a simple 'WAVEFORMAT' (wFormatTag=1)
              //      or (especially for 24-bit samples( a 'WAVEFORMATEXTENSIBLE' (wFormatTag=65534) !
              i = _read(pWIO->hFile, &pWIO->ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
              if ( (DWORD)i != chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ )
               { ok = FALSE;
               }
              // Note that the actual chunk size may differ from what we
              // just read, even though it's a "fmt "-chunk !!
            }
           else if (strncmp(chunk_hdr.id, "data", 4) == 0)
            { // bingo, it's the one and only DATA-Chunk:
              pWIO->dwFileDataSize = chunk_hdr.dwChunkSize;  // netto size of the samples in BYTES
              found_data_chunk = TRUE;
              // 2014-03-10 : Don't count on a valid entry in 'dwChunkSize' !
              //              If a program was terminated abnormally,
              //              it may not have updated this header entry
              //              but that doesn't mean the data in the recording
              //              are useless !
              //  Other programs don't even bother to write a proper 'chunk size',
              //  they simply SEQUENTIALLY write header, then the samples, then close the file,
              //  leaving chunk_hdr.dwChunkSize = zero .
              //              Thus, since 2014-03-10 :
              //              If, while reading more samples from the 'data' chunk,
              //              we will silently increase pWIO->dwFileDataSize on the go,
              //              so that when the end-of-file is reached,
              //              we can REWIND it properly in WaveIO_WindToSampleIndex().
            }
           else if (strncmp(chunk_hdr.id, "inf1", 4) == 0)
            { // it's the prorietary 'Info Chunk, type 1' (since 2010-07) :
              if( (pszExtraInfo!=NULL) && ((DWORD)iExtraInfoLength>chunk_hdr.dwChunkSize) )
               { // ok, the caller's string buffer is large enough. Copy:
                 i = _read(pWIO->hFile, pszExtraInfo, chunk_hdr.dwChunkSize );
                 pszExtraInfo[chunk_hdr.dwChunkSize] = '\0'; // ALWAYS keep C-strings terminated !
                 // WaveIO.cpp doesn't care about the contents of this string,
                 // but AudioFileIO.cpp will parse some of the contents .
               }
            }
           else // unknown chunk type, ignore it, it's ok !
            {
              i = 0;  // no "extra" bytes read from unknown chunk. Just Skip it (below).
              // 2011-02-09: got here with a chunk type "PAD " in a file
              //   recorded with a ZOOM H-1, 96 kSamples/sec, 24 bit/s, 2 channels,
              //   which could NOT be properly read by SL .
              // 2014-02-20: got here with chunk_hdr.id = "fact" whatever that was supposed to be.
            }
           // Skip the chunk, no matter if it was unknown...
           //  (except for the "data" chunk)
           // Reason: also the known chunks may be LONGER
           // in the file than we expected in our structure !
           // For example, the size of the "fmt" header may be indicated
           //  as 16, 18, 40 bytes, or whatever, but we only read a maximum of
           //  40 bytes above (in QFile_Read), so we must possibly SKIP
           //  a few bytes in the seek-command below :
           if( (!found_data_chunk) && ok && ( chunk_hdr.dwChunkSize > (DWORD)i) )
            {
             if (_lseek( pWIO->hFile, chunk_hdr.dwChunkSize-i, SEEK_CUR ) < 0)
              { strcpy(pWIO->cErrorString,"lseek failed");
                ok = FALSE;  // seeking the end of the unknown chunk failed !
              }
              // Note: PellesC knew 'lseek', but the linker complained about missing '_lseek'. Eeek.
              //       Replaced lseek by _lseek and the problem was gone. Same for 'tell'. Hell.
            }
         } // end if <reading a chunk header ok>
      } // end while <ok AND "data"-chunk not found>

     if (!found_data_chunk) ok=FALSE;
     /* Arrived here with ok=TRUE, we should have skipped all chunks. */
     /* Test the "fmt "-chunk's contents if the format is supported:  */
     if (ok)
      {
        if( (pWIO->ChunkFormat.wFormatTag != 1)     // standard Microsoft PCM
          &&(pWIO->ChunkFormat.wFormatTag != 65534) // 'WAVE_FORMAT_EXTENSIBLE'
          )
         {
          sprintf(pWIO->cErrorString,"bad FormatTag:%d",(int)pWIO->ChunkFormat.wFormatTag );
          ok = FALSE;    // Format category, only 1=Microsoft PCM supported
         }
        if( (pWIO->ChunkFormat.wChannels<1) || (pWIO->ChunkFormat.wChannels>8) )
         {
          sprintf(pWIO->cErrorString,"bad NrChannels:%d",(int)pWIO->ChunkFormat.wChannels );
          ok = FALSE;    // Number of channels, 1=mono, 2=stereo
         }
        if(   (pWIO->ChunkFormat.dwSamplesPerSec < 1)
           || (pWIO->ChunkFormat.dwSamplesPerSec > AUDIO_FILE_MAX_SAMPLE_RATE) ) // -> c:\cbproj\SoundUtl\AudioFileDefs.h
         {
          sprintf(pWIO->cErrorString,"bad SamplesRate:%d",(int)pWIO->ChunkFormat.dwSamplesPerSec);
          ok = FALSE;    // Sampling rate not supported
         }
        else
         { // at least we know the 'nominal' sample rate..
           pWIO->dbl_SampleRate = (double)pWIO->ChunkFormat.dwSamplesPerSec;
         }
        if( (pWIO->ChunkFormat.wBlockAlign < 1) || (pWIO->ChunkFormat.wBlockAlign > 16) )
         {
          sprintf(pWIO->cErrorString,"bad BlockAlign:%d",(int)pWIO->ChunkFormat.wBlockAlign );
          ok = FALSE;    // Data block size ?? Bytes per Sample ???
         }
        if(  (pWIO->ChunkFormat.wBitsPerSample==8)   // can we handle this ?...
          || (pWIO->ChunkFormat.wBitsPerSample==16)
          || (pWIO->ChunkFormat.wBitsPerSample==24)  // supported since 2010-05-27
          || (pWIO->ChunkFormat.wBitsPerSample==32) )
         {
           pWIO->iSizeOfDataType = pWIO->ChunkFormat.wBitsPerSample / 8;
           pWIO->iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I; // initial guess...
           if( pWIO->ChunkFormat.wBitsPerSample==8 )
            { // "PCM" wave audio files with EIGHT BITS PER (single) SAMPLE
              // are an exception: they use UNSIGNED integers (range 0..255),
              // not SIGNED integers as one would except (-128...+127) !
              pWIO->iDataTypeFlags = AUDIO_FILE_DATA_TYPE_UNSIGNED;
            }
         }
        else  // neither 8, 16, 24, or 32 bits/sample -> too exotic for me :o)
         {
          sprintf(pWIO->cErrorString,"bad BitsPerSample:%d",(int)pWIO->ChunkFormat.wBitsPerSample );
          ok = FALSE;    // Sample size, usually 8 or 16 bits per sample
         }
        if( (pWIO->ChunkFormat.wBitsPerSample==16) && (pWIO->ChunkFormat.wBlockAlign==1) )
         {  // 16 bits per sample and "block_align=1" is impossible (see formula)
            // but it occurred in a sample from G4JNT. Fix that bug HERE:
            pWIO->ChunkFormat.wBlockAlign = (WORD)
               (pWIO->ChunkFormat.wChannels * (pWIO->ChunkFormat.wBitsPerSample+7) / 8);
         }
      } // end if <ok to check the "fmt "-chunk ?>

     if (ok)
      {
        // save file-internal position of 1st audio sample
        // (just in case someone wants to "Rewind" etc).
        pWIO->dwFilePos_Data = _tell( pWIO->hFile );
          // caution: tell returns a "SIGNED long" value, which is stupid
          //           because disk files may be up to 4 GBytes !
        if( pWIO->dwFilePos_Data == 0xFFFFFFFF )
         { strcpy(pWIO->cErrorString,"ftell failed");
           ok = FALSE;
         }
        pWIO->dwCurrFilePos  = pWIO->dwFilePos_Data;
        // Note: This is not the position (file offset) of the "data chunk",
        //       but the position of the FIRST SAMPLE in the data chunk !
        if( fReOpenForWriting )
         { // if the file is 're-opened' for writing, set the current writing position
           // to the END of the file:
           pWIO->dwCurrFilePos = _lseek( pWIO->hFile, 0, SEEK_END ); // set current position to the file's end
         } // end if < re-opened for writing >
      } // end if < header analysis 'ok' >

   } // end if < header ok >

  if (!ok)
   {
    if(pWIO->cErrorString[0]=='\0')
      strcpy(pWIO->cErrorString,"other 'open' error");
    WaveIO_CloseFile( pWIO );  // close input file if opened but bad header
    return FALSE;
   }
  else
   {
    pWIO->OpenedForReading = ok;
    return TRUE;
   }
} // WaveIO_InOpenInternal(..)

/***************************************************************************/
BOOL WaveIO_InOpen( T_WaveIO *pWIO,
                char *pszFileName,
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength ) // [in] size of pszExtraInfo
  /* Opens a wave file for READING .
   * Returns TRUE on success and FALSE on any error .
   */
{
  return WaveIO_InOpenInternal( pWIO, pszFileName, FALSE/*!fReOpenForWriting*/,
                         pszExtraInfo, iExtraInfoLength );
} // WaveIO_InOpen(..)


/************************************************************************/
DWORD WaveIO_GetTotalCountOfSamples(T_WaveIO *pWIO)
  /* Returns the total number of sampling points in the file .
   */
{
  if( pWIO->ChunkFormat.wBlockAlign>0 )
   { // subtract recording duration from "recording start time"
     // (because up to now, only the DOS FILE TIME is known)
     return pWIO->dwFileDataSize/*bytes*/ / pWIO->ChunkFormat.wBlockAlign;
   } // end if( fSubtractRecLengthFromTime )
  else
     return 0;
} // WaveIO_GetTotalCountOfSamples()


/***************************************************************************/
DWORD WaveIO_GetCurrentSampleIndex(T_WaveIO *pWIO)
  /* Reads the current sample index that will be used on the next
   * READ- or WRITE- access to the audio samples in the opened file.
   * Returns a negative value on any error.
   *  Caution, since 2008-03 we need to play with wave files > 2 GByte
   *  so use DWORD, not "long" as the file offset .
   * Return value : sample index; 0xFFFFFFFF = ERROR ("4 Gig minus one")
   */
{
 DWORD dw = 0xFFFFFFFF;
 if ( (pWIO->hFile>=0) && (pWIO->ChunkFormat.wBlockAlign>0) )
  {
   // ex: dw = tell( pWIO->FileHandle );  // check this : will "tell" work beyond 2 GByte ?
   // better use our local copy of the "current file position", it's faster :
   dw = pWIO->dwCurrFilePos;
   if( dw >= pWIO->dwFilePos_Data )
       dw -= pWIO->dwFilePos_Data;
   else dw = 0;
   if(pWIO->ChunkFormat.wBlockAlign>0)
      dw /= (DWORD)pWIO->ChunkFormat.wBlockAlign;
  }
 return dw;
} // WaveIO_GetCurrentSampleIndex()


/***************************************************************************/
BOOL WaveIO_WindToSampleIndex(T_WaveIO *pWIO, DWORD dwNewSampleIndex)
  /* Sets the current sample index that will be used on the next
   * READ- or WRITE- access to the audio samples an opened file.
   * Returns TRUE if ok ,  otherwise FALSE.
   * As long as you don't use this routine/method,
   * the audio file will be played in a strictly "sequential" way.
   */
{
 DWORD dw = 0xFFFFFFFF;
 if (   (pWIO->ChunkFormat.wBlockAlign>0) && (pWIO->hFile >= 0) )
  {
   dw = dwNewSampleIndex * pWIO->ChunkFormat.wBlockAlign;
   if ( dw<pWIO->dwFileDataSize )
    {
      pWIO->dwCurrFilePos = pWIO->dwFilePos_Data + dw;
      return( _lseek( pWIO->hFile, pWIO->dwCurrFilePos , SEEK_SET ) > 0);
    }
  }
 return FALSE;
} // WaveIO_WindToSampleIndex()



/***************************************************************************/
LONG WaveIO_ReadSamples( T_WaveIO *pWIO, BYTE* pData, LONG Length )
  /* Reads some samples from a wave file which has been opened for READING,
   *            !!! WITHOUT !!! conversion of the sample format in memory .
   * Returns the NUMBER OF AUDIO SAMPLE POINTS if successful,
   *     (not the number of bytes; one sample point may contain up to 8 bytes)
   *   or a negative value if errors occurred.
   * Notes:
   *  - "Length" is the size of the caller's buffer capacity in BYTES.
   *     The actual number of AUDIO SAMPLES will depend on the sample
   *     format. For 16-bit mono recordings, the number of samples
   *     will only be half the "Length".
   *     Generally, the   number of audio samples read   is:
   *          n_samples = Length / pWIO->ChunkFormat.wBlockAlign .
   *  - since 11/2005, up to FOUR channels with SIXTEEN BITS each are possible
   *          ( pWIO->ChunkFormat.wBlockAlign = 16 ) .
   */
{
 int  i;
 long i32;


 if ( pWIO->OpenedForReading && (pWIO->ChunkFormat.wBlockAlign>0) && (pWIO->hFile >= 0)
    )
  {
   i = _read(pWIO->hFile, pData, Length);
   // 2014-02-20: got here with i=511(!), Length=512, pWIO->dwCurrFilePos=58
   //             at the begin of a long wave audio file.
   //             How can this be ? Punishment because we didn't use _rtl_read() ?
   //         Missed something in _open() to open the damned file in BINARY mode !
   // Added 2014-03-10 : Reached the end but .dwFileDataSize not set yet ?
   if( i>=0 && i<Length )  // obviously reached the END OF THE "DATA" CHUNK :
    { if( pWIO->dwFileDataSize == 0 )  // .. but the indicated length was ZERO..
       { i32 = pWIO->dwCurrFilePos + i - pWIO->dwFilePos_Data;
         if( i32>0 )
          { pWIO->dwFileDataSize = (DWORD)i32;
          }
       }
    }
   if (i>0)
    {
      pWIO->dwCurrFilePos += i;

      return i / pWIO->ChunkFormat.wBlockAlign;
    }
  }
 return -1;
} // WaveIO_ReadSamples(..)

/***************************************************************************/
LONG WaveIO_ReadSampleBlocks( T_WaveIO *pWIO, // preferred API (used by Spectrum Lab)
     int channel_nr,    // channel_nr for 1st destination block
     int nr_samples,    // number of samples (or, more precisely, length of destination blocks:  pFltDest1[0..nr_samples-1] )
     float *pFltDest1, // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
     float *pFltDest2, // 2nd destination block (~"RIGHT"), may be NULL (!)
     char *pszExtraInfo, int iMaxStrlen ) // [out,optional] extra string with user data for wave.rd_info
  /* Reads some samples from a wave file and converts them to numbers (*)
   *                    in the range of +- 1.0 (not +/- 32767 anymore !).
   * Input parameters:
   *   channel_nr: 0= begin with the first channel (LEFT if stereo recording)
   *               1= begin with the second channel (RIGHT if stereo, or Q if complex)
   *     The UPPER BYTE of 'channel_nr' can optionally define
   *         HOW MANY CHANNELS SHALL BE PLACED IN ONE DESTINATION BLOCK:
   *    (channel_nr>>8) : 0x00 or 0x01 = only ONE channel per destination block
   *                      0x02         = TWO channels per destination block
   *                                    (often used for complex values aka I/Q)
   *
   *   nr_samples: Max number of FLOATs which can be stored in float dest[0..nr_samples-1]
   *               (this is NOT always the same as the number of sampling points,
   *                for example if the output are I/Q samples, all placed in
   *                pFltDest1, there are only nr_samples/2 complex sampling points).
   *   *pFltDestX = Pointers to destination arrays (one block per channel, sometimes "paired")
   * Return value:
   *      the NUMBER OF FLOATING-POINT VALUES placed in each destination block
   *                 (not necessarily the number of sample points!) ,
   *      or a negative value if an error occurred.
   * Note that returning less samples than you may expect is not an error
   * but indicates reaching the file's end !
   *   ( This convension later caused trouble when the input is not a FILE
   *     but a STREAM, see WPH_IO_Plugin.cpp ) .
   * (*) float may be either SINGLE- or DOUBLE-precision float,
   *     defined in some header file, depending on SWI_FLOAT_PRECISION .
   */
{
  BYTE *bp, *temp_array;
  short *sp;  
  int  i;
  int  asize;
  int  n_samples_read;
  int  iResult = -1;
  BOOL ok;
  BOOL fTwoChannelsPerBlock = FALSE;
  float flt;

   fTwoChannelsPerBlock =  ((channel_nr & AUDIO_FILE_TWO_CHANNELS_PER_BLOCK) != 0)
                        && (pWIO->ChunkFormat.wChannels>1);
   channel_nr &= 0x000F;  // remove all 'flags', leave only the start-channel number

   if ( (!pWIO->OpenedForReading)
      || (pWIO->hFile<0)
      || (pWIO->ChunkFormat.wBlockAlign<=0)
      || (pWIO->ChunkFormat.wBitsPerSample <= 1)  )
     return -1;

   if(fTwoChannelsPerBlock)
    { nr_samples /= 2;    // here: TWO destination array indices per sample point
      // so if the caller can accept 4096 floating-point values in his array (pFltDest1),
      // he will only read 2048 sampling points in one call of this subroutine !
    }

   asize = nr_samples * pWIO->ChunkFormat.wBlockAlign/* strange name for "bytes per sample-point"*/;
   if (asize<=0)
     return -1;
   temp_array = malloc(asize);  // allocate temporary array, C style
   if(temp_array==NULL)
     return -1;

   n_samples_read = WaveIO_ReadSamples( pWIO, // Returns the NUMBER OF AUDIO SAMPLE POINTS!
            temp_array, /* temporary BYTE-array for conversion     */
            asize );    /* size in BYTE, not "number of samples" !! */
   if (n_samples_read > nr_samples)
       n_samples_read = nr_samples;  // emergency limit, should never happen
   ok = FALSE;
   if  (pWIO->ChunkFormat.wBitsPerSample <= 8)
    { /* up to 8 bits per sample : use a BYTE-pointer for addressing... */
     ok = TRUE;
     bp = temp_array;
     if( (pWIO->ChunkFormat.wChannels>1)  // Number of channels, 1=mono, 2=stereo
      && (channel_nr < pWIO->ChunkFormat.wChannels) )
      { // channel_nr: 0=left, 1=right, ..
        bp += channel_nr;             // skip 'unwanted' channels
      }
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;   // TWO destination array indices per sample point
        if( pWIO->iDataTypeFlags & AUDIO_FILE_DATA_TYPE_UNSIGNED )
         { // here for UNSIGNED 8-bit samples ( raw value range 0..255 ) :
           for(i=0; i<n_samples_read/*!*/; i++ )
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              flt = (unsigned char)bp[0];
              *(pFltDest1++) = (flt-127.0) * (1.0 / 127.0);  // 1st channel in file ("I")
              flt = (unsigned char)bp[1];
              *(pFltDest1++) = (flt-127.0) * (1.0 / 127.0);  // 2nd channel in file ("Q")
              if( (pFltDest2!=NULL) && (pWIO->ChunkFormat.wChannels>2) )
               { flt = (unsigned char)bp[2];
                 *(pFltDest2++) = (flt-127.0) * (1.0 / 127.0); // 3rd channel in file ("I")
                 flt = (unsigned char)bp[3];
                 *(pFltDest2++) = (flt-127.0) * (1.0 / 127.0); // 4th channel in file ("Q")
               }
              bp += pWIO->ChunkFormat.wBlockAlign; /* "block align" ~~ "bytes per sample point" */
            } // end for < 8-bit SIGNED samples >
         }
        else  // 8-bit SIGNED ( raw value range -128 ... + 127 )  :
         {
           for(i=0; i<n_samples_read/*!*/; i++ )
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              flt = (signed char)bp[0];
              *(pFltDest1++) = flt * (1.0 / 127.0);  // 1st channel in file ("I")
              flt = (signed char)bp[1];
              *(pFltDest1++) = flt * (1.0 / 127.0);  // 2nd channel in file ("Q")
              if( (pFltDest2!=NULL) && (pWIO->ChunkFormat.wChannels>2) )
               { flt = (signed char)bp[2];
                 *(pFltDest2++) = flt * (1.0 / 127.0); // 3rd channel in file ("I")
                 flt = (signed char)bp[3];
                 *(pFltDest2++) = flt * (1.0 / 127.0); // 4th channel in file ("Q")
               }
              bp += pWIO->ChunkFormat.wBlockAlign; /* "block align" ~~ "bytes per sample point" */
            } // end for < 8-bit SIGNED samples >
         }
      }
     else // still 8 bits per value, but not "two channels per output block" ...
      { iResult = n_samples_read;  // ONE destination array index per sample point
        if( (pWIO->ChunkFormat.wChannels<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<n_samples_read; ++i)
            { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
              *pFltDest1++ = ( (float)(*bp) - 127.0 ) * (1.0 / 127.0);
              bp += pWIO->ChunkFormat.wBlockAlign;
            }
         }
        else
         {  // TWO or more channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<n_samples_read; ++i)
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              *pFltDest1++ = ( (float)(*bp)   - 127.0 ) * (1.0 / 127.0);
              *pFltDest2++ = ( (float)(*(bp+1))-127.0 ) * (1.0 / 127.0);
              bp += pWIO->ChunkFormat.wBlockAlign;
            }
         } // end if <two separate destination blocks>
      } // end else < not TWO channels per destination block >
   } // end if (8-bit-samples)
  else
  if( (pWIO->ChunkFormat.wBitsPerSample <= 16) && (pWIO->ChunkFormat.wBlockAlign>1) )
   { /* 9..16 bits per sample, we use a 16bit-int-pointer for addressing. */
     ok = TRUE;
     sp = (short*)temp_array; // should be even address, so what ??
     if( (pWIO->ChunkFormat.wChannels>1) // Number of channels, 1=mono, 2=stereo
      && (channel_nr < pWIO->ChunkFormat.wChannels) )
      { // channel_nr: 0=left, 1=right, ..
        sp += channel_nr;     // skip 'unwanted' channels
      }
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;  // TWO destination array indices per sample point
        switch( pWIO->iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                 |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
              ) )
         {
           case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = (float)( * sp  ) * (1.0/32768.0); // 1st channel ("I")
                *pFltDest1++ = (float)(*(sp+1)) * (1.0/32768.0); // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (pWIO->ChunkFormat.wChannels>2) )
                 { *pFltDest2++ = (float)(*(sp+2)) * (1.0/32768.0); // 3rd channel ("I")
                   *pFltDest2++ = (float)(*(sp+3)) * (1.0/32768.0); // 4th channel ("Q")
                 }
                sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
           case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadUns16Intel(sp);   // 1st channel ("I")
                *pFltDest1++ = ReadUns16Intel(sp+1); // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (pWIO->ChunkFormat.wChannels>2) )
                 { *pFltDest2++ = ReadUns16Intel(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadUns16Intel(sp+3); // 4th channel ("Q")
                 }
                sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
           case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadSigned16Motorola(sp);    // 1st channel ("I")
                *pFltDest1++ = ReadSigned16Motorola(sp+1);  // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (pWIO->ChunkFormat.wChannels>2) )
                 {
                   *pFltDest2++ = ReadSigned16Motorola(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadSigned16Motorola(sp+1); // 4th channel ("Q")
                 }
                sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
           case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
              | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadUns16Motorola(sp);    // 1st channel ("I")
                *pFltDest1++ = ReadUns16Motorola(sp+1);  // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (pWIO->ChunkFormat.wChannels>2) )
                 {
                   *pFltDest2++ = ReadUns16Motorola(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadUns16Motorola(sp+3); // 4th channel ("Q")
                 }
                sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
         } // end switch( pWIO->iDataTypeFlags & ... )
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      {
        iResult = n_samples_read;  // ONE destination array index per sample point
        if( (pWIO->ChunkFormat.wChannels<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           switch( pWIO->iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                   ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                    |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
                 ) )
            {
              case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
               { for(i=0; i<n_samples_read; ++i)
                  { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
                    *pFltDest1++ = (float)(*sp) * (1.0 / 32768.0);
                    sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break;
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Intel(sp);
                   sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
              case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadSigned16Motorola(sp);
                   sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_MOTOROLA (signed)
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
                 | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Motorola(sp);
                   sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED,MOTOROLA
            } // end switch( pWIO->iDataTypeFlags &  ... )
         }
        else
         {  // TWO channels must be copied (and "split" into two destination blocks) :
           switch( pWIO->iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                   ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                    |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
                 ) )
            {
              case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
               { for(i=0; i<n_samples_read; ++i)
                  { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
                    *pFltDest1++ = (float)(  *sp  ) * (1.0 / 32768.0);
                    *pFltDest2++ = (float)(*(sp+1)) * (1.0 / 32768.0);
                    sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Intel(sp);
                   *pFltDest2++ = ReadUns16Intel(sp+1);
                   sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
              case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadSigned16Motorola(sp);
                   *pFltDest2++ = ReadSigned16Motorola(sp+1);
                   sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_MOTOROLA (signed)
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
                 | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Motorola(sp);
                   *pFltDest2++ = ReadUns16Motorola(sp+1);
                   sp += (pWIO->ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED,MOTOROLA
            } // end switch( pWIO->iDataTypeFlags &  ... )
         } // end if <two separate destination blocks>
      } // end else < not TWO channels per destination block >
    } // end if (16-bit-samples)
   else if( pWIO->ChunkFormat.wBitsPerSample == 24 ) // added 2010-05-27 ...
    {   // and modified 2011-02, because some stupid asshole decided
        // to REVERSE the byte order in a 24-bit-per-sample recording.
        // In M1F1-int24-AFsp.wav, considered a 'valid' PCM wave file,
        //   the byte order seemed to be LOW1-MID1-HIGH1 LOW2-MID2-HIGH2
        //   (1 = left channel,   2 = right channel) .
        //
     Conv24BitSamplesToFloat( temp_array, n_samples_read,
                 pWIO->ChunkFormat.wChannels,  // number of channels in the SOURCE
                 fTwoChannelsPerBlock?2:1,
                 pFltDest1,  // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
                 pFltDest2); // 2nd destination block (~"RIGHT"), may be NULL (!)
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;   // TWO destination array indices per sample point
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      {
        iResult = n_samples_read;  // ONE destination array index per sample point
      }
    }
   else if( pWIO->iSizeOfDataType == 4 ) // actually 32-bit FLOATING POINT (?)
    { if( pWIO->iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
       { // must be 32-bit 'IEEE single precision float'; only INTEL byte order is supported here:
         Conv32BitSamplesToFloat( temp_array, n_samples_read,
                 pWIO->ChunkFormat.wChannels,  // number of channels in the SOURCE
                 fTwoChannelsPerBlock?2:1,
                 pWIO->dblGainFactor,
                 pFltDest1,  // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
                 pFltDest2); // 2nd destination block (~"RIGHT"), may be NULL (!)
         if( fTwoChannelsPerBlock )
          { // two channels (usually I+Q) per destination block (since 2005-11-01) :
            iResult = 2*n_samples_read;   // TWO destination array indices per sample point
          }
         else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
          {
            iResult = n_samples_read;  // ONE destination array index per sample point
          }
       } // end if( pWIO->iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
    }
   else if( pWIO->iSizeOfDataType == 8 ) // actually 64-bit FLOATING POINT (?)
    { if( pWIO->iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
       { // must be 64-bit 'IEEE double precision float'; only INTEL byte order is supported here:
         Conv64BitSamplesToFloat( temp_array, n_samples_read,
                 pWIO->ChunkFormat.wChannels,  // number of channels in the SOURCE
                 fTwoChannelsPerBlock?2:1,
                 pWIO->dblGainFactor,
                 pFltDest1,  // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
                 pFltDest2); // 2nd destination block (~"RIGHT"), may be NULL (!)
         if( fTwoChannelsPerBlock )
          { // two channels (usually I+Q) per destination block (since 2005-11-01) :
            iResult = 2*n_samples_read;   // TWO destination array indices per sample point
          }
         else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
          {
            iResult = n_samples_read;  // ONE destination array index per sample point
          }
       } // end if( pWIO->iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
    }
   else
    { // cannot handle this sample format !!
     ok = FALSE;
     iResult = -1;  // error
    }

   free( temp_array );
   return iResult;

} // WaveIO_ReadSampleBlocks(...)

/***************************************************************************/
BOOL WaveIO_WriteHeader( T_WaveIO *pWIO )
  /* Writes the 'header' for the audio file (usually a RIFF WAVE file).
   *  This was once done immediately in OutOpen(), but since 2011-02
   *  writing the header is postponed until the first chunk of samples
   *  is available, and a PRECISE(!) timestamp is present in pChunkInfo .
   *  All relevant information has already been set in OutOpen(),
   *  but not written to the file yet (by the time we get here).
   */
{
  T_WAV_CHUNK_HDR  chunk_hdr;
  BOOL ok = TRUE;
  BOOL fWriteHeader = TRUE;
  // ex: char sz1kInfo[1024];  // only used in the bigger brother, WaveIO.cpp
  int  i;

  // prepare a 'format chunk' ("fmt ") in memory even if not writing to RIFF wave:
  memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
  strncpy(chunk_hdr.id, "fmt ", 4);
  if( (pWIO->iBitsPerSample <= 16) && (pWIO->iNumChannels<=2) )
   { pWIO->ChunkFormat.wFormatTag = 1;     // Format category:  1=Microsoft PCM
     chunk_hdr.dwChunkSize = WAVEIO_SIZEOF_STANDARD_WAVEFORMAT;    /* 16 byte */
     pWIO->ChunkFormat.cbSize = 16;   // doesn't matter, not written to file
   }
  else
   { pWIO->ChunkFormat.wFormatTag = 65534; // Format category:  'WAVE_FORMAT_EXTENSIBLE'
     chunk_hdr.dwChunkSize = WAVEIO_SIZEOF_WAVE_FORMAT_EXTENSIBLE; /* 40 byte */
     // > For the WAVEFORMATEXTENSIBLE structure, the Format.cbSize  field
     // > must be set to 22 and the SubFormat field ("GUID") must be set to
     // > KSDATAFORMAT_SUBTYPE_PCM .
     pWIO->ChunkFormat.cbSize = 22;
   }


  if( pWIO->OpenedForWriting )  // See OutOpen() ...
   {
     // Fill and write the header of the WAV-file:
     memset( &pWIO->RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
     strncpy(pWIO->RiffHdr.id_riff, "RIFF", 4);
     strncpy(pWIO->RiffHdr.wave_id, "WAVE", 4);
     pWIO->RiffHdr.len =  sizeof(T_WAV_RIFF_HDR)
                 + 2*sizeof(T_WAV_CHUNK_HDR)
                 + WAVEIO_SIZEOF_STANDARD_WAVEFORMAT/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/;
      // (the length of the audio samples ("data") is still unknown,
      //  will be ADDED later before closing)
     if( fWriteHeader )
       {
         ok = (_write(pWIO->hFile, &pWIO->RiffHdr, sizeof(T_WAV_RIFF_HDR) ) == sizeof(T_WAV_RIFF_HDR) );
       }
   } // end if <file opened successfully>
  else
   { ok = FALSE;
   }


  // 2011-02-13 : Moved the "inf1" header further below,
  //                    between the "fmt " and the "data" chunk,
  //                    because Linux didn't like those files ...
  // Note: the chunk format thingy has already been init'd in OutOpen() !

  if( (fWriteHeader) && (ok) )
   {
    // Write the T_WAV_CHUNK_FORMAT to the output RIFF wave file:
    i = _write(pWIO->hFile, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
    if (i != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
    i = _write(pWIO->hFile, &pWIO->ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
    if (i != (int)chunk_hdr.dwChunkSize )   ok = FALSE; // file-write-error ?
   }

  if( (fWriteHeader) && (ok) )
   {
    // Fill and write a DATA-chunk to the output file:
    memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
    strncpy(chunk_hdr.id, "data", 4);
    chunk_hdr.dwChunkSize = 0; // still unknown, will be updated before closing.
    i = _write(pWIO->hFile, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
    if (i != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
    // save file-internal position of 1st audio sample, which will also be
    // used to write the correct "data"-chunk size into the header LATER,
    // when closing the file.
    pWIO->dwFilePos_Data = _tell( pWIO->hFile );
    if (pWIO->dwFilePos_Data==0xFFFFFFFF)    // pWIO->dwFilePos_Data should be 44 now
        ok = FALSE;
    pWIO->dwCurrFilePos = pWIO->dwFilePos_Data;
   }

  pWIO->fMustWriteHeader = FALSE;  // "done" (at least, we tried)

  if (!ok)
   {
     WaveIO_CloseFile( pWIO );  // close output file, could not generate a header.
     return FALSE;
   }
  else
   {
     return TRUE;
   }
} // end WaveIO_WriteHeader()

/***************************************************************************/
BOOL WaveIO_OutOpen( T_WaveIO *pWIO,
           char *file_name,
           int file_mode,        // AUDIO_FILE_MODE_OVERWRITE etc
           int iAudioFileOptions, // AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
           int bits_per_sample,  // supported 8, 16, or 24 bits/sample
           int channels,
           double dblSampleRate) // [in] sampling rate in Hz, as precise as possible
  /* Creates and Opens an audio file for WRITING audio samples.
   *         Returns TRUE on success and FALSE on any error.
   * Input parameters..
   *   bits_per_sample:  must be either 8 or 16 (bits for a single channel)
   *   channels:         1=mono, 2=stereo recording etc
   *   sample_rate:      number of samples per second.
   *                     If the output format is AUDIO_FILE_FORMAT_RIFF,
   *                     the sample rate should be above 8000 samples/second.
   */
{
 BOOL ok = TRUE;

  WaveIO_CloseFile( pWIO );   // Close "old" file if it was open


  // round the sample rate to a 32-bit INTEGER value (for the RIFF header)
  pWIO->dbl_SampleRate = dblSampleRate;

  // coarse check of the arguments..
  if( (bits_per_sample != 8) && (bits_per_sample != 16) && (bits_per_sample != 24) )
      return FALSE;
  pWIO->iBitsPerSample = bits_per_sample;
  if( (channels<1) || (channels>2) )
      return FALSE;    // only mono or stereo recording supported
                       // ..sorry for those 'quattrophiles' ;-)
  pWIO->iNumChannels = channels;
  // 2010-05-10: now accepting 10 MSample/sec ... SETI may use this,
  //             of course this will NEVER work for real-time processing !
  if( (pWIO->dbl_SampleRate < 1) || (pWIO->dbl_SampleRate > AUDIO_FILE_MAX_SAMPLE_RATE) )
      return FALSE;

  pWIO->iAudioFileOptions = iAudioFileOptions;
  pWIO->dwCurrFilePos = pWIO->dwFilePos_Data = 0;  // default file position if no header
  pWIO->dwFileDataSize= 0;  // "netto" size of audio samples still zero
  pWIO->iFileMode   = file_mode;  // used in WriteHeader() [postponed since 2011-02]
  strncpy(pWIO->sz255FileName, file_name, 255); // save the filename for the "Get"-function,
                                            // and for the postponed WriteHeader call .

  // prepare a 'format chunk' ("fmt ") in memory even if not writing to RIFF wave:
  memset( &pWIO->ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX) );
  if( (pWIO->iBitsPerSample <= 16) && (pWIO->iNumChannels<=2) )
   { pWIO->ChunkFormat.wFormatTag = 1;  // Format category:  1=Microsoft PCM
     pWIO->ChunkFormat.cbSize = 16;     // doesn't matter, not written to file
   }
  else
   { pWIO->ChunkFormat.wFormatTag = 65534; // Format category:  'WAVE_FORMAT_EXTENSIBLE'
     // > For the WAVEFORMATEXTENSIBLE structure, the Format.cbSize  field
     // > must be set to 22 and the SubFormat field ("GUID") must be set to
     // > KSDATAFORMAT_SUBTYPE_PCM .
     pWIO->ChunkFormat.cbSize = 22;
   }
  pWIO->ChunkFormat.wChannels  = (WORD)pWIO->iNumChannels; //  1=mono, 2=stereo
  pWIO->ChunkFormat.dwSamplesPerSec     // Sampling rate     , samples per second
                 = (long)(pWIO->dbl_SampleRate+0.5);   // ROUNDED TO 32bit-INTEGER VALUE!
    // ToDo: Should we use the 'nominal' sampling rate here, even if we know better ?
    // Some audio processing software may not be comfortable with 47999 samples/second.
  pWIO->ChunkFormat.wBitsPerSample    // Sample size, usually 8 or 16 bits per sample
                 = (WORD)pWIO->iBitsPerSample;
  pWIO->ChunkFormat.wBlockAlign       // Data block size , Bytes per Sample :
                 = (WORD)( (long)pWIO->ChunkFormat.wChannels
                 *  ( ( pWIO->ChunkFormat.wBitsPerSample + 7 ) / 8 ) );
  pWIO->ChunkFormat.dwAvgBytesPerSec  // For buffer estimation
                 = pWIO->ChunkFormat.wBlockAlign
                 * pWIO->ChunkFormat.dwSamplesPerSec;
  // The above was the mandatory 16-byte 'standard WAVEFORMAT' part .
  // Next: Fill the (optional) rest of the WAVE_FORMAT_EXTENSIBLE in memory,
  //       regardless of chunk_hdr.dwChunkSize :
  pWIO->ChunkFormat.Samples.wValidBitsPerSample = pWIO->ChunkFormat.wBitsPerSample;
  memcpy( pWIO->ChunkFormat.b16GUID, WaveIO_b16GUID_PCM, 16 ); // GUIDo-droid ??


  //
  // Note: Since 2011-02, writing of the WAVE FILE HEADERS is postponed
  //       until the first call of WriteSamples(), called through AudioFileIO.cpp (!)
  //   But we already CREATE the file here, to avoid wasting time later.
  // Create new file, truncate existing file to length "zero", or APPEND data ?
  pWIO->fMustWriteHeader = TRUE;
  if( pWIO->iFileMode == AUDIO_FILE_MODE_APPEND )
   { // if the file already exists, the new data shall be APPENDED :
     if( WaveIO_InOpenInternal( pWIO, pWIO->sz255FileName, TRUE/*fReOpenForWriting*/,
                         NULL,0/*don't care for the old info*/) )
      { pWIO->fMustWriteHeader = FALSE;  // do NOT write an "all-new" header in this case !
      } // end if < "APPEND"-mode, and file successfully opened >
     else
      { pWIO->dwFileDataSize= 0;  // tried to append, but file doesn't exist yet -> CREATE it further below
      }
   }
  if(pWIO->hFile<=0)
   { 
#   ifdef __BORLANDC__
     _fmode = O_BINARY;  // anachronism from the VERY dark ages. Affects _creat() !
     // PellesC didn't know _fmode. Nnnggrrrrrrr.
#   endif
     // Borland's help file suggests S_IREAD | S_IWRITE, defined in sys/stat.h .
     // Without these parameters from the umpteenth header-file, the file
     // _creat()-ed below was WRITE-PROTECTED after it was closed.  Holy s..t.
     pWIO->hFile = _creat( pWIO->sz255FileName, S_IREAD | S_IWRITE);
     // note that 'zero' is a valid FILE HANDLE !!
     pWIO->fMustWriteHeader = TRUE;
   }
  ok = (pWIO->hFile>=0);

  pWIO->OpenedForWriting = ok;  // required for some polling functions,
                            // "it's ok to call WriteSamples_Float() now" .

  return TRUE;   // not really sure if we will be able to CREATE the file...

} // WaveIO_OutOpen(..)


/***************************************************************************/
LONG WaveIO_WriteSamples( T_WaveIO *pWIO, BYTE* pData, LONG Length/*nBytes*/ )
  /* Writes some samples in 'raw' form (without any conversion)
   *       to an audio file which has been opened for WRITING.
   * Returns the number of SAMPLES written,
   *       or a NEGATIVE value if there was an error.
   * Note: "Length" is the size of the caller's buffer capacity in BYTES.
   *       The actual number of AUDIO SAMPLES will depend on the sample
   *       format.
   *       For 16-bit mono recordings (which is the only supported format
   *       up 'til now), the number of samples will only be half the "Length"
   *       etc.
   */
{
  int n_written=0;

  /* Ensure that it's really possible to WRITE audio samples to a file now.
   */
   if ( (!pWIO->OpenedForWriting)
      || (pWIO->ChunkFormat.wBlockAlign<=0)
      || (pWIO->ChunkFormat.wBitsPerSample <= 1)
      || (Length <= 0)
      || (pWIO->hFile < 0)
        )
     return -1;


  // If not done yet, write the WAVE HEADERS now (before the samples):
  if( pWIO->fMustWriteHeader )
   { if( ! WaveIO_WriteHeader( pWIO ) )
      { WaveIO_CloseFile( pWIO );
        return -1;
      }
     pWIO->fMustWriteHeader = FALSE;
   }


  /* Try to append the new samples to the wave file.
   * Note that "WindToSampleIndex()" does not work on OUTPUT files yet !!!
   */
   n_written = _write(pWIO->hFile, pData, Length );
   if (n_written != Length)
    {
     /* Writing to the output file failed  !!!
      * Seems we're in trouble now, because the reason may be a full
      * harddisk and Windooze is going to crash in a few moments :-(
      */
     WaveIO_CloseFile( pWIO );
     return -1;  // also set a breakpoint here !
    }

  /* Keep the "current file position" updated so we don't have to call
   *    the "tell()"-function too often.
   * This parameter will also be READ from the spectrum analyser main window
   * to display the actual file size.
   */
  pWIO->dwCurrFilePos += n_written;
  if(pWIO->dwCurrFilePos-pWIO->dwFilePos_Data >= pWIO->dwFileDataSize)
     pWIO->dwFileDataSize = pWIO->dwCurrFilePos-pWIO->dwFilePos_Data;
  return n_written / pWIO->ChunkFormat.wBlockAlign;
} // end WaveIO_WriteSamples(..)

/***************************************************************************/
LONG WaveIO_WriteSamples_Float( // app-layer output, using NORMALIZED floats
       T_WaveIO *pWIO,
       float *pfltSrc,    // [in] caller's source buffer, 32-bit floats
       int nSamplePoints,            // [in] number of sample points
       int nSourceChannelsPerSample) // [in] one or two floats per sample point ?
  /* Writes some samples in the floating point format (usually normalized to -1.0 ... +1.0)
   *       to an audio file which has been opened for WRITING.
   *    This is the preferred method for Spectrum Lab since 2011 (better suited to 24-bit precision).
   *    The actually used 'peak' value range should be specified in pChunkInfo->dblSampleValueRange.
   *
   * Returns the number of SAMPLE POINTS written,
   *       or a NEGATIVE value if there was an error.
   */
{
  int iChannel, nChannels, nBytesToSend, nDestChannelsPerSample, iRest;
  float flt, fltScalingFactor;
  union { BYTE b[4];  long i32; } t;
  LONG nSamplesWritten = 0;
  BYTE *pbTempBuffer, *pb;
  SHORT *pI16;

  nDestChannelsPerSample   = pWIO->ChunkFormat.wChannels;
  nChannels = (nSourceChannelsPerSample < nDestChannelsPerSample)
             ? nSourceChannelsPerSample : nDestChannelsPerSample;

  /* Make sure that it's really possible to WRITE audio samples to a file now.
   */
   if ( (!pWIO->OpenedForWriting)
      || (nSamplePoints <= 0 )
      || (nSourceChannelsPerSample <= 0)
      || (pWIO->ChunkFormat.wBlockAlign<= 0)
      || (pWIO->ChunkFormat.wBitsPerSample <= 1)
      || (pWIO->hFile < 0)
        )
    { return -1;
    }

  // Allocate a temporary buffer, in which the 8-, 16-, or 24-bit values
  // can be built, before flushing them to the wave file :
  nBytesToSend = (DWORD)pWIO->ChunkFormat.wBlockAlign * nSamplePoints;
  pbTempBuffer = malloc( nBytesToSend );
  if( pbTempBuffer == NULL )
   { sprintf(pWIO->cErrorString,"out of memory in WriteSamples");
     return -1;
   }
  // NO LAZY RETURN AFTER THIS POINT ! Don't forget to free that buffer....
  memset( pbTempBuffer, 0, nBytesToSend );
  pb = pbTempBuffer;            // buffer used for 8-bit unsigned integer
  pI16 = (SHORT*)pbTempBuffer;  // same buffer used for 16-bit integer

  // For best speed, use DIFFERENT LOOPS for each of the supported
  // "bits per sample" (strictly: bits per integer in each channel of a sample point)
  switch(pWIO->ChunkFormat.wBitsPerSample)  // which sample-resolution, 8, 16 or 24 bit ?
   {
     case  8:   // input: normalized floats (-1.0 .... +1.0) ,
                // output: UNSIGNED 8-bit   (  0  .... 255 ) .
        fltScalingFactor = 127.0;
        nSamplesWritten = nSamplePoints;
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for UNSIGNED 8-bit samples ( output value range 0..255 ) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // up to 8 bits per sample:
              flt = 127.0 + pfltSrc[iChannel] * fltScalingFactor;
              if( flt< 0.0  ) flt= 0.0;
              if( flt> 255.0) flt= 255.0;
              pb[iChannel] = (BYTE)flt;
            } // end for < 8-bit UNSIGNED samples >
           pb      += nDestChannelsPerSample;   // next sample group in destination (unused channels already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // next sample group in SOURCE
         }
        break;

     case 16:   // input: normalized floats (-1.0   .... +1.0) ,
                // output: SIGNED 16-bit    (-32767 .... +32767 ) .
        fltScalingFactor = 32767.0;
        nSamplesWritten = nSamplePoints;
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for UNSIGNED 8-bit samples ( output value range 0..255 ) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // scale and copy 16 bits per sample:
              flt = pfltSrc[iChannel] * fltScalingFactor;
              if( flt<-32767.0 ) flt=-32767.0;
              if( flt> 32767.0 ) flt= 32767.0;
              pI16[iChannel] = (SHORT)flt;
            } // end for < 16-bit SIGNED samples >
           pI16 += nDestChannelsPerSample; // next sample group in destination (unused channels already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // next sample group in SOURCE
         }
        break;

     case 24:   // input: normalized floats (-1.0   .... +1.0) ,
                // output: SIGNED 24-bit    (-(2^23-1).. 2^23-1 ) .
        fltScalingFactor = 8388607.0/* 2^23 - 1 */;
        nSamplesWritten = nSamplePoints;
        iRest = nDestChannelsPerSample - nSourceChannelsPerSample;
        if( iRest<0 )  iRest=0;
        iRest *= 3; // ... bytes per sample for ONE channel
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for UNSIGNED 8-bit samples ( output value range 0..255 ) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // scale and copy 24 bits per sample:
              flt = pfltSrc[iChannel] * fltScalingFactor;
              if( flt<-8388607.0 ) flt=-8388607.0;
              if( flt> 8388607.0 ) flt= 8388607.0;
              t.i32 = (long)flt;
              *pb++ = t.b[0];   // least significant byte (bits  7 ... 0)
              *pb++ = t.b[1];   // middle byte            (bits 15 ... 8)
              *pb++ = t.b[2];   // most significant byte  (bits 23 ...16)
            } // end for < 16-bit SIGNED samples >
           pb += iRest; // skip unused channels in destination (already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // inc for next sample in the SOURCE block
         } // end while
        // Arrived here, pb (pointer into destination buffer) should point here:
        if( pb != (pbTempBuffer + nBytesToSend ) )
         { iRest = pb-pbTempBuffer;  // <<< set breakpoint here !
         }
        break;

     default:  // non supported sample resolution ...
        sprintf(pWIO->cErrorString,"bad BitsPerSample:%d",(int)pWIO->ChunkFormat.wBitsPerSample );
        nSamplesWritten = -1;  // this is a 'permanent' error
        nBytesToSend = 0;
        break;
   } // end switch(pWIO->ChunkFormat.wBitsPerSample)

  if( nBytesToSend>0 )
   { WaveIO_WriteSamples( pWIO, pbTempBuffer, nBytesToSend );
   }

  free( pbTempBuffer );
  return nSamplesWritten;

} // end WaveIO_WriteSamples_Float(...)


/************************************************************************/
DWORD WaveIO_GetCurrentFileSize(T_WaveIO *pWIO)
  /* Returns the current file size (if there is an opened WAVE file).
   * Useful especially when logging audio samples  to keep track
   * of the used disk size (stop before Windows is getting into trouble).
   */
{
 if (pWIO->OpenedForWriting || pWIO->OpenedForReading)
       return pWIO->dwCurrFilePos;
  else return 0L;
} // end WaveIO_GetCurrentFileSize()

double WaveIO_GetSampleRate(T_WaveIO *pWIO)  // this is the FILE's sampling rate !
{ return pWIO->dbl_SampleRate;
}
void   WaveIO_SetSampleRate(T_WaveIO *pWIO, double dblSampleRate)
{ pWIO->dbl_SampleRate = dblSampleRate;
}

//---------------------------------------------------------------------------
int    WaveIO_GetNrChannels(T_WaveIO *pWIO)
{ return pWIO->ChunkFormat.wChannels;
}

//---------------------------------------------------------------------------
void   WaveIO_SetNrChannels(T_WaveIO *pWIO, int iNrChannels)
{ // Only applies to RAW FILES !  For RIFF WAVE files,
  //  the number of channels will be read from the file header .
  pWIO->wRawFileNumberOfChannels = (WORD)iNrChannels;
}

//---------------------------------------------------------------------------
int    WaveIO_GetBitsPerSample(T_WaveIO *pWIO)
{ return pWIO->ChunkFormat.wBitsPerSample;
}


//---------------------------------------------------------------------------
double WaveIO_GetCurrentRecordingTime( T_WaveIO *pWIO )
{ double d;   // no "float" here, we need 64 bit floats to encode the time !
  d = (double)WaveIO_GetCurrentSampleIndex(pWIO);
  if(pWIO->dbl_SampleRate>0)
     d /= pWIO->dbl_SampleRate;
  return d;    // **NOT** "+ Start Time" ! ! !
}
double WaveIO_GetRecordingStartTime(T_WaveIO *pWIO)
{ return pWIO->dblStartTime; // seconds elapsed since 00:00:00 GMT, January 1, 1970.
}
void   WaveIO_SetRecordingStartTime(T_WaveIO *pWIO, double dblStartTime)
{ pWIO->dblStartTime = dblStartTime;
}

void WaveIO_GetFileName(T_WaveIO *pWIO, char *pszDest, int iMaxLen)
{ strncpy( pszDest, pWIO->sz255FileName, iMaxLen );
}


BOOL WaveIO_AllParamsIncluded(T_WaveIO *pWIO)
{ return FALSE;
}



/***************************************************************************/
BOOL WaveIO_CloseFile( T_WaveIO *pWIO )
  /* Closes the WAV-file (if opened for READING exor WRITING.
   * Returns TRUE on success and FALSE on any error.
   */
{
 BOOL ok=TRUE;
 int  n_written;
 T_WAV_CHUNK_HDR  chunk_hdr;

  if (pWIO->OpenedForWriting && (pWIO->hFile>=0) )
   {
    /* If the file has been opened for WRITING, we may have to correct
     * some header entries (depending on the file type & size, because we did
     *  not know how long the file was going to get when writing the header).
     */
    if( pWIO->dwFilePos_Data > (long)sizeof(T_WAV_CHUNK_HDR) )
     { /* If the position of the "data" chunk structure is known:
        * wind the file position back to 'data chunk header'...
        */
      if( _lseek( pWIO->hFile, pWIO->dwFilePos_Data-sizeof(T_WAV_CHUNK_HDR), SEEK_SET ) > 0)
       {
        strncpy(chunk_hdr.id, "data", 4);
        // The size of the "data" chunk is known now, write it to the file:
        chunk_hdr.dwChunkSize = pWIO->dwFileDataSize;
        n_written = _write(pWIO->hFile, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR));
        if (n_written != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
       } /* end if <seeking the "data"-chunk length entry ok> */
      if (ok)
       { /* If all accesses up to now were ok, also update the "total" file
          * size in the RIFF header.
          */
         pWIO->RiffHdr.len = pWIO->dwFilePos_Data + pWIO->dwFileDataSize;
         // Rewind to the beginning of the file
         //     (where the RIFF header should be :)
         if( _lseek( pWIO->hFile, 0L, SEEK_SET ) == 0)
          {
           n_written = _write(pWIO->hFile, &pWIO->RiffHdr, sizeof(T_WAV_RIFF_HDR) );
           if( n_written != sizeof(T_WAV_RIFF_HDR) )
            { // writing the size-corrected RIFF header failed:
              ok = FALSE;
            }
          }
       } /* end if <ok to correct the file size in the RIFF header> */
     } /* end if  <RIFF WAVE  and  pWIO->dwFilePos_Data > sizeof(..)     */
   } // end of (..OpenedForWriting)


  /* Now close the file handle, no matter if it was for READING or WRITING.
   */
   _close(pWIO->hFile);
   pWIO->hFile = -1;  // forget this handle (it's now invalid)

  pWIO->OpenedForReading = FALSE;
  pWIO->OpenedForWriting = FALSE;

  return ok;
} // WaveIO_CloseFile(..)



/* EOF <WaveIO.c> */



