/*
 *  File:    ?\WSQ2\WSQ_MainWin.c
 *  Date:    2017-12-03 (made WSQ2 compatible with WSQCall, with 1.46484375 Hz tone spacing)
 *  Authors: Con,  ZL2AFP (inventor and author of the original code),
 *           Wolf, DL4YHF (separated GUI and codec, added a few modifications).
 *  Purpose: WSQ main window (with the very windows-specific stuff) .
 *   Formerly part of the original code  main_wsq_2 _seconds.c   by ZL2AFP .
 *   Moved into an extra module to separate GUI and 'algorithms' by DL4YHF .
 *
 *
 *      #####################################################################
 *
 *
 *        This software is provided 'as is', without warranty of any kind,
 *        express or implied. In no event shall the author be held liable
 *        for any damages arising from the use of this software.
 *
 *        Permission to use, copy, modify, and distribute this software and
 *        its documentation for non-commercial purposes is hereby granted,
 *        provided that the above copyright notice and this disclaimer appear
 *        in all copies and supporting documentation.
 *
 *        The software must NOT be sold or used as part of a any commercial
 *        or "non-free" product.
 *
 *      #####################################################################
 */

#ifdef __BORLANDC__
// prevent that bugger from bugging us with 'initialized data in header, cannot pre-compile..." :
# pragma hdrstop
#endif


/*------------------------------------------------------------------------------
 *
 *
 *
 */

#define USE_RICH_EDITORS 1  /* use a "Rich" text editor ? 1 = yes, 0 = no */
         // (even after all these years, a "normal, poor" text edit control
         //  seems to be restricted to 64 kByte of text !
         //  So set USE_RICH_EDITORS to 1, and send EM_EXLIMITTEXT to make sure
         //  the RichText-control will REALLY accept lots of text. )

#include <windows.h>
#include <windowsx.h>
#include <richedit.h>  // RichEdit, part of windoze, no 'special' DLL
#include <commctrl.h>  // Common Controls, part of windoze, no 'special' DLL
#include <commdlg.h>   // Common Dialogs, also a standard windows header

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
// ex: #include <conio.h>  // heavens, no.

#include <stdarg.h>
#include <tchar.h>




#include "pa_host.h"
#include "portaudio.h"
#ifdef  UseAudioIO  // 'UseAudioIO' MAY be defined in the project settings,
# include "AudioIO/AudioIO.h" // ... if DL4YHF's Audio-I/O-DLL host shall be used
  // AudioIO.h defines AUDIO_IO_H; that's what we check after this point
  // whenever something Audio-I/O-specific is required in this module .
#endif // UseAudioIO ?

#include "dspmath.h"
#include "coeffs.h"

// Ex: include <complex.h>
   // DL4YHF 2014-02-15 : Including 'complex.h' in a *.c (not cpp) module
   // caused a megaton of obfuscated error messages from Borland C++Builder V6,
   // most occurred in "_mbstate.h" whatever that is.
   // It later turned out that WSQ doesn't need complex.h at all.
#include "WinStuff.h"   // windows-specific stuff like ReadIntFromIniFile(), etc (YHF)
#include "WaveIO.h"     // audio file reader/writer by DL4YHF (added 2014-02-19)

#include "main.h"       // menu identifiers and similar macro-constants
// ex: #include "wsq_varicode tables.c"  // avoid spaces in filenames, even if M$ says it's ok

#include "WSQ_Codec.h"        // WSQ_DecodeAudio(), WSQ_BeginTransmission(), WSQ_GenerateAudio(), ..
#include "WSQ_Interactive.h"  // header for the 'interactive' WSQ decoder
#include "CW_Generator.h"     // CW_GenerateAudio() [to send callsign of similar in Morse code]

int g_iNominalSamplingRateForSoundcard = 32000;  // DL4YHF: Seems to be the soundcard's sampling frequency
#define SAMPLES_PER_BUFFER 32 // frames per buffer (for PortAudio), originally only FOUR sample points per call (!!)
#define CHANNELS_PER_SAMPLE 1

#define SMOOTH_K    0.13333333

#define ASCII_XON   0x11 //for serial comms routine
#define ASCII_XOFF  0x13

#define TIMER1_INTV_MS 50 /* interval of timer 1 in milliseconds */

// Background colours for the two receive text windows.
//   Also usable as 2nd parameter for SetTextColor(),
//   but NOT as an 'RGBQUAD' due to the stupidly incompatible colour formats:
#define C_RGB_RX1 RGB(250,230,160) // a bit more reddish than the 2nd channel
#define C_RGB_RX2 RGB(240,240,140) // a bit more yellowish than the 1st channel


//---------------------------------------------------------------------------
// Variables  (global but mostly 'internal', for convenience)
//   DL4YHF' preferrence : If the name of a variable leaves any doubt
//                         about its data type, use the following prefixes:
//                         b   : BYTE. The elder will know. Nowadays 'uint8_t'.
//                         w   : WORD. Still used in the windows world for 16 bit unsigned int.
//                         dw  : DWORD. 32 bit unsigned integer
//                         i   : integer, not necessarily 32 bit
//                         i32 : integer, definitely 32 bit
//                         flt : single precision float
//                         dbl : double precision float
//                         sz  : string terminated with a zero byte
//                         sz255: string which can be filled (by callee) with up to 255 characters
//                         f   : BOOL (boolean, aka 'flag', only TRUE or FALSE)
//---------------------------------------------------------------------------

  char g_szAppTitle[256];    // now set at runtime, may contain the INSTANCE INDEX in the title
  char g_szShortTitle[256];
  const char g_szClassName[]   = "WSQ";
  char g_szIniFileName[20] = "WSQ_Config.ini"; // Name of the INI file used by this application
      //  Windoze will place this somewhere, by its own gusto.. usually C:\Windows\WSQ_Config.ini.
      //  Don't try to specify a full path to the ini file - you may end up in a non-writeable folder.
      //  Since 2014-03-15, the 2nd, 3rd, 4th running instance of WSQ uses a different INI file name.
  HANDLE g_hInstanceDetectionMutex = NULL;
  int    g_iInstanceNr = 0;  // Instance counter.  We may run MORE THAN ONE INSTANCE of WSQ
          // to compare different settings, especially different FFT sizes and sampling rates.

  BOOL    fAutoSwitchTxToRx; // TRUE=enabled, FALSE=disabled (no 'automatic' switch)
  BOOL    fSwitchFromTxToRx; // set in Audio-callback (TX), polled in main loop
  int     iAutoSwitchTxRxState;  // small state machine for automatic TX->RX-switch
  int     iAutoSwitchTxRxTimer;  // "timer" (actual counts initial idle symbols) for the above state machine
  int     iDisplayedEncoderTone = -1;

  char    title[80], baud[80],comports[80],stopbit_s[80],parity[80], bitsperbyte_s[80];

  static int  iComPortNr=0;  // 0 ="don't use", else 1="COM1", etc  .
                             // Also usable (for PTT control) WITHOUT a synthesizer configuration file !
                             // Thus iComPortNr is now saved in the ini-file on exit.
  static BOOL fReleaseCOMonRx;
  static int baudrate, stopbits, bitsperbyte;
  char    szComportBuffer[256];
  HANDLE  hComm=NULL;  // handle to the serial port, used to drive the synthesizer when active
                       // (and/or PTT control?). NULL when not opened.

  // ex: char synthesizer[33][80]; // frequency-setting command strings.
          // Not sure if synthesizer[0] is a command to actually TURN OFF the synthesizer;
          // if it is, then, for 33 different 'real' WSQ tones, we need 34 strings:
  char synthesizer[34][80]; // frequency-setting command strings; [0] = command to 'turn off' (mod. 2014-03-19)
          // Loaded from the synthesizer setup file in

  // not used anywhere: long int   NumberofBytes = 0;   // number of bytes to save to wave file

  int    g_nInputDevices,            g_nOutputDevices;
  char   g_sz255InputDevice[256],    g_sz255OutputDevice[256];
  PortAudioStream *g_PAStream_Recv, *g_PAStream_Send;

  T_WaveIO g_AnalysedWaveFile;  // one instance of an audio file reader (added 2014-02-20)
  T_WaveIO g_RecordedWaveFile; // one instance of an audio file writer (added 2014-02-20)
  char   AnalysedWaveFileName[MAX_PATH+101]; // saved in the config (INI file) !
  char   RecordedWaveFileName[MAX_PATH+101]; // saved in the config (INI file) !
  int    nCharsEmittedDuringWaveAnalysis, nDotsEmittedDuringWaveAnalysis;
  char   WSQ_sz255LastError[256];

  DWORD  g_dw50MillisecondCounter;  // postscaler for the 50-ms-timer (used for flashing buttons, etc, not time-critical)
  SYSTEMTIME g_LastDisplayedTime;

#ifdef AUDIO_IO_H     // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
  T_AIO_DLLHostInstanceData AIO_Host; // one instance of DL4YHF's "audio-I/O-DLL" host
  char   AudioIoDllFileName[MAX_PATH+101];
  BOOL   fUseAudioIOAsInput, fUseAudioIOAsOutput;
         // Because it's up to the user in which sequence he launches
         // the audio SOURCE and DESTINATION (different applications),
         // fUseAudioIOAsInput = TRUE   doesn't mean we are receiving
         // audio from the SOURCE (if WSQ is the DESTINATION) yet.
  BOOL   fAudioInputAlive; // TRUE when successfully reading input;
         // FALSE as long as no data arrive from the 'audio source' .
         // So far, only used in combination with 'fUseAudioIOAsInput' .
  int    iAudioInputAliveTimer;
  HWND   hwndAIOControlPanel;
#endif // AUDIO_IO_H ?
  BOOL   fRecordingWaveFile    = FALSE;
  BOOL   fAnalysingWaveFile    = FALSE;
  BOOL   fDecodeWaveFileInLoop = FALSE;
  BOOL   fDecodeWaveStatistics = FALSE;
  BOOL   RXbutton              = TRUE;
  BOOL   TXbutton              = FALSE;  // reflects the state of the 'transmit' button *and* the transmitter itself
  BOOL   TXbuttonFlashOff      = FALSE;
  BOOL   Pause                 = FALSE;
  BOOL   fShowSymbolMarkers    = TRUE;
  BOOL   fShowSymbolNumbers    = FALSE;  // .. as overlay on the scrolling waterfall ?
  BOOL   Pausebutton           = FALSE;
  // ex: BOOL SpectrumAveragebutton = FALSE; // replaced by extern T_WSQDecoder.symbol_avrg_mode
  int    iWFColorPalette       = 0; // one of the following values:
#   define WF_PALETTE_BLUE    0
#   define WF_PALETTE_GREEN   1
#   define WF_PALETTE_SUNRISE 2
#   define WF_PALETTE_LINRAD  3
  extern unsigned long SunriseColors[256]; // imported from SunriseColors.c
  extern unsigned long LinradColors[256];  // imported from LinradColors.c
  BOOL   fShowSpectrumGraph = TRUE; // option to show the most recent spectrum as a 'curve' (graph) over the spectrogram


  // Added by DL4YHF ...
  float  g_WfallFmin_Hz, g_WfallFmax_Hz;
  float  fltCWSecondsPerDot;
  char   sz255CWid[256];
  int    iCWTxCharIndex=0; // index into sz255CWid[] during CW transmission
  int    iCWidState = 0; // 0=passive, 1=request to send CW ID, 2=sending CW ID
  int    iCWidAlways= 0;
  T_CWGenerator CW_Generator;

  static HBRUSH hBrush;  // MUST be static! // YHF: Now also used in subroutines
  static HBRUSH hBrushButton, hBrushButtonAlt, hBrushBorder;
  static HBRUSH hBrush_receive[WSQ_MAX_DECODER_CHANNELS], hBrush_send;
  static HCURSOR hCursorEastWest, hCursorNorthSouth, hCursorArrow, hCursorWSQI;
  static int iPrevKeysAndButtons=0;  // for the mouse-move / button up/down-handler
  static int iMouseOverArea,iDraggingArea,iMouseMoveStartX,iMouseMoveStartY;
  static int iDragStartLayoutY0, iDragStartLayoutY1, iDragStartLayoutY2;
  static float fltMouseMoveStartXRel,fltMouseMoveStartYRel;
  static float fltMouseMoveStartFreq,fltMouseMoveStartFSpan; // for dragging waterfall or frequency scale


  HWND    g_hwndRXbutton, g_hwndTXbutton, g_hwndPausebutton;

  HWND    g_hwndMain;
  HWND    hwndTrackbar[2];
#  define TRACKBAR_CONTRAST   0
#  define TRACKBAR_BRIGHTNESS 1
  HMENU   g_hMainMenu;           // Handle of the main menu, etc...
  HMENU   g_hWFallContextMenu;
  float   g_fltClickedFrequency; // used for the above waterfall context menu
  BOOL    fMouseDragged;   // .. in the waterfall, to tell a 'click' from a 'dragged waterfall'

  HANDLE  hBitmapRX_ON, hBitmapTX_ON, hBitmapRX_OFF, hBitmapTX_OFF, hBitmapPause_ON, hBitmapPause_OFF;

  HINSTANCE hThisInstance;
  // HINSTANCE g_hInst;  // replaced by WS_hInstance .. don't duplicate info !
  HWND      hCtl_RX[WSQ_MAX_DECODER_CHANNELS], hCtl_TX;  // [rich] edit controls for RX and TX

  HDC       hdc2, hdcScroll;
  HBITMAP   hbmScroll, hbmScrollOld; // scrolling waterfall bitmaps, must be preserved between WM_PAINT messages
# define USE_DIB_SECTION 1 /* use "DIB" = "device-independent bitmap" (faster) ? */
#if( USE_DIB_SECTION )  // use "DIB" = "device-independent bitmap" ?
  BITMAPINFO g_bmiScrollingWF;       // 'bitmap info', used by CreateDIBSection(),
                                     // contains height+width from 'creation' of the scrolling waterfall bitmap.
  DWORD *g_pdwWfallPixelData = NULL; // pointer to the 'pixel data' within the waterfall bitmap,
                                     // for fast direct image maninpulation without SetPixel/SetPixelV
  DWORD  g_dwWfallPixelStride=0;     // access Pixel[y][x] as g_pdwWfallPixelData[y*g_dwWfallPixelStride + x]
  DWORD  g_dwWfallPixelsYMax;
#endif // USE_DIB_SECTION ?
  static int  s_nNewWaterfallLines = 0;
  static BOOL s_fRedrawAllWFLines = TRUE;

  typedef union
   { DWORD dw;
     BYTE b[4];
   } T_RGBColor;
#if( USE_DIB_SECTION )  // use "DIB" = "device-independent bitmap", and direct access to pixels via pointer (which uses RGBQUADs)
#  define RGB_BYTE_BLUE  0
#  define RGB_BYTE_GREEN 1
#  define RGB_BYTE_RED   2
#  define RGB_BYTE_ALPHA 3
#else // do not direct access to the waterfall's pixels but 'SetPixel' (which uses COLORREFs)
#  define RGB_BYTE_BLUE  0
#  define RGB_BYTE_GREEN 1
#  define RGB_BYTE_RED   2
#  define RGB_BYTE_ALPHA 3
#endif
  T_RGBColor spectrogram_rgb[ FFT_MAX_FREQ_BINS ]; // ex: brightness, but these are R+G+B components (using funny incompatible offsets, see above)
  char      sz80[84];
  int       iNumVisibleChars;


  float     ss, tmp, pk, agc_decay, agcthr, agcgain;

  int       gain,  iWFBrightness_Percent, iWFContrast_Percent;
  float     noise_blank ;

  HFONT     RXfont;

  FILE      *Wavfile;

  // Main window layout (flexible + sizeable since 2014-02-16) :
  //
  //   Title
  //   Menu
  //   -------------------------------------------------y=0  ----   ---
  //   | Main Rx Text (editor)                         |         |   |   25 % (?)
  //   |-----------------------------------------------|y0       |  <-- splitter 1
  //   | Second, optional Rx Text (editor)             |         |   |   25 % (?)
  //   -------------------------------------------------y1(290)  |  <-- splitter 2
  //   | Tx Text (editor)                              |         |   |   10 % (?)
  //   -------------------------------------------------y2(360)  |  <-- splitter 3
  //   |graph_  |------------------------------------| |         |   |
  //   | rect   | Scrolling Waterfall Bitmap         | |         |   | remaining
  //   |        |------------------------------------| |         |   | height
  //   --------------------------------------------------y5(488) |   |  (nn %)
  //   | Brt.+Contr.| Peak-      |  _____  _____  ____ |_y6(490) |   |
  //   | (trackbars)| Indicator  | |Pause|| Tx  || Rx ||         |   |
  //   -------------------------------------------------yEnd(537)-  ---
  //   |        |   |            | |      |      |     |
  //   x=0     x1  x2           x3 x4     x5    x6     xEnd(821)
  //               (260)     (634)(638)  (700) (762)
  //
  //  xEnd + yEnd are actually the 'endstops', just outside the client area.
  //  Values in parentheses are the defaults from ZL2AFP's original code.
  //  Vertical splitter positions depend on window height (client height)
  //           and the following 'percentages' :
  int Layout_iSplitPercent[3] = { 25, 25, 15 }; // splitter positions in PERCENT OF CLIENT HEIGHT, input for UpdateMainWindowLayout()
  int Layout_xWfallLeft  =  60, // left edge of waterfall bitmap (scrolling bitmap area)
      Layout_yWfallTop   = 370, // upper edge of the scrolling waterfall bitmap
      Layout_iWfallWidth = 755, // width of the scrolling waterfall in pixels
      Layout_iWfallHeight=  86, // height of the scrolling waterfall in pixels

      Layout_x2=260,     // separator line between trackbar and Peak indicator box;
                         // next pixel to the right is 'commands_rect'
      Layout_x3=634,     // vertical separator between 'commands_rect' (left)
                         // and the button area (right)
      Layout_x4=388+250, // left edge of 'Pause' button
      Layout_x5=450+250, // left edge of 'Tx' button
      Layout_x6=512+250, // left edge of 'Rx' button (762?)
      Layout_xEnd=821,   // horizontal endstop; width of the client area MINUS ONE
      Layout_iTrackbarWidth=80, // width of each trackbar (contrast+brightness)
#     define C_TRACKBAR_BOX_X1   4
#     define C_TRACKBAR_Y_OFFSET 2
#     define C_TRACKBAR_HEIGHT  20
#     define C_TRACKBAR_H_SEPARATOR 10
      Layout_iCurveHeight=221,   // height (ex: 221)
      Layout_iBtnWidth = 58, // width (the BITMAPS for these buttons are 58 pixels wide)
      Layout_iBtnHeight= 24, // height (the BITMAPS for these buttons are 24 pixels high)

      Layout_y0=145,     // border between 'main' and 'secondary' Rx text (QSO + 'monitor')
      Layout_y1=290,     // lower edge of Rx text, upper edge of Tx text
      Layout_y2=360,     // upper edge of the drawing area for WM_PAINT
      // Layout_y4 : removed again
      Layout_y5=488,     // upper edge of the box around trackbar, indicators, and buttons
      Layout_y6=490,     // upper edge of buttons, and (approx.) the trackbar
      Layout_yEnd=537,   // vertical endstop; height of the client area MINUS ONE
      Layout_yMeterTop,  // upper end of the bargraph (ex: 366)
      Layout_yMeterBottom; // lower end of the bargraph (ex: 460)


  RECT  rctClientArea, graph_rect, commands_rect; // also set in UpdateMainWindowLayout() now

//---------------------------------------------------------------------------
// internal 'forward' references (because C compilers are stupid)
//---------------------------------------------------------------------------

void OnWmCreate(HWND hwnd /*, CREATESTRUCT *pCS*/ ); // handler for WM_CREATE (forward decl.)
void UpdateVariableMenuItems(HMENU hMenu);
void SaveSettingsInIniFile(void);
void CreateBitmapForWaterfall(void);
int  FreqToScreenXoffset( double freq_Hz );


//---------------------------------------------------------------------------
// Functions
//---------------------------------------------------------------------------

void SetCurrentDirectoryToPathOfExecutable(void)
{
  // Because some nasty file selector boxes like to switch
  // the "current directory" by their own gusto,
  // set the current directory back to where it should be:
  // 
  // Without this, for example the help files will not be found
  // by ShellExecute() because the CurrentWD points "somewhere else",
  // for example into the folder from where you just loaded
  // a wave file for analysis. This is seriously broken because
  // the file selector returns a full path anyway..
  char sz255PathToExecutable[256], *cp;
  GetModuleFileName(0, sz255PathToExecutable, 255 );
  cp = strrchr( sz255PathToExecutable, '\\' );
  if( cp==NULL )  // quite unlikely that micro$oft uses POSIX one fine day, but...
   {  cp = strrchr( sz255PathToExecutable, '/' );
   }
  if( cp!=NULL )
   { *cp = '\0';  // strip the name of the executable, leave only the PATH to it .
     SetCurrentDirectory( sz255PathToExecutable );
     // On the author's PC, the result in sz255PathToExecutable was:
     //  "C:\\cbproj\\WSQ2" . Ok. ShellExecute() should be able to "open" (verb)
     //  the help file, located in a subdirectory within the current directory.
   }
}

//---------------------------------------------------------------------------
void UpdateMainWindowLayout( int iClientWidth, int iClientHeight )
  // Updates the main window layout (parameters) for a given size
  // of the main window (in geek speak: based on the window's client area).
  // Called from the WM_CREATE handler for the first time,
  //   and (later) from the WM_SIZE handler when the window is RESIZED.
  //   Input : Layout_iSplitPercent[]
  //   Output: Layout_x.., Layout_y.. .
  // See also: ApplyLayoutForChildWindows() [moves 'child windows' to new positions]
{
  int h;
  RECT rct;
  // Note: The child windows don't necessarily exist yet when getting here.
  //       That's why the client width + height may have to be passed as arguments.
  //       Only when the main window already exists, they may be ZERO = "detect here":
  if( iClientWidth==0 && iClientHeight==0 )
   {  GetClientRect( g_hwndMain, &rct );
      iClientWidth = rct.right-1;
      iClientHeight= rct.bottom-1;
      LimitInteger( &iClientWidth, 10, 10000 ); // kludge to avoid div-by-zero
      LimitInteger( &iClientHeight,10, 10000 );
   }

  // Thus UpdateMainWindowLayout() only sets a few simple integers (Layout_xyz) !
  Layout_xEnd = iClientWidth - 1;  // -> 821 with ZL2AFP's original layout
  Layout_yEnd = iClientHeight- 1;  // -> 537 with ZL2AFP's original layout

  // Let's begin with the easy stuff in the lower right corner: the buttons..
  Layout_y6/*490*/ = Layout_yEnd - (Layout_iBtnHeight + 4);
  Layout_x6/*762*/ = Layout_xEnd - (Layout_iBtnWidth  + 2); // "Rx"
  Layout_x5/*700*/ = Layout_x6   - (Layout_iBtnWidth  + 3); // "Tx"
  Layout_x4/*638*/ = Layout_x5   - (Layout_iBtnWidth  + 3); // "Pause"
  Layout_x3/*634*/ = Layout_x4   - 4;
  Layout_x2/*260*/ = Layout_x3   - 374; // fixed width of the Peak indicator text
  Layout_xWfallLeft/*60*/  = 60; // LEFT edge of the scrolling waterfall bitmap is fixed ("anchor left")
  Layout_iWfallWidth/*755*/= Layout_xEnd/*821*/ - Layout_xWfallLeft/*60*/ - 6;

  // If the window gets WIDER, the trackbars (C+B sliders) also get wider,
  // because that's the most easily resizeable item now. May change in future !
  Layout_iTrackbarWidth = (Layout_x2 - C_TRACKBAR_BOX_X1 - 3*C_TRACKBAR_H_SEPARATOR) / 2;

  // Vertical screen layout:
  //   -------------------------------------------------y=0  ----   ---
  //   | Main Rx Text (editor)                         |         |   |   30 % (?)
  //   |-----------------------------------------------|y0       |  --- splitter 1
  //   | Second, optional Rx Text (editor)             |         |   |   30 % (?)
  //   -------------------------------------------------y1(290)  |  --- splitter 2
  //   | Tx Text (editor)                              |         |   |   10 % (?)
  //   -------------------------------------------------y2(360)  |  --- splitter 3
  //   | remaining height: waterfall, buttons, etc     |                 30 %
  //   -------------------------------------------------
  // Avoid 'completely wrong' splitter percentages :
  LimitInteger( &Layout_iSplitPercent[0], 5, 80 );
  LimitInteger( &Layout_iSplitPercent[1], 5, 85 - Layout_iSplitPercent[0] );
  LimitInteger( &Layout_iSplitPercent[2], 5, 90 - Layout_iSplitPercent[0] - Layout_iSplitPercent[1] );
  h = (iClientHeight * Layout_iSplitPercent[0] ) / 100;
  #define MIN_EDITOR_HEIGHT 40 /*pixels*/
  LimitInteger( &h, MIN_EDITOR_HEIGHT, iClientHeight - 100 );
  Layout_y0 = h;
  h = (iClientHeight * Layout_iSplitPercent[1] ) / 100;
  LimitInteger( &h, MIN_EDITOR_HEIGHT, iClientHeight - Layout_y0 - 100 );
  Layout_y1 = Layout_y0 + h;
  if( WSQ_Decoder.nChannels==1 )
   { // 2nd Rx text editor not in use : splitter 1 (!) has no effect, same in OnMouseEvent() !
     Layout_y0 = Layout_y1;
   }
  h = (iClientHeight * Layout_iSplitPercent[2] ) / 100;
  LimitInteger( &h, MIN_EDITOR_HEIGHT, iClientHeight - Layout_y1 - 100 );
  Layout_y2 = Layout_y1 + h;
  h = iClientHeight - Layout_y2;  // remaining height for waterfall+sliders&buttons..
  if( h < 100 )
   {  h = 100;                    // ... shouldn't be less than 100
      Layout_y2 = iClientHeight-h;
   }
# define SEPARATOR_BELOW_TX_TEXT 6
# define WFALL_BORDER_TOP        8  // a bit more space for the tx-frequency-indicators
# define WFALL_BORDER_BOTTOM     4  
# define WFALL_BORDER_WIDTH      6  // less space on the sides of the waterfall bitmap
  Layout_yWfallTop = Layout_y2 + SEPARATOR_BELOW_TX_TEXT + WFALL_BORDER_TOP; // upper edge of waterfall bitmap
  Layout_y5 = Layout_y6 - 2;     // y5=bottom of graph_rect (like y6, anchored to bottom of window)
  Layout_iWfallHeight = Layout_y5 - Layout_yWfallTop - 22/*frequency scale, etc*/ - WFALL_BORDER_TOP - WFALL_BORDER_BOTTOM;
  Layout_yMeterTop    = Layout_y2 + SEPARATOR_BELOW_TX_TEXT; // upper end of the bargraph (ex: 366)
  Layout_yMeterBottom = Layout_y5 - 28;   // lower end of the bargraph (ex: 460)

} // UpdateMainWindowLayout()


void ApplyLayoutForChildWindows(void)
  // Moves the 'child windows' (windowed controls) according to the layout .
  // Called after UpdateMainWindowLayout(), but only if those 'children' exist !
{
  MoveWindow( g_hwndPausebutton,
        Layout_x4,         // new x1, ex: 388+250
        Layout_y6,         // new y1, ex: 490
        Layout_iBtnWidth,  // width
        Layout_iBtnHeight, // height
        TRUE/*bRepaint*/ );
  MoveWindow( g_hwndTXbutton,
        Layout_x5,         // pos x1, ex: 450+250
        Layout_y6,         // pos y1, ex: 490
        Layout_iBtnWidth,  // width
        Layout_iBtnHeight, // height
        TRUE/*bRepaint*/ );
  MoveWindow( g_hwndRXbutton,
        Layout_x6,         // pos x1, ex: 512+250,
        Layout_y6,         // pos y1, ex: 490
        Layout_iBtnWidth,  // width
        Layout_iBtnHeight, // height
        TRUE/*bRepaint*/ );

  MoveWindow( hwndTrackbar[TRACKBAR_CONTRAST],
        C_TRACKBAR_BOX_X1 + C_TRACKBAR_H_SEPARATOR,        // x
        Layout_y6+C_TRACKBAR_Y_OFFSET, // y
        Layout_iTrackbarWidth, // width
        C_TRACKBAR_HEIGHT,     // height
        TRUE/*bRepaint*/ );
  MoveWindow( hwndTrackbar[TRACKBAR_BRIGHTNESS],
        C_TRACKBAR_BOX_X1 + 2*C_TRACKBAR_H_SEPARATOR + Layout_iTrackbarWidth, // x
        Layout_y6+C_TRACKBAR_Y_OFFSET, // y
        Layout_iTrackbarWidth, // width
        C_TRACKBAR_HEIGHT,     // height
        TRUE/*bRepaint*/ );


  // Since MoveWindow not only moves a window but resizes it, use it for the editors:
  MoveWindow( hCtl_RX[0],  // main receive window, topmost edit window, ALWAYS visible
        0,    // pos x1 (in window client coords, 0= upper left corner of parent window)
        0,    // pos y1
        Layout_xEnd,  // width in pixels
        Layout_y0-1,  // height in pixels (leaving a few pixels for the 'splitter')
        TRUE/*bRepaint*/ );
  MoveWindow( hCtl_RX[1],  // 2nd receive window, only SOMETIMES visible
        0,    // pos x1 (in window client coords, 0= upper left corner of parent window)
        Layout_y0,    // pos y1
        Layout_xEnd,  // width in pixels
        Layout_y1-Layout_y0-1, // height in pixels
        TRUE/*bRepaint*/ );

  MoveWindow( hCtl_TX,
        0,            // pos x1
        Layout_y1,    // pos y1, ex: 290
        Layout_xEnd,           // width in pixels,  ex: 825
        Layout_y2-Layout_y1-1, // height in pixels, ex: 70
        TRUE/*bRepaint*/ );

} // ApplyLayoutForChildWindows()


//---------------------------------------------------------------------------
int IdentifyClientCoord( int iClientX, int iClientY,
         float *pfltXrel,   // optional output: relative 'X' coord, 0...1 within the area
         float *pfltYrel )  // optional output: relative 'X' coord, 0...1 within the area
  // Identifies the 'area' for a given client coordinate.
  // Possible return values, roughly ordered by vertical position ("from top to bottom"):
# define CLIENT_AREA_UNKNOWN    0
# define CLIENT_AREA_RX_TEXT_1  1
# define CLIENT_AREA_RX_TEXT_2  2
# define CLIENT_AREA_TX_TEXT    3
# define CLIENT_AREA_WATERFALL  4
# define CLIENT_AREA_FREQ_SCALE 5
# define CLIENT_AREA_TX_FREQ    6  /* indicator for the TRANSMIT frequency, can be pulled via mouse */
# define CLIENT_AREA_RX1_F_LO   7  /* indicator for the 1st decoder's start frequency (i+0) */
# define CLIENT_AREA_RX2_F_LO   8  /* indicator for the 2nd decoder's start frequency (i+1) */
# define CLIENT_AREA_RX1_F_HI   9  /* indicator for the 1st decoder's end   frequency */
# define CLIENT_AREA_RX2_F_HI  10  /* indicator for the 2nd decoder's end   frequency */
{
  int x,x1,x2,y,y1,y2,h;
  /* TONE SPACING for TX taken from "Decoder1" settings : */
  double dblToneSpacing_Hz = (WSQ_Decoder.channel[0].iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */

  // Check the different 'areas' within the WSQ window, ordered from top to bottom.
  // See DL4YHF's ASCII-art "Main window layout" for details !
  // With a 'form manager' as in Borlands's VCL this would be easier,
  //   but here, with plain-vanilla C / Win32 programming, it's still manageable..
  if( iClientX<0 || iClientX>Layout_xEnd || iClientY<0 || iClientY>Layout_yEnd )
   { return CLIENT_AREA_UNKNOWN;   // cannot be in the client area (but "borders")
   }
  if( iClientY<Layout_y0 )
   { return CLIENT_AREA_RX_TEXT_1;
   }
  if( (iClientY>Layout_y0+1) && iClientY<Layout_y1 )
   { return CLIENT_AREA_RX_TEXT_2;
   }
  if( (iClientY>Layout_y1+1) && iClientY<Layout_y2 )
   { return CLIENT_AREA_TX_TEXT;
   }
  x = iClientX - Layout_xWfallLeft;
  y = iClientY - (Layout_yWfallTop-WFALL_BORDER_TOP);
  if( x>=0 && x<Layout_iWfallWidth && y>=0 && y<WFALL_BORDER_TOP )
   { // inside the narrow 'top area' (just above the waterfall) :
     if( pfltXrel != NULL )
      { *pfltXrel = (float)x / (float)Layout_iWfallWidth;
      }
     if( pfltYrel != NULL )
      { *pfltYrel = 0;  // no meaningful information in this coord
      }
     /* indicator for the TRANSMIT frequency, can be pulled via mouse. */
     x1 = FreqToScreenXoffset( WSQ_Encoder.fltLowestToneFreq );
     x2 = FreqToScreenXoffset( WSQ_Encoder.fltLowestToneFreq + dblToneSpacing_Hz * (WSQ2_NUM_TONES-1) );
     if( (x >= x1 ) && (x <= x2 ) )
      { return CLIENT_AREA_TX_FREQ;
      }
   }
  // inside the scrolling waterfall area, or on one of its 'internals' ?
  x = iClientX - Layout_xWfallLeft;
  y = iClientY - Layout_yWfallTop;
  if( x>=0 && x<Layout_iWfallWidth && y>=0 && y<Layout_iWfallHeight )
   { if( pfltXrel != NULL )
      { *pfltXrel = (float)x / (float)Layout_iWfallWidth;
      }
     if( pfltYrel != NULL )
      { *pfltYrel = (float)y / (float)Layout_iWfallHeight;
      }
     x1 = FreqToScreenXoffset( WSQ_Decoder.channel[0].fmin_Hz );
     x2 = FreqToScreenXoffset( WSQ_Decoder.channel[0].fmax_Hz );
     if( x>=x1-2 && x<=x1+2 )
      { return CLIENT_AREA_RX1_F_LO; /* indicator for the 1st decoder's start frequency */
      }
     if( x>=x2-2 && x<=x2+2 )
      { return CLIENT_AREA_RX1_F_HI; /* indicator for the 1st decoder's end   frequency */
      }
     if( WSQ_Decoder.nChannels>=2 )
      { x1 = FreqToScreenXoffset( WSQ_Decoder.channel[1].fmin_Hz );
        x2 = FreqToScreenXoffset( WSQ_Decoder.channel[1].fmax_Hz );
        if( x>=x1-2 && x<=x1+2 )
         { return CLIENT_AREA_RX2_F_LO;
         }
        if( x>=x2-2 && x<=x2+2 )
         { return CLIENT_AREA_RX2_F_HI;
         }
      }
     return CLIENT_AREA_WATERFALL;
   }
  // inside the frequency scale ?
  x = iClientX - Layout_xWfallLeft;
  y1= Layout_yWfallTop  + Layout_iWfallHeight + WFALL_BORDER_BOTTOM; // upper end of frequency scale
  y2= Layout_y5; // lower end of frequency scale
  h = y2-y1-1;
  y = iClientY - y1;
  if( x>=0 && x<Layout_iWfallWidth && y>=0 && y<h )
   { if( pfltXrel != NULL )
      { *pfltXrel = (float)x / (float)Layout_iWfallWidth;
      }
     if( pfltYrel != NULL )
      { *pfltYrel = (float)y / (float)h;  // just for completeness, 'y' not used for anything yet
      }
     return CLIENT_AREA_FREQ_SCALE;
   }

  return CLIENT_AREA_UNKNOWN;
} // end IdentifyClientCoord()


int FreqToScreenXoffset( double freq_Hz ) // DL4YHF 2014-02-16
{ // Convers a frequency [Hz] into a screen offset (client abscissa, "x"),
  // for the waterfall area (and the associated frequency scale).
  // Note: To avoid desaster with invalid coordinates, the result will be
  //       clipped inside the 'waterfall coordinate range', which is
  //       Layout_xWfallLeft ... Layout_xWfallLeft + Layout_iWfallWidth - 1.
  float fltQuot = g_WfallFmax_Hz-g_WfallFmin_Hz;
  int x = 0;
  if( fltQuot > 0 )
   { x = (int)( ( 0.5*fltQuot + (freq_Hz-g_WfallFmin_Hz) * (double)Layout_iWfallWidth ) / fltQuot );
   }
  if( x<0 )
   {  x=0;
   }
  if( x>=Layout_iWfallWidth )
   {  x= Layout_iWfallWidth-1;
   }
  return x;
} // FreqToScreenXoffset()

int FreqToScreenX( double freq_Hz ) // DL4YHF 2014-02-16
{ // Convers a frequency [Hz] into a screen X coord (client abscissa, "x"),
  // for the waterfall area (and the associated frequency scale).
  // Note: To avoid desaster with invalid coordinates, the result will be
  //       clipped inside the 'waterfall coordinate range', which is
  //       Layout_xWfallLeft ... Layout_xWfallLeft + Layout_iWfallWidth - 1.
  return Layout_xWfallLeft + FreqToScreenXoffset( freq_Hz );
} // FreqToScreenX()

double ScreenXoffsetToFreq( int x ) // DL4YHF 2014-02-28
{ // Inverse to FreqToScreenXoffset() , i.e. :
  // x = (int)( (0.5 + (freq_Hz-g_WfallFmin_Hz) * Layout_iWfallWidth
  //         / ( g_WfallFmax_Hz-g_WfallFmin_Hz ) ) );
  double freq_Hz;
  if( Layout_iWfallWidth > 2 )
   { freq_Hz = g_WfallFmin_Hz + (double)x * ( g_WfallFmax_Hz-g_WfallFmin_Hz ) / (double)Layout_iWfallWidth;
   }
  else
   { freq_Hz = g_WfallFmin_Hz;
   }
  return freq_Hz;
} // ScreenXoffsetToFreq()


int ChooseFile( HWND hParent,
                char *cpFilter, // [in] something like "Setup Files Only (*setup*.txt)\0*setup*.txt\0All Files (*.*)\0*.*\0\0"
                char *cpTitle,  // [in] something like "Select synthesizer setup file"
                char *sResult)  // [out] zero-terminated string, use a buffer with at least (MAX_PATH+101) chars
{
    OPENFILENAME ofn;    // To compile with BORLAND (and some other compilers),
                         // declare locals BEFORE executable code .
    sResult[0] = 0;
 // OPENFILENAME ofn;    // Borland : "declaration isn't allowed here'. Right.
                         // In "C" (not C++) local variables should be declared
                         // before the execuatable code .
    memset(&ofn, 0, sizeof(OPENFILENAME));
    ofn.lStructSize = sizeof(OPENFILENAME);
    ofn.hInstance   = WS_hInstance;
    ofn.Flags       = OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY;
    ofn.hwndOwner   = hParent;
    ofn.lpstrFilter = cpFilter; // ex: ("Setup Files Only (*setup*.txt)\0*setup*.txt\0All Files (*.*)\0*.*\0\0");
    ofn.lpstrTitle  = cpTitle;  // ex. ("Select synthesizer setup file");
    ofn.lpstrFile   = sResult;
    ofn.nMaxFile    = MAX_PATH+100;
    ofn.lpstrCustomFilter = NULL;
    ofn.nMaxCustFilter    = 0L;
    if (IDOK == GetOpenFileName(&ofn))
    {
        return 1;
    }
    return 0;
}

void Start_Portaudio(void);


void SerialPuts(HANDLE hComm, char *txstring)
{
  DWORD    iBytesWritten;

  if(hComm == NULL) return;

  WriteFile(hComm, txstring, (DWORD) strlen(txstring), &iBytesWritten,NULL);
}

void SendToSynthesizer( int iTone/*0..32*/ )  // called from WSQ2_Codec() during transmit, when SynthesizerActive==TRUE
{
  char buffer[256];
  if( hComm != NULL )
   {
     // Convert integer into string for the serial comms
     //   need [tone+1] as synthesizer[0] is zero Hz :
     sprintf(buffer, "%s\r", synthesizer[ iTone+1]);
     SerialPuts(hComm, buffer);
   }  
}

HANDLE SerialInit(char *ComPortName, DWORD BaudRate, int ByteSize, int StopBit, char ParityChar, char Protocol, int RxTimeOut, int TxTimeOut, int RxBufSize, int TxBufSize)
{
  HANDLE          hCom;
  // BOOL         bPortReady; // never used, removed to avoid warning from BCB
  DCB             dcb;
  COMMTIMEOUTS    CommTimeouts;
  int             Parity;
  char szPortName[80];

  // Modified 2014-03-19 to defeat problems with VSPE (Virtual Serial Port Emulator),
  //          as suggested by F5WK - thanks Michel :
  sprintf(szPortName,"\\\\.\\%s",ComPortName);  // added to enable opening VSPE virtual com port

    switch(ParityChar) {
        case 'N':
        case 'n':
            Parity = NOPARITY;
            break;
        case 'E':
        case 'e':
            Parity = EVENPARITY;
            break;
        case 'O':
        case 'o':
            Parity = ODDPARITY;
            break;
        default:
            return NULL;    // illegal parameter !
    }

    switch(StopBit)
    {
        case 1:
            StopBit = ONESTOPBIT;
            break;
        case 2:
            StopBit = TWOSTOPBITS;
            break;
        default:
            return NULL;    // illegal parameter !
    }

    hCom = CreateFile(  szPortName, // 2014-03-19: use the stuff with backslashes before the 'legacy' COM port name. Works better with VSPE.
                        GENERIC_READ | GENERIC_WRITE,
                        0,                 // exclusive access
                        NULL,              // no security
                        OPEN_EXISTING,
                        0,                 // no overlapped I/O
                        NULL);             // null template

    if(hCom == INVALID_HANDLE_VALUE) return NULL;

    SetupComm(hCom, RxBufSize, TxBufSize); // set Rx and Tx buffer sizes

    // Port settings are specified in a Data Communication Block (DCB).

    GetCommState(hCom, &dcb);

    dcb.BaudRate    = BaudRate;
    dcb.ByteSize    = ByteSize;
    dcb.Parity      = Parity;
    dcb.StopBits    = StopBit;
    dcb.fAbortOnError = TRUE;

    switch(Protocol) {
    case 'D':   // DTR/DSR
    case 'd':
        // set XON/XOFF
        dcb.fOutX   = FALSE;
        dcb.fInX    = FALSE;
        // set RTSCTS
        dcb.fOutxCtsFlow = FALSE;
        dcb.fRtsControl = RTS_CONTROL_DISABLE;
        // set DSRDTR
        dcb.fOutxDsrFlow = TRUE;
        dcb.fDtrControl = DTR_CONTROL_HANDSHAKE;
        break;
    case 'R':   // RTS/CTS
    case 'r':
        // set XON/XOFF
        dcb.fOutX   = FALSE;
        dcb.fInX    = FALSE;
        // set RTSCTS
        dcb.fOutxCtsFlow = TRUE;
        dcb.fRtsControl = RTS_CONTROL_HANDSHAKE;
        // set DSRDTR
        dcb.fOutxDsrFlow = FALSE;
        dcb.fDtrControl = DTR_CONTROL_DISABLE; 
        break;
    case 'X':   // XON/XOFF
    case 'x':
        // set XON/XOFF
        dcb.fOutX   = TRUE;
        dcb.fInX    = TRUE;
        dcb.fTXContinueOnXoff = TRUE;
        dcb.XoffChar = ASCII_XOFF;
        dcb.XoffLim = RxBufSize - (RxBufSize / 4);
        dcb.XonChar = ASCII_XON;
        dcb.XonLim = RxBufSize - (RxBufSize / 2);
        // set RTSCTS
        dcb.fOutxCtsFlow = FALSE;
        dcb.fRtsControl = RTS_CONTROL_DISABLE; 
        // set DSRDTR
        dcb.fOutxDsrFlow = FALSE;
        dcb.fDtrControl = DTR_CONTROL_DISABLE;
        break;
    case 'N':   // NOPROTOCOL
    case 'n':
    default:
        // set XON/XOFF
        dcb.fOutX   = FALSE;
        dcb.fInX    = FALSE;
        // set RTSCTS
        dcb.fOutxCtsFlow = FALSE;
        dcb.fRtsControl = RTS_CONTROL_DISABLE;
        // set DSRDTR
        dcb.fOutxDsrFlow = FALSE;
        dcb.fDtrControl = DTR_CONTROL_DISABLE;
        break;
    }

    SetCommState(hCom, &dcb);

    // Set timeouts
    CommTimeouts.ReadIntervalTimeout = RxTimeOut;
    CommTimeouts.ReadTotalTimeoutMultiplier = 0;
    CommTimeouts.ReadTotalTimeoutConstant = RxTimeOut;

    CommTimeouts.WriteTotalTimeoutMultiplier = 0;
    CommTimeouts.WriteTotalTimeoutConstant = TxTimeOut;

    SetCommTimeouts(hCom, &CommTimeouts);

    return hCom;
}
/*-----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------*/
static LRESULT CALLBACK PTTDlgProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  int i;
  char szTemp[256];


  // This sets up the comport for the PTT line
  // Uses RTS and DTR just in case there are differences in rigs

  switch (uMsg)
    {
        case WM_INITDIALOG:
            {
              SendMessage(GetDlgItem(hwndDlg,IDM_PTT), CB_ADDSTRING, 0, (LPARAM)"NONE" );
              for(i=1;i<100;i++) // populate the comobobox with all possible comports
               {
                wsprintf(szTemp,"%s%d", "COM",i);
                SendMessage(GetDlgItem(hwndDlg,IDM_PTT), CB_ADDSTRING, 0, (LPARAM)szTemp);
               }

              // Select the item in combobox that is the desired comport number
              SendMessage(GetDlgItem(hwndDlg,IDM_PTT), CB_SETCURSEL, iComPortNr, 0);

              // make sure the window stays on top
              SetWindowPos (hwndDlg, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);

            }
            break;

         case WM_COMMAND:
            switch (GET_WM_COMMAND_ID(wParam, lParam))
            {

             case IDC_CANCEL:
                 EndDialog(hwndDlg, 0);
                break;

             case IDOK:
          /*      HANDLE SerialInit(char *ComPortName,
                        DWORD BaudRate,
                        int ByteSize,   = 8
                        int StopBit,    = 1
                        char ParityChar,= None, Even, Odd
                        char Protocol,  = N for RTS/DTR etc all turned off
                        int RxTimeOut,  = 50msec?
                        int TxTimeOut,
                        int RxBufSize,
                        int TxBufSize)
         */


                 i = SendMessage(GetDlgItem(hwndDlg,IDM_PTT),CB_GETCURSEL,0,0); // -> "the index of the currently selected item"
                 // ex: SendMessage(GetDlgItem(hwndDlg,IDM_PTT),CB_GETLBTEXT,i, (LPARAM)szComportBuffer); // -> "the string from the combo box"
                 // As long as the combo list contains "COM1","COM2", ... "COM99",
                 //  (i.e. not only the ports which really exist on the system),
                 // the ITEM INDEX is actually the same as the "COM port number":
                 if( i>=0 )  // 'NONE' means neither PTT- nor synthesizer control !
                  {  iComPortNr = i;  // don't rely on this ! the COM PORT NAME may be something very different,
                     //  at least Microsoft threatens us with using 'very exotic' names instead of "COMn" ...
                     //  fortunately the hardware manufacturers stick to the "legacy" names !
                  }
                 if( hComm != NULL )
                  {  CloseHandle(hComm);  // close the *old* COM in case it is open
                     hComm = NULL;
                  }
                 if( iComPortNr > 0 )     // OPEN the new selected COM port ..
                  { sprintf( szTemp, "COM%d", (int)iComPortNr );
                    hComm = SerialInit( szTemp, baudrate, bitsperbyte, stopbits, parity[0], 'N', 50, 50, 512, 512) ;
                    if (hComm == NULL)//INVALID_HANDLE_VALUE)
                     { MessageBox(0,"Comport already open or no comport available","Comport Error",0);   //error occured alert user about error
                     }
                    else
                     {
                       if (EscapeCommFunction(hComm,CLRDTR) == 0)              //clear RTS and DTR
                        { MessageBox(0,"Error Clearing DTR","Comport Error",0); //should never get here but, just in case..
                        }
                       if (EscapeCommFunction(hComm,CLRRTS) == 0)
                        { MessageBox(0,"Error Clearing CTS","Comport Error",0);
                        }
                     }
                  }
                EndDialog(hwndDlg, 0);
                break;

            }
            break;

        case WM_CLOSE:

            EndDialog(hwndDlg, 0);
            return TRUE;
         }

    return FALSE;
} // PttDlgProc()
/*-----------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
void RxTextWriter(
        T_WSQDecoder *pDecoder,
        int iChannel,  // [in] decoder index; 0=first RX text, 1=second, etc
        int iFlags,    // [in] bit combination of  WSQ_TEXT_FLAG_.... (from WSQ_Codec.h)
        char *pszText )
  // Called via function pointer from ReceiveAudioCallback() .
  // 'Writes' (appends) the received character into the receive text window .
  // >  (No need to mess with a separate worker thread since this is
  // >   already taken care of by PortAudio!)
  // The same applies to when the "received audio" is actually a WAVE FILE.
{
  unsigned char *cp, *cp2;
  // int i,text_length;
  unsigned char sz255TextForEditor[260];
  HWND hwndEditor = hCtl_RX[iChannel];
  DWORD dwLine, dwColumn, dwStartPos, dwEndPos;

  // Replace certain characters before appending the text to the edit control.
  // For example, windows doesn't handle single CARRIAGE RETURN or NEW LINE
  // characters properly (the appear as hollow boxes), so:
  cp = (unsigned char*)pszText;
  cp2= sz255TextForEditor;
  while( (*cp!=0) && ((cp2-sz255TextForEditor)<250) )
   { switch(*cp)
      { case 13:
        case 10:
           *cp2++ = '\r';
           *cp2++ = '\n';
           break;
        case 31:           // erroneously decoded character ->
           // *cp2++ = 0xBF;  // questionmark standing upside down (too ugly)
           *cp2++ = 0x1F;  // undefined character, printed as 'hollow rectangle'
           break;
        default:
           if( *cp>=32 && *cp<=127 )
            { *cp2++ = *cp;
            }
           else
            { sprintf( (char*)cp2, "{%d}", (int)(*cp) ); // stupid (char*) cast added to make PellesC happy (Borland didn't complain)
              *cp2 += strlen( (char*)cp2 ); // again, stupid cast added to eliminate warning from PellesC ("Assignment of unsigned char* to const char*" - yucc)
             }
           break;
      }
     ++cp;
   }
  *cp2 = '\0';


#if(0)
  // make sure caret stays at end of text
  text_length = SendMessage(hwndEditor, WM_GETTEXTLENGTH, 0, 0);
  SendMessage(hwndEditor, EM_SETSEL, text_length, text_length);

  // emulate keypresses using WM_CHAR (is this really legal?)
  cp = pszText;
  while( *cp != 0 )
   { SendMessage ( hwndEditor, WM_CHAR, *cp++, 0); // this removes the cursor
   }
  if( iDraggingArea==CLIENT_AREA_UNKNOWN )
   { SendMessage ( g_hwndMain, WM_MOUSEMOVE, 0, 0); // counteract it by forcing a mouse move
   }
#else  // modified by DL4YHF
  if( iFlags & WSQ_TEXT_FLAG_FINAL )
   { // The text is a 'finally decoded character' from INTERACTIVE decoder.
     // The last line (=the "Ticker") in the editor must be removed before,
     // and rebuilt after adding sz255TextForEditor :
     SetCaretToEndOfText( hwndEditor );
     GetTextCaretLineAndColumn( hwndEditor, &dwLine, &dwColumn );
     ReplaceTextLine( hwndEditor, dwLine, ""/*pszSource*/, FALSE/*cannot undo*/ );
     // Also delete the end-of-line (CR+LF) which separated the 'last line' from the older characters:
     GetTextSelection( hwndEditor, &dwStartPos, &dwEndPos );
     if( dwStartPos > 2 )
      { dwStartPos -= 2;
        SetTextSelection( hwndEditor, dwStartPos, dwEndPos );
      }
     strcat( (char*)sz255TextForEditor, "\r\n" );
     ReplaceSelectedText( hwndEditor, (char*)sz255TextForEditor, FALSE/*cannot undo*/ );
     // Append a new 'last line' (without CR+NL), with all characters
     // which may still be modified by dragging their symbols on the waterfall:
     WSQI_GetTextForTicker( pDecoder, iChannel, (char*)sz255TextForEditor );
     SetCaretToEndOfText( hwndEditor );
     ReplaceSelectedText( hwndEditor, (char*)sz255TextForEditor, FALSE/*cannot undo*/ );
     SetCaretToEndOfText( hwndEditor );
   }
  else // ! if( iFlags & WSQ_TEXT_FLAG_FINAL ) ...
   {
     // If the 'interactive' decoder (frequency shooter) is NOT active,
     // the character can be simply be appended to the end of the editor text:
     SetCaretToEndOfText( hwndEditor );
     ReplaceSelectedText( hwndEditor, (char*)sz255TextForEditor, FALSE/*cannot undo*/ );
     SetCaretToEndOfText( hwndEditor );
   }
#endif

  // Update the 'dot statistics'. Results are only displayed during FILE analysis,
  //  a test file with a long sequence of DOTS (in WSQ2) was used for testing.
  if( fAnalysingWaveFile )
   { cp = (unsigned char*)pszText;
     while( *cp != 0 )
      { ++nCharsEmittedDuringWaveAnalysis;
        if( *cp=='.' )
         { ++nDotsEmittedDuringWaveAnalysis;
         }
        ++cp;
      }
   } // end if( fAnalysingWaveFile )


} // end RxTextWriter()


//---------------------------------------------------------------------------
int InteractiveDecoderActive(void)
  // Returns nonzero if the 'interactive' WSQ decoder, aka 'Frequency Shooter',
  // is currently active. Actually, the return value also defines
  // into which of the receive text windows the interactively decoded text
  // should be printed: 1=1st RX-window, 2=2nd, 1+2=both .
  // Called from the mouse event handler to decide 'what to do' with a left-click.
{ int iWhich = 0;
  if( WSQ_Decoder.channel[0].fInteractive )
   {  iWhich |= 1;
   }
  if( (WSQ_Decoder.nChannels>=2) && WSQ_Decoder.channel[1].fInteractive )
   {  iWhich |= 2;
   }
  return iWhich; // 0=none of the decoders in 'interactive' mode, 1=1st, 2=2nd, 3=both
} // end InteractiveDecoderActive()


//---------------------------------------------------------------------------
void PrintToRxWindows( int iWhich,  // 1=1st RX-window, 2=2nd, 1+2=both
                char *pszFormat, ... )  // format string as for sprintf
{
  va_list parameter;              /* Parameter-Liste fr VA_... Macros */
  char szText[255];               /* Puffer fr formatierten String    */
  int i, text_length;
  HWND hwndEditor;

  va_start( parameter, pszFormat );
  vsprintf( szText, pszFormat, parameter );
  va_end(parameter);

  for(i=0; i<WSQ_MAX_DECODER_CHANNELS; ++i )
   { if( (iWhich & (1<<i) ) && (i<WSQ_Decoder.nChannels) )
      { hwndEditor = hCtl_RX[i];
        text_length = SendMessage(hwndEditor, WM_GETTEXTLENGTH, 0, 0);
        SetTextSelection( hwndEditor, text_length/*dwStartPos*/, text_length/*dwEndPos*/ );
        ReplaceSelectedText( hwndEditor, szText, FALSE/*fCanUndo*/ );
      }
   }

} // end PrintToRxWindows()


//---------------------------------------------------------------------------
void RedrawWaterfall(int nNewLines)  // -> InvalidateRect(), to redraw the waterfall in the WSQ GUI .
{                                    //    The actual 'painting' is done in OnWmPaint()

  s_nNewWaterfallLines = nNewLines;  // evaluated and cleared in OnWmPaint()

  // redraw the waterfall
  InvalidateRect( g_hwndMain,     // window handle
                  &graph_rect,    // redraw only the spectrum and waterfall area   // CONST RECT*, NULL => force redraw of entire client area
                  FALSE);

  // redraw the commands area (with parameters which need to be updated after each symbol from the decoder, along with the waterfall)
  InvalidateRect ( g_hwndMain,     // window handle
                  &commands_rect,  // redraw only the commands area   // CONST RECT*, NULL => force redraw of entire client area
                  FALSE);
} // end RedrawWaterfall()

//---------------------------------------------------------------------------
void RedrawAll(void)
{
  s_fRedrawAllWFLines = TRUE;
  InvalidateRect( g_hwndMain, NULL, FALSE );  // -> WM_PAINT -> OnWmPaint()
} // end RedrawAll()

//---------------------------------------------------------------------------
void RedrawWaterfallAndFreqScale(void)
{
  s_fRedrawAllWFLines = TRUE;
  InvalidateRect( g_hwndMain, &graph_rect, FALSE );  // -> WM_PAINT -> OnWmPaint()
} // end RedrawWaterfallAndFreqScale()



/*------------------------------------------------------------------------------
 *
 *      Receive audio stream callback function  (for Portaudio) .
 *
 *      All receive audio processing is done in this function!
 *
 *      (No need to mess with a separate worker thread since this is
 *      already taken care of by PortAudio!)
 */

static int ReceiveAudioCallback( void *inputBuffer,
                          void *outputBuffer,
                          unsigned long framesPerBuffer,
                          PaTimestamp outTime,
                          void *userData)
{
  int nSlicesRcvd; // .. or number of lines which must be added to the spectrogram
  float *pfltInput = (float*)inputBuffer;
#define L_TEST_SIGNAL 0 // TEST: replace input with a pure sinewave of known amplitude ? 1=YES 0=NO .
#if( L_TEST_SIGNAL )
  int iSample;
  float fltInput;
  float sampleRate = g_iNominalSamplingRateForSoundcard;
  static float fltTestSigPhase=0.0;
#endif // L_TEST_SIGNAL ?


  // ----------------------------------------------------------------------------
  // Argument inputBuffer may be NULL during start-up so...

  if (inputBuffer == NULL)
    return 0;

#ifdef AUDIO_IO_H     // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
  if( fUseAudioIOAsInput )  // using an Audio-I/O-DLL for input (not portaudio)
    return 0;       // ... so discard the input, do NOT call WSQ_DecodeAudio()
#endif // AUDIO_IO_H ?

  // Read input buffer, process data, etc ...  2014-02-20: moved to WSQ2_Codec.c
  if( ! fAnalysingWaveFile )
   {

#if( L_TEST_SIGNAL )
    for(iSample=0; iSample<framesPerBuffer; ++iSample )
     {
       fltInput = sin( 2.0 * 3.141592654 * fltTestSigPhase );
       fltTestSigPhase += ( 1001.12345678/*Hz*/ / sampleRate );
       if( fltTestSigPhase > 1.0 )
        {  fltTestSigPhase -= 1.0;
        }
       pfltInput[2*iSample] = fltInput;
#     if( CHANNELS_PER_SAMPLE==2 )
       pfltInput[2*iSample+1] = 0.0;
#     endif
     }
#endif // L_TEST_SIGNAL ?


     if( fRecordingWaveFile && g_RecordedWaveFile.OpenedForWriting )
      { WaveIO_WriteSamples_Float( &g_RecordedWaveFile, // app-layer output, using NORMALIZED floats
           pfltInput, framesPerBuffer,
           CHANNELS_PER_SAMPLE );
      }
     WSQ_Decoder.nLinesVisibleInSpectrogram = Layout_iWfallHeight;// <- important for the INTERACTIVE decoder

     nSlicesRcvd = WSQ_DecodeAudio( // here: for audio received from PortAudio (i.e. soundcard)
        &WSQ_Decoder,  // [in] everything the WSQ decoder needs to know :)
        pfltInput,     // [in] an array of samples (pointer to float)
        framesPerBuffer,     // [in] number of samples to process (what PA calls a 'frame' is a single SAMPLE POINT with 1 or 2 channels)
        CHANNELS_PER_SAMPLE, // [in] nChannelsPerSample, originally TWO(!) channels, even though only the 1st channel was actually used
        (float)g_iNominalSamplingRateForSoundcard,  // [in] precise sampling rate from the input [Hz]
        RxTextWriter); // [in] callback to write received characters into the receive text editor

     if( nSlicesRcvd>0 )  // number of lines to be added to the waterfall display
      { RedrawWaterfall(nSlicesRcvd);   // -> InvalidateRect(), only once per call
      }
   }
  return 0; // a non-zero return will stop the stream
} // ReceiveAudioCallback()


//---------------------------------------------------------------------------
void ShowAnalysedFileStatistics(void)
{
  if( nCharsEmittedDuringWaveAnalysis > 0 )
   { PrintToRxWindows( 1 | 2, "\r\nStatistics: %d chars decoded; %d dots = %d %% .\r\n",
        (int)nCharsEmittedDuringWaveAnalysis,
        (int)nDotsEmittedDuringWaveAnalysis,
        (int)((100*nDotsEmittedDuringWaveAnalysis) / nCharsEmittedDuringWaveAnalysis) );
     // Grep for references to "WSQ_dots_at_-28dB_SNR_synthetic.wav" in WSQ_Codec.c
     //  to find out for what this 'dot character statistic' was used :
     //  The 'dot percentage' was actually a quality indicator.
   }
} // end ShowAnalysedFileStatistics()


//---------------------------------------------------------------------------
void ContinueWaveFileAnalysis(void)
{
  // To keep things simple, pass the samples from the wave file to the decoder
  // in the same chunk size (FramesPerBuffer) as for real-time processing.
  // Must not be more than 1/16th of the FFT size !
  // Because for various reasons the timer shouldn't run faster than a 50 ms interval,
  // thus approx. 32000 samples * 50 / 1000 = 1600 samples would be processed
  // per timer interval, thus WSQ_DecodeAudio() is called multiple times below.
  // To analyse wave files much faster than in real time, we use larger chunks here:
# undef  L_SAMPLES_PER_CHUNK
# define L_SAMPLES_PER_CHUNK 32768
  int   nSamplesRead, nSamples2, iSample;
  int   nSlicesRcvd = 0;
  float fltChunk[L_SAMPLES_PER_CHUNK];
  float sampleRate;
  if( g_AnalysedWaveFile.OpenedForReading )
   { sampleRate = WaveIO_GetSampleRate(&g_AnalysedWaveFile);  // be prepared for NON-32000 !
     nSamplesRead = WaveIO_ReadSampleBlocks( &g_AnalysedWaveFile,
           0,   // channel_nr for 1st destination block
           L_SAMPLES_PER_CHUNK, // nr_samples (per destination block, length of destination arrays)
           fltChunk,            // 1st destination block (~"LEFT" audio channel)
           NULL,                // 2nd destination block (~"RIGHT"), may be NULL
           NULL,0);  // [out,optional] extra string with user data for wave.rd_info
     if( nSamplesRead > 0 )
      { iSample = 0;
        while( iSample<nSamplesRead )
         { nSamples2 = nSamplesRead - iSample;
           if( nSamples2 > 256 )
            {  nSamples2 = 256;
            }
           nSlicesRcvd += WSQ_DecodeAudio( // here: for audio read from an AUDIO FILE (*.wav)
             &WSQ_Decoder,        // [in] everything the WSQ decoder needs to know :)
             &fltChunk[iSample],  // [in] an array of samples (pointer to float)
             nSamples2,           // [in] number of samples to process
             1,                   // [in] nChannelsPerSample, here: only ONE channel
             sampleRate,          // [in] precise sampling rate from the input [Hz]
             RxTextWriter);       // [in] callback to write received characters into the receive text editor
           iSample += nSamples2;
         }
      } // end if( nSamplesRead > 0 )
     if( nSamplesRead < L_SAMPLES_PER_CHUNK )  // obviously reached the end of the wave file..
      { if( fDecodeWaveStatistics ) // show character statistics (used during tests) :
         { ShowAnalysedFileStatistics();
         }
        if( fDecodeWaveFileInLoop )
         { // Re-initialize the decoder, to clear averages, integrators, and similar:
           WSQ_BeginReception( &WSQ_Decoder ); // prepares / clears all decoders before an rx-over
           nCharsEmittedDuringWaveAnalysis = nDotsEmittedDuringWaveAnalysis = 0;

           // Wind back to the 1st sample in the file.
           if( ! WaveIO_WindToSampleIndex( &g_AnalysedWaveFile, 0/*sample index*/ ) )
            { // Wind him up ? He can't stop, he keeps on going round.... NO, he doesn't !
              // There's something wrong with the file, or the file reader, so close it:
              WaveIO_CloseFile( &g_AnalysedWaveFile );
              PrintToRxWindows( 1 | 2, "\r\nCould not wind back file '%s'\r\n", g_AnalysedWaveFile.sz255FileName );
            }
           else
            { PrintToRxWindows( 1 | 2, "\r\n--- Replay '%s' ---\r\n", g_AnalysedWaveFile.sz255FileName );
            }
         }
        else // no loop -> finished 'audio file analysis'
         { WaveIO_CloseFile( &g_AnalysedWaveFile );
         }
      }
   } // end if( g_AnalysedWaveFile.OpenedForReading )
  if( ! g_AnalysedWaveFile.OpenedForReading )
   { fAnalysingWaveFile = FALSE;  // done !
   }
  if( nSlicesRcvd>0 )  // add new lines to the waterfall display ?
   { RedrawWaterfall(nSlicesRcvd);   // -> InvalidateRect(), only once per call
   }

} // ContinueWaveFileAnalysis()

//---------------------------------------------------------------------------
#ifdef AUDIO_IO_H // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
void WSQ_ReadAndProcessSamplesFromAudioIO(void)
{ // Peridically called from the timer event handler (approx. 50 ms intervals)
  // when the input is read from an audio-I/O-DLL (instead of a soundcard) .
  // At 50 ms intervals, 32000 samples per seconds, and 2 channels per sample,
  // 3200 single floats must be processed per call. But since the timer interval
  // may be slower, and to prevent buffer overflows, use a 16384-element array here.
  // (almost the same as for WAVE FILE processing, which also PULLS OUT the samples
  //  from the driver instead of getting them PUSHED IN as with Portaudio)
# undef  L_SAMPLES_PER_CHUNK
# define L_SAMPLES_PER_CHUNK 16384
  int   nSamplesRead, nSamples2, iSample;
  int   nSlicesRcvd = 0;
  long  i32Result;
  float fltChunk[L_SAMPLES_PER_CHUNK];
  T_ChunkInfo chunkInfo;
  float fltSampleRate;
  T_WSQDecoder *pDecoder = &WSQ_Decoder;

  ChunkInfo_Init( &chunkInfo );
  i32Result = AIO_Host_ReadInputSamplePoints(
                 &AIO_Host, // [in,out] DLL-host instance data
                 fltChunk,  // [in] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
                 L_SAMPLES_PER_CHUNK, // [in] number of SAMPLE POINTS(!) to read
                 1,         // [in] number of samples PER SAMPLE POINT (may be less than provided by source)
                 AIO_TIMEOUT_NON_BLOCKING, // [in] max timeout in milliseconds, or AIO_TIMEOUT_NON_BLOCKING
                 &chunkInfo,// [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h
                 NULL);     // [out,optional] "been-waiting-flag" (32-bit integer)
  if( i32Result >= AIO_NO_ERROR ) // successfully read another chunk of samples
   {                              // from the Audio-I/O- or ExtIO-DLL ..
     nSamplesRead = i32Result;
     fltSampleRate = chunkInfo.dblPrecSamplingRate;
     if( fltSampleRate <= 0 ) // no valid sampling rate in chunk-info ->
      { fltSampleRate = pDecoder->fltInputSamplingRate; // use the previous one
      }
     // Instead of starting/stopping the remote receiver (or whatever feeds the DLL),
     // samples acquired from the Audio-I/O-DLL (or Winrad-compatible ExtIO-DLL)
     // are simply DISCARDED here while *NOT RECEIVING* :
     if( RXbutton )
      {
        if( nSamplesRead > 0 )
         { iSample = 0;
           while( iSample<nSamplesRead )
            { nSamples2 = nSamplesRead - iSample;
              if( nSamples2 > 256 )
               {  nSamples2 = 256;
               }
              nSlicesRcvd += WSQ_DecodeAudio(  // here: for audio received from an Audio-I/O-DLL
                pDecoder,            // [in] everything the WSQ decoder needs to know :)
                &fltChunk[iSample],  // [in] an array of samples (pointer to float)
                nSamples2,           // [in] number of samples to process
                1,                   // [in] nChannelsPerSample, here: only ONE channel
                fltSampleRate,       // [in] hopefully precise sampling rate [Hz]
                RxTextWriter);       // [in] callback to write received characters into the receive text editor
              iSample += nSamples2;
            }
         } // end if( nSamplesRead > 0 )
      } // end if < WSQ in 'RX' state > ?
   } // end if < successfully read another block of samples from the interface-DLL >
  if( nSlicesRcvd>0 )  // add new lines to the waterfall display ?
   { RedrawWaterfall(nSlicesRcvd);  // -> InvalidateRect(), only once per call
     // Typically, a block of samples received via Audio-I/O-DLL
     // resulted in FOUR new waterfall lines painted here.
   }

} // end WSQ_ReadAndProcessSamplesFromAudioIO()
#endif // AUDIO_IO_H ?

//---------------------------------------------------------------------------
#ifdef AUDIO_IO_H // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
void WSQ_StartInputFromAudioIoDLL(void)
{
  double dblSampleRate;
  INT32 i32Result;

  fUseAudioIOAsInput = TRUE;   // *WANT* to use the DLL as an audio source now, but:
  fAudioInputAlive   = FALSE;  // the external source may not be able to send yet !
  iAudioInputAliveTimer = 1000 / TIMER1_INTV_MS; // start alive-checking timer [1 second]
  i32Result = AIO_Host_OpenAudioInput(  // Includes "Start()" !!
     &AIO_Host, // [in,out] DLL-host instance data
     NULL,      // [in] char *pszAudioStreamID; 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE
                // (here: optional, use NULL to tap "the first available source")
     (char*)g_szClassName, // [in] char *pszAudioReaderID; short but unique name which identifies
                // the audio reader. This name may appear on the "control panel" / "patch panel"
                // of certain Audio-I/O DLLs .
     g_iNominalSamplingRateForSoundcard, // [in] "wanted" sample rate, often IGNORED (reasons below..)
                // Usually dictated by the FEEDING SOURCE;
                //   the sampling rate is NOT resampled by the DLL !
                //   dblWantedSampleRate can be used, however, for hardware drivers
                //   (like soundcard or SDR drivers) if the hardware supports
                //   different 'native ' sample rates.
     &dblSampleRate, // [out(!)] realized sample rate,
                // usually dictated by the FEEDING SOURCE;
                // the sampling rate is NOT resampled by the DLL !
     CHANNELS_PER_SAMPLE, // [in] number of channels PER SAMPLE POINT (same as for portaudio)
     2000,      // [in] iTimeout_ms; max timeout in milliseconds for THIS call
                // (may have to wait for a short time, for example when...
                //   - the I/O lib needs to communicate with an external device (to open it)
                //   - the I/O lib needs to wait until the 'writer' (=the other side)
                //     has called AIO_Host_OpenAudioOutput() and set the sampling rate)
     AIO_OPEN_OPTION_NORMAL, // [in] bit combination of AIO_OPEN_OPTION_... flags
     (T_ChunkInfo*)NULL);    // [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL
  if( i32Result < 0 )
   { // Something went wrong when trying to open the Audio-I/O-DLL for input..
     sprintf( WSQ_sz255LastError, "AudioIO::%s", AIO_Host_ErrorCodeToString( &AIO_Host, i32Result) );
     // ex: fUseAudioIOAsInput = FALSE;  // do *NOT* use it as an audio source !
     // Modified 2014-02-23 : Try again later, maybe the user has not launched
     // the external 'audio source' (for example Spectrum Lab) yet .
     // Keep fUseAudioIOAsInput=TRUE but fAudioInputAlive=FALSE;
     // which means "WANT to receive but the source isn't ready yet; waiting.."
   }
  else // AIO_Host_OpenAudioInput() was successful ...
   { fAudioInputAlive = TRUE;   // Ok, may begin to read audio samples from the DLL
   }
} // WSQ_StartInputFromAudioIoDLL()

void WSQ_StopInputFromAudioIoDLL(void)
{
  AIO_Host_CloseAudioInput( &AIO_Host ); // .. but don't unload the DLL yet !
  fAudioInputAlive = FALSE;
} // WSQ_StopInputFromAudioIoDLL()

void WSQ_StartOutputToAudioIoDLL(void)
{
} // WSQ_StartOutputToAudioIoDLL()

void WSQ_StopOutputToAudioIoDLL(void)
{
  AIO_Host_CloseAudioOutput( &AIO_Host );
} // WSQ_StopOutputToAudioIoDLL()

//---------------------------------------------------------------------------
void WSQ_ShowControlPanelOfAudioIoDLL(void)
{  AIO_Host_ShowControlPanel( // NON-MODAL (not blocking) !
                           &AIO_Host, // [in] DLL-host instance data
                          g_hwndMain, // [in] handle of the host's window ("parent")
                &hwndAIOControlPanel, // [out] handle of the DLL's control panel
            AIO_DISPLAY_SHOW_WINDOW,
                                NULL, // [in,optional] pvUserDataForCallback
                               NULL); // [in,optional] user callback function
   // Note: Opening the DLL's native control panel if already open won't hurt.
   // Even when opened from another application which also hosts the SAME DLL.
   // So far, only tested with DL4YHF's "in_AudioIO.dll" but not with ExtIO !
} // end WSQ_ShowControlPanelOfAudioIoDLL()
#endif // AUDIO_IO_H ?

//---------------------------------------------------------------------------
char *WSQ_GetTypeAndStatusOfCurrentInputDevice(void) // used somewhere in the menu
{
  if( fAnalysingWaveFile )
   { if( g_AnalysedWaveFile.OpenedForReading )
      {  return ("Audio file, not open");
      }
     else
      { return ("Audio file");
      }
   }
#ifdef AUDIO_IO_H     // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
  else if( fUseAudioIOAsInput )  // currently using an Audio-I/O-DLL for input (not portaudio)
   { if( fAudioInputAlive )
      { return ("Audio I/O DLL [alive]");
      }
     else  // DLL successfully loaded, but the source (sender) isn't alive yet:
      { return ("Audio I/O DLL [waiting]");
      }
   }
#endif // AUDIO_IO_H ?
  else // guess the input device is a SOUNDCARD..
   { return ("Soundcard");
   }

} // end WSQ_GetTypeAndStatusOfCurrentInputDevice()

//---------------------------------------------------------------------------
// For transmission ....
//---------------------------------------------------------------------------

char TxCharReader(void)
  // Called via function pointer from SendAudioCallback() -> WSQ_GenerateAudio()
  // whenever the WSQ encoder needs the next character for transmission.
  // If there's nothing to send at the moment, a zero byte is returned.
  // Also implements the optional automatic 'TX to RX switch' function.
{
  int  len;
  char cResult = '\0';

  // For *.c (not *.cpp) modules, Borland doesn't allow declaring locals,
  // and especially not declaring local array with size defined at RUNTIME.
  char buf[65536];  // DL4YHF 2014-02-15 : Not sure how long the text will be.. > 64 kByte ?


  len = GetWindowTextLength(hCtl_TX) + 1; // add 1 for the final NULL
  if (len > 1)
   {
     if( len>sizeof(buf) )  // Houston we've got a problem !
      { return 0;           // however, transmitting 64k of chars would take AGES..
      }
     GetWindowText(hCtl_TX, buf, len);
     //... do stuff with text ...
     cResult = buf[0];  // get first character in buffer


     SetWindowText(hCtl_TX, &buf[1] );   // move the text to send along one as we remove the first character
     SendMessage(hCtl_TX, EM_SETSEL, len, len); //put caret at end of text

     // echo the sent character to the receive screen

     if(buf[0]==13) goto NoExtraLF; // echo only the CR from received text

     SendMessage ( hCtl_RX[0], WM_CHAR, buf[0], 0);  // This hides the cursor so
NoExtraLF:
     if( iDraggingArea==CLIENT_AREA_UNKNOWN )
      { SendMessage ( g_hwndMain, WM_MOUSEMOVE, 0, 0);    // counteract it by forcing a mouse move
      }
     iAutoSwitchTxRxState = 1;  // 'got something to send' -> initial longer idle time is over

   } // end if (len>1)
  else // Transmit buffer is completely empty.  Switch back from TX to RX ?
   { if( fAutoSwitchTxToRx )
      { // send CR+LF (for the remote receiver), send IDLE char, then stop transmission.
        switch( iAutoSwitchTxRxState )
         { case 0:  // initial state, didn't send ANYTHING TYPED BY OPERATOR yet
               // In this state, the operator has a few more seconds to think
               // before the program switches back to receive .
               // During that time, either the 'diddle'/staircase/zero-symbol may be sent.
               if( ++iAutoSwitchTxRxTimer > 10 )  // over 20 'idle' characters (or symbols) sent
                { // -> time to turn off, maybe the operator fell asleep
                  ++iAutoSwitchTxRxState;  // next state
                }
               break;
           case 1:
               cResult = 13;           // send carriage return
               ++iAutoSwitchTxRxState; //  next state for next time..
               break;
           case 2:
               cResult = 0;            // send IDLE character
               ++iAutoSwitchTxRxState;
               break;
           default:
               if( iCWidAlways && (iCWidState==0) )
                {  iCWidState = 1;
                   iCWTxCharIndex=0; // index into sz255CWid[] during CW transmission
                   CW_InitGenerator( &CW_Generator, WSQ_Encoder.fltLowestToneFreq, fltCWSecondsPerDot );
                }
               if( iCWidState == 1 ) // 0=passive, 1=request to send CW ID, 2=sending CW ID
                {  iCWidState = 2;
                   iCWTxCharIndex=0; // index into sz255CWid[] during CW transmission                
                }
               else
                {  fSwitchFromTxToRx = TRUE; // set in Audio-callback (TX), polled in main loop
                }
               break;
         }
      }
   }

  return cResult;

} // end TxCharReader()

//---------------------------------------------------------------------------
char TxCharReaderForCW(void)
  // Called via function pointer from SendAudioCallback() -> CW_GenerateAudio()
  // whenever the CW (Morse) generator needs the next character for transmission.
  // If there's nothing to send at the moment, a zero byte is returned.
{
  if( iCWTxCharIndex < (int)strlen(sz255CWid) )
   { return sz255CWid[iCWTxCharIndex++];
   }
  else
   { return 0;
   } 

} // end TxCharReaderForCW()

/*------------------------------------------------------------------------------
 *
 *      Transmit audio stream callback function (for Portaudio) .
 *
 *      All audio processing is done in this function!
 *
 *      (No need to mess with a separate worker thread since this is
 *      already taken care of by PortAudio!)
 */

static int SendAudioCallback (void *inputBuffer,
                          void *outputBuffer,
                          unsigned long framesPerBuffer,
                          PaTimestamp outTime,
                          void *userData)
{
  int i,n;
  if( fSwitchFromTxToRx )    // want to switch back to RX, but main thread didn't notice yet
   { // -> fill the buffer with 'silence'
     n = framesPerBuffer * CHANNELS_PER_SAMPLE;
     for(i=0; i<n; ++i)
      { ((float *)outputBuffer)[i] = 0.0;
      }
   }
  else if( iCWidState == 2 ) // 0=passive, 1=request to send CW ID, 2=sending CW ID
   { if( ! CW_GenerateAudio( &CW_Generator,
        (float *)outputBuffer,  // // [out] an array of samples (floating point)
        framesPerBuffer,        // [in] number of samples to be produced now
        CHANNELS_PER_SAMPLE,    // [in] number of channels per sample point ("frame")
        (float)g_iNominalSamplingRateForSoundcard,       // [in] precise sampling rate from the output [Hz]
        TxCharReaderForCW) )  // [in] callback to read the next character from the transmit text editor
      {  // end of the CW-ID-transmission :
        iCWidState = 0;
        fSwitchFromTxToRx = TRUE; // set in Audio-callback (TX), polled in main loop
      }
   }
  else
   { // Produce WSQ audio stream ...  2014-02-20: moved to WSQ2_Codec.c
     if( WSQ_GenerateAudio( &WSQ_Encoder,
        (float *)outputBuffer,  // // [out] an array of samples (floating point)
        framesPerBuffer,        // [in] number of samples to be produced now
        CHANNELS_PER_SAMPLE,    // [in] number of channels per sample point ("frame")
        (float)g_iNominalSamplingRateForSoundcard,       // [in] precise sampling rate from the output [Hz]
        TxCharReader) )  // [in] callback to read the next character from the transmit text editor
      { // if the 'generator' has begun a new tone (symbol)..

        // display the commands area
        InvalidateRect ( g_hwndMain,   // window handle
           &commands_rect,  // redraw only the commands area   // CONST RECT*, NULL => force redraw of entire client area
           FALSE);
      }
   }

  return 0; // a non-zero return will stop the stream
} // SendAudioCallback()


//---------------------------------------------------------------------------
int SigNoiseToMeterY( float fltSig_to_Noise_dB )
{
  float f;
  int   y;
  // The original meter SNR range was approx. +5 .. -32 dB, thus :
  f = (float)( Layout_yMeterBottom - Layout_yMeterTop) / (float)(32+5);
  y = Layout_yMeterTop + (int)( ( 5.0 - fltSig_to_Noise_dB ) * f );
  LimitInteger( &y, Layout_yMeterTop, Layout_yMeterBottom );
  return y;
} // end SigNoiseToMeterY()

float Voltage_to_dBfs( float fltNormalizedVoltage/*0..1*/ )
{
  if( fltNormalizedVoltage >= 1e-10 )
   {  return 20.0 * log10( fltNormalizedVoltage ); // ->  -200 to 0 dB "over" full scale
   }
  else
   {  return -200.0; // dB "over" full scale
   }
} // end Voltage_to_dBfs()

void CreateBitmapForWaterfall(void)   // Layout_iWfallWidth,Layout_iWfallHeight -> hbmScroll,g_bmiScrollingWF
{
#if( USE_DIB_SECTION )  // use "DIB" = "device-independent bitmap" ?
  // > CreateDIBSection() creates a device-independent bitmap (DIB) that applications
  // > can write to directly. The function gives you a pointer to the location
  // > of the bitmap's bit values. You can (..) let the operating system allocate
  // > the memory for the bitmap.
  memset( &g_bmiScrollingWF, 0, sizeof(g_bmiScrollingWF) );
  g_bmiScrollingWF.bmiHeader.biSize = sizeof( g_bmiScrollingWF.bmiHeader );
  g_bmiScrollingWF.bmiHeader.biWidth  = Layout_iWfallWidth;  // checked later, in OnWmPaint(), because
  g_bmiScrollingWF.bmiHeader.biHeight = Layout_iWfallHeight; // the bitmap size may have to be increased after resizing !
  g_bmiScrollingWF.bmiHeader.biPlanes = 1;
  g_bmiScrollingWF.bmiHeader.biBitCount = 32; // use FOUR, not THREE bytes per pixel !
  g_bmiScrollingWF.bmiHeader.biCompression = BI_RGB;
  hbmScroll = CreateDIBSection(
                hdc2, // HDC hdc, handle to device context
   &g_bmiScrollingWF, // CONST BITMAPINFO *pbmi, pointer to structure containing bitmap size, format, and color data
      DIB_RGB_COLORS, // UINT iUsage, color data type indicator: RGB values or palette indices
   (void*)&g_pdwWfallPixelData, // VOID *ppvBits, pointer to variable to receive a pointer to the bitmap's bit values
                NULL, // HANDLE hSection, optional handle to a file mapping object
                  0); // DWORD dwOffset offset to the bitmap bit values within the file mapping object
  if( hbmScroll != NULL )
   { g_dwWfallPixelStride = g_bmiScrollingWF.bmiHeader.biWidth;
     g_dwWfallPixelsYMax  = g_bmiScrollingWF.bmiHeader.biHeight-1;
   }
  else
   { g_dwWfallPixelStride = 0;
     g_dwWfallPixelsYMax  = 0;
   }
#else  // ! USE_DIB_SECTION  :
  hbmScroll = CreateCompatibleBitmap (hdc2, MAX_WATERFALL_BITMAP_WIDTH, MAX_WATERFALL_BITMAP_HEIGHT);
#endif //   USE_DIB_SECTION  ?
} // end CreateBitmapForWaterfall()


#define SPECTRUM_MIN_DB -110.0
#define SPECTRUM_MAX_DB 0.0

//---------------------------------------------------------------------------
LONG OnWmPaint (HWND hWnd)
  // Reaction to the WM_PAINT message  (some of which is "borrowed" from SAQrx,
  //  other parts based on DL4YHF's Spectrum Lab) .
  // This monster function produces all 'graphic' output of the WSQ main window.
{
  double freq, fstep, substep;
  int pixel_brightness, red, green, blue;
  int i,n,x,y,w,h,x1,y1,x2,y2,x_occupied,iDecoderIndex;
  int iLine, nLinesToUpdate;
  BOOL first_pixel;
  T_WSQEncoder *pEncoder = &WSQ_Encoder;
  T_WSQDecoder *pDecoder = &WSQ_Decoder;
  T_WSQDecoderChannel *pChannel;
  T_SpecBufEntry      *pSpectrum;
  T_WSQI_Point  *pWSQIPoint;
  /* default TONE SPACING taken from "Decoder1" settings : */
  double dblToneSpacing_Hz = (WSQ_Decoder.channel[0].iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */
  float flt, fltFreqToBinIdx, fltPeak, fltPeak_dBfs;
  int   iBinIdx, iPrevBinIdx;
  long  i32SpectrumIndex;
  char  szLabel[80];    // also used for the yellow info text
#if( USE_DIB_SECTION )  // use "DIB" = "device-independent bitmap" ?
  DWORD *pdwPixel;
#endif

  HDC hDC;
  HDC hdcMem;  // off-screen device context (for painting "in memory")
  HPEN      hPen=NULL, hOldPen=NULL;
  PAINTSTRUCT ps;
  // ex: BITMAPINFO DibInfo;

  // ex: HANDLE  vertFont;
  // ex: HANDLE  hOldFont;
  HBRUSH  hSolidBrush, hOldBrush;
  HBRUSH  hBrush_SmeterGreen;
  HBRUSH  hOldSmeterBrushGreen;
  HANDLE  SmeterFont,VerySmallFont,hOldFont4Mem; // , hOldFont4WFall;
  HBITMAP hbmMem, hbmOld;

  hDC = BeginPaint (hWnd, &ps);

  // Get the size of the program window's client area:
  //
  // NOTE! The coordinates returned by GetClientRect() are:
  //
  //   rctClientArea.left == rctClientArea.top == 0 (always)
  //
  //   rctClientArea.right = WIDTH of client area
  //   (rightmost visible X == rctClientArea.right-1
  //
  //   rctClientArea.bottom = HEIGHT of client area
  //   (bottommost visible Y == rctClientArea.bottom-1
  //

  // set up limited rectangle to invalidate:
  graph_rect.left   = 0;
  graph_rect.top    = Layout_y2 + 2;   // ex: 362 (Layout_y2 = 360)
  graph_rect.right  = Layout_xEnd - 5; // ex: 820;
  graph_rect.bottom = Layout_y5;       // ex: 460;

  // set up limited rectangle to invalidate commands_rect =
  commands_rect.left   = Layout_x2+1;  // ex: 261;
  commands_rect.top    = Layout_y5+1;  // ex: 489;
  commands_rect.right  = Layout_x3-1;  // ex: 633;
  commands_rect.bottom = Layout_yEnd-5; // ex: 513;

  GetClientRect (hWnd, &rctClientArea);
  // In the original code (with fixed window size) :
  //  rctClientArea.right  = 823  (not 825),
  //  rctClientArea.bottom = 519


  // Create a compatible device context (DC) for drawing off-screen:

  hdcMem = CreateCompatibleDC (((LPPAINTSTRUCT)(&ps))->hdc);

  // Create a temporary off-screen bitmap to do the off-screen drawing on:
  hbmMem = CreateCompatibleBitmap (((LPPAINTSTRUCT)(&ps))->hdc,
                                   rctClientArea.right-rctClientArea.left,
                                   rctClientArea.bottom-rctClientArea.top);

  // Direct all drawing operations onto the off-screen bitmap:
  hbmOld = (HBITMAP) SelectObject (hdcMem, hbmMem);



  // We can now begin the drawing operations, first erase the background:
  FillRect (hdcMem, &rctClientArea, (HBRUSH) GetStockObject(DKGRAY_BRUSH));

  SetBkMode(hdcMem, TRANSPARENT);  //so that numbers don;t have white boxes around them

#if(0)
  vertFont = CreateFont(
          15,0,//   'height,width(default=0)
          900,0, //  'escapement(angle),orientation (1/10 degree)
          400, //_    'weight (default=0,normal=400,bold=700)
          FALSE, //_ 'Italic
          FALSE, //_ 'Underline
          FALSE, //_ 'StrikeThru
          ANSI_CHARSET, OUT_CHARACTER_PRECIS,// _
          CLIP_DEFAULT_PRECIS, PROOF_QUALITY,//_
          FIXED_PITCH | FF_MODERN,"Courier New" );
  hOldFont4Mem = SelectObject(hdcMem,vertFont);
  SetTextColor(hdcMem, 0x00F0C080);
  // TextOut (hdcMem, 10,450, "Hi there, I'm rotated by 90 degrees", 10);
  DeleteObject (SelectObject (hdcMem,hOldFont4Mem));
#endif // (0)

  VerySmallFont = CreateFont(
          9,0, //   'height,width(default=0)
          0,0, //  'escapement(angle),orientation (1/10 degree)
          400, //_    'weight (default=0,normal=400,bold=700)
          FALSE, //_ 'Italic
          FALSE, //_ 'Underline
          FALSE, //_ 'StrikeThru
          ANSI_CHARSET, OUT_CHARACTER_PRECIS,// _
          CLIP_DEFAULT_PRECIS, PROOF_QUALITY,//_
          FIXED_PITCH | FF_MODERN,"Courier New" );

  SmeterFont = CreateFont(
          13,0,//   'height,width(default=0)
          0,0, //  'escapement(angle),orientation (1/10 degree)
          400, //_    'weight (default=0,normal=400,bold=700)
          FALSE, //_ 'Italic
          FALSE, //_ 'Underline
          FALSE, //_ 'StrikeThru
          ANSI_CHARSET, OUT_CHARACTER_PRECIS,// _
          CLIP_DEFAULT_PRECIS, PROOF_QUALITY,//_
          FIXED_PITCH | FF_MODERN,"Courier New" );


  // Draw waterfall display box and "status bar" :
  SetTextAlign (hdcMem, TA_LEFT);
  SetTextColor(hdcMem, 0x00808080);

  hPen = CreatePen (PS_SOLID, 1, 0x00808080);
  hOldPen = (HPEN)SelectObject (hdcMem, hPen);

  x1 = Layout_xWfallLeft- WFALL_BORDER_WIDTH;
  y1 = Layout_yWfallTop - WFALL_BORDER_TOP;
  x2 = Layout_xWfallLeft + Layout_iWfallWidth + WFALL_BORDER_WIDTH;
  y2 = Layout_yWfallTop  + Layout_iWfallHeight+ WFALL_BORDER_BOTTOM;
  MoveToEx (hdcMem, x1, y1, NULL);
  LineTo (hdcMem,   x2, y1 );
  LineTo (hdcMem,   x2, y2 );  // box around the waterfall
  LineTo (hdcMem,   x1, y2 );
  LineTo (hdcMem,   x1, y1 );

  // Add frequency markers.  Modified by DL4YHF for flexible layout + freq-range.
  // Select a suitable frequency step for the ticks .. must depend on the frequency span, AND the display width.
  GetTextExtent( hdcMem, "1234 Hz", &w, &h );  // typical with ("1000 Hz") : 54 pixels
  w += 5;   // add 'minimum spacing' between labels
  flt = (float)w/*pixels*/ * (g_WfallFmax_Hz - g_WfallFmin_Hz) / Layout_iWfallWidth; // -> "Hertz per tick", but a very odd number
  if     ( flt <=  1.0 ) {  fstep =  1.0;   substep=0.25; }
  else if( flt <=  2.0 ) {  fstep =  2.0;   substep=0.5;  } // not 2.5 - no decimals in the frequency labels !
  else if( flt <=  5.0 ) {  fstep =  5.0;   substep=1.0;  }
  else if( flt <= 10.0 ) {  fstep = 10.0;   substep=2.5;  }
  else if( flt <= 25.0 ) {  fstep = 25.0;   substep=5.0;  }
  else if( flt <= 50.0 ) {  fstep = 50.0;   substep=10.0; }
  else if( flt <= 100.0) {  fstep = 100.0;  substep=25.0; }
  else if( flt <= 250.0) {  fstep = 250.0;  substep=50.0; }
  else if( flt <= 500.0) {  fstep = 500.0;  substep=100.0; }
  else                   {  fstep = 1000.0; substep=100.0; }
  freq  = g_WfallFmin_Hz;
  freq -= fmod( freq, fstep );
  x1 = Layout_xWfallLeft;
  x2 = Layout_xWfallLeft + Layout_iWfallWidth;
  y1 = Layout_yWfallTop  + Layout_iWfallHeight + WFALL_BORDER_BOTTOM;
  y2 = Layout_y5 - h - 2;
  x_occupied = x1 + 35;
  while( freq < g_WfallFmax_Hz ) // draw frequency markers (vertical lines + labels) ...
   { x = FreqToScreenX( freq );
     if( x>x1 && x<x2 )  // inside the frequency scale
      {
        MoveToEx (hdcMem, x, y1 , NULL);  // waterfall frequency markers
        LineTo   (hdcMem, x, y2 );
        if( (x>x_occupied) && (x+50)<x2 )  // enough clearance for a frequency label ?
         { sprintf( szLabel, "%d Hz", (int)(freq+0.5) );
           GetTextExtent( hdcMem, szLabel, &w, &h );  // typical with ("1000 Hz") : 54 pixels
           LeftAlignedTextOut( hdcMem, x-(w/2), y2, szLabel );
           x_occupied = x + w + 3/*pixels*/;  // just in case a label is 'surprisingly long'
         }
      }
     freq += fstep;
   }
  // Similar for the 'sub-steps' (with smaller ticks and no labels) :
  freq  = g_WfallFmin_Hz;
  freq -= fmod( freq, substep );
  while( freq < g_WfallFmax_Hz ) // draw frequency markers (vertical lines + labels) ...
   { x = FreqToScreenX( freq );
     if( x>x1 && x<x2 )  // inside the frequency scale
      { MoveToEx (hdcMem, x, y1 , NULL);  // waterfall sub-step markers
        LineTo   (hdcMem, x, y1+(y2-y1)/2 );
      }
     freq += substep;
   }



  // Put long box for "status bar" along the bottom for controls below
  MoveToEx (hdcMem, 5/*ex:5*/,               Layout_y5, NULL);
  LineTo   (hdcMem, Layout_xEnd-2/*ex:820*/, Layout_y5 );
  LineTo   (hdcMem, Layout_xEnd-2/*ex:820*/, Layout_yEnd-4/*ex:514*/ );
  LineTo   (hdcMem, 5/*ex:5*/, Layout_yEnd-4/*ex:514*/ );
  LineTo   (hdcMem, 5/*ex:5*/, Layout_yEnd-30/*ex:488*/ );         // frame around status bar

  MoveToEx (hdcMem, Layout_x6-4/*ex:758*/, Layout_y5 , NULL);  // separator between 'Tx' and 'Rx' button
  LineTo   (hdcMem, Layout_x6-4/*ex:758*/, Layout_yEnd-4/*ex:514*/ );

  MoveToEx (hdcMem, Layout_x5-4/*ex:696*/, Layout_y5 , NULL);  // separator between 'Pause' and 'Tx' button
  LineTo   (hdcMem, Layout_x5-4/*ex:696*/, Layout_yEnd-4/*ex:514*/ );

  MoveToEx (hdcMem, Layout_x4-4/*ex:634*/, Layout_y5 , NULL);  // separator before 'Pause' button
  LineTo   (hdcMem, Layout_x4-4/*ex:634*/, Layout_yEnd-4/*ex:514*/ );


  MoveToEx (hdcMem, Layout_x2/*ex:260*/, Layout_y5 , NULL);  // separator markers
  LineTo   (hdcMem, Layout_x2/*ex:260*/, Layout_yEnd-4/*ex:514*/ );


  // Clean up (intermediately)
  SelectObject (hdcMem, hOldPen);
  DeleteObject (hPen);

  // Added 2014-03-10 : If necessary, delete and re-create the bitmap for the scrolling waterfall:
  if(  (Layout_iWfallWidth  > g_bmiScrollingWF.bmiHeader.biWidth )
     ||(Layout_iWfallHeight > g_bmiScrollingWF.bmiHeader.biHeight) )
   {
     SelectObject(hdcScroll, hbmScrollOld); // restore the old whatever-it-is,
         // counterpart to 'hbmScrollOld = (HBITMAP)SelectObject(hdcScroll, hbmScroll)' :
     // > An application should always replace a new object with the original,
     // > default object after it has finished drawing with the new object.
     // At this point, hbmScroll is not "selected into" the decive context 'hdcScroll'...
     // so we should be able to delete both safely (the bitmap, and its device context):
     DeleteObject(hbmScroll);
     // Not necessary : DeleteDC(hdcScroll);
     // Create a NEW (larger) off-screen bitmap to do the off-screen drawing on waterfall display:
     CreateBitmapForWaterfall();   // Layout_iWfallWidth,Layout_iWfallHeight -> hbmScroll,g_bmiScrollingWF

     // Direct waterfall drawing operations onto the off-screen bitmap again:
     hbmScrollOld = (HBITMAP) SelectObject (hdcScroll, hbmScroll);

   } // end if < size of the scrolling waterfall bitmap must be increased >


  // Draw the spectrogram (waterfall, buffered in an array of spectra)
  hPen = CreatePen (PS_SOLID, 1, 0x00DDDDDD);//0x0000FF10L);
  hOldPen = (HPEN)SelectObject (hdcMem, hPen);

 // rPixelsPerdB = (float)(CURVEHEIGHT) / (SPECTRUM_MAX_DB-SPECTRUM_MIN_DB);

  SetBkMode(hdcScroll, TRANSPARENT);  // use transparent letters for the 'symbol numbers' on the waterfall
  SetTextColor(hdcScroll, C_RGB_RX1 );
  // How many lines of the waterfall must be updated ?
  nLinesToUpdate = s_nNewWaterfallLines;
  if( s_fRedrawAllWFLines )          // update the ENTIRE spectrogram (after resizing, etc) ?
   { nLinesToUpdate = Layout_iWfallHeight;
   }
  if( nLinesToUpdate > 0 )
   { s_nNewWaterfallLines = 0;
     // Scroll the (off-screen) waterfall image down by as many lines as required:
     if( nLinesToUpdate < Layout_iWfallHeight )
      { BitBlt(  // scroll within the waterfall bitmap :
         hdcScroll, // << destination (device context)
         0,         // x-coordinate of destination rectangle's upper-left corner
         nLinesToUpdate,      // y-coordinate of destination rectangle's upper-left corner
         Layout_iWfallWidth,  // width of the DESTINATION rectangle in pixels
         Layout_iWfallHeight-nLinesToUpdate, // height " " " " " "
         hdcScroll, // << source (device context)
         0, 0,      // coordinate of the source rectangle's upper-left corner
         SRCCOPY);
      }

     // Plot another spectrum (usually one, but maybe more during file-analysis)
     //      as a line of pixels in the waterfall,
     //      with the most recent (iLine=y=0) at the top.
     //      To avoid overwriting the text labels, the drawing sequence is reversed:
  #if(0)
     for(iLine=0; iLine < nLinesToUpdate; ++iLine )
  #else
     for( iLine=nLinesToUpdate-1; iLine>=0; --iLine )
  #endif
      { pSpectrum = WSQ_GetSpectrumFromBuffer( pDecoder, iLine ); // result may be NULL when "not available" !
        if( pSpectrum != NULL )
         { // Since neither the width of the window, not the FFT size is fixed anymore,
           // the ratio between 'screen pixels' and 'FFT frequency bins' is arbitrary.
           //
           // Principle used below:
           // Running along the frequency axis, the program looks into
           //    a RANGE(!) of FFT bins(!) to get the color of a pixel.
           // There may be less FFT bins than pixels, there may be more,
           // and only if we're lucky there is a 'perfect match' (one pixel per bin).
           if( pSpectrum->fltFreqBinWidth_Hz > 0.0 )
            { fltFreqToBinIdx = 1.0 / pSpectrum->fltFreqBinWidth_Hz;
            }
           else // avoid div-by-zero..
            { fltFreqToBinIdx = 0.0;
            }
           iPrevBinIdx = -1;
           for (x = 0; x < Layout_iWfallWidth; ++x)
            {
              // Convert the x-coord to a frequency (in Hertz):
              freq = ScreenXoffsetToFreq( x );
              // Convert the pixel's center frequency to an FFT frequency bin index:
              iBinIdx = (int)( 0.5 + (freq - pSpectrum->fltCenterFreqOf1stBin) * fltFreqToBinIdx);
              if( iBinIdx>0 && iBinIdx<pSpectrum->nUsedBins )
               {
                 if( iPrevBinIdx < 0 )
                  {  iPrevBinIdx = iBinIdx-1;  // 1st loop
                     if( iPrevBinIdx < 0 )
                      {  iPrevBinIdx = 0;
                      }
                  }
                 if( iPrevBinIdx < iBinIdx ) // Is this a new FFT-bin at all ?
                  { fltPeak = pSpectrum->fltFreqBins[++iPrevBinIdx];
                    while( iPrevBinIdx < iBinIdx ) // MORE THAN ONE BIN per pixel : get the peak..
                     { flt = pSpectrum->fltFreqBins[++iPrevBinIdx];
                       if( flt>fltPeak )
                        {  fltPeak = flt;
                        }
                     }
                    // Convert the peak (~voltage) into a pixel colour .
                    //  'iWFBrightness_Percent'  = trackbar position, ranging from 0 to 100 .
                    //  'fltPeak' = FFT bin amplitude (proportional to a voltage),
                    //         normalized to 0....1 for a full-swing sinewave w/o AGC .
                    //  'pixel_brightness' = 0..255 .
                    //  The factor below was empirically found to have a similar
                    //  brightness (with the same scroller position) as in ZL2AFP's original.
                    //  2014-03-04 : Now using a logarithmic measure for more dynamic range,
                    //               and two separate sliders to adjust the displayed magnitude range.
                    fltPeak_dBfs = Voltage_to_dBfs( fltPeak/*0..1*/ ); // ->  -120 to 0 dB, typically noise -60 dB "over" full scale
                    // How to "apply" the contrast and brightness percentage
                    //     to the logarithmized voltages (here: in dBfs) ?
                    // - 'Contrast' is a kind of GAIN adjustment, i.e. multiplied with the dB value
                    // - 'Brightness' is an OFFSET adjustment, i.e. added or subtracted from the brightness
                    // Meaningful dB range with 'min' contrast   (0  %) : 120 dB  ("full dynamic range visible)
                    // Meaningful dB range with 'max' contrast   (100%) :   3 dB  ("super-sensitive display")
                    // Average fltPeak_dBfs with AGC (in WSQ_Codec) enabled: approx. -70 dB,
                    //  thus -70 dB should give a usable waterfall display with both sliders set to 50 % .
                    // ex: pixel_brightness = (int) ( 2.55 * (float)iWFBrightness_Percent  + 0.1 * (float)iWFContrast_Percent * (fltPeak_dBfs + 60.0) );
                    flt = (  fltPeak_dBfs/*-200..0 dBfs; -70 dBfs with AGC enabled    */
                           + 2.0*(float)iWFBrightness_Percent/* adds 100 dBfs at 50%, sum = +30 */
                           - 30.0 /* result is ZERO for 'mid pixel brightness' */
                          ) // now MULTIPLY with the 'contrast' control value:
                        * ( 0.5 + 0.1 * (float)iWFContrast_Percent );
                    pixel_brightness = 127/* mid range of 0..255*/ + (int)flt;
                    LimitInteger( &pixel_brightness, 0, 255 );
                    switch( iWFColorPalette )
                     { case WF_PALETTE_BLUE :
                       default :  // Blue waterfall display
                          red   = pixel_brightness;  // -> 0..255 (for each colour component)
                          green = pixel_brightness;
                          blue  = pixel_brightness * 2;
                          break;
                       case WF_PALETTE_GREEN  : // Green waterfall display
                          red   = pixel_brightness;
                          green = pixel_brightness * 2;
                          blue  = pixel_brightness;
                          break;
                       case WF_PALETTE_SUNRISE: // 'sunrise' palette from Spectrum Lab
                          red   = SunriseColors[pixel_brightness] & 0xFF;
                          green =(SunriseColors[pixel_brightness] >> 8) & 0xFF;
                          blue  =(SunriseColors[pixel_brightness] >> 16) & 0xFF;
                          break;
                       case WF_PALETTE_LINRAD : // 'Linrad'-like palette (rainbow-like)
                          red   = LinradColors[pixel_brightness] & 0xFF;
                          green =(LinradColors[pixel_brightness] >> 8) & 0xFF;
                          blue  =(LinradColors[pixel_brightness] >> 16) & 0xFF;
                          break;
                     }
                    LimitInteger( &red,   0, 255 ); // just for safety..
                    LimitInteger( &green, 0, 255 );
                    LimitInteger( &blue,  0, 255 );

                  } // end if( iBinIndex > iPrevBinIdx )
                 else  // not a new FFT frequency bin index: use the same R,G,B colour as for the left neighbour
                  {
                  }

               } // end if( iBinIdx>0 && iBinIdx<pSpectrum->nUsedBins )
              else // pixel position is not covered by a valid FFT frequency bin:
               { red = green = blue = 0;
               }
              spectrogram_rgb[x].b[RGB_BYTE_RED]   = red;
              spectrogram_rgb[x].b[RGB_BYTE_GREEN] = green;
              spectrogram_rgb[x].b[RGB_BYTE_BLUE]  = blue;
              spectrogram_rgb[x].b[RGB_BYTE_ALPHA] = 0x00;
            } // end for x...

           // mark the peaks (measured symbol frequencies) in the spectrogram ?
           if( fShowSymbolMarkers )
            {
              for( iDecoderIndex=0; (iDecoderIndex<pDecoder->nChannels)
                                 && (iDecoderIndex<WSQ_MAX_DECODER_CHANNELS);
                 ++iDecoderIndex )
               { freq = pSpectrum->fltSymbolPeakFreq[iDecoderIndex];
                 x = FreqToScreenXoffset( freq );
                 if( x>=0 && x<FFT_MAX_FREQ_BINS )
                  { // Set pixel with a channel-depending colour
                    // (original colour was yellow: red=255, green=255, blue=0)
                    red=green=blue = 0;
                    switch( iDecoderIndex )
                     { case 0:  // first rx window : slightly more reddish
                          red  = 255;
                          green= 240;
                          break;
                       default: // other channels : less reddish..
                          red  = 240;
                          green= 255;
                          break;
                     }
                    spectrogram_rgb[x].b[RGB_BYTE_RED]   = red;
                    spectrogram_rgb[x].b[RGB_BYTE_GREEN] = green;
                    spectrogram_rgb[x].b[RGB_BYTE_BLUE]  = blue;
                    spectrogram_rgb[x].b[RGB_BYTE_ALPHA] = 0x00;
                 }
               } // end for < mark the symbol-frequencies for all decoders >
            } // end if( fShowSymbolMarkers )


#        if( USE_DIB_SECTION )  // use "DIB" = "device-independent bitmap" ?
           if( g_pdwWfallPixelData != NULL ) // (a bit more complex, but MUCH faster than SetPixel!)
            { if( iLine <= (int)g_dwWfallPixelsYMax )
               { // Ex: pdwPixel = g_pdwWfallPixelData + iLine * g_dwWfallPixelStride;
                 // You thought the topmost line (y=0) is at the lowest address ?
                 // Wrong, the bitmap is "standing upside down" in memory !
                 pdwPixel = g_pdwWfallPixelData + (g_dwWfallPixelsYMax-iLine) * g_dwWfallPixelStride;
                 for (x = 0; x < Layout_iWfallWidth; ++x)
                  { *pdwPixel++ = spectrogram_rgb[x].dw; // copy "RGBQUADS", not "COLORREFS" !
                  }
               }
            }
#        else // ! USE_DIB_SECTION : doomed to use 'SetPixel' or 'SetPixelV'...
           for (x = 0; x < Layout_iWfallWidth; ++x)
            { SetPixelV( hdcScroll, x, iLine/*y*/, spectrogram_rgb[x].dw );
               // SetPixel has the speed of a tranquilized snail,
               // and copies  "COLORREFS", not "RGBQUADS", which are incompatible !
            }
#        endif // USE_DIB_SECTION ?

           if( fShowSymbolNumbers )
            { // T_SpecBufEntry.sz15DecoderInfo[] contains the text
              // to be displayed in the waterfall .
              //  This is usually an empty string, but may contain a 'tone number'
              //  or 'quality indicator' once per symbol.
              //  Content may be decoder-specific, thus every decoder has its own:
              for( iDecoderIndex=0; (iDecoderIndex<pDecoder->nChannels)
                                 && (iDecoderIndex<WSQ_MAX_DECODER_CHANNELS);
                 ++iDecoderIndex )
               {
                 x = 10;
                 if( pSpectrum->sz15DecoderInfo[iDecoderIndex][0] != '\0' )
                  { LeftAlignedTextOut( hdcScroll, x,iLine/*y*/, pSpectrum->sz15DecoderInfo[iDecoderIndex] );
                  }
               }
            } // end if( fShowSymbolNumbers )
         } // end if( pSpectrum != NULL )
      } // end  for(iLine... )
   } // end if( nLinesToUpdate > 0 )

  // Clean up (intermediately)
  SelectObject (hdcMem, hOldPen);
  DeleteObject (hPen);


  BitBlt( // copy the waterfall into the destination bitmap (always!)
        hdcMem,  // << destination (device context)
        Layout_xWfallLeft, Layout_yWfallTop, // x,y for the destination
        Layout_iWfallWidth,           // width in pixels, ex:755
        Layout_iWfallHeight,          // height in pixels, ex:85
        hdcScroll, // << source (device context)
        0, 1,   //<--1 to avoid showing the first line being updated
        SRCCOPY);

  // Add the rectangular markers from the 'interactive' decoder ?
  if( InteractiveDecoderActive() )
   {
     hPen = CreatePen( PS_SOLID, 1, 0x007FCFFF );
     hOldPen = (HPEN)SelectObject( hdcMem, hPen );
     SetTextColor( hdcMem, 0x007FCFFF );
     x1 = Layout_xWfallLeft+3;
     x2 = x1 + Layout_iWfallWidth - 4;
     y1 = Layout_yWfallTop;
     y2 = y1 + Layout_iWfallHeight - 1;
     pSpectrum = WSQ_GetSpectrumFromBuffer( pDecoder, 0/*iLine:topmost*/ );
     if( pSpectrum != NULL )
      {
        for(i=0; i<Layout_iWfallHeight; ++i )
         { i32SpectrumIndex = pSpectrum->i32SpectrumIndex -/*!*/ i;
           if( (pWSQIPoint=WSQI_GetMarkerForWFall( i32SpectrumIndex)) != NULL )
            { x = Layout_xWfallLeft + FreqToScreenXoffset( pWSQIPoint->freq_Hz );
              y = Layout_yWfallTop  + i;
              if( x>x1 && x<x2 && y>y1 && y<y2 )
               { MoveToEx (hdcMem, x-2, y-7 , NULL);  // frame, similar as the mouse pointer for the interactive decoder
                 LineTo   (hdcMem, x+3, y-7 );
                 LineTo   (hdcMem, x+3, y+7 );
                 LineTo   (hdcMem, x-2, y+7 );
                 LineTo   (hdcMem, x-2, y-7 );
                 if( pWSQIPoint->decoded_char > 0 )
                  { memset(sz80, 0, sizeof(sz80) );
                    switch( pWSQIPoint->decoded_char )
                     { case 10:  strcpy(sz80, "NL" ); break;
                       case 13:  strcpy(sz80, "CR" ); break;
                       case 32:  strcpy(sz80, "' '"); break;
                       default:  sz80[0] = (char)pWSQIPoint->decoded_char;
                          break;
                     }
                    LeftAlignedTextOut( hdcMem, x+4, y-5, sz80 );
                  }
               } // end if < marker inside the visible area of the spectrogram >
            } // end if < marker from the interactive decoder in this line >
         } // for(i=0; i<Layout_iWfallHeight; ++i )
      } // end if( pSpectrum != NULL )
     SelectObject (hdcMem, hOldPen);
     DeleteObject (hPen);
   } // end if( InteractiveDecoderActive() )

  // Display the received peak frequencies and the sent tones....
  hPen = CreatePen (PS_SOLID, 1, 0x00DDDDDD);//0x0000FF10L);
  hOldPen = (HPEN)SelectObject (hdcMem, hPen);
  SetTextColor (hdcMem, 0x0000A0A0);  // dark yellow
  SetTextAlign (hdcMem, TA_LEFT);
  sz80[0] = '\0';
  switch( iMouseOverArea )
   { case CLIENT_AREA_TX_FREQ   :
        sprintf( sz80,"TX range: %0.1f .. %0.1f Hz",
                   (float) WSQ_Encoder.fltLowestToneFreq,
                   (float)(WSQ_Encoder.fltLowestToneFreq + dblToneSpacing_Hz * (WSQ2_NUM_TONES-1)) );
        break;
     case CLIENT_AREA_RX1_F_LO  :
     case CLIENT_AREA_RX2_F_LO  :
        i = iMouseOverArea - CLIENT_AREA_RX1_F_LO;
        sprintf( sz80,"RX%d fmin = %0.1f Hz",(int)(i+1),(float)WSQ_Decoder.channel[i].fmin_Hz );
        break;
     case CLIENT_AREA_RX1_F_HI  :
     case CLIENT_AREA_RX2_F_HI  :
        i = iMouseOverArea - CLIENT_AREA_RX1_F_HI;
        sprintf( sz80,"RX%d fmax = %0.1f Hz",(int)(i+1),(float)WSQ_Decoder.channel[i].fmax_Hz );
        break;
     default:  // normal info display, if the mouse pointer is NOT over a 'special' area:
        if(TXbutton==TRUE)  // currently transmitting ?
         {
           if(pEncoder->SynthesizerActive)
            { sprintf (sz80, "Synth cmd   %s", synthesizer[pEncoder->tone+1]);
            }
           else
            { sprintf (sz80, "Audio freq  %0.2f Hz", pEncoder->fltLowestToneFreq + (float)pEncoder->tone*dblToneSpacing_Hz );
            }
         }
        if(RXbutton==TRUE)
         { iNumVisibleChars = sprintf (sz80, "Peak hits = %d", pDecoder->peak_hits);
           TextOut (hdcMem, Layout_x2+270, Layout_y6+3 , sz80, iNumVisibleChars);
           sprintf (sz80, "Peak %0.1f Hz  %0.1f dB",
              (float)pDecoder->channel[0].fltSymbolPeakFreq, pDecoder->channel[0].fltSig_to_Noise_dB );
         }
        break;
    }
  TextOut (hdcMem, Layout_x2+10, Layout_y6+3 , sz80, strlen(sz80) );

  // Display the S/N meter reading===========================
  x1 = 5;                      // ex:  5
  x2 = Layout_xWfallLeft - 30; // ex: 30
  y1 = Layout_yMeterTop;    // upper end of the bargraph
  y2 = Layout_yMeterBottom; // lower end of the bargraph

  // First make the meter background box
  hSolidBrush = CreateSolidBrush (0x00505050);
    hOldBrush   = (HBRUSH)SelectObject (hdcMem, hSolidBrush);
    Rectangle (hdcMem, x1, y1, x2, y2);
    SelectObject (hdcMem, hOldBrush );
  DeleteObject (hSolidBrush);

  hBrush_SmeterGreen = CreateSolidBrush (0x0000B000);
  hOldSmeterBrushGreen = (HBRUSH)SelectObject (hdcMem, hBrush_SmeterGreen);
  SetBkMode(hdcMem, TRANSPARENT);
  x = x1;
  w = x2-x1;  // -> width of the meter (bargraph) if there was only ONE decoder
  if(WSQ_Decoder.nChannels>1)
   { w /= WSQ_Decoder.nChannels;
   }
  for( iDecoderIndex=0; (iDecoderIndex<WSQ_Decoder.nChannels)
                     && (iDecoderIndex<WSQ_MAX_DECODER_CHANNELS);
     ++iDecoderIndex )
   { pChannel = &pDecoder->channel[iDecoderIndex];
     y = SigNoiseToMeterY( pChannel->fltSig_to_Noise_dB );
     Rectangle (hdcMem, x, y, x+w, y2);
     x += w;
   }
  SelectObject (hdcMem, hOldSmeterBrushGreen);
  DeleteObject (hBrush_SmeterGreen);

  hOldFont4Mem = SelectObject(hdcMem,SmeterFont);   // < remains selected until the next SelectObject(hdcMem,hOldFont4Mem)
  SetTextColor(hdcMem, 0x00FFFFFF);//0x00F0C080);
  for(i=0; i<=3; ++i )
   { n = i * -10;                            // -30dB, -20dB, -10dB, 0 dB
     y = SigNoiseToMeterY( n );
     MoveToEx (hdcMem, x1, y , NULL);
     LineTo (hdcMem,   x2, y );
     sprintf( szLabel, "%d", (int)n );
     LeftAlignedTextOut( hdcMem, x+2, y-5, szLabel );
     if( i==0 )
      { LeftAlignedTextOut( hdcMem, x+10, y-5, "dB" );
      }
   }

  // Clean up (intermediately)
  SelectObject (hdcMem, hOldPen);
  DeleteObject (hPen);

  hPen = CreatePen (PS_SOLID, 1, 0x000000FF);
  hOldPen = (HPEN)SelectObject (hdcMem, hPen);
  y = SigNoiseToMeterY( -25/*dB*/ );
  MoveToEx (hdcMem, x1, y, NULL);   // -25dB red marker
  LineTo (hdcMem,   x2, y );

  SelectObject (hdcMem, hOldPen);
  DeleteObject (hPen);

  // Show the individual SQUELCH LEVELS for each decoder:
  hPen = CreatePen (PS_SOLID, 1, 0x00007FFF);
  hOldPen = (HPEN)SelectObject (hdcMem, hPen);
  x = x1;
  for( iDecoderIndex=0; (iDecoderIndex<WSQ_Decoder.nChannels)
                     && (iDecoderIndex<WSQ_MAX_DECODER_CHANNELS);
     ++iDecoderIndex )
   { pChannel = &pDecoder->channel[iDecoderIndex];
     y = SigNoiseToMeterY( pChannel->iSquelch_dB );
     MoveToEx (hdcMem, x, y, NULL);   // orange squelch marker
     LineTo (hdcMem,   x+w, y );
     x += w;
   }
  // Add label for signal-to-noise meter.  'SmeterFont' still selected for hdcMem !
  LeftAlignedTextOut(hdcMem, x1, y2+2,  " Signal" );
  LeftAlignedTextOut(hdcMem, x1, y2+14, "to Noise" );
  LeftAlignedTextOut(hdcMem, C_TRACKBAR_BOX_X1 + 2, Layout_y6+C_TRACKBAR_Y_OFFSET+4, "C");
  LeftAlignedTextOut(hdcMem, C_TRACKBAR_BOX_X1 + 2+ C_TRACKBAR_H_SEPARATOR+Layout_iTrackbarWidth, Layout_y6+C_TRACKBAR_Y_OFFSET+4, "B");

  SelectObject(hdcMem,hOldFont4Mem); //  'SmeterFont' not selected for hdcMem anymore.


  // Draw the most recent spectrum as a curve, overlaid on the spectrogram ?
  if( fShowSpectrumGraph )
   {
     pSpectrum = WSQ_GetSpectrumFromBuffer( pDecoder, 0/*most recent*/ ); // result may be NULL when "not available" !
     if( pSpectrum != NULL )
      { if( pSpectrum->fltFreqBinWidth_Hz > 0.0 )
         { fltFreqToBinIdx = 1.0 / pSpectrum->fltFreqBinWidth_Hz;
         }
        else // avoid div-by-zero..
         { fltFreqToBinIdx = 0.0;
         }
        iPrevBinIdx = -1;
        first_pixel = TRUE;
        fltPeak_dBfs= -200;
        y1 = Layout_yWfallTop;
        y2 = y1 + Layout_iWfallHeight-1;
        for (x = 0; x < Layout_iWfallWidth; ++x)
         {
           // Convert the x-coord to a frequency (in Hertz):
           freq = ScreenXoffsetToFreq( x );
           // Convert the pixel's center frequency to an FFT frequency bin index:
           iBinIdx = (int)( 0.5 + (freq - pSpectrum->fltCenterFreqOf1stBin) * fltFreqToBinIdx);
           if( iBinIdx>0 && iBinIdx<pSpectrum->nUsedBins )
            {
              if( iPrevBinIdx < 0 )
               { iPrevBinIdx = iBinIdx-1;  // 1st loop
                 if( iPrevBinIdx < 0 )
                  {  iPrevBinIdx = 0;
                  }
               }
              if( iPrevBinIdx < iBinIdx ) // Is this a new FFT-bin at all ?
               { fltPeak = pSpectrum->fltFreqBins[++iPrevBinIdx];
                 while( iPrevBinIdx < iBinIdx ) // MORE THAN ONE BIN per pixel : get the peak..
                  { flt = pSpectrum->fltFreqBins[++iPrevBinIdx];
                    if( flt>fltPeak )
                     {  fltPeak = flt;
                     }
                  }
                 // Convert the peak (~voltage) into dB, and scale into an y-coordinate
                 fltPeak_dBfs = Voltage_to_dBfs( fltPeak/*0..1*/ ); // ->  -120 to 0 dB, typically noise -60 dB "over" full scale
               }
              y = y2 - (int)( (float)Layout_iWfallHeight * ( fltPeak_dBfs + 120.0 ) / 120.0) ;
              LimitInteger( &y, y1, y2 );
              if( first_pixel )
               { MoveToEx( hdcMem, Layout_xWfallLeft+x, y, NULL);
                 first_pixel = FALSE;
               }
              else
               { LineTo( hdcMem, Layout_xWfallLeft+x, y );
               }
            } // end if( iBinIdx>0 && iBinIdx<pSpectrum->nUsedBins )
         } // end for x...
      } // end if( pSpectrum != NULL )
   } // end if ( fShowSpectrumGraph )

  hOldFont4Mem = SelectObject(hdcMem,VerySmallFont);   // < remains selected until the next SelectObject(hdcMem,hOldFont4Mem)


  // Draw IFK spectrum limit markers (DL4YHF: and, since 2014-03,
  //    the 'transmitted' and 'expected' tone frequencies)
  for( iDecoderIndex=0; (iDecoderIndex<WSQ_Decoder.nChannels)
                     && (iDecoderIndex<WSQ_MAX_DECODER_CHANNELS);
     ++iDecoderIndex )
   { pChannel = &pDecoder->channel[iDecoderIndex];
     y1 = Layout_yWfallTop;
     y2 = y1 + Layout_iWfallHeight + 4;
     if( pDecoder->nChannels>1 )  // if there's more than one decoder, use slightly smaller indicators (U-shaped)
      { y1 += 4 * iDecoderIndex;
        y2 = y1 + Layout_iWfallHeight - 4;
      }
     y = y2-1;
     h = -4; // here: offset added for the 'ticks' is NEGATIVE
     // If there are MULTIPLE decoders, use an interted "U" for the 1st,
     // and a "U" shaped indicator for the 2nd (and possibly any further) channel
     //  with the following intention:
     // The ticks for the 1st decoder's tone frequencies are ABOVE of the waterfall,
     // because the decoder's text output is also on the top.
     // The ticks for the 2nd decoder are on the LOWER END of the waterfall.
     // This way, even if the decoder frequency ranges overlap,
     //           the ticks painted further below don't .
     // Showing the 'ticks' also gives an indication about the 'tolerance margins'
     // on both sides of the ideal (expected) tone frequencies:
     //    |    tone numbers:             |
     //    |    0 1 2 3 4 5 ..       32   |
     //    |____|_|_|_|_|_|....|_|_|_|____|
     //         |
     //        1000 Hz (default for the 'lowest tone', here: 'base frequency')
     //
     switch( iDecoderIndex )
      { case 0:
           hPen = CreatePen (PS_SOLID, 1, 0x00007FFFL);  // REDDISH ORANGE for the main decoder (which is ALWAYS enabled)
           if( WSQ_Decoder.nChannels > 1 )
            { y = Layout_yWfallTop-2;
              h = 4;   // here: offset added for the 'ticks' is POSITIVE
            }
           break;
        default:    // an optional WSQ decoder instance:
           hPen = CreatePen (PS_SOLID, 1, 0x0000CFCFL);  // YELLOW-ISH ORANGE for the 2nd (optional) decoder
           break;
      }
     hOldPen = (HPEN)SelectObject (hdcMem, hPen);
     x1 = FreqToScreenX( pChannel->fmin_Hz ); // so far, there's only ONE decoder
     x2 = FreqToScreenX( pChannel->fmax_Hz );
     MoveToEx (hdcMem, x1, y1, NULL);
     LineTo (hdcMem,   x1, y2 );
     MoveToEx (hdcMem, x2, y1, NULL);
     LineTo (hdcMem,   x2, y2 );
     MoveToEx(hdcMem,  x1, y,  NULL);
     LineTo (hdcMem,   x2, y );
     for(i=0; i<WSQ2_NUM_TONES; ++i)
      { flt = pChannel->fltBaseFreq + (float)i * dblToneSpacing_Hz;
        x = FreqToScreenX( flt );
        MoveToEx (hdcMem, x, y, NULL);
        LineTo (hdcMem,   x, y+h );
      }
     SelectObject (hdcMem, hOldPen);
     DeleteObject (hPen);
   } // end for < all decoders >

  // Similar as WSJT-X, show the frequency range of the TRANSMITTED tones
  //  as two small arrows pointing down from the TX WINDOW towards the spectrogram.
  // If the TRANSMIT FREQUENCY RANGE is outside the 1st decoder's range,
  // use a double-width line as a 'warning' :
  i = 1;   // line width in pixel
  freq = pEncoder->fltLowestToneFreq + dblToneSpacing_Hz * (WSQ2_NUM_TONES-1);
  if( (pEncoder->fltLowestToneFreq<pDecoder->channel[0].fmin_Hz) || (freq>pDecoder->channel[0].fmax_Hz) )
   { ++i;
   }
  if( iMouseOverArea==CLIENT_AREA_TX_FREQ )
   { ++i;
   }
  SetTextColor (hdcMem, 0x000000FFL);
  hPen = CreatePen (PS_SOLID, i, 0x000000FFL);
  hOldPen = (HPEN)SelectObject (hdcMem, hPen);
  x1 = FreqToScreenX( pEncoder->fltLowestToneFreq );
  x2 = FreqToScreenX( freq );
  y1 = Layout_yWfallTop-WFALL_BORDER_TOP+1;
  y2 = Layout_yWfallTop;
  MoveToEx(hdcMem, x1-3, y2-3, NULL );
  LineTo(hdcMem, x1, y2 );
  LineTo(hdcMem, x1+4, y2-4 ); // 4, not 3, because LineTo omits the last pixel
  MoveToEx(hdcMem, x1, y2, NULL);
  LineTo(hdcMem, x1, y1 );
  LineTo(hdcMem, x2, y1 );
  LineTo(hdcMem, x2, y2 );
  MoveToEx(hdcMem, x2-3, y2-3, NULL );
  LineTo(hdcMem, x2, y2 );
  LineTo(hdcMem, x2+4, y2-4 );
  LeftAlignedTextOut(hdcMem, (x1+x2)/2 - 8, y1+1,  "TX" );
  if(TXbutton==TRUE)  // currently transmitting ? show the momentary frequency
   {
     if( iCWidState == 2 )  // sending CW ID ?
      { if( CW_Generator.ramp > 0.5 )
         { flt = CW_Generator.fltCenterFrequency;
         }
        else
         { flt = 0.0;
         }
      }
     else                   // sending WSQ :
      { flt = pEncoder->fltLowestToneFreq + (float)pEncoder->tone*dblToneSpacing_Hz;
      }
     if( flt != 0.0 )
      {
        i = 5;  // radius of the indicator disk
        x = FreqToScreenX( flt );
        y = Layout_yWfallTop + i+1;
        hSolidBrush = CreateSolidBrush (0x0000CFFF);
          hOldBrush = (HBRUSH)SelectObject (hdcMem, hSolidBrush);
          Ellipse( hdcMem, x-i, y-i, x+i, y+i );
          SelectObject (hdcMem, hOldBrush );
        DeleteObject (hSolidBrush);
      }
   }
  SelectObject (hdcMem, hOldPen);
  DeleteObject (hPen);

  SelectObject(hdcMem, hOldFont4Mem );

  // ex: DeleteObject (SelectObject (hdcMem,hOldSmeterFont));
  DeleteObject (SmeterFont );
  DeleteObject (VerySmallFont);
  SelectObject (hdcMem, hPen);
  DeleteObject (hPen);


  BitBlt( // "Blit" the finished drawing to the physical screen
         ((LPPAINTSTRUCT)(&ps))->hdc, // << destination (visible device context)
         rctClientArea.left, rctClientArea.top,
         rctClientArea.right-rctClientArea.left,
         rctClientArea.bottom-rctClientArea.top,
         hdcMem, // << source (off-screen device context)
         0, 0,
         SRCCOPY);

  // Done with off-screen bitmap, pen, and DC, get rid of them

  SelectObject (hdcMem, hOldPen);
  DeleteObject (hOldPen);

  SelectObject (hdcMem, hbmOld);
  DeleteObject (hbmMem);
  DeleteDC (hdcMem);  // hdcMem (off-screen bitmap) doesn't exist anymore now
  DeleteDC (hDC);


  EndPaint(hWnd, &ps);

  s_fRedrawAllWFLines = FALSE;  // 'done' (after handling the WM_PAINT message)

  return 0L;          // WM_PAINT message sucessfully handled
} // OnWmPaint()


//---------------------------------------------------------------------------
BOOL OnMouseWheel( int iClientX, int iClientY, int iDelta )
{
  float fc, sp, fmin, fmax, f1, f2, xRel, yRel;  // relative X- and Y-coord, 0...1 within the 'identified area'
  int iArea = IdentifyClientCoord( iClientX, iClientY, &xRel, &yRel );
  switch( iArea )
   { case CLIENT_AREA_RX_TEXT_1 :
     case CLIENT_AREA_RX_TEXT_2 :
     case CLIENT_AREA_TX_TEXT   :
        return FALSE;  // message not handled here
     case CLIENT_AREA_FREQ_SCALE:
     case CLIENT_AREA_WATERFALL : // zoom in/out of the waterfall,
        // keeping the original center frequency (under the mouse),
        // only modify the span:
        sp = g_WfallFmax_Hz - g_WfallFmin_Hz;
        fc = g_WfallFmin_Hz + xRel * sp;
        if( iDelta > 0 )  // zoom in
         { sp /= 1.5;
         }
        if( iDelta < 0 )  // zoom out
         { sp *= 1.5;
         }
#       define SPECDISP_MIN_SPAN 10/*Hz*/
        if( sp < SPECDISP_MIN_SPAN ) // don't zoom in 'too extremely'
         {  sp = SPECDISP_MIN_SPAN;  // (but exaggerated zoom allows to see the sub-bin frequency interpolation of the 'symbol frequency' overlay)
         }
        f1 = fc - xRel * sp;
        f2 = fc + (1.0-xRel) * sp;
        WSQ_GetDisplayableFreqRange( &WSQ_Decoder, &fmin, &fmax );
        LimitFloat( &f1, fmin, fmax-SPECDISP_MIN_SPAN );
        LimitFloat( &f2, f1+SPECDISP_MIN_SPAN, fmax );
        if( (f1!=g_WfallFmin_Hz) || (f2!=g_WfallFmax_Hz) )
         { g_WfallFmin_Hz = f1;
           g_WfallFmax_Hz = f2;
           RedrawWaterfallAndFreqScale();
         }
        return TRUE;
     default:
        return FALSE;  // message not handled here
   }
  // return FALSE;        // WM_MOUSEWHEEL message NOT handled
} // OnMouseWheel()


//---------------------------------------------------------------------------
void UpdateLastLineInRxWindowForInteractiveMode(
        int iChannel ) // [in] decoder index; 0=first RX text, 1=second, etc
{
  T_WSQDecoder *pDecoder = &WSQ_Decoder;
  HWND hwndEditor = hCtl_RX[iChannel];
  char sz255LastLine[256];
  DWORD dwLine;

  // Replace the 'last line' of text in the editor (RX window)
  // with an updated string (from the interactive WSQ decoder):
  SetCaretToEndOfText( hwndEditor );
  GetTextCaretLineAndColumn( hwndEditor, &dwLine, NULL );
  WSQI_GetTextForTicker( pDecoder, iChannel, sz255LastLine );
  ReplaceTextLine( hwndEditor, dwLine, sz255LastLine, FALSE/*cannot undo*/ );
  SetCaretToEndOfText( hwndEditor );
} // end UpdateLastLineInRxWindowForInteractiveMode()


//---------------------------------------------------------------------------
void EnableMouseEventsInEditors( BOOL fEnable )
{
  int i;
  // > EnableWindow enables or disables mouse and keyboard input to the specified window or control.
  for(i=0; i<WSQ_MAX_DECODER_CHANNELS; ++i )
   { EnableWindow( hCtl_RX[i], fEnable );
   }
  EnableWindow( hCtl_TX, fEnable );
}

//---------------------------------------------------------------------------
BOOL OnMouseEvent( int iClientX, int iClientY, int iKeysAndButtons )
  // Only called for the MAIN WINDOW, not for children like the text editors,
  // but could also abused as 'vertical splitter' BETWEEN the editors.
{
  BOOL fResult = FALSE; // message handled here ? TRUE=yes, FALSE=no .
  BOOL fSpecialCursor = FALSE;
  float fmin, fmax, f1, f2, xRel, yRel;  // relative X- and Y-coord, 0...1 within the 'identified area'
  int i, iClientWidth, iClientHeight;
  int iArea = IdentifyClientCoord( iClientX, iClientY, &xRel, &yRel );
  int interactive_decoders, iChannel, iBufIdxOffset;
  BOOL fJustPressed, fJustReleased, fModified=FALSE;
  POINT pt;
  RECT rct;
  // T_SpecBufEntry *pSpectrum;


  GetClientRect( g_hwndMain, &rct);   // here to retrieve parameters for the initial window layout
  iClientWidth = rct.right-1;
  iClientHeight= rct.bottom-1;
  LimitInteger( &iClientWidth, 10, 10000 ); // kludge to avoid div-by-zero
  LimitInteger( &iClientHeight,10, 10000 );

  fJustPressed  = (iKeysAndButtons & MK_LBUTTON) > (iPrevKeysAndButtons & MK_LBUTTON);
  fJustReleased = (iKeysAndButtons & MK_LBUTTON) < (iPrevKeysAndButtons & MK_LBUTTON);
  if( fJustPressed )
   { iMouseMoveStartX = iClientX;
     iMouseMoveStartY = iClientY;
     iDragStartLayoutY0 = Layout_y0;
     iDragStartLayoutY1 = Layout_y1;
     iDragStartLayoutY2 = Layout_y2;
     iDraggingArea = iArea;
     fltMouseMoveStartXRel = xRel;
     fltMouseMoveStartYRel = yRel;
   }
  if( iArea != iMouseOverArea )
   { iMouseOverArea = iArea;
     InvalidateRect( g_hwndMain, &commands_rect, FALSE ); // -> redraw the yellow info-text in OnWmPaint()
   }
  switch( iArea )  // ... default actions with or without 'dragging'
   { case CLIENT_AREA_RX_TEXT_1 :
     case CLIENT_AREA_RX_TEXT_2 :
     case CLIENT_AREA_TX_TEXT   :
        break;  // message not handled here (yet ?)
     case CLIENT_AREA_TX_FREQ   : // Indicator for the TRANSMIT frequency.
     case CLIENT_AREA_RX1_F_LO  : // indicator for the 1st decoder's start frequency
     case CLIENT_AREA_RX1_F_HI  : // indicator for the 1st decoder's end   frequency
     case CLIENT_AREA_RX2_F_LO  : // indicator for the 2nd decoder's start frequency
     case CLIENT_AREA_RX2_F_HI  : // indicator for the 2nd decoder's end   frequency
        // All these controls can be pulled via mouse .
        // NO BREAK HERE ! Similar handler as for moving the waterfall sideways:
     case CLIENT_AREA_WATERFALL : // grab the waterfall (to move up/down in frequency)
        WSQ_GetDisplayableFreqRange( &WSQ_Decoder, &fmin, &fmax );
        iBufIdxOffset = iClientY - Layout_yWfallTop; // -> 0=most recent spectrum
        // pSpectrum = WSQ_GetSpectrumFromBuffer( &WSQ_Decoder, iBufIdxOffset );
        interactive_decoders = InteractiveDecoderActive();
        iChannel  = 0;
        if( interactive_decoders )
         {  fSpecialCursor = TRUE;
            SetCursor(hCursorWSQI);
            if( interactive_decoders==2) // special case: only the 2nd decoder in interactive mode
             {  iChannel = 1;
             }
            if( iDraggingArea == CLIENT_AREA_WATERFALL )
             { iDraggingArea = -1;  // suppress dragging the waterfall in 'interactive decoder mode'
               // (the frequency scale can be dragged as an alternative)
             }
         }
        if( iKeysAndButtons & MK_LBUTTON )
         { if( fJustPressed )
            { fltMouseMoveStartFreq = g_WfallFmin_Hz; // default for dragging the WATERFALL
              fltMouseMoveStartFSpan= g_WfallFmax_Hz - g_WfallFmin_Hz;
              if( interactive_decoders )
               { f1 = g_WfallFmin_Hz + xRel * (g_WfallFmax_Hz - g_WfallFmin_Hz);
                 WSQI_OnMouseDown( f1, WSQ_Decoder.i32SpectrumCounter - iBufIdxOffset );
                 UpdateLastLineInRxWindowForInteractiveMode(iChannel); // here: after mouse button pressed
               }
              switch( iDraggingArea )   // dragging something 'special' (not the waterfall) ?
               {
                 case CLIENT_AREA_WATERFALL : // dragging the waterfall (sideways)
                    break;
                 case CLIENT_AREA_TX_FREQ:
                    fltMouseMoveStartFreq = WSQ_Encoder.fltLowestToneFreq;
                    break;
                 case CLIENT_AREA_RX1_F_LO:
                 case CLIENT_AREA_RX2_F_LO:
                    iChannel = iDraggingArea - CLIENT_AREA_RX1_F_LO;
                    fltMouseMoveStartFreq = WSQ_Decoder.channel[iChannel].fmin_Hz;
                    break;
                 case CLIENT_AREA_RX1_F_HI:
                 case CLIENT_AREA_RX2_F_HI:
                    iChannel = iDraggingArea - CLIENT_AREA_RX1_F_HI;
                    fltMouseMoveStartFreq = WSQ_Decoder.channel[iChannel].fmax_Hz;
                    break;
                 default:
                    break;
               }
            } // end if( fJustPressed )
           else // already pressed, so move the waterfall (or other controls) along the frequency axis ?
            { WSQ_GetDisplayableFreqRange( &WSQ_Decoder, &fmin, &fmax );
              f1 = fltMouseMoveStartFreq + (xRel-fltMouseMoveStartXRel) * fltMouseMoveStartFSpan;
              LimitFloat( &f1, fmin, fmax );
              switch( iDraggingArea )
               { case CLIENT_AREA_WATERFALL : // dragging the waterfall (sideways)
                    f1 = fltMouseMoveStartFreq + (fltMouseMoveStartXRel-xRel/*!*/) * fltMouseMoveStartFSpan;
                    f2 = f1 + fltMouseMoveStartFSpan;
                    LimitFloat( &f1, fmin, fmax-SPECDISP_MIN_SPAN );
                    LimitFloat( &f2, f1+SPECDISP_MIN_SPAN, fmax );
                    if( (f1!=g_WfallFmin_Hz) || (f2!=g_WfallFmax_Hz) )
                     { g_WfallFmin_Hz = f1;
                       g_WfallFmax_Hz = f2;
                       RedrawWaterfallAndFreqScale();
                       fMouseDragged = TRUE;
                       SetCursor(hCursorEastWest);
                       fSpecialCursor = TRUE;
                     }
                    break;
                 case CLIENT_AREA_TX_FREQ   : // Indicator for the TRANSMIT frequency
                    if( f1 != WSQ_Encoder.fltLowestToneFreq )
                     { WSQ_Encoder.fltLowestToneFreq = f1;
                       fModified = fMouseDragged = TRUE;
                       SetCursor(hCursorEastWest);
                       fSpecialCursor = TRUE;
                     }
                    break;
                 case CLIENT_AREA_RX1_F_LO:
                 case CLIENT_AREA_RX2_F_LO:
                    i = iDraggingArea - CLIENT_AREA_RX1_F_LO;
                    f2 = WSQ_Decoder.channel[i].fmin_Hz;
                    if( f1 != f2 )
                     {
                       WSQ_SetDecoderFmin( &WSQ_Decoder.channel[i], f1 );
                       fModified = fMouseDragged = TRUE;
                       SetCursor(hCursorEastWest);
                       fSpecialCursor = TRUE;
                     }
                    break;
                 case CLIENT_AREA_RX1_F_HI:
                 case CLIENT_AREA_RX2_F_HI:
                    i = iDraggingArea - CLIENT_AREA_RX1_F_HI;
                    f2 = WSQ_Decoder.channel[i].fmax_Hz;
                    if( f1 != f2 )
                     {
                       WSQ_SetDecoderFmax( &WSQ_Decoder.channel[i], f1 );
                       fModified = fMouseDragged = TRUE;
                       SetCursor(hCursorEastWest);
                       fSpecialCursor = TRUE;
                     }
                    break;
                 default :  // cannot drag this, whatever it is:
                    break;
               } // end switch( iDraggingArea )
              if( fModified )
               { InvalidateRect( g_hwndMain, &commands_rect, FALSE );
                 InvalidateRect( g_hwndMain, &graph_rect, FALSE );  // -> WM_PAINT -> OnWmPaint()
               }
            } // end if( iDraggingArea == iArea )
         } // end if < mouse moved over waterfall (or elements on it) with the LEFT button pressed >
        else if( (interactive_decoders>0) && ( (iKeysAndButtons & MK_LBUTTON) < (iPrevKeysAndButtons & MK_LBUTTON) ) )
         { // 'interactive WSQ decoder' is active, using LEFT mouse button to 'shoot a frequency' :
           // Pass the FREQUENCY and TIME to the interactive decoder, and let WSQ_Interactive.c do the rest
           f1 = g_WfallFmin_Hz + xRel * (g_WfallFmax_Hz - g_WfallFmin_Hz);
           WSQI_OnMouseUp( f1, WSQ_Decoder.i32SpectrumCounter - iBufIdxOffset );
           UpdateLastLineInRxWindowForInteractiveMode(iChannel); // here: after mouse button pressed
         }
        else // maybe the RIGHT mouse button has just been released ? -> show context menu..
        if(  ( (iKeysAndButtons & MK_RBUTTON) < (iPrevKeysAndButtons & MK_RBUTTON) )
           ||( (!fMouseDragged) && ( (iKeysAndButtons & MK_LBUTTON) < (iPrevKeysAndButtons & MK_LBUTTON) ) )
          )
         {
           g_fltClickedFrequency = g_WfallFmin_Hz + xRel * (g_WfallFmax_Hz - g_WfallFmin_Hz);
           WS_PrintMenuItem(g_hWFallContextMenu,IDM_WF_POPUP_CLICKED_FREQ,
               "Clicked on %0.1f Hz ...", (float)g_fltClickedFrequency );
           pt.x = iClientX;
           pt.y = iClientY;
           ClientToScreen( g_hwndMain, &pt );
           // > TrackPopupMenu displays a shortcut menu at the specified location
           // > and tracks the selection of items on the menu. (...)
           TrackPopupMenu( g_hWFallContextMenu, TPM_BOTTOMALIGN | TPM_LEFTALIGN,
               pt.x, pt.y,  // [in] SCREEN(!) coordinate
               0, g_hwndMain/*owner*/,
               NULL/*"no-dismissal area":anything outside this menu*/ );
         }
        else // left mouse button NOT down : show 'readout cursor' info ?
         {
         }
        break;
     case CLIENT_AREA_FREQ_SCALE:  // can be dragged sideways, similar as the waterfall
        if( iKeysAndButtons & MK_LBUTTON )
         { if( fJustPressed )
            { fltMouseMoveStartFreq = g_WfallFmin_Hz; // same as for dragging the WATERFALL
              fltMouseMoveStartFSpan= g_WfallFmax_Hz - g_WfallFmin_Hz;
            }
           else // already pressed, so move the waterfall (or other controls) along the frequency axis ?
            { WSQ_GetDisplayableFreqRange( &WSQ_Decoder, &fmin, &fmax );
              f1 = fltMouseMoveStartFreq + (fltMouseMoveStartXRel-xRel/*!*/) * fltMouseMoveStartFSpan;
              f2 = f1 + fltMouseMoveStartFSpan;
              LimitFloat( &f1, fmin, fmax-SPECDISP_MIN_SPAN );
              LimitFloat( &f2, f1+SPECDISP_MIN_SPAN, fmax );
              if( (f1!=g_WfallFmin_Hz) || (f2!=g_WfallFmax_Hz) )
               { g_WfallFmin_Hz = f1;
                 g_WfallFmax_Hz = f2;
                 RedrawWaterfallAndFreqScale();
                 fMouseDragged = TRUE;
                 SetCursor(hCursorEastWest);
                 fSpecialCursor = TRUE;
               }
            } // end else < left mouse button already pressed >
         }
        break;
     default:
        break;  // don't care about this 'screen area' here
   } // end switch( iArea )
  if( fJustReleased )
   { iDraggingArea = CLIENT_AREA_UNKNOWN;
     fMouseDragged = FALSE;
   }

  iPrevKeysAndButtons = iKeysAndButtons;

  // Set the cursor, depending on the type of the 'screen area', to let user discover the splitters:
  if( ! fSpecialCursor )
   { SetCursor(hCursorArrow);
   }
  SetWindowLong(g_hwndMain, DWL_MSGRESULT, 1); // ensure mouse is default arrow if outside the spectrum display

  return fResult;     // TRUE:WM_MOUSEWHEEL message handled, FALSE:not...

} // end OnMouseEvent()



/*------------------------------------------------------------------------------
 *
 *      Reaction to WM_SIZE message
 */
BOOL OnWmSize(HWND hWnd, int fwSizeType, int nWidth, int nHeight )
{
  // About WM_SIZE :
  // > If the MoveWindow function is called for a child window
  // > as a result of the WM_SIZE message, the bRedraw parameter
  // > should be nonzero to cause the window to be repainted.
  //
  // We only want to care about resizing the MAIN WINDOW, thus:
  if( hWnd == g_hwndMain )
   {
     s_fRedrawAllWFLines = TRUE;  // don't scroll the waterfall, but redraw it completely
     UpdateMainWindowLayout( nWidth, nHeight ); // -> Layout_xyz
     ApplyLayoutForChildWindows(); // -> MoveWindow(), ...
     // no need to 'redraw' anything here because windows
     // invalidates the window's client area anyway.
     return TRUE;          // WM_SIZE message sucessfully handled
   }
  else // not interested in resizing anything else, let the system handle it:
   { return FALSE;
   }
} // OnWmSize()

//---------------------------------------------------------------------------
PaDeviceID AudioDeviceNameToPaDeviceID( char *szName, BOOL for_input )
{
  int i, numDevices = Pa_CountDevices();
  const PaDeviceInfo *pPaDeviceInfo;

  for(i=0;i<numDevices && i<100; ++i )
   { pPaDeviceInfo = Pa_GetDeviceInfo(i);
     if( for_input )  // searching for an INPUT device ...
      { if( pPaDeviceInfo->maxInputChannels >= 1 )
         { if( strcmp( szName,pPaDeviceInfo->name) == 0 )
            { return i;
            }
         }
      }
     else             // searching for an OUTPUT device ...
      { if( pPaDeviceInfo->maxOutputChannels >= 1 )
         { if( strcmp( szName,pPaDeviceInfo->name) == 0 )
            { return i;
            }
         }
      }
   }
  // Arrived here: no match found -> use the "default" device
  if( for_input )
   { return Pa_GetDefaultInputDeviceID();
   }
  else
   { return Pa_GetDefaultOutputDeviceID();
   }
} // end AudioDeviceNameToPaDeviceID()

/*------------------------------------------------------------------------------
 *
 *      Start Portaudio
 */

void Start_Portaudio(void)
{
  char szErr80[84];
  PaError PaInitresultCode;
  PaDeviceID devId;

  Pa_StopStream (g_PAStream_Recv);
  Pa_Sleep( 500 );        // wait a while before restarting

  // Re-Initialize PortAudio subsystem
  PaInitresultCode = Pa_Initialize();
  if (PaInitresultCode == paNoError)
   {
     devId = AudioDeviceNameToPaDeviceID( g_sz255InputDevice, TRUE/*input*/ );
     PaInitresultCode = Pa_OpenStream( // ex: Pa_OpenDefaultStream(..)
                 &g_PAStream_Recv,     // PortAudioStream** stream,
                 devId,                // PaDeviceID inputDevice,
                 CHANNELS_PER_SAMPLE,  // int numInputChannels,
                 paFloat32,            // PaSampleFormat inputSampleFormat, most convenient format on a fast machine...
                 NULL,                 // void *inputDriverInfo,
                 paNoDevice,           // PaDeviceID outputDevice,

                 0,                    // int numOutputChannels,
                 paFloat32,            // PaSampleFormat outputSampleFormat,
                 NULL,                 // void *outputDriverInfo,
                 g_iNominalSamplingRateForSoundcard, // sampling frequency
                 SAMPLES_PER_BUFFER, 0, // 'frames' (sample points) per buffer, let PA determine numBuffers
                 paNoFlag,              // PaStreamFlags streamFlags,
                 ReceiveAudioCallback,  // callback function
                 NULL);      // we have no user data pointer to pass to the callback for now
     if (PaInitresultCode == paNoError)
      {  PaInitresultCode = Pa_StartStream (g_PAStream_Recv);
      }
   }

  if (PaInitresultCode != paNoError)
   {
     Pa_Terminate();
     if (PaInitresultCode == paHostError)
      {
         MessageBox ( g_hwndMain, szErr80, g_szClassName/*ex:"LMS"*/, MB_APPLMODAL | MB_OK | MB_ICONEXCLAMATION);
         PostQuitMessage (1);
      }
     else
      {
         MessageBox ( g_hwndMain, Pa_GetErrorText(PaInitresultCode), g_szClassName/*ex:"LMS"*/,
                         MB_APPLMODAL | MB_OK | MB_ICONEXCLAMATION);
         PostQuitMessage (1);
      }
   }
}

void Start_Portaudio_Send(void)
{
  char szErr80[84];
  PaError PaInitresultCode;
  PaDeviceID devId;


  Pa_StopStream (g_PAStream_Send);
  Pa_Sleep( 500 );        // wait a while before restarting

  // Re-Initialize PortAudio subsystem
  PaInitresultCode = Pa_Initialize ();
  if (PaInitresultCode == paNoError)
   {
     devId = AudioDeviceNameToPaDeviceID( g_sz255OutputDevice, FALSE/*e.g. output*/ );
     PaInitresultCode = Pa_OpenStream( // ex: Pa_OpenDefaultStream(..)
                 &g_PAStream_Send,     // PortAudioStream** stream,
                 paNoDevice,           // PaDeviceID inputDevice,
                 0,                    // int numInputChannels,
                 paFloat32,            // PaSampleFormat inputSampleFormat,
                 NULL,                 // void *inputDriverInfo,
                 devId,                // PaDeviceID outputDevice,

                 CHANNELS_PER_SAMPLE,  // int numOutputChannels,
                 paFloat32,            // PaSampleFormat outputSampleFormat,
                 NULL,                 // void *outputDriverInfo,
                 g_iNominalSamplingRateForSoundcard, // sampling frequency
                 SAMPLES_PER_BUFFER, 0, // 'frames' (sample points) per buffer, let PA determine numBuffers
                 paNoFlag,             // PaStreamFlags streamFlags,
                 SendAudioCallback,    // callback function
                 NULL);      // we have no user data pointer to pass to the callback for now
     if (PaInitresultCode == paNoError)
      { PaInitresultCode = Pa_StartStream (g_PAStream_Send);
      }
   }

  if (PaInitresultCode != paNoError)
   {
     Pa_Terminate();
     if (PaInitresultCode == paHostError)
      {
        MessageBox ( g_hwndMain, szErr80, g_szClassName/*ex:"LMS"*/, MB_APPLMODAL | MB_OK | MB_ICONEXCLAMATION);
        PostQuitMessage (1);
      }
     else
      {
        MessageBox ( g_hwndMain, Pa_GetErrorText(PaInitresultCode), g_szClassName/*ex:"LMS"*/,
                     MB_APPLMODAL | MB_OK | MB_ICONEXCLAMATION);
        PostQuitMessage (1);
      }
   }
}



/*------------------------------------------------------------------------------
 *
 *      Show About box
 */

void vShowAboutBox (HWND hwnd)
{
  char sz1k[1024];

  sprintf(sz1k,
            "Weak signal IFK sender/receiver with:\n"
              "- 28/32 varicode v3.0\n"
              "- serial port PTT\n"
              "- synthesizer control\n"
              "- selectable setup file\n"
              "- selectable peaks count\n"
              "- selectable waterfall colour\n"
              "- compatible with WSQ and WSQCall\n"
              "\n"
              "This version was compiled %s\n"
              "Original by Con Wassilieff ZL2AFP\n"
              "Modified by Wolf Buescher DL4YHF\n\n"
              "Currently 0.512 baud\n",  (char*)__DATE__ );
  MessageBox (hwnd, sz1k, "About WSQ2 v1.2", 0);
}


/*------------------------------------------------------------------------------
 *
 *      Show Help box
 */

void vShowHelpBox (HWND hwnd)
{
  MessageBox (hwnd,
              "To be advised\n",

              "WSQ2 Help", 0);
}


//---------------------------------------------------------------------------
void Set3ButtonStates( int iButtonStates )  // Sets the state of the three buttons 'RX','TX','PAUSE' :
 #define BS_PAUSE 0
 #define BS_RX    1
 #define BS_TX    2
{
  SendMessage(g_hwndTXbutton,STM_SETIMAGE, IMAGE_BITMAP,
                (LPARAM)( (iButtonStates==BS_TX) ? hBitmapTX_ON : hBitmapTX_OFF) );
  SendMessage(g_hwndRXbutton, STM_SETIMAGE, IMAGE_BITMAP,
                (LPARAM)( (iButtonStates==BS_RX) ? hBitmapRX_ON : hBitmapRX_OFF) );
  SendMessage(g_hwndPausebutton, STM_SETIMAGE, IMAGE_BITMAP,
                (LPARAM)( (iButtonStates==BS_PAUSE) ? hBitmapPause_ON: hBitmapPause_OFF) );
} // end Set3ButtonStates()

//---------------------------------------------------------------------------
void SwitchFromTxToRx(void)
{
  int  i;
  char buffer_off[34];  // command (string) for synthesizer

  Set3ButtonStates( BS_RX ); /* Set the ON button image for 'RX', the others OFF */
  TXbutton = FALSE;
  RXbutton = TRUE;
  Pausebutton = FALSE;

  // Start a new line on each receive screen to separate received from sent text
  for(i=0; i<WSQ_MAX_DECODER_CHANNELS; ++i )
   { SendMessage ( hCtl_RX[i], WM_CHAR, 0xD, 0);
   }
  // Start the receiver ...
#ifdef AUDIO_IO_H // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
  if( fUseAudioIOAsInput ) // input from soundcard replaced by Audio-I/O-DLL ->
   { // Nothing to do here, because input from the DLL keeps running;
     // see handler for WM_TIMER -> WSQ_ReadAndProcessSamplesFromAudioIO()
   }
  else
#endif // def'd AUDIO_IO_H ?
   { // input directly from soundcard (Portaudio), not via Audio-I/O-DLL:
     Pa_StartStream( g_PAStream_Recv);  // Start the receiver (Portaudio)
   }
  Pa_StopStream (g_PAStream_Send);      // Stop the sender

  if( hComm != NULL )
   {
     // include command to send frequency data to the synthesizer
     // to set frequency to zero when on receive (ie turn the synth off)
     sprintf(buffer_off, "%s\r", synthesizer[0]);
     SerialPuts(hComm, buffer_off);

     EscapeCommFunction(hComm,CLRDTR);
     EscapeCommFunction(hComm,CLRRTS);

     if( fReleaseCOMonRx ) // close serial port, here in SwitchFromTxToRx() ..
      { Sleep( 100 );        // just a precaution.. wait until characters have been flushed to serial port
        CloseHandle(hComm);  // close the serial port, here: after transmission
        hComm = NULL;        // handle to the COM port is invalid now
        // Note: Windows seems to keep DTR and RTS *cleared* for the closed port,
        //       which means the PTT remains passive. But this gives other programs
        //       like WSPR, WSJT (to name just a few) to use the same COM port
        //       while WSQ is in RECEIVE mode.
      }
   } // end if( hComm != NULL )

  fSwitchFromTxToRx = FALSE;   // DID switch back to receive now


} // end SwitchFromTxToRx()


//---------------------------------------------------------------------------
char *DecoderAlgorithmToString(int iAlgorithm)
{
  switch( iAlgorithm )
   { case WSQ_ALGO_ORIGINAL : return "almost original (ZL2AFP)";
     case WSQ_ALGO_ALTERN_1 : return "modified by DL4YHF (Alt.1)";
     default:                 return "unknown";
   }
}

//---------------------------------------------------------------------------
void UpdateVariableMenuItems(HMENU hMenu)
{
  T_WSQDecoder *pDecoder = &WSQ_Decoder;
  T_WSQEncoder *pEncoder = &WSQ_Encoder;


  // New items (added by DL4YHF) will be only updated here .
  // A few old items (by ZL2AFP) are updated in the menu event handler .
  // That may change some day, to have all menu-related stuff in one place .
  // Only strings for static menu items (w/o checkmarks or variable text)
  // are solely defined in the old-fashioned resource file.
  // Note:   Functions prefixed WS_ are implemented in DL4YHF's "WinStuff.c" .
  //         Most functions w/o prefix are declared in some windows header.
  WS_CheckMenuItem( hMenu, IDM_LONG_FFT, pDecoder->fUseLongerFFT ); // 'FFT window longer than WSQ symbol' ? Originally just a TEST (2014-03-15); details in WSQ_Codec.c
  WS_CheckMenuItem( hMenu, IDM_AGC_OFF,  (pDecoder->agc_mode == WSQ_AGC_OFF)  ); // "off" added by DL4YHF 2014-03-01
  WS_CheckMenuItem( hMenu, IDM_AGC_FAST, (pDecoder->agc_mode == WSQ_AGC_FAST) );
  WS_CheckMenuItem( hMenu, IDM_AGC_SLOW, (pDecoder->agc_mode == WSQ_AGC_SLOW) );
  WS_CheckMenuItem(hMenu, IDM_NB_ENABLED, pDecoder->nb_enabled );
  WS_PrintMenuItem(hMenu, IDM_AGC_NB_START_FREQ,"AGC/NB reference start: %d Hz", (int)pDecoder->iAGC_NB_StartFreq );
  WS_PrintMenuItem(hMenu, IDM_AGC_NB_END_FREQ,  "AGC/NB reference  end : %d Hz", (int)pDecoder->iAGC_NB_EndFreq );

  WS_CheckMenuItem( hMenu, IDM_SYMBOL_MARKERS,  fShowSymbolMarkers);
  WS_CheckMenuItem( hMenu, IDM_SYMBOL_NUMBERS,  fShowSymbolNumbers);
  WS_CheckMenuItem( hMenu, IDM_SHOW_SPECTRUM_GRAPH, fShowSpectrumGraph);
  WS_CheckMenuItem( hMenu, IDM_PALETTE_BLUE,   (iWFColorPalette==WF_PALETTE_BLUE) );
  WS_CheckMenuItem( hMenu, IDM_PALETTE_GREEN,  (iWFColorPalette==WF_PALETTE_GREEN));
  WS_CheckMenuItem( hMenu, IDM_PALETTE_SUNRISE,(iWFColorPalette==WF_PALETTE_SUNRISE));
  WS_CheckMenuItem( hMenu, IDM_PALETTE_LINRAD, (iWFColorPalette==WF_PALETTE_LINRAD));
  WS_CheckMenuItem( hMenu, IDM_3_PEAKS, (pDecoder->peak_hits==3) );
  WS_CheckMenuItem( hMenu, IDM_6_PEAKS, (pDecoder->peak_hits==6) );
  WS_CheckMenuItem( hMenu, IDM_SYMBOL_AVRG_OFF , pDecoder->symbol_avrg_mode==WSQ_SYM_AVRG_OFF  );
  WS_CheckMenuItem( hMenu, IDM_SYMBOL_AVRG_FAST, pDecoder->symbol_avrg_mode==WSQ_SYM_AVRG_FAST );
  WS_CheckMenuItem( hMenu, IDM_SYMBOL_AVRG_SLOW, pDecoder->symbol_avrg_mode==WSQ_SYM_AVRG_SLOW );
  WS_CheckMenuItem( hMenu, IDM_SYNTHESIZER_ON,   pEncoder->SynthesizerActive);
  WS_CheckMenuItem( hMenu, IDM_SYNTHESIZER_OFF, !pEncoder->SynthesizerActive);
  WS_CheckMenuItem( hMenu, IDM_RECORD_WAVE, fRecordingWaveFile && g_RecordedWaveFile.OpenedForWriting );
  if( fRecordingWaveFile )
   { WS_PrintMenuItem(hMenu,IDM_RECORD_WAVE, "&Recording file %s",g_RecordedWaveFile.sz255FileName );
   }
  else
   { WS_PrintMenuItem(hMenu,IDM_RECORD_WAVE, "&Record wave file" );
   }
  WS_CheckMenuItem( hMenu, IDM_DECODE_WAVE, fAnalysingWaveFile && g_AnalysedWaveFile.OpenedForReading );
  WS_CheckMenuItem( hMenu, IDM_DECODE_WAVE_LOOP, fDecodeWaveFileInLoop);
  WS_CheckMenuItem( hMenu, IDM_DECODE_WAVE_STATISTICS, fDecodeWaveStatistics);

#ifdef AUDIO_IO_H     // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
  if( AudioIoDllFileName[0] == '\0' ) // no DLL has been selected -> cannot use any for input !
   { WS_PrintMenuItem( hMenu, IDM_SELECT_AUDIO_IO_DLL, "&Select Audio I/O DLL" );
     WS_EnableMenuItem(hMenu, IDM_USE_AUDIO_IO_DLL_FOR_INPUT,  FALSE ); // FALSE->"grayed", TRUE->ENABLED
     WS_EnableMenuItem(hMenu, IDM_USE_AUDIO_IO_DLL_FOR_OUTPUT, FALSE );
   }
  else // AudioIoDllFileName is, at least, non-empty (which doesn't mean a lot..)
   {
     // When *currently using* the DLL for input, show its name:
     if( fUseAudioIOAsInput )
      { WS_PrintMenuItem( hMenu, IDM_SELECT_AUDIO_IO_DLL,
           "&Selected Input : %s, %s",
           WSQ_GetTypeAndStatusOfCurrentInputDevice(),
           AudioIoDllFileName ); // dare to show the FULL PATH here ;)
      }
     else // otherwise, show this:
      { WS_PrintMenuItem( hMenu, IDM_SELECT_AUDIO_IO_DLL, "&Select Audio I/O DLL (current input: %s)",
           WSQ_GetTypeAndStatusOfCurrentInputDevice() );
      }
     WS_EnableMenuItem(hMenu, IDM_USE_AUDIO_IO_DLL_FOR_INPUT, TRUE );
     WS_CheckMenuItem( hMenu, IDM_USE_AUDIO_IO_DLL_FOR_INPUT, fUseAudioIOAsInput );
     WS_EnableMenuItem(hMenu, IDM_USE_AUDIO_IO_DLL_FOR_OUTPUT,TRUE );
     WS_CheckMenuItem( hMenu, IDM_USE_AUDIO_IO_DLL_FOR_OUTPUT,fUseAudioIOAsInput );
   }
#else                 // no support for Audio-I/O -> DISABLE the menu item
  WS_EnableMenuItem( hMenu, IDM_SELECT_AUDIO_IO_DLL,        FALSE ); // compiled w/o AudioIO.c ..
  WS_EnableMenuItem( hMenu, IDM_USE_AUDIO_IO_DLL_FOR_INPUT, FALSE );
  WS_EnableMenuItem( hMenu, IDM_USE_AUDIO_IO_DLL_FOR_OUTPUT,FALSE );
#endif // AUDIO_IO_H ?

  WS_PrintMenuItem( hMenu, IDM_LAST_ERROR, "Last error: %s", WSQ_sz255LastError );

  WS_CheckMenuItem( hMenu, IDM_AUTO_TX_RX,   fAutoSwitchTxToRx );
  WS_CheckMenuItem( hMenu, IDM_SEND_DIDDLE,  pEncoder->fDiddleOnTxIdle );
  WS_CheckMenuItem( hMenu, IDM_SEND_CW_ID,  (iCWidState>0) );
  WS_CheckMenuItem( hMenu, IDM_CW_ID_ALWAYS, iCWidAlways );
  WS_PrintMenuItem( hMenu, IDM_DEFINE_CW_SPEED, "CW speed : %.2f seconds per dot",(float)fltCWSecondsPerDot );

  /* Added by DL4YHF 2014-03-19 ("because the WSQ frequency was occupied by JT9-1") : */
  WS_PrintMenuItem( hMenu, IDM_LOWEST_TONE_ON_TX,
               "Lowest tone on TX : %.1f Hz", (float)pEncoder->fltLowestToneFreq );
  WS_PrintMenuItem( hMenu, IDM_DEC_NUM_CHANNELS, "Number of Channels: %d",   (int)pDecoder->nChannels );
  WS_PrintMenuItem( hMenu, IDM_DEC1_Fo, "Decoder 1 Base Frequency  : %.1f Hz",(float)pDecoder->channel[0].fltBaseFreq );
  WS_PrintMenuItem( hMenu, IDM_DEC2_Fo, "Decoder 2 Base Frequency  : %.1f Hz",(float)pDecoder->channel[1].fltBaseFreq );
  WS_PrintMenuItem( hMenu, IDM_DEC1_BW, "Decoder 1 Frequency Span  : %.1f Hz",(float)pDecoder->channel[0].fltBandwidth);
  WS_PrintMenuItem( hMenu, IDM_DEC2_BW, "Decoder 2 Frequency Span  : %.1f Hz",(float)pDecoder->channel[1].fltBandwidth);
  WS_PrintMenuItem( hMenu, IDM_DEC1_SQL,"Decoder 1 Squelch Level   : %d dB", (int)pDecoder->channel[0].iSquelch_dB);
  WS_PrintMenuItem( hMenu, IDM_DEC2_SQL,"Decoder 2 Squelch Level   : %d dB", (int)pDecoder->channel[1].iSquelch_dB);
  // WS_PrintMenuItem( hMenu, IDM_DEC1_ALGO,"Decoder 1 Algorithm : %s", DecoderAlgorithmToString(pDecoder->channel[0].iAlgorithm) );
  // WS_PrintMenuItem( hMenu, IDM_DEC2_ALGO,"Decoder 2 Algorithm : %s", DecoderAlgorithmToString(pDecoder->channel[1].iAlgorithm) );
  WS_EnableMenuItem(hMenu, IDM_DEC2_Fo, (pDecoder->nChannels>=2) );
  WS_EnableMenuItem(hMenu, IDM_DEC2_BW, (pDecoder->nChannels>=2) );
  WS_EnableMenuItem(hMenu, IDM_DEC2_Fo, (pDecoder->nChannels>=2) );
  WS_EnableMenuItem(hMenu, IDM_DEC2_SQL,(pDecoder->nChannels>=2) );

  WS_CheckMenuItem( hMenu, IDM_DEC1_MODE_OLD,     pDecoder->channel[0].iWSQmode == WSQ_MODE_OLD );     // "old WSQ with 4 bins / tone"
  WS_CheckMenuItem( hMenu, IDM_DEC1_MODE_WSQCALL, pDecoder->channel[0].iWSQmode == WSQ_MODE_WSQCall ); // "WSQCall with 3 bins / tone"
  WS_CheckMenuItem( hMenu, IDM_DEC1_ALGO1, pDecoder->channel[0].iAlgorithm == WSQ_ALGO_ORIGINAL );
  WS_CheckMenuItem( hMenu, IDM_DEC1_ALGO2, pDecoder->channel[0].iAlgorithm == WSQ_ALGO_ALTERN_1 );
  WS_CheckMenuItem( hMenu, IDM_DEC1_INTERACTIVE, pDecoder->channel[0].fInteractive );
  WS_CheckMenuItem( hMenu, IDM_DEC2_MODE_OLD,     pDecoder->channel[1].iWSQmode == WSQ_MODE_OLD );     // "old WSQ with 4 bins / tone"
  WS_CheckMenuItem( hMenu, IDM_DEC2_MODE_WSQCALL, pDecoder->channel[1].iWSQmode == WSQ_MODE_WSQCall ); // "WSQCall with 3 bins / tone"
  WS_CheckMenuItem( hMenu, IDM_DEC2_ALGO1, pDecoder->channel[1].iAlgorithm == WSQ_ALGO_ORIGINAL );
  WS_CheckMenuItem( hMenu, IDM_DEC2_ALGO2, pDecoder->channel[1].iAlgorithm == WSQ_ALGO_ALTERN_1 );
  WS_CheckMenuItem( hMenu, IDM_DEC2_INTERACTIVE, pDecoder->channel[1].fInteractive );
  // A stupid "POPUP" (menu item with submenus) as no ID in the resource file
  // so there's no way to enable/disable its availability via WS_EnableMenuItem().
  // Only "MENUITEM" has an ID (to access it programmatically). Sigh.
  //     One of the many "joys" of bare-bone Windoze API programming.
  WS_EnableMenuItem(hMenu, IDM_DEC2_ALGO1,(pDecoder->nChannels>=2));
  WS_EnableMenuItem(hMenu, IDM_DEC2_ALGO2,(pDecoder->nChannels>=2));
  WS_EnableMenuItem(hMenu, IDM_DEC2_INTERACTIVE, (pDecoder->nChannels>=2) );
  if( iComPortNr <= 0 )
   { WS_PrintMenuItem( hMenu, IDM_PTT_CONTROLS, "COM port / PTT control : none" );
   }
  else
   { WS_PrintMenuItem( hMenu, IDM_PTT_CONTROLS, "COM port / PTT control : COM%d",(int)iComPortNr );
   }
  WS_CheckMenuItem( hMenu, IDM_RELEASE_COM_ON_RX, fReleaseCOMonRx );

} // end UpdateVariableMenuItems()


//---------------------------------------------------------------------------
void StartTransmit(void)
{
  int i;
  char sz80Temp[80];

   Set3ButtonStates( BS_TX ); /* Set the ON button image for 'TX', the others OFF */
   TXbutton = TRUE;
   RXbutton = FALSE;
   Pausebutton = FALSE;
   iDisplayedEncoderTone = -1; // force update of the display (small circle on top of the waterfall)

   // Switching from anything to 'transmit' always terminates file-analysis-mode:
   if( fAnalysingWaveFile )
    { WaveIO_CloseFile( &g_AnalysedWaveFile ); // no problem if not opened at all..
      fAnalysingWaveFile = FALSE;  // file analysis stopped because the OP wants to have a QSO
      if( fDecodeWaveStatistics )  // show character statistics (used during tests) :
       { ShowAnalysedFileStatistics();
       }
    }
   // If the serial port shall be used (for PTT- and/or synthesizer control),
   //  and the port is not opened yet, open it now:
   if( (hComm == NULL) /*&& WSQ_Encoder.SynthesizerActive*/ && (iComPortNr>0) )
    { sprintf( sz80Temp, "COM%d", (int)iComPortNr );
      hComm = SerialInit( sz80Temp/*name*/, baudrate, bitsperbyte, stopbits, parity[0], 'N', 50, 50, 512, 512) ;
      if (hComm == NULL)
       {
         sprintf( WSQ_sz255LastError, "Comport (%s) already open or no comport available", sz80Temp );
       }
    } // end if < need to open the serial port for transmission ? >

   // Start a new line on each receive screen to put the sent echoed text (Hex D)
   for(i=0; i<WSQ_MAX_DECODER_CHANNELS; ++i )
    { SendMessage ( hCtl_RX[i], WM_CHAR, 0xD, 0);
    }
   WSQ_BeginTransmission( &WSQ_Encoder ); // prepare the encoder before transmission
   iAutoSwitchTxRxState = 0;  // initial state of the 'auto-Tx-to-Rx-switch' machine
   iAutoSwitchTxRxTimer = 0;  // initial-idle timer, used for the " " "
   fSwitchFromTxToRx = FALSE; // no 'request to switch back to RECEIVE' yet
   Pa_StartStream(g_PAStream_Send); // Start the sender (sometimes failed - callback never called - YHF 2014-03)
   Pa_StopStream (g_PAStream_Recv); // Stop the receiver

   if (hComm != NULL)
    { // PTT control : set DTR and RTS (some use this, others that ?  - YHF)
      EscapeCommFunction(hComm,SETDTR);
      EscapeCommFunction(hComm,SETRTS);
    }

} // end StartTransmit()


/*------------------------------------------------------------------------------
 *
 *      Windows message callback handler
 *
 *      (This function is called by the Windows function's DispatchMessage() )
 */

LRESULT CALLBACK WindowProcedure (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  int i, index;
  char FileName[MAX_PATH+101]; // For selecting + opening setup- or wave files
  FILE *setupfile;
  char sz80Temp[80];
  char buffer_off[34];  // to convert tone value in to a character
  // RECT rct;
  POINT pt;
  SYSTEMTIME          systime_utc;
  T_WSQEncoder        *pEncoder = &WSQ_Encoder;
  T_WSQDecoder        *pDecoder = &WSQ_Decoder;
  T_WSQDecoderChannel *pChannel = &pDecoder->channel[0];

  switch (message)
   {
     case WM_CREATE:  // this is the right place to create any child window,
                      // including the editors for RX and TX test ...
        OnWmCreate(hwnd);
        break; // end case WM_CREATE

     case WM_HSCROLL:
      {
        // waterfall contrast or brightness slider control ?
        if (lParam == (int)hwndTrackbar[TRACKBAR_CONTRAST])
         {
           iWFContrast_Percent = SendMessage(hwndTrackbar[TRACKBAR_CONTRAST], TBM_GETPOS, 0,0); //get the trackbar position
           SetFocus(hwnd);    // set trackbar focus elsewhere to avoid having the focus box around the control
           RedrawWaterfallAndFreqScale();
         }
        if (lParam == (int)hwndTrackbar[TRACKBAR_BRIGHTNESS])
         {
           iWFBrightness_Percent = SendMessage(hwndTrackbar[TRACKBAR_BRIGHTNESS], TBM_GETPOS, 0,0); //get the trackbar position
           SetFocus(hwnd);    // set trackbar focus elsewhere to avoid having the focus box around the control
           RedrawWaterfallAndFreqScale();           
         }
      } break; // end case WM_HSCROLL

     case WM_CTLCOLORSTATIC:
        return (int) hBrush;  // don't need to use SetBkColor as hBrush does it all

     case WM_CTLCOLOREDIT:    // this one works for NORMAL (non-rich) edit controls and trackbars,
      {                       //      but failed to set the background of a RICH EDIT control .
        // control's window handle is lParam, not hWnd
#     if( ! USE_RICH_EDITORS )
        if( (HWND)lParam==hCtl_RX[0] ) // Receive screen graphics (1st channel)
         { // HDC of control is wParam
           SetTextColor((HDC)wParam, RGB(0,0,120));
           SetBkColor((HDC)wParam, C_RGB_RX1 ); // a bit more reddish than the 2nd channel
           SetBkMode((HDC)wParam,OPAQUE);
           // return new background brush
           return (int) hBrush_receive[0];
         }
        if( (HWND)lParam==hCtl_RX[1] ) // since 2014-02-21, two RX channels
         { SetTextColor((HDC)wParam, RGB(0,0,120));
           SetBkColor((HDC)wParam,C_RGB_RX2 );  // a bit more yellowish than the 1st channel
           SetBkMode((HDC)wParam,OPAQUE);
           // return new background brush.
           return (int) hBrush_receive[1];
         }
#     else // no "plain old text edit controls but RICH EDIT" : Incompatible !
        // Found somewhere, about WM_CTLCOLOREDIT :
        // > Rich Edit: This message is not supported. To set the background
        // > color for a rich edit control, use the EM_SETBKGNDCOLOR message.
#     endif // USE_RICH_EDITORS ?

        if((HWND)lParam==hCtl_TX)  // Send screen graphics
         {
           // HDC of control is wParam
           SetBkMode((HDC)wParam,OPAQUE);
           SetTextColor((HDC)wParam, RGB(255,255,0));
           SetBkColor((HDC)wParam,RGB(80,80,120));
           // return new background brush
           return (int) hBrush_send;//(LRESULT)GetStockObject(WHITE_BRUSH);
         }
        // just use the default settings otherwise
        // ex: return DefWindowProc(hwnd, message, wParam, lParam);
        //     (will be called below, AFTER the switch..case)
      } break; // end case WM_CTLCOLOREDIT

     case WM_COMMAND: // sent by 'windowed control elements' (buttons, etc) to the parent (here: to main window)
      { // <- unnecessary but helps with certain editors to fold / unfold the following section
        switch( wParam )
         {
           case IDC_TX_BUTTON:
              if(TXbutton == TRUE)
               { break;
               }
              else
               { StartTransmit();
                 if (hComm != NULL)
                  { // Send command with frequency data to the synthesizer
                    // to set frequency to lowest non-zero tone when on transmit (ie turn the synth back on)
                    // otherwise it will send zero Hz until the first character
                    sprintf(buffer_off, "%s\r", synthesizer[1]);
                    SerialPuts(hComm, buffer_off);

                    // PTT control : set DTR and RTS (some use this, others that ?  - YHF)
                    EscapeCommFunction(hComm,SETDTR);
                    EscapeCommFunction(hComm,SETRTS);
                  }
               }
              return 0; // end case IDC_TX_BUTTON


           case IDC_RX_BUTTON:
              if(RXbutton == TRUE) //if already depressed then quit - we want to simulate radiobutton behaviour...
               {
                 // ... only during file analysis, pressing 'RX' stops file analysis mode
                 //     and switches back to real-time processing :
                 if( fAnalysingWaveFile )
                  { WaveIO_CloseFile( &g_AnalysedWaveFile ); // no problem if not opened at all..
                    fAnalysingWaveFile = FALSE; // file analysis stopped because the OP wants to receive 'real time' data
                    if( fDecodeWaveStatistics ) // show character statistics (used during tests) :
                     { ShowAnalysedFileStatistics();
                     }
                  }
                 break;
               }
              else
               { SwitchFromTxToRx();
               }
              return 0; // end case IDC_RX_BUTTON

           case IDC_PAUSE_BUTTON:
              if(Pausebutton == TRUE) //if already depressed then quit - we want to simulate radiobutton behaviour
               { break;
               }
              else
               {
                 Set3ButtonStates( BS_PAUSE ); /* Set the PAUSE button image ON, the others OFF */
                 TXbutton = FALSE;
                 RXbutton = FALSE;
                 Pausebutton = TRUE;

                 Pa_StopStream( g_PAStream_Recv); // Stop the receiver
                 Pa_StopStream (g_PAStream_Send); // Stop the sender

                 // include command to send frequency data to the synthesizer
                 // to set frequency to zero when on receive (ie turn the synth off)
                 sprintf(buffer_off, "%s\r", synthesizer[0]);
                 SerialPuts(hComm, buffer_off);

                 EscapeCommFunction(hComm,CLRDTR);
                 EscapeCommFunction(hComm,CLRRTS);
                }
              return 0; // end case IDC_PAUSE_BUTTON



           // Menu events .....
           case IDM_RECORD_WAVE   :
              // Ask for the name of an audio 'logfile', try to create it, write the header,
              // and (if successful) save the incoming (received) audio in it.
              // The rest of the file will be written when passing sample blocks
              // to the WSQ decoder (-> WSQ_DecodeAudio), except during 'file analysis' .
              WaveIO_CloseFile( &g_RecordedWaveFile ); // no problem if NOT opened
              fRecordingWaveFile = FALSE;
              // If the analysed file had already been selected before,
              // it shall appear as 'pre-selected' in the following dialog:
              if( WS_SelectFileToSave( hwnd, WS_hInstance,
                    "Select wave file to record", // [in] char *pszTitle,
                RecordedWaveFileName,             // [in,out] default filename + buffer for result
                sizeof(RecordedWaveFileName)-1,   // [in] int iMaxLength
                "Wave Audio Files (*.wav)\0*.wav\0\0", // [in] char *pszFilter,
                "wav" ) ) // [in] char *pszDefaultExtension, only needed if user fails to ENTER an extension
               { // user clicked 'Ok' in the above dialog ..
                 if( WaveIO_OutOpen( &g_RecordedWaveFile, RecordedWaveFileName,
                                     AUDIO_FILE_MODE_APPEND,   // file_mode
                                     AUDIO_FILE_OPTION_NORMAL, // file options
                                     16, // bits_per_sample
                                     1,  // channels, only record the first even if Portaudio delivers TWO !
                                     g_iNominalSamplingRateForSoundcard ) )
                  { fRecordingWaveFile = TRUE;
                  }
                 else
                  { MessageBox( g_hwndMain, "Could not open output file", "WSQ", MB_OK );
                  }
               } // end if ( ChooseFile... )
              break; // end case IDM_RECORD_WAVE

           case IDM_DECODE_WAVE:
              // Ask for the name of an input audio file, try to open it,
              // and (if successful) begin to analyse it.
              // The rest of the file will be analysed in the TIMER EVENT handler,
              // to avoid messing with multiple threads.
              Pa_StopStream (g_PAStream_Recv); // Stop the receiver
              Set3ButtonStates( BS_PAUSE ); /* Show "PAUSE" before analysing the wave file */
              TXbutton = FALSE;
              RXbutton = FALSE;
              Pausebutton = TRUE;
              WaveIO_CloseFile( &g_AnalysedWaveFile ); // no problem if NOT opened
              if( fAnalysingWaveFile && fDecodeWaveStatistics )
               { ShowAnalysedFileStatistics();
               }
              fAnalysingWaveFile = FALSE;
              // If the analysed file had already been selected before,
              // it shall appear as 'pre-selected' in the following dialog:
              if( WS_SelectFileToOpen( hwnd, WS_hInstance,
                    "Select wave file for input", // [in] char *pszTitle,
                AnalysedWaveFileName,             // [in,out] default filename + buffer for result
                sizeof(AnalysedWaveFileName)-1,   // [in] int iMaxLength
                "Wave Audio Files (*.wav)\0*.wav\0\0", // [in] char *pszFilter,
                "wav" ) ) // [in] char *pszDefaultExtension, only needed if user fails to ENTER an extension
               { // user clicked 'Ok' in the above dialog ..
                 nCharsEmittedDuringWaveAnalysis = nDotsEmittedDuringWaveAnalysis = 0; // reset statistics
                 if( WaveIO_InOpen( &g_AnalysedWaveFile, AnalysedWaveFileName, NULL,0) )
                  { fAnalysingWaveFile = TRUE; // flag for the timer event handler
                    Set3ButtonStates( BS_RX ); // Show "RX" while analysing the file
                    TXbutton = FALSE;
                    RXbutton = TRUE;
                    Pausebutton = FALSE;
                    // Re-initialize the decoder, to clear averages, integrators, and similar,
                    // because the initial characters shall not be affected by previously received signals:
                    WSQ_BeginReception( &WSQ_Decoder ); // prepares / clears all decoders before an rx-over
                    PrintToRxWindows( 1 | 2, "\r\n--- Playing '%s', %d Samples/sec ---\r\n",
                           g_AnalysedWaveFile.sz255FileName,
                           (int)(0.5+g_AnalysedWaveFile.dbl_SampleRate) );
                  }
               } // end if ( ChooseFile... )
              break; // end case IDM_DECODE_WAVE

           case IDM_DECODE_WAVE_LOOP:
              fDecodeWaveFileInLoop = !fDecodeWaveFileInLoop;
              break;
           case IDM_DECODE_WAVE_STATISTICS:
              fDecodeWaveStatistics = !fDecodeWaveStatistics;
              break;
           case IDM_AUTO_TX_RX  : // enable  AUTOMATIC switch from TX to RX
              fAutoSwitchTxToRx = !fAutoSwitchTxToRx;
              return 0;
           case IDM_LOWEST_TONE_ON_TX :
              WS_RunFloatInputDialog( hwnd, g_szShortTitle,
                 "Lowest tone frequency for TRANSMIT",
                 "(300..15000, default 1000 Hz)", &pEncoder->fltLowestToneFreq,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              return 0;
           case IDM_SEND_DIDDLE : // send 'diddles' for sync when TX idling
              pEncoder->fDiddleOnTxIdle = !pEncoder->fDiddleOnTxIdle;
              return 0;
           case IDM_SEND_CW_ID   :
              iCWTxCharIndex=0; // index into sz255CWid[] during CW transmission
              CW_InitGenerator( &CW_Generator, WSQ_Encoder.fltLowestToneFreq, fltCWSecondsPerDot );
              if(TXbutton == TRUE)  // already transmitting ->
               { // just wait for the end of the WSQ transmit over, THEN send CW ID
                 iCWidState = 1; // 0=passive, 1=request to send CW ID, 2=sending CW ID
               }
              else  // not transmitting yet -> start to transmit and send ONLY the CW ID
               { iCWidState = 2; // 0=passive, 1=request to send CW ID, 2=sending CW ID
                 StartTransmit();
               }
              return 0;
           case IDM_CW_ID_ALWAYS :
              iCWidAlways = !iCWidAlways;
              return 0;
           case IDM_DEFINE_CW_ID :
              WS_RunStringEditDialog( hwnd, g_szShortTitle,
                 "CW ID text (callsign or 'wsq de' callsign,",
                 "maximum 255 characters)",
                 sz255CWid,       // default value
                 WS_EDIT_NORMAL,  // edit options
                 sz255CWid, 255,  // destination, max length
                 WS_NO_HELP_CONTEXT );
              return 0;
           case IDM_DEFINE_CW_SPEED:
              WS_RunFloatInputDialog( hwnd, g_szShortTitle,
                 "CW dot length; QRSS or standard speed",
                 "(0.05 .. 3 sec per dot, std=0.1, >=3:QRSS)", &fltCWSecondsPerDot,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              LimitFloat( &fltCWSecondsPerDot, 0.05/*normal CW*/, 30.0/*QRSS30*/ );
              return 0;

           case IDM_SYMBOL_MARKERS:
              fShowSymbolMarkers = !fShowSymbolMarkers;
              RedrawWaterfallAndFreqScale();
              return 0;
           case IDM_SYMBOL_NUMBERS:
              fShowSymbolNumbers = !fShowSymbolNumbers;
              RedrawWaterfallAndFreqScale();
              return 0;

           case IDM_3_PEAKS:
              pDecoder->peak_hits = 3;
              return 0;

           case IDM_6_PEAKS:
              pDecoder->peak_hits = 6;
              return 0;

           case IDM_SPECTRUM_FREQ_RANGE:
              i = (int)(0.5 + g_WfallFmin_Hz );
              WS_RunIntegerInputDialog( hwnd, g_szShortTitle,
                 "Waterfall start frequency",
                 "(0..3000 Hz)", &i,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              LimitInteger( &i, 0, 3000 );
              g_WfallFmin_Hz = i;
              i = (int)(0.5 + g_WfallFmax_Hz );
              WS_RunIntegerInputDialog( hwnd, g_szShortTitle,
                 "Waterfall end frequency",
                 "(300..4000 Hz)", &i,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              LimitInteger( &i, g_WfallFmin_Hz+100, 4000 );
              g_WfallFmax_Hz = i;
              RedrawWaterfallAndFreqScale();
              return 0;

           case IDM_SPECTRUM_AMPL_RANGE:
              return 0;

           case IDM_SHOW_SPECTRUM_GRAPH:
              fShowSpectrumGraph = !fShowSpectrumGraph;
              RedrawWaterfallAndFreqScale();
              return 0;

           case IDM_PALETTE_BLUE:
              iWFColorPalette=WF_PALETTE_BLUE;
              RedrawWaterfallAndFreqScale();
              return 0;
           case IDM_PALETTE_GREEN:
              iWFColorPalette=WF_PALETTE_GREEN;
              RedrawWaterfallAndFreqScale();
              return 0;
           case IDM_PALETTE_SUNRISE:
              iWFColorPalette=WF_PALETTE_SUNRISE;
              RedrawWaterfallAndFreqScale();
              return 0;
           case IDM_PALETTE_LINRAD:
              iWFColorPalette=WF_PALETTE_LINRAD;
              RedrawWaterfallAndFreqScale();
              return 0;

           case IDM_LONG_FFT: // 'FFT window longer than WSQ symbol' ? Originally just a TEST (2014-03-15); details in WSQ_Codec.c
              pDecoder->fUseLongerFFT = !pDecoder->fUseLongerFFT;
              return 0;

           case IDM_AGC_OFF:
              pDecoder->agc_mode = WSQ_AGC_OFF;
              return 0;

           case IDM_AGC_FAST:
              pDecoder->agc_mode = WSQ_AGC_FAST;
              return 0;

           case IDM_AGC_SLOW:
              pDecoder->agc_mode = WSQ_AGC_SLOW;
              return 0;

           case IDM_AGC_NB_START_FREQ: /* modify AGC/noiseblanker reference start freq */
              WS_RunIntegerInputDialog( hwnd, g_szShortTitle,
                 "Start frequency of AGC/NB reference range",
                 "(300..3000 Hz)", &pDecoder->iAGC_NB_StartFreq,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              return 0;

           case IDM_AGC_NB_END_FREQ  : /* modify AGC/noiseblanker reference end freq   */
              WS_RunIntegerInputDialog( hwnd, g_szShortTitle,
                 "Start frequency of AGC/NB reference range",
                 "(300..3000 Hz)", &pDecoder->iAGC_NB_EndFreq,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              return 0;

           case IDM_PTT_CONTROLS:
              return DialogBox(hThisInstance, MAKEINTRESOURCE(DLG_PTT), NULL, (DLGPROC)PTTDlgProc);
           case IDM_RELEASE_COM_ON_RX:
              fReleaseCOMonRx = !fReleaseCOMonRx;
              return 0;
           case IDM_SYNTHESIZER_ON:
              pEncoder->SynthesizerActive = TRUE;
              return 0;
           case IDM_SYNTHESIZER_OFF:
              pEncoder->SynthesizerActive = FALSE;
              return 0;

           case IDM_SYMBOL_AVRG_OFF:
              pDecoder->symbol_avrg_mode = WSQ_SYM_AVRG_OFF;
              return 0;
           case IDM_SYMBOL_AVRG_FAST:
              pDecoder->symbol_avrg_mode = WSQ_SYM_AVRG_FAST;
              return 0;
           case IDM_SYMBOL_AVRG_SLOW:
              pDecoder->symbol_avrg_mode = WSQ_SYM_AVRG_SLOW;
              return 0;

           case IDM_SPLITTER_POS_DEFAULT :
              Layout_iSplitPercent[0] = 25;
              Layout_iSplitPercent[1] = 25;
              Layout_iSplitPercent[2] = 15;
              UpdateMainWindowLayout(0,0);
              ApplyLayoutForChildWindows();
              RedrawAll();
              return 0;
           case IDM_SPLITTER_POS_LARGE_WF:
              Layout_iSplitPercent[0] = 10;
              Layout_iSplitPercent[1] = 10;
              Layout_iSplitPercent[2] = 5;
              UpdateMainWindowLayout(0,0);
              ApplyLayoutForChildWindows();
              RedrawAll();
              return 0;
           case IDM_DEC_NUM_CHANNELS:
#            if(0)
              i = WSQ_Decoder.nChannels;
              WS_RunIntegerInputDialog( hwnd, g_szShortTitle,
                 "number of decoder channels", // char *pszLabel1
                 "(1..2)",                     // char *pszLabel2
                 &i, WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
#            else  // as long at there are only ONE or TWO decoders, keep it simpler:
              i = WSQ_Decoder.nChannels+1;
              if( i>WSQ_MAX_DECODER_CHANNELS )
               {  i=1;
               }
#            endif
              LimitInteger( &i, i, WSQ_MAX_DECODER_CHANNELS );
              if( i != WSQ_Decoder.nChannels )
               {  WSQ_Decoder.nChannels = i;
                  UpdateMainWindowLayout(0,0);
                  ApplyLayoutForChildWindows();
                  RedrawWaterfallAndFreqScale();
               }
              return 0;

           case IDM_DEC1_Fo         :
           case IDM_DEC2_Fo         :  // must be id IDM_DEC1_Fo+1, etc..
              index = wParam-IDM_DEC1_Fo; // -> 0..WSQ_MAX_DECODER_CHANNELS-1
              pChannel = &pDecoder->channel[index];
              sprintf( sz80Temp, "Decoder %d 'base' frequency (lowest tone)", index+1 );
              WS_RunFloatInputDialog( hwnd, g_szShortTitle,
                 sz80Temp, "(300..15000, default 1000 Hz)", &pChannel->fltBaseFreq,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              if( pChannel->fltBaseFreq <= 0 )
               {  pChannel->fltBaseFreq = 1000;
               }
              return 0;
           case IDM_DEC1_BW         :
           case IDM_DEC2_BW         :
              index = wParam-IDM_DEC1_BW; // -> 0..WSQ_MAX_DECODER_CHANNELS-1
              pChannel = &pDecoder->channel[index];
              sprintf( sz80Temp, "Decoder %d frequency span", index+1 );
              WS_RunFloatInputDialog( hwnd, g_szShortTitle,
                 sz80Temp, "(64..14000 Hz)", &pChannel->fltBandwidth,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              return 0;
           case IDM_DEC1_SQL        :
           case IDM_DEC2_SQL        :
              index = wParam-IDM_DEC1_SQL; // -> 0..WSQ_MAX_DECODER_CHANNELS-1
              pChannel = &pDecoder->channel[index];
              sprintf( sz80Temp, "Decoder %d Squelch Level", index+1 );
              WS_RunIntegerInputDialog( hwnd, g_szShortTitle,
                 sz80Temp, "( 0 .. -30 dB SNR in 3 kHz b/w)", &pChannel->iSquelch_dB,
                 WS_EDIT_NORMAL, WS_NO_HELP_CONTEXT );
              return 0;
           case IDM_DEC1_MODE_OLD:
              pDecoder->channel[0].iWSQmode = WSQ_MODE_OLD;
              return 0;
           case IDM_DEC1_MODE_WSQCALL:
              pDecoder->channel[0].iWSQmode = WSQ_MODE_WSQCall;
              return 0;
           case IDM_DEC1_ALGO1:
              pDecoder->channel[0].iAlgorithm = WSQ_ALGO_ORIGINAL;
              return 0;
           case IDM_DEC1_ALGO2:
              pDecoder->channel[0].iAlgorithm = WSQ_ALGO_ALTERN_1;
              return 0;
   //      case IDM_DEC1_ALGO3:
   //         pDecoder->channel[0].iAlgorithm = WSQ_ALGO_ALTERN_2;
   //         return 0;
           case IDM_DEC1_INTERACTIVE:
              pDecoder->channel[0].fInteractive = !pDecoder->channel[0].fInteractive;
              if( pDecoder->channel[0].fInteractive )
               {  pDecoder->channel[1].fInteractive = FALSE;
               }
              return 0;
           case IDM_DEC2_MODE_OLD:
              pDecoder->channel[1].iWSQmode = WSQ_MODE_OLD;
              return 0;
           case IDM_DEC2_MODE_WSQCALL:
              pDecoder->channel[1].iWSQmode = WSQ_MODE_WSQCall;
              return 0;
           case IDM_DEC2_ALGO1:
              pDecoder->channel[1].iAlgorithm = WSQ_ALGO_ORIGINAL;
              return 0;
           case IDM_DEC2_ALGO2:
              pDecoder->channel[1].iAlgorithm = WSQ_ALGO_ALTERN_1;
              return 0;
   //      case IDM_DEC2_ALGO3:
   //         pDecoder->channel[1].iAlgorithm = WSQ_ALGO_INTERACTIVE;
   //         return 0;
           case IDM_DEC2_INTERACTIVE:
              pDecoder->channel[1].fInteractive = !pDecoder->channel[1].fInteractive;
              return 0;

           case IDM_WF_POPUP_DEC1_Fo: // clicked into the waterfall to set the frequency where 'we expect the lowest tone'
           case IDM_WF_POPUP_DEC2_Fo: // (the actual decoder frequency range may begin several Hz 'lower')
              index = wParam-IDM_WF_POPUP_DEC1_Fo; // -> 0..WSQ_MAX_DECODER_CHANNELS-1
              WSQ_SetDecoderBaseFreq( pDecoder, index, g_fltClickedFrequency );
              RedrawWaterfallAndFreqScale();
              return 0;
           case IDM_WF_POPUP_DEC1_BW: // clicked into the waterfall to set the frequency span, at least 64 Hz
           case IDM_WF_POPUP_DEC2_BW:
              return 0;
           case IDM_WF_POPUP_LOWEST_TONE_ON_TX:
              pEncoder->fltLowestToneFreq = g_fltClickedFrequency;
              return 0;


           case IDM_OPEN_SYNTH_FILE:
              if( ChooseFile( hwnd,
                          "Setup Files Only (*setup*.txt)\0*setup*.txt\0All Files (*.*)\0*.*\0\0",
                          "Select synthesizer setup file",
                          FileName) )
               { // user clicked 'Ok' in the above dialog ..
                 if ((setupfile = fopen(FileName, "r")) == NULL)
                  { sprintf( WSQ_sz255LastError, "Couldn't open %s .",(char*)FileName );
                    break;
                  }

                 fgets(title,80,setupfile);  // Must read all of these in to keep the order correct
                 fgets(sz80Temp,80,setupfile);
                 fgets(comports, 80,setupfile);
                 fgets(baud,80,setupfile);
                 fgets(parity, 80, setupfile);
                 fgets(bitsperbyte_s, 80, setupfile);
                 fgets(stopbit_s,80,setupfile);

                 // ex: for (index=0;index<33;index++) // aren't there 34(!) strings in setup.txt ? (YHF 2014-03-19)
                 for (index=0;index<34/*!*/;index++) // modified, now char synthesizer[34][80];
                  { fgets(synthesizer[index],80,setupfile);
                    // [0]     : > to set frequency to zero when on receive (ie turn the synth off)
                    // [1..33] : 33 WSQ tones
                  }

                 fclose(setupfile);

                 // For the synthesizer control, a COM port number is mandatory.
                 // Only if the operator didn't already select a COM port number himself (through the menu),
                 // use the COM port definition from ZL2AFP's synthesizer setup file:
                 if( iComPortNr <= 0 )
                  { iComPortNr = atoi(comports); // here: INITIAL COM port number from synthesizer setup file
                  }
               }
              return 0;

           case IDM_AUDIO_CONTROL_PANEL:
              ShellExecute(hwnd, "open", "rundll32.exe",
                 "shell32.dll,Control_RunDLL mmsys.cpl,,2",NULL, SW_SHOWNORMAL);// 2 opens "Audio" tab directly
              return 0;

#         ifdef AUDIO_IO_H     // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
           case IDM_SELECT_AUDIO_IO_DLL: /* added by DL4YHF, 2014-02-22 */
              fUseAudioIOAsInput = FALSE;
              AIO_Host_FreeDLL(&AIO_Host);  // unload the old DLL from memory (before picking a new one)
              if( WS_SelectFileToOpen( hwnd, WS_hInstance,
                "Select Audio-I/O-DLL for input (e.g. in_AudioIO.dll)", // [in] char *pszTitle,
                AudioIoDllFileName,             // [in,out] default filename + buffer for result
                sizeof(AudioIoDllFileName)-1,   // [in] int iMaxLength
                "Audio-I/O or ExtIO DLL Files (*.dll)\0*.dll\0\0", // [in] char *pszFilter,
                "dll" ) ) // [in] char *pszDefaultExtension, only needed if user fails to ENTER an extension
               { // user clicked 'Ok' in the above dialog ..
                 // Try to load the DLL (by the DLL host in AudioIO.c),
                 // and if successful, use that DLL FOR INPUT instead of the soundcard.
                 if( AIO_Host_LoadDLL( &AIO_Host, AudioIoDllFileName) == AIO_NO_ERROR )
                  { // Looks like the NEW DLL is a valid Audio-I/O (or ExtIO)-compatible DLL : USE it !
                    fUseAudioIOAsInput = TRUE;
                    WSQ_ShowControlPanelOfAudioIoDLL();
                    WSQ_StartInputFromAudioIoDLL(); // strictly said: "TRY TO OPEN" (source may not be ready to send yet)
                    // If source can already send, WSQ_StartInputFromAudioIoDLL() sets fAudioInputAlive=TRUE.
                    // Otherwise, will try again (in the timer) until the source can be connected.
                  }
               }
              return 0;

           case IDM_USE_AUDIO_IO_DLL_FOR_INPUT: // just a simple 'toggler' (on/off)
              if( fUseAudioIOAsInput )
               { WSQ_StopInputFromAudioIoDLL();
                 fUseAudioIOAsInput = FALSE;
               }
              else
               { fUseAudioIOAsInput = TRUE;
                 WSQ_StartInputFromAudioIoDLL(); // -> fAudioInputAlive=TRUE, hopefully...
               }
              return 0;

           case IDM_USE_AUDIO_IO_DLL_FOR_OUTPUT:
              if( fUseAudioIOAsOutput )
               { WSQ_StopOutputToAudioIoDLL();  // -> fUseAudioIOAsOutput = FALSE;
               }
              else
               { WSQ_StartOutputToAudioIoDLL(); // -> fUseAudioIOAsOutput = TRUE;
               }
              return 0;
#         endif // AUDIO_IO_H ?


           case IDM_HELPCONTENTS:
              //    vShowHelpBox (hwnd);
              // ex: ShellExecute(hwnd, "open", "WSQhelp.htm", NULL, NULL, SW_SHOWNORMAL);
              // To avoid hassle with the stupid "current working directory" .....
              SetCurrentDirectoryToPathOfExecutable();
              ShellExecute(hwnd, "open", "html\\wsq_manual.htm", NULL, NULL, SW_SHOWNORMAL);
              return 0;


           case IDM_HELPABOUT:
              vShowAboutBox (hwnd);
              return 0;

           case IDM_QUIT     :
              SendMessage(hwnd,WM_CLOSE,0,0);  // polite way to terminate
              break;

           default:
              if( (wParam >= IDM_FIRST_AUDIO_INPUT_DEVICE) && (wParam <= IDM_LAST_AUDIO_INPUT_DEVICE) )
               { // one of the dynamically created menu items with 'audio input devices' :
                 WS_GetMenuText(g_hMainMenu, wParam, g_sz255InputDevice, 255 );
                 for(i=0; i<g_nInputDevices; ++i)
                  { WS_CheckMenuItem( g_hMainMenu, IDM_FIRST_AUDIO_INPUT_DEVICE+i,
                                (int)wParam == (IDM_FIRST_AUDIO_INPUT_DEVICE+i) );
                  }
                 if( RXbutton )
                  { // REstart the receiver (with the NEW input device) immediately ?
#                  ifdef AUDIO_IO_H // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
                    if( ! fUseAudioIOAsInput ) // input from soundcard (not Audio-I/O-DLL ) ..
#                  endif // def'd AUDIO_IO_H ?
                      { Start_Portaudio();  // actually STOPS and RE-STARTS portaudio for input
                      }
                  }
               }
              if( (wParam >= IDM_FIRST_AUDIO_OUTPUT_DEVICE) && (wParam <= IDM_LAST_AUDIO_OUTPUT_DEVICE) )
               { // one of the dynamically created menu items with 'audio output devices' :
                 WS_GetMenuText(g_hMainMenu, wParam, g_sz255OutputDevice, 255 );
                 for(i=0; i<g_nInputDevices; ++i)
                  { WS_CheckMenuItem( g_hMainMenu, IDM_FIRST_AUDIO_OUTPUT_DEVICE+i,
                                 (int)wParam == (IDM_FIRST_AUDIO_OUTPUT_DEVICE+i) );
                  }
                 Start_Portaudio_Send();  // kludge to 'apply' g_sz255OutputDevice
                 if( ! TXbutton )
                  { // stop again, because we don't really want to transmit
                    Pa_StopStream (g_PAStream_Send);
                  }
               }
              break;

          } // end switch( wParam )
      } break; // end case WM_COMMAND

     case WM_DESTROY:
      {
        KillTimer (hwnd, 0);  // what if SetTimer() was never called? *#*
        KillTimer (hwnd, 1);
        WaveIO_CloseFile( &g_AnalysedWaveFile ); // no problem if NOT opened
        Pa_StopStream (g_PAStream_Recv);   // result ignored...
        Pa_CloseStream (g_PAStream_Recv);
        Pa_Terminate ();
        CloseHandle(hComm);
        SaveSettingsInIniFile(); // Save a few settings (NOT for the synthesizer) in an old-fashioned ini file
        WSQ_DeleteDecoder( &WSQ_Decoder ); // clean up, free memory
        DestroyCursor( hCursorWSQI );      // delete cursor created by CreateCursor()
        if( g_hInstanceDetectionMutex!=NULL )  // Delete 'my' mutex to tell everyone we're out of business
         { ReleaseMutex( g_hInstanceDetectionMutex );
           CloseHandle( g_hInstanceDetectionMutex );  // this removes "our" mutex from the system
           g_hInstanceDetectionMutex = NULL;
         }
        PostQuitMessage(0);
      } break;

     case WM_TIMER:   // will periodically get here every   milliseconds.
      {
        switch (wParam)
         {
           case 0:    // redraw the spectrum and waterfall area only
              InvalidateRect (hwnd, // window handle
                      &graph_rect,  // redraw only the spectrum and waterfall area
                      FALSE); // FALSE => do not erase when BeginPaint() is called
              return 0L;
           case 1:    // a faster timer, since 2014-02-20 used to analyse WAVE FILES
              ++g_dw50MillisecondCounter; // doesn't matter if this rolls over (after almost 7 years)
              // Added 2014-04-29 to have the current date+time visible on 'WSQ grabbers' :
              GetSystemTime( &systime_utc ); // -> SYSTEMTIME (in UTC)
              if( systime_utc.wSecond != g_LastDisplayedTime.wSecond )
               { // time to update the time (and date) displayed in the menu line
                 g_LastDisplayedTime = systime_utc;
                 WS_PrintMenuItem( g_hMainMenu, IDM_TIME,
                        "%04d-%02d-%02d %02d:%02d:%02d",
                        (int)g_LastDisplayedTime.wYear,
                        (int)g_LastDisplayedTime.wMonth,
                        (int)g_LastDisplayedTime.wDay,
                        (int)g_LastDisplayedTime.wHour,
                        (int)g_LastDisplayedTime.wMinute,
                        (int)g_LastDisplayedTime.wSecond );
                 // Because the menu didn't repaint itself automatically:
                 DrawMenuBar( hwnd );       
               }
              if( fAnalysingWaveFile )
               { if( Pausebutton )
                  {   // Paused during file-analysis mode : "do nothing" (do NOT close the file!)
                  }
                 else
                  { ContinueWaveFileAnalysis(); // -> CLEARS fAnalysingWaveFile when done
                    if( ! fAnalysingWaveFile )
                     { // File analysis done -> switch to PAUSE state
                       Set3ButtonStates( BS_PAUSE ); /* Set the PAUSE button image ON, the others OFF */
                       TXbutton = FALSE;
                       RXbutton = FALSE;
                       Pausebutton = TRUE;
                     }
                  } // end < analysing file, NOT paused >
               }
              else // NOT analysing a wave file..
               {
#              ifdef AUDIO_IO_H // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
                 if( fUseAudioIOAsInput )
                  {
                    // Ideally, WSQ_StartInputFromAudioIoDLL() has already set fAudioInputAlive=TRUE.
                    // Otherwise, will try again (here, in the timer) until the source can be connected;
                    // but not every 50 milliseconds !
                    if( ! fAudioInputAlive )
                     { if( iAudioInputAliveTimer > 0 )
                        { --iAudioInputAliveTimer;  // wait a second before trying again
                        }
                       else
                        { iAudioInputAliveTimer = 1000 / 50/*ms*/;  // restart alive-checking timer
                          WSQ_StartInputFromAudioIoDLL(); // -> fAudioInputAlive=TRUE when successful
                        }
                     }
                    if( fAudioInputAlive ) // audio input (from the 'bridge' to WSQ) is alive:
                     { WSQ_ReadAndProcessSamplesFromAudioIO();
                     }
                  }
#              endif // def'd AUDIO_IO_H ?
               }  // end else < NOT analysing a wave file >

              if( fSwitchFromTxToRx )  // set in Audio-callback (TX), HERE: polled in main loop...
               {  SwitchFromTxToRx();  // almost the same effect as clicking the 'RX' button
                  fSwitchFromTxToRx = FALSE; // "done" (in a thread-safe way)
               }

              if( TXbutton )
               { if( fAutoSwitchTxToRx && ( iAutoSwitchTxRxState > 1 ) )
                  { // Still transmitting, but just about to switch back to RECEIVE
                    // because there's nothing more to send -> flash the 'TX' button
                    if( (g_dw50MillisecondCounter & 7) == 0 )
                     { SendMessage(g_hwndTXbutton,STM_SETIMAGE, IMAGE_BITMAP, (LPARAM)hBitmapTX_ON);
                       TXbuttonFlashOff = FALSE;
                     }
                    if( (g_dw50MillisecondCounter & 7) == 3 )
                     { SendMessage(g_hwndTXbutton,STM_SETIMAGE, IMAGE_BITMAP, (LPARAM)hBitmapTX_OFF);
                       TXbuttonFlashOff = TRUE;
                     }
                  }
                 else // TX button should not FLASH but be steadily ON :
                  { if( TXbuttonFlashOff )  // had been "flashed OFF" so turn it on again (IMMEDIATELY)
                     { SendMessage(g_hwndTXbutton,STM_SETIMAGE, IMAGE_BITMAP, (LPARAM)hBitmapTX_ON);
                       TXbuttonFlashOff = FALSE;
                     }
                  }
                 // Update the small circles on top of the waterfall indicating the current TX frequency:
                 if( iCWidState == 2 )  // sending CW ID ?
                  { if( CW_Generator.ramp > 0.5 )
                     { i = CW_Generator.fltCenterFrequency;
                     }
                    else
                     { i = 0;
                     }
                  }
                 else                   // sending WSQ :
                  { i = pEncoder->tone;
                  }
                 if( i != iDisplayedEncoderTone )
                  { iDisplayedEncoderTone = i;
                    InvalidateRect( g_hwndMain, &graph_rect, FALSE); // -> "Ellipse()"...
                  }
               } // end if ( TXbutton )
              else // NOT transmitting (but paused or receiving) :
               { if( iDisplayedEncoderTone != -1 )
                  { iDisplayedEncoderTone = -1;
                    InvalidateRect( g_hwndMain, &graph_rect, FALSE);
                  }
               }
              if( WSQI_fRedrawMarkers )
               { WSQI_fRedrawMarkers = FALSE;
                 InvalidateRect( g_hwndMain, &graph_rect, FALSE);
               }
              if( WSQI_fGotNewCharacter )
               { WSQI_fGotNewCharacter = FALSE;
                 UpdateLastLineInRxWindowForInteractiveMode(
                    InteractiveDecoderActive()==2 ? 1 : 0/*iChannel*/ );
               }
              return 0L;
         }
      } break; // end case WM_TIMER

     case WM_PAINT:
        return OnWmPaint (hwnd);

#  if(1)  // (1) if erasing of the background (prior to painting) isn't necessary....
     case WM_ERASEBKGND: // Sent to a window to erease the background (for example, when resizing).
        // In some cases, this is the reason for the annoying flicker.
        // When everything in the client area is filled 'opaquely' when painting anyway,
        // there's no need to erase BEFORE painting, and most of the flicker can be eliminated
        // by letting windows know "we erased the background" (by returning NONZERO)
        // but not erase anything in reality.
        // > Return Values
        // > An application should return nonzero if it erases the background; otherwise, it should return zero.
        return 1; // ... not playing by the above rules: don't erase the background but say "we did" !
#  endif // (0,1)

     case WM_SIZE:   // sent to a window after its size has changed
      { if( OnWmSize(hwnd, wParam/*fwSizeType*/, LOWORD(lParam)/*nWidth*/, HIWORD(lParam)/*nHeight*/ ) )
         { return 0L;  // handled the message HERE; don't call DefWindowProc()
         }
      } break;

     case WM_INITMENU:
        // > Sent when a menu is about to become active.
        // > It occurs when the user clicks an item on the menu bar
        // > or presses a menu key. This allows the application
        // > to modify the menu before it is displayed.
        // Unfortunately (but not suprisingly, as often) this did NOT work properly.
        // Try something else:
        UpdateVariableMenuItems(g_hMainMenu);
        return 0L; // handled the message HERE; don't call DefWindowProc()

     case WM_MOUSEMOVE:
      { // > wParam = fwKeys :
        // > Indicates whether various virtual keys are down (MK_CONTROL,MK_LBUTTON,MK_MBUTTON,MK_RBUTTON,MK_SHIFT)
        // > LOWORD(lParam) : x-coordinate, relative to the upper-left corner of the client area
        // > HIWORD(lParam) : y-coordinate, ...
        if( hwnd == g_hwndMain )  // don't want to handle mouse-event in child windows
         { fMouseDragged = TRUE; // flag to tell a 'click' from a 'dragged waterfall' in OnMouseEvent()
           if( OnMouseEvent( LOWORD(lParam)/*x*/, HIWORD(lParam)/*y*/, wParam/*keys+mouse buttons*/ ) )
            { return 0L; // handled the message HERE; don't call DefWindowProc()
            }
         }
        else // not directed to the main window, but (possibly) one of the 'windowed children' ?
        if( iDraggingArea != CLIENT_AREA_UNKNOWN ) // must INTERCEPT the message while dragging a splitter.
         { // Because in this case, the coordinate is the CHILD WINDOW'S coordinate, transform it back and forth:
           if( ClientToScreen( hwnd, &pt ) )
            { if( ScreenToClient( g_hwndMain, &pt ) )
               { if( OnMouseEvent( pt.x, pt.y, wParam/*keys+mouse buttons*/ ) )
                  { return 0L; // a kludge .. not really happy with this .. DL4YHF 2014-03-07
                  }
               }
            }
         }
      } break; // end case WM_MOUSEMOVE

     case WM_LBUTTONDOWN:
     case WM_RBUTTONDOWN:
        SetFocus(g_hwndMain);  // set trackbar focus elsewhere to avoid editor windows gulping up WM_MOUSEWHEEL (!)
        fMouseDragged = FALSE; // flag to tell a 'click' from a 'dragged waterfall'
        // NO BREAK HERE ! fall through the next cases...
     case WM_LBUTTONUP:
     case WM_RBUTTONUP:
        // Same parameters as for WM_MOUSEMOVE, so to keep it simple, use the same handler..
        if( hwnd == g_hwndMain )  // don't want to handle mouse-event in child windows
         {
           if( OnMouseEvent( LOWORD(lParam)/*x*/, HIWORD(lParam)/*y*/, wParam/*keys+mouse buttons*/ ) )
            { return 0L; // handled the message HERE; don't call DefWindowProc()
            }
         }
       break; // end case WM_RBUTTONDOWN, WM_RBUTTONUP  (here used for the WATERFALL's context menu)

     case WM_MOUSEWHEEL:
      { // > The high-order word of 'wParam' indicates the distance the wheel is rotated,
        // >   expressed in multiples or divisions of WHEEL_DELTA, which is 120.
        // >   A positive value indicates that the wheel was rotated forward,
        // >   away from the user; a negative value indicates that the wheel
        // >   was rotated backward, toward the user.
        // >   The low-word of 'wParam' indicates whether various virtual keys are down.
        // > The low-order word of 'lParam' specifies the x-coordinate of the pointer,
        // >   relative to the upper-left corner of the screen.
        // >   The high-order word of 'lParam' specifies the y-coordinate of the pointer,
        // >   relative to the upper-left corner of the screen.
        // Unfortunately, WM_MOUSEWHEEL is trapped by the editor windows
        // when they have the focus (input cursor).
        pt.x = LOWORD(lParam);
        pt.y = HIWORD(lParam);
        i = HIWORD(wParam);  // HIWORD is unsigned, but the 'delta' is signed, thus:
        if( i & 0x8000 )
         {  i |= 0xFFFF0000; // expand sign from 16 to 32 bit
         }
        i /= WHEEL_DELTA;
        if( ScreenToClient( hwnd, &pt ) )
         { if( OnMouseWheel( pt.x, pt.y, i/*delta*/ ) )
            { return 0L; // handled the message HERE; don't call DefWindowProc()
            }
         }
        // Outside the client coord: let DefWindowProc process the message !
      } break; // end case WM_MOUSEWHEEL


#ifdef  WM_AUDIO_IO_CONTROL  // Support DL4YHF's "Audio-I/O" (audio stream interface DLL)
     case WM_AUDIO_IO_CONTROL:  // msg from the Audio-I/O control panel to the host:
        // WM_AUDIO_IO_CONTROL( wParam= function code, lParam= function value)
        switch( wParam ) // check the 'function code' in MESSAGE.wParam :
         { case WM_AUDIO_IO_FC_NONE:         /* just a "ping" */
              break;
           case WM_AUDIO_IO_FC_PANEL_CLOSED: /* control panel closed by user */
              break;
         } // end switch wParam, here: "function code"
        return 0L;
#endif // def  WM_AUDIO_IO_CONTROL ?

     default:          /* for messages that we don't deal with */
      { return DefWindowProc (hwnd, message, wParam, lParam);
      }
   }

  return DefWindowProc (hwnd, message, wParam, lParam);
}


/*-----------------------------------------------------------------------------
 * Handler for the WM_CREATE message
 */
void OnWmCreate(HWND hwnd /*, CREATESTRUCT *pCS*/ )
{

  // Serial comms setup parameters
  FILE *setupfile;
  // DCB dcbSerialParams = {0};
  // COMMTIMEOUTS timeouts = {0};
  // int comport_default     = 1;    //default comport
  // int baudrate_default    = 9600; //default baudrate for serial comms
  // int stopbits_default    = 1;
  // int bitsperbyte_default = 8;
  // char parity_default[80] = "N";
  char sz255Temp[256];
  int  i,y,h,numDevices;
  HMENU hmInputDevices, hmOutputDevices;
  const PaDeviceInfo *pPaDeviceInfo;
  COLORREF bgcolor_receive[2], bgcolor_send;


  if ((setupfile = fopen("setup.txt", "r")) == NULL)  // Load comport and other details
   {
     // Removed by DL4YHF: iComPortNr  = 1;  // COM port number now saved in / loaded from the INI file !
     baudrate    = 9600;
     stopbits    = 1;
     bitsperbyte = 8;
     parity[0]   = 'N';
   }
  else // setupfile valid, i.e. "setup.txt" was opened SUCCESSFULLY ....
   {
     // ex: setupfile = fopen("setup.txt", "r"); // this MUST be in here!
     // YHF: Hmmm.. ? already opened "setup.txt" successfully, so why open it AGAIN ?
     fgets(title,80,setupfile);     // -> "WSQ2 setup for serial comms and PTT\n"
     fgets(sz255Temp,80,setupfile); // -> "\n"
     fgets(comports, 80,setupfile); // -> "1\n"  (by default, not good for USB<->RS-232 adapters)
     fgets(baud,80,setupfile);      // -> "9600\n"
     fgets(parity, 80, setupfile);  // -> "N\n"
     fgets(bitsperbyte_s, 80, setupfile);    // -> "8\n"
     fgets(stopbit_s,80,setupfile);          // -> "1\n"

     for (i=0;i<34/*!*/;i++) // modif. 2014-03-19, ex: for (i=0;i<33;i++)
      { fgets(synthesizer[i],80,setupfile);
      }

     fclose(setupfile);

     // ex: iComPortNr  = atoi(comports);  // removed 2014-03, COM port number now saved in / loaded from the INI file !
     baudrate    = atoi(baud);
     stopbits    = atoi(stopbit_s);
     bitsperbyte = atoi(bitsperbyte_s);

     // Note: As suggested in the RSGB LF group, the serial port remains CLOSED
     //       because other programs (like WSJT, WSPR) may want to use it
     //       while WSQ is in RECEIVE mode. Thus, the serial port will be
     //       opened "when needed", in  ___ .
   } // end if < synthesizer setup file successfully loaded >

  // Graphic stuff.. plain windows GDI style, without VCL, MFC or the like .
  //                 What an incredible bulk !
# if( USE_RICH_EDITORS )
  LoadLibrary("RICHED32.DLL");  // See Win32ProgRef, "About Rich Edit Controls"
#endif

  // Microsoft: "LoadCursor has been superseded". Who cares, let them re-invent the wheel every few years.
  //            "LoadImage" has far too many function arguments for this purpose:
  hCursorEastWest  = LoadCursor(hThisInstance,IDC_SIZEWE ); // "Double-pointed arrow pointing west and east"
  hCursorNorthSouth= LoadCursor(hThisInstance,IDC_SIZENS ); // "Double-pointed arrow pointing north and south"
  hCursorArrow     = LoadCursor(hThisInstance,IDC_ARROW  ); // "Standard arrow"
  hCursorWSQI      = WSQI_CreateFrameCursor( hThisInstance ); // special cursor for the 'interactive' decoder
  hBrush = CreateSolidBrush ( RGB(64, 64, 64) ); // brush to colour the trackbars dark gray to match the background

  // ex: hBrush_receive = CreateSolidBrush ( RGB(250,230,160) );  //colour the receive screen yellow-ish
  bgcolor_receive[0] = RGB(250,230,160); // 1st RX screen yellow-ish, a bit more red-ish than the 2nd channel
  bgcolor_receive[1] = RGB(240,240,140); // 2ns RX screen yellow-ish, less red-ish than the 1st channel
  bgcolor_send       = RGB( 80, 80,120); // blue-ish colour for the send screen

  hBrush_receive[0]=CreateSolidBrush( bgcolor_receive[0] ); // "brush" for the 1st RX screen's background
  hBrush_receive[1]=CreateSolidBrush( bgcolor_receive[1] ); // "brush" for the 2nd RX screen's background
  hBrush_send  = CreateSolidBrush ( bgcolor_send );

  // Create an owner drawn button and colours for same
  hBrushButton = CreateSolidBrush (0x00504000);
  hBrushButtonAlt = CreateSolidBrush (0x00B09000);
  hBrushBorder =  CreateSolidBrush (0x00F0C080);

  // Update the main window layout. Similar must be done when resizing,
  // thus we use a common subroutine for this purpose:
  GetClientRect (hwnd, &rctClientArea);   // here to retrieve parameters for the initial window layout
  UpdateMainWindowLayout( rctClientArea.right-1/*iClientWidth*/, rctClientArea.bottom-1/*iClientHeight*/ );
        // Note: the child windows don't exist here yet.
        //  UpdateMainWindowLayout() only sets a few simple integers.

  // Textbox for received text
  // int hFont = (int)GetStockObject(ANSI_VAR_FONT);

  RXfont=CreateFont(20,10,0,0,700,FALSE,FALSE,FALSE,DEFAULT_CHARSET,OUT_OUTLINE_PRECIS,
                    CLIP_DEFAULT_PRECIS,ANTIALIASED_QUALITY, VARIABLE_PITCH,TEXT("Arial"));

  for(i=0; i<WSQ_MAX_DECODER_CHANNELS; ++i )
   { switch(i)
      { case 0:  // "main" rx window : topmost, between y=0 and y=y0
            y = 0;
            h = Layout_y0 - 1;
            break;
        case 1:  // 2nd rx window / "monitor" (for a different frequency range)
            y = Layout_y0;
            h = Layout_y1 - Layout_y0;
            break;
        default: // even more RX channels ? More work required here !
            y = Layout_y1-1;
            h = 1;
            break;
      }
     hCtl_RX[i] = CreateWindowEx(  // rx text ..
               0, // dwExStyle (WS_EX_...)
#             if( USE_RICH_EDITORS )
               RICHEDIT_CLASS, "",
#             else
               "edit", "",
#             endif
               WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_TABSTOP
                        | ES_MULTILINE | ES_AUTOVSCROLL |WS_VSCROLL,
               0,            // pos x1 (in window client coords, 0= upper left corner of parent window)
               y,            // pos y1
               Layout_xEnd,  // width in pixels,  ex: 825
               h,            // width in pixels
               hwnd/*parent*/, NULL/*menu*/, GetModuleHandle(""),  NULL);
     SendMessage( hCtl_RX[i], WM_SETFONT,  (WPARAM) RXfont, MAKELPARAM(TRUE, 0));


#  if( USE_RICH_EDITORS )
     // Increase the upper limit of the amount of text in the Rich Edit control.
     //  (the default value seems ridiculous, only 32 kByte !)  We want 2 MByte.
     // See email from ZL2AFP, 2017-12-05, WSQ2 RX text limit / stopped printing
     // after running for a couple of days (when USE_RICH_EDITORS was 0) .
     // So : Don't leave this to fate .. let Windoze know we expect LOTS of text:
     SendMessage( hCtl_RX[i], EM_EXLIMITTEXT, (WPARAM)0, (LPARAM)(DWORD)(2048L*1024) );
     // Because RichEdit doesn't support WM_CTLCOLOREDIT, set the BACKGROUND COLOUR this way:
     SendMessage( hCtl_RX[i], EM_SETBKGNDCOLOR, (WPARAM)0, (LPARAM)bgcolor_receive[i] );
#  endif // USE_RICH_EDITORS ?

   } // end for < all decoder channels / receive text controls >


  // Textbox for sending text
  hCtl_TX = CreateWindowEx(0, // tx text ..
#             if( USE_RICH_EDITORS )
               RICHEDIT_CLASS, "",
#             else
               "edit", "",
#             endif
              WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_TABSTOP | ES_MULTILINE | ES_AUTOVSCROLL |WS_VSCROLL,
               0,            // pos x1
               Layout_y1,    // pos y1, ex: 290
               Layout_xEnd,         // width in pixels,  ex: 825
               Layout_y2-Layout_y1, // height in pixels, ex: 70
               hwnd, NULL, GetModuleHandle(""),  NULL);
#if( USE_RICH_EDITORS )
  SendMessage( hCtl_TX, EM_EXLIMITTEXT, (WPARAM)0, (LPARAM)(DWORD)(2048L*1024) );
  SendMessage( hCtl_TX, EM_SETBKGNDCOLOR, (WPARAM)0, (LPARAM)bgcolor_send );
#endif // USE_RICH_EDITORS ?


  // Create Pause button==================================================
  g_hwndPausebutton = CreateWindowEx (
        0,                                 // no extended styles
        "static",                          // class name
        "Pause",                           // title (caption)
        WS_CHILD | WS_VISIBLE | SS_BITMAP | SS_NOTIFY,
        Layout_x4, // pos x1, ex: 388+250
        Layout_y6, // pos y1, ex: 490
        Layout_iBtnWidth,  // width,  ex: 110 (?? - YHF)
        Layout_iBtnHeight, // height, ex: 20
        hwnd,                              // parent window
        (HMENU) IDC_PAUSE_BUTTON,          // control identifier (don't need it subsequently so make it NULL
        GetModuleHandle(NULL),             // instance
        NULL                               // no WM_CREATE parameter
      );

  hBitmapPause_OFF = LoadBitmap(WS_hInstance, "IMG_BITMAP_PAUSE_OFF");
  hBitmapPause_ON = LoadBitmap(WS_hInstance, "IMG_BITMAP_PAUSE_ON");
  /* Set the button image */
  SendMessage(g_hwndPausebutton,STM_SETIMAGE, 0,(LPARAM) hBitmapPause_OFF);

  // Create Send button =================================================
  g_hwndTXbutton = CreateWindowEx ( // tx button ..
        0,                                    // no extended styles
        "static",                             // class name
        "Pass",                               // title (caption)
        WS_CHILD | WS_VISIBLE | SS_BITMAP | SS_NOTIFY,// style
        Layout_x5, // pos x1, ex: 450+250
        Layout_y6, // pos y1, ex: 490
        Layout_iBtnWidth,  // width,  ex: 110 (?? - YHF)
        Layout_iBtnHeight, // height, ex: 20
        hwnd,                     // parent window
        (HMENU) IDC_TX_BUTTON,    // control identifier (don't need it subsequently so make it NULL
        GetModuleHandle(NULL),    // instance
        NULL                      // no WM_CREATE parameter
      );

  hBitmapTX_OFF = LoadBitmap(WS_hInstance, "IMG_BITMAP_TX_OFF");
  hBitmapTX_ON = LoadBitmap(WS_hInstance, "IMG_BITMAP_TX_ON");
  /* Set the button image */
  SendMessage(g_hwndTXbutton,STM_SETIMAGE, 0,(LPARAM) hBitmapTX_OFF); //Set default button to USB

  // Create Receive button===============================================
  g_hwndRXbutton = CreateWindowEx (  // rx button ..
        0,                                   // no extended styles
        "static",                            // class name
        "PassAll",                           // title (caption)
        WS_CHILD | WS_VISIBLE | SS_BITMAP | SS_NOTIFY,// | BS_AUTOCHECKBOX | BS_PUSHLIKE,
        Layout_x6, // pos x1, ex: 512+250,
        Layout_y6, // pos y1, ex: 490
        Layout_iBtnWidth,  // width,  ex: 110 (?? - YHF)
        Layout_iBtnHeight, // height, ex: 20
        hwnd,                                // parent window
        (HMENU) IDC_RX_BUTTON,               // control identifier (don't need it subsequently so make it NULL
        GetModuleHandle(NULL),               // instance
        NULL                                 // no WM_CREATE parameter
      );

  hBitmapRX_OFF = LoadBitmap(WS_hInstance, "IMG_BITMAP_RX_OFF");
  hBitmapRX_ON  = LoadBitmap(WS_hInstance, "IMG_BITMAP_RX_ON");
  /* Set the button image */
  SendMessage(g_hwndRXbutton,STM_SETIMAGE, 0,(LPARAM) hBitmapRX_ON);




  // Waterfall display contrast + brightness trackbar =======================
  hwndTrackbar[TRACKBAR_CONTRAST] = CreateWindowEx ( // contrast trackbar
           0,                                  // no extended styles
           "MSCTLS_TRACKBAR32",                // class name
           "Trackbar Control",                 // title (caption)
           WS_CHILD | WS_VISIBLE | TBS_NOTICKS  | TBS_BOTH | TBS_HORZ,// style
           C_TRACKBAR_BOX_X1 + C_TRACKBAR_H_SEPARATOR,         // x
           Layout_y6+C_TRACKBAR_Y_OFFSET,  // y
           Layout_iTrackbarWidth, // width
           C_TRACKBAR_HEIGHT,     // height
           hwnd,                  // parent window
           NULL,                  // control identifier (don't need it subsequently so make it NULL
           hThisInstance,         // instance
           NULL                   // no WM_CREATE parameter
          );
  hwndTrackbar[TRACKBAR_BRIGHTNESS] = CreateWindowEx ( // brightness trackbar
           0,                             // no extended styles
           "MSCTLS_TRACKBAR32",           // class name
           "Trackbar Control",            // title (caption)
           WS_CHILD | WS_VISIBLE | TBS_NOTICKS  | TBS_BOTH | TBS_HORZ,// style
           C_TRACKBAR_BOX_X1 + 2*C_TRACKBAR_H_SEPARATOR + Layout_iTrackbarWidth, // x
           Layout_y6+C_TRACKBAR_Y_OFFSET, // y
           Layout_iTrackbarWidth, // width
           C_TRACKBAR_HEIGHT,     // height
           hwnd,                  // parent window
           NULL,                  // control identifier (don't need it subsequently so make it NULL
           hThisInstance,         // instance
           NULL                   // no WM_CREATE parameter
          );
  for(i=TRACKBAR_CONTRAST; i<=TRACKBAR_BRIGHTNESS; ++i)
   { SendMessage(hwndTrackbar[i], TBM_SETRANGE,
           (WPARAM) TRUE,                      // redraw flag
           (LPARAM) MAKELONG(0, 100));         // min. & max. positions
     SendMessage(hwndTrackbar[i], TBM_SETPAGESIZE,
            0, (LPARAM) 10);                    // new page size
     SendMessage(hwndTrackbar[i], TBM_SETPOS,
           (WPARAM) TRUE,                      // redraw flag
           (LPARAM) (i==TRACKBAR_CONTRAST) // set initial brightness level
                  ? iWFContrast_Percent
                  : iWFBrightness_Percent );
   }


  // Waterfall display setup================================================
  hdc2 = GetDC(HWND_DESKTOP);   //Get a DC from somewhere for the waterfall display

  hdcScroll = CreateCompatibleDC (hdc2);

  // Create an off-screen bitmap to do the off-screen drawing on waterfall display:
  CreateBitmapForWaterfall();   // Layout_iWfallWidth,Layout_iWfallHeight -> hbmScroll,g_bmiScrollingWF

  // Direct waterfall drawing operations onto the off-screen bitmap:
  hbmScrollOld = (HBITMAP) SelectObject (hdcScroll, hbmScroll);

  // Load the Audio-I/O-DLL (if configured in the previous session) =======
#ifdef AUDIO_IO_H     // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
  if( fUseAudioIOAsInput )
   { if( AIO_Host_LoadDLL( &AIO_Host, AudioIoDllFileName) == AIO_NO_ERROR )
      { // Looks like the NEW DLL is a valid Audio-I/O (or ExtIO)-compatible DLL : START it !
        WSQ_ShowControlPanelOfAudioIoDLL(); // one fine day, we'll support SDRs with VFO control and I/Q input this way
        WSQ_StartInputFromAudioIoDLL();
        // At this point, the control panel of in_AudioIO.dll should show "WSQ"
        // as one of the 'listeners' attached to the DLL because we passed our
        // 'class name' when loading the DLL as argument 'pszAudioReaderID' .
      }
     else // Failed to load an audio-I/O-DLL : automatically 'fall back' to Portaudio (i.e. soundcard)
      { fUseAudioIOAsInput = FALSE;
        Pa_StartStream (g_PAStream_Recv); // Restart receiver (if the SOUNDCARD is used for input)
      }
   } // if( fUseAudioIOAsInput ) ?
#endif // AUDIO_IO_H ?


  // Start a periodic timer that posts a WM_TIMER message each 50ms
  // SetTimer (hwnd, 0, 10000, NULL);
  SetTimer( hwnd, 1, TIMER1_INTV_MS, NULL);   // since 2014-02-20, used to analyse WAVE FILES

  // Load the main menu from the resource (statically linked to the executable):
  g_hMainMenu = LoadMenu(hThisInstance, MAKEINTRESOURCE(ID_MENU));
  // Insert a few 'dynamic' items, mainly sub-menus, for example the list of audio devices:
  hmInputDevices = CreatePopupMenu();
  hmOutputDevices= CreatePopupMenu();

  // let Portaudio enumerate the audio INPUT devices ("soundcards" or similar) :
  Pa_Initialize();
  numDevices = Pa_CountDevices();
  g_nInputDevices = g_nOutputDevices = 0;  // also required later to set/clear checkmarks, thus not local
  for(i=0;i<numDevices && i<100; ++i )
   { pPaDeviceInfo = Pa_GetDeviceInfo(i);
     if( pPaDeviceInfo->maxInputChannels >= 1 )
      { InsertMenu( hmInputDevices, -1, MF_BYPOSITION|MF_STRING, IDM_FIRST_AUDIO_INPUT_DEVICE+g_nInputDevices,
                    pPaDeviceInfo->name );
        WS_CheckMenuItem( hmInputDevices, IDM_FIRST_AUDIO_INPUT_DEVICE+g_nInputDevices,
                    strcmp( g_sz255InputDevice,pPaDeviceInfo->name) == 0 );
        ++g_nInputDevices;
        // Note: We don't care about those "device numbers" here because they
        //       are subject to change whenever an USB device is plugged in/out.
        //       Only the NAMES of those audio devices will NOT change.
      }
     if( pPaDeviceInfo->maxOutputChannels >= 1 )
      { InsertMenu( hmOutputDevices, -1, MF_BYPOSITION|MF_STRING, IDM_FIRST_AUDIO_OUTPUT_DEVICE+g_nOutputDevices,
                    pPaDeviceInfo->name );
        WS_CheckMenuItem( hmOutputDevices, IDM_FIRST_AUDIO_OUTPUT_DEVICE+g_nOutputDevices,
                    strcmp( g_sz255OutputDevice,pPaDeviceInfo->name) == 0 );
        ++g_nOutputDevices;
      }
   }
  InsertMenu(
        g_hMainMenu, // handle of menu
        IDM_AUDIO_CONTROL_PANEL, // UINT uPosition, menu item that new menu item precedes
        MF_BYCOMMAND | MF_STRING | MF_POPUP, // menu item flags
        (UINT)hmInputDevices,    // menu item identifier or handle of drop-down menu or submenu
        "Audio input devices");  // menu item content
  InsertMenu(
        g_hMainMenu, // handle of menu
        IDM_AUDIO_CONTROL_PANEL, // UINT uPosition, menu item that new menu item precedes
        MF_BYCOMMAND | MF_STRING | MF_POPUP, // menu item flags
        (UINT)hmOutputDevices,   // menu item identifier or handle of drop-down menu or submenu
        "Audio output devices"); // menu item content
  SetMenu(hwnd, g_hMainMenu);

  // Initialize PortAudio subsystem ========================================
  Start_Portaudio();                // Start receiver (here: initial call, in OnWmCreate)
  Pa_StopStream (g_PAStream_Recv);  // temporarily stop receiver to avoid screen flicker
                                    // when starting sender

  Start_Portaudio_Send();           // Start sender
  Pa_StopStream (g_PAStream_Send);  // Stop sender until needed later

#ifdef AUDIO_IO_H
  if( ! fUseAudioIOAsInput )
#endif // AUDIO_IO_H ?
   { Pa_StartStream (g_PAStream_Recv); // Restart receiver (if the SOUNDCARD is used for input)
   }

  // Create a free-floating popup menu ("context menu") for the WATERFALL display:
  g_hWFallContextMenu = CreatePopupMenu();
  AppendMenu(g_hWFallContextMenu, MF_STRING, IDM_WF_POPUP_CLICKED_FREQ, "Clicked on x Hz ..." );
  for(i=0; i<WSQ_MAX_DECODER_CHANNELS; ++i )
   { sprintf( sz255Temp, "Set as new base frequency for decoder %d",(int)(i+1) );
     AppendMenu(g_hWFallContextMenu, MF_STRING, IDM_WF_POPUP_DEC1_Fo+i, sz255Temp );
     sprintf( sz255Temp, "Set as new upper freq-limit for decoder %d",(int)(i+1) );
     AppendMenu(g_hWFallContextMenu, MF_STRING, IDM_WF_POPUP_DEC1_BW+i, sz255Temp );
   }
  AppendMenu(g_hWFallContextMenu, MF_STRING, IDM_WF_POPUP_LOWEST_TONE_ON_TX, "Set lowest tone on TX" );
} // OnWmCreate()

//---------------------------------------------------------------------------
void SaveSettingsInIniFile(void) // only called on 'normal' program termination,
                                 // BEFORE destroying the main window (g_hwndMain)
{
  int i;
  char section[88];
  RECT rct;
  T_WSQEncoder        *pEncoder = &WSQ_Encoder;  
  T_WSQDecoder        *pDecoder = &WSQ_Decoder;
  T_WSQDecoderChannel *pChannel;


  // Save a few settings (NOT for the synthesizer) from an old-fashioned ini file..

  strcpy( section,"About_WSQ" );
  WriteStringToIniFile( g_szIniFileName,section,"1", "Configuration for WSQ (Weak Signal QSO mode for LF+MF)." );
  WriteStringToIniFile( g_szIniFileName,section,"2", "This file can be safely deleted to restore the defaults." );
  WriteStringToIniFile( g_szIniFileName,section,"3", "Details at www.qsl.net/dl4yhf/WSQ/ ." );
  WriteStringToIniFile( g_szIniFileName,section,"4", "" );

  // > The GetWindowRect function retrieves the dimensions of the **bounding**
  // > rectangle of the specified window. The dimensions are given in screen
  // > coordinates that are relative to the upper-left corner of the screen.
  if( GetWindowRect( g_hwndMain, &rct) )
   {  strcpy( section,"MainWindow" );
      WriteIntToIniFile(g_szIniFileName,section,"Top", rct.top );
      WriteIntToIniFile(g_szIniFileName,section,"Left",rct.left);
      WriteIntToIniFile(g_szIniFileName,section,"Width",rct.right - rct.left); // **bounding**, thus not "minus one" !
      WriteIntToIniFile(g_szIniFileName,section,"Height",rct.bottom-rct.top );
      WriteIntToIniFile(g_szIniFileName,section,"Rx1Splitter", Layout_iSplitPercent[0] );
      WriteIntToIniFile(g_szIniFileName,section,"Rx2Splitter", Layout_iSplitPercent[1] );
      WriteIntToIniFile(g_szIniFileName,section,"RxTxSplitter",Layout_iSplitPercent[2] );
   }
  strcpy( section,"AudioSettings" );
  WriteStringToIniFile( g_szIniFileName,section,"InputDeviceName", g_sz255InputDevice );
  WriteStringToIniFile( g_szIniFileName,section,"OutputDeviceName",g_sz255OutputDevice );
#ifdef AUDIO_IO_H     // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
  WriteIntToIniFile(g_szIniFileName,section,"UseAudioIOAsInput", fUseAudioIOAsInput);
  WriteIntToIniFile(g_szIniFileName,section,"fUseAudioIOAsOutput",fUseAudioIOAsOutput);
  WriteStringToIniFile(g_szIniFileName,section,"AudioIODllFileName",AudioIoDllFileName);
#endif // AUDIO_IO_H ?

  // Audio-file related configuration...
  strcpy( section,"AudioFiles" );
  WriteStringToIniFile(g_szIniFileName,section,"AnalysedWaveFile", AnalysedWaveFileName);
  WriteStringToIniFile(g_szIniFileName,section,"RecordedWaveFile", RecordedWaveFileName);
  WriteIntToIniFile(g_szIniFileName,section,"PlayInLoop",fDecodeWaveFileInLoop);
  WriteIntToIniFile(g_szIniFileName,section,"ShowStatistics",fDecodeWaveStatistics);

  // Parameters for automatic RX/TX changeover, etc:
  strcpy( section,"Params" );
  WriteIntToIniFile(g_szIniFileName,section,"AutoSwitchTxRx",   fAutoSwitchTxToRx );
  WriteIntToIniFile(g_szIniFileName,section,"COM_port_number",  iComPortNr );
  WriteIntToIniFile(g_szIniFileName,section,"ReleaseCOMonRx",   fReleaseCOMonRx );

  WriteDoubleToIniFile(g_szIniFileName,section,"CW_sec_per_dot",fltCWSecondsPerDot);
  WriteStringToIniFile(g_szIniFileName,section,"CW_id_text",    sz255CWid);
  WriteIntToIniFile(g_szIniFileName,section,"CW_id_always",     iCWidAlways );

  WriteIntToIniFile(g_szIniFileName,section,"Wfall_fmin", g_WfallFmin_Hz );
  WriteIntToIniFile(g_szIniFileName,section,"Wfall_fmax", g_WfallFmax_Hz );
  WriteIntToIniFile(g_szIniFileName,section,"Wfall_brightness", iWFBrightness_Percent );
  WriteIntToIniFile(g_szIniFileName,section,"Wfall_contrast",   iWFContrast_Percent   );
  WriteIntToIniFile(g_szIniFileName,section,"Wfall_palette",    iWFColorPalette       );
  WriteIntToIniFile(g_szIniFileName,section,"ShowSpectrumGraph",fShowSpectrumGraph    );
  WriteIntToIniFile(g_szIniFileName,section,"ShowSymbolMarkers",fShowSymbolMarkers    );
  WriteIntToIniFile(g_szIniFileName,section,"ShowSymbolNumbers",fShowSymbolNumbers    );

  // WSQ-encoder and -decoder specific settings :
  strcpy( section,"WSQ" );
  WriteIntToIniFile(g_szIniFileName,section,"NumDecoders", pDecoder->nChannels );
  WriteDoubleToIniFile(g_szIniFileName,section,"LowestToneOnTX",pEncoder->fltLowestToneFreq);
  WriteIntToIniFile(g_szIniFileName,section,"SendDiddlesOnIdle",pEncoder->fDiddleOnTxIdle );
  WriteIntToIniFile(g_szIniFileName,section,"UseLongerFFT",pDecoder->fUseLongerFFT);
  WriteIntToIniFile(g_szIniFileName,section,"SymbolAvrg",  pDecoder->symbol_avrg_mode);
  WriteIntToIniFile(g_szIniFileName,section,"AGCMode",     pDecoder->agc_mode);
  WriteIntToIniFile(g_szIniFileName,section,"SynthActive", pEncoder->SynthesizerActive);
  // Note: The COM port number for the synthesizer (and PTT) seems to be loaded from setup.txt,
  //    so it doesn't make sense to store that information in WSQ_Config.ini (=the "GUI config").
  //
  for(i=0; i<WSQ_MAX_DECODER_CHANNELS; ++i )
   { sprintf( section,"Decoder%d",1+i );
     pChannel = &pDecoder->channel[i];
     WriteDoubleToIniFile(g_szIniFileName,section,"BaseFreq", pChannel->fltBaseFreq );
     WriteDoubleToIniFile(g_szIniFileName,section,"FreqSpan", pChannel->fltBandwidth);
     WriteIntToIniFile(g_szIniFileName,section,"sql_dB",pChannel->iSquelch_dB);
     WriteIntToIniFile(g_szIniFileName,section,"mode",     pChannel->iWSQmode );   // 4 (old) or 3 FFT bins per tone (WSQ, WSQCall) ?
     WriteIntToIniFile(g_szIniFileName,section,"algorithm",pChannel->iAlgorithm );
     WriteIntToIniFile(g_szIniFileName,section,"interactive", pChannel->fInteractive);
   }
} // end SaveSettingsInIniFile()


/*------------------------------------------------------------------------------
 *
 *      Window$ main()
 */

int WINAPI WinMain( HINSTANCE hThisInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpszArgument,
                    int nShowHow )
{
  int i,bw,sql,mode,algo;
  int x,y,h,w;
  float flt;
  char section[88], sz80[88];
  T_WSQDecoder *pDecoder = &WSQ_Decoder;
  T_WSQEncoder *pEncoder = &WSQ_Encoder;

  //HWND hwnd;              // this is the handle for our window
  MSG messages;             // here messages to the application are saved
  WNDCLASSEX wc;            // data structure for the windowclass

  InitCommonControls(); // loads common controls DLL
  WS_hInstance = hThisInstance;

  // If you also thought 'hThisInstance' can be used to find out
  // if "this" is the FIRST INSTANCE of this program running, or the 2nd, 3rd,.. : WRONG.
  // To find out the 'instance number' (which DL4YHF uses here for different
  // configuration files), use the old 'CreateMutex' trick, borrowed from Spectrum Lab:
  // > A mutex object is used because it can be detected by all applications
  // >  (it's a very "global" object). The system closes its handle
  // >  automatically when the process terminates.
  // >  The mutex object is destroyed when its last handle has been closed.
  // >  Must be executed before any other child window is created....
  g_iInstanceNr = 0;  // Instance counter. 0 = "i am the *FIRST* instance", etc
  while(g_iInstanceNr<10) // look for mutexes of already running instances..
   { sprintf( sz80, "WSQ_Inst%d", (int)(g_iInstanceNr) );
     g_hInstanceDetectionMutex = CreateMutex( NULL, FALSE, sz80 ); // PellesC: "invalid expression",
         // because '::CreateMutex()' doesn't belong in a *.c (not *.cpp) file.
         // Borland accepted '::CreateMutex()' as well as 'CreateMutex()' here.
     if( GetLastError()==ERROR_ALREADY_EXISTS )  // ex: ::GetLastError() -> PellesC: "invalid expression"
      { // the mutex for this instance already exists :
        //   FORGET the handle (it's not ours!) and try the next mutex in the next loop:
        if(g_hInstanceDetectionMutex!=NULL)
         { CloseHandle(g_hInstanceDetectionMutex); // closes THIS HANDLE to the mutex object,
           // but doesn't remove the mutex itself, because (from Win32 API):
           // > The mutex object is destroyed when its last handle has been closed.
         }
        ++g_iInstanceNr;  // 1 = "I AM THE *SECOND* INSTANCE" (array index!!), etc
      }
     else
      { // Mutex did NOT exist: break loop; APPL_iInstanceNr is valid ,
        // and leave "our" instance-detection-mutex as it is.
        // It will be removed when THIS instance terminates.
        break;  // found an "unoccupied" instance (-mutex)
      }
   } // end while < loop to detect the instance number, using MUTEXES >
  if( g_iInstanceNr > 0 )  // use different INI file names for the 2nd, 3rd, 4th instance of WSQ:
   { sprintf( g_szIniFileName, "WSQ_Config_%d.ini", (int)(g_iInstanceNr+1) );
     sprintf( g_szAppTitle, "WSQ2 [%d] Weak Signal QSO by ZL2AFP, adapted by DL4YHF, compiled %s", (int)(g_iInstanceNr+1), (char*)__DATE__ );
     sprintf(g_szShortTitle,"WSQ2 [%d] Weak Signal QSO", (int)(g_iInstanceNr+1) );
   }
  else
   { sprintf( g_szAppTitle, "WSQ2 Weak Signal QSO by ZL2AFP, adapted by DL4YHF, compiled %s", (char*)__DATE__ );
     strcpy(g_szShortTitle, "WSQ2 Weak Signal QSO" );
   }

  memset( &g_AnalysedWaveFile, 0, sizeof(g_AnalysedWaveFile) );
  memset( &g_RecordedWaveFile, 0, sizeof(g_RecordedWaveFile) );

  g_dw50MillisecondCounter = 0;
  iMouseOverArea = iDraggingArea = CLIENT_AREA_UNKNOWN;

#ifdef AUDIO_IO_H // option to use DL4YHF's Audio-I/O-DLL host for input ?
  fAudioInputAlive = FALSE;   // not receiving input from an external audio source yet
  AIO_Host_InitInstanceData( &AIO_Host );  // details about this in AudioIO.h !
  fUseAudioIOAsInput = fUseAudioIOAsOutput = FALSE;
  hwndAIOControlPanel = (HWND)0;
#endif // AUDIO_IO_H ?


  // Read a few settings (NOT for the synthesizer) from an old-fashioned ini file:
  strcpy( section,"AudioSettings" );
  ReadStringFromIniFile( g_szIniFileName,section,"InputDeviceName", "", g_sz255InputDevice, sizeof(g_sz255InputDevice)-1 );
  ReadStringFromIniFile( g_szIniFileName,section,"OutputDeviceName","", g_sz255OutputDevice,sizeof(g_sz255OutputDevice)-1 );
#ifdef AUDIO_IO_H     // optional support for DL4YHF's Audio-I/O-DLL-host as an input device ?
  fUseAudioIOAsInput = ReadIntFromIniFile(g_szIniFileName,section,"UseAudioIOAsInput",0);
  fUseAudioIOAsOutput= ReadIntFromIniFile(g_szIniFileName,section,"fUseAudioIOAsOutput",0);
  ReadStringFromIniFile(g_szIniFileName,section,"AudioIODllFileName",
      ""/*pszDefault*/, AudioIoDllFileName, sizeof(AudioIoDllFileName)-1 );
#endif // AUDIO_IO_H ?

  strcpy( section,"AudioFiles" );
  ReadStringFromIniFile(g_szIniFileName,section,"AnalysedWaveFile", "", AnalysedWaveFileName, sizeof(AnalysedWaveFileName)-1 );
  ReadStringFromIniFile(g_szIniFileName,section,"RecordedWaveFile", "", RecordedWaveFileName, sizeof(RecordedWaveFileName)-1 );
  fDecodeWaveFileInLoop = ReadIntFromIniFile(g_szIniFileName,section,"PlayInLoop",0);
  fDecodeWaveStatistics = ReadIntFromIniFile(g_szIniFileName,section,"ShowStatistics",0);

  strcpy( section,"Params" );
  g_WfallFmin_Hz = ReadIntFromIniFile(g_szIniFileName,section,"Wfall_fmin",  850/*Hz*/ );
  g_WfallFmax_Hz = ReadIntFromIniFile(g_szIniFileName,section,"Wfall_fmax", 1230/*Hz*/ );
  if( g_WfallFmax_Hz <= g_WfallFmin_Hz )
   { g_WfallFmin_Hz =  850.0;
     g_WfallFmax_Hz = 1230.0;
   }
  iWFBrightness_Percent= ReadIntFromIniFile(g_szIniFileName,section,"Wfall_brightness", 50/*percent*/ );
  iWFContrast_Percent = ReadIntFromIniFile(g_szIniFileName,section, "Wfall_contrast",   50/*percent*/ );
  iWFColorPalette     = ReadIntFromIniFile(g_szIniFileName,section, "Wfall_palette",   iWFColorPalette);
  fShowSpectrumGraph  = ReadIntFromIniFile(g_szIniFileName,section,"ShowSpectrumGraph",fShowSpectrumGraph);
  fShowSymbolMarkers  = ReadIntFromIniFile(g_szIniFileName,section,"ShowSymbolMarkers",fShowSymbolMarkers);
  fShowSymbolNumbers  = ReadIntFromIniFile(g_szIniFileName,section,"ShowSymbolNumbers",fShowSymbolNumbers);

  fAutoSwitchTxToRx   = ReadIntFromIniFile(g_szIniFileName,section,"AutoSwitchTxRx", TRUE );
  iComPortNr = ReadIntFromIniFile(g_szIniFileName,section,"COM_port_number",  iComPortNr );
  fReleaseCOMonRx = ReadIntFromIniFile(g_szIniFileName,section,"ReleaseCOMonRx", TRUE );
  fltCWSecondsPerDot = ReadDoubleFromIniFile(g_szIniFileName,section,"CW_sec_per_dot",fltCWSecondsPerDot);
  LimitFloat( &fltCWSecondsPerDot, 0.05/*normal CW*/, 30.0/*QRSS30*/ );
  ReadStringFromIniFile( g_szIniFileName,section,"CW_id_text", "", sz255CWid, sizeof(sz255CWid)-1 );
  iCWidAlways = ReadIntFromIniFile(g_szIniFileName,section,"CW_id_always", iCWidAlways );

  // WSQ-encoder and -decoder specific settings :
  strcpy( section,"WSQ" );
  i = ReadIntFromIniFile(g_szIniFileName,section,"NumDecoders",1);

  // Init the WSQ encoder and set its parameters to meaningful defaults :
  WSQ_InitEncoder( &WSQ_Encoder );

  // Init the 'common' data for the WSQ en- and decoder, and set meaningful defaults.
  WSQ_InitDecoder( &WSQ_Decoder,
                   g_iNominalSamplingRateForSoundcard, // initial sampling rate, may change later
                   WSQ_INPUT_REAL, // input options, future plan: flag for quadrature input
                   i ); // number of decoder channels
  // Some of the 'defaults' may be overwritten from the ini file:
  pEncoder->fltLowestToneFreq= ReadDoubleFromIniFile(g_szIniFileName,section,"LowestToneOnTX",pEncoder->fltLowestToneFreq);
  pEncoder->fDiddleOnTxIdle  = ReadIntFromIniFile(g_szIniFileName,section,"SendDiddlesOnIdle",pEncoder->fDiddleOnTxIdle);
  pDecoder->fUseLongerFFT    = ReadIntFromIniFile(g_szIniFileName,section,"UseLongerFFT",pDecoder->fUseLongerFFT);
  pDecoder->symbol_avrg_mode = ReadIntFromIniFile(g_szIniFileName,section,"SymbolAvrg",pDecoder->symbol_avrg_mode);
  pDecoder->agc_mode = ReadIntFromIniFile(g_szIniFileName,section,"AGCMode",pDecoder->agc_mode);
  pEncoder->SynthesizerActive = ReadIntFromIniFile(g_szIniFileName,section,"SynthActive", FALSE );

  // Individual settings for the WSQ decoders (to compare DIFFERENT settings side-by-side) :
  for(i=0; i<WSQ_MAX_DECODER_CHANNELS; ++i )
   { sprintf( section,"Decoder%d",1+i );
     flt= ReadDoubleFromIniFile(g_szIniFileName,section,"BaseFreq", 1000.0 );
     bw = ReadIntFromIniFile(g_szIniFileName,section,"FreqSpan", 80 + 30*i );
     sql= ReadIntFromIniFile(g_szIniFileName,section,"sql_dB",-30 );
     mode = ReadIntFromIniFile(g_szIniFileName,section,"mode", WSQ_MODE_WSQCall );
     algo = ReadIntFromIniFile(g_szIniFileName,section,"algorithm", WSQ_ALGO_ALTERN_1 );
     LimitFloat(  &flt, 300,15000 );  // ususally processing audio (300..3000 Hz), but who knows
     LimitInteger( &bw, 64,  3000 );  // WSQ2 requires at least 64 Hz, reject nonsense
     WSQ_InitDecoderChannel( &WSQ_Decoder, i/*channel index*/, flt, bw, sql, mode, algo );
     WSQ_Decoder.channel[i].fInteractive = ReadIntFromIniFile(g_szIniFileName,section,"interactive", WSQ_Decoder.channel[i].fInteractive );
   }
  WSQI_BeginReception();  // clear markers for the INTERACTIVE WSQ decoder (only once)

  // Set up the main window
  wc.cbSize        = sizeof (WNDCLASSEX);
  wc.style         = CS_HREDRAW | CS_VREDRAW; // or 0?
  wc.lpfnWndProc   = WindowProcedure;
  wc.cbClsExtra    = 0 ;
  wc.cbWndExtra    = 0 ;
  wc.hInstance     = hThisInstance;
  wc.hIcon         = LoadIcon(hThisInstance,TEXT("PROGRAM_ICON"));
  wc.hCursor       = NULL;//LoadCursor (NULL, IDC_ARROW);
  wc.hbrBackground = (HBRUSH) (COLOR_BTNFACE);  // black
  wc.lpszMenuName  = NULL;  // no menu yet (loaded later, in OnWmCreate(), from a 'resource')
  wc.lpszClassName = g_szClassName;
  wc.hIconSm       = LoadIcon(hThisInstance,TEXT("PROGRAM_ICON"));

  if (!RegisterClassEx (&wc))//(HBRUSH)GetStockObject(BLACK_BRUSH);//
  {
    MessageBox (NULL, "Window Registration Failed!", "Error!",
                MB_ICONEXCLAMATION | MB_OK);
    return 0;
  }

  // The class is registered, let's create the program's main window :
  strcpy( section,"MainWindow" );  // ... using position+size from previous session if possible
  y = ReadIntFromIniFile(g_szIniFileName,section,"Top",  10 );
  x = ReadIntFromIniFile(g_szIniFileName,section,"Left", 10 );
  w = ReadIntFromIniFile(g_szIniFileName,section,"Width",831);
  h = ReadIntFromIniFile(g_szIniFileName,section,"Height",573);
  Layout_iSplitPercent[0] = ReadIntFromIniFile(g_szIniFileName,section,"Rx1Splitter", Layout_iSplitPercent[0] );
  Layout_iSplitPercent[1] = ReadIntFromIniFile(g_szIniFileName,section,"Rx2Splitter", Layout_iSplitPercent[1] );
  Layout_iSplitPercent[2] = ReadIntFromIniFile(g_szIniFileName,section,"RxTxSplitter",Layout_iSplitPercent[2] );
  g_hwndMain = CreateWindowEx(  // here: WSQ main window..
           0,                             // use WS_EX_CLIENTEDGE for "sunken edge" (else 0)
           g_szClassName,                 // window class name
           g_szAppTitle,                  // title Text, e.g. "WSQ2 Weak Signal QSO by ZL2AFP"
           // DWORD dwStyle,        window style. DL4YHF tried to make the window sizeable...
            // WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX, // original by ZL2AFP
            // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX, // non-sizeable window
               WS_OVERLAPPEDWINDOW,  // quite normal window (guess with 'sizing border' even w/o WS_SIZEBOX)
            // WS_POPUP | WS_DLGFRAME ,   // window with no title, quite unusual as main window
            // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_SIZEBOX, // sizeable
            // WS_CAPTION,   // with title, but no icon and no system menu nor close symbol
            // WS_OVERLAPPEDWINDOW, WS_THICKFRAME,
           x,y,w,h,        // [in] position and size
           NULL,           // HWND hWndParent, "is optional for pop-up windows"
           NULL,           // No menu yet (loaded later, in OnWmCreate(), from a 'resource')
           hThisInstance,  // Program instance handle
           NULL );         // No Window creation data
  // Note (DL4YHF) : the 'child windows' are now created in the WM_CREATE handler, not here
  if( g_hwndMain == NULL )
   {
     MessageBox (NULL, "Window Creation Failed!", "Error!",
                 MB_ICONEXCLAMATION | MB_OK);
     return 0;
   }



  // Make the window visible on the screen
  ShowWindow ( g_hwndMain, SW_SHOW);//nShowHow );

  //  UpdateWindow (hwnd);// not needed since we use a timer tick for redraw


  // Run the message loop. It will run until GetMessage() returns 0
  while (GetMessage (&messages, NULL, 0, 0) > 0)
  {
    // Translate virtual-key messages into character messages
    TranslateMessage (&messages);
    // Send message to WindowProcedure
    DispatchMessage (&messages);
  }

  // The program return-value is 0 - The value that PostQuitMessage() gave

  return messages.wParam;

} // WinMain()

