/*
 *  File:    ?\WSQ2\WSQ_Interactive.c  (module prefix WSQI_)
 *  Date:    2014-03-27
 *  Author:  Wolf, DL4YHF (separated GUI and codec, added modifications).
 *  Purpose: WSQ 'Interactive' Decoder aka 'Frequency Shooter' .
 *           A 'click and decode' video game integrated in DL4YHF's WSQ GUI :)
 *           Seriously: The human eye/brain may be better suited to recognize
 *           very weak WSQ traces in a slowly scrolling spectrogram.
 *           Click into the spectrogram wherever you think(!) you see
 *           a WSQ signal, maybe you can 'beat' the machine and get
 *           better decode on a live WSQ transmission !
 *
 *
 *      #####################################################################
 *
 *
 *        This software is provided 'as is', without warranty of any kind,
 *        express or implied. In no event shall the author be held liable
 *        for any damages arising from the use of this software.
 *
 *        Permission to use, copy, modify, and distribute this software and
 *        its documentation for non-commercial purposes is hereby granted,
 *        provided that the above copyright notice and this disclaimer appear
 *        in all copies and supporting documentation.
 *
 *        The software must NOT be sold or used as part of any commercial
 *        or "non-free" product.
 *
 *      #####################################################################
 */

/*------------------------------------------------------------------------------
 *
 *
 *
 */

#ifdef __BORLANDC__
# pragma hdrstop
#endif

#include "switches.h" /* SWITCHES.H must be included before anything else! */
         /* Contains stuff like SWI_USE_OOURAS_FFT, SWI_USE_KISS_FFT, etc. */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include <windows.h>  // using HCURSOR and CreateCursor() here

#include "WSQ_Codec.h"         // WSQ_DecodeAudio(), WSQ_GenerateAudio(), ..
#include "WSQ_Interactive.h"   // header for the 'interactive' WSQ decoder

// Special cursor for the interactive decoder : Appears like a transparent rectangle.
// Height 2 + 16 + 2 = 20 pixels (16 waterfall lines per WSQ symbol + frame)
// Width: 2 +  5 + 2 = 9  pixels (transparent 'hole in the middle' should be odd)
//
// AND mask	XOR mask	Display
//    0      0        Black  (here: OUTER frame, 1 pixel)
//    0      1        White  (here: inner frame, 1 pixel)
//    1      0        Screen (here: transparent 'hole in the middle')
//    1      1        Reverse screen

#if(0)  // complete frame
static const BYTE ANDmaskCursor[] =  // outer frame BLACK, inner frame WHITE, interior TRANSPARENT
{
  0x00,0x7F,    // 0  :  0 0 0 0  0 0 0 0  0 1 1 1  1 1 1 1  "AND": 0 = black or white but not transparent
  0x00,0x7F,    // 1  :  0 0 0 0  0 0 0 0  0 1 1 1  1 1 1 1  "AND": 0 = black or white but not transparent
  0x3E,0x7F,    // 2  :  0 0 1 1  1 1 1 0  0 1 1 1  1 1 1 1  "AND": 1 = transparent
  0x3E,0x7F,    // 3
  0x3E,0x7F,    // 4
  0x3E,0x7F,    // 5
  0x3E,0x7F,    // 6
  0x3E,0x7F,    // 7
  0x3E,0x7F,    // 8
  0x3E,0x7F,    // 9
  0x3E,0x7F,    // 10
  0x3E,0x7F,    // 11
  0x3E,0x7F,    // 12
  0x3E,0x7F,    // 13
  0x3E,0x7F,    // 14
  0x3E,0x7F,    // 15
  0x3E,0x7F,    // 16
  0x3E,0x7F,    // 17
  0x00,0x7F,    // 18
  0x00,0x7F     // 19
};

// rectangle-shaped cursor XOR mask
static const BYTE XORmaskCursor[] =  // outer frame BLACK, inner frame WHITE, interior TRANSPARENT
{
  0x00,0x00,  // 0  :  0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = black, 1 = white (with "AND"=0)
  0x7F,0x00,  // 1  :  0 1 1 1  1 1 1 1  0 0 0 0  0 0 0 0  "XOR": 0 = black, 1 = white (with "AND"=0)
  0x41,0x00,  // 2  :  0 1 0 0  0 0 0 1  0 0 0 0  0 0 0 0
  0x41,0x00,  // 3
  0x41,0x00,  // 4
  0x41,0x00,  // 5
  0x41,0x00,  // 6
  0x41,0x00,  // 7
  0x41,0x00,  // 8
  0x41,0x00,  // 9
  0x41,0x00,  // 10
  0x41,0x00,  // 11
  0x41,0x00,  // 12
  0x41,0x00,  // 13
  0x41,0x00,  // 14
  0x41,0x00,  // 15
  0x41,0x00,  // 16
  0x41,0x00,  // 17
  0x7F,0x00,  // 18
  0x00,0x00   // 19
};
#else // just the "four corners" of the frame :
static const BYTE ANDmaskCursor[] =  // "Four Corner"-shape cursor with a lot of transparency
{               //                |center pixel
  0x1C,0x7F,    // 0  :  0 0 0 1  1 1 0 0  0 1 1 1  1 1 1 1  "AND": 0 = black or white but not transparent, 1=transparent
  0x1C,0x7F,    // 1  :  0 0 0 1  1 1 0 0  0 1 1 1  1 1 1 1  "AND": 0 = black or white but not transparent
  0x3E,0x7F,    // 2  :  0 0 1 1  1 1 1 0  0 1 1 1  1 1 1 1  "AND": 1 = transparent
  0x3E,0x7F,    // 3
  0x3E,0x7F,    // 4
  0x3E,0x7F,    // 5
  0xFF,0xFF,    // 6   :  1 1 1 1  1 1 1 1  1 1 1 1  1 1 1 1  "AND": 1 = transparent
  0xFF,0xFF,    // 7   :  1 1 1 1  1 1 1 1  1 1 1 1  1 1 1 1  "AND": 1 = transparent
  0xFF,0xFF,    // 8   :  1 1 1 1  1 1 1 1  1 1 1 1  1 1 1 1  "AND": 1 = transparent
  0xFF,0xFF,    // 9   :  1 1 1 1  1 1 1 1  1 1 1 1  1 1 1 1  "AND": 1 = transparent
  0xFF,0xFF,    // 10  :  1 1 1 1  1 1 1 1  1 1 1 1  1 1 1 1  "AND": 1 = transparent
  0xFF,0xFF,    // 11  :  1 1 1 1  1 1 1 1  1 1 1 1  1 1 1 1  "AND": 1 = transparent
  0xFF,0xFF,    // 12  :  1 1 1 1  1 1 1 1  1 1 1 1  1 1 1 1  "AND": 1 = transparent
  0xFF,0xFF,    // 13  :  1 1 1 1  1 1 1 1  1 1 1 1  1 1 1 1  "AND": 1 = transparent
  0x3E,0x7F,    // 14
  0x3E,0x7F,    // 15
  0x3E,0x7F,    // 16
  0x3E,0x7F,    // 17
  0x1C,0x7F,    // 18
  0x1C,0x7F     // 19
};

// Four-Corner-shaped cursor XOR mask
static const BYTE XORmaskCursor[] =  // "Four Corner" cursor... outer frame BLACK, inner frame WHITE, interior TRANSPARENT
{             //                |center pixel
  0x00,0x00,  // 0  :  0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = black, 1 = white (with "AND"=0)
  0x63,0x00,  // 1  :  0 1 1 0  0 0 1 1  0 0 0 0  0 0 0 0  "XOR": 0 = black, 1 = white (with "AND"=0)
  0x41,0x00,  // 2  :  0 1 0 0  0 0 0 1  0 0 0 0  0 0 0 0  "XOR": 0 = transparent (with "AND"=1)
  0x41,0x00,  // 3
  0x41,0x00,  // 4
  0x41,0x00,  // 5
  0x00,0x00,  // 6  : 0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = transparent (with "AND"=1)
  0x00,0x00,  // 7  : 0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = transparent (with "AND"=1)
  0x00,0x00,  // 8  : 0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = transparent (with "AND"=1)
  0x00,0x00,  // 9  : 0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = transparent (with "AND"=1)
  0x00,0x00,  // 10 : 0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = transparent (with "AND"=1)
  0x00,0x00,  // 11 : 0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = transparent (with "AND"=1)
  0x00,0x00,  // 12 : 0 0 0 0  0 0 0 0  0 0 0 0  0 0 0 0  "XOR": 0 = transparent (with "AND"=1)
  0x41,0x00,  // 13
  0x41,0x00,  // 14
  0x41,0x00,  // 15
  0x41,0x00,  // 16
  0x41,0x00,  // 17
  0x63,0x00,  // 18
  0x00,0x00   // 19
};
#endif // frame rectangle or just "four corners" for the mouse pointer ?

#define WSQI_MAX_POINTS 200
T_WSQI_Point WSQI_Points[WSQI_MAX_POINTS];
BOOL         WSQI_fRedrawMarkers;
BOOL         WSQI_fGotNewCharacter;

//---------------------------------------------------------------------------
CPROT void WSQI_BeginReception(void)  // actually CLEARS ALL MARKERS !
{
  int i;
  memset( &WSQI_Points, 0, sizeof(WSQI_Points) );
  for(i=0; i<WSQI_MAX_POINTS; ++i )
   { WSQI_Points[i].i32SpectrumIndex = SPECTRUM_INDEX_UNUSED;
   }
  WSQI_fRedrawMarkers = TRUE;
} // end WSQI_BeginReception()


//---------------------------------------------------------------------------
HCURSOR WSQI_CreateFrameCursor( HINSTANCE hInst )
{ // Creates a custom cursor (mouse pointer) at run time.
  // No fooling around with antiquated resource- or icon editors.
  return CreateCursor( hInst,   // app. instance
             3,                 // horizontal position of hot spot
             10,                // vertical position of hot spot
             16,                // cursor width
             20,                // cursor height
             ANDmaskCursor,     // AND mask
             XORmaskCursor );   // XOR mask
} // end WSQI_CreateFrameCursor()


//---------------------------------------------------------------------------
void WSQI_EnterSymbol( float freq_Hz, long  iSpectrumIndex )
  // Called from WSQI_OnMouseUp() when the operator clicked into the waterfall,
  // but also from WSQ_DecodeAudio() when the AUTOMATIC decoder has detected
  // a new symbol (which was above the squelch level) .
{
  int i, j, /*curr_char,*/ curr_nibble, prev_nibble;
  float flt, prev_freq_Hz;
  double dblToneSpacing_Hz = (WSQ_Decoder.channel[0].iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */



  // Insert the new point in the array .
  // The OLDEST entry, with the lowest 'spectrum index', is at array index ZERO.
  // Keep WSQI_Points[] sorted by increasing 'spectrum index' (waterfall line counter).
  // When running out of enties in WSQI_Points[], the OLDEST entry is removed.
  i = 0;
  while( i<WSQI_MAX_POINTS )
   {
     if( WSQI_Points[i].i32SpectrumIndex == iSpectrumIndex )
      { // REPLACE this item (don't insert another one with the same index) :
        WSQI_Points[i].freq_Hz = freq_Hz;
        WSQI_Points[i].delta_f = 0;
        WSQI_Points[i].decoded_char = 0;
        WSQI_fRedrawMarkers = TRUE;
        break;
      }
     else if( WSQI_Points[i].i32SpectrumIndex > iSpectrumIndex )
      { // new entry must be inserted BEFORE [i] ('out of sequence', most likely a MANUAL edit)
        for(j=WSQI_MAX_POINTS-1; j>i; --j)
         { WSQI_Points[j] = WSQI_Points[j-1]; // scroll items at the end further down
         }
        WSQI_Points[i].freq_Hz = freq_Hz;
        WSQI_Points[i].delta_f = 0;
        WSQI_Points[i].decoded_char = 0;
        WSQI_Points[i].i32SpectrumIndex = iSpectrumIndex;
        WSQI_fRedrawMarkers = TRUE;
        break;
      }
     ++i;
   }
  if( i>=WSQI_MAX_POINTS ) // all entries in WSQI_Points[] are occupied ->
   { // remove the OLDEST entry (lowest index), scroll up the rest,
     // and set the new entry at the end (highest index).
     for(i=0; i<WSQI_MAX_POINTS-1; ++i)
      { WSQI_Points[i] = WSQI_Points[i+1];
      }
     WSQI_Points[WSQI_MAX_POINTS-1].freq_Hz = freq_Hz;
     WSQI_Points[i].delta_f = 0;
     WSQI_Points[i].decoded_char = 0;
     WSQI_Points[WSQI_MAX_POINTS-1].i32SpectrumIndex = iSpectrumIndex;
   }
  // At this point, WSQI_Points[] is again sorted by 'age' (spectrum index).
  // The OLDEST index, with the LOWEST spectrum index, is at WSQI_Points[0] .
  // Try to decode as many characters as possible. The result may(!) be displayed
  // in the spectrogram.
  curr_nibble  = prev_nibble = 0;
  prev_freq_Hz = WSQI_Points[0].freq_Hz;
  for(i=0; i<WSQI_MAX_POINTS; ++i )  // decode again, beginning with the OLDEST entry
   { if( WSQI_Points[i].i32SpectrumIndex == SPECTRUM_INDEX_UNUSED )
      { break;  // reached the end of the list; no further decodes are possible
      }
     if( i>0 )
      {
        flt = (WSQI_Points[i].freq_Hz - prev_freq_Hz) / dblToneSpacing_Hz;
        WSQI_Points[i].delta_f = flt;
        curr_nibble = RoundFloat( flt );  // round, don't truncate !
        if(curr_nibble < 0)
         { curr_nibble += 33;  // allow for wrap-around (17 tones for 16 tone differences)
         }
        --curr_nibble;   // 1 is added at the TX end so need to subtract it here
        // Because single-symbol characters will be delayed by one symbol in WSQ_DecodeVaricode(),
        // "date them back" here [i-1].
        // Effect: When moving one of the markers on the spectrogram,
        // the character printed right next to it, and the SUBSEQUENT character
        // will change - not the TWO subsequent characters !
        // Try yourself by moving a character 'in the middle' of the spectrogram.
        WSQI_Points[i-1].decoded_char = WSQ_DecodeVaricode( curr_nibble, prev_nibble );
      }
     prev_freq_Hz = WSQI_Points[i].freq_Hz;
     prev_nibble  = curr_nibble;
   }
  WSQI_fGotNewCharacter = TRUE;

} // end WSQI_EnterSymbol()


//---------------------------------------------------------------------------
void WSQI_OnMouseDown(
       float fltClickedFreq, // [in] audio frequency in Hertz
       long  iSpectrumIndex) // [in] unique index to identify the clicked spectrum (~FFT counter)
{
  int i,j;

  if( (WSQI_Points[0].i32SpectrumIndex==0) && (WSQI_Points[1].i32SpectrumIndex==0) )
   { WSQI_BeginReception();  // init array (if caller didn't do this explicitly)
   }
  // If the time (spectrum index) is close to an already existing point,
  // REMOVE the old point from the list:
  i = 0;
  while( i<WSQI_MAX_POINTS )
   {
     if(  (WSQI_Points[i].i32SpectrumIndex > (iSpectrumIndex-5) )
        &&(WSQI_Points[i].i32SpectrumIndex < (iSpectrumIndex+5) ) )
      { // REMOVE this item from the list :
        for(j=i; j<(WSQI_MAX_POINTS-1); ++j )
         { WSQI_Points[j] = WSQI_Points[j+1];
         }
        WSQI_Points[WSQI_MAX_POINTS-1].i32SpectrumIndex = SPECTRUM_INDEX_UNUSED;
        WSQI_fRedrawMarkers = TRUE;
      }
     else
      { ++i;
      }
   }
} // end WSQI_OnMouseDown()


//---------------------------------------------------------------------------
void WSQI_OnMouseUp(
       float fltClickedFreq, // [in] audio frequency in Hertz
       long  iSpectrumIndex) // [in] unique index to identify the clicked spectrum (~FFT counter)
{
  WSQI_EnterSymbol( fltClickedFreq, iSpectrumIndex );
} // end WSQI_OnMouseUp()

//---------------------------------------------------------------------------
T_WSQI_Point *WSQI_GetMarkerForWFall(
       long iSpectrumIndex ) // [in] highest (newest) spectrum index
  // Function tailored for the waterfall-update-function in WSQ_MainWin.c !
  // Returns the address of T_WSQI_Point if a marker should be painted
  //                     into this line of the spectrogram, otherwise NULL .
{
  int i;
  for(i=0; i<WSQI_MAX_POINTS; ++i )
   {
     if( WSQI_Points[i].i32SpectrumIndex == iSpectrumIndex )
      { return &WSQI_Points[i];
      }
   }
  return NULL;
} // end WSQI_GetMarkerForWFall()


//---------------------------------------------------------------------------
void WSQI_OnNewSpectrum( T_WSQDecoder *pDecoder, int iChannel,
        void (*pRxCharWriter)     // [in] callback to write received characters into the receive text editor
              (T_WSQDecoder *pDecoder, int iChannel, int iFlags, char *pszText) )
  // Called from WSQ_DecodeAudio() to inform the 'interactive' WSQ decoder
  // about a new spectrum which has just been appended to the spectrum buffer.
  // The OLDEST spectrum will soon be scrolled out of view (in the spectrogram),
  // and if it contained a decoded characters (automatic or interactive),
  // that character will now be transferred from the LAST LINE IN THE RX TEXT
  // to the older lines (where it cannot be modified anymore by clicking
  // into the spectrogram to move the WSQ symbol markers) .
{
  int i;
  char c2[2];
  long i32SpectrumIndex = pDecoder->i32SpectrumCounter
                        - pDecoder->nLinesVisibleInSpectrogram;
  // Run through all entries in WSQI_Points[] .
  // If there's a character decoded for the scrolled-out line, emit it.
  for(i=0; i<WSQI_MAX_POINTS; ++i)
   { if( WSQI_Points[i].i32SpectrumIndex == i32SpectrumIndex )
      { if( WSQI_Points[i].decoded_char > 0 )
         { c2[0] = (char)WSQI_Points[i].decoded_char;
           c2[1] = '\0';
           pRxCharWriter( pDecoder, iChannel, WSQ_TEXT_FLAG_FINAL, c2 );  // write the 'interactively' decoded  character into the RX window
           // (the GUI will insert this character
           //   BEFORE the "last line" with the intermediate decoder output )
         }
      }
   }
} // end WSQI_OnNewSpectrum()

//---------------------------------------------------------------------------
void WSQI_GetTextForTicker( T_WSQDecoder *pDecoder, int iChannel,
        char *psz255Dest )  // [out] text for the "last line" of the RX text window, without line-breaks
  // Retrieves the text for the "last line" in the RX window
  // when using the interactive decoder .
{
  int i, chr, nCharsEmitted=0;
  long i32OldestSpectrumIndex =
           pDecoder->i32SpectrumCounter // index of the spectrum on top of the spectrogram (="newest")
         - pDecoder->nLinesVisibleInSpectrogram;
  // Run through all entries in WSQI_Points[] ...
  // > The OLDEST index, with the LOWEST spectrum index, is at WSQI_Points[0] .
  //
  nCharsEmitted = 0;
  for(i=0; i<WSQI_MAX_POINTS && nCharsEmitted<250; ++i)
   { if(   (WSQI_Points[i].i32SpectrumIndex != SPECTRUM_INDEX_UNUSED)
        && (WSQI_Points[i].i32SpectrumIndex > i32OldestSpectrumIndex) )  // "sufficiently NEW spectrum" ?
      {
        // If there's a character decoded, append it to the string.
        // Replace 'constrol characters' like Carriage Return and New Line by dummies (spaces).
        if( (chr=WSQI_Points[i].decoded_char) > 0 )
         { switch(chr)
            { case 10:
                 strcpy( psz255Dest+nCharsEmitted, "<NL>" );
                 nCharsEmitted += 4;
                 break;
              case 13:
                 strcpy( psz255Dest+nCharsEmitted, "<CR>" );
                 nCharsEmitted += 4;
                 break;
              default:
                 psz255Dest[nCharsEmitted++] = (char)chr;
                 break;
            }
         }
      }
   }
  psz255Dest[nCharsEmitted] = '\0';

} // end WSQI_GetTextForTicker()



