/*
 *  File:    ?\WSQ2\WSQ_Codec.h
 *  Date:    2017-12-03 (made WSQ2 compatible with WSQCall, with 1.46484375 Hz tone spacing)
 *  Authors: Con,  ZL2AFP (inventor and author of the original code),
 *           Wolf, DL4YHF (separated GUI and codec, added a few modifications).
 *  Purpose: WSQ (Weak Signal QSO) Encoder / Decoder .
 *   Formerly part of  main_wsq_2 _seconds.c   (=original code)  by ZL2AFP .
 *   Moved into an extra module to separate GUI and 'algorithms' by DL4YHF .
 *   Since this codec is assumed to support different symbol lenghts
 *   sooner or later, the module prefix is "WSQ", not "WSQ2" .
 *
 *      #####################################################################
 *
 *
 *        This software is provided 'as is', without warranty of any kind,
 *        express or implied. In no event shall the author be held liable
 *        for any damages arising from the use of this software.
 *
 *        Permission to use, copy, modify, and distribute this software and
 *        its documentation for non-commercial purposes is hereby granted,
 *        provided that the above copyright notice and this disclaimer appear
 *        in all copies and supporting documentation.
 *
 *        The software must NOT be sold or used as part of a any commercial
 *        or "non-free" product.
 *
 *      #####################################################################
 */

#ifndef _WSQ_CODEC_H_
#define _WSQ_CODEC_H_

#define USE_KISS_FFT 1 /* use Mark Borgerding's radix-N KISS FFT ? 1=yes, 0=no */
    // Even though the CPU usage in WSQ2 is not dominated by the CPU (yet?),
    // use KISS FFT because it's a bit faster than the old textbook-style FFT.
    // See benchmarks in WSQ_Codec.c !

#if ( USE_KISS_FFT )
// Here we use the FFT with real (not complex) input -> use kiss_fftr, not just kiss_fft :
# include "kiss_fftr.h" // KISS FFT for REAL input (but complex output, of course!)
#endif // use KISS FFT ?

#include "IIR_Bandpass.h"  // bandpass used by 'Anti-AGC' and noise blanker

#ifndef  TRUE  /* types which used to be defined in windef.h : */
# define TRUE  1
# define FALSE 0
  typedef unsigned long  DWORD;
  typedef int            BOOL;
  typedef unsigned char  BYTE;
  typedef unsigned short WORD;
#endif

#define WSQ_MAX_DECODER_CHANNELS 2


#define WSQ_OLD_TONE_SPACING 1.953125 /* single tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz, i.e. 4 FFT bins per tone */
#define WSQCall_TONE_SPACING 1.46484375 /* single tone spacing FOR WSQCall (2017): (3/2.048) Hz,   3 FFT bins per tone */
#define WSQ2_NUM_TONES   33      /* total number number of IFK tones   */
#define WSQ2_SYMBOL_TIME 2.048   /* seconds per symbol (fixed!)        */

#define FFT_MAX_INPUT_SIZE 65536 // time domain input points for the FFT (MAXIMUM)
#define FFT_MAX_FREQ_BINS (FFT_MAX_INPUT_SIZE/2+1) // number of bins in FFT output (maximum)
#define SPEC_BUF_MAX_BINS 16384  // should be enough for the usual 'SSB' audio range

#define WSQ_SLICES_PER_SYMBOL 16  // eached symbol is 'sliced' into 16 sampling points (in the decoder)
    // Since one symbol is 2.048 seconds long, a single 'slice' is 128 ms .
    // This is also the (fixed) scrolling speed of the waterfall (circa 8 lines per second) .

typedef struct tSpecBufEntry
{
  float fltCenterFreqOf1stBin;
  float fltFreqBinWidth_Hz;
  float fltFreqBins[ SPEC_BUF_MAX_BINS ];  // array of FFT frequency bins [voltages], used for plotting the spectrum or waterfall
  signed char i8FreqOffsets[ SPEC_BUF_MAX_BINS ]; // array of frequency offsets for increased frequency resolution;
         // unit: 1/127 th FFT bin width (scaled for signed 8-bit integer, offset range = +/- a single FFT bin).
         // May be added to the theoretic 'bin center frequency' for improved frequency accuracy.
         // Resolution approx. 0.488 Hz (=bin width) / 127 = 3.8 mHz .
  int   nUsedBins;
    // Added for convenience (plotted along with the spectrum) : Decoder data..
  float fltSymbolPeakFreq[WSQ_MAX_DECODER_CHANNELS];   // displayed as overlay in the waterfall
  float fltSymbolPeakMag_dB[WSQ_MAX_DECODER_CHANNELS];
  char  sz15DecoderInfo[WSQ_MAX_DECODER_CHANNELS][16]; // text to be displayed in the waterfall
         //  Usually an empty string, but may contain a 'tone number'
         //  or 'quality indicator' once per symbol.
         //  Content may depends on the settings,
         //  for example if the option 'Show Symbol Numbers' is selected in the GUI.
  long  i32SpectrumIndex; // non-circular counter of lines emitted to the waterfall
                          // (used as a timebase for the 'interactive' WSQ decoder)


} T_SpecBufEntry;

typedef struct tWSQPeakHistoryEntry  // used for symbol synchronisation by DL4YHF's decoder algorithm
{
  float frequency; // [Hz]
  float magnitude; // [dBfs]
} T_WSQPeakHistoryEntry;

typedef struct tWSQDecoderChannel // for a multi-channel WSQ decoder ...
{           // .. put everything which applies to ONE DECODER CHANNEL here:
   // Configuration parameters, saved in a file between two sessions :
   float fltBaseFreq;  // audio frequency of the lowest WSQ tone, originally 1000 Hz, saved in the configuration (integer!)
   float fltBandwidth; // decoder's allowed frequency range (span) in Hz, also saved in config
   int   iSquelch_dB; // decoder's squelch level in dB (SNR), saved in config

   int   iWSQmode;    // one of the following:
#        define WSQ_MODE_OLD      0  /* compatible with "Old" WSQ, 4 bins per tone, 4/2.048s = 1.953125 Hz tone spacing */
#        define WSQ_MODE_WSQCall  1  /* compatible with "Old" WSQ, 3 bins per tone, 3/2.048s = 1.46484375 Hz tone spacing */

   int   iAlgorithm;  // one of the following:
#        define WSQ_ALGO_ORIGINAL 0
#        define WSQ_ALGO_ALTERN_1 1
   BOOL  fInteractive; // aka 'Frequency Shooter', can be combined with any of the above algorithms

   // Derived from the configuration :
   float fmin_Hz, fmax_Hz;  // frequency range at which THIS decoder may look (derived from the above)

   // Used during operation, not part of the configuration :
   float ss, tmp, pk, agc_decay, agcthr, agcgain;
   float fltSig_to_Noise_dB;
   int   iPeakBinIndex, iPrevPeakBin;      // << for the 'relatively original' algorithm
   float fltPeakFreqHz, fltPrevPeakFreqHz; // << for DL4YHF's modified algorithm
   T_WSQPeakHistoryEntry sPeakMagHistory[WSQ_SLICES_PER_SYMBOL]; //   "  "  "  "
   int iSymbolInhibitCountdown; // only used by the DL4YHF decoder, inhibits the emission
            // of a new symbol if the previous one is less than a half(!) symbol time ago
   int peak_counter;
   int curr_nibble, prev_nibble;
   int symbol_bin_index;  // FFT frequency *bin index* of the last decoded symbol.
       // in the original code the name was just 'symbol' which is misleading.
   int prev_symbol_bin_index;
   float fltSymbolPeakFreq;

} T_WSQDecoderChannel;

typedef struct tWSQDecoder  // 'Common' data shared by all decoders using the same FFT for input..
{
  int   nChannels; // number of decoder channels fed from the same FFT, 1..WSQ_MAX_DECODER_CHANNELS
  int   symbol_avrg_mode; // affects the WATERFALL DISPLAY but also to ALL decoders. Possible settings:
#       define WSQ_SYM_AVRG_OFF  0
#       define WSQ_SYM_AVRG_FAST 1
#       define WSQ_SYM_AVRG_SLOW 2

  int   peak_hits;          // configuration parameter, applies to ALL decoders

  int   agc_mode;   // one of the following:
#       define WSQ_AGC_OFF  0
#       define WSQ_AGC_SLOW 1
#       define WSQ_AGC_FAST 2
  int   nb_enabled; // noiseblanker enabled (TRUE,FALSE)
  int   iAGC_NB_StartFreq, iAGC_NB_EndFreq; // Frequency range for the AGC- and noiseblanker reference.
                    // There should be ONLY NOISE (atmospheric) but no man-made signals in this range.
  float fltAGC_ss;  // not sure what 'ss' stands for but it's a "working register" for the AGC. Possibly 'slow sample'.
  float fltAGC_pk;  // AGC peak value, slowly decaying in WSQ_DecodeAudio()

  float fltInputSamplingRate; // may change when samples are fed into the decoder
  BOOL  fUseLongerFFT;   // 'FFT window longer than WSQ symbol' ? Originally just a TEST (2014-03-15); details in WSQ_Codec.c
  int   iFftInputSize;   // number of samples in the time domain entering each FFT
  int   iDecimation;     // 1:decimate by one (i.e. don't decimate)  or
                         // 4:decimate by four (as in ZL2AFP's original code)
  int   iFftNumFreqBins; // number of frequency bins (in the output from each FFT)
  float fltFftBinWidth;  // width of each frequency bin [Hz]

#if( USE_KISS_FFT )
  kiss_fftr_cfg kiss_FFTR_object; // black box created by kiss_fftr_alloc()
  kiss_fft_cpx  complexFreqBins[FFT_MAX_FREQ_BINS];
#endif // use KISS FFT ?

  // Spectrum buffer, also used to plot the waterfall (in WSQ_MainWin.c) :
  T_SpecBufEntry SpecBufDummy;
  T_SpecBufEntry *SpectrumBuffer;
  int   nSpecBufAllocEntries; // number of allocated entries
  int   nSpecBufUsedEntries; // number of 'used' entries, runs from 0 to nSpecBufAllocEntries
  int   iSpecBufIndex;       // circular buffer index, runs from 0 to nSpecBufAllocEntries-1
  long  i32SpectrumCounter;  // non-circular counter of lines emitted to the waterfall
                             // (used as a timebase for the 'interactive' WSQ decoder)

  int   nLinesVisibleInSpectrogram; // required for the INTERACTIVE decoder
                             // to emit decoders 'finally' to the RX text window
                             // when a certain entry in the spectrum buffer
                             // is scrolled out of view on the bottom of the spectrogram.

  DWORD dwMagic31415926;     // magic number, set when buffer has been allocated


  // YHF: The original code split each symbol into 16 'slices' .
  //      Each symbol is (always) 2.048 seconds long (in WSQ2) .
  //      Each slice (of a symbol) was 128 milliseconds long .
  //      Due to the raised-cosine-like window, the center slice
  //      should theoretically get the largest amplitude of the entire symbol.
  //      In future versions, there may possibly be less slices per symbol.
  int    iSampleCounter, iDecimateSampleCounter, iSampleOfSlice;
  int    iDecimatorIdxIn;
  float  fltDecimatorQueueI[64];
  float  fltFFTWindow[FFT_MAX_INPUT_SIZE];
  float  fltDecimatedInput[FFT_MAX_INPUT_SIZE*2+2]; // two times larger than necessary because this may be COMPLEX one day ?
  float  fltSmoothedFFT[FFT_MAX_FREQ_BINS];

  float  fltFftBufRe[FFT_MAX_INPUT_SIZE*2+2]; // buffers for FFT on raw input samples
  float  fltFftBufIm[FFT_MAX_INPUT_SIZE*2+2]; // (used for input in the time domain AND for output in the frequency domain)
  float  fltPrevFftBinsRe[FFT_MAX_FREQ_BINS]; // complex frequency bins from PREVIOUS FFT
  float  fltPrevFftBinsIm[FFT_MAX_FREQ_BINS]; // (used for precise frequency measurements based on phases)


  int    hangcnt; // used for the AGC

  T_IIR_Filter iirFilter; // IIR bandpass filter used by 'Anti-AGC' and noise blanker

  T_WSQDecoderChannel channel[WSQ_MAX_DECODER_CHANNELS];

} T_WSQDecoder;  // so far, only ONE common instance of this :  WSQ_Decoder

typedef struct tWSQEncoder
{
  // Configuration :
  BOOL  fDiddleOnTxIdle; // TRUE:send "diddles" when transmitter in idle (RTTY slang),
                         // FALSE:send an unmodulated tone (which gives no sync info).
  float fltLowestToneFreq;  // frequency of the lowest tone in Hertz. Originally 1000.0 .
    // Made adjustable (via GUI) when the 'default WSQ frequency' was occupied by JT9-1 .

  // Used during operation, not part of the configuration (would be 'private' in C++):
  int   nSamplesPerSymbol;
  int   iSampleOfSymbolCounter;
  BOOL  SynthesizerActive;

  int   iTxNibble1, iTxNibble2;  // first and optional 2nd nibble of character, for TX
  float phase, fAngle;
  short curslot, prevslot;
  short tone, prevtone; // range: 0..32 (because we have 33 different tones)
 
} T_WSQEncoder;  // so far, only ONE common instance of this :  WSQ_Encoder

extern T_WSQEncoder WSQ_Encoder;
extern T_WSQDecoder WSQ_Decoder;

//---------------- Prototypes -----------------------------------------------

// For peaceful coexistence between "C" and "C++" :
#ifndef CPROT
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  /* ! "cplusplus" */
#endif


CPROT void LimitInteger( int *piValue,  int min, int max );
CPROT void LimitFloat( float *fltValue, float min, float max );
CPROT int  RoundFloat( double number );

CPROT float WSQ_VoltageRatioTodB( float fltVoltageRatio ); // used by some of DL4YHF's decoder variants..
CPROT BOOL  WSQ_CalcFftPeakMagnitudeAndFrequency( // but may be used directly by the GUI (and others)
                  T_SpecBufEntry *pSpectrum, // [in] spectrum (with info header)
                  int    iPeakBinIndex,      // [in] frequency bin index of the 'peak'
                  float  *pfltPeakMag_dB,    // [out] peak magnitude  [dB]
                  float  *pfltPeakFrequency);// [out] peak frequency  [Hz]

CPROT void SendToSynthesizer( int iTone/*0..32*/ ); // called from WSQ2_Codec() during transmit, when SynthesizerActive==TRUE.
                  // Since 2014-03-26, also called from CW_Generator.c to send the optional CW ID.
                  // Implemented in the main module, not in any of the 'generators'.

CPROT void WSQ_InitDecoder( T_WSQDecoder *pDecoder,
                      float fltInitialSamplingRate,
                      int inputOptions, // future plan: flag for quadrature input, etc:
#                      define WSQ_INPUT_REAL    0
#                      define WSQ_INPUT_COMPLEX 1
                      int nDecoderChannels );
CPROT void WSQ_DeleteDecoder( T_WSQDecoder *pDecoder ); // clean up, free memory                      

CPROT void WSQ_InitDecoderChannel( T_WSQDecoder *pDecoder,
         int   iChannel,    // 0..WSQ_MAX_DECODER_CHANNELS-1, used to identify different decoder output windows
         float fltBaseFreq, // audio frequency of the lowest WSQ tone, originally 1000 Hz
         float fltBandwidth,  // decoder's allowed frequency range (span) in Hz
         int   iSquelch_dB, // squelch level, dB SNR (typically -25 or so..)
         int   iMode,       // WSQ_MODE_OLD or WSQ_MODE_WSQCall (4 or 3 FFT-bins per tone spacing; added 2017-12-03)
         int   iAlgorithm); // WSQ_ALGO_ORIGINAL or WSQ_ALGO_ALTERN_1, maybe more in future

CPROT void WSQ_SetDecoderBaseFreq( T_WSQDecoder *pDecoder,  // "base frequency" := "tone zero"
         int   iChannel,    // 0..WSQ_MAX_DECODER_CHANNELS-1
         float fltFbase);   // audio frequency of the lowest WSQ tone, originally 1000 Hz

CPROT void WSQ_SetDecoderBandwidth( T_WSQDecoder *pDecoder,
         int   iChannel,      // 0..WSQ_MAX_DECODER_CHANNELS-1
         float fltBandwidth); // decoder's allowed frequency range (span) in Hz

         // Alternative methods to set the decoder passband, used by mouse event handler:
CPROT void WSQ_SetDecoderFmin( T_WSQDecoderChannel *pChannel, float fmin );
CPROT void WSQ_SetDecoderFmax( T_WSQDecoderChannel *pChannel, float fmax );

CPROT void WSQ_BeginReception(T_WSQDecoder *pDecoder); // prepares / clears all decoders before an rx-over
         // Not really necessary, but used during file analysis to provide a 'fresh start'.
         // Resets low-pass filters, symbol averages, integrators, and similar stuff.

CPROT int WSQ_DecodeVaricode( int curr_nibble, int prev_nibble );         

CPROT int WSQ_DecodeAudio(  // returns the number of symol slices receive (= number of lines to be plotted in the spectrogram)
        T_WSQDecoder *pDecoder, // [in] struct will data for all active decoders
        float *pfltSamples, int nSamples, int nChannelsPerSample, float sampleRate,
        void (*pRxTextWriter)(T_WSQDecoder *pDecoder, int iChannel, int iFlags,  char *pszText) );
#       define WSQ_TEXT_FLAG_NORMAL 0
#       define WSQ_TEXT_FLAG_FINAL  1 // finally decoded character (from INTERACTIVE decoder),
                                      // 'last line' must be removed before, and restored afterwards adding pszText
#       define WSQ_TEXT_FLAG_PRELIM 2 // preliminary output into the 'last line' (ticker)
#       define WSQ_TEXT_FLAG_DEBUG  4
CPROT T_SpecBufEntry * WSQ_GetSpectrumFromBuffer( T_WSQDecoder *pDecoder, int iBufIdxOffset ); // result may be NULL !

CPROT void WSQ_GetDisplayableFreqRange( T_WSQDecoder *pDecoder, float *pfltFmin, float *pfltFmax ); // .. for the spectrogram

CPROT void WSQ_InitEncoder( T_WSQEncoder *pEncoder );
CPROT void WSQ_BeginTransmission(T_WSQEncoder *pEncoder); // prepares WSQ_GenerateAudio() before a tx-over
CPROT BOOL WSQ_GenerateAudio(
        T_WSQEncoder *pEncoder,
        float *pfltSamples, int nSamples, int nChannelsPerSample, float sampleRate,
        char (*pTxCharReader)(void) );

#endif // ndef _WSQ_CODEC_H_

