/*
 *  File:    ?\WSQ2\WSQ2_Codec.c
 *  Date:    2014-03-15
 *  Authors: Con,  ZL2AFP (inventor of WSQ and author of the original code),
 *           Wolf, DL4YHF (separated GUI and codec, added modifications).
 *  Purpose: WSQ2 Encoder / Decoder .
 *   Formerly part of the original code  main_wsq_2 _seconds.c   by ZL2AFP .
 *   Moved into an extra module to separate GUI and 'algorithms' by DL4YHF .
 *
 *
 *      #####################################################################
 *
 *
 *        This software is provided 'as is', without warranty of any kind,
 *        express or implied. In no event shall the author be held liable
 *        for any damages arising from the use of this software.
 *
 *        Permission to use, copy, modify, and distribute this software and
 *        its documentation for non-commercial purposes is hereby granted,
 *        provided that the above copyright notice and this disclaimer appear
 *        in all copies and supporting documentation.
 *
 *        The software must NOT be sold or used as part of any commercial
 *        or "non-free" product.
 *
 *      #####################################################################
 */

/*------------------------------------------------------------------------------
 *
 *
 *
 */

#ifdef __BORLANDC__
// prevent that bugger from bugging us with 'initialized data in header, cannot pre-compile..." :
# pragma hdrstop
#endif

#include "switches.h" /* SWITCHES.H must be included before anything else! */
         /* Contains stuff like SWI_USE_OOURAS_FFT, SWI_USE_KISS_FFT, etc. */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <windows.h>  // LONGLONG, LARGE_INTEGER, QueryPerformanceCounter()

#include "QuickSrt.h"   // DL4YHF's implementation of QuickSort
#include "dspmath.h"
#include "coeffs.h"

// Ex: include <complex.h>
   // DL4YHF 2014-02-15 : Including 'complex.h' in a *.c (not cpp) module
   // caused a megaton of obfuscated error messages from Borland C++Builder V6,
   // most occurred in "_mbstate.h" whatever that is.
   // It later turned out that WSQ doesn't need complex.h at all.

#include "WSQ_Codec.h"       // WSQ_DecodeAudio(), WSQ_GenerateAudio(), ..
#include "WSQ_Interactive.h" // interactive decoder, aka 'Frequency Shooter' mode

#include "WSQ_Varicode_Tables.c"

#define SMOOTH_K    0.13333333

#define ASCII_XON   0x11 //for serial comms routine
#define ASCII_XOFF  0x13

//---------------------------------------------------------------------------
// Variables  (global but mostly 'internal', for convenience)
//---------------------------------------------------------------------------

  // ... some of them declared in WSQ_Codec.h, at least those which must be accessible
  //     in both the graphic user interface AND the encoder / decoder.

T_WSQDecoder WSQ_Decoder;       // so far, only one global object of this type,..
T_WSQEncoder WSQ_Encoder;       // and definitely only ONE encoder (signal generator)


//---------------------------------------------------------------------------
// Internal forward references
//---------------------------------------------------------------------------

int   WSQ_GetSuitableFftInputSize( T_WSQDecoder *pDecoder, float sampleRate );
float WSQ_CalcNoiseInRange( T_SpecBufEntry *pSpectrum, int i1, int i2 );
void  WSQ_UpdateDecoderParams( T_WSQDecoderChannel *pChannel );
void  WSQ_UpdateFFTWindowTable( T_WSQDecoder *pDecoder );

//---------------------------------------------------------------------------
// Functions
//---------------------------------------------------------------------------

void LimitInteger( int *piValue,  int min, int max ) // YHF, for laziness..
{ if( *piValue < min )
   {  *piValue = min;
   }
  if( *piValue > max )
   {  *piValue = max;
   }
}

void LimitFloat( float *fltValue,  float min, float max )
{ if( *fltValue < min )
   {  *fltValue = min;
   }
  if( *fltValue > max )
   {  *fltValue = max;
   }
}

//---------------------------------------------------------------------------
int RoundFloat(double number)
{ // It's weird that C has exotic stuff like 'floor','ceil', but not 'round'.
  // Borland didn't have round(), PellesC had it and complained about 
  // incompatible declaration, WB was fed up by this and RENAMED his own
  // 'round()' function to RoundFloat()...
    return (number >= 0) ? (int)(number + 0.5) : (int)(number - 0.5);
}

//---------------------------------------------------------------------------
float WSQ_VoltageRatioTodB( float fltVoltageRatio )
{
   if( fltVoltageRatio > 1e-10 )
      return 20*log10( fltVoltageRatio );
   else
      return -200.0;  // to keep things simple, limit dynamic range to 200 dB
}

//---------------------------------------------------------------------------
void WSQ_InitDecoder( T_WSQDecoder *pDecoder,
                      float fltInitialSamplingRate,
                      int inputOptions, // future plan: flag for quadrature input, etc etc
                      int nDecoderChannels )
{ memset( pDecoder, 0, sizeof(T_WSQDecoder) );

  pDecoder->nSpecBufAllocEntries = 800; // number of lines buffered for the waterfall
         // (and, possibly, for some algorithms to 'look back in time')
         // 800 lines * 16384 bins * 4 bytes = 50 MByte (ok, even for very old PCs)
  pDecoder->SpectrumBuffer = (T_SpecBufEntry*)malloc( pDecoder->nSpecBufAllocEntries * sizeof(T_SpecBufEntry) );
  pDecoder->nSpecBufUsedEntries = 0;
  pDecoder->i32SpectrumCounter  = 0; // non-circular counter of lines emitted to the waterfall
                                     // (used as a timebase for the 'interactive' WSQ decoder)
  pDecoder->dwMagic31415926 = 0x31415926; // 'open for business' (WSQ_InitDecoder has been called)

  LimitInteger( &nDecoderChannels, 0, WSQ_MAX_DECODER_CHANNELS ); // so far, only one or two decoders
  pDecoder->nChannels = nDecoderChannels;
  pDecoder->peak_hits = 6;      // default "peaks hit parade" count
  pDecoder->fUseLongerFFT = TRUE;
  pDecoder->fltInputSamplingRate = fltInitialSamplingRate;
  pDecoder->iDecimation  = (pDecoder->fltInputSamplingRate >= 22000 ) ? 4 : 1;
  pDecoder->iFftInputSize= WSQ_GetSuitableFftInputSize( pDecoder, pDecoder->fltInputSamplingRate );
  pDecoder->fltFftBinWidth = pDecoder->fltInputSamplingRate
                           / (float)( pDecoder->iDecimation * pDecoder->iFftInputSize );
  WSQ_UpdateFFTWindowTable(pDecoder); // f(pDecoder->iFftInputSize)
#if( SWI_USE_KISS_FFT )  // already create a black box for the real forward 'KISS' FFT ?
  pDecoder->kiss_FFTR_object = NULL; // not yet. See WSQ_DecodeAudio() !
#endif // use KISS FFT ?


  // Default frequency range for the AGC- and noiseblanker reference.
  // There should be ONLY NOISE (atmospheric) but no man-made signals in this range:
  pDecoder->iAGC_NB_StartFreq = 800; /* Hz */
  pDecoder->iAGC_NB_EndFreq   = 900; /* Hz */

  WSQ_BeginReception(pDecoder); // prepares / clears all decoders before the 1st rx-over
} // end WSQ_InitDecoder()

//---------------------------------------------------------------------------
void WSQ_DeleteDecoder( T_WSQDecoder *pDecoder )
{
  if( pDecoder->dwMagic31415926 == 0x31415926 )
   { pDecoder->dwMagic31415926 = 0;
     Sleep(100);  // give other threads the chance to STOP using the decoder
     if(  (pDecoder->nSpecBufAllocEntries > 0 )
        &&(pDecoder->SpectrumBuffer != NULL ) )
      { pDecoder->nSpecBufAllocEntries = 0;
        free( pDecoder->SpectrumBuffer );
        pDecoder->SpectrumBuffer = NULL;
      }
   }
} // end WSQ_DeleteDecoder()

//---------------------------------------------------------------------------
void WSQ_InitDecoderChannel( T_WSQDecoder *pDecoder,
         int   iChannel,    // 0..WSQ_MAX_DECODER_CHANNELS-1, used to identify different decoder output windows
         float fltBaseFreq, // audio frequency of the lowest WSQ tone, originally 1000 Hz
         float fltBandwidth,// decoder's allowed frequency range (span) in Hz
         int   iSquelch_dB, // squelch level, dB SNR (typically -25 or so..)
         int   iWSQMode,    // WSQ_MODE_OLD or WSQ_MODE_WSQCall (4 or 3 FFT-bins per tone spacing; added 2017-12-03)
         int   iAlgorithm)  // WSQ_ALGO_ORIGINAL or WSQ_ALGO_ALTERN_1, maybe more in future
{
  T_WSQDecoderChannel *pChannel;
  if( iChannel>=0 && iChannel<WSQ_MAX_DECODER_CHANNELS )
   { pChannel = &pDecoder->channel[iChannel];
     memset( (void*)pChannel, 0, sizeof(T_WSQDecoderChannel) );
     pChannel->fltBaseFreq= fltBaseFreq; // audio frequency of the lowest WSQ tone, originally 1000 Hz
     pChannel->fltBandwidth= fltBandwidth;
     pChannel->iSquelch_dB= iSquelch_dB;
     pChannel->iWSQmode   = iWSQMode;   // tone spacing for WSQ_MODE_OLD or WSQ_MODE_WSQCall ?
     pChannel->iAlgorithm = iAlgorithm; // ex: WSQ_ALGO_ORIGINAL
     WSQ_UpdateDecoderParams(pChannel); // Update those 'derived', internally used parameters for this decoder instance
   }
} // end WSQ_InitStruct_Decoder()

//---------------------------------------------------------------------------
void WSQ_UpdateDecoderParams( T_WSQDecoderChannel *pChannel )
  // Update those 'derived', internally used parameters for a decoder instance.
  // Called after modifying the base frequency or decoder bandwidth (span).
{
  double dblToneSpacing_Hz = (pChannel->iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */
  double dblMinBandwidth = dblToneSpacing_Hz * (double)(WSQ2_NUM_TONES-1);
  pChannel->fmin_Hz = pChannel->fltBaseFreq - 0.5 * ( pChannel->fltBandwidth - dblMinBandwidth );
  pChannel->fmax_Hz = pChannel->fmin_Hz + pChannel->fltBandwidth;
} // end WSQ_UpdateDecoderParams()

//---------------------------------------------------------------------------
void WSQ_SetDecoderBaseFreq( T_WSQDecoder *pDecoder, // "base frequency" := "tone zero"
         int   iChannel,    // 0..WSQ_MAX_DECODER_CHANNELS-1
         float fltBaseFreq) // audio frequency of the lowest WSQ tone, originally 1000 Hz
{
  T_WSQDecoderChannel *pChannel;
  if( iChannel>=0 && iChannel<WSQ_MAX_DECODER_CHANNELS )
   { pChannel = &pDecoder->channel[iChannel];
     pChannel->fltBaseFreq = fltBaseFreq;    // audio frequency of the lowest WSQ tone, originally 1000 Hz
     WSQ_UpdateDecoderParams( pChannel );
   }
} // end WSQ_SetDecoderBaseFreq()

//---------------------------------------------------------------------------
void WSQ_SetDecoderBandwidth( T_WSQDecoder *pDecoder,
         int   iChannel,      // 0..WSQ_MAX_DECODER_CHANNELS-1
         float fltBandwidth)  // decoder's allowed frequency range (span) in Hz
{
  T_WSQDecoderChannel *pChannel;
  if( iChannel>=0 && iChannel<WSQ_MAX_DECODER_CHANNELS )
   { pChannel = &pDecoder->channel[iChannel];
     pChannel->fltBandwidth = fltBandwidth;
     WSQ_UpdateDecoderParams( pChannel );
   }
} // end WSQ_SetDecoderBandwidth()

// Alternative methods to set the decoder passband, used by mouse event handler:

//---------------------------------------------------------------------------
void WSQ_SetDecoderFmin( T_WSQDecoderChannel *pChannel,
        float fmin ) // begin of the decoder's passband, ususally a few Hz below(!) the "base frequency"
{
  double dblToneSpacing_Hz = (pChannel->iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */
  double dblMinBandwidth = dblToneSpacing_Hz * (double)(WSQ2_NUM_TONES-1);

  // inverse to pChannel->fmin_Hz = pChannel->fltBaseFreq - 0.5 * ( pChannel->fltBandwidth - fltMinBandwidth ) :
  pChannel->fltBaseFreq = fmin + 0.5 * ( pChannel->fltBandwidth - dblMinBandwidth );
  // pChannel->fmin_Hz = fmin;
  // pChannel->fmax_Hz = fmin + pChannel->fltBandwidth;
  WSQ_UpdateDecoderParams(pChannel);
} // end WSQ_SetDecoderFmin()

//---------------------------------------------------------------------------
void WSQ_SetDecoderFmax( T_WSQDecoderChannel *pChannel,
        float fmax ) // end of the decoder's passband, ususally a few Hz below(!) the "base frequency"
{
  double dblToneSpacing_Hz = (pChannel->iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */
  double dblMinBandwidth = dblToneSpacing_Hz * (double)(WSQ2_NUM_TONES-1);
  double dblNewBandwidth;
  // inverse to pChannel->fmax_Hz = pChannel->fltBaseFreq + fltMinBandwidth + 0.5 * ( pChannel->fltBandwidth - fltMinBandwidth );
  // solved for bandwidth (!) :
  dblNewBandwidth = ( fmax - pChannel->fltBaseFreq - dblMinBandwidth ) * 2.0 + dblMinBandwidth;
  if( dblNewBandwidth < dblMinBandwidth )
   {  dblNewBandwidth = dblMinBandwidth;
   }
  pChannel->fltBandwidth = dblNewBandwidth;
  WSQ_UpdateDecoderParams(pChannel);
} // end WSQ_SetDecoderFmax()


//---------------------------------------------------------------------------
void WSQ_BeginReception(T_WSQDecoder *pDecoder) // prepares / clears all decoders before an rx-over
  // Not really necessary, but used during file analysis to provide a 'fresh start'.
  // Resets low-pass filters, symbol averages, integrators, and similar stuff.
{
  int i,iChannel;
  T_WSQDecoderChannel *pChannel;

  // Initialize the common part (used for ALL channels of this decoder):
  pDecoder->iSampleCounter = pDecoder->iDecimateSampleCounter = pDecoder->iSampleOfSlice = 0;
  pDecoder->iDecimatorIdxIn= 0;
  for( i=0; i<64; ++i )
   { pDecoder->fltDecimatorQueueI[i] = 0.0;
   }
  for( i=0; i<FFT_MAX_INPUT_SIZE; ++i)
   { pDecoder->fltDecimatedInput[i] = 0.0;
   }
  for( i=0; i<FFT_MAX_FREQ_BINS; ++i)
   { pDecoder->fltSmoothedFFT[i] = 0.0;
   }
  for( i=0; i<FFT_MAX_INPUT_SIZE; ++i)
   { pDecoder->fltFftBufRe[i] = 0.0;
     pDecoder->fltFftBufIm[i] = 0.0;
   }
  pDecoder->hangcnt = 0; // used for the AGC

  // Initialize all 'decoder channels' :
  for( iChannel=0; iChannel<WSQ_MAX_DECODER_CHANNELS; ++iChannel )
   { pChannel = &pDecoder->channel[iChannel];
     pChannel->ss = pChannel->tmp = pChannel->pk = pChannel->agc_decay = pChannel->agcthr = pChannel->agcgain = 0.0;
     pChannel->fltSig_to_Noise_dB = 0.0;
     pChannel->iPeakBinIndex = pChannel->iPrevPeakBin = 0;
     pChannel->peak_counter= 0;
     pChannel->curr_nibble = pChannel->prev_nibble = 0;
     pChannel->symbol_bin_index = pChannel->prev_symbol_bin_index = 0;
     pChannel->fltSymbolPeakFreq = 0.0;
     pChannel->iSymbolInhibitCountdown = WSQ_SLICES_PER_SYMBOL / 2; // don't emit the next symbol before (at least) a HALF symbol duration
   }

} // end WSQ_BeginReception()

//---------------------------------------------------------------------------
int WSQ_GetSuitableFftInputSize( T_WSQDecoder *pDecoder, float sampleRate )
  // Other input parameters: pDecoder->iDecimation (usually 4),
  //                         pDecoder->fUseLongerFFT (usually TRUE) .
{
  double decimatedSR = sampleRate, dFftBinWidth_Hz, dFftWindowLength_s;
  double dblToneSpacing_Hz = (pDecoder->channel[0].iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */
  int i = 1024;     // pick the nearest suitable power of two....
  if( pDecoder->iDecimation > 1 )
   {  decimatedSR /= (float)pDecoder->iDecimation;
   }
  // About the decision for a reasonable FFT size ...  by DL4YHF, 2014-03-15 :
  // This was formerly fixed to FFT_MAX_INPUT_SIZE = 16384 for an input sampling rate of 32000/4 Hz,
  // but since 2014-02-26 the FFT size is VARIBLE, to realize a suitable bin width
  // for any input sampling rate :
  // The original plan was to have an FFT frequency bin width of (WSQ2_SGL_TONE_SPACING/4),
  // i.e. 32000 Hz / (4*16384) = 0.48828125 Hz = 1 / symbol_length_in_seconds .
  //        |         |   |___ FFT input size in the original WSQ decoder
  //        |         |_______ input decimation in the original WSQ decoder
  //        |_________________ audio sampling rate in the original WSQ decoder
  //
  // Later (2014-03-15, DL4YHF) the question arose if the total FFT window
  // could be longer, because due to the cos^2-similar windowing,
  // some of the energy (signal power * time) at the begin and end of the WSQ symbol
  // are lost.  So, just for comparison, a "longer" window was tried:
  //___________________________________________________________________________
  //|                                                                          |
  //|       Original :              |   "Longer FFT" by DL4YHF :               |
  //|                               |                                          |
  //| |<-- WSQ symbol time --->|    |        |<-- WSQ symbol time --->|        |
  //|             ___            1  |                  _____          .        |
  //|          _--   --             |             __---     ---__     .        |
  //|         /        \            |           /                \    .        |
  //|        /          \           |          /                   \  .... 0.5 |
  //|      _/            \_         |        /                       \         |
  //|   __-                -__      |     __/                          \__     |
  //| --                      -- 0  |  ---                                ---  |
  //| |<----- FFT window ----->|    |  |<---------- FFT window ------------->| |
  //|__________________________________________________________________________|
  //
  // With the WSQ symbol beginning and ending at the "-6 dB" points of the FFT
  // window, the window would be about TWO TIMES longer than a WSQ symbol
  // (because the cos^2 aka 'raised cosine' aka 'Hann' window contains
  //  a FULL cosine PERIOD, so the -6 dB points, 0.5, are at 25 and 75 percent
  //  of the window length)
  // Of course this will cause some ISI (inter-symbol interference) but that
  // may be tolerable. To compare the two window lengths shown above,
  // the 'longer FFT' option can be modified in the new WSQ GUI since 2014-03-15.
  // To generated a WSQ-like 33-tone staircase, with noise added by Spectrum Lab,
  // use the following expression for the 'arbitrary waveform editor' :
  //    round(33*x-.5)/16-1
  //    That rising staircase produces a string of SPACE characters,
  //    except for an occasional 'a' after jumping from the highest to the lowest tone.
  // The 33-tone test 'staircase' should take 33 * 2.048 seconds;
  //    ->  'frequency modulator frequency' :   1 / (33*2.048s) = 0.014796 Hz.
  //    'Deviation' for the frequency modulator: 31 Hz (to "both sides") .
  //    Center frequency of the FM'ed sinewave : 1031 Hz .
  //    Noise generator in SL set to "-45 dB / sqrt(Hz)" .
  //    Sinewave level for 50% copy with the original algorithm,
  //       'fast' symbol averaging :  -35 dB ("over" full scale).
  //     SNR indicator (bar) mostly below -25 dB, with 'SLOW' average
  //     and modulation turned off (i.e. 'carrier only') SNR around -27 dB .
  //
  while( i < FFT_MAX_FREQ_BINS )
   { dFftBinWidth_Hz    = decimatedSR / (double)i;
     dFftWindowLength_s = (double)i / decimatedSR;
     if( pDecoder->fUseLongerFFT )
      {
        if( dFftWindowLength_s >= (1.5 * WSQ2_SYMBOL_TIME) )
         { break;  // the "longer FFT" is now long enough (at least 1.5 times longer than a WSQ symbol)
         }
      }
     else // similar to the original code: total FFT window with ideally same length as one WSQ symbol...
      { // With a bin width above 1/3rd of the WSQ tone spacing the original decoder won't work.
        // With a bin width below 1/6th of the WSQ tone spacing, the FFT window gets too long -> won't work (?)
        // Thus:
        if( (i>=FFT_MAX_INPUT_SIZE) || (dFftBinWidth_Hz <= (dblToneSpacing_Hz/3) )  )
         { break;
         }
      }
     i *= 2;  // try the next power of two for the FFT input size
   }
  return i;  // number of samples in the time domain entering each FFT
} // end WSQ_GetSuitableFftInputSize()


//---------------------------------------------------------------------------
void WSQ_UpdateFFTWindowTable( T_WSQDecoder *pDecoder ) // f(pDecoder->iFftInputSize)
{
  int i;
  float flt = 2.0 * 3.141592654 / (float)pDecoder->iFftInputSize;
  for ( i=0; i<pDecoder->iFftInputSize; i++ )
   {
#   if(0)   // original code : // Blackman-Something window
     pDecoder->fltFFTWindow[i] = (0.42-0.5*cos(6.2832*(i)/pDecoder->iFftInputSize) + 0.08*cos(2*6.2832*(i)/pDecoder->iFftInputSize));
#   else    // DL4YHF's modified code : cos^2 aka 'raised cosine' window aka 'Hann window' : easier for interpolated frequency readout
            // The guy after whom this function was named was Julius von Hann, not Hanning..
     pDecoder->fltFFTWindow[i] =  0.5 * (1.0 + cos((float)(i-pDecoder->iFftInputSize/2)*flt) );
#   endif
   }
} // end WSQ_UpdateFFTWindowTable()


//--------------------------------------------------------------------------
int WSQ_DecodeVaricode( int curr_nibble, int prev_nibble )
  // Varicode (character) decoder .
  // Returns the decoded character, usually ranging from 32 to 127 (ASCII) .
{
  int curr_character = 0;

  // > If you study the WSQ Varicode V4 table you will see that
  // > the first number of all code groups is always in the range 0 - 28 (not 0 - 26, WB).
  // > The most used characters have just one code.
  // > Following numbers in a code group (if used) are always in the range 29 - 31 (not 27 - 31, WB).
  // > In this way the receiver will always know when a new code starts.
  // > We call the first number the Initial Symbol, while the others
  // > are called Continuation Symbols.  (...)
  // > At the receiver, when an Initial code is received directly after
  // > a previous Initial code, the earlier one must relate to a single-symbol character,
  // > so is looked up in the table and printed, and the new code retained for next time.
  // > If the new code is a Continuation code, the previous code and the new code
  // > are looked up in the table as a pair, and the result printed.

  // WB: The reason for delaying (retaining) the new code (here: curr_nibble)
  //     becomes obvious from the following example:
  //     code {1,30} = digit '1';
  //     code {1} *not* followed by a continuation symbol = character 'a' .
  //
  // single-nibble characters
  if( (prev_nibble < 29) && (curr_nibble < 29) )
   {
     curr_character = wsq_varidecode[prev_nibble];
     if((curr_character<10)|(curr_character>127))
      { curr_character = 31; // to avoid "bonk" sound (esp. ascii 7)
      }
     if((curr_character>10)&(curr_character<31))
      { curr_character = 31; // - substitute a non-printing character
      }
   }

  // double-nibble characters
  if( (prev_nibble < 29) & (curr_nibble > 28) )
   {
     curr_character = wsq_varidecode[prev_nibble * 32 + curr_nibble];
     // YHF 2014-02-23: Got here with prev_nibble=28 + curr_nibble=30
     //                 when sending the 'IDLE' character from further below.
     //                 -> curr_character = 0x00 .
     //  Should not be replaced by a 'printing' 31 like an 'unknown character' .
     if( curr_character == 0x00 )
      { // 'IDLE' character, VAR={28,30}, now treated as NON-PRINTING (DL4YHF 2014-02-23)
      }
     else // not the IDLE character but some 'printing' character..
      {
        if((curr_character<10)||(curr_character>127))
         { curr_character = 31; // not sure what this is, but looks like a hollow square in an edit control
         }
        if((curr_character>10)&&(curr_character<31))
         { curr_character = 31;
         }
      }
   }
   
  return curr_character;

} // end WSQ_DecodeVaricode()

/*------------------------------------------------------------------------------
 *
 *      Receive audio processing function
 *
 *      All receive audio processing is done in this function!
 *
 *      (No need to mess with a separate worker thread since this is
 *      already taken care of by PortAudio!)
 */
int WSQ_DecodeAudio(
        T_WSQDecoder *pDecoder,   // [in] data for all active decoders
        float *pfltSamples,       // [in] an array of samples (pointer to float)
        int   nSamples,           // [in] number of samples to process (originally only FOUR(!!!) per call)
        int   nChannelsPerSample, // [in] originally TWO(!) channels, only the first was used here
        float sampleRate,         // [in] sampling rate from the input [Hz] .
              // Usually 32000 for soundcard, but may be different for FILE analysis.
              // With exotic signal sources, the sampling rate
              // may not even be an INTEGER ! (for example, Audio-I/O with 'calibrated' SR)
        void (*pRxCharWriter)     // [in] callback to write received characters into the receive text editor
              (T_WSQDecoder *pDecoder, int iChannel, int iFlags, char *pszText) )
  //
  // Call stack (when Portaudio is used for input):
  // WinMMPa_OutputThreadProc() -> PaHost_BackgroundManager()
  //  -> Pa_TimeSlice() -> Pa_CallConvertInt16() -> ReceiveAudioCallback()
  //      -> WSQ_DecodeAudio()
{
  int   nSlicesRcvd = 0; // return value : number of slices == spectrum lines for plotting
  int   iChannel;
  T_SpecBufEntry      *pSpectrum;
  T_WSQDecoderChannel *pChannel;
  double dblToneSpacing_Hz = (pDecoder->channel[0].iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */

  float fltInput, agc_decay, agcthr, agcgain, tmp;
  float fltMinBandwidth = dblToneSpacing_Hz * (float)(WSQ2_NUM_TONES-1);
  double decimatedSR, dFftBinWidth_Hz, d1, d2;

  float flt,sig,noise,rDecimated;
  float re,im, rNormalizeFFT;
  int   i, iSample, iBin, nSamplesPerSlice;
  double dSecondsPerSlice;
  int   iPeakBinIndex;
  float fltPeakMag_dB, fltPeakFrequency;
  double        val, max = 0.0;
  int           nibble;
  unsigned char curr_character;
  int           low_limit,high_limit; // FFT bin indices for the WSQ2 'passband' (decoder range)
  char          sz80[88];
  LONGLONG i64TimerFreqcy, i64Time1, i64Time2, i64TimeDiff_us;

#undef  L_TEST
#define L_TEST 0 /* select one of the "sinewave-injection tests" further below ? */
#if( L_TEST )
  static float fltTestSigPhase=0.0;   // evil static, only used for TESTING
#endif

  UNREFERENCED( max );

  pDecoder->iDecimation = 4;
  decimatedSR = sampleRate / (double)pDecoder->iDecimation;
  i = WSQ_GetSuitableFftInputSize( pDecoder, sampleRate );
  if ( i != pDecoder->iFftInputSize )
   { pDecoder->iFftInputSize = i;
     WSQ_UpdateFFTWindowTable(pDecoder); // f(pDecoder->iFftInputSize)
#   if( SWI_USE_KISS_FFT )  // modify the mysterious black box for the real forward FFT :
     if( pDecoder->kiss_FFTR_object != NULL )  // KISS FFT(real) object already allocated ?
      { // de-allocate it because we're using a different FFT size now:
        kiss_fftr_free( pDecoder->kiss_FFTR_object ); // couterpart for kiss_fftr_alloc()
        pDecoder->kiss_FFTR_object = NULL;     // force re-creation further below
      }
#   endif // use KISS FFT ?
   } // end if < FFT size has been changed >
  pDecoder->fltFftBinWidth= dFftBinWidth_Hz = decimatedSR / (double)pDecoder->iFftInputSize;
     // In the original code, the symbol length was
     //  16384 points / (32000Hz/4) = 2.048 seconds (thus the name WSQ2);
     //   |___ FFT "input" size (i.e. number of points in the time domain)
     // -> 1/16 th of the symbol length = 2.048 s / 16 = 0.128 seconds [per FFT] :
  dSecondsPerSlice = (double)WSQ2_SYMBOL_TIME / (double)WSQ_SLICES_PER_SYMBOL;  // -> 0.128 s
     // But if the sampling rate isn't an integer, the *realized* 'number of seconds per slice'
     // may not be exactly 0.128 seconds (important for the phase-measurements further below) !
  nSamplesPerSlice = (int)( 0.5 + dSecondsPerSlice * decimatedSR );
  dSecondsPerSlice = (double)nSamplesPerSlice / decimatedSR; // 'precise' measurement interval

#if( SWI_USE_KISS_FFT )  // create a mysterious black box for the real forward FFT :
  if( pDecoder->kiss_FFTR_object == NULL )
   { pDecoder->kiss_FFTR_object = kiss_fftr_alloc(
                 pDecoder->iFftInputSize, // int nfft, guess 'nfft' shall be the number of samples in the time domain
                 0/*not inverse*/, NULL,NULL );
   }
#endif // use KISS FFT ?

  pDecoder->iFftNumFreqBins = (pDecoder->iFftInputSize/2)+1;   // number of frequency bins (in the output from each FFT)
  // A 1024 point REAL FFT produces 513(!) frequency bins, including the "DC" bin !
  // Conversion of FFT frequency bins into into power dB :
  // A pure input sin wave ... A*sin(wt)... will produce an fft output
  //   peak of (N*A/4)^2  where N is FFT_SIZE.
  rNormalizeFFT = 4.0 / (float)pDecoder->iFftInputSize;  // factor to normalize FFT result to amplitudes (voltages), range 0..1

  // AGC parameters (based upon Winrad)------------------------------
  if (pDecoder->agc_mode != WSQ_AGC_FAST)
     agc_decay = .9999; // slow agc
  else
     agc_decay = .9995; // fast agc

  agcthr = 60.e-5;                    // too many magic numbers here..
  agcgain = .05;                      //
  agcgain = pow(agcgain, 2.2) * 2.0;

  // Read input buffer, process data, and emit spectra + decoded characters
  for (iSample = 0; iSample < nSamples; ++iSample )
   {
     // Get interleaved soundcard samples from input buffer

     fltInput  = pfltSamples[0];        // so far, only process ONE channel..
     pfltSamples += nChannelsPerSample; // .. but who knows...

#   if(L_TEST==1) // TEST: replace non-decimated input with a pure sinewave of known amplitude (-> ok)
     fltInput = sin( 2.0 * 3.141592654 * fltTestSigPhase );
     fltTestSigPhase += ( 1000.1234567/*Hz*/ / sampleRate );
     if( fltTestSigPhase > 1.0 )
      {  fltTestSigPhase -= 1.0;
      }
#   endif // TEST ?


     // "Borrowed" from SAQrx.......
     // Decimation filter, R = 4, nTaps = 64 (hardcoded... ;-)

     if (--pDecoder->iDecimatorIdxIn < 0)  // fill buffer backwards (that convolution stuff...)
      { pDecoder->iDecimatorIdxIn = 63;    // (well, the filter is symmetrical, but anyway...)
      }

     pDecoder->fltDecimatorQueueI[ pDecoder->iDecimatorIdxIn ] = fltInput;

     // Decimate Fs and do the heavy stuff at lower speed

     if( ++ pDecoder->iDecimateSampleCounter >= 4)  // decimation factor = R = 4
      {
        pDecoder->iDecimateSampleCounter = 0;

        // Fs = 8000 Hz code goes here, after the first decimation filter ..
        // DL4YHF : .. things may be different now, especially when analysing wave files
        //             with 11025 or 22050 or even 44100 S/sec .
        rDecimated  = 0.0;          // clear MAC accumulators
        for (i = 0; i < 64; i++)
         {
           flt = rDecIntCoeffs[i];
           rDecimated += pDecoder->fltDecimatorQueueI[pDecoder->iDecimatorIdxIn] * flt;

           if (++pDecoder->iDecimatorIdxIn >= 64)
            { pDecoder->iDecimatorIdxIn = 0;
            }
         }

        rDecimated *= 1.0e-8;      // normalize (FIR gain ~10^8 !!!)

        // rDecimated (ex: and rQ11k) now contains the downsampled (but not complex) signal
        // ready for processing at Fs = 8000 Hz, covering approcimately 0..4 kHz
        // (actually less due to the filter's roll-off).

#     if(L_TEST==2) // TEST: replace DECIMATED input with a pure sinewave of known amplitude ?
       rDecimated = sin( 2.0 * 3.141592654 * fltTestSigPhase );
       fltTestSigPhase += ( 1000.1234567/*Hz*/ / decimatedSR );
       if( fltTestSigPhase > 1.0 )
        {  fltTestSigPhase -= 1.0;
        }
#     endif // L_TEST==2 ?

        // To have the NEWEST sample at index zero when it's time for the next FFT,
        // the 'samples within the new slice' must be filled in THIS order:
        pDecoder->fltDecimatedInput[ (nSamplesPerSlice-1) - pDecoder->iSampleOfSlice ] = rDecimated;

        // The original code split each symbol into 16 slices for sync, i.e. 0.128 s.
        //  Got enough samples (ex: 'k') for the next symbol-sync "slice" ?
        if( ++pDecoder->iSampleOfSlice >= nSamplesPerSlice )
         {  pDecoder->iSampleOfSlice = 0;

           // fill one sixteenth (?) of the FFT window with 'new' samples
           //   (see graphics explaining the 'memmove' further below)
           for ( i = 0; i < nSamplesPerSlice; i++) // circular buffer for symbol sync
            {
              // AGC section ---------------------------------------------------------------
              if(pDecoder->agc_mode != WSQ_AGC_OFF)
               {
                 tmp = fabs( pDecoder->fltDecimatedInput[i]);   // -> ranging from 0 to 1

                 if(pDecoder->fltAGC_ss < tmp)
                  { pDecoder->fltAGC_ss = (9.0 * pDecoder->fltAGC_ss + tmp) / 10.0; // slow down attack time
                    pDecoder->fltAGC_pk = tmp;
                    pDecoder->hangcnt = (pDecoder->agc_mode == WSQ_AGC_FAST) ? 1 : 6;
                  }


                  // ex: pDecoder->fltDecimatedInput[i] *= 0.29 / max(pDecoder->fltAGC_ss, agcthr);
                  // PellesC : "Expected called object to have function type, but found 'double'. "
                  // What the heck.. ? The problem seemed to be the max() macro, got rid of it:
                  tmp = ( pDecoder->fltAGC_ss>agcthr ) ? pDecoder->fltAGC_ss : agcthr;
                  if( tmp > 0 )
                   {  tmp = 0.29 / tmp;  // origin of this magic number ?
                      pDecoder->fltDecimatedInput[i] *= tmp;
                   }

                  if(pDecoder->hangcnt == 0)
                   { pDecoder->fltAGC_ss *= agc_decay;
                   }

                  pDecoder->fltAGC_pk *= agc_decay;
               } // end of AGC section

              if(pDecoder->hangcnt > 0)
               { pDecoder->hangcnt--;
               }
              // ----------------------------------------------------------------------------

            } // end for ( i = 0; i < nSamplesPerSlice; i++)

#         if(L_TEST==3) // TEST: replace the FFT-input shortly BEFORE multiplying with the windowing function ?
           for ( i = 0; i < pDecoder->iFftInputSize; i++ )
            {
              pDecoder->fltDecimatedInput[i] = sin( 2.0 * 3.141592654 * fltTestSigPhase );
              // This should produce a peak at pSpectrum->fltFreqBins[4] = 1.0
              fltTestSigPhase += ( 4.0 * dFftBinWidth_Hz / decimatedSR );
              if( fltTestSigPhase > 1.0 )
               {  fltTestSigPhase -= 1.0;
               }
            }
#         endif // L_TEST==3 ?


           // Multiply the (decimated?) input for the FFT with the windowing function.
           //  Result is the input (windowed time domain samples) for the REAL FFT :
           for ( i = 0; i < pDecoder->iFftInputSize; i++ )
            { pDecoder->fltFftBufRe[i] = pDecoder->fltDecimatedInput[i] * pDecoder->fltFFTWindow[i];
            }

           // Benchmark to compare different implementations of the REAL FFT
           //  (except for that bulky monster which everyone is talking about):
           QueryPerformanceCounter( (LARGE_INTEGER *) &i64Time1 );
#         if( SWI_USE_KISS_FFT )  // use KISS FFT for the real forward FFT :
           if( pDecoder->kiss_FFTR_object != NULL )
            { kiss_fftr(
                 pDecoder->kiss_FFTR_object, // [in] mysterious black box
                 pDecoder->fltFftBufRe,      // [in] time domain samples
                 pDecoder->complexFreqBins); // [out] complex frequency bins (a la KISS FFT)
              // To avoid having to rewrite the code further below,
              // separate the KISS FFT output into separate arrays
              // for the real and imaginary part of the complex spectrum:
              for(i=0; i<pDecoder->iFftNumFreqBins; ++i )
               { pDecoder->fltFftBufRe[i] = pDecoder->complexFreqBins[i].r;
                 pDecoder->fltFftBufIm[i] = pDecoder->complexFreqBins[i].i;
               }
            }
#         else // ! SWI_USE_KISS_FFT :
           // Here:  Not using the KISS FFT but an even simpler one, based on
           // Steven W. Smith, "The Scientists and Engineer's Guide to Digital Signal Processing" [SGDSP] :
           dspmath_CalcRealFft( pDecoder->iFftInputSize,
                                pDecoder->fltFftBufRe,
                                pDecoder->fltFftBufIm );
#         endif // use KISS FFT ?
           // Benchmark to compare different implementations of the REAL FFT
           //  (except for that bulky monster which everyone is talking about):
           QueryPerformanceCounter( (LARGE_INTEGER *) &i64Time2 );
           QueryPerformanceFrequency( (LARGE_INTEGER *)&i64TimerFreqcy );
           if( i64TimerFreqcy > 0 )
            { i64TimeDiff_us = 1000000LL * (i64Time2 - i64Time1) / i64TimerFreqcy/*Hz*/;
              i64TimeDiff_us = i64TimeDiff_us;
              // Test results (on the author's Z61m notebook, 1.66 GHz T5500):
              // Plain old FFT, coded by DL4YHF based on [SGDSP], 32768 point input :
              //   6181 us, 6210 us, 6274 us, 6196 us .
              // KISS FFT (real), also 32768 point input, thus radix-4 butterflies :
              //   3296 us, 3344 us, 3321 us, 3293 us .
              // In WSQ2, Takuya Ooura's FFT was a tiny bit SLOWER(!) than KISS:
              //   3688 us, 3885 us, 3718 us, 3809 us .
              // Strange/unexplained/compiler dependent (?) :
              //   In Spectrum Lab, Ooura's FFT was 2 times faster than KISS !
              // Even with 6 ms per FFT, calulated every 128 milliseconds,
              // the CPU load caused by the FFT is neglectable .
              // But: This may change in future version, if the FFT is used
              //      to cross-correlate the received signal with a template
              //      of a single WSQ symbol (=tone WITH A DEFINED LENGTH).
            } // end if < FFT benchmarks available ? >

           // Store the new FFT (frequency bins aka spectrum)
           // in a cicurlar buffer aka FIFO ( pDecoder->SpecBuf[SPEC_BUF_SIZE] ).
           if(   (pDecoder->iSpecBufIndex < pDecoder->nSpecBufAllocEntries)
              && (pDecoder->nSpecBufAllocEntries > 0 )
              && (pDecoder->SpectrumBuffer != NULL ) )
            { pSpectrum = pDecoder->SpectrumBuffer + pDecoder->iSpecBufIndex;
              pDecoder->iSpecBufIndex = ( pDecoder->iSpecBufIndex + 1 ) % pDecoder->nSpecBufAllocEntries;
            }
           else  // spectrum buffer (array) not allocated -> use the 'dummy'
            { pSpectrum = &pDecoder->SpecBufDummy;
            }
           ++pDecoder->i32SpectrumCounter;  // non-circular counter of lines emitted to the waterfall
                                            // (used as a timebase for the 'interactive' WSQ decoder)
           pSpectrum->i32SpectrumIndex      = pDecoder->i32SpectrumCounter;
           pSpectrum->fltCenterFreqOf1stBin = 0.0;
           pSpectrum->fltFreqBinWidth_Hz    = dFftBinWidth_Hz;
           pSpectrum->nUsedBins    = pDecoder->iFftNumFreqBins;
           LimitInteger( &pSpectrum->nUsedBins, 0, SPEC_BUF_MAX_BINS );
           for (iBin = 0; iBin < pSpectrum->nUsedBins; ++iBin)
            {
              re = pDecoder->fltFftBufRe[iBin];
              im = pDecoder->fltFftBufIm[iBin];
              flt= sqrt(re*re+im*im) * rNormalizeFFT;  // -> voltage, normalized to 0...1 (1 for a "full scale sinewave")
              switch( pDecoder->symbol_avrg_mode )
               { case WSQ_SYM_AVRG_OFF :
                    pDecoder->fltSmoothedFFT[iBin] = flt;
                    break;
                 case WSQ_SYM_AVRG_FAST:
                    flt = 3.0 * SMOOTH_K * ( flt - pDecoder->fltSmoothedFFT[iBin]);
                    pDecoder->fltSmoothedFFT[iBin] += flt;
                    break;
                 case WSQ_SYM_AVRG_SLOW:
                    flt = SMOOTH_K * ( flt - pDecoder->fltSmoothedFFT[iBin]);
                    pDecoder->fltSmoothedFFT[iBin] += flt;
                    break;
               }
              pSpectrum->fltFreqBins[iBin] = pDecoder->fltSmoothedFFT[iBin];

              // The PHASE (angle) contained in the complex frequency bins
              // may be used by some decoders to increase the requency resolution
              // by comparing angles in consecutive overlapping FFTs .
              // This is similar (but not equal) to the REASSIGNED SPECTROGRAM
              // as implemented in Spectrum Lab (in fact, inspired by it).
              // The two consecutive FTs (pSpectrum and pPrevSpectrum)
              // are separated by <nSamplesPerSlice> * <decimatedSR>, e.g. 0.128 seconds.
              // Multiply the complex frequency bin from the 'previous' FFT
              //  by the complex conjugate of the recent FFT :
              re = pDecoder->fltPrevFftBinsRe[iBin] * pDecoder->fltFftBufRe[iBin]
                 + pDecoder->fltPrevFftBinsIm[iBin] * pDecoder->fltFftBufIm[iBin];
              im = pDecoder->fltPrevFftBinsIm[iBin] * pDecoder->fltFftBufRe[iBin]
                 - pDecoder->fltPrevFftBinsRe[iBin] * pDecoder->fltFftBufIm[iBin];
              // Measure the fractinal part ("limited to one circle")
              // of the number of complex pointer rotations between the two FFTs :
              d1 = dspmath_CalculateAngle( re, im ) / (2.0 * 3.141592654); // -> fractional part number of complex pointer rotations
              if( d1 > 0.5 )
               {  d1 -= 1.0;  // wrap to range -0.5 ... +0.5 [rotations]
               }
              // Example: A 1001 Hz signal would let the complex pointer (in its FFT bin)
              //   rotate 1001 Hz * 0.128 sec = 128.128 times (peak in bin number 2050).
              //  With a bin width of 0.48828125 Hz, a signal in the 'neighbour bin'
              //   would let the complex pointer rotate 1001.48828125 Hz * 0.128 sec = 128.062 times.
              if( iBin==3043 )
               {  iBin = iBin;  // <<< set a 'conditional breakpoint' for testing HERE
                  // With iBin=2050, got here with d1 = circa 0.128 [rotations]
                  //      using a 1001 Hz test signal generated by Spectrum Lab.
                  // With iBin=2044, got here with d1 = circa -0.255 [rotations]
                  //      using a 998 Hz test signal. d2 (theoretic fc, below): 998.0468 Hz.
               }
              //  Here we only have the fractional part; the 'additional' rotations
              //  caused by the signal not being 'ideally centered' in the frequency bin
              //  must be below one.  Due to the large overlap (here: 15/16th)
              //  between consecutive FFTs, this is always the case (see numbers above).
              // Calculate the theoretic number of rotations if the signal was *centered* in this FFT bin:
              d2 = (double)iBin * dFftBinWidth_Hz; // result with iBin=2050 : 1000.97656 Hz .
              // Note: For an FFT with REAL input, bin 'zero' (with the 'DC' component)
              //       is CENTERED at 0.0000 Hz, and ends at 0.5 * dFftBinWidth .
              //       Thus, the center frequency of the 2nd bin (index ONE)
              //       is 1 * dFftBinWidth_Hz,  not 1.5 * dFftBinWidth_Hz !
              // Convert the FFT bin's theoretic center frequency (d2)
              //  into 'fractional part number of complex pointer rotations' (cycles)
              //  to compare it with the 'measured' value (d1) :
              d2 = fmod( d2 * dSecondsPerSlice, 1.0 );
              if( d2 > 0.5 )
               {  d2 -= 1.0;  // wrap to range -0.5 ... +0.5 [rotations]
               }
              // Example: At this point, d1=+0.128 [measured rotations for the 1001.0 Hz test signal] ,
              //   d2=+0.125 [rotations if the signal was perfectly centered, i.e. 1000.97656 Hz ].
              d2 = d1 - d2;  // -> difference between the 'fractional' parts of the number of rotations
              // As shown in the example, the difference between the fractional parts (here: d2 = +0.003)
              // should be close to zero (remember, the unit is 'circles', not degrees or radians).
              // For simplicity, assume frequencies are positive and the pointer rotates clockwise.
              // Thus if the signal is ABOVE the FFT bin's center frequency,
              // d2 will be 'slightly positive' as in the 1001 Hz example.
              if( d2 > 0.5 )
               {  d2 -= 1.0;  // wrap to range -0.5 ... +0.5 [rotations]
               }
              if( d2 < -0.5 )
               {  d2 += 1.0;  // wrap both ways.  abs(result) will be WAY BELOW 0.5 !
               }

              // Convert the offset ('number of rotations') into a frequency:
              d2 /= dSecondsPerSlice;  // -> Example : d2 = 0.0234 Hz;  1000.976 + 0.0234 = 1001 Hz (ok)
              // Convert that frequency into a multiple of FFT frequency bins(!) for compact storage:
              flt = d2 / dFftBinWidth_Hz;  // -> Example: flt = 0.048 [frequency bins]
              // Plausibility check: if the frequency offset (d2, now in Hz)
              // exceeds +/-1 frequency bin widths, discard it. Since the WSQ decoder
              // will look at the 'peak bin' and its two neighbours anyway,
              // it doesn't really matter, but because the 'frequency offset'
              // will be stored in a memory-saving 8-bit integer array,
              // the offset must be limited anyway.
              LimitFloat( &flt, -1.0, 1.0 ); // limit the max. offset to one FFT bin (to either side)
              // For memory-efficient storage, scale the signal's frequency offset
              // (relative to the FFT bin's theoretic center frequency)
              // to +/-127 which represents +/- TWO FFT frequency bin widths:
              flt *= 127.0;  // scaling factor from +/- 1 to +/- 127 for 8-bit unsigned integer
              pSpectrum->i8FreqOffsets[iBin] = (signed char)(int)flt;  // unit: 1/127 th FFT bin width (signed 8-bit integer)
              // Test result using a slow frequency modulated triangle (*):
              //    html/freq_measurement_via_phase.png .
              // For comparison, same test signal but using the interpolation method:
              //    html/freq_measurement_via_interpolation.png
              // (*) test signal generated by SL with the following settings:
              //     sine generator center frequency = 1001 Hz, -30 dBfs,
              //     frequency modulator 0.01 Hz, deviation=5 Hz, triangle,
              //     noise generator set to '-45 dBfs / sqrt(Hz)' .

              // Save the complex frequency bins for the next time (next FFT):
              pDecoder->fltPrevFftBinsRe[iBin] = pDecoder->fltFftBufRe[iBin];
              pDecoder->fltPrevFftBinsIm[iBin] = pDecoder->fltFftBufIm[iBin];
            } // end for < all FFT frequency bins >

           for( iChannel=0; iChannel < WSQ_MAX_DECODER_CHANNELS; ++iChannel )
            { pSpectrum->fltSymbolPeakFreq[iChannel]   = -1;
              pSpectrum->fltSymbolPeakMag_dB[iChannel] = -200.0; /* dB */
              // these 'peaks' may later be plotted as an 'overlay' on the spectrogram
            }

           // Inform the 'interactive' WSQ decoder about the new spectrum.
           // The OLDEST spectrum will scoll out of view (in the spectrogram),
           // and if it contained a decoded characters (automatic or interactive),
           // that character will now be transferred from the LAST LINE IN THE RX TEXT
           // to the older lines (where it cannot be modified anymore by clicking
           // into the spectrogram to move the WSQ symbol markers) .
           WSQI_OnNewSpectrum( pDecoder, 0/*iChannel*/, pRxCharWriter );


           // Go on with the 'automatic' WSQ decoding process (multiple channels)....
           LimitInteger( &pDecoder->nChannels,  0, WSQ_MAX_DECODER_CHANNELS );
           for( iChannel=0; iChannel<pDecoder->nChannels; ++iChannel )
            { pChannel = &pDecoder->channel[iChannel];

              // Update the internally used parameters for this decoder instance:
              dblToneSpacing_Hz = (pChannel->iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */

              pChannel->fmin_Hz = pChannel->fltBaseFreq - 0.5 * ( pChannel->fltBandwidth - fltMinBandwidth );
              pChannel->fmax_Hz = pChannel->fmin_Hz + pChannel->fltBandwidth;
              low_limit  = (int)( 0.5 + pChannel->fmin_Hz / dFftBinWidth_Hz);
              high_limit = (int)( 0.5 + pChannel->fmax_Hz / dFftBinWidth_Hz);


              if( pChannel->iAlgorithm >= WSQ_ALGO_ALTERN_1 )
               {
                 // Find the maximum bin within the decoder's passband:
                 max = -999.9;
                 iPeakBinIndex = 0;
                 for (iBin = low_limit; iBin < high_limit; ++iBin)
                  {
#                  define L_PEAK_FINDING_METHOD 1  // by DL4YHF.. tried different 'peak-finding methods'
#                  if(L_PEAK_FINDING_METHOD==1)  // use the 'simple loop' (to find the peak) ?
                    val = pSpectrum->fltFreqBins[iBin];
                    if(val > max)
                     { max  = val;
                       iPeakBinIndex = iBin;
                     }
                    // Result with WSQ_dots_at_-28dB_SNR_synthetic.wav (string of dots, SNR = -28 dB),
                    //  *short* FFT, symbol averaging OFF :
                    //  196 chars decoded; 36 dots = 18 % .
#                  endif // L_PEAK_FINDING_METHOD==1 ?
#                  if(L_PEAK_FINDING_METHOD==2) // Try something else, which *may* (!) work better
                    // if the WSQ tone exactly hits the border between two frequency bins:
                    // (auf deutsch: 'frequency bin' = 'Frequenzeimer', kein Witz!).
                    // Get the signal's magnitude (in dB "over" whatever, full-scale?),
                    //   independent of its relative position within an frequency bin,
                    //   because (with the method 1) the peak APPEARS to be 1.4 dB weaker
                    //   if the frequency just hits the border between two FFT bins.
                    //   Worth the effort ? Obviously not (2014-03) :
                    // Result with WSQ_dots_at_-28dB_SNR_synthetic.wav (string of dots, SNR = -28 dB),
                    //  *short* FFT, symbol averaging OFF :
                    //  199 chars decoded; 39 dots = 19 %  [neglectable improvement compared with the 'simple loop']
                    WSQ_CalcFftPeakMagnitudeAndFrequency( // ... by interpolation with the neighbours :
                                  pSpectrum, // [in] spectrum (with info header)
                                  iBin,      // [in] frequency bin index of the alleged 'peak'
                          &fltPeakMag_dB,    // [out] peak magnitude  [dBfs]
                          &fltPeakFrequency);// [out] peak frequency  [Hz] (A)
                    if(fltPeakMag_dB > max)
                     {
                       max  = fltPeakMag_dB;     // peak magnitude (dB "over" whatever)
                       iPeakBinIndex = iBin;
                     }
#                  endif // L_PEAK_FINDING_METHOD==2 ?
                  } // end for( iBin... )

                 WSQ_CalcFftPeakMagnitudeAndFrequency( // ... by interpolation with the neighbours :
                                  pSpectrum, // [in] spectrum (with info header)
                           iPeakBinIndex,    // [in] frequency bin index of the alleged 'peak'
                          &fltPeakMag_dB,    // [out] peak magnitude  [dBfs]
                          &fltPeakFrequency);// [out] peak frequency  [Hz] (A)

                 // Try to get a more "accurate"(?) frequency measurement.
                 // WB 2014-03-15: Not sure yet if this is BETTER or WORSE
                 //  than the neighbour-bin-based frequency interpolation:
                 // Increase the frequency resolution by comparing the PHASES
                 //  in the current and previous spectrum (FFT) :
                 flt = (float)pSpectrum->i8FreqOffsets[iPeakBinIndex];
                 // Convert frequency offset from +/- 1/127th of an FFT bin-width to Hz :
                 flt *= ( dFftBinWidth_Hz / 127.0 );
                 flt += (float)iPeakBinIndex * dFftBinWidth_Hz;  // (B)
                 // 'flt' now holds the precise peak frequency based on phase measurement.
                 // The difference between the interpolation-based fltPeakFrequency (A)
                 // and the phase-measurement based frequency (B) could be used as
                 // a quality indicator. On MF, (B) appeared to be slightly better -
                 // see screenshots in ..\WSQ2\Notes_DL4YHF\freq_measurement_xyz.png .
                 // Thus:
                 fltPeakFrequency = flt;   // use (B) instead of (A)

                 // Noise level estimation (by DL4YHF) :
                 // Here, use a different algorithm to find the noise level
                 //  (once suggested by G4JNT, used in DL4YHF's Spectrum Lab):
                 noise/*V*/  = WSQ_CalcNoiseInRange( pSpectrum, low_limit, high_limit );
                 // The above 'noise' is actually a noise voltage density,
                 // which must be converted into dB ....
                 noise/*dB*/ = WSQ_VoltageRatioTodB( noise ); // relative to full scale, if there wasn't the optional AGC
                 // Normalize the noise level for 3000 Hz bandwidth,
                 //  because that's what most other amateur radio programs do.
                 // Example:
                 //    A 'receiver' (here: a single FFT BIN..)
                 //    with 0.5Hz bandwidth (*) has a 'noise-reducing gain'
                 //    of  +3.01 dB (= 10*log10(1.0Hz/0.5Hz) )
                 //    compared to a receiver with 1.0 Hz bandwidth.
                 //    (*) The equivalent receiver bandwidth is not the same as the
                 //    FFT bin width - due to windowing, it's 1.5 times wider.
                 noise/*dB*/ += 10.0*log10(3000.0/*Hz*/ / (1.5 * dFftBinWidth_Hz) );
                 // Theoretic result with 3000 Hz 'receiver bandwidth' :
                 // Noise level = -50 dBfs + 10 * log(3000 Hz / 1 Hz) = -15.2 dBfs;
                 // measured result with '-50 dBfs' (in 1 Hz) from SL:
                 //  noise = -15.29 dB  (OK).
                 //
                 pChannel->fltSig_to_Noise_dB = fltPeakMag_dB - noise/*dB*/;
                 // The SNR measured this way was slightly (one or two dB) different
                 // from the results returned by ZL2AFP's original code,
                 // but no attempt was made to "correct" this
                 // by adding or subtracting an unexplainable 'magic' number here.
                 // SNR meter "tests", with SL's noise generator at -50 dBfs/sqrt(Hz),
                 //     i.e. noise level in 3000 Hz = -15.2 dBfs (details above).
                 //     In SL, load configurations/WSQ2_Noise_Test1.usr .
                 //
                 // Sinewave 'carrier level' in SL: -10 dBfs | -20 dBfs | -30 dBfs | -40 dBfs
                 // ----------------------------------------+----------+----------+---------
                 // Theoretic SNR (in 3000 Hz)    :  +5.2 dB | - 4.8 dB | -14.8 dB | -24.8 dB
                 // Displayed with 'short FFT'    :  +5.3 dB | - 4.9 dB |  -15 dB  | - 24 dB  (ok)
                 // Displayed with 'long FFT'     :  +5.3 dB | - 4.9 dB |  -15 dB  | - 25 dB  (ok)
                 //
                 // -> the bandwidth normalisation works as it should:
                 //    The SNR shown in the WSQ GUI remains unchanged
                 //    when switching between 'long' and 'normal' FFT .
                 //
                 // To create a WSQ2 signal which gave 'poor but occasional copy'
                 //  with the original decoder, use SL + WSQ2_Noise_Test1.usr,
                 //  with the WSQ2 signal (generated by the digimode terminal)
                 //  amplitude set to '1 % of full swing' i.e. -40 dBfs,
                 //  noise generator at -47 dBfs/sqrt(Hz),
                 //  i.e. noise level in 3000 Hz = -47 dBfs + 10 * log(3000 Hz / 1 Hz) = -12.2 dBfs;
                 //  theoretic SNR of the WSQ2 signal = -40 dB - (-12.2dB) = -27.8 dB (SNR in 3 kHz).
                 //  With an endless string of '.' characters sent in WSQ2, about 10..20 percent 'copy',
                 //  in other words unreadable (with both ZL2AFP's original and DL4YHF's 1st 'alternate' algorithm).
                 //  To let the tests run 'faster than life', a test file with the above parameters
                 //  was saved as ?\WaveArchive\WSQ..\WSQ_dots_at_-28dB_SNR_synthetic.wav .
                 //  Instead of counting 'good' and 'bad' characters manually,
                 //  a character-counting / statistic display was added in WSQ_MainWin.c
                 //  which dumps the total number of characters, and number of decoded 'dots'
                 //  when ending a wave file analysis. Test results (2014-03-22) :
                 //  WSQ_dots_at_-28dB_SNR_synthetic.wav (string of dots, SNR = -28 dB):
                 //   Almost orig.,  no  avrg, short FFT:  196 chars decoded; 36 dots = 18 % .
                 //   Almost orig.,  no  avrg, long  FFT:  179 chars decoded; 52 dots = 29 % .
                 //   Almost orig., slow avrg, short FFT:  183 chars decoded; 43 dots = 23 % .
                 //   Almost orig., slow avrg, long  FFT:  150 chars decoded; 47 dots = 31 % .
                 //   DL4YHF alt.1,  no  avrg, short FFT:  196 chars decoded; 36 dots = 18 % . (still the simple peak-searching loop, but more accurate frequency by the 'phase'-method)
                 //   DL4YHF alt.1,  no  avrg, long  FFT:  187 chars decoded; 54 dots = 28 % .
                 //   DL4YHF alt.1, slow avrg, long  FFT:  171 chars decoded; 52 dots = 30 % .


               }
              else // close to ZL2AFP's original algorithm :
               {
                 // find the maximum bin within the decoder's passband
                 max = 0.0;
                 iPeakBinIndex = 0;
                 for (iBin = low_limit; iBin < high_limit; ++iBin)
                  { val = pDecoder->fltSmoothedFFT[iBin];
                    if(val > max)
                     { max  = val;
                       iPeakBinIndex = iBin;
                     }
                  }
                 fltPeakMag_dB    = pDecoder->fltSmoothedFFT[iPeakBinIndex];
                 fltPeakFrequency = pSpectrum->fltCenterFreqOf1stBin + (double)iPeakBinIndex * dFftBinWidth_Hz;

                 // Find signal/noise ratio (very approximately)
                 noise = 0;
                 for (iBin = low_limit; iBin < high_limit; ++iBin)
                  { noise += pDecoder->fltSmoothedFFT[iBin];
                  }
                 sig = pDecoder->fltSmoothedFFT[iPeakBinIndex-1]
                     + pDecoder->fltSmoothedFFT[iPeakBinIndex]
                     + pDecoder->fltSmoothedFFT[iPeakBinIndex+1];
                 noise -= sig;    // actually subtracting voltages here, anyway..
                 if( noise > 1e-9 ) // Modified by DL4YHF to avoid div-by-zero / "log10 Domain error" (exception)
                  { flt = sig / (noise)*1.0;
                    // ZL2AFP: normalise to 3000Hz b/w??? - this agrees with PATHSIM simulations as is
                    // DL4YHF: .. but not with the 'long FFT' option; see measurements listed in WSQ_GetSuitableFftInputSize().
                    pChannel->fltSig_to_Noise_dB = WSQ_VoltageRatioTodB( flt );
                  }
                 else
                  { pChannel->fltSig_to_Noise_dB = -200.0; // dummy for 'invalid peak' [dB]
                  }
               }  // end else < close to ZL2AFP's original algorithm >

              //  NOT FUNCTIONAL YET (just a 'planned experiment', 2014-03) :
              // Symbol synchronisation by the 'peaks' caused by the cos^2 windowing ?
              // When the symbol is best aligned with the FFT window,
              // the *interpolated* magnitude will reach its peak,
              // which can be nicely seen by the S(NR)-meter.
              //    Actually, the S(NR)-meter, which LOOKED LIKE "QSB",
              //    inspired DL4YHF to try this algorithm.
              for( i=WSQ_SLICES_PER_SYMBOL-1; i>0; --i )
               { pChannel->sPeakMagHistory[i] = pChannel->sPeakMagHistory[i-1];
               }
              pChannel->sPeakMagHistory[0].magnitude = fltPeakMag_dB;
              pChannel->sPeakMagHistory[0].frequency = fltPeakFrequency;
              // At this point,
              // sPeakMagHistory[0] = NEWEST peak entry in the history,
              // sPeakMagHistory[WSQ_SLICES_PER_SYMBOL-1] = oldest.
              //
              // Ideally, the peak magnitudes in sPeakMagHistory[] look like this
              //  when the next symbol is 'ready for decoding' :
              //                                |
              //             __---__            |- max. magnitude (dB over whatever)
              //           ./   .   \.          |
              //         /      .     \         | cos^2 shape
              //       /        .       \       | caused by FFT window
              //     /          .         \     |
              //    /           .          \    |
              //  ------------------------------ - noise level
              //   |            |           |
              //  [0]          [7]        [15]  <- index into sPeakMagHistory[]
              //   |<---one symbol time --->|
              //  "presence"             "past"
              //
              //
              //

              if( pChannel->iSymbolInhibitCountdown > 0 )
               { --pChannel->iSymbolInhibitCountdown;  // when counted down to zero, another symbol MAY be 'emitted'
               }

              // For later: mark the peak symbol in a colour which identifies THIS decoder channel
              pSpectrum->fltSymbolPeakFreq[iChannel]   = fltPeakFrequency;  // displayed as overlay in the waterfall
              pSpectrum->fltSymbolPeakMag_dB[iChannel] = fltPeakMag_dB; // also displayed as overlay in the waterfall
              pChannel->fltPeakFreqHz = fltPeakFrequency;
              nibble = -99;   // no new nibble (~ valid symbol) decoded yet..
              UNREFERENCED( nibble );
              switch( pChannel->iAlgorithm )
               { case  WSQ_ALGO_ALTERN_1 :  // a few 'alternative' algorithms by DL4YHF, March 2014:
                  {
#                  if(0)  // symbol synchronisation derived from the 'peak magnitude history' ?
                    // Since we cannot be sure about the correctness of the PREVIOUS symbol
                    // (it may have been down in the noise, or a false peak),
                    // accept ANY peak in the history if the last symbol
                    // was emitted at least A HALF SYMBOL TIME ago :
                    if( pChannel->iSymbolInhibitCountdown == 0 )
                     { flt = pChannel->sPeakMagHistory[0].magnitude;
                       for(i=j=0; i<WSQ_SLICES_PER_SYMBOL; ++i)
                        { if( pChannel->sPeakMagHistory[i].magnitude > flt )
                           { flt = pChannel->sPeakMagHistory[i].magnitude;
                             j = i;
                           }
                        }
                       if( j == (WSQ_SLICES_PER_SYMBOL-1)/2 )  // peak ok..
                        {
                          flt = pChannel->fltPeakFreqHz - pChannel->fltPrevPeakFreqHz;
                          pChannel->fltPrevPeakFreqHz = pChannel->fltPeakFreqHz;
                          // Convert the frequency difference [Hz] into a fractional WSQ tone number:
                          flt /= dblToneSpacing_Hz;
                          nibble = RoundFloat(flt);  // round, don't truncate, to the nearest WSQ 'nibble'
                        } // end if < peak ok >
                     } // end if( pChannel->iSymbolInhibitCountdown == 0 )
#                  else  // older stuff, similar to the original code:
                    // "when the frequency is stable, it may be a new symbol"
                    // > now see if we have several peaks in a row, indicating the presence of a symbol
                    pChannel->iPeakBinIndex = iPeakBinIndex;
                    if(pChannel->iPeakBinIndex == pChannel->iPrevPeakBin)  // <<< may be improved, looks like a 'too-hard decision' (YHF)
                     { pChannel->peak_counter++;
                     }
                    pChannel->iPrevPeakBin = iPeakBinIndex;

                    if(pChannel->peak_counter > pDecoder->peak_hits)
                     { // more than 6 peaks (normally peak_hits = 6) out of a possible 16 indicates a symbol rather than noise
                       pChannel->symbol_bin_index  = iPeakBinIndex;
                       pChannel->fltSymbolPeakFreq = pSpectrum->fltCenterFreqOf1stBin + (double)iPeakBinIndex * dFftBinWidth_Hz;
                       pChannel->peak_counter = 0;  // reset counter ready for next symbol
                     }

                    // Determine the difference between current and previous symbol
                    // ex: nibble = ((float)(pChannel->symbol_bin_index/4) - (float)(pChannel->prev_symbol_bin_index/4));
                    // But: The FFT bin width may be too different to be handled by the above code.
                    // With the following modification, even a file sampled with 22050 S/sec
                    // could be decoded without errors:
                    flt = (float)(pChannel->symbol_bin_index - pChannel->prev_symbol_bin_index) * dFftBinWidth_Hz;
                    // At this point, 'flt' is the frequency difference in Hz.
                    // Convert this into the closest WSQ tone number ('nibble' ideally ranging from 0 to 32)
                    nibble = RoundFloat( flt / dblToneSpacing_Hz ); // round, don't truncate !
                    // 2014-03-15: The above code snipped was copied also
                    //             into the 'almost orignal code' further below,
                    //             because it seemed to produce better results..
#                  endif // (0,1) ?
                  } break; // end case WSQ_ALGO_ALTERN_1
                 default :
                  { // Close to ZL2AFP's original algorithm, but modified for different FFT sizes :
                    //
                    // > now see if we have several peaks in a row, indicating the presence of a symbol
                    pChannel->iPeakBinIndex = iPeakBinIndex;
                    if(pChannel->iPeakBinIndex == pChannel->iPrevPeakBin)  // <<< may be improved, looks like a 'too-hard decision' (YHF)
                     { pChannel->peak_counter++;
                     }
                    pChannel->iPrevPeakBin = iPeakBinIndex;

                    if(pChannel->peak_counter > pDecoder->peak_hits)
                     { // more than 6 peaks (normally peak_hits = 6) out of a possible 16 indicates a symbol rather than noise
                       pChannel->symbol_bin_index  = iPeakBinIndex;
                       pChannel->fltSymbolPeakFreq = pSpectrum->fltCenterFreqOf1stBin + (double)iPeakBinIndex * dFftBinWidth_Hz;
                       pChannel->peak_counter = 0;  // reset counter ready for next symbol
                     }

                    // Determine the difference between current and previous symbol
                    // ex: nibble = ((float)(pChannel->symbol_bin_index/4) - (float)(pChannel->prev_symbol_bin_index/4));
                    // But: The FFT bin width may be too different to be handled by the above code.
                    // With the following modification, even a file sampled with 22050 S/sec
                    // could be decoded without errors:
                    flt = (float)(pChannel->symbol_bin_index - pChannel->prev_symbol_bin_index) * dFftBinWidth_Hz;
                    // At this point, 'flt' is the frequency difference in Hz.
                    // Convert this into the closest WSQ tone number ('nibble' ideally ranging from 0 to 32)
                    nibble = RoundFloat( flt / dblToneSpacing_Hz ); // round, don't truncate !
                  } // end case < decoder algorithm close to ZL2AFP's original code>
               } // end switch( pChannel->iAlgorithm )

              if(nibble < 0)
               { nibble = nibble + 33;  // allow for wrap-around (17 tones for 16 tone differences)
               }
              nibble = nibble - 1;      // 1 is added at the TX end so need to subtract it here

              if(nibble < 0)goto skip;  // -1 is our idle symbol, indicating we already have our symbol
                                        // -> Wait for next one to arrive

              // Arrived here: symbol finished, about to emit a new character...
              pChannel->iSymbolInhibitCountdown = WSQ_SLICES_PER_SYMBOL / 2; // only used by the DL4YHF decoder

              if( pChannel->fInteractive && ( pChannel->fltSig_to_Noise_dB >= pChannel->iSquelch_dB ) )
               { // Pass on the symbol to the 'interactive' decoder :
                 WSQI_EnterSymbol( pChannel->fltPeakFreqHz, pSpectrum->i32SpectrumIndex );
               }

              // Added 2014-03-10 : Show the 'symbol number' (tone number, 0..32, but FRACTIONAL)
              sprintf( pSpectrum->sz15DecoderInfo[iChannel], "%0.1f",
                        (float)( (pChannel->fltSymbolPeakFreq - pChannel->fltBaseFreq) / dblToneSpacing_Hz) );

              // Varicode decoder. Same subroutine called from WSQ_Codec.c and WSQ_Interactive.c :
              pChannel->curr_nibble = nibble;
              curr_character = WSQ_DecodeVaricode( pChannel->curr_nibble, pChannel->prev_nibble );
              pChannel->prev_nibble = pChannel->curr_nibble;

              // Emit the character if signal >= squelch level for this decoder ?
              if( (curr_character>0) && (pChannel->fltSig_to_Noise_dB >= pChannel->iSquelch_dB) )
               { sz80[0] = curr_character;
                 sz80[1] = '\0';
                 if( ! pChannel->fInteractive )
                  { pRxCharWriter( pDecoder, iChannel, WSQ_TEXT_FLAG_NORMAL, sz80 );  // write received character into RX window
                  }
                 else //  INTERACTIVE decoder active :
                  { // In INTERACTIVE decoder mode, the last half dozen of characters
                    // (those still visible in the spectrogram) may have to be REPLACED
                    // when the operator moves the symbol markers in the spectrogram.
                    // Principle: Instead of emitting the decoded character immediately
                    // to the normal editor text (including carriage return + new line),
                    // the automatic decoder (WSQ_DecodeAudio) emits each *SYMBOL*
                    // to a queue for the interactive decoder (WSQI_EnterSymbol) .
                    // As long as the WSQ symbols are in that queue (where they can be edited),
                    //  they are displayed in A SINGLE LINE in the editor.
                    //
                    // When a line with an 'interactively decoded' character
                    // falls out of that buffer, it will be removed from the 'last line'
                    // and append as normally decoded text (including CR+NL)
                    // in the last-but-one line of the text editor.
                  }
               } // end if < valid character, AND signal above squelch level >

skip:
              pChannel->prev_symbol_bin_index = pChannel->symbol_bin_index;

            } // end for( iChannel ...

           // Move the now "OLD" samples along 1/16th of a symbol length at a time.
           // Principle:
           //  newest(!) sample                            oldest sample    |
           //  | at index 0                          at index 16383(!) |    |
           //  |                                                       |    |
           //  |<-------------------- FFT window --------------------->|    |
           //    ( <pDecoder->iFftInputSize> decimated input samples)       |
           //  |<"slice">|                                                  |
           //   _________ _____________________________________________     |
           //  |         |                                             |    |
           //  |1/16th.. |  15/16th    of FFT window length            |    |
           //  |_________|_____________________________________________|    |
           //    "new"   .                  "old" samples      .       .    |
           //  |_________.__memmove_source_____________________|       .    |
           //  .                                                       .    |
           //  .>>copy>>>|_______________memmove_destination___________|    |
           //                                                               |
           // Note: Because source and destination overlap,                 |
           //       use memmove() rather than memcpy() here.                |
#         if(0)
           memmove( &pDecoder->fltDecimatedInput[nSamplesPerSlice],     // dest,
                    &pDecoder->fltDecimatedInput[0],                    // source
             sizeof(float)*(pDecoder->iFftInputSize-nSamplesPerSlice)); // # bytes
#         else   // equivalent without memmove():
           for ( i = pDecoder->iFftInputSize-1; i >= nSamplesPerSlice; --i )
            { pDecoder->fltDecimatedInput[i] = pDecoder->fltDecimatedInput[i-nSamplesPerSlice];
            }
#         endif
           // pDecoder->fltDecimatedInput[0..nSamplesPerSlice-1] will be overwritten
           // in the next loops. It's not necessary to clear them here.

           ++nSlicesRcvd; // count the number of slices == spectrum lines for plotting

         } // end if( ... >= nSamplesPerSlice )

      } // end if < time to decimate ... >

   } // for (iSample ...

  return nSlicesRcvd;

} // WSQ_DecodeAudio()

//---------------------------------------------------------------------------
BOOL WSQ_CalcFftPeakMagnitudeAndFrequency(  // .. using interpolation with neighbour bins
                  T_SpecBufEntry *pSpectrum, // [in] spectrum (with info header)
                  int    iPeakBinIndex,      // [in] frequency bin index of the 'peak'
                  float  *pfltPeakMag_dB,    // [out] peak magnitude  [dB]
                  float  *pfltPeakFrequency) // [out] peak frequency  [Hz]
// Based on a suggestion by Markus Vester (DF6NM) ..
//
// > Da wir die FFT-Bandbreite und die Fensterfunktion
// > (bzw. ihre Fouriertransformierte) kennen, wissen wir
// >  vorab schon die Krmmung k des Spektrallinien-Scheitels.
// >  Es gelte in dessen Umgebung fr die Pegel (in dB) nherungsweise
// >      A(f) = Amax - k * ((f-fmax)/df)^2
// >  mit dem FFT-Bin-Abstand df.
// >  Dieses k ist also einfach die Nachbarkanalunterdrckung,
// >  wenn Du einen Trger genau in der Mitte eines Bins platziert httest.
// >  Damit kannst Du dann so vorgehen:
// >  1. Suche das Bin mit dem hchsten Pegel im Cursor-Fensterchen.
// >  2. Whle zustzlich den strkeren der beiden Frequenznachbarn aus.
// >  3. Das wirkliche Maximum liegt zwischen diesen beiden Rasterfrequenzen
// >     f1 und f2=f1+df. Fr deren Amplituden gilt nach obigem
// >     A1 = Amax-k*((fmax-f1)/df)^2 bzw. A2 = Amax-k*((f2-fmax)/df)^2.
// >  Durch Auflsen der beiden Gleichungen ergeben sich die
// >     interpolierte Frequenz des Maximums :
// >     fmax = (f1+f2)/2 + df*(A2-A1)/(2*k)
// >  und dessen Pegel
// >     Amax = (A1+A2)/2 + k/4 + (A2-A1)^2/(4*k) ,
// >  zumindest falls ich mich nicht verrechnet habe ;-).
// >  Fr eine amplitudenlineare Skala sollte es hnlich gehen,
// >  es scheint aber einfacher, die Werte von A1 und A2 zu logarithmieren
// >  und die Interpolation in dB durchzufhren.

{
   float pk_freq = pSpectrum->fltCenterFreqOf1stBin
                 + pSpectrum->fltFreqBinWidth_Hz * (float)iPeakBinIndex;
   float pk_ampl;         // amplitude (here actually proportinal to RMS)
   float pk_dB = -200.0;  // dummy for 'cannot retrieve dB-value'
   float f1, ampl1, dB1, f2, ampl2, dB2, kdB, half_bin_width;
   float pk_bin_Fc = pk_freq;
   BOOL  interpolation_ok = FALSE;


   if( iPeakBinIndex>=1 && (iPeakBinIndex+1)<pSpectrum->nUsedBins )
    { // Improve the accuracy by interpolation ...
      pk_ampl = pSpectrum->fltFreqBins[iPeakBinIndex];
      ampl1   = pSpectrum->fltFreqBins[iPeakBinIndex-1];
      ampl2   = pSpectrum->fltFreqBins[iPeakBinIndex+1];
      if( (ampl1>pk_ampl) || (ampl2>pk_ampl) )  // iPeakBinIndex isn't a peak ! the RIGHT neighbour is stronger..
       { // Don't interpolate at all; instead return the values for the center bin
         if( pfltPeakMag_dB != NULL )   // [out] peak magnitude  [dB]
          { *pfltPeakMag_dB = pk_dB;
          }
         if( pfltPeakFrequency != NULL) // [out] peak frequency  [Hz]
          { *pfltPeakFrequency = pk_freq;
          }
         return FALSE;
       } // end if < "no peak at all" >

      // At this point:
      //  pk_ampl = amplitude of middle bin,
      //  ampl1   = left neighbour, ampl2 = right neighbour
      // Which of the two neighbours has the higher peak ?
      // Overwrite the "weaker neighbour" with the center value.
      // dB1 and dB2 will be used as input for interpolation.
      if( ampl1 > ampl2 )  // the left (LF) neighbour is stronger:
       { ampl2 = pk_ampl;  // right neighbour (..2) replaced by center
         f2 = pk_freq;
         f1 = pk_freq - pSpectrum->fltFreqBinWidth_Hz;
       }
      else                 // the right (HF) neighbour is stronger:
       { ampl1 = pk_ampl;  // left neighbour (..1) replaced by center
         f1 = pk_freq;
         f2 = pk_freq + pSpectrum->fltFreqBinWidth_Hz;
       }
     // For the maximum interpolation, the voltages must be converted to dB :
     dB1 = WSQ_VoltageRatioTodB( ampl1 );
     dB2 = WSQ_VoltageRatioTodB( ampl2 );

     // A constant (for cos^2 and similar FFT windows) of 5.9 is a compromise between 6.02 dB and 5.6 dB ..
     kdB = 5.9;
     // Tnx DF6NM:
     // > Die Konstante 5.9 (fr cos^2-FFT-Window) ist ein Kompromiss
     // > zwischen 6.02 dB und 5.6 dB,
     // > der den Restfehler fr alle mglichen Trgerlagen
     // > (mittig im Eimer bis genau zwischen zwei Sthlen) minimieren soll.
     // > Damit erfolgt die eigentliche Interpolation:
     // > frc = (f1 + f2) / 2 + (f2 - f1) * (dB2 - dB1) / (2 * kdB)
     // > dBc = (dB1 + dB2) / 2 + kdB / 4 + (dB2 - dB1) * (dB2 - dB1) / (4 * kdB)
     pk_freq = (  f1 + f2 ) / 2.0 + ( f2 - f1) * ( dB2 - dB1) / (2.0 * kdB);
     pk_dB   = ( dB1 + dB2) / 2.0 + kdB / 4.0 + (dB2 - dB1) * (dB2 - dB1) / (4.0 * kdB);

     // > Zuletzt noch ein Plausibilittscheck, der bei stark
     // > abweichenden Pegeln (durch Rauschen oder am Bildschirmrand)
     // > eine Extrapolation in unsinnige Frequenz-Entfernungen verhindern soll.
     // > Eigentlich soll ja der strkere Nachbar nie mehr als 6dB
     // > unter dem Maximum sein.
     // > IF (dB2 - dB1) > 1.5 * kdB THEN frc = f2: dBc = dB2
     // > IF (dB1 - dB2) > 1.5 * kdB THEN frc = f1: dBc = dB1
     interpolation_ok = TRUE;
     // keep it simple...
     half_bin_width = 0.5 * pSpectrum->fltFreqBinWidth_Hz;
     if( pk_freq > pk_bin_Fc+half_bin_width )
      {  pk_freq = pk_bin_Fc+half_bin_width;
         pk_dB   = dB2;
         interpolation_ok = FALSE;
      }
     if( pk_freq < pk_bin_Fc-half_bin_width )
      {  pk_freq = pk_bin_Fc-half_bin_width;
         pk_dB   = dB1;
         interpolation_ok = FALSE;
      }
   }

  // Write some result values back to the caller's var-space :
  if( pfltPeakMag_dB != NULL )   // [out] peak magnitude  [dB]
   { *pfltPeakMag_dB = pk_dB;
   }
  if( pfltPeakFrequency != NULL) // [out] peak frequency  [Hz]
   { *pfltPeakFrequency = pk_freq;
   }

  return interpolation_ok;
} // end WSQ_CalcFftPeakMagnitudeAndFrequency()

//---------------------------------------------------------------------------
float WSQ_CalcNoiseInRange(
         T_SpecBufEntry *pSpectrum, // [in] spectrum (with info header)
                   int i1, int i2 ) // [in] FFT frequency bin index range
  /*
   *  Noise level detection as suggested by G4JNT ('CFAR'-algorithm) .
   *  Coded by DL4YHF, October 2000, for Spectrum Lab. Later adapted for WSQ.
   *
   * "Automatic" method of noise level determination -based on mail from G4JNT
   * -------------------------------------------------------------------------
   * A particularly good means for noise level detection is to generate an array
   * of amplitudes in each FFT bin, then sort these into order of increasing
   * amplitude.
   * The amplitude of the lower quartile value (eg bin number 256 in a sorted
   * set of 1024 points) PLUS 6dB is then a very good estimate of the mean
   * noise level.
   * This technique then automatically throws away very high values
   * (strong signals) that would otherwise affect thresholding
   * based on averaging.
   *
   *  Inputs:
   *    a_spectrum = points to a spectrum array (FFT output), with array in dB.
   *    i1, i2 = index range for the source array  to be processed
   *  Returns:
   *    an estimation of the mean noise level,
   *    same 'amplitude' (or magnitude) unit as in pSpectrum->fltFreqBins[].
   */
{
 T_QSORT_SAMPLE_PAIR *temp_array;
 int i;
 int n_values;
 float noise;

  // Check (and limit) arguments ..
  LimitInteger( &i1, 0, pSpectrum->nUsedBins );
  LimitInteger( &i2, 0, pSpectrum->nUsedBins );
  n_values = i2-i1+1;  // caution, see notes on SPEC_DT_COMPLEX_ARRAY below !
  if( n_values > 2 )
   {
     // Generate an array of amplitudes which can be sorted without affecting
     // the caller's array
     temp_array = malloc( n_values * sizeof( temp_array[0] ) );
     for(i=0;i<n_values;++i)
      {
        temp_array[i].amplitude = pSpectrum->fltFreqBins[i1++];
        temp_array[i].bin_index = i;  // used to get the frequency after sorting
      }

     // Sort the array into order of increasing amplitude (or magnitude?) .
     // The QuickSort function is implemented in QuickSrt.c .
     QuickSort(temp_array, n_values, QUICKSORT_INCREASING );

     // The amplitude of the lower quartile value (eg bin number 256 in a sorted
     // set of 1024 points) PLUS 6(?)dB is a good estimate of the mean
     // noise level (suggested by G4JNT) .
     i = n_values/4;
#   if(1)
     noise = temp_array[i].amplitude;
#   else // tried to "make the noise-reading less noisy" but the following is nonsense:
     noise = (temp_array[i-2].amplitude + temp_array[i-1].amplitude + temp_array[i].amplitude
            + temp_array[i+1].amplitude + temp_array[i+2].amplitude) / 5.0;
#   endif
     // ex: noise_dB += ?;  // add 6(?) dB
     // here, the contents of T_SpecBufEntry.fltFreqBins are VOLTAGES, thus:
     noise *= 1.75; // multiply by X instead of adding Y dB (dealing with VOLTAGES here)
                    // should have been *2 (~ +6 dB) but wasn't..  WHY ?

     // Clean up temporary array
     free( temp_array );

     return noise;
  }
 else // something wrong with the input parameters:
  {
     return 0.0;
  }
} // end  WSQ_CalcNoiseInRange()


//---------------------------------------------------------------------------
T_SpecBufEntry * WSQ_GetSpectrumFromBuffer( T_WSQDecoder *pDecoder, int iBufIdxOffset ) // result may be NULL !
{
  int iBufIdx;
  if(  (pDecoder->nSpecBufAllocEntries > 0 )
    && (pDecoder->SpectrumBuffer != NULL )
    && (iBufIdxOffset < pDecoder->nSpecBufAllocEntries ) )
   { iBufIdx = pDecoder->iSpecBufIndex - iBufIdxOffset - 1;
     // The LAST (newest) entry in the spectrum buffer, retrieved with iBufIdxOffset=0,
     // is at pDecoder->iSpecBufIndex MINUS ONE (due to the post-increment in WSQ_DecodeAudio) !
     iBufIdx = ( iBufIdx + pDecoder->nSpecBufAllocEntries ) % pDecoder->nSpecBufAllocEntries; // handle circular index wrap-around
     LimitInteger( &iBufIdx, 0, pDecoder->nSpecBufAllocEntries-1);  // just for safety (shouldn't be necessary)
     return pDecoder->SpectrumBuffer + iBufIdx;
   }
  else
   { return NULL;
   }
} // end WSQ_GetSpectrumFromBuffer()

//---------------------------------------------------------------------------
void WSQ_GetDisplayableFreqRange( T_WSQDecoder *pDecoder, float *pfltFmin, float *pfltFmax )
  // Retrieves the displayable frequency range for the spectrogram.
  // Used by the GUI to limit the range when 'zooming out', etc.
  // Takes sampling rate, decimation, limited FFT-buffer-size,
  //       and (future plan) real / quadrature into account.
{
  float f,decimatedSR,fmax;

  decimatedSR = pDecoder->fltInputSamplingRate;
  if( pDecoder->iDecimation > 1 )
   { decimatedSR /= (float)pDecoder->iDecimation;
   }
  fmax = 0.5 * decimatedSR;
  if( pDecoder->iFftInputSize <= 0 )   // not decided for an FFT size yet ?
   { pDecoder->iFftInputSize = WSQ_GetSuitableFftInputSize( pDecoder, pDecoder->fltInputSamplingRate );
   }
  if( pDecoder->fltFftBinWidth <= 0 )  // not calculated yet ? (may be adjusted on-the-fly in DecodeAudio!)
   { pDecoder->fltFftBinWidth = decimatedSR / (float)pDecoder->iFftInputSize;
   }
  if( pDecoder->fltFftBinWidth > 0.0 )
   { f = (float)SPEC_BUF_MAX_BINS * pDecoder->fltFftBinWidth;
     if( fmax > f )      // upper frequency limited by the number of buffered FFT frequency bins ?
      {  fmax = f;
      }
   }
  if( pfltFmin != NULL )
   { *pfltFmin = 0.0;    // may change in future: with I/Q input, "negative" frequencies are possible (-fmax)
   }
  if( pfltFmax != NULL )
   { *pfltFmax = fmax;
   }
} // end WSQ_GetDisplayableFreqRange()


//---------------------------------------------------------------------------
void WSQ_InitEncoder( T_WSQEncoder *pEncoder )
{ memset( pEncoder, 0, sizeof(T_WSQEncoder) );
  pEncoder->fDiddleOnTxIdle   = TRUE;
  pEncoder->fltLowestToneFreq = 1000.0; // frequency of the lowest tone in Hertz. Originally 1000.0 .
}

//---------------------------------------------------------------------------
void WSQ_BeginTransmission( T_WSQEncoder *pEncoder )
{ // Cleans up some of the former statics in the 'generator' .
  // Called from the main program shortly before transmission.
  pEncoder->iSampleOfSymbolCounter = 99999;  // kludge to begin 1st symbol immediately
  pEncoder->iTxNibble1 = pEncoder->iTxNibble2 = -1;  // no pending nibbles
  pEncoder->phase   = pEncoder->fAngle   = 0.0;
  pEncoder->curslot = pEncoder->prevslot = pEncoder->tone = pEncoder->prevtone = 0; // don't leave anything to fate
} // end WSQ_BeginTransmission()

/*------------------------------------------------------------------------------
 *
 *      Transmit audio generator function
 *
 *      All audio processing is done in this function!
 *
 *      (No need to mess with a separate worker thread since this is
 *      already taken care of by PortAudio!)
 */
BOOL WSQ_GenerateAudio( T_WSQEncoder *pEncoder,
        float *pfltSamples, // [out] an array of samples (pointer to float)
        int   nSamples,     // [in] number of samples to be produced now
        int   nChannelsPerSample, // [in] originally TWO(!) channels, for output to PortAudio
        float sampleRate,  // [in] sampling rate for the output [Hz]
        char (*pTxCharReader)(void) ) // [in] callback to read the next character from the transmit text editor .
                                      //  All the ugly windows-specific stuff happens THERE :-)
   // Return value: TRUE when a new tone has begun
   //  (the GUI will update some indicator then)
{ int  iSample;

  unsigned char ch;      // doesn't need to be static anymore, due to WSQ_iTxNibble2
  // YHF 2014-03-19, when the de-factor standard 'WSQ' frequency on MF, 474.2 kHz + 1000 Hz,
  //                 was occupied by a very long lasting WSJTX/JT9-2 signal :
  float fltFirstToneFreq_Hz = (float)pEncoder->fltLowestToneFreq;
        // In ZL2AFP's original code, the 'first tone' was fixed at 1000 Hz .
        // Thus with the default settings (i.e. without WSQ_Config.ini),
        // fltFirstToneFreq_Hz should be exactly 1000.0 [Hz] .
  double dblToneSpacing_Hz = (WSQ_Decoder.channel[0].iWSQmode==WSQ_MODE_OLD)
                       ? WSQ_OLD_TONE_SPACING  /* tone spacing FOR THE "OLD" WSQ: (4/2.048) Hz */
                       : WSQCall_TONE_SPACING; /* tone spacing FOR WSQCall (2017): (3/2.048) Hz */
  // DL4YHF 2014-02-23 (trying to understand some magic numbers) :
  //   1.953125 = spacing between IFK tones.
  //   Details from http://www.qsl.net/zl1bpu/SOFT/WSQ.htm :
  // [1]
  // > In an IFK mode, the data is coded in the difference between tones,
  // > rather than by defining the actual tones used.
  // > Text is transmitted as groups of codes in the range 0 - 31.
  // [2]
  // > The tone numbers to be transmitted are determined by adding each code number
  // > to the immediate previously transmitted code number, then adding one.
  // > If the result is greater than 32, then 33 is subtracted from the tone number.
  // [3]
  // > The resulting tone number is in the range 1 - 33, and is converted
  // > (using a look-up table) into an audio tone for transmission
  // > in 1.953125Hz (4/2.048) steps. Audio tones are generated by a sine-wave
  // > Numerically Controlled Oscillator (NCO), a type of software synthesizer.
  // [4]
  // > The 33 tones used are numbered sequentially from the lowest to the highest,
  // > with the lowest tone being 1000Hz. This is also the transmission starting tone.
  //
  // In the adapted variant by DL4YHF, the transmit 'center frequency' matches
  // the *CENTER FREQUENCY* of the *first* decoder instance
  // (so the operator will always answer on the frequency of that decoder) .
  //
  // [5]
  // > Why codes 0 - 31 in the table, 1-32 IFK differences, and yet there are
  // > 33 tones ? It's a bit like making a post-and-rail fence 32m long
  // > with posts every metre - you need 32m of railing, but 33 posts, not 32 !
  // [6]
  // > In addition, the IFK modulator always adds one to keep the tones
  // > in rotation (so repeated '0' symbols don't give a constant tone),
  // > but this does not change the number of differences- there are still 33 tones.
  //
  // YHF: The above ([6], repeated '0' symbols don't give a constant tone)
  //  could not be obvserved in the original software compiled 2013-12-26 .
  //  In fact, during idle, the transmitted tone was CONSTANT, making it
  //  difficult to tell an 'idle WSQ transmitter' from QRM (steady carriers),
  //  and to identify the begin of a WSQ transmission on an aribtrary frequency.
  //
  // [7]
  // > If you study the table above you will see that the first number
  // > of all code groups is always in the range 0 - 26.
  // YHF: In fact, the first number in the "VAR" column ranges from 0 to 28 (= Initial Symbols),
  //               the second number (if any) ranges from 29 to 31 (= Continuation Symbols) .
  // [8]
  // > At the receiver, when an Initial code is received directly
  // > after a previous Initial code, the earlier one must relate to
  // > a single-symbol character, so is looked up in the table and printed,
  // > and the new code retained for next time.
  // > If the new code is a Continuation code, the previous code
  // > and the new code are looked up in the table as a pair,
  // > and the result printed.
  BOOL fNewSymbol = FALSE;

  pEncoder->nSamplesPerSymbol = (int)(0.5 + (float)WSQ2_SYMBOL_TIME * sampleRate);
  // Example (32 kS/sec) : 65536 samples per WSQ2 symbol .
  for (iSample = 0; iSample < nSamples; ++iSample )
   {

     if( pEncoder->iSampleOfSymbolCounter++ >= pEncoder->nSamplesPerSymbol)  // get a new symbol every 2.048 seconds ...
      {
        pEncoder->iSampleOfSymbolCounter = 0;
        // At the end of the symbol interval, send a Continuation Symbol ? [YHF]
        if( pEncoder->iTxNibble2 > 28) // send the 2nd nibble (aka Continuation Symbol) ?
         {                   // Since 2014-02-23, also used for the 'diddle' option !
           pEncoder->iSampleOfSymbolCounter = 0;
           pEncoder->tone = ( pEncoder->prevtone + pEncoder->iTxNibble2 + 1) % 33;
           fNewSymbol = TRUE;

           // Send frequency data to the synthesizer (when enabled):
           if(pEncoder->SynthesizerActive)
            { SendToSynthesizer( pEncoder->tone );  // ex: SerialPuts(hComm,..)
            }

           pEncoder->prevtone = pEncoder->tone;
           pEncoder->iTxNibble2 = 0;  // Continuation Symbol has been sent
         } // if < Continuation Symbol pending > ?
        else // no 'continuation symbol' pending -> get the next character from the TX text editor...
         {
           ch = pTxCharReader();  // read next char from TX text; returns zero when empty
           if( ch ) // note that 'ch' is only used here -> doesn't need to be static
            { // here: transmitter NOT in 'idle' ...
              pEncoder->iTxNibble1 = (int)wsq_varicode[ch][0];
              pEncoder->iTxNibble2 = (int)wsq_varicode[ch][1]; // ZERO when there's no 2nd nibble (i.e. single-symbol-character)

              // With varicode
              pEncoder->tone = (pEncoder->prevtone + pEncoder->iTxNibble1 + 1) % 33;
              fNewSymbol = TRUE;

              // include command to send frequency data to the synthesizer (how do I send the right info?? )
              if(pEncoder->SynthesizerActive)
               { SendToSynthesizer( pEncoder->tone/*0..32*/ );  // ex: SerialPuts(hComm,..)
               }
/*
Comment from Murray
The only oddity I found is that when the synth is newly turned on (and will be on a default frequency)
it might not make much sense the first time you start the program and then press transmit,
but if you stop and start again, it seems to come right. I've not investigated why that is as yet.
Sure to be an A isn't listening to B or B isn't talking to A problem.
*/
            } // end if( ch )
           else
            { // Arrived here ? The transmitter is idling, i.e. "nothing to send" .
              // The original WSQ code kept repeating the previous symbol (frequency).
              // In DL4YHF's modified code, there is an option to 'send diddles on idle' which,
              // similar to RTTY but also to PSK31, provides synchronisation for the receiver.
              // Also, there is the option to automatically switch from TRANSMIT to RECEIVE
              // (but that is handled in the main module, not here in the codec) .
              if( pEncoder->fDiddleOnTxIdle )
               { // OPTION: When the transmitter is idle, use the two 'edge frequencies'
                 //        (alternatively send the LOWEST and HIGHEST of the IFK tones)
                 pEncoder->iTxNibble1 = pEncoder->iTxNibble2 = 0;  // no 'real data' !
                 if( pEncoder->prevtone <= WSQ2_NUM_TONES/2 )
                  { pEncoder->tone = 32;  // diddle, highest possible tone
                  }
                 else
                  { pEncoder->tone = 0;   // diddle, lowest possible tone
                  }
                 // Send frequency data to the synthesizer (when enabled):
                 if(pEncoder->SynthesizerActive)
                  { SendToSynthesizer( pEncoder->tone );  // ex: SerialPuts(hComm,..)
                  }
                 fNewSymbol = TRUE;  // flag for updating the 'Audio frequency' indicator in the GUI
                 // The above 'diddle' produced a string of '=' characters in the decoder.
                 // Try to reproduce this 'by hand'. From the DEcoder:
                 // > [nibble := difference between consecutive tones aka symbols]
                 // > if(nibble < 0)
                 // >  { nibble = nibble + 33; //allow for wrap-around (17 tones for 16 tone differences)
                 // >  }
                 // > nibble = nibble - 1;     // 1 is added at the Jason TX end so need to subtract it here
                 //
                 // tone #0 - #32 -> nibble1 := ( (0-32) + 33) - 1  = 0
                 // tone #32 - #0 -> nibble2 := ( (32-0)     ) - 1  = 31
                 // From 'WSQ Varicode V3.png' : VAR=(0,31)   ->  "=",  q.e.d .
                 //
                 // Would be nicer to have a non-printing character detected for the 'diddle' (idle tone).
                 // On the other hand, the '=' character rotated by 90 looks like the 'diddle'
                 // in the spectrogram, so just leave it as-is for now, but remember this:
                 // The two-symbol 'diddle' ends with the upper tone as shown above,
                 // to avoid misinterpreting the first 'real' character.
               } // end if( WSQ_fDiddleOnTxIdle )
              else  // don't send 'diddles' on idle but ... umm.. a constant tone or staircase ?
               { // tone = (prevtone + 1) % 33; // according to [6], but the decoder interpreted this as SPACE character ("VAR=0").
                 // According to 'WSQ Varicode V3.png', the IDLE character is "VAR 28,30" . But:
#              if(1)
                 pEncoder->iTxNibble1 = 28;
                 pEncoder->iTxNibble2 = 30;  // -> receiver printed hollow squares
                 pEncoder->tone = (pEncoder->prevtone + pEncoder->iTxNibble1 + 1) % 33;
                 fNewSymbol = TRUE;
                 if(pEncoder->SynthesizerActive)
                  { SendToSynthesizer( pEncoder->tone );  // ex: SerialPuts(hComm,..)
                  }
#              endif
               }
            } // end else < nothing to send, transmitter is 'idle' >
         } // end else < no CONTINUATION symbol pending for transmission >
      } // end if( pEncoder->iSampleOfSymbolCounter ... )  [1st within the sample-generating loop]

     pEncoder->phase = 2.0*3.141592654* ( fltFirstToneFreq_Hz/*1000?*/ + pEncoder->tone*dblToneSpacing_Hz/*1.953125 Hz ?*/) / sampleRate;
     // ex: phase = 2*3.14159* (1000 + tone*1.953125) / sampleRate;  // <<=====change this to suit mode 1.46484375
     //     phase = 2*3.14159* (1000 + tone*1.46484375) / sampleRate; // 3-bin tone spacing (no good)

     pfltSamples[0] = sin(pEncoder->fAngle); // originally, only the 1st output channel was filled
     if( nChannelsPerSample>1 )    // the 2nd channel could be used to generate quadrature output...
      {  pfltSamples[1] = cos(pEncoder->fAngle); // .. for an image-cancelling TX mixer
      }
     pfltSamples += nChannelsPerSample;

     pEncoder->fAngle += pEncoder->phase;
     if(pEncoder->fAngle>=(2.0 * 3.141592654))
      { pEncoder->fAngle = pEncoder->fAngle - 2.0 * 3.141592654;
      }


     pEncoder->prevtone = pEncoder->tone;
     pEncoder->prevslot = pEncoder->curslot;
   }

  return fNewSymbol;
} // WSQ_GenerateAudio()


