/*--------------------------------------------------------------------------*/
/* IIR_Bandpass.c =   Filter Designer and DSP runtime function              */
/*                    for a chain of IIR bandpasses  .                      */
/*                                                                          */
/*  Written by Wolfgang Buescher (DL4YHF), (c) 2003 - 2014 .                */
/*                                                                          */
/*  Based on an idea by Jim Moritz (M0BMU) using analogue design methods .  */
/*                                                                          */
/*                                                                          */
/*  Revision date and terms of use :   see IIR_Bandpass.c !                 */
/*                                                                          */
/*--------------------------------------------------------------------------*/


#ifndef _IIR_BANDPASS_H_
#define _IIR_BANDPASS_H_ 1



// some CONSTANTS ...
#ifndef  PI
 #define PI 3.14159265358979323846
#endif


// Macro to avoid annoying warnings about 'assigned a value that is never used':
#ifndef UNREFERENCED
# ifdef __BORLANDC__
#  define UNREFERENCED(x)  (void)x
# else
#  define UNREFERENCED(x)  x=x
# endif
#endif // ndef UNREFERENCED ?


// max. filter order for this implementation of the IIR algorithm:
#define IIR_FILTER_MAX_COEFFICIENTS (3*20)  /* three coeffs per biquad! */
  // Note: If this is not enough, use the filter with FFT CONVOLUTION instead !

  // Filter response types for the IIR bandpass designer suggested by M0BMU.
  // The trailing number is the number of 2nd-order IIR stages/sections
  // (biquads) created by this design.
#define FILTER_RESPONSE_BESSEL_10       0
#define FILTER_RESPONSE_BUTTERWORTH_10  1
#define FILTER_RESPONSE_CHEBYSHEV01_10  2
#define FILTER_RESPONSE_CHEBYSHEV1_10   3
#define FILTER_RESPONSE_LINPHASE05_10   4
#define FILTER_RESPONSE_LINPHASE005_10  5
#define FILTER_RESPONSE_GAUSSIAN6DB_10  6
#define FILTER_RESPONSE_GAUSSIAN12DB_10 7



typedef struct tIIR_Filter // Structure with ALL DATA needed to "run" an FIR- or IIR filter
{ // (also used as destination structure for some filter designers)

  // DESIGN INPUT parameters for the entire chain:
  double dblFcenter;        // filter center frequency [Hertz]
  double dblBandwidth;      // total filter bandwidth  [Hertz]
  double dblSampleRate;     // samples per second (for the design)  
  int iFilterResponseType;  // like FILTER_RESPONSE_BESSEL_10 etc

  // DESIGN OUTPUT parameters, calculated in IIRBandpass_Design():
  int    max_coeff;   // 1..<SND_IIR_FILTER_MAX_COEFFICIENTS>, 2=2nd order filter
                      // or <numer_of_stages> * <coeffs_per_stage> !!!
  // (in this case, the bandpass is implemented as a chain
  //  of 2nd order IIR filters aka 'biquads'.
  //  For each stage (biquad) in that chain, THREE coefficients are used.
  //  Thus, with IIR_FILTER_MAX_COEFFICIENTS=99,  up to 33 biquads are possible.
  float alpha[IIR_FILTER_MAX_COEFFICIENTS+1];
  float beta[IIR_FILTER_MAX_COEFFICIENTS+1];

  // RUNTIME workspace, used when 'running a signal' through the filter:
  float z[IIR_FILTER_MAX_COEFFICIENTS+1];   // delay line

} T_IIR_Filter;



//---------------- Prototypes -----------------------------------------------

// For peaceful coexistence between "C" and "C++" :
#ifndef CPROT
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  /* ! "cplusplus" */
#endif

//---------------------------------------------------------------------------
CPROT int IIRBandpass_Design(
     double dblFcenter, double dblBandwidth, double dblSampleRate,
     int iFilterResponseType,  // [in] e.g. FILTER_RESPONSE_BESSEL_10, etc etc
     T_IIR_Filter *pFilter );  // [out] destination structure with the filter coeffs

//---------------------------------------------------------------------------
CPROT void IIRBandpass_ProcessRealSignal(
     float *input_samples,  float *output_samples, int number_of_samples,
     T_IIR_Filter *pFilter );  // [in,modified] structure with filter coeffs and delay line



#endif // _IIR_BANDPASS_H_
