/*--------------------------------------------------------------------------*/
/* IIR_Bandpass.c =   Filter Designer and DSP runtime function              */
/*                    for a chain of IIR bandpasses  .                      */
/*                                                                          */
/*  Written by Wolfgang Buescher (DL4YHF), (c) 2003 - 2014 .                */
/*                                                                          */
/*  Based on an idea by Jim Moritz (M0BMU) using analogue design methods .  */
/*                                                                          */
/*  Much later, this module was also used (as a copy) in DL4YHF's adaption  */
/*  of ZL2AFP's WSQ (Weak-Signal QSO mode); with a combination of the       */
/*  multi-stage IIR-bandpass designer *plus* the IIR filter itself :        */
/*              [ c:\cbproj] \WSQ2\IIR_Bandpass.c                           */
/*  Original:   [ c:\cbproj] \SoundUtl\DesignIIRbandpass.c                  */
/*                                                                          */
/*  Revision date:   2003-02-11  (YYYY-MM-DD)                               */
/*                                                                          */
/*                                                                          */
/*   #####################################################################  */
/*                                                                          */
/*      This software is provided 'as is', without warranty of any kind,    */
/*      express or implied. In no event shall the author be held liable     */
/*      for any damages arising from the use of this software.              */
/*                                                                          */
/*      Permission to use, copy, modify, and distribute this software and   */
/*      its documentation for non-commercial purposes is hereby granted,    */
/*      provided that the above copyright notice and this disclaimer        */
/*      appear in all copies and supporting documentation.                  */
/*                                                                          */
/*      The software must NOT be sold or used as part of any commercial     */
/*      or "non-free" product.                                              */
/*                                                                          */
/*   #####################################################################  */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include <math.h>

#include "IIR_Bandpass.h"

//---------------------------------------------------------------------------
static double pow2(double x)
{ return x*x;
}

//---------------------------------------------------------------------------
int IIRBandpass_Design(
   double dblFcenter,           // [in] filter center frequency [Hertz]
   double dblBandwidth,         // [in] total filter bandwidth  [Hertz]
   double dblSampleRate,        // [in] samples per second, required for normalization
      int iFilterResponseType,  // [in] e.g. FILTER_RESPONSE_BESSEL_10, see *.h
   T_IIR_Filter *pFilter )      // [out] destination structure with the filter coeffs
  //
  // Designs a chain (cascade) of IIR-type BANDPASS FILTERS,
  //            with carefully distributed poles,
  //      for a given bandwidth, center frequency, and sampling rate .
  //
  // Input parameters:
  //      Fcenter, Bandwidth, SampleRate : values in Hertz(!)
  //      FilterResponseType: FILTER_RESPONSE_xxxx (see FilterTypes.h)
  //
  // Return values :  1 = ok
  //                 <0 = Error
{
  int    iNrOfSections;
                  // Somewhere else called this "number of stages",
                  // but that's misleading.
                  // A 'Section' is a 2nd-order IIR filter here !
  int    i, iSectionIndex, iOddEvenIndex, iCoeffIndex;
  double alpha[10]; // filter parameters from Jim's spreadsheet
  double beta[10];
  double a,b,c,d,e,g,q,m,w,qbp,f0a,f0b;  // temporary values from Jim's formulas
  double temp;
  double sfc;      // center frequency of the current section
  double sbw;      // bandwidth of the current section
  double section_gain;     // gain for the current section
  double k,r,omega_a,omega_n;
  double a0,a1,a2,b1,b2;   // helpers to calculate a single IIR2-section
  // double T_sample;

  if( (dblBandwidth<=0) || (dblSampleRate<=0) )
    return -1;
  // T_sample = 1.0 / dblSampleRate;

  a0=a1=a2=b1=b2=0.0;      // ensure everything is defined and initialized...
  section_gain = 0.0;
  qbp = dblFcenter / dblBandwidth; // cell 'E5' of spreadsheet
  if(qbp<=0)
     return -2;

  iNrOfSections = 0;

  // First take a look at the filter response type.
  // Depending on the type of the filter, the 'alpha' and 'beta' parameters
  // are preset in a CASE list (which can be easily expanded if you have
  // other good 'filter recipes' or full 'filter cookbook' .
  // The following 'alpha' and 'beta' parameters are taken from a spreadsheet
  // table written by Jim Moritz, M0BMU.
  // See \SpecLab\Filters_by_M0BMU\filtercalc.xls .
  // Notes:
  //   - For a 10th-order filter, five of these parameter pairs are required.
  //   - In EXCEL, a group of 5 lines * 2 columns can be selected and copied
  //     into the clipboard, and pasted into the "C"-source code. Very neat !
  switch(iFilterResponseType)
   {
    case FILTER_RESPONSE_BESSEL_10      :
         // Recipe for a 10th order Bessel filter.
         // Copied from M0BMU's table ( pole positions alpha, beta ) .
         // Looks a bit too narrow (compared with others), but...
         //  (from Maxim's "Analog Filter Design Demystified") :
         // > The Bessel filter gives a constant propagation delay across
         // > the input frequency spectrum. Therefore, applying a square wave
         // > (consisting of a fundamental and many harmonics) to the input
         // > of a Bessel filter yields an output square wave with no overshoot
         // > (all the frequencies are delayed by the same amount).
         // > Other filters delay the harmonics by different amounts,
         // > resulting in an overshoot on the output waveform.
         iNrOfSections = 10;
#if(0) // from M0BMU
         alpha[0] = 0.5172;   beta[0] = 0.5092;
         alpha[1] = 0.5412;   beta[1] = 0.4682;
         alpha[2] = 0.6   ;   beta[2] = 0.3896;
         alpha[3] = 0.7326;   beta[3] = 0.2792;
         alpha[4] = 1.151 ;   beta[4] = 0.1437;
#else  // from http://www.rfcafe.com/references/electrical/bessel-poles.htm,
       // "bessel filter poles" (not sure if these are also "alpha" and "beta",
         alpha[0] = 1.9335;
         alpha[1] = 0.8684;
         alpha[2] = 1.8478;
         alpha[3] = 1.6669;
         alpha[4] = 1.3649;
         beta[0] = 0.2424;
         beta[1] = 2.2996;
         beta[2] = 0.7295;
         beta[3] = 1.2248;
         beta[4] = 1.7388;
#endif
         break;

    case FILTER_RESPONSE_BUTTERWORTH_10 :
         // Recipe for a 10th order Butterworth filter.
         // Copied from M0BMU's table.
         iNrOfSections = 10;
         alpha[0] = 0.9877;   beta[0] = 0.1564;
         alpha[1] = 0.8910;   beta[1] = 0.4540;
         alpha[2] = 0.7071;   beta[2] = 0.7071;
         alpha[3] = 0.454 ;   beta[3] = 0.891 ;
         alpha[4] = 0.1564;   beta[4] = 0.9877;
         break;

    case FILTER_RESPONSE_CHEBYSHEV01_10 :
         // Recipe for an 8th order Chebyshev filter
         //            with 0.1 dB passband ripple.
         // Copied from M0BMU's table.
         iNrOfSections = 8;
         alpha[0] = 0.3058;   beta[0] = 0.1952;
         alpha[1] = 0.2592;   beta[1] = 0.5558;
         alpha[2] = 0.1732;   beta[2] = 0.8319;
         alpha[3] = 0.06082;  beta[3] = 0.9812;
         break;

    case FILTER_RESPONSE_CHEBYSHEV1_10  :
         // Recipe for an 8th order Chebyshev filter
         //            with 1.0 dB passband ripple.
         // Copied from M0BMU's table.
         iNrOfSections = 8;
         alpha[0] = 0.1737;   beta[0] = 0.1956;
         alpha[1] = 0.1473;   beta[1] = 0.5571;
         alpha[2] = 0.0984;   beta[2] = 0.8337;
         alpha[3] = 0.03456;  beta[3] = 0.9836;
         break;

    case FILTER_RESPONSE_LINPHASE05_10  :
         // Recipe for a 10th order 'linear phase' filter
         //            with 0.5 degree phase ripple.
         //    (may need this one day for digital demodulators!)
         // Copied from M0BMU's table.
         iNrOfSections = 10;
         alpha[0] = 0.5249;   beta[0] = 0.3487;
         alpha[1] = 0.5193;   beta[1] = 1.0429;
         alpha[2] = 0.5051;   beta[2] = 1.7261;
         alpha[3] = 0.4741;   beta[3] = 2.385;
         alpha[4] = 0.3708;   beta[4] = 2.994;
         break;

    case FILTER_RESPONSE_LINPHASE005_10 :
         // Recipe for a 10th order 'linear phase' filter
         //            with 0.05 degree phase ripple.
         //    (may need this one day for digital demodulators!)
         // Copied from M0BMU's table.
         iNrOfSections = 10;
         alpha[0] = 0.7592;   beta[0] = 0.3413;
         alpha[1] = 0.7467;   beta[1] = 1.0195;
         alpha[2] = 0.7159;   beta[2] = 1.6836;
         alpha[3] = 0.6475;   beta[3] = 2.3198;
         alpha[4] = 0.4777;   beta[4] = 2.9128;
         break;

    case FILTER_RESPONSE_GAUSSIAN6DB_10 :
         // Bandpass with Transitional Gaussian response to -6dB.
         // Copied from M0BMU's table.
         iNrOfSections = 10;
         alpha[0] = 0.3384;   beta[0] = 0.2101;
         alpha[1] = 0.3164;   beta[1] = 0.618 ;
         alpha[2] = 0.2677;   beta[2] = 0.9852;
         alpha[3] = 0.1849;   beta[3] = 1.2745;
         alpha[4] = 0.06706;  beta[4] = 1.4389;
         break;

    case FILTER_RESPONSE_GAUSSIAN12DB_10:
         // Bandpass with Transitional Gaussian response to -12dB.
         // Copied from M0BMU's table.
         // Used in SpecLab's FSK(RTTY) demodulator .
         iNrOfSections = 10;
         alpha[0] = 0.4535;   beta[0] = 0.2794;
         alpha[1] = 0.4352;   beta[1] = 0.8289;
         alpha[2] = 0.3886;   beta[2] = 1.3448;
         alpha[3] = 0.2908;   beta[3] = 1.7837;
         alpha[4] = 0.1136;   beta[4] = 2.0599;
         break;

    default:
         return -4;   // cannot create this type of filter response (yet)..

   } // end switch(iFilterResponseType)

  // Save the DESIGN INPUT parameters for the entire chain;
  //   it may be handy to know them later for some applications
  pFilter->dblFcenter   = dblFcenter;    // filter center frequency [Hertz]
  pFilter->dblBandwidth = dblBandwidth;  // total filter bandwidth  [Hertz]
  pFilter->dblSampleRate= dblSampleRate; // samples per second
  pFilter->iFilterResponseType = iFilterResponseType; // FILTER_RESPONSE_BESSEL_10, etc

  // Prepare the general filter coefficient structure:
  pFilter->max_coeff = 3*iNrOfSections-1;

  // Calculate the coefficients for all defined stages.
  // Note that we are stepping by two in this loop, because every section
  // will produce two different IIR2-bandpasses (for odd+even indices).
  for(iSectionIndex=0; iSectionIndex<iNrOfSections; iSectionIndex+=2)
   {
     // DL4YHF's "C" implementation of M0BMU's filter calculation spreadsheet.
     // Note that these coeffs are for a time-continuous (aka "analog") filter;
     //       we will use the bilinear transform to turn this into
     //       a time-discrete ("digital") filter further below .
     a = alpha[iSectionIndex/2];
     b = beta[iSectionIndex/2];
     c = pow2(a) + pow2(b);
     d = 2*a / qbp;
     e = c/pow2(qbp) + 4;
     temp = e*e-4*d*d; // temporary argument for square root, must not be negative
     if (temp<0)
        return -10;  // something fishy, set breakpoint here
     g = sqrt(temp);
     q = sqrt((e+g) / (2*d*d));
     m = a*q / qbp;
     temp = pow2(m) - 1;
     if (temp<=0)
        return -11;  // something fishy, set breakpoint here
     w = m + sqrt(temp);
     f0a = dblFcenter / w;
     f0b = w * dblFcenter;

     for(iOddEvenIndex=0; iOddEvenIndex<=1; ++iOddEvenIndex)
      {
       // set the properties for a single filter section...
       if(iOddEvenIndex==0)  // odd section index, use f0a :
        {
         sfc = f0a;          // section's center frequency
         sbw = f0a / q;      // section's bandwidth
        }
       else                  // even section index, use f0b :
        {
         sfc = f0b;
         sbw = f0b / q;
        }
       // To achieve unity gain for the overall filter,
       // with equally distributed gains for all sections (Jim's "K"-formula):
       // This is not required for floating point maths, but for integer..
       section_gain =  sqrt(  pow2(pow2(sfc)-pow2(dblFcenter))
                            + pow2(dblFcenter * sbw) )
                                / (dblFcenter * sbw) ;

       // Now that we have the center frequency & bandwidth of the current
       // section, we can generate the coefficients for a single second-order
       // IIR bandpass. This would be a second-order continuous time bandpass
       //  in the analog world, i.e. in the s-domain;
       //  But the digital filter 'operates' in the z-domain !
       // Fortunately, it's possible to transform the 'analog filter'
       //   into an equivalent 'digital filter' : the bilinear transform.
       //  From http://en.wikipedia.org/wiki/Bilinear_transform ,
       //  "Frequency warping" (oder, im deutschen Wiki, "Frequenzverzerrung"):
       // > To determine the frequency response of a continuous-time filter,
       // > the transfer function H_a(s)  is evaluated at s = j * omega
       // > which is on the j * omega axis.
       // > Likewise, to determine the frequency response of a discrete-time
       // > filter, the transfer function H_d(z) is evaluated at z = e^(j*omega*T)
       // > which is on the unit circle, |z| = 1 .
       //     ( T is the sample time, T_sample; reciprocal of the sample rate )
       // > When the actual frequency of \omega\ is input to the discrete-time
       // > filter designed by use of the bilinear transform, it is desired
       // > to know at what frequency, \omega_a\ , for the continuous-time filter
       // > that this \omega\ is mapped to.  (.....)
       // > Every point on the unit circle in the discrete-time filter z-plane,
       // > z = e^( j * omega * T) is mapped to a point on the j * omega axis
       // > on the continuous-time filter s-plane, s = j \omega_a \ .
       // > That is, the discrete-time to continuous-time frequency mapping
       // > of the bilinear transform is
       // >   omega_a = (2/T) * tan( omega * T / 2)
       // > and the inverse mapping is
       // >   omega = (2/T) * arctan( omega_a * T / 2)   .
       // > The discrete-time filter behaves at frequency \omega \ the same way
       // > that the continuous-time filter behaves at frequency
       // >    (2/T) * tan(omega*T/2) .  Specifically, the gain and phase shift
       // > that the discrete-time filter has at frequency \omega \ is the same
       // > gain and phase shift that the continuous-time filter has
       // > at frequency (2/T) * tan(omega * T/2) . This means
       // > that every feature, every "bump" that is visible in the frequency response
       // > of the continuous-time filter is also visible in the discrete-time
       // > filter, but at a different frequency. (!)
       // > For low frequencies (that is, when omega << 2/T or omega_a << 2/T),
       // >   \omega\  approx \omega_a\ .
       // Below:
       //      omega_a = continuous-time filter frequency (from "analog" design);
       //      omega   = discrete-time filter frequency (for the "digital" implementation);
       //      omega_n = omega normalized for the sampling rate
       //
       omega_a = 2*PI* sfc;    // Kreisfrequenz "analog" (Mittenfrequenz der Filterstufe, s-Domne)
    // omega = (2.0 / T_sample) * atan( omega_a * T_sample / 2.0); // -> Kreisfrequenz z-Domne
       omega_n = omega_a /dblSampleRate; // omega normalized to the sample rate (0...1)
          // (Note: using omega instead of omega_a for the following algorithm
          //        [omega_n] resulted in "completely wrong" frequencies!)  .
          // But, a 3700 Hz wide Chebyshev bandpass with fc=77.5 kHz, at fs=192 kHz,
          //      designed with the algorithm below had a very skewed passband;
          //      decaying by roughly 5 dB from the lower to the upper end .
          //      No skewed passband at fc=~~15 kHz.  Reason ??
          //      The peaks in the passband ripple appeared at the frequencies
          //      'sfc' in Hz, so no problem with that.
       // A part from CalculateIIR2Coeffs was copied here for clarity:
       // Calculate coefficients for a single 2nd-order IIR-bandpass.
       // The following formulas are taken from
       //   "The Scientist and Engineer's Guide to Digital Signal Processing",
       //   Second Edition, page 326 (recursive narrow band filters) .
       //   Locally saved as ?/literatur/DSP_Guide/CH19.PDF .
       // "sfc" is the section's center frequency,
       // "sbw" is the section's bandwidth.
       r = 1.0 - 3 * sbw/dblSampleRate;
       k = ( 1.0 - 2.0*r*cos(omega_n) + r*r ) / ( 2.0-2.0*cos(omega_n) );
       // calculate coefficients for a band-pass filter:
       a0 = 1.0 - k;
       a1 = 2.0 * (k-r) * cos(omega_n);
       a2 = r*r - k;
       b1 = 2*r*cos(omega_n);
       b2 = -r*r;

       // Multiply all alpha-coeffs of "this" 2nd-order IIR section
       // with the section's individual "gain".
       // Then write the filter coefficients into the destination structure.
       iCoeffIndex = 3 * (iSectionIndex + iOddEvenIndex);
       if(iCoeffIndex<IIR_FILTER_MAX_COEFFICIENTS-2)
        {
         pFilter->alpha[iCoeffIndex]   = a0  * section_gain;
         pFilter->beta [iCoeffIndex]   = 0.0;
         pFilter->alpha[iCoeffIndex+1] = a1  * section_gain;
         pFilter->beta [iCoeffIndex+1] = -b1;
         pFilter->alpha[iCoeffIndex+2] = a2  * section_gain;
         pFilter->beta [iCoeffIndex+2] = -b2;
        }
       else // illegal coeff_index
        {
         return -2;
        }
      } // end for(iOddEvenIndex...
   } // end for(iSectionIndex...

 return 1; // ok

} // end IIRBandpass_Design()


/***************************************************************************/
void IIRBandpass_ProcessRealSignal(
          float *input_samples,
          float *output_samples,
          int    number_of_samples,
          T_IIR_Filter *pFilter )      // [in,modified] structure with filter coeffs and delay line
    /* Let a block of samples run through a conventional digital filter
     * with convolution in the time domain (IIR or FIR, no "FFT Filter").
     * Used by the real-time audio processing thread
     * and by the digimode-decoder (for MARK and SPACE separation).
     * The filter coefficients and the filter's 'memories' are passed
     * in a pointer to a T_FILTER_DATA struct (see SoundThd.h) .
     * Note: It doesn't hurt if the input and output blocks are the same memory.
     */
{
  int   jmax,stage,n_stages,i,j,k;
  float x,y,z;
  float *alpha, *beta;
  float *queue_in_ptr, *coeff_ptr, *queue_rd_ptr;


  if (pFilter->max_coeff > IIR_FILTER_MAX_COEFFICIENTS)
      pFilter->max_coeff = IIR_FILTER_MAX_COEFFICIENTS;

  //------------------------------------------------------------------
  // algorithm for CASCADED SECOND ORDER  IIR - filters
  // consisting of 2nd(!) order filters (aka "biquads") in a chain.
  // Example: 5 stages *  2 nd-order IIR's in a chain:
  //          cascaded  = 3 (3 coefficients per stage)
  //          max_coeff = 14 (coefficient indices 0..14 are valid)
  n_stages = (pFilter->max_coeff+1) / 3;
  for(i=0; i<number_of_samples; ++i)
   {
     x = input_samples[i];  // filter input = "X"
     k = 0;                 // coefficient array index
     alpha = pFilter->alpha;  // (one pointer ref less for access in loop)
     for(stage=0; stage<n_stages; ++stage)
      {  // y0 = x*alpha0 + z0
         y = x * (*alpha++) + pFilter->z[k];   // k=0: no "beta" coefficient here !
         if (y > 1e20)
          { // avoid floating point errors (crash!) when the filter oscillates:
            // 2011-03-23: Got here with state=0, 1-kHz LP @ 192kS, y=1.04e20,
            //    pFilter->alpha[0..2] = 0.00411564, 0.008231,  0.00411564
            //    pFilter->beta [0..2] = 0.0       , 1.990531, -0.9905792
            y = 1e20;
          }
         ++k;
         // z0 = x*alpha1 + z1 - y*beta1
         pFilter->z[k-1] = x * (*alpha++) + pFilter->z[k] - y * pFilter->beta[k];
         ++k;
         // z1 = x*alpha2 - y*beta2
         pFilter->z[k-1] = x * (*alpha++) - y * pFilter->beta[k];
         x = y; // output from this stage = input for next stage
         ++k;   // coefficient index for next stage (biquad) of the cascade
      } // end for(stage... )
     output_samples[i] = y;
   } // end for <all samples in the filter buffer>

} // end IIRBandpass_ProcessRealSignal()

