/*
 *  File:    ?\WSQ2\CW_Generator.c
 *  Date:    2014-03-15
 *  Author:  DL4YHF (recycled from Spectrum Lab).
 *  Purpose: CW (Morse Code) generator .
 *   Used to send a station identifier (or other) in Morse code,
 *   as required / recommended in some countries .
 *
 *
 *      #####################################################################
 *
 *        This software is provided 'as is', without warranty of any kind,
 *        express or implied. In no event shall the author be held liable
 *        for any damages arising from the use of this software.
 *
 *        Permission to use, copy, modify, and distribute this software and
 *        its documentation for non-commercial purposes is hereby granted,
 *        provided that the above copyright notice and this disclaimer appear
 *        in all copies and supporting documentation.
 *
 *        The software must NOT be sold or used as part of any commercial
 *        or "non-free" product.
 *
 *      #####################################################################
 */

#include <string.h>
#include <math.h>

#include "CW_Generator.h"

//---------------------------------------------------------------------
// Constant tables ..

static const WORD CW_Table[59]=
  // The word in the CW table is divided into 8 groups of two bits starting
  // at the msb side.  The two bits represent one of four possible states.
  //  00 - end of character
  //  01 - DOT
  //  10 - DASH
  //  11 - SPACE of two dot times
{
   0xF000,      // 1111 1100 0000 0000b  ( 32)       WORD SPACE
   0x0000,      // 0000 0000 0000 0000b  ( 33)  !
   0x0000,      // 0000 0000 0000 0000b  ( 34)  "
   0x0000,      // 0000 0000 0000 0000b  ( 35)  #
   0x0000,      // 0000 0000 0000 0000b  ( 36)  $
   0x0000,      // 0000 0000 0000 0000b  ( 37)  %
   0x0000,      // 0000 0000 0000 0000b  ( 38)  &
   0x0000,      // 0000 0000 0000 0000b  ( 39)  '
   0x0000,      // 0000 0000 0000 0000b  ( 40)  (
   0x0000,      // 0000 0000 0000 0000b  ( 41)  )
   0x566C,      // 0101 0110 0110 1100b  ( 42)  *    ...-.-  SK
   0x6670,      // 0110 0110 0111 0000b  ( 43)  +    .-.-.   AR
   0xA5AC,      // 1010 0101 1010 1100b  ( 44)  ,    --..--
   0x0000,      // 0000 0000 0000 0000b  ( 45)  -
   0x666C,      // 0110 0110 0110 1100b  ( 46)  .    .-.-.-
   0x9670,      // 1001 0110 0111 0000b  ( 47)  /    -..-.
   0xAAB0,      // 1010 1010 1011 0000b  ( 48)  0    -----
   0x6AB0,      // 0110 1010 1011 0000b  ( 49)  1    .----
   0x5AB0,      // 0101 1010 1011 0000b  ( 50)  2    ..---
   0x56B0,      // 0101 0110 1011 0000b  ( 51)  3    ...--
   0x55B0,      // 0101 0101 1011 0000b  ( 52)  4    ....-
   0x5570,      // 0101 0101 0111 0000b  ( 53)  5    .....
   0x9570,      // 1001 0101 0111 0000b  ( 54)  6    -....
   0xA570,      // 1010 0101 0111 0000b  ( 55)  7    --...
   0xA970,      // 1010 1001 0111 0000b  ( 56)  8    ---..
   0xAA70,      // 1010 1010 0111 0000b  ( 57)  9    ----.
   0x0000,      // 0000 0000 0000 0000b  ( 58)  :
   0x0000,      // 0000 0000 0000 0000b  ( 59)  ;
   0x0000,      // 0000 0000 0000 0000b  ( 60)  <
   0x95B0,      // 1001 0101 1011 0000b  ( 61)  =    -...-   BT
   0x0000,      // 0000 0000 0000 0000b  ( 62)  >
   0x5A5C,      // 0101 1010 0101 1100b  ( 63)  ?    ..--..
   0x0000,      // 0000 0000 0000 0000b  ( 64)  @
   0x6C00,      // 0110 1100 0000 0000b  ( 65)  A    .-
   0x95C0,      // 1001 0101 1100 0000b  ( 66)  B    -...
   0x99C0,      // 1001 1001 1100 0000b  ( 67)  C    -.-.
   0x9700,      // 1001 0111 0000 0000b  ( 68)  D    -..
   0x7000,      // 0111 0000 0000 0000b  ( 69)  E    .
   0x59C0,      // 0101 1001 1100 0000b  ( 70)  F    ..-.
   0xA700,      // 1010 0111 0000 0000b  ( 71)  G    --.
   0x55C0,      // 0101 0101 1100 0000b  ( 72)  H    ....
   0x5C00,      // 0101 1100 0000 0000b  ( 73)  I    ..
   0x6AC0,      // 0110 1010 1100 0000b  ( 74)  J    .---
   0x9B00,      // 1001 1011 0000 0000b  ( 75)  K    -.-
   0x65C0,      // 0110 0101 1100 0000b  ( 76)  L    .-..
   0xAC00,      // 1010 1100 0000 0000b  ( 77)  M    --
   0x9C00,      // 1001 1100 0000 0000b  ( 78)  N    -.
   0xAB00,      // 1010 1011 0000 0000b  ( 79)  O    ---
   0x69C0,      // 0110 1001 1100 0000b  ( 80)  P    .--.
   0xA6C0,      // 1010 0110 1100 0000b  ( 81)  Q    --.-
   0x6700,      // 0110 0111 0000 0000b  ( 82)  R    .-.
   0x5700,      // 0101 0111 0000 0000b  ( 83)  S    ...
   0xB000,      // 1011 0000 0000 0000b  ( 84)  T    -
   0x5B00,      // 0101 1011 0000 0000b  ( 85)  U    ..-
   0x56C0,      // 0101 0110 1100 0000b  ( 86)  V    ...-
   0x6B00,      // 0110 1011 0000 0000b  ( 87)  W    .--
   0x96C0,      // 1001 0110 1100 0000b  ( 88)  X    -..-
   0x9AC0,      // 1001 1010 1100 0000b  ( 89)  Y    -.--
   0xA5C0       // 1010 0101 1100 0000b  ( 90)  Z    --..
}; // end CW_Table[]

//---------------------------------------------------------------------------
void CW_InitGenerator( T_CWGenerator *pGenerator,
        float fltCenterFrequency, float fltSecondsPerDot )
{
  memset( pGenerator, 0, sizeof( T_CWGenerator ) );
  pGenerator->fltCenterFrequency = fltCenterFrequency;
  pGenerator->fltSecondsPerDot = fltSecondsPerDot;
  pGenerator->ramp     = pGenerator->phase = 0.0;
  pGenerator->ramp_inc = 0.0;
  pGenerator->wTxShiftReg = CW_Table[0];
  pGenerator->iSampleOfSymbolCounter = pGenerator->nSamplesPerSymbol = 0;
} // end CW_InitGenerator()


//---------------------------------------------------------------------------
BOOL CW_GenerateAudio( T_CWGenerator *pGenerator,
        float *pfltSamples, int nSamples, int nChannelsPerSample, float sampleRate,
        char (*pTxCharReader)(void) )
{ int  iSample;
  int  nSamplesPerDot = (int)(0.5 + pGenerator->fltSecondsPerDot * sampleRate);
  unsigned char ch;
  int  symbol = (pGenerator->wTxShiftReg >> 14) & 3;
  // Ramp time shall be 10 ms to reduce keying clicks.
  // Too many badly clicking transmitters out there, anyway.. !
  // Ramp from 0 to 1 within 10 ms = (0.01 * sampleRate) samples,
  //  -> increment (or decrement) pGenerator->ramp by the following amount PER SAMPLE:
  float ramp_inc  = 1.0 / ( 0.01/*seconds*/ * sampleRate );
  float phase_inc = 2.0*3.141592654 * pGenerator->fltCenterFrequency / sampleRate;


  for (iSample = 0; iSample < nSamples; ++iSample )
   {
     if( pGenerator->iSampleOfSymbolCounter++ >= pGenerator->nSamplesPerSymbol)
      {
        pGenerator->iSampleOfSymbolCounter = 0;
        if( ! pGenerator->sending_dot_gap ) // send the one-dot-gap after each dash or dot:
         { pGenerator->ramp_inc = -ramp_inc; // "ramp down" (relative amplitude from 1 to 0)
           pGenerator->sending_dot_gap = TRUE;
           pGenerator->nSamplesPerSymbol = nSamplesPerDot;
         }
        else // have sent the dot-gap, time for the next dash, dot, or inter-character space
         { pGenerator->sending_dot_gap = FALSE;
           // Shift the next symbol (dash,dot,space) into the two most significant bits
           //  in the transmit shift register:
           pGenerator->wTxShiftReg <<= 2;
           // The word in the CW table is divided into 8 groups of two bits starting
           // at the msb side.  The two bits represent one of four possible states.
           //  00 - end of character
           //  01 - DOT
           //  10 - DASH
           //  11 - SPACE of two dot times
           // If the transmit shift register is EMPTY, read the next character for transmission,
           // and convert it into Morse code (for the tx shift register).
           if( pGenerator->wTxShiftReg == 0 )
            { ch = pTxCharReader();
              if( ch>='a' && ch<='z' )
               {  ch = ch-'a'+'A';
               }
              if( ch>=' ' && ch<='Z')
               {   pGenerator->wTxShiftReg = CW_Table[ ch-' '];  // look up pattern
               }
              else // nothing more to send !
               { pGenerator->wTxShiftReg = 0;
               }
            } // end if < tx shift register empty >
           symbol = (pGenerator->wTxShiftReg >> 14) & 3;
           switch( symbol )
            { case 0 :  // end of character (or text?)
                 pGenerator->ramp_inc = -ramp_inc; // "ramp down"
                 if( pGenerator->wTxShiftReg == 0 )
                  { pGenerator->nSamplesPerSymbol = 0;  // end of transmission
                  }
                 else
                  { pGenerator->nSamplesPerSymbol = 3*nSamplesPerDot;
                  }
                 break;
              case 1 :  // dot
                 pGenerator->nSamplesPerSymbol =  nSamplesPerDot;
                 pGenerator->ramp_inc = +ramp_inc; // "ramp up"
                 SendToSynthesizer( 16 );  // ex: SerialPuts(hComm,..)
                 break;
              case 2 :  // dash
                 pGenerator->nSamplesPerSymbol = 3*nSamplesPerDot;
                 pGenerator->ramp_inc = ramp_inc; // "ramp up"
                 SendToSynthesizer( 16 );  // ex: SerialPuts(hComm,..)
                 break;
              case 3 :  // space of two dot times (a gap of one dot follows EACH symbol, and is considered part of the symbol)
                 pGenerator->nSamplesPerSymbol = 2*nSamplesPerDot;
                 pGenerator->ramp_inc = -ramp_inc; // "ramp down"
                 break;
            } // end switch( symbol )
         } // end if < send dot-gap >
      } // end if( pGenerator->iSampleOfSymbolCounter++ >= pGenerator->nSamplesPerSymbol)
     pGenerator->ramp += pGenerator->ramp_inc;
     if( pGenerator->ramp < 0.0 )
      {  pGenerator->ramp = 0.0;
      }
     if( pGenerator->ramp > 1.0 )
      {  pGenerator->ramp = 1.0;
      }
     pfltSamples[0] = pGenerator->ramp * sin(pGenerator->phase); // originally, only the 1st output channel was filled
     if( nChannelsPerSample>1 )    // the 2nd channel could be used to generate quadrature output...
      {  pfltSamples[1] = pGenerator->ramp * cos(pGenerator->phase); // .. for an image-cancelling TX mixer
      }
     pfltSamples += nChannelsPerSample;

     pGenerator->phase += phase_inc;
     if(pGenerator->phase >=(2.0 * 3.141592654))
      { pGenerator->phase -= 2.0 * 3.141592654;
      }
   }

  // return TRUE as long as there is something more to send:
  return (pGenerator->iSampleOfSymbolCounter < pGenerator->nSamplesPerSymbol )
    ||   (pGenerator->wTxShiftReg != 0 );

} // end CW_GenerateAudio()
