//----------------------------------------------------------------------
// File:  \cbproj\AudioIO\AudioIO.h ('master' file, various copies elsewhere)
// Author: Wolfgang Buescher (DL4YHF)
// Date:   2014-02-22
// Purpose: Prototypes, data types and error codes for the "Audio I/O" functions
//          (module prefix "AIO_" , first used in the
//           "winamp-to-SpecLab" plugins, but may be used elsewhere, too...)
//
//      For more information, see ?\AudioIO\manual\AudioIO_Manual.pdf
//                   or visit www.qsl.net/d/dl4yhf/AudioIO/index.html .
//
// Included by:
//  - all projects compiling the DLLs (for example, in_AudioIO.dll ,
//        compiled from \cbproj\AudioIO\aio2winamp.c )
//  - all projects USING such DLLs (for example DL4YHF's Spectrum Lab,
//        using the wrapper C:\cbproj\AudioIO\AudioIO.c )
//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Copyright, License, and Disclaimer (for the AudioIO-skeleton)
//      Copyright (c) 2008...2065, Wolfgang Bscher (DL4YHF)
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are met:
//   1. Redistributions of source code must retain the above copyright notice,
//      this list of conditions and the following disclaimer.
//   2. Redistributions in binary form must reproduce the above copyright notice,
//      this list of conditions and the following disclaimer in the
//      documentation and/or other materials provided with the distribution.
//   3. Neither the name of the author nor the name of other contributors
//      may be used to endorse or promote products derived from this software
//      without specific prior written permission.
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
//  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
//   OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
//   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//   LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
//
//
// Revision history:
//  V0.03, 2014-02-22:  Integrated Audio-I/O in WSQ2. Details in AudioIO.C .
//                      Another COPY of AudioIO.h now exists 
//                      in \cbproj\WSQ2\AudioIO\AudioIO.h .
//
//  V0.02, 2012-08-08:  Added support for I2PHD's "ExtIO"-DLLs, which are not
//                      really compatible with DL4YHF's "Audio-I/O"-DLL-API,
//                      but this DLL-HOST (AudioIO.C) will map the necessary
//                      calls to the ExtIO-DLL functions, if an ExtIO-DLL
//                      was loaded instead of an Audio-I/O-DLL .
//
//  V0.01, 2011-06-13:  Re-activated the "Audio I/O DLL" project.
//                      Put all statics inside a structure,
//                      to allow multiple instances used in ONE application.
//                Plan: Use this DLL also as an input for SPECTRUM LAB,
//                      more or less as a single-input / multiple-output node:
//
//        ___________         ____________        __________
//       |           |       |            |      |          | ---> Winamp,ASIO,..
//       | Soundcard | --->  | SpecLab #1 | ---> | Audio-IO | ---> SpecLab #2
//       |___________|       |____________|      |__________| ---> SpecLab #3
//        ______   /|\        (with SR-        .               .
//       |      |   |          calibrator      .               .
//       | GPS  |->-|          + resampler)    .               .
//       |______|                              .               .
//                                           One Writer     Multiple Readers
//
//  V0.00, 2008-03-27:  Project started as a WINAMP-"input"-plugin
//                      to send audio data from DL4YHF's "Spectrum Lab"
//                      into Winamp, to play with Icecast/Shoutcast streaming .
//
//----------------------------------------------------------------------

#ifndef  AUDIO_IO_H
 #define AUDIO_IO_H  /* protect against multiple (recursive) inclusion */

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !

#ifndef _CHUNK_INFO_H_
# include "ChunkInfo.h" // T_ChunkInfo in c:\cbproj\SoundUtl\ChunkInfo.h
#endif


#ifndef CPROT /* for peaceful co-existence between C and C++ ... */
// Note on "extern": do not trust the lousy example from Borland !!
//   >extern "c" = wrong    (not recognized by Borland C++ V4.0)
//   >extern "C" = correct  (though the help system says something different)
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  /* ! "cplusplus" */
#endif

// NO-WAY ! ! : #include "vcl.h" // stuff like 'bool' (must get rid of this Borland-specific craze,
    // but the function pointer declarations for Winrad's ExtIO used it)

//----------------------------------------------------------------------
// Common data types.
//   Most of them defined somewhere else; if not, define them here .
//----------------------------------------------------------------------

#ifndef BYTE
 #define BYTE unsigned char /* unsigned 8 bit integer, NOT USED AS "CHARACTER" */
#endif

#ifndef  INT32  // signed 32 bit integer, compiler-independent !
         // On most compilers, 'int' is 32 bit, but it may be 16 or 64 bit.
         // Data Type Prefix simply 'i' because this is the standard .
         // Unfortunately, many compilers (except Borland to name one)
         // are too stupid to evaluate sizeof(<some_builtin_type>) 
         // in a preprocessor condition like the one below.
# if(defined __BORLANDC__)  /* <<< add your compiler here if it's smart enough */
#  if(sizeof(int)==4)  /* this is what we expect under Win32... */
#    define INT32 int
#  else  /* this would happen with cross compilers for 16-bit systems: */
#    if(sizeof(long)==4)
#      define INT32 long
#    else
#       error "Neither sizeof(int) nor sizeof(long) is four bytes ?!? "
#    endif
#  endif
# else  // not one of the smart compilers (cannot compare sizeof(int) in the prepro)..
#  define INT32 int  // does this dumb compiler really use 32bit-"int" ?
# endif
#endif // ndef INT32

#ifndef  INT16   // signed 16 bit integer */
         // Data Type Prefix 'i16' because this is something special :)
# define INT16 short
#endif // ndef INT16

// ex: FLT64 =  64-bit floating point, IEEE 754 format .
// Data Type Prefix 'dbl' (double precision;  52 bit mantissa) .
// Since most compilers (except Borland) are too stupid for something like..
// #if( sizeof(double)!=8 )
// #  error "sizeof(double) isn't 8 ... how can this be ?"
// #endif
// ... we rely on "double" always being "64-bit floating point" .


//----------------------------------------------------------------------
// Audio I/O Error codes (negative return values often mean "failed")
//   DON'T MODIFY, only extend ! Modifying existing codes will render
//          any audio I/O, and the host which uses it, INCOMPATIBLE !
//   When adding new error codes, also modify SoundUtl\Sound.cpp::
//     ConvertErrorCode_AIO_to_SOUND() .
//----------------------------------------------------------------------
#define AIO_NO_ERROR                 0
#define AIO_GENERAL_ERROR            -1  /* use this error code if you don't find anything better below */
#define AIO_ERROR_DLL_NOT_LOADED     -2
#define AIO_ERROR_DLL_NOT_COMPATIBLE -3
#define AIO_ERROR_NOT_IMPLEMENTED    -4  /* tried to call a function which is not implemented in the DLL */
#define AIO_ERROR_INVALID_PARAMETER  -5
#define AIO_ERROR_OUT_OF_RESOURCES   -6
#define AIO_ERROR_FORMAT_NOT_SUPPORTED -10
#define AIO_ERROR_SAMPRATE_NOT_SUPPORTED -11
#define AIO_ERROR_FILE_NOT_FOUND    -12
#define AIO_ERROR_INPUT_NOT_OPEN    -14
#define AIO_ERROR_OUTPUT_NOT_OPEN   -15
#define AIO_INVALID_HANDLE          -17  /* used as ERROR CODE but also for non-initialized "handles"  */
#define AIO_ERROR_CANCELLED         -18  /* used in AIO_ShowAboutBox, or when something was "cancelled by the user" */
#define AIO_ERROR_CANNOT_WRITE_YET  -20
#define AIO_ERROR_CANNOT_READ_YET   -21
#define AIO_ERROR_NOT_INITIALIZED   -22


//----------------------------------------------------------------------
// Operating mode of the AUDIO PLAYER (for example, Winamp) .
//    The player *may* be controlled through the AudioIO-compatible plugin,
//    or know what it should do from the "URL" entered in the audio player.
//    For example, if the user enters "audio://" in Winamp,
//    that audio player will try to connect to in_AudioIO.DLL,
//    which replaces the input from the soundcard.
// Details in \cbproj\AudioIO\manual\AudioIO_Manual.odt/pdf .
//----------------------------------------------------------------------
#define AIO_PLAYER_MODE_OFF    0   // the audio player seems to be "off"
#define AIO_PLAYER_MODE_AUDI   1   // receiving from AudioIO (URL = "audi://")
#define AIO_PLAYER_MODE_RSV2   2   // reserved for future extensions
#define AIO_PLAYER_MODE_RSV3   3   // reserved for future extensions
#define AIO_PLAYER_MODE_RSV4   4   // reserved for future extensions
#define AIO_PLAYER_MODE_TONE   5   // receiving test tone    (URL = "tone://")


//----------------------------------------------------------------------
// Special values for the 'iTimeout_ms' parameter
//  (max timeout value in milliseconds, blocking or non-blocking mode)
//----------------------------------------------------------------------
#define AIO_TIMEOUT_NON_BLOCKING 0
#define AIO_TIMEOUT_DEFAULT      500


//----------------------------------------------------------------------
// Windows messages, used for interaction between the DLL's control panel
// and the application (usually not required for the application)
//----------------------------------------------------------------------
// ex: #define WM_AUDIO_IO_CONTROL (WM_APP+0x314)  /* WM_APP: 0x8000 .. 0xBFFF */
#define WM_AUDIO_IO_CONTROL (WM_USER+0x314)
   // WM_AUDIO_IO_CONTROL( wParam= function code, lParam= function value)
   // Possible values for the 'function code' in MESSAGE.wParam :
#define WM_AUDIO_IO_FC_NONE           0    /* just a "ping" */
#define WM_AUDIO_IO_FC_PANEL_CLOSED   10   /* control panel closed by user */

//----------------------------------------------------------------------
// Data structures (rarely used in the DLL interface, but possibly in the app)
//   Avoid changing them, because every modification here may cause
//   incompatibility between the DLL and the using applications .
//----------------------------------------------------------------------

#ifdef  C_AIO_RAW_STREAM_PARAMS
  // Beware: The T_AIO_RawStreamParams structure is declared
  //             in \cbproj\WinampPluginHost\WPH_Core.h
  //         AND in \cbproj\AudioIO\AudioIO.h .
  // DON'T MODIFY ! Modifying this struct will cause "DLL hell" .
# if( C_AIO_RAW_STREAM_PARAMS != 2 )
#   error "Terror ! Incompatible def's of T_AIO_RawStreamParams in WPH_Core.h and AudioIO.h !!"
# endif
#else
# define C_AIO_RAW_STREAM_PARAMS 2
typedef struct t_AIO_RawStreamParams
{ // For new projects / DLLs, use the T_ChunkInfo structure as defined
  //  in c:\cbproj\SoundUtl\ChunkInfo.h instead of T_AIO_RawStreamParams !
  double  dblPreciseSampleRate; // "calibrated" number of samples per second .
             // Set this field to zero if you don't know it.
             // AudioIO will replace it with the "nominal" sample rat then .
  INT32  iNominalSampleRate; // something like 11025, 22050, 44100, ...
  INT32  iNumChannels;       // 1=mono, 2=stereo, 4=quadrophonic or 2*I/Q
  // ex: iBitsPerSample;     // usually 16 (bits per sample)
  INT32  iStreamOptions;     // set to zero by default
  INT32  iReserve1;          // fill up the structure to 20 bytes
} T_AIO_RawStreamParams;
#endif // ! defined C_AIO_RAW_STREAM_PARAMS ?

// An ultimatively simple, lock-free circular buffer for incoming samples.
//    Only used by a few DLLs, most noteably: Winrad-ExtIO-compatibles.
//    Allocated when required; size depends on the sampling rate.
typedef struct t_AIO_SampleBuffer
{ int   nChannelsPerSample;
  long  i32AllocatedSize_nSamples, i32HeadIndex, i32TailIndex;  // units: SAMPLE POINTS, not "single floating point values"
  long  i32MagicP;    // contains 0x31415926 if the struct is initialized
  float *pfltSamples; // buffer with <nChannelsPerSample*i32AllocatedSize_nSamples> single floating point values
  long  i32MagicE;    // contains 0x27182818 if the struct is initialized
} T_AIO_SampleBuffer;


// Function pointer type for CALLBACK functions (from the DLL into the application)
typedef INT32 _stdcall (*tpCallbackFromDLL)(
        void  *pvUserData,      // [in] address of some user data passed to the callback
        INT32 i32FunctionCode,
        INT32 iParam1,
        INT32 iParam2);

// function pointer TYPES (prototypes) for functions loaded from the DLL ...
//   Note that 'DLLIMPORT' doesn't necessary include '__stdcall' - so beware !
#define DLLIMPORT __declspec(dllimport)
typedef const char* DLLIMPORT (*tpErrorCodeToString)(int iAIOErrorCode);
typedef INT32 DLLIMPORT (*tpGetOutputParams)(char *pszAudioStreamID, double *pdblSampleRate,INT32 *piNumChannels);
typedef INT32 DLLIMPORT (*tpGetWinampParams)(double *pdblSampleRate,INT32 *piNumChannels,
                                       int *piWinampInitialized, int *piWinampInputPaused );

typedef INT32 DLLIMPORT (*tpOpenAudioOutput)(
        char *pszAudioStreamID,   // [in] 'Stream ID' for the Audio-I/O-DLL
        char *pszDescription,     // [in] a descriptive name, usually the name of the application
        double dblSampleRate,     // [in] samples per second, "precise"
        INT32  iNumChannels,      // [in] 1=mono, 2=stereo, 4=quadrophonic
        INT32 iTimeout_ms,        // [in] max timeout in milliseconds for THIS call
        INT32 iOpenOptions);      // [in] bit combination of AIO_OPEN_OPTION_... flags:
#             define AIO_OPEN_OPTION_NORMAL 0  /* 'nothing special' */

typedef void  DLLIMPORT (*tpCloseAudioOutput)(INT32 iAudioIoHandle);

typedef INT32 DLLIMPORT (*tpWriteOutputSamplePoints)(
        INT32 iAudioIoHandle,     // [in] handle to the input "device"
        float *pfltSource,        // [in] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points",
                                  //      normalized to -1.0 ... +1.0 for full swing
        INT32 iNumSamplePoints,   // [in] number of SAMPLE POINTS(!) to write or send
        INT32 nChannelsPerSample, // [in] number of samples PER SAMPLE POINT
        INT32 iTimeout_ms ,       // [in] max timeout in milliseconds, or AIO_TIMEOUT_NON_BLOCKING
        T_ChunkInfo *pInChunkInfo,// [in,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL(!)
        INT32 *piHaveWaited_ms);  // [out,optional] "have been waiting here for XX milliseconds"
        // Return : see specification of AIO_Host_WriteOutputSamplePoints() in AudioIO.cpp .
        //        ( actually the return value, when non-negative, indicates SUCCESS
        //          and the number of remaining (unused) entries
        //          in the recipient's buffer )

typedef INT32 DLLIMPORT (*tpCanWriteOutput)(INT32 iAudioIoHandle);

typedef INT32 DLLIMPORT (*tpOpenAudioInput)( // returns a HANDLE (details in AudioIO.c) !
        char *pszAudioStreamID,   // [in] 'Stream ID' for the Audio-I/O-DLL
        char *pszAudioReaderID,   // [in] short but unique name to identify the READER, usually the name of the application
        double  dblWantedSampleRate,  // [in(!)] "wanted" sample rate, often IGNORED (reasons below..)
        double *pdblOutSampleRate,    // [out(!)] samples per second, see notes for AIO_Host_OpenAudioInput() .
        INT32 nChannelsPerSample,     // [in] number of channels PER SAMPLE POINT
        INT32 iTimeout_ms,            // [in] max timeout in milliseconds for THIS call
        INT32 iOpenOptions,           // [in] bit combination of AIO_OPEN_OPTION_... flags
        T_ChunkInfo *pOutChunkInfo ); // [out] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h
typedef void  DLLIMPORT (*tpCloseAudioInput)(INT32 iAudioIoHandle);
typedef INT32 DLLIMPORT (*tpReadInputSamplePoints)(
        INT32 iAudioIoHandle,     // [in] audio-handle, from ..OpenAudioOutput()
        float *pfltDest,          // [out] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
        INT32 iNumSamplePoints,   // [in] number of SAMPLE POINTS(!) to read
        INT32 nChannelsPerSample, // [in] number of channels PER SAMPLE POINT
        INT32 iTimeout_ms ,       // [in] max timeout in milliseconds, or AIO_TIMEOUT_NON_BLOCKING
        T_ChunkInfo *pOutChunkInfo, // [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL (!)
        INT32 *piHaveWaited_ms);    // [out,optional] "have been waiting here for XX milliseconds"

typedef INT32 DLLIMPORT (*tpCanReadInput)(INT32 iAudioIoHandle);

typedef INT32 DLLIMPORT (*tpShowAboutBox)(HWND hwndParent);
typedef INT32 DLLIMPORT (*tpShowControlPanel)(
       HWND   hwndParent,            // [in] handle to the host's window
       HWND   *phwndControlPanel,    // [out] handle of the DLL's control panel
       INT32  iDisplayOptions ,      // AIO_DISPLAY_SHOW_WINDOW, AIO_DISPLAY_HIDE_WINDOW, etc (bit combination)
#      define AIO_DISPLAY_SHOW_WINDOW 1
#      define AIO_DISPLAY_HIDE_WINDOW 2
       void  *pvUserDataForCallback,         // [in,optional] address of some user data passed to the callback
       tpCallbackFromDLL pCallbackFunction); // [in,optional] address of a callback function, may be NULL


#if( SWI_SUPPORT_EXTIO_DLL ) // support Winrad's ExtIO-DLLs (here: HOST SIDE) ?
  // Note: When these DLLs were compiled, both "int" and "long" were in fact 32-bit integer.
  //       A concise data type is used here to avoid running into trouble
  //       when compiling the DLL HOST (!) on a 64-bit compiler .
  //       FROM WHERE, and IN WHICH ORDER these functions should be called,
  //       is described in   c:\cbproj\ExtIO\ExtIO_Notes_DL4YHF.htm  !
  //       In Winrad_Extio.pdf, prototypes for the following functions
  //       were garnished with __stdcall  __declspec(dllexport) .
  //       But in "defines.h", for example:
  //       typedef bool  (__stdcall * pfnInitHW)(char *name, char *model, int& index);
  typedef BOOL  (__stdcall * pfnInitHW)(char *name, char *model, int *pi32Index);
  typedef BOOL  (__stdcall * pfnOpenHW)(void);
  typedef void  (__stdcall * pfnCloseHW)(void);
  typedef int   (__stdcall * pfnStartHW)(int extLOfreq); // sets "VFO", aka "LO", aka "NCO" frequency !
  typedef void  (__stdcall * pfnStopHW)(void);
  typedef void  (__stdcall * pfnSetCallback)(void (*) (int, int, float, void *));
  typedef int   (__stdcall * pfnSetHWLO)(long);
  typedef long  (__stdcall * pfnGetHWLO)(void);
  typedef int   (__stdcall * pfnGetStatus)(void);
  typedef long  (__stdcall * pfnGetHWSR)(void);
  typedef void  (__stdcall * pfnRawDataReady)(long samprate, void *Ldata, void *Rdata, int numsamples);
  typedef void  (__stdcall * pfnShowGUI)(void);
  typedef void  (__stdcall * pfnHideGUI)(void);
  typedef void  (__stdcall * pfnTuneChanged)(long tunefreq);
  typedef long  (__stdcall * pfnGetTune)(void);
  typedef void  (__stdcall * pfnModeChanged)(char mode);
  typedef char  (__stdcall * pfnGetMode)(void);
  typedef void  (__stdcall * pfnIFLimitsChanged)(long lowfreq, long highfreq);
  typedef void  (__stdcall * pfnFiltersChanged)(int loCut, int hiCut, int pitch);
  typedef void  (__stdcall * pfnGetFilters)(int *piLoCut, int *piHiCut, int *piPitch);
# define EXTIO_HWT_NONE      0  /* formerly an ENUM but these values are carved in stone, */
# define EXTIO_HWT_SDR14     1  /* because they are used INSIDE THE EXTIO-DLLs too .      */
# define EXTIO_HWT_SDRX      2
# define EXTIO_HWT_USB_Int16 3  /* DO NOT MODIFY THESE DEFINES ! */
    // > Thehardwaredoesitsowndigitizationandtheaudiodataarereturnedtothe host
    // > viathecallbackdevice.Datamustbein16-bit(short)format,littleendian.
# define EXTIO_HWT_Soundcard 4
    // > Theaudiodataarereturnedviathesoundcardmanagedbythe host.Theexternal
    // > hardwarejustcontrolstheLO,andpossiblyapreselector,underDLLcontrol.
# define EXTIO_HWT_USB_Int24 5
    // > Thehardwaredoesitsowndigitizationandtheaudiodataarereturnedtothe host
    // > viathecallbackdevice.Dataarein24-bitintegerformat,littleendian.
# define EXTIO_HWT_USB_Int32 6
    // > Thehardwaredoesitsowndigitizationandtheaudiodataarereturnedtothe host
    // > viathecallbackdevice.Dataarein32-bitintegerformat,littleendian.
# define EXTIO_HWT_USB_Float32 7
    // > Thehardwaredoesitsowndigitizationandtheaudiodataarereturnedtothe host
    // > viathecallbackdevice.Dataarein32-bitfloatformat,littleendian.
#endif // SWI_SUPPORT_EXTIO_DLL ?


// A structure for the DLL host (not required for the DLL itself!) .
// Added 2011-06-12 to make the DLL-host instantiatable, which means
//   "we can have MORE THAN ONE SUCH DLL" loaded by a single application .
typedef struct tAIO_DLLHostInstanceData
{
   // Complete path + filename of the DLL which has been loaded for this instance:
   char sz255DLLPathAndName[256];
   char sz511OriginalDirectory[512];
   BOOL fCurrentDirectorySetToDLL;   // TRUE:"temporarily set current directory to the path of the DLL" (SetCurrentDirectory)
                                     // FALSE:"the 'current directory' is back to where it should be"
   T_AIO_SampleBuffer buf;  // lock-free, circular buffer for SOME (not all) DLLs

#define AIO_HOST_USE_CRITTER 1
#if( AIO_HOST_USE_CRITTER )
   CRITICAL_SECTION Critter; // added 2012-11-18 when suspecting multi-threading issues
   int nCritterEntryCounter; // added 2012-11-19 for debugging purposes only
#endif                       // (unexplainable crashes with with an ExtIO-DLL)
   BOOL fCritterInitialized; // this especially applies to the CRITICAL SECTION (but not to the loaded DLL)

   // function pointers into the DLL (when loaded)...
   tpErrorCodeToString m_pErrorCodeToString;
   tpGetWinampParams  m_pGetWinampParams;
   tpGetOutputParams  m_pGetOutputParams;
   tpShowAboutBox     m_pShowAboutBox;
   tpShowControlPanel m_pShowControlPanel;

   tpOpenAudioOutput  m_pOpenAudioOutput;
   tpCloseAudioOutput m_pCloseAudioOutput;
  // tpWriteAudioOutput m_pWriteAudioOutput; // discarded, use WriteOutputSamplePoints instead !
   tpCanWriteOutput   m_pCanWriteOutput;
   tpWriteOutputSamplePoints m_pWriteOutputSamplePoints; // with T_ChunkInfo, added 2011-06

   // Audio-INPUT from an Audio-I/O-dll, added here (in AudioIO.h) 2011-06-12 :
   tpOpenAudioInput   m_pOpenAudioInput;
   tpCloseAudioInput  m_pCloseAudioInput;
   tpReadInputSamplePoints m_pReadInputSamplePoints;
   tpCanReadInput     m_pCanReadInput;


   // Other DLL-Host-Instance-related stuff :
   BOOL      fOpenForBusiness; // flag to avoid multithreading-trouble, see AIO_Host_FreeDLL()
   BOOL      fControlPanelVisible; // flag if the DLL's control panel (or "GUI") is CURRENTLY visible
   BOOL      fSRChangedByDriver;   // flag set BY THE ExtIO-DLL when *the driver* changed the sampling rate
   BOOL      fVFOChangedByDriver;  // flag set BY THE ExtIO-DLL when *the driver* changed the VFO frequency
   BOOL      fStopRequestedByDriver;  // flag set BY THE ExtIO-DLL when *the driver* wants us to stop "him" ?!
   BOOL      fStartRequestedByDriver; // flag set BY THE ExtIO-DLL when *the driver* wants us to start "him" again?!
   HINSTANCE h_AudioIoDLL; // handle to the DLL (NULL when not loaded)
   INT32     iAudioInputHandle, iAudioOutputHandle; // AIO_INVALID_HANDLE when not *opened*
   double    dblInputSamplingRate;     // added 2012-08-12 (may have to GUESS this!!)
   long      i32ExternalVfoFrequency;  // added 2012-08-11 for ExtIO-DLL/"StartHW" .
   int       nIQPairsPerExtIoCallback; // added 2012-08-12; result from "StartHW" .

# if( SWI_SUPPORT_EXTIO_DLL )   // support Winrad-compatible ExtIO-DLLs ?
   // Added 2012-08-08 for the ExtIO-compatible part :
   BOOL  fExtIOLoaded, fExtIOHWready, fHWopened, fHWstarted;  // keep track of WHAT was already called
   INT32 i32ExtIOHWtype; // EXTIO_HWT_ ... ("hardware type" for ExtIO-DLL)
   char  devNameIn[128], devNameOut[128], s[512], sepb[16], sep;
   char  extHWname[128], extHWmodel[128];
   pfnInitHW           m_pExtIO_InitHW;
   pfnOpenHW           m_pExtIO_OpenHW;
   pfnCloseHW          m_pExtIO_CloseHW;
   pfnStartHW          m_pExtIO_StartHW;
   pfnStopHW           m_pExtIO_StopHW;
   pfnSetCallback      m_pExtIO_SetCallback;
   pfnSetHWLO          m_pExtIO_SetHWLO;
   pfnGetHWLO          m_pExtIO_GetHWLO;
   pfnGetStatus        m_pExtIO_GetStatus;
   pfnGetHWSR          m_pExtIO_GetHWSR;
   pfnRawDataReady     m_pExtIO_RawDataReady;
   pfnShowGUI          m_pExtIO_ShowGUI;
   pfnHideGUI          m_pExtIO_HideGUI;
   pfnTuneChanged      m_pExtIO_TuneChanged;
   pfnGetTune          m_pExtIO_GetTune;
   pfnModeChanged      m_pExtIO_ModeChanged;
   pfnGetMode          m_pExtIO_GetMode;
   pfnIFLimitsChanged  m_pExtIO_IFLimitsChanged;
   pfnFiltersChanged   m_pExtIO_FiltersChanged;
   pfnGetFilters       m_pExtIO_GetFilters;
# endif // SWI_SUPPORT_EXTIO_DLL ?



} T_AIO_DLLHostInstanceData;


//---------------------------------------------------------------------------
// Methods for a simple, lock-free, sample buffer T_AIO_SampleBuffer .
//    Details only in C:\cbproj\AudioIO\AudioIO.C .
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
CPROT BOOL AIO_SampleBuffer_IsValid( T_AIO_SampleBuffer *pBuf );
CPROT BOOL AIO_SampleBuffer_Init( T_AIO_SampleBuffer *pBuf,
             long  nSamplePointsPerBuffer, int   nChannelsPerSample );
CPROT void AIO_SampleBuffer_Delete( T_AIO_SampleBuffer *pBuf );
CPROT long AIO_SampleBuffer_GetNumSamplesFree( T_AIO_SampleBuffer *pBuf );
CPROT long AIO_SampleBuffer_GetNumSamplesBuffered( T_AIO_SampleBuffer *pBuf );
CPROT BOOL AIO_SampleBuffer_WriteSamples_Float( T_AIO_SampleBuffer *pBuf,
             float *pfltSrc, int iNumSamplePoints,  int nChannelsPerSample);
CPROT long AIO_SampleBuffer_ReadSamples( T_AIO_SampleBuffer *pBuf,
             float *pfltDest, int iMaxSamplePoints, int nChannelsPerSample);
//---------------------------------------------------------------------------



//----------------------------------------------------------------------
// Prototypes for the Audio I/O  "API"
//   These are the functions implemented in AudioIO.c .
//   Do NOT confuse these functions with the functions IN THE DLL !
//   (the functions in the DLL do NOT contain a pointer to the instance-data
//    in their argument list ! To avoid conflicts between function names
//    in the 'DLL host' (AudioIO.C) and in the DLL (like aio2winamp.c),
//    the functions IMPLEMENTED in AudioIO.C begin with AIO_Host_ ,
//    while functions IMPLEMENTED in the DLL begin with AIO_ (w/o "Host") .
//  Without these modifications, the (wxDev-C++-)project "in_AudioIO" could not
//  be compiled anymore in 2011-06-14 !
//----------------------------------------------------------------------


//----------------------------------------------------------------------
CPROT void AIO_Host_InitInstanceData(
        T_AIO_DLLHostInstanceData *pInstData); // [in,out] instance data to be initialized

//----------------------------------------------------------------------
CPROT void AIO_Host_DeleteInstanceData(  // counterpart to AIO_Host_InitInstanceData()
        T_AIO_DLLHostInstanceData *pInstData);


//----------------------------------------------------------------------
CPROT INT32 AIO_Host_LoadDLL(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        char *pszName); // Dynamically loads the DLL into memory.

//----------------------------------------------------------------------
CPROT void AIO_Host_FreeDLL(
        T_AIO_DLLHostInstanceData *pInstData); // [in,out] DLL-host instance data
   // Should be called by application when terminating. Frees resources.

//----------------------------------------------------------------------
CPROT const char *AIO_Host_ErrorCodeToString(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        INT32 iAIOError );
   // Converts a (negative) Audio-I/O error code into a human readable string.
   // Since some error codes (numbers below minus 1000) may be DLL-specific,
   // this function is also located inside the DLL (and not in AudioIO.c) .

//----------------------------------------------------------------------
CPROT INT32 AIO_Host_GetOutputParams(T_AIO_DLLHostInstanceData *pInstData,
        char *pszAudioStreamID, // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE
           // (here: optional, use NULL to query info about "the first available source")
        double *pdblSampleRate, // [out] samples per second, "precise", optional
        INT32 *piNumChannels);  // [out] 1=mono, 2=stereo, 4=quadrophonic
   // Called by the application to query the current settings
   //  for a certain "outgoing" audio connection - BEFORE OPENING IT .
   // (*) For an explanation of the "source name", see AIO_OpenAudioOutput() .

//----------------------------------------------------------------------
CPROT INT32 AIO_Host_GetWinampParams( // OPTIONAL function (only if winamp is supported)
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        double *pdblSampleRate,     // [out] samples per second used by Winamp
        INT32 *piNumChannels,       // [out] 1=mono, 2=stereo   used by Winamp
        INT32 *piWinampInitialized, // [out] 0=winamp not connected to the DLL,  1=winamp connected(*)
        INT32 *piWinampInputPaused);// [out] 0=winamp's input is paused, 1=winamp's input is running
  // (*) "connected" means, Winamp is there, has loaded the plugin DLL,
  //     and has said "hello" by calling the "Init()"-function of the Winamp-plugin.
  //     As soon as Winamp calls the "Quit"-function, the flag gets cleared.
  // All pointers in the argument list are OPTIONAL.
  //  If you don't need some of the above values, just pass a NULL pointer .

//----------------------------------------------------------------------
CPROT long AIO_Host_GetNominalSampleRate( // -> current sampling rate in Hertz.
         T_AIO_DLLHostInstanceData *pInstData); // [in] DLL-host instance data
  // Only used for the ExtIO-DLLs, after the flag
  //  'pInstData->fSRChangedByDriver' has been set somewhere.
  // Most other drivers will NOT change the sampling rate by their own gusto.

//----------------------------------------------------------------------
CPROT BOOL  AIO_Host_HasVFO( // -> TRUE=VFO supported, FALSE=no. Used for ExtIO-DLLs.
       T_AIO_DLLHostInstanceData *pInstData); // [in] DLL-host instance data

//----------------------------------------------------------------------
CPROT long  AIO_Host_GetVFOFrequency( // -> current VFO tuning frequency in Hertz.
       T_AIO_DLLHostInstanceData *pInstData); // [in] DLL-host instance data
  // Required for the ExtIO-DLLs, after the flag
  //  'pInstData->fVFOChangedByDriver' has been set somewhere (in a callback).
  // Most other drivers will NOT change the VFO frequency by their own gusto.
  // Spectrum Lab periodically calls this function IN THE MAIN THREAD,
  // from CSound::GetVFOFrequency(), to update the VFO display in the main
  // window after -for example- the tuning frequency has been changed
  // unexpectedly by the ExtIO-DLL *itself* (for example, FiFi-SDR, Si570) .

//----------------------------------------------------------------------
CPROT long   AIO_Host_SetVFOFrequency( // Return value explained in AudioIO.c !
       T_AIO_DLLHostInstanceData *pInstData, // [in] DLL-host instance data
       long   i32VFOFrequency );             // [in] new VFO frequency in Hertz
  // IMPORTANT: For the ExtIO-DLLs, this function must be called
  //  BEFORE opening the "driver" for input, because for some reason,
  //  the VFO frequency must be passed as a function argument in "StartHW" !
  //  (failing to do so caused ExtIO_RTL.DLL to open a stupid message box
  //   telling something about "PLL not locked", instead of silently returning
  //   with an error code, and leaving the user interaction to the HOST) .

//----------------------------------------------------------------------
CPROT INT32 AIO_Host_OpenAudioOutput(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        char  *pszAudioStreamID, // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE (here: mandatory)
        char *pszDescription,    // [in] a descriptive name which identifies
           // the audio source ("audio producer"). This name may appear
           // on a kind of "patch panel" on the configuration screen of the Audio-I/O DLL .
           // For example, Spectrum Lab will identify itself as "SpectrumLab1"
           // for the first running instance, "SpectrumLab2" for the 2nd, etc .
        double dblSampleRate,    // [in] samples per second, "precise" if possible
        INT32 iNumChannels,      // [in] 1=mono, 2=stereo, 4=quadrophonic
        INT32 iTimeout_ms,       // [in] max timeout in milliseconds for THIS call
        INT32 iOpenOptions);     // [in] bit combination of AIO_OPEN_OPTION_... flags
  // Called from the external audio source (writer) to open the DLL for "output"
  //   ("output" must seen from that application's point of view) .
  // If this was successfull, it may begin to periodically call
  // AIO_WriteAudioOutput() to send chunks of audio blocks .
  // (*) pszMyName should be a descriptive name
  //     which identifies the audio source. This name may appear
  //     on a kind of "patch panel" in a future version of Audio-I/O .
  //     For example, Spectrum Lab will identify itself as "SpectrumLab1"
  //     for the first running instance, "SpectrumLab2" for the 2nd, etc .
  //
  // Return value:
  //     >= 0 on success,  in this case it's a "handle" value
  //                        which must be passed to AIO_WriteAudioOutput later.
  //      < 0 on failure (like Winamp's output-plugins) .
  //
  // Note: Having successfully opened the output for "writing"
  //  doesn't mean the destination (like Winamp) is already prepared
  //  so send anything (especially not when streaming MP3 to the web, etc).
  //  That's why we need to call AIO_CanWriteOutput() sometimes .
  // Details, especially about the calling sequence
  //           of AIO_Host_OpenAudioOutput() and AIO_Host_OpenAudioInput() :
  //          ONLY in c:\cbproj\AudioIO\aio2winamp.c::AIO_OpenAudioOutput() !


//----------------------------------------------------------------------
CPROT INT32 AIO_Host_CanWriteOutput(
       T_AIO_DLLHostInstanceData *pInstData); // [in,out] DLL-host instance data
  // Returns number of bytes(!) possible to write at a given time.
  // Never will decrease unless you call Write (or Close)..
  //       like in Winamp's output plugin .


#if(0) // Removed 2011-07; use AIO_Host_WriteOutputSamplePoints() instead:
//----------------------------------------------------------------------
CPROT INT32 AIO_Host_WriteAudioOutput(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        BYTE *pbSource,          // [in] source buffer (*)
        INT32 iNumBytesToWrite,  // [in] number of BYTES(!) to write
        INT32 iTimeout_ms );     // [in] max timeout in milliseconds, or AIO_TIMEOUT_NON_BLOCKING
  // Called from external audio source to write data into this plugin's
  // internal audio queue .  Outdated .  Details in AudioIO.c .
#endif

//----------------------------------------------------------------------
CPROT INT32 AIO_Host_WriteOutputSamplePoints(
        T_AIO_DLLHostInstanceData *pInstData, // [in] DLL-host instance data
        float *pfltSource,        // [in] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points",
                                  //      USUALLY (**) normalized to -1.0 ... +1.0 for full ADC swing
        INT32 iNumSamplePoints,   // [in] number of SAMPLE POINTS(!) to write or send
        INT32 nChannelsPerSample, // [in] number of samples PER SAMPLE POINT
        INT32 iTimeout_ms ,       // [in] max timeout in milliseconds, or AIO_TIMEOUT_NON_BLOCKING
        T_ChunkInfo *pInChunkInfo, // [in,optional (*)] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h
        INT32 *piHaveWaited_ms);  // [out,optional] "have been waiting here for XX milliseconds"
  // Return : >= 0 on success (the number indicates HOW MANY SAMPLES can be
  //                           placed in the buffer if we'd immediately
  //                           call AIO_WriteAudioOutput() again ) .
  //          < 0  : one of the NEGATIVE error codes defined in AudioIO.H .
  // Unlike the older (and now discarded) 'audio output' functions, this one
  //  supports the 'T_ChunkInfo' struct defined in c:\cbproj\SoundUtl\ChunkInfo.h,
  //  including timestamp, calibrated sample rate, GPS date+time+position, etc.
  //  It also supports multi-channel audio (grouped with MULTIPLE CHANNELS per "sample point") .
  // (*) Passing in the 'T_ChunkInfo' may be optional,
  //     i.e. pInChunkInfo MAY be NULL, but it's highly recommended
  //     that EACH audio-writer provides at least a 'minimum' T_ChunkInfo
  //     as described in ..\SoundUtl\ChunkInfo.h
  //     (dwValidityFlags=0, dblSampleValueRange=1.0)
  // (**) If the input is NOT normalized to -1.0 ... +1.0 ,
  //     the 'T_ChunkInfo' structure must be passed in,
  //     with pInChunkInfo->dblSampleValueRange set to the sample value range.
  // All details in AudioIO.c (AudioIO.h contains only an excerpt) .

//----------------------------------------------------------------------
CPROT void AIO_Host_CloseAudioOutput(
       T_AIO_DLLHostInstanceData *pInstData); // [in,out] DLL-host instance data
  // Polite applications call this function when they have nothing more to send,
  // or when they shut down.


//----------------------------------------------------------------------
CPROT INT32 AIO_Host_OpenAudioInput(  // Includes "Start()" !!
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        char *pszAudioStreamID,   // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE
           // (here: optional, use NULL to tap "the first available source")
        char *pszAudioReaderID,   // [in] short but unique name which identifies
           // the audio reader. This name may appear on the "control panel" / "patch panel"
           // of certain Audio-I/O DLLs .
           // For example, Spectrum Lab will identify itself as "SpectrumLab1"
           // for the first running instance, "SpectrumLab2" for the 2nd, etc .
        double  dblWantedSampleRate,  // [in(!)] "wanted" sample rate, often IGNORED (reasons below..)
           // Usually dictated by the FEEDING SOURCE;
           //   the sampling rate is NOT resampled by the DLL !
           //   dblWantedSampleRate can be used, however, for hardware drivers
           //   (like soundcard or SDR drivers) if the hardware supports
           //   different 'native ' sample rates.
        double *pdblOutSampleRate,    // [out(!)] realized sample rate,
           // usually dictated by the FEEDING SOURCE;
           // the sampling rate is NOT resampled by the DLL !
        INT32 nChannelsPerSample,     // [in] number of channels PER SAMPLE POINT
        INT32 iTimeout_ms ,           // [in] max timeout in milliseconds for THIS call
           // (may have to wait for a short time, for example when...
           //   - the I/O lib needs to communicate with an external device (to open it)
           //   - the I/O lib needs to wait until the 'writer' (=the other side)
           //     has called AIO_Host_OpenAudioOutput() and set the sampling rate)
        INT32 iOpenOptions,           // [in] bit combination of AIO_OPEN_OPTION_... flags
        T_ChunkInfo *pOutChunkInfo);  // [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL (!)
  //
  // Return value: 'HANLDE' for further calls (stored in T_AIO_DLLHostInstanceData).
  //     >= 0 on success,  in this case it's a "handle" value
  //             which must be passed to AIO_ReadInputSamplePoints later.
  //      < 0 on failure (like Winamp's output-plugins) .
  // Details, especially about the calling sequence
  //           of AIO_Host_OpenAudioOutput() and AIO_Host_OpenAudioInput() :
  //          ONLY in c:\cbproj\AudioIO\aio2winamp.c::AIO_OpenAudioOutput() !


//----------------------------------------------------------------------
CPROT BOOL AIO_Host_Start(
        T_AIO_DLLHostInstanceData *pInstData); // [in,out] DLL-host instance data
  // Only required for the ExtIO-DLLs. Nothing else really needs this.

//----------------------------------------------------------------------
CPROT BOOL AIO_Host_Stop(
        T_AIO_DLLHostInstanceData *pInstData); // [in,out] DLL-host instance data
  // Only required for the ExtIO-DLLs. Nothing else really needs this.


//----------------------------------------------------------------------
CPROT INT32 AIO_Host_SetInputCallback( // sets the address of an optional callback for INPUT
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data, including the internal 'handle', etc
        void  *pvUserDataForCallback,         // [in] address of some user data passed to the callback
        tpCallbackFromDLL pCallbackFunction); // [in] address of a callback function

//----------------------------------------------------------------------
CPROT int AIO_Host_ReadInputSamplePoints( // returns the number of sample-POINTS, or a negative error code
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data, including the internal 'handle', etc
        float *pfltDest,          // [out] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
        int   iNumSamplePoints,   // [in] maximum number of SAMPLE POINTS(!) to read
        int   nChannelsPerSample, // [in] number of channels PER SAMPLE POINT (may be less than provided by source)
        int   iTimeout_ms ,       // [in] max timeout in milliseconds, or AIO_TIMEOUT_NON_BLOCKING
        T_ChunkInfo *pOutChunkInfo, // [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL (!)
        int   *piHaveWaited_ms);  // [out,optional] "have been waiting here for XX milliseconds"
  // Returns the NUMBER OF SAMPLE POINTS acually read,
  //         and placed in the caller's buffer .
  // Details / specification in \cbproj\AudioIO\AudioIO.c !

//----------------------------------------------------------------------
CPROT void AIO_Host_CloseAudioInput(
       T_AIO_DLLHostInstanceData *pInstData); // [in,out] DLL-host instance data
  // Polite applications call this function before they terminate !


//----------------------------------------------------------------------
CPROT INT32 AIO_Host_ShowAboutBox(
       T_AIO_DLLHostInstanceData *pInstData, // [in] DLL-host instance data
       HWND   hwndParent );   // [in] handle to the host's window
  // Shows the DLL's "about" box as a modal (blocking) dialog.
  // Certain special DLLs may provide some additonal CONFIGURATION here.
  // Return value should be AIO_NO_ERROR if the box was closed with the "OK" button,
  // or AIO_ERROR_CANCELLED when an optional "Cancel"-button was clicked .

//----------------------------------------------------------------------
CPROT BOOL  AIO_Host_HasControlPanel( // -> TRUE=yes, FALSE=no
       T_AIO_DLLHostInstanceData *pInstData); // [in] DLL-host instance data

//----------------------------------------------------------------------
CPROT BOOL  AIO_Host_ControlPanelVisible( // -> TRUE=yes, FALSE=no
       T_AIO_DLLHostInstanceData *pInstData); // [in] DLL-host instance data

//----------------------------------------------------------------------
CPROT INT32 AIO_Host_ShowControlPanel( // NON-MODAL (not blocking) !
       T_AIO_DLLHostInstanceData *pInstData, // [in] DLL-host instance data
       HWND   hwndParent,            // [in] handle to the host's window
       HWND   *phwndControlPanel,    // [out] handle of the DLL's control panel
       INT32  iDisplayOptions   ,    // AIO_DISPLAY_SHOW_WINDOW, AIO_DISPLAY_HIDE_WINDOW, etc
       void  *pvUserDataForCallback,         // [in,optional] address of some user data passed to the callback
       tpCallbackFromDLL pCallbackFunction); // [in,optional] address of a callback function, may be NULL
  // Shows the DLL's own "control panel" box as a non-modal (= non-blocking) window.
  // Certain special DLLs may provide some additonal CONFIGURATION here.
  // Return value: AIO_NO_ERROR, if the control panel could successfully be opened.
  // Notes:
  //   -  Don't try to open the control panel multiple times.
  //   -  If the Audio-DLL-Host actually hosts a Winrad-compatible ExtIO-DLL,
  //      there is no handle to the DLL's own control panel,
  //      so unfortunately, so CHECK *phwndControlPanel - it may be NULL !

#endif // ndef  AUDIO_IO_H

/* EOF <AudioIO.H >.  Leave an empty line after this for certain compilers ! */


