//----------------------------------------------------------------------
// File:  C:\cbproj\AudioIO\AudioIO.c (original 'master' file, copies below)
// Author: Wolfgang Buescher (DL4YHF)
// Date:   2014-02-22
// Purpose: Interface / DLL HOST(!) for the "Audio-I/O" system (audio exchange) .
//     Dynamically loads an AudioIO-PLUGIN-DLL, for example in_AudioIO.dll,
//      compiled from  \cbproj\AudioIO\aio2winamp.c   .
//      Later modified to support Winrad-compatible ExtIO-DLLs as well .
//     The DLL itself contains a circular audio buffer IN SHARED MEMORY .
//     Winamp can also load the same DLL (from the same location !!),
//        and other programs can use the same DLL at the same time
//        (if they support AudioIO, or can use the Winamp plugin interface) .
//     So both applications ("processes") can exchange audio streams
//      without the need for multiple soundcards, virtual audio cables,
//      shared files, windows messages, etc .
//
//  AudioIO.C is just an interface between the AudioIO-DLL and the application.
//     It dynamically loads the DLL into memory on request, to avoid the issues
//     with compiler-specific name mangling, the need for import libs, etc .
//
//  For more information, see ?\AudioIO\manual\AudioIO_Manual.pdf
//                   or visit www.qsl.net/d/dl4yhf/AudioIO/index.html .
//
//  ExtIO-specific notes are in  C:\cbproj\ExtIO\ExtIO_Notes_DL4YHF.htm
//     ( describes using an ExtIO-DLL from the *host's* point of view,
//       i.e. for applications which USE such DLLs rather than IMPLEMENT one )
//
// Copies (besides the 'master file', see above):
//    C:\cbproj\WSQ2\AudioIO\AudioIO.c (possibly copied there via BATCH FILE)
//
// Used in:
//    -  The Audio-I/O "Test Client" (to simulate multiple readers)
//    -  Spectrum Lab (actually the first application acting as a SOURCE)
//    -  WSQ2 (Weak Signal QSO mode by ZL2AFP, adapted by DL4YHF)
//
// Notes:
//    Don't confuse the AudioIO system with a simple Winamp-plugin .
//    The initial plan (in 2008-03) was to use AudioIO as a simple(!)
//    way to chain several audio-processing applications; winamp being only
//    one of them.
//    Somehow related to this subject is the project "Winamp plugin host",
//    see \cbproj\WinampPluginHost\*.cpp  .  But the latter is, as the name
//    says, just a simple(?) HOST for a few of winamp's plugins.
//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Copyright, License, and Disclaimer (for the AudioIO-skeleton)
//      Copyright (c) 2008...2065, Wolfgang Bscher (DL4YHF)
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are met:
//   1. Redistributions of source code must retain the above copyright notice,
//      this list of conditions and the following disclaimer.
//   2. Redistributions in binary form must reproduce the above copyright notice,
//      this list of conditions and the following disclaimer in the
//      documentation and/or other materials provided with the distribution.
//   3. Neither the name of the author nor the name of other contributors
//      may be used to endorse or promote products derived from this software
//      without specific prior written permission.
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
//  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
//   OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
//   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//   LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
//
//
// Revision history:
//  V0.03, 2014-02-22:  Integrated Audio-I/O in WSQ2, see \cbproj\WSQ2\*.* .
//                      Eliminates the need for VAC when Spectrum Lab is used
//                      as preprocessor for ZL2AFP's Weak-Signal QSO mode.
//
//  V0.02, 2012-08-08:  Added support for I2PHD's "ExtIO"-DLLs, which are not
//                      really compatible with DL4YHF's "Audio-I/O"-DLL-API,
//                      but this DLL-HOST (AudioIO.C) will map the necessary
//                      calls to the ExtIO-DLL functions, if an ExtIO-DLL
//                      was loaded instead of an Audio-I/O-DLL .
//
//  V0.01, 2011-06-13:  Re-activated the "Audio I/O DLL" project.
//                      Put all statics inside a structure,
//                      to allow multiple instances used in ONE application.
//                Plan: Use this DLL also as an input for SPECTRUM LAB,
//                      more or less as a single-input / multiple-output node:
//
//        ___________         ____________        __________
//       |           |       |            |      |          | ---> Winamp,ASIO,..
//       | Soundcard | --->  | SpecLab #1 | ---> | Audio-IO | ---> SpecLab #2
//       |___________|       |____________|      |__________| ---> SpecLab #3
//        ______   /|\        (with SR-        .               .
//       |      |   |          calibrator      .               .
//       | GPS  |->-|          + resampler)    .               .
//       |______|                              .               .
//                                           One Writer     Multiple Readers
//
//  V0.00, 2008-03-27:  Project started as a WINAMP-"input"-plugin
//                      to send audio data from DL4YHF's "Spectrum Lab"
//                      into Winamp, to play with Icecast/Shoutcast streaming .
//
//----------------------------------------------------------------------------

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !
                       // If Borland C++ is too stupid to open this file
                       // in the editor (via context menu), it may be here:
                       // \CBproj\SpecLab\SWI_V01\SWITCHES.H
#include <windows.h>
#include "AudioIO.h"   // error codes, data types, and prototypes for AudioIO
#ifdef UseUtility1      // Availability of DL4YHF's "Utility One" may be defined in the project settings..
# include "Utility1.h"  // Contains DEBUGGER_BREAK() etc. Original file in C:\cbproj\SoundUtl\utility1.h .
#else  // if DL4YHF's "Utility One" is NOT available, use the following replacements:
# define UTL_NamedMalloc(name,size) malloc(size)
# define UTL_free(block)            free(block)
# define UTL_WriteRunLogEntry(format_string,x)  /*dummy,problematic*/
#endif // UseUtility1 ?

#if( SWI_SUPPORT_EXTIO_DLL )   // support Winrad-compatible ExtIO-DLLs ?
static T_AIO_DLLHostInstanceData *ExtIO_pInstData = NULL; // address of instance data for ExtIO-callback
  // (this ugly variable is the reason for only being able to load
  //  ONE instance of an ExtIO-DLL at any given time)
#endif // SWI_SUPPORT_EXTIO_DLL ?

//----------------------------------------------------------------------
// Debugging stuff
//----------------------------------------------------------------------
int AudioIO_iSourceCodeLine = 0;
#define DOBINI() AudioIO_iSourceCodeLine=__LINE__
  // Allows to check the last 'known' sourcecode line with the BCB debugger,
  // even if the program has crashed somewhere else (and won't single-step).

//----------------------------------------------------------------------
// Methods for a simple, lock-free, sample buffer T_AIO_SampleBuffer .
// Note that this buffer type is only used for the interface
//    between the DLL HOST and the application (Spectrum Lab),
//    but not (necessarily) in the DLL itself.
//    It also doesn't carry the important 'Audio-Chunk-Info' along
//    (which is no big deal because the ExtIO-DLLs, for which this
//     buffer was implemented in SL, doesn't support audio-chunk-infos at all)
//  Implementers of an Audio-I/O-DLL are free to decide any buffering method.
//----------------------------------------------------------------------

//---------------------------------------------------------------------------
BOOL AIO_SampleBuffer_IsValid( T_AIO_SampleBuffer *pBuf )
{
  return (pBuf->i32MagicP == 0x31415926) && (pBuf->i32MagicE==0x27182818)
      && (pBuf->nChannelsPerSample>=1)   && (pBuf->nChannelsPerSample<=8)
      && (pBuf->i32AllocatedSize_nSamples>0) && (pBuf->pfltSamples != NULL);
}

//---------------------------------------------------------------------------
BOOL AIO_SampleBuffer_Init( T_AIO_SampleBuffer *pBuf,
        long  nSamplePointsPerBuffer, // maximum required buffer size as "number of sample points"
        int   nChannelsPerSample )   // aka number of channels per "sample point"
{
  long i,nFloatsToAllocate;
  if( (pBuf==NULL) || (nChannelsPerSample<1) || (nChannelsPerSample>8)
     || (nSamplePointsPerBuffer<16) || (nSamplePointsPerBuffer>20000000) )
   { // Caller passed in 'garbage'; refuse to do anything
     return FALSE;
   }

  // Has this buffer been initialized before, and do we need to free it (avoid memory leaks) ?
  if( AIO_SampleBuffer_IsValid( pBuf ) )
   { if( (pBuf->nChannelsPerSample==nChannelsPerSample)
      && (pBuf->i32AllocatedSize_nSamples==nSamplePointsPerBuffer) )
      { // No need to free and re-allocate this buffer; just clean it:
        pBuf->i32HeadIndex = pBuf->i32TailIndex = 0;
        return TRUE;
      }
     AIO_SampleBuffer_Delete( pBuf );  // otherwise delete the old instance
   }

  // Make this buffer INvalid by clearing the magic numbers. May have to bail out.
  pBuf->i32MagicP = pBuf->i32MagicE = 0;
  nFloatsToAllocate = (long)nSamplePointsPerBuffer * (long)nChannelsPerSample;
  if(  (nFloatsToAllocate <= 0)  // oops... range/sign overflow, or whatever..
    || (nFloatsToAllocate > 20000000L ) ) // more than 20 Mega-Samples ? Never !
   { return FALSE;   // bail out; and leave the buffer "invalid"
   }
  pBuf->pfltSamples = UTL_NamedMalloc( "AIObuf", nFloatsToAllocate * sizeof(float) );
  if( pBuf->pfltSamples==NULL )
   { return FALSE;   // bail out; the system seems to run out of memory
   }
  // Don't leave anything to fate; clear the buffer.
  //  We only want to have valid floating point numbers in there, even when empty.
  for( i=0; i<nFloatsToAllocate; ++i)
   { pBuf->pfltSamples[i] = 0.0;
   }
  pBuf->nChannelsPerSample = nChannelsPerSample;
  pBuf->i32AllocatedSize_nSamples = nSamplePointsPerBuffer;
  pBuf->i32HeadIndex = pBuf->i32TailIndex = 0;  // buffer is now EMPTY
  // Open for business: Make this buffer valid by setting those magic numbers:
  pBuf->i32MagicP=0x31415926;
  pBuf->i32MagicE=0x27182818;
  return TRUE;
} // end AIO_SampleBuffer_Init()

//---------------------------------------------------------------------------
void AIO_SampleBuffer_Delete( T_AIO_SampleBuffer *pBuf )
  // Important to free up resources, when a buffer isn't used anymore
{
  if( AIO_SampleBuffer_IsValid( pBuf ) )
   { if( pBuf->i32AllocatedSize_nSamples > 0 )
      {  pBuf->i32AllocatedSize_nSamples = 0;
         Sleep(100); // give other threads the chance to STOP USING THIS BUFFER
         UTL_free( pBuf->pfltSamples );
         pBuf->pfltSamples = NULL;
      }
   }
  pBuf->i32MagicP = pBuf->i32MagicE = 0;
} // end AIO_SampleBuffer_Delete()

//---------------------------------------------------------------------------
long AIO_SampleBuffer_GetNumSamplesBuffered( T_AIO_SampleBuffer *pBuf )
{
  long n = 0;
  // Note that this classic circular FIFO is EMPTY when head==tail index !
  // The maximum capacity is ONE SAMPLE LESS THAN ALLOCATED ! (go figure)

  if( (pBuf->i32MagicP == 0x31415926) && (pBuf->i32MagicE==0x27182818) )
   { n = pBuf->i32HeadIndex - pBuf->i32TailIndex;
     if( n<0 ) n += pBuf->i32AllocatedSize_nSamples; // circular wrap
   }
  return n;
} // end AIO_SampleBuffer_GetNumSamplesBuffered()

//---------------------------------------------------------------------------
long AIO_SampleBuffer_GetNumSamplesFree( T_AIO_SampleBuffer *pBuf )
  // Retrieves this buffer's momentary unused ("free") capacity,
  //  expressed as NUMBER OF SAMPLE POINTS (with X channels per point) .
  //  Used by some callers to decide if a given number of samples
  //  fits into the buffer, before stuffing new samples in .
{
  long n = 0;
  if( (pBuf->i32MagicP == 0x31415926) && (pBuf->i32MagicE==0x27182818) )
   { n = pBuf->i32HeadIndex - pBuf->i32TailIndex;
     if( n<0 ) n += pBuf->i32AllocatedSize_nSamples; // circular wrap
     // At this point, 'n' holds the number of USED entries .
     // This classic circular FIFO is EMPTY when head==tail index !
     // The maximum capacity is ONE SAMPLE LESS THAN ALLOCATED ! Thus:
     n = pBuf->i32AllocatedSize_nSamples - 1 - n;
   }
  return n;
} // end AIO_SampleBuffer_GetNumSamplesFree()


//---------------------------------------------------------------------------
BOOL AIO_SampleBuffer_WriteSamples_Float( T_AIO_SampleBuffer *pBuf,
        float *pfltSource,        // [in] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
        int   iNumSamplePoints,   // [in] number of SAMPLE POINTS(!) to read
        int   nChannelsPerSample) // [in] number of channels PER SAMPLE POINT (may be less than "initialized")
  // Example: If the source contains 512 sample points with 2 channels each,
  //          pfltSrc must point to an array of single-precision FLOATING POINT
  //          values with at least 1024 (!) single floats !
{
  float *pfltDst;
  int   n, iChannel, nCommonChannels;
  long  i32HeadIndex, i32BufSize;
  n = AIO_SampleBuffer_GetNumSamplesFree( pBuf );
  if ( iNumSamplePoints > n )
   { return FALSE;  // buffer not valid, or too few free entries remaining
   }
  i32HeadIndex = pBuf->i32HeadIndex;  // use local copy for speed...
  i32BufSize   = pBuf->i32AllocatedSize_nSamples;
  // the unit of the two values above are SAMPLE POINTS, with N channels per POINT !
  nCommonChannels = pBuf->nChannelsPerSample;
  if( nCommonChannels > nChannelsPerSample )
   {  nCommonChannels = nChannelsPerSample;
   }
  if( i32HeadIndex >= i32BufSize )
   {  i32HeadIndex = 0;   // circular index wrap
   }
  pfltDst = &pBuf->pfltSamples[ i32HeadIndex * pBuf->nChannelsPerSample ];
  if( nChannelsPerSample==2 && pBuf->nChannelsPerSample==2 )
   { // optimized function to enter I/Q data, or "stereo", in both SOURCE + DEST:
     while( iNumSamplePoints-- )
      { *pfltDst++ = *pfltSource++;
        *pfltDst++ = *pfltSource++;
        ++i32HeadIndex;  // an index of SAMPLE POINTS, with N channels per point !
        if( i32HeadIndex >= i32BufSize )
         {  i32HeadIndex = 0;    // next circular index wrap
            pfltDst = pBuf->pfltSamples; // no multiplication because index=0
         }
      }
   }
  else // universal, but slighty slower :
   {
     while( iNumSamplePoints-- )
      { iChannel = 0;
        while( iChannel < nCommonChannels )
         { pfltDst[iChannel] = pfltSource[iChannel];
           ++iChannel;
         }
        pfltSource += nChannelsPerSample;    // number of channels per sample in the SOURCE (caller's non-circular buffer)
        pfltDst += pBuf->nChannelsPerSample; // number of channels per sample in the DESTINATION
        ++i32HeadIndex;  // an index of SAMPLE POINTS, with N channels per point !
        if( i32HeadIndex >= i32BufSize )
         {  i32HeadIndex = 0;
            pfltDst = pBuf->pfltSamples;
         }
      }
   }

  // Updating pBuf->i32HeadIndex actually makes the new data available for the reader:
  pBuf->i32HeadIndex = i32HeadIndex;  // ... put datt back where it belongs :)
  return TRUE;
} // end AIO_SampleBuffer_WriteSamples_Float()

//---------------------------------------------------------------------------
BOOL AIO_SampleBuffer_WriteSamples_Int16( T_AIO_SampleBuffer *pBuf,
        INT16 *pi16Source,        // [in] audio samples, as 16-bit integers, grouped as "sample points"
        int   iNumSamplePoints,   // [in] number of SAMPLE POINTS(!) to read
        int   nChannelsPerSample) // [in] number of channels PER SAMPLE POINT (may be less than "initialized")
  // Places <iNumSamplePoints> with <nChannelsPerSample> values per point
  //        in a FIFO. Called from a DIFFERENT THREAD than the sample reader !
  // Example: If the source contains 512 sample points with 2 channels each,
  //          pfltSrc must point to an array of 16-bit signed integers
  //          values with at least 1024 (!) single numbers, or 2048 bytes !
{
  float *pfltDst;   // the destination is always a FLOATING POINT buffer !
  int   n, iChannel, nCommonChannels;
  long  i32HeadIndex, i32BufSize;
  n = AIO_SampleBuffer_GetNumSamplesFree( pBuf );
  if ( iNumSamplePoints > n )
   { return FALSE;  // buffer not valid, or too few free entries remaining
   }
  i32HeadIndex = pBuf->i32HeadIndex;  // use local copy for speed...
  i32BufSize   = pBuf->i32AllocatedSize_nSamples;
  nCommonChannels = pBuf->nChannelsPerSample;
  if( nCommonChannels > nChannelsPerSample )
   {  nCommonChannels = nChannelsPerSample;
   }
  if( i32HeadIndex >= i32BufSize )
   {  i32HeadIndex = 0;   // circular index wrap
   }
  pfltDst = &pBuf->pfltSamples[ i32HeadIndex * pBuf->nChannelsPerSample ];
  while( iNumSamplePoints-- )
   { iChannel = 0;
     while( iChannel < nCommonChannels )
      { // Note: Spectrum Lab expects floats normalized to +/- 1.0  !
        pfltDst[iChannel] = (float)pi16Source[iChannel] * (1.0/32768.0);
        ++iChannel;
      }
     pi16Source += nChannelsPerSample;    // number of channels per sample in the SOURCE (caller's non-circular buffer)
     pfltDst += pBuf->nChannelsPerSample; // number of channels per sample in the DESTINATION  (circular buffer)
     ++i32HeadIndex;  // an index of SAMPLE POINTS, with N channels per point !
     if( i32HeadIndex >= i32BufSize )     // circular index wrap...
      {  i32HeadIndex = 0;
         pfltDst = pBuf->pfltSamples;
      }
   } // end while < more samples >

  // Updating pBuf->i32HeadIndex actually makes the new data available for the reader:
  pBuf->i32HeadIndex = i32HeadIndex;  // ... put datt back where it belongs :)
  return TRUE;
} // end AIO_SampleBuffer_WriteSamples_Int16()

//---------------------------------------------------------------------------
BOOL AIO_SampleBuffer_WriteSamples_Int24( T_AIO_SampleBuffer *pBuf,
        BYTE  *pbSource,          // [in] audio samples, as 24-bit integers, grouped as "sample points"
        int   iNumSamplePoints,   // [in] number of SAMPLE POINTS(!) to read
        int   nChannelsPerSample) // [in] number of channels PER SAMPLE POINT (may be less than "initialized")
  // Example: If the source contains 512 sample points with 2 channels each,
  //          pfltSrc must point to an array of 16-bit signed integers
  //          values with at least 1024 (!) single numbers, or 2048 bytes !
{
  float *pfltDst;   // the destination is always a FLOATING POINT buffer !
  union { BYTE b[4]; long i32; } temp;  // union for 24-bit trickery
  int   n, iChannel, nCommonChannels, nSkippedBytesPerSample;
  long  i32HeadIndex, i32BufSize;
  n = AIO_SampleBuffer_GetNumSamplesFree( pBuf );
  if ( iNumSamplePoints > n )
   { return FALSE;  // buffer not valid, or too few free entries remaining
   }
  i32HeadIndex = pBuf->i32HeadIndex;  // use local copy for speed...
  i32BufSize   = pBuf->i32AllocatedSize_nSamples;
  nCommonChannels = pBuf->nChannelsPerSample;
  if( nCommonChannels > nChannelsPerSample )
   {  nCommonChannels = nChannelsPerSample;
   }
  nSkippedBytesPerSample = 3 * (pBuf->nChannelsPerSample - nChannelsPerSample);
  if( i32HeadIndex >= i32BufSize )
   {  i32HeadIndex = 0;   // circular index wrap
   }
  pfltDst = &pBuf->pfltSamples[ i32HeadIndex * pBuf->nChannelsPerSample ];
  while( iNumSamplePoints-- )
   { iChannel = 0;
     while( iChannel < nCommonChannels )
      {
        // As usual on Intel CPUs, the least significant byte is first:
        temp.b[0] = *pbSource++;   // bits 0..7
        temp.b[1] = *pbSource++;   // bits 8..15
        temp.b[2] = *pbSource++;   // bits 16..23 (bit 23 = sign)
        if( temp.b[2] & 0x80 )     // expand the sign into bits 31..24 :
         { temp.b[3] = 0xFF;   // now temp.i32 is NEGATIVE, range -1.. - 2^23
         }
        else
         { temp.b[3] = 0x00;   // now temp.i32 is non-negative, range 0..2^23-1
         }
        // Note: Spectrum Lab expects floats normalized to +/- 1.0  :
        pfltDst[iChannel] = (float)temp.i32 * (1.0 / 8388608.0/*2^23*/ );
        ++iChannel;
      }
     pbSource += nSkippedBytesPerSample;
     pfltDst += pBuf->nChannelsPerSample; // number of channels per sample in the DESTINATION
     ++i32HeadIndex;  // an index of SAMPLE POINTS, with N channels per point !
     if( i32HeadIndex >= i32BufSize )
      {  i32HeadIndex = 0;
         pfltDst = pBuf->pfltSamples;
      }
   } // end while < more samples >

  // Updating pBuf->i32HeadIndex actually makes the new data available for the reader:
  pBuf->i32HeadIndex = i32HeadIndex;  // ... put datt back where it belongs :)
  return TRUE;
} // end AIO_SampleBuffer_WriteSamples_Int24()

//---------------------------------------------------------------------------
BOOL AIO_SampleBuffer_WriteSamples_Int32( T_AIO_SampleBuffer *pBuf,
        INT32 *pi32Source,        // [in] audio samples, as 32-bit integers, grouped as "sample points"
        int   iNumSamplePoints,   // [in] number of SAMPLE POINTS(!) to read
        int   nChannelsPerSample) // [in] number of channels PER SAMPLE POINT (may be less than "initialized")
  // Example: If the source contains 512 sample points with 2 channels each,
  //          pfltSrc must point to an array of 32-bit signed integers
  //          values with at least 1024 (!) single numbers, or 4096 bytes !
{
  float *pfltDst;   // the destination is always a FLOATING POINT buffer !
  int   n, iChannel, nCommonChannels;
  long  i32HeadIndex, i32BufSize;
  n = AIO_SampleBuffer_GetNumSamplesFree( pBuf );
  if ( iNumSamplePoints > n )
   { return FALSE;  // buffer not valid, or too few free entries remaining
   }
  i32HeadIndex = pBuf->i32HeadIndex;  // use local copy for speed...
  i32BufSize   = pBuf->i32AllocatedSize_nSamples;
  nCommonChannels = pBuf->nChannelsPerSample;
  if( nCommonChannels > nChannelsPerSample )
   {  nCommonChannels = nChannelsPerSample;
   }
  if( i32HeadIndex >= i32BufSize )
   {  i32HeadIndex = 0;   // circular index wrap
   }
  pfltDst = &pBuf->pfltSamples[ i32HeadIndex * pBuf->nChannelsPerSample ];
  while( iNumSamplePoints-- )
   { iChannel = 0;
     while( iChannel < nCommonChannels )
      { // Note: Spectrum Lab expects floats normalized to +/- 1.0  !
        pfltDst[iChannel] = (float)pi32Source[iChannel] * (1.0/2147483648.0);
        ++iChannel;
      }
     pi32Source += nChannelsPerSample;    // number of channels per sample in the SOURCE (caller's non-circular buffer)
     pfltDst += pBuf->nChannelsPerSample; // number of channels per sample in the DESTINATION
     ++i32HeadIndex;  // an index of SAMPLE POINTS, with N channels per point !
     if( i32HeadIndex >= i32BufSize )
      {  i32HeadIndex = 0;
         pfltDst = pBuf->pfltSamples;
      }
   } // end while < more samples >

  // Updating pBuf->i32HeadIndex actually makes the new data available for the reader:
  pBuf->i32HeadIndex = i32HeadIndex;  // ... put datt back where it belongs :)
  return TRUE;
} // end AIO_SampleBuffer_WriteSamples_Int32()

//---------------------------------------------------------------------------
long AIO_SampleBuffer_ReadSamples( T_AIO_SampleBuffer *pBuf,
        float *pfltDest,          // [out] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
        int   iMaxSamplePoints,   // [in] MAXIMUM(!) number of SAMPLE POINTS to read
        int   nChannelsPerSample) // [in] number of channels PER SAMPLE POINT .
                                  //      Usually TWO for I/Q pairs in ExtIO .
  // Reads the next chunk of data from the buffer, removing it from there.
  //       The destination are always FLOATING POINT NUMBERS (for Spectrum Lab).
  // Example: If the caller wants to read 512 sample points with 2 channels each,
  //          pfltDest must point to an array of single-precision FLOATING POINT
  //          values with at least 1024 (!) single floats !
  // Return value: number of sample-points (usually, I/Q pairs) actually read.
  //          If caller needs EXACTLY a certain number of samples (but not less),
  //          he'd call AIO_SampleBuffer_GetNumSamplesBuffered() before .
{
  float *pfltSrc;
  int   n, iChannel, nCommonChannels;
  long  i32TailIndex, i32BufSize;
  n = AIO_SampleBuffer_GetNumSamplesBuffered( pBuf );
  if( iMaxSamplePoints > n )
   {  iMaxSamplePoints = n;
   }
  if( (iMaxSamplePoints<=0) || (nChannelsPerSample<=0) )
   { return 0;  // buffer not valid, or nothing in there at all, or garbage-in
   }
  i32TailIndex = pBuf->i32TailIndex;  // use local copy for speed...
  i32BufSize   = pBuf->i32AllocatedSize_nSamples;
  // Note that the units for buffer indices and sizes are SAMPLE POINTS,
  //      not "single floating point values" !
  nCommonChannels = pBuf->nChannelsPerSample;
  if( nCommonChannels > nChannelsPerSample )
   {  nCommonChannels = nChannelsPerSample;
   }
  if( i32TailIndex >= i32BufSize )
   {  i32TailIndex = 0;   // circular index wrap
   }
  pfltSrc = &pBuf->pfltSamples[ i32TailIndex * pBuf->nChannelsPerSample ];
  n = iMaxSamplePoints;   // already limited number of sample points (to return)
  if( nChannelsPerSample==2 && pBuf->nChannelsPerSample==2 )
   { // optimized function to enter I/Q data, or "stereo", in both SOURCE + DEST:
     while( n-- )
      { *pfltDest++ = *pfltSrc++;
        *pfltDest++ = *pfltSrc++;
        ++i32TailIndex;  // this is an index of SAMPLE POINTS, with N channels per POINT !
        if( i32TailIndex >= i32BufSize )
         {  i32TailIndex = 0;            // next circular index wrap
            pfltSrc = pBuf->pfltSamples; // no multiplication because index=0
         }
      }
   }
  else // universal, but slighty slower :
   {
     while( n-- )
      { iChannel = 0;
        while( iChannel < nCommonChannels )
         { pfltDest[iChannel] = pfltSrc[iChannel];
           ++iChannel;
         }
        pfltDest+= nChannelsPerSample;    // number of channels per sample in the DESTINATION (caller's non-circular buffer)
        pfltSrc += pBuf->nChannelsPerSample; // number of channels per sample in the source (our circular buffer)
        ++i32TailIndex; // an index of SAMPLE POINTS, with N channels per POINT !
        if( i32TailIndex >= i32BufSize )
         {  i32TailIndex = 0;
            pfltSrc = pBuf->pfltSamples; // no multiplication because index=0
         }
      }
   }

  // Updating pBuf->i32HeadIndex actually makes the new data available for the reader:
  pBuf->i32TailIndex = i32TailIndex;  // ... put datt back where it belongs :)

  return iMaxSamplePoints;
} // end AIO_SampleBuffer_ReadSamples()



//----------------------------------------------------------------------
void AIO_Host_InitInstanceData(
        T_AIO_DLLHostInstanceData *pInstData) // [in] address of DLL-host instance data
  // Must be called ONCE (and ONLY once) for each instance, before anything else.
  // Prerequisites: None .
{
  // Initialize the critical section (for thread synchronisation).
  // Not sure if this beast is really necessary, but some ExtIO-DLLs
  // reacted strange in a multi-threaded application...
#if( AIO_HOST_USE_CRITTER )
  InitializeCriticalSection( &pInstData->Critter );
  pInstData->nCritterEntryCounter = 0;
#endif
  pInstData->fCritterInitialized = TRUE; // applies to the CRITICAL SECTION (not to a "loaded DLL")

  pInstData->i32ExternalVfoFrequency = -1; // mark the VFO frequency as 'invalid'
   // (we need a VALID VFO frequency to prevent a stupid message popup from ExtIO_RTL.DLL)


  // Forget any function pointers into the DLL :
  pInstData->m_pErrorCodeToString= NULL;
  pInstData->m_pGetWinampParams = NULL;
  pInstData->m_pGetOutputParams = NULL;
  pInstData->m_pOpenAudioOutput = NULL;
  pInstData->m_pCloseAudioOutput= NULL;
 // pInstData->m_pWriteAudioOutput= NULL;
  pInstData->m_pWriteOutputSamplePoints = NULL;  // added 2011-06
  pInstData->m_pCanWriteOutput  = NULL;
  pInstData->m_pOpenAudioInput  = NULL;           // added 2011-06 ...
  pInstData->m_pCloseAudioInput = NULL;
  pInstData->m_pReadInputSamplePoints = NULL;
  pInstData->m_pCanReadInput    = NULL;


  // Other DLL-Host-Instance-related stuff :
  pInstData->h_AudioIoDLL = NULL; // handle to the DLL (NULL when not loaded)
  pInstData->iAudioInputHandle = pInstData->iAudioOutputHandle = AIO_INVALID_HANDLE; // nothing *opened* yet
  pInstData->fOpenForBusiness = FALSE; // NOT open for business, because no DLL loaded yet, but..


#if( SWI_SUPPORT_EXTIO_DLL )   // support Winrad-compatible ExtIO-DLLs ?
  // Added 2012-08-08 for the ExtIO-compatible part :
  // if( ExtIO_pInstData != NULL )  // We're in trouble because an ExtIO-DLL is already loaded !
  //  { DEBUGGER_BREAK(); // set breakpoint here, or use the 'common' breakpoint in C:\cbproj\SpecLab\DebugU1.cpp
  //  }
  // ex: ExtIO_pInstData = pInstData;  // address of instance data for ExtIO-callback (ONLY)
  // No-No ! The Audio-I/O-DLL-Host may run in two instances: One for input, one for output.
  //         ExtIO-DLLs can only be loaded in ONE INSTANCE, because there is no way
  //         to pass a pointer to the instance-data to the callback function.
  //         Thus, the ugly global/static 'ExtIO_pInstData' for the callback
  //         must only be occupied by that instance which REALLY hosts
  //         an ExtIO-DLL !
  pInstData->m_pExtIO_InitHW  = NULL;
  pInstData->m_pExtIO_OpenHW  = NULL;
  pInstData->m_pExtIO_CloseHW = NULL;
  pInstData->m_pExtIO_StartHW = NULL;
  pInstData->m_pExtIO_SetCallback = NULL;
  pInstData->m_pExtIO_SetHWLO = NULL;
  pInstData->m_pExtIO_GetHWLO = NULL;
  pInstData->m_pExtIO_GetStatus = NULL;
  pInstData->m_pExtIO_GetHWSR = NULL;
  pInstData->m_pExtIO_RawDataReady = NULL;
  pInstData->m_pExtIO_ShowGUI = NULL;
  pInstData->m_pExtIO_HideGUI = NULL;
  pInstData->m_pExtIO_TuneChanged = NULL;
  pInstData->m_pExtIO_GetTune = NULL;
  pInstData->m_pExtIO_IFLimitsChanged = NULL;
  pInstData->m_pExtIO_FiltersChanged = NULL;
  pInstData->m_pExtIO_GetFilters = NULL;
#endif // SWI_SUPPORT_EXTIO_DLL ?

} // end AIO_Host_InitInstanceData()

//----------------------------------------------------------------------
void AIO_Host_DeleteInstanceData(  // counterpart to AIO_Host_InitInstanceData()
        T_AIO_DLLHostInstanceData *pInstData)
  // In Spectrum Lab, called from
{
  if(   ( pInstData->h_AudioIoDLL != NULL )
#    if( SWI_SUPPORT_EXTIO_DLL )
      ||( ExtIO_pInstData == pInstData)  // did THIS INSTANCE host an ExtIO-DLL ?
#    endif // SWI_SUPPORT_EXTIO_DLL ?
    )
   {  AIO_Host_FreeDLL(pInstData);  // oops.. been in use before ?   Unload the "old" DLL .
   }
  pInstData->fOpenForBusiness = FALSE; // definitely NOT open for business anymore

  AIO_SampleBuffer_Delete( &pInstData->buf ); // free this buffer (if allocated)

#if( AIO_HOST_USE_CRITTER )
  if( pInstData->fCritterInitialized )
   {  pInstData->fCritterInitialized = FALSE; // don't anyone try to ENTER now !
      Sleep(10);
      DeleteCriticalSection( &pInstData->Critter );
   }
#else
  pInstData->fCritterInitialized = FALSE;
#endif // AIO_HOST_USE_CRITTER ?



} // end AIO_Host_DeleteInstanceData()

//----------------------------------------------------------------------
static INT32 WindowsErrorToAIOErrorCode( DWORD dwWindowsErrorCode,
                                         INT32 default_AIO_error )
{
  switch( dwWindowsErrorCode  )
   { // Help system on "GetLastError" :
     // > For a complete list of error codes, see the WINNT.H header file .
     // As so often, that's complete rubbish. You won't find error codes
     // like 'ERROR_FILE_NOT_FOUND' in WINNT.H . Instead, the most important
     // lower error codes are in ERROR.H, not in WINNT.H !
     case ERROR_PATH_NOT_FOUND :
     case ERROR_FILE_NOT_FOUND :  // don't expect to see THIS error code, but:
     case ERROR_MOD_NOT_FOUND  :  // error returned by LoadLibrary() when "file not found"
        return AIO_ERROR_FILE_NOT_FOUND;
     default:
        return default_AIO_error; // shit happened ... but no idea why
   }
} // end WindowsErrorToAIOErrorCode()


#if( SWI_SUPPORT_EXTIO_DLL )   // support Winrad-compatible ExtIO-DLLs ?
//------------------------------------------------------------------------------------
void ExtIO_Callback(INT32 cnt, INT32 status, float IQoffs, void *IQdata)
  // Callback function for Winrad's ExtIO-DLL .
  //    CALLING CONVENTION UNKNOWN / NOT SPECIFIED !  'CDECL' ? Details below.
  // From  c:\cbproj\ExtIO\Winrad_Extio.pdf :
  // > Invokedwhenanewbufferofaudiodataisready,orwhenan
  // > asynchronouseventmustbecommunicatedbytheDLL.Ofcourse
  // > thenewbufferofaudiodataisonlysentbyDLLsthatcontrolHW
  // > thathavetheirowninternaldigitizersanddonotdepend
  // > onthesoundcardforinput.InthiscaseitsuptotheDLL
  // > todecidewhichI/Oportisusedtoreadfrom
  // > the HW the digitized audio data stream.
  // Note: Alberto's Winrad-ExtIO documentation doesn't specify the
  //       calling convention for this callback function explicitly :
  // > ThecallbackfunctioninWinradthattheDLLisexpectedtocall,isdefinedasfollows:
  // > void extIOCallback(int cnt, int status, float IQoffs, short IQdata[] )
  //    Since Winrad was also compiled with Borland C++ Builder (32 bit),
  //    there's a good chance the callback function it WILL be compatible .
  //    Remember, this can/will cause problems with different compilers
  //    sooner or later ! Borland's calling convention for "normal C functions",
  //    when not decorated with one of those dirty dozen ugly keywords,
  //    is as follows:  < to do : find this out; the BCB documentation doesn't help >
  //
{


  // CAUTION-CAUTION-CAUTION !
  //     Don't make any assumption about THE CALLER of this function !
  //     It may be the worker thread in an ExtIO-DLL, it may be some
  //     flaky MFC-stuff (like C:\WINDOWS\system32\MFC42.DLL), or whatever !


  // Unfortunately there is no extra parameter for the address of the
  // "instance data" in this API, so we must use an ugly static variable:
  T_AIO_DLLHostInstanceData *pInstData  // [in,out] DLL-host instance data
    = ExtIO_pInstData;

  if( pInstData==NULL )
   { return;   // should never happen !
   }

  // cnt :  the number of samples returned.
  if(cnt < 0)
   {
     switch(status)
      {
        case 0 :
        case 1 :
        case 2 :
        case 3 :
        case 4 :
           break;

        case 100 : // sampling speed has changed in the external HW
           // > Thisstatusvalueindicatesthatasamplingfrequencychange
           // >hastakenplace,eitherbyahardwareaction,
           // > orbyaninteractionoftheuserwiththeDLLGUI..
           // >WhenWinradreceivesthisstatus,itcallsimmediatelyafter
           // > theGetHWSR()APItoknowthenewsamplingrate.
           // (WB: of course not from the callback, but from the main task)
           // MainForm->srchanged = true;
           pInstData->fSRChangedByDriver = TRUE;
           break;

        case 101 : // LO frequency has changed in the external HW
           // > ThisstatusvalueindicatesthatachangeoftheLOfrequency
           // > hastakenplace,eitherbyahardwareaction,
           // > orbyaninteractionoftheuserwiththeDLLGUI..
           // > WhenWinradreceivesthisstatus,itcallsimmediatelyafter
           // > theGetHWLO()APItoknowthenewLOfrequency.
           // MainForm->extLOchanged = true;
           pInstData->fVFOChangedByDriver = TRUE;
           break;

        case 102 : // DLL has blocked changing the LO frequency.
           // > This status value indicates that the DLL has
           // > temporarily blocked any change to the LOfrequency.
           // > Thismayhappen,e.g.,whentheDLLhasstartedrecording
           // > onaWAVfiletheincoming raw data. As the center frequency
           // > has been written into the WAV file header,
           // > changingitduringtherecordingwouldbeanerror.
           // MainForm->LOlocked = true;
           break;

        case 103 :  // DLL has UN-blocked changing the LO frequency.
           // indicatesthatchangestotheLOfrequencyareagainacceptedbytheDLL
           // MainForm->LOlocked = false;
           break;

        case 104 : // LO freq. has changed,  must keep the Tune freq. unchanged
           // (must immediately call GetHWLO() )
           // > Indicates that a change of the LO frequency has taken place,
           // > andthatWinradshouldactsotokeeptheTunefrequencyunchanged.
           // > WhenWinradreceivesthisstatus,itcallsimmediatelyafter
           // > theGetHWLO()APItoknowthenewLOfrequency .
           pInstData->fVFOChangedByDriver = TRUE;
           break;

        case 105 : // a change of the Tune freq. is being requested.
           // Must call GetTune() to know which value is wanted
           // MainForm->mustGetTune = true;
           break;

        case 106 : // a change of demod. mode is being requested.
           // Must call GetMode() to know the new mode
           // MainForm->mustGetMode = true;
           break;

        case 107 : // The DLL wants Winrad to Start
           // > This status value indicates that the DLL is asking Winrad
           // > to behave as if the user hadpressedtheStartbutton.
           // > IfWinradisalreadystarted,thisisequivalenttoano-op.
           pInstData->fStartRequestedByDriver = TRUE;
           break;

        case 108 : // The DLL wants Winrad to Stop
           // > This status value indicates that the DLL is asking Winrad
           // > to behave as if the user hadpressedtheStopbutton.
           pInstData->fStopRequestedByDriver = TRUE;
           break;

        case 109 : // a change in the band limits is being requested
           // > the DLL is asking Winrad to change the passband limits
           // > and/ortheCWpitch . (..)
           // Must call GetFilters()
           // MainForm->mustGetFilters = true;
           break;
      } // end switch(status)
    return;
   }  // end if(cnt < 0)

  //----------- not a status change, but a sending of I/O data -----------
  // if(!MainForm->running) return;
  if( (pInstData->i32ExtIOHWtype == EXTIO_HWT_SDR14) && (status != 2/*RUNNING*/) )
   { return;
   }

  if(IQdata == NULL)
   { return;
   }

  // Ignore 'cnt' here ?
  // > cnt
  // > is the number of samples returned. As the data is complex (I/Q pairs),
  // > then there are two 16 bitvaluespersample.Ifnegative,thenthe
  // >callbackwascalledjusttoindicateastatuschange,nodatareturned.
  // > PresentlyWinraddoesnotusethisvalue,butratherthereturnvalue
  // > oftheStartHW()API, to allocate the buffers and process the audio data
  // > returned by the DLL. The cnt value ischeckedonlyfornegativevalue,
  // > meaningastatuschange.
  //  The question is if all existing ExtIO-DLL pass in a 'valid' cnt,
  //  or sloppily just pass some non-negative junk in cnt.
  //  ExtIO_SDR14.DLL behaved well:  cnt == pInstData->nIQPairsPerExtIoCallback == 2048 .
  //
  // Ignore the 'I/Q offset' here ?  Certainly yes, because we don't know
  // how to apply it.
  switch( pInstData->i32ExtIOHWtype )  // the 'hardware type' actually defines the DATA TYPE:
   {
     case EXTIO_HWT_SDR14      : // Not sure if this must be supported here at all...
        // but with ExtIO_SDR14.DLL, got here with pInstData->i32ExtIOHWtype==1 !
        // Looking at ?/WinradSources/Sound_io.cpp, EXTIO_HWT_SDR14 uses 16-bit integers
        // so treat this the same way as the next case:
     case EXTIO_HWT_USB_Int16  : // I/Q data as pairs of 16-bit integers
        AIO_SampleBuffer_WriteSamples_Int16( &pInstData->buf,
             (INT16*)IQdata, pInstData->nIQPairsPerExtIoCallback, 2/*channels/sample*/ );
        break;
     case EXTIO_HWT_USB_Int24  : // I/Q data as pairs of 24-bit integers
        AIO_SampleBuffer_WriteSamples_Int24( &pInstData->buf,
             (BYTE*)IQdata, pInstData->nIQPairsPerExtIoCallback, 2/*channels/sample*/ );
        break;
     case EXTIO_HWT_USB_Int32  : // I/Q data as pairs of 32-bit integers
        AIO_SampleBuffer_WriteSamples_Int32( &pInstData->buf,
             (INT32*)IQdata, pInstData->nIQPairsPerExtIoCallback, 2/*channels/sample*/ );
        break;
     case EXTIO_HWT_USB_Float32: // I/Q data as pairs of 32-bit floats
        AIO_SampleBuffer_WriteSamples_Float( &pInstData->buf,
             (float*)IQdata, pInstData->nIQPairsPerExtIoCallback, 2/*channels/sample*/ );
        break;
     default :                   // OOPS ?!
        break;
   } // end switch( pInstData->i32ExtIOHWtype )


}
#endif // SWI_SUPPORT_EXTIO_DLL ?


static void SetCurrentDirectoryForDLL( char *pszPathAndName )
{ char sz511DirPath[512];
  char *cp;
  // Isolate the directory path, without the name of the DLL:
  strncpy( sz511DirPath, pszPathAndName, 511);
  sz511DirPath[511] = '\0';  // always terminate a C-string !
  cp = strrchr( sz511DirPath, '\\' );
  if( cp==NULL ) // oops... Linux ?
   { cp= strrchr( sz511DirPath, '/' );
   }
  if( cp != NULL )
   { *cp = '\0'; // Remove the FILENAME from the DIRECTORY PATH.
                 // The result does NOT end with a backslash !
   }
  SetCurrentDirectory( sz511DirPath );
} // end SetCurrentDirectoryForDLL()

//----------------------------------------------------------------------
static void EnterCritter( T_AIO_DLLHostInstanceData *pInstData )
{
#if( AIO_HOST_USE_CRITTER )
  if( pInstData->nCritterEntryCounter>0)
   { // added 2012-11-19 for debugging purposes only ..
     pInstData->nCritterEntryCounter = pInstData->nCritterEntryCounter; // << set breakpoint HERE
     // 2012-11-19 : Got here in Spectrum Lab when called through
     //   TSoundThread::Execute() -> SoundThd_InitializeAudioDevices()
     //    -> CSound::InOpen() -> AIO_Host_OpenAudioInput() -> EnterCritter() .
     // 2012-11-19 : Got here in Spectrum Lab when called through
     //   TSpectrumLab::Timer1Timer() -> SndThd_On50msTimer()
     //    -> AIO_Host_GetNominalSampleRate() -> EnterCritter() .  That's ok.
   }
  ++pInstData->nCritterEntryCounter;
  EnterCriticalSection( &pInstData->Critter );
  // 2012-11-19: Got stuck in EnterCriticalSection() when nCritterEntryCounter was TWO,
  //   called through SndThd_On50msTimer() -> AIO_Host_GetNominalSampleRate() .
  //   "The other" (still pending) call could not be tracked (helpless debugger,
  //   unable to tell the state of all other threads except the currently
  //   'single-stepped' one) .
#endif
} // end EnterCritter()

//----------------------------------------------------------------------
static void LeaveCritter( T_AIO_DLLHostInstanceData *pInstData )
{
#if( AIO_HOST_USE_CRITTER )
  LeaveCriticalSection( &pInstData->Critter );
  --pInstData->nCritterEntryCounter; // just here to check balance of Enter/Leave critical sections
#endif
} // end LeaveCritter()

//----------------------------------------------------------------------
static INT32 LogCriticalError( INT32 aio_error_code )
{
  aio_error_code = aio_error_code;   // << set a breakpoint here !
  UTL_WriteRunLogEntry( "AIO_Host: Critical error (%d) !",(int)aio_error_code );
  return aio_error_code;
} // end LogCriticalError()

//----------------------------------------------------------------------
INT32 AIO_Host_LoadDLL(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        char *pszPathAndName) // [in] path+name of the DLL to be loaded.
                              //      If NULL, we make a guess for WINAMP.
  // Loads an Audio-I/O compatible DLL, for example in_AudioIO.dll .
  // Must be called exactly ONCE before any other function in this module.
  // Calling it TWICE will unload the "old" DLL, and load a "new" DLL .
  //
  // CAUTION-CAUTION-CAUTION ! !
  //   Certain DLLs must be loaded from Winamp's plugin directory,
  //   otherwise the access to the common shared memory will fail !
  //   (this doesn't apply to 'normal' Audio-I/O-Libraries,
  //    and of course not to Winrad-ExtIO-compatible DLLs)
  //
  // Return value:  AIO_NO_ERROR  or one of the AIO_... error codes .
  //
{
  // char *cp, *cp2;
  char *cpFilenameWithoutPath;
  char sz511GuessedPathAndName[512];
  INT32  i32Result = AIO_ERROR_NOT_IMPLEMENTED;


  if( !pInstData->fCritterInitialized )
   { return LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
   }

  DOBINI();
  EnterCritter( pInstData );
  DOBINI();
   {  // NO 'LAZY RETURN' AFTER THIS POINT !


     if(  (pInstData->h_AudioIoDLL != NULL)
#    if( SWI_SUPPORT_EXTIO_DLL )
        ||( ExtIO_pInstData == pInstData)  // did THIS INSTANCE host an ExtIO-DLL ?
#    endif // SWI_SUPPORT_EXTIO_DLL ?
       )
      {  AIO_Host_FreeDLL(pInstData);  // oops.. been in use before ?   Unload the "old" DLL .
      }

     //
     // Added 2012-10-26: Certain ExtIO-DLLs didn't find other DLLs (which THEY
     //  needed to run) in 'their own' working directory. To fix that, this DLL-host
     //  now temporarily sets the 'current working directory' to where the DLL itself
     //  is located;
     //  but only while loading + initialising that DLL. After that,
     //  the 'current working directory' (an anachronism) is set back to where it was:
     if( ! pInstData->fCurrentDirectorySetToDLL )
      { GetCurrentDirectory( 511, pInstData->sz511OriginalDirectory );
      }
     SetCurrentDirectoryForDLL( pszPathAndName );
     pInstData->fCurrentDirectorySetToDLL = TRUE;

     if( pszPathAndName != NULL )
      { // The application has specified the filename (possibly including the path)
        //  of the audio-I/O-compatible DLL .   So use that one first :
        pInstData->h_AudioIoDLL = LoadLibrary(pszPathAndName);
        // Note: if LoadLibrary fails, it returns NULL (!= INVALID_HANDLE_VALUE).
      }
     else  // the application did NOT specify a path + name .
      { // In this case, try to find the DLL in the winamp plugin directory .
        // Unfortunately Billy's men decided to make the name of the program
        // folder depend on the language of the windows installation,
        // so there is no easy way to find it.
        // To avoid fooling around with the windoze registry, we simply
        // try the following for "default" english and german installations :
        cpFilenameWithoutPath = strrchr( pszPathAndName, '\\' );   // Windoze-compatible path-separator
        if( cpFilenameWithoutPath==NULL ) // oops... Linux ?!
         { cpFilenameWithoutPath = strrchr( pszPathAndName, '/' ); // Penguin-compatible path-separator
         }
        if( cpFilenameWithoutPath != NULL )
         { ++cpFilenameWithoutPath;  // now points to the 1st character of the filename, WITHOUT path
         }
        else
         { cpFilenameWithoutPath = pszPathAndName; // this thingy doesn't seem to have a path at all !
         }
        strcpy( sz511GuessedPathAndName, "c:\\Program Files\\Winamp\\Plugins\\" ); // windoze XP
        strcat( sz511GuessedPathAndName, cpFilenameWithoutPath );
        pszPathAndName = sz511GuessedPathAndName;
        pInstData->h_AudioIoDLL = LoadLibrary(pszPathAndName);
        if( pInstData->h_AudioIoDLL == NULL )
         { // doesn't seem to be there... try something else : GERMAN ...
           strcpy( sz511GuessedPathAndName, "c:\\Programme\\Winamp\\Plugins\\" ); // windoze XP, germanski
           strcat( sz511GuessedPathAndName, cpFilenameWithoutPath );
           pInstData->h_AudioIoDLL = LoadLibrary(pszPathAndName);
         }
        if( pInstData->h_AudioIoDLL == NULL )
         { // doesn't seem to be there... try something else : ITALIAN ...
           strcpy( sz511GuessedPathAndName,  "c:\\Programmi\\Winamp\\Plugins\\" ); // windoze XP, italiano
           strcat( sz511GuessedPathAndName, cpFilenameWithoutPath );
           pInstData->h_AudioIoDLL = LoadLibrary(pszPathAndName);
         }
        if( pInstData->h_AudioIoDLL == NULL )  // Still out of luck ? Try something else ...
         { strcpy( sz511GuessedPathAndName, cpFilenameWithoutPath );
           pInstData->h_AudioIoDLL = LoadLibrary(pszPathAndName);
         }
      }
     if( pInstData->h_AudioIoDLL==NULL )
      { // What went wrong in 'LoadLibrary' (Win32 API) ? Translate this into an AIO-error-code !
        i32Result = WindowsErrorToAIOErrorCode( GetLastError(), AIO_ERROR_DLL_NOT_LOADED );
      }
     // Save the complete path and name ("fully specified" or "guessed").
     //     Required for AudioIO_DLL_ControlUnit, to avoid loading twice.
     strncpy( pInstData->sz255DLLPathAndName, pszPathAndName, 255 );
     pInstData->sz255DLLPathAndName[255] = '\0';


     if( pInstData->h_AudioIoDLL!=NULL )
      {
        // Get the function pointers of everything we need to call (inside the Audio-I/O-DLL) .
        // Note: When compiling DLLs with certain compilers (like Borland C++Builder),
        //       the names in the DLL get MANGLED so GetProcAddress() won't find them.
        // Cure: Use a *.DEF-file, for example c:\cbproj\AudioIO\in_AudioIO_BCB4.def .
        pInstData->m_pErrorCodeToString=(tpErrorCodeToString)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_ErrorCodeToString");
        pInstData->m_pShowAboutBox    =(tpShowAboutBox)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_ShowAboutBox");
        pInstData->m_pShowControlPanel=(tpShowControlPanel)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_ShowControlPanel");
        pInstData->m_pGetWinampParams =(tpGetWinampParams)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_GetWinampParams");
        // Audio INPUT functions in the DLL :
        pInstData->m_pGetOutputParams =(tpGetOutputParams)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_GetOutputParams");
        pInstData->m_pOpenAudioOutput =(tpOpenAudioOutput)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_OpenAudioOutput");
        pInstData->m_pCloseAudioOutput=(tpCloseAudioOutput)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_CloseAudioOutput");
        // pInstData->m_pWriteAudioOutput=(tpWriteAudioOutput)GetProcAddress(pInstData->h_AudioIoDLL,"AIO_WriteAudioOutput");
        pInstData->m_pCanWriteOutput  =(tpCanWriteOutput)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_CanWriteOutput");
        pInstData->m_pWriteOutputSamplePoints=(tpWriteOutputSamplePoints)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_WriteOutputSamplePoints");
        // Audio INPUT functions in the DLL :
        pInstData->m_pOpenAudioInput  =(tpOpenAudioInput)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_OpenAudioInput");
        pInstData->m_pCloseAudioInput =(tpCloseAudioInput)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_CloseAudioInput");
        pInstData->m_pReadInputSamplePoints=(tpReadInputSamplePoints)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_ReadInputSamplePoints");
        pInstData->m_pCanReadInput    =(tpCanReadInput)GetProcAddress(pInstData->h_AudioIoDLL,
              "AIO_CanReadInput" );

#    if( SWI_SUPPORT_EXTIO_DLL )   // support Winrad-compatible ExtIO-DLLs ?
        pInstData->fExtIOLoaded = pInstData->fExtIOHWready = pInstData->fHWopened = pInstData->fHWstarted = FALSE;
        pInstData->m_pExtIO_InitHW  = (pfnInitHW) GetProcAddress(pInstData->h_AudioIoDLL, "InitHW" );
        pInstData->m_pExtIO_OpenHW  = (pfnOpenHW) GetProcAddress(pInstData->h_AudioIoDLL, "OpenHW");
        pInstData->m_pExtIO_CloseHW = (pfnCloseHW)GetProcAddress(pInstData->h_AudioIoDLL, "CloseHW" );
        pInstData->m_pExtIO_StartHW = (pfnStartHW)GetProcAddress(pInstData->h_AudioIoDLL, "StartHW");
        pInstData->m_pExtIO_StopHW  = (pfnStopHW) GetProcAddress(pInstData->h_AudioIoDLL, "StopHW");
        pInstData->m_pExtIO_SetCallback = (pfnSetCallback) GetProcAddress(pInstData->h_AudioIoDLL, "SetCallback");
        pInstData->m_pExtIO_SetHWLO = (pfnSetHWLO)GetProcAddress(pInstData->h_AudioIoDLL, "SetHWLO");
        pInstData->m_pExtIO_GetHWLO = (pfnGetHWLO)GetProcAddress(pInstData->h_AudioIoDLL, "GetHWLO");
        pInstData->m_pExtIO_GetStatus=(pfnGetStatus)GetProcAddress(pInstData->h_AudioIoDLL,"GetStatus");
        pInstData->m_pExtIO_GetHWSR = (pfnGetHWSR) GetProcAddress(pInstData->h_AudioIoDLL, "GetHWSR");
        pInstData->m_pExtIO_RawDataReady= (pfnRawDataReady)GetProcAddress(pInstData->h_AudioIoDLL,"RawDataReady");
        pInstData->m_pExtIO_ShowGUI = (pfnShowGUI)GetProcAddress(pInstData->h_AudioIoDLL,  "ShowGUI");
        pInstData->m_pExtIO_HideGUI = (pfnHideGUI)GetProcAddress(pInstData->h_AudioIoDLL,  "HideGUI");
           // the following stuff usually doesn't exist :
        pInstData->m_pExtIO_TuneChanged = (pfnTuneChanged)GetProcAddress(pInstData->h_AudioIoDLL,"TuneChanged");
        pInstData->m_pExtIO_GetTune = (pfnGetTune) GetProcAddress(pInstData->h_AudioIoDLL, "GetTune");
        pInstData->m_pExtIO_ModeChanged = (pfnModeChanged)GetProcAddress(pInstData->h_AudioIoDLL,"ModeChanged");
        pInstData->m_pExtIO_GetMode = (pfnGetMode) GetProcAddress(pInstData->h_AudioIoDLL, "GetMode");
        pInstData->m_pExtIO_IFLimitsChanged = (pfnIFLimitsChanged)GetProcAddress(pInstData->h_AudioIoDLL, "IFLimitsChanged");
        pInstData->m_pExtIO_FiltersChanged  = (pfnFiltersChanged)GetProcAddress(pInstData->h_AudioIoDLL,  "FiltersChanged");
        pInstData->m_pExtIO_GetFilters  = (pfnGetFilters)GetProcAddress(pInstData->h_AudioIoDLL, "GetFilters");
        // Check if the mandatory entry points *for an ExtIO-compatible DLL* are present.
        // If ANY of these functions are missing, we're not calling ANYTHING inside the ExtIO-DLL:
        if(  pInstData->m_pExtIO_InitHW == NULL
          || pInstData->m_pExtIO_CloseHW == NULL
          || pInstData->m_pExtIO_OpenHW == NULL
          || pInstData->m_pExtIO_StartHW == NULL
          || pInstData->m_pExtIO_StopHW == NULL
          || pInstData->m_pExtIO_SetCallback == NULL
          || pInstData->m_pExtIO_SetHWLO == NULL
          || pInstData->m_pExtIO_GetStatus == NULL)
         { // It's NOT a Winrad-ExtIO-compatible DLL, so DO NOT TREAT IT AS SUCH:
           pInstData->fExtIOLoaded  = FALSE;
           pInstData->fExtIOHWready = FALSE;
         }
        else
         { // Bingo, it's an ExtIO-compatible DLL. Initialize it as described
           // in Alberto's Winrad - specifications for the external I/O DLL :
           // CoInitialize(0);   // initialize "COM" AGAIN. This is stupid, because unknown other modules may do the same. One of them is the ASIO-host.
           // Moved this COM-stuff into utility1.cpp::UTL_Init(), where CoInitialize() is called exactly ONCE.
           if( ExtIO_pInstData == NULL )  // can only load ONE SINGLE INSTANCE of such a DLL !
            {
              ExtIO_pInstData = pInstData;  // address of instance data for ExtIO-callback
              pInstData->fExtIOLoaded  = TRUE;
              pInstData->extHWname[0]  = '\0';
              pInstData->extHWmodel[0] = '\0';
              pInstData->i32ExtIOHWtype= 0;
              // Initialisation sequence for ExtIO-compatible DLLs .
              // Details in
              //  >>>>>>> C:\cbproj\ExtIO\ExtIO_Notes_DL4YHF.htm#calling_sequence <<<<<<< !
              // > In Winrad, InitHW() is called once from TMainForm::FormShow(),
              // > directly after loading the DLL  .
              // 2012-08-11: WITHOUT calling InitHW(), FreeLibrary() did NOT crash later.
              // 2012-08-11: Removing some of the DLLIMPORT-stuff in AudioIO.h seemed to cure this.
              // 2012-12-27: Modified the treatment of the result from InitHW,
              //             because some compilers treat 'bool' as an EIGHT BIT VALUE
              //             (and only set the AL register, not EAX, to a meaningful result),
              //             while others (like Borland when compiling C, not C++)
              //             treat 'bool' as a 32-bit value, and return 'bool' in EAX .
              //             The '(BOOL)(BYTE)' double cast operator takes care of both,
              //             by ignoring the upper 24 bits in EAX :
              //  call dword ptr [eax+0x000007do] ; pInstData->m_pExtIO_InitHW()
              //  xor  edx, edx    ; EDX := 0   (32 bit register, dl=least significant byte in it)
              //  mov  dl, al      ; AL = 8 bits(!) returned as "bool" from InitHW()
              //  mov  [ecx+0x000003ac],edx  ; store pInstData->fExtIOHWready as 32-bit 'BOOL' (not 8-bit 'bool')
              //             Details in C:\cbproj\RTL_SDR\ExtIO\ExtIO_RTL.cpp .
#            ifdef UseUtility1
              UTL_WriteRunLogEntry( "AIO_Host_LoadDLL: Calling InitHW in ExtIO-DLL..." );
#            endif
              pInstData->fExtIOHWready = (BOOL)((BYTE)(pInstData->m_pExtIO_InitHW(
                                pInstData->extHWname, pInstData->extHWmodel,
                              & pInstData->i32ExtIOHWtype) ) );
#            ifdef UseUtility1
              UTL_WriteRunLogEntry( "InitHW returned: HW=\"%s\", Model=\"%s\", Type=%d, Rdy=0x%lX",
                 pInstData->extHWname, pInstData->extHWmodel,
                 (int)pInstData->i32ExtIOHWtype,  (long)pInstData->fExtIOHWready );
#            endif                 
              // 2012-08-09 : Exception "CoInitialize wurde nicht aufgerufen". Holy shit.
              //              IT WAS, but in another thread.
              //              Calling CoInitialize AGAIN (here) seemed to cure it,
              //              but the program (and the stupid debugger) froze then.
              //  Bloody Windows. Damned DLLs. Evil 'bool'. Stupid incompatible calling conventions !
              //     At least, with the *seemingly* superfluous CoInitialize-stuff,
              //     got here with pInstData->extHWname = "SDR-IQ   S/N = G000154" ,
              //                   pInstData->extHWmodel= "N/A" ,
              //                   pInstData->i32ExtIOHWtype = 1 = EXTIO_HWT_SDR14, ok,
              //         but       pInstData->fExtIOHWready = 77068545 (=suspicious) .
              //     First guess: Incompatible calling convention. Return in AX
              //                  or on the stack ? Check this by single-stepping
              //                  through the DISASSEMBLED CODE with the debugger (yucc).
              //        Results:  No obvious problem. ESP before the "call dword ptr"
              //                   was 0x0497F9E8, and 0x0497F9F4 thereafter,
              //                   which looks like 12 bytes = 3 DWORDs popped by CALLEE,
              //                   as expected for "STDCALL" (ALL arguments pushed on
              //                   the stack; stack cleaned up by CALLEE = called function).
              //            Maybe someone doesn't care (*) about the difference between 'bool'
              //            and 'integer' (77068545 would mean TRUE because it's non-zero) ?
              //                  Tried again with the SDR-IQ *DISCONNECTED*, and got:
              //                   pInstData->extHWname = "SDR-IQ/14 not connected or power off" ,
              //                   pInstData->extHWmodel= "Boh?"  (a joke, but ok),
              //                   pInstData->i32ExtIOHWtype = 1 = EXTIO_HWT_SDR14, ok,
              //         and       pInstData->fExtIOHWready = 0 = proper value for "FALSE" .
              //      (*) The reason for this annoyance was discovered later:
              //      "bool" is an evil type, especially when mixing "C" with C++ !
              //    TRIED THIS with Borland C++Builder, in a "C" and in a "C++" file:
              //     int i=sizeof(bool); // in a "C" file, compiled with Borland C++Builder: sizeof(bool)=4 .
              //     int i=sizeof(bool); // in a CPP file, compiled with Borland C++Builder: sizeof(bool)=1 !
              //      This is really stupid and annoying.  Avoid "bool" in any API function,
              //      especially when the same header shall be included from "C" *and* "C++" !
              //      ExtIO-DLLs compiled from a C++ module will return 'bool' in AL (8 bit) .
              //      ExtIO-DLLs compiled from a C module will return 'bool' in EAX (32 bit) .
              //      Thus, the DLL HOST (like DL4YHF's C:\cbproj\AudioIO\AudioIO.c) must
              //      ignore bits 31..8 in the value returned by InitHW(), and ideally only
              //      look at the value in AL ('least significant byte' of the EAX register).
              //      For DEVELOPERS OF ExtIO-DLLs : Don't declare 'InitHW' as "bool" !
              //      Instead, declared it as 'int' or 'long', which forces the compiler
              //      to set the ENTIRE EAX REGISTER (not only AL) to either 0 or 1 .
              //
              // 2012-10-28 : Tested with FiFi-SDR / ExtIO_Si570.dll by PE0FKO,
              //              found at http://pe0fko.nl/ExtIO_Si570/ .
              //    InitHW returned: HW="SoftRock Si570", Model="pe0fko", Rdy=0x1,
              //           and pInstData->i32ExtIOHWtype = 4 = EXTIO_HWT_Soundcard .
              //
              // 2012-11-19 : Tested with ExtIO_RTL.dll by Jose Araujo and a Noxon DAB stick
              //             (after other ExtIOs for "RTL SDR" failed miserably),
              //         and got:
              //                   pInstData->extHWname = "NOXON" ,
              //                   pInstData->extHWmodel= "DAB Stick" ,
              //                   pInstData->i32ExtIOHWtype = 3 = EXTIO_HWT_USB_Int16, ok,
              //         and       pInstData->fExtIOHWready = 1 = proper value for "TRUE"
              // As described in C:\cbproj\ExtIO\ExtIO_Notes_DL4YHF.htm ,
              // call SetCallback() immediately after InitHW() :
#            ifdef UseUtility1
              UTL_WriteRunLogEntry( "SetCallback for ExtIO-DLL" );
#            endif
              pInstData->m_pExtIO_SetCallback(ExtIO_Callback); // did the damn thing crash HERE ?
#            ifdef UseUtility1
              UTL_WriteRunLogEntry( "SetCallback returned" );
#            endif              
              // As described in C:\cbproj\ExtIO\ExtIO_Notes_DL4YHF.htm ,
              //   Winrad calls OpenHW() immediately after SetCallback(),
              //   but only in SelExtHWClick(), not in FormShow() .
              //   Thus better do NOT call OpenHW from here .

              // SelExtHW->Caption = extHWname;
            } // end if < ExtIO-DLL not already loaded in another instance ? >
         } // end else ('bingo')..
#    endif // SWI_SUPPORT_EXTIO_DLL ?

      } // end if( pInstData->h_AudioIoDLL!=NULL )


     // Are we happy with this, or is there something important missing ?
     if(   (pInstData->m_pOpenAudioOutput != NULL)
#      if( SWI_SUPPORT_EXTIO_DLL )
         ||( pInstData->fExtIOLoaded )
#      endif
         // && ...
       )
      { // ok, at least the "mandatory" functions are there -> ready to use ..
        pInstData->fOpenForBusiness = TRUE;
        i32Result = AIO_NO_ERROR;   // 0 = no error (but the result is no "handle" or similar)
      }
     else // something important is missing in the DLL . Kick it out, whatever it was
      { FreeLibrary(pInstData->h_AudioIoDLL);
        pInstData->h_AudioIoDLL = NULL;
#      if( SWI_SUPPORT_EXTIO_DLL )
        if( ExtIO_pInstData == pInstData )
         {  ExtIO_pInstData = NULL;
         }
#      endif
        i32Result = AIO_ERROR_DLL_NOT_LOADED;
      }

     if( pInstData->fCurrentDirectorySetToDLL )
      { // Set the 'current working directory' for THIS application(!) back where it should be :
        SetCurrentDirectory( pInstData->sz511OriginalDirectory );
        pInstData->fCurrentDirectorySetToDLL = FALSE; // the 'CWD' is now BACK TO NORMAL
      }
   } // end of the (optional) critical section
  LeaveCritter( pInstData );


  return i32Result;

} // end AIO_Host_LoadDLL()


//----------------------------------------------------------------------
void AIO_Host_FreeDLL(  // Should be called by application when terminating
        T_AIO_DLLHostInstanceData *pInstData) // [in,out] DLL-host instance data
{

  if( !pInstData->fCritterInitialized )
   { LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
     return;
   }

  DOBINI();
  EnterCritter( pInstData ); // NO 'LAZY RETURN' AFTER THIS POINT !
   { DOBINI();

     // Note: If still open, the DLL will close it's own control window
     //       WHEN THE LAST APPLICATION which loaded it HAS TERMINATED.
     //   Details in \cbproj\AudioIO\aio2winamp.c, DllMain() or DllEntryPoint() .
     //
     if( pInstData->fOpenForBusiness )  // don't let any thread call anything here now...
      { pInstData->fOpenForBusiness = FALSE;
        // Just for safety, wait a few milliseconds before "really" unloading
        // the DLL from memory. There may be some poorly written worker thread,
        // still "working" inside one of the functions which have officially
        // been declared invalid (see above) by setting the function pointers NULL.
        Sleep( 50/*milliseconds*/ );

        // In the meantime, all functions called by various worker-treads
        //  should have 'returned'..
        // make all references to the DLL invalid and unload it from memory .
      }

     // Make sure that an ill-behaving application doesn't call anything
     //    inside the DLL when the DLL is unloaded from memory !
     pInstData->m_pErrorCodeToString=NULL;
     pInstData->m_pGetWinampParams = NULL;
     pInstData->m_pGetOutputParams = NULL;
     pInstData->m_pOpenAudioOutput = NULL;
     pInstData->m_pCloseAudioOutput= NULL;
     // pInstData->m_pWriteAudioOutput= NULL;
     pInstData->m_pCanWriteOutput  = NULL;
     pInstData->m_pOpenAudioInput  = NULL;
     pInstData->m_pCloseAudioInput = NULL;
     pInstData->m_pCanReadInput    = NULL;
     pInstData->m_pReadInputSamplePoints   = NULL;
     pInstData->m_pWriteOutputSamplePoints = NULL;

     pInstData->m_pShowAboutBox    = NULL;
     pInstData->m_pShowControlPanel= NULL;

#   if( SWI_SUPPORT_EXTIO_DLL )   // support Winrad-compatible ExtIO-DLLs ?
     // Added 2012-08-08 for the ExtIO-compatible part :
     if(pInstData->fHWstarted && (pInstData->m_pExtIO_StopHW != NULL)  )
      {
#      ifdef UseUtility1
        UTL_WriteRunLogEntry( "AIO_Host_FreeDLL: Calling StopHW in ExtIO-DLL..." );
#      endif
        pInstData->m_pExtIO_StopHW();
#      ifdef UseUtility1
        UTL_WriteRunLogEntry( "AIO_Host_FreeDLL: Back from StopHW in ExtIO-DLL..." );
#      endif
      }
     pInstData->fHWstarted = FALSE;
     if(pInstData->m_pExtIO_CloseHW != NULL ) // && pInstData->fHWopened )
      {
#      ifdef UseUtility1
        UTL_WriteRunLogEntry( "AIO_Host_FreeDLL: Calling CloseHW in ExtIO-DLL..." );
#      endif
        pInstData->m_pExtIO_CloseHW();
        // 2012-08-09 : Terror in the ExtIO-DLL: "Illegal window handle" ?!
        //              -> avoid calling CloseHW() without previous OpenHW() ??
#      ifdef UseUtility1
        UTL_WriteRunLogEntry( "AIO_Host_FreeDLL: Back from CloseHW in ExtIO-DLL..." );
#      endif        
      }
     pInstData->m_pExtIO_InitHW  = NULL;
     pInstData->m_pExtIO_OpenHW  = NULL;
     pInstData->m_pExtIO_CloseHW = NULL;
     pInstData->m_pExtIO_StartHW = NULL;
     pInstData->m_pExtIO_SetCallback = NULL;
     pInstData->m_pExtIO_SetHWLO = NULL;
     pInstData->m_pExtIO_GetHWLO = NULL;
     pInstData->m_pExtIO_GetStatus = NULL;
     pInstData->m_pExtIO_GetHWSR = NULL;
     pInstData->m_pExtIO_RawDataReady = NULL;
     pInstData->m_pExtIO_ShowGUI = NULL;
     pInstData->m_pExtIO_HideGUI = NULL;
     pInstData->m_pExtIO_TuneChanged = NULL;
     pInstData->m_pExtIO_GetTune = NULL;
     pInstData->m_pExtIO_IFLimitsChanged = NULL;
     pInstData->m_pExtIO_FiltersChanged = NULL;
     pInstData->m_pExtIO_GetFilters = NULL;
     pInstData->fExtIOLoaded = pInstData->fExtIOHWready = pInstData->fHWopened = FALSE;
     if( ExtIO_pInstData == pInstData )  // did THIS INSTANCE host an ExtIO-DLL ?
      {  ExtIO_pInstData = NULL; // it doesn't host it anymore, and ANOTHER ONE may be loaded
      }
#   endif // SWI_SUPPORT_EXTIO_DLL ?


     // Now, after all function pointers are invalid, unload the DLL
     if( pInstData->h_AudioIoDLL != NULL )
      { // there was REALLY a DLL loaded ...
        // Note: If the Audio-I/O-DLL is *ALSO* a winamp plugin, and winamp
        //       is still using that DLL, windows will NOT unloaded it yet .
#      ifdef UseUtility1
        UTL_WriteRunLogEntry( "AIO_Host_FreeDLL: Calling FreeLibrary..." );
#      endif        
        FreeLibrary(pInstData->h_AudioIoDLL);
        // 2012-08-11: Got stuck in FreeLibrary() / ExtIO-DLL .
        pInstData->h_AudioIoDLL = NULL;
#      ifdef UseUtility1
        UTL_WriteRunLogEntry( "AIO_Host_FreeDLL: Survived, hooray." );
#      endif        
      }
   } // end of a long 'critical section'    
  LeaveCritter( pInstData );

} // end AIO_Host_FreeDLL()


//----------------------------------------------------------------------
const char *AIO_Host_ErrorCodeToString(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        int iAIOErrorCode )
  // Converts a (negative) Audio-I/O error code into a human readable string.
  // Since some error codes (numbers below minus 1000) may be DLL-specific,
  // this function is also located inside the DLL (and not in AudioIO.c) .
{
  const char *cp;
  if( (pInstData!=NULL) && pInstData->fOpenForBusiness && pInstData->fCritterInitialized
    && (pInstData->m_pErrorCodeToString!=NULL) )
   { // If we have loaded a valid Audio-I/O DLL, let the DLL say what the error means:
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     cp = (*pInstData->m_pErrorCodeToString)( (INT32)iAIOErrorCode );
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
     return cp;
   }
  else // If we don't have a valid Audio-I/O DLL in memory, use this "table":
   { switch( iAIOErrorCode )
      { // Caution: When adding new error strings here, make sure to modify:
        //  - C:\cbproj\AudioIO\AudioIO.c    (Audio-I/O-"Host" for applications)
        //  - c:\cbproj\AudioIO\aio2winamp.c (reference implementation of a DLL)
        case AIO_NO_ERROR              :  return "no error";                      // 0
        case AIO_GENERAL_ERROR         :  return "general error";                 // -1
        case AIO_ERROR_DLL_NOT_LOADED  :  return "DLL not loaded";                // -2
        case AIO_ERROR_DLL_NOT_COMPATIBLE: return "DLL not compatible";           // -3
        case AIO_ERROR_NOT_IMPLEMENTED :  return "function not implemented";      // -4
        case AIO_ERROR_INVALID_PARAMETER: return "invalid parameter";             // -5
        case AIO_ERROR_OUT_OF_RESOURCES:  return "out of resources";              // -6
        case AIO_ERROR_FORMAT_NOT_SUPPORTED: return "format not supported";       // -10
        case AIO_ERROR_SAMPRATE_NOT_SUPPORTED: return "samplerate not supported"; // -11
        case AIO_ERROR_FILE_NOT_FOUND  :  return "file not found";                // -12
        case AIO_ERROR_INPUT_NOT_OPEN  :  return "input not open";                // -14
        case AIO_ERROR_OUTPUT_NOT_OPEN :  return "output not open";               // -15
        case AIO_INVALID_HANDLE        :  return "invalid handle";                // -17
        case AIO_ERROR_CANCELLED       :  return "cancelled by user";             // -18
        case AIO_ERROR_CANNOT_WRITE_YET:  return "cannot write yet";              // -20
        case AIO_ERROR_CANNOT_READ_YET :  return "cannot read yet";               // -21
        default:
          if( iAIOErrorCode >= 0 )
            { return "positive results are no errors";
            }
          else
            { return "unknown error code - DLL unloaded ?"; /* (*) */
            }
          // (*) at least, such an error should not occurr when the DLL
          //     is not loaded. The DLL would know what it means !
      } // end m_pErrorCodeToString
   } // end if < Audio-I/O DLL *not* loaded >
} // end AIO_Host_ErrorCodeToString()   [ interface to the DLL, not the DLL itself ]

//----------------------------------------------------------------------
BOOL  AIO_Host_HasVFO( // -> TRUE=VFO (*) supported, FALSE=no. Used for ExtIO-DLLs.
       T_AIO_DLLHostInstanceData *pInstData)  // [in] DLL-host instance data
   // (*) "VFO" may be any kind of "externally controlled frequency converter".
   //     We don't care about the details. SL only needs to know if there's
   //     some kind of 'external frequency conversion' or not.
{
#if( SWI_SUPPORT_EXTIO_DLL )
   // If an ExtIO-DLL is loaded, AND it contains an entry point for 'GetHWLO',
   // we assume this 'driver' controls an external hardware *WITH* a VFO:
   if( pInstData->fExtIOLoaded && (pInstData->m_pExtIO_GetHWLO!=NULL) )
    { return TRUE;
    }
#endif // SWI_SUPPORT_EXTIO_DLL

  return FALSE;   // everything else is assumed NOT to support a VFO

} // end AIO_Host_HasVFO()

//----------------------------------------------------------------------
long AIO_Host_GetNominalSampleRate( // -> current (IF-) sampling rate in Hertz.
         T_AIO_DLLHostInstanceData *pInstData) // [in] DLL-host instance data
  // Only used for the ExtIO-DLLs, after the flag
  //  'pInstData->fSRChangedByDriver' has been set somewhere.
  // Most other drivers will NOT change the sampling rate by their own gusto.
{
  long i32SampRate_Hz = 0;


  if( !pInstData->fCritterInitialized )
   { return LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
   }

#if( SWI_SUPPORT_EXTIO_DLL )
  // If an ExtIO-DLL is loaded, AND it contains an entry point for 'GetHWLO',
  // we assume this 'driver' controls an external hardware *WITH* a VFO:
  if(   pInstData->fExtIOLoaded && pInstData->fCritterInitialized
    && (pInstData->m_pExtIO_GetHWSR!=NULL) )
   {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     i32SampRate_Hz = pInstData->m_pExtIO_GetHWSR(); // for ExtIO, only integer !!
     // (thus the application -SpectrumLab- will look up this "nominal SR"
     //  in a table, and use the CALIBRATED sampling rate from that table
     //  for its own processing)
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
   }
#endif // SWI_SUPPORT_EXTIO_DLL
  // pInstData->fSRChangedByDriver = FALSE; // clear this flag; the application took notice
  // Removed; the ugly "flag-polling-and-clearing" is now in SoundThd.cpp !
  return i32SampRate_Hz;
} // end AIO_Host_GetNominalSampleRate()

//----------------------------------------------------------------------
long AIO_Host_GetVFOFrequency( // -> current VFO tuning frequency in Hertz.
       T_AIO_DLLHostInstanceData *pInstData) // [in] DLL-host instance data
  // Only used for the ExtIO-DLLs, after the flag
  //  'pInstData->fVFOChangedByDriver' has been set somewhere.
  // Most other drivers will NOT change the VFO frequency by their own gusto.
  // Spectrum Lab periodically calls this function IN THE MAIN THREAD,
  // from CSound::GetVFOFrequency(), to update the VFO display in the main
  // window after -for example- the tuning frequency has been changed
  // unexpectedly by the ExtIO-DLL *itself* (for example, FiFi-SDR, Si570) .

{
  long i32VFOFreq_Hz;

  if( !pInstData->fCritterInitialized )
   { return LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
   }


#if( SWI_SUPPORT_EXTIO_DLL )
  // If an ExtIO-DLL is loaded, AND it contains an entry point for 'GetHWLO',
  // we assume this 'driver' controls an external hardware *WITH* a VFO:
  if( pInstData->fExtIOLoaded && pInstData->fCritterInitialized
    && (pInstData->m_pExtIO_GetHWLO!=NULL)
    && (pInstData->fVFOChangedByDriver || (pInstData->i32ExternalVfoFrequency<=0) )
    )
   {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     i32VFOFreq_Hz = pInstData->m_pExtIO_GetHWLO(); // for ExtIO, only integer
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();

     if( i32VFOFreq_Hz != pInstData->i32ExternalVfoFrequency )
      { // Surprise surprise !
        pInstData->i32ExternalVfoFrequency = i32VFOFreq_Hz; // polled in Sound.cpp, CSound::InReadStereo()
      }
   }
#else
  // everything else is assumed NOT to support a VFO ...
#endif // SWI_SUPPORT_EXTIO_DLL
  pInstData->fVFOChangedByDriver = FALSE; // clear this flag; the application took notice

  return pInstData->i32ExternalVfoFrequency;

} // end AIO_Host_GetVFOFrequency()

//----------------------------------------------------------------------
long AIO_Host_SetVFOFrequency( // -> current VFO tuning frequency in Hertz.
       T_AIO_DLLHostInstanceData *pInstData, // [in] DLL-host instance data
       long i32VFOFrequency )                // [in] new VFO frequency in Hertz
   // Return value (only valid if the loaded driver supports a VFO,
   //   which, at the moment, can only be a Winrad-compatible ExtIO-DLL;
   //   thus this is almost the same as Winrad's "SetHWLO" :
   //  0 : Thefunctiondidcompletewithouterrors.
   //  <0(anegativenumberN) :
   //      Thespecifiedfrequencyislowerthantheminimumthatthehardware
   //     iscapabletogenerate.TheabsolutevalueofNindicates
   //      whatistheminimumsupportedbytheHW.
   //  >0(apositivenumberN)Thespecifiedfrequencyisgreater
   //      thanthemaximumthatthehardwareiscapabletogenerate.
   //      ThevalueofNindicateswhatisthemaximumsupportedbytheHW.
  // IMPORTANT: For the ExtIO-DLLs, this function must be called
  //  BEFORE opening the "driver" for input, because for some reason,
  //  the VFO frequency must be passed as a function argument in "StartHW" !
  //  (failing to do so caused ExtIO_RTL.DLL to open a stupid message box
  //   telling something about "PLL not locked", instead of silently returning
  //   with an error code, and leaving the user interaction to the HOST) .
  //  When called "early" (before calling StartHW), the VFO frequency is
  //  stored internally, but not sent to the 'radio' yet [2012-11-22] .
{
   long i32Result = 0;
   long i32ReadBackFrequency;

   // 2012-11-19: Got a message box from Jose's ExtIO_RTL.dll telling a story
   //             about "PLL not locked", BEFORE getting here at all !
   //             Shortly after, the program (or the DLL?) crashed
   //             with yet another access violation .
   //             Tried to intercept ALL calls of the ExtIO-DLL
   //             to spot errors in the CALLING SEQUENCE,
   //             which may possibly be caused by the multithreading in SL .
   //  Looking at C:\RTL-SDR\ExtIO_RTL_mit_Sourcen\ExtIO_RTL-master\src\ExtIO_RTL.cpp,
   //     there is in fact this stupid call to the BLOCKING MESSAGE BOX function:
   //  > MessageBox(NULL, TEXT("PLL not locked!"),TEXT("Error!"), MB_OK|MB_ICONERROR);
   //     .. which causes SetHWLO() blocking the caller (and SL's high-priority
   //        audio processing thread, which calls AIO_Host_SetVFOFrequency() !
   //  Too bad. Looks like we'd better not call SetHWLO() at all ...
   //  .. but SetHWLO() is also called from StartHW() in ExtIO_RTL.cpp,
   //     which means that even AIO_Host_Start()->StartHW() is in danger
   //     of blocking the process for ETERNAL (until the operator has closed
   //     the stupid Message Box .. nnnngrrrr ) .
   //
   pInstData->i32ExternalVfoFrequency = i32VFOFrequency; // maybe for later / "Start()"


   if( !pInstData->fCritterInitialized )
    { return LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
    }


#if( SWI_SUPPORT_EXTIO_DLL )
   // If an ExtIO-DLL is loaded, AND it contains an entry point for 'GetHWLO',
   // we assume this 'driver' controls an external hardware *WITH* a VFO:
  if( pInstData->fExtIOLoaded
   && pInstData->fHWstarted // added 2012-11-22 to defeat the 'PLL'-trouble mentioned below
   && pInstData->fCritterInitialized
   && (pInstData->m_pExtIO_SetHWLO!=NULL) )
   {
     DOBINI();   // 2012-11-19 : Got stuck in THIS LINE ?!
     EnterCritter( pInstData );
     DOBINI();
     i32Result = pInstData->m_pExtIO_SetHWLO( i32VFOFrequency );
     // 2012-11-22, ExtIO_RTL.DLL : "PNG ! PLL not locked !" (stupid message popup,
     //   even though i32VFOFrequency was 434000000 (434 MHz) which is perfectly valid
     //   as a 'tuning frequency' for the Noxon "DAB/DAB+ USB-Stick" . Drecksding. )
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
      // When trying to set approx. 24 MHz via ExtIO_SDR14.DLL,
      // the return value was 34208500 = approx. 34 MHz,
      // even if the input value (frequency) was perfectly valid.
      // DLL non-conformant with the ExtIO-DLL-specification ?
      // Or did someone not pay attention to the dreadful calling conventions in "C" ?
      if( (i32Result!=0) && (pInstData->m_pExtIO_GetHWLO!=NULL) )
       { i32ReadBackFrequency = pInstData->m_pExtIO_GetHWLO();
         if( i32ReadBackFrequency==i32VFOFrequency )
          { // Actually, SETTING the frequency was successful,
            // it's just the DLL being a bit stupid !
            // 2012-11-22 : Got here with ExtIO_RTL.DLL, i32Result=-1,
            //    i32ReadBackFrequency = 100 000 000 (?!) . Stupid.
            //
            // 2012-08-15 : Got here with ExtIO_SDR14.DLL,
            // i32Result=34205872, i32ReadBackFrequency=9 [Hz], i32VFOFrequency=9 [Hz],
            // so according to the Winrad ExtIO-spec the result should have
            // been ZERO (because setting was successful) :
            // ex: i32Result = 0;
            // But unfortunately, ExtIO_SDR14.DLL is even more stupid:
            // Tried to set it to 70 MHz (VFO), and read back 70 MHz,
            // which of course is NOT OK ! Thus, even more stuff required
            // to return the correct value compatible with Alberto's spec:
            if( i32Result>0 && i32ReadBackFrequency>i32Result )
             { // stupid ExtIO-DLL; returns a value which can NEVER be true !
               // leave i32Result as-is, guess it's REALLY the maximum frequency
             }
            else if( i32Result>0 && i32ReadBackFrequency<-i32Result )
             { // stupid ExtIO-DLL #2; returns a TOO LOW value which can also not be true !
               i32Result = -1;  // this is just GUESSWORK !
               // > Thespecifiedfrequencyislowerthantheminimumthatthehardware
               // >iscapabletogenerate.TheabsolutevalueofNindicates
               // > whatistheminimumsupportedbytheHW.
             }
            else if( i32Result<0 && i32ReadBackFrequency<i32Result )
             { // keep i32Result as-is (NEGATIVE)
             }
            else // the frequency is between +/- i32Result, so GUESS it was successful,
             { i32Result = 0;   // indicate "No Error" to the caller
             }
          }
       }
    }
#endif // SWI_SUPPORT_EXTIO_DLL

  return i32Result;

} // end AIO_Host_SetVFOFrequency()

//----------------------------------------------------------------------
BOOL AIO_Host_Start(
        T_AIO_DLLHostInstanceData *pInstData) // [in,out] DLL-host instance data
  // Only required for the ExtIO-DLLs. Nothing else really needs this.
{
   BOOL fResult = TRUE;


  if( !pInstData->fCritterInitialized )
   { LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
     return FALSE;
   }


#if( SWI_SUPPORT_EXTIO_DLL )
   // If an ExtIO-DLL is loaded, AND it contains an entry point for 'GetHWLO',
   // we assume this 'driver' controls an external hardware *WITH* a VFO:
   if( pInstData->fExtIOLoaded && pInstData->fCritterInitialized
     && (pInstData->m_pExtIO_StartHW!=NULL) )
    {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     // Caution: In ExtIO_RTL.cpp, StartHW() calls SetHWLO(), and SetHWLO()
     //          may decide to open a stupid MODAL MESSAGE BOX which blocks this call,
     //          and thus occupies the Critical Section for eternity.  YUCC !
     //  To avoid this, the parameter passed to StartHW() MUST be a valid VFO-frequency.
     //  But what to do if the shiny bloated C++ class (CSound) failed to set
     //  a proper VFO frequency, because it has NO IDEA about the radio's
     //  valid frequency range ? Alberto has given an example in Winrad_Extio.pdf
     //  to set a valid default value in the DLL implementation, which can
     //  possibly be queried from the DLL somehow. As usual, the question is HOW,
     //  and WHERE, and if the calling sequence is allowed at all.
     //  So, we only try to retrieve the goddamned VFO frequency from the driver
     //  if the DLL-hosting application hasn't set its own frequency yet:
     if( pInstData->i32ExternalVfoFrequency < 0 ) // app hasn't set this yet..
      { pInstData->i32ExternalVfoFrequency = AIO_Host_GetVFOFrequency(pInstData);
        //
      }
     pInstData->nIQPairsPerExtIoCallback = pInstData->m_pExtIO_StartHW(pInstData->i32ExternalVfoFrequency); // here in AIO_Host_Start()
     // Crashed when trying to "return to the caller" in the debugger from here,
     // but no problem when "single stepping" until the end, and THEN returning.
     // This bug only happened with ExtIO_SDR.DLL, not no other ExtIO-DLL. Sigh..
     // 2012-11-23: Got here with pInstData->nIQPairsPerExtIoCallback = -1
     //             when trying to use ExtIO_RTL.DLL in Spectrum Lab .
     //             Caller was TSoundThread::Execute() -> CSound::Start() .
     DOBINI();
     LeaveCritter( pInstData );
     // 2012-11-23: Crashed in LeaveCritter() above, after ExtIO_RTL.DLL
     //             had returned with pInstData->nIQPairsPerExtIoCallback = -1 ?!
     DOBINI();
      // > ThisentryiscalledbyWinradeachtimetheuserpressestheStart
      // > buttonontheWinradmainscreen,afterhavingpreviouslyspecified
      // > thattheDLLisincontroloftheexternalhardware . (...)
      // > Return value: An integer specifying how many I/Q pairs are
      // > returned by the DLL each time the callback function is invoked.
      // WB 2012-11-19: This is obviously NOT TRUE, or some authors didn't care.
      //    For example, ExtIO_USRP.dll :: StartHW() returned ONE ?! !?
      //
      if( pInstData->nIQPairsPerExtIoCallback < 0 )
       {
#       ifdef UseUtility1
         DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, UTL_USE_CURRENT_TIME,
           "ExtIO-DLL: StartHW() failed, retval = %d !", (int)pInstData->nIQPairsPerExtIoCallback );
#       endif
         fResult = FALSE;
       }
      else
       { pInstData->fHWstarted = fResult = TRUE;
       }
    }
#endif // SWI_SUPPORT_EXTIO_DLL ?
   return fResult;
} // end AIO_Host_Start()

//----------------------------------------------------------------------
BOOL AIO_Host_Stop(
        T_AIO_DLLHostInstanceData *pInstData) // [in,out] DLL-host instance data
  // Only required for the ExtIO-DLLs. Nothing else really needs this.
{
   BOOL fResult = TRUE;

  if( !pInstData->fCritterInitialized )
   { LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
     return FALSE;
   }

#if( SWI_SUPPORT_EXTIO_DLL )
   // If an ExtIO-DLL is loaded, AND it contains an entry point for 'GetHWLO',
   // we assume this 'driver' controls an external hardware *WITH* a VFO:
   if( pInstData->fExtIOLoaded && (pInstData->m_pExtIO_StopHW!=NULL) )
    {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     pInstData->m_pExtIO_StopHW();  // <<  set breakpoint HERE !
     // 2012-11-19: The goddamned 'ExtIO_USRP.DLL' never returned from the
     //   above call.   WB gave up at this point .
     pInstData->fHWstarted = FALSE;
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
    }
#endif // SWI_SUPPORT_EXTIO_DLL ?
   return fResult;
} // end AIO_Host_Stop()


//----------------------------------------------------------------------
INT32 AIO_Host_ShowAboutBox(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        HWND   hwndParent )  // [in] handle to the host's window
  // Shows the DLL's "about" box as a modal (blocking) dialog.
  // Certain special DLLs may provide some additonal CONFIGURATION here.
  // Return value should be AIO_NO_ERROR if the box was closed with the "OK" button,
  // or AIO_ERROR_CANCELLED when an optional "Cancel"-button was clicked .
{
  if( (pInstData!=NULL) && pInstData->fOpenForBusiness && (pInstData->m_pShowAboutBox!=NULL) )
   { // If we have loaded a valid Audio-I/O DLL, let it show the 'About' box:
     return (*pInstData->m_pShowAboutBox)( hwndParent );
   }
  else // If we don't have a valid Audio-I/O DLL in memory, use this "table":
   { return AIO_ERROR_NOT_IMPLEMENTED;
   } // end if < Audio-I/O DLL *not* loaded >
} // end AIO_Host_ShowAboutBox()   [ interface to the DLL, not the DLL itself ]

//----------------------------------------------------------------------
CPROT BOOL AIO_Host_HasControlPanel( // -> TRUE=yes, FALSE=no
       T_AIO_DLLHostInstanceData *pInstData) // [in] DLL-host instance data
{
  if( (pInstData!=NULL) && pInstData->fOpenForBusiness )
   { // Have loaded a valid Audio-I/O DLL, let it show its Control Panel..
     if( pInstData->m_pShowControlPanel!=NULL )
      { // looks like an Audio-I/O-DLL (not an ExtIO-DLL) WITH a control panel:
        return TRUE;
      }
#  if( SWI_SUPPORT_EXTIO_DLL )
     // not an Audio-I/O-DLL but possibly an ExtIO-DLL ?
     if( pInstData->fExtIOLoaded && (pInstData->m_pExtIO_ShowGUI!=NULL) )
      { return TRUE;
      }
#  endif // SWI_SUPPORT_EXTIO_DLL
   }
  return FALSE;
} // end AIO_Host_HasControlPanel()

//----------------------------------------------------------------------
CPROT BOOL AIO_Host_ControlPanelVisible( // -> TRUE=yes, FALSE=no
       T_AIO_DLLHostInstanceData *pInstData) // [in] DLL-host instance data
{
  if( AIO_Host_HasControlPanel( pInstData ) )
   { return pInstData->fControlPanelVisible;
   }
  else
   { return FALSE;
   }
} // end AIO_Host_ControlPanelVisible()


//----------------------------------------------------------------------
CPROT INT32 AIO_Host_ShowControlPanel( // NON-MODAL (not blocking) !
       T_AIO_DLLHostInstanceData *pInstData, // [in] DLL-host instance data
       HWND   hwndParent,            // [in] handle to the host's window
       HWND   *phwndControlPanel,    // [out] handle of the DLL's control panel
       INT32  iDisplayOptions,       // [in] AIO_DISPLAY_SHOW_WINDOW, AIO_DISPLAY_HIDE_WINDOW, etc
       void  *pvUserDataForCallback,        // [in,optional] address of some user data passed to the callback
       tpCallbackFromDLL pCallbackFunction) // [in,optional] address of a callback function, may be NULL
  // Shows the DLL's own "control panel" box as a non-modal (= non-blocking) window.
  // Certain special DLLs may provide some additonal CONFIGURATION here.
  // Return value: AIO_NO_ERROR, if the control panel could successfully be opened.
  // Note: Don't try to open the control panel multiple times.
{
  if( (pInstData!=NULL) && pInstData->fOpenForBusiness )
   { // Have loaded a valid Audio-I/O DLL, let it show its Control Panel..
     if( pInstData->m_pShowControlPanel!=NULL )
      { // looks like an Audio-I/O-DLL (not an ExtIO-DLL) :
        return (*pInstData->m_pShowControlPanel)(hwndParent,phwndControlPanel,iDisplayOptions,
                                                 pvUserDataForCallback,pCallbackFunction);
        // Hint: For source-level debugging, AND stepping *INTO* the DLL (on a sourcecode level),
        // compile the Audio-I/O-DLL with BCB. There's an extra project file for that:
        //   c:\cbproj\AudioIO\in_AudioIO_BCB4_DONT_USE.bpr  .
        // The IMPLEMENTATION of the first DLL with an own control panel is in
        //   c:\cbproj\AudioIO\aio2winamp.c :: AIO_ShowControlPanel() .
      }
#  if( SWI_SUPPORT_EXTIO_DLL ) // support Winrad's ExtIO-DLLs (here: HOST SIDE) ?
     else // not an Audio-I/O-DLL but possibly an ExtIO-DLL ?
     if( pInstData->fExtIOLoaded && pInstData->fCritterInitialized )
      { // looks like a Winrad-compatible ExtIO-DLL (not DL4YHF's Audio-I/O-DLL)
        switch( iDisplayOptions )
         { case AIO_DISPLAY_SHOW_WINDOW:
              if (pInstData->m_pExtIO_ShowGUI!=NULL)
               {
                 // Not here: EnterCriticalSection( &pInstData->Critter );
                 // (who knows if ShowGUI() is a stupid MODAL DIALOG,
                 //  or always returns immediately ? Not documented anywhere)
                 pInstData->m_pExtIO_ShowGUI();
                 // Bonito's "ExtIO_FiFi.dll" will crash at this point,
                 // somewhere after calling USER32.IsWindowVisible. Useless.
                 // Not here: LeaveCriticalSection( &pInstData->Critter );
                 return AIO_NO_ERROR;
               }
              break;
           case AIO_DISPLAY_HIDE_WINDOW:
              if (pInstData->m_pExtIO_HideGUI!=NULL)
               {
                 DOBINI();
                 EnterCritter( pInstData );
                 DOBINI();
                 pInstData->m_pExtIO_HideGUI();
                 DOBINI();
                 LeaveCritter( pInstData );
                 DOBINI();
                 return AIO_NO_ERROR;
               }
              break;
         } // end switch( iDisplayOptions )
      } // end if( pInstData->fExtIOLoaded )
#  endif // SWI_SUPPORT_EXTIO_DLL
   }
  // Arrived here: Function not supported, or no suitable DLL loaded
  return AIO_ERROR_NOT_IMPLEMENTED;
} // end AIO_Host_ShowControlPanel()   [ interface to the DLL, not the DLL itself ]


//----------------------------------------------------------------------
INT32 AIO_Host_GetOutputParams(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        char *pszAudioStreamID,   // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE
           // (here: optional, use NULL to query info about "the first available source")
        double *pdblSampleRate, // [out] samples per second, "precise", optional
        INT32 *piNumChannels)   // [out] 1=mono, 2=stereo, 4=quadrophonic
   // Called by the application to query the current settings
   //  for a certain "outgoing" audio connection - BEFORE OPENING IT .
   // (*) For an explanation of the "source name", see AIO_Host_OpenAudioOutput() .
{
  INT32 i32Result = AIO_ERROR_DLL_NOT_LOADED;  // negative error code !

  if( !pInstData->fCritterInitialized )
   { return LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
   }

  if( (pInstData!=NULL) && pInstData->fOpenForBusiness && pInstData->fCritterInitialized
    && (pInstData->m_pGetOutputParams!=NULL) )
   {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     i32Result =  (*pInstData->m_pGetOutputParams)(pszAudioStreamID, pdblSampleRate, piNumChannels );
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
   }
  return i32Result;
} // end AIO_GetOutputParams()  [ interface to the DLL, not the DLL itself ]


//----------------------------------------------------------------------
INT32 AIO_Host_GetWinampParams(  // Optional, not required in many cases...
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        double *pdblSampleRate,     // [out] samples per second used by Winamp
        INT32 *piNumChannels,       // [out] 1=mono, 2=stereo   used by Winamp
        INT32 *piWinampInitialized, // [out] 0=winamp not connected to the DLL,  1=winamp connected(*)
        INT32 *piWinampInputPaused) // [out] 0=winamp's input is paused, 1=winamp's input is running
  // (*) "connected" means, Winamp is there, has loaded the plugin DLL,
  //     and has said "hello" by calling the "Init()"-function of the Winamp-plugin.
  //     As soon as Winamp calls the "Quit"-function, the flag gets cleared.
  // All pointers in the argument list are OPTIONAL.
  //  If you don't need some of the above values, just pass a NULL pointer .
{
  INT32 i32Result = AIO_ERROR_DLL_NOT_LOADED;  // negative error code !


  // As a special service, if the DLL is *not* loaded, we set all results
  // to ZERO here (instead of leaving them possibly uninitialized) :
  if(pdblSampleRate)
   { *pdblSampleRate = 0.0; // SharedData.dblSampleRate;
   }
  if(piNumChannels )
   { *piNumChannels = 0;    // SharedData.iNumChannels;
   }
  if(piWinampInitialized)
   { *piWinampInitialized = 0; // SharedData.iWinampInitialized;
   }
  if(piWinampInputPaused)
   { *piWinampInputPaused = 0; // SharedData.iWinampInputPaused;
   }

  if( (pInstData!=NULL) && pInstData->fOpenForBusiness && pInstData->fCritterInitialized
     && pInstData->m_pGetWinampParams )
   {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     i32Result = (*pInstData->m_pGetWinampParams)(pdblSampleRate, piNumChannels,
                                   piWinampInitialized, piWinampInputPaused );
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
   }
  return i32Result;
} // end AIO_Host_GetWinampParams()  [ interface to the DLL, not the DLL itself ]

//----------------------------------------------------------------------
INT32 AIO_Host_OpenAudioOutput( // 'creates' an audio source, and opens it for 'writing'
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        char *pszAudioStreamID,   // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the SOURCE (here: mandatory)
        char *pszDescription,     // [in] a descriptive name which identifies
           // the audio source ("audio producer"). This name may appear
           // on a kind of "patch panel" on the configuration screen of the Audio-I/O DLL .
           // For example, Spectrum Lab will identify itself as "SpectrumLab1"
           // for the first running instance, "SpectrumLab2" for the 2nd, etc .
        double dblSampleRate,   // [in] samples per second, "precise"
        INT32  iNumChannels,    // [in] 1=mono, 2=stereo, 4=quadrophonic
        INT32 iTimeout_ms,      // [in] max timeout in milliseconds for THIS call
        INT32 iOpenOptions )    // [in] bit combination of AIO_OPEN_OPTION_... flags
  // Called from external audio source to open the DLL for "output"
  //   ("output" must seen from that application's point of view) .
  // If this was successfull, it may begin to periodically call
  // AIO_WriteAudioOutput() to send chunks of audio blocks .
  // (*) pszMyName should be a descriptive name
  //     which identifies us for the user. This name may appear
  //     on a kind of "patch panel" in a future version of Audio-I/O .
  //     For example, see AudioIO_Test_Client.c  .
  //
  // Return value:
  //     >= 0 on success ,  in this case it's a "handle" value
  //                        which must be passed to AIO_WriteAudioOutput later.
  //      < 0 on failure (like Winamp's output-plugins) .
  //
  // Note: Having successfully opened the output for "writing"
  //  doesn't mean the destination (like Winamp) is already prepared
  //  so send anything (especially not when streaming MP3 to the web, etc).
  //  That's why we need to call AIO_CanWriteOutput() sometimes .
{  INT32 h;
   if( (pInstData!=NULL) && pInstData->fOpenForBusiness && pInstData->fCritterInitialized
     && pInstData->m_pOpenAudioOutput )
    {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     h = (*pInstData->m_pOpenAudioOutput)(pszAudioStreamID, pszDescription,
                     dblSampleRate, iNumChannels, iTimeout_ms, iOpenOptions);
     // When successfull, the returned value (h) is a NON-NEGATIVE HANDLE .
     // ! Thus, ZERO is a perfectly valid result. For example, in_AudioIO.dll,
     // ! implemented in c:\cbproj\AudioIO\aio2winamp.c, actually uses this
     // ! 'handle' as an ARRAY INDEX into up to eight(?) 'audio sources' :
     // ! unsigned int iAudioSource = (unsigned int)g_iWinampAudioHandle >> 8;

     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
     if( h<0 )
      { pInstData->iAudioOutputHandle = AIO_INVALID_HANDLE;
        return h;
      }
     else
      { pInstData->iAudioOutputHandle = h;
        return h;
      }
   }
  else
   { return AIO_ERROR_DLL_NOT_LOADED;  // negative error code !
   }
} // end AIO_Host_OpenAudioOutput()

//----------------------------------------------------------------------
INT32 AIO_Host_CanWriteOutput(
        T_AIO_DLLHostInstanceData *pInstData) // [in,out] DLL-host instance data
  // Returns number of bytes(!) possible to write at a given time.
  // Never will decrease unless you call Write (or Close)..
  //       like in Winamp's output plugin .
{
  INT32 i32Result = AIO_ERROR_DLL_NOT_LOADED;  // negative error code !

   if( (pInstData!=NULL) && pInstData->fOpenForBusiness && pInstData->fCritterInitialized
     && pInstData->m_pCanWriteOutput )
    { if ( pInstData->iAudioOutputHandle>=0 )
       {
         DOBINI();
         EnterCritter( pInstData );
         DOBINI();
         i32Result = (*pInstData->m_pCanWriteOutput)(pInstData->iAudioOutputHandle);
         DOBINI();
         LeaveCritter( pInstData );
         DOBINI();
       }
      else
       { i32Result = AIO_ERROR_OUTPUT_NOT_OPEN;  // negative error code !
       }
    }
  return i32Result;
} // end AIO_Host_CanWriteOutput()


#if(0) // Removed 2011-07; use AIO_Host_WriteOutputSamplePoints() instead:
//----------------------------------------------------------------------
INT32 AIO_Host_WriteAudioOutput(
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        BYTE *pbSource,          // [in] source buffer (*)
        INT32 iNumBytesToWrite,  // [in] number of BYTES(!) to write
        INT32 iTimeout_ms )      // [in] max timeout in milliseconds, 0=non-blocking
  // Called from external audio source to write data into this plugin's
  // internal audio queue .
  //  (*) Even if this is a BYTE-POINTER, the buffer should be aligned
  //      to 4-byte-boundaries when more than 8 bits per sample are used .
  //      The sample format is the common "PCM-Wave" format :
  //           UNSIGNED integer for 8-bit-per-sample (0..255),
  //           SIGNED integer for everything else (16 bit: -32768...+32767) .
  // Return : >= 0 on success (the number indicates HOW MANY SAMPLES can be
  //                           placed in the buffer if we'd immediately
  //                           call AIO_WriteAudioOutput() again ) .
  //          0 = AIO_NO_ERROR = "nothing written but no error" (?)
  //          < 0  : one of the NEGATIVE error codes defined in AudioIO.H .
  //
{  if( (pInstData!=NULL) && pInstData->fOpenForBusiness && pInstData->m_pOpenAudioOutput )
    { if ( pInstData->iAudioOutputHandle>=0 )
       { return (*pInstData->m_pWriteAudioOutput)(pInstData->iAudioOutputHandle,
                  pbSource, iNumBytesToWrite, iTimeout_ms );
       }
      else
       { return AIO_ERROR_OUTPUT_NOT_OPEN;  // negative error code !
       }
    }
   else
    { return AIO_ERROR_DLL_NOT_LOADED;  // negative error code !
    }
} // end AIO_Host_WriteAudioOutput()
#endif

//----------------------------------------------------------------------
CPROT INT32 AIO_Host_WriteOutputSamplePoints(
        T_AIO_DLLHostInstanceData *pInstData, // [in] DLL-host instance data
        float *pfltSource,        // [in] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points",
                                  //      USUALLY (**) normalized to -1.0 ... +1.0 for full ADC swing
        INT32 iNumSamplePoints,   // [in] number of SAMPLE POINTS(!) to write or send
        INT32 nChannelsPerSample, // [in] number of samples PER SAMPLE POINT
        INT32 iTimeout_ms ,       // [in] max timeout in milliseconds, 0=non-blocking
        T_ChunkInfo *pInChunkInfo,// [in,optional (*)] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h
        INT32 *piHaveWaited_ms)   // [out,optional] "have been waiting here for XX milliseconds"
  // Return : >= 0 on success (the number indicates HOW MANY SAMPLES can be
  //                           placed in the buffer if we'd immediately
  //                           call AIO_WriteAudioOutput() again  .
  //                   This is NOT the "number of samples placed in the output"
  //                   - on a successfull call, that'd be == iNumSamplePoints) .
  //                   Actually, a non-negative return value indicates SUCCESS
  //                   and the number of remaining (unused) entries
  //                   in the recipient's buffer .
  //          < 0  : one of the NEGATIVE error codes defined in AudioIO.H .

  // Unlike the older (and now discarded) 'audio output' functions, this one
  //  supports the 'T_ChunkInfo' struct defined in c:\cbproj\SoundUtl\ChunkInfo.h,
  //  including timestamp, calibrated sample rate, GPS date+time+position, etc.
  //  It also supports multi-channel audio (grouped with MULTIPLE CHANNELS per "sample point") .
  //
  // (*) Passing in the 'T_ChunkInfo' may be optional, but it's recommended
  //     that EACH audio-writer provides at least a 'minimum' T_ChunkInfo
  //     as described in ..\SoundUtl\ChunkInfo.h
  //     (with .dwValidityFlags=0, .dblSampleValueRange=1.0)
  // (**) If the input is NOT normalized to -1.0 ... +1.0 ,
  //     the 'T_ChunkInfo' structure must be passed in,
  //     with pInChunkInfo->dblSampleValueRange specifying the sample value range.
  // All details in AudioIO.c (AudioIO.h contains only an excerpt) .
{
  INT32 i32Result = AIO_ERROR_OUTPUT_NOT_OPEN;  // negative error code !

  if( pInstData!=NULL && pInstData->fOpenForBusiness && pInstData->fCritterInitialized
     && pInstData->iAudioOutputHandle >= 0 ) // note: a 'handle' of ZERO is perfectly valid !
   {
     // If the loaded DLL supports it, use AIO_WriteAudioOutput2() .
     // Otherwise, fall back to the "old" AIO_WriteAudioOutput() without T_ChunkInfo.
     if( pInstData->m_pWriteOutputSamplePoints != NULL )
      {
        DOBINI();
        EnterCritter( pInstData );
        DOBINI();
        i32Result = (*pInstData->m_pWriteOutputSamplePoints)(pInstData->iAudioOutputHandle,
                 pfltSource, iNumSamplePoints, nChannelsPerSample,
                 iTimeout_ms,  // [in] max timeout in milliseconds, 0=non-blocking
                 pInChunkInfo, // [in,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL !
                 piHaveWaited_ms);// [out,optional] "been-waiting-flag" (32-bit integer, NUMBER OF MILLISECONDS)
        DOBINI();
        LeaveCritter( pInstData );
        DOBINI();
      }
     else
      { i32Result = AIO_ERROR_DLL_NOT_LOADED;  // negative error code !
      }
   }
  return i32Result;

} // end AIO_Host_WriteAudioOutput2()


//----------------------------------------------------------------------
INT32 AIO_Host_OpenAudioInput(  // API to open the decive, and begin to read samples from it
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        char *pszAudioStreamID,   // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE
           // (here: optional, use NULL to tap "the first available source")
        char *pszAudioReaderID,   // [in] short but if possible UNIQUE name which identifies
           // the audio reader ("audio consumer"). This name may appear
           // on a kind of "patch panel" on the configuration screen of the Audio-I/O DLL .
           // For example, Spectrum Lab will identify itself as "SpectrumLab1"
           // for the first running instance, "SpectrumLab2" for the 2nd, etc .
        double  dblWantedSampleRate,  // [in(!)] "wanted" sample rate, often IGNORED (reasons below..)
        double *pdblOutSampleRate,    // [out(!)] "realized" sample rate, samples per second,
                  // usually dictated by the FEEDING SOURCE;
                  // the sampling rate is NOT resampled by the DLL !
        INT32 nChannelsPerSample,     // [in] number of channels PER SAMPLE POINT
        INT32 iTimeout_ms ,           // [in] max timeout in milliseconds for THIS call
           // (may have to wait for a short time, for example when...
           //   - the I/O lib needs to communicate with an external device (to open it)
           //   - the I/O lib needs to wait until the 'writer' (=the other side)
           //     has called AIO_Host_OpenAudioOutput() and set the sampling rate, etc
        INT32 iOpenOptions,           // [in] bit combination of AIO_OPEN_OPTION_... flags
        T_ChunkInfo *pOutChunkInfo )  // [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL !
  // Called from the application (like SpecLab) to open the DLL for "input"
  //   ("input" must seen from that application's point of view) .
  // If this was successfull, it may begin to periodically call
  // AIO_ReadInputSamplePoints() to read chunks of audio sample-POINTS .
  // (*) pszAudioSourceName should be a descriptive name
  //     which identifies the SOURCE for the user. This name may appear
  //     on a kind of "patch panel" in a future version of Audio-I/O .
  //     For example, see AudioIO_Test_Client.c  .
  //
  // Return value:
  //     >= 0 on success ,  in this case it MAY be a "handle" value
  //                        which must be passed to AIO_WriteAudioOutput later.
  //      < 0 on failure (like Winamp's output-plugins) .
  //
  // Note: Having successfully opened the output for "reading"
  //  doesn't mean the source (like Winamp) is already prepared
  //  so send anything !
{  INT32 h;
   INT32 i32SampRate_Hz;

  if( pInstData==NULL)
   { return LogCriticalError( AIO_ERROR_NOT_INITIALIZED );  // negative error code !
   }
  if( !pInstData->fCritterInitialized)
   { return LogCriticalError( AIO_ERROR_NOT_INITIALIZED );
   }
  if( !pInstData->fOpenForBusiness )
   { return LogCriticalError( AIO_ERROR_DLL_NOT_LOADED );
   }
  pInstData->dblInputSamplingRate = dblWantedSampleRate; // meaningful default, if the DLL doesn't tell us
  if( pInstData->m_pOpenAudioInput!=NULL )   // looks like an Audio-I/O-DLL (not ExtIO) ...
   {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     h = (*pInstData->m_pOpenAudioInput)(pszAudioStreamID, pszAudioReaderID,
               dblWantedSampleRate/*in*/, pdblOutSampleRate/*out*/,
               nChannelsPerSample, iTimeout_ms, iOpenOptions,
               pOutChunkInfo ); // [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL !
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
     if( h<0 )
      { pInstData->iAudioInputHandle = AIO_INVALID_HANDLE;
        return h;
      }
     else
      { // The handle returned by AIO_OpenAudioInput (implemented in the DLL)
        // must be passed to AIO_Host_ReadInputSamplePoints later, to identify
        // the READER. Remember, the audio-exchange-buffer supports one writer
        // but MULTIPLE READERS. This handle identifies it. But don't make
        // any assumption about what this 'handle' actually is: It's entirely
        // up to the DLL !  For example, see \cbproj\AudioIO\aio2winamp.c .
        // Because the handle MAY(!) be a table index, a 'handle value' of zero
        // is perfectly valid !
        pInstData->iAudioInputHandle = h;
        return h;
      }
   }
# if( SWI_SUPPORT_EXTIO_DLL )   // support I2PHD's ExtIO-DLLs (Winrad-compatible) ?
   else  if( pInstData->fExtIOLoaded && pInstData->fExtIOHWready )
    { // looks like a Winrad-compatible ExtIO-DLL (not DL4YHF's Audio-I/O-DLL) .
      // Details about the CALLING SEQUENCE of all these 'Init', 'Open', and 'Start'-stuff
      //         are in c:\cbproj\ExtIO\ExtIO_Notes_DL4YHF.htm  .
      // fExtIOHWready==TRUE (checked above) means InitHW() was successful .
      if( ( ! pInstData->fHWopened )   // should we "open the hardware" ?
        &&( ! pInstData->fHWstarted) ) // may we call 'OpenHW' at the moment ?
       { // Note: SDRsharp also refuses to call OpenHW if already 'started' !
         if(  pInstData->m_pExtIO_OpenHW != NULL )
          {
            DOBINI();
            // Examined pInstData->Critter at this point:
            //  pInstData->Critter = { :022C2A88, -1, 0, NULL, NULL, 0 }
            EnterCritter( pInstData );
            // Examined pInstData->Critter at this point:
            //  pInstData->Critter = { :022C2A88, 0, 1, :00000F64, NULL, 0 }
            DOBINI();
            pInstData->fHWopened = pInstData->m_pExtIO_OpenHW();
            // (Some developers don't seem to give a damn about the BOOLEAN type.
            //  Often got here with trash like pInstData->fHWopened = 437170625 ,
            //  for example with BorIP_und_HDSDR\ExtIO_USRP.dll  )
            // 2012-11-19: Plausible results with ExtIO_RTL.dll by Jose Araujo :
            //  pInstData->fHWopened = 1 == TRUE .
            DOBINI();
            // Examined pInstData->Critter at this point:
            //  pInstData->Critter = { :022C2A88, 0, 1, :00000F64, NULL, 0 }
            LeaveCritter( pInstData );
            // Examined pInstData->Critter at this point:
            //  pInstData->Critter = { :022C2A88, -1, 0, NULL, NULL, 0 }
            DOBINI();
            //  (this already opened the control panel for the SDR-14/IQ. Why ?)
            // Details about OpenHW() only in Winrad_Extio.PDF !
            // the result may be FALSE, which means
            //  "some error occurred, the external HW cannot be controlled by the DLL"
            if( ! pInstData->fHWopened )
             {
#             ifdef UseUtility1
               DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, UTL_USE_CURRENT_TIME,
                 "ExtIO-DLL: OpenHW() failed !" );
#             endif
               return AIO_ERROR_INPUT_NOT_OPEN;
             }
          }
       } // end if < ExtIO-Hardware not "opened" yet ? >

           // CAUTION ! Some ExtIO-DLLs must be treated
           // like raw eggs... It's unknown if
           // see notes / descriptions of calling sequence
           //  in >>>  c:\cbproj\ExtIO\ExtIO_Notes_DL4YHF.htm <<<  !
      // To allocate a sample-buffer with suitable size, the sampling rate
      // must be known in advance.
      // Unfortunately, GetHWSR() is NOT mandatory, so we may have to guess !
      if( pInstData->m_pExtIO_GetHWSR != NULL )
       {
         DOBINI();
         // Examined pInstData->Critter at this point:
         //  pInstData->Critter = { :022C2A88, -1, 0, NULL, NULL, 0 }
         EnterCritter( pInstData );  // 2012-11-19 : Crashed HERE ?!
         // Examined pInstData->Critter at this point:
         //  pInstData->Critter = { :022C2A88, 0, 1, :00000AFC, NULL, 0 }
         DOBINI();
         i32SampRate_Hz = pInstData->m_pExtIO_GetHWSR();
         // With ExtIO_SDR14.DLL, got here with i32SampRate_Hz = 158730 .
         // With BorIP_und_HDSDR\ExtIO_USRP.DLL, got here with SR = 1000000 ,
         //      which is nonsense, at least for the RTL-SDR there's no such sampling rate.
         // With ExtIO_RTL.DLL, got here with i32SampRate_Hz = 2400000 (2.4 MSamples!)
         //      which is more than many 'old workhorses' can handle,
         //      but since there is no way to retrieve all VALID hardware sampling rates
         //      from the DLL, we'll have to accept that for the moment.
         DOBINI();
         // Examined pInstData->Critter at this point:
         //  pInstData->Critter
         LeaveCritter( pInstData );
         // Examined pInstData->Critter at this point:
         //  pInstData->Critter
         DOBINI();
         pInstData->dblInputSamplingRate = i32SampRate_Hz;
       }
      // For simplicity: Make the buffer large enough for ONE second of audio:
      AIO_SampleBuffer_Init( &pInstData->buf,
          (long)pInstData->dblInputSamplingRate, // [in] nSamplePointsPerBuffer
                                            2 ); // [in] nChannelsPerSample

      // Besides 'OpenHW' without parameters, there may also be 'StartHW',
      // which SETS the external radio's VFO frequency (center frequency).
      // Details, as usual, in >>>  c:\cbproj\ExtIO\ExtIO_Notes_DL4YHF.htm <<< !
      //  > In WinradSources\main.cpp, StartHW() is *only* called
      //  > from TMainForm::StartStopClick(), when 'starting' (i.e. when
      //  > the user clicks winrad's START/STOP button),
      //  > after creating the processing threads, ....
      // Because AudioIO.c originally didn't care about RADIO FREQUENCIES
      // and "VFO SETTINGS", there was no way to retrieve the "frequency"
      // required for StartHW() here.
      // This is more or less what the SDR14-control-DLL calls "NCO" frequency
      // and sets through CSDR14Ctrl::SetNCOfrequency() .
      if( pInstData->m_pExtIO_StartHW != NULL ) // note: don't treat POINTERS like a BOOLEAN value !
       {
         DOBINI();
         Sleep(500);   // wait some time before calling 'StartHW' ?!
         EnterCritter( pInstData );
         DOBINI();
         pInstData->nIQPairsPerExtIoCallback = pInstData->m_pExtIO_StartHW( pInstData->i32ExternalVfoFrequency ); // here in AIO_Host_OpenAudioInput()
         // See notes on the "PLL not locked"-problem in ExtIO_RTL.cpp !
         // With ExtIO_RTL.DLL loaded in Spectrum Lab (a GUI application),
         //    got here with pInstData->nIQPairsPerExtIoCallback = 32768 (!),
         //    which means at a sampling rate of 2.4 MSamples/sec,
         //    the callback will phone home (call back) every 13 milliseconds .
         // With the same ExtIO_RTL.DLL loaded in the 'Audio-IO-Test-Client'
         //    (a boring CONSOLE application), got here with ..nIQPairs.. = 0 (!).
         //    despite using the same input parameters as in SpecLab. WTF... ?
         DOBINI();
         LeaveCritter( pInstData );
         DOBINI();
         // > Returnvalue:
         // > An integer specifying how many I/Q pairs are returned
         // > by the DLL each time the callback function is invoked (later).
         // > Thisinformationisusedofcourseonlywhentheinputdata
         // > arenotcomingfromthesoundcard,butthroughthecallbackdevice.
         // > Ifthenumberisnegative,thatmeansthatanerrorhasoccurred,
         // > (the host)interruptsthestartingprocessandreturnstotheidlestatus.
         // > ThenumberofI/Qpairsmustbeatleast512,
         // > oranintegermultipleofthatvalue .
         if( pInstData->nIQPairsPerExtIoCallback < 0 )
          {
#          ifdef UseUtility1
            DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, UTL_USE_CURRENT_TIME,
              "ExtIO-DLL: StartHW() failed !" );
#          endif
            return AIO_ERROR_INPUT_NOT_OPEN;
          }
         else
          { pInstData->fHWstarted = TRUE;
          }
       }

      // Added 2012-10-28 to support FiFi-SDR: Some ExtIO-DLLs (ExtIO_Si570.dll)
      //  rely on the ordinary soundcard as an input devices. Check this:
      if( pInstData->i32ExtIOHWtype == EXTIO_HWT_Soundcard /*4*/ )
       { // > Theaudiodataarereturnedviathesoundcardmanagedbythe host.
         // > TheexternalhardwarejustcontrolstheLO,
         // > andpossiblyapreselector,underDLLcontrol.
         // In this case, AudioIO.c is considered the 'host' (for the DLL),
         // so the ugly soundcard processing shall be performed here.
         // In Spectrum Lab, there is already another software module for the
         // soundcard-input processing (Sound.cpp) so we keep the soundcard-stuff
         // out of here. Instead, the soundcard processing remains where it was:
         // in C:\cbproj\SoundUtl\Sound.cpp : CSound::InOpen() .
         //
       } // end if( pInstData->i32ExtIOHWtype == EXTIO_HWT_Soundcard /*4*/ )

      if( pdblOutSampleRate != NULL )
       { *pdblOutSampleRate = pInstData->dblInputSamplingRate; // maybe from GetHWSR(), maybe not..
       }
      return AIO_NO_ERROR;
    }
# endif // SWI_SUPPORT_EXTIO_DLL ?
   else
    { return AIO_ERROR_DLL_NOT_LOADED;  // negative error code !
    }
} // end AIO_Host_OpenAudioInput()


//----------------------------------------------------------------------
CPROT int AIO_Host_ReadInputSamplePoints( // returns the number of sample-POINTS, or a negative error code
        T_AIO_DLLHostInstanceData *pInstData, // [in,out] DLL-host instance data
        float *pfltDest,        // [out] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
        int   iNumSamplePoints, // [in] number of SAMPLE POINTS(!) to read
        int   nChannelsPerSample, // [in] number of samples PER SAMPLE POINT (may be less than provided by source)
        int   iTimeout_ms ,     // [in] max timeout in milliseconds, 0=non-blocking
        T_ChunkInfo *pOutChunkInfo, // [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h
        int   *piHaveWaited_ms) // [out,optional] "have been waiting here for XX milliseconds"
  //
  // Called from the application (for example SpecLab) to read the next chunk
  //  of data from the "audio I/O DLL"s internal audio queue .
  // Return : >= 0 on success (==NUMBER OF SAMPLE POINTS acually read,
  //                           and placed in the caller's buffer ) .
  //          < 0  : one of the NEGATIVE error codes defined in AudioIO.H .
{
  long n;
  int  iResult;
  int  waited_ms = 0;
  if( pInstData==NULL)
   { return AIO_ERROR_NOT_INITIALIZED;  // negative error codes...
   }
  if( !pInstData->fCritterInitialized)
   { return AIO_ERROR_NOT_INITIALIZED;
   }
  if( !pInstData->fOpenForBusiness )
   { return AIO_ERROR_DLL_NOT_LOADED;
   }
#if( SWI_SUPPORT_EXTIO_DLL )   // support Winrad-compatible ExtIO-DLLs ?
  // (needs no audio-input-handle, so check this before other stuff..)
  if( pInstData->fExtIOLoaded )
   { // Is the hosted ExtIO-DLL really open for business ?
     if( ! pInstData->fHWopened )
      { return AIO_ERROR_INPUT_NOT_OPEN;
      }

     // If necessary, WAIT FOR samples being filled in by the ExtIO-DLL;
     // and if there's enough, retrieve as many samples from the lock-free
     // buffer as the caller wants.
     while( iTimeout_ms > 0 )  // WAIT for the data if nothing available ?
      { n = AIO_SampleBuffer_GetNumSamplesBuffered( &pInstData->buf );
        if( n>=iNumSamplePoints )  // ok, enough data buffered, don't wait any further
         { break;
         }
        Sleep(20);  // wait for some other thread to deliver more samples
        iTimeout_ms -= 20;
        waited_ms += 20;
      } // end while( iTimeout_ms > 0 )
     if( piHaveWaited_ms != NULL )
      { *piHaveWaited_ms = waited_ms;
      }
     return AIO_SampleBuffer_ReadSamples( &pInstData->buf, pfltDest,
                             iNumSamplePoints, nChannelsPerSample );
   }
#endif // SWI_SUPPORT_EXTIO_DLL ?

  if(  pInstData->iAudioInputHandle<0 )
   { iResult = AIO_ERROR_INPUT_NOT_OPEN;  // negative error code !
   }
  else if( pInstData->m_pReadInputSamplePoints )
   {
     DOBINI();
     EnterCritter( pInstData );
     DOBINI();
     iResult = (*pInstData->m_pReadInputSamplePoints)(pInstData->iAudioInputHandle,
               pfltDest, iNumSamplePoints, nChannelsPerSample, iTimeout_ms,
               pOutChunkInfo, piHaveWaited_ms );
     // When using in_AudioIO.dll, the above call will be made into
     //      c:\cbproj\AudioIO\aio2winamp.c :: AIO_ReadInputSamplePoints() .
     // Borland C++ Builder allows single-stepping on the sourcecode level even INTO THE DLL,
     // because the DLL itself was built with BCB, and contains debug info.
     DOBINI();
     LeaveCritter( pInstData );
     DOBINI();
   }
  else
   { iResult = AIO_ERROR_DLL_NOT_LOADED;  // negative error code !
   }
  return iResult;
} // end AIO_Host_ReadInputSamplePoints()

//----------------------------------------------------------------------
void AIO_Host_CloseAudioOutput(
      T_AIO_DLLHostInstanceData *pInstData) // [in,out] DLL-host instance data
  // Polite applications call this function when they have nothing more to send,
  // or when they shut down.
{
  if( pInstData!=NULL )
   { if( pInstData->fOpenForBusiness && pInstData->fCritterInitialized
       && (pInstData->m_pCloseAudioOutput) && (pInstData->iAudioOutputHandle>=0) )
      {
        DOBINI();
        EnterCritter( pInstData );
        DOBINI();
        (*pInstData->m_pCloseAudioOutput)(pInstData->iAudioOutputHandle);
        DOBINI();
        LeaveCritter( pInstData );
        DOBINI();
      }
     pInstData->iAudioOutputHandle = AIO_INVALID_HANDLE; // negative "handle" value
   }
}


//----------------------------------------------------------------------
CPROT void AIO_Host_CloseAudioInput(
       T_AIO_DLLHostInstanceData *pInstData) // [in,out] DLL-host instance data
  // Polite applications call this function when they shut down.
{
  if( pInstData!=NULL  &&  pInstData->fCritterInitialized )
   { if( pInstData->fOpenForBusiness
       && (pInstData->m_pCloseAudioInput) && (pInstData->iAudioInputHandle>=0) )
      {
        DOBINI();
        EnterCritter( pInstData );
        DOBINI();
#      ifdef UseUtility1
        UTL_WriteRunLogEntry( "AIO_Host: Calling 'CloseAudioInput' in the DLL" );
#      endif
        (*pInstData->m_pCloseAudioInput)(pInstData->iAudioInputHandle);
#      ifdef UseUtility1
        UTL_WriteRunLogEntry( "AIO_Host: Survived 'CloseAudioInput' in the DLL" );
#      endif
        DOBINI();
        LeaveCritter( pInstData );
        DOBINI();
      }
     pInstData->iAudioInputHandle = AIO_INVALID_HANDLE; // negative "handle" value

#   if( SWI_SUPPORT_EXTIO_DLL )   // support Winrad-compatible ExtIO-DLLs ?
     if( pInstData->fExtIOLoaded && (pInstData->m_pExtIO_StopHW!=NULL) )
      {
       // About StopHW , from the ExtIO-specification:
       // > CalledbyWinradeachtimetheuserpressestheStopbutton
       // > ontheWinradmainscreen.Itcanbeusedby the DLL
       // > for whatever task (..) .
       // > IftheexternalHWsendstheaudiodataviatheUSBport,
       // >oranyotherhardwareportmanagedbytheDLL,
       // >   whenthisentryiscalled,theHWshouldbecommanded
       // >   bytheDLLtostopsendingdata.
       if( pInstData->fHWstarted )   // only call StopHW when really started ?
        {
          DOBINI();
          EnterCritter( pInstData );
          DOBINI();
          pInstData->fHWstarted = FALSE;
#        ifdef UseUtility1
          UTL_WriteRunLogEntry( "AIO_Host: Calling 'StopHW' in the ExtIO-DLL" );
#        endif
          pInstData->m_pExtIO_StopHW();
#        ifdef UseUtility1
          UTL_WriteRunLogEntry( "AIO_Host: Survived 'StopHW' in the ExtIO-DLL" );
#        endif
          DOBINI();
          LeaveCritter( pInstData );
          DOBINI();
          // 2012-10-26, in ExtIO_RTLSDR.dll : Crash, bang, boom !  Why ?
          // somewhere after ntdll.RtlActivateActivationContextUnsafeFast() ,
          //        ntdll.NtWaitForSingleObject(),  ntdll.KiFastSystemCall(),
          //        ntdll.RtlDeactivateActivationContextUnsafeFast() ,
          //        KERNEL32.CloseHandle(), ntdll.ZwClose(),
          //        ntdll.KiFastSystemCall(),  ntdll.RtlRaiseException() .
        }
       // Note: Do NOT call ExtIO's "CloseHW" here yet;
       //       Winrad only seems to do that if the ExtIO-DLL is not required
       //       anymore, because CloseHW()...
       //   >iscalledbyWinradwhentheUserindicatesthatthecontrol
       //   > oftheexternalHWisnolongerneededorwanted.  ("Select Input")
      }
     pInstData->fHWstarted = FALSE;
#   endif // SWI_SUPPORT_EXTIO_DLL ?
   }
} // end AIO_Host_CloseAudioInput()



/* EOF <AudioIO.C >.  Leave an empty line after this for certain compilers ! */






