//---------------------------------------------------------------------------
//
// File  : ?\cbproj\YHF_Tools\wsock.c
// Author: Wolfgang Buescher (DL4YHF)
// Date  : 2005-06-10
// Compiler: Borland C++Builder V4
//
// Description:
//  Wrapper between application and SOCKET SERVICES to communicate via IP
//    ( UDP, or maybe TCP ) .
//  May be adapted for a MICROCONTROLLER one day, so use plain "C" only,
//  and avoid the necessity to include WINSOCK.H in the application !
// Notes:
//   - No C++-stuff (may have to implement this on a microcontroller) ,
//     though the module may be included from C++ modules .
//   - No VCL-stuff, no MFC, no CLX, no other tools .
//   - No dynamic memory (leave that to the application - we may not have it)
//   - No other flashy soapy gimmicks too ..
//   - this module is MAY be used for the little HTTP server inside SpecLab .
//
//---------------------------------------------------------------------------

#ifdef __WIN32__
 #include <windows.h>
 #include <winsock.h>
 // #include <winsock2.h>  // winsock2.h is about 3 times larger :-(
#endif // __WINDOWS__

#include <stdio.h>

#include "wsock.h" // old wrapper for some "winsock" services, located in YHF_Tools



//---------------------------------------------------------------------------
// Internal defines (shall not be placed in WSOCK.H because of OS-dependency)
//---------------------------------------------------------------------------


// Internal variables
char    WSOCK_sz80CurrentAction[84] = "";
HWND    WSOCK_hwndMyWindowHandle = 0;  // WINDOW HANDLE for messages from WINSOCK.DLL
BOOLEAN WSOCK_fStarted = FALSE;
BOOLEAN WSOCK_fLogEvents = FALSE;   // log communication events (connect, disconnect,...) ?
BOOLEAN WSOCK_fLogWinMsgs= FALSE;   // log windows messages (sent by WINSOCK) ?
BOOLEAN WSOCK_fLogErrors = TRUE;    // log error messages ?
int     WSOCK_iLastWSError   = 0;
WSOCKET WSOCK_ServerListenerSocket = WSOCKET_INVALID;
T_WSOCK_Client WSOCK_Clients[WSOCK_MAX_CLIENTS];
T_WSOCK_Server WSOCK_Servers[WSOCK_MAX_SERVERS];


//---------------------------------------------------------------------------
// internal forward declarations
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Non-application-specific stuff which may go into an own module one fine day
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
BOOLEAN WSOCK_Init( HWND hwndMainWindow )
   // Should be called ONCE before using the "WSOCK" interface
{
 WSADATA info;
 int i;

   WSOCK_hwndMyWindowHandle = hwndMainWindow;


   if( WSOCK_fStarted )
     { // MessageBox(NULL, "WinSock already initialized.", "Winsock Test", MB_OK);
       return TRUE;
     }

   // Clear a few internal vars in WoBu's "WSOCK"-interface module:
   WSOCK_ServerListenerSocket = INVALID_SOCKET;
   for( i=0; i<WSOCK_MAX_CLIENTS; ++i )
    {
      memset(  &WSOCK_Clients[i], 0, sizeof(T_WSOCK_Client) ) ;
      WSOCK_Clients[i].socket = INVALID_SOCKET;
    }
   for( i=0; i<WSOCK_MAX_SERVERS; ++i )
    {
      memset(  &WSOCK_Servers[i], 0, sizeof(T_WSOCK_Server) ) ;
      WSOCK_Servers[i].peer_socket = INVALID_SOCKET;
    }

   // ex: if ( WSAStartup( MAKEWORD(1,1), &info) != 0)
   // Strange message from BCB V4: "Suggest parentheses to clarify precedence", why ?!?!
   // Seemed to be related with the shitty "MAKEWORD" macro; defined in windef.h as:
   //   #define MAKEWORD(a, b)      ((WORD)(((BYTE)(a)) | ((WORD)((BYTE)(b))) << 8))
   if ( (WSAStartup( 0x0101, &info)) != 0)
    { // MessageBox(NULL, "Cannot initialize WinSock!", "WSAStartup", MB_OK);
      return FALSE;
    }
   else
    {
      WSOCK_fStarted=TRUE;
      // Mem_Messages->Lines->Add(" WSAStartup successfull. ");
      // UpdateMyAddressInfo();
      return TRUE;
    }
} // end WSOCK_Init()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
CPROT void WSOCK_Exit(void)   // cleans up.  Call upon closure of application.
{
  int i;

  // If active, tell WINSOCK not to send us windows messages any longer:
  // > To cancel all notification (that is, to indicate that Windows Sockets
  // > should send no further messages related to network events on the socket)
  // > lEvent should be set to zero.
  if( WSOCK_hwndMyWindowHandle != 0 )
   {
     for( i=0; i<WSOCK_MAX_SERVERS; ++i )
      {
        if( WSOCK_Servers[i].peer_socket != INVALID_SOCKET )
         { WSOCK_Disconnect( &WSOCK_Servers[i].peer_socket );
         }
      }
     if( WSOCK_ServerListenerSocket != INVALID_SOCKET )
         WSAAsyncSelect(WSOCK_ServerListenerSocket, WSOCK_hwndMyWindowHandle, 0, 0);
     for( i=0; i<WSOCK_MAX_CLIENTS; ++i )
      {
        if( WSOCK_Clients[i].socket != INVALID_SOCKET )
         { WSOCK_Disconnect( &WSOCK_Clients[i].socket );
         }
      }
     Sleep(50);  // not sure if this helps to clear the message queue..
   }
   WSOCK_Disconnect( &WSOCK_ServerListenerSocket );   // closesocket(), etc etc etc
   WSACleanup();
} // end WSOCK_Exit()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
BOOLEAN WSOCK_GetOwnName( char *pszMyName, int iMaxLen )
{
  // If no error occurs, gethostname returns zero. Otherwise, it returns
  // SOCKET_ERROR and a specific error code can be retrieved by calling WSAGetLastError.
  if( gethostname(pszMyName, iMaxLen ) == SOCKET_ERROR)  /* who are we ? */
   {
     WSOCK_iLastWSError = WSAGetLastError();
     pszMyName[0] = 0;
     return FALSE;
   }
  return TRUE;
} // end WSOCK_GetOwnName()

//---------------------------------------------------------------------------
BOOLEAN WSOCK_GetOwnIPaddress(char *psz20MyIPaddressAsString )
{
  char   sz80MyName[84];
  struct hostent *hp;
  struct sockaddr_in sa;


  if( WSOCK_GetOwnName( sz80MyName, 80  ) )
   {
     hp = gethostbyname(sz80MyName);  /* get our address info (doesn't require DNS) */
     // Remarks
     //
     // gethostbyname returns a pointer to a hostent structure. The contents
     // of this structure correspond to the hostname name.
     // The pointer which is returned points to a structure
     // which is allocated by Windows Sockets.
     // The application must never attempt to modify this structure
     // or to free any of its components. Furthermore, only one copy
     // of this structure is allocated per thread, and so the application
     // should copy any information which it needs before issuing
     // any other Windows Sockets function calls.
     //
     // gethostbyname does not resolve IP address strings passed to it.
     //  Such a request is treated exactly as if an unknown host name were passed.
     //  An application with an IP address string to resolve should use inet_addr
     //  to convert the string to an IP address, then gethostbyaddr to obtain the
     //  hostent structure.
     // gethostbyname will resolve the string returned by a successful call to gethostname.
     if (hp == NULL)                             /* we don't exist !? */
      { WSOCK_iLastWSError = WSAGetLastError();
        WSOCK_LogError( "GetOwnIPaddress failed", WSOCK_iLastWSError );
      }
     else
      {
        memset(&sa, 0, sizeof(struct sockaddr_in)); /* clear address */
        memcpy((char *)&sa.sin_addr, hp->h_addr, hp->h_length);   /* copy address info */
        sa.sin_family = hp->h_addrtype; /* retrieve (IP-) address as string... */
        strcpy( psz20MyIPaddressAsString, inet_ntoa(sa.sin_addr));
        return TRUE;
      } // end if (WSOCK_fStarted)
   }
  psz20MyIPaddressAsString[0] = '\0';
  return FALSE;
} // end WSOCK_GetOwnIPaddress()

//---------------------------------------------------------------------------
BOOLEAN WSOCK_GetIPaddressByName(
          char *pszHisName,                 // [in]  something like "Penti4"
          char *psz20HisIPaddressAsString ) // [out] something like "192.168.0.200"
{
  struct sockaddr_in sa;
  struct hostent *hp;

  hp = gethostbyname(pszHisName);           /* get HIS address info */
  if (hp == NULL)                           /* HE doen't exist !    */
   { WSOCK_iLastWSError = WSAGetLastError();
     WSOCK_LogError( "gethostbyname failed", WSOCK_iLastWSError );
     psz20HisIPaddressAsString[0] = '\0';
     return FALSE;
   }
  else
   {
     memset(&sa, 0, sizeof(struct sockaddr_in)); /* clear address */
     memcpy((char *)&sa.sin_addr, hp->h_addr, hp->h_length);   /* copy address info */
     sa.sin_family = hp->h_addrtype; /* retrieve (IP-) address as string... */
     strcpy( psz20HisIPaddressAsString, inet_ntoa(sa.sin_addr));
     return TRUE;
   }
} // end WSOCK_GetIPaddressByName()


//---------------------------------------------------------------------------
BOOLEAN WSOCK_SRV_EstablishListener(WSOCKET *pSocket, unsigned short portnum)
  // SERVER SIDE: Code to establish a socket ("plug in the phone") .
  //              Based on Figure 1 from "Winsock Quick and Dirty Primer" .
  //
{ char   myname[256];
  SOCKET s;
  struct sockaddr_in sa;
  struct hostent *hp;
  int    iError;

  memset(&sa, 0, sizeof(struct sockaddr_in)); /* clear our address    */
  gethostname(myname, sizeof(myname));        /* who are we?          */
  hp = gethostbyname(myname);  /* get our address info (doesn't require DNS) */
  if (hp == NULL)                             /* we don't exist !?    */
   { WSOCK_iLastWSError = WSAGetLastError();
     WSOCK_LogError( "EstablishListener: GetHostByName failed", WSOCK_iLastWSError );
     *pSocket = INVALID_SOCKET;
     return FALSE;
   }
  sa.sin_family = hp->h_addrtype;             /* this is our host address */
  sa.sin_port = htons(portnum);               /* this is our port number */
  s = socket(AF_INET, SOCK_STREAM, 0);        /* create the socket */
  if (s == INVALID_SOCKET)
   { WSOCK_iLastWSError = WSAGetLastError();
     WSOCK_LogError( "EstablishListener: Creation of socket failed", WSOCK_iLastWSError );
     *pSocket = INVALID_SOCKET;
     return FALSE;
   }

  //  bind the socket to the internet address  ...
  //
  // bind( SOCKET s, const struct sockaddr FAR*  name, int namelen ) :
  // This routine is used on an unconnected connectionless or connection-oriented
  // socket, before subsequent connects or listens. When a socket is created
  // with socket, it exists in a name space (address family), but it has
  // no name assigned. bind establishes the local association of the socket
  // by assigning a local name to an unnamed socket.
  // As an example, in the Internet address family, a name consists of three parts:
  // the address family, a host address, and a port number which identifies the
  // application. In Windows Sockets 2, the name parameter is not strictly
  // interpreted as a pointer to a "sockaddr" structure. It is cast this way
  // for Windows Sockets compatibility. Service Providers are free to regard
  // it as a pointer to a block of memory of size namelen. The first two bytes
  // in this block (corresponding to "sa_family" in the "sockaddr" declaration)
  // must contain the address family that was used to create the socket.
  // Otherwise, an error WSAEFAULT will occur.
  //
  // If an application does not care what local address is assigned to it,
  // it can specify the manifest constant value ADDR_ANY for the sa_data
  // field of the name parameter. This allows the underlying service provider
  // to use any appropriate network address, potentially simplifying application
  // programming in the presence of multihomed hosts (that is, hosts that have
  // more than one network interface and address). For TCP/IP, if the port
  // is specified as zero, the service provider will assign a unique port
  // to the application with a value between 1024 and 5000.
  // The application can use getsockname after bind to learn the address
  // and the port that has been assigned to it, but note that if the Internet
  // address is equal to INADDR_ANY, getsockname will not necessarily be able
  // to supply the address until the socket is connected, since several addresses
  // can be valid if the host is multihomed.
  if ( bind(s, (struct sockaddr *)&sa, sizeof(struct sockaddr_in)) == SOCKET_ERROR)
   {
     WSOCK_iLastWSError = WSAGetLastError();
     WSOCK_LogError( "EstablishListener: bind() failed", WSOCK_iLastWSError );
     closesocket(s);
     *pSocket = INVALID_SOCKET;
     return FALSE;
   }
  listen(s, 3);                               /* max # of queued connects */

  // Make the socket asynchronous so we get a message
  //   whenever there's data waiting on the socket (instead of BLOCKING CALLS)
  // Note:  WSOCK_hwndMyWindowHandle must be set to the application's "window handle"
  //        before getting here !
  iError = WSAAsyncSelect(  // see Win32 Programmer's reference ...
         s,                 // SOCKET s,   [in] descriptor identifying the socket for which event notification is required
         WSOCK_hwndMyWindowHandle,   // HWND hWnd,  [in] handle identifying the window which should receive a message when a network event occurs
         WM_WINSOCK_SRV,       // unsigned int wMsg, [in] message to be received when a network event occurs
         FD_ACCEPT | FD_READ | FD_WRITE | FD_CONNECT | FD_CLOSE ); // long lEvent, [in] bitmask which specifies a combination of network events in which the application is interested
  if( iError != 0 )
   { // The return value is zero if the application's declaration of interest
     // in the network event set was successful.
     // Otherwise, the value SOCKET_ERROR is returned, and a specific error number
     // can be retrieved by calling WSAGetLastError.
     WSOCK_iLastWSError = WSAGetLastError();
     WSOCK_LogError( "EstablishListener: AsyncSelect failed", WSOCK_iLastWSError );
     closesocket(s);
     *pSocket = INVALID_SOCKET;
     return FALSE;

   }

  *pSocket = s;   // hooray
  return TRUE;

}

//---------------------------------------------------------------------------
BOOLEAN WSOCK_CLI_BeginConnect(
       T_WSOCK_Client *pClient, // [out] contains "socket", etc
       char *hostname_or_addr,         // [in] IP address or hostname
       unsigned short wHisPortNumber)  // [in]
  // CLIENT SIDE: Sets up a connection using plain socket services  .
  //  Once based on Figure 3 from "Winsock Quick and Dirty Primer" ("call_socket"),
  //  and some other tricks found on the web .
  // Return value : a SOCKET for the windows variant,
  //                or a kind of "handle" for the planned microcontroller variant .
{ struct sockaddr_in sa;
  struct hostent     *hp;
  short  hostent_addrtype;
  SOCKET s;
  int    iResult, iError;
  char   *cp;

  if( pClient==NULL )
     return FALSE;


  // What is the first argument; a NAME or an ADDRESS ?
  // Note: gethostbyname does not resolve IP address strings passed to it.
  // Such a request is treated exactly as if an unknown host name were passed.
  // An application with an IP address string to resolve should use inet_addr()
  // to convert the string to an IP address,
  // then gethostbyaddr() to obtain the hostent structure.
  cp = (char*)hostname_or_addr;
  while(*cp>='0' && *cp<='9') ++cp;
  if( *cp=='.' && cp>hostname_or_addr ) // something like '192.' .. must be an IP address !
   { DWORD dwIPaddr;
     dwIPaddr = inet_addr( hostname_or_addr );
     memset(&sa,0,sizeof(sa));          // clear entire 'sockaddr_in' structure
     // In the Internet address family, a name consists of three parts:
     //   the address family, a host address, and a port number which identifies the application
     memcpy((char *)&sa.sin_addr, &dwIPaddr, 4);   /* set address */
     sa.sin_family = AF_INET;    // address family for internetwork: UDP, TCP, etc.
     sa.sin_port = htons((u_short)wHisPortNumber);
     hostent_addrtype = AF_INET;
   }
  else // everything else will hopefully be resolved by a DNS lookup:
   {
     hp = gethostbyname(hostname_or_addr);
     if (hp == NULL) /* we don't know who this host is ? (or the trivial "not-initialized"-error) */
      { WSOCK_iLastWSError = WSAGetLastError();
        pClient->socket = INVALID_SOCKET;
        pClient->iConnState = WSOCK_CS_DISCONNECTED;
        return FALSE;
      }
     memset(&sa,0,sizeof(sa));
     // set address (usually 4 byte, for example { 192,168,0,200 } ) :
     memcpy((char *)&sa.sin_addr, hp->h_addr, hp->h_length);
     sa.sin_family = hp->h_addrtype; // for example, 2 = AF_INET
     sa.sin_port = htons((u_short)wHisPortNumber);
     hostent_addrtype = hp->h_addrtype;
   }

  // Create a stream socket which is bound to a specific service provider
  //  Note: For hostent_addrtype = 2 = AF_INET, this will be an IP connection :
  //        with SOCK_STREAM -> TCP (sequenced, reliable, two-way, connection-based byte streams)
  //        with SOCK_DGRAM  -> UDP (connectionsless, unreliable buffers sent as small datagrams)
  s = socket(
        hostent_addrtype, // int af,   [in] An address family specification
        SOCK_STREAM,      // int type, [in] A type specification for the new socket
        0);               // int protocol [in] Particular protocol, specific to the indicated address family

  if (s == INVALID_SOCKET)
   { WSOCK_iLastWSError = WSAGetLastError();
     WSOCK_LogError( "ConnectToHost: Failed to create socket", WSOCK_iLastWSError );
     pClient->socket = INVALID_SOCKET;
     pClient->iConnState = WSOCK_CS_DISCONNECTED;
     return FALSE;
   }

  // Make the socket "asynchronous" (non-blocking) so we get a message
  //   whenever there's data waiting on the socket (instead of BLOCKING CALLS)
  iResult = WSAAsyncSelect(  // see Win32 Programmer's reference ...
    s,              // SOCKET s,   [in] descriptor identifying the socket for which event notification is required
    WSOCK_hwndMyWindowHandle,// HWND hWnd,  [in] handle identifying the window which should receive a message when a network event occurs
    WM_WINSOCK_CLI, // unsigned int wMsg, [in] message to be received when a network event occurs
    FD_READ | FD_WRITE | FD_CONNECT | FD_CLOSE);  // long lEvent, [in] bitmask which specifies a combination of network events in which the application is interested
  // > WSAAsyncSelect() is used to request that the Windows Sockets DLL
  // > should send a message to the window hWnd whenever it detects
  // > any of the network events specified by the lEvent parameter.
  // > The message which should be sent is specified by the wMsg parameter.
  // > The socket for which notification is required is identified by s.
  // > This function automatically sets socket s to nonblocking mode,
  // > regardless of the value of lEvent.
  if( iResult == SOCKET_ERROR)
   { WSOCK_iLastWSError = WSAGetLastError();
     WSOCK_LogError( "ConnectToHost: WSAAsyncSelect failed", WSOCK_iLastWSError );
     closesocket(s);
     pClient->socket = INVALID_SOCKET;
     pClient->iConnState = WSOCK_CS_DISCONNECTED;
     return FALSE;
   }

  // Begin the connection attempt.  This will almost certainly
  // not complete immediately.
  if( connect(s,
      (struct sockaddr*)&sa, // [in] The name of the peer to which the socket is to be connected
       sizeof(sa)) == SOCKET_ERROR)
   {
     iError = WSAGetLastError();
     if (iError == WSAEWOULDBLOCK)
      {  // not an "error" , but started asynchronous connection (attempt!) ...
        if(WSOCK_fLogEvents)
           WSOCK_LogEvent("Asynchronous connection attempt started.");
        SetTimer(
          WSOCK_hwndMyWindowHandle, // handle of window for timer messages
            WSOCK_TIMER_ID,    // timer identifier
          WSOCK_TIMEOUT_MS,    // time-out value in milliseconds
                       0 );    // address of timer procedure
        strcpy(WSOCK_sz80CurrentAction, "Connecting to server" );
      }
     else  // other, "real" error :
      {
        WSOCK_iLastWSError = iError;
        WSOCK_LogError( "ConnectToHost: Async connect failed", WSOCK_iLastWSError );
        closesocket(s);
        pClient->socket = INVALID_SOCKET;
        pClient->iConnState = WSOCK_CS_DISCONNECTED;
        return FALSE;
      }
   }
  else  // connect() immediately successfull, though async(non-blocking) call :
   {
     // We'll have to fake an FD_CONNECT message to ourselves.
     if(WSOCK_fLogEvents)
        WSOCK_LogEvent( "Wonder of wonders!  Asynch connect completed immediately!");
     pClient->socket = s;
     pClient->iConnState = WSOCK_CS_CONNECTED;
     // ex: PostMessage(WSOCK_hwndMyWindowHandle, WM_WINSOCK_CLI, s, MAKELPARAM(FD_CONNECT, 0) );
     // Again, the annoying warning (from BCB V4): "Suggest parentheses to clarify precedence". YUCC.
     // So, like the "MAKEWORD"-stuff, WoBu got rid of the "MAKELPARAM", too
     //  (who wants to "makel around with an LPARAM" anyway.. grrrrrrrrr) :
     //  LPARAM MAKELPARAM( WORD wLow, WORD wHigh ) simply puts two 16-bit WORDS
     //  together in a 32-bit DWORD, which is then cast into an "LPARAM" a la Microsoft.
     // Why Borland C++Builder V4 saw a problem here, remains a mystery.
     PostMessage(WSOCK_hwndMyWindowHandle, WM_WINSOCK_CLI, s, (LPARAM)((FD_CONNECT) | (0<<16)) );
     return TRUE;
   }

  pClient->socket = s;
  pClient->iConnState = WSOCK_CS_CONNECTING;
  return TRUE;
}

//---------------------------------------------------------------------------
BOOLEAN WSOCK_Disconnect( SOCKET *pSocket )
    // Gracefully shuts a connection down (client or server).
    //  Returns true if we're successful, false otherwise.
{
  BOOLEAN fResult = FALSE;

  if( *pSocket != INVALID_SOCKET )
   {  // if the SERVER socket already exists, kick it out (to establish NEW socket)

     // Disallow any further data sends.  This will tell the other side
     // that we want to go away now.  If we skip this step, we don't
     // shut the connection down nicely.
     if( shutdown( *pSocket, SD_SEND ) == SOCKET_ERROR)
      {
        fResult = FALSE;
      }
     else
      {
        // Receive any extra data still sitting on the socket.  After all
        // data is received, this call will block until the remote host
        // acknowledges the TCP control packet sent by the shutdown above.
        // Then we'll get a 0 back from recv, signalling that the remote
        // host has closed its side of the connection.
        BYTE bTrashBuffer[2048];
        while (1)
         {
           int nNewBytes = recv(*pSocket, (char*)bTrashBuffer, sizeof(bTrashBuffer), 0);
           if (nNewBytes == SOCKET_ERROR)
            {
              fResult = FALSE;
              break;
            }
           else if (nNewBytes != 0)
            {
              if(WSOCK_fLogEvents || WSOCK_fLogErrors )
                 WSOCK_LogEvent( "Received %d unexpected bytes during shutdown.",(int)nNewBytes);
            }
           else
            {
              // Okay, we're done!
              fResult = TRUE;
              break;
            }
         } // end while < more bytes to be dumped >
      }

     // Close the socket, no matter how "graceful" the above steps were
     if( closesocket( *pSocket ) == SOCKET_ERROR )
         fResult = FALSE;
     *pSocket = INVALID_SOCKET;
   } // end if < valid socked > ?
   
  return fResult;

} // end WSOCK_Disconnect()


//---------------------------------------------------------------------------
int WSOCK_GetLastError(void)
{
  int iLastError = WSAGetLastError();
  if( iLastError!=0 )
   {  return iLastError;
   }
  else
   {  return WSOCK_iLastWSError;
   }
}

//---------------------------------------------------------------------------
char * WSOCK_ErrorCodeToString( int iWSAErrorCode )
 // Turns error codes returned by WSAGetLastError() into text .
 //
{
  static char sz80OtherError[84];  // <<< avoid using this static var wherever possible
                                   //  (it may cause problems with multi-threading one day)
  switch( iWSAErrorCode )
   {
    case WSAEINTR          : return "Interrupted function call";
    case WSAEBADF          : return "Invalid socket descriptor";
    case WSAEACCES         : return "Permission denied";
    case WSAEFAULT         : return "Bad address";
    case WSAEINVAL         : return "Invalid argument";
    case WSAEMFILE         : return "Too many open files or sockets";
    case WSAEWOULDBLOCK    : return "Call of WINSOCK routine would block";
    case WSAEINPROGRESS    : return "Operation now in progress";
    case WSAEALREADY       : return "Operation already in progress";
    case WSAENOTSOCK       : return "Socket operation on non-socket";
    case WSAEDESTADDRREQ   : return "Destination address required";
    case WSAEMSGSIZE       : return "Message too long";
    case WSAEPROTOTYPE     : return "Protocol wrong type for socket";
    case WSAENOPROTOOPT    : return "Bad protocol option";
    case WSAEPROTONOSUPPORT: return "Socket type not supported";
    case WSAESOCKTNOSUPPORT: return "Socket not supported";
    case WSAEOPNOTSUPP     : return "Operation not supported";
    case WSAEPFNOSUPPORT   : return "Protocol family not supported";
    case WSAEAFNOSUPPORT   : return "Address family not supported by protocol family";
    case WSAEADDRINUSE     : return "Address already in use";
    case WSAEADDRNOTAVAIL  : return "Cannot assign requested address";
    case WSAENETDOWN       : return "Network is down";
    case WSAENETUNREACH    : return "Network is unreachable";
    case WSAENETRESET      : return "Network dropped connection on reset";
    case WSAECONNABORTED   : return "Software caused connection abort";
    case WSAECONNRESET     : return "Connection reset by peer";
    case WSAENOBUFS        : return "No buffer space available";
    case WSAEISCONN        : return "socket already connected";
    case WSAENOTCONN       : return "Socket is not connected";
    case WSAESHUTDOWN      : return "Cannot send after socket shutdown";
    case WSAETOOMANYREFS   : return "too many refs";
    case WSAETIMEDOUT      : return "Connection timed out";
    case WSAECONNREFUSED   : return "Connection refused";
    case WSAELOOP          : return "loop detected";
    case WSAENAMETOOLONG   : return "name too long";
    case WSAEHOSTDOWN      : return "Host is down";
    case WSAEHOSTUNREACH   : return "No route to host";
    case WSAENOTEMPTY      : return "not empty";
    case WSAEPROCLIM       : return "Too many processes";
    case WSAEUSERS         : return "users";
    case WSAEDQUOT         : return "quot";
    case WSAESTALE         : return "stale";
    case WSAEREMOTE        : return "remote";
    case WSAEDISCON        : return "Graceful shutdown in progress";
    case WSASYSNOTREADY    : return "Network subsystem is unavailable";
    case WSAVERNOTSUPPORTED: return "WINSOCK.DLL version out of range";
    case WSANOTINITIALISED : return "WSAStartup not yet performed";
    case WSAHOST_NOT_FOUND : return "Host not found";
    case WSATRY_AGAIN      : return "Non-authoritative host not found";
    case WSANO_RECOVERY    : return "non-recoverable error";
    case WSANO_DATA        : return "valid name but no data";
    default                :
         sprintf(sz80OtherError, "Other WINSOCK error, code %d",(int)iWSAErrorCode);
         return sz80OtherError;
   } // end switch( iWSAErrorCode )

} // end WSOCK_ErrorCodeToString()


//---------------------------------------------------------------------------
void WSOCK_SRV_OnAccept(HWND hwnd, SOCKET socket, int nErrorCode)
   // Handles notification of incoming connections .
   // Called from WSOCK_SRV_HandleMessage() in wsock.c .
{
 SOCKADDR_IN SockAddr;
 // LPREQUEST   lpReq;
 SOCKET      peerSocket;
 int         nRet, nLen, i;

  (void)hwnd;  // unused so far, but who knows..
  (void)socket; // unused parameter, but "  " 
  (void)nErrorCode; //  "   "         "  "  "

  //
  // accept the new socket descriptor
  //
  nLen = sizeof(SOCKADDR_IN);
  peerSocket = accept(WSOCK_ServerListenerSocket, (LPSOCKADDR)&SockAddr, &nLen);
      // SOCKET accept ( SOCKET s, struct sockaddr FAR* addr, int FAR* addrlen )
      //            accepts a connection on a socket
      // Parameters
      // s [in] A descriptor identifying a socket which is listening for connections after a listen.
      // addr [out] An optional pointer to a buffer which receives the address of the connecting
      //            entity, as known to the communications layer.
      //            The exact format of the addr argument is determined
      //            by the address family established when the socket was created.
      // addrlen [out] An optional pointer to an integer which contains
      //            the length of the address addr.
      // Remarks
      //
      // This routine extracts the first connection on the queue of pending
      // connections on s, creates a new socket and returns a handle
      // to the new socket. The newly created socket has the same properties
      // as s including asynchronous events registered with WSAAsyncSelect
      // or with WSAEventSelect, but not including the listening socket's
      // group ID, if any. If no pending connections are present on the queue,
      // and the socket is not marked as nonblocking, accept blocks the caller
      // until a connection is present. If the socket is marked nonblocking
      // and no pending connections are present on the queue, accept returns
      // an error as described below. The accepted socket cannot be used to
      // accept more connections. The original socket remains open.
      //
      // The argument addr is a result parameter that is filled in
      // with the address of the connecting entity, as known to the
      // communications layer. The exact format of the addr parameter
      // is determined by the address family in which the communication
      // is occurring. The addrlen is a value-result parameter;
      // it should initially contain the amount of space pointed to by addr;
      // on return it will contain the actual length (in bytes) of the address
      // returned. This call is used with connection-oriented socket types
      // such as SOCK_STREAM. If addr and/or addrlen are equal to NULL,
      // then no information about the remote address of the accepted socket
      // is returned.
  if (peerSocket == SOCKET_ERROR)
   {
     nRet = WSAGetLastError();
     if (nRet != WSAEWOULDBLOCK)
      {
       //
       // Just log the error and return
       //
       WSOCK_LogError("accept() error", nRet );
       return;
      }
   }

  //
  // Make sure we get async notices for this socket
  //
  nRet = WSAAsyncSelect(  // see Win32 Programmer's reference ...
         peerSocket,      // SOCKET s,   [in] descriptor identifying the socket for which event notification is required
         WSOCK_hwndMyWindowHandle,   // HWND hWnd,  [in] handle identifying the window which should receive a message when a network event occurs
         WM_WINSOCK_SRV,  // unsigned int wMsg, [in] message to be received when a network event occurs
         FD_READ | FD_WRITE | FD_CLOSE); // long lEvent, [in] bitmask which specifies a combination of network events in which the application is interested

  if (nRet == SOCKET_ERROR)
   {
     nRet = WSAGetLastError();
     closesocket(peerSocket);
     // log the error and return
     WSOCK_LogError( "accept() error", nRet );
     return;
   }

  // Add the connection to the list of server connections:
  for(i=0; i<WSOCK_MAX_SERVERS; ++i)
   {
     if( WSOCK_Servers[i].peer_socket == INVALID_SOCKET )
      { // found a free entry in the server's "connection" table :
        WSOCK_Servers[i].peer_socket = peerSocket;
        if( WSOCK_SRV_OnConnect( &WSOCK_Servers[i] ) )
         {
           if( WSOCK_fLogEvents )
               WSOCK_LogEvent( "Server accepted connection on socket %d from: %s",
                               peerSocket, inet_ntoa(SockAddr.sin_addr));
         }
        else // WSOCK accepted connection, but the APPLICATION (!) refused ...
         {
           if( WSOCK_fLogEvents || WSOCK_fLogErrors )
               WSOCK_LogEvent( "Application rejected connection on socket %d from: %s",
                         peerSocket, inet_ntoa(SockAddr.sin_addr));
           closesocket(peerSocket);
           WSOCK_Servers[i].peer_socket = INVALID_SOCKET;
         }
        return; // WSOCK_SRV_OnAccept() successful
      }
   }

  // Arrived here: The maximum number of server connections was exceeded !
  closesocket(peerSocket);

} // end WSOCK_SRV_OnAccept()


#if(0) // here just an EXAMPLE; the following routine will be APPLICATION-SPECIFIC !!
//---------------------------------------------------------------------------
BOOLEAN WSOCK_SRV_OnConnect( T_WSOCK_Server *pServer )
  // SERVER SIDE: Called by WSOCK when new connection has just been accepted.
  // If return=FALSE, the connection will immediately be closed again.
{
  return TRUE;   // ok, don't reject the connection !
} // end WSOCK_SRV_OnConnect()
#endif // (0) // <<< just an EXAMPLE for WSOCK_SRV_OnConnect()

#if(0) // here just an EXAMPLE; the following routine will be APPLICATION-SPECIFIC !!
//---------------------------------------------------------------------------
void WSOCK_SRV_OnDisconnect( T_WSOCK_Server *pServer, int iErrorCode )
  // SERVER SIDE: Called by WSOCK when a connection has been cut (for various reasons) .
  // Just a NOTIFICATION ! The application doesn't has to do anything in the simplest case.
{
}
#endif // (0) // <<< just an EXAMPLE for WSOCK_SRV_OnConnect()


#if(0) // here just an EXAMPLE; the following routine will be APPLICATION-SPECIFIC !!
//---------------------------------------------------------------------------
void WSOCK_SRV_OnRead( T_WSOCK_Server *pServer, int nErrorCode)
   // SERVER SIDE: Handles notification of readiness for reading ("data arrived").
   //              Called after "the other side SENT something" .
{
  static BYTE buf[2048];
 // LPREQUEST   lpReq;
  int         i, nRet;

  int bytes_read;

  do
   {
    // The Windows Sockets recv function receives data from a socket...
    bytes_read = WSOCK_Receive( // ex: recv( // see help on recv in Win32 programmer's reference !
        pServer->peer_socket,   // [in] descriptor identifying a connected socket
        (char*)buf,   // char FAR* buf, [out] A buffer for the incoming data
        sizeof(buf)); // int len,       [in] The length of buf

    if( WSOCK_fLogEvents )   // Append this to the event log ?
     { char sz255Temp[256]; char *cp;
       sprintf(sz255Temp, "Received %d bytes from socket %d : \"", (int)bytes_read, (int)pServer->peer_socket );
       cp = sz255Temp + strlen(sz255Temp);
       for(i=0; i<bytes_read && i<80; ++i)
        { if( buf[i]<32 ) break;
          else *cp++=buf[i];
        }
       *cp++='\"'; *cp='\0';
       WSOCK_LogEvent( sz255Temp );
     }

    if (bytes_read >= 0)
     { // Do something with the data ...
     }
   } while( bytes_read == sizeof(buf) );

} // end WSOCK_SRV_OnRead()
#endif // (0) // <<< just an EXAMPLE for WSOCK_SRV_OnRead()


//---------------------------------------------------------------------------
void WSOCK_SRV_OnClose(SOCKET socket, int nErrorCode)
   // SERVER SIDE: Handles notification of socket closure .
   //              Hasn't got anything to do with closing a "window" etc !
{
  int i;

  // The client must have reset the connection.
  // Clean up.

  (void)nErrorCode; // unused parameter (so far..)


  // Is it one of the SERVER's peer sockets ?
  for(i=0; i<WSOCK_MAX_SERVERS; ++i)
   { if( socket == WSOCK_Servers[i].peer_socket )
      {
        // Notify the application about the disconnect...
        WSOCK_SRV_OnDisconnect( &WSOCK_Servers[i], 0/*iErrorCode*/ );

        // close the server's peer socket, make it available for new connections
        if( WSOCK_Servers[i].peer_socket != INVALID_SOCKET )
            closesocket( WSOCK_Servers[i].peer_socket );
        WSOCK_Servers[i].peer_socket = INVALID_SOCKET;
      }
   } // end for

  // Is this the SERVER's listener socket ? ...
  if( socket == WSOCK_ServerListenerSocket)
   { // NOT WSOCK_Disconnect( &WSOCK_ServerListenerSocket ); ! !
     closesocket( WSOCK_ServerListenerSocket );
     WSOCK_ServerListenerSocket = INVALID_SOCKET;
   }


} // end WSOCK_SRV_OnClose()


//---------------------------------------------------------------------------
void WSOCK_SRV_HandleMessage(HWND hwnd, WPARAM wParam, LPARAM lParam)
  // SERVER SIDE: Handler for windows messages sent to the application
  //              through the (user-defined!) WM_WINSOCK message .
  // Called whenever the WINDOWS SOCKET service wants to inform
  // the IP server about "something" .
  //
  // Important Note (from Win32 programmer's reference) :
  // > The Windows Sockets DLL will not continually flood an application
  // > with messages for a particular network event.
  // > Having successfully posted notification of a particular event
  // > to an application window, no further message(s) for that network
  // > event will be posted to the application window
  // > until the application makes the function call which implicitly
  // > re-enables notification of that network event.
  //
{
  int i;
  int iSrvTabIndex;
  int nErrorCode = WSAGETSELECTERROR(lParam);
  // The WSAGETSELECTEVENT macro extracts the event code from the lParam
  // in the response to a WSAAsyncSelect() :
  WORD wEventCode = WSAGETSELECTEVENT(lParam);

  iSrvTabIndex = -1;
  for(i=0; i<WSOCK_MAX_SERVERS; ++i)
   { if( WSOCK_Servers[i].peer_socket == (SOCKET)wParam )
      { iSrvTabIndex = i;
        break;
      }
   }

  switch(wEventCode)
   {
    case FD_ACCEPT: // notification of incoming connections
         if( WSOCK_fLogWinMsgs )
          { WSOCK_LogEvent( "Server received FD_ACCEPT msg, socket %d ",(int)wParam );
          }
         WSOCK_SRV_OnAccept(hwnd, (SOCKET)wParam, nErrorCode);
         break;

    case FD_CONNECT: // notification of completed connection
         if( WSOCK_fLogWinMsgs )
          { WSOCK_LogEvent( "Server received FD_CONNECT msg, socket %d ",(int)wParam );
          }
         break;

    case FD_READ:  // notification of readiness for reading
         // (called after "the other guy SENT something"; for us: there's something to read)
         if( WSOCK_fLogWinMsgs )
          { WSOCK_LogEvent( "Server received FD_READ msg, socket %d ",(int)wParam );
          }
         if( iSrvTabIndex >= 0 )
          { WSOCK_SRV_OnRead( &WSOCK_Servers[i] );
          }
         break;

    case FD_WRITE: // notification of readiness for writing
         if( WSOCK_fLogWinMsgs )
          { WSOCK_LogEvent( "Server received FD_WRITE msg, socket %d ",(int)wParam );
          }
         if( iSrvTabIndex >= 0 )
          { WSOCK_SRV_OnWrite( &WSOCK_Servers[i] );
          }
         break;

    case FD_CLOSE: // notification of socket closure
         if( WSOCK_fLogWinMsgs )
          { WSOCK_LogEvent( "Server received FD_CLOSE msg, socket %d ",(int)wParam );
          }
         WSOCK_SRV_OnClose((SOCKET)wParam, nErrorCode);
         break;

    case FD_OOB:       // notification of the arrival of out-of-band data
  //case FD_QOS:       // notification of socket Quality of Service (QOS) changes
  //case FD_GROUP_QOS: // notification of socket group Quality of Service (QOS) changes
    default:           // what else ?!
         if( WSOCK_fLogWinMsgs )
          { WSOCK_LogEvent( "Server received 'other' event (code %d)",(int)wEventCode );
          }
         break;
   }
} // end WSOCK_SRV_HandleMessage( )


//---------------------------------------------------------------------------
//   CLIENT Functions ...
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
T_WSOCK_Client * WSOCK_FindClientBySocket( WSOCKET socket )
{
 int i;
  // Check the list of server connections:
  for(i=0; i<WSOCK_MAX_CLIENTS; ++i)
   { if( WSOCK_Clients[i].socket == socket )
      { // found the entry in the client's "connection" table :
        return &WSOCK_Clients[i];
      }
   }
  return NULL;  // no client seems to belong to this socket !
} // end WSOCK_FindClientBySocket()


//---------------------------------------------------------------------------
T_WSOCK_Client * WSOCK_AddClient( void )
{
 int i;
  // Check the list of server connections:
  for(i=0; i<WSOCK_MAX_CLIENTS; ++i)
   { if( WSOCK_Clients[i].iConnState == WSOCK_CS_UNUSED_ENTRY )
      { // found an unused entry in the client's "connection" table :
        WSOCK_Clients[i].iConnState = WSOCK_CS_DISCONNECTED;
        return &WSOCK_Clients[i];
      }
   }
  return NULL;  // all client structs seem to be occupied !
} // end WSOCK_AddClient()


//---------------------------------------------------------------------------
void WSOCK_DisconnectClient( T_WSOCK_Client *pClient )
{
  if( pClient )
   {
    if( pClient->socket != WSOCKET_INVALID )
        WSOCK_Disconnect( &pClient->socket );
    pClient->socket     = WSOCKET_INVALID;
   }
}

//---------------------------------------------------------------------------
void WSOCK_DeleteClient( T_WSOCK_Client *pClient )
{
  if( pClient )
   {
    if( pClient->socket != WSOCKET_INVALID )
        WSOCK_Disconnect( &pClient->socket );
    pClient->socket     = WSOCKET_INVALID;
    pClient->iConnState = WSOCK_CS_UNUSED_ENTRY;
   }
} // end WSOCK_DeleteClient()



#if(0) // here just an EXAMPLE; the following routine will be APPLICATION-SPECIFIC !!
//---------------------------------------------------------------------------
void WSOCK_CLI_OnRead( T_WSOCK_Client *pClient )
   // CLIENT SIDE: Handles notification of readiness for reading .
   //       Called after "the other side SENT something" / "something to read" .
{
  static BYTE buf[2048];
 // LPREQUEST   lpReq;
  int         i, nRet, bytes_read;

  if( pClient == NULL )
     return;

  do
   {
    // The Windows Sockets recv function receives data from a socket...
    bytes_read = WSOCK_Receive( // see help on recv in Win32 programmer's reference !
        pClient->socket,   // SOCKET s,      [in] A descriptor identifying a connected socket
        buf,               // char FAR* buf, [out] A buffer for the incoming data
        sizeof(buf));      // int len,       [in] The length of buf

    if( WSOCK_fLogEvents )   // Append this to the event log ?
     { char sz255Temp[256]; char *cp;
       sprintf(sz255Temp, "Client received %d bytes from socket %d : \"", (int)bytes_read, (int)socket );
       cp = sz255Temp + strlen(sz255Temp);
       for(i=0; i<bytes_read && i<80; ++i)
        { if( buf[i]<32 ) break;
          else *cp++=buf[i];
        }
       *cp++='\"'; *cp='\0';
       WSOCK_LogEvent( sz255Temp );
     }

    if (bytes_read >= 0)
     { // Do something with the data ...
     }
   } while( bytes_read == sizeof(buf) );

} // end WSOCK_CLI_OnRead()
#endif // (0)


//---------------------------------------------------------------------------
void WSOCK_CLI_HandleMessage(HWND hwnd, WPARAM wParam, LPARAM lParam)
  // CLIENT SIDE: Handler for windows messages sent to the application
  //              through the (user-defined!) WM_WINSOCK message .
  // Called whenever the WINDOWS SOCKET service wants to inform
  // the IP client about "something" .
  //
  // Important Note (from Win32 programmer's reference) :
  // > The Windows Sockets DLL will not continually flood an application
  // > with messages for a particular network event.
  // > Having successfully posted notification of a particular event
  // > to an application window, no further message(s) for that network
  // > event will be posted to the application window
  // > until the application makes the function call which implicitly
  // > re-enables notification of that network event.
  //
{
  int nErrorCode = WSAGETSELECTERROR(lParam);
  // The WSAGETSELECTEVENT macro extracts the event code from the lParam
  // in the response to a WSAAsyncSelect() :
  WORD wEventCode = WSAGETSELECTEVENT(lParam);
  T_WSOCK_Client *pClient = WSOCK_FindClientBySocket( (SOCKET)wParam );

  (void)hwnd; // unused parameter .. so what ?

  KillTimer(
       WSOCK_hwndMyWindowHandle, // handle of window that installed timer
       WSOCK_TIMER_ID );         // timer identifier

  if (nErrorCode != 0)
   {
     WSOCK_LogError( "Client: Error notification", nErrorCode );
     switch (nErrorCode)
      {
        case WSAECONNRESET:
             if( WSOCK_fLogWinMsgs | WSOCK_fLogErrors )
                 WSOCK_LogEvent( "Client closed socket %d due to 'connection reset'",(int)wParam );
             break;
        case WSAECONNREFUSED:
             if( WSOCK_fLogWinMsgs | WSOCK_fLogErrors )
                 WSOCK_LogEvent( "Client closed socket %d due to 'connection refused'",(int)wParam );
             break;
        case WSAETIMEDOUT:
             if( WSOCK_fLogWinMsgs | WSOCK_fLogErrors )
                 WSOCK_LogEvent( "Client closed socket %d due to 'timeout'",(int)wParam );
        default:
             if( WSOCK_fLogWinMsgs | WSOCK_fLogErrors )
                 WSOCK_LogEvent( "Unexpected error, Client closes socket %d ",(int)wParam );
             break;
      }
     // For most of the above errors, the application will be notified
     // and the socket closed !
     if( pClient!= NULL && (pClient->socket!=INVALID_SOCKET) )
      { // Notify the APPLICATION about this event :
        WSOCK_CLI_OnDisconnect( pClient, nErrorCode );
        if( pClient->socket != INVALID_SOCKET ) // may have been closed by appl. already!
            closesocket( pClient->socket );
        pClient->socket = INVALID_SOCKET;
        if(pClient->iConnState == WSOCK_CS_CONNECTED )
           pClient->iConnState = WSOCK_CS_DISCONNECTED;
      }
     return;
   } // end if < error >

  if( pClient!= NULL )
   {
     switch(wEventCode)
      {
        case FD_ACCEPT: // notification of incoming connections  (never occurred for the CLIENT)
            if( WSOCK_fLogWinMsgs )
             { WSOCK_LogEvent( "Client received FD_ACCEPT msg, socket %d ",(int)wParam );
             }
            // WSOCK_CLI_OnAccept(hwnd, (SOCKET)wParam, nErrorCode);
            break;

        case FD_CONNECT: // notification of completed connection (important for the CLIENT)
            if( WSOCK_fLogWinMsgs )
             { WSOCK_LogEvent( "Client received FD_CONNECT msg, socket %d ",(int)wParam );
             }
            pClient->iConnState = WSOCK_CS_CONNECTED;
            WSOCK_CLI_OnConnect( pClient );
            break;

        case FD_READ:  // notification of readiness for reading
            // (called after "the other guy SENT something"; for us: "there's something to read")
            if( WSOCK_fLogWinMsgs )
             { WSOCK_LogEvent( "Client received FD_READ msg, socket %d ",(int)wParam );
             }
            WSOCK_CLI_OnRead( pClient );
            break;

        case FD_WRITE: // notification of readiness for writing ("may send now if we like")
            // Note: the CLIENT only seems to receive this message ONCE AFTER CONNECT;
            //       or ONCE after a send() returned WSAEWOULDBLOCK ! ! !
            // This is not very clever.. some applications want to be called back
            //     after sending ONE packet successfully, to send the next packet
            //     in the "OnRead"-callback !
            if( WSOCK_fLogWinMsgs )
             { WSOCK_LogEvent( "Client received FD_WRITE msg, socket %d ",(int)wParam );
             }
            WSOCK_CLI_OnWrite( pClient );
            break;

        case FD_CLOSE: // notification of socket closure (here: CLIENT side)
            if( WSOCK_fLogWinMsgs )
             { WSOCK_LogEvent( "Client received FD_CLOSE msg, socket %d ",(int)wParam );
             }
            // Also notify the APPLICATION about this event :
            WSOCK_CLI_OnDisconnect( pClient, 0 );

            // NOTE: The APPLICATION may have already closed the socket..
            if( pClient->socket != INVALID_SOCKET ) // .. if not, close it HERE
                closesocket( pClient->socket );
            pClient->socket = INVALID_SOCKET;
            pClient->iConnState = WSOCK_CS_DISCONNECTED;
            break;

        case FD_OOB:       // notification of the arrival of out-of-band data
     // case FD_QOS:       // notification of socket Quality of Service (QOS) changes
     // case FD_GROUP_QOS: // notification of socket group Quality of Service (QOS) changes
        default:           // what else ?!
            if( WSOCK_fLogWinMsgs )
             { WSOCK_LogEvent( "Client received 'other' event (code %d)",(int)wEventCode );
             }
            break;
      }
   } // end if( pClient!= NULL )


} // end WSOCK_CLI_HandleMessage( )


//---------------------------------------------------------------------------
int WSOCK_BeginToSend( WSOCKET socket, BYTE *pbSrc, int iLen )
{
  int iResult;

  // The Windows Sockets send function sends data on a connected socket.
  iResult = send(
           socket,// SOCKET s, [in] descriptor identifying a connected socket
           (const char*)pbSrc, // const char * buf, [in] buffer containing the data to be transmitted
           iLen,  // int len,          [in] length of the data in buf
           0 );   // int flags,        [in] specifies the way in which the call is made
     // Remarks on 'send()' :
     // > send is used to write outgoing data on a connected socket.
     // > For message-oriented sockets, care must be taken not to exceed
     // > the maximum packet size of the underlying provider,
     // > which can be obtained by getting the value of socket option
     // > SO_MAX_MSG_SIZE. If the data is too long to pass atomically
     // > through the underlying protocol the error WSAEMSGSIZE is returned,
     // > and no data is transmitted.
     // > Note that the successful completion of a send does not indicate
     // > that the data was successfully delivered.
     // > If no buffer space is available within the transport system to hold
     // > the data to be transmitted, send will block unless the socket
     // > has been placed in a nonblocking I/O mode.
     // > On nonblocking stream-oriented sockets, the number of bytes written
     // > can be between 1 and the requested length, depending on buffer
     // > availability on both the local and foreign hosts.
     // > The select, WSAAsyncSelect or WSAEventSelect call can be used
     // > to determine when it is possible to send more data.
     // > Calling send with a len of zero is to be treated by implementations
     // > as successful. In this case, send can return zero as a valid
     // > return value.
     // > For message-oriented sockets, a zero-length transport datagram is sent.
   // BUT, as always, things are more complicated than you may think !
   //
   // Especially if your application relies on the FD_WRITE event .
   //
   // From www.gamedev.net/reference/programming/features/asyncsock/page5.asp,
   //  written by Drew Sikora :
   //
   // > Now the FD_WRITE event is a bit more complicated. Mainly because
   // > of the way it is called. First of all, an FD_WRITE event is always
   // > generated when you first establish a connection with another socket.
   // > So you think all you have to do is plunk in a send() function
   // >              with whatever else youll need and youll be fine .
   // > Oh if that were only true. Listen carefully: the FD_WRITE event
   // > is triggered only when there is more room in the buffer with which
   // > to write data. Huh? Allow me to explain.
   // >
   // > First off, the buffer, as I use it here, is the network buffer,
   // > which you write information to. That information is then sent
   // > over the network (intranet or internet) to the receiving socket.
   // > So youd think that since there will always be space left to write data
   // > if you dont fill it up all at once, that FD_WRITE events will just
   // > keep coming and coming right? Wrong! Read the rules again.
   // > It says when there is more room. Not enough room, more room.
   // > This means you have to fill it up first! How do you do that?
   // >
   // > The general idea is to create an infinite while loop in which you will
   // > continuously send data until you max out the buffer. When is happens,
   // > send() will return the error WSAWOULDBLOCK. This means that if it were
   // > a blocking socket, it would wait (stop execution) for more room
   // > in the buffer and then send. But since it isnt, you get the error.
   // > So now that youve filled up the buffer, you just have to wait until
   // > more room becomes available so you can write again. And bingo!
   // > Up pops another FD_WRITE event.
   // > Do you have any idea how much trouble I had to go through to figure this out?
   // > You people are so darned lucky! Heres an example of an FD_WRITE event handler:
   // > (..)
   // > The implementation isnt so hard after all! It was only the concept
   // > that was messing with your head. If you use this technique, the loop
   // > will end when the network buffer becomes full.  (COMPLETELY FULL, that is..)
   // > Then, when more space is available - another FD_WRITE event will
   // > be triggered, and you can send more data.
   // > Before you jump to make use of your newfound knowledge, allow me
   // > to clarify the use of the FD_WRITE event. Do not expect to be using
   // > this event at all if you are not going to be sending large pieces of data
   // > at once. The reason for this is simple - if you base all you information
   // > transactions on FD_WRITE and never send enough to fill up the buffer,
   // > after that first trial FD_WRITE event generated at the beginning
   // > of the connection - no other FD_WRITE event will be triggered !
   // > Therefore for games, where you send as little data as possible,
   // > just forget about setting a socket for the FD_WRITE event and use send()
   // > wherever you need it.
   // > FD_WRITE is best used for transferring files, in my experience.
   //
   // DL4YHF: What a nice article, it contains everything which is MISSING
   //         (or at least hard to find) in the original winsock documentation.
  if( iResult>=0 )
   {
     if( WSOCK_fLogEvents )   // Append this to the event log ?
         WSOCK_LogEvent( "WSOCK sending %d bytes to socket %d .", (int)iResult, (int)socket );

     return iResult;  // number of bytes sent
   }
  else // send() failed ?
   { iResult = WSOCK_GetLastError();
     if (iResult == WSAEWOULDBLOCK)
      { // not an "error" , but started asynchronous connection (attempt!) ...
        return WSOCK_BUFFER_FULL;
      }
     else
      {
        WSOCK_LogError( "BeginToSend: TX error", iResult );
        return WSOCK_ERROR;
      }
   }
} // end WSOCK_BeginToSend()


//---------------------------------------------------------------------------
int WSOCK_CLI_BeginToSend( T_WSOCK_Client *pClient, BYTE *pbSrc, int iLen )
{
 int iResult = WSOCK_ERROR;
  if( pClient!=NULL && pbSrc!=NULL && iLen>=0 )
   { // there's really something to transmit NOW:
     iResult = WSOCK_BeginToSend( pClient->socket, pbSrc, iLen );
     if( iResult>=0 && (iResult!=WSOCK_BUFFER_FULL) )
      { // See DL4YHF's note on FD_WRITE ... make sure "OnWrite()" will be called
        // ex: PostMessage(WSOCK_hwndMyWindowHandle, WM_WINSOCK_CLI, pClient->socket, MAKELPARAM(FD_WRITE, 0) );
        // Because Borland C++Builder "sugggested parentheses to clarify precedence" (whoa??!) :
        // MAKELPARAM(FD_WRITE/*loword*/, 0/*hiword*/ ) -> (LPARAM)((FD_WRITE) | (0<<16))
        PostMessage(WSOCK_hwndMyWindowHandle, WM_WINSOCK_CLI, pClient->socket, (LPARAM)((FD_WRITE)|(0<<16)) );
        // After this UGLY modification, BCB was happy and compiled this unit with 0 errors and 0 warnings.
      }
   }
  return iResult;
} // end WSOCK_CLI_BeginToSend()

//---------------------------------------------------------------------------
int WSOCK_SRV_BeginToSend( T_WSOCK_Server *pServer, BYTE *pbSrc, int iLen )
{
 int iResult = WSOCK_ERROR;
  if( pServer!=NULL && pbSrc!=NULL && iLen>=0 )
   { // there's really something to transmit NOW:
     iResult = WSOCK_BeginToSend( pServer->peer_socket, pbSrc, iLen );
     if( iResult>=0 && (iResult!=WSOCK_BUFFER_FULL) )
      { // See DL4YHF's note on FD_WRITE : It will only be sent if the buffer
        // is *COMPLETELY* full , not just "partially" !
        PostMessage(WSOCK_hwndMyWindowHandle, WM_WINSOCK_CLI, pServer->peer_socket, (LPARAM)((FD_WRITE)|(0<<16)) );
      }
   }
  return iResult;
} // end WSOCK_SRV_BeginToSend()


//---------------------------------------------------------------------------
int WSOCK_Receive( WSOCKET socket, BYTE *pbDestSrc, int iMaxLen )
{
  int iError;
  int nBytes = recv( socket, (char*)pbDestSrc, iMaxLen, 0 );
  if (nBytes == SOCKET_ERROR)
   { iError = WSOCK_GetLastError();
     if (iError == WSAEWOULDBLOCK)
      {  // not an "error" , it simply says "there is nothing in the buffer" !
        // Append this to the event log ?
        if( WSOCK_fLogEvents )
            WSOCK_LogEvent( "WSOCK received nothing " );
        return WSOCK_BUFFER_EMPTY;
      }
     else
      { WSOCK_LogError( "WSOCK Receive Error", iError );
      }
   }
  else
   {
    // Append this to the event log ?
    if( WSOCK_fLogEvents )
        WSOCK_LogEvent( "WSOCK rcvd %d bytes from socket %d ", (int)nBytes, (int)socket );
   }
  return nBytes;
} // end WSOCK_Receive()


// EOF <wsock.c>  .  Leave an empty line after this for GNU/Mingw/DevCpp !
