/***************************************************************************/
/*  C:\cbproj\SoundUtl\utility1.cpp :                                      */
/*  - Some small string conversion functions etc     by DL4YHF.            */
/*  - dynamic memory allocation with "housekeeping" for debugging purposes */
/*  - MODULE PREFIX "UTL" ..   for historic reasons                        */
/*  - "Master" in C:\cbproj\SoundUtl\utility1.cpp                          */
/*  - Copies exist in C:\cbproj\WSQ2 and (as C-module) C:\cbproj\CalcEd,   */
/*           to have 'all files in one place' in the project directory .   */
/*                                                                         */
/*  Latest changes :                                                       */
/*   2020-10-11 :  Modified UTL_NamedMalloc(etc) to *GUARANTEE*            */
/*                 8-byte alignment, which -to the author's big suprise-   */
/*                 Borland C++Builder V6's standard "malloc()" DID NOT !   */
/*   2007-09-30 :  Added UTL_NamedMalloc() for debugging purposes .        */
/***************************************************************************/

#include <time.h>
#include <stdio.h>
#include <string.h>

#include <float.h>  // caution; there's something evil about _fpclass()
#include <math.h>

#include <io.h>     // to write the optional logfile
#include <fcntl.h>  // a mystery why we need another stupid header for O_RDWR (!)
#include <Windows.h> // even more stupid headers - but that's "C" (btw this one defines stuff like DWORD, WORD, BYTE, CRITICAL_SECTION, ...)

#include "utility1.h"

//---------- Internal data types ----------------------------------------
typedef struct
{
  void *pvBlock;   // actually points to a T_UTL_AllocBlockHdr (see below), followed by the 'netto' data
  DWORD dwAllocatedSize; // 'officially' allocated size (without debugging- and misaligned overhead)
  char  c8BlockName[8];  // optional NAME of a memory block (only for debugging)
} T_UTL_AllocBlockInfo;

typedef struct
{
  int  iIndexIntoInfo;       // first FOUR BYTES
  BYTE *pbMisalignedBlock;   // next FOUR BYTES (8 bytes total, important to keep alignment)
  // The user data ("officially allocated block") follows directly
  //  after this T_UTL_AllocBlockHdr structure .
  // OLD (before defeating the MISALIGNED results from Borland's malloc()) :
  //      When the user application frees a block, it's easy to find
  //      the related entry in the T_UTL_AllocBlockInfo array  :
  //      simply subtract sizeof(T_UTL_AllocBlockHdr) from the address .
  // NEW (since 2020-10, after fixing the stupid MISALIGNED result from malloc()) :
  //      When the user application frees a block, it's still easy to find
  //      the related entry in the T_UTL_AllocBlockInfo array  :
  //      simply subtract sizeof(T_UTL_AllocBlockHdr) from the to-be-freed address.
  //      This gives a pointer to a (properly aligned) T_UTL_AllocBlockHdr
  //      directly BEFORE the (properly aligned) 'payload' .
  //      The T_UTL_AllocBlockHdr contains the ORIGINAL (possibly misaligned)
  //      address, returned by the crappy old malloc(), required to free()
  //      the T_UTL_AllocBlockHdr *and* the 'payload' (in a single call).
  //      See UTL_NamedMalloc() for details about the misaligned 'scrap bytes'
  //      *BEFORE* the T_UTL_AllocBlockHdr (the T_UTL_AllocBlockHdr itself is
  //      aligned to 8 bytes; pbMisalignedBlock may point to a location
  //      that is ZERO to SEVEN(!) bytes *before* the T_UTL_AllocBlockHdr).
  //
} T_UTL_AllocBlockHdr;

#define UTL_MAX_ALLOC_BLOCKS 262144
  /* 2012-03-23: increased from 100000 to 262144 (=256*1024), for reasons   */
  /*        explained in C:\cbproj\SpecLab\TDScope.CPP ("Loran" experiment) */
  /* 2010-02-27: increased from 20000 to 100000,                            */
  /* because *EVERY* T_SPECTRUM is now allocated dynamically ,              */
  /* and each sample in the time domain scope's display buffer may(!) now   */
  /* have its own moving average buffer (CMovingAvrgFilter) .               */
  /* No big issue because every T_UTL_AllocBlockInfo is only 8..16 byte in size */


//---------- GLOBAL variables -------------------------------------------
//  ( WoBu dared to use such things .. )
double UTL_dblAdcMaxInputVoltage=   1.0; // can be used for ABSOLUTE displays, like "100 mV"
double UTL_dblAdcInputImpedance = 600.0; // required for VOLTAGE / POWER conversions
       // Note: THE ABOVE CONVERSION PARAMETERS MUST NEVER BE ZERO.
       //       FOR PERFORMANCE REASONS, NO "DIVIDE-BY-ZERO"-CHECK IS PERFORMED
       //       WHEN USING THESE VALUES ANYWHERE IN THE PROGRAM !

const char *UTL_months1[12]=
   { "JAN","FEB","MAR","APR","MAY","JUN","JUL","AUG","SEP","OCT","NOV","DEC" };
const char *UTL_months2[12]=
   { "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec" };

char       UTL_sz255LogFile[256]="";  // optional name of LOGFILE (""=don't log)
static int UTL_iLogFileNrOfErrors = 0;
BOOL       UTL_fRunLog_UseOutputDebugString = FALSE; // .. in UTL_WriteRunLogEntry(), BESIDES the logfile
/*static*/ T_UTL_AllocBlockInfo UTL_sAllocBlocks[UTL_MAX_ALLOC_BLOCKS];
           // 2008-08-17: Removed the 'static' attribute to simplify debugging.
           // (a 'static' variable cannot always be displayed in the debugger)
static long UTL_i32NrOfAllocatedBlocks = 0;
static long UTL_i32MaxNrOfAllocatedBlocks = 0;
static long UTL_i32NextUnusedAllocBlock = 0;
static BOOL UTL_fInitialized=FALSE;
static int  UTL_iNrOfBadBlockShown=0;
CRITICAL_SECTION UTL_csMalloc; // who knows if all this is thread-safe ?
CRITICAL_SECTION UTL_csRunLog; // same here: UTL_WriteRunLogEntry() should be thread-safe

int UTL_iSourceCodeLine=__LINE__; // added 2008-08-03 to trace "sqrt domain error"
#define DOBINI() UTL_iSourceCodeLine=__LINE__

char  UTL_CPUID_VendorID[16];     // actually a 12-character string with the CPU's manufacturer ID string
DWORD UTL_dw4CPUID_ProcInfoAndFeatures[4];
int   UTL_iWindowsMajorVersion=0; // 0=unknown, 5=Win2003 or XP, 6=Vista or Win7 (!), and the story goes on...


void UTL_InitMallocHouskeeping(void);

int  UTL_CheckMemNodes_iBlockIndex = 0;           // only for debugging purposes !
T_UTL_AllocBlockInfo *UTL_CheckMemNodes_pBlockInfo = NULL;  // only for debugging purposes !
T_UTL_AllocBlockHdr  *UTL_CheckMemNodes_pBlockHeader= NULL; // only for debugging purposes !

//---------------------------------------------------------------------------
// Dynamically loaded functions (mostly from "KERNEL32") that didn't exist
// in Borland's stoneage "windows.h" (or headers included from there) :
typedef BOOL (__stdcall *tGetSystemTimes)(LPFILETIME, LPFILETIME, LPFILETIME);
tGetSystemTimes pGetSystemTimes = NULL; // NULL if GetSystemTimes() not available


//---------- Implementation of functions ---------------------------------

/***************************************************************************/
static void AppendBlockName( char *pszDest, char *c8BlockName )
{
  char *cp = pszDest + strlen(pszDest);
  int  ci;
  for(ci=0; ci<8; ++ci)
   { if( c8BlockName[ci]!='\0' )
         *cp++ = c8BlockName[ci];
     else break;
   }
  *cp = '\0';
}


//---------------------------------------------------------------------------
BOOL UTL_CheckMemoryNodes(void)  // caution, may be UTTERLY SLOW !
{
  BOOL fResult = TRUE;
  char sz80[84];
  char *cp;   int ci;
  T_UTL_AllocBlockInfo *pBlockInfo;
  T_UTL_AllocBlockHdr  *pBlockHeader;

  if( UTL_fInitialized )
   {
     DOBINI();
     EnterCriticalSection( &UTL_csMalloc ); // no one else calls malloc now, right ?
     DOBINI();
     for(int i=0; i<UTL_MAX_ALLOC_BLOCKS; ++i)
      { pBlockInfo = &UTL_sAllocBlocks[i];
        if( pBlockInfo->pvBlock != NULL )
         {  // Here is an allocated block. Check it...
            DWORD size = pBlockInfo->dwAllocatedSize;
            BYTE *pbBlock = (BYTE*)pBlockInfo->pvBlock;
            // Note: pvBlock actually points to a T_UTL_AllocBlockHdr (see below), followed by the 'netto' data
            pBlockHeader = (T_UTL_AllocBlockHdr*)pbBlock;
            // 2012-09-27: Access violation here ? Don't trust pBlockInfo->pvBlock !
            //   Just because it's non-NULL doesn't mean it's a guaranteed valid pointer,
            //   especially since the application may have trashed a block of memory
            //   by overwriting the maximum length. Guess this is what happened...
            // 2012-11-03: Similar problem again, unfortunately no idea where exactly
            //   the allocated block was corrupted. Had run an I/Q file analysis before.
            //   Because the debugger was unable to examine 'i' (local variable),
            //   the program now copies the most important stuff into
            //   the following GLOBAL variables, which are ONLY for debugging:
            UTL_CheckMemNodes_iBlockIndex = i;
            UTL_CheckMemNodes_pBlockInfo  = pBlockInfo;
            UTL_CheckMemNodes_pBlockHeader= pBlockHeader;
            // 2012-11-03: The exception happened with UTL_CheckMemNodes_iBlockIndex=22,
            //             *UTL_CheckMemNodes_pBlockInfo = { 0x045E8010, 131072, "SndChkIn" },
            //              UTL_CheckMemNodes_pBlockHeader = 0x045E8010 .
            //             *UTL_CheckMemNodes_pBlockHeader = { ???? } = invalid address.
            //  The PREVIOUS block, UTL_sAllocBlocks[21] = { NULL, 0, "SndChunk" } .
            if( pBlockHeader->iIndexIntoInfo != i )
             {     fResult = FALSE;
                   // DEBUGGER_BREAK();  // SET BREAKPOINT HERE,  or in DebugU1.cpp !
                   strcpy( sz80, "Memory allocation error " );
                   if( pBlockInfo->c8BlockName[0] != '\0' )
                    { // the defective block has a name : report it !
                      strcat( sz80, "in " );
                      AppendBlockName( sz80, pBlockInfo->c8BlockName );
                    }
                   strcat( sz80, ": block START corrupted" );
                   UTL_WriteRunLogEntry( sz80 );
             }
            pbBlock += sizeof(T_UTL_AllocBlockHdr);
            if( (pbBlock[size]!=0x11) || (pbBlock[size+1]!=0x22) )
             {  // Wrong 'magic code' after the allocated block ?
                // The application must have "destroyed" this by accident.
                // Show this incident in the error history, if possible.
                // Setting a breakpoint in the line below is highly recommended !
                fResult = FALSE;
                // DEBUGGER_BREAK();  // SET BREAKPOINT HERE,  or in DebugU1.cpp !
                if( UTL_iNrOfBadBlockShown < 10 )
                 {
                   strcpy( sz80, "Memory allocation error " );
                   if( pBlockInfo->c8BlockName[0] != '\0' )
                    { // the defective block has a name : report it !
                      strcat( sz80, "in " );
                      cp = sz80+strlen(sz80);
                      for(ci=0; ci<8; ++ci)
                       { if( pBlockInfo->c8BlockName[ci]!='\0' )
                            *cp++ = pBlockInfo->c8BlockName[ci];
                         else break;
                       }
                      *cp = '\0';
                    }
                   strcat( sz80, ": end of block exceeded" );
                   UTL_WriteRunLogEntry( sz80 );
                   ++UTL_iNrOfBadBlockShown;
                 }
       //       DEBUGGER_BREAK();      // SET BREAKPOINT HERE,  or in DebugU1.cpp !
             }
         }
      }
     DOBINI();
     LeaveCriticalSection( &UTL_csMalloc );
     DOBINI();
   } // end if( UTL_fInitialized )

  return fResult;

} // end UTL_CheckMemoryNodes();



//---------------------------------------------------------------------------
void UTL_ExitFunction(void) // automagically called on exit, but may be called
                            // from the application (for testing purposes)
{
  char sz80[84];
  T_UTL_AllocBlockInfo *pBlockInfo;
  int iStillAllocatedBlocks = 0;
  char *cp;
  char *cpNameOfBadBlock = "";
  DOBINI();
  if( UTL_fInitialized )
   {
     UTL_WriteRunLogEntry( "ExitFunction: Checking memory nodes.." );

     DOBINI();
     UTL_CheckMemoryNodes();
     EnterCriticalSection( &UTL_csMalloc ); // no one else calls malloc now, right ?
     for(int i=0; i<UTL_MAX_ALLOC_BLOCKS; ++i)
      { pBlockInfo = &UTL_sAllocBlocks[i];
        // Note: T_UTL_AllocBlockInfo.pvBlock actually points
        //       to a T_UTL_AllocBlockHdr, followed by the 'netto' data .
        if( pBlockInfo->pvBlock != NULL )
         {  // Too bad ! someone seems to have forgotten
            //  to free his memory block. Cannot do that here,
            //  because 'he' (whoever 'he' is) may still be USING it .
            // DEBUGGER_BREAK();  // SET BREAKPOINT HERE,  or in DebugU1.cpp !
            if( iStillAllocatedBlocks == 0 )
             { cpNameOfBadBlock = pBlockInfo->c8BlockName;
             }
            ++iStillAllocatedBlocks;
         }
      }
     if( iStillAllocatedBlocks > 0 )
      { sprintf( sz80, "WARNING: Application didn't free %d memory block%c when quitting.",
                 (int)iStillAllocatedBlocks, (char)(iStillAllocatedBlocks==1)?' ':'s' );
        UTL_WriteRunLogEntry( sz80 );
        strcpy( sz80, "WARNING: Name of 1st non-freed block = \"" );
        cp = sz80+strlen(sz80);
        for(int i=0; (i<=7) && (*cpNameOfBadBlock!=0); ++i)
         { *cp++ = *cpNameOfBadBlock++;
         }
        strcpy(cp,"\" .");
        UTL_WriteRunLogEntry( sz80 );
      }
     else
      { if( (UTL_sz255LogFile[0] > 32) || UTL_fRunLog_UseOutputDebugString )
         { sprintf( sz80, "Ok, all %d dynamic memory blocks were freed.",
                          (int)UTL_i32MaxNrOfAllocatedBlocks  );
           UTL_WriteRunLogEntry( sz80 );
         }
      }

     if( UTL_sz255LogFile[0] > 32 )
      { UTL_WriteRunLogEntry( "Reached last termination step; closing logfile." );
      } // end if < also log the message text AS A FILE ? >
     UTL_fInitialized = FALSE;  // important ! don't try to 'Enter' again
     LeaveCriticalSection( &UTL_csMalloc );
     DeleteCriticalSection( &UTL_csMalloc );

     CoUninitialize();  // De-Init COM (formerly done in VARIOUS other modules).
      // 2012-08-09 : Moved this ugly CoInitialize/CoUninitialize-BULLSHIT
      //              HERE - into c:\CBproj\SoundUtl\UTILITY1.cpp -
      //              which ensures this ugly stuff is called exactly ONCE.
      DOBINI();


   } // end if( UTL_fInitialized )


  DOBINI();
        // 2008-05-17 : Shit.. the "phantom breakpoint" (ntdll.DgbBreakPoint)
        //  is back again. Now here (occasionally), after terminating the C code,
        //  but not in any of the self-written modules (but in the VCL). Damn .
        // In a Delphi forum, someone wrote:
        // > Delphi (Win32)
        // > Es kann vorkommen, dass sich ein Programm einwandfrei kompillieren
        // > lsst, jedoch beim Start aus Delphi nach einiger Zeit
        // > das CPU-Fenster geffnet wird. Dort steht dann hufig
        // >    ntdll.DbgBreakPoint    (exactly the same happens in BCB V4)
        // > Dies liegt daran, da Microsoft in manchen Dlls die Funktion
        // > ntdll.DbgBreakPoint vergessen hat.
        // > Microsoft hat ein paar Dlls versehentlich mit Debug-Informationen
        // > ausgeliefert, die noch Breakpoints enthalten, was der Debugger
        // > natrlich meldet. Man muss in so einem Falle zur Laufzeit
        // > den Code patchen. (..)
        // ? Hard to believe. To patch NTDLL.DLL in memory sounds like
        // ? a bad idea since it will probably cause protection faults, etc...

  // Note: even AFTER returning from this atexit()-function,
  //       the damned runtime lib will call some goddamned C++ destructors,
  //      for example: CCorrelator::~CCorrelator() .
  // Furthermore, EVEN LATER, the runtime library (or whatever)
  //       will call VCLINIT.CPP::__ExitVCL(), which then calls VclExit() .
  // But that's not all yet: EVEN LATER, the runtime library(?)
  //       decides to call CSpecEventLog::~CSpecEventLog() ,
  //              CSpectrumBufferPreview::~CSpectrumBufferPreview() [module uses the VCL!],
  //                    CPlotSampleArray::~CPlotSampleArray() [a couple of times],
  //                       CNoiseBlanker::~CNoiseBlanker(),
  //                    CAmFmDeModulator::~CAmFmDeModulator() etc [see T_SOUND_BLACKBOX],
  //                        TChirpFilter::~TChirpFilter(),
  //                          CHumFilter::~CHumFilter(),
  //              CSoundDecimatingBuffer::~CSoundDecimatingBuffer(),
  //              CSoundUpsamplingBuffer::~CSoundUpsamplingBuffer(),
  //                     CSpectrumPlayer::~CSpectrumPlayer(), ....
  //                    CAudioFileWriter::~CAudioFileWriter(),
  //                    CAudioFileReader::~CAudioFileReader() ,
  //         T_PhaseAmplMeter::~TPhaseAmplMeter() -> CGoertzel::~CGoertzel(),
  //                          CFftFilter::~CFftFilter(),
  //                    C_AnyAudioFileIO::~C_AnyAudioFileIO(),
  //                  -> C_WPH_IO_Plugin::~C_WPH_IO_Plugin(),
  //                            CRtcmDec::~CRtcmDec(),
  //                    Sytem::AnsiString::~AnsiString() [long after VclExit()!],
  //                            ComCtrls::Finalization(),
  //         when finally THE DEBUGGER ITSELF crashed with an access violation
  //         in some module called 'bordbk41' (?) or similar .
  //          (at that time, only
  //           __cleanup(),  exit(), and __startup()! remained on the
  //           call stack,   when the debugger went QRT ..  shit ! )
  //         On a second attempt, it became obvious, that after all the above
  //         destruction / atexit - stuff was through, the program was still
  //         stuck in an endless loop inside perseususb.dll, with the
  //         following (possibly corrupted) call stack :
  //       __startup()
  //         -> exit()
  //             -> atexit()
  //                 -> __terminate() -> kernel32.dll -> kernel32.dll -> ....
  //                     -> ntdll.dll -> ntdll.dll
  //                         -> perseususb.dll (at 0x10003B72)
  //                             -> perseususb.dll (at 0x10002BF0) .
  // 
  //  Conclusion : Better add some application-callable "Close"-method
  //       explicitly, instead of relying on the strange desctructor-calling-
  //       sequence of the runtime lib, and/or Borland's VCL .
  //
} // end UTL_ExitFunction()


//---------------------------------------------------------------------------
BOOL UTL_IsCPUIDAvailable(void)
  // Checks if the CPU on which this code will run
  //  supports the CPUID instruction .
  //  (http://en.wikipedia.org/wiki/CPUID : 0x0FA2 as a 16-bit word).

{

  // Implementation for Borland C++Builder based on a code snippet by 'Gambit',
  //   see http://www.borlandtalk.com/reading-unique-cpu-number-vt8951.html .
  __asm
   {
     PUSHFD // direct access to flags no possible, only via stack
     POP    EAX     // flags to EAX
     MOV    EDX,EAX // save current flags
     XOR    EAX,0x00200000 // not ID bit
     PUSH   EAX     // onto stack
     POPFD          // from stack to flags, with not ID bit
     PUSHFD         // back to stack
     POP    EAX     // get back to EAX
     XOR    EAX,EDX // check if ID bit affected
     SHR    EAX,21  // Result=Bit #21
   }
  return _EAX;   // just to make the C compiler happy.. return value in EAX register
} // end UTL_IsCPUIDAvailable()

void UTL_GetCPUID( DWORD dwEAX/*aka "CPUID leaf"*/, DWORD *pFourDWORDs )
{
  __asm
   {
     PUSH  EBX     // Save affected register
     PUSH  EDI
     MOV   EDI,pFourDWORDs // cpuID result in four DWORDS, need pFourDWORDs (pointer) in EDI for 'STOSD'. Boy, those were the days...
     MOV   EAX,dwEAX  // function code in EAX, for example 1 = "Processor Info and Feature Bits" (*)
     DW    0xA20F  // CPUID Command
     // > This returns the CPU's stepping, model, and family information in EAX
     // > (also called the signature of a CPU), feature flags in EDX and ECX,
     // > and additional feature info in EBX.
     STOSD         // CPUID[1] (EAX from CPUID(1)) :  "Stepping" (b3..0), "Model"(7..4), "Family"(11..8), .... details below
     MOV   EAX,EBX
     STOSD         // CPUID[2] (EBX from CPUID(1)) :
     MOV   EAX,ECX
     STOSD         // CPUID[3] (ECX from CPUID(1)) :
     MOV   EAX,EDX
     STOSD         // CPUID[4] (EDX from CPUID(1)) :
     POP   EDI     // Restore registers
     POP EBX
   } // end __asm
  // (*) EAX=1: Processor Info and Feature Bits, according to the Wikipedia article:
  // > This returns the CPU's stepping, model, and family information in EAX
  // > (also called the signature of a CPU), feature flags in EDX and ECX,
  // > and additional feature info in EBX.
  // >   The format of the information in EAX is as follows:
  // >
  // >     3:0  Stepping
  // >     7:4  Model
  // >     11:8  Family
  // >     13:12  Processor Type
  // >     19:16  Extended Model
  // >     27:20  Extended Family

}



//---------------------------------------------------------------------------
void UTL_Init(void) // should be called "EARLY" ON STARTUP - doesn't harm if called twice
{
  int i;
  DWORD dw4[4];
  DWORD dwMaxCpuIdFunctionInEAX;
  OSVERSIONINFO windoze_version_info;


  if( ! UTL_fInitialized )
   {
     CoInitialize(0);  // Init COM (formerly done in VARIOUS other modules).
      // 2012-08-09 : Moved this ugly CoInitialize/CoUninitialize-BULLSHIT
      //              HERE - into c:\CBproj\SoundUtl\UTILITY1.cpp -
      //              which ensures this ugly stuff is called exactly ONCE.
      DOBINI();
     UTL_i32NrOfAllocatedBlocks = 0;
     UTL_i32MaxNrOfAllocatedBlocks = 0;
     InitializeCriticalSection( &UTL_csRunLog );
     InitializeCriticalSection( &UTL_csMalloc );
     atexit(UTL_ExitFunction);  // make sure the critical section will be properly released


     EnterCriticalSection( &UTL_csMalloc );  // see if we 'survive' this ;-)
     for(i=0; i<UTL_MAX_ALLOC_BLOCKS; ++i)
      {
       memset(&UTL_sAllocBlocks[i], 0, sizeof(T_UTL_AllocBlockInfo) );
      }
     LeaveCriticalSection( &UTL_csMalloc );

     // Added 2015-03 : Find out on which CPU we're running,
     //                 and which extensions (to the basic x86 instruction set)
     //                 are available .
     memset( UTL_dw4CPUID_ProcInfoAndFeatures, 0, sizeof(UTL_dw4CPUID_ProcInfoAndFeatures) );
     memset( UTL_CPUID_VendorID, 0, sizeof(UTL_CPUID_VendorID) );
     if( UTL_IsCPUIDAvailable() )  // -> TRUE when the "CPUID" instruction is available
      {
         UTL_GetCPUID( 0/*EAX*/, dw4 ); // from http://en.wikipedia.org/wiki/CPUID#EAX.3D0:_Get_vendor_ID :
         // > This returns the CPU's manufacturer ID string  a twelve-character
         // > ASCII string stored in EBX, EDX, ECX (in that order).
         // > The highest basic calling parameter (largest value that EAX
         // > can be set to before calling CPUID) is returned in EAX.
         *(DWORD*)(UTL_CPUID_VendorID+0) = dw4[1]; // EBX
         *(DWORD*)(UTL_CPUID_VendorID+4) = dw4[3]; // EDX
         *(DWORD*)(UTL_CPUID_VendorID+8) = dw4[2]; // ECX
         dwMaxCpuIdFunctionInEAX = dw4[0];         // EAX

         // On a Lenovo Z61m with a 'Centrino Duo', got here with
         // UTL_CPUID_VendorID = "GenuineIntel" and dwMaxCpuIdFunctionInEAX=10 (!)
         //  (which confirms the "CPUID"-instruction works as it should)
         //
         //
         if( dwMaxCpuIdFunctionInEAX >= 1 )  // may we call 'CPUID' with EAX=1 ?
          { UTL_GetCPUID( 1/*EAX*/, UTL_dw4CPUID_ProcInfoAndFeatures ); // reads four DWORDS from "CPUID" with EAX=1 : "Processor Info and Feature Bits"
          }
         // On a Lenovo Z61m with Intel 'Centrino Duo', got here with
         // UTL_dw4CPUID_ProcInfoAndFeatures[0] = EAX from CPUID(1)) = 0x0000 06F2 =
         //   0000 0000 0000 0000 0000 0110 1111 0010 :
         //   |||| |||| |||| |||| |||| |||| |||| |\\\_ b3..0 = "stepping" :
         //   |||| |||| |||| |||| |||| |||| \\\\_\____ b7..4 = "model"
         //   |||| |||| |||| |||| |||| \\\\___________ b11..8 = "family" :
         //    > If "family" (here: 6) is lower than 15,
         //    > only the "family" and "model" fields should be used.
         //
         // UTL_dw4CPUID_ProcInfoAndFeatures[1] = EBX from CPUID(1)) = 0x0002 0800
         //
         // UTL_dw4CPUID_ProcInfoAndFeatures[2] = ECX from CPUID(1)) = 0x0000 E3BD =
         //   0000 0000 0000 0000 1110 0011 1011 1101 :
         //   |||| |||| |||| |||| |||| |||| |||| ||||_ b0 "SSE3" ("Prescott New Instructions") available
         //   |||| |||| |||| |||| |||| |||| |||| |||__ b1 PCLMULQDQ : 0=not available
         //   |||| |||| |||| |||| |||| |||| |||| ||___ b2 "dtes64" (64-bit debug store)
         //   |||| |||| |||| |||| |||| |||| |||| |____ b3 "monitor"
         //   |||| |||| |||| |||| |||| |||| ||||______ b4 "CPL qualified debug store"
         //   |||| |||| |||| |||| |||| |||| |||_______ b5 "Virtual Machine Extensions"
         //   |||| |||| |||| |||| |||| |||| ||________ b6 0=NO "Safer Mode Extensions"
         //   |||| |||| |||| |||| |||| |||| |_________ b7 "Enhanced SpeedStep"
         //   |||| |||| |||| |||| |||| ||||___________ b8 "Thermal Monitor 2"
         //   |||| |||| |||| |||| |||| |||____________ b9 "Supplemental SSE3 instructions"
         //   |||| |||| |||| |||| |||| ||_____________ b10 "L1 Context ID" (0=not supported)
         //   |||| |||| |||| |||| |||| |______________ b11 reserved
         //   |||| |||| |||| |||| ||||________________ b12 "FMA3" : 0 = "Fused Multiply-Add" *NOT* supported. Aaaaargh !
         //   |||| |||| |||| |||| |||_________________ b13 "CMPXCHG16B"
         //   |||| |||| |||| |||| ||__________________ b14 "xptr"
         //   |||| |||| |||| |||| |___________________ b15 "pdcm"
         //   |||| |||| |||| ||||_____________________ (bit 16 = reserved)
         //   |||| |||| |||| |||______________________ b17 "dca"
         //   |||| |||| |||| ||_______________________ b18 "SSE4.1 instructions": 0=not supported
         //   |||| |||| |||| |________________________ b19 "SSE4.2 instructions": 0=not supported
         //   \\\\_\\\\_\\\\__________________________ none of the other 'features' supported
         //
         //
         // UTL_dw4CPUID_ProcInfoAndFeatures[3] = EDX from CPUID(1)) = 0xBFEB FBFF :
         //   1011 1111 1110 1011 1111 1011 1111 1111
         //   |||| |||| |||| |||| |||| |||| |||| ||||_ b0 "fpu" (Onboard x87 FPU)
         //   |||| |||| |||| |||| |||| |||| |||| |||__ b1 "vme"
         //   |||| |||| |||| |||| |||| |||| |||| ||___ b2 "debugging extensions CR4"
         //   |||| |||| |||| |||| |||| |||| |||| |____ b3 "Page Size Extension"
         //   |||| |||| |||| |||| |||| |||| ||||______ b4 "Time Stamp Counter"
         //   |||| |||| |||| |||| |||| |||| |||_______ b5 "Model-specific registers"
         //   |||| |||| |||| |||| |||| |||| ||________ b6 "Physical Address Extension"
         //   |||| |||| |||| |||| |||| |||| |_________ b7 "Machine Check Extension"
         //   |||| |||| |||| |||| |||| ||||___________ b8 "CMPXCHG8" (compare and swap 8)
         //   |||| |||| |||| |||| |||| |||____________ b9 "Onboard APIC"
         //   |||| |||| |||| |||| |||| ||_____________ b10 (reserved)
         //   |||| |||| |||| |||| |||| |______________ b11 SYSTENTER and SYSEXIT
         //   |||| |||| |||| |||| ||||________________ b12 "mtrr"
         //   |||| |||| |||| |||| |||_________________ b13 "pge"
         //   |||| |||| |||| |||| ||__________________ b14 "mca"
         //   |||| |||| |||| |||| |___________________ b15 "cmov and FCMOV"
         //   |||| |||| |||| ||||_____________________ b16 "Page Attribute Table"
         //   |||| |||| |||| |||______________________ b17 "pse-36"
         //   |||| |||| |||| ||_______________________ b18 "Processor Serial Number"
         //   |||| |||| |||| |________________________ b19
         //   |||| |||| ||||__________________________ b20 (reserved)
         //   |||| |||| |||___________________________ b21
         //   |||| |||| ||____________________________ b22
         //   |||| |||| |_____________________________ b23 "MMX instructions"
         //   |||| ||||_______________________________ b24
         //   |||| |||________________________________ b25
         //   |||| ||_________________________________ b26
         //   |||| |__________________________________ b27
         //   ||||____________________________________ b28
         //   |||_____________________________________ b29
         //   ||______________________________________ b30 "ia64" (0)
         //   |_______________________________________ b31
         //
         // Even with little hope that "FMA4" ( 4 operands fused multiply-add )
         // are available on the Lenovo Z61m, ask the 'CPUID' instruction for it:
         // EAX=80000000h: "Get Highest Extended Function Supported" :
         UTL_GetCPUID( 0x80000000/*EAX*/, dw4 );
         // On the Lenovo Z61m, got here with dw4[0] = EAX = 0x80000008 (ok for a "Core Duo")
         if( dw4[0] >= (DWORD)0x80000001 )
          { // At least the CPU says 'it's ok to ask for "Extended Processor Info and Feature Bits"..
            UTL_GetCPUID( 0x80000001/*EAX*/, dw4 );
            // On the Lenovo Z61m, got here with ...
            //   dw4[0] (EAX) = 0 (none of the "AMD feature flags")
            //   dw4[1] (EBX) = 0 (none of the "AMD feature flags")
            //   dw4[2] (ECX) = 1 = "lahf_lm" but no other "AMD feature flags"
            //                    (ECX bit 16 NOT set -> FMA4 *not* supported)
            //   dw4[3] (EDX) = 0x2010 0000
            //                    | |_______ bit 20 = "No-Execute bit" supported
            //                    |_________ bit 29 = "long mode"
          }

      } // end if < 'CPUID' instruction available ? >
     else // 'CPUID' instruction not available ? ?
      {  // Must be running on what Wikipedia calls an 'Earlier Intel 486' ?!
      }


     // Added 2020-12-17 : Find out under which Windoze version we're running.
     //   For example, Sound_OpenSoundcardInputControlPanel() needs this info
     //   to decide how to open the dreadful ever-chaning soundcard volume
     //   control panel ("sndvol32.exe", "sndvol.exe", and whatever comes next)
     // Must find out the windoze version (MAJOR version, at least)
     // because of a zillion of subtle and annoying differences
     //  (especially between XP, Makes-Me-Vomit-"Vista", Windows 7 / 8 / 10 ..)
     // We don't want to have dozens of GetVersionEx-calls all over the place.
     windoze_version_info.dwOSVersionInfoSize = sizeof( OSVERSIONINFO );
     if( GetVersionEx( &windoze_version_info ) )
      { UTL_iWindowsMajorVersion = windoze_version_info.dwMajorVersion;
        if( windoze_version_info.dwMajorVersion >= 5 )
         { // Win2000, Win XP, etc..  , but definitely NOT win98:
         }
        if( windoze_version_info.dwMajorVersion >= 6 )
         { // Vista (Kotz, Brech, Wrg) oder Windows 7 ?
           // Warum verwenden die Vollpfosten fr BEIDES die gleiche "Major Version" ?
           // Elender Schwachsinn ! Windows "Vista" ist wohl Windows V6.0,
           //                       und "Windows 7" ist wohl Windows V6.1 .
         }
      }
     else // GetVersionEx() failed. WHY AM I NOT SURPRISED ?
      { // Leave UTL_iWindowsMajorVersion as-is. Maybe some other function
        // is smarter than GetVersionEx() and UTL_iWindowsMajorVersion has
        // already been set somewhere else.
      }

     // Dynamically "load" (i.e. retrieve the addresses of) some Kernel
     // functions that were unknown at the time Borland developed their
     // variant of "windows.h" (or headers included from there).
     //
     // To measure the CPU load, we'd like to have GetModuleHandle()
     // and                for the implementation of CalculateCPULoad() :
     // Use GetModuleHandle() and GetProcAddress() to load these functions
     // dynamically at runtime. Beware, depending on the windows version,
     // these functions MAY or MAY NOT exist at all - so check all function
     // pointers for NULL before trying to invoke them !
     pGetSystemTimes = (tGetSystemTimes)GetProcAddress(GetModuleHandle("KERNEL32"), "GetSystemTimes");


     UTL_fInitialized = TRUE;
     DOBINI();
   }
}


//---------------------------------------------------------------------------
static float CalculateCPULoad(unsigned long long idleTicks, unsigned long long totalTicks)
{
   static unsigned long long _previousTotalTicks = 0;
   static unsigned long long _previousIdleTicks = 0;

   unsigned long long totalTicksSinceLastTime = totalTicks-_previousTotalTicks;
   unsigned long long idleTicksSinceLastTime  = idleTicks-_previousIdleTicks;

   float ret = 1.0f-((totalTicksSinceLastTime > 0) ? ((float)idleTicksSinceLastTime)/totalTicksSinceLastTime : 0);

   _previousTotalTicks = totalTicks;
   _previousIdleTicks  = idleTicks;
   return ret;
}

//---------------------------------------------------------------------------
static unsigned long long FileTimeToInt64(const FILETIME & ft)
{ return (((unsigned long long)(ft.dwHighDateTime))<<32)
         |((unsigned long long)ft.dwLowDateTime);
}

//---------------------------------------------------------------------------
float UTL_GetCPULoad(void) // Call PERIODICALLY(!) at regular intervals (*) !
  // Returns 1.0f for "CPU fully pinned", 0.0f for "CPU idle", or somewhere in between.
  // Returns -1.0 on error.
  // (*) You'll need to call this at regular intervals, since it measures
  //     the load between the previous call and the current one.
  //     In Spectrum Lab, periodically called from SpecMain.cpp : Timer1Timer().
{
   FILETIME idleTime, kernelTime, userTime;
   // Borland's "windows.h" neither knew GetSystemTimes() nor FileTimeToInt64().
   // Someone (Remy Lebeau) wrote, 15 years ago:
   // > GetSystemTimes() is only available on XP SP1, 2003 Server, and Vista
   // > (Longhorn). BCB6 was released years before any of them were released.
   // > Use GetModuleHandle() and GetProcAddress() to load the function
   // > dynamically at runtime.
   // Added that code in UTL_Init(), 2020-06.
#if(0)
   if( GetSystemTimes(&idleTime, &kernelTime, &userTime) )
#else
   if( ( pGetSystemTimes != NULL ) // successfully "loaded" from KERNEL32.DLL ?
    && ( pGetSystemTimes(&idleTime, &kernelTime, &userTime) ) ) // <- call via function pointer !
#endif // GetSystemTimes() directly available, or dynamically loaded ?
    { return CalculateCPULoad( FileTimeToInt64(idleTime),
                               FileTimeToInt64(kernelTime)
                              +FileTimeToInt64(userTime) );
    }

   // Arrived here ? Something went wrong, or GetSystemTimes() not available, so ..
   return -1.0f;
} // UTL_GetCPULoad()

//---------------------------------------------------------------------------
void UTL_OpenRunLongFile( char *pszLogFileName )
{
  SYSTEMTIME ti;
  if( pszLogFileName && (pszLogFileName[0]>0) )
   {  DeleteFile( pszLogFileName ); // write a new, "clean" file when it's due
      strncpy( UTL_sz255LogFile, pszLogFileName, 255);
      ::GetSystemTime( &ti );       // retrieves the system time IN UTC (hooray)
      UTL_WriteRunLogEntry( "Logfile created on %04d-%02d-%02d %02d:%02d:%02d UTC",
          (int)ti.wYear, (int)ti.wMonth, (int)ti.wDay,
          (int)ti.wHour, (int)ti.wMinute,(int)ti.wSecond );
   }
} // end UTL_OpenRunLongFile()

//---------------------------------------------------------------------------
void UTL_WriteRunLogEntry( char * pszFormatString, ... )
  // Opens the "run-log"-file, appends a line of text, and closes again (!)
  //    because this program / the OS / the PC may crash and we don't want
  //    to lose the debugging info .
  // To avoid flooding the harddisk with error messages (if the same error
  //  is detected & written over and over again), the logfile is
  //  deliberately limited to a certain number of messages.
  //
  // Note: In SPECTRUM LAB, besides the "run log"-file, there's also a
  //       scrolling text display ("Debugging Display" / erasable "Error History").
  //       To show IMPORTANT messages also there (without confusing the user),
  //       call DEBUG_EnterErrorHistory() ... it calls UTL_WriteRunLogEntry(),
  //       not vice versa !
{
  va_list arglist;  // very 'arglistig' if not handled properly (crash,bang,boom)
  SYSTEMTIME ti;
  char temp[1024];
  char *cp;

  if( ! UTL_fInitialized )
     return;

  if( UTL_iLogFileNrOfErrors > 2000 )   // 2009-07 : increased from 1000 to 2000
     return;

  EnterCriticalSection( &UTL_csRunLog );

  cp = temp;
  ::GetSystemTime( &ti );
  wsprintf( cp, "%02d:%02d:%02d.%01d ",
                (int)ti.wHour,  (int)ti.wMinute,
                (int)ti.wSecond,(int)ti.wMilliseconds / 100 );
  cp += strlen(cp);

  // Print to string and append to edit control
  va_start(arglist, pszFormatString);
  vsprintf(cp, pszFormatString, arglist);
  va_end(arglist);    // MUST be called, always !
  cp += strlen(cp);

  // Send this message to the SYSTEM DEBUGGER ?
  if( UTL_fRunLog_UseOutputDebugString )
   { OutputDebugString( temp );  // here: WITHOUT trailing CR+NL  !
     // > The OutputDebugString function sends a string to the debugger for the current application.
     // (In Borland C++Builder erscheinen diese Strings im 'Ereignisprotokoll'
     //  mit dem Vorsatz "ODS:", was wohl "Output Debug String" heissen soll)
   }

  // Append this message to the "run-log" file ?
  if( UTL_sz255LogFile[0] > 32 )
   { int iHandle = _rtl_open(UTL_sz255LogFile, O_RDWR);
     if(iHandle>=0)
      { // successfully OPENED the file, jump to its end :
        lseek( iHandle, 0, SEEK_END );
      }
     else // could not OPEN the file, so CREATE it :
      { iHandle = _rtl_creat(UTL_sz255LogFile, 0/*attrib*/ );
      }
     if(iHandle>=0)
      { strcpy( cp, "\r\n" );  // actually APPENDS CR+NL to 'temp'
        _rtl_write( iHandle, temp, strlen(temp) );
        _rtl_close( iHandle );
        ++UTL_iLogFileNrOfErrors;
      }
   } // end if < also log the message text AS A FILE ? >

  LeaveCriticalSection( &UTL_csRunLog );

} // end UTL_WriteRunLogEntry()


//---------------------------------------------------------------------------
void *UTL_malloc(DWORD size)
   // Replacement for malloc(),
   //  with additional housekeeping for debugging purposes .
   // Caution: not 100% thread-safe, but no big issue.. !
{
  return UTL_NamedMalloc( NULL/*no name*/, size);
}

//---------------------------------------------------------------------------
void *UTL_NamedMalloc( char * pszMemoryBlockName, DWORD size)
   // Like malloc(), but this function allows the additional definition
   // of a NAME for the allocated memory; which can help to track down
   // non-freed memory blocks (when an application terminates, it can
   // list the names of all erroneously NON-FREED memory blocks
   // in the error log).
   // Added 2007-09-30 when problems with non-freed blocks occurred in Spectrum Lab .
   // Since 2020-10-11, UTL_NamedMalloc() also GUARANTEES that, if successfull,
   //       the returned block ALWAYS begins at an 8-byte aligned address
   //        - something that Borland C++Builder V6's standard "malloc()"
   //          did NOT - see WB's observations in VorbisFileIO.cpp !
   //       Since we don't know in advance what the basic malloc() will deliver,
   //       make the INTERNALLY ALLOCATED block larger than necessary (by four
   //       bytes, since Borland's malloc() always returned 4-byte aligned blocks,
   //       so in the worst case we need to 'waste' 4 bytes before the address
   //       returned by UTL_NamedMalloc(). Later, when UTL_free() calls free(),
   //       we need to pass in the ORIGINAL (possibly misaligned) address
   //       delivered by malloc(). Guess where we store that address now ...
{
  int i,j;
  BYTE *pbBlock = NULL;  // allocation may FAIL, in that case return a NULL pointer !
  BYTE *pbMisalignedBlock;
  T_UTL_AllocBlockInfo *pBlockInfo;
  T_UTL_AllocBlockHdr  *pBlockHeader;
  int ci;


  if( ! UTL_fInitialized )
     UTL_Init();
  if( ! UTL_fInitialized )
     return NULL;

  EnterCriticalSection( &UTL_csMalloc );  // make this stuff thread-safe

  i = UTL_i32NextUnusedAllocBlock;
  j = 10;
  do  // look in the 'neighbourhood' of the last known 'unused' info-block :
   {
     if( (i>=0) && (i<UTL_MAX_ALLOC_BLOCKS) && (UTL_sAllocBlocks[i].pvBlock==NULL) )
      { // no need for a long search (for the next unused block)..
        UTL_i32NextUnusedAllocBlock = i+1;  // for the NEXT time (!)
        break;
      }
     else
      { ++i;
        if(i>=UTL_MAX_ALLOC_BLOCKS)
         { i=0;
         }
      }
   }while(j--);
  if( j<=0 ) i=-1; // no luck in the 'neighbourhood' of UTL_i32NextUnusedAllocBlock
  if( (i<0) || (i>=UTL_MAX_ALLOC_BLOCKS) )  // -> search the entire, HUGE, array
   { for(i=0; i<UTL_MAX_ALLOC_BLOCKS; ++i)
      { if( UTL_sAllocBlocks[i].pvBlock == NULL )
         { UTL_i32NextUnusedAllocBlock = i+1; // for the NEXT time (!)
           break;
         }
      }
   }
  if( i>=0 && i<UTL_MAX_ALLOC_BLOCKS && UTL_sAllocBlocks[i].pvBlock==NULL )  // found an unused block ?
   {
     pBlockInfo = &UTL_sAllocBlocks[i];
     //  Note: Allocate 4 additional "trailing" bytes for internal purposes,
     //         ("integrity testing", see UTL_free() ) !
     //        plus 4 bytes for an index back into UTL_sAllocBlocks[i] .
     DOBINI();
     pbMisalignedBlock = (BYTE*)malloc( size
                           + 8/*bytes "wasted" to defeat MISALIGNMENT in ANY case*/
                           + 4/*integrity check pattern*/
                           + sizeof(T_UTL_AllocBlockHdr) );
     DOBINI();
     if( pbMisalignedBlock != NULL )
      { pbBlock = (BYTE*)(((DWORD)pbMisalignedBlock + 7UL) & 0xFFFFFFF8UL); // only works with 32-bit pointers !
        // Examples (hex addresses, least significant bits):
        //  pbMisalignedBlock    ..0  ..1  ..2  ..3  ..4  ..5 ..6 ..7 ..8 .09
        //  pbMisalignedBlock+7  ..7  ..8  ..9  ..A  ..B  ..C ..D ..E ..F .10
        //  pbBlock              ..0  ..8  ..8  ..8  ..8  ..8 ..8 ..8 ..8 .10
        //  #bytes "wasted"        0    7    6    5    4    3   2   1   0   7
        //   (BEFORE the aligned block)
        ++UTL_i32NrOfAllocatedBlocks;   // just for debugging (!)
        if(UTL_i32MaxNrOfAllocatedBlocks < UTL_i32NrOfAllocatedBlocks )
         { UTL_i32MaxNrOfAllocatedBlocks = UTL_i32NrOfAllocatedBlocks;
         }
        pBlockHeader = (T_UTL_AllocBlockHdr*)pbBlock;
        pBlockHeader->iIndexIntoInfo = i;
        pBlockHeader->pbMisalignedBlock = pbMisalignedBlock; // important for "free()" !
        pBlockInfo->pvBlock = (void*)pbBlock;
        // Note: T_UTL_AllocBlockInfo.pvBlock actually points to
        // an T_UTL_AllocBlockHdr (see below), followed by the 'netto' data !
        pbBlock += sizeof(T_UTL_AllocBlockHdr); // result pointer (points to "netto" data)
        pBlockInfo->dwAllocatedSize = size; // this is the 'officially' allocated size, for the user
 #if( 1 )   //
            // Copy the optional NAME of a memory block (only for debugging) .
            // Do NOT use a string manipulation function like "strcpy" here..
            // the destination isn't necessarily a ZERO-TERMINATED C-string !
        if( pszMemoryBlockName != NULL )
         { for(ci=0; ci<8; ++ci)
            { if( *pszMemoryBlockName!='\0' )
               { pBlockInfo->c8BlockName[ci] = *pszMemoryBlockName++;
               }
              else
               { pBlockInfo->c8BlockName[ci] = '\0';
               }
            }
         }
        else // the poor li'l memory block hasn't got a name :
         { for(ci=0; ci<8; ++ci)
            { pBlockInfo->c8BlockName[ci] = '\0';
            }
         }
 #else
       (void)pszMemoryBlockName;  // suppress "unused" warning
 #endif // SWI_ALLOW_TESTING
        *(pbBlock + size)   = 0x11;
        *(pbBlock + size+1) = 0x22; // magic code after "officially" allocated block
      } // end if < native malloc successfull ? >
   } // end if < found an UNUSED block >

  LeaveCriticalSection( &UTL_csMalloc );
  DOBINI();

  return (void*)pbBlock;
} // end UTL_NamedMalloc()

#if( SWI_ALLOW_TESTING )
/***************************************************************************/
BOOL UTL_IsMallocBlockValid( void * block )
{
  int i; DWORD size; BYTE *pbBlock; T_UTL_AllocBlockInfo *pBlockInfo;

  if( block==NULL )
   {
     DEBUGGER_BREAK();  // SET BREAKPOINT HERE,  or in DebugU1.cpp !
     return FALSE;
   }

  // Note: T_UTL_AllocBlockInfo.pvBlock actually points
  //       to a T_UTL_AllocBlockHdr, followed by the 'netto' data .
  // The caller of UTL_IsMallocBlockValid() doesn't know about that,
  // and passes a pointer to the NETTO data.   So :
  pbBlock = (BYTE*)block - sizeof(T_UTL_AllocBlockHdr);
  for(i=0; i<UTL_MAX_ALLOC_BLOCKS; ++i)
   {
     if( UTL_sAllocBlocks[i].pvBlock == (void*)pbBlock )
      { pBlockInfo = &UTL_sAllocBlocks[i];
        size = pBlockInfo->dwAllocatedSize;
        if( (pbBlock[size]!=0x11) || (pbBlock[size+1]!=0x22) )
         { // missing magic code after allocated block ?
           // The application must have "destroyed" this by accident.
           DEBUGGER_BREAK();  // SET BREAKPOINT HERE,  or in DebugU1.cpp !
           return FALSE;
         }
        else return TRUE;
      }
   } // end for
  return FALSE;
} // end UTL_IsMallocBlockValid()
#endif // ( SWI_ALLOW_TESTING )

//---------------------------------------------------------------------------
void UTL_free(void * block)
   // Replacement for malloc() / free() ,
   //  with additional housekeeping for debugging purposes
{
  int i;
  // BOOL found_block = FALSE;
  BOOL ok_to_free  = TRUE;
  DWORD size; BYTE *pbBlock;
  T_UTL_AllocBlockInfo *pBlockInfo;
  T_UTL_AllocBlockHdr  *pBlockHeader;
  char sz255[256];
 // #if( SWI_ALLOW_TESTING )
 // char *cp; int ci;
 // #endif // SWI_ALLOW_TESTING

  DOBINI();
  if( UTL_fInitialized && (block!=NULL) )
   {
     DOBINI();
     EnterCriticalSection( &UTL_csMalloc );  // make this stuff thread-safe
     DOBINI();
     // Note: T_UTL_AllocBlockInfo.pvBlock actually points
     //       to a T_UTL_AllocBlockHdr, followed by the 'netto' data .
     // The caller of UTL_IsMallocBlockValid() doesn't know about that,
     // and passes a pointer to the NETTO data.   So :
     pbBlock = ((BYTE*)block) - sizeof( T_UTL_AllocBlockHdr ); //
     pBlockHeader = (T_UTL_AllocBlockHdr*)pbBlock;
     i = pBlockHeader->iIndexIntoInfo;  // 2015-01-15 : access violation after "killing" audio thread
     if( i>=0 && i<UTL_MAX_ALLOC_BLOCKS )
      { if( UTL_sAllocBlocks[i].pvBlock == (void*)pbBlock )
         {  pBlockInfo = &UTL_sAllocBlocks[i];
            size = pBlockInfo->dwAllocatedSize;
            pbBlock = (BYTE*)pBlockInfo->pvBlock + sizeof(T_UTL_AllocBlockHdr);
            if( (pbBlock[size]!=0x11) || (pbBlock[size+1]!=0x22) )
             {  // missing magic code after allocated block ?
                // The application must have "destroyed" this by accident.
                // DEBUGGER_BREAK();  // SET BREAKPOINT HERE,  or in DebugU1.cpp !
                strcpy( sz255, "SERIOUS BUG " );
                if( pBlockInfo->c8BlockName[0] != '\0' )
                 { // the defective block has a name : tell it to the user !
                   strcat( sz255, "in " );
                   AppendBlockName( sz255, pBlockInfo->c8BlockName );
                 }
                strcat( sz255, ": end of alloc-block overwritten" );
                UTL_WriteRunLogEntry( sz255 );
                // Already caught one of these SERIOUS BUGS in SpecLab ,
                //  where "debug_run_log.txt" will be written if you specify
                //  the command line parameter /debug or \debug  .
                // It later turned out to be EVEN MORE HELPFUL to have
                //  *names* for critical (or suspicious) memory allocations.
                ok_to_free = FALSE; // better don't try to free 'corrupted' memory
             }
            UTL_sAllocBlocks[i].pvBlock = NULL;
            UTL_sAllocBlocks[i].dwAllocatedSize = 0;
            UTL_i32NextUnusedAllocBlock = i;
         } // end if( UTL_sAllocBlocks[i].pvBlock == (void*)pbBlock )
        else
         {  // DEBUGGER_BREAK();  // SET BREAKPOINT HERE,  or in DebugU1.cpp !
            // 2012-11-03 : Got here when called from SoundThd_DeleteAllAudioChunkBuffers(),
            //              pBlockHeader->iIndexIntoInfo = 0,
            //              AllocBlocks[0].c8Blockname = "Spectrum" .
            sprintf( sz255, "UTL_free: Failed to deallocate block #%d, \"",
                            (int)i );
            AppendBlockName( sz255, UTL_sAllocBlocks[i].c8BlockName );
            strcat( sz255, "\", but address in header doesn't match" );
            UTL_WriteRunLogEntry( sz255 );
            ok_to_free = FALSE; // better don't try to free 'corrupted' memory
         }
      } // end if < valid index into the alloc-block-info >
     else
      { UTL_WriteRunLogEntry( "UTL_free: Someone tried to free an un-allocated memory block !" );
      }

     DOBINI();
     if( ok_to_free && (pBlockHeader->pbMisalignedBlock!=NULL) )
      { // ex: free(pBlockHeader); // frees the 'internal block header' AND the user's netto data !
        free( pBlockHeader->pbMisalignedBlock ); // free the 'internal block header',
               // the 0..7 "wasted", misaligned bytes BEFORE the aligned block,
               // AND (also in the same 'originally mallocated block')
               // the user's netto data !
      }
     else
      {  // since 2012-11-03, better do NOT free obviously 'corrupted' blocks
      }
     if( UTL_i32NrOfAllocatedBlocks > 0 )
      { --UTL_i32NrOfAllocatedBlocks;   // just for debugging (!)
      }
     DOBINI();
     LeaveCriticalSection( &UTL_csMalloc );
     DOBINI();
   }
}

//---------------------------------------------------------------------------
long UTL_GetAllocdMemTotal(void)  // result in BYTES !
{ long lResult=0;
  int i;
  if( UTL_fInitialized )
   {
     EnterCriticalSection( &UTL_csMalloc );  // make this stuff thread-safe
     for(i=0; i<UTL_MAX_ALLOC_BLOCKS; ++i)
      { if( UTL_sAllocBlocks[i].pvBlock != NULL )
           lResult += (long)UTL_sAllocBlocks[i].dwAllocatedSize;
      }
     LeaveCriticalSection( &UTL_csMalloc );
   }
  return lResult;
}

//---------------------------------------------------------------------------
long UTL_GetAllocdMemNumBlocks(void)
{ long lResult=0;
  int i;
  if( UTL_fInitialized )
   {
     EnterCriticalSection( &UTL_csMalloc );  // make this stuff thread-safe
     for(i=0; i<UTL_MAX_ALLOC_BLOCKS; ++i)
      { if( UTL_sAllocBlocks[i].pvBlock != NULL )
           ++lResult;
      }
     LeaveCriticalSection( &UTL_csMalloc );
   }
  return lResult;
}


typedef union // type used for 8/16-bit-fiddling in UTL_GetBlockChecksum()
{
   unsigned short int both;
   struct
    {
      unsigned char lo;
      unsigned char hi;
    } sbyte;
} LO_HI;

unsigned short int CRC_TABLE[] = {
0x0000  ,0x1021  ,0x2042  ,0x3063  ,0x4084  ,0x50A5  ,0x60C6  ,0x70E7  ,
0x8108  ,0x9129  ,0xA14A  ,0xB16B  ,0xC18C  ,0xD1AD  ,0xE1CE  ,0xF1EF  ,
0x1231  ,0x0210  ,0x3273  ,0x2252  ,0x52B5  ,0x4294  ,0x72F7  ,0x62D6  ,
0x9339  ,0x8318  ,0xB37B  ,0xA35A  ,0xD3BD  ,0xC39C  ,0xF3FF  ,0xE3DE  ,
0x2462  ,0x3443  ,0x0420  ,0x1401  ,0x64E6  ,0x74C7  ,0x44A4  ,0x5485  ,
0xA56A  ,0xB54B  ,0x8528  ,0x9509  ,0xE5EE  ,0xF5CF  ,0xC5AC  ,0xD58D  ,
0x3653  ,0x2672  ,0x1611  ,0x0630  ,0x76D7  ,0x66F6  ,0x5695  ,0x46B4  ,
0xB75B  ,0xA77A  ,0x9719  ,0x8738  ,0xF7DF  ,0xE7FE  ,0xD79D  ,0xC7BC  ,
0x48C4  ,0x58E5  ,0x6886  ,0x78A7  ,0x0840  ,0x1861  ,0x2802  ,0x3823  ,
0xC9CC  ,0xD9ED  ,0xE98E  ,0xF9AF  ,0x8948  ,0x9969  ,0xA90A  ,0xB92B  ,
0x5AF5  ,0x4AD4  ,0x7AB7  ,0x6A96  ,0x1A71  ,0x0A50  ,0x3A33  ,0x2A12  ,
0xDBFD  ,0xCBDC  ,0xFBBF  ,0xEB9E  ,0x9B79  ,0x8B58  ,0xBB3B  ,0xAB1A  ,
0x6CA6  ,0x7C87  ,0x4CE4  ,0x5CC5  ,0x2C22  ,0x3C03  ,0x0C60  ,0x1C41  ,
0xEDAE  ,0xFD8F  ,0xCDEC  ,0xDDCD  ,0xAD2A  ,0xBD0B  ,0x8D68  ,0x9D49  ,
0x7E97  ,0x6EB6  ,0x5ED5  ,0x4EF4  ,0x3E13  ,0x2E32  ,0x1E51  ,0x0E70  ,
0xFF9F  ,0xEFBE  ,0xDFDD  ,0xCFFC  ,0xBF1B  ,0xAF3A  ,0x9F59  ,0x8F78  ,
0x9188  ,0x81A9  ,0xB1CA  ,0xA1EB  ,0xD10C  ,0xC12D  ,0xF14E  ,0xE16F  ,
0x1080  ,0x00A1  ,0x30C2  ,0x20E3  ,0x5004  ,0x4025  ,0x7046  ,0x6067  ,
0x83B9  ,0x9398  ,0xA3FB  ,0xB3DA  ,0xC33D  ,0xD31C  ,0xE37F  ,0xF35E  ,
0x02B1  ,0x1290  ,0x22F3  ,0x32D2  ,0x4235  ,0x5214  ,0x6277  ,0x7256  ,
0xB5EA  ,0xA5CB  ,0x95A8  ,0x8589  ,0xF56E  ,0xE54F  ,0xD52C  ,0xC50D  ,
0x34E2  ,0x24C3  ,0x14A0  ,0x0481  ,0x7466  ,0x6447  ,0x5424  ,0x4405  ,
0xA7DB  ,0xB7FA  ,0x8799  ,0x97B8  ,0xE75F  ,0xF77E  ,0xC71D  ,0xD73C  ,
0x26D3  ,0x36F2  ,0x0691  ,0x16B0  ,0x6657  ,0x7676  ,0x4615  ,0x5634  ,
0xD94C  ,0xC96D  ,0xF90E  ,0xE92F  ,0x99C8  ,0x89E9  ,0xB98A  ,0xA9AB  ,
0x5844  ,0x4865  ,0x7806  ,0x6827  ,0x18C0  ,0x08E1  ,0x3882  ,0x28A3  ,
0xCB7D  ,0xDB5C  ,0xEB3F  ,0xFB1E  ,0x8BF9  ,0x9BD8  ,0xABBB  ,0xBB9A  ,
0x4A75  ,0x5A54  ,0x6A37  ,0x7A16  ,0x0AF1  ,0x1AD0  ,0x2AB3  ,0x3A92  ,
0xFD2E  ,0xED0F  ,0xDD6C  ,0xCD4D  ,0xBDAA  ,0xAD8B  ,0x9DE8  ,0x8DC9  ,
0x7C26  ,0x6C07  ,0x5C64  ,0x4C45  ,0x3CA2  ,0x2C83  ,0x1CE0  ,0x0CC1  ,
0xEF1F  ,0xFF3E  ,0xCF5D  ,0xDF7C  ,0xAF9B  ,0xBFBA  ,0x8FD9  ,0x9FF8  ,
0x6E17  ,0x7E36  ,0x4E55  ,0x5E74  ,0x2E93  ,0x3EB2  ,0x0ED1  ,0x1EF0
}; // end CRC_TABLE[]



//---------------------------------------------------------------------------
long UTL_GetBlockChecksum(void *pBlock, int iBlockSize)
  // Often used to check if the contents of a STRUCTURE has been changed.
  // Not 100% bulletproof but fast.
{
#if(0)  // old stuff, replaced by a real CRC in 2018-11-29 :
  WORD wSum1=0, wSum2=0;
  WORD *pW = (WORD*)pBlock;
  // DOBINI();
  iBlockSize /= sizeof(WORD);
  while(iBlockSize--)
   { wSum1 += *pW;  // 2013-02-09 : Access violation here, called from TCircuitForm::Timer1Timer() -> UpdateCircuitDisplay() ! !
     wSum2 += (WORD)( ((*pW + iBlockSize) ^ iBlockSize) );
     ++pW;
   }
  // DOBINI();
  return (long)wSum1 + ((long)wSum2 << 2);
#else    // table-driven CRC "rescued" from another project:
  int i;
  LO_HI temp;
  unsigned short int crc = 0x4846;   // "HF" as seed - guess who... 
  for(i=0;i<iBlockSize;i++)
   {
     temp.both = CRC_TABLE[crc >> 8];
     crc   = temp.sbyte.hi ^ (crc & 0xff);
     crc <<= 8;
     crc  |= temp.sbyte.lo ^ ((BYTE*)pBlock)[i];
   }
  return crc;
#endif
}

//---------------------------------------------------------------------------
int  UTL_SafeStrlen( char * pszSource, int iMaxLen )
{ int len=0;
  if( pszSource==NULL )
    return 0;
  while( (*pszSource!='\0') && (len<iMaxLen) )
   { ++len;
     ++pszSource;
   }
  return len;
}

//---------------------------------------------------------------------------
void UTL_CopyShortenedString( char * pszDest, char * pszSource, int iMaxLen, int iWordBreakAfter)
{
 char *cp;
 int iSrcLength;
 DOBINI();
 iSrcLength = strlen( pszSource );

 if( iSrcLength <= iMaxLen )
    strcpy( pszDest, pszSource );
 else
   { cp=pszDest;
     while( *pszSource && (iMaxLen>2) )
      {
        if( iWordBreakAfter<=0 && *pszSource==' ' )
          break;

        *cp++ = *pszSource++;
        iMaxLen--; iWordBreakAfter--;
      }
     *cp++ = '.';  *cp++ = '.'; *cp = '\0';
   }
 DOBINI();
} // end UTL_CopyShortenedString()

//---------------------------------------------------------------------------
void UTL_TimeToString(long now_sec, char *dst,  char time_separator)
  // See also (far more advanced and versatile) : UTL_FormatDateAndTime() !
{
  int h=(now_sec/3600)%24; int m=(now_sec/60)%60; int s=now_sec%60;
  sprintf(dst,"%02d%c%02d%c%02d",
       (int)h,(char)time_separator,(int)m,(char)time_separator,(int)s);
}

//---------------------------------------------------------------------------
void UTL_TimeToDateString(long now_sec, char *dst, char date_separator)
{ // see Borland's help on mktime(), gmtime()  and tm !
  struct tm *gmt;
  gmt = gmtime(&now_sec);
  sprintf(dst,"%s%c%02d%c%04d",
      UTL_months1[gmt->tm_mon % 12],(char)date_separator,
              (int)gmt->tm_mday,  (char)date_separator,
              (int)gmt->tm_year+1900);
}

//---------------------------------------------------------------------------
char *UTL_UnitCodeToString( int unit_code, BOOL fMustBeUnique )
{
  switch(unit_code)
   {
    case SCALE_UNIT_BLANK  :                 // blank unit, but tech unit (like "1.2G")
    case SCALE_UNIT_NONE   : return "";      // definitely NO unit
    case SCALE_UNIT_UNKNOWN: return "?";     // UNKNOWN unit is something different
    case SCALE_UNIT_NORM_V:  return "";      // voltage, normalized to +/- 1.0 for full ADC input swing
    case SCALE_UNIT_PERCENT: return "%";     // similar, but expressed as "percentage" (100 % = ADC clipping)
    case SCALE_UNIT_VOLT:    return "V";     // voltage (V), 'calibrated' using UTL_dblAdcMaxInputVoltage

    case SCALE_UNIT_WATT:    return "W";     // power (W), 'calibrated' with UTL_dblAdcMaxInputVoltage + UTL_dblAdcInputImpedance
    case SCALE_UNIT_NORM_PWR: return "Wn";   // power, normalized to 0...1 for full A/D converter input range

    case SCALE_UNIT_dB:      return "dB";    // dB , unknown reference level (0dB = UTL_dblAdcMaxInputVoltage)
    case SCALE_UNIT_dBfs: if(fMustBeUnique)  // dB , reference level is  "full scale" (0 dBfs = clipping)
                             return "dBfs";  // too ugly (now only used for config-files)
                          else               // because no-one knew what "dBfs" should mean
                             return "dB";    // this looks better in diagrams !
    case SCALE_UNIT_dBV:     return "dBV";   // dB , reference level is 1 volt rms across any impedance
    case SCALE_UNIT_dBuV:    return "dBV";  // dBuV, reference level is 1 microvolt rms across any impedance
    case SCALE_UNIT_dBm:     return "dBm";   // dB , reference level is 1 milliwatt (often across 600 ohms)

    case SCALE_UNIT_DEGREE:  return "";
    case SCALE_UNIT_HERTZ:   return "Hz";
    case SCALE_UNIT_SECONDS:
    case SCALE_UNIT_TIME:    return "s";
    case SCALE_UNIT_SAMPLES_PER_SECOND:
                             return "S/s";

    default:                 return "?";
   }
} // end UTL_UnitCodeToString()

//---------------------------------------------------------------------------
BOOL UTL_AreStringsEqualWidhWildcard( char *pszString,  char *pszRefWithWildcards )
  // returns TRUE if the two strings are "equal" (case ignored, '?'=wildcard) .
{
  char c1, cRef;

  do
   { c1  = *(pszString++);
     cRef= *(pszRefWithWildcards++);
     // convert both chars to lower case to compare them further below:
     if(c1>='A' && c1<='Z')
        c1 = (char)( c1-'A'+'a' );
     if(cRef>='A' && cRef<='Z')
        cRef = (char)( cRef-'A'+'a' );
     switch( cRef ) // anything "special" in the reference-string ?
      { case '\0':  // end of the reference, FINISHED searching
          return c1=='\0';
        case '?':   // this character matches ANY character..
          break;
        default:
          if(c1!=cRef)
            return FALSE;
          break;
      }
   }while(c1 && cRef);
  return (c1==cRef);

} // end UTL_AreStringsEqualWidhWildcard()

//---------------------------------------------------------------------------
int UTL_ParseUnitString( char **ppszUnitString, double *pdblTechPrefixFactor )
{
 char *cp = *ppszUnitString;
 int    iScaleUnit = SCALE_UNIT_UNKNOWN;
 double dblTechPrefixFactor = 1.0;

  DOBINI();

  // Skip space character(s)
  while(*cp==' ')
    ++cp;

  // check if there is a "technical prefix" like u=Micro, m=milli, k=kilo, M=Mega, etc
  switch(*cp)
   { case 'f': dblTechPrefixFactor = 1e-15; ++cp; break;  // femto
     case 'p': dblTechPrefixFactor = 1e-12; ++cp; break;  // pico
     case 'n': dblTechPrefixFactor = 1e-9;  ++cp; break;  // nano
     case 'u': dblTechPrefixFactor = 1e-6;  ++cp; break;  // mikro (mu)
     case 'm': dblTechPrefixFactor = 1e-3;  ++cp; break;  // milli
     case 'k': dblTechPrefixFactor = 1e3;   ++cp; break;  // kilo
     case 'M': dblTechPrefixFactor = 1e6;   ++cp; break;  // Mega
     case 'G': dblTechPrefixFactor = 1e9;   ++cp; break;  // Giga
     case 'T': dblTechPrefixFactor = 1e12;  ++cp; break;  // Tera
     default:
          break;
   }
  if(pdblTechPrefixFactor) *pdblTechPrefixFactor = dblTechPrefixFactor;
  switch(cp[0])
   {
     case 'V': ++cp; iScaleUnit = SCALE_UNIT_VOLT;    break;
     case 'W': ++cp; iScaleUnit = SCALE_UNIT_WATT;
          if(*cp=='n') // "normalized" power ("Wn") ?
           { ++cp;
             iScaleUnit = SCALE_UNIT_NORM_PWR;
           }
          break;
     case '': ++cp; iScaleUnit = SCALE_UNIT_DEGREE;  break;
     case 's': ++cp; iScaleUnit = SCALE_UNIT_SECONDS; break;
     case 'S':
          switch(cp[1])
           { case '/' :
                switch(cp[2])
                 { case 'S':
                      cp+=3;
                      iScaleUnit = SCALE_UNIT_SAMPLES_PER_SECOND;
                      break;
                   default:
                      break;
                 }
                break;
             default:
                break;
           }
          break;
     case 'H':
          switch(cp[1])
           { case 'z': cp += 2;
                  iScaleUnit = SCALE_UNIT_HERTZ;
                  break;
             default:
                  break;
           }
          break;
     case 'd': ++cp;
          switch(*cp)
           { case 'B': ++cp;
               switch(*cp)
                { case 'f': ++cp;
                            if(*cp=='s') ++cp;
                            iScaleUnit = SCALE_UNIT_dBfs;
                            break;
                  case 'F': ++cp;
                            if(*cp=='S') ++cp;
                            iScaleUnit = SCALE_UNIT_dBfs;
                            break;
                  case 'V': ++cp;
                            iScaleUnit = SCALE_UNIT_dBV;
                            break;
                  case 'u':
                  case '':
                            ++cp;
                            if(*cp=='S') ++cp;
                            iScaleUnit = SCALE_UNIT_dBuV;
                            break;
                  case 'm': ++cp;
                            iScaleUnit = SCALE_UNIT_dBm;
                            break;
                  default:  // everything else considered simply "dB", unknown reference level
                            iScaleUnit = SCALE_UNIT_dB;
                            break;
                } // end switch < letter after unit "dB" >
               break;
             default: iScaleUnit = SCALE_UNIT_UNKNOWN; break;
           }
          break;
     case '%': ++cp; iScaleUnit = SCALE_UNIT_PERCENT; break;
     case '?': ++cp; iScaleUnit = SCALE_UNIT_UNKNOWN; break;
     default:  iScaleUnit = SCALE_UNIT_NORM_V;        break;
   }

  *ppszUnitString = cp;
  DOBINI();
  return iScaleUnit;
} // end UTL_ParseUnitString()


//---------------------------------------------------------------------------
void UTL_FloatToFixedLengthString( double dblValue, char *pszDest, int iNrOfChars )
   // Converts a floating-point number into a string,
   // with the best possible resolution for a given string length.
   // Something the lousy "sprintf"-function cannot do ! ?!
{
 int iDigitsBeforePoint, iDigitsAfterPoint;
 char *cp = pszDest;

  DOBINI();

     // sprintf(str80,"%5.5lg",(double)dblAmpl);  // doesn't work (no 5-char-limit!)
     // Don't trust the help system which says about the printf format string:
     // > % [flags] [width] [.prec] [F|N|h|l|L] type_char
     // > [width] (Optional) Width specifier
     // >          Minimum number of characters to print, padding with blanks or zeros
     // > [prec]  (Optional) Precision specifier
     // >          Maximum number of characters to print;
     // >          for integers, minimum number of digits to print
     // What the ugly help system won't tell you, [prec] seems to be the
     //         COUNT OF NON-ZERO DIGITS in the string , not "characters" .
     // "%.3" doesn't work, it always produces 3 fractional digits regardless of the digits be4 the point
     // "%5.5" doesn't work, it does not obey the "maximum number of characters to print"
     // "%5.3" looks ugly, it produces trailing spaces (baaah) instead of showing as many fractional digits as possible.
     // Is it really so complicated in "C" to produce a simple fixed-length floating point string
     // with a TRULY limited length ? NGRRRR  ....
  // How many decimal places needed before the comma ?
  iDigitsBeforePoint = 1;  // there must be at least a ZERO character before the decimal point
  if(dblValue<0)
   { *cp++ = '-'; --iNrOfChars; dblValue=-dblValue; }
  if(dblValue>1.0)
   { // value above one, how many additional digits before the decimal point ?
     // Examples: log10(1)=0,      (int)0 = 0
     //           log10(9)=0.95,   (int)0.95 = 0 (!!!)
     //           log10(10)=1.0,   (int)1.0  = 1
     iDigitsBeforePoint += (int)log10(dblValue);
   }
  iDigitsAfterPoint = iNrOfChars - 1/*for point*/ - iDigitsBeforePoint;
  if(iDigitsAfterPoint<0) iDigitsAfterPoint=0;
  sprintf(cp,"%*.*lf",(int)iNrOfChars,(int)iDigitsAfterPoint,(double)dblValue);
  DOBINI();

} // end UTL_FloatToFixedLengthString()


//---------------------------------------------------------------------------
int UTL_FloatToTechNotation2(
          char *pszDest,             // destination string
          int iMaxLen,               // max length of destination
          int iOptionsAndNrOfDigits, // how many DIGITS (+ flags)
          char * pszUnit,            // "V", "A", "Samples", .... (difference between UTL_FloatToTechNotation and ..2)
          double dblValue )          // floating point value to be formatted
   // Formats a floating-point value into a string with TECHNICAL NOTATION,
   //  with exponents like p(ico) .. k(ilo) .. M(ega) plus physical UNIT .
   // Return value:
   //     NEGATIVE: error
   //     POSITIVE: success, returns the length of the generated string.
   // Note: An INVERSE function (that can parse the string generated
   //  by UTL_FloatToTechNotation / UTL_FloatToTechNotation2) is CLI_CalcSumWithDefault() -> CLI_CalcArg() -> CLI_ParseTechUnit() .
   //  Variants of utility1.cpp : UTL_FloatToTechNotation[2]()
   //  may exist elsewhere, for example:
   //    - C:\cbproj\SoundUtl\utility1.cpp  ["master"]
   //    - C:\cbproj\CalcEd\PlotWin.c
{
 int  i, iExponent, iDigitsBeforePoint, iDigitsAfterPoint;
 char * cp = pszDest;
 char * cp2;

  DOBINI();

   // How many decimal places needed before the comma ?
  iDigitsBeforePoint = 1;  // there must be at least a ZERO character before the decimal point
  iExponent = 0;
  if(dblValue<0)           // deal with negative values early :
   { *cp++ = '-'; --iMaxLen; dblValue=-dblValue;
   }

  if( dblValue > 1e-16)  // avoid 0.00000 femtoV,  write 0.000000 V instead
   {
    // Find a suitable "technical" exponent
    //   - in fact a PREFIX before the unit like n(nano), m(milli), k(kilo), M(Mega), etc
    //   - exponents must be multiples of 3 (or -3) .
    //  Examples (check):   0.1 V -> 100 mV    999 V -> 999 V    1000 V -> 1 kV
    while( (dblValue < 1.0) && (iExponent>-15) ) // 1e-15 = femto, don't go below
      { iExponent -= 3;   dblValue *= 1e3;  }
    while( (dblValue > 1000) && (iExponent<12) ) // 1e12  = Tera, don't go higher
      { iExponent += 3;   dblValue *= 1e-3; }
   }
  else // "technical" exponent notation makes no sense for this unit :
   {
     iExponent = 0;   // refuse to produce nonsense like "mdB", "k" etc
   }
  if(dblValue>1.0)
   { // If the value is (still) above one, how many additional digits are required before the decimal point ?
     // Examples: log10(1)=0,      (int)0 = 0
     //           log10(9)=0.95,   (int)0.95 = 0 (!!!)
     //           log10(10)=1.0,   (int)1.0  = 1
     iDigitsBeforePoint += (int)log10(dblValue);
   }
  iDigitsAfterPoint = (iOptionsAndNrOfDigits & 15) - 1/*for point*/ - iDigitsBeforePoint;
  if(iDigitsAfterPoint<0) iDigitsAfterPoint=0;
  sprintf(cp,"%*.*lf",(int)(iOptionsAndNrOfDigits & 15),(int)iDigitsAfterPoint,(double)dblValue);
  if( iOptionsAndNrOfDigits & UTL_FMT_OPTION_REMOVE_TRAILING_ZEROES)
   { cp2 = cp + strlen(cp);
     while( (cp2>(cp+1)) && cp2[-1]=='0' )
      { --cp2;
        *cp2 = '\0';
      }
     if( (cp2>(cp+1)) && cp2[-1]=='.' )
      { --cp2;
        *cp2 = '\0';
      }
   } // end if < remove trailing zeroes ? >
  i = strlen(cp);
  iMaxLen -= i;
  if(iMaxLen>3)
   { cp += i;
    if( iOptionsAndNrOfDigits & UTL_FMT_OPTION_SPACE_BEFORE_UNIT)
      { *cp++ = ' ';  --iMaxLen; }
    switch(iExponent)
      { case -15: *cp++='f'; break;  // femto
        case -12: *cp++='p'; break;  // pico
        case -9:  *cp++='n'; break;  // nano
        case -6:  *cp++='u'; break;  // mikro (mu)
        case -3:  *cp++='m'; break;  // milli
        case  0:  break;
        case  3:  *cp++='k'; break;  // kilo
        case  6:  *cp++='M'; break;  // Mega
        case  9:  *cp++='G'; break;  // Giga
        case 12:  *cp++='T'; break;  // Tera
        default:  sprintf(cp,"e%d",(int)iExponent);
                  cp+=strlen(cp);
                  break;
      }
    *cp = '\0';
    if(iMaxLen>(int)strlen(pszUnit) )
      { strcpy(cp, pszUnit); i = strlen(cp);
        cp+=i;
        iMaxLen-=i;  // for future extensions, keep this up-to-date
      }
   }

  (void)iMaxLen;     // "never used" --- oh, shut up, compiler !

  DOBINI();

  return pszDest - cp;   // returns NUMBER OF CHARACTERS in produced string
} // end UTL_FloatToTechNotation2()


//---------------------------------------------------------------------------
int UTL_FloatToTechNotation(
          char *pszDest,             // destination string
          int iMaxLen,               // max length of destination
          int iOptionsAndNrOfDigits, // how many DIGITS (+ flags)
          int iScaleUnit,            // SCALE_UNIT_xxxx (difference between UTL_FloatToTechNotation and ..2)
          double dblValue )          // floating point value to be formatted
   // "Formats" a floating-point value into a TECHNICAL NOTATION,
   //  with exponents like p(ico) .. k(ilo) .. M(ega) plus physical UNIT .
   //  iScaleUnit = SCALE_UNIT_TIME expects dblValue in seconds,
   //     but (depending on the absolute value) may decide to use seconds, minutes, or hours .
   //
   // Return value:
   //     NEGATIVE: error
   //     POSITIVE: success, returns the length of the generated string.
   //
   // Note: An INVERSE function (that can parse the string generated
   //  by UTL_FloatToTechNotation / UTL_FloatToTechNotation2) is CLI_CalcSumWithDefault() -> CLI_CalcArg() -> CLI_ParseTechUnit() .
{
 int  i, iExponent, iDigitsBeforePoint, iDigitsAfterPoint;
 char * cp = pszDest;
 char * cp2;
 char * pszUnit = UTL_UnitCodeToString( iScaleUnit, FALSE/*short*/ );

  DOBINI();

  if( iScaleUnit == SCALE_UNIT_TIME ) // dblValue in seconds, but...
   { pszUnit = "s";
     if( fabs(dblValue) >= 3600.0 )   // better use HOURS
      { pszUnit = "hrs";
        dblValue /= 3600.0;
      }
     else if( fabs(dblValue) > 60.0 ) // .. or MINUTES ?
      { pszUnit = "min";
        dblValue /= 60.0;
      }
   } // end if < SCALE_UNIT_TIME >


  // How many decimal places needed before the comma ?
  iDigitsBeforePoint = 1;  // there must be at least a ZERO character before the decimal point
  iExponent = 0;
  if(dblValue<0)           // deal with negative values early :
   { *cp++ = '-'; --iMaxLen; dblValue=-dblValue;
   }

  if(   (dblValue > 1e-16)  // avoid 0.00000 femtoV,  write 0.000000 V instead
     &&((iScaleUnit & SCALE_UNIT_MASK_dB) != SCALE_UNIT_dB )  // avoid millidecibel ..
     && (iScaleUnit != SCALE_UNIT_DEGREE)        // and  kilodegrees  ..
     && (iScaleUnit != SCALE_UNIT_PERCENT)       // and  millipercent ..
     && (iScaleUnit != SCALE_UNIT_NONE   )       // and  milli<dimensionless> !
    )
   {
    // Find a suitable "technical" exponent
    //   - in fact a PREFIX before the unit like n(nano), m(milli), k(kilo), M(Mega), etc
    //   - exponents must be multiples of 3 (or -3) .
    //  Examples (check):   0.1 V -> 100 mV    999 V -> 999 V    1000 V -> 1 kV
    while( (dblValue < 1.0) && (iExponent>-15) ) // 1e-15 = femto, don't go below
      { iExponent -= 3;   dblValue *= 1e3;  }
    while( (dblValue > 1000) && (iExponent<12) ) // 1e12  = Tera, don't go higher
      { iExponent += 3;   dblValue *= 1e-3; }
   }
  else // "technical" exponent notation makes no sense for this unit :
   {
     iExponent = 0;   // refuse to produce nonsense like "mdB", "k" etc
   }
  if(dblValue>1.0)
   { // If the value is (still) above one, how many additional digits are required before the decimal point ?
     // Examples: log10(1)=0,      (int)0 = 0
     //           log10(9)=0.95,   (int)0.95 = 0 (!!!)
     //           log10(10)=1.0,   (int)1.0  = 1
     iDigitsBeforePoint += (int)log10(dblValue);
   }
  iDigitsAfterPoint = (iOptionsAndNrOfDigits & 15) - 1/*for point*/ - iDigitsBeforePoint;
  if(iDigitsAfterPoint<0) iDigitsAfterPoint=0;
  sprintf(cp,"%*.*lf",(int)(iOptionsAndNrOfDigits & 15),(int)iDigitsAfterPoint,(double)dblValue);
  if( iOptionsAndNrOfDigits & UTL_FMT_OPTION_REMOVE_TRAILING_ZEROES)
   { cp2 = cp + strlen(cp);
     while( (cp2>(cp+1)) && cp2[-1]=='0' )
      { --cp2;
        *cp2 = '\0';
      }
     if( (cp2>(cp+1)) && cp2[-1]=='.' )
      { --cp2;
        *cp2 = '\0';
      }
   } // end if < remove trailing zeroes ? >
  i = strlen(cp);
  iMaxLen -= i;
  if(iMaxLen>3)
   { cp += i;
    if( iOptionsAndNrOfDigits & UTL_FMT_OPTION_SPACE_BEFORE_UNIT)
      { *cp++ = ' ';  --iMaxLen; }
    switch(iExponent)
      { case -15: *cp++='f'; break;  // femto
        case -12: *cp++='p'; break;  // pico
        case -9:  *cp++='n'; break;  // nano
        case -6:  *cp++='u'; break;  // mikro (mu)
        case -3:  *cp++='m'; break;  // milli
        case  0:  break;
        case  3:  *cp++='k'; break;  // kilo
        case  6:  *cp++='M'; break;  // Mega
        case  9:  *cp++='G'; break;  // Giga
        case 12:  *cp++='T'; break;  // Tera
        default:  sprintf(cp,"e%d",(int)iExponent);
                  cp+=strlen(cp);
                  break;
      }
    *cp = '\0';
    if(iMaxLen>(int)strlen(pszUnit) )
      { strcpy(cp, pszUnit); i = strlen(cp);
        cp+=i;
        iMaxLen-=i;  // for future extensions, keep this up-to-date
      }
   }

  (void)iMaxLen;     // "never used" --- oh, shut up, compiler !

  DOBINI();

  return pszDest - cp;   // returns NUMBER OF CHARACTERS in produced string
} // end UTL_FloatToTechNotation()


//---------------------------------------------------------------------------
BOOL UTL_LimitDouble( double *pdbl, double min, double max )
  // Returns TRUE if the value (*pdbl) HAD TO BE LIMITED;
  //    or   FALSE if limiting was not necessary .
{
  BOOL fLimited = FALSE;
  if( *pdbl < min )
   {  *pdbl = min;  fLimited = TRUE;
   }
  if( *pdbl > max )
   {  *pdbl = max;  fLimited = TRUE;
   }
  return fLimited;
} // end UTL_LimitDouble()



//---------------------------------------------------------------------------
char *UTL_LastApiErrorCodeToString( DWORD dwLastError )
  // Converts a windoze API error code (as returned by GetLastError() )
  // into a string.  Also removes the CR/NL-nonsense which WINDOZE likes to append.
  // Returns a pointer to a STATIC string which is only valid
  // until the next call of this function !
{
 LPVOID lpMsgBuf;
 DWORD  dwLength;
 static char szStaticStringResult[100];

  DOBINI();

 /* Convert the GetLastError()-code into a readable string: */
 FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
        NULL,
        dwLastError,        // .. from GetLastError()
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
             // (dont expect other languages if you replace "LANG_NEUTRAL" !)
            (LPTSTR) &lpMsgBuf,
             0,
             NULL );

 // Convert the string and free the message's buffer.
 strncpy(szStaticStringResult, (char*)lpMsgBuf, sizeof(szStaticStringResult)-1);
 szStaticStringResult[sizeof(szStaticStringResult)-1] = '\0'; // keep string terminated


 // Free the buffer which has been allocated somehow in FormatMessage(!)
 LocalFree( lpMsgBuf );

 // If the result string contains this foolish CR and/or NL characters at the end,
 // trow them out..
 dwLength = strlen(szStaticStringResult);
 if( (dwLength>0) && (szStaticStringResult[dwLength-1] == '\r') )  // crazy1
  { --dwLength;
    szStaticStringResult[dwLength] =  '\0';
  }
 if( (dwLength>0) && (szStaticStringResult[dwLength-1] == '\n') )  // crazy2
  { --dwLength;
    szStaticStringResult[dwLength] =  '\0';
  }
 if( (dwLength>0) && (szStaticStringResult[dwLength-1] == '\r') )  // crazy3
  { --dwLength;
    szStaticStringResult[dwLength] =  '\0';
  }

  DOBINI();

  return szStaticStringResult;

} // end UTL_LastApiErrorCodeToString()



//---------------------------------------------------------------------------
double UTL_ConvertYMDhmsToUnix( int year, int month, int day,
                                int hour, int minute, double dblSecond )
  // Converts a date into the UNIX format, which is explained below.
  // Shall be a better(?) replacement for the old "mktime" function.
  //      NO lousy global variables like "_timezone" and "_daylight" are used.
  //      We always use UTC (~GMT) here, and double values are much better
  //      in the year 2038 (when programs using the old "mktime" will fail).
  //  Input parameters: full year (like 2001),
  //                    month: 1..12, day: 1..31,
  //                    hour: 0..23, minute: 0..59, second: 0..59 .
  //  Return:  UNIX-Format = SECONDS elapsed since January 1st, 1970, 00:00:00 .
  //  Example: 22.10.2001, 11:12:13  returns 1003749133 UNIX-SECONDS.
  //          (DD.MM.YYYY, hh:mm:ss)
{
  // Nach Infos von http://www.pci.uzh.ch/pfister/formelsammlung.html ,
  //                  (Lokal gespeichert unter "Rolfs Formelsammlung")
  //  Julianisches Datum (JD)
  // JD = Anzahl Tage seit 4713 vor Christus
  // bis und mit 4.Oktober 1582: B = int((y+4716)/4) - 1181
  //        ab 15.Oktober 1582: B = int(y/400)-int(y/100)+int(y/4)
  // (Wegen der Gregoranischen Kalenderreform folgte auf den 4.Okt. sogleich der 15.)
  //  JD = 365*y - 679004 + B + int(30.6*m) + d + h/24 + 2400000.5
  // MJD = JD - 2400000.5  (Modifiziertes Julianisches Datum)
  //          (WB: MJD = Anzahl Tage seit 17. November 17 1858 Mitternacht)
  // h = Tageszeit in Stunden
  // d = Tag
  // Y = fr Jahre vor Christus: Y = -1 - Jahreszahl
  //     fr Jahre nach Christus: Y = Jahreszahl
  // wenn Monat<=2 : y = Y-1   m = Monat+13
  //          sonst: y = Y     m = Monat+1
  // int() = Abrundung auf ganze Zahl
  //
  // ... Conversion into C :
  // The "Modified Julian Date" is the number of DAYS passed since
  //     >>  Midnight, November 17, 1858  <<
  // The UNIX date-and-time format is the number of SECONDS passed since
  //     >>  Midnight, January 1, 1970    <<
  long   b, m, y;
  double mjd;

  if( month<= 2) { y=year-1; m=month+13; }
         else    { y=year;   m=month+1;  }
  if( (year>=1582) || (month>=10) || (day>=15) )
       b = (int)(y/400) - (int)(y/100) + (int)(y/4);
   else // before October 4, 1582:
       b = (int)((y+4716)/4) - 1181;
   mjd = 365.0*(double)y - 679004 + b + (int)((double)m*30.6) + day;
   // Now convert the result from Modified Julian DAYS (w/o hours)
   // to UNIX-Date in "SECONDS since January 1, 1970",
   //   because the second is the SI unit for the time !
   // The MJD of Jan 1, 1970 has been calculated by the above routine
   // as:  40587 [days between Jan1, 1970 00:00:00  and  Nov 17, 1858, 00:00:00]
   return ((double)mjd - 40587.0) * 86400.0/* seconds per day*/
         + (double)hour*3600.0 + (double)minute*60.0 + dblSecond;
   // Note: In contrast to UNIX, this routine may also return valid negative
   //       numbers (there were good times even before January 1, 1970 ;-)
   //       and -hopefully- this routine will still work after year 2038 .
} // end UTL_ConvertYMDhmsToUnix()



//---------------------------------------------------------------------------
void UTL_SplitUnixDateTimeToYMDhms( double dblUnixDateTime,
         int *piYear,
         int *piMonth,
         int *piDay,
         int *piHour, int *piMinute, double *pdblSecond ) // [out] time of day
  // Inverse function to UTL_ConvertYMDhmsToUnix() .
  // Splits a date-and-time in UNIX format (seconds since Jan 1st, 1970, 00:00:00 UTC)
  // into year, month, day, hour, minute, and (fractional) seconds .
  // Much better than the ugly, thread-unsafe, global-variable-dependent,
  //                      stupid, stonage function named 'gmtime()' !
{
  double dbl, dblUnixDay, dblSecondsOfDay, dblHour, dblMin;
  long   jdayno, j, d, m, y;

  // Split DAY and SECONDS-OF-DAY (here: still UNIX, not MJD, and not JD):
  dblUnixDay = floor( dblUnixDateTime / 86400.0/*seconds per day*/ );
  dblSecondsOfDay = dblUnixDateTime - dblUnixDay * 86400.0;
  dbl = dblSecondsOfDay / 3600.0;      // -> hours of day, fractional
  dblHour = floor(dbl); dbl=60.0*(dbl-dblHour);  // -> hours of day
  dblMin  = floor(dbl); dbl=60.0*(dbl-dblMin);   // -> minutes of hour
  if( piHour != NULL )
   { *piHour = (int)dblHour;
   }
  if( piMinute!= NULL )
   { *piMinute = (int)dblMin;
   }
  if( pdblSecond!= NULL )
   { *pdblSecond = dbl;   // remaining "seconds of minute", fractional
   }

  // Now for the trickier part: Split year, month of year, and day of month.
  // From   http://en.wikipedia.org/wiki/Julian_day :
  // > Unix Time (seconds since January 1st, 1970, UTC)
  // >  calculated from Julian Day by : (JD - 2440587.5) * 86400
  // So, calculate 'jdayno' (Julian Day, JD) for the algo further below.
  //  "UnixDay" = JulianDay - 2440587.5, so :
  jdayno = (long)( dblUnixDay + 2440587.5 );  // -> Julian Day, INTEGER

  // Die folgende Berechnung stammt aus Google Code Search, Stichwort
  //  "jdater - sets day, month and year given jdayno" (haystack.edu .. date.c)
  //  "inverse of jday method" .
  // Was in der Beschreibung leider komplett fehlt, ist eine Spezifikation
  //  der ZAEHLWEISE, speziell fuer jdayno (0...x oder 1...x),
  //  Monat(0..11 oder 1..12) und Tag (0..30 oder 1..31) !
  // Von gmtime() ist man ja Elendes gewohnt (Tage von 1..31 aber Monate von 0..11).
  // Hier scheint aber alles AB EINS gezaehlt zu werden (Pascal-Freaks?),
  // auch der 'jdayno' beginnt die Zaehlung scheinbar bei EINS, nicht NULL, darum:
  ++jdayno;
  // Original formula from haystack.edu starts here:
  j = jdayno - 1721119;
  y = (4*j - 1)/146097;
  j  =  4*j - 1 - 146097*y;
  d = j/4;
  j = (4*d + 3)/1461;
  d = 4*d + 3 - 1461*j;
  d = (d + 4)/4;
  m = (5*d - 3)/153;
  d = 5*d - 3 - 153*m;
  d = (d + 5)/5;
  y = 100*y + j;
  if (m < 10)
   {  m = m + 3;
   }
  else
   {  m = m - 9;
      y = y + 1;
   }

  if( piYear != NULL )
   { *piYear = y;
   }
  if( piMonth != NULL )
   { *piMonth = m;
   }
  if( piDay != NULL )
   { *piDay = d;
   }
} // end UTL_SplitUnixDateTimeToYMDhms()


//---------------------------------------------------------------------------
void UTL_FormatDateAndTime( char *format, double dblUnixDateTime, char *dest)
  // Example for format:  "YYYY-MM-DD hh:mm:ss"  (preferred ISO8601)
  //           ->result:  "2003-06-01 14:28:02"  (placed in dest)
  // The input parameter dblDateTime is similar to the UNIX time:
  //         "SECONDS elapsed since January 1st, 1970, 00:00:00" .
{
 char   fmt,fmt2,fmt3;
 int    i;
 int   year,month,day,hour,min;
 double dblSec, dblSec2;

  DOBINI();
  UTL_SplitUnixDateTimeToYMDhms( dblUnixDateTime,
                                 &year, &month, &day, &hour, &min, &dblSec );
  DOBINI();
  while( (fmt = *format++) != 0)
   {
    fmt2 = *format; // to look ahead at following letters in the format string
    fmt3 = *(format+1);
    switch(fmt)
     {
       case 'Y':  // year  (1 to 4 digits)
            if(fmt2=='Y')
             {
              ++format;  // skip 2nd letter in format string
              if(fmt3=='Y')
               {
                ++format;  // skip 3rd letter in format string
                if(*format=='Y')
                 {
                  ++format;  // skip 4th letter in format string
                  sprintf(dest,"%04d",(int)year );
                 }
                else // 3-letter format (??)
                 {
                  sprintf(dest,"%03d",(int)(year%1000) );
                 }
               }
              else // 2-letter format:
               {
                sprintf(dest,"%02d",(int)(year%100) );
               }
             } // end if (fmt2..)
            else // 1-letter format (used for automatic filenames somewhere)
             {
              sprintf(dest,"%01d",(int)(year%10) );
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;

       case 'M':  // month (2 digits or THREE LETTERS (Jan,Feb, etc) )
            if(fmt2=='M' || fmt2=='m')
             {
              ++format;  // skip 2nd letter in format string
              if(fmt3=='M') // "MMM" = { JAN, FEB, .. }
               {
                ++format;  // skip 3rd letter in format string
                if( month<1 ) month=1;
                strcpy(dest,UTL_months1[(month-1) % 12]);
               }
              else
              if(fmt3=='m') // "Mmm" = { Jan, Feb, .. }
               {
                ++format;  // skip 3rd letter in format string
                if( month<1 ) month=1;
                strcpy(dest,UTL_months2[(month-1) % 12]);
               }
              else // 2-letter format: numerical, 1..12
               {
                sprintf(dest,"%02d",(int)month );
               }
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)month );
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;

       case 'D':  // day (2 digits)
            if(fmt2=='D')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)day);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)day);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 'h':  // hour
            if(fmt2=='h')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)hour);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)hour);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 'm':  // minute (always 2 digits)
            if(fmt2=='m')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)min);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)min);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 's':  // second (at least 2 digits, possible 'fractional' !)
            if(fmt2=='s')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)dblSec );
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)dblSec );
             }
            dest += strlen(dest);
            if( format[0]=='.' && format[1]=='s')
             { // looks like tenths of a second after a decimal point !
               format+=2; // skip ".s" from the format string
               dblSec2 = 10.0 * dblSec;
               i = fmod( dblSec2, 10.0);
               if (i<0)   // ??
                   i=0;
               if (i>9)   // ????
                   i=9;   // set breakpoint here if fmod behaves strange...
               *dest++ = '.';
               *dest++ = (char)('0'+i);  // append 1/10 second(s)
               // Modified 2015-01 : Even more decimal places (1/100, 1/1000 s)
               while(*format=='s')
                 { // gosh.. he wants to see hundreds of a second :
                   ++format; // skip another "s" from the format string
                   dblSec2 *= 10.0;
                   i = fmod( dblSec2, 10.0);
                   *dest++ = (char)('0'+i);  // append 1/100, 1/1000 second(s), any maybe even more
                   // (see AFilePrp.cpp: GPS-sync'd recordings may have timestamps
                   //      with microsecond accuracy, or at least resolution )
                 }
               *dest   = '\0';
             }
            fmt = 0; // format letter has been handled
            break;

       default: // all other characters will be passed 1:1 to "dest"
            break;
     } // end switch(fmt)

    if (fmt) // character from format string could not be handled:
     {
      *dest++ = fmt;
     }
   } // end while(fmt)
  *dest = '\0';  // null-terminate the destination string

  DOBINI();
} // end UTL_FormatDateAndTime()


//---------------------------------------------------------------------------
BOOL UTL_IsAsciiLetter(char c)
{
  return ( (c>='a' && c<='z') || (c>='A' && c<='Z') );
}

//---------------------------------------------------------------------------
BOOL UTL_CheckAndSkipToken(BYTE **ppbSource, char *pszToken)
{
 BYTE *bp;
  bp = *ppbSource;
  while(*bp==' ' || *bp=='\t') // trailing spaces before a token may be skipped
    { ++bp; }
  while( (*bp == *pszToken) && (*pszToken) )
   { ++bp;
     ++pszToken;
   }
  if( *pszToken==0 )  // reached end of token -> FOUND !
   { *ppbSource = bp;
     return TRUE;
   }
  else  // token not found
   { return FALSE;
   }
} // end UTL_CheckAndSkipToken()

//---------------------------------------------------------------------------
char UTL_SkipSpaces(BYTE **ppbSource) // returns the next NON-WHITESPACE character
{ BYTE *bp;
  bp = *ppbSource;
  while(*bp==' ' || *bp=='\t')   // skip SPACES and TABS (for reading "text data files")
    { ++bp;
    }
  *ppbSource = bp;
  return *bp;
} // end UTL_SkipSpaces()


//---------------------------------------------------------------------------
void UTL_SkipToEndOfString(
        char **ppszSource)  // [in,out] sourcecode pointer
{
  while(**ppszSource!='\0')
   { ++*ppszSource;
   }
} // end UTL_SkipToEndOfString()

//---------------------------------------------------------------------------
int UTL_SkipToEndOfLine( char **ppszSource, char *cpEndstop )
  // Skips anything up to, and including, the "end of a line"
  //       in an HTTP header (request, response, doesn't matter),
  //       including ONE COMBINATION of CR+NL .
  // Returns the number of characters actually skipped,
  //       including the CR and NL .
  // Thus, when skipping an EMPTY LINE, the return value is 2.
  // Only when NOTHING was skipped at all, the return value is 0.
{
  char *cp = *ppszSource;
  int  nCharsSkipped;
  while( (*cp!='\0') && (cp<cpEndstop) )
   { if( *cp=='\r' ) // reached the end of the line ..
      { ++cp;        // skip '\r' (CR), and hopefully also '\n' (NL)
        if( (cp<cpEndstop) && (*cp=='\n') )
         { ++cp;
         }
        break;
      }
     ++cp;
   }
  nCharsSkipped = cp - *ppszSource;
  *ppszSource = cp;
  return nCharsSkipped;
} // end UTL_SkipToEndOfLine()

//---------------------------------------------------------------------------
int UTL_CopyStringUntilEOL( BYTE *pbSource, char *pszDest, int iMaxDestLen )
  // Copies a string, until (but not including) an END-OF-LINE marker,
  //                  or until the end-of-string marker (trailing zero),
  //                  or until a maximum string length as specified .
  // The return value is the NUMBER OF BYTES actually copied from pbSource.
  // Note: The destination will ALWAYS be terminated with a trailing zero,
  //       i.e. it will ALWAYS be a valid "C"-string .
  //       If iMaxDestLen is 1 (ONE), there will ONLY be that trailing zero.
  //       iMaxDestLen < 1 is **ILLEGAL** !
{
  BYTE *bp = pbSource;
  while( (iMaxDestLen>1) && *bp!='\r' && *bp!='\n' && *bp!='\0' )
    { *pszDest++ = (char)*bp++;
    }
  if( iMaxDestLen>0 )
    { *pszDest++ =  '\0';
    }
#ifdef __BORLANDC__
  (void)pszDest; // ".. assigned but never used" .. shut up Borland !
#endif

  return bp - pbSource;
} // end UTL_CopyStringUntilEOL()


//---------------------------------------------------------------------------
long UTL_ParseInteger(BYTE **ppbSource, int ndigits)
  // Also good for HEXADECIMAL NUMBERS WITH '0x' PREFIX since 2006-01 !
{
 long ret=0;
 int  neg=0;
 BYTE *bp;
 BYTE c;
  bp = *ppbSource;
  while(*bp==' ' || *bp=='\t')   // skip SPACES and TABS (for reading "text data files")
    { ++bp; }
  if(*bp=='-')
    { ++bp; neg=1; }
  else
  if(*bp=='+')
    { ++bp; }
  if( bp[0]=='0' && bp[1]=='x' ) // hexadecimal (C-style) ?
   { bp += 2;  // skip hex prefix
     while(ndigits>0)
      {
        c=*bp;
        if( c>='0' && c<='9' )
         { ++bp;
           --ndigits;
           ret = 16*ret + (c-'0');
         }
        else if(c>='a' && c<='f')
         { ++bp;
           --ndigits;
           ret = 160*ret + (c-'a'+10);
         }
        else if(c>='A' && c<='F')
         { ++bp;
           --ndigits;
           ret = 16*ret + (c-'A'+10);
         }
        else
           break;
      }
   }
  else // decimal :
   { while( (c=*bp)>='0' && (c<='9') && (ndigits>0) )
      { ++bp;
        --ndigits;
        ret = 10*ret + (c-'0');
      }
   }
  *ppbSource = bp;
  return neg ? -ret : ret;
} // end UTL_ParseInteger()


//---------------------------------------------------------------------------
BOOL UTL_ParseFormattedDateAndTime(BYTE *format, double *pdblDateTime,
                                   BYTE **ppSource )
  // Example for format:  "DD.MMM YYYY hh:mm:ss.s"
  //          &  source:  "01.May 2001 14:28:02.5"
  // More of less the inverse to UTL_FormatDateAndTime() .
  // The returned value is similar to the UNIX time:
  //         "SECONDS elapsed since January 1st, 1970, 00:00:00" .
{
 char   fmt,fmt2,fmt3;
 struct tm *gmt;
 int    i;
 long   date_and_time;
 int iYear, iMonth, iDay, iHour, iMinute;
 double dblSecond;
 double dblPowerOfTen;

  DOBINI();

  // Split the default date+time value, if some components are missing.
  if( (*pdblDateTime>=0) && (*pdblDateTime < (double)0x7FFFFFFF) )
    date_and_time = (long)*pdblDateTime;
   else
    date_and_time = (long)fmod(*pdblDateTime, 24*60*60);
  gmt = gmtime(&date_and_time);   // fill structure with default values...
  iYear  = gmt->tm_year+1900;
  iMonth = gmt->tm_mon+1;
  iDay   = gmt->tm_mday;
  iHour  = gmt->tm_min;
  iMinute= gmt->tm_min;
  dblSecond= (double)gmt->tm_sec + fmod(*pdblDateTime, 1.0);

  while( ((fmt=*format++)!=0) && (**ppSource!=0) )
   {
    fmt2 = *format; // to look ahead at following letters in the format string
    fmt3 = *(format+1);
    switch(fmt)
     {
       case 'Y':  // Year  (1 to 4 digits).
       case 'y':  //  Only if there is an OBVIOUS delimiter in the format string,
                  //  we could ignore the number of 'Y's one fine day.
                  //  Think of this format string: "YYYYMMDDhhmmss"
            if(fmt2=='Y')
             {
              ++format;  // skip 2nd letter in format string
              if(fmt3=='Y')
               {
                ++format;  // skip 3rd letter in format string
                if(*format=='Y')
                 {
                  ++format;  // skip 4th letter in format string
                  iYear = UTL_ParseInteger(ppSource,4);
                 }
                else // 3-letter format (??)
                 {
                  iYear = (iYear - (iYear%1000))
                        + UTL_ParseInteger(ppSource,2);
                 }
               }
              else // 2-letter format:
               {
                iYear = (iYear - (iYear%100))
                      + UTL_ParseInteger(ppSource,2);
               }
             } // end if (fmt2..)
            else // 1-letter format (used for automatic filenames somewhere)
             {
              iYear = (iYear - (iYear%10))
                    + UTL_ParseInteger(ppSource,1);
             }
            break;

       case 'M':  // month (2 digits or THREE LETTERS (Jan,Feb, etc) )
            if(fmt2=='M' || fmt2=='m')
             {
              ++format;  // skip 2nd letter in format string
              if( (fmt3=='M')   // "MMM" = { JAN, FEB, .. }
                ||(fmt3=='m') ) // "Mmm" = { Jan, Feb, .. }
               {
                ++format;  // skip 3rd letter in format string
                for(i=0;i<12;++i)
                 {
                   if(strnicmp( (char*)*ppSource,UTL_months1[i],3)==0)
                     { iMonth = i+1;  break;  }
                   if(strnicmp( (char*)*ppSource,UTL_months2[i],3)==0)
                     { iMonth = i+1;  break;  }
                 } // end for
                for(i=0;i<=2;++i)
                 if(UTL_IsAsciiLetter(**ppSource) )
                    ++*ppSource;
               }
              else // 2-letter format: numerical, 1..12
               {
                iMonth = UTL_ParseInteger(ppSource,2);
               }
             } // end if (fmt2..)
            else // 1-letter format (??)  .. possibly used in filenames
             {
              iMonth = (iMonth-(iMonth%10)) + UTL_ParseInteger(ppSource,1);
             }
            break;

       case 'D':  // day (2 digits)
            if(fmt2=='D')
             {
              ++format;  // skip 2nd letter in format string
              iDay = UTL_ParseInteger(ppSource,2);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              iDay = (iDay-(iDay%10)) + UTL_ParseInteger(ppSource,1);
             }
            break;

       case 'h':  // hour
            if(fmt2=='h')
             {
              ++format;  // skip 2nd letter in format string
              iHour = UTL_ParseInteger(ppSource,2);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              iHour = (iHour-(iHour%10)) + UTL_ParseInteger(ppSource,1);
             }
            break;

       case 'm':  // minute (usually 2 digits)
            if(fmt2=='m')
             {
              ++format;  // skip 2nd letter in format string
              iMinute = UTL_ParseInteger(ppSource,2);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              iMinute = (iMinute-(iMinute%10)) + UTL_ParseInteger(ppSource,1);
             }
            break;

       case 's':  // second (at least 2 digits, possible 'fractional' !)
            if(fmt2=='s')
             {
              ++format;  // skip 2nd letter in format string
             } // end if (fmt2..)
            dblSecond = UTL_ParseInteger(ppSource,2);
            if( format[0]=='.' && (**ppSource=='.') && format[1]=='s')
             { // looks like tenths of a second after a decimal point !
               ++format;     // skip "." from format string
               ++*ppSource;  // skip "." from source string
               dblPowerOfTen = 1.0;
               // Modified 2015-01-24:  If the time may have decimal places
               //   in the format string, e.g. "ss.s",
               //   then the parser may parse even MORE digits *after* the dot.
               // Reason: in AFilePrp.cpp .
               while( /*ex: *format=='s' &&*/ (**ppSource>='0') && (**ppSource<='9') )
                {
                  dblPowerOfTen *= 0.1;
                  dblSecond += (double)UTL_ParseInteger(ppSource,1) * dblPowerOfTen;
                }
               while( *format=='s' )
                { ++format;  // skip any number of 's'es from the format string after the decimal point
                }
             }
            break;

       default: // all other characters will be skipped...
            if(**ppSource>=' ')
              ++*ppSource;
            break;
     } // end switch(fmt)
   } // end while(fmt)
  *pdblDateTime = UTL_ConvertYMDhmsToUnix(
      iYear, iMonth, iDay, iHour, iMinute, dblSecond );

  DOBINI();
  return true;
} // end UTL_ParseFormattedDateAndTime()



//---------------------------------------------------------------------------
char *UTL_StringToTime( char *cp,  double dblDefaultTime,
                        double *pdblDestTime,
                        char date_separator, char time_separator  )
   /* Converts a string in various formats into a 32-bit value in SECONDS.
    *
    * Inputs:
    *   cp           : pointer to null-terminated source string.
    *   default_time : may be used if some components (date,time) may be
    *                  missing in the source string. FORMAT: UTC (GMT).
    * Outputs:
    *     dst_time   : seconds elapsed since Jan 1 1970, 00:00:00 UTC.
    *           If only time-of-day required, use value MODULO (24*3600) .
    * Return value: pointer to the character FOLLOWING the time-expression.
    */
{
 char c;
 long l;
 int  n_digits;
 int  i_hms;
 int  i_dmy;
 int  date_format;
 struct tm gmt;

  DOBINI();

  l = (long)dblDefaultTime;
  gmt = *gmtime( &l );
  i_hms  = 0;
  i_dmy  = 0;
  date_format = 0;
  do
   {
     c = *cp;
     l = 0;
     n_digits = 0;
     while(c>='0' && c<='9') // looks like a decimal value. Convert it.
      {
        l = l * 10 + (c-'0');
        c = *++cp;
        ++n_digits;
      }
     if (n_digits>0)
      {
       // next decimal value has been converted.   ':'=time ...
       if( (c==time_separator) || ( (c!=date_separator) && (c!='-') && (i_hms>0) ) )
        { // the converted number was hour or minute:
          i_dmy = 0;
          switch(i_hms)
           {
             case 0: // hours
                   gmt.tm_hour = l;  i_hms=1; break;
             case 1: // minutes
                   gmt.tm_min  = l;  i_hms=2; break;
             case 2: // seconds
                   gmt.tm_sec  = l;  i_hms=3; break;
             default: // ????
                   break;
           }
          if (*cp!='\0') // skip separator to next digit
             ++cp;
        }
       else // not ':'..
       // date format "1" :  DD.MM.YYYY,  DD.MM YYYY
       if(  (date_format != 2)
         && (     (c==date_separator)
               || ((c==' ') && (i_dmy==1) )
               || ((c!=time_separator) && (i_dmy==2) ) )  //   no special char 3rd number
         )
        {
          if(c==date_separator) date_format=1;
          i_hms = 0;
          switch(i_dmy)
           {
             case 0: // day
                   gmt.tm_mday= l; i_dmy=1; break;
             case 1: // month
                   gmt.tm_mon = l; i_dmy=2; break;
             case 2: // year
                   gmt.tm_year =l; i_dmy=3; break;
             default: // ????
                   break;
           }
          if (*cp!='\0') // skip separator to next digit
             ++cp;
        }
       else
       // date format "2" :  YYYY-MM-DD
       if(c=='-')
        {
          date_format=2;
          i_hms = 0;
          switch(i_dmy)
           {
             case 0: // year
                   gmt.tm_year= l; i_dmy=1; break;
             case 1: // month
                   gmt.tm_mon = l; i_dmy=2; break;
             case 2: // day
                   gmt.tm_mday=l;  i_dmy=3; break;
             default: // ????
                   break;
           }
          if (*cp!='\0') // skip separator to next digit
             ++cp;
        }
     } // end if(n_digits>0)
   } while ( (c!=0) && (*cp) && (n_digits>0) );


  if(pdblDestTime != NULL)
   {
    gmt.tm_isdst = 0;
    // mktime seems to convert a  tm struct into a LOCAL time
    // which is definitely not what we want here !
    // The global long variable _timezone contains the difference
    // in seconds between GMT and local standard time (in PST,
    // _timezone is 8 x 60 x 60).
    // The global variable _daylight is used to tell the RTL's
    // functions (mktime & localtime) whether they should take
    // daylight saving time into account if it runs into a date
    // that would normally fall into that catagory.
    // It is set to 1 if the daylight savings time conversion
    // should be applied. These values are set by tzset,
    // not by the user program directly.
    if(gmt.tm_year>=1900)
       gmt.tm_year=gmt.tm_year-1900;
    if(gmt.tm_mon>=1)
       gmt.tm_mon =gmt.tm_mon - 1;   // YES, they count months from 0..11,
                                     // but days from 1..31 - brain damaged >:-|
    gmt.tm_isdst=0; // don't want to see this rotten daylight saving thing anymore
    l = mktime(&gmt);
    if(l<0) // mktime failed... reason is internal addition of timezone,
     {      // when converting the TIME ONLY (like hh:mm:ss).
       ++gmt.tm_mday;
       l = mktime(&gmt) - 24*3600;
     }
    // mktime SEEMS TO BE AFFECTED BY _timezone + _daylight or something else
    //  which is not what we want here (what a mess !!!).
    // Try to convert the result from "mktime" (whatever it is) into GMT.
    // Example: the string 22:00:00 should be converted to 79200,
    //          but at this point it was 75600 (in DL, May 2001).
    // The value of _timezone was -3600, so must SUBTRACT (!) the zone here:
    l -= _timezone;
    if (l<0)
        l += 24*3600;
    *pdblDestTime = l;  // value 0 here is [Jan 1 1970,] 00:00:00 GMT(!)
   }

  DOBINI();

  return cp;
} // end StringToTime


// easier stuff follows :-)

//---------------------------------------------------------------------------
double UTL_VgainToDecibel(double voltage_factor)
{  // Convert a voltage gain factor into decibels.
   // A factor of zero is illegal, a factor of 2 is +6 dB
  if(voltage_factor>0)
     return 20*log10(voltage_factor);
  else // illegal value, most likely a factor of "zero":
     return -300.0;    // 2008-08-08 : changed from -200 to -300,
                       // to be compatible with the floating point error handling
                       // in SoundMaths.c : _matherr(), log10() .
} // end UTL_VgainToDecibel

double UTL_DecibelToVgain(double decibels)
{  // Convert a decibel value into a VOLTAGE GAIN FACTOR.
   // 0 decibels should be factor 1.0 .
   return  pow(10.0, decibels/20.0);  // don't use pow10 here, it takes an INTEGER argument !
} // end UTL_DecibelToVgain

//---------------------------------------------------------------------------
double UTL_dBFS_to_other_dB_Offset( int iToScaleUnit )
  // Returns an OFFSET which must be ADDED
  //    to turn a "decibel-FULL-SCALE"-value (dBfs)
  //    into "some other kind of dB" (dBV for example) .
{
  // "previous" parameters to tell if re-calculation is really necessary...
  static double dblOldAdcInputImpedance = -1.0;
  static double dblAdcMaxInputVoltage = -1.0;
  static double dblDecibelOffset[32];
  int i;

  DOBINI();

  iToScaleUnit &= 31;    // remove all "unit flag bits"

  if( (dblOldAdcInputImpedance != UTL_dblAdcInputImpedance)
    ||(dblAdcMaxInputVoltage   != UTL_dblAdcMaxInputVoltage) )
   {
     // remember the current parameters to avoid unnecessary recalculation:
     dblOldAdcInputImpedance = UTL_dblAdcInputImpedance;
     dblAdcMaxInputVoltage   = UTL_dblAdcMaxInputVoltage;

     // remove "garbage" in the lookup table  :
     for(i=0;i<=31;++i)
         dblDecibelOffset[i] = 0.0;   // no offset by default..

     // recalculate all necessary dB-conversion-offsets :
     dblDecibelOffset[SCALE_UNIT_dBfs] = 0.0;  // reference level "full scale" -> no dB offset
     dblDecibelOffset[SCALE_UNIT_dB]   = 0.0;  // offset for unknown reference level
     dblDecibelOffset[SCALE_UNIT_dBV]  =       // reference level is 1 volt
        // positive value must be added if input voltage range is above 1 V ->
        20 * log10( UTL_dblAdcMaxInputVoltage );

     dblDecibelOffset[SCALE_UNIT_dBuV]  =     // reference level is 1 microvolt rms across any impedance
        // positive value must be added if input voltage range is above 1V ->
        20 * log10( UTL_dblAdcMaxInputVoltage / 1e-6 );

     dblDecibelOffset[SCALE_UNIT_dBm]  =       // reference level is 1 milliwatt (often across 600 ohms)
        // positive addition if voltage range is above U=sqrt(P*R)=sqrt(1mW * Rin) ....
        20 * log10( UTL_dblAdcMaxInputVoltage / sqrt( 1e-3 * UTL_dblAdcInputImpedance ) );
        // Example: MaxInputVoltage=2 V, InputImpedance=600 Ohm  -> [dBm] = [dBfs] + 8.2dB

   } // end if < ADC input range   or   input impedance changed >

  DOBINI();

  return dblDecibelOffset[ iToScaleUnit ];

} // end UTL_dBFS_to_other_dB_Offset()


//---------------------------------------------------------------------------
double UTL_AnyAmplitudeToNormalizedV(
          double dblAnyAmplitude,    // input value (to be converted)
             int iFromScaleUnit )    // unit of input value, SCALE_UNIT_xxxxx
   // Converts "any possible amplitude/magnitude value"
   //          INTO normalized ADC units (voltage) .
   // Caution, uses some ugly global variables as invisible parameters
   //  ( UTL_dblAdcMaxInputVoltage   and   UTL_dblAdcInputImpedance ) !
{
 switch( iFromScaleUnit )
  {
   case SCALE_UNIT_NONE     :      // NO unit, or UNKNOWN unit :
   case SCALE_UNIT_UNKNOWN  :      // convert "something unknown" to VOLTAGE (how?)
        return dblAnyAmplitude;    // not possible to convert anything here !

   case SCALE_UNIT_NORM_V:         // voltage, normalized to +/- 1.0 for full ADC input swing
        return dblAnyAmplitude;
   case SCALE_UNIT_PERCENT:        // voltage, but expressed as "percentage" (100 % = ADC clipping)
        return dblAnyAmplitude * 0.001;
   case SCALE_UNIT_VOLT:           // voltage (V), 'calibrated' using UTL_dblAdcMaxInputVoltage
        return dblAnyAmplitude / UTL_dblAdcMaxInputVoltage;

   case SCALE_UNIT_WATT     :      // from POWER/Watt INTO normalized ADC units (voltage)
        // P = U*I = U/R; U=sqrt( P * R )
        if( dblAnyAmplitude < 0 )  // a POWER can never be negative; someone must have fooled with the PHASE
              return -sqrt( -dblAnyAmplitude * UTL_dblAdcInputImpedance ) / UTL_dblAdcMaxInputVoltage;
         else return  sqrt(  dblAnyAmplitude * UTL_dblAdcInputImpedance ) / UTL_dblAdcMaxInputVoltage;
   case SCALE_UNIT_NORM_PWR :      // power, normalized to 0...1 for full A/D converter input range
        if( dblAnyAmplitude < 0 )
              return -sqrt( -dblAnyAmplitude );
         else return  sqrt(  dblAnyAmplitude );

   case SCALE_UNIT_dB    :         // from dB , unknown reference level (using UTL_dblAdcMaxInputVoltage as user-defined reference)
        return pow(10.0, dblAnyAmplitude/20.0) / UTL_dblAdcMaxInputVoltage;

   case SCALE_UNIT_dBfs  :         // from dB , reference level is  "full scale" (0 dBfs = clipping), INTO normalized ADC units (voltage)
        return pow(10.0, dblAnyAmplitude/20.0);

   case SCALE_UNIT_dBV   :         // from dBV , reference level 1 Vrms,  INTO normalized ADC units (voltage)
        return pow(10.0, dblAnyAmplitude/20.0) / UTL_dblAdcMaxInputVoltage;

   // ex: case SCALE_UNIT_dBu:  dropped support for this stoneage unit; nobody uses 600 Ohm equipment these days
        // The 0.775 volt value is "a 0 dBm sine wave fed into 600 ohms" (audiophile stuff)
        // ( but the exact value is 0.7745966692 V = sqrt( 1e-3 watt * 600 ohm )
        // return 0.7745966692 * pow(10.0, dblAnyAmplitude/20.0) / UTL_dblAdcMaxInputVoltage;

   case SCALE_UNIT_dBuV  :         // from "dB above 1 microvolt RMS" INTO normalized ADC units (voltage)
        return 1e-6 * pow(10.0, dblAnyAmplitude/20.0) / UTL_dblAdcMaxInputVoltage;

   case SCALE_UNIT_dBm   :         // from dB , reference level 1 milliwatt (across the devices DEFINED input impedance)
        return sqrt( 1e-3 * UTL_dblAdcInputImpedance) * pow(10.0, dblAnyAmplitude/20.0) / UTL_dblAdcMaxInputVoltage;

   default:
        return dblAnyAmplitude;
  } // end switch( iFromScaleUnit )

} // end UTL_AnyAmplitudeToNormalizedV()


//---------------------------------------------------------------------------
double UTL_ConvertAnyAmplitudeToNormalizedPower(
          double dblAnyAmplitude,    // input value (to be converted)
             int iFromScaleUnit )    // unit of input value, SCALE_UNIT_xxxxx
   // Converts "any possible amplitude/magnitude value" INTO a power
   //          (or, precisely "something proportinal to an energy") .
   // Purpose: Averaging !  The inverse function, to convert the averaged value
   //          back into that "something", is UTL_ConvertNormalizedPowerToAnyAmplitude() .
   // Caution, this function MAY use some ugly global variables, for example
   //          UTL_dblAdcMaxInputVoltage   and   UTL_dblAdcInputImpedance !
   //    (that's the reason why we prefer NORMALIZED powers internally)
{
 switch( iFromScaleUnit )
  {
   case SCALE_UNIT_NONE     :      // NO unit, or UNKNOWN unit :
   case SCALE_UNIT_UNKNOWN  :      // convert "something unknown" to VOLTAGE (how?)
        return dblAnyAmplitude * dblAnyAmplitude;  // assume it's a voltage; square it

   case SCALE_UNIT_NORM_V:         // voltage, just square it for some power-proportional thing
        return dblAnyAmplitude * dblAnyAmplitude;  // voltage squared for the result
   case SCALE_UNIT_PERCENT:        // voltage, but expressed as "percentage" (100 % = ADC clipping)
        dblAnyAmplitude *= 0.01;   // from "voltage percentage" to +/- 1.0 .
        return dblAnyAmplitude * dblAnyAmplitude;  // voltage squared for the result
   case SCALE_UNIT_VOLT:           // voltage (V), 'calibrated' using UTL_dblAdcMaxInputVoltage
        if( UTL_dblAdcMaxInputVoltage > 0.0 )
            dblAnyAmplitude /= UTL_dblAdcMaxInputVoltage;
        return dblAnyAmplitude * dblAnyAmplitude;  // voltage squared for the result

   case SCALE_UNIT_WATT     : // from POWER/Watts INTO something power-proportinal...
   case SCALE_UNIT_NORM_PWR : // power, normalized to 0...1 for full A/D converter input range
        // P = U*I = U/R; U=sqrt( P * R )
        return dblAnyAmplitude;   // just ignore the scaling factor in this case !

   case SCALE_UNIT_dB    :        // from dB , various reference level, all treated the same here:
   case SCALE_UNIT_dBfs  :
        return pow(10.0, dblAnyAmplitude/*here: dB*/ / 10.0 );  // dBfs -> normalized power
   case SCALE_UNIT_dBV   :
   case SCALE_UNIT_dBm   :
        return pow(10.0, dblAnyAmplitude/*here: dB*/ / 10.0 );  // dB -> normalized power

   case SCALE_UNIT_dBuV  :  // dBV ("dBuV") into normalized power (1.0 for full ADC input)
        dblAnyAmplitude = 1e-6 * pow(10.0, dblAnyAmplitude/*here: dB*/ / 20.0 );  // dBuV -> voltage in VOLTS
        if( UTL_dblAdcMaxInputVoltage > 0.0 )
            dblAnyAmplitude /= UTL_dblAdcMaxInputVoltage;
        return dblAnyAmplitude * dblAnyAmplitude;  // voltage squared for the result

   default:
        return dblAnyAmplitude * dblAnyAmplitude;
  } // end switch( iFromScaleUnit )

} // end UTL_ConvertAnyAmplitudeToNormalizedPower()

//---------------------------------------------------------------------------
double UTL_ConvertNormalizedPowerToAnyAmplitude(
          double dblPowerOrEnergy, // input value (to be converted)
             int iToScaleUnit )    // unit of output value, SCALE_UNIT_xxxxx
   // This is the inverse function to UTL_ConvertAnyAmplitudeToNormalizedPower() .
{
  double dblResult;

 if( dblPowerOrEnergy < 1e-100 ) // this is illegal (for a power or energy) ...
     dblPowerOrEnergy = 1e-100;  // could have been a numeric rounding error (?)
 if( dblPowerOrEnergy > 1e100 )
     dblPowerOrEnergy = 1e100;
 // (no matter if it was an error, after this we can safely calculate the square root)

 switch( iToScaleUnit )
  {
   case SCALE_UNIT_NONE     :      // NO unit, or UNKNOWN unit :
   case SCALE_UNIT_UNKNOWN  :      // convert to "something unknown" (how?)
        return sqrt( dblPowerOrEnergy );  // back from "power" to "voltage"

   case SCALE_UNIT_NORM_V:         // voltage, just square it for some power-proportional thing
        return sqrt( dblPowerOrEnergy );  // back from "power" to "voltage"
   case SCALE_UNIT_PERCENT:        // voltage, but expressed as "percentage" (100 % = ADC clipping)
        return sqrt( dblPowerOrEnergy ) * 100.0;  // back from "power" to "voltage in percent"
   case SCALE_UNIT_VOLT:           // voltage (V), 'calibrated' using UTL_dblAdcMaxInputVoltage
        dblResult = sqrt( dblPowerOrEnergy );
        if( UTL_dblAdcMaxInputVoltage > 0.0 )
            dblResult *= UTL_dblAdcMaxInputVoltage;
        return dblResult;

   case SCALE_UNIT_WATT     :     // POWER/Watts ...
        return dblPowerOrEnergy;  // already ignored the scaling factor in this case !
   case SCALE_UNIT_NORM_PWR :     // power, normalized to 0...1 for full A/D converter input range
        return dblPowerOrEnergy;

   case SCALE_UNIT_dB    :        // from power into dB , various reference level, all treated the same here:
   case SCALE_UNIT_dBfs  :
   case SCALE_UNIT_dBV   :
   case SCALE_UNIT_dBm   :
        // Often CRASHED(!) here with a floating-point error (same old story, over and over..)
        // Trying to "run until return" with the debugger is wishful thinking; it never worked.
        return 10.0 * log10(dblPowerOrEnergy);  // power(energy) -> dB, various reference levels

   case SCALE_UNIT_dBuV  :  // convert from normalized power (1.0 = max ADC input) to dBV ("dBuV")
        if( UTL_dblAdcMaxInputVoltage > 0.0 )
         { dblPowerOrEnergy *= sqrt(UTL_dblAdcMaxInputVoltage);
         }
        dblResult = 10.0 * log10(dblPowerOrEnergy);  // power(energy) -> dBV
        // Now convert from dBV to dBuV :
        dblResult += 60.0;  // 10*log(1e6) = 60
        return dblResult;

   default:
        return sqrt( dblPowerOrEnergy );  // back from "power" to "voltage" or whatever linear thing it was
  } // end switch( iFromScaleUnit )

} // end UTL_ConvertNormalizedPowerToAnyAmplitude()



//---------------------------------------------------------------------------
double UTL_NormalizedVToAnyAmplitude(
          double dblVoltage,       // input value (normalized ADC units, ~voltage)
             int iToScaleUnit )    // unit of input value, SCALE_UNIT_xxxxx
   // Converts a NORMALIZED VOLTAGE into "any possible amplitude/magnitude value" .
   // Caution, uses some ugly global variables as invisible parameters
   //  ( UTL_dblAdcMaxInputVoltage   and   UTL_dblAdcInputImpedance ) !
   // Note: Many of the following conversion formulas have been copied
   //       from UTL_NormalizedVToAnyAmplitude() into other functions also,
   //       for example into UTL_ConvertVoltagesToOtherUnits()  !
{
 switch( iToScaleUnit )
  {
   case SCALE_UNIT_UNKNOWN  :      // convert normalized voltage to "something unknown" (how?)
        return dblVoltage;         // not possible to convert anything here !

   case SCALE_UNIT_NORM_V   :      // convert normalized voltage to normalized voltage ?!
        return dblVoltage;

   case SCALE_UNIT_PERCENT  :      // convert normalized voltage to "PERCENTAGE of full scale"
        return dblVoltage * 100.0;

   case SCALE_UNIT_VOLT     :      // convert normalized voltage to VOLTAGE (peanuts)
        return dblVoltage * UTL_dblAdcMaxInputVoltage;

   case SCALE_UNIT_WATT     :      // convert voltage to  POWER/Watt
        dblVoltage *= UTL_dblAdcMaxInputVoltage;  // normalized -> absolute voltage
        return dblVoltage * dblVoltage / UTL_dblAdcInputImpedance; // voltage->power,  P = U*I = U/R
   case SCALE_UNIT_NORM_PWR :     // power, normalized to 0...1 for full A/D converter input range
        return dblVoltage * dblVoltage;

   case SCALE_UNIT_dB    :         // convert normalized voltage to  dB , unknown reference level
        dblVoltage *= UTL_dblAdcMaxInputVoltage;  // apply "user-defined reference" (!)
        if(dblVoltage>0)  return 20*log10(dblVoltage);
                    else  return -200.0;  // illegal value, assume "ultimately weak signal"

   case SCALE_UNIT_dBfs  :         // convert normalized voltage to  dB , reference level is  "full scale"
        if(dblVoltage>0)  return 20*log10(dblVoltage);
                    else  return -200.0;  // illegal value, assume "ultimately weak signal"

   case SCALE_UNIT_dBV   :         // convert normalized voltage to  dB , reference level 1 Vrms
        dblVoltage *= UTL_dblAdcMaxInputVoltage;  // normalized -> absolute voltage
        if(dblVoltage>0)  return 20*log10(dblVoltage);
                    else  return -200.0;  // illegal value, assume "ultimately weak signal"

   case SCALE_UNIT_dBuV  :         // convert normalized voltage to  dBV (dBuV)
        dblVoltage = dblVoltage * 1e6;
        if( UTL_dblAdcMaxInputVoltage > 0.0 )
         { dblVoltage *= UTL_dblAdcMaxInputVoltage;
         }
        if(dblVoltage>0)  return 20*log10(dblVoltage); // voltage(relative) -> dBuV
                    else  return -200.0;  // illegal value, assume "ultimately weak signal"

   case SCALE_UNIT_dBm   :         // convert normalized voltage to   dBm (dBmilliwatt) ,
        //  reference level is 1 milliwatt across the devices DEFINED input impedance (!)
        dblVoltage *= UTL_dblAdcMaxInputVoltage;  // normalized -> absolute voltage
        dblVoltage /= sqrt( 1e-3 * UTL_dblAdcInputImpedance);
        if(dblVoltage>0)  return 20*log10(dblVoltage);
                    else  return -200.0;  // illegal value, assume "ultimately weak signal"

   default:
        return dblVoltage;
  } // end switch( iToScaleUnit )

} // end UTL_NormalizedVToAnyAmplitude()

//---------------------------------------------------------------------------
double UTL_ConvertAmplitudeUnit( double dblAmplitudeInAnyUnit,
             int iFromUnit, int iToUnit )
{
  double dblNormalizedVoltage;

  if( iFromUnit == iToUnit )
   {  return dblAmplitudeInAnyUnit;  // no need to convert if both units equal
   }
  if( iFromUnit == SCALE_UNIT_NORM_V )
   {  dblNormalizedVoltage = dblAmplitudeInAnyUnit; // no need to convert INPUT unit
   }
  else
   {  dblNormalizedVoltage = UTL_AnyAmplitudeToNormalizedV( dblAmplitudeInAnyUnit, iFromUnit );
   }
  return UTL_NormalizedVToAnyAmplitude( dblNormalizedVoltage, iToUnit );
} // end UTL_ConvertAmplitudeUnit()

//---------------------------------------------------------------------------
void UTL_ConvertNormalizedVsToOtherUnits(
              int iToScaleUnit,    // desired output unit ( SCALE_UNIT_xxxxx )
              int iCountOfFloats,  // how many 'floating point values' (not necessarily "number of FFT bins")
              T_Float *pfltSrc,    // pointer to INPUT array
              T_Float *pfltDst )   // pointer to OUTPUT array (may be the same as the input array ! )
{
  // Notes:
  // -  Many of the following conversion formulas have been copied
  //    into this routine from UTL_NormalizedVToAnyAmplitude() ,
  //    to save a lot of subroutine calls in the loops.
  // -  It may look like a waste of resources to use DOUBLE instead of FLOAT
  //    in the following temporary variables. BUT:
  //    The compiler will internally convert to double precision anyway,
  //    because the math functions like log10 do not exist for 'float'  !
  float fltIn, fltOut, fltFactor=1.0;

  DOBINI();

 switch( iToScaleUnit )
  {
   case SCALE_UNIT_UNKNOWN  :      // convert normalized voltages to "something unknown" (how?)
        while(iCountOfFloats--)         // not possible to convert anything here,
           *pfltDst++  = *pfltSrc++;    //   just copy
        break;

   case SCALE_UNIT_NORM_V   :      // convert normalized voltages to normalized voltages ?!
        while(iCountOfFloats--)         // not possible to convert anything here,
           *pfltDst++  = *pfltSrc++;    //   just copy
        break;

   case SCALE_UNIT_PERCENT  :      // convert normalized voltages to "PERCENTAGE of full scale"
        while(iCountOfFloats--)
           *pfltDst++  = (*pfltSrc++) * 100.0;
        break;

   case SCALE_UNIT_VOLT     :      // convert normalized voltages to VOLTAGE
        while(iCountOfFloats--)
           *pfltDst++  = *pfltSrc++ * UTL_dblAdcMaxInputVoltage;
        break;

   case SCALE_UNIT_WATT     :      // convert voltage to  POWER/Watt
        while(iCountOfFloats--)
         { fltIn = *pfltSrc++ * UTL_dblAdcMaxInputVoltage; // normalized -> absolute voltage
           fltOut = fltIn * fltIn / UTL_dblAdcInputImpedance;   // voltage -> power,  P = U*I = U/R
           *pfltDst++  = fltOut;
         }
        break;
   case SCALE_UNIT_NORM_PWR :     // convert to power, normalized to 0...1 for full A/D converter input range
        while(iCountOfFloats--)
         { fltIn = *pfltSrc++;
           fltOut = fltIn * fltIn; // voltage -> power,  P = U*I = U/R; here: R=1 ("normalized")
           *pfltDst++  = fltOut;
         }
        break;

   case SCALE_UNIT_dB    :
        // convert normalized voltages to  dB , unknown reference level in UTL_dblAdcMaxInputVoltage
        fltFactor = UTL_dblAdcMaxInputVoltage;
        if( fltFactor < 1e-30 )
         {  fltFactor = 1.0; // avoid log10( 0 )  etc
         }
        while(iCountOfFloats--)
         {
           DOBINI();
           fltIn = *pfltSrc++;
           try  // if you though you can catch floating point errors this way ...
            {   // sorry, but this doesn't work . We'll still suffer from shit
                // like "sqrt: DOMAIN error" ,
                //      "invalid floating point operation",    and the like .
#if( (0) && SWI_ALLOW_TESTING )
              if( fltIn == UTL_UNINITIALIZED_ARRAY_VALUE )
               { UTL_WriteRunLogEntry( "UTL: Processing non-initialized array value !" );
               }
#endif // SWI_ALLOW_TESTING
#if(0) // DO NOT USE _fpclass() - IT'S TOTALLY BUGGED IN BORLAND C++BUILDER !!
              if( _fpclass( fltIn ) == _FPCLASS_PN  ) // check the floating point "class",
                // but calling _fpclass() seems to have some really nasty side-effect
                //  - it "posions" other floating-point operations (see WatchWin.cpp) .
                // The effect is, after calling _fpclass(X) *ONCE*, all following
                // calculations with 'double' precision will only give 'single' (float)
                // results !   Don't believe this ? See \cbproj\fpclass_test !
#else
              if( fltIn>1e-30 )
#endif
               { // Only if the number is "valid", "positive" (non-zero) ...
                 DOBINI();
                 fltIn *= fltFactor;
                 // without the _fpclass check, the program occasionally crashed;
                 //                          a try-except block didn't CATCH it .
                 // Call tree : TSpectrumLab::Timer1Timer()
                 //               -> ProcessSpectraFromInputStream()
                 //                    -> AnalyseSpectraAndUpdateDisplay()
                 //                         -> CLI_MakeTCliSpectrum()
                 //                             -> UTL_ConvertNormalizedVsToOtherUnits()
                 // Some explanations on this obscure NAN / INF - stuff can be found at
                 // http://homepages.borland.com/efg2lab/Mathematics/NaN.htm#QuietAndSignaling ,
                 // locally saved (by WB) as     /downloads/programming/Exploring_NaN.htm .
                 // What they call "IsNaN" in Delphi is _isnan() in some C dialects;
                 // anyway using _fpclass instead of _isnan() & _finite() sounded better .
                 fltOut =  20*log10(fltIn);  // should be safe to calculate log10 now - IS IT REALLY ?
                 DOBINI();
               }
              else // the floating-point value is "BAD"; don't use it..
               {
                 DOBINI();
                 fltOut =  -600.0;  // illegal value, assume "ultimately weak signal" (in dB)
                 // (set a breakpoint above; and try to find the cause for this !!)
               }
             } // end try...
            catch(...)  // something went wrong with the damned floating point stuff ->
             { DOBINI();
               fltOut =  -600.0;  // illegal value, assume "ultimately weak signal" (in dB)
             }
           *pfltDst++  = fltOut;
           DOBINI();
         }
        DOBINI();
        break;

   case SCALE_UNIT_dBfs  :
        // convert normalized voltages to  dB , reference level is  "full scale"
        fltFactor=1.0;    // ... so NO CALIBRATION FACTOR in this case !
        while(iCountOfFloats--)
         { fltIn = *pfltSrc++;
           if(fltIn>0)  fltOut =  20*log10(fltIn);
                  else  fltOut =  -200.0;  // illegal value, assume "ultimately weak signal"
           *pfltDst++  = fltOut;
         }
        break;

   case SCALE_UNIT_dBV   :
        // convert normalized voltages to  dB , reference level 1 Vrms
        fltFactor = UTL_dblAdcMaxInputVoltage;
        while(iCountOfFloats--)
         { fltIn = *pfltSrc++ * fltFactor;
           if(fltIn>0)  fltOut =  20*log10(fltIn);
                  else  fltOut =  -200.0;  // illegal value, assume "ultimately weak signal"
           *pfltDst++  = fltOut;
         }
        break;

#if(0) // "dBu" not supported anymore (since 2009-08)
   case SCALE_UNIT_dBu   :
        // convert normalized voltages to  "dB above 0.775 Vrms"
        // The 0.775 volt value is "a 0 dBm sine wave fed into 600 ohms" (audiophile stuff)
        // ( but the exact value is 0.7745966692 V = sqrt( 1e-3 watt * 600 ohm )
        fltFactor = UTL_dblAdcMaxInputVoltage / 0.7745966692;
        while(iCountOfFloats--)
         { fltIn = *pfltSrc++ * fltFactor;
           if(fltIn>0)  fltOut =  20*log10(fltIn);
                  else  fltOut =  -200.0;  // illegal value, assume "ultimately weak signal"
           *pfltDst++  = fltOut;
         }
        break;
#endif // (0)

   case SCALE_UNIT_dBuV  :
        // Convert normalized voltages to  "dB above 1 microvolt RMS"
        fltFactor = UTL_dblAdcMaxInputVoltage / 1e-6;
        while(iCountOfFloats--)
         { fltIn = *pfltSrc++ * fltFactor;
           if(fltIn>0)  fltOut =  20*log10(fltIn);
                  else  fltOut =  -200.0;  // illegal value, assume "ultimately weak signal"
           *pfltDst++  = fltOut;
         }
        break;

   case SCALE_UNIT_dBm   :
        // convert normalized voltages to   dBm (dBmilliwatt) ,
        //  reference level is 1 milliwatt across the devices DEFINED input impedance (!)
        fltFactor = UTL_dblAdcMaxInputVoltage / sqrt( 1e-3 * UTL_dblAdcInputImpedance);
        while(iCountOfFloats--)
         { fltIn = *pfltSrc++ * fltFactor;
           if(fltIn>0)  fltOut =  20*log10(fltIn);
                else  fltOut =  -200.0;  // illegal value, assume "ultimately weak signal"
           *pfltDst++  = fltOut;
         }
        break;

   default:
        while(iCountOfFloats--)         // no idea how to convert,
           *pfltDst++  = *pfltSrc++;    //   copy this stuff 1:1  (unconverted)
        break;

  } // end switch( iToScaleUnit )

 (void)fltFactor; // .."is assigned a value that is never used" ... oh, shut up

  DOBINI();

} // end UTL_ConvertNormalizedVsToOtherUnits()


/* EOF <utility1.c> */
