//-----------------------------------------------------------------------
// File:    C:\cbproj\SoundUtl\audiomsg.h
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Contains basic information about a block of audio samples,
//          which will be passed from one application to another
//          using WM_COPYDATA messages, or TCP/IP packets, or even UDP frames.
// Platform: Win32 (for Linux TCP/IP may work, but never tested) .
// Compiler: Borland C++Builder,  [wx]Dev-Cpp (GCC descendants).
// Original: c:\CBproj\SoundUtl\audiomsg.h
// History: See audiomsg.c
//
// Used in the following projects : ... see AUDIOMSG.C !
//-----------------------------------------------------------------------

#ifndef AUDIO_MSG_H
#define AUDIO_MSG_H

#ifndef CPROT
// Note on "extern": do not trust the lousy example from Borland !
//   >extern "c" = nonsense (not recognized by Borland C++ V4.0)
//   >extern "C" = correct  (though the help system says something different)
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  /* ! "cplusplus" */
#endif

#ifndef SHORT
 #define SHORT short int
#endif

#define AUDIOMSG_MAGIC_STRUCT_HEADER (long)0x80008000

#define AUDIOMSG_STRUCT_CODE_AudioMsgInfo     1 /* code for T_AudioMsgInfo    */
#define AUDIOMSG_STRUCT_CODE_AudioShortHeader 2 /* code for T_AudioShortHeader*/
#define AUDIOMSG_STRUCT_CODE_AudioMsgInfoExt  3 /* code for T_AudioMsgInfoExt */
#define AUDIOMSG_STRUCT_CODE_AudioMsgQuery    4 /* code for T_AudioMsgQuery   */

#define AUDIOMSG_VERSION_NUMBER 1   /* Version number of THIS module */

// bitmasks for T_AudioMsgInfo.i32StreamOptions :
#define AUDIOMSG_OPT_SEND_RESPONSE 0x00000001 /* also for TCP/IP and UDP */
#define AUDIOMSG_OPT_BLOCK_SENDER  0x00000002 /* only for WM_COPYDATA-"streams" (local) */

// bitmasks for the result code of the audio data message
//   (returned by AudioMsgBuf_SendAudio_xyz() )
#define AUDIOMSG_RESULT_MASK_FLAG  0xF0000000
#define AUDIOMSG_RESULT_MASK_DATA  0x0FFFFFFF /* mask for 24 bit "returned data" */
#define AUDIOMSG_RESULT_FLAG_OK    0x80000000 /* message has been delivered      */
#define AUDIOMSG_RESULT_CANT_SEND  0x00000000 /* message could not be sent       */
      // (note: ZERO=FALSE is also returned by the SendMessage()-function in case of errors) */

// possible values for T_AudioMsgInfo.i32InfoStringContent
#define AUDIOMSG_INFO_STR_UNUSED       0
#define AUDIOMSG_INFO_STR_COMMAND      1    /* for remote control, parsed by interpreter (future plan) */
#define AUDIOMSG_INFO_STR_RESPONSE     2
#define AUDIOMSG_INFO_STR_TRACK_NAME   21
#define AUDIOMSG_INFO_STR_ARTIST       22
#define AUDIOMSG_INFO_STR_STREAM_NAME  23


//--------------------------------------------------------------------------
//  Data types ...
//--------------------------------------------------------------------------

typedef struct // T_AudioMsgInfo :  DO NOT CHANGE ! Used in various packages !!
{ // Can be sent via WM_COPYDATA, via UDP/IP,  via TCP/IP, or... (who knows?)
  // Spectrum Lab ACCEPTS (and sends, since 2007-11) any of these:
  //    T_AudioMsgInfo, T_AudioMsgInfoExt, T_AudioShortHeader .

  // Byte[0x00..0x0F] identify this structure...
  long i32Magic0x80008000;  // magic byte sequence 0x80008000 (bytes: 0, -128, 0, -128);
       // should never occur in "normal audio data" (or, at least, can be replaced)
  long i32SizeOfStruct;      // here: 144, may be required to skip the struct in a stream or file
  long i32StructCode;        // 1 for "T_AudioMsgInfo", etc (AUDIOMSG_STRUCT_CODE_..)
  long i32Reserve;           // leave this component ZERO for the moment
     // sum: 16 byte

  // Information (parameters) about the audio stream, without making things too complicated
  double dblSampleRate;     // expect something like 11025, 22050, 44100 or 48000 here.
    //  using DOUBLE PRECISION FLOAT because this may be an exactly calibrated value.
  long iDataType_Bits;     // Bits per sample (for ONE channel). Usually 8 or 16 .
  long i32NumChannels;       // number of audio channels. Usually one or two.
  long i32Volume;            // output volume, set by Winamp plugin, 0...255
  long i32Balance;           // thingy called 'pan' in Winamp, -128 ... 128
     // sum: 16+24 = 40 byte

  // Information (variables) about the stream itself
  long i32NrSamplePointsInBlock;  // number of sampling points following this "info"-header
      // (Note: one SAMPLE POINT consists of ( i32NumChannels * iDataType_Bits / 8 ) bytes
  long i32TotalNumSamplePoints;   // number of sampling points which have been sent so far
      // (usually within the current "track". Only look at the lower 31 bits, and guess the rest)
      //  Can be easily converted into "total playing time" using the sample rate .
  long i32NrSamplePointsInTxBuffer; // number of sampling points left in the sender's buffer
      // ( AFTER he sent this block. Can be used by the receiver to estimate
      //   if he accepts the data fast enough, or a sample rate mismatch is pending)
  long i32StreamOptions;     // combination of AUDIOMSG_OPT_ xxxx (see def's)
     // sum: 16+24+16 = 56 byte

  // "info string", may contain track name or similar, or even a REMOTE CONTROL COMMAND (!)
  long i32InfoStringContent; // AUDIOMSG_INFO_STR_ xxxx (see def's)
  char sz80InfoString[84];   // 80 characters + trailing ZERO byte + DWORD-alignment
     // sum: 16+24+16 + 4+84 = 144 byte
     //   (up to this point, T_AudioMsgInfo *AND* T_AudioMsgInfoExt are equal,
     //    except  i32SizeOfStruct and  i32StructCode ) .

} T_AudioMsgInfo;
#ifdef __BORLANDC__   /* only Borland is smart enough for this : */
 #if(sizeof(T_AudioMsgInfo)!= 144 )
    #error "Something wrong with sizeof(T_AudioMsgInfo) !"
 #endif
#endif // BORLANDC


typedef struct // T_AudioShortHeader :  DO NOT CHANGE !
{        // MAY BE used to send an audio stream SIMPLY through UDP/IP .
         // Spectrum Lab ACCEPTS (and sends, since 2007-11) any of these:
         //    T_AudioMsgInfo, T_AudioMsgInfoExt, T_AudioShortHeader .
         //
         // However, if you plan to use a simple microcontroller with ethernet port
         // as "audio source", this structure (=T_AudioShortHeader) may be easier .
  // Byte[0x00..0x0F] identify this structure... SAME LAYOUT AS IN T_AudioMsgInfo !
  long i32Magic0x80008000;  // magic byte sequence 0x80008000 (bytes: 0, -128, 0, -128);
       // should never occur in "normal audio data" (or, at least, can be replaced)
  long i32SizeOfStruct;      // here: 40, may be required to skip the struct in a stream
  long i32StructCode;        // 2 (AUDIOMSG_STRUCT_CODE_AudioShortHeader) for "T_AudioShortHeader"
  long i32Flags;             // leave this component ZERO for the moment
     // sum: 16 byte
  #define C_AUDIOMSG_HEADER_IDENTIFIER_PART_SIZE 16

  // Information (parameters) about the audio stream, without making things too complicated
  double dblSampleRate;     // something like 11025, 22050, 44100 or 48000 samples/sec.
    //  using DOUBLE PRECISION FLOAT because this may be an exactly calibrated value.
  WORD wBitsPerSample;      // Bits per sample (for ONE channel). Usually 8 or 16 (integer) or 32 (float).
  WORD wNumChannels;        // number of audio channels. Usually one or two.
     // sum: 16 + 8+2+2 = 28 byte

  // Information (variables) about the stream itself
  long i32NrSamplePointsInBlock;  // number of sampling points following this "info"-header
      // (Note: one SAMPLE POINT consists of ( NumChannels * BitsPerSample / 8 ) bytes
  long i32TotalNumSamplePoints;   // number of sampling points which have been sent so far
      // (usually within the current "track". Only look at the lower 31 bits, and guess the rest)
      //  Can be easily converted into "total playing time" using the sample rate,
      //  but also to check if UDP packets have been lost (if so, insert silence,
      //  assuming the number of samples is the same as in the previous block ).
  // sum: 28+4+4+4 = 40 byte. Finito.
  // The audio data block follows directly after the header, containing
  // i32NrSamplePointsInBlock * ((wBitsPerSample+7)/8) * wNumChannels  bytes.

} T_AudioShortHeader;
#ifdef __BORLANDC__
  // The Borland C compiler is smart enough for the following. Others aren't !
  #if( sizeof( T_AudioShortHeader ) != 40 )
    #error "Something wrong with sizeof(T_AudioShortHeader) !"
  #endif
#endif
#define C_AUDIOMSG_MIN_HEADER_SIZE 40


typedef struct // T_AudioMsgInfoExt :  DO NOT CHANGE !
{ // MAY BE used to send an audio stream SIMPLY through UDP/IP or WM_COPYDATA .
  // Spectrum Lab ACCEPTS (and sends, since 2007-11) any of these:
  //    T_AudioMsgInfo, T_AudioShortHeader, T_AudioExtendedInfo .
  //
  // This structure was added in 2007-11 (besides the already existing
  //   T_AudioMsgInfo and T_AudioShortHeader), when a new application
  //   required a very precise timestamp including calendar date + time,
  //   down to the microsecond range, of WHEN a sample had been
  //   recorded. Adding new members to T_AudioMsgInfo would be a bad idea
  //   (would cause problems with already existing applications),
  //
  // Byte[0x00..0x0F] identify this structure...
  long i32Magic0x80008000;  // magic byte sequence 0x80008000 (bytes: 0, -128, 0, -128);
       // should never occur in "normal audio data" (or, at least, can be replaced)
  long i32SizeOfStruct;      // here: 144, may be required to skip the struct in a stream
  long i32StructCode;        // 3 for "T_AudioMsgInfoExt" (AUDIOMSG_STRUCT_CODE_AudioMsgInfoExt)
  long i32Reserve;           // leave this component ZERO for the moment
     // sum: 16 byte

  // Information (parameters) about the audio stream, without making things too complicated
  double dblSampleRate;     // expect something like 11025, 22050, 44100 or 48000 here.
    //  using DOUBLE PRECISION FLOAT because this may be an exactly calibrated value.
  long iDataType_Bits;      // Data type, actually "bits per sample for ONE channel".
                             //  Valid: 8 for 8-bit UNSIGNED integer (0..255),
                             //        16 for 16-bit SIGNED integer (+/-32767),
                             //        32 for 32-bit FLOATING POINT (range +/- 1.0).
  long i32NumChannels;       // number of audio channels. Usually one or two.
  long i32Volume;  // output volume, set by Winamp plugin, 0...255 [discarded]
  long i32Balance; // thingy called 'pan' in Winamp, -128 ... 128  [discarded]
     // sum: 16+24 = 40 byte

  // Information (variables) about the stream itself
  long i32NrSamplePointsInBlock;  // number of sampling points following this "info"-header
      // (Note: one SAMPLE POINT consists of ( i32NumChannels * iDataType_Bits / 8 ) bytes
  long i32TotalNumSamplePoints;   // number of sampling points which have been sent so far
      // (usually within the current "track". Only look at the lower 31 bits, and guess the rest)
      //  Can be easily converted into "total playing time" using the sample rate .
  long i32NrSamplePointsInTxBuffer; // number of sampling points left in the sender's buffer
      // ( AFTER he sent this block. Can be used by the receiver to estimate
      //   if he accepts the data fast enough, or a sample rate mismatch is pending)
  long i32StreamOptions;     // combination of AUDIOMSG_OPT_ xxxx (see def's)
     // sum: 16+24+16 = 56 byte

  // "info string", may contain track name or similar, or even a REMOTE CONTROL COMMAND (!)
  long i32InfoStringContent; // AUDIOMSG_INFO_STR_ xxxx (see def's)
  char sz80InfoString[84];   // 80 characters + trailing ZERO byte + DWORD-alignment
     // sum: 16+24+16 + 4+84 = 144 byte
     //   (up to this point, T_AudioMsgInfo *AND* T_AudioMsgInfoExt are equal,
     //    except  i32SizeOfStruct and  i32StructCode ) .

  // ------------------------------------------------------------------------
  // Members which only exist in the T_AudioMsgInfoExt structure,
  //         but not in T_AudioMsgInfo >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
  long double ldblUnixDateAndTime; // current date+timestamp in UNIX format, which is:
         // "Seconds passed since Januar 1st, 1970, 00:00:00.0" (in UTC).
         // Note: Microsoft's "long double" sucks - it's the same as "double". Stupid.
         //       Borland's "long double" is 80 bit (64 bit mantissa) .
         //       Not sure if Borland's "long double" is compatible
         //       with Microsoft's "extended" floating point type .
         // The time refers to the next "audio" sample sent in this block
         // (not to the "begin of recording" ! We may eliminate slight changes
         //  in the sampling clock, or unstable oscillators this way)
         // Should be set by the caller, before calling AudioMsgBuf_SendWmCopydata(),
         // because otherwise, SL's spectrogram display may simply stop .
  double dblReserved[16-2];  // reserved for future "double" members (64 bit IEEE float)
     // sum: 144 + 16*8 = 272 byte

  long   i32Reserved[16];    // up to 16 32-bit integers
     // sum: 272 + 16*4 = 336 byte

} T_AudioMsgInfoExt;
#ifdef __BORLANDC__
  // The Borland C compiler is smart enough for the following. Others aren't !
  #if( sizeof( T_AudioMsgInfoExt ) != 336 )
    #error "Something wrong with sizeof(T_AudioMsgInfoExt) !"
  #endif
#endif
#define C_AUDIOMSG_MAX_HEADER_SIZE 336


typedef struct // T_AudioMsgQuery :  DO NOT CHANGE !
{ // MAY BE used to 'query' (or poll) something from 'the other' side,
  //        using WM_COPYDATA to communicate with another application
  //        running on the same PC as the application which sends the query.
  //
  // This structure was added in 2012-03 (besides the already existing
  //   T_AudioMsgInfo and T_AudioShortHeader).
  //   The RESULT (returned by the windows SendMessage() function)
  //   can only be a single 32-bit value. The meaning of that value
  //   depends on T_AudioMsgQuery.i32FunctionCode - see details below.
  //
  // As for the other messages, Byte[0x00..0x0F] identify this structure..
  long i32Magic0x80008000;  // magic byte sequence 0x80008000 (bytes: 0, -128, 0, -128);
       // should never occur in "normal audio data" (or, at least, can be replaced)
  long i32SizeOfStruct;      // here: 144, may be required to skip the struct in a stream
  long i32StructCode;        // 3 for "T_AudioMsgInfoExt" (AUDIOMSG_STRUCT_CODE_AudioMsgInfoExt)
  long i32Reserve;           // leave this component ZERO for the moment
     // sum: 16 byte

  long i32FunctionCode;      // function code; one of the following:
#    define C_AUDIOMSG_FUNCTION_GET_VERSION_NUMBER     0
#    define C_AUDIOMSG_FUNCTION_GET_NUM_CHANNELS       1
#    define C_AUDIOMSG_FUNCTION_GET_SAMPLERATE         2
#    define C_AUDIOMSG_FUNCTION_GET_FREE_BUFFER_SPACE  4
#    define C_AUDIOMSG_FUNCTION_GET_USED_BUFFER_SPACE  5
  long i32IntegerParam;       // optional 32-bit INTEGER parameter
  double dblFloatParam;       // optional 64-bit floating point parameter
  char sz80StringParam[84];   // optional string parameter; max. 80 characters + trailing ZERO byte + DWORD-alignment

} T_AudioMsgQuery;

// Pack all possible "audio message headers" in a common union,
//    used to pass the address of "any such header" to subroutines :
typedef union
{ T_AudioMsgInfo     ami;    // the old, "short" audio-message-info-header
  T_AudioMsgInfoExt  ami2;   // the newer, "extended" audio-message-header
                             // with timestamp and description of the content
  T_AudioShortHeader ash;
  BYTE  b[C_AUDIOMSG_MAX_HEADER_SIZE];        // header as an array of BYTES
  DWORD dw[(C_AUDIOMSG_MAX_HEADER_SIZE+3)/4]; // header as an array of DWORDS
} T_AudioMsgAnyHeader;

/* Buffer to receive input from another application through messages or streams */
typedef struct tAudioMsgBuffer
{
  T_AudioMsgInfoExt  info;   // extended audio-message-header, contains
                             // sample rate, bits/sample, number of channels,
                             // timestamp and description of the content
  // Note: "info" must be followed DIRECTLY by "bTxBuffer" because both are sent
  //   in a single call of SendMessage() (using WM_COPYDATA), or by other means.
  BYTE bTxBuffer[16384*2*4]; // transmit-buffer for WM_COPYDATA (for various data types)
     // Large enough to send 16384 samples * 2 channel * 4 bytes per 'float' .
     // Note: Since 2012, Spectrum Lab uses 32-bit floating point numbers
     //      'normalized' to +/- 1.0 internally; regardless of the ADC resolution.
     //  The 'audio message buffer' uses the same format internally.
     //  8- or 16-bit integer samples are only generated on-the-fly when required.
     //  bTxBuffer[] is just a 'temporary storage' for any data type (int, float).



  //---The following members of T_AudioMsgBuffer are NOT sent via WM_COPYDATA---
  #define AUDIO_MSG_BUF_MAX_SAMPLE_POINTS 65536 /* MUST BE A POWER OF TWO ! */
  #define AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE 4
  float fltSampleBuffer[AUDIO_MSG_BUF_MAX_SAMPLE_POINTS][AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE];
  LONGLONG m_i64SampleCountIn;  // total number of sample points ENTERED in the buffer.
  LONGLONG m_i64SampleCountOut; // total number of sample points READ FROM the buffer.
  // The lower bits of the above 64-bit counters are used as indices
  //  into the circular buffer (using bitwise and, with 2^N minus 1) .

  // To retrieve the timestamp without thread locking, TWO storages are used.
  // At least ONE of these TWO entries will be valid .
  // See AudioMsgBuf_SetCurrentInputTimestamp() for details.
  struct
   { BOOL     valid;            // marks this entry as valid (TRUE) or invalid (FALSE)
     LONGLONG i64SampleCount;   // sample counter to which THIS timestamp applies
     long double ldblTimestamp; // timestamp in UNIX format
  // double      dblSampleRate; // accurate sample rate "near" the timestamp
  // (SR assumed to be fairly constant; use T_AudioMsgBuffer.info.dblSampleRate)
   } tstamps[2];

  int  iTxErrors;
  int  iRxErrors;

} T_AudioMsgBuffer;



//--------------------------------------------------------------------------
//  Prototypes ...  ( functions commented and explained in audiomsg.c only )
//--------------------------------------------------------------------------

CPROT void AudioMsgInfo_to_AudioMsgInfoExt( T_AudioMsgInfo *pSrc, T_AudioMsgInfoExt *pDst );
CPROT void AudioShortHeader_to_AudioMsgInfoExt( T_AudioShortHeader *pSrc, T_AudioMsgInfoExt *pDst );

// Functions using the T_AudioMsgBuffer structure (would be 'methods' in C++) .
CPROT BOOL AudioMsgBuf_Init(T_AudioMsgBuffer **ppBuf, double dblSampleRate, int iNumChannels );
CPROT void AudioMsgBuf_Delete(T_AudioMsgBuffer **ppBuf);
CPROT long AudioMsgBuf_CalcSampleBlockSizeInBytes( T_AudioMsgAnyHeader *pAMsgHdr );
CPROT BOOL AudioMsgBuf_FillFromInt16( T_AudioMsgBuffer *pBuf, SHORT *pi16Src, int iNrOfWords );
CPROT long AudioMsgBuf_GetUsedBufferSpace_nSamples( T_AudioMsgBuffer *pBuf );
CPROT long AudioMsgBuf_GetFreeBufferSpace_nSamples( T_AudioMsgBuffer *pBuf );
CPROT long AudioMsgBuf_ReadSamplesIntoSeparateChannelBlocks( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToRead, float *pfltChannelA, float *pfltChannelB );
// "Write" samples into the buffer's internal FIFO, in different flavours:
CPROT long AudioMsgBuf_WriteSamplesFromSeparateChannelBlocks( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToWrite, float *pfltChannelA, float *pfltChannelB );
CPROT long AudioMsgBuf_WriteSamplesInterleaved_Uns8( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToWrite, int iChannelsPerSample, unsigned char *pu8Source );
CPROT long AudioMsgBuf_WriteSamplesInterleaved_Int16( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToWrite, int iChannelsPerSample, short int *pi16Source );
CPROT long AudioMsgBuf_WriteSamplesInterleaved_Float32( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToWrite, int iChannelsPerSample, float *pfltSource );


CPROT long AudioMsgBuf_SendAudioViaWmCopydata( T_AudioMsgBuffer *pBuf, HWND hwndMyWindow, HWND hwndHisWindow, int iDataType_Bits );
CPROT long AudioMsgBuf_HandleWmCopydata( T_AudioMsgBuffer *pBuf, COPYDATASTRUCT *pCDS );
CPROT long AudioMsgBuf_HandleQuery( T_AudioMsgBuffer *pBuf, T_AudioMsgQuery *pQuery );
CPROT void   AudioMsgBuf_SetCurrentSamplingRate( T_AudioMsgBuffer *pBuf, double dblSampleRate );
CPROT double AudioMsgBuf_GetCurrentSamplingRate( T_AudioMsgBuffer *pBuf ); // "local" !
CPROT void   AudioMsgBuf_SetNumberOfChannels( T_AudioMsgBuffer *pBuf, int iNumChannels );
CPROT int    AudioMsgBuf_GetNumberOfChannels( T_AudioMsgBuffer *pBuf );    // "local" !
CPROT void   AudioMsgBuf_SetCurrentInputTimestamp( T_AudioMsgBuffer *pBuf, long double ldblUnixDateAndTime );
CPROT BOOL   AudioMsgBuf_GetOutputTimestamp( T_AudioMsgBuffer *pBuf, long double *pldblUnixDateAndTime );

#ifdef USE_WSOCK // use WSOCK.C to send audio via UDP, too ?
 CPROT long AudioMsgBuf_SendAudio_UDP( T_AudioMsgBuffer *pBuf, SOCKET s, char *pszDestIp, int iDestPort );
#endif

// Functions added 2012-03-28, only supported (or used) in Spectrum Lab V2.77 b24 or later,
CPROT long AudioMsg_PollRemoteNumChannels_WMSG( HWND hwndMyWindow, HWND hwndHisWindow );
CPROT long AudioMsg_PollRemoteSamplingRate_WMSG( HWND hwndMyWindow, HWND hwndHisWindow );
CPROT long AudioMsg_PollRemoteBufferSpace_WMSG( HWND hwndMyWindow, HWND hwndHisWindow );


#endif /* AUDIO_MSG_H */
