//-----------------------------------------------------------------------
// File:    audiomsg.c
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Contains basic information about a block of audio samples,
//          which will be passed from one application to another
//          using WM_COPYDATA messages, or TCP/IP packets, or even UDP frames.
//          NO C++ CLASS, BECAUSE THIS MODULE MAY BE USED ONE DAY
//          ON A TARGET WHICH ONLY SUPPORTS "C" (but not C++) .
// Used in:
//  - Spectrum Lab,
//  - Winamp2Speclab-output-plugin,  see \cbproj\winamp2speclab\readme.txt
//  - AudioMessageTest,
//  -  .. ?
//
// Original: c:\CBproj\SoundUtl\audiomsg.cpp
// History:
//   2012-03-29  AudioMsgBuf_SendWmCopydata() now uses the 'larger'
//               T_AudioMsgInfoExt, which contains timestamp etc.
//               For backward compatibility, Spectrum Lab still supports also
//               the 'small' headers (T_AudioMsgInfo and T_AudioShortHeader) .
//             + Audio samples blocks can be 32-bit FLOATING POINT numbers now,
//
//   2007-02-07  Used this module in the project "AudioMessageTest" .
//                See notes in AudioMsgBuf_SendAudio_UDP() if DevCpp
//                complains about missing 'htons@4', 'inet_addr@4', etc !
//   2006-03-18  Added the option to send audio-messages via UDP/IP packets.
//   2005-07-09  Initial version, uses WM_COPYDATA messages only
//
//-----------------------------------------------------------------------

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !

#ifdef USE_WSOCK       // use WSOCK.C to send audio via UDP, too ?
  // (USE_WSOCK must be defined under project options..conditionals or similar)
  // #include <winsock.h> // winsock.h was ok for Borland C++Builder V4, but not for V6
  #include <winsock2.h> // in BCB6, winsock.h lacks some defines,
                        // but winsock2.h must obviously be included BEFORE windows.h
#endif

#include <windows.h>
// Important: NO VCL-STUFF in this module !

#include "utility1.h"  // uses UTL_NamedMalloc instead of malloc (for debugging), and to *GUARANTEE* 8-byte alignment
#ifdef USE_WSOCK       // use WSOCK.C to send audio via UDP, too ?
  // (USE_WSOCK must be defined under project options..conditionals or similar)
  // #include <winsock.h> // winsock.h was ok for Borland C++Builder V4, but not for V6
  // not here: #include <winsock2.h> // in BCB6, winsock.h lacks some defines,
                        // but winsock2.h must obviously be included BEFORE windows.h
#  include "wsock.h"   // old wrapper for some "winsock" services, located in YHF_Tools
  int AMSG_iUdpSendErrors = 0;
#endif // USE_WSOCK, defined somewhere under "project options".."compiler"
#include "audiomsg.h"  // original file location: c:\CBproj\SoundUtl\audiomsg.h



/***************************************************************************/
void AudioMsgInfo_to_AudioMsgInfoExt( T_AudioMsgInfo *pSrc, T_AudioMsgInfoExt *pDst )
  // Converts the old (small) T_AudioMsgInfo structure
  // into the newer (larger) T_AudioMsgInfoExt .
  // Fields which do ONLY exist in T_AudioMsgInfoExt are NOT modified in pDst.
  // An application which SENDS audio (through WM_COPYDATA, etc) may support
  //   one, two, or all three types of 'audio message headers'.
  // An application which RECEIVES audio this way (usually: Spectrum Lab)
  //   should support ALL types of 'audio message headers'
  //   for the sake of compatibility.
{
  pDst->dblSampleRate   = pSrc->dblSampleRate;
  pDst->iDataType_Bits= pSrc->iDataType_Bits;
  pDst->i32NumChannels  = pSrc->i32NumChannels;
  pDst->i32Volume       = pSrc->i32Volume;   // discarded; only existed for Winamp
  pDst->i32Balance      = pSrc->i32Balance;  //  " " "

  pDst->i32NrSamplePointsInBlock   = pSrc->i32NrSamplePointsInBlock;
  pDst->i32TotalNumSamplePoints    = pSrc->i32TotalNumSamplePoints;
  pDst->i32NrSamplePointsInTxBuffer= pSrc->i32NrSamplePointsInTxBuffer;
  pDst->i32StreamOptions           = pSrc->i32StreamOptions;
  pDst->i32InfoStringContent       = pSrc->i32InfoStringContent;
  strncpy( pDst->sz80InfoString, pSrc->sz80InfoString, 80 );
} // end AudioMsgInfo_to_AudioMsgInfoExt()

/***************************************************************************/
void AudioShortHeader_to_AudioMsgInfoExt( T_AudioShortHeader *pSrc, T_AudioMsgInfoExt *pDst )
  // Converts the 'very small' T_AudioShortHeader structure
  // into the larger T_AudioMsgInfoExt .
  // Fields which do ONLY exist in T_AudioMsgInfoExt are NOT modified in pDst.
  // See notes about the three(?) different types of 'audio message headers'
  //           in AudioMsgInfo_to_AudioMsgInfoExt() .
{
  pDst->dblSampleRate   = pSrc->dblSampleRate;
  pDst->iDataType_Bits= pSrc->wBitsPerSample;
  pDst->i32NumChannels  = pSrc->wNumChannels;
  pDst->i32NrSamplePointsInBlock   = pSrc->i32NrSamplePointsInBlock;
  pDst->i32TotalNumSamplePoints    = pSrc->i32TotalNumSamplePoints;
} // end AudioShortHeader_to_AudioMsgInfoExt()


/***************************************************************************/
long AudioMsgBuf_CalcSampleBlockSizeInBytes( T_AudioMsgAnyHeader *pAMsgHdr )
  // Calculates the size of an (uncompressed) block of audio samples,
  // which immediately follows one of the 'audio message headers' in a stream.
{
  if( pAMsgHdr!=NULL )
   { if( pAMsgHdr->ami.i32Magic0x80008000 == AUDIOMSG_MAGIC_STRUCT_HEADER )
      { switch( pAMsgHdr->ami.i32StructCode )
         { case AUDIOMSG_STRUCT_CODE_AudioMsgInfo:     /* T_AudioMsgInfo    */
              return( (pAMsgHdr->ami.iDataType_Bits /* Bits per sample (for ONE channel). Usually 8 or 16 . */
                      *pAMsgHdr->ami.i32NumChannels) / 8)
                     * pAMsgHdr->ami.i32NrSamplePointsInBlock;
           case AUDIOMSG_STRUCT_CODE_AudioShortHeader: /* T_AudioShortHeader*/
              return( (pAMsgHdr->ash.wBitsPerSample /* Bits per sample (for ONE channel). Usually 8 or 16 . */
                      *pAMsgHdr->ash.wNumChannels) / 8)
                     * pAMsgHdr->ash.i32NrSamplePointsInBlock;
           case AUDIOMSG_STRUCT_CODE_AudioMsgInfoExt:  /* T_AudioMsgInfoExt */
              return( (pAMsgHdr->ami2.iDataType_Bits /* Bits per sample (for ONE channel). Usually 8 or 16 . */
                      *pAMsgHdr->ami2.i32NumChannels) / 8)
                     * pAMsgHdr->ami2.i32NrSamplePointsInBlock;
           case AUDIOMSG_STRUCT_CODE_AudioMsgQuery : // received a T_AudioMsgQuery : NOT SUPPORTED HERE YET !
              return  0; // this special message does NOT carry audio samples !

           default: // with all other structs, it's impossible to calculate
                    // the size of an uncompressed audio block IN BYTES !
              break;
         } // end switch
      } // end if < header valid >
   }
  return 0;  // something invalid; assume a block length of ZERO


} // end AudioMsgBuf_CalcSampleBlockSizeInBytes()



/***************************************************************************/
BOOL AudioMsgBuf_Init(T_AudioMsgBuffer **ppBuf,
              double dblSampleRate, // samples/second, possibly "calibrated"
              int    iNumChannels ) // 1="mono",  2="stereo"
   // Initializes one of the buffers which were first used
   //  to send and receive audio data through WM_COPYDATA messages (etc) .
   //  The caller may modify some members in the T_AudioMsgBuffer struct
   //  afterwards if necessary (we don't want to pass everything in the
   //  argument) .
   // Note: AudioMsgBuf_Init() internally allocates memory. To free these resources,
   // an application MUST call AudioMsgBuf_Delete(T_AudioMsgBuffer **ppBuf) when finished .
{
  T_AudioMsgBuffer *pBuf;
  if( *ppBuf!=NULL )
   { pBuf= *ppBuf;
   }
  else // memory for the audio-message-buffer doesn't exist -> allocate it now:
   { pBuf=(T_AudioMsgBuffer*)UTL_NamedMalloc( "AudioMsg", sizeof(T_AudioMsgBuffer) );
   }

  if( pBuf!=NULL )
   {
     memset( &pBuf->info, 0, sizeof(pBuf->info) );
     pBuf->info.i32SizeOfStruct = sizeof(pBuf->info);
     pBuf->info.i32Magic0x80008000 = AUDIOMSG_MAGIC_STRUCT_HEADER;
     pBuf->info.i32StructCode = AUDIOMSG_STRUCT_CODE_AudioMsgInfoExt;
     pBuf->info.dblSampleRate = dblSampleRate; // just a 'default'..
     pBuf->info.iDataType_Bits= 16;  // future plan: support other sample sizes
     pBuf->info.i32NumChannels = iNumChannels;
     *ppBuf = pBuf;  // NOW it's ok to use this buffer (in another thread)
     return TRUE;
   }
  return FALSE;
} // end AudioMsgBuf_Init()

/***************************************************************************/
void AudioMsgBuf_Delete(T_AudioMsgBuffer **ppBuf)
  // see notes in AudioMsgBuf_Init() for an explanation why we need this ;-)
{
  T_AudioMsgBuffer *pBuf = *ppBuf;
  if( pBuf!=NULL )
   { if(  (pBuf->info.i32SizeOfStruct == (long)sizeof(T_AudioMsgInfo) )
        &&(pBuf->info.i32Magic0x80008000 == AUDIOMSG_MAGIC_STRUCT_HEADER ) )
      { pBuf->info.i32Magic0x80008000 = 0; // let everyone know this buffer isn't valid any longer
        UTL_free( pBuf );
        *ppBuf = NULL;
      }
   }
} // end AudioMsgBuf_Delete()


/***************************************************************************/
long AudioMsgBuf_GetUsedBufferSpace_nSamples( T_AudioMsgBuffer *pBuf )
  // Returns the number of audio sample points in the LOCAL BUFFER
  //  (not the buffer at 'the other end' of the communication link)
{
  LONGLONG i64;

  if( pBuf==NULL )
   { return 0;
   }

  i64 = pBuf->m_i64SampleCountIn - pBuf->m_i64SampleCountOut;
  if( i64<0 )
   {  i64=0;
   }
  if( i64>AUDIO_MSG_BUF_MAX_SAMPLE_POINTS) // oops...
   {  i64=AUDIO_MSG_BUF_MAX_SAMPLE_POINTS;
   }
  return (long)i64;
} // end AudioMsgBuf_GetUsedBufferSpace_nSamples()

/***************************************************************************/
long AudioMsgBuf_GetFreeBufferSpace_nSamples( T_AudioMsgBuffer *pBuf )
  // Returns the free (usable) buffer space the LOCAL BUFFER
  //  (not the buffer at 'the other end' of the communication link),
  // measured in SAMPLE POINTS.
  // One sample point may contain one or more channels (stereo or I/Q data).
{
  LONGLONG i64;

  if( pBuf==NULL )
   { return 0;
   }

  i64 = AudioMsgBuf_GetUsedBufferSpace_nSamples(pBuf); // -> number of USED sample entries
  //  For a circular FIFO of this type with 65536 entries,
  //   the max. 'netto' usable size is only 65535 entries !
  return AUDIO_MSG_BUF_MAX_SAMPLE_POINTS - 1 - (long)i64;

} // end AudioMsgBuf_GetFreeBufferSpace_nSamples()


//---------------------------------------------------------------------------
// "Read" samples from the buffer's internal FIFO, in different flavours ..
//---------------------------------------------------------------------------

/***************************************************************************/
long AudioMsgBuf_ReadSamplesIntoSeparateChannelBlocks( T_AudioMsgBuffer *pBuf,
       long nSamplePointsToRead, T_Float *pfltChannelA, T_Float *pfltChannelB )
  // Used by Spectrum Lab to read samples from the T_AudioMsgBuffer ("input").
  // Return value: number of sample-points placed in pBuf->bTxBuffer[],
  //               and drained from the circular FIFO in T_AudioMsgBuffer .
{
  long i,nSamplesRead;
  DWORD dwBufIdx;

  if( pBuf==NULL )
   { return 0;
   }

  dwBufIdx = (DWORD)pBuf->m_i64SampleCountOut & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);

  nSamplesRead = AudioMsgBuf_GetUsedBufferSpace_nSamples( pBuf );
  if( nSamplesRead > nSamplePointsToRead )
   {  nSamplesRead = nSamplePointsToRead;
   }

  // Read the samples from fltSampleBuffer[AUDIO_MSG_BUF_MAX_SAMPLE_POINTS][AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE] :
  if( (pfltChannelB==NULL) || pBuf->info.i32NumChannels<2 )
   { // only retrieve ONE audio channel:
     for(i=0; i<nSamplesRead; ++i)
      { pfltChannelA[i] = pBuf->fltSampleBuffer[dwBufIdx][0];
        dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
      }
   }
  else // pfltChannelB != NULL  *AND*  pBuf->info.i32NumChannels>=2 :
   { // retrieve TWO audio channels (no further checking WITHIN the loop for speed reasons):
     for(i=0; i<nSamplesRead; ++i)
      { pfltChannelA[i] = pBuf->fltSampleBuffer[dwBufIdx][0];
        pfltChannelB[i] = pBuf->fltSampleBuffer[dwBufIdx][1];
        dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
      }
   }
  // Count total number of SAMPLE POINTS, too .
  // This is important for AudioMsgBuf_GetTimestamps();
  // and actually drains the 'read' samples from the FIFO :
  pBuf->m_i64SampleCountOut += nSamplesRead;

  return nSamplesRead;
} // AudioMsgBuf_ReadSamplesIntoSeparateChannelBlocks()


/***************************************************************************/
long AudioMsgBuf_ReadSamplesIntoTxBuffer( T_AudioMsgBuffer *pBuf,
           long nSamplesToSend, int iDataTypeForTransmission )
  // Prepares the transmit-buffer (for WM_COPYDATA, etc) to send
  //       another block of samples .
  //       Various data types are supported .
  // The input is read from (DRAINED) the T_AudioMsgBuffer's circular FIFO .
  // Parameters:
  //   iDataTypeForTransmission : 8  = 8-bit unsigned (0..255),
  //                              16 = 16-bit signed (+/-32k),
  //                              32 = 32-bit *FLOATING POINT* (-1.0 .. +1.0) .
  // Return value: number of sample-points placed in pBuf->bTxBuffer[],
  //               and drained from the circular FIFO in T_AudioMsgBuffer .
{
  int  iSample, nSamplesMax, iChannel, nChannels;
  float f, *pfltSrc;
  unsigned char *pu8Dst; short int *pi16Dst; float *pfltDst; // pointers for various data types
  DWORD dwBufIdx;

  if( pBuf==NULL || nSamplesToSend<=0 )
   { return FALSE;
   }
  nChannels = pBuf->info.i32NumChannels;
  if( nChannels<1 )
   { return FALSE;
   }

  dwBufIdx = (DWORD)pBuf->m_i64SampleCountOut & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);

  nSamplesMax = AudioMsgBuf_GetUsedBufferSpace_nSamples( pBuf );
  if( nSamplesToSend > nSamplesMax )
   {  nSamplesToSend = nSamplesMax;
   }
  // How many samples can be stuffed into T_AudioMsgBuffer.bTxBuffer[] ?
  nSamplesMax = sizeof( pBuf->bTxBuffer )
              / (  (iDataTypeForTransmission/8) // -> 1, 2, or 4 bytes per 'number'
                 * nChannels ); // number of channels PER SAMPLE POINT
  if( nSamplesToSend > nSamplesMax )
   {  nSamplesToSend = nSamplesMax;
   }


  // Read the samples from fltSampleBuffer[AUDIO_MSG_BUF_MAX_SAMPLE_POINTS][AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE]
  // and convert/scale them into the format used for 'transmission' :
  switch( iDataTypeForTransmission )
   { case 8:  // transmit the data as 8-bit unsigned integers, scaled into 0..255
        pBuf->info.iDataType_Bits = 8;
        pu8Dst = (unsigned char*)pBuf->bTxBuffer;
        for(iSample=0; iSample<nSamplesToSend; ++iSample)
         { pfltSrc = pBuf->fltSampleBuffer[dwBufIdx];
           for(iChannel=0; iChannel<nChannels; ++iChannel )
            { f = 127.0 * (1.0 + *pfltSrc++);
              if( f>255.0 ) f=255.0;
              if( f< 0.0  ) f= 0.0 ;
              *pu8Dst++ = (unsigned char)f;
            }
           dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
         }
        break;
     case 16: // transmit the data as 16-bit signed integers, scaled into +/-32k
        pBuf->info.iDataType_Bits = 16;
        pi16Dst = (short int*)pBuf->bTxBuffer;
        for(iSample=0; iSample<nSamplesToSend; ++iSample)
         { pfltSrc = pBuf->fltSampleBuffer[dwBufIdx];
           for(iChannel=0; iChannel<nChannels; ++iChannel )
            { f = 32767.0 * *pfltSrc++;
              if( f>32767.0 ) f=32767.0;
              if( f<-32767.0) f=-32767.0;  // Note: -32768 is AVOIDED
              *pi16Dst++ = (SHORT)f;
            }
           dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
         }
        break;
     case 32: // transmit the data as 32-bit *FLOATING POINTs*, range -1.0 .. +1.0
        pBuf->info.iDataType_Bits = 32;
        pfltDst = (float*)pBuf->bTxBuffer;
        for(iSample=0; iSample<nSamplesToSend; ++iSample)
         { pfltSrc = pBuf->fltSampleBuffer[dwBufIdx];
           for(iChannel=0; iChannel<nChannels; ++iChannel )
            { *pfltDst++ = *pfltSrc++;
            }
           dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
         }
        break;
   } // end switch( iDataTypeForTransmission )

  // Update the 'info' field, which will be sent along with the samples in bTxBuffer[]:
  pBuf->info.i32NrSamplePointsInBlock = nSamplesToSend;

  // Count the total number of output SAMPLE POINTS, too .
  // This is important for AudioMsgBuf_GetTimestamps();
  // and actually drains the 'read' samples from the FIFO :
  pBuf->m_i64SampleCountOut += nSamplesToSend; // LSBits used as FIFO TAIL INDEX !


  return nSamplesToSend;

} // end AudioMsgBuf_ReadSamplesIntoTxBuffer()


//---------------------------------------------------------------------------
// "Write" samples into the buffer's internal FIFO, in different flavours ..
//---------------------------------------------------------------------------

/***************************************************************************/
long AudioMsgBuf_WriteSamplesFromSeparateChannelBlocks( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToWrite, T_Float *pfltChannelA, T_Float *pfltChannelB )
  // Used by Spectrum Lab to put samples into an audio-message-buffer ("output").
{
  long i,nSamplesWritten;
  DWORD dwBufIdx;

  if( pBuf==NULL )
   { return 0;
   }

  dwBufIdx = (DWORD)pBuf->m_i64SampleCountIn & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);

  nSamplesWritten = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf );
  if( nSamplesWritten > nSamplePointsToWrite )
   {  nSamplesWritten = nSamplePointsToWrite;
   }

  // Write the samples into fltSampleBuffer[AUDIO_MSG_BUF_MAX_SAMPLE_POINTS][AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE] :
  if( (pfltChannelB==NULL) || pBuf->info.i32NumChannels<2 )
   { // only write ONE audio channel:
     for(i=0; i<nSamplesWritten; ++i)
      { pBuf->fltSampleBuffer[dwBufIdx][0] = pfltChannelA[i];
        dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
      }
   }
  else // pfltChannelB != NULL  *AND*  pBuf->info.i32NumChannels>=2 :
   { // write TWO audio channels (no further checking WITHIN the loop for speed reasons):
     for(i=0; i<nSamplesWritten; ++i)
      { pfltChannelA[i] = pBuf->fltSampleBuffer[dwBufIdx][0];
        pfltChannelB[i] = pBuf->fltSampleBuffer[dwBufIdx][1];
        dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
      }
   }
  pBuf->m_i64SampleCountIn += nSamplesWritten; // sample counter AND FIFO 'Head' index !

  return nSamplesWritten;
} // end AudioMsgBuf_WriteSamplesFromSeparateChannelBlocks()


/***************************************************************************/
long AudioMsgBuf_WriteSamplesInterleaved_Uns8( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToWrite, int iChannelsPerSample, unsigned char *pu8Source )
  // Used internally to put samples received via WM_COPYDATA (or similar)
  //   into an audio-message-buffer (later read as "input" by Spectrum Lab).
  // Here: source samples as an interleaved array of 8-bit UNSIGNED integers.
{
  long  nSamplesWritten = 0;
  DWORD dwBufIdx;
  int  iSample, iChannel;
  float *pfltDst;

  if( pBuf!=NULL && pu8Source!=NULL
     && iChannelsPerSample>=1
     && iChannelsPerSample<=AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE )
   {
     dwBufIdx = (DWORD)pBuf->m_i64SampleCountIn & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
     nSamplesWritten = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf );
     if( nSamplesWritten > nSamplePointsToWrite )
      {  nSamplesWritten = nSamplePointsToWrite;
      }

     // Write the samples into fltSampleBuffer[AUDIO_MSG_BUF_MAX_SAMPLE_POINTS][AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE] :
     for(iSample=0; iSample<nSamplesWritten; ++iSample)
      { pfltDst = pBuf->fltSampleBuffer[dwBufIdx];
        dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
        for(iChannel=0; iChannel<iChannelsPerSample; ++iChannel)
         { *pfltDst++ = (1/128.0) * ((float)(*pu8Source++) - 128.0);
         }
      }
     pBuf->m_i64SampleCountIn += nSamplesWritten; // sample counter AND FIFO 'Head' index !
   } // end if < parameters valid >

  return nSamplesWritten;
} // end AudioMsgBuf_WriteSamplesInterleaved_Uns8()

/***************************************************************************/
long AudioMsgBuf_WriteSamplesInterleaved_Int16( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToWrite, int iChannelsPerSample, short int *pi16Source )
  // Used internally to put samples received via WM_COPYDATA (or similar)
  //   into an audio-message-buffer (later read as "input" by Spectrum Lab).
  // Here: source samples as an interleaved array of 8-bit UNSIGNED integers.
{
  long  nSamplesWritten = 0;
  DWORD dwBufIdx;
  int  iSample, iChannel;
  float *pfltDst;

  if( pBuf!=NULL && pi16Source!=NULL
     && iChannelsPerSample>=1
     && iChannelsPerSample<=AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE )
   {
     dwBufIdx = (DWORD)pBuf->m_i64SampleCountIn & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
     nSamplesWritten = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf );
     if( nSamplesWritten > nSamplePointsToWrite )
      {  nSamplesWritten = nSamplePointsToWrite;
      }

     // Write the samples into fltSampleBuffer[AUDIO_MSG_BUF_MAX_SAMPLE_POINTS][AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE] :
     for(iSample=0; iSample<nSamplesWritten; ++iSample)
      { pfltDst = pBuf->fltSampleBuffer[dwBufIdx];
        dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
        for(iChannel=0; iChannel<iChannelsPerSample; ++iChannel)
         { *pfltDst++ = (1/32767.0) * (float)*pi16Source++;
         }
      }
     pBuf->m_i64SampleCountIn += nSamplesWritten; // sample counter AND FIFO 'Head' index !
   } // end if < parameters valid >

  return nSamplesWritten;
} // end AudioMsgBuf_WriteSamplesInterleaved_Int16()

/***************************************************************************/
long AudioMsgBuf_WriteSamplesInterleaved_Float32( T_AudioMsgBuffer *pBuf,
           long nSamplePointsToWrite, int iChannelsPerSample, float *pfltSource )
  // Used internally to put samples received via WM_COPYDATA (or similar)
  //   into an audio-message-buffer (later read as "input" by Spectrum Lab).
  // Here: source samples as an interleaved array of 8-bit UNSIGNED integers.
{
  long  nSamplesWritten = 0;
  DWORD dwBufIdx;
  int  iSample, iChannel;
  float *pfltDst;

  if( pBuf!=NULL && pfltSource!=NULL
     && iChannelsPerSample>=1
     && iChannelsPerSample<=AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE )
   {
     dwBufIdx = (DWORD)pBuf->m_i64SampleCountIn & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
     nSamplesWritten = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf );
     if( nSamplesWritten > nSamplePointsToWrite )
      {  nSamplesWritten = nSamplePointsToWrite;
      }

     // Write the samples into fltSampleBuffer[AUDIO_MSG_BUF_MAX_SAMPLE_POINTS][AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE] :
     for(iSample=0; iSample<nSamplesWritten; ++iSample)
      { pfltDst = pBuf->fltSampleBuffer[dwBufIdx];
        dwBufIdx = (dwBufIdx+1) & (AUDIO_MSG_BUF_MAX_SAMPLE_POINTS-1);
        for(iChannel=0; iChannel<iChannelsPerSample; ++iChannel)
         { *pfltDst++ = *pfltSource++;
         }
      }
     pBuf->m_i64SampleCountIn += nSamplesWritten; // sample counter AND FIFO 'Head' index !
   } // end if < parameters valid >

  return nSamplesWritten;
} // end AudioMsgBuf_WriteSamplesInterleaved_Float32()



/***************************************************************************/
long AudioMsg_SendQuery_WMSG(
        HWND hwndMyWindow,   // [in] "my" window handle
        HWND hwndHisWindow,  // [in] "his" window handle
        long i32FunctionCode, // [in] C_AUDIOMSG_FUNCTION_GET_FREE_BUFFER_SPACE, etc
        long i32Parameter,    // [in] optional 32-bit INTEGER parameter
        double dblFloatParam, // [in] optional 64-bit floating point parameter
        char *psz80StringParam) // [in] optional string parameter, up to 80 characters, may be NULL
  //
  // Sends a 'query' to the remote 'audio message receiver', using WM_COPYDATA.
  // The result depends on the function code, with one exception:
  //  0 = AUDIOMSG_RESULT_CANT_SEND means "cannot send the message"
  //      (this is dictated by the windows API, result from SendMessage )
{
 LRESULT lResult;
 COPYDATASTRUCT cds;
 T_AudioMsgQuery query;

  if( hwndMyWindow==NULL || hwndHisWindow==NULL)
      return AUDIOMSG_RESULT_CANT_SEND;
  if( hwndMyWindow==hwndHisWindow )
      return AUDIOMSG_RESULT_CANT_SEND;  // too dangerous to send a WM_COPYDATA message to "myself" !
  // Fill out all(!) fields in the T_AudioMsgQuery structure:
  memset( &query, 0, sizeof(query) );    // also clear the 'reserved' fields !
  query.i32SizeOfStruct = sizeof(query);
  query.i32Magic0x80008000 = AUDIOMSG_MAGIC_STRUCT_HEADER;
  query.i32StructCode = AUDIOMSG_STRUCT_CODE_AudioMsgQuery;
  query.i32FunctionCode = i32FunctionCode; // e.g. C_AUDIOMSG_FUNCTION_GET_FREE_BUFFER_SPACE
  query.i32IntegerParam = i32Parameter;    // optional 32-bit INTEGER parameter
  query.dblFloatParam   = dblFloatParam;   // optional 64-bit floating point parameter
  if( psz80StringParam != NULL )            // optional string parameter ?
   { strncpy( query.sz80StringParam, psz80StringParam, 80 );
   }
  // Prepare the windows-specific "copydatastruct" :
  cds.dwData = ('A')+((DWORD)'U'>>8)+((DWORD)'S'>>16)+((DWORD)'D'>>24);
  // module="AU" + command="SD"(Stream Data)
  cds.cbData = sizeof(T_AudioMsgQuery); // number of bytes in data block
  cds.lpData = &query;  // pointer to data block : Here, we send a T_AudioMsgQuery structure.
  lResult = SendMessage( // "SEND", not "POST"  ! (send may block the caller - see notes)
              (HWND)hwndHisWindow,  // HWND hWnd = handle of destination window
              WM_COPYDATA,          // UINT Msg  = message to send
              (WPARAM)hwndMyWindow, // HANDLE OF SENDING WINDOW
              (LPARAM)&cds ); // 2nd  msg parameter = pointer to COPYDATASTRUCT
     // Notes:
     //  - WM_COPYDATA can only be SENT, not POSTED !
     //  - If the destination window is this application itself,
     //    the SendMessage()-function will not return while we wait
     //    somewhere without calling the message loop !
     //    (Keep this in mind if this function *seems to* crash !
     //  - if SendMessage() cannot send, it returns ZERO = AUDIOMSG_RESULT_CANT_SEND
  return lResult;
} // end AudioMsg_SendQuery_WMSG()

/***************************************************************************/
long AudioMsg_PollRemoteBufferSpace_WMSG( HWND hwndMyWindow, HWND hwndHisWindow )
  // Polls the remote 'audio message receiver' for his free buffer space.
  // Used by some applications to find out if Spectrum Lab (=the receiver)
  // is ready to accept enough data, BEFORE generating new data.
  // In contrast to AudioMsgBuf_SendWmCopydata(), THIS function
  // will NEVER block the caller .
  // Note: The return value is a number of *buffer entries*,
  //       NOT NECESSARILY a number of "audio sampling points"
  //      (divide the return value by the number of audio channels to get that)
  // Return = 0 = AUDIOMSG_RESULT_CANT_SEND means "cannot send ANYTHING" now .
  //
{
  return AudioMsg_SendQuery_WMSG( hwndMyWindow, hwndHisWindow,
           C_AUDIOMSG_FUNCTION_GET_FREE_BUFFER_SPACE, // long i32FunctionCode
           0 ,    // optional 32-bit INTEGER parameter, here: unused
           0.0,   // optional 64-bit floating point parameter, here: unused
           NULL); // optional string parameter, here: unused (NULL)

} // end AudioMsg_PollRemoteBufferSpace_WMSG()


/***************************************************************************/
long AudioMsg_PollRemoteSamplingRate_WMSG( HWND hwndMyWindow, HWND hwndHisWindow )
{
  return AudioMsg_SendQuery_WMSG( hwndMyWindow, hwndHisWindow,
           C_AUDIOMSG_FUNCTION_GET_SAMPLERATE, // long i32FunctionCode
           0 ,    // optional 32-bit INTEGER parameter, here: unused
           0.0,   // optional 64-bit floating point parameter, here: unused
           NULL); // optional string parameter, here: unused (NULL)
} // end AudioMsg_PollRemoteSamplingRate_WMSG()


/***************************************************************************/
long AudioMsg_PollRemoteNumChannels_WMSG( HWND hwndMyWindow, HWND hwndHisWindow )
{
  return AudioMsg_SendQuery_WMSG( hwndMyWindow, hwndHisWindow,
           C_AUDIOMSG_FUNCTION_GET_NUM_CHANNELS, // long i32FunctionCode
           0 ,    // optional 32-bit INTEGER parameter, here: unused
           0.0,   // optional 64-bit floating point parameter, here: unused
           NULL); // optional string parameter, here: unused (NULL)
} // end AudioMsg_PollRemoteNumChannels_WMSG()



/***************************************************************************/
long AudioMsgBuf_SendAudioViaWmCopydata( T_AudioMsgBuffer *pBuf,
        HWND hwndMyWindow,  // [in] the sender's window handle
        HWND hwndHisWindow, // [in] the recipient's window handle
        int iDataType_Bits) // [in] data type for transmission: 8 = 8 bit unsigned,
                            // 16=16 bit signed integer, 32= 32-bit floating point
  // Sends a block of audio data through a windows MESSAGE (actually WM_COPYDATA),
  //       using the T_AudioMsgInfoExt header before the array of samples.
  // Return value:
  //  The FREE NUMBER OF ENTRIES (single values) IN THE RECEIVER'S BUFFER !
  //  This allows the sender (here: the caller) to keep the receiver's buffer 'as full as possible'.
  //  The caller may decide how many values to send (and, possibly, WHEN to send the next block).
  // Note: The return value is a number of *buffer entries*,
  //       NOT NECESSARILY a number of "audio sampling points"
  //      (divide the return value by the number of audio channels to get that)
  // Return = 0 = AUDIOMSG_RESULT_CANT_SEND means
  //      "the receiver couldn't handle the data" .
{
 LRESULT lResult;
 COPYDATASTRUCT cds;
 long  nSamplesToSend;
 int   nBytesPerSamplePoint; // one 'sample point' may contain ONE OR MORE channels !

  if( pBuf==NULL || hwndMyWindow==NULL || hwndHisWindow==NULL)
      return AUDIOMSG_RESULT_CANT_SEND;
  if( hwndMyWindow==hwndHisWindow )
      return AUDIOMSG_RESULT_CANT_SEND;  // too dangerous to send a WM_COPYDATA message to "myself" !

  nBytesPerSamplePoint = (iDataType_Bits/8) * pBuf->info.i32NumChannels;
  if(nBytesPerSamplePoint<1)
   {  return AUDIOMSG_RESULT_CANT_SEND;  // not properly initialized ?
   }
  nSamplesToSend = AudioMsgBuf_GetUsedBufferSpace_nSamples(pBuf);
  if(nSamplesToSend > 32768/nBytesPerSamplePoint)  // keep size of the WM_COPYDATA message
   { nSamplesToSend = 32768/nBytesPerSamplePoint;  // below 64 kByte (32 kByte + header)
   }
  pBuf->info.i32NrSamplePointsInBlock = AudioMsgBuf_ReadSamplesIntoTxBuffer(pBuf, nSamplesToSend, iDataType_Bits );
  pBuf->info.iDataType_Bits = iDataType_Bits; // << part of the sent 'info header'
  cds.dwData = ('A')+((DWORD)'U'>>8)+((DWORD)'S'>>16)+((DWORD)'D'>>24);
  // module="AU" + command="SD"(Stream Data)
  cds.cbData = sizeof(T_AudioMsgInfoExt) // # of bytes in data block including info header ..
             + pBuf->info.i32NrSamplePointsInBlock * nBytesPerSamplePoint;
  cds.lpData = &pBuf->info;  // pointer to data block : we begin sending with the AUDIO INFO header,
                             // directly followed by the audio samples in T_AudioMsgBuffer .
                             // So the "receiver" will always know the most important parameters.
  lResult = SendMessage( // "SEND", not "POST"  ! (send may block the caller - see notes)
              (HWND)hwndHisWindow,  // HWND hWnd = handle of destination window
              WM_COPYDATA,          // UINT Msg  = message to send
              (WPARAM)hwndMyWindow, // HANDLE OF SENDING WINDOW
              (LPARAM)&cds ); // 2nd  msg parameter = pointer to COPYDATASTRUCT
     // Notes:
     //  - WM_COPYDATA can only be SENT, not POSTED !
     //  - If the destination window is this application itself,
     //    the SendMessage()-function will not return while we wait
     //    somewhere without calling the message loop !
     //    (Keep this in mind if AudioMsg_Send() *seems to* crash !
     //  - if SendMessage() cannot send, it returns ZERO = AUDIOMSG_RESULT_CANT_SEND
  return lResult;  // -> FREE SPACE (#samples) remaining at the remote end (!)
} // end AudioMsgBuf_SendWmCopydata()


/***************************************************************************/
long AudioMsgBuf_HandleQuery( T_AudioMsgBuffer *pBuf, T_AudioMsgQuery *pQuery )
  // Implements the 'handler' for received T_AudioMsgQuery-messages
  //  via WM_COPYDATA .  First implemented in Spectrum Lab .
{
  long i32Result = 0;
  if( pBuf!=NULL && pQuery!=NULL )
   { switch( pQuery->i32FunctionCode )
      { case C_AUDIOMSG_FUNCTION_GET_VERSION_NUMBER   :
           // The 'other side' wants to know our version number:
           i32Result = AUDIOMSG_VERSION_NUMBER; // (version of AUDIOMSG.C / AUDIOMSG.H)
           break;

        case C_AUDIOMSG_FUNCTION_GET_NUM_CHANNELS     :
           i32Result = pBuf->info.i32NumChannels;
           break;

        case C_AUDIOMSG_FUNCTION_GET_SAMPLERATE       :
           i32Result = (long)(pBuf->info.dblSampleRate + 0.5);
           break;

        case C_AUDIOMSG_FUNCTION_GET_USED_BUFFER_SPACE:
           // The 'other side' wants to the amount of USED buffer space on this end.
           i32Result = AudioMsgBuf_GetUsedBufferSpace_nSamples( pBuf );
           break;

        case C_AUDIOMSG_FUNCTION_GET_FREE_BUFFER_SPACE:
           // The 'other side' wants to the amount of free buffer space on this end.
           i32Result = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf );
           break;

        default:
           break;
      } // end switch( pQuery->i32FunctionCode )
   }
  return i32Result;
} // end AudioMsgBuf_HandleQuery()


#ifdef USE_WSOCK // use WSOCK.C to send audio via UDP, too ?
/***************************************************************************/
long AudioMsgBuf_SendAudio_UDP( T_AudioMsgBuffer *pBuf, SOCKET s,
                                char *pszDestIp, int iDestPort )
  // Sends a block of audio data through a UDP packet (User Datagram Protocol) .
  // 's' must be a valid UDP socket, created by the caller like this:
  //     s=socket(AF_INET,SOCK_DGRAM,0);
  //
  // Return value:
  //   0 = "couldn't send" or "the receiver couldn't handle the data" .
  //  >0 = "ok" (possibly the number of samples which can be accepted)
  //
  // Notes:
  //  - The return value is a number of *buffer entries*, NOT NECESSARILY
  //       a number of "audio sampling points" (divide by pBuf->info.i32NumChannels for that)
  //       Return=0 means "the receiver couldn't handle the data",
  //          or (here): "cannot send" (most likely due to network trouble) .
  //  - When using this function in a DevCpp-project, the linker complained
  //     about missing functions like 'htons@4', 'inet_addr@4', etc .
  //     Cure: In DevCpp, select "Project".."Project Options".."Parameters"(!)..
  //           "Add Library Or Object", then add "libwsock32.a" .
{
  struct sockaddr_in sa;
  int total_len, msg_len, bytes_sent, rc, optlen;
  unsigned int uiMaxMsgSize;

  if( pBuf==NULL )
      return AUDIOMSG_RESULT_CANT_SEND;

  // prepare the socket-"address"
  sa.sin_family=AF_INET;
  // ex: sa.sin_port=htons(iDestPort);     // destination port number (UDP)
  // Because BCB complained "conversion may lose significant digits" :
  sa.sin_port=htons((unsigned short int)iDestPort); // destination port number (UDP)
  // ex: sa.sin_addr=inet_addr(pszDestIp); // pszDestIp = something like "192.168.0.201";
  //     what a lousy example. Guess that guy never tested it himself. Better:
  sa.sin_addr.s_addr=inet_addr(pszDestIp); // pszDestIp = something like "192.168.0.201";

  // Begin sending with the AUDIO INFO header (pBuf->Info),
  //   followed directly by the audio samples in T_AudioMsgBuffer .
  // So the "receiver" will always know the most important parameters.
  total_len = sizeof(T_AudioMsgInfo) // number of bytes in data block (including info header)
      + sizeof(SHORT) * pBuf->info.i32NrSamplePointsInBlock * pBuf->info.i32NumChannels;

  // Isn't this block too long ? See documentation of socket services:
  // > For message-oriented sockets, care must be taken not to exceed
  // > the maximum packet size of the underlying subnets, which can be obtained
  // > by getting the value of socket option SO_MAX_MSG_SIZE
  optlen = sizeof(uiMaxMsgSize);
  rc = getsockopt(        s, // SOCKET s,     descriptor identifying a socket
                 SOL_SOCKET, // int level,
            SO_MAX_MSG_SIZE, // int optname,
                             // BCB4: SO_MAX_MSG_SIZE known,
                             // BCB6: SO_MAX_MSG_SIZE undefined (only in winsock2.h).
                             //  Bullshit, over and over.
       (char*)&uiMaxMsgSize, // char FAR* optval,  ( often result = 65507 )
                  &optlen ); // int FAR*  optlen
  (void)rc;  // "rc is assigned a value that is never used". A-ha. So what ?
  bytes_sent = 0;
  do // Repeat this loop if the block is too large for being sent in a single UDP packet:
   {
     // Limit the block size to what can REALLY be sent in a single message:
     msg_len = total_len;
     if( msg_len > (int)uiMaxMsgSize )
         msg_len = uiMaxMsgSize;

     // Try to send the data to the receiver (no big hassle, no blocking, no threading here)
     // Caution: sendto may block the caller for a second if the UDP destination doesn't work ?!
     //         (but still the result code was == msg_len in that case)
     rc = sendto(        s, // SOCKET s,     descriptor identifying a socket
         (void*)&pBuf->info,// char * buf,   buffer containing the data to be transmitted
                   msg_len, // int len,      length of the data in buf
                         0, // int flags,    the way in which the call is made
            (SOCKADDR*)&sa, // sockaddr *to, pointer to the address of the target socket
       sizeof(SOCKADDR_IN));// int tolen,    size of the address in 'to'.

     if(rc==SOCKET_ERROR)
      { ++pBuf->iTxErrors;
        if( pBuf->iTxErrors < 10 )
         {
           WSOCK_LogError( "Failed to send to UDP socket", WSAGetLastError() );
            // printf("Fehler: sendto, fehler code: %d\n",WSAGetLastError());
         }
        return 0;
      }
     else
      { bytes_sent += rc;
        // printf("%d Bytes gesendet!\n", rc);
      }

   }while( bytes_sent<total_len );


  return 1;
} // end AudioMsgBuf_SendAudio_UDP()
#endif // defined USE_WSOCK


/***************************************************************************/
void   AudioMsgBuf_SetNumberOfChannels( T_AudioMsgBuffer *pBuf, int iNumChannels )
{
  if( iNumChannels < 1 )
   {  iNumChannels = 1;
   }
  if( iNumChannels > AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE )
   {  iNumChannels = AUDIO_MSG_BUF_MAX_CHANNELS_PER_SAMPLE;
   }
  if( pBuf!=NULL )
   {
     if( pBuf->info.i32NumChannels != iNumChannels )
      { pBuf->info.i32NumChannels = iNumChannels;
        // At the moment, no other actions required here.
        // This may change in future, for example if the buffer
        // had been dynamically allocated.
      }
   }
} // end AudioMsgBuf_SetNumberOfChannels()

/***************************************************************************/
int    AudioMsgBuf_GetNumberOfChannels( T_AudioMsgBuffer *pBuf )
{ if( pBuf!=NULL )
   { return pBuf->info.i32NumChannels;
   }
  else
   { return 0;
   }
} // end AudioMsgBuf_GetNumberOfChannels()

/***************************************************************************/
void AudioMsgBuf_SetCurrentSamplingRate( T_AudioMsgBuffer *pBuf,
      double dblSampleRate )      // [in] precise sample rate of the next block
  //   which will be written into the audio-sample-buffer .
  // Call this shortly ***before*** entering new samples into the buffer.
{
  if( pBuf!=NULL )
   { if( dblSampleRate > 0.0 ) // avoid div-by-zero somewhere else
      { pBuf->info.dblSampleRate = dblSampleRate;
      }
   }
} // end AudioMsgBuf_SetCurrentSamplingRate()

/***************************************************************************/
double AudioMsgBuf_GetCurrentSamplingRate( T_AudioMsgBuffer *pBuf )
{ if( pBuf!=NULL )
   { return pBuf->info.dblSampleRate;
   }
  else
   { return 0.0;
   }
} // end AudioMsgBuf_GetCurrentSamplingRate()

/***************************************************************************/
void AudioMsgBuf_SetCurrentInputTimestamp( T_AudioMsgBuffer *pBuf,
      long  double dblUnixDateAndTime ) // [in] current date+timestamp in UNIX format
  // Sets the "time of recording" (absolute) for the next audio sample
  //   which will be written into the audio-sample-buffer .
  // Call this shortly ***before*** entering new samples into the buffer.
{
  int  qi;

  if( pBuf==NULL )
   { return;
   }

  // Overwrite the OLDEST of the timestamps in the tiny queue (lowest index).
  // Note that the READING THREAD will never need the "oldest" entry.
  // This cumbersome method is only required if the WRITER and READER
  // and interrupt each other in an unpredictable way (which may be the case).
  if( pBuf->tstamps[0].i64SampleCount < pBuf->tstamps[1].i64SampleCount )
   { qi = 0;  // overwrite pBuf->tstamps[0] because it's the OLDER entry
   }
  else
   { qi = 1;  // overwrite pBuf->tstamps[1] ..
   }
  pBuf->tstamps[qi].valid = FALSE;  // mark this entry as INVALID
  pBuf->tstamps[qi].ldblTimestamp = dblUnixDateAndTime; // set new timestamp
  pBuf->tstamps[qi].i64SampleCount= pBuf->m_i64SampleCountIn;
  pBuf->tstamps[qi].valid = TRUE;   // make this entry VALID (last step!)

} // end AudioMsgBuf_SetCurrentInputTimestamp()


//---------------------------------------------------------------------------
BOOL AudioMsgBuf_GetOutputTimestamp( T_AudioMsgBuffer *pBuf,
      long double *pldblUnixDateAndTime) // [out] current date+timestamp in UNIX format, 80 bit float !
  // Returns the time of recording  for the next audio sample
  //   which will be retrieved from the buffer (related to output) .
  // Call this function shortly ***before*** reading output samples,
  //   because the result depends on the OUTPUT SAMPLE COUNTER (!)
  // Return value:  TRUE when successful, FALSE otherwise .
{
  int  trial, qi=-1;
  long double ts;
  LONGLONG sc;
  double sr/*samplerate*/, offs;

  if( pBuf==NULL )
   { return FALSE;
   }

  // Because the READER may be interrupted by the WRITER: Try this twice..
  for( trial=0; trial<=1; ++trial )
   { // Use the most recent timestamp info (which must be VALID, too);
     if(   (pBuf->tstamps[0].i64SampleCount > pBuf->tstamps[1].i64SampleCount )
        && (pBuf->tstamps[0].valid) )
      { qi = 0;  // use Buf->tstamps[0] because it's the NEWEST
      }
     else if (pBuf->tstamps[1].valid )
      { qi = 1;  // use pBuf->tstamps[1] (because it's the newest, or the other is currently invalid)
      }
     else        // none of the two entries is valid at the moment ?!
      { qi = -1;
      }
     if( qi>=0 )
      { ts = pBuf->tstamps[qi].ldblTimestamp;  // timestamp [Unix seconds]
        sc = pBuf->tstamps[qi].i64SampleCount; // sample counter and buffer index
    //  sr = pBuf->tstamps[qi].dblSampleRate;  // sample rate [Hz]
        sr = pBuf->info.dblSampleRate;         // sample rate [Hz]
        if( !pBuf->tstamps[qi].valid )
         { qi = -1; // bad luck; interrupted -> try again (one more time)
         }
      }
   }
  if( (qi>=0) && (sr>0) )
   { // Calculate the time offset (in seconds) for the timestamp
     //  of the next sample which will be read from the buffer (i.e. OUTPUT).
     //  Because the buffer's tail index (m_i64SampleCountOut) usually lags
     //  sc, the offset is usally negative .
     offs = (double)(pBuf->m_i64SampleCountOut - sc) / sr;
     if( pldblUnixDateAndTime )
      { *pldblUnixDateAndTime = ts + offs;
      }
     return TRUE;
   }
  else
   { return FALSE; // << set breakpoint here !
   }

} // end AudioMsgBuf_GetOutputTimestamp()


//---------------------------------------------------------------------------
long AudioMsgBuf_HandleWmCopydata( T_AudioMsgBuffer *pBuf, COPYDATASTRUCT *pCDS )
  // Handler for the WM_COPYDATA message with audio data in Spectrum Lab .
{
  int  nWaitingLoops;
  long i32FreeBufEntries;
  long i32Result = 0;
  int  iBytesPerSamplePoint; // .. with multiple channels per "point" (!)
  long i32NSamplesRcvd = 0;
  BYTE *pbSrc = (BYTE*)pCDS->lpData;     // pointer to datablock with header
  T_AudioMsgInfo *pAudioMsgInfo=(T_AudioMsgInfo*)pbSrc;
  if( pAudioMsgInfo->i32Magic0x80008000 == AUDIOMSG_MAGIC_STRUCT_HEADER )
   { // It's one of the messages defined in AUDIOMSG.H, but which one ?
     switch(pAudioMsgInfo->i32StructCode)
      { case AUDIOMSG_STRUCT_CODE_AudioMsgInfo:
           // It's the old (small) T_AudioMsgInfo structure,
           // immediately followed by another block of samples.
            { // Save the 'short' T_AudioMsgInfo for later:
              AudioMsgInfo_to_AudioMsgInfoExt( pAudioMsgInfo/*source*/, &pBuf->info/*dest*/ );
              iBytesPerSamplePoint = (pBuf->info.iDataType_Bits/8) * pBuf->info.i32NumChannels;
              pbSrc += sizeof(T_AudioMsgInfo);
              if( iBytesPerSamplePoint > 0 )
               { i32NSamplesRcvd = (pCDS->cbData - sizeof(T_AudioMsgInfo)) / iBytesPerSamplePoint;
               }
              else
               { i32NSamplesRcvd = 0;
               }
              // Because there's no timestamp in T_AudioMsgInfo,
              // simply INCREMENT the timestamp in T_AudioMsgInfoExt
              // by the number of samples divided by the sample rate:
              if(  i32NSamplesRcvd>0 && pBuf->info.dblSampleRate>0
                && pBuf->info.i32NumChannels>0 )
               { pBuf->info.ldblUnixDateAndTime +=
                    (long double)i32NSamplesRcvd /
                     ( pBuf->info.dblSampleRate * pBuf->info.i32NumChannels);
               }
            } break;
        case AUDIOMSG_STRUCT_CODE_AudioShortHeader:
           // It's just a T_AudioShortHeader structure,
           //  also leading a block of samples.
            { // Copy the available info from the 'short' audio message header:
              AudioShortHeader_to_AudioMsgInfoExt( (T_AudioShortHeader*)pbSrc/*source*/, &pBuf->info/*dest*/ );
              iBytesPerSamplePoint = (pBuf->info.iDataType_Bits/8) * pBuf->info.i32NumChannels;
              pbSrc += sizeof(T_AudioShortHeader);
              if( iBytesPerSamplePoint > 0 )
               { i32NSamplesRcvd = (pCDS->cbData - sizeof(T_AudioShortHeader)) / iBytesPerSamplePoint;
               }
              else
               { i32NSamplesRcvd = 0;
               }
              // Because there's no timestamp in the 'short' audio header,
              // simply INCREMENT the timestamp in T_AudioMsgInfoExt
              // by the number of samples divided by the sample rate.
              // Without this, the spectrogram display in SL may stop, because
              // SL relies on the TIMESTAMPS, not just the "number of samples".
              if(  i32NSamplesRcvd>0 && pBuf->info.dblSampleRate>0
                 && pBuf->info.i32NumChannels>0 )
               { pBuf->info.ldblUnixDateAndTime +=
                    (long double)i32NSamplesRcvd /
                    ( pBuf->info.dblSampleRate * pBuf->info.i32NumChannels);
               }
             } break;
        case AUDIOMSG_STRUCT_CODE_AudioMsgInfoExt:
           // It's the larger T_AudioMsgInfoExt structure,
           //  which also leads a block of samples,
           //  but (unlike the older AudioMsgInfo)
           //  contains precise TIMESTAMPS and other info.
             { T_AudioMsgInfoExt *pAudioMsgInfoExt=(T_AudioMsgInfoExt*)pbSrc;
               AudioMsgBuf_SetCurrentInputTimestamp( pBuf,
                             pAudioMsgInfoExt->ldblUnixDateAndTime );
               pBuf->info.dblSampleRate  = pAudioMsgInfoExt->dblSampleRate;
               pBuf->info.iDataType_Bits = pAudioMsgInfoExt->iDataType_Bits;
               pBuf->info.i32NumChannels = pAudioMsgInfoExt->i32NumChannels;
               pBuf->info.i32NrSamplePointsInBlock= pAudioMsgInfoExt->i32NrSamplePointsInBlock;
               pBuf->info.i32TotalNumSamplePoints = pAudioMsgInfoExt->i32TotalNumSamplePoints;
               iBytesPerSamplePoint = (pBuf->info.iDataType_Bits/8) * pBuf->info.i32NumChannels;
               pbSrc += sizeof(T_AudioMsgInfoExt);
               if( iBytesPerSamplePoint > 0 )
                { i32NSamplesRcvd = (pCDS->cbData - sizeof(T_AudioMsgInfoExt)) / iBytesPerSamplePoint;
                  // The 'calculated' number of samples (from the size of the WM_COPYDATA block)
                  // should be the same as indicated in the block header:
                  if( i32NSamplesRcvd != pBuf->info.i32NrSamplePointsInBlock )
                   { i32NSamplesRcvd = i32NSamplesRcvd; // << set breakpoint here
                   }
                }
               else
                { i32NSamplesRcvd = 0;
                }
             } break;
        case AUDIOMSG_STRUCT_CODE_AudioMsgQuery : // received a T_AudioMsgQuery
             { i32NSamplesRcvd = 0;  // this message does NOT carry audio data !
               i32Result = AudioMsgBuf_HandleQuery( pBuf, (T_AudioMsgQuery*)pbSrc );
             } break; // end case AUDIOMSG_STRUCT_CODE_AudioMsgQuery
      } // end switch(pAudioMsgInfo->i32StructCode)
   } // end if < valid AUDIO MESSAGE INFO struct >

  if( i32NSamplesRcvd > 0 ) // only if NOT a T_AudioMsgQuery..
   {
     i32FreeBufEntries = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf );
     nWaitingLoops = 10;  // max waiting time ~20 * 50 milliseconds
     while( (i32NSamplesRcvd >= i32FreeBufEntries )
        // && (pBuf->info.i32StreamOptions & AUDIOMSG_OPT_BLOCK_SENDER)
           && (nWaitingLoops-- > 0 )  )
      { // Too few free entries in the buffer.. so block the sender a few milliseconds
        // (hopefully the buffer can accept more samples then).
        // CRUDE, BUT WORKS OK FOR THE WINAMP-TO-SPECLAB PLUGIN !
        Sleep(50);
        i32FreeBufEntries = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf );
      }
     if( i32NSamplesRcvd>0 && i32NSamplesRcvd<=i32FreeBufEntries )
      { // SL's input buffer can handle this chunk of data now
        // -> copy the received audio samples into the circular FIFO;
        //    and -if necessary- convert the data type into the internal format.
        switch( pBuf->info.iDataType_Bits )
         { case 8:
              AudioMsgBuf_WriteSamplesInterleaved_Uns8( pBuf,
                 i32NSamplesRcvd, pBuf->info.i32NumChannels, (unsigned char *)pbSrc );
              break;
           case 16:
              AudioMsgBuf_WriteSamplesInterleaved_Int16( pBuf,
                 i32NSamplesRcvd, pBuf->info.i32NumChannels, (short int *)pbSrc );
              break;
           case 32:
              AudioMsgBuf_WriteSamplesInterleaved_Float32( pBuf,
                 i32NSamplesRcvd, pBuf->info.i32NumChannels, (float *)pbSrc );
              break;
          }
        // What to return ?
        // For simplicity, return the FREE NUMBER OF ENTRIES IN THE BUFFER !
        // This allows the sender to keep our buffer 'as full as possible' .
        // 'He' may decide how many values to send (and, possibly, WHEN to send the next block).
        i32Result = AudioMsgBuf_GetFreeBufferSpace_nSamples( pBuf );
      }
     else // timed out waiting for a free buffer...
      {
        i32Result = 0; // <<< set breakpoint here !
      }
   } // end if( i32NSamplesRcvd > 0 )
  return i32Result;
} // end AudioMsgBuf_HandleWmCopydata()

/* EOF < audiomsg.c > */
