/*========================================================================*/
/*                                                                        */
/* YHF_MultiLang.cpp  =  Hilfsroutinen fuer die Internationalisierung     */
/*                   einer BORLAND-C++Builder-Applikation .               */
/*  - verzichtet auf Borland's fehlerbehafteten Resourcen-DLL-Experten    */
/*  - verwendet stattdessem einen einfachen(?)                            */
/*      "tabellengesttzen bersetzer",                                   */
/*      um Anzeige- und Meldungstexte mehrsprachig halten zu koennen .    */
/*                                                                        */
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/*                                                                        */
/* YHF_MultiLang.cpp =  mother's little helper for internationalization   */
/*                   of a windoze application written in C++Builder .     */
/*  - does NOT require Borland's bugged Resource-DLL-Wizard               */
/*  - uses a simple(?) table-based "translator",                          */
/*      to hold two or more language alternatives .                       */
/*                                                                        */
/*                                                                        */
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
/*                                                                        */
/*  Written by Wolfgang Buescher (DL4YHF) .                               */
/*  Use of this sourcecode for commercial purposes strictly forbidden !   */
/*                                                                        */
/* Latest changes:   (YYYY-MM-DD, Author)                                 */
/*  2005-09-11, WoBu:   Less spaces before language code in combo list .  */
/*  2005-07-10, WoBu:   Now showing the LINE NUMBER in error messages .   */
/*  2005-06-28, WoBu:   Fixed a bug in YHF_SetLanguage() which caused     */
/*                      ignoring the flag fMayTranslateForms sometimes.   */
/*  2004-11-21, WoBu:   Added the option to load translations             */
/*                      from SEVERAL files, now in "translations" folder. */
/*                      First used for the French translation of WinPic,  */
/*                      thanks to Alain Pierre (french.txt) .             */
/*  2004-09-13, WoBu:   Now also translating a TPanel's CAPTION .         */
/*  2004-06-04, WoBu:   Checked the possibility for multi-byte character  */
/*                      support - to find it almost impossible ,          */
/*         because the standard library (sprintf) doesn't support it too. */
/*========================================================================*/

//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include <vcl.h>
#include <io.h>
#include <fcntl.h>
#include <buttons.hpp>   // TBitBtn
#include <ComCtrls.hpp>  // TTabSheet
#include <CheckLst.hpp>  // TCheckListBox
#include <IniFiles.hpp>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <dir.h>         // uses findfirst, findnext to find all translation files


#pragma hdrstop

#include "QFile.h"          // DL4YHF's "quick file routines" required since 2004-11
#include "YHF_MultiLang.h"  // non-project-specific header for multi-language support
#include "translation.h"    // project specific: how many languages, etc ?


//---------------------------------------------------------------------------
//   Data Types
//---------------------------------------------------------------------------
typedef struct
{ char scISO639FromLang[4]; // ISO639-1 code of the source language, should be "en" = english .
  char scISO639ToLang[4];   // ISO639-1 code of the translation, like "de","fr","es","it","gr"..
                            // scISO639ToLang[0] = '\0' marks the end of the list .
  char sz40LanguageNameForDisplay[42]; // like "Deutsch","English","Francais" (read from file)
  char sz255TranslationFileName[256];  // Name of the translation file, found by YHF_FindTranslationFiles()
} T_TranslationFileInfo;  // -> YHF_TranslationFileInfo[YHF_MAX_TRANSLATIONS], filled in YHF_FindTranslationFiles()



//---------------------------------------------------------------------------
//   Variables (global, say "yucc" if you feel better then ...)
//---------------------------------------------------------------------------
#define YHF_MAX_TRANSLATIONS 16  /* a bit optimistic, 16 different translations ?! */
T_TranslationFileInfo YHF_TranslationFileInfo[YHF_MAX_TRANSLATIONS]; // result from
int  YHF_iCountTranslationsFromFiles = 0;
char YHF_scISO639_TranslateFrom[4] = "\0"; // ISO639-1 code of previous language used in all forms
char YHF_scISO639_TranslateTo  [4] = "en"; // ISO639-1 code of current language
int  YHF_iLanguageTestMode = 0;  // 0=normal,  1=highlight translated controls
char YHF_sz255UntranslatedString[256];
int  YHF_iTranslationOk = 0;     // additional result from YHF_TranslateString()
int  YHF_iCountTranslationErrors=0;
#define YHF_MAX_PREV_ERRORS 64
char * YHF_ML_pszPrevTranslationErrors[YHF_MAX_PREV_ERRORS];  // to avoid showing the same error twice
WORD   YHF_ML_wPrevErrorIndex = 0;
char * YHF_pszzLoadedTable = NULL; // translation table loaded in ..SetLanguage()
char * YHF_pszzTranslationTable=(char*)TranslationTable[0]; // pointer to language-dependent translation table


  // Format of the translation table
  // -----------------------------------
  // It is planned to load these string from an ASCII file one day,
  //       to be compatible with future extensions, EVERY LINE contains
  //       its own language, using ISO 639-1(!) 2-letter language codes .
  //  Some examples of ISO 639-1 language codes :
  //   en:  english   (may be abbreviated as e: here)
  //   de:  german    (may be abbreviated as d: here)
  //   fr:  french    (may be abbreviated as f: here)
  //   it:  italian   (may be abbreviated as i: here)
  //   es:  spanish   (may be abbreviated as s: here)
  //   da:  danish
  //   nl:  dutch; flemish
  //   pt:  portuguese
  //   sv:  swedish
  //
  //  If a line begins with ':' , it contains the reference string
  //  which is used to find the right table entry. It MUST NOT
  //  be modified by the translator. The language used for the reference
  //  is usually ENGLISH (e:) so it's not necessary to have an english
  //  translation coded in lines beginning with "e:" or "en:" !
  //
  // Note: the english "master" MUST NOT BE MODIFIED by the translator,
  //       because EXACTLY the same string is somewhere in the C-code
  //       or one of the windows (called "forms" in Borland-C++Builder).
  //
  // The strings contain special characters, which must be handled
  // carefully to avoid malfunction of the program. Some characters are:
  //  '&'  as PREFIX mark the shortcut (letter) in menus, like "E&xit".
  //       If possible, use the same hotkey letters in your translation
  //       to avoid dupes in a menu !
  //  '%'  often indicates a format string, which is a placeholder for
  //       a number or a text. Be careful not to change the sequence after this.
  //       If there is more than one percent character in a string,
  //       be careful to keep the same parameter sequence ! Eample:
  //        ":Problem Nr. %d : found=%s,  read=%s",
  //        "d:Problem Nr. %d: gefunden=%s gelesen=%s",
  //          ( %d is a decimal number, %s is a string parameter )




//---------------------------------------------------------------------------
//  Subroutines
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void  YHF_ShowTranslationError(
                char *pszUntranslatedString,
                char *pszFoundInModule,
                char *pszFoundInComponent )
{

  if(YHF_iCountTranslationErrors==0)
   { APPL_ShowMsg( 0, 127/*level:IMPORTANT*/,
       "------ UNTRANSLATED STRINGS FOUND (FromLang=%s, ToLang=%s) ------",
       YHF_scISO639_TranslateFrom, YHF_scISO639_TranslateTo );
   }
  ++YHF_iCountTranslationErrors;

  // Format the untranslated string so it can be easily copied into the
  // translation table (like in TRANSLATION.C) .
  // A valuable help for internationalizing the whole project .
  if( pszFoundInModule )
   {
    if( pszFoundInComponent )
     {   // "module" (=Borland-FORM)   and    "component" specified :
      APPL_ShowMsg(0,  127/*level:IMPORTANT*/, "  \"%s\",  // in %s.%s",
           pszUntranslatedString, pszFoundInModule, pszFoundInComponent );
     }
    else // "module" specified but not the "component" where the error occurred :
     {
      APPL_ShowMsg(0,  127, "  \"%s\",  // in %s",
           pszUntranslatedString, pszFoundInModule );
     }
   }
  else // no "module" specified..
   {
    if( pszFoundInComponent )
     { APPL_ShowMsg(0,  127, "  \"%s\",  // %s",
                pszUntranslatedString, pszFoundInComponent );
       // used for messages like > "blabla" // missing translation <
     }
    else
     { APPL_ShowMsg(0,  127, "  \"%s\",  //",
                pszUntranslatedString );
     }
   }

} // end YHF_TranslationError()


//---------------------------------------------------------------------------
void YHF_Short2LongLangId( char cShortId, char *pcId1, char *pcId2 )
{
  switch( cShortId )
   {
    case 'e':  *pcId1='e';  *pcId2='n';  break;
    case 'd':  *pcId1='d';  *pcId2='e';  break;
    case 'f':  *pcId1='f';  *pcId2='r';  break;
    case 'i':  *pcId1='i';  *pcId2='t';  break;
    case 's':  *pcId1='e';  *pcId2='s';  break;
    default:   *pcId1='?';  *pcId1='?';  break;
   }
} // end YHF_Short2LongLangId()

//---------------------------------------------------------------------------
char *YHF_ML_SkipEndOfLineAndTerminateString( char *psz )
  // Skips all characters until the end of a string,
  //  and terminates the string with a null character ('\0') .
  //  Used to load CR/NL-separated string from textfiles .
  // Note: The "string" may be terminated by  CR/NL or '\0' character !
{ while(1)
   { if( (BYTE)*psz>=32)
       ++psz;
     else  // some control character or END-OF-STRING :
      { if( *psz=='\0')  // it is already ZERO-TERMINATED string (ONE or TWO zero-bytes)
         { ++psz;
           if(*psz=='\0') ++psz;     // SKIP one or two zero-bytes but not more !
           return psz;
         }
        if(psz[0]=='\r' && psz[1]=='\n') // it was a CR/NL-terminated TEXT LINE
         { psz[0]=psz[1]='\0';      // .. now it's a double-zero-terminated STRING
          return psz+2;
         }
        if(psz[0]=='\n' && psz[1]=='\r') // it was a stupid editor with NL/CR-termination
         { psz[0]=psz[1]='\0';      // .. now it's a double-zero-terminated STRING also
          return psz+2;
         }
        else
          ++psz;         // none of these special combinations, continue
      }
   }
} // end YHF_ML_SkipEndOfLineAndTerminateString()


//---------------------------------------------------------------------------
char *YHF_FindTranslationReference( char *pszzTranslationTable, char *pszReference )
{ // Searches for a string REFERENCE in the translation table.
  // Used, for example, to find the lines after pszReference="<languages>", etc.
  //
  char  *psz = pszzTranslationTable; // ex: = YHF_pszzTranslationTable;
  char  *pszFound;
  char  *pszOld;

  // Repeat search loop until translation found or end of table.
  // The end of the table is marked by at least three zero-bytes.
  do
   {
     pszOld = psz;   // to detect 'pointer movement' during analysis

     // Skip garbage up to the next REFERENCE string, indicated by a colon after a single ZERO-byte:
     if(psz[0]==':')
      { // looks like a 'REFERENCE' string, compare it..
        ++psz;  // skip the leading colon
        pszFound = psz;
        psz = YHF_ML_SkipEndOfLineAndTerminateString( psz );
        // Is this exactly the string we're looking for ?
         if( strcmp( pszFound, pszReference )==0 )
          { // found the translation, return immediately !
            return pszFound;
          } // end if( strcmp... )
      }
     else
     if(psz[0]=='\0' && psz[1]=='\0' && psz[2]=='0')
      { return NULL; // end of the translation table, nothing found
      }
     else
      { psz = YHF_ML_SkipEndOfLineAndTerminateString( psz );
      }
   }while ( psz>pszOld );   // emergency break if no 'pointer movement' was detected
  return NULL;  // reference name not found !
} // end YHF_FindTranslationReference()

//---------------------------------------------------------------------------
char *YHF_GetNextTranslationEntry( char *pszPrevious )
{
  if(! pszPrevious ) return NULL;

  return YHF_ML_SkipEndOfLineAndTerminateString( pszPrevious );

} // end YHF_GetNextTranslationEntry()

//---------------------------------------------------------------------------
char *YHF_TranslateStringInternal(
          char * pcISO639_SrcLanguage,  // en, de, fr, it, es, ... (source language)
          char * pcISO639_DstLanguage,  // en, de, fr, it, es, ... (destination language)
          const char * pszString2Translate )
{ // Translates a string FROM the specified language into the CURRENT language.
  // If the string-identifer is not found, the identifier itself is returned,
  // because in that case it's most likely that no "translation" is avaliable.
  // In language test mode (YHF_iLanguageTestMode>0),
  //    the string is copied to YHF_sz255UntranslatedString if it looks
  //    like something which 'should be translatable'.
  char  *psz = YHF_pszzTranslationTable;
  char  *pszOld;
  char  *pszLang1;
  char  *pszLang2;
  char  cId1,    cId2;
  char  cSrcId1, cSrcId2;
  char  cDstId1, cDstId2;
  BOOL  fEndOfTable = FALSE;
  int   iLineNr=1;  // required for error messages only !
  static int iPrevErrorLine=-1;


  YHF_iTranslationOk = 1;                 // by default: "translation was ok"

  if(YHF_iLanguageTestMode)
   { YHF_sz255UntranslatedString[0]='\0';
   }

  if( pcISO639_SrcLanguage[0] )  // source is NOT the REFERENCE language:
   { cSrcId1 = pcISO639_SrcLanguage[0];
     cSrcId2 = pcISO639_SrcLanguage[1];
   }
  else  // source = REFERENCE language (which is english) :
   { cSrcId1 = 'e';
     cSrcId2 = 'n';
   }
  cDstId1 = pcISO639_DstLanguage[0];  // destination language must always be specified
  cDstId2 = pcISO639_DstLanguage[1];


  if(  cDstId1==cSrcId1 && cDstId2==cSrcId2 )
    return (char*)pszString2Translate;   // no need to translate
  if(pszString2Translate[0]=='_')
    ++pszString2Translate;               // not able to translate, skip "error"-marker
  if(strlen(pszString2Translate)<=2)
    return (char*)pszString2Translate;   // not worth to translate


  // Repeat search loop until translation found or end of table.
  // The end of the table is marked by at least three zero-bytes.
  do
   {
     pszOld = psz;   // to detect 'pointer movement' during analysis

     // Skip garbage up to the next REFERENCE string, indicated by a colon after a single ZERO-byte:
     if(psz[0]!=':')
      { if(psz[0]=='\0' && psz[1]=='\0' && psz[2]=='0')
         {fEndOfTable=TRUE; // end of the translation table !
           break;
         }
        else
        if(psz[0]==';')   // line has been commented out; skip it !
         { psz = YHF_ML_SkipEndOfLineAndTerminateString( psz );
           ++iLineNr;
           break;
         }
        if( (BYTE)psz[0]<32 && psz[1]==':')
         {            // one garbage character, followed by REF string -> ok
           ++psz;     // skip ONE garbage character
         }
        else
        if( (BYTE)psz[0]<32 && (BYTE)psz[1]<32 && psz[2]==':')
         {            // TWO garbage characters (ex: CR/NL), followed by REF string -> also ok
           psz+=2;    // skip TWO garbage characters
           ++iLineNr;
         }
      }



     pszLang1 = NULL;  // "old" language string
     pszLang2 = NULL;  // "translation" also unknown

     // Look at the next character, which should be
     //  ':' as an indicator for the REFERENCE ID
     if(*psz==':')    // found another REFERENCE string,
      { // which marks the begin of a block of translations with the same meaning.
        // Consider the reference string as "english", just in case we are
        // translating 'back' from some other language..
        ++psz;
        if( cSrcId1=='e' &&  cSrcId2=='n')
            pszLang1 = psz;    // use this as default for the source
        if( cDstId1=='e' &&  cDstId2=='n')
            pszLang2 = psz;    // use this as default for the destination
        psz = YHF_ML_SkipEndOfLineAndTerminateString( psz );
        ++iLineNr;

        // Skip through all translations of 'this' string to find the right
        // language(s) ... Note that the REFERENCE string may be involved as
        // SOURCE, as DESTINATION, or NOT AT ALL !
        while( 1 )     // while <more lines within the group>
         {
          cId1 = psz[0];   // Look at the next TWO characters,
          cId2 = psz[1];   //   they should be another LANGUAGE ID ...
          if(cId1==':')    // Found the next reference indicator,
             break;        //   so leave this loop and 'compare'
          if(cId1==';')    // Line was 'commented out' -> skip it
           { psz = YHF_ML_SkipEndOfLineAndTerminateString( psz );
             ++iLineNr;
             continue;
           }
          if(  (BYTE)cId1>=32  && (BYTE)cId2>=32) // there must be AT LEAST TWO CHARACTERS in a line..
           {
            if(cId2==':')  // abbreviated language code ?
             { YHF_Short2LongLangId( cId1, &cId1, &cId2 );
               psz += 2;   // skip X: (1-letter language code + colon)
             }
            else               // no abbreviated language code ..
            if(psz[2]==':')    // two-letter language code (as it should be)
             {
               psz += 3;  // skip XX: (2-letter language code + colon)
             }
            else            // unknown format of this line; ignore it
             {              //   (but skip to the begin of the next line)
               if(iLineNr!=iPrevErrorLine)
                { char sz80Tmp[84];
                  strncpy(sz80Tmp,psz, 80);
                  sz80Tmp[80]=0;
                  APPL_ShowMsg( 0, 127/*level:IMPORTANT*/,
                    "TranslateString: Bad table entry, line %d, \"%s\" .",(int)iLineNr, sz80Tmp );
                  iPrevErrorLine = iLineNr; // don't show this error again  
                }
             }

            // Check if this is the SOURCE language :
            if( cId1==cSrcId1 && cId2==cSrcId2 )
             { // ok, this is the source-LANGUAGE
               //  (but not necessarily a matching STRING)
               pszLang1 = psz;
             }

            // Check if this is the DESTINATION language :
            if( cId1==cDstId1 && cId2==cDstId2 )
             { // ok, this is the destination-LANGUAGE
               //  (but not necessarily a matching STRING)
               pszLang2 = psz;
             }

            // Advance the 'source pointer' to the end of the STRING
            //     or the end of a CR/NL-terminated LINE from a text file:
            psz = YHF_ML_SkipEndOfLineAndTerminateString( psz );
            ++iLineNr;

           } // end if(cDstId1 && cDstId2)
          else // did NOT get two non-zero characters...
           {
             fEndOfTable=TRUE; // ... must be the end of the translation table !
             break;
           } // end else < not two characters >
         } // end while < more entries in a translation block >

       // Found the source- and destination language ?
       if( pszLang1 && pszLang2)
        {
         // Is this exactly the string we're looking for ?
         if( strcmp( pszLang1, pszString2Translate )==0 )
          { // Bingo, found the translation.  Don't waste time but return !
            return pszLang2;
          } // end if( strcmp... )
        }
      } // end if(psz[0]==':')
     else
      { // This is not the beginning of a language-translation-block :
        psz = YHF_ML_SkipEndOfLineAndTerminateString( psz );
        ++iLineNr;
      }
   }while (!fEndOfTable && (psz>pszOld) );   // emergency break if no 'pointer movement' was detected

  YHF_iTranslationOk = 0;

  if(YHF_iLanguageTestMode)
   {
     strncpy(YHF_sz255UntranslatedString, pszString2Translate, 255);
     YHF_sz255UntranslatedString[255]='\0'; // always terminate, sz255 has 256 byte
   }


  return (char*)pszString2Translate; // no string alternative found; use the "identifier" itself
} // end YHF_TranslateStringInternal()


//---------------------------------------------------------------------------
char *YHF_TranslateString( const char * pszString2Translate )
{ // Called via macro TS(x) = TranslateString  and  TE(x) = TranslateFromEnglish.
  // If the string-identifer is not found, the identifier itself is returned,
  // because in that case it's most likely that no "translation" is avaliable.
  // In language test mode (YHF_iLanguageTestMode>0),
  //    the string is copied to YHF_sz255UntranslatedString if it looks
  //    like something which 'should be translatable'.
  int  i;
  BOOL fShownErrorBefore = FALSE;
  char *pszResult=YHF_TranslateStringInternal(
                    "en",                     // source language (REFERENCE, english)
                    YHF_scISO639_TranslateTo, // destination language = variable
                    pszString2Translate );
  if(YHF_iLanguageTestMode)
   { if(YHF_sz255UntranslatedString[0])
      {  // Looks like the string could NOT be translated.
         // Has this error been shown before, or is it a "new" error ?
         // Consider this: shortly after program start, there may be
         // a lot of new translations,
         // but later, there may be "periodic" translations of the same string
         // over and over, which shall NOT fill the list of untranslated strings.
         // So : Avoid showing the same error twice !
         // Has this error been displayed before ?  Only look at the string's memory address !
         for(i=0; i<YHF_MAX_PREV_ERRORS && !fShownErrorBefore; ++i)
          { if( pszString2Translate == YHF_ML_pszPrevTranslationErrors[i] )
                fShownErrorBefore = TRUE;
          }

         if( !fShownErrorBefore )
          {
           YHF_ShowTranslationError( YHF_sz255UntranslatedString,
              NULL, "missing translation" );
           if( YHF_ML_wPrevErrorIndex >= YHF_MAX_PREV_ERRORS )
               YHF_ML_wPrevErrorIndex = 0;
           YHF_ML_pszPrevTranslationErrors[YHF_ML_wPrevErrorIndex] = (char*)pszString2Translate;
           YHF_ML_wPrevErrorIndex++;
          } // end if < not shown this translation error before >
      } // end if < untranslated string ? >
   }
  return pszResult;
} // end YHF_TranslateString()

#if(0)
//---------------------------------------------------------------------------
char *YHF_GetTranslatedText( const char **ppsz )
{
  const char *psz = NULL;

  if(ppsz!=NULL)
   {
    psz = ppsz[YHF_iTranslateTo];
    if(psz != NULL)
     { if(*psz == '\0') // don't accept an empty string !
          psz=NULL;
     }
    if(!psz)
     psz = ppsz[0];     // return untranslated string ("master language")
   }

  if(psz==NULL)         // nothing found ? no text available.
     psz = "n/a";

  return (char*)psz;

} // end YHF_GetTranslatedText()
#endif


//---------------------------------------------------------------------------
int  YHF_SetLanguage( char * pszISO639Language,  // "en","de","fr","it","es"...
                      char * pszTranslationFileName, // name of OPTIONAL translation file
                      BOOL fMayTranslateForms ) // TRUE = may translate the application's forms now
  // Sets the ISO 639-1 code of the new language,
  //       and -optionally- loads the new language from a file .
  //  Examples:
  //   en:  english
  //   de:  german (deutsch)
  //   fr:  french
  //   it:  italian
  //   es:  spanish (espania, not estonia ;)
  //   da:  danish
  //   nl:  dutch; flemish
  //   pt:  portuguese
  //   sv:  swedish (sverige)
  // Which of these languages can be used, depends on the
  //       LINKED or LOADED translation table; unknown at compile-time !
  // If the specified language can't be found in the built-in translation table,
  //       the routine tries to find the required file itself,
  //       using the info from YHF_FindTranslationFiles() .
{
  int  i;
  int  fh;
  long length;
  long src_index, dst_index;
  char c;
  char *pszTabEntry;
  T_TranslationFileInfo *pTranslationInfo;

  // If the "old" (present) language is NOT ENGLISH,
  //    translate everything back into english
  //    before switching to the new language .
  // This must be done BEFORE unloading the old language !
  // Note: at this step, YHF_scISO639_TranslateTo is the language which all forms
  //       HAVE BEEN TRANSLATED TO in a previous call .
  //    ( YHF_scISO639_TranslateTo[0]==0 && YHF_scISO639_TranslateTo[1]==0 )
  //    means "the forms have not been translated yet, should be in english )
  if(   ( YHF_scISO639_TranslateTo[0]!='\0'|| YHF_scISO639_TranslateTo[1]!='\0')
     && ( YHF_scISO639_TranslateTo[0]!='e' || YHF_scISO639_TranslateTo[1]!='n' ) )
   {
     if( fMayTranslateForms )   // switching back to the reference language
      {           // is only possible if all forms may be translated NOW ..
        YHF_scISO639_TranslateTo[0] = 'e';
        YHF_scISO639_TranslateTo[1] = 'n';
        YHF_TranslateAllForms();
        // Why is this necessary ?   Think...
        //  Example: First translate from german into english,
        //           then load the englisch->french translation table,
        //           then translate from english->french
        //           because translating "directly" from german into french
        //           will usually be impossible because it's another file !
        YHF_scISO639_TranslateFrom[0] = 'e';   // now all forms are in ENGLISH
        YHF_scISO639_TranslateFrom[1] = 'n';
      }
   } // end if < present language is NOT ENGLISH >

  for(i=0; i<YHF_MAX_PREV_ERRORS; ++i)       // forget any "old" missing translations..
   { YHF_ML_pszPrevTranslationErrors[i] = NULL; // to avoid showing the same error twice
   }
  YHF_ML_wPrevErrorIndex = 0;

  // for backward compatibility, accept some old dialing codes too..
  if( (strcmp(pszISO639Language,"01")==0 )
     || (pszISO639Language[0]=='\0')  )
   { YHF_scISO639_TranslateTo[0] = 'e';
     YHF_scISO639_TranslateTo[1] = 'n';
   }
  else if( strcmp(pszISO639Language,"33")==0 )
   { YHF_scISO639_TranslateTo[0] = 'f';
     YHF_scISO639_TranslateTo[1] = 'r';
   }
  else if( strcmp(pszISO639Language,"49")==0 )
   { YHF_scISO639_TranslateTo[0] = 'd';
     YHF_scISO639_TranslateTo[1] = 'e';
   }
  else   // arrived here: hope it's a 2-digit language code a la ISO 639-1 !
   {
     YHF_scISO639_TranslateTo[0] = pszISO639Language[0];
     YHF_scISO639_TranslateTo[1] = pszISO639Language[1];
   }
  YHF_scISO639_TranslateTo[2] = '\0';    // for safety, remove trailing junk

  // If no translation file passed in argument #2 :  Auto-detect ?
  if(pszTranslationFileName==NULL || pszTranslationFileName[0]==0 )
   {
    // Check if the requested language can be found in YHF_TranslationFileInfo :
    for(i=0; i<YHF_MAX_TRANSLATIONS && YHF_TranslationFileInfo[i].scISO639ToLang[0]>32; ++i)
     { pTranslationInfo = &YHF_TranslationFileInfo[i];
       if(  YHF_scISO639_TranslateTo[0]==pTranslationInfo->scISO639ToLang[0]
         && YHF_scISO639_TranslateTo[1]==pTranslationInfo->scISO639ToLang[1]
         && pTranslationInfo->sz255TranslationFileName[0]>32 )
        { // bingo, there seems to be an extra file for this language -> USE IT !
          pszTranslationFileName = pTranslationInfo->sz255TranslationFileName;
          break;  // no need to search any further
        }
     } // end if(pszTabEntry)
   } // end if (pszTranslationFileName==0)


  // If a translation table HAS ALREADY BEEN LOADED, unload that table now, and free resources:
  if( YHF_pszzLoadedTable )   // UNLOAD the old table (because size of new table will be different)
   {
    if( YHF_pszzTranslationTable==YHF_pszzLoadedTable ) // back to the fixed table
        YHF_pszzTranslationTable=(char*)TranslationTable[0];
    free( YHF_pszzLoadedTable );    // first get rid of the "old" !
    YHF_pszzLoadedTable = NULL;
   }


  // OPTIONAL: If the name of a translation file is specified or automatically detected,
  //           try to load that file now :
  if( pszTranslationFileName )
   { // a translation file shall be loaded....
     fh=_rtl_open(pszTranslationFileName, O_RDONLY );
     if(fh>0)
      {
       length = lseek(fh, 0L, SEEK_END);  // get the file size
       lseek(fh, 0L, SEEK_SET);           // rewind to start
       if(length>10 && length < 256000)
        {
         YHF_pszzLoadedTable = (char*)malloc( length + 8 ); // allocate buffer
         if( YHF_pszzLoadedTable )
          {
           memset( YHF_pszzLoadedTable, 0, length+8); // provide "termination"
           if( _rtl_read( fh, YHF_pszzLoadedTable, length ) > 0 ) // load table
            {
             // Replace certain characters in the table.
             src_index=0; dst_index=0;
             while(src_index<length  && dst_index<length)
              { c = YHF_pszzLoadedTable[src_index];
                if( c=='\\' )
                 {  // ooops, there is such a nasty backslash sequence...
                    // replace it with the proper control character !
                   switch(YHF_pszzLoadedTable[src_index+1])
                    {
                     case 'r':   // CARRIAGE RETURN
                       c='\r';
                       src_index++; // skip one MORE char from source
                       break;
                     case 'n':   // NEW LINE
                       c='\n';
                       src_index++; // skip one MORE char from source
                       break;
                     case 't':   // TABULATOR
                       c='\t';
                       src_index++; // skip one MORE char from source
                       break;
                    }
                 }
                YHF_pszzLoadedTable[dst_index++] = c;
                // Note: dst_index must never exceed src_index  !
                ++src_index;
              } // end while(src_index<length)

             // If some characters have been removed in the above steps,
             //  the "ugly rest" must be cleared now:
             if(dst_index < length)
                memset( YHF_pszzLoadedTable+dst_index, 0, length-dst_index);

             YHF_pszzTranslationTable=YHF_pszzLoadedTable;  // use the table now
             // Note: the loaded table contains CR+NL terminations
             //       instead of ZERO-BYTES.
             //  YHF_TranslateStringInternal() will zero-terminate
             //       the translated strings before returning them !
            }
          } // end if < RAM buffer for translation file could be allocated >
        } // end if <reasonable file length>
       _rtl_close(fh);
       APPL_ShowMsg(0, 0, "Loaded translation file  \"%s\" .", pszTranslationFileName );
      } // end if <translation file could be opened>
     else
      {
       APPL_ShowMsg(0, 127, "Cannot open translation file  \"%s\" .",
                            pszTranslationFileName );
      }
   } // end if( pszTranslationFileName )


  if( fMayTranslateForms    // switching back to the reference language
   &&(  ( YHF_scISO639_TranslateTo[0] != YHF_scISO639_TranslateFrom[0] )
      ||( YHF_scISO639_TranslateTo[1] != YHF_scISO639_TranslateFrom[1] ) )
    )
   { // is only possible if all forms may be translated NOW ..
    YHF_TranslateAllForms();
   }

  return 1;
} // end YHF_SetLanguage()

//---------------------------------------------------------------------------
const char * YHF_GetLanguage( void )
  // Returns the ISO 639-1 code of the new language .
{
  return YHF_scISO639_TranslateTo;
}


//---------------------------------------------------------------------------
BOOL YHF_DumpTranslatorFile( char *pszDestFileName )
 // Dumps the currently used 'translation table' into a text file.
 // Used mainly for testing purposes, but ...
 //
 // Can be used to turn the built-in translation table (translation.c)
 //      into a simple text file which can be edited by the user,
 //      because the dump format is the same as the file
 //      which can be loaded with YHF_SetLanguage()  !
 // Since 2004-11-21, this routine can dump the BUILT-IN table
 //      as well as the currently LOADED TRANSLATION .
{
 int handle;
 char  sz4kTemp[4004];
 char  c;

 char  *psz = YHF_pszzTranslationTable;
 char  *psz2;
 char  *pszDest;
 if(!psz)
    return FALSE;
 if ((handle = _rtl_creat(pszDestFileName, 0)) < 0)
    return FALSE;

 while(1)
  {
    psz2 = YHF_ML_SkipEndOfLineAndTerminateString( psz );
    if(psz2==psz)
       break;
    // Copy & Replace some important control characters :
    pszDest=sz4kTemp;
    while( ((c=*psz++)!=0)  && (psz<psz2) && (pszDest<(sz4kTemp+4000)) )
     { if( (BYTE)c >=32 ) *pszDest++=c;
       else if( c=='\n' ) { *pszDest++='\\'; *pszDest++='n';  }
       else if( c=='\r' ) { *pszDest++='\\'; *pszDest++='r';  }
       else if( c=='\t' ) { *pszDest++='\\'; *pszDest++='t';  }
     }
    *pszDest++='\0';  // terminate the "C"-style destination string
    _rtl_write( handle, sz4kTemp, strlen(sz4kTemp) );
    _rtl_write( handle, "\r\n", 2  );
    psz=psz2;
    if(psz[0]=='\0')
      break;
  }

 _rtl_close(handle);
 return TRUE;

} // end YHF_DumpTranslatorFile()


//---------------------------------------------------------------------------
void YHF_CloseTranslator(void)
 // Cleans up before termination.
 // Frees dynamically allocated memory if required.
 // Must be called upon termination of the main application.
{
  if( YHF_pszzLoadedTable )   // UNLOAD the old table
   {
    YHF_pszzTranslationTable=(char*)TranslationTable[0];
    free( YHF_pszzLoadedTable );    // first get rid of the "old" !
    YHF_pszzLoadedTable = NULL;
   }
} // end YHF_CloseTranslator()


//---------------------------------------------------------------------------
BOOL YHF_FindTranslationFiles( char * pszTranslationFilePattern )
 // Searches for translation files in the specified directory,
 //   with the specified search pattern. Example:
 //   pszTranslationFilePattern = "C:\\CBproj\\WinPicPr\\translations\\*.txt"
 //
 // New since 2004-11-21 !
{
  BOOL found;
  T_QFile myQFile;
  struct ffblk ffblk;
  int done;
  T_TranslationFileInfo *pTranslationInfo;
  int iNumTranslationsFound = 0;
  int i;
  char sz255FullFileName[256];
  char sz255Temp[256];
  char *cp;

  // First clear the table of translations (avoid garbage in YHF_TranslationFileInfo):
  for(i=0; i<YHF_MAX_TRANSLATIONS; ++i)
   {  memset( &YHF_TranslationFileInfo[i], 0, sizeof(T_TranslationFileInfo) );
   }
  YHF_iCountTranslationsFromFiles = 0;


  done = findfirst( pszTranslationFilePattern, &ffblk, 0);
  while (!done  && (iNumTranslationsFound<YHF_MAX_TRANSLATIONS) )
   {
     // Found another translation file in the specified directory (+search pattern) .
     strncpy(sz255FullFileName, pszTranslationFilePattern, 255);
     if( (cp=strrchr(sz255FullFileName,'\\')) != NULL)
          cp[1]='\0';
     else sz255FullFileName[0]='\0';
     strncat(sz255FullFileName, ffblk.ff_name, 255);

     //  Open the file to see what's in it (there MAY be more than one language in it !) :
     if( QFile_Open( &myQFile, sz255FullFileName, O_RDONLY ) )
      { // Find the ":<languages>" reference section :
        found = FALSE;
        for(i=0; i<10 && !found; ++i)  // skip a possible "header" !
         { if( QFile_ReadLine( &myQFile, sz255Temp, 255 ) < 0 )
              break;
           else
            { if(strcmp(sz255Temp,":<languages>")==0)
               {  // found the language reference section for THIS file:
                 found=TRUE;
                 break;
               }
            }
         } // end for
        if( found )  // found the language reference section in the file :
         { char szSourceLanguage[4] = {0,0,0,0};
          for(i=0; i<10 && (iNumTranslationsFound<YHF_MAX_TRANSLATIONS); ++i)
           { pTranslationInfo = &YHF_TranslationFileInfo[iNumTranslationsFound];
             if( QFile_ReadLine( &myQFile, sz255Temp, 255 ) <= 3 )
                break;
             else if(sz255Temp[0]==':')   // end of the reference section
                break;
             else if( sz255Temp[1]==':' || sz255Temp[2]==':' )
              { // found the language reference section for THIS file. Example:
                // 1st line:  "en:English"  (means the "reference language" is english)
                // next line: "fr:Francais" (means "file contains translation into french")
                cp = sz255Temp;
                if( i==0 )  // first line -> "reference" language (source)
                 {
                   szSourceLanguage[0] = *cp++;
                   if(*cp!=':') szSourceLanguage[1] = *cp++;
                        else    szSourceLanguage[1] = 0;
                   if(*cp==':') ++cp;
                 }
                else // i>0, second line ->  "translated into" language (destination)
                 {
                   pTranslationInfo->scISO639FromLang[0] = szSourceLanguage[0];
                   pTranslationInfo->scISO639FromLang[1] = szSourceLanguage[1];
                   pTranslationInfo->scISO639ToLang[0] = *cp++;
                   if(*cp!=':') pTranslationInfo->scISO639ToLang[1] = *cp++;
                   if(*cp==':') ++cp;
                   strncpy( pTranslationInfo->sz40LanguageNameForDisplay, cp, 40 );
                   strncpy( pTranslationInfo->sz255TranslationFileName, sz255FullFileName, 255 );
                   ++iNumTranslationsFound;
                 }
              }
           } // end for < all lines in the language reference section of this file >
         } // end if( found )

        QFile_Close( & myQFile );  // close the language translation file again

      } // end if( QFile_Open(.. )

     done = findnext(&ffblk);
   } // end while (!done) ..

  YHF_iCountTranslationsFromFiles = iNumTranslationsFound;

  return (iNumTranslationsFound > 0);

} // end YHF_FindTranslationFiles()



//---------------------------------------------------------------------------
BOOL YHF_UpdateLanguageCombo( TComboBox *pCombo )
 // Fills a TComboBox (Borland VCL) with a list of all available languages
 // ( new since 2004-11-21: first look for all built-in translations,
 //               and then for all translations in "language-files" )
{
  int  i,j;
  BOOL found;
  char *cp;
  char *pszTabEntry;
  char *pszLangName;
  char szComboLine[256];
  T_TranslationFileInfo *pTranslationInfo;

  #define C_SPACE_PADDED_LENGTH 12  /* ex: 20, changed 2005-09-11 to make language codes VISIBLE */

  pCombo->Clear();  // empty the combo box at runtime...

  // First part of the list : BUILT-IN translations !
  pszTabEntry = YHF_FindTranslationReference( (char*)TranslationTable, "<languages>");
  if(pszTabEntry)
   { // ok, there seems to be a list of supported languages in the TABLE or FILE..

     while( (pszTabEntry=YHF_GetNextTranslationEntry(pszTabEntry)) != NULL)
      {
        if( *pszTabEntry<'a'  )
            break;  // reached the end of this list !
        pszLangName = strchr(pszTabEntry,':');
        if(pszLangName)
         { strncpy(szComboLine,pszLangName+1, 20);
           szComboLine[20]=0;
           cp = szComboLine+strlen(szComboLine);
           do{ *cp++=' '; *cp='\0';
             }while(strlen(szComboLine)<C_SPACE_PADDED_LENGTH);
           *cp++='(';
           *cp++=pszTabEntry[0];
           if(pszTabEntry[1]<='z')
              *cp++=pszTabEntry[1];
           *cp++=')';
           *cp='\0';
           pCombo->Items->Add( szComboLine );
         }
      }
   }

  // After the built-in translations, append the results from YHF_FindTranslationFiles() :
  for(i=0; i<YHF_iCountTranslationsFromFiles; ++i)
   {
     pTranslationInfo = &YHF_TranslationFileInfo[i];
     if( pTranslationInfo->scISO639ToLang[0] != '\0' )
      {
        strncpy(szComboLine,pTranslationInfo->sz40LanguageNameForDisplay, 20);
        szComboLine[20]=0;
        cp = szComboLine+strlen(szComboLine);
        do{ *cp++=' '; *cp='\0';
          }while(strlen(szComboLine)<C_SPACE_PADDED_LENGTH);
        *cp++='(';
        *cp++=pTranslationInfo->scISO639ToLang[0];
        if(pTranslationInfo->scISO639ToLang[0]<='z')
           *cp++=pTranslationInfo->scISO639ToLang[1];
        *cp++=')';
        *cp='\0';
        // only add this item to the list IF IT'S NOT ALREADY IN THERE !
        found = FALSE;
        for(j=0; j<pCombo->Items->Count; ++j)
         { if( pCombo->Items->Strings[j] == szComboLine )
            { found=TRUE; break;
            }
         }
        if( ! found )
         { pCombo->Items->Add( szComboLine );
         }
      }
   } // end for(i=0; i<YHF_iCountTranslationsFromFiles; ++i)


  if(pCombo->Items->Count < 1)
   { // couldn't find the <languages> list in the table.
     // Fill the combo box with a fixed list instead !
     pCombo->Items->Add( "English   (en)");
     pCombo->Items->Add( "Deutsch   (de)");
   }

  // Now select the current language in the newly filled combo :
  for(i=0; i<pCombo->Items->Count; ++i)
   {
     cp = strchr( pCombo->Items->Strings[i].c_str(), '(');
     if(cp)
      { if( strncmp(cp+1, YHF_scISO639_TranslateTo, 2) == 0)
         { pCombo->ItemIndex = i;
           return TRUE;
         }
      }
   }
  pCombo->ItemIndex = -1;
  return FALSE;
} // end YHF_UpdateLanguageCombo()


//---------------------------------------------------------------------------
void YHF_TranslateAllForms(void)
  // Translates all 'Forms' (windows) of the application into the language
  //  which has been previously been set with YHF_SetLanguage() .
  // USUALLY CALLED FROM YHF_SetLanguage() with fMayTranslateForms=TRUE !
{
 #define TSI(s) YHF_TranslateStringInternal(YHF_scISO639_TranslateFrom,YHF_scISO639_TranslateTo,s)
 int iForm,iComp,count;
 int iItemIndex;
 TForm *pForm;
 TComponent *pComp;
 BOOL fHasTooltip;
 AnsiString s;

 static char scISO639_OldLanguage[4] = "\0\0\0\0";  // "current" language OF ALL FORMS(!)


 YHF_iCountTranslationErrors=0;

 YHF_scISO639_TranslateFrom[0] = scISO639_OldLanguage[0];
 YHF_scISO639_TranslateFrom[1] = scISO639_OldLanguage[1];

 // Remember the "new" language, will need it if the user wants to switch back
 scISO639_OldLanguage[0] = YHF_scISO639_TranslateTo[0];
 scISO639_OldLanguage[1] = YHF_scISO639_TranslateTo[1];


  // Loop through all 'Forms' and all their 'Components', to find out which
  // of them need to be translated; using strings from a very own table.
  //  (to avoid the crashy Resource-DLL-Wizard which is too stupid to produce path names)
  // The "Tag" value of certain VCL components defines if translation is used
  // or not:
  //    (Tag & 8) set means: Translate ALSO THE STRINGS IN THE LIST
  //    (Tag & 16) means:    DO NOT translate me !
  count = Screen->FormCount;  // does this work ?
  for(iForm=0; iForm<count; ++iForm)
   {
     pForm = Screen->Forms[iForm];
     if((pForm->Tag & 16)==0)
      { pForm->Caption = TSI(pForm->Caption.c_str());
        if(YHF_iLanguageTestMode && !YHF_iTranslationOk)
         { pForm->Caption = "_"+pForm->Caption;
           YHF_ShowTranslationError( YHF_sz255UntranslatedString,
                                     pForm->Name.c_str(),
                                     NULL );
         }
      }

     for( iComp=0; (iComp<pForm->ComponentCount); iComp++ )
      {
       YHF_iTranslationOk = 1;

       pComp = pForm->Components[iComp];
       if((pComp->Tag & 16) == 0)
        { // Only if this VCL component MAY be translated :
          // Check the type of this component; most need special treatment..
         fHasTooltip = FALSE;

         /* Check if the component is a TLabel:   */
         if (pComp->ClassNameIs("TLabel") )
          {
           // cast the component to a TStringGrid pointer
           TLabel *label = (TLabel*)pComp;
           // Translate the label's CAPTION :
           label->Caption = TSI(label->Caption.c_str());
           if(YHF_iLanguageTestMode)
            { if(YHF_iTranslationOk)
                label->Color =  clGreen;
              else
                label->Color =  clRed;
            }
           else // no 'test mode'; mark missing translations with an underscore
           if(!YHF_iTranslationOk)
            { label->Caption = "_"+label->Caption;
            }
          } /* end if <component is a TLabel>       */
         else
         if (pComp->ClassNameIs("TPanel") )
          { TPanel *pnl = (TPanel *)pComp;
            pnl->Caption = TSI(pnl->Caption.c_str());
            if(YHF_iLanguageTestMode)
             { if(!YHF_iTranslationOk)
                 pnl->Font->Style = TFontStyles() << fsItalic;
             }
            else // no 'test mode'; mark missing translations with an underscore
            if(!YHF_iTranslationOk)
             { pnl->Caption = "_"+pnl->Caption;
             }
          } /* end if <component is a TButton>           */
         else
         if (pComp->ClassNameIs("TButton") )
          { TButton *btn = (TButton *)pComp;
            fHasTooltip = TRUE;
            btn->Caption = TSI(btn->Caption.c_str());
            if(YHF_iLanguageTestMode)
             { if(!YHF_iTranslationOk)
                 btn->Font->Style = TFontStyles() << fsItalic;
             }
            else // no 'test mode'; mark missing translations with an underscore
            if(!YHF_iTranslationOk)
             { btn->Caption = "_"+btn->Caption;
             }
          } /* end if <component is a TButton>           */
         else
         if (pComp->ClassNameIs("TBitBtn") )
          { TBitBtn *btn = (TBitBtn *)pComp;
            fHasTooltip = TRUE;
            btn->Caption = TSI(btn->Caption.c_str());
            if(YHF_iLanguageTestMode)
             { if(!YHF_iTranslationOk)
                 btn->Font->Style = TFontStyles() << fsItalic;
             }
            else // no 'test mode'; mark missing translations with an underscore
            if(!YHF_iTranslationOk)
             { btn->Caption = "_"+btn->Caption;
             }
          } /* end if <component is a TBitBtn>           */
         else
         if (pComp->ClassNameIs("TRadioButton") )
          { TRadioButton *btn = (TRadioButton *)pComp;
            fHasTooltip = TRUE;
            btn->Caption = TSI(btn->Caption.c_str());
            if(YHF_iLanguageTestMode)
             { if(!YHF_iTranslationOk)
                 btn->Font->Style = TFontStyles() << fsItalic;
             }
            else // no 'test mode'; mark missing translations with an underscore
            if(!YHF_iTranslationOk)
             { btn->Caption = "_"+btn->Caption;
             }
          } /* end if <component is a TRadioButton>     */
         else
         if (pComp->ClassNameIs("TSpeedButton") )
          { fHasTooltip = TRUE; // speed buttons have no text, but tooltips ("Hint")
          }
         else
         if (pComp->ClassNameIs("TTabSheet") )
          { TTabSheet *ts = (TTabSheet *)pComp;
            ts->Caption = TSI(ts->Caption.c_str());
            // if(YHF_iLanguageTestMode)
             { if(!YHF_iTranslationOk)
                ts->Caption = "_"+ts->Caption;
             }
          } /* end if <component is a TTabSheet>           */
         else
         if (pComp->ClassNameIs("TCheckBox") )
          { TCheckBox *chk = (TCheckBox *)pComp;
            chk->Caption = TSI(chk->Caption.c_str());
            if(YHF_iLanguageTestMode)
             { if(YHF_iTranslationOk)
                chk->Color =  clGreen;
               else
                chk->Color =  clRed;
             }
            else // no 'test mode'; mark missing translations with an underscore
            if(!YHF_iTranslationOk)
             { chk->Caption = "_"+chk->Caption;
             }
          } /* end if <component is a TTabSheet>           */
         else
         if (pComp->ClassNameIs("TMenuItem") )
          { // works for entries in a "TMainMenu" and "TPopupMenu" .
            TMenuItem *mi = (TMenuItem *)pComp;
            mi->Caption = TSI(mi->Caption.c_str());
            // if(YHF_iLanguageTestMode)
             { if(!YHF_iTranslationOk)
                { if(mi->Caption.c_str()[0]!='_')
                     mi->Caption =  "_"+mi->Caption;
                }
             }
          }
         else
         if (pComp->ClassNameIs("TGroupBox") )
          { TGroupBox *gb = (TGroupBox *)pComp;
            gb->Caption = TSI(gb->Caption.c_str());
            // if(YHF_iLanguageTestMode)
             { if(!YHF_iTranslationOk)
                { if(gb->Caption.c_str()[0]!='_')
                     gb->Caption =  "_"+gb->Caption;
                }
             }
          }
         else
         if (pComp->ClassNameIs("TListBox") )
          { TListBox *lb = (TListBox *)pComp;
            if(lb->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
             {
              for(iItemIndex=0; iItemIndex<lb->Items->Count; ++iItemIndex)
               { // translate all 'Items' (TStrings) in this listbox..
                s = lb->Items->Strings[iItemIndex];
                s = TSI(s.c_str());
                // if(YHF_iLanguageTestMode)
                if(!YHF_iTranslationOk)
                  { if(s.c_str()[0]!='_')
                       s =  s+"_";  // APPEND(!) the 'translation error marker' here !
                    // (why ? see LB_CompareOperators in NewEventDialog..
                    //  the first few characters are used to identify the selected entry)
                  }
                lb->Items->Strings[iItemIndex] = s;
               } // end for <all items>
             } // end if <component's tag says 'transl8 me'>
          } // end if <component is a TListBox>
         else
         if (pComp->ClassNameIs("TComboBox") )
          { TComboBox *cb = (TComboBox *)pComp;
            if(cb->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
             {
              for(iItemIndex=0; iItemIndex<cb->Items->Count; ++iItemIndex)
               { // translate all 'Items' (TStrings) in this listbox..
                s = cb->Items->Strings[iItemIndex];
                s = TSI(s.c_str());
                // if(YHF_iLanguageTestMode)
                if(!YHF_iTranslationOk)
                  { if(s.c_str()[0]!='_')
                       s =  s+"_";  // APPEND(!) the 'translation error marker' here !
                    // (why ? see LB_CompareOperators in NewEventDialog..
                    //  the first few characters are used to identify the selected entry)
                  }
                cb->Items->Strings[iItemIndex] = s;
               } // end for <all items>
             } // end if <component's tag says 'transl8 me'>
          } // end if <component is a TComboBox>
         else
         if (pComp->ClassNameIs("TRadioGroup") )
          { TRadioGroup *rg = (TRadioGroup *)pComp;
            if(rg->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
             {
              rg->Caption = TSI(rg->Caption.c_str());
              for(iItemIndex=0; iItemIndex<rg->Items->Count; ++iItemIndex)
               { // translate all 'Items' (TStrings) in this radio group..
                s = rg->Items->Strings[iItemIndex];
                s = TSI(s.c_str());
              // if(YHF_iLanguageTestMode)
                if(!YHF_iTranslationOk)
                { if(s.c_str()[0]!='_')
                       s =  s+"_";  // APPEND(!) the 'translation error marker' here !
                    // (why ? see LB_CompareOperators in NewEventDialog..
                    //  the first few characters are used to identify the selected entry)
                }
                rg->Items->Strings[iItemIndex] = s;
               } // end for <all items>
             } // end if <component's tag says 'transl8 me'>
          } // end if <component is a TRadioGroup>
         else
         if (pComp->ClassNameIs("TCheckListBox") )
          { TCheckListBox *cl = (TCheckListBox *)pComp;
            if(cl->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
             {
              for(iItemIndex=0; iItemIndex<cl->Items->Count; ++iItemIndex)
               { // translate all 'Items' (TStrings) in this checklist..
                s = cl->Items->Strings[iItemIndex];
                s = TSI(s.c_str());
                // if(YHF_iLanguageTestMode)
                if(!YHF_iTranslationOk)
                  { if(s.c_str()[0]!='_')
                       s =  s+"_";  // APPEND(!) the 'translation error marker' here !
                    // (why ? see LB_CompareOperators in NewEventDialog..
                    //  the first few characters are used to identify the selected entry)
                  }
                cl->Items->Strings[iItemIndex] = s;
               } // end for <all items>
             } // end if <component's tag says 'transl8 me'>
          } // end if <component is a TCheckListBox>

         if(  ( YHF_iLanguageTestMode>0 )
           && (!YHF_iTranslationOk )
           && ( YHF_sz255UntranslatedString[0]>0 ) )
          {
           // Format the untranslated string so it can be easily copied into the
           // translation table (like in TRANSLATION.C) .
           // A valuable help for internationalizing the whole project .
           YHF_ShowTranslationError(
                YHF_sz255UntranslatedString,
                pForm->Name.c_str() ,
                pComp->Name.c_str() );
          } // end if <show untranslatable string in global error list>

         /* Check if the component has a "hint" (tooltip string):   */
         if( fHasTooltip )
          {
           TControl *co = (TControl *)pComp;
           if( co->Hint != "" )
            { co->Hint = TSI(co->Hint.c_str());
              if(  ( YHF_iLanguageTestMode>0 ) && (!YHF_iTranslationOk )
                && ( YHF_sz255UntranslatedString[0]>0 ) )
               {
                 YHF_ShowTranslationError(
                   YHF_sz255UntranslatedString,
                   pForm->Name.c_str() ,
                   AnsiString(pComp->Name + ".Hint").c_str() );
               }
            }
          } // end if < component has a "hint" >


        } // end if <may translate this component, due to its 'Tag' value>
      } /* end for < all components of the form >      */
  } // end for<all forms of the application>

  if( (YHF_iLanguageTestMode>0) && (YHF_iCountTranslationErrors>0) )
   {
    APPL_ShowMsg(0,  127/*level:IMPORTANT*/,
      "Found %d untranslated strings in all forms.",(int)YHF_iCountTranslationErrors );
   }

} // end YHF_TranslateAllForms()




