//---------------------------------------------------------------------------
// YHF_Jpeg.c  =  a small wrapper for the Independent Jpeg Group's Code.
//
//  A result of WoBu's very first experiments to decompress JPEG files.
//
//  Using:
//     the IJG's JPEG library   Copyright (C) 1994-1997, Thomas G. Lane
// BUT NO LONGER USING LIBTIFF since the removal of LZW compression (what a dog)
//    neither PAINTLIB          Copyright (c) 1996-1998, Ulrich von Zadow
//        nor LIBTIFF           Copyright (c) 1988-1997, Sam Leffler etc
//
// Some trivial modifications were made in the IJPEG code
// to reduce the number of warnings detected by Borland's C++Builder V4 .
// (I did not want to turn the warnings off in the compiler options).
//
// To WoBu's big surprise, decoding a JPEG image worked immediately
// on the very first attempt.  Well done guys ... though there are a few
// 'design flaws' which required slight modifications to make it work STABLE
// under Windoze XP. (uninitialized properties of some classes, which lead
// to the removal of the PAINTLIB stuff).
//
//---------------------------------------------------------------------------


#include <vcl.h>
#include <stdio.h>

#pragma hdrstop

#ifdef _WINDOWS
 #include <windows.h>
#endif

#include "YHF_Jpeg.h"   // header for this module, to check the prototypes

/*
 * Include file for users of JPEG library.
 * You will need to have included system headers that define at least
 * the typedefs FILE and size_t before you can include jpeglib.h.
 * (stdio.h is sufficient on ANSI-conforming systems.)
 * You may also wish to include "jerror.h".
 */
extern "C"
{
 #include "jpeglib.h"

 #include "jerror.h"

 #define JPEG_INTERNAL_OPTIONS	/* cjpeg.c,djpeg.c need to see xxx_SUPPORTED */

 //  (without this, sometime RGB_RED, RGB_PIXELSIZE etc wouldn't be defined)

 #include "jconfig.h"

 #include "jmorecfg.h"

// #error dobini 

}

/*
 * <setjmp.h> is used for the optional error recovery mechanism shown in
 * the second part of the example.
 */

#include <setjmp.h>


char YHF_Jpeg_ErrorMessage[256] = "not called yet";



/******************** JPEG DECOMPRESSION SAMPLE INTERFACE *******************/

/* This half of the example shows how to read data from the JPEG decompressor.
 * It's a bit more refined than the above, in that we show:
 *   (a) how to modify the JPEG library's standard error-reporting behavior;
 *   (b) how to allocate workspace using the library's memory manager.
 *
 * Just to make this example a little different from the first one, we'll
 * assume that we do not intend to put the whole image into an in-memory
 * buffer, but to send it line-by-line someplace else.  We need a one-
 * scanline-high JSAMPLE array as a work buffer, and we will let the JPEG
 * memory manager allocate it for us.  This approach is actually quite useful
 * because we don't need to remember to deallocate the buffer separately: it
 * will go away automatically when the JPEG object is cleaned up.
 */


/*
 * ERROR HANDLING:
 *
 * The JPEG library's standard error handler (jerror.c) is divided into
 * several "methods" which you can override individually.  This lets you
 * adjust the behavior without duplicating a lot of code, which you might
 * have to update with each future release.
 *
 * Our example here shows how to override the "error_exit" method so that
 * control is returned to the library's caller when a fatal error occurs,
 * rather than calling exit() as the standard error_exit method does.
 *
 * We use C's setjmp/longjmp facility to return control.  This means that the
 * routine which calls the JPEG library must first execute a setjmp() call to
 * establish the return point.  We want the replacement error_exit to do a
 * longjmp().  But we need to make the setjmp buffer accessible to the
 * error_exit routine.  To do this, we make a private extension of the
 * standard JPEG error handler object.  (If we were using C++, we'd say we
 * were making a subclass of the regular error handler.)
 *
 * Here's the extended error handler struct:
 */

struct my_error_mgr
{
  struct jpeg_error_mgr pub;	/* "public" fields */
  jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

/*
 * Here's the routine that will replace the standard error_exit method:
 */
void my_error_exit(j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
  my_error_ptr myerr = (my_error_ptr) cinfo->err;

  /* Always display the message. */
  /* We could postpone this until after returning, if we chose. */
  (*cinfo->err->output_message) (cinfo);

  /* Return control to the setjmp point */
  longjmp(myerr->setjmp_buffer, 1);
}

/***************************************************************************/
BOOL YHF_ReadJpegFile(char * filename,
           Graphics::TBitmap *bmp)
/*
 * Routine for JPEG decompression.  We assume that the source file name
 * is passed in.
 * Return TRUE on success,   FALSE on error.
 * If there was an error, a description can be found in
 *  YHF_Jpeg_ErrorMessage .
 */
{
  /* This struct contains the JPEG decompression parameters and pointers to
   * working space (which is allocated as needed by the JPEG library).
   */
  struct jpeg_decompress_struct cinfo;
  /* We use our private extension JPEG error handler.
   * Note that this struct must live as long as the main JPEG parameter
   * struct, to avoid dangling-pointer problems.
   */
  struct my_error_mgr jerr;
  /* More stuff */
  FILE * infile;		/* source file */
  JSAMPARRAY buffer;		/* Output row buffer */
  int row_stride;		/* physical row width in output buffer */
  int x;
  int iDestBitsPerPixel    // 8, 16, 32 bits/pixel   for the target bitmap
     = GetDeviceCaps( bmp->Canvas->Handle, BITSPIXEL );


  /* In this example we want to open the input file before doing anything else,
   * so that the setjmp() error recovery below can assume the file is open.
   * VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
   * requires it in order to read binary files.
   */

  if ((infile = fopen(filename, "rb")) == NULL)
   {
    // fprintf(stderr, "can't open %s\n", filename);
    return FALSE;
   }

  /* Step 1: allocate and initialize JPEG decompression object */
  memset(&cinfo, 0, sizeof(cinfo) );  // will be done in "jpeg_stdio_src" anyway

  /* We set up the normal JPEG error routines, then override error_exit. */
  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit = my_error_exit;
  /* Establish the setjmp return context for my_error_exit to use. */
  if (setjmp(jerr.setjmp_buffer)) {
    /* If we get here, the JPEG code has signaled an error.
     * We need to clean up the JPEG object, close the input file, and return.
     */
    jpeg_destroy_decompress(&cinfo);
    fclose(infile);
    return FALSE;
  }
  /* Now we can initialize the JPEG decompression object. */
  jpeg_create_decompress(&cinfo);

  /* Step 2: specify data source (eg, a file) */
  jpeg_stdio_src(&cinfo, infile);

  /* Step 3: read file parameters with jpeg_read_header() */
  (void) jpeg_read_header(&cinfo, TRUE);
  /* We can ignore the return value from jpeg_read_header since
   *   (a) suspension is not possible with the stdio data source, and
   *   (b) we passed TRUE to reject a tables-only JPEG file as an error.
   * See libjpeg.doc for more info.
   */

  /* Step 4: set parameters for decompression */
  // Many of the defaults are already set by jpeg_read_header(), so we do nothing here.
  cinfo.dct_method = JDCT_FLOAT;  // best for a modern CPU.

  /* Step 5: Start decompressor */
  jpeg_start_decompress(&cinfo);
  /* We can ignore the return value since suspension is not possible.
   * After jpeg_start_decompress() we have the correct scaled
   * output image dimensions available, as well as the output colormap
   * if we asked for color quantization.
   */
  /* JSAMPLEs per row in output buffer */
  row_stride = cinfo.output_width * cinfo.output_components;
  /* Make a one-row-high sample array that will go away when done with image */
  buffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);
  bmp->Width  = cinfo.output_width;
  bmp->Height = cinfo.output_height;
  // added by WoBu: prepare a target TBitmap object ...

#if( ! USE_PIXEL_PROPERTY )
  // cannot modify the bloody "PixelFormat" here..
  bmp->PixelFormat = pf32bit; // seems IMPOSSIBLE to change this even for "offscreen" bitmaps ?!
#endif // (! USE_PIXEL_PROPERTY)

  /* Step 6: while (scan lines remain to be read) */
  /*           jpeg_read_scanlines(...); */

  /* Here we use the library's state variable cinfo.output_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   */
  while (cinfo.output_scanline < cinfo.output_height)
   {
    /* jpeg_read_scanlines expects an array of pointers to scanlines.
     * Here the array is only one element long, but you could ask for
     * more than one scanline at a time if that's more convenient.
     */
    jpeg_read_scanlines(&cinfo, buffer, 1);
    /* Assume put_scanline_someplace wants a pointer and sample count. */
    if(cinfo.output_scanline < bmp->Height)
     {
#if(USE_PIXEL_PROPERTY) // here the SLOWEST POSSIBLE WAY to paint into a bitmap, VCL style:
       for(x=0; x<cinfo.output_width && x<bmp->Width; ++x)
        {
          BYTE * pSrcPixel = buffer[0]+x*4/*RGB_PIXELSIZE*/;
          bmp->Canvas->Pixels[x][cinfo.output_scanline] = (TColor)
               (  ((DWORD) pSrcPixel[0])          // red
                | ((DWORD) pSrcPixel[1] << 8)     // green
                | ((DWORD) pSrcPixel[2] << 16) ); // blue
        }
#else  // the next better way to access the pixels in a VCL-style TBitmap:
       BYTE *pDestPixel;
       BYTE *pSrcPixel = buffer[0];
       // x = pBitmap->Width;
       x = bmp->Width;
       if(x>cinfo.output_width)
          x=cinfo.output_width;
       pDestPixel = (BYTE *)bmp->ScanLine[cinfo.output_scanline];
       if(pDestPixel)
        {
         switch(iDestBitsPerPixel)
          {
            case 32:
                while(x--)
                  { // Hi, this is an INTEL CPU !
                    // Borland's info on TColor tells you
                    //  > $000000FF is pure red (are we pascal users ?!)
                    // but we are playing with BYTES here.... strange order:
                    *pDestPixel++ = pSrcPixel[RGB_BLUE];
                    *pDestPixel++ = pSrcPixel[RGB_GREEN];
                    *pDestPixel++ = pSrcPixel[RGB_RED];
                    *pDestPixel++ = 0;  // is THIS the so-called "alpha" channel ???
                    pSrcPixel += RGB_PIXELSIZE; // most likely, RGB_PIXELSIZE=4
                  }
                break;
            default:
                // cannot handle this destination format !
                break;
          } // end switch(iDestBitsPerPixel)
        }
#endif // USE_PIXEL_PROPERTY

     } // end if < output_scanline fits into target bitmap >
   } // end while < scanlines from decoder >

  /* Step 7: Finish decompression */
  (void) jpeg_finish_decompress(&cinfo);
  /* We can ignore the return value since suspension is not possible
   * with the stdio data source.
   */

  /* Step 8: Release JPEG decompression object */

  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_decompress(&cinfo);

  /* After finish_decompress, we can close the input file.
   * Here we postpone it until after no more JPEG errors are possible,
   * so as to simplify the setjmp error logic above.  (Actually, I don't
   * think that jpeg_destroy can do an error exit, but why assume anything...)
   */
  fclose(infile);

  /* At this point you may want to check to see whether any corrupt-data
   * warnings occurred (test whether jerr.pub.num_warnings is nonzero).
   */

  /* And we're done! */
  return TRUE;
} // end ReadJpegFile()


/***************************************************************************/
BOOL YHF_WriteJpegFile(char * filename, int quality,  Graphics::TBitmap *bmp)
/*
 * Routine for JPEG compression.  We assume that the target file name,
 * a compression quality factor, and a pointer to the source bitmap
 * are passed in.
 * Return TRUE on success,   FALSE on error.
 * If there was an error, a description can be found in
 *  YHF_Jpeg_ErrorMessage .
 */
{
  /* This struct contains the JPEG compression parameters and pointers to
   * working space (which is allocated as needed by the JPEG library).
   * It is possible to have several such structures, representing multiple
   * compression/decompression processes, in existence at once.  We refer
   * to any one struct (and its associated working data) as a "JPEG object".
   */
  struct jpeg_compress_struct cinfo;
  /* This struct represents a JPEG error handler.  It is declared separately
   * because applications often want to supply a specialized error handler
   * (see the second half of this file for an example).  But here we just
   * take the easy way out and use the standard error handler, which will
   * print a message on stderr and call exit() if compression fails.
   * Note that this struct must live as long as the main JPEG parameter
   * struct, to avoid dangling-pointer problems.
   */
  struct my_error_mgr jerr;
  /* More stuff */
  FILE * outfile = NULL;        // target file
  JSAMPROW row_pointer[1];	// pointer to JSAMPLE row[s]
  int row_stride;		// physical row width in image buffer
  BYTE *pbLineBuffer;           // buffer to collect one line of pixels
  int  x;
  int iSourceBitsPerPixel    // 8, 16, 32 bits/pixel   from the source bitmap
     = GetDeviceCaps( bmp->Canvas->Handle, BITSPIXEL );


  /* Step 1: allocate and initialize JPEG compression object */

  /* We have to set up the error handler first, in case the initialization
   * step fails.  (Unlikely, but it could happen if you are out of memory.)
   * This routine fills in the contents of struct jerr, and returns jerr's
   * address which we place into the link field in cinfo.
   */

  /* We set up the normal JPEG error routines, then override error_exit. */
  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit = my_error_exit;
  /* Establish the setjmp return context for my_error_exit to use. */
  if (setjmp(jerr.setjmp_buffer)) {
    /* If we get here, the JPEG code has signaled an error.
     * We need to clean up the JPEG object, close the input file, and return.
     */
    jpeg_destroy_compress(&cinfo);
    if(outfile)
       fclose(outfile);
    return FALSE;
  }

  /* Now we can initialize the JPEG compression object. */
  jpeg_create_compress(&cinfo);

  /* Step 2: specify data destination (eg, a file) */
  /* Note: steps 2 and 3 can be done in either order. */

  /* Here we use the library-supplied code to send compressed data to a
   * stdio stream.  You can also write your own code to do something else.
   * VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
   * requires it in order to write binary files.
   */
  if ((outfile = fopen(filename, "wb")) == NULL)
   {
    //  fprintf(stderr, "can't open %s\n", filename);
    jpeg_destroy_compress(&cinfo);
    return FALSE;
   }
  jpeg_stdio_dest(&cinfo, outfile);

  /* Step 3: set parameters for compression */

  /* First we supply a description of the input image.
   * Four fields of the cinfo struct must be filled in:
   */
  cinfo.image_width = bmp->Width; /* image width and height, in pixels */
  cinfo.image_height = bmp->Height;
  cinfo.input_components = 4;	  /* # of color components per pixel */
                                  /* (for odd reasons, must use "4" for RGB) */
  cinfo.in_color_space = JCS_RGB; /* colorspace of input image       */
  /* Now use the library's routine to set default compression parameters.
   * (You must set at least cinfo.in_color_space before calling this,
   * since the defaults depend on the source color space.)
   */
  jpeg_set_defaults(&cinfo);
  /* Now you can set any non-default parameters you wish to.   */
  cinfo.dct_method = JDCT_FLOAT;  // best for a modern CPU. Nothing else supported ?!?
  jpeg_set_quality(&cinfo, quality, TRUE /* limit to baseline-JPEG values */);

  /* Step 4: Start compressor */

  /* TRUE ensures that we will write a complete interchange-JPEG file.
   * Pass TRUE unless you are very sure of what you're doing.
   */
  jpeg_start_compress(&cinfo, TRUE);

  /* Step 5: while (scan lines remain to be written) */
  /*           jpeg_write_scanlines(...); */

  /* Here we use the library's state variable cinfo.next_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   * To keep things simple, we pass one scanline per call; you can pass
   * more if you wish, though.
   */
  row_stride = cinfo.image_width * cinfo.input_components;
  pbLineBuffer = (BYTE*)malloc(row_stride);
  if(pbLineBuffer==NULL)
   { // chaos, very low memory..
     jpeg_destroy_compress(&cinfo);
     fclose(outfile);
     return FALSE;
   }
  while (cinfo.next_scanline < cinfo.image_height)
   {
    // collect one line of pixels in a buffer, organized like the JPEG
    // compressor wants them....
    BYTE *pDestPixel = pbLineBuffer;
    BYTE *pSrcPixel;
    x = bmp->Width;
    if(x>cinfo.image_width)
       x=cinfo.image_width;
    pSrcPixel = (BYTE *)bmp->ScanLine[cinfo.next_scanline];
    if(pSrcPixel)
      {
       switch(iSourceBitsPerPixel)
        {
         case 32:
              while(x--)
                  { // Hi, this is an INTEL CPU !
                    // Borland's info on TColor tells you
                    //  > $000000FF is pure red (are we pascal users ?!)
                    // but we are playing with BYTES here.... strange order:
                    pDestPixel[RGB_BLUE] = *pSrcPixel++; // 1st SOURCE byte: BLUE
                    pDestPixel[RGB_GREEN]= *pSrcPixel++; // 2nd SOURCE byte: GREEN
                    pDestPixel[RGB_RED]  = *pSrcPixel++; // 3rd SOURCE byte: RED
                    if(cinfo.input_components>3)
                       pDestPixel[3] = 0x00;  // unused component ?!
                    pSrcPixel++;  // skip the so-called "alpha" channel ???
                    pDestPixel += cinfo.input_components; // should be 3 comps
                  }
                break;
          default:
                // cannot handle this destination format !
                break;
        } // end switch(iSourceBitsPerPixel)
      } // end if <pointer to source bitmap's pixel data exists>


    // jpeg_write_scanlines expects an array of pointers to scanlines.
    // Here the array is only one element long, but you could pass
    // more than one scanline at a time if that's more convenient.
    row_pointer[0] = pbLineBuffer;
    (void) jpeg_write_scanlines(&cinfo, row_pointer, 1);
   } // end while (next_scanline ...)

  /* Step 6: Finish compression */
  jpeg_finish_compress(&cinfo);

  /* After finish_compress, we can close the output file */
  fclose(outfile);

  /* Step 7: release JPEG compression object                                */
  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_compress(&cinfo);


  // clean up
  if(pbLineBuffer)
     free(pbLineBuffer);
  /* And we're done! */
  return TRUE;
} // end WriteJpegFile()



/***************************************************************************/
BOOL SaveBmpAsFile(char *sFName, Graphics::TBitmap *bmp )
  // Saves a (VCL-)TBitmap as a file.
  // The following target formats are possible
  //  (detected by the file extension) :  JPG and BMP .
  // Return TRUE on success,   FALSE on error.
  // If there was an error, a description can be found in
  //  YHF_Jpeg_ErrorMessage .
{
 char *pszExtension = strrchr(sFName,'.');  // find the file name's extension.

 if(pszExtension == NULL)
   {
    sprintf(YHF_Jpeg_ErrorMessage, "Error: No extension in \"%s\"", sFName );
    return FALSE;
   }

 if(   (stricmp(pszExtension,".bmp")==0)  )
  {  // try to save the TBitmap as BMP file
     sprintf(YHF_Jpeg_ErrorMessage, "saved as Windoze bitmap");
     bmp->SaveToFile( sFName ); // no notification about success/failure :-(
     return TRUE;
  } // end if <save as BMP>


 if(   (stricmp(pszExtension,".jpg")==0)
    || (stricmp(pszExtension,".jpeg")==0) )
  {  // try to save the bitmap as JPEG file
    if( WriteJpegFile( sFName,
                          50, // quality
                        bmp)) // Graphics::TBitmap *bmp
     {
       sprintf(YHF_Jpeg_ErrorMessage, "saved as JPEG file");
       return TRUE;
     }
    else
     {
       sprintf(YHF_Jpeg_ErrorMessage, "Error saving JPEG file");
       return FALSE;
     }
  } // end if <save as JPEG>


 sprintf(YHF_Jpeg_ErrorMessage, "Error encoding image: Extension \"%s\" unknown !",
                         pszExtension );
 return FALSE;
} // end SaveBmpAsFile()


/***************************************************************************/
BOOL LoadBmpFromFile(char *sFName, Graphics::TBitmap *bmp )
  // Loads a (VCL-)TBitmap as a file.
  // The following target formats are possible
  //  (detected by the file extension) :  JPG and BMP .
  // Return TRUE on success,   FALSE on error.
  // If there was an error, a description can be found in
  //  YHF_Jpeg_ErrorMessage .
{
 char *pszExtension = strrchr(sFName,'.');  // find the file name's extension.

 if(pszExtension == NULL)
   {
    sprintf(YHF_Jpeg_ErrorMessage, "Error: No extension in \"%s\"", sFName );
    return FALSE;
   }

 if(   (stricmp(pszExtension,".bmp")==0)  )
  {  // try to load a BMP file into the TBitmap
     try
      {
        bmp->LoadFromFile( sFName ); // no notification about success/failure :-(
        sprintf(YHF_Jpeg_ErrorMessage, "loaded Windoze bitmap");
        return TRUE;
      }
     catch(...)
      {
        sprintf(YHF_Jpeg_ErrorMessage, "Error loading Windoze bitmap file");
        return FALSE;
      }
  } // end if <load from BMP>


 if(   (stricmp(pszExtension,".jpg")==0)
    || (stricmp(pszExtension,".jpeg")==0) )
  {  // try to save the bitmap as JPEG file
     // use the original IJPEG code, without PAINTLIB's bugged wrapping classes...
    if( ReadJpegFile( sFName,
                        bmp)) // Graphics::TBitmap *bmp
     {
       sprintf(YHF_Jpeg_ErrorMessage, "loaded JPEG file");
       return TRUE;
     }
    else
     {
       sprintf(YHF_Jpeg_ErrorMessage, "Error reading JPEG file");
       return FALSE;
     }
  } // end if <save as JPEG>


 sprintf(YHF_Jpeg_ErrorMessage, "Error encoding image: Extension \"%s\" unknown !",
                         pszExtension );
 return FALSE;
} // end LoadBmpFromFile()


// EOF


