/*------------------------------------------------------------------*/
/* File: C:\cbproj\YHF_Tools\YHF_Inet.c                             */
/*       (everything else is just a COPY)                           */
/*                                                                  */
/* DL4YHF's 'internet' utilities based on the socket services .     */
/* Initially used to read audio streams from remote servers .       */
/*       (c) 2011 by Wolfgang Buescher                              */
/*                                                                  */
/*                                                                  */
/* Autor: Wolfgang Buescher (DL4YHF)                                */
/*  Used in:                                                        */
/*    *  Spectrum Lab, to retrieve the contents of a remote *.M3U   */
/*    *  in SL's "OggVorbis"-DLL, to read Ogg/Vorbis streams        */
/*                                                                  */
/* Revision history (YYYY-MM-DD):                                   */
/*                                                                  */
/*  2011-12-26 :                                                    */
/*   Changed from the old (simple, but inefficient) polling-method  */
/*      to asynchronous sockets (with a window message sent when    */
/*      new data have arrived. NO MULTI-THREADING) . Details at     */
/*      http://tangentsoft.net/wskfaq/articles/io-strategies.html . */
/*      Old version saved as C:\cbproj\YHF_Tools\YHF_Inet_simple.c  */
/*                                                                  */
/*  2011-11-30 :                                                    */
/*  Created YHF_Inet.c as an extra layer between an application     */
/*      (CLIENT) to read files from remote internet servers.        */
/*      Contains an afwul lot of specialized 'string parsers'       */
/*      to separate URLs (or do they call it URIs these days?),     */
/*      for example, to ...                                         */
/*   - find out if a given "filename" is the name of a LOCAL FILE   */
/*      or an INTERNET RESOURCE (to pick the right API to access    */
/*      them, because unfortunately, under windoze, a SOCKET        */
/*      is a SOCKET and not a FILE HANDLE, and vice versa... )      */
/*   - parse strings like "http://67.207.143.181/vlf6.m3u"  ,       */
/*      to retrieve the PROTOCOL,  HOST NAME,  FILENAME ;           */
/*   - Simplify the task of opening a TCP/IP connection, and        */
/*      (on top of that) implement a simple HTTP CLIENT .           */
/*                                                                  */
/*------------------------------------------------------------------*/


// ex: #include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !



/* NO VCL IN THIS UNIT !!!!! (or other BORing stuff) */
#include <string.h>
#include <windows.h>
#include <stdarg.h>
#include <stdio.h>   // not really 'standard I/O' but vsprintf used here
#pragma hdrstop

#include "YHF_Inet.h"     // header for THIS module
#include "YHF_CmdLine.h"  // DL4YHF's command-line-parser utility
#include "Utility1.h"     // UTL_CheckAndSkipToken() + similar parsing stuff
#include "ErrorCodes.h"   // DL4YHF's error codes (also used here since 2011)



//------------ internal data types ------------------------------------------



//------------ Functions ----------------------------------------------------

//---------------------------------------------------------------------------
char * INET_WinsockErrorCodeToString( int iWSAErrorCode )
 // Turns error codes returned by WSAGetLastError() (!!!!) into text .
 //
{
  static char sz80OtherError[84];  // <<< avoid using this static var wherever possible
                                   //  (it may cause problems with multi-threading one day)
  switch( iWSAErrorCode )
   {
    case WSAEINTR          : return "Interrupted function call";
    case WSAEBADF          : return "Invalid socket descriptor";
    case WSAEACCES         : return "Permission denied";
    case WSAEFAULT         : return "Bad address";
    case WSAEINVAL         : return "Invalid argument";
    case WSAEMFILE         : return "Too many open files or sockets";
    case WSAEWOULDBLOCK    : return "Call of WINSOCK routine would block";
    case WSAEINPROGRESS    : return "Operation now in progress";
    case WSAEALREADY       : return "Operation already in progress";
    case WSAENOTSOCK       : return "Socket operation on non-socket";
    case WSAEDESTADDRREQ   : return "Destination address required";
    case WSAEMSGSIZE       : return "Message too long";
    case WSAEPROTOTYPE     : return "Protocol wrong type for socket";
    case WSAENOPROTOOPT    : return "Bad protocol option";
    case WSAEPROTONOSUPPORT: return "Socket type not supported";
    case WSAESOCKTNOSUPPORT: return "Socket not supported";
    case WSAEOPNOTSUPP     : return "Operation not supported";
    case WSAEPFNOSUPPORT   : return "Protocol family not supported";
    case WSAEAFNOSUPPORT   : return "Address family not supported by protocol family";
    case WSAEADDRINUSE     : return "Address already in use";
    case WSAEADDRNOTAVAIL  : return "Cannot assign requested address";
    case WSAENETDOWN       : return "Network is down";
    case WSAENETUNREACH    : return "Network is unreachable";
    case WSAENETRESET      : return "Network dropped connection on reset";
    case WSAECONNABORTED   : return "Software caused connection abort";
    case WSAECONNRESET     : return "Connection reset by peer";
    case WSAENOBUFS        : return "No buffer space available";
    case WSAEISCONN        : return "socket already connected";
    case WSAENOTCONN       : return "Socket is not connected";
    case WSAESHUTDOWN      : return "Cannot send after socket shutdown";
    case WSAETOOMANYREFS   : return "too many refs";
    case WSAETIMEDOUT      : return "Connection timed out";
    case WSAECONNREFUSED   : return "Connection refused";
    case WSAELOOP          : return "loop detected";
    case WSAENAMETOOLONG   : return "name too long";
    case WSAEHOSTDOWN      : return "Host is down";
    case WSAEHOSTUNREACH   : return "No route to host";
    case WSAENOTEMPTY      : return "not empty";
    case WSAEPROCLIM       : return "Too many processes";
    case WSAEUSERS         : return "users";
    case WSAEDQUOT         : return "quot";
    case WSAESTALE         : return "stale";
    case WSAEREMOTE        : return "remote";
    case WSAEDISCON        : return "Graceful shutdown in progress";
    case WSASYSNOTREADY    : return "Network subsystem is unavailable";
    case WSAVERNOTSUPPORTED: return "WINSOCK.DLL version out of range";
    case WSANOTINITIALISED : return "WSAStartup not yet performed";
    case WSAHOST_NOT_FOUND : return "Host not found";
    case WSATRY_AGAIN      : return "Non-authoritative host not found";
    case WSANO_RECOVERY    : return "non-recoverable error";
    case WSANO_DATA        : return "valid name but no data";
    default                :
         sprintf(sz80OtherError, "Other WINSOCK error, code %d",(int)iWSAErrorCode);
         return sz80OtherError;
   } // end switch( iWSAErrorCode )

} // end INET_WinsockErrorCodeToString()

/****************************************************************************/
int INET_GuessAudioFileTypeFromName(char * pszFilename,
                             int iDefaultAudioFileType) // [in] to solve "*.dat"-ambiguity !)
{
  char *cp = strrchr(pszFilename,'.');
  char *cpFirstColon = strchr(pszFilename,':');
  if(cp!=NULL)
   {
     if(stricmp(cp+1,"raw")==0)
      { return AUDIO_FILE_FORMAT_RAW;
      }
     if(stricmp(cp+1,"dat")==0)        // .dat produced by AUDIO UTILITIES, also headerless raw data
      { // In this case, there is an ambiguity between "old" raw files,
        // and the 8-bit I/Q format used by the SETI folks (also *.dat) :
        switch( iDefaultAudioFileType )
         { case AUDIO_FILE_FORMAT_SETI_1: // SETI 8-bit raw I/Q, *.dat, 8738133 samples/sec
              return iDefaultAudioFileType;
           default:  // cannot be MP3, WAW, etc !
              return AUDIO_FILE_FORMAT_RAW;
         }
      }
     if(stricmp(cp+1,"wav")==0)
      { return AUDIO_FILE_FORMAT_RIFF; // (2) standard "RIFF wave audio" (*.wav)
      }
     if( (stricmp(cp+1,"ogg")==0) || (stricmp(cp+1,"ogx")==0) || (stricmp(cp+1,"oga")==0) )
      { return AUDIO_FILE_FORMAT_OGG;  // Ogg(container) / Vorbis(codec) (*.ogg)
        //  .ogg, .spx = "audio/ogg" nach http://tools.ietf.org/html/rfc5334
        //  .ogx       = "application/ogg"
        //  .oga       = "audio/ogg" (im Gegensatz zu .ogg NUR Audio, kein Video)
      }
     if(stricmp(cp+1,"txt")==0)
      { return AUDIO_FILE_FORMAT_TEXT;
      }
 //  if(stricmp(cp+1,"jas")==0)
 //   { return AUDIO_FILE_FORMAT_JASON;
 //   }
     if(stricmp(cp+1,"m3u")==0)
      { return AUDIO_FILE_FORMAT_M3U; // not an AUDIO FILE FORMAT but a 'playlist' (text)
        // To detect the REAL type of this medium, the file (*.m3u) must be opened,
        // or loaded from a remote server !
        // Example:   http://67.207.143.181/vlf6.m3u   may contain something like this:
        //            http://67.207.143.181:80/vlf6    .
        // Opening that *STREAM* (through the socket services) turns up
        // the TRUE "file" format later (mp3 compression or ogg/vorbis) .
      }
     if(stricmp(cp+1,"mp3")==0)
      { return AUDIO_FILE_FORMAT_MP3; // not supported by SpecLab (requires plugin)
      }

   } // end if < saw something which LOOKS LIKE a file extension >

  if( strnicmp( pszFilename, "http://", 7)== 0 ) // added 2011-12-05
   { if( iDefaultAudioFileType == AUDIO_FILE_FORMAT_UNKNOWN )
      { return AUDIO_FILE_FORMAT_OTHER_WEB_STREAM;
      }
     else
      { return iDefaultAudioFileType;
      }
   }

  if( strnicmp( pszFilename, "rawtcp://", 8)== 0 ) // added 2011-12-11 (dummy for "raw TCP", not HTTP)
   { if( iDefaultAudioFileType == AUDIO_FILE_FORMAT_UNKNOWN )
      { return AUDIO_FILE_FORMAT_OTHER_WEB_STREAM;
      }
     else
      { return iDefaultAudioFileType;
      }
   }

  return iDefaultAudioFileType;
} // end INET_GuessAudioFileTypeFromName()


/****************************************************************************/
int INET_GuessAudioFileTypeFromData(
       BYTE *pbFileData,   int nBytes,
       int iDefaultAudioFileType )  // [in] default type (AUDIO_FILE_FORMAT_..) when no match was found
{
  if( (pbFileData!=NULL) && (nBytes>=8) )
   { if( strncmp( (char*)pbFileData, "OggS", 4) == 0 )
      { return AUDIO_FILE_FORMAT_OGG;  // Ogg(container) / Vorbis(codec) (*.ogg)
      }
   }

  return iDefaultAudioFileType;
} // end INET_GuessAudioFileTypeFromData()


/****************************************************************************/
int INET_ContentTypeToAudioFileFormat( char *pszContentType, int iDefaultAudioFileType )
{
  BYTE *bp = (BYTE*)pszContentType;

  if( UTL_CheckAndSkipToken( &bp, "audio/x-mpegurl" ) )
   {  // This is what http://67.207.143.181/vlf6.m3u replied.
      // According to http://webdesign.about.com/od/sound/a/sound_mime_type.htm ,
      //  "audio/x-mpegurl" is only the MIME-type for "m3u" .
      // It doesn't tell us anything about the audio stream itself !
      //  (the stream itself may be OGG/VORBIS instead of MPEG)
      return AUDIO_FILE_FORMAT_M3U; // ok, it's definitely M3U
   }
  else if( UTL_CheckAndSkipToken( &bp, "audio/mpeg" ) )
   {  // Now we know this URL (pszAudioFileName) is not the name
      // of a PLAYLIST (*.m3u) but the name of an MP3 file or stream !
      return AUDIO_FILE_FORMAT_MP3; // not M3U but MP3
   }
  else if( UTL_CheckAndSkipToken( &bp, "application/ogg" ) )  // RFC 3534 - what a misnomer !
   {  // This is the content-type returned from http://67.207.143.181:80/vlf1 :
      return AUDIO_FILE_FORMAT_OGG;
   }
  else if( UTL_CheckAndSkipToken( &bp, "audio/ogg" ) ) // RFC 5334
   {  return AUDIO_FILE_FORMAT_OGG;
   }
  else if( UTL_CheckAndSkipToken( &bp, "text/html" ) ) // RFC 5334
   {  // This is, most likely, NOT an audio file,
      // but most likely a "404"-like error message from an Icecast server.
      // We still don't know anything until we READ the damned HTML page
      // and try to parse it.
      return iDefaultAudioFileType; // !
   }
  else
   {  return iDefaultAudioFileType;
   }
} // end INET_ContentTypeToAudioFileFormat()

/****************************************************************************/
char * INET_AudioFileFormatToContentType( int iAudioFileType )
{
  static char  sz39Unknown[40];
  switch( iAudioFileType )
   { case AUDIO_FILE_FORMAT_M3U:
        return "audio/x-mpegurl";
     case AUDIO_FILE_FORMAT_MP3:
        return "audio/mpeg";
     case AUDIO_FILE_FORMAT_OGG:
        return "audio/ogg";
     default:
        sprintf( sz39Unknown, "unknown(%d)", (int)iAudioFileType );
        return sz39Unknown;
   }
} // end INET_AudioFileFormatToContentType()


/****************************************************************************/
int INET_SplitURL(
          char * pszFilenameOrURL,   // [in] filename or URL (will be DESTROYED / "chopped into pieces" !)
          T_INET_UrlParts *parts )   // [out] pointers to the 'parts'
  // Breaks up a filename (also for local files), or a URL, into pieces .
  // The ORIGINAL STRING (pszFilenameOrURL) WILL BE DESTROYED !
  // Loosely based on 'HTParse.c' from www.w3.org/Library/src/HTParse.c .
  //
  // [in] pszFilenameOrURL : usually a URL, with the following format :
  //               scheme://domain:port/path?query_string#fragment_id
  //      Example: http://67.207.143.181/vlf6.m3u
  // [out] T_INET_UrlParts *parts : Simple zero-terminated C-strings, example:
  //         parts->access   =
  //         parts->host     = hostname like "qsl.net" or IP string like "192.168.0.123", WITHOUT PORT NUMBER
  //         parts->absolute =
  //         parts->relative =
  //         parts->fragment =
  //         parts->iProtocol: may contain 0=unknown, or INET_PROTOCOL_HTTP, etc .
  //         parts->iPort    : port NUMBER (0=not explicitly set in INET_SplitURL)
  // Return value: ?
{
  int  iResult = 0;
  char * p;
  char * after_access = pszFilenameOrURL;
  memset(parts, 0, sizeof(T_INET_UrlParts) );

  /* Look for fragment identifier */
  if ((p = strchr(pszFilenameOrURL, '#')) != NULL)
   {
	  *p++ = '\0';
     parts->pszFragment = p;
   }


  if ((p = strchr(pszFilenameOrURL, ' ')) != NULL) *p++ = '\0';

  for(p=pszFilenameOrURL; *p; p++)
   {

	  // Look for any whitespace. This is very bad for pipelining
     // as it makes the request invalid
     if (isspace((int) *p))
      {
	     char *orig=p, *dest=p+1;
	     while ((*orig++ = *dest++) != 0);  // scroll up by one character, delete the space character
	     p = p-1;
      }
     if (*p=='/' || *p=='#' || *p=='?')
      { break;
      }
     if (*p==':')
      {
        *p = 0;
        parts->pszAccess = after_access; /* Scheme has been specified */

        // The combination of gcc, the "-O" flag and the HP platform is
        // unhealthy. The following three lines is a quick & dirty fix, but is
        // not recommended. Rather, turn off "-O".
        /*		after_access = p;*/
        /*		while (*after_access == 0)*/
        /*		    after_access++;*/

        after_access = p+1;

        if (0==stricmp("URL", parts->pszAccess))
         {
		     parts->pszAccess = NULL;  /* Ignore IETF's URL: pre-prefix */
         }
        else
         { break;
         }
      }
   }

  // Added by DL4YHF: If there's this 'access'-thingy, examine it further:
  parts->iProtocol = INET_PROTOCOL_UNKNOWN; // may contain 0=unknown, or INET_PROTOCOL_HTTP, etc .
  if( parts->pszAccess != NULL )
   { if( stricmp(parts->pszAccess, "http" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_HTTP;
      }
     else if( stricmp(parts->pszAccess, "https" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_HTTPS;
      }
     else if( stricmp(parts->pszAccess, "file" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_FILE;  // it's a LOCAL FILE !
      }
     else if( stricmp(parts->pszAccess, "ftp" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_FTP;
      }
     else if( stricmp(parts->pszAccess, "mailto" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_FTP;
      }
     else if( stricmp(parts->pszAccess, "rawtcp" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_RAW_TCP;
      }

   }

  p = after_access;
  if (*p=='/')
   {
	  if (p[1]=='/')
      {
        parts->pszHost = p+2;   // host has been specified
        *p=0;                   // Terminate access
        p=strchr(parts->pszHost,'/');	// look for end of host name if any
        if(p)
         {
           *p=0;			        // Terminate host
           parts->pszAbsolute = p+1;   // Root has been found
         }
        // Added by WoBu: If there's a port number after the host name,
        //                separate it, and convert it into integer :
        p=strchr(parts->pszHost,':'); // look for separator between host and port
        if(p)
         { *p=0;             // Terminate host again (now before the port)
           parts->iPort = atoi(p+1);
         }

      }
     else
      {
        parts->pszAbsolute = p+1;		// Root found but no host
      }
   }
  else
   {
     parts->pszRelative = (*after_access) ? after_access : NULL; /* NULL for "" */
   }
  return iResult;
} // end INET_SplitURL()



/****************************************************************************/
BOOL INET_IsInternetStream( char * pszFilenameOrURL )
   // Returns TRUE when pszFilenameOrURL looks like an 'internet' resource;
   //      or FALSE when it's most likely the path+name of a 'local' file .
{
  char *cpFirstColon = strchr(pszFilenameOrURL,':');

  if( (cpFirstColon != NULL) &&  (cpFirstColon > (pszFilenameOrURL+2) ) )
   { // something like http:/ ... doesn't look like a file ...
     return TRUE;   // there's colon, but not something like "C:\blabla" ... guess it's an internet stream
   }
  else
   { return FALSE;  // not an (internet-)STREAM but a (local) FILE
   }
} // end INET_IsInternetStream()

/****************************************************************************/
CPROT int INET_InitClient(
     T_INET_Client *pInetClient,    // [out] client data struct
     T_InetCallback pPICallback,    // [in, optional] progress indicator callback
     void *pvUserCallbackData)      // [in, optional] user callback data
{
  if( pInetClient==NULL )
   { return ERROR_ILLEGAL_CMD_PARAMETER;
   }
  memset( pInetClient, 0, sizeof( T_INET_Client ) );
  pInetClient->sock = INVALID_SOCKET;         // note: this is NOT ZERO (thus the need for 'magic')
  pInetClient->pPICallback = pPICallback;
  pInetClient->pvUserCallbackData = pvUserCallbackData;
  pInetClient->iState = INET_CLIENT_STATE_PASSIVE;
  pInetClient->dwMagic17724538 = 0x17724538;  // declare this struct valid
  pInetClient->dwMagic98696044 = 0x98696044;

  return NO_ERRORS;
} // end INET_InitClient()


/****************************************************************************/
int INET_InvokeCallback(
     T_INET_Client *pInetClient, // [in,out] web client data
     int  iEventType,            // [in] INET_EVENT_TYPE_INFO / ..ERROR ?
     int  iProgressPercent,      // [in] progress in percent, or an 'event code'
     char * pszFormatString,     // [in] format string like "Trying to connect host %s"
     ...)                        // [in] 'value' to be shown, must match the format string
{
  va_list arglist;
  char sz255[256];
  int result=0;

  // Print to string and append to edit control
  va_start(arglist, pszFormatString);
  vsprintf(sz255, pszFormatString, arglist);
  if( pInetClient->pPICallback != NULL )
   { result = pInetClient->pPICallback( pInetClient->pvUserCallbackData, pInetClient,
                                        iEventType, sz255, iProgressPercent );
   }
  va_end(arglist);  // important to avoid stack corruption - must ALWAYS be called !

  return result;
} // end INET_InvokeCallback()

/****************************************************************************/
static void INET_Client_ProcessHTTPResponse(
              T_INET_Client *pInetClient)    // [in,out] web client data
  // Called after reception of data during state INET_CLIENT_STATE_WAIT_HTTP_RESP .
  // Checks pInetClient->c1kRxBuffer[0...pInetClient->nBytesInRxBuffer-1]
  //        for a valid HTTP response (typically, a HTTP GET response);
  //        and if the response is complete, switches from state
  //        from state INET_CLIENT_STATE_WAIT_HTTP_RESP
  //        into state INET_CLIENT_STATE_WAIT_FILE_DATA .
  // Bytes which don't belong to the HTTP GET response but to the 'file data'
  //        remain in pInetClient->c1kRxBuffer[0...pInetClient->nBytesInRxBuffer-1] .
{
  // Details in http://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Server_response :
  // > A server response is followed by a blank line and text of the requested page.
  // >  ...
  // > Most of the header lines are optional. When Content-Length is missing
  // > the length is determined in other ways.
  // Here: Don't care for all those 'header lines', only care for the
  //  'blank line' which separates the response-stuff (with all those chatty
  //  'header lines') from the real "file data", which is what the caller
  //  wants to retrieve .
  //
  // Note: Don't assume anything. Especially not what a 'blank line' looks like.
  //   A blank line may be "\n\n" or "\r\n\r\n" (same stupidity as in text files).
  //
  // Note: We're playing with the internet here, and the internet is often
  //       a crappy, short-living place. No big surprise to receive a response
  //       like the following (from Deutschlandfunk's Ogg Vorbis stream,
  //       http://www.dradio.de/streaming/dlf_mq_ogg.m3u ) :
  //           pInetClient->nBytesInRxBuffer = 242
  //           pInetClient->c1kRxBuffer =
  //    "HTTP/1.0 301 Moved Permanently\r\n"
  //    "Server: squid/2.5.STABLE14\r\n"
  //    "Date: Sun, 11 Dec 2011 21:36:57 GMT\r\n"
  //    "Content-Length: 0\r\n"
  //    "Location: http://www.dradio.de\r\n"
  //    "Set-Cookie: NSC-esbeip_esbeip_fffffffffasheblahdideldumm;"
  //    "path=/;httponly\r\n\r\n" .
  //  Trying the same with Firefox (which sends a much longer GET request)
  //  resulted in an M3U file with the following content (which is the "real" URL):
  //    http://dradio.ic.llnwd.net/stream/dradio_dlf_medium_a.ogg
  //
  int  iIndexOfFirstNonHeaderByte = 0; // buffer-index of the last 'header'-byte : NOT FOUND
  int  i, n, nBytesInRxBuffer = pInetClient->nBytesInRxBuffer;
  char c, *cp, *cpRxBuffer = pInetClient->c1kRxBuffer;
  for(i=0; i<nBytesInRxBuffer-1; ++i)
   { if( cpRxBuffer[i]=='\n' )  // Ok, this may be the end of a line, 'UNIX' or 'DOS' text format....
      { // what's the NEXT character, another '\n', or "\r\n" ?
        if( cpRxBuffer[i+1] == '\n' )  // yes, must be a UNIX-like server ..
         { iIndexOfFirstNonHeaderByte = i+2;   // buffer-index of the last 'header'-byte
           break;
         }
        else
        if((cpRxBuffer[i+1] == '\r' )   // DOS- or WINDOWS text format (with CR+NL as end-of-line markers)
         &&(cpRxBuffer[i+2] == '\n' ) )
         { iIndexOfFirstNonHeaderByte = i+3;   // buffer-index of the last 'header'-byte
           break;
           // Note: Even a server which was definitely running UNIX (or LINUX)
           //       used CR+NL as end-of-line marker in the HTTP responses ;-)
           // The complete HTTP transfer was read in an entire block,
           // including both the HTTP 'GET' response *AND* the file data :
           // cpRxBuffer: "HTTP/1.0 200 OK\r\nContent-Type: audio/x-mpegurl\r\n\r\nhttp://67.207.143.181:80/vlf6\r\n"
         }
      }
   }
  if( iIndexOfFirstNonHeaderByte > 0 ) // found the 'blank line' which marks the end of the response-header-lines !
   { // Since the rx-buffer is 'ours', we can truncate the 'response text' with a zero-byte here,
     // and pass it to the application (just in case the application wants to parse it).
     // In other words, an opportunity for the application to parse the received HTTP GET RESPONSE.
     cpRxBuffer[iIndexOfFirstNonHeaderByte-1] = '\0';
     cp = cpRxBuffer;    // parse the 'header lines' in the HTTP GET response...
     while(*cp != '\0')
      { // Only check for the 'initial response line' and a few 'header lines'
        // (in the HTTP "GET" response), look for a few special tokens .
        // > The header lines are in the usual text header format,
        // > which is: one line per header,
        // >   of the form   "Header-Name: value", ending with CRLF.
        if( UTL_CheckAndSkipToken( (BYTE**)&cp, "HTTP/" ) )
         { // This must be the "Initial Response Line" .
           // Followed by the HTTP protocol version ("1.0" or "1.1", or whatever)
           // and status code as a machine-readable 3-digit integer .
           pInetClient->iHttpMajorVersion = UTL_ParseInteger( (BYTE**)&cp, 1/*digits*/ ); // almost certain the result is ONE
           if( *cp=='.' ) ++cp;
           pInetClient->iHttpMinorVersion = UTL_ParseInteger( (BYTE**)&cp, 1/*digits*/ ); // this result may be ZERO (HTTP 1.0) or ONE (HTTP 1.1)
           pInetClient->iHttpStatusCode = UTL_ParseInteger( (BYTE**)&cp, 3/*digits*/ );   // this result will hopefully be TWOHUNDRED (don't ask why)
         }
        else
        if( UTL_CheckAndSkipToken( (BYTE**)&cp, "Content-Type:" ) )
         { // This thingy defines, more or less, the type of the 'netto data'
           // which will follow immediately after the 'header lines'.
           UTL_SkipSpaces( (BYTE**)&cp );
           UTL_CopyStringUntilEOL( (BYTE*)cp, pInetClient->sz80ContentType, 80 );
         }
        // Increment 'cp' until the end of this line, and skip CR+NL (or only NL?)
        while( (c=*cp++) != '\0')
         { if(c=='\n')  // not 100 % foolproof but reasonably simple
            { break;
            }
         }
      } // end while(*cp != '\0') ...

     if( pInetClient->pPICallback != NULL )
      { pInetClient->pPICallback( pInetClient->pvUserCallbackData, pInetClient,
          INET_EVENT_TYPE_PARSE_RESPONSE, cpRxBuffer, iIndexOfFirstNonHeaderByte );
      }


     // After this, there MAY be bytes remaining in the RX-buffer which belong
     // to the 'netto' file data. We don't want to lose them, thus scroll up
     // the buffer contents, so the first 'file data' byte will be at index ZERO.
     n = pInetClient->nBytesInRxBuffer - iIndexOfFirstNonHeaderByte;
     if( n>0 )  // some "non-header-bytes" will remain in the RX-buffer !
      { for(i=0; i<n; ++i)
         { pInetClient->c1kRxBuffer[i] = pInetClient->c1kRxBuffer[i+iIndexOfFirstNonHeaderByte];
         }
        pInetClient->nBytesInRxBuffer = n;
      }
     else
      { pInetClient->nBytesInRxBuffer = 0;  // the RX-buffer is now "completely empty"
      }

     if( pInetClient->iState == INET_CLIENT_STATE_WAIT_HTTP_RESP )
      {  // had been waiting for reception of the (HTTP-?) response,
         // but after this, we're waiting for the 'file data' :
         pInetClient->iState = INET_CLIENT_STATE_WAIT_FILE_DATA;
      }
   } // end if( iIndexOfFirstNonHeaderByte > 0 )
  else  // Reception of the HTTP GET RESPONSE is obviously not complete yet :
   {
   }
} // end INET_Client_ProcessHTTPResponse()

/****************************************************************************/
int INET_OpenAndReadFromServer(
              T_INET_Client *pInetClient,    // [in,out] web client data
              char * pszFilenameOrURL,       // [in] filename or URL
              int  iTimeout_ms,              // [in] max. timeout in milliseconds
              int  iOptions,                 // [in] bitwise combination of the following:
              // INET_RD_OPTION_NORMAL                : return after reception of and number of bytes
              // INET_RD_OPTION_RETURN_AFTER_RESPONSE : return after initial response; no need to wait for the CONTENTS
              // INET_RD_OPTION_WAIT_FOR_ALL_DATA     : return only when all <iMaxLength> bytes received, or timeout
              BYTE *pbDest, int iMaxLength)  // [in,out] destination buffer for the first part of the NETTO DATA
                                             //   (aka "content", not the HTTP response headers)
  // Used, for example, to examine the contents of an M3U file
  //       on a remote server ("stream") .
  // Returns the number of bytes actually placed in pbDest,
  //       or a NEGATIVE ERROR CODE ("error code MADE NEGATIVE") .
  // Note: The error codes in ErrorCodes.h are all POSITIVE !
  //       To tell them from a 'successful result' (NUMBER OF BYTES READ),
  //       we make those errors NEGATIVE here.
  // Note: The TCP/IP connection is NOT automatically closed on return,
  //       because the caller may want to read more bytes from the stream !
{
  struct sockaddr_in sin;
  struct in_addr addr;
  struct hostent *host;
  int result,len,progress_percent;
  int nBytesToSend, nBytesSent;
  int nBytesRcvdNow, nBytesToRead;
  int nBytesInDest = 0;
  int nMillisecondsWaited = 0;
  int iWSAErrorCode;
  unsigned long ul;
  char szChoppedURL[INET_MAX_URL_LENGTH+1];
  char sz255Request[256];
  T_INET_UrlParts UrlParts;
  BOOL fRecvdBefore;

  if( pInetClient==NULL )
   { return -ERROR_ILLEGAL_CMD_PARAMETER;
   }

  if(  (pInetClient->dwMagic17724538 == 0x17724538)
    && (pInetClient->dwMagic98696044 == 0x98696044)
    && (pInetClient->sock != INVALID_SOCKET) )  // oops ?!
   { // both magic numbers valid (pi^0.5, pi^2) -> was open but not closed yet !!
     INET_CloseClient( pInetClient );  // bear with the caller; close old connection autoMAGICally
     pInetClient->sock = INVALID_SOCKET; // redundant but defensive (don't close twice)
   }

  if( pszFilenameOrURL==NULL )
   { return -1;  // ERROR: something terribly wrong with the input arguments
   }

  len = strlen(pszFilenameOrURL);
  if( len<1 || len>INET_MAX_URL_LENGTH )
   { return -1;  // ERROR: something terribly wrong with the input arguments
   }
  strcpy(szChoppedURL, pszFilenameOrURL );
  result = INET_SplitURL( szChoppedURL, &UrlParts );
  if( UrlParts.pszHost == NULL )
   { return -INET_ERROR_BAD_URL_SYNTAX;  // ERROR: not a valid "host"-name
   }
  pInetClient->iProtocol = UrlParts.iProtocol;
  pInetClient->iState = INET_CLIENT_STATE_PASSIVE;
  pInetClient->nBytesInRxBuffer = 0;
  pInetClient->ulTotalBytesReceived = 0; // for diagnostic purposes
  pInetClient->ulTotalBytesSent     = 0;

  if( INET_IsInternetStream( pszFilenameOrURL ) )
   { // it's an internet resource, not a FILE ON THE LOCAL DISK:
     pInetClient->is_internet_resource = TRUE;
     INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, 0/*progress*/,
         "Creating socket" );
     pInetClient->sock = socket(AF_INET, SOCK_STREAM, 0);  /* init socket descriptor */
     if( pInetClient->sock==INVALID_SOCKET )
      { INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, 0/*progress*/,
           "Failed to create socket" );
        return -INET_ERROR_SOCKET_NOT_OPEN;  // ERROR: something went wrong with the socket
        // Note that the error codes in ErrorCodes.h are all POSITIVE !
        // To tell them from a 'successful result' (NUMBER OF BYTES READ),
        // we make those errors NEGATIVE here.  Any better solution ?
      }

     // Note that some say gethostbyname does not resolve IP address strings passed to it !
     // Such a request is treated exactly as if an unknown host name were passed.
     // An application with an IP address string to resolve should use inet_addr
     // to convert the string to an IP address, then gethostbyaddr to obtain the hostent structure.
     // Anyway, with UrlParts.pszHost == "67.207.143.181", gethostbyname() returned NON-NULL !
     //   Maybe it's a stupid IP STRING (like "192.168.0.222"),
     //   so we don't need to look up a 'host by name' at all:
     addr.s_addr = inet_addr(UrlParts.pszHost); // maybe it's an "IP-String"..
     host = gethostbyaddr((char*)&addr, sizeof(addr), AF_INET);
     if( host==NULL )
      { // Obviously not a simple "dotted IP string", so:
        host = gethostbyname(UrlParts.pszHost);
        // > If no error occurs, gethostbyname returns a pointer
        // > to the hostent structure described above. Otherwise, it returns a NULL pointer.
      }
     if( host==NULL ) // neither a stupid IP STRING ("192.168.0.222"), nor a known host name !
      { closesocket( pInetClient->sock );  pInetClient->sock=INVALID_SOCKET;
        INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, 0/*progress*/,
           "gethostbyname(%s) failed", UrlParts.pszHost );
        return -INET_ERROR_SOCKET_NOT_OPEN;  // ERROR: something else went wrong with the socket
      }

     // Prepare the sockaddr_in structure which we need to connect to the remote host
     memcpy(&sin.sin_addr.s_addr, host->h_addr, host->h_length);
        // Note: There isn't a h_addr member in the 'hostent' structure.
        //  But the compiler didn't complain. Is it a stupid MACRO ?
        //  Yes it is. #define h_addr  h_addr_list[0]. No surprise that the debugger
        //  cannot evaluate it.  The h_addr_list[]-thingy is just another monster,
        //  it's actually a char FAR * FAR * ;  (forget about the stoneage-"FAR")
        //  This is just another annoyance of the obfuscated socket library.
        //  Um diesen elenden Dreck mit dem Debugger zu inspizieren, WATCH this:
        //  (int)(unsigned char)host->h_addr_list[0][0], ..[0][1], ..[0][2], ..[0][3].
        //  The result should be the IP address .
     sin.sin_family = AF_INET;
     if( UrlParts.iPort<=0 )
      {
        switch( pInetClient->iProtocol )
         { case INET_PROTOCOL_HTTP:
           default:
              UrlParts.iPort = 80;  // default for HTTP
              break;
         } // end switch( pInetClient->iProtocol )
      }
     sin.sin_port = htons(UrlParts.iPort);

     // Connect to the socket service described by the sockaddr_in struct :
     INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, 0/*progress*/,
         "Trying to connect host %s", UrlParts.pszHost );
     // Note: It seems important to 'connect' BEFORE making the socket non-blocking,
     //       otherwise connect() will usually fail, and there's no way
     //       to instruct 'connect' to NOT return until the connection
     //       is established. To keep it simple, do NOT use a loop here:
     if (connect(pInetClient->sock, (struct sockaddr *)&sin, sizeof(sin)) < 0)
      { iWSAErrorCode = WSAGetLastError();
        closesocket( pInetClient->sock );   pInetClient->sock=INVALID_SOCKET;
        INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, 0/*progress*/,
             "Failed to connect %s : %s", UrlParts.pszHost,
             INET_WinsockErrorCodeToString(iWSAErrorCode) );
        return -INET_ERROR_COULDNT_CONNECT;
      }
     INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, 0/*progress*/,
         "Connected to %s", UrlParts.pszHost );


     // Now, AFTER successfully connecting the remote "host" (server),
     //      try to make this socket "non-blocking".
     //      This is not very clear in the Win32 programmer's reference !
#if(1)
     // FIONBIO: 'Enable or disable nonblocking mode on socket' .
     ul = 1;  // nonzero for nonblocking mode
     result = ioctlsocket( pInetClient->sock, FIONBIO, &ul );
     if(result == SOCKET_ERROR)
      { iWSAErrorCode = WSAGetLastError();
        closesocket( pInetClient->sock ); pInetClient->sock=INVALID_SOCKET;
        INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, 0/*progress or event*/,
           "Failed to make socket non-blocking : %s",
           INET_WinsockErrorCodeToString(iWSAErrorCode) );
        return -INET_ERROR_SOCKET_NOT_OPEN;  // ERROR: something else went wrong with the socket
      }
#endif

     pInetClient->fConnectionLost = FALSE;  // clear this 'old' flag for the application

     // If the protocol is HTTP (which will often be the case when starting
     //  to read an audio stream hidden in an M3U "playlist") ,
     //  the client (i.e. THIS END) must inform the remote server
     //  what to do. In HTTP parlance: Send a GET command, plus some red tape
     //  (barebone web programming always comes along with an AWFUL lot of red tape..)
     // From a simple introduction on HTTP from http://www.jmarshall.com/easy/http/#requestline :
     // > To retrieve the file at the URL
     // >     http://www.somehost.com/path/file.html
     // > first open a socket to the host www.somehost.com, port 80
     // > (use the default port of 80 because none is specified in the URL).
     // > Then, send something like the following through the socket:
     // >     GET /path/file.html HTTP/1.0
     // >     From: someuser@jmarshall.com
     // >     User-Agent: HTTPTool/1.0
     // >     [blank line here]
     // > The server should respond with something like the following,
     // > sent back through the same socket:
     // >     HTTP/1.0 200 OK
     // >     Date: Fri, 31 Dec 1999 23:59:59 GMT
     // >     Content-Type: text/html
     // >     Content-Length: 1354
     // >     <html> ........
     // From Wikipedia :
     // > The request message consists of the following:
     // >
     // >  *  Request line, such as GET /images/logo.png HTTP/1.1,
     // >     which requests a resource called /images/logo.png from server
     // >  *  Headers, such as Accept-Language: en
     // >  *  An empty line.
     // >  *  An optional message body.
     // > The request line and headers must all end with <CR><LF>
     // > (that is, a carriage return followed by a line feed).
     // > The empty line must consist of only <CR><LF> and no other whitespace .
     pInetClient->iState = INET_CLIENT_STATE_SENDING_REQUEST;
     switch( pInetClient->iProtocol )
      { case INET_PROTOCOL_HTTP :
           strcpy( sz255Request, "GET " );
           if( UrlParts.pszAbsolute != NULL )
            { if( UrlParts.pszAbsolute[0] != '/' )
               { strcat( sz255Request, "/" );
               }
              strcat( sz255Request, UrlParts.pszAbsolute );
            }
           else if( UrlParts.pszRelative != NULL )
            { strcat( sz255Request, UrlParts.pszRelative );
            }
           strcat( sz255Request, " HTTP/1.0\r\n\r\n" );   // note the 2nd -EMPTY- line !
           nBytesToSend = strlen(sz255Request);
           nBytesSent = send( pInetClient->sock, sz255Request, nBytesToSend, 0 );
           if( nBytesSent != nBytesToSend )
            { INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, 0/*progress or event*/,
                     "Failed to send GET-request" );
            }
           else
            { pInetClient->iState = INET_CLIENT_STATE_WAIT_HTTP_RESP;
              INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, 1/*progress or event*/,
                     "%s", sz255Request );
            }
           break; // end case INET_PROTOCOL_HTTP
         case INET_PROTOCOL_RAW_TCP :  // added 2011-12-11 to experiment with timestamped Ogg/Vorbis streams
           // Nothing to send in this case (no REQUEST);
           //  the "file" data (in fact, a stream) should just start to pour in !
           pInetClient->iState = INET_CLIENT_STATE_WAIT_FILE_DATA;
           break; // end case INET_PROTOCOL_RAW_TCP
         default:
           break;
      } // end switch( pInetClient->iProtocol )

     if( iTimeout_ms>0 ) // immediately READ SOMETHING (HTTP response or similar) after opening the connection ?
      {
        // Try to read the first NNN bytes from the remote host.
        // Note that, at this point, the socket is NON-BLOCKING which means
        // revc() will return IMMEDIATELY if there are no data waiting
        // in the network buffer !
        nMillisecondsWaited = 0;
        fRecvdBefore = FALSE;
        INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, 0/*progress*/,
           "Waiting for reception of first data" );

        while ( pInetClient->iState == INET_CLIENT_STATE_WAIT_HTTP_RESP )
         {
           if( nMillisecondsWaited > iTimeout_ms )
            { // been waiting too long (at least longer than the caller can accept) ->
              INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, 0/*progress or event*/,
                  "Timed out waiting for response" );
              break;
            }
           progress_percent = (100 * nMillisecondsWaited) / iTimeout_ms;
           nBytesToRead = sizeof(pInetClient->c1kRxBuffer) - pInetClient->nBytesInRxBuffer;
           nBytesRcvdNow = recv( // why do so many people use 'read' instead of 'recv' ?
                 pInetClient->sock,   // [in] socket; descriptor identifying a connected socket
                 pInetClient->c1kRxBuffer // [out] buffer for the received HTTP response (or similar)
                  + pInetClient->nBytesInRxBuffer,
                 nBytesToRead,        // [in] MAX. number of bytes acceptable in that buffer
                 0 );                 // [in] flags; specifies how the call is made
              // > If no error occurs, recv returns the number of bytes received.
              // > If the connection has been gracefully closed, the return value
              // > is zero. Otherwise, a value of SOCKET_ERROR is returned,
              // > and a specific error code can be retrieved by calling WSAGetLastError.
              // > If no incoming data is available at the socket,
              // > the recv call waits for data to arrive
              // > unless the socket is nonblocking. In this case, a value of
              // > SOCKET_ERROR is returned with the error code set to WSAEWOULDBLOCK.
              // > The select, WSAAsyncSelect, or WSAEventSelect calls can be used
              // > to determine when more data arrives.
           if( nBytesRcvdNow > 0 )  // hooray, successfully "received something" !
            { pInetClient->nBytesInRxBuffer += nBytesRcvdNow;
              pInetClient->ulTotalBytesReceived += nBytesRcvdNow;
              fRecvdBefore = TRUE;
              INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, progress_percent,
                   "Receiving response.. %d bytes", (int)pInetClient->nBytesInRxBuffer );
              // Still waiting for the reception of the HTTP RESPONSE,
              //       or already receiving the netto "file data" ?
              INET_Client_ProcessHTTPResponse( pInetClient );
              // Note: INET_Client_ProcessHTTPResponse() may switch the state
              //       from INET_CLIENT_STATE_WAIT_HTTP_RESP
              //         to INET_CLIENT_STATE_WAIT_FILE_DATA,
              //       which would terminate this 'waiting loop' .
              if( pInetClient->iState != INET_CLIENT_STATE_WAIT_HTTP_RESP )
               { // Finished waiting for the GET-response ("headers"),
                 // may already have read the first part of the "netto file data"
                 // because the network buffer won't care for 'empty lines'
                 // after the HTTP GET header(s) !
                 if( iOptions & INET_RD_OPTION_RETURN_AFTER_RESPONSE )
                  { // return after initial response; no need to wait for the CONTENTS
                    // (despite that, bytes which don't belong to the HTTP GET response
                    //   -but to the 'file data' aka 'content'- remain in
                    //  pInetClient->c1kRxBuffer[0...pInetClient->nBytesInRxBuffer-1] )
                    break;
                  }
                 else
                  { nBytesInDest = INET_ReadDataFromServer( pInetClient,
                                    iTimeout_ms, iOptions, pbDest, iMaxLength );
                  }
               } // end if( pInetClient->iState != INET_CLIENT_STATE_WAIT_HTTP_RESP )
            }
           else // nothing received this time, or connection closed, or another ERROR ?
            { if( nBytesRcvdNow==SOCKET_ERROR )  // obviously a REAL ERROR... or what ?!!??!!?!
               { iWSAErrorCode = WSAGetLastError();
                 if( iWSAErrorCode == WSAEWOULDBLOCK )  // winsock's way of saying 'nothing received' ?
                  { // No, not really. As usual, the documentation is poor.
                    // recv() often returned ZERO when nothing was received.
                  }
                 else // not just 'nothing received' but a REAL ERROR.  Which ?
                  { INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, progress_percent,
                      "Error from recv: %s",
                      INET_WinsockErrorCodeToString(iWSAErrorCode) );
                    break;
                  }
               }
              else  // nothing received, but no "socket error"
               { // Got here even with a BLOCKING socket(!),
                 // which did NOT block, but returned ZERO !
               }
              if( fRecvdBefore )  // nothing received now, but something received before
               { // -> network buffer is completely empty; "done" !
                 break;
               }
              else
               { // Wait a few milliseconds more, typically for the 'initial response'
                 if( iTimeout_ms>0 )
                  { Sleep(50/*ms*/);   // .. give the remote server ("host") time to reply
                    nMillisecondsWaited += 50/*ms*/;
                  }
                 else // call is made "completely non-blocking" :
                  { break;
                  }
               }
            }
         } // end while
      } // end if( iTimeout_ms>0 )
     // Note: The connection REMAINS OPEN !
   } // end if < not a file but an internet resource, most likely a STREAM or M3U on remote server >
  else
   { // it's a FILE on the local disk:
     pInetClient->is_internet_resource = FALSE;
     pInetClient->fConnectionLost = FALSE;
   } // end else < file on the local disk>

  INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, 0/*progress*/,
       "Returning from 'OpenAndRead' with %d bytes", (int)nBytesInDest );

  return nBytesInDest;
} // end INET_OpenAndReadFromServer()

/****************************************************************************/
int INET_ReadDataFromServer(
     T_INET_Client *pInetClient,    // [in,out] web client data
     int  iTimeout_ms,              // [in] max. timeout in milliseconds
     int  iOptions,                 // [in] bitwise combination of the following:
     // INET_RD_OPTION_NORMAL                : return after reception of any number of bytes
     // INET_RD_OPTION_WAIT_FOR_ALL_DATA     : return only when all <iMaxLength> bytes received, or timeout
     BYTE *pbDest, int iMaxLength)  // [in,out] destination buffer
  // Used, for example, to examine the contents of an M3U file
  // on a remote server ("stream") .
  // Returns the number of bytes actually placed in pbDest,
  //       or a NEGATIVE ERROR CODE ("error code MADE NEGATIVE") .
  // Note: The error codes in ErrorCodes.h are all POSITIVE !
  //       To tell them from a 'successful result' (NUMBER OF BYTES READ),
  //       we make those errors NEGATIVE here.
  // Note: The TCP/IP connection is NOT automatically closed on return,
  //       because the caller may want to read more bytes from the stream !
{
  int i,n,nBytesRcvdNow;
  int nBytesToRead, nBytesRead = 0;
  int nMillisecondsWaited = 0;
  int iWSAErrorCode;
  BOOL fRecvdBefore;
  unsigned long ulPrevTotalBytesReceived = pInetClient->ulTotalBytesReceived;

  if( pInetClient->sock==INVALID_SOCKET )
   { return -INET_ERROR_SOCKET_NOT_OPEN;
   }

  if( pInetClient->is_internet_resource )
   { if( pInetClient->iState != INET_CLIENT_STATE_WAIT_HTTP_RESP )
      { // Finished waiting for the GET-response ("headers"),
        // may already have read the first part of the "netto file data"
        // because the network buffer won't care for 'empty lines'
        // after the HTTP GET header(s) !
        // Bytes which don't belong to the HTTP GET response
        // but to the 'file data' remain in
        // pInetClient->c1kRxBuffer[0...pInetClient->nBytesInRxBuffer-1] .
        if( pInetClient->nBytesInRxBuffer > 0 )
         { // before attempting to RECEIVE more (from the socket),
           // deliver those data from the internal RX-buffer :
           nBytesToRead = pInetClient->nBytesInRxBuffer;
           if( nBytesToRead > iMaxLength )
            {  nBytesToRead = iMaxLength;
            }
           if( nBytesToRead > 0 )
            {  memcpy( pbDest, pInetClient->c1kRxBuffer, nBytesToRead );
               pbDest     += nBytesToRead;
               nBytesRead += nBytesToRead;
               iMaxLength -= nBytesToRead;
               if( iMaxLength>0 )
                { *pbDest = 0;  // append a trailing zero, as a service for TEXT STRINGS
                }
               n = pInetClient->nBytesInRxBuffer - nBytesToRead;
               if( n>0 )  // some "non-header-bytes" still remain in the RX-buffer !
                { for(i=0; i<n; ++i)   // (this is unusual but .. who knows)
                   { pInetClient->c1kRxBuffer[i] = pInetClient->c1kRxBuffer[i+nBytesToRead];
                   }
                  pInetClient->nBytesInRxBuffer = n;
                }
               else  // the RX-buffer is now "completely empty"; no need to "move up" !
                { pInetClient->nBytesInRxBuffer = 0;
                }
            } // end if( nBytesToRead > 0 )
         } // end if( pInetClient->nBytesInRxBuffer > 0 )
        else
         { nBytesRead = nBytesRead;
         }
      } // end if( pInetClient->iState != INET_CLIENT_STATE_WAIT_HTTP_RESP )
   }

  if( iMaxLength>0 )  // still more bytes 'acceptable' by the caller ?
   { // Try to read the next N bytes from the remote host (file "contents") .
     // Note that, at this point, the socket is NON-BLOCKING which means
     // revc() will return IMMEDIATELY if there are no data waiting
     // in the network buffer !
     nMillisecondsWaited = 0;
     fRecvdBefore = FALSE;
     while( (nMillisecondsWaited <= iTimeout_ms)  &&  (iMaxLength>0) )
      { nBytesRcvdNow = recv(
              pInetClient->sock,   // [in] socket; descriptor identifying a connected socket
              (char*)pbDest,       // [out] (remaining) buffer for the incoming data
              iMaxLength,          // [in] (remaining) length of the above buffer
              0 );                 // [in] flags; specifies how the call is made
        if( nBytesRcvdNow>0 )  // hooray, successfully "received something" !
         { pInetClient->ulTotalBytesReceived += nBytesRcvdNow;
           nBytesRead += nBytesRcvdNow;
           pbDest     += nBytesRcvdNow;
           iMaxLength -= nBytesRcvdNow;
           if( iMaxLength>0 )
            { *pbDest = 0;  // append a trailing zero, as a service for TEXT STRINGS
            }
           fRecvdBefore = TRUE;
         }
        else // nothing received this time; wait a few milliseconds more..
         { if( nBytesRcvdNow < 0 ) // a NEGATIVE return from recv() indicates an error.. which ?
            { iWSAErrorCode = WSAGetLastError();
              if( iWSAErrorCode == WSAEWOULDBLOCK )  // winsock's way of saying 'nothing received' ?
               { // Not really an error. As usual, the documentation is poor.
                 // recv() often returned ZERO when nothing was received.
                 nBytesRcvdNow = 0;
               }
              else // not just 'nothing received' but a REAL ERROR.  Which ?
               { INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, iWSAErrorCode,
                      "Error from recv: %s",
                      INET_WinsockErrorCodeToString(iWSAErrorCode) );
                 break;
               }
            }
           if( fRecvdBefore  // nothing received now, but something received before
             && ( (iOptions & INET_RD_OPTION_WAIT_FOR_ALL_DATA) == 0 ) )
            { // -> network buffer seems to be empty; "done" !
              break;
            }
           else
            { // Wait a few milliseconds more, typically for the 'initial response'
              if( iTimeout_ms > 0 )
               {
                 Sleep(50/*ms*/);   // .. give the remote server ("host") time to reply
                 nMillisecondsWaited += 50/*ms*/;
               }
              else // call is made "completely non-blocking" :
               { break;
               }
            }
         }
      } // end while < more bytes to read from the remote host >
   } // if( iMaxLength>0 )

  if( (ulPrevTotalBytesReceived==0) && (pInetClient->ulTotalBytesReceived>0) )
   {
    INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_INFO, 0/*progress*/,
           "Received first %d bytes", (int)pInetClient->ulTotalBytesReceived );
   }

  return nBytesRead;
} // end INET_ReadDataFromServer()


/****************************************************************************/
static int INET_SendImmediately(  // "continues" writing/sending if necessary
     T_INET_Client *pInetClient,    // [in,out] web client data
     int  iTimeout_ms,              // [in] max. timeout in milliseconds
     int  iOptions,                 // [in] combination of INET_WR_OPTION_....
     BYTE *pbSource, int iLength)   // [in] source data, and number of bytes to send
  // Beware of the goddamned Nagle algorithm .. only send as large blocks as possible !              
  // Return value : number of bytes actually written, or a NEGATIVE error code
  //        compatible with c:\cbproj\SoundUtl\ErrorCodes.cpp::ErrorCodeToString()
{
  int iErrorCodeOrNumBytesWritten = 0;
  int nLoops = 20;   // don't try to send() more than this number of times
  int i,n,nBytesRcvdNow;
  int nBytesToRead, nBytesRead = 0;
  int nMillisecondsWaited = 0;
  int iWSAErrorCode;
  BOOL fRecvdBefore;

  if( pInetClient->sock==INVALID_SOCKET )
   { return -INET_ERROR_SOCKET_NOT_OPEN;
   }


  if( (pbSource!=NULL) && (iLength>0) )  // is there really something to send ?
   {
     if( pInetClient->is_internet_resource )
      { // It's indeed an 'internet' connection, most likely TCP/IP-based ...
        switch( pInetClient->iProtocol )
         { case INET_PROTOCOL_HTTP    :
           case INET_PROTOCOL_RAW_TCP :
              while( (iLength > 0) && (nLoops>0) )
               { --nLoops;
                 n = send( pInetClient->sock, pbSource, iLength, 0/*flags*/ );
                 if( n>0 )  // ok, successfully "sent something" !
                  { pInetClient->ulTotalBytesSent += n;
                    iLength -= n;
                    pbSource+= n;
                    iErrorCodeOrNumBytesWritten += n;
                  }
                 else // nothing sent this time; wait a few milliseconds more.. it's a NON-BLOCKING socket !
                  {
                    iWSAErrorCode = WSAGetLastError();
                    // Got here REPEATEDLY with iWSAErrorCode == 10035 == WSAEWOULDBLOCK;
                    // no matter how long we waited. That's not what a socket should do !
                    switch( iWSAErrorCode )
                     { case WSAEWOULDBLOCK : // winsock's way of saying 'TX-buffer it full' ?
                          // Not really an error. Most likely, the network buffer is full .
                          // If the caller wants to, wait a few milliseconds here.
                          if( iTimeout_ms > 0 )
                           {
                             Sleep(50/*ms*/);   // .. give the remote server ("host") time to reply
                             nMillisecondsWaited += 50/*ms*/;
                           }
                          else // call is made "completely non-blocking" :
                           { iErrorCodeOrNumBytesWritten = -INET_ERROR_CANNOT_SEND;
                             nLoops = 0;
                           }
                          break;
                        // everything else must be a REAL error. The question is which.
                        // After the 'WSAWOULDBLOCK'-thingy, the next error we often got
                        // was iWSAErrorCode == 10053 == WSAECONNABORTED .
                        case WSAECONNABORTED :
                          // The windoze documentation had this to say:
                          // > Software caused connection abort.
                          // > An established connection was aborted by the software
                          // > in your host computer, possibly due to
                          // > a data transmission time-out or protocol error.
                          // > The application should close the socket as it is no longer usable.
                          // In fact, got here when trying to send Ogg/Vorbis
                          //  to   67.207.139.49:8901  ,
                          //  and Paul's server replied with TCP 'RST' flag:
                          // > Some host TCP stacks may implement a half-duplex close sequence,
                          // > as Linux or HP-UX do. If such a host
                          // > actively closes a connection but still has not
                          // > read all the incoming data the stack already received
                          // > from the link, this host sends a RST instead of a FIN.
                          INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, iWSAErrorCode,
                              "Error in send: Connection aborted." );
                          iErrorCodeOrNumBytesWritten = -INET_ERROR_CONNECTION_ABORT;
                          nLoops = 0;
                          INET_CloseClient( pInetClient );
                          pInetClient->fConnectionLost = TRUE;  // flag for the application
                          break;

                        case WSAECONNRESET :
                          // > An existing connection was forcibly closed by the remote host.
                          INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, iWSAErrorCode,
                              "Error in send: Connection reset (by remote host). Closing socket." );
                          iErrorCodeOrNumBytesWritten = -INET_ERROR_CONNECTION_RESET;
                          nLoops = 0;
                          INET_CloseClient( pInetClient );
                          pInetClient->fConnectionLost = TRUE;  // flag for the application
                          break;

                        default :
                          // Some other kind of most likely 'unrecoverable' error :
                          INET_InvokeCallback( pInetClient, INET_EVENT_TYPE_ERROR, iWSAErrorCode,
                              "Error from send: %s . Closing socket.",
                              INET_WinsockErrorCodeToString(iWSAErrorCode) );
                          iErrorCodeOrNumBytesWritten = -INET_ERROR_CONNECTION_ABORT;
                          nLoops = 0;
                          INET_CloseClient( pInetClient );
                          pInetClient->fConnectionLost = TRUE;  // flag for the application
                          break;
                     } // end switch( iWSAErrorCode )
                  } // end else <iErrorCodeOrNumBytesWritten<=0>
                 if( nMillisecondsWaited  > iTimeout_ms )
                  { break;
                  }
               }
              break; // end case INET_PROTOCOL_HTTP
           default:  // any other protocol isn't supported here yet !
              break;
         } // end switch( pInetClient->iProtocol )
      } // end if( pInetClient->is_internet_resource )
   } // end if < valid source >

  return iErrorCodeOrNumBytesWritten;
} // end INET_SendImmediately()



/****************************************************************************/
CPROT int INET_WriteDataToServer(  // "continues" writing/sending if necessary
     T_INET_Client *pInetClient,    // [in,out] web client data
     int  iTimeout_ms,              // [in] max. timeout in milliseconds
     int  iOptions,                 // [in] combination of INET_WR_OPTION_....
     BYTE *pbSource, int iLength)   // [in] source data, and number of bytes to send
  // To minimize the number of calls of send() ,
  //    the caller may specify the following flags in iOptions :
  //    INET_WR_OPTION_DONT_FLUSH : "Dont try to send already,
  //                                 I know there's more to send in the next call".
  //    INET_WR_OPTION_FLUSH_NOW  : "Send any data waiting in the buffer" .
  //    If none of the flags is set, the data-to-be may(!) be sent without buffering,
  //    especially when the block is 'quite long'( iLength > ? kByte) .
  //
  // Return value : number of bytes actually written, or a NEGATIVE error code
  //        compatible with c:\cbproj\SoundUtl\ErrorCodes.cpp::ErrorCodeToString()
{
  int iErrorCodeOrNumBytesWritten = 0;
  int n, nLoops = 2;

  if( pInetClient->sock==INVALID_SOCKET )
   { return -INET_ERROR_SOCKET_NOT_OPEN;
   }


  if( (pbSource!=NULL) && (iLength>0) )  // is there really something to send ?
   {
     // If there's already 'something' in the buffer, we cannot bypass it !
     // If the caller says 'INET_WR_OPTION_DONT_FLUSH' we also wont bypass the buffer.
     if(    (pInetClient->nBytesInTxBuffer > 0)
         || (iOptions & INET_WR_OPTION_DONT_FLUSH ) )
      {
        // Will the 'new' data fit inside the internal buffer ?
        while( ( (pInetClient->nBytesInTxBuffer + iLength) > sizeof(pInetClient->cTxBuffer) )
              && (nLoops>0)  )
         { // The 'new' data won't fit into the buffer, so flush the buffer FIRST:
           iErrorCodeOrNumBytesWritten = INET_SendImmediately(  // "continues" writing/sending if necessary
                  pInetClient, iTimeout_ms, iOptions,
                  pInetClient->cTxBuffer, pInetClient->nBytesInTxBuffer );
           if( iErrorCodeOrNumBytesWritten < 0 )
            { return iErrorCodeOrNumBytesWritten;
            }
           else // iErrorCodeOrNumBytesWritten >= 0
            { pInetClient->nBytesInTxBuffer -= iErrorCodeOrNumBytesWritten;
              if( pInetClient->nBytesInTxBuffer > 0 )
               { // oops.. the damned buffer couldn't be flushed COMPLETELY !
                 // This shouldn't happen, but it WILL (occasionally) !
                 // Did you know that memcpy() may fail if souce+dest overlap ? Thus:
                 memmove( pInetClient->cTxBuffer/*dest*/,
                    pInetClient->cTxBuffer+iErrorCodeOrNumBytesWritten/*src*/,
                    pInetClient->nBytesInTxBuffer); /* number of bytes to move*/
               }
              else // Good news: successfully flushed the buffer completely.
               { pInetClient->nBytesInTxBuffer = 0;
               }
            }
           --nLoops;
         } // end while

      } // end if( pInetClient->nBytesInTxBuffer > 0 )

     // Append the 'new' data to the buffer if possible :
     if( (pInetClient->nBytesInTxBuffer + iLength) < sizeof(pInetClient->cTxBuffer) )
      { // the 'new' data fit inside the buffer, so append them to those data
        // which still wait for transmission:
        memmove( pInetClient->cTxBuffer + pInetClient->nBytesInTxBuffer/*dest*/,
                 pbSource, iLength );
        pInetClient->nBytesInTxBuffer += iLength;
        iErrorCodeOrNumBytesWritten = iLength;
        iLength = 0;
      }
   } // end if < valid source >

  // "Send any data waiting in the buffer" now ?
  if( iOptions & INET_WR_OPTION_FLUSH_NOW )
   { iErrorCodeOrNumBytesWritten = INET_SendImmediately(  // "continues" writing/sending if necessary
                  pInetClient, iTimeout_ms, iOptions,
                  pInetClient->cTxBuffer, pInetClient->nBytesInTxBuffer );
     if( iErrorCodeOrNumBytesWritten < 0 )
      { return iErrorCodeOrNumBytesWritten;
      }
     else
     if( iErrorCodeOrNumBytesWritten == pInetClient->nBytesInTxBuffer )
      { // Bingo, the internal buffer is now COMPLETELY empty :
        pInetClient->nBytesInTxBuffer = 0;
        if( (pbSource!=NULL) && (iLength>0) ) // more data which did NOT fit inside the buffer ?
         { iErrorCodeOrNumBytesWritten = INET_SendImmediately(  // "continues" writing/sending if necessary
               pInetClient, iTimeout_ms, iOptions,
               pbSource,    iLength );
           if( iErrorCodeOrNumBytesWritten < 0 )
            { return iErrorCodeOrNumBytesWritten;
            }
           else
           if( iErrorCodeOrNumBytesWritten == iLength )
            { // Bingo-bingo; also flushed out everything which was passed in NOW !
              return iLength;
            }
           else // sent some bytes, but not ALL :
            {   // copy the remaining (un-sent) bytes into the buffer
                // even though the caller asked us to 'send them all' .
                // (this happens a bit further below, common code)
            }
         }
        else // successfully flushed all, and there's nothing remaining to send:
         { return iErrorCodeOrNumBytesWritten; // here: number of 'old' bytes flushed
         }
      }
     else  // argh ... sent something but not ALL .  Shame on winsock !
      { // In that case, we can NOT send the remaining <iLength> bytes now !
        // Append the remaining bytes into the internal buffer and return.
      }
   } // end if( iOption & INET_WR_OPTION_FLUSH_NOW )

  if( (pbSource!=NULL) && (iLength>0) ) // STILL something remaining which could NOT be sent yet ?
   {
     if( (iLength+pInetClient->nBytesInTxBuffer) <= sizeof(pInetClient->cTxBuffer) )
      { memmove( pInetClient->cTxBuffer + pInetClient->nBytesInTxBuffer,
                 pbSource, iLength );
        pInetClient->nBytesInTxBuffer += iLength;
      }
     if( iOptions & INET_WR_OPTION_FLUSH_NOW )
      {  return 0;  // let the caller know we were unable to send NOW !
      }
     else
      {  return iLength;  // let the caller know we 'accepted' his new data
                          // (even though they were not sent yet)
      }
   } // end if < valid source >

  return iErrorCodeOrNumBytesWritten;
} // end INET_WriteDataToServer()



/****************************************************************************/
void INET_CloseClient(
        T_INET_Client *pInetClient )   // [in,out] web client data
{
  if( pInetClient != NULL )
   { if( pInetClient->sock!=INVALID_SOCKET )
      {
        INET_InvokeCallback( pInetClient,
             INET_EVENT_TYPE_INFO, 0/*progress or event*/,
             "Closing socket connection" );
        // ex: close(sock); // NOT FOR WINDOZE ! A "SOCKET" is a not "FILE HANDLE" !
        // A socket is a socket, and a file is a file. They are DIFFERENT WORLDS !
        // From the Borland C++Builder help files:
        // > When a session has been completed, a closesocket must be performed.
        closesocket( pInetClient->sock );
        pInetClient->sock = INVALID_SOCKET; // don't try to close a 2nd time
      }
     pInetClient->nBytesInRxBuffer = pInetClient->nBytesInTxBuffer = 0;
   }
} // end INET_CloseClient()


/* EOF < YHF_Inet.c > */
