/*------------------------------------------------------------------*/
/*  YHF_Comm.h                                                      */
/*                                                                  */
/* DL4YHF's inter-application  communication module.                */
/*                                                                  */
/*       Uses windows' WM_COPYDATA-Message mechanism                */
/*       to pass data from one application to another.              */
/*                                                                  */
/*       (c) 2001..2004 by Wolfgang Buescher                        */
/*                                                                  */
/* Autor: Wolfgang Buescher (www.qsl.net/DL4YHF)                    */
/*                                                                  */
/* Some "registered" programs which use module YHF_Comm ,           */
/*      along with their "window names" (for YHF_FindWindowByName): */
/*  - "SpectrumLab" : DL4YHF's audio spectrum analyzer              */
/*  - "RdfCalc"     : DL4YHF's Radio-Direction-Finder + map plotter */
/*  - "CanTestW"    : WoBu/DL4YHF's CAN-Tester for Windows  (QRL)   */
/*                                                                  */
/* Revision history (YYYY-MM-DD):                                   */
/*  2004-10-01 :                                                    */
/*        Minor corrections (YHF_FindWindowByName) + clarifications */
/*  2003-09-08 :                                                    */
/*        Used for communication between SpectrumLab and RdfCalc .  */
/*  2001-08-30 :                                                    */
/*        Now also used at DL4YHF's QRL (won't tell you for what..) */
/*  2001-08-27 :                                                    */
/*        Started as private experiment in DL4YHF's Spectrum Lab.   */
/*                                                                  */
/*------------------------------------------------------------------*/

#ifndef _YHF_COMM_  // avoid multiple inclusion
 #define _YHF_COMM_ 1

//------------ Defines ------------------------------------------------------
#define YHF_CMD_SOFTWARE_VERSION  1

  // Note on queue length: the more applications can simultaneously
  //  send commands to this application, the more queue entries you
  //  need. QUEUE_LENGTH=20 should be more than enough for 5 "users".
#define YHF_CMD_QUEUE_LENGTH    20

  // The size of the largest data block which can be passed in a
  //  single windows message should be defined here (with a little
  //  safety margin):
#define YHF_CMD_MAX_DATA_SIZE 1024


  // Some return values for the SendMessage()-routine. DON'T MODIFY.
#define YHF_CMD_RESULT_TRUE  1
#define YHF_CMD_RESULT_FALSE 0 // following microsoft's guidelines for WM_COPYDATA
// The following bitmasks can be ORed to the RESULT value in a WM_COPYDATA msg,
// to indicate "immediate results" etc.
// "Immediate" means, there will definitely not be a "responding" windows
//  message later.
// Can only be used by HandleCopydataMessage() itself before WndProc returns.
// DO NOT CHANGE WITHOUT INCREMENTING THE SOFTWARE VERSION NUMBER !!!
#define YHF_CMD_RESULT_IMMEDIATE       0x80000000
#define YHF_CMD_RESULT_IMMEDIATE_WORD  0x81000000
#define YHF_CMD_RESULT_IMMEDIATE_INT16 0x82000000
#define YHF_CMD_RESULT_IMMEDIATE_ERROR 0x83000000
#define YHF_CMD_RESULT_ERROR_SIZE      0x83000001



//------------ Data Types ---------------------------------------------------
#ifndef BYTE
 #define BYTE unsigned char    /*  8 Bit !!!! */
 // fuer WINDOZE: vgl. Definitionen in <windef.h> ...
 // typedef unsigned char       BYTE;
#endif

#ifndef SIGNED_BYTE
 #define SIGNED_BYTE signed char /* 8 Bit mit Vorzeichen */
#endif

#ifndef  INT8
 #define INT8 signed char   /* 8 Bit mit Vorzeichen  */
#endif

#ifndef  INT16
 #define INT16 short int    /* 16 Bit mit Vorzeichen */
#endif

#ifndef  INT32
 #define INT32 signed long /* "long" ist DERZEIT immer 32 Bit Integer */
#endif

#ifndef WORD
 #define WORD unsigned short int /* 16 Bit beim C++Builder!*/
#endif

#ifndef LWORD
 #define LWORD unsigned long   /* 32 Bit !!!! */
#endif

typedef union { // 8/16/32-bit - adressable "long"-data type
    BYTE b[4];
    WORD w[2];
    INT32 l;
    LWORD ul;
} T_YHF_BLONG;

typedef union
{ BYTE  bData[YHF_CMD_MAX_DATA_SIZE];     // block of BYTES (or other formats)
  char  cData[YHF_CMD_MAX_DATA_SIZE];     // array of CHARS (8 bit each)
  WORD  wData[YHF_CMD_MAX_DATA_SIZE/2];   // array of WORDs (16 bit)
  INT32 lData[YHF_CMD_MAX_DATA_SIZE/4];   // array of LONGs (32 bit)

  // .. add your own data types here to SEND them in a WM_COPYDATA message ..

} T_YHF_DATA_UNION;

typedef struct
{ char Module1;  // 2-letter module for switch-list,  like "CL"=Command Line interpreter
  char Module2;
  char Command1; // 2-letter command for switch-list, like "EC"=execute command
  char Command2;
} T_YHF_CMD_CHARS;

typedef union
{ T_YHF_CMD_CHARS c;
  T_YHF_BLONG bl;
} T_YHF_CMD_UNION;


typedef struct
{
  LWORD dwSender;  // window handle of the application which sent this command

  // The next 4 bytes are carried as "COPYDATASTRUCT.dwData"
  // between all applications programmed by DLYHF (=WoBu).
  // They are often interpreted as 4 characters, with the first two
  // identifying the MODULE and the last two the COMMAND itself.
  // How they shall be interpreted, depends on the application
  //  (which is outside the sope of this module).
  T_YHF_CMD_UNION cu;


  // The contents and format of the data passed in a COPYDATASTRUCT
  // depend very much on MODULE and COMMAND :
  long lDataSize;
  T_YHF_DATA_UNION du;

} T_YHF_COMMAND;


//------------ Variables ----------------------------------------------------
extern HWND YHF_MyWindowHandle;  // only for diagnostic purposes - YOU shouldn't need this !


//------------ Prototypes ---------------------------------------------------

/****************************************************************************/
void YHF_InitMessageHandler( HWND hwndMyMainWindow );


/****************************************************************************/
long YHF_GetMyWindowHandle( void );

/****************************************************************************/
BOOL YHF_CreateCommWindow(char *pszWindowClassName, WNDPROC pWndProc );
void YHF_DestroyCommWindow( void );
   // Creates/destroys an invisible top-level window which can act as a receiver
   // for WM_COPYDATA messages.  For details, see YHF_Comm.cpp !


/****************************************************************************/
long  YHF_HandleCopydataMessage(
      long  dwSender, // identifies the window passing the data via WM_COPYDATA
      long  dwData,   // 32 bits of data passed in COPYDATASTRUCT.dwData
      long  cbData,   // size of datablock in bytes
      void* lpData ); // pointer to datablock. THIS BLOCK IS READ-ONLY !!!
  // Application specific handler for received WM_COPYDATA messages.
  // Used to communicate between different applications.
  //
  // For implementation notes see Win32 Programmer's Reference on WM_COPYDATA.
  //
  // General Info (no application specific details) :
  // This routine is called from a message handler after an other application
  // has SENT a WM_COPYDATA message to this application (using SendMessage,
  // not PostMessage so be careful..).
  // This routine should return AS SOON AS POSSIBLE, and it should NOT send
  // an answering message itself (because the other -sending- application
  // is blocked in SendMessage() until THIS routine returns).
  //     In Borland C++Builder Applications, look for a WindowProc method
  //     of the main window to see the caller of this subroutine.
  // The return value of this routine will be the RESULT value
  // of the calling application from SendMessage().
  // Microsoft suggests to return only TRUE or FALSE,
  //  but WoBu decided this would be a waste of 31 bits..


/****************************************************************************/
T_YHF_COMMAND *YHF_PeekCommandMessage(
    char cModule1, char cModule2,  // MODULE identification (two letters)
    BOOL fRemove );
  // Checks the internal "command message" queue
  // for a message and places the message (if any) in the specified structure.
  // In contrast to the Win API's PeekMessage function, the command(message)
  // will not be removed from the internal queue until RemoveCommandMessage
  // is called (this is necessary to avoid COPYING LARGE DATA BLOCKS).
  // If a message is not removed from the queue within a few seconds,
  //  it will be removed automatically to avoid blocking the queue.
  // Return:  NULL = no message found
  //          <>   = message found, returns a pointer to the message in RAM.
  //                 DONT FORGET TO REMOVE THIS MESSAGE AFTER PROCESSING IT !


/****************************************************************************/
void YHF_RemoveCommandMessage(  T_YHF_COMMAND *pCmd );
  // Removes a "command message" from the internal queue.
  // See PeekCommandMessage() for details.



/****************************************************************************/
long YHF_SendCommandMessage(
           long hDestWindow,  // handle of destination window
       T_YHF_COMMAND *pCmd ); // pointer to message being sent
  // Sends a "command message" to an other application.
  //
  // The caller must identify himself by entering
  //     the HANDLE OF HIS MAIN WINDOW in pCmd->dwSender.
  //
  // This routine does not return IMMEDIATELY !
  //  It returns after the message has been sent to
  //  and processed by   the receiver.
  //  The return value is the result code (~acknowledge)
  //  which the partner returned in a windows message.
  // See PeekCommandMessage() for details how this works
  // on the "other side".


/****************************************************************************/
long YHF_FindWindowByName( char *pszWindowName );
  // see YHF_Comm.cpp for details !

#endif // _YHF_COMM_  to avoid multiple inclusion

// EOF <YHF_Comm.h>
