/*------------------------------------------------------------------*/
/*  YHF_Comm.cpp                                                    */
/*                                                                  */
/* DL4YHF's inter-application  communication module.                */
/*                                                                  */
/*       Uses windows' WM_COPYDATA-Message mechanism                */
/*       to pass data from one application to another.              */
/*                                                                  */
/*       (c) 2001..2004 by Wolfgang Buescher                        */
/*                                                                  */
/* Autor: Wolfgang Buescher (www.qsl.net/DL4YHF)                    */
/*                                                                  */
/* Some "registered" programs which use module YHF_Comm ,           */
/*      along with their "window names" (for YHF_FindWindowByName): */
/*  - "SpectrumLab" : DL4YHF's audio spectrum analyzer, 1st instance*/
/*  - "RdfCalc"     : DL4YHF's Radio-Direction-Finder + map plotter */
/*  - "CanTestW"    : WoBu/DL4YHF's CAN-Tester for Windows  (QRL)   */
/*                                                                  */
/* Revision history (YYYY-MM-DD):                                   */
/*  2004-10-01 :                                                    */
/*        Minor corrections (YHF_FindWindowByName) + clarifications */
/*  2003-09-08 :                                                    */
/*        Used for communication between SpectrumLab and RdfCalc .  */
/*  2001-08-30 :                                                    */
/*        Now also used at DL4YHF's QRL (won't tell you for what..) */
/*  2001-08-27 :                                                    */
/*        Started as private experiment in DL4YHF's Spectrum Lab.   */
/*                                                                  */
/*------------------------------------------------------------------*/


/* NO VCL IN THIS UNIT !!!!! (or other BORing stuff) */
// #include <stdio.h>
// #include <io.h>
// #include <fcntl.h>
// #include <stdarg.h>
#include <string.h>
// #include <stdlib.h>
#include <ctype.h>
// #include <time.h>
// #include <dos.h>
#include <windows.h>
// #include <math.h>

#pragma hdrstop


#include "YHF_Comm.h"  // header for this module

//------------ internal data types ------------------------------------------
typedef struct
{
  int status;
  T_YHF_COMMAND cmd;
} T_YHF_CMD_LIST_ENTRY;


//------------ Variables ----------------------------------------------------
T_YHF_CMD_LIST_ENTRY YHF_CommandList[YHF_CMD_QUEUE_LENGTH];
int YHF_CommandListIndexIn = 0;
int YHF_CommandListCounter = 0;  // total counter for command list usage
HWND YHF_MyWindowHandle = NULL;
char YHF_sz80MyCommsWindowClassName[84] = "";
HWND YHF_MyCommsWindowHandle = NULL;



//------------ Implementation of Routines -----------------------------------

/****************************************************************************/
void YHF_InitMessageHandler( HWND hwndMyMainWindow )
{
  YHF_MyWindowHandle = hwndMyMainWindow;
  memset(YHF_CommandList, 0, sizeof(YHF_CommandList) );
  YHF_CommandListIndexIn = 0;
  YHF_CommandListCounter = 0;  // total counter for command list usage
} // end InitMessageHandler()


/****************************************************************************/
long YHF_GetMyWindowHandle( void )
{
  return (long)YHF_MyWindowHandle;
}

/****************************************************************************/
BOOL YHF_CreateCommWindow(char *pszWindowClassName, WNDPROC pWndProc )
   // Creates an invisible top-level window which can act as a receiver
   // for WM_COPYDATA messages.  In contrast to the visible main window
   // (created by Borland's VCL or something), the name of this inivisible
   // 'communication window' can be freely specified by the application.
   // First used in DL4YHF's Spectrum Lab to have multiple instances running
   // with DIFFERENT CLASS NAMES at the same time .
{
  if( YHF_MyCommsWindowHandle != NULL )
   {  YHF_DestroyCommWindow();
   }

  WNDCLASS wc = {
        0,                              // style
        pWndProc?pWndProc:DefWindowProc,// lpfnWndProc
        0,                              // cbClsExtra
        0,                              // cbWndExtra
        NULL,                           // this file's HINSTANCE
        NULL,                           // hIcon
        LoadCursor(NULL, IDC_ARROW),    // hCursor
        (HBRUSH)(COLOR_BTNFACE+1),      // hbrBackground
        NULL,                           // lpszMenuName
        TEXT(pszWindowClassName),       // lpszClassName
    };

  if( RegisterClass(&wc) )
   { // Successfully registered a new class with the given name . Save the name:
     strncpy( YHF_sz80MyCommsWindowClassName , pszWindowClassName, 80 );
     // Now create an invisible window using it ...
     YHF_MyCommsWindowHandle = CreateWindow(
       pszWindowClassName, // LPCTSTR lpClassName,  pointer to registered class name
       NULL,               // LPCTSTR lpWindowName, pointer to window name
       WS_OVERLAPPED,      // DWORD dwStyle, window style
       0,                  // int x,      horizontal position of window
       0,                  // int y,      vertical position of window
       0,                  // int nWidth, window width
       0,                  // int nHeight, window height
       NULL,        // HWND hWndParent, handle to parent or owner window
       NULL,        // HMENU hMenu,     handle to menu or child-window identifier
       NULL,        // HANDLE hInstance,  handle to application instance
       NULL );      // LPVOID lpParam, pointer to window-creation data
     // Note that this window will NOT be sub-classed, because it shall be
     // a "top level window" - otherwise FindWindow() will not find it !
     return (YHF_MyCommsWindowHandle != NULL);
   }
  return FALSE;

} // end YHF_CreateCommWindow()

/****************************************************************************/
void YHF_DestroyCommWindow( void )
{
  if( YHF_MyCommsWindowHandle != NULL )
   {  DestroyWindow( YHF_MyCommsWindowHandle );
      YHF_MyCommsWindowHandle = NULL;
   }
  if( YHF_sz80MyCommsWindowClassName[0]>0 )
   {  UnregisterClass( YHF_sz80MyCommsWindowClassName, NULL );
      YHF_sz80MyCommsWindowClassName[0] = 0;
   }
} // end YHF_DestroyCommWindow()

/****************************************************************************/
long YHF_HandleCopydataMessage(
      long  dwSender, // identifies the window passing the data via WM_COPYDATA
      long  dwData,   // 32 bits of data passed in COPYDATASTRUCT.dwData
      long  cbData,   // size of datablock in bytes
      void* lpData )  // pointer to datablock. THIS BLOCK IS READ-ONLY !!!
  // Application specific handler for received WM_COPYDATA messages.
  // Used to communicate between different applications.
  //
  // For implementation notes see Win32 Programmer's Reference on WM_COPYDATA.
  //
  // General Info (no application specific details) :
  // This routine is called from a message handler after an other application
  // has SENT a WM_COPYDATA message to this application (using SendMessage,
  // not PostMessage so be careful..).
  // This routine should return AS SOON AS POSSIBLE, and it should NOT send
  // an answering message itself (because the other -sending- application
  // is blocked in SendMessage() until THIS routine returns).
  //     In Borland C++Builder Applications, look for a WindowProc method
  //     of the main window to see the caller of this subroutine.
  // The return value of this routine will be the RESULT value
  // of the calling application from SendMessage().
  // Microsoft suggests to return only TRUE or FALSE.
{
  // Application specific details about the usage of the WM_COPYDATA-
  // message in W.B.'s CAN-TESTER FOR WINDOWS:
  //  - Serves as a gateway for other applications to the tester's
  //    routines, especially for future extensions
  //    like external test scripts, download utility, etc.
  //  - The dwData parameter is used as a primary "sub-divider"
  //    for incoming command requests. To avoid the need for a C specific
  //    header file containing command definitions, we use a character-based
  //    system.
  //  - The lpData block is usually copied into an internal queue,
  //    using a structure which is different from the windoze stuff.
  //    (making it easier to migrate from windowe to something else)
  //  - The handling of the queued data takes place in the applications
  //    idle method or in a timer-method, where we have enough time to
  //    create an answering windows message (which returns the data from a
  //    READ-command, etc)
  // Notes:
  // - A similar application is in W.B.'s project "Spectrum Lab",
  //   see caller :  SpecLab_CommWindowProc()  in module SpecMain.cpp .
  // - At least under Win XP, an application can send a WM_COPYDATA-message
  //   to itself. This is a nice feature for testing purposes !
 T_YHF_BLONG   blCommand;
 T_YHF_COMMAND *pCmd;
 int   i;

  if(cbData>YHF_CMD_MAX_DATA_SIZE)
     return YHF_CMD_RESULT_ERROR_SIZE;

  blCommand.l = dwData; // to split the 32 bit "command" into 4 bytes :
  // b[0]="Module1", b[1]="Module2", b[2]="Command1",  b[3]="Command2" )
  if(blCommand.b[0] == 0)
   { // no valid Module letter:  message handled here ..
     switch(blCommand.b[2])  // branch on 1st command letter:
      {
        case 'l':  // command list index
             switch(blCommand.b[3]) // branch on 2nd command letter:
              { case 'i':   // command list index
                     return YHF_CMD_RESULT_IMMEDIATE_INT16
                          | YHF_CommandListIndexIn;
                case 'c':   // command list counter
                     return YHF_CMD_RESULT_IMMEDIATE_INT16
                          | YHF_CommandListCounter;
                default:
                     break;
              }
             return YHF_CMD_RESULT_FALSE;

        case 'v':  // Version of DL4YHF's communication handler
             return YHF_CMD_RESULT_IMMEDIATE_INT16
                          | YHF_CMD_SOFTWARE_VERSION;

        default:   // 1st command letter unknown - place message in queue !
             break;
      }
   } // end if < 1st command letter ZERO >

  // Try to enter the message into the queue.
  for(i=0; i<YHF_CMD_QUEUE_LENGTH; ++i)
   {
    if(YHF_CommandListIndexIn >= YHF_CMD_QUEUE_LENGTH)
       YHF_CommandListIndexIn = 0;
    if( YHF_CommandList[YHF_CommandListIndexIn].status == 0)
     { // bingo, found an unused index in the command message list.
       // EnterCriticalSection  ?  no, not really required - hopefully
       YHF_CommandListCounter++;
       YHF_CommandList[YHF_CommandListIndexIn].status = 1; // occupied
       pCmd = &YHF_CommandList[YHF_CommandListIndexIn].cmd;
       pCmd->dwSender = dwSender;
       pCmd->cu.c.Module1 = blCommand.b[0];
       pCmd->cu.c.Module2 = blCommand.b[1];
       pCmd->cu.c.Command1= blCommand.b[2];
       pCmd->cu.c.Command2= blCommand.b[3];
       if(cbData>YHF_CMD_MAX_DATA_SIZE/*1024 bytes?*/)
        { // Problem: must truncate the data block because it's too large !
          // ( This should never happen because passing large blocks
          //   in a WM_COPYDATA message is not what the message was made for..)
          cbData = YHF_CMD_MAX_DATA_SIZE;
        }
       if(cbData>0)
        { pCmd->lDataSize = cbData;
          memcpy( /*dest:*/pCmd->du.bData, /*source:*/lpData,  /*len:*/cbData);
        }
       else  // no data bytes in this message ?
        { pCmd->lDataSize = 0;
          pCmd->du.bData[0] = '\0';
        }
       // LeaveCriticalSection
       return YHF_CMD_RESULT_TRUE;
     } // end if <empty queue location found>
    else // no empty entry in the queue found..
     {
       YHF_CommandListIndexIn++;
     }
   } // end for..

  return YHF_CMD_RESULT_FALSE;  // couldn't handle WM_COPYDATA message...
} // end HandleCopydataMessage()


/****************************************************************************/
T_YHF_COMMAND *YHF_PeekCommandMessage(
    char cModule1, char cModule2, // MODULE identification (two letters)
    BOOL fRemove )  // remove message from buffer or leave it in there ?
  // Checks the internal "command message" queue
  // for a message and places the message (if any) in the specified structure.
  // In contrast to the Win API's PeekMessage function, the command(message)
  // will not be removed from the internal queue until RemoveCommandMessage
  // is called (this is necessary to avoid COPYING LARGE DATA BLOCKS).
  // If a message is not removed from the queue within a few seconds,
  //  it will be removed automatically to avoid blocking the queue.
  // Return:  NULL  = no message found
  //         <>NULL = message found, returns a pointer to the message in RAM.
  //                  DONT FORGET TO REMOVE THIS MESSAGE AFTER PROCESSING IT !
  // Usage examples:
  //   \cbproj\RDF_Calc\rdf_gui.cpp : uses this function for remote control
  //                  of the RDF Calculator (and its map display) through
  //                  external applications like Spectrum Lab .
{
 int i,j;
 T_YHF_COMMAND *pCmd;

  j = YHF_CommandListIndexIn;  // try to start with the OLDEST(!) entry

  // Try to get a command or message from the queue.
  for(i=0; i<YHF_CMD_QUEUE_LENGTH; ++i)
   {
     if( (++j) >= YHF_CMD_QUEUE_LENGTH)
        j = 0;
     if(YHF_CommandList[j].status > 0)
      {
       pCmd = &YHF_CommandList[j].cmd;
       if( (cModule1=='*' || (pCmd->cu.c.Module1==cModule1) )
        && (cModule2=='*' || (pCmd->cu.c.Module1==cModule1) ) )
        {  // found an entry matching the caller's module  OR WILDCARD.
         if(fRemove)
            YHF_CommandList[j].status = 0;
         return pCmd;
        }
      } // end if <found a used entry in the command / message queue>
   } // end for..

  return NULL;
} // end PeekCommandMessage()


/****************************************************************************/
void YHF_RemoveCommandMessage(  T_YHF_COMMAND *pCmd )
  // Removes a "command message" from the internal queue.
  // See PeekCommandMessage() for details.
{
 int i;
  if(  (pCmd >= &YHF_CommandList[0].cmd)
     &&(pCmd <= &YHF_CommandList[YHF_CMD_QUEUE_LENGTH-1].cmd) )
   {
    i = (pCmd - &YHF_CommandList[0].cmd) / sizeof(T_YHF_CMD_LIST_ENTRY);
    if (i>=0 && i<YHF_CMD_QUEUE_LENGTH)
     {
       YHF_CommandList[i].status = 0;
     }
   }
} // end RemoveCommandMessage()
//---------------------------------------------------------------------------


/****************************************************************************/
long YHF_SendCommandMessage(
           long hDestWindow,  // handle of DESTINATION window
       T_YHF_COMMAND *pCmd )  // pointer to message being sent
  // Sends a "command message" to an other application.
  //
  // The caller should identify himself by entering
  //     the HANDLE OF HIS MAIN WINDOW in pCmd->dwSender
  //     or set pCmd->dwSender to NULL (then the handle passed during
  //     module initialisation will be used)
  //
  // This routine does not return IMMEDIATELY !
  //  It returns after the message has been sent to
  //  and processed by   the receiver.
  //  The return value is the result code (~acknowledge)
  //  which the partner returned in a windows message.
  // See PeekCommandMessage() for details how this works
  // on the "other side".
{
 COPYDATASTRUCT cds;

  cds.dwData = pCmd->cu.bl.l;   // 4 characters: module + command
  cds.cbData = pCmd->lDataSize; // count of bytes in data block
  cds.lpData = pCmd->du.bData;  // pointer to data block

  return SendMessage( // "SEND", not "POST"  !
    (HWND)hDestWindow,      // HWND hWnd = handle of destination window
    WM_COPYDATA,            // UINT Msg  = message to send
    (WPARAM)pCmd->dwSender, // HANDLE OF SENDING WINDOW :
    (LPARAM)&cds ); // 2nd  msg parameter = pointer to COPYDATASTRUCT
  // Note: WM_COPYDATA can only be SENT, not POSTED !

} // end YHF_SendCommandMessage()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
long YHF_FindWindowByName( char *pszWindowName )
 // Returns the handle of a message-destination window,
 //  which can be used as parameter long hDestWindow
 //  when sending messages through YHF_SendCommandMessage() .
 // If the return value is 0 (zero), no such top-level window exists.
 //
 // Note for Borland C++ Builder  users :
 //    The windows API function FindWindow() is used for this purpose.
 //    It is described in the "Win32 Programmer's Reference" by MS .
 //    No fancy BORLAND STUFF is used here !
 //    The name of your program's window is the name of the
 //    instance variable of the "main form". In the sourcecode of the
 //    associated "main unit", you will see something like this for example:
 //    > //------------------------------
 //    > #pragma package(smart_init)
 //    > #pragma resource "MyOwnRes.res"
 //    > #pragma resource "*.dfm"
 //    > TSpectrumLab *SpectrumLab;
 //    In this example, "TSpectrumLab" is the name of a CLASS (descendant of a TForm),
 //      and "SpectrumLab" is the name of the instance which C++Builder
 //          creates automatically.
 //      Unfortunately the name of your WINDOW (which can be found by FindWindow() )
 //          would then be "TSpectrumLab", not "SpectrumLab".
 //          I didn't find a simple way to assign a name to the applications
 //          top-level window during runtime yet ... would be nicer if the
 //          window's name was "SpectrumLab" in this case, right ?
 //      To bypass this problem (at least a bit), the function
 //      YHF_FindWindowByName() cheats a little:
 //      If a window named <pszWindowName> cannot be found, try again
 //      with an upper case 'T' before it ("T"+pszWindowName) to let things
 //      look better for the user, who shall not have to care for this
 //      internal stuff. By the way this is one more reason why you should
 //      give your Borland-"forms" a descriptive name - use something better
 //      than the automatically generated name "TForm1" for your main form.
 //
 // The problem with "TWO OR MORE SPECTRUM LABS COMMUNICATION WITH EACH OTHER"
 // was fixed later, grep for YHF_CreateCommWindow() to see the details.
 // (The trick is to open an invisible window with an alternative class name,
 //  which can be done during runtime. Example (SpecLab):
 //  First instance = "SpecLab1",  2nd instance = "SpecLab2", etc . )
 //
 // Implemented 2003-09-08 .
 //
{
  char sz255[256];

 // The FindWindow function retrieves the handle to the top-level window
 // whose class name and/or window name match the specified strings.
 // This function does not search child windows.
 //
 long lResult = (long)FindWindow(
    pszWindowName, // pointer to class name, for example "SpectrumLab" .
    NULL );        // LPCTSTR lpWindowName = pointer to window name ("title"?!)
    // (ugly term: this is NOT NECESSARILY the window's TITLE -
    //  it may be the name of a 'form' created with Borland C++ Builder)

 if(lResult==0)    // Second attempt required ?  (since 2004-10-01)
  { // Didn't find the window this way. Try something else,
    // maybe the program was written with Borland C++ Builder.
    // In that case the window name will begin with a "T",
    // as explained in YHF_Comm.cpp, function YHF_FindWindowByName() .
    sz255[0]='T'; sz255[255]='\0';
    strncpy(sz255+1, pszWindowName, 254);
    lResult = (long)FindWindow( sz255, NULL ); // better luck this time ?
  } // end if < second attempt, especially for programs written with Borland C++Builder >

 return lResult;

} // end YHF_FindWindowByName( )
