/***************************************************************************/
/*  WaveIO.cpp:                                                            */
/*   Unit to import and export *.wav - files and similar..                 */
/*  - by DL4YHF August '2000  ...  February 2011                           */
/*  - based on the description of the WAVE-file format                     */
/*    found at http://www.wotsit.org  (excellent site with file formats!)  */
/*  - does NOT use any fancy windows gimmics, just standard file I/O.      */
/*  - has also been converted into a plain-vanilla "C" source in June 2003 */
/*     which compiles happily under Borland C for DOS, see \cproj\WaveTool */
/*  Info about wave files :                                                */
/*  http://www-mmsp.ece.mcgill.ca/documents/audioformats/wave/Samples.html */
/*    contained the following samples (used for testing THIS module) :     */
/*  -   M1F1-int24-AFsp.wav (138 kB)   :  WAVE file, stereo 24-bit data,   */
/*                  doesn't play in the stupid f**d windows media player,  */
/*                  but works properly when played back in Spectrum Lab .  */
/*                  See notes in Read24BitIntegerAsFloat() further below.  */
/*  -   M1F1-int24WE-AFsp.wav (138 kB) :  WAVE (WAVE_FORMAT_EXTENSIBLE),   */
/*                                        stereo 24-bit data.              */
/*                  Plays in windoze media player and SL;                  */
/*                  confirms the usual 'Intel' byte order (lo/mid/hi) .    */
/*                                                                         */
/*  Used in (at least) the following projects :                            */
/*     - DL4YHF's Spectrum Lab                                             */
/*      (DL4YHF's Curve Editor  uses a 'downsized' variant of this module, */
/*         see \cbproj\CurveEdit\LoadSaveWave.c )                          */
/*                                                                         */
/* Latest changes :                                                        */
/*   2011-02-13: Added WriteSamples_Float(), and changed the sequence      */
/*               of the "inf1" chunk (now AFTER the "fmt " chunk) because  */
/*            the files with the "inf1" chunk 1st didn't play under Linux. */
/*-------------------------------------------------------------------------*/

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#include <windows.h> // definition of data types like LONGLONG (wtypes.h) etc
#include <stdlib.h>
#include <stdio.h>
#include <io.h>          /* _rtl_open and other file routines       */
#include <fcntl.h>       /* O_RDONLY  and other file flags          */
#include <time.h>
#include "utility1.h"    // some helper routines by DL4YHF
#include "QFile.h"       // DL4YHF's "Quick File Routines" (tried this 2008-03)
// #include <exception>

#pragma hdrstop
#pragma warn -8004 // <var is a assigned a value that is never used> - so what

#define _I_AM_WAVE_IO_ 1
#include "WaveIO.h"  // header for this file with "single source"-variables

 // Found this GUID in an 'existing' wave file (pls55.wav), here as a BYTE array:
const BYTE WaveIO_b16GUID_PCM[16] = /* taken from an 'existing' wave file */
//  [0]  [1]  [2]  [3]  [4]  [5]  [6]  [7]  [8]  [9]  [10] [11] [12] [13] [14] [15]
  { 0x01,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x80,0x00,0x00,0xaa,0x00,0x38,0x9b,0x71 };
 // The MS site says it should be this "GUID" :
 //  00000001 - 0000 - 0010 - 8000 - 00aa00389b71
 // Note the strange different byte orders in bytes[6/7] and [8/9] !


extern double GuessExactFileSampleRate(double dblIndicatedSRate);

// Helper functions (no class methods, to make them easy to use...)

/***************************************************************************/
static T_Float ReadUns16Intel( short *sp )
{ return (T_Float)(*(unsigned short*)(sp)) - 32767.0;
} // end ReadUns16Intel()

/***************************************************************************/
static T_Float ReadSigned16Motorola( short *sp )
{ union { BYTE b[2];  short i16; } t16;
  t16.b[0] = ((BYTE*)sp)[1];
  t16.b[1] = ((BYTE*)sp)[0];
  return (T_Float)t16.i16;
} // end ReadSigned16Motorola()

/***************************************************************************/
static T_Float ReadUns16Motorola( short *sp )
{ union { BYTE b[2];  unsigned short u16; } t16;
  t16.b[0] = ((BYTE*)sp)[1];
  t16.b[1] = ((BYTE*)sp)[0];
  return (T_Float)t16.u16 - 32768.0;
} // end ReadUns16Motorola()


/***************************************************************************/
T_Float Read24BitIntegerAsFloat( BYTE *pbSource )
{
  union { BYTE b[4];  long i32; } t;

  // We'll always run on a machine with 'INTEL BYTE ORDER', so : b[0] = LSB  .
  // The following worked properly with a test file [ M1F1-int24-AFsp.wav ]  !
  t.b[0] = *pbSource++;   // bits 7..0
  t.b[1] = *pbSource++;   // bits 15..8
  t.b[2] = *pbSource  ;   // bits 23..16
  if( t.b[2] & 0x80 )     // expand the sign-bit to bits 31..24 :
       t.b[3] = 0xFF;
  else t.b[3] = 0x00;
  return (T_Float)t.i32 * ( 1.0 / 256.0 );  // normalize to +/- 32767 for historic reasons
             // (without sacrificing the 24-bit resolution ! ! )
} // end Read24BitIntegerAsFloat()


/***************************************************************************/
void Conv24BitSamplesToFloat(
        BYTE *pbSource,          // [in] source data, 24-bit signed integers, possibly 'n' channels interleaved
        int  nSamplePoints,      // [in] number of SAMPLE POINTS (with possibly more than one channel each)
        int  nSrcChannelsPerSample, // [in] number of channels per sample point IN THE SOURCE
        int  nDstChannelsPerSample, // [in] number of channels per sample point IN THE DESTINATION
        T_Float *pFltDest1,      // [out] 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
        T_Float *pFltDest2)      // [out] 2nd destination block (~"RIGHT"), may be NULL (!)
{
  int i;
  DWORD dwAddressIncrementPerSample = 3 * nSrcChannelsPerSample;
  //ex: if(   ( (nChannelsPerSample==2) && (pFltDest2==NULL) )
  //       || ( (nChannelsPerSample==4) && (pFltDest2!=NULL) )  )
  if( nDstChannelsPerSample>1 )
   { // two channels (usually I+Q) per destination block,
     // which means TWO destination array indices (in pFltDest*) per sample point
     for(i=0; i<nSamplePoints; i++ )
      { // up to 8 bits per sample, TWO SEPARATE destination blocks:
        *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // 1st channel ("I")
        *pFltDest1++ = Read24BitIntegerAsFloat( pbSource+3 );  // 2nd channel ("Q")
        if( (nSrcChannelsPerSample==4) && (pFltDest2!=NULL) )
         { *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+6 ); // 3rd channel ("I")
           *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+9 ); // 4th channel ("Q")
         }
        pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .
      }
   }
  else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
   { // (in other words, ONE destination array index per sample point)
     if( (nDstChannelsPerSample<2) || (pFltDest2==NULL) )
      { // only ONE channel to be copied...
        for(i=0; i<nSamplePoints; i++ )
         { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
           *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // one DESTINATION channel
           pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .

         }
      }
     else // (m_ChunkFormat.wChannels>=2) && (pFltDest2!=NULL) :
      {  // TWO channels must be copied (and "split" into two destination blocks) :
        for(i=0; i<nSamplePoints; i++ )
         { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
           *pFltDest1++ = Read24BitIntegerAsFloat( pbSource   );  // 1st DESTINATION channel
           *pFltDest2++ = Read24BitIntegerAsFloat( pbSource+3 );  // 2nd DESTINATION channel
           pbSource += dwAddressIncrementPerSample; // increment the 'byte'-pointer
                           // for the next sampling point in the SOURCE block .
         }
      }
   } // end else < not TWO channels per destination block >
} // end Conv24BitSamplesToFloat()


/*------------- Implementation of the class  C_WaveIO ---------------------*/

/***************************************************************************/
C_WaveIO::C_WaveIO()
  /* Constructor of the Wave-IO Class.
   * Initializes all properties etc.
   */
{
   memset( &m_RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
   memset( &m_ChunkHdr,    0 , sizeof(T_WAV_CHUNK_HDR)    );
   memset( &m_ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX) );

# if (WAVEIO_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
   m_QFile.iHandle    = -1;
# else
   m_FileHandle       = -1;
# endif
   m_OpenedForReading = FALSE;
   m_OpenedForWriting = FALSE;
   m_dblStartTime     = 0.0;
   m_dwCurrFilePos    = 0;
   m_dwFileDataSize   = 0;
   m_iFileFormat      = AUDIO_FILE_FORMAT_UNKNOWN; // RAW (*.dat, *.raw), RIFF (*.wav), or SETI's "raw" (*.dat) ?
   m_iAudioFileOptions= AUDIO_FILE_OPTION_NORMAL;  // see c:\cbproj\SoundUtl\AudioFileDefs.h
   m_iBitsPerSample   = 16;         // just a "meaningful default" !
   m_cErrorString[0]  = '\0';

   m_dbl_SampleRate = 11025.0;      // initial guess, only important for RAW files
   m_iSizeOfDataType = 1;     // just a GUESS for reading RAW files,
   m_wRawFileNumberOfChannels = 1;  // should be properly set before opening such files
   m_iNumChannels    = 1;
   m_fMustWriteHeader = FALSE;

} // end of C_WaveIO's constructor

/***************************************************************************/
C_WaveIO::~C_WaveIO()
  /* Destructor of the Wave-IO Class.
   *  - Flushes all output files of this class (if any)
   *  - Closes all handles (if still open)
   *  - Frees all other resources that have been allocated.
   */
{


} // end C_WaveIO's destructor


/***************************************************************************/
BOOL C_WaveIO::InOpenInternal( char * pszFileName,
                BOOL fReOpenForWriting,
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength ) // [in] size of pszExtraInfo
  /* Opens a wave file for READING, or an (existing) file for WRITING .
   * Returns TRUE on success and FALSE on any error .
   */
{
 int i;
 T_WAV_CHUNK_HDR  chunk_hdr;
 struct ftime f_time;
 char *cp;
 BOOL fExtensionIsWav;
 BOOL fSubtractRecLengthFromTime = FALSE;
 BOOL ok = TRUE;
 BOOL found_data_chunk;

  CloseFile();   // Close "old" file if it was open

  // Look at the file extension..
  cp = strrchr(pszFileName, '.');
  if(cp)
       fExtensionIsWav = (strnicmp(cp,".wav",4)==0);
  else fExtensionIsWav = FALSE;
  // ex: m_iFileFormat = AUDIO_FILE_FORMAT_UNKNOWN;  // not sure whether RAW, RIFF, or SETI ...


  // Erase old chunks:
  memset( &m_RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
  memset( &m_ChunkHdr,    0 , sizeof(T_WAV_CHUNK_HDR)    );
  memset( &m_ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX) );
  memset( m_sz255FileName,0 , 255);
  // removed from here: m_dbl_SampleRate = 0.0;  // still unknown (!)

  if( fReOpenForWriting )
   {
#    if (WAVEIO_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
      ok = QFile_Open(&m_QFile, pszFileName, QFILE_O_RDWR );
#    else
      m_FileHandle = _rtl_open( pszFileName, O_RDWR );
      ok = (m_FileHandle>0);
#    endif  // ! WAVEIO_USE_QFILE
   }
  else  // open for READING (only) :
   {
#    if (WAVEIO_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
      ok = QFile_Open(&m_QFile, pszFileName, QFILE_O_RDONLY );
#    else
      m_FileHandle = _rtl_open( pszFileName, O_RDONLY);
      ok = (m_FileHandle>0);
#    endif  // ! WAVEIO_USE_QFILE
   }
  if(ok)
   {
    strncpy(m_sz255FileName, pszFileName, 255);
    // Get the "file time" as a first coarse value when the file has been
    // recorded. Note that "getftime" returns its own time structure
    // which is very different from our internal "time()"-like format.
    // "getftime" returns the date relative to 1980,
    // but SpecLab uses the Unix-style format which is
    //    "Seconds elapsed since 00:00:00 GMT(=UTC), January 1, 1970"
    // how many SECONDS between the UNIX-time(est'd 1970) and 1980 ?
    // T_Float d1,d2; BYTE *pSource;
    // pSource = "1970-01-01 00:00:00";
    // UTL_ParseFormattedDateAndTime( "YYYY-MM-DD hh:mm:ss", &d1, &pSource); // -> 0 (!)
    // pSource = "1980-01-01 00:00:00";
    // UTL_ParseFormattedDateAndTime( "YYYY-MM-DD hh:mm:ss", &d2, &pSource);
    // d2-=d1;
     // (should be something like 365*24*60*60 = 31536000, in fact it's 315532800)

    if( getftime(
#       if (WAVEIO_USE_QFILE)
         m_QFile.iHandle,
#       else
         m_FileHandle,
#       endif
        &f_time)==0) // "0" = "no error"
     {
       m_dblStartTime =
         UTL_ConvertYMDhmsToUnix( // parameters for this routine:
         f_time.ft_year+1980,  // int year  (FULL year, like 2002, not crap like "year-1980",
         f_time.ft_month,      // int month (1..12)
         f_time.ft_day,        // int day   (1..31)
         f_time.ft_hour,    // int hour  (0..23) - not the annoying "AM PM" stuff :-)
         f_time.ft_min,     // int minute (0..59)
         2*f_time.ft_tsec); // T_Float dblSecond  (0.000...59.999)
       // By order of the Prophet, "dblStartTime" should really contain the RECORDING start time,
       //  in UTC, or -if a GPS receiver is connected- the current GPS time .
       // The FILE TIME is usually the time of the last write access, which will be
       // the LAST sample. So we must subtract the duration of the recording
       // as soon as we know it [but we don't know it here because the file header
       // has not been analysed yet].
       fSubtractRecLengthFromTime = TRUE;
     }
    else m_dblStartTime=0;


    // analyse the header of the WAV-file (which if POSSIBLY A RIFF-format)
#  if (WAVEIO_USE_QFILE)
    i = QFile_Read( &m_QFile,   &m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
#  else
    i = _rtl_read(m_FileHandle, &m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
#  endif
    if( i==sizeof(T_WAV_RIFF_HDR) )
     { // reading the RIFF header successful:
      if(strncmp(m_RiffHdr.id_riff, "RIFF", 4) == 0)
        {
          m_iFileFormat = AUDIO_FILE_FORMAT_RIFF;  // bingo, it's the common windoze - *.WAV-format
        }
      else
        { // 'raw' audio samples without header.. usually *.DAT, very ambiguous !!
          if(   (m_iFileFormat != AUDIO_FILE_FORMAT_RAW)
             && (m_iFileFormat != AUDIO_FILE_FORMAT_SETI_1) // added 2010-05-10
            )
           { m_iFileFormat = AUDIO_FILE_FORMAT_RAW;
           }
          if( m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1 ) // added 2010-05-10
           { // Info about the SETI raw file format, by Ignacio Nacho, 2010 :
             // > Currently the sample rate is 8738133 samples/sec and the recording is
             // > in I/Q format, 8 bit signed per sample. So the recorded raw file is:
             // > 1st byte: In-phase of first sample.
             // > 2nd byte: Quadrature of first sample.
             // > 3rd byte:: In-phase of second sample.
             // > 4th byte: Quadrature of second sample
             m_dbl_SampleRate = 8738133.0; // Samples/second for SETI (8.7 MSamples/sec,
                                           // this only works in FILE ANALYSIS MODE...)
             m_ChunkFormat.wChannels = m_wRawFileNumberOfChannels = 2;
             m_iSizeOfDataType = 1;   // 8 bit per ADC value ( times 2 for I/Q )
             m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I; // SIGNED INTEGER (?)
             m_ChunkFormat.wBitsPerSample = (WORD)( m_iSizeOfDataType * 8 );
           }
          if( fExtensionIsWav )
            {                                // .. should NEVER have the extension "wav"
              strcpy(m_cErrorString,"no RIFF id"); // .. so this is an error !
              ok = FALSE;
            }
        } // end else < NO RIFF-header found >
      if( m_iFileFormat == AUDIO_FILE_FORMAT_RIFF )
       { // if the audio file has a RIFF header, the RIFF-ID should be "WAVE" :
        if(strncmp(m_RiffHdr.wave_id, "WAVE", 4) != 0)
         { strcpy(m_cErrorString,"no WAVE id");
           ok = FALSE;
         }
       }
     }
   } // end if <file opened successfully>
  else
   { strcpy(m_cErrorString,"_open failed");
     // ex: ok = FALSE;
     return FALSE;
   }


  // For RIFF WAVE files (*.wav), analyse all CHUNKS, until the first "data" chunk has been found
  if( m_iFileFormat == AUDIO_FILE_FORMAT_RIFF )
   {
     memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
     found_data_chunk = FALSE;
     while( (ok) && (! found_data_chunk) )
      {
        // Read the next chunk HEADER from the input file:
#      if (WAVEIO_USE_QFILE)
        i = QFile_Read( &m_QFile,   &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
#      else
        i = _rtl_read(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
#      endif
        // 2008-04-26: Got here with chunk_hdr.id="fmt", chunk_hdr.dwChunkSize=18 ,
        //   with a 32kHz stereo wave file saved by "sndrec32" from windoze XP .
        //
        if( i!=sizeof(T_WAV_CHUNK_HDR) )
         { strcpy(m_cErrorString,"bad header size");
           ok = FALSE;
         }
        else // reading the chunk header was ok. Analyse it:
         {
           i = 0; // number of bytes read from a "known" chunk.
           if(   (strncmp(chunk_hdr.id, "fmt ", 4) == 0)
              && (chunk_hdr.dwChunkSize >= WAVEIO_SIZEOF_STANDARD_WAVEFORMAT )
              && (chunk_hdr.dwChunkSize <= sizeof(T_WAV_CHUNK_FORMAT_EX) )
             )
            { // whow, it's *AT LEAST* a standard T_WAV_CHUNK_FORMAT - structure,
              //   read a T_WAV_CHUNK_FORMAT_xyz from the input file .
              // BUT: The 'fmt ' chunk may be a simple 'WAVEFORMAT' (wFormatTag=1)
              //      or (especially for 24-bit samples( a 'WAVEFORMATEXTENSIBLE' (wFormatTag=65534) !
#            if (WAVEIO_USE_QFILE)
              i = QFile_Read( &m_QFile,   &m_ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
#            else
              i = _rtl_read(m_FileHandle, &m_ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
#            endif
              if ( (DWORD)i != chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ )
               { ok = FALSE;
               }
              // Note that the actual chunk size may differ from what we
              // just read, even though it's a "fmt "-chunk !!
            }
           else if (strncmp(chunk_hdr.id, "data", 4) == 0)
            { // bingo, it's the one and only DATA-Chunk:
              m_dwFileDataSize = chunk_hdr.dwChunkSize;  // netto size of the samples in BYTES
              found_data_chunk = TRUE;
            }
           else if (strncmp(chunk_hdr.id, "inf1", 4) == 0)
            { // it's the prorietary 'Info Chunk, type 1' (since 2010-07) :
              if( (pszExtraInfo!=NULL) && ((DWORD)iExtraInfoLength>chunk_hdr.dwChunkSize) )
               { // ok, the caller's string buffer is large enough. Copy:
#            if (WAVEIO_USE_QFILE)
                 i = QFile_Read( &m_QFile,   pszExtraInfo, chunk_hdr.dwChunkSize );
#            else
                 i = _rtl_read(m_FileHandle, pszExtraInfo, chunk_hdr.dwChunkSize );
#            endif
                 pszExtraInfo[chunk_hdr.dwChunkSize] = '\0'; // ALWAYS keep C-strings terminated !
                 // WaveIO.cpp doesn't care about the contents of this string,
                 // but AudioFileIO.cpp will parse some of the contents .
               }
            }
           else // unknown chunk type, ignore it, it's ok !
            {
              i = 0;  // no "extra" bytes read from unknown chunk. Skip it (below).
              // 2011-02-09: got here with a chunk type "PAD " in a file
              // recorded with a ZOOM H-1, 96 kSamples/sec, 24 bit/s, 2 channels,
              // which could NOT be properly read by SL .
            }
           // Skip the chunk, no matter if it was unknown...
           //  (except for the "data" chunk)
           // Reason: also the known chunks may be LONGER
           // in the file than we expected in our structure !
           // For example, the size of the "fmt" header may be indicated
           //  as 16, 18, 40 bytes, or whatever, but we only read a maximum of
           //  40 bytes above (in QFile_Read), so we must possibly SKIP
           //  a few bytes in the seek-command below :
           if( (!found_data_chunk) && ok && ( chunk_hdr.dwChunkSize > (DWORD)i) )
            {
#           if (WAVEIO_USE_QFILE)
             if (QFile_Seek( &m_QFile, chunk_hdr.dwChunkSize-i, QFILE_SEEK_CUR ) < 0)
#           else
             if (lseek( m_FileHandle, chunk_hdr.dwChunkSize-i, SEEK_CUR ) < 0)
#           endif
              { strcpy(m_cErrorString,"lseek failed");
                ok = FALSE;  // seeking the end of the unknown chunk failed !
              }
            }
         } // end if <reading a chunk header ok>
      } // end while <ok AND "data"-chunk not found>

     if (!found_data_chunk) ok=FALSE;
     /* Arrived here with ok=TRUE, we should have skipped all chunks. */
     /* Test the "fmt "-chunk's contents if the format is supported:  */
     if (ok)
      {
        if( (m_ChunkFormat.wFormatTag != 1)     // standard Microsoft PCM
          &&(m_ChunkFormat.wFormatTag != 65534) // 'WAVE_FORMAT_EXTENSIBLE'
          )
         {
          sprintf(m_cErrorString,"bad FormatTag:%d",(int)m_ChunkFormat.wFormatTag );
          ok = FALSE;    // Format category, only 1=Microsoft PCM supported
         }
        if( (m_ChunkFormat.wChannels<1) && (m_ChunkFormat.wChannels>8) )
         {
          sprintf(m_cErrorString,"bad NrChannels:%d",(int)m_ChunkFormat.wChannels );
          ok = FALSE;    // Number of channels, 1=mono, 2=stereo
         }
        if(   (m_ChunkFormat.dwSamplesPerSec < 1)
           || (m_ChunkFormat.dwSamplesPerSec > AUDIO_FILE_MAX_SAMPLE_RATE) ) // -> c:\cbproj\SoundUtl\AudioFileDefs.h
         {
          sprintf(m_cErrorString,"bad SamplesRate:%d",(int)m_ChunkFormat.dwSamplesPerSec);
          ok = FALSE;    // Sampling rate not supported
         }
        else
         { // at least we know the 'nominal' sample rate..
           // Problem: RIFF WAVe header only uses integer value for sampling rate.
           // Try to be smart and reconstruct a 'precise' sample rate.
           // A guess (true in 90% of all cases):
           // Sampling rate is a standard soundcard sampling rate
           // or some "decimated" value from that.
           // Decimation factors can be any combination of 2^n * 3^m , 0..n..8, 0..m..8 .
           m_dbl_SampleRate = GuessExactFileSampleRate( m_ChunkFormat.dwSamplesPerSec );
         }
        if( (m_ChunkFormat.wBlockAlign < 1) || (m_ChunkFormat.wBlockAlign > 16) )
         {
          sprintf(m_cErrorString,"bad BlockAlign:%d",(int)m_ChunkFormat.wBlockAlign );
          ok = FALSE;    // Data block size ?? Bytes per Sample ???
         }
        if(  (m_ChunkFormat.wBitsPerSample==8)   // can we handle this ?...
          || (m_ChunkFormat.wBitsPerSample==16)
          || (m_ChunkFormat.wBitsPerSample==24)  // supported since 2010-05-27
          || (m_ChunkFormat.wBitsPerSample==32) )
         {
           m_iSizeOfDataType = m_ChunkFormat.wBitsPerSample / 8;
           m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I; // initial guess...
           if( m_ChunkFormat.wBitsPerSample==8 )
            { // "PCM" wave audio files with EIGHT BITS PER (single) SAMPLE
              // are an exception: they use UNSIGNED integers (range 0..255),
              // not SIGNED integers as one would except (-128...+127) !
              m_iDataTypeFlags = AUDIO_FILE_DATA_TYPE_UNSIGNED;
            }
         }
        else  // neither 8, 16, 24, or 32 bits/sample -> too exotic for me :o)
         {
          sprintf(m_cErrorString,"bad BitsPerSample:%d",(int)m_ChunkFormat.wBitsPerSample );
          ok = FALSE;    // Sample size, usually 8 or 16 bits per sample
         }
        if( (m_ChunkFormat.wBitsPerSample==16) && (m_ChunkFormat.wBlockAlign==1) )
         {  // 16 bits per sample and "block_align=1" is impossible (see formula)
            // but it occurred in a sample from G4JNT. Fix that bug HERE:
            m_ChunkFormat.wBlockAlign = (WORD)
               (m_ChunkFormat.wChannels * (m_ChunkFormat.wBitsPerSample+7) / 8);
         }
      } // end if <ok to check the "fmt "-chunk ?>

     if (ok)
      {
        // save file-internal position of 1st audio sample
        // (just in case someone wants to "Rewind" etc).
#      if (WAVEIO_USE_QFILE)
        m_dwFilePos_Data = QFile_Seek( &m_QFile, 0, QFILE_SEEK_CUR );
#      else
        m_dwFilePos_Data = tell( m_FileHandle );
          // caution: tell returns a "SIGNED long" value, which is stupid
          //           because disk files may be up to 4 GBytes !
#      endif
        if( m_dwFilePos_Data == 0xFFFFFFFF )
         { strcpy(m_cErrorString,"ftell failed");
           ok = FALSE;
         }
        m_dwCurrFilePos  = m_dwFilePos_Data;
        // Note: This is not the position (file offset) of the "data chunk",
        //       but the position of the FIRST SAMPLE in the data chunk !
        if( fReOpenForWriting )
         { // if the file is 're-opened' for writing, set the current writing position
           // to the END of the file:
#         if (WAVEIO_USE_QFILE)
           m_dwCurrFilePos = QFile_Seek( &m_QFile, 0, QFILE_SEEK_END );
#         else
           m_dwCurrFilePos = lseek( m_FileHandle, 0, SEEK_END ); // set current position to the file's end
#         endif
         } // end if < re-opened for writing >
      } // end if < header analysis 'ok' >

   } // end if( m_iFileFormat == AUDIO_FILE_FORMAT_RIFF )
  else   // not AUDIO_FILE_FORMAT_RIFF,  but AUDIO_FILE_FORMAT_RAW = 'raw' audio samples without header
   { //------------------------------------------------------------------------------------

#   if (WAVEIO_USE_QFILE)
     m_dwFileDataSize = QFile_Seek( &m_QFile, 0, QFILE_SEEK_END );
#   else
     m_dwFileDataSize = lseek( m_FileHandle, 0, SEEK_END ); // determine the netto size in BYTES
#   endif

     m_dwFilePos_Data = m_dwCurrFilePos = 0;  // no header to skip, jump back to the 1st byte
#   if (WAVEIO_USE_QFILE)
     QFile_Seek( &m_QFile, m_dwCurrFilePos, QFILE_SEEK_SET );
#   else
     lseek( m_FileHandle, m_dwCurrFilePos, SEEK_SET );
#   endif

     // Only a few properties should be set for successfully reading a RAW file.
     // The caller should set them before or after opening the file for reading .
     // (preferred: Tell us everything we need to know BEFORE opening the file)
     UpdateRawFileParams();  // -> m_ChunkFormat.wBlockAlign, etc
   }

  // Arrived here, the file-pointer is set to the first audio sample in the file.

  if( fSubtractRecLengthFromTime
     && (m_dbl_SampleRate>0) && (m_ChunkFormat.wBlockAlign>0) )
   { // subtract recording duration from "recording start time"
     // (because up to now, only the DOS FILE TIME is known)
     m_dblStartTime -= m_dwFileDataSize/*bytes*/
        / ( m_dbl_SampleRate * m_ChunkFormat.wBlockAlign );
   } // end if( fSubtractRecLengthFromTime )

  if (!ok)
   {
    if(m_cErrorString[0]=='\0')
      strcpy(m_cErrorString,"other 'open' error");
    CloseFile();  // close input file if opened but bad header
    return FALSE;
   }
  else
   {
    m_OpenedForReading = ok;
    return TRUE;
   }
} // C_WaveIO::InOpenInternal(..)

/***************************************************************************/
BOOL C_WaveIO::InOpen(
                char *pszFileName,
                int iAudioFileFormat,  // AudioFileDefs.h: AUDIO_FILE_FORMAT_..
                char *pszExtraInfo,    // [out] optional info text (ASCII), NULL when unused
                int iExtraInfoLength ) // [in] size of pszExtraInfo
  /* Opens a wave file for READING .
   * Returns TRUE on success and FALSE on any error .
   */
{
  // Because we're not sure if *.DAT is 'any RAW file format'
  //         or SETI's "raw file format", let the caller pass
  //         the (assumed) audio file format to us :
  m_iFileFormat = iAudioFileFormat;
  return InOpenInternal( pszFileName, FALSE/*!fReOpenForWriting*/,
                         pszExtraInfo, iExtraInfoLength );
} // C_WaveIO::InOpen(..)


/************************************************************************/
void C_WaveIO::UpdateRawFileParams(void)
{
 if(   (m_iFileFormat==AUDIO_FILE_FORMAT_RAW )
    || (m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1) // added 2010-05-10
   )
  { // If the most important parameters of the RAW file have been specified
    // *after* opening the file, calculate the missing parameters now :
    if( m_iSizeOfDataType < 1 )
        m_iSizeOfDataType = 1;
    if( m_wRawFileNumberOfChannels<1 )
        m_wRawFileNumberOfChannels=1;
    m_ChunkFormat.wBitsPerSample = (WORD)( 8 * m_iSizeOfDataType );
    m_ChunkFormat.wChannels   = (WORD)m_wRawFileNumberOfChannels;
    m_ChunkFormat.wBlockAlign = (WORD)(m_iSizeOfDataType * m_wRawFileNumberOfChannels);
  }
} // end C_WaveIO::UpdateRawFileParams()

/************************************************************************/
DWORD C_WaveIO::GetTotalCountOfSamples(void)
  /* Returns the total count of sampling points in the file .
   */
{
  if( m_ChunkFormat.wBlockAlign>0 )
   { // subtract recording duration from "recording start time"
     // (because up to now, only the DOS FILE TIME is known)
     return m_dwFileDataSize/*bytes*/ / m_ChunkFormat.wBlockAlign;
   } // end if( fSubtractRecLengthFromTime )
  else
     return 0;
} // C_WaveIO::GetTotalCountOfSamples()


/***************************************************************************/
DWORD C_WaveIO::GetCurrentSampleIndex(void)
  /* Reads the current sample index that will be used on the next
   * READ- or WRITE- access to the audio samples in the opened file.
   * Returns a negative value on any error.
   *  Caution, since 2008-03 we need to play with wave files > 2 GByte
   *  so use DWORD, not "long" as the file offset .
   * Return value : sample index; 0xFFFFFFFF = ERROR ("4 Gig minus one")
   */
{
 DWORD dw = 0xFFFFFFFF;
 if (
#    if (WAVEIO_USE_QFILE)
       (m_QFile.iHandle>=0)
#    else
       (m_FileHandle>=0)
#    endif
     && (m_ChunkFormat.wBlockAlign>0) )
  {
   // ex: dw = tell( m_FileHandle );  // check this : will "tell" work beyond 2 GByte ?
   // better use our local copy of the "current file position", it's faster :
   dw = m_dwCurrFilePos;
   if( dw >= m_dwFilePos_Data )
       dw -= m_dwFilePos_Data;
   else dw = 0;
   if(m_ChunkFormat.wBlockAlign>0)
      dw /= (DWORD)m_ChunkFormat.wBlockAlign;
  }
 return dw;
} // C_WaveIO::GetCurrentSampleIndex()


/***************************************************************************/
BOOL C_WaveIO::WindToSampleIndex(DWORD dwNewSampleIndex)
  /* Sets the current sample index that will be used on the next
   * READ- or WRITE- access to the audio samples an opened file.
   * Returns TRUE if ok ,  otherwise FALSE.
   * As long as you don't use this routine/method,
   * the audio file will be played in a strictly "sequential" way.
   */
{
 DWORD dw = 0xFFFFFFFF;
 if (   (m_ChunkFormat.wBlockAlign>0)
     && (dwNewSampleIndex>=0)
#   if ( WAVEIO_USE_QFILE )
     && (m_QFile.iHandle >= 0)
#   else
     && (m_FileHandle >= 0)
#   endif
    )
  {
   dw = dwNewSampleIndex * m_ChunkFormat.wBlockAlign;
   if ( (dw>=0) && (dw<m_dwFileDataSize) )
    {
      m_dwCurrFilePos = m_dwFilePos_Data + dw;
#    if (WAVEIO_USE_QFILE)
      return( QFile_Seek( &m_QFile, m_dwCurrFilePos , SEEK_SET ) > 0);
#    else
      return( lseek( m_FileHandle, m_dwCurrFilePos , SEEK_SET ) > 0);
#    endif
    }
  }
 return FALSE;
} // C_WaveIO::WindToSampleIndex()



/***************************************************************************/
LONG C_WaveIO::ReadSamples( BYTE* pData, LONG Length )
  /* Reads some samples from a wave file which has been opened for READING,
   *            !!! WITHOUT !!! conversion of the sample format in memory .
   * Returns the NUMBER OF AUDIO SAMPLE POINTS if successful,
   *     (not the number of bytes; one sample point may contain up to 8 bytes)
   *   or a negative value if errors occurred.
   * Notes:
   *  - "Length" is the size of the caller's buffer capacity in BYTES.
   *     The actual number of AUDIO SAMPLES will depend on the sample
   *     format. For 16-bit mono recordings, the number of samples
   *     will only be half the "Length".
   *     Generally, the   number of audio samples read   is:
   *          n_samples = Length / m_ChunkFormat.wBlockAlign .
   *  - since 11/2005, up to FOUR channels with SIXTEEN BITS each are possible
   *          ( m_ChunkFormat.wBlockAlign = 16 ) .
   */
{
 int  i;

 if( (m_ChunkFormat.wBlockAlign==0)
    &&  (   (m_iFileFormat==AUDIO_FILE_FORMAT_RAW)
         || (m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1) ) // added 2010-05-10
   )
  { // If the most important parameters of the RAW file have been specified
    // *after* opening the file, calculate the missing parameters now :
    UpdateRawFileParams();   // -> m_ChunkFormat.wBlockAlign, etc
  }

 if ( m_OpenedForReading && (m_ChunkFormat.wBlockAlign>0)
#   if ( WAVEIO_USE_QFILE )
     && (m_QFile.iHandle >= 0)
#   else
     && (m_FileHandle >= 0)
#   endif
    )
  {
# if ( WAVEIO_USE_QFILE )
   i = QFile_Read( &m_QFile,   pData, Length);
# else
   i = _rtl_read(m_FileHandle, pData, Length);
# endif
   if (i>0)
    {
      m_dwCurrFilePos += i;
      return i / m_ChunkFormat.wBlockAlign;
    }
  }
 return -1;
} // C_WaveIO::ReadSamples(..)

/***************************************************************************/
LONG C_WaveIO::ReadSampleBlocks( // preferred API (used by Spectrum Lab)
     int channel_nr,    // channel_nr for 1st destination block
     int nr_samples,    // number of samples (or, more precisely, length of destination blocks:  pFltDest1[0..nr_samples-1] )
     T_Float *pFltDest1, // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
     T_Float *pFltDest2, // 2nd destination block (~"RIGHT"), may be NULL (!)
     T_ChunkInfo *pOutChunkInfo, // [out,optional] additional results, see ChunkInfo.h, may be a NULL pointer (!)
     char *pszExtraInfo, int iMaxStrlen ) // [out,optional] extra string with user data for wave.rd_info
  /* Reads some samples from a wave file and converts them to numbers (*)
   *                                         in the range of +- 32768.0 .
   * Input parameters:
   *   channel_nr: 0= begin with the first channel (LEFT if stereo recording)
   *               1= begin with the second channel (RIGHT if stereo, or Q if complex)
   *     The UPPER BYTE of 'channel_nr' can optionally define
   *         HOW MANY CHANNELS SHALL BE PLACED IN ONE DESTINATION BLOCK:
   *    (channel_nr>>8) : 0x00 or 0x01 = only ONE channel per destination block
   *                      0x02         = TWO channels per destination block
   *                                    (often used for complex values aka I/Q)
   *
   *   nr_samples: Max number of FLOATs which can be stored in T_Float dest[0..nr_samples-1]
   *               (this is NOT always the same as the number of sampling points,
   *                for example if the output are I/Q samples, all placed in
   *                pFltDest1, there are only nr_samples/2 complex sampling points).
   *   *pFltDestX = Pointers to destination arrays (one block per channel, sometimes "paired")
   * Return value:
   *      the NUMBER OF FLOATING-POINT VALUES placed in each destination block
   *                 (not necessarily the number of sample points!) ,
   *      or a negative value if an error occurred.
   * Note that returning less samples than you may expect is not an error
   * but indicates reaching the file's end !
   *   ( This convension later caused trouble when the input is not a FILE
   *     but a STREAM, see WPH_IO_Plugin.cpp ) .
   * (*) T_Float may be either SINGLE- or DOUBLE-precision float,
   *     defined in some header file, depending on SWI_FLOAT_PRECISION .
   */
{
  BYTE *temp_array;
  int  i;
  int  asize;
  int  n_samples_read;
  int  iResult;
  BOOL ok;
  BOOL fTwoChannelsPerBlock = FALSE;
  T_Float flt;

   fTwoChannelsPerBlock =  ((channel_nr & AUDIO_FILE_TWO_CHANNELS_PER_BLOCK) != 0)
                        && (m_ChunkFormat.wChannels>1);
   channel_nr &= 0x000F;  // remove all 'flags', leave only the start-channel number

   if ( (!m_OpenedForReading)
#     if ( WAVEIO_USE_QFILE )
      || (m_QFile.iHandle<0)
#     else
      || (m_FileHandle<0)
#     endif
      || (m_ChunkFormat.wBlockAlign<=0)
      || (m_ChunkFormat.wBitsPerSample <= 1)  )
     return -1;

   if(fTwoChannelsPerBlock)
    { nr_samples /= 2;    // here: TWO destination array indices per sample point
      // so if the caller can accept 4096 floating-point values in his array (pFltDest1),
      // he will only read 2048 sampling points in one call of this subroutine !
    }

   asize = nr_samples * m_ChunkFormat.wBlockAlign/* strange name for "bytes per sample-point"*/;
   if (asize<=0)
     return -1;
   temp_array = new BYTE[asize];  // allocate temporary array, C++ style
   if(temp_array==NULL)
     return -1;

   n_samples_read = ReadSamples( // Returns the NUMBER OF AUDIO SAMPLE POINTS!
            temp_array, /* temporary BYTE-array for conversion     */
            asize );    /* size in BYTE, not "count of samples" !! */
   if (n_samples_read > nr_samples)
       n_samples_read = nr_samples;  // emergency limit, should never happen
   ok = FALSE;
   if  (m_ChunkFormat.wBitsPerSample <= 8)
    { /* up to 8 bits per sample : use a BYTE-pointer for addressing... */
     ok = TRUE;
     BYTE *bp = temp_array;
     if( (m_ChunkFormat.wChannels>1)  // Number of channels, 1=mono, 2=stereo
      && (channel_nr < m_ChunkFormat.wChannels) )
      { // channel_nr: 0=left, 1=right, ..
        bp += channel_nr;             // skip 'unwanted' channels
      }
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;   // TWO destination array indices per sample point
        if( m_iDataTypeFlags & AUDIO_FILE_DATA_TYPE_UNSIGNED )
         { // here for UNSIGNED 8-bit samples ( raw value range 0..255 ) :
           for(i=0; i<n_samples_read/*!*/; i++ )
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              flt = (unsigned char)bp[0];
              *(pFltDest1++) = (flt-127.0) * (32767.0 / 127.0);  // 1st channel in file ("I")
              flt = (unsigned char)bp[1];
              *(pFltDest1++) = (flt-127.0) * (32767.0 / 127.0);  // 2nd channel in file ("Q")
              if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
               { flt = (unsigned char)bp[2];
                 *(pFltDest2++) = (flt-127.0) * (32767.0 / 127.0); // 3rd channel in file ("I")
                 flt = (unsigned char)bp[3];
                 *(pFltDest2++) = (flt-127.0) * (32767.0 / 127.0); // 4th channel in file ("Q")
               }
              bp += m_ChunkFormat.wBlockAlign; /* "block align" ~~ "bytes per sample point" */
            } // end for < 8-bit SIGNED samples >
         }
        else  // 8-bit SIGNED ( raw value range -128 ... + 127 )  :
         {
           for(i=0; i<n_samples_read/*!*/; i++ )
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              flt = (signed char)bp[0];
              *(pFltDest1++) = flt * (32767.0 / 127.0);  // 1st channel in file ("I")
              flt = (signed char)bp[1];
              *(pFltDest1++) = flt * (32767.0 / 127.0);  // 2nd channel in file ("Q")
              if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
               { flt = (signed char)bp[2];
                 *(pFltDest2++) = flt * (32767.0 / 127.0); // 3rd channel in file ("I")
                 flt = (signed char)bp[3];
                 *(pFltDest2++) = flt * (32767.0 / 127.0); // 4th channel in file ("Q")
               }
              bp += m_ChunkFormat.wBlockAlign; /* "block align" ~~ "bytes per sample point" */
            } // end for < 8-bit SIGNED samples >
         }
      }
     else // still 8 bits per value, but not "two channels per output block" ...
      { iResult = n_samples_read;  // ONE destination array index per sample point
        if( (m_ChunkFormat.wChannels<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           for(i=0; i<n_samples_read; ++i)
            { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
              *pFltDest1++ = ( (T_Float)(*bp) - 127.0 ) * (32767.0 / 127.0);
              bp += m_ChunkFormat.wBlockAlign;
            }
         }
        else
         {  // TWO or more channels must be copied (and "split" into two destination blocks) :
           for(i=0; i<n_samples_read; ++i)
            { // up to 8 bits per sample, TWO SEPARATE destination blocks:
              *pFltDest1++ = ( (T_Float)(*bp)   - 127.0 ) * (32767.0 / 127.0);
              *pFltDest2++ = ( (T_Float)(*(bp+1))-127.0 ) * (32767.0 / 127.0);
              bp += m_ChunkFormat.wBlockAlign;
            }
         } // end if <two separate destination blocks>
      } // end else < not TWO channels per destination block >
   } // end if (8-bit-samples)
  else
  if( (m_ChunkFormat.wBitsPerSample <= 16) && (m_ChunkFormat.wBlockAlign>1) )
   { /* 9..16 bits per sample, we use a 16bit-int-pointer for addressing. */
     ok = TRUE;
     short *sp = (short*)temp_array; // should be even address, so what ??
     if( (m_ChunkFormat.wChannels>1) // Number of channels, 1=mono, 2=stereo
      && (channel_nr < m_ChunkFormat.wChannels) )
      { // channel_nr: 0=left, 1=right, ..
        sp += channel_nr;     // skip 'unwanted' channels
      }
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;  // TWO destination array indices per sample point
        switch( m_iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                 |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
              ) )
         {
           case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = (T_Float)(*sp);        // 1st channel ("I")
                *pFltDest1++ = (T_Float)(*(sp+1));    // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
                 { *pFltDest2++ = (T_Float)(*(sp+2)); // 3rd channel ("I")
                   *pFltDest2++ = (T_Float)(*(sp+3)); // 4th channel ("Q")
                 }
                sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
           case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadUns16Intel(sp);   // 1st channel ("I")
                *pFltDest1++ = ReadUns16Intel(sp+1); // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
                 { *pFltDest2++ = ReadUns16Intel(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadUns16Intel(sp+3); // 4th channel ("Q")
                 }
                sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
           case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadSigned16Motorola(sp);    // 1st channel ("I")
                *pFltDest1++ = ReadSigned16Motorola(sp+1);  // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
                 {
                   *pFltDest2++ = ReadSigned16Motorola(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadSigned16Motorola(sp+1); // 4th channel ("Q")
                 }
                sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
           case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
              | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
            { for(i=0; i<n_samples_read/*!*/; i++ )
               { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                *pFltDest1++ = ReadUns16Motorola(sp);    // 1st channel ("I")
                *pFltDest1++ = ReadUns16Motorola(sp+1);  // 2nd channel ("Q")
                if( (pFltDest2!=NULL) && (m_ChunkFormat.wChannels>2) )
                 {
                   *pFltDest2++ = ReadUns16Motorola(sp+2); // 3rd channel ("I")
                   *pFltDest2++ = ReadUns16Motorola(sp+3); // 4th channel ("Q")
                 }
                sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
               }
            } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
         } // end switch( m_iDataTypeFlags & ... )
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      {
        iResult = n_samples_read;  // ONE destination array index per sample point
        if( (m_ChunkFormat.wChannels<2) || (pFltDest2==NULL) )
         { // only ONE channel to be copied...
           switch( m_iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                   ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                    |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
                 ) )
            {
              case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
               { for(i=0; i<n_samples_read; ++i)
                  { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
                    *pFltDest1++ = (T_Float)(*sp);
                    sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break;
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Intel(sp);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
              case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadSigned16Motorola(sp);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_MOTOROLA (signed)
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
                 | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Motorola(sp);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED,MOTOROLA
            } // end switch( m_iDataTypeFlags &  ... )
         }
        else
         {  // TWO channels must be copied (and "split" into two destination blocks) :
           switch( m_iDataTypeFlags &  // see bitmasks defined in AudioFileDefs.h :
                   ( AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01, if NOT set, it's a SIGNED integer */
                    |AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04, if NOT set, it's the usual INTEL byte order*/
                 ) )
            {
              case AUDIO_FILE_DATA_TYPE_SIGNED_I/*0x00*/:  // SIGNED 16 bit integer, little endian ("Intel")
               { for(i=0; i<n_samples_read; ++i)
                  { // up to 8 bits per sample, coding is "UNSIGNED" (crazy, isn't it)
                    *pFltDest1++ = (T_Float)(*sp);
                    *pFltDest2++ = (T_Float)(*(sp+1));
                    sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_SIGNED_I
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/: // 16 bit UNSIGNED, little endian ("Intel")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Intel(sp);
                   *pFltDest2++ = ReadUns16Intel(sp+1);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED
              case AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/:  // SIGNED 16 bit integer, big endian ("Motorola")
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadSigned16Motorola(sp);
                   *pFltDest2++ = ReadSigned16Motorola(sp+1);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_MOTOROLA (signed)
              case AUDIO_FILE_DATA_TYPE_UNSIGNED/*0x01*/ // 16 bit UNSIGNED, big endian ("Motorola")
                 | AUDIO_FILE_DATA_TYPE_MOTOROLA/*0x04*/ :
               { for(i=0; i<n_samples_read/*!*/; i++ )
                  { // up to 8 bits per sample, TWO SEPARATE destination blocks:
                   *pFltDest1++ = ReadUns16Motorola(sp);
                   *pFltDest2++ = ReadUns16Motorola(sp+1);
                   sp += (m_ChunkFormat.wBlockAlign / 2); // pointer to 16-bit-values!!
                  }
               } break; // end case AUDIO_FILE_DATA_TYPE_UNSIGNED,MOTOROLA
            } // end switch( m_iDataTypeFlags &  ... )
         } // end if <two separate destination blocks>
      } // end else < not TWO channels per destination block >
    } // end if (16-bit-samples)
   else if( m_ChunkFormat.wBitsPerSample == 24 ) // added 2010-05-27 ...
    {   // and modified 2011-02, because some stupid asshole decided
        // to REVERSE the byte order in a 24-bit-per-sample recording.
        // In M1F1-int24-AFsp.wav, considered a 'valid' PCM wave file,
        //   the byte order seemed to be LOW1-MID1-HIGH1 LOW2-MID2-HIGH2
        //   (1 = left channel,   2 = right channel) .
        //
     Conv24BitSamplesToFloat( temp_array, n_samples_read,
                 m_ChunkFormat.wChannels,  // number of channels in the SOURCE
                 fTwoChannelsPerBlock?2:1,
                 pFltDest1,  // 1st destination block (~"LEFT",  sometimes "I" AND "Q" )
                 pFltDest2); // 2nd destination block (~"RIGHT"), may be NULL (!)
     if( fTwoChannelsPerBlock )
      { // two channels (usually I+Q) per destination block (since 2005-11-01) :
        iResult = 2*n_samples_read;   // TWO destination array indices per sample point
      }
     else // ! fTwoChannelsPerBlock .. may be two separated destination blocks
      {
        iResult = n_samples_read;  // ONE destination array index per sample point
      }
    }
   else
    { // cannot handle this sample format !!
     ok = FALSE;
     iResult = -1;  // error
    }

   delete[] temp_array;
   return iResult;

} // C_WaveIO::ReadSampleBlocks(...)

/***************************************************************************/
BOOL C_WaveIO::WriteHeader(
           T_ChunkInfo *pChunkInfo) // << precise sample rate, 'radio frequency', date and time, etc (must not be NULL)
  /* Writes the 'header' for the audio file (usually a RIFF WAVE file).
   *  This was once done immediately in OutOpen(), but since 2011-02
   *  writing the header is postponed until the first chunk of samples
   *  is available, and a PRECISE(!) timestamp is present in pChunkInfo .
   *  All relevant information has already been set in OutOpen(),
   *  but not written to the file yet (by the time we get here).
   */
{
  T_WAV_CHUNK_HDR  chunk_hdr;
  BOOL ok = TRUE;
  BOOL fWriteHeader = TRUE;
  char sz1kInfo[1024];

  // prepare a 'format chunk' ("fmt ") in memory even if not writing to RIFF wave:
  memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
  strncpy(chunk_hdr.id, "fmt ", 4);
  if( (m_iBitsPerSample <= 16) && (m_iNumChannels<=2) )
   { m_ChunkFormat.wFormatTag = 1;     // Format category:  1=Microsoft PCM
     chunk_hdr.dwChunkSize = WAVEIO_SIZEOF_STANDARD_WAVEFORMAT;    /* 16 byte */
     m_ChunkFormat.cbSize = 16;   // doesn't matter, not written to file
   }
  else
   { m_ChunkFormat.wFormatTag = 65534; // Format category:  'WAVE_FORMAT_EXTENSIBLE'
     chunk_hdr.dwChunkSize = WAVEIO_SIZEOF_WAVE_FORMAT_EXTENSIBLE; /* 40 byte */
     // > For the WAVEFORMATEXTENSIBLE structure, the Format.cbSize  field
     // > must be set to 22 and the SubFormat field ("GUID") must be set to
     // > KSDATAFORMAT_SUBTYPE_PCM .
     m_ChunkFormat.cbSize = 22;
   }


  if( m_OpenedForWriting )  // See OutOpen() ...
   {
    // Update the "start time" of this recording from the audio-chunk-info,
    //        and some other details which may be important for post processing.
    //  (because it will be more accurate than the timestamp when we CREATED the file):
    if( pChunkInfo != NULL )
     { m_dblStartTime = pChunkInfo->dblUnixDateAndTime; // ex: TIM_GetCurrentUnixDateAndTime();
       // By order of the Prophet, "dblStartTime" shall be a time in UTC,
       //    or -if a GPS receiver is connected- the current GPS time .
     }

    if(m_iFileFormat==AUDIO_FILE_FORMAT_RIFF)
     {
      // Fill and write the header of the WAV-file:
      memset( &m_RiffHdr,     0 , sizeof(T_WAV_RIFF_HDR)     );
      strncpy(m_RiffHdr.id_riff, "RIFF", 4);
      strncpy(m_RiffHdr.wave_id, "WAVE", 4);
      m_RiffHdr.len =  sizeof(T_WAV_RIFF_HDR)
                 + 2*sizeof(T_WAV_CHUNK_HDR)
                 + WAVEIO_SIZEOF_STANDARD_WAVEFORMAT/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/;
       // (the length of the audio samples ("data") is still unknown,
       //  will be ADDED later before closing)
      if( fWriteHeader )
       {
#       if ( WAVEIO_USE_QFILE )
         ok = QFile_Write( &m_QFile, (BYTE*)&m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
#       else
         ok = (_rtl_write(m_FileHandle, &m_RiffHdr, sizeof(T_WAV_RIFF_HDR) ) == sizeof(T_WAV_RIFF_HDR) );
#       endif
       }
     } // end if <RIFF file format>
   } // end if <file opened successfully>
  else
   { ok = FALSE;
   }


  // 2011-02-13 : Moved the "inf1" header further below,
  //                    between the "fmt " and the "data" chunk,
  //                    because Linux didn't like those files ...
  // Note: the chunk format thingy has already been init'd in OutOpen() !

  if( (fWriteHeader) && (m_iFileFormat==AUDIO_FILE_FORMAT_RIFF) && (ok) )
   {
    // Write the T_WAV_CHUNK_FORMAT to the output RIFF wave file:
#   if ( WAVEIO_USE_QFILE )
    ok &= QFile_Write( &m_QFile, (BYTE*)&chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
    ok &= QFile_Write( &m_QFile, (BYTE*)&m_ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
#  else
    i = _rtl_write(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
    if (i != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
    i = _rtl_write(m_FileHandle, &m_ChunkFormat, chunk_hdr.dwChunkSize/*ex:sizeof(T_WAV_CHUNK_FORMAT)*/ );
    if (i != chunk_hdr.dwChunkSize )   ok = FALSE; // file-write-error ?
#  endif
   }


  // Fill and write an "inf1"-chunk to the output file ?
  // 2011-02-13 : Moved the "inf1" header to this place,
  //                    between the "fmt " and the "data" chunk,
  //                    because Linux didn't like those files ...
#if(1) && ( WAVEIO_USE_QFILE )  // only possible with the 'QFile' abstraction layer:
  if( (fWriteHeader) && (m_iFileFormat==AUDIO_FILE_FORMAT_RIFF) && (ok)
      && (m_iAudioFileOptions & AUDIO_FILE_OPTION_ALLOW_EXTRA_INFOS_IN_HEADER) )
   { if( ChunkInfoToString( sz1kInfo, sizeof(sz1kInfo)-1/*maxlen*/, pChunkInfo ) )
      { // Modified in 2011-02-19 to improve the timestamp accuracy:
        //  The "chunk info text" is now prepared here, when writing
        //  the first audio sample - because only at this time, we know
        //  the PRECISE timestamp of the next sample written to the file
        // (accurate enough for TOA/ATD RDF, etc) .
        memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
        strncpy(chunk_hdr.id, "inf1", 4);
        chunk_hdr.dwChunkSize = (strlen(sz1kInfo) + 4/*!*/) & 0x000FFFC;
        ok &= QFile_Write( &m_QFile, (BYTE*)&chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
        ok &= QFile_Write( &m_QFile, (BYTE*)sz1kInfo, chunk_hdr.dwChunkSize );
      }
   } // end if < write an extra chunk with "additional info" ? >
#endif // ( WAVEIO_USE_QFILE )


  if( (fWriteHeader) && (m_iFileFormat==AUDIO_FILE_FORMAT_RIFF) && (ok) )
   {
    // Fill and write a DATA-chunk to the output file:
    memset(&chunk_hdr, 0, sizeof(T_WAV_CHUNK_HDR) );
    strncpy(chunk_hdr.id, "data", 4);
    chunk_hdr.dwChunkSize = 0; // still unknown, will be updated before closing.
#  if ( WAVEIO_USE_QFILE )
    ok &= QFile_Write( &m_QFile, (BYTE*)&chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
#  else
    i = _rtl_write(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR) );
    if (i != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
#  endif
    // save file-internal position of 1st audio sample, which will also be
    // used to write the correct "data"-chunk size into the header LATER,
    // when closing the file.
#  if (WAVEIO_USE_QFILE)
    m_dwFilePos_Data = QFile_Seek( &m_QFile, 0, QFILE_SEEK_CUR );
#  else
    m_dwFilePos_Data = tell( m_FileHandle );
#  endif
    if (m_dwFilePos_Data==0xFFFFFFFF)    // m_dwFilePos_Data should be 44 now
        ok = FALSE;
    m_dwCurrFilePos = m_dwFilePos_Data;
   }

  m_fMustWriteHeader = FALSE;  // "done" (at least, we tried)

  if (!ok)
   {
    CloseFile();  // close output file, could not generate a header.
    return FALSE;
   }
  else
   {
    return TRUE;
   }
} // end C_WaveIO::WriteHeader()

/***************************************************************************/
BOOL C_WaveIO::OutOpen(
           char *file_name,
           int iAudioFileFormat, // AudioFileDefs.h: AUDIO_FILE_FORMAT_..
           int file_mode,        // AUDIO_FILE_MODE_OVERWRITE etc
           int iAudioFileOptions, // AUDIO_FILE_OPT_ALLOW_EXTRA_INFOS_IN_HEADER, etc [-> c:\cbproj\SoundUtl\AudioFileDefs.h]
           int bits_per_sample,  // supported 8, 16, or 24 bits/sample
           int channels,
           T_ChunkInfo *pInChunkInfo ) // initial sample rate, date+time (here: NOT optional, ptr must not be NULL)
  /* Creates and Opens an audio file for WRITING audio samples.
   *         Returns TRUE on success and FALSE on any error.
   * Input parameters..
   *   iAudioFileFormat: required to solve the ambiguity of different "raw"
   *                     file formats, which all have the extension "*.dat",
   *                     which is rather meaningless but that's the way it is.
   *   additional_info:  a null-terminated text string with additional info
   *                     which will be included in the exported file.
   *                     NULL if this feature is not used.
   *                     SpecLab uses this to save (GPS-)date, (GPS-)time,
   *                     precise sample rate, possibly the name of a (GPS-)log,
   *                     and other receiver settings when the recording started.
   *                     In Spectrum Lab, this string is 'produced'
   *                     using ChunkInfo.c::ChunkInfoToString() .
   *   bits_per_sample:  must be either 8 or 16 (bits for a single channel)
   *   channels:         1=mono, 2=stereo recording etc
   *   sample_rate:      number of samples per second.
   *                     If the output format is AUDIO_FILE_FORMAT_RIFF,
   *                     the sample rate should be above 8000 samples/second.
   */
{
#if ( ! WAVEIO_USE_QFILE )
 int i;
#endif
 BOOL ok = TRUE;
 int iFileFormat;

  CloseFile();   // Close "old" file if it was open

  if( pInChunkInfo==NULL )
   { return FALSE;
   }

  // round the sample rate to a 32-bit INTEGER value (for the RIFF header)
  m_dbl_SampleRate = pInChunkInfo->dblPrecSamplingRate;

  // coarse check of the arguments..
  if(  (iAudioFileFormat!=AUDIO_FILE_FORMAT_RAW)
    && (iAudioFileFormat!=AUDIO_FILE_FORMAT_SETI_1) // also "*.dat" but other file format !
    && (iAudioFileFormat!=AUDIO_FILE_FORMAT_RIFF) )
      return FALSE;
  if( (bits_per_sample != 8) && (bits_per_sample != 16) && (bits_per_sample != 24) )
      return FALSE;
  m_iBitsPerSample = bits_per_sample;
  if( (channels<1) || (channels>2) )
      return FALSE;    // only mono or stereo recording supported
                       // ..sorry for those 'quattrophiles' ;-)
  m_iNumChannels = channels;
  // 2010-05-10: now accepting 10 MSample/sec ... SETI may use this,
  //             of course this will NEVER work for real-time processing !
  if( (m_dbl_SampleRate < 1) || (m_dbl_SampleRate > AUDIO_FILE_MAX_SAMPLE_RATE) )
      return FALSE;

  m_iAudioFileOptions = iAudioFileOptions;
  m_dwCurrFilePos = m_dwFilePos_Data = 0;  // default file position if no header
  m_dwFileDataSize= 0;  // "netto" size of audio samples still zero
  m_iFileFormat = iAudioFileFormat;  // remember the file format when closing!
  m_iFileMode   = file_mode;  // used in WriteHeader() [postponed since 2011-02]
  strncpy(m_sz255FileName, file_name, 255); // save the filename for the "Get"-function,
                                            // and for the postponed WriteHeader call .

  // prepare a 'format chunk' ("fmt ") in memory even if not writing to RIFF wave:
  memset( &m_ChunkFormat, 0 , sizeof(T_WAV_CHUNK_FORMAT_EX) );
  if( (m_iBitsPerSample <= 16) && (m_iNumChannels<=2) )
   { m_ChunkFormat.wFormatTag = 1;     // Format category:  1=Microsoft PCM
     m_ChunkFormat.cbSize = 16;   // doesn't matter, not written to file
   }
  else
   { m_ChunkFormat.wFormatTag = 65534; // Format category:  'WAVE_FORMAT_EXTENSIBLE'
     // > For the WAVEFORMATEXTENSIBLE structure, the Format.cbSize  field
     // > must be set to 22 and the SubFormat field ("GUID") must be set to
     // > KSDATAFORMAT_SUBTYPE_PCM .
     m_ChunkFormat.cbSize = 22;
   }
  m_ChunkFormat.wChannels  = (WORD)m_iNumChannels; //  1=mono, 2=stereo
  m_ChunkFormat.dwSamplesPerSec     // Sampling rate     , samples per second
                 = (long)(m_dbl_SampleRate+0.5);   // ROUNDED TO 32bit-INTEGER VALUE!
    // ToDo: Should we use the 'nominal' sampling rate here, even if we know better ?
    // Some audio processing software may not be comfortable with 47999 samples/second.
  m_ChunkFormat.wBitsPerSample    // Sample size, usually 8 or 16 bits per sample
                 = (WORD)m_iBitsPerSample;
  m_ChunkFormat.wBlockAlign       // Data block size , Bytes per Sample :
                 = (WORD)( (long)m_ChunkFormat.wChannels
                 *  ( ( m_ChunkFormat.wBitsPerSample + 7 ) / 8 ) );
  m_ChunkFormat.dwAvgBytesPerSec  // For buffer estimation
                 = m_ChunkFormat.wBlockAlign
                 * m_ChunkFormat.dwSamplesPerSec;
  // The above was the mandatory 16-byte 'standard WAVEFORMAT' part .
  // Next: Fill the (optional) rest of the WAVE_FORMAT_EXTENSIBLE in memory,
  //       regardless of chunk_hdr.dwChunkSize :
  m_ChunkFormat.Samples.wValidBitsPerSample = m_ChunkFormat.wBitsPerSample;
  memcpy( m_ChunkFormat.b16GUID, WaveIO_b16GUID_PCM, 16 ); // GUIDo-droid ??


  //
  // Note: Since 2011-02, writing of the WAVE FILE HEADERS is postponed
  //       until the first call of WriteSamples(), called through AudioFileIO.cpp (!)
  //   But we already CREATE the file here, to avoid wasting time later.
  // Create new file, truncate existing file to length "zero", or APPEND data ?
  m_fMustWriteHeader = TRUE;
  if( m_iFileMode == AUDIO_FILE_MODE_APPEND )
   { // if the file already exists, the new data shall be APPENDED :
     iFileFormat = m_iFileFormat; // save this info, if InOpenInternal() fails
     if( InOpenInternal( m_sz255FileName, TRUE/*fReOpenForWriting*/,
                         NULL,0/*don't care for the old info*/) )
      { m_fMustWriteHeader = FALSE;  // do NOT write an "all-new" header in this case !
      } // end if < "APPEND"-mode, and file successfully opened >
     else
      { m_iFileFormat = iFileFormat;  // set the file format again (after InOpenInternal failed)
        m_dwFileDataSize= 0;
      }
   }
#if ( WAVEIO_USE_QFILE )
  if(m_QFile.iHandle<=0)
   { ok = QFile_Create( &m_QFile, m_sz255FileName, 0/*attr="normal"*/ );
   }
  else ok = TRUE;
#else
  if(m_FileHandle<=0)
   { m_FileHandle = _rtl_creat( m_sz255FileName, 0);
   }
  ok = (m_FileHandle>0);
#endif

  m_OpenedForWriting = ok;  // required for some polling functions,
                            // "it's ok to call WriteSamples_Float() now" .

  return TRUE;   // not really sure if we will be able to CREATE the file...

} // C_WaveIO::OutOpen(..)


/***************************************************************************/
LONG C_WaveIO::WriteSamples( BYTE* pData, LONG Length/*nBytes*/ )
  /* Writes some samples in 'raw' form (without any conversion)
   *       to an audio file which has been opened for WRITING.
   * Returns the number of SAMPLES written,
   *       or a NEGATIVE value if there was an error.
   * Note: "Length" is the size of the caller's buffer capacity in BYTES.
   *       The actual number of AUDIO SAMPLES will depend on the sample
   *       format.
   *       For 16-bit mono recordings (which is the only supported format
   *       up 'til now), the number of samples will only be half the "Length"
   *       etc.
   */
{
  int n_written=0;

  /* Ensure that it's really possible to WRITE audio samples to a file now.
   */
   if ( (!m_OpenedForWriting)
      || (m_ChunkFormat.wBlockAlign<=0)
      || (m_ChunkFormat.wBitsPerSample <= 1)
      || (Length <= 0)
#    if (WAVEIO_USE_QFILE)
      || (m_QFile.iHandle < 0)
#    else
      || (m_FileHandle < 0)
#    endif
        )
     return -1;

  /* Try to append the new samples to the wave file.
   * Note that "WindToSampleIndex()" does not work on OUTPUT files yet !!!
   */
#  if ( WAVEIO_USE_QFILE )
   if( QFile_Write( &m_QFile, pData, Length ) )  // -> QFile_Write returns TRUE or FALSE only !
    { n_written = Length;
    }
#  else
   n_written = _rtl_write(m_FileHandle, pData, Length );
#  endif
   if (n_written != Length)
    {
     /* Writing to the output file failed  !!!
      * Seems we're in trouble now, because the reason may be a full
      * harddisk and Windooze is going to crash in a few moments :-(
      */
     CloseFile();
     return -1;  // also set a breakpoint here !
    }

  /* Keep the "current file position" updated so we don't have to call
   *    the "tell()"-function too often.
   * This parameter will also be READ from the spectrum analyser main window
   * to display the actual file size.
   */
  m_dwCurrFilePos += n_written;
  if(m_dwCurrFilePos-m_dwFilePos_Data >= m_dwFileDataSize)
     m_dwFileDataSize = m_dwCurrFilePos-m_dwFilePos_Data;
  return n_written / m_ChunkFormat.wBlockAlign;
} // end C_WaveIO::WriteSamples(..)

/***************************************************************************/
LONG C_WaveIO::WriteSamples_Float( // app-layer output, using NORMALIZED floats
       T_Float *pfltSrc,     // [in] caller's source buffer, 32-bit floats
       T_ChunkInfo *pChunkInfo, // [in] number of samples, channels, precise sample rate, 'radio frequency', date and time, etc
          // btw, T_ChunkInfo is defined in c:\cbproj\SoundUtl\ChunkInfo.h .
       char * pszExtraInfo ) // [in] extra 'info' (usually 'data') added to info chunk
  /* Writes some samples in the 'normalized' floating point format (-1.0 ... +1.0)
   *       to an audio file which has been opened for WRITING.
   * Returns the number of SAMPLE POINTS written,
   *       or a NEGATIVE value if there was an error.
   */
{
  int iChannel, nChannels, nSamplePoints, nBytesToSend, nSourceChannelsPerSample, nDestChannelsPerSample, iRest;
  T_Float flt, fltScalingFactor;
  union { BYTE b[4];  long i32; } t;
  LONG nSamplesWritten = 0;
  BYTE *pbTempBuffer, *pb;
  SHORT *pI16;

  if( pChunkInfo == NULL )
   { return -1;
   }
  nSamplePoints            = (int)pChunkInfo->dwNrOfSamplePoints;
  nSourceChannelsPerSample = pChunkInfo->nComponentsPerSamplePoint;
  nDestChannelsPerSample   = m_ChunkFormat.wChannels;
  nChannels = (nSourceChannelsPerSample < nDestChannelsPerSample)
             ? nSourceChannelsPerSample : nDestChannelsPerSample;

  /* Make sure that it's really possible to WRITE audio samples to a file now.
   */
   if ( (!m_OpenedForWriting)
      || (nSamplePoints <= 0 )
      || (nSourceChannelsPerSample <= 0)
      || (m_ChunkFormat.wBlockAlign<= 0)
      || (m_ChunkFormat.wBitsPerSample <= 1)
      || (pChunkInfo->dblSampleValueRange == 0.0)
#    if (WAVEIO_USE_QFILE)
      || (m_QFile.iHandle < 0)
#    else
      || (m_FileHandle < 0)
#    endif
        )
    { return -1;
    }

  /* If the header still needs to be written, do that NOW (we know the timestamp..) */
  if( m_fMustWriteHeader )
   { WriteHeader(
        pChunkInfo); // << precise sample rate, 'radio frequency', date and time, etc (must not be NULL)
     m_fMustWriteHeader = FALSE;  // "done"
   }



#if(0)
  // Also emit a new GPS position to the file ? (since 2010-07-07)
  if(  (m_iPosUpdateCounter_WR != pChunkInfo->gps.iUpdateCounter )
        &&( m_dblInfoWriteInterval_sec != 0 )
        &&( qfAuxFile.fOpened )    )
   {
     WriteChunkInfo( m_dwSamplePointsWritten/*index*/, pChunkInfo, pszExtraInfo );
     m_iPosUpdateCounter_WR = pChunkInfo->gps.iUpdateCounter;
   } // end if < periodically emit 'position info' (with GPS data, etc) ?
#endif

  // Allocate a temporary buffer, in which the 8-, 16-, or 24-bit values
  // can be built, before flushing them to the wave file :
  nBytesToSend = (DWORD)m_ChunkFormat.wBlockAlign * nSamplePoints;
  pbTempBuffer = new BYTE[ nBytesToSend ];
  if( pbTempBuffer == NULL )
   { sprintf(m_cErrorString,"out of memory in WriteSamples");
     return -1;
   }
  // NO LAZY RETURN AFTER THIS POINT ! Don't forget to free that buffer....
  memset( pbTempBuffer, 0, nBytesToSend );
  pb = pbTempBuffer;            // buffer used for 8-bit unsigned integer
  pI16 = (SHORT*)pbTempBuffer;  // same buffer used for 16-bit integer

  // For best speed, use DIFFERENT LOOPS for each of the supported
  // "bits per sample" (strictly: bits per integer in each channel of a sample point)
  switch(m_ChunkFormat.wBitsPerSample)  // which sample-resolution, 8, 16 or 24 bit ?
   {
     case  8:   // input: normalized floats (-1.0 .... +1.0) ,
                // output: UNSIGNED 8-bit   (  0  .... 255 ) .
        fltScalingFactor = 127.0/* 2^7 - 1 */
          /  pChunkInfo->dblSampleValueRange; // works with +/- 1.0 but also with +/-32767 sample value range !
        nSamplesWritten = nSamplePoints;
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for UNSIGNED 8-bit samples ( output value range 0..255 ) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // up to 8 bits per sample:
              flt = 127.0 + pfltSrc[iChannel] * fltScalingFactor;
              if( flt< 0.0  ) flt= 0.0;
              if( flt> 255.0) flt= 255.0;
              pb[iChannel] = (BYTE)flt;
            } // end for < 8-bit UNSIGNED samples >
           pb      += nDestChannelsPerSample;   // next sample group in destination (unused channels already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // next sample group in SOURCE
         }
        break;

     case 16:   // input: normalized floats (-1.0   .... +1.0) ,
                // output: SIGNED 16-bit    (-32767 .... +32767 ) .
        fltScalingFactor = 32767.0/* 2^15 - 1 */
          /  pChunkInfo->dblSampleValueRange; // works with +/- 1.0 but also with +/-32767 sample value range !
        nSamplesWritten = nSamplePoints;
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for UNSIGNED 8-bit samples ( output value range 0..255 ) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // scale and copy 16 bits per sample:
              flt = pfltSrc[iChannel] * fltScalingFactor;
              if( flt<-32767.0 ) flt=-32767.0;
              if( flt> 32767.0 ) flt= 32767.0;
              pI16[iChannel] = (SHORT)flt;
            } // end for < 16-bit SIGNED samples >
           pI16 += nDestChannelsPerSample; // next sample group in destination (unused channels already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // next sample group in SOURCE
         }
        break;

     case 24:   // input: normalized floats (-1.0   .... +1.0) ,
                // output: SIGNED 24-bit    (-(2^23-1).. 2^23-1 ) .
        fltScalingFactor = 8388607.0/* 2^23 - 1 */
          /  pChunkInfo->dblSampleValueRange; // works with +/- 1.0 but also with +/-32767 sample value range !
        nSamplesWritten = nSamplePoints;
        iRest = nDestChannelsPerSample - nSourceChannelsPerSample;
        if( iRest<0 )  iRest=0;
        iRest *= 3; // ... bytes per sample for ONE channel
        while( nSamplePoints > 0 )
         { --nSamplePoints;
           // here for UNSIGNED 8-bit samples ( output value range 0..255 ) :
           for(iChannel=0; iChannel<nChannels; ++iChannel)
            { // scale and copy 24 bits per sample:
              flt = pfltSrc[iChannel] * fltScalingFactor;
              if( flt<-8388607.0 ) flt=-8388607.0;
              if( flt> 8388607.0 ) flt= 8388607.0;
              t.i32 = (long)flt;
              *pb++ = t.b[0];   // least significant byte (bits  7 ... 0)
              *pb++ = t.b[1];   // middle byte            (bits 15 ... 8)
              *pb++ = t.b[2];   // most significant byte  (bits 23 ...16)
            } // end for < 16-bit SIGNED samples >
           pb += iRest; // skip unused channels in destination (already ZEROED by memset)
           pfltSrc += nSourceChannelsPerSample; // inc for next sample in the SOURCE block
         } // end while
        // Arrived here, pb (pointer into destination buffer) should point here:
        if( pb != (pbTempBuffer + nBytesToSend ) )
         { iRest = pb-pbTempBuffer;  // <<< set breakpoint here !
         }
        break;

     default:  // non supported sample resolution ...
        sprintf(m_cErrorString,"bad BitsPerSample:%d",(int)m_ChunkFormat.wBitsPerSample );
        nSamplesWritten = -1;  // this is a 'permanent' error
        nBytesToSend = 0;
        break;
   } // end switch(m_ChunkFormat.wBitsPerSample)

  if( nBytesToSend>0 )
   { WriteSamples( pbTempBuffer, nBytesToSend );
   }

  delete[] pbTempBuffer;
  return nSamplesWritten;

} // end C_WaveIO::WriteSamples_Float(...)


/************************************************************************/
DWORD C_WaveIO::GetCurrentFileSize(void)
  /* Returns the current file size (if there is an opened WAVE file).
   * Useful especially when logging audio samples  to keep track
   * of the used disk size (stop before Windows is getting into trouble).
   */
{
 if (m_OpenedForWriting || m_OpenedForReading)
       return m_dwCurrFilePos;
  else return 0L;
} // end C_WaveIO::GetCurrentFileSize()

/************************************************************************/
  // Some 'Get' - and 'Set' - routines for the AUDIO FILE class ..
BOOL   C_WaveIO::SetFileFormat(int iAudioFileFormat)
{ // This is only important BEFORE opening an audio file,
  // so the 'wrapper' knows which module to call..
  // .. especially for RAW audio files .
  if(  (iAudioFileFormat!=AUDIO_FILE_FORMAT_RAW)
    && (iAudioFileFormat!=AUDIO_FILE_FORMAT_RIFF)
    && (iAudioFileFormat!=AUDIO_FILE_FORMAT_SETI_1) )
   { return FALSE;  // format not supported by this class
   }
  if( m_iFileFormat != iAudioFileFormat )
   { // close any OLD file before switching to the new format:
     CloseFile();
   } // if( m_iAudioFileFormat = iAudioFileFormat )
  m_iFileFormat = iAudioFileFormat;
  return TRUE;
}
int    C_WaveIO::GetFileFormat(void)
{ return m_iFileFormat;
}
double C_WaveIO::GetSampleRate(void)  // this is the FILE's sampling rate !
{ return m_dbl_SampleRate;
}
void   C_WaveIO::SetSampleRate(double dblSampleRate)
{ m_dbl_SampleRate = dblSampleRate;
}

//---------------------------------------------------------------------------
int    C_WaveIO::GetNrChannels(void)
{ return m_ChunkFormat.wChannels;
}

//---------------------------------------------------------------------------
void   C_WaveIO::SetNrChannels(int iNrChannels)
{ // Only applies to RAW FILES !  For RIFF WAVE files,
  //  the number of channels will be read from the file header .
  m_wRawFileNumberOfChannels = (WORD)iNrChannels;
  if(  (m_iFileFormat==AUDIO_FILE_FORMAT_RAW)
    || (m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1) ) // added 2010-05-10
   { UpdateRawFileParams();
   }
}

//---------------------------------------------------------------------------
int    C_WaveIO::GetBitsPerSample(void)
{ return m_ChunkFormat.wBitsPerSample;
}

//---------------------------------------------------------------------------
void   C_WaveIO::SetDataType(int iSizeOfDataType, int iDataTypeFlags)
{ // Only applies to RAW FILES !  For RIFF WAVE files,
  //  the number of bits per (single) sample will be read from the file header .
  // In Spectrum Lab, this function is called after the user has possibly
  //    modified some audio file format parameters in the
  //    'Audio File Properties' dialog (see AFilePrp.cpp) .
  m_iSizeOfDataType = iSizeOfDataType;
  m_iDataTypeFlags  = iDataTypeFlags;  // AUDIO_FILE_DATA_TYPE_SIGNED_I .. etc
  if(  (m_iFileFormat==AUDIO_FILE_FORMAT_RAW)
    || (m_iFileFormat==AUDIO_FILE_FORMAT_SETI_1) ) // added 2010-05-10
   { UpdateRawFileParams();
   }
}

//---------------------------------------------------------------------------
void C_WaveIO::GetDataType(
        int *piSizeOfDataType, // [out] 0=unknown, 1=BYTEs, 2=short INTs, 4=floats, 8=double's
        int *piDataTypeFlags)  // [out] data type flags, see AudioFileDefs.h
{
  if( piSizeOfDataType != NULL )
   { *piSizeOfDataType = m_iSizeOfDataType;
   }
  if( piDataTypeFlags  != NULL )
   { *piDataTypeFlags  = m_iDataTypeFlags;
   }
} // end C_WaveIO::GetDataType()


T_Float C_WaveIO::GetFreqOffset(void)
{ return 0.0;
}

T_Float C_WaveIO::GetNcoFrequency(void)
{ // only important for decimated files with I/Q-samples and complex mixing
  return 0.0;
} // end C_WaveIO::GetNcoFrequency()

BOOL   C_WaveIO::SetNcoFrequency(T_Float dblNcoFrequency)
{ // only important for decimated files with I/Q-samples and complex mixing
  dblNcoFrequency = dblNcoFrequency;
  return FALSE;  // not supported
} // end C_WaveIO::SetNcoFrequency()


double C_WaveIO::GetCurrentRecordingTime(void)
{ double d;   // no "T_Float" here, we need 64 bit floats to encode the time !
  d = (double)GetCurrentSampleIndex();
  if(m_dbl_SampleRate>0)
     d /= m_dbl_SampleRate;
  return d;    // **NOT** "+ Start Time" ! ! !
}
double C_WaveIO::GetRecordingStartTime(void)
{ return m_dblStartTime; // seconds elapsed since 00:00:00 GMT, January 1, 1970.
}
void   C_WaveIO::SetRecordingStartTime(double dblStartTime)
{ m_dblStartTime = dblStartTime;
}


void C_WaveIO::GetFileName(char *pszDest, int iMaxLen)
{ strncpy( pszDest, m_sz255FileName, iMaxLen );
}


bool   C_WaveIO::AllParamsIncluded(void)
{ return FALSE;
}



/***************************************************************************/
BOOL C_WaveIO::CloseFile( void )
  /* Closes the WAV-file (if opened for READING exor WRITING.
   * Returns TRUE on success and FALSE on any error.
   */
{
 BOOL ok=TRUE;
#if (!WAVEIO_USE_QFILE)
 int  n_written;
#endif
 T_WAV_CHUNK_HDR  chunk_hdr;

#    if (WAVEIO_USE_QFILE)
  if (m_OpenedForWriting && (m_QFile.iHandle>=0) )
#    else
  if (m_OpenedForWriting && (m_FileHandle>=0) )
#    endif
   {
    /* If the file has been opened for WRITING, we may have to correct
     * some header entries (depending on the file type & size, because we did
     *  not know how long the file was going to get when writing the header).
     */
    if(  (m_iFileFormat==AUDIO_FILE_FORMAT_RIFF)
      && (m_dwFilePos_Data > (long)sizeof(T_WAV_CHUNK_HDR) ) )
     { /* It's a RIFF WAVE file, and the position
        * of the "data" chunk structure is known:
        * wind the file position back to 'data chunk header'...
        */
#    if (WAVEIO_USE_QFILE)
      if( QFile_Seek( &m_QFile, m_dwFilePos_Data-sizeof(T_WAV_CHUNK_HDR), SEEK_SET ) > 0)
#    else
      if( lseek( m_FileHandle, m_dwFilePos_Data-sizeof(T_WAV_CHUNK_HDR), SEEK_SET ) > 0)
#    endif
       {
        strncpy(chunk_hdr.id, "data", 4);
        // The size of the "data" chunk is known now, write it to the file:
        chunk_hdr.dwChunkSize = m_dwFileDataSize;
#      if ( WAVEIO_USE_QFILE )
        ok = QFile_Write( &m_QFile, (BYTE*)&chunk_hdr, sizeof(T_WAV_CHUNK_HDR));
#      else
        n_written = _rtl_write(m_FileHandle, &chunk_hdr, sizeof(T_WAV_CHUNK_HDR));
        if (n_written != sizeof(T_WAV_CHUNK_HDR) ) ok = FALSE;
#      endif
       } /* end if <seeking the "data"-chunk length entry ok> */
      if (ok)
       { /* If all accesses up to now were ok, also update the "total" file
          * size in the RIFF header.
          */
         m_RiffHdr.len = m_dwFilePos_Data + m_dwFileDataSize;
         // Rewind to the beginning of the file
         //     (where the RIFF header should be :)
#    if (WAVEIO_USE_QFILE)
         if( QFile_Seek( &m_QFile, 0L, QFILE_SEEK_SET ) == 0)
#    else
         if( lseek( m_FileHandle, 0L, SEEK_SET ) == 0)
#    endif
          {
#         if ( WAVEIO_USE_QFILE )
           ok &= QFile_Write( &m_QFile,  (BYTE*)&m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
#         else
           n_written = _rtl_write(m_FileHandle, &m_RiffHdr, sizeof(T_WAV_RIFF_HDR) );
           if( n_written != sizeof(T_WAV_RIFF_HDR) )
            { // writing the size-corrected RIFF header failed:
              ok = FALSE;
            }
#         endif
          }
       } /* end if <ok to correct the file size in the RIFF header> */
     } /* end if  <RIFF WAVE  and  m_dwFilePos_Data > sizeof(..)     */
   } // end of (..OpenedForWriting)


  /* Now close the file handle, no matter if it was for READING or WRITING.
   */
#  if (WAVEIO_USE_QFILE)
   QFile_Close( &m_QFile ); // sets the handle invalid by itself
#  else
   _rtl_close(m_FileHandle);
   m_FileHandle = -1;  // must forget this handle now !!
#  endif

  m_OpenedForReading = FALSE;
  m_OpenedForWriting = FALSE;

  return ok;
} // C_WaveIO::CloseFile(..)



/* EOF <WaveIO.cpp> */



