//---------------------------------------------------------------------------
//
// File:    C:\cbproj\SoundUtl\VorbisStreamServer.cpp  (module prefix VSS_)
// Date:    2014-04-29
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Output server for 'raw TCP' Ogg/Vorbis or non-compressed audio streams,
//          compatible with Paul Nicholson's VLF RX Toolchain .
//     NOT FOR HTTP, but "raw" TCP/IP !
//          (for HTTP-based streaming, see c:\cbproj\SpecLab\SpecHttpSrv.cpp)
//
//  Parametrized via dialog in c:\cbproj\SpecLab\AudioStreamServer_GUI.cpp  :
//       > Used when SL shall act as *server* for *outgoing* streams,
//       > i.e. SL waits until a remote client initiates the connection .
//
//  If HTTP (not 'raw TCP') is used as the transfer protocol,
//      outgoing samples are *pulled* from the 'Vorbis Stream Server'
//      via Spectrum Lab's built-in HTTP server :
//      Remote client requests "_audiostream.ogg"
//       -> c:\cbproj\SpecLab\SpecHttpSrv.cpp :: OnHttpFileRequestOrCompletion()
//          calls OnHttpAudioStreamRequest()
//          -> VorbisStreamOutputServer.GetInitialHeadersForStreamOutputServer();
//
// Latest revisions:
//  2014-04-29 : Put this module to work for COMPRESSED timestamped streams,
//               based on top of HTTP ("_audiostream.ogg"), not raw TCP/IP.
//  2020-10-25 : Decided to add the VLF-RX-tools-compatible SERVER
//               for timestamped, non-compressed streams
//                    in C:\cbproj\SoundUtl\VorbisStreamServer.cpp ,
//                NOT in C:\cproj\SoundUtl\AudioViaTCP.C ,
//               because all the VLF-TX-tool-compatible handling of
//               timestamps (in VT_BLOCKs) was already there, for
//               *compressed* streams (via HTTP: "_audiostream.ogg" ).
//               Most of the API (application interface) remains unchanged:
//
//---------------------------------------------------------------------------

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#pragma hdrstop  // no precompiled headers after this ..

//#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include "wsock.h"  // old wrapper for some "winsock" services, located in YHF_Tools

#pragma hdrstop  // no precompiled headers after this ..

#include "Timers.h"      // TIM_GetCurrentUTC() (->UNIX format), used for logfile,
                         // high-resolution 'current time'-function, T_TIM_Stopwatch, etc
#include "utility1.h"    // some helper functions by DL4YHF
#include "ErrorCodes.h"  // Combination of WINDOWS error codes + DL4YHF's own
#include "YHF_Inet.h"    // DL4YHF's 'internet' helper functions (uses sockets)
#include "http_intf.h"  // HTTP server interface, also included by the application
#include "audiomsg.h"   // \CBproj\SoundUtl\audiomsg.c : "audio messages" (sent through TCP/IP etc)
#include "VorbisStreamServer.h" /* Server for Vorbis-compressed 'TCP' streams */



// Global variables for the 'Vorbis Stream Server' :
BOOL VSS_enabled = FALSE;



// Global "output" buffer, filled from real-time audio processing thread,
//   read by the audio-via-TCP-server to send this data :
#define VSS_TX_BUFFER_SIZE 65535  /* Number of 16-bit entries. Must be a power of two ! */
// static short  VorbisStreamServer_i16TxAudioBuffer[ VSS_TX_BUFFER_SIZE ]; // TX-buffer for uncompressed audio data
// static DWORD  VorbisStreamServer_dwTxBufferHeadIdx;   // head index for the TX-buffer (used by multiple readers)
// static double VorbisStreamServer_dblTxBufSampleRate; // precise sampling rate of the data in VorbisStreamServer_i16TxAudioBuffer
// static double VorbisStreamServer_dblUnixDateAndTime; // timestamp for the LATEST (newest) sample in "

typedef enum  { PS_CONNECTING, PS_SENDING_AUDIO, PS_RECEIVING_AUDIO } tProtState;
typedef enum  { RS_WAIT_HEADER,RS_HEADER, RS_AUDIO_SAMPLES          } tRxState;


// CONNECTION structure (used by the audio-via-TCP SERVER, one of these for each 'accepted client')
typedef struct tVSSConnection
{
  SOCKET  Socket;
  tProtState prot_state;
  tRxState   rx_state;
  T_AudioMsgAnyHeader hdr;    // any kind of audio-message-header
  long  i32RxByteCount;       // number of bytes received for the present eRxState
  long  i32RxBytesWanted;     // number of bytes to receive (usually audio samples)
  SHORT *pi16RxAudioBuffer;   // buffer for received audio samples, usually points to ACTCP_i16RxAudioBuffer[]
  BOOL  fWaitingToContinueWrite; // added 2008-01-02, see long story in OnWrite() !
  DWORD dwConnectTime;        // value of GetTickCount() when the connection was created
  DWORD dwRecv; // statistics: number of bytes received for this connection (from server's point of view)
  DWORD dwSend; // statistics: number of bytes sent for this connection (from server's point of view)
} T_VSSConnection;

#define VSS_SRV_MAX_CONNECTIONS 10 /* max number of clients to which the server may be connected */
static T_VSSConnection VSS_server_connections[VSS_SRV_MAX_CONNECTIONS];

static T_AudioFileInfoCallback VSS_pCallback = NULL; // address of a user callback function,
                                    // invoked via 'CallBack()' and nothing else
static void *VSS_pvCallbackData = NULL; // user-defined address passed to the callback function

// Internal linkage
static SOCKET VSS_listenSocket = INVALID_SOCKET;  // socket to listen for incoming connects

static int VSS_iHttpHdlrBusy = 0; // prevents calling the periodic handler while we're processing async msgs


// Forward declarations (prototypes for "internal" functions, for what it's worth)
static void VSS_OnContinueWrite( T_VSSConnection *pSrvConn );
static void VSS_OnAccept( SOCKET socket, int nErrorCode);
static void VSS_OnRead(SOCKET socket, int nErrorCode);
static void VSS_SendHttpString( T_VSSConnection * pSrvConn, char *pszMsg );
static void VSS_OnWrite(SOCKET socket, int nErrorCode);
static void VSS_OnContinueWrite( T_VSSConnection *pSrvConn );
static void VSS_OnClose(SOCKET socket, int nErrorCode);
static void VSS_BeginToSendData( T_VSSConnection * pSrvConn );
static void VSS_SendData( T_VSSConnection * pSrvConn );
static void VSS_CloseConnection( T_VSSConnection * pSrvConn );



//===========================================================================
//  Common functions (for both SERVER- and CLIENT functions)
//===========================================================================

//---------------------------------------------------------------------------
void VSS_Init( void )   // should be called ONCE to initialize this module
{
} // end VSS_Init()


//---------------------------------------------------------------------------
static int CallBack(
               int iEventType, // one of the VORBIS_IO_EVENT_TYPE_.. - values
               char *pszInfo, // string parameter: Info, Error message, or Server response
               int iParam1 )  // integer parameter: progress percentage or error code
{
  int iResult = 0;
  if( VSS_pCallback != NULL )
   { iResult = VSS_pCallback(
                    VSS_pvCallbackData, // [in] address of any user-speficed data
                    iEventType, // one of the following (or maybe a bit combination?)
                           //  AUDIO_FILE_IO_EVENT_TYPE_INFO  (0)
                           //  AUDIO_FILE_IO_EVENT_TYPE_ERROR (1)
                           //  AUDIO_FILE_IO_EVENT_TYPE_PARSE_RESPONSE (2)
                           //  AUDIO_FILE_IO_EVENT_FLAG_SHOW_TIMESTAMP (8)
                    pszInfo,   // info string or HTTP response
                    iParam1);  // progress percentage or numeric error code
     // In Spectrum Lab, the above call takes us from
     //   StartAudioStream() -> C_VorbisFileIO::OutOpen() -> CallBack_Info() -> CallBack()
     //     to AudioStreamOut_GUI.cpp::MyVorbisIOCallback(), where the strings
     //     passed this way will be appended to the connection log .
   }
  return iResult;
} // end CallBack()

//---------------------------------------------------------------------------
static void CallBack_Info( char *pszFormat, ... )
  // Text messages 'emitted' this way will appear on SL's
  //      "Analyse / play audio stream" control panel,
  //      through a long-and-winding (but thread-safe) road
  //      explained in CallBack_Error() further below .
{
  char sz255[256], *cp;
  va_list arglist;
  va_start(arglist, pszFormat);
  UTL_FormatDateAndTime("hh:mm:ss.s ", TIM_GetCurrentUTC(), sz255 );
  cp = sz255+strlen(sz255);
  vsnprintf(cp, 255-(cp-sz255), pszFormat, arglist);
  va_end(arglist);
  CallBack( AUDIO_FILE_IO_EVENT_TYPE_INFO, sz255, 0 );
} // end CallBack_Info()

//---------------------------------------------------------------------------
static void VSS_LogError( double dblUnixTime, // use 0 for "current" time
                            char *pszFormat, ... )
  // Error messages 'emitted' this way will appear on SL's
  //      "Analyse / play audio stream" control panel,
  //      through a long-and-winding (but thread-safe) road
  //      explained further below .

{
  char sz255[256], *cp;
  va_list arglist;
  va_start(arglist, pszFormat);
  if( dblUnixTime <= 0.0 )
   {  dblUnixTime = TIM_GetCurrentUTC();
   }
  UTL_FormatDateAndTime("hh:mm:ss.s ", dblUnixTime, sz255 );
  cp = sz255+strlen(sz255);
  vsnprintf(cp, 255-(cp-sz255), pszFormat, arglist);
  va_end(arglist);
  CallBack( AUDIO_FILE_IO_EVENT_TYPE_ERROR, sz255, 0 );
} // end VSS_LogError()


//---------------------------------------------------------------------------
void VSS_LogInfo( char *pszFormat, ...)
{
  va_list arglist;   // really arg listig, this arglist-thingy :o)
  char szTemp[384];

  // Write text to string and append to event list
  va_start(arglist, pszFormat);
  vsnprintf( szTemp, sizeof(szTemp)-1, pszFormat, arglist);
  va_end(arglist);
  CallBack( AUDIO_FILE_IO_EVENT_TYPE_INFO, szTemp, 0 );

}


//---------------------------------------------------------------------------
void VSS_LogWSockError( LPCSTR lpText, int nErrorCode)
{
  VSS_LogError( 0.0, "%s : %s", lpText, WSOCK_ErrorCodeToString( nErrorCode ) );
}



//---------------------------------------------------------------------------
void VSS_DoNetworkStuff(void)
  // Check for timeouts, continues transmission with "limited bandwidth", etc .
  // Periodically called (every 50ms) from the application's "window procedure",
  //  possibly using a WM_TIMER event or something similar .
{
  int i;
  T_VSSConnection *pSrvConn;


  if( VSS_iHttpHdlrBusy > 0 )
     return;  // don't enter while we're working in VSS_HandleAsyncMsg(),
              // and prevent recursive calls while we're still "in here" .

  ++VSS_iHttpHdlrBusy;        // no "lazy return" after this !

  // Loop through all "connections" (in the HTTP server), and check if any of them
  // needs our special attention.
  // This feature was added 2008-01-02 for "slow transfers" in Spectrum Lab .
  // On this occasion, we can also update some statistics and housekeeping .
  for(i=0; i<VSS_SRV_MAX_CONNECTIONS ; ++i)
   { pSrvConn = &VSS_server_connections[i];

     // Is a client connected here, and waiting for something ?
     if( (pSrvConn->Socket!=INVALID_SOCKET) && (pSrvConn->fWaitingToContinueWrite) )
      { VSS_OnContinueWrite( pSrvConn );
      }

   } // end for

  if( VSS_iHttpHdlrBusy>0 )
    --VSS_iHttpHdlrBusy;

} // end VSS_DoNetworkStuff()


//---------------------------------------------------------------------------
BOOL VSS_Start(  // Starts "serving" audio streams to multiple clients, i.e. starts LISTENING for clients on a 'raw TCP port' (not HTTP).
   int iAudioServerPort, // port number to connect the audio server via TCP/IP
   T_AudioFileInfoCallback pInfoCallback) // [in, optional] progress indicator callback (or similar)
  //
  // See also : AVTCP_StartServer() [alternative, used in Spectrum Lab] !
  //
{
  SOCKADDR_IN  saServer;
  char  szBuf[256];
  int   i,nRet;

  // Init a few internal variables..
  memset( VSS_server_connections, 0, sizeof(VSS_server_connections) );
  for(i=0; i<VSS_SRV_MAX_CONNECTIONS; ++i)
   { VSS_server_connections[i].Socket = INVALID_SOCKET;  // marks an UNUSED ENTRY
   }

  // Save pointers to the callback functions ..
  VSS_pCallback = pInfoCallback;
  VSS_pvCallbackData = NULL;  

  // Save the Window handle and windows message ID for further use
  VSS_iHttpHdlrBusy = 0;

  // Create a TCP/IP stream socket
  VSS_listenSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP );
  if (VSS_listenSocket == INVALID_SOCKET)
   {
     VSS_LogWSockError( "Could not create listen socket", WSAGetLastError());
     return FALSE;
   }

  // Always use the specified port number.
  //  Since this is not a standard service like HTTP, getservbyname doesn't work.
  saServer.sin_port = htons( (u_short)iAudioServerPort);

  // Fill in the rest of the address structure
  saServer.sin_family = AF_INET;
  saServer.sin_addr.s_addr = INADDR_ANY;

  // Bind our name to the socket
  nRet = bind(VSS_listenSocket, (LPSOCKADDR)&saServer, sizeof(struct sockaddr));
  if (nRet == SOCKET_ERROR)
   {
     VSS_LogWSockError( "bind() error", WSAGetLastError());
     closesocket(VSS_listenSocket);
     VSS_listenSocket = INVALID_SOCKET;
     return FALSE;
   }

  // Set the socket to listen
  nRet = listen(VSS_listenSocket, SOMAXCONN);
  if (nRet == SOCKET_ERROR)
   {
    VSS_LogWSockError( "listen() error", WSAGetLastError());
    closesocket(VSS_listenSocket);
    VSS_listenSocket = INVALID_SOCKET;
    return FALSE;
   }

  // Display the host name and address
  gethostname(szBuf, sizeof(szBuf));
  VSS_LogInfo( "Started Audio-via-TCP Server on %s:%d",  // computer_name:port
                   szBuf, htons(saServer.sin_port));
  return TRUE;
} // end VSS_Start()

//---------------------------------------------------------------------------
void VSS_Stop(void)
{
  int i;
  T_VSSConnection *pSrvConn;

  // Close the listening socket
  closesocket(VSS_listenSocket);
  VSS_listenSocket = INVALID_SOCKET;

  // Close all open connections
  for(i=0; i<VSS_SRV_MAX_CONNECTIONS ; ++i)
   { pSrvConn = &VSS_server_connections[i];
     if( pSrvConn->Socket != INVALID_SOCKET )
      { closesocket(pSrvConn->Socket);
        pSrvConn->Socket = INVALID_SOCKET;
      }
   }

  VSS_LogInfo("Audio-via-TCP Server Stopped");

  // Forget callback vectors (after the last call of VSS_LogInfo)
  VSS_pCallback = NULL;


} // end VSS_Stop()

//---------------------------------------------------------------------------
BOOL VSS_IsRunning(void)
{
  return VSS_listenSocket != INVALID_SOCKET;

} // end VSS_IsRunning()

//---------------------------------------------------------------------------
T_VSSConnection * VSS_SocketToServerConnection( SOCKET sock )
  // Looks for a certain entry in the server's table of connections .
  // With sock=INVALID_SOCKET , this function can be used to find a free entry.
  // Return NULL means "not found" in the table .
{ int i;
  T_VSSConnection *pSrvConn;

  for(i=0; i<VSS_SRV_MAX_CONNECTIONS ; ++i)
   { pSrvConn = &VSS_server_connections[i];
     if( pSrvConn->Socket == sock )
      { return pSrvConn;
      }
   }
  return NULL;
} // end VSS_SocketToServerConnection()

//---------------------------------------------------------------------------
void VSS_HandleAsyncMsg( WPARAM wParam, LPARAM lParam)
  // Called from the application's "window procedure" after reception
  //  of the windows message for the Audio-via-TCP server
  //  (sent to the application from the windows socket service).
{
  int nErrorCode;

  ++VSS_iHttpHdlrBusy;        // no "lazy return" after this !

  nErrorCode = WSAGETSELECTERROR(lParam);

  switch(WSAGETSELECTEVENT(lParam))
   {
     case FD_ACCEPT:
        // Winsock received the connect() request and wants us
        //  to send back an accept() .
        VSS_OnAccept( (SOCKET)wParam, nErrorCode);
        break;

     case FD_READ:
        // Winsock notified us that the network buffer has more data for us
        // to collect.  Internally, we will call recv() . But caution, there are
        // many traps and pitfalls (even with this simple function, grrrr).
        // > The windows socket services will not continually flood an application
        // > with messages for a particular network event. Having successfully
        // > posted notification of a particular event to an application window,
        // > no further message(s) for that network event will be posted
        // > to the application window until the application makes the function
        // > call that implicitly reenables notification of that network event.
        // BUT (!) ...
        // > Any call to the reenabling routine, even one that fails, results
        // > in reenabling of message posting for the relevant event.
        // > For FD_READ, FD_OOB, and FD_ACCEPT events, message posting is
        // > level-triggered. This means that if the reenabling routine is called
        // > and the relevant condition is still met after the call,
        // > a WSAAsyncSelect message is posted to the application. This allows
        // > an application to be event-driven and not be concerned
        // > with the amount of data that arrives at any one time.
        // >         Consider the following sequence:
        // > - Network transport stack receives 100 bytes of data on socket s
        // >         and causes Windows Sockets 2 to post an FD_READ message.
        // > - The application issues recv( s, buffptr, 50, 0) to read 50 bytes.
        // > - Another FD_READ message is posted because there is still data to be read.
        // > With these semantics, an application need not read all available data
        // > in response to an FD_READ message  a single recv in response
        // >          to each FD_READ message is appropriate.
        VSS_OnRead( (SOCKET)wParam, nErrorCode);
        break;

     case FD_WRITE:
        // Winsock tells us that there is room in the buffer to send,
        //   and we must/should/may(?) send using the send() function .
        // 2008-01-02: Caution ! There may be a bug in microsoft's Winsock ...
        //  titled "BUG: Winsock--Lost FD_WRITE Notifications". From Q129974:
        // > SYMPTOMS
        // > In both of the following situations, the initial FD_WRITE notifications are never received:
        // > * Create a datagram socket and call WSAAsyncSelect() with interest
        // >   in FD_WRITE notification.
        // >
        // > -or-
        // > * Create a stream socket, call connect() to connect to a TCP server
        // >   and call WSAAsyncSelect() with interest in FD_WRITE notification.
        // >
        // For a better explanation of the FD_WRITE event, see VSS_SRV_OnWrite().
        VSS_OnWrite((SOCKET)wParam, nErrorCode);
        break;

     case FD_CLOSE:
        // Winsock tells us that the other sides socket has shut down.
        VSS_OnClose((SOCKET)wParam, nErrorCode);
        break;
   }

  if( VSS_iHttpHdlrBusy>0 )
    --VSS_iHttpHdlrBusy;

} // end VSS_HandleAsyncMsg()

//---------------------------------------------------------------------------
void VSS_OnAccept( SOCKET socket, int nErrorCode)
  // Called (indirectly) from Winsock service
  //  when a connection to our "audio-via-TCP-server" was requested.
{
  SOCKADDR_IN SockAddr;
  T_VSSConnection *pSrvConn;
  SOCKET      peerSocket;
  int         nRet;
  int         nLen;

#ifdef __BORLANDC__
  (void)nErrorCode; (void)socket;  // make BCB happy (parameter never used)
#else
  nErrorCode = nErrorCode; // make other compilers happy (parameter never used)
  socket = socket;
#endif

  // accept the new socket descriptor
  nLen = sizeof(SOCKADDR_IN);
  peerSocket = accept(VSS_listenSocket/*in*/, (LPSOCKADDR)&SockAddr/*out*/, &nLen/*out*/);
  if (peerSocket == SOCKET_ERROR)
   {
     nRet = WSAGetLastError();
     if (nRet != WSAEWOULDBLOCK)
      { // Just log the error and return
        VSS_LogWSockError( "accept() error", WSAGetLastError());
        return;
      }
   }

  // ex: Make sure we get async notices for this socket
  // nRet = WSAAsyncSelect(peerSocket, VSS_hMainWindow, VSS_iWindowsMsg, FD_READ | FD_WRITE | FD_CLOSE);


  // Too many connections, or will we REALLY accept this request
  //  (and add it to the list) ?
  pSrvConn = VSS_SocketToServerConnection( INVALID_SOCKET );  // search for an "empty" server-entry
  if( pSrvConn==NULL )
   { // number of "acceptable" connections already exceeded !
     closesocket(peerSocket);
     VSS_LogError(0, "Connection rejected from %s (too many connections)",
                       inet_ntoa(SockAddr.sin_addr) );
   }
  else
   {
     // Fill in the list of connected clients (THIS is the server)
     pSrvConn->Socket = peerSocket;
     pSrvConn->dwConnectTime = GetTickCount();
     pSrvConn->fWaitingToContinueWrite = FALSE;  // added 2008-01-02 for "slow sending" (bandwidth limited)
     pSrvConn->dwRecv = 0;
     pSrvConn->dwSend = 0;
     pSrvConn->prot_state = PS_CONNECTING;  // see state machine in VSS_OnRead()...
     pSrvConn->rx_state = RS_WAIT_HEADER;
     pSrvConn->i32RxByteCount = 0;
     VSS_LogInfo("Connection accepted on socket %d from %s",
                 (int)peerSocket, inet_ntoa(SockAddr.sin_addr) );
     // Shortly after this, we should get an "OnRead"-event for this socket !
   }
} // end VSS_OnAccept()


//---------------------------------------------------------------------------
void VSS_HandleRcvdBytes(T_VSSConnection *pSrvConn,
            BYTE *pbSource, int nBytesToProcess )
  // Handle the data, in the audio server,
  //  depending on what we are receiving at the moment :
  //  - "garbage/waiting for a header",  or
  //  - "header",  or
  //  - "audio samples" .
  // Fasten seat belt, we're using a tricky state machine...
{
  // BOOL fHeaderOk;
  // int  nBytesToProcessInBlock;
  // BYTE *pbDst;

  while( nBytesToProcess > 0 )
   {
     // What are we receiving at the moment; an audio-message-header,
     // or digitized samples, or are we just waiting for the arrival of
     // the next recognizeable pattern ?
     switch( pSrvConn->rx_state )
      { case RS_WAIT_HEADER: // waiting for the begin of a header
           break; // end case RS_WAIT_HEADER
        case RS_HEADER :     // receiving an AUDIO MESSAGE HEADER
           break; // end case RS_HEADER
        case RS_AUDIO_SAMPLES: // receiving AUDIO SAMPLES : Pack them into a buffer,
           break; // end case RS_AUDIO_SAMPLES
        default:  // oops.. switch back to the default state !
           pSrvConn->rx_state = RS_WAIT_HEADER;
           pSrvConn->i32RxByteCount = 0;
           break;
      } // end switch( pSrvConn->rx_state )
   }// end while( nBytesToProcess > 0 )
} // end VSS_HandleRcvdBytes()

//---------------------------------------------------------------------------
void VSS_OnRead(SOCKET socket, int nErrorCode)
  // Called (indirectly) from Winsock service
  //  when something shall be 'read' from the audio-via-TCP server (message FD_READ).
  //  This is usually the 2nd step after a connection was requested.
{
 BYTE  buf[2048];          // ex: "static", but we don't want static stuff in here
 char *cp;
 T_VSSConnection *pSrvConn;

 int          nRet;

#ifdef __BORLANDC__
  (void)nErrorCode;   // make BCB happy (parameter never used)
#else
  nErrorCode = nErrorCode; // make other compilers happy (parameter never used)
#endif

  // Find this socket in the linked list
  pSrvConn = VSS_SocketToServerConnection(socket);
  if (pSrvConn == NULL)
   {
     // Not in the list. Log the error, read the data to clear the buffers,
     // and close the connection
     nRet = 0;
     while(nRet != SOCKET_ERROR)
           nRet = recv(socket, (char*)buf, sizeof((char*)buf)-1, 0);
     closesocket(socket);
     return;
   }

  if( pSrvConn->prot_state != PS_RECEIVING_AUDIO ) // only if NOT "receiving audio" already..
   {
     // Recv the data
     //   For the FD_READ event, this is usually a string which tells us
     //   a lot about the visitor... including his OS and the browser he's using !
     //   A typical "request"-string easily exceeds 500 characters .
     // Zero the buffer so the recv is null-terminated
     memset(buf, 0, sizeof(buf));
     nRet = recv(socket, (char*)buf, sizeof( (char*)buf)-1, 0);
     if (nRet == SOCKET_ERROR)
      {
        if (WSAGetLastError() == WSAEWOULDBLOCK)
           return;
        VSS_LogWSockError( "recv()", WSAGetLastError() );
        VSS_CloseConnection(pSrvConn);
        return;
      }

     // Keep statistics
     pSrvConn->dwRecv += nRet;
   }
  else // pSrvConn->prot_state == PS_RECEIVING_AUDIO
   { // In this case, we will let the revc() put the data directly
     // in an audio sample buffer, to avoid unnecessary copy-operations .
     nRet = 0; // nothing received
#   ifdef __BORLANDC__
     (void)nRet;
#   endif
   }

  // Are we just CONNECTING, RECEIVING AUDIO, or what ?
  switch( pSrvConn->prot_state )
   { case PS_CONNECTING:   // we're just beginning to connect, "talking in text mode" (!)
        // "Let him in"  ?
        //   If the FD_READ event was caused by a web browser (HTTP-request),
        //   the string in <buf> will tells us a lot about the visitor,
        //   including his OS and the browser he's using !  For example:
        // buf = "GET /HTTP/1.1\r\nHost: hf-test2:1024\r\nUser-Agent: Mozilla/5.0..."
        // Here, in the AUDIO-VIA-TCP/IP-SERVER, we don't deal with HTTP .
        //   A typical "request"-string easily exceeds 500 characters .
        cp = (char*)buf;
        if( strnicmp( cp, "GET ", 4) == 0 )   // HTTP-"GET" not supported !
         { // No, send "bad request" error
           VSS_SendHttpString( pSrvConn, "400 Bad Request - this is not an HTTP server !");
           VSS_LogError(0, "AudioViaTCP rejected GET-request");
           VSS_CloseConnection( pSrvConn );
           return;
         }
        if( strnicmp( cp, "GET_AUDIO", 9) == 0 )
         { // client wants to receive an audio stream from this server .
           // let him join in (truly "accept" the connection) :
           pSrvConn->prot_state = PS_SENDING_AUDIO;
           VSS_LogInfo("AudioViaTCP accepted GET_AUDIO request");
           VSS_BeginToSendData( pSrvConn );
         }
        else
         { // No idea what to do with this request. Helplessly send an HTTP-like error:
           VSS_SendHttpString( pSrvConn, "501 Not Implemented" );
           VSS_LogError(0, "AudioViaTCP rejected unknown request");
           VSS_CloseConnection( pSrvConn );
           return;
         }
        break; // end case PS_CONNECTING
     case PS_SENDING_AUDIO:   // we're sending audio data (BINARY data blocks)
        break; // end case PS_SENDING_AUDIO
     case PS_RECEIVING_AUDIO: // we're receiving audio data (BINARY data blocks)
        do
         {
           //   BTW no need to call "memset" in this case...
           nRet = recv(socket, (char*)buf, sizeof( (char*)buf)-1, 0);
           if (nRet == SOCKET_ERROR)
            { // Is it really an ERROR, or is it just "zero bytes received" ?!
              if (WSAGetLastError() != WSAEWOULDBLOCK)
               { VSS_LogWSockError( "recv()", WSAGetLastError() );
                 VSS_CloseConnection(pSrvConn);
               }
              return;
            }
           // Got here: Something received. Keep the statistics ..
           pSrvConn->dwRecv += nRet;
           // .. and handle the received bytes, depending on the decoder state:
           VSS_HandleRcvdBytes( pSrvConn, buf, nRet );
         }while(nRet>0); // repeat this loop until the network buffer is empty (!)
        break; // end case PS_RECEIVING_AUDIO
     default:
        break;
   } // end switch (prot_state)

} // end VSS_OnRead()

//---------------------------------------------------------------------------
void VSS_OnWrite(SOCKET socket, int nErrorCode)
   // Called from .._HandleAsyncMsg() on the FD_WRITE event.
   //
   // As always, this function is more complicated than you may think !
   // From www.gamedev.net/reference/programming/features/asyncsock/page5.asp,
   //  written by Drew Sikora :
   //
   // > Now the FD_WRITE event is a bit more complicated. Mainly because
   // > of the way it is called. First of all, an FD_WRITE event is always
   // > generated when you first establish a connection with another socket.
   // > So you think all you have to do is plunk in a send() function
   // >              with whatever else youll need and youll be fine .
   // > Oh if that were only true. Listen carefully: the FD_WRITE event
   // > is triggered only when there is more room in the buffer with which
   // > to write data. Huh? Allow me to explain.
   // >
   // > First off, the buffer, as I use it here, is the network buffer,
   // > which you write information to. That information is then sent
   // > over the network (intranet or internet) to the receiving socket.
   // > So youd think that since there will always be space left to write data
   // > if you dont fill it up all at once, that FD_WRITE events will just
   // > keep coming and coming right? Wrong! Read the rules again.
   // > It says when there is more room. Not enough room, more room.
   // > This means you have to fill it up first! How do you do that?
   // >
   // > The general idea is to create an infinite while loop in which you will
   // > continuously send data until you max out the buffer. When is happens,
   // > send() will return the error WSAWOULDBLOCK. This means that if it were
   // > a blocking socket, it would wait (stop execution) for more room
   // > in the buffer and then send. But since it isnt, you get the error.
   // > So now that youve filled up the buffer, you just have to wait until
   // > more room becomes available so you can write again. And bingo!
   // > Up pops another FD_WRITE event.
   // > Do you have any idea how much trouble I had to go through to figure this out?
   // > You people are so darned lucky! Heres an example of an FD_WRITE event handler:
   // > (..)
   // > The implementation isnt so hard after all! It was only the concept
   // > that was messing with your head. If you use this technique, the loop
   // > will end when the network buffer becomes full.  (COMPLETELY FULL, that is..)
   // > Then, when more space is available - another FD_WRITE event will
   // > be triggered, and you can send more data.
   // > Before you jump to make use of your newfound knowledge, allow me
   // > to clarify the use of the FD_WRITE event. Do not expect to be using
   // > this event at all if you are not going to be sending large pieces of data
   // > at once. The reason for this is simple - if you base all you information
   // > transactions on FD_WRITE and never send enough to fill up the buffer,
   // > after that first trial FD_WRITE event generated at the beginning
   // > of the connection - no other FD_WRITE event will be triggered !
   // > Therefore for games, where you send as little data as possible,
   // > just forget about setting a socket for the FD_WRITE event and use send()
   // > wherever you need it.
   // > FD_WRITE is best used for transferring files, in my experience.
   //
   // DL4YHF: What a nice article, it contains everything which is MISSING
   //         (or at least hard to find) in the original winsock documentation.
{
  T_VSSConnection *pSrvConn;
  BYTE buf[1024];
  int nRet;

 #ifdef __BORLANDC__
  (void)nErrorCode;   // make BCB happy (parameter never used)
#else
  nErrorCode = nErrorCode; // make other compilers happy (parameter never used)
#endif

  pSrvConn = VSS_SocketToServerConnection(socket);
  if ( pSrvConn == NULL)
   {
     //
     // Not in our list ?  Dump the received data, and bark out .
     nRet = 0;
     while(nRet != SOCKET_ERROR)
           nRet = recv(socket, (char*)buf, sizeof( (char*)buf )-1, 0);
     closesocket(socket);
     return;
   }

  // Continue sending the stream (or whatever it is)
  VSS_SendData( pSrvConn );
} // end VSS_OnWrite()

//---------------------------------------------------------------------------
void VSS_OnContinueWrite( T_VSSConnection *pSrvConn )
   // Called periodically for a socket opened for SENDING DATA,
   //   for example for a file transfer which hasn't been sent completely.
   //   The reason why we need this function (in addition to HttpSrv_OnWrite)
   //   is explained in VSS_OnWrite() .
   // Caller: periodically, usually from some fancy (and jittery) windows timer .
{
   pSrvConn->fWaitingToContinueWrite = FALSE; // "done"
} // end VSS_OnContinueWrite()

//---------------------------------------------------------------------------
void VSS_OnClose(SOCKET socket, int nErrorCode)
{
  T_VSSConnection * pSrvConn;

#ifdef __BORLANDC__
  (void)nErrorCode;   // make BCB happy (parameter never used)
#else
  nErrorCode = nErrorCode; // make other compilers happy (parameter never used)
#endif

  // Have we already deleted this entry?
  pSrvConn = VSS_SocketToServerConnection(socket);
  if (pSrvConn == NULL)
     return;

  // It's still in our list: The client must have reset the connection.

  // Clean up.
  VSS_CloseConnection( pSrvConn );
}


//---------------------------------------------------------------------------
void VSS_CloseConnection( T_VSSConnection * pSrvConn )
{
  DWORD dwElapsedTime_ms;

  dwElapsedTime_ms = (GetTickCount() - pSrvConn->dwConnectTime);

  // Log the event ...
  VSS_LogInfo("Closing socket %d, ti=%.1f s, rcvd=%ld, sent=%ld byte.",
           (int)(pSrvConn->Socket),
          (float)dwElapsedTime_ms * 1e-3,
          (long)(pSrvConn->dwRecv),
          (long)(pSrvConn->dwSend)  );

  // Don't call HttpSrv_OnContinueWrite periodically (connection being closed !)
  pSrvConn->fWaitingToContinueWrite = FALSE;

  // Close the socket
  closesocket(pSrvConn->Socket);

  pSrvConn->Socket = INVALID_SOCKET; // this marks the list entry as "unused" !


} // end VSS_CloseConnection()


//---------------------------------------------------------------------------
void VSS_BeginToSendData( T_VSSConnection * pSrvConn )
  // Begin to send a file (requested from this HTTP server) .
  // Call Tree :
  //    VSS_SRV_HandleAsyncMsg()
  //      -> VSS_OnRead()        ("someone wants to read a file from us")
  //              -> VSS_BeginToSendData()
  //                   -> VSS_SRV_SendData()
{

  // Open the file for reading
  // if( 1 )
   {
     VSS_SendHttpString( pSrvConn, "404 Not Found" );
     VSS_CloseConnection( pSrvConn );
     return;
   }

  // Send as much data as we can
  // VSS_SendData( pSrvConn );
  // ( this is only the FIRST part of the audio stream ! )

} // end VSS_BeginToSendData()

//---------------------------------------------------------------------------
void VSS_SendHttpString( T_VSSConnection * pSrvConn, char *pszMsg )
{
  int nRet;
  static char szMsg[512];
  wsprintf(szMsg, "<body><h1>%s</h1></body>", pszMsg );
  nRet = send( pSrvConn->Socket, szMsg, strlen(szMsg), 0);
  if (nRet == SOCKET_ERROR)
   {
     if (WSAGetLastError() != WSAEWOULDBLOCK)
      {
        VSS_LogWSockError( "send()@SendError()", WSAGetLastError() );
        nRet = 0;
      }
   }
  pSrvConn->dwSend += nRet;
} // end VSS_SendHttpString()

//---------------------------------------------------------------------------
void VSS_SendData( T_VSSConnection * pSrvConn )
  // Server side: Sends "as much as possible" of the audio stream,
  //              which has already been prepared for sending.
{
  int nBytesToSend, nBytesSent;
  BYTE buf[4096];

  do
   {
     nBytesToSend = 0;

     // Send this buffer to the client
     nBytesSent = send( pSrvConn->Socket, (char*)buf, nBytesToSend, 0);
     if (nBytesSent == SOCKET_ERROR)
      { // This sometimes happens if the client has closed the connection,
        //  usually while debugging because the web browser "times out".
        if (WSAGetLastError() != WSAEWOULDBLOCK)
         {
           VSS_LogWSockError( "send()", WSAGetLastError() );

           VSS_CloseConnection( pSrvConn );
           return; // return immediately because pReq is not valid any longer
         }
        nBytesSent = 0;
      }

     // Keep track of what has actually been sent (for the statistics)
     pSrvConn->dwSend += nBytesSent;

     // Are the protocol stack buffers full?
     if (nBytesSent < (int)nBytesToSend)
      {
        // We'll have to finish later .  *ppReq remains valid in this case .
        return;
      }
   }while(nBytesToSend>0);


} // end VSS_SendData()





// EOF < VorbisStreamServer.cpp >
