/***************************************************************************/
/*  C:\cbproj\SoundUtl\VorbisFileIO.cpp :                                  */
/*   Unit to import and export *.ogg - files (Vorbis in Ogg container),    */
/*       and read (analyse) Ogg/Vorbis encoded streams                     */
/*      (using c:\cbproj\YHF_Tools\YHF_Inet.c::INET_ReadDataFromServer() ) */
/*  Written by DL4YHF, begun in May 2010 .                                 */
/*  Based on the WaveIO.cpp, the API is mostly compatible .                */
/*    Uses libogg V1.2.0 and libvorbis V1.3.1, locally saved in            */
/*           C:\cproj\ogg_vorbis\libvorbis_1_3_1\*.*   (audio codec)       */
/*      and  C:\cproj\ogg_vorbis\libogg_1_2_0\*.*      (stream container)  */
/*    Does NOT use any windows gimmics, just standard file I/O.            */
/*    The INTERFACE (*.h) doesn't even use the OGG / VORBIS headers !      */
/*                                                                         */
/*  Used in (at least) the following projects :                            */
/*     - DL4YHF's Spectrum Lab                                             */
/*                                                                         */
/* Short overview / most important functions for AUDIO STREAMING ("out"):  */
/* ----------------------------------------------------------------------  */
/*  After initialising an instance of class C_VorbisFileIO,                */
/*  periodically call WriteSamples_Float() from the DSP thread             */
/*     (in Spectrum Lab: SoundThd.cpp : SoundThd_WaveSaveProcess() ) .     */
/*                                                                         */
/*  Whenever a client connects, e.g. asking for "_audiostream.ogg" in SL,  */
/*             call GetInitialHeadersForStreamOutputServer() as explained  */
/*             in that method.                                             */
/*                                                                         */
/*  As long as the stream (e.g. TCP socket) is open and 'ready to send',   */
/*             call GetNextPagesForStreamOutputServer() as explained there.*/
/*                                                                         */
/*                                                                         */
/* Modification History :                                                  */
/*                                                                         */
/*  2022-03-05: Tried to fix Renato Romero's problem with timestamped VLF  */
/*              stream, where a backlash accumulated .  Details in         */
/*      bugfixes_and_notes\renato_problem_with_timestamped_VLF_streams.txt */
/*                                                                         */
/*  2020-10-14: Problem with DECIMATION in combination with streamd re-    */
/*              ceived through VorbisFileIO.cpp. From DF6NM :              */
/*    > Ein Problem scheint noch zu sein dass eine schmalbandige FFT       */
/*    > FFT mit Vordezimation (sowohl real, also auch komplex)             */
/*    > nicht richtig funktioniert, so als ob der Dezimationsfaktor        */
/*    > nicht angewendet wrde.                                            */
/*   ( -> postarchiv/df6nm/stream_49334_vorbis_vs_49337_uncompressed.png ) */
/*    > Das Problem mit der Dezimation taucht nicht von Anfang an auf,     */
/*    > sondern erst auf wenn ich den Stream stoppe und dann wieder "Start"*/
/*    > drcke (10:21:42 UT im Bildchen). Danach wirkt die Dezimation      */
/*    > (Faktor 4) nicht mehr vor der FFT. Die Skalierung mit              */
/*    > one-pixel-per-FFT-bin bercksichtigt den Faktor trotzdem, daher    */
/*    > erscheint das Spektrogramm "blockig". Dieser Effekt ist nicht neu, */
/*    > er trat auch jedesmal mit vorbis auf wenn der Stream aus           */
/*    > irgendeinem Grund unterbrochen war und wir versuch(t)en            */
/*    > ihn hndisch neu zu starten.                                       */
/*              Grep for the date (2020-10-14) to get the hole story.      */
/*                                                                         */
/*  2020-10-11: Added support for streams consisting of VT_BLOCKs only,    */
/*              as used by P. Nicholson's VLF-RX-tools .                   */
/*              New:  type  AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS .  */
/*              Tested with live VLF streams by DK7FC :                    */
/*       tcp://129.206.29.82:49336                                         */
/*            uncompressed, VT_BLOCKs with time stamps, 8 kS/s             */
/*       tcp://129.206.29.82:49335                                         */
/*            uncompressed, 8-byte floats, VT_BLOCKs, time stamps, 48 kS/s */
/*            (gave a reddish glow on the author's DSL line, 3 MBit/second)*/
/*                                                                         */
/*  2014-04-21: Added support for UNCOMPRESSED streams (for LAN streams).  */
/*              Grep for the date (2014-04-21) to find the modifications.  */
/*                                                                         */
/*  2013-07-05: Minor changes to support the PALAOA audio stream, which    */
/*              uses a new 'serial' for the Vorbis data each MINUTE (!).   */
/*              + Increased the initial timeout limit in VorbisRead(),     */
/*                because some of the VLF 'natural radio' streams          */
/*                were quite sluggish to deliver the first Ogg page.       */
/*                                                                         */
/*  2011-12-14: To receive the TIMESTAMP stream from Paul Nicholson's      */
/*              VLF-tools, the decoder (and the encoder) was rewritten     */
/*              so it doesn't use 'vorbisfile' anymore. The old version    */
/*              was rescued as VorbisFileIO_old_using_vorbisfile.cpp .     */
/*           To test SENDING timestamped Ogg/Vorbis streams locally, use   */
/*  C:\cbproj\Winsock_Experiments\TCP_Dummy_Sink_for_AudioStreams_Proj.exe */
/*              running on the same PC as Spectrum Lab .                   */
/*                                                                         */
/*  2010-07-02: When trying to put all OGG- and VORBIS libraries into      */
/*              a DLL ( project OggVorbis4SL.bpr -> *.dll, *.lib ),        */
/*              the stupid linker didn't find the functions listed below,  */
/*              even though the IMPORT LIBRARY (for the DLL) and the       */
/*              application which should use the DLL + LIB were compiled   */
/*              EXACTLY with the same compiler- & linker settings .        */
/*              Notice the leading underscores ..                          */
/*                 _ov_open_callbacks() , _ov_info(), _ov_read(),          */
/*                 _vorbis_info_init(), _vorbis_encode_init_vbr(),         */
/*                 _vorbis_comment_init(), _vorbis_comment_add_tag(),      */
/*                 _ogg_stream_init(), _vorbis_analysis_init(),            */
/*                 _vorbis_block_init(), _vorbis_analysis_headerout() ,    */
/*                 _ogg_stream_packetin(), _ogg_stream_flush(),   ....  !  */
/*             (In other words, the stupid thing didn't find ANY of the    */
/*              ogg- and vorbis function in the stupid import library )    */
/*               ->  temporarily switched back to the old projects,        */
/*                   which contains the ENTIRE OGG- & VORBIS sources       */
/*                   besides the ENTIRE Spectrum Lab sources. Holy Shit .  */
/*                 The DLL-using-project was rescued in                    */
/*           c:\cbproj\SpecLab\DevSteps\SpecLab_using_OggVorbis_SOURCE.zip */
/*       and c:\cbproj\SpecLab\DevSteps\SpecLab_using_OggVorbis_DLL.zip .  */
/*  To avoid garnishing zillions of Ogg/Vorbis functions (in their libs)   */
/*  with the stupid annoying declspec(dllexport)-stuff,  VorbisFileIO.cpp  */
/*  was also placed in OggVorbis4SL.dll + OggVorbis4SL.lib ,  and only     */
/*  those functions which REALLY needed to be in the stupid import lib     */
/*  were decorated with the "Deckelspeck" keywords . This is disgusting.   */
/*-------------------------------------------------------------------------*/


#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !
                       // (use the same SWITCHES when compiling SL and the DLL,
                       //  most likely \CBproj\SpecLab\SWI_V01\SWITCHES.H  )

#include <windows.h> // definition of data types like LONGLONG (wtypes.h) etc
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <winsock.h> // must be included before YHF_Inet.h ('SOCKET', etc)
#include <io.h>          /* _rtl_open and other file functions       */
#include <fcntl.h>       /* O_RDONLY  and other file flags          */
#include <time.h>
#include "Timers.h"      // TIM_GetCurrentUTC() (->UNIX format), used for logfile,
                         // high-resolution 'current time'-function, T_TIM_Stopwatch, etc
#include "utility1.h"    // some helper functions by DL4YHF
#include "ErrorCodes.h"  // Combination of WINDOWS error codes + DL4YHF's own
#include "YHF_Inet.h"    // DL4YHF's 'internet' helper functions (uses sockets)
#include "QFile.h"       // DL4YHF's "Quick File Functions" (for local files)
#include <StringLib.h>   // Header for all functions in DL4YHF's "String Library"
#ifndef C_SND_MATH_INVALID_FLOAT_VALUE
# include "SoundMaths.h"
#endif
// #include <exception>

#pragma hdrstop
#pragma warn -8004 // <var is a assigned a value that is never used> - so what


#define OV_EXCLUDE_STATIC_CALLBACKS /* don't need OV_CALLBACKS_DEFAULT & Co .. */
#include "vorbis/codec.h"      // C:\cbproj\ogg_vorbis\libvorbis_1_3_1\include\vorbis\codec.h
#include "vorbis/vorbisfile.h" // used for an 'easy start' to DECODE ogg files with vorbis audio
#include "vorbis/vorbisenc.h"

#define _I_AM_VORBIS_FILE_IO_ 1
#include "VorbisFileIO.h"      // header for THIS file
#include "AudioFileIO.h"       // AudioSampleDataTypeToString(), etc

#ifndef  int32_t
# define int32_t long
#endif
#ifndef  int64_t
# define int64_t  __int64
#endif
#ifndef  uint64_t
# define uint64_t unsigned __int64
#endif


// Bavarian debugging stuff ...
int  VorbisFileIO_iSourceCodeLine = __LINE__;  // last source code line (WATCH this!)
long VorbisFileIO_i32DesHobI = 0;              // last value 'i had'
#define DOBINI()   VorbisFileIO_iSourceCodeLine=__LINE__
#define DESHOBI(x) {VorbisFileIO_iSourceCodeLine=__LINE__;VorbisFileIO_i32DesHobI=(long)x;}


//
//  Stream header: to be sent once at start of stream
//
struct VT_OGG_HDR   /* compatible with Paul's VLF Tools -  don't modify ! */
{                   // This is the INITIAL special header.
   int32_t magic;   // Magic pattern, defined as HDR_MAGIC a few lines below
   int32_t chans;   // Number of samples per frame
   int32_t rate;    // Nominal sample rate
   int32_t spare;
};

#define HDR_MAGIC 0x1996c3a3  /* by Paul Nicholson - don't modify ! */

//
//  Data block: sent whenever new timestamp data comes in on the SR stream
//

struct VT_OGG_BLK   /* compatible with Paul's VLF Tools -  don't modify ! */
{
   int32_t magic;   // must contain BLK_MAGIC = 0x4f60f817 to be valid
   int32_t tbreak;  // Timestamp discontinuity detected
   uint64_t posn;   // Frame count to which the rest of this data applies
                    //  (WB: "frame count" = sample-point index ?? Don't bet on this !)
   int32_t spare1;
   int32_t spare2;
   int32_t secs;    // Timestamp, seconds
   int32_t nsec;    // and nanoseconds  (ex: microseconds)
   double srcal;    // Sample rate calibration factor (ideally 1.0)
   double glat;     // Geographic coordinates
   double glong;
   double gasl;     // Metres above sea level
};

#define BLK_MAGIC 0x4f60f817  /* by Paul Nicholson - don't modify ! */

struct tStreamDemux
{
   int serial;         // 2013-07-05: tStreamDemux.serial=-1 now marks an UNUSED entry
   ogg_stream_state os;  // type declared in c:\cbproj\ogg_vorbis\libogg_1_2_0\include\ogg\ogg.h
   int (*handler)(struct tVorbisFileHelper *pvfh, ogg_packet * op);
   int activity_detector; // added 2013-07-05 to recycle the OLDEST (orphaned) entry
};



// Internal data for any C_VorbisFileIO object. Later used for streams, too.
//    (We don't want to declare this stuff in the header VorbisFileIO.h
//     because it's purely INTERNAL. )
typedef struct tVorbisFileHelper // T_VorbisFileHelper ...
{
        // hex numbers in squared brackets are BYTE-OFFSETS within this struct.
        // We're compiling for a 32-bit target, thus pointers are 4 bytes .
  class C_VorbisFileIO *pThis; // [00] address of the class instance which 'owns' this helper-struct
  T_AudioFileInfoCallback pInfoCallback; // [04] address of a user callback function, set by C_VorbisFileIO::SetInfoCallback(),
                                    // invoked via 'CallBack()' and nothing else
  void *pvUserCallbackData;         // [08] address of something user-defined, passed along in the user callback function
  DWORD dwAlignmentDummy1;          // [0C] dummy to let 'bBuffer' begin on an 8-byte boundary

# define BUFFER_MAX_BYTES 65536     // buffer data and headers, for received and transmitted streams
  BYTE   bBuffer[BUFFER_MAX_BYTES]; // [10] used when the already received bytes turn out to be *NOT* an OggVorbis stream,
                                    // or as the circular FIFO for an outgoing stream with 'multiple readers'.
                                    // BEWARE: For efficient copying of 8-byte floating point samples
                                    //         (and 'LONGLONGs', ancient name for 64-bit integer),
                                    //         bBuffer must be aligned to 8-byte address boundaries.
                                    //         Added the byte-offset-counting for the first part
                                    //         of the
  DWORD  dwBufferUsage;             // [10010] ... BUFFER_MAX_BYTES
  DWORD  dwBufferIndex;             // [10014] 0 ... dwBufferUsage-1 (steps by one for each byte consumed from the buffer)
  LONGLONG i64NumTotalBytesProcessed; // [10018] for an extra criterion to process bytes "only in the VERY first byte"
# define BUFFER_MAX_PAGES 64        // max number of Ogg pages (or similar) in bBuffer[]
  struct
   { int iBufIndex;     // [10020,..] (byte-)index into T_VorbisFileHelper.bBuffer where this page begins
     int iSize_Byte;    // size of this Ogg page, measured in byte
     DWORD dwPageIndex; // unique serial number, starts counting OGG pages at zero
   }BufferPages[BUFFER_MAX_PAGES]; // "directory" of all Ogg pages currently stored in bBuffer.
                                   // Usage explained in GetNextPagesForStreamOutputServer().
  int    iBufferPagesHead; // index into BufferPages[] for the next page written in WriteOggPage()
  int    nBufferPagesUsed; // only BufferPages[0..nBufferPagesUsed-1] may be read in GetNextPagesForStreamOutputServer()
  DWORD  dwPageIndex;      // unique serial number, incremented in WriteOggPage(), copied to BufferPages[iBufferPagesHead++]
  T_ChunkInfo chunk_info;  // filled in ProcessHeaderFromNonCompressedStream()
  int    iStreamFormat;    // may contain one of the following :
# define STREAM_FORMAT_UNKNOWN        0
# define STREAM_FORMAT_OGG_VORBIS     1
# define STREAM_FORMAT_NON_COMPRESSED 2 // Non-compressed, but usually with T_StreamHeader blocks
# define STREAM_FORMAT_VLF_RX_BLOCKS  3 // also non-compressed, but segmented into VT_BLOCKs, without T_StreamHeaders
  BOOL   fStreamFormatDetectedFromData; // TRUE=yes (recognized iStreamFormat FROM THE INPUT STREAM/FILE);
                                        // FALSE=no (someone told us it's "non-compressed" but we can never be sure
                                        //           if it's STREAM_FORMAT_VLF_RX_BLOCKS or not.)
  BOOL   fMissingVTBlock;
  int    iVTFlags; // VTFLAG_FLOAT4, VTFLAG_FLOAT8, VTFLAG_INT1, VTFLAG_INT2, VTFLAG_INT4 (compatible with VLFRX-tools)
  int    iAudioFileOptions; // AUDIO_FILE_OPTION_NON_BLOCKING_CALLS, etc [-> AudioFileDefs.h]



  // Info about the received audio stream;
  //  set as soon as we know from the ogg/vorbis headers :
  double dblNominalSampleRate; // number of sample points  per second, nominal value, but possibly fractional !
  int    nChannelsPerSampleInStream;  // nomenclature: a "sample" (sample point) may contain more than one channel
  int    nChannelsPerSampleRequested; // number of active channels (per sample) requested by the caller.
                                      // Usually set in C_VorbisFileIO::ReadSampleBlocks(), but may be ZERO
                                      // if the application 'will accept anything' (since 2021-05-20).
  int    nSamplesPerHeaderInStream;   // number of sample-points (with one or more channels per point) per header or VT_BLOCK
  int    iSampleCountdownBetweenHeaders; // "countdown" for sample-points following a stream-header or VT_BLOCK .
                             // As long as this is nonzero, we expect another SAMPLE POINT
                             // in ReadAndDecodeNonCompressed(), but don't check
                             // for the next stream-header or VT_BLOCK yet .
  BOOL   fGotAllHeaders;     // flag set when received enough data to know the basic Vorbis parameters
                             // (or, for 'uncompressed streams with headers', got the relevant headers)
  BOOL   have_vorbis;        // added 2011-12-15
  BOOL   want_tstamp, have_tstamp;  // ... for both Encoder AND decoder !
  BOOL   fEncoderInit2;      // flag for the 'postponed' part of decoder-init


  // The following members of T_VorbisFileHelper are used for both
  //     'local files'   and   'internet streams' (usually HTTP) :
  BOOL fConnecting;
  BOOL fOpenedForReading;
  BOOL fOpenedForWriting;
  BOOL fPlainOggVorbis;  // PFLAG from Paul's vtvorbis
  BOOL fMustClearVorbisInfo,  // reminder to call vorbis_info_clear() [etc]
       fMustClearDSP, fMustClearBlock, fMustClearComment;


  // bitstream settings for the encoder (and decoder?) :
  ogg_stream_state osv,  // Stream state - vorbis stream
                   ost;  //              - timestamp stream
                         // An ogg_stream_state takes physical pages,
                         // weld into a logical stream of packets .
  ogg_sync_state   oy;   // oy: not a creature by Stephen King, but :
          // > ogg_sync_state tracks the synchronization of the current page.
          // > It is used during decoding to track the status of data
          // > as it is read in, synchronized, verified, and parsed into pages
          // > belonging to the various logical bistreams in the current
          // > physical bitstream link.
//ogg_packet       opack; // one raw packet of data for decode  .  ex "op" .
  vorbis_info      vinfo;
  vorbis_comment   vcomment; // struct that stores all the user comments
  vorbis_dsp_state vdsp;     // central working state for the packet->PCM decoder
  vorbis_block     vblock;   // local working space for packet->PCM decode
  int   nRcvdVorbisHeaders;  // Count of vorbis headers received (important: 3 headers when opening)

  ogg_packet header;
  ogg_packet header_comm; // not sure if it's ok to use a LOCAL (stack) var for this
  ogg_packet header_code;
  int   vt_packetno;

  #define MAX_DEMUX 10 // Added 2011-12-15, to demultiplex streams as in Paul's vtvorbis.c
  struct tStreamDemux demux[MAX_DEMUX];
  int    ndemux; // number of 'installed' demultiplexers in demux[]
                 // (2013-07-05: this may give problems with the Palaoa streams,
                 //              which seems to use new 'serials' after a bunch
                 //              of seconds ... grep for the modification date !
                 // 2013-07-05: tStreamDemux.serial=-1 now marks an UNUSED entry,
                 //             and tStreamDemux.activity_detector will be used
                 //             to recycle the OLDEST (orphaned) entry .

  LONGLONG i64NumSamplesWritten;
  LONGLONG i64NumSamplesWrittenAtLastTimestamp;
  long     nBitsPerSecond, nBitrateCounter;
  double   dblStartTimeOfBitrateMeasurement;
  double   dblTimestampLatency; // parameter readable as AUDIO_FILE_PARAM_TIMESTAMP_LATENCY
  T_TIM_Stopwatch stopwatch; // .. to measure and limit the time spent in ReadSampleBlocks() [and anything called from there], WAITING for input from the network
  int      iMaxTimeToSpendReadingSamples_ms; // <- set in ReadSampleBlocks(), tested against pvfh->stopwatch in any function called from there
  int      iWaitedForInput_ms; // number of milliseconds spent WAITING for data from the network,
                               // in the last call of C_VorbisFileIO::ReadSampleBlocks() .

  #define MAX_DECODER_OUTPUT_SIZE (4*65536)
  float  fltDecoderOutput[MAX_DECODER_OUTPUT_SIZE]; // internal buffer for the output from Vorbis decompressor
  int    nDecoderOutputBufferUsage;  // number of single floats READ INTO fltDecoderOutput[]
  int    nDecoderOutputBufferPeakUsage;  // peak value of the above, only for diagnostics
  int    iDecoderOutputReadIndex; // index to read the next sample from fltDecoderOutput[]
  long double dblStartTime;       // Unix timestamp of the FIRST sample read from the file or stream
  BOOL   fGotStartTimeFromTimestamp;
  // long i32CurrentFilePosition;


  // The following members of T_VorbisFileHelper are only used for LOCAL FILES:
#if (VORBIS_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
  T_QFile m_QFile;        // data structures and buffer for "quick file access"
  T_QFile qfStreamLog; // a different file to log webstreams (raw in- or output)
#else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland ("_rtl_"-nonsense):
  INT     m_FileHandle;
#endif


  // The following members of T_VorbisFileHelper are only used for INTERNET STREAMS:
  char  sz80ServerPassword[84];  // password; a remote audio server may ask for this
  char  sz255StreamLogfile[256]; // Name of the optional "webstream logfile"
  BOOL  fWriteLogfile;           // TRUE=yes, FALSE=no
  BOOL  fIsInternetStream;   // TRUE: using InetClient;  FALSE: using m_QFile.
  BOOL  fStoreSamplesForMultipleReaders;  //  used to encode audio for MULTIPLE remote clients (last few seconds of audio)
  T_INET_Client InetClient;  // DL4YHF's "internet client" (so far, only for HTTP).
                             // Implemented in C:\cbproj\YHF_Tools\YHF_Inet.c .
  int   inetReadOptions;     // for VorbisRead() : INET_RD_OPTION_WAIT_FOR_ALL_DATA or INET_RD_OPTION_NORMAL (=do NOT wait for 'all requested bytes')
  int   inetTimeout_ms;      // for VorbisRead() : maximum number of milliseconds to "wait for reception" .
                             // Since 2022-03, set in ReadSampleBlocks() depending on the number of samples,
                             // unless pvfh->iAudioFileOptions contains AUDIO_FILE_OPTION_NON_BLOCKING_CALLS.

  BOOL  fStoreInitialHeaders;              // used to encode audio for MULTIPLE remote clients (persistent headers)
  #define INITIAL_HEADER_BUFFER_SIZE 4096  // details in
  BYTE  bInitialHeaderBuffer[INITIAL_HEADER_BUFFER_SIZE];
  int   nInitialHeaderBufferedBytes;

  //
  // Timing queue for the DECODER (i.e. incoming audio stream with GPS timestamps).
  //   For streams with timestamps, compatible with Paul's VLF-tools .
  //   Here we save received VT_OGG_BLK packets while they wait for the
  //   associated vorbis audio to be decoded.
  #define MAXTQ 100
  struct VT_OGG_BLK tq[MAXTQ];
  int   iTimestampQueueLength;
  LONGLONG i64TimestampCounter;
  LONGLONG i64NumSamplesDecoded;      // Number of sample-points emitted by the DECODER (ex: nout)
  LONGLONG i64NumSamplesReadByApp;    // Number of sample-points which have been read by the APPLICATION
  LONGLONG i64PrevTstampSamplePosn;   // only to check timestamps for plausibility (which FAILED with a stereo stream)
  long double ldblPrevTstampUnixTime; // "  "   "      "         "    "


  // Timestamp checker for the ENCODER
  uint64_t nSamplesOutputAtLastEncoderTimestamp;
  double   dblUnixTimeAtLastEncoderTimestamp;

  // For debugging:
  int   nWriteErrors;
  int   nSuccessfullCallsOfReadAndDecode;

} T_VorbisFileHelper;

int VFIO_nErrorsInHistory = 0; // added 2016-01 to avoid flooding the error history

LONGLONG VFIO_i64InspectMeAfterDebuggerException = 0;

/*----- Internal functions / Callbacks / forward declarations ----*/
/*----------- NO CLASS MEMBERS OR CLASS METHODS HERE ! -----------*/

extern "C" int VorbisFileIO_InetCallback(
          void *pvUserCallbackData, struct tInetClient *pInetClient,
          int iEventType, char *pszInfo, int iProgressPercent);

//---------------------------------------------------------------------------
void VFIO_DebuggerBreak( T_VorbisFileHelper *pvfh, char * pszInfo, int iSourceLine )
{
  iSourceLine = iSourceLine;  // <<<< always set a debugger breakpoint here <<<<
  // if( VFIO_nErrorsInHistory < 100 )
  //  { ++VFIO_nErrorsInHistory;
  //    //No-No: DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, UTL_USE_CURRENT_TIME, pszInfo );
  //    //   The CALLER will enter relevant errors in the history, no need to duplicate that here
  //  }
}

char* VorbisErrorCodeToString( int vorbis_error_code )
{
  static char sz40UnknownCode[44];
  static char sz40NotAnError[44];
  if( vorbis_error_code > 0 )
   { sprintf(sz40NotAnError, "Code %d is not an error !",(int)vorbis_error_code);
     return sz40NotAnError;  // stupid caller, kick the bucket :)
   }
  // For an explanation, see vorbis(libvorbis_1_3_1)/doc/vorbis-errors.txt :
  switch( vorbis_error_code )
   {
     case 0         : return "ok (no error)";
     case OV_FALSE  : return "FALSE / info not available";    // not available because no playback, etc
     case OV_EOF    : return "End of file";
     case OV_HOLE   : return "lost packet(s)";  // 'hole in the data'
     case OV_EREAD  : return "media read error";
     case OV_EFAULT : return "internal fault";
     case OV_EIMPL  : return "feature not implemented";
     case OV_EINVAL : return "invalid argument value";
     case OV_ENOTVORBIS: return "not a Vorbis file/stream/packet";
     case OV_EBADHEADER: return "invalid Vorbis bitstream header";
     case OV_EVERSION  : return "Vorbis version mismatch";
     case OV_ENOTAUDIO : return "packet is no audio data";
     case OV_EBADPACKET: return "invalid packet submitted";
     case OV_EBADLINK  : return "invalid stream section / bad link";
     case OV_ENOSEEK   : return "bitstream is not seekable";
     // In additon to the above 'Ogg/Vorbis' error codes, added the following
     // more 'general' errors in 2020-10-11 :
     case OV_ERR_TIMEOUT: return "timeout";

     default: sprintf(sz40UnknownCode, "unknown error code (%d)", (int)vorbis_error_code );
        return sz40UnknownCode; // caller should process this result a.s.a.p. !
   }
} // end VorbisErrorCodeToString()

//---------------------------------------------------------------------------
char* StreamFormatToString(int iStreamFormat)
{ switch( iStreamFormat )
   { case STREAM_FORMAT_UNKNOWN        : return "unknown";
     case STREAM_FORMAT_OGG_VORBIS     : return "OggVorbis";
     case STREAM_FORMAT_NON_COMPRESSED : return "NonCompr.";
     case STREAM_FORMAT_VLF_RX_BLOCKS  : return "VLF-RX";
     default: return "???";
   }
} // end StreamFormatToString()

//---------------------------------------------------------------------------
char * VFIO_VTFlagsToString(int iVTFlags) // returns names compatible with AudioSampleDataTypeToString() !
{
  switch( iVTFlags & VTFLAG_FMTMASK )
   { case VTFLAG_FLOAT4: // 4 byte floats
        return "float32";
     case VTFLAG_FLOAT8: // 8 byte floats
        return "float64";
     case VTFLAG_INT1  : // 1 byte signed integers
        return "int8";
     case VTFLAG_INT2  : // 2 byte signed integers
        return "int16";
     case VTFLAG_INT4  : // 4 byte signed integers
        return "int32";
     default:

        return "unknown";

   }
} // end VFIO_VTFlagsToString()


  // Looking for more of these short little ..ToString() helpers in plain "C" ?
  //  * AudioSampleDataTypeToString()  in AudioFileIO.cpp,
  //  * ChunkInfoToString()            in ChunkInfo2.c,
  //  * CLI_FormatNumberToString()     in C:\cbproj\SpecLab\Cli_rout.cpp
  //  * ErrorCodeToString()            in C:\cbproj\SoundUtl\ErrorCodes.cpp
  //  * UTL_TimeToString()             in C:\cbproj\SoundUtl\utility1.cpp


//---------------------------------------------------------------------------
BOOL VFIO_StringToVTFlags(
        char **ppcSource,  // [in,incremented] : "Ogg", "OggVorbis", "Uncompressed" [,"int8" / "int16" / .. ]
        int *piVTFlags )   // [in,out]: VTFLAG_INT2_.. (preset with meaningful default by caller)
{
  int iTokenIndex = QFile_SkipStringFromList( ppcSource, "int8\0int16\0int32\0float32\0float64\0\0" );
  if( piVTFlags != NULL )
   { *piVTFlags &= ~VTFLAG_FMTMASK;  // leave other flags unchanged (only modifiy the data-type-flags)
     switch( iTokenIndex )
      { case 0 : *piVTFlags |= VTFLAG_INT1; // 1 byte signed integers
                 break;
        case 1 : *piVTFlags |= VTFLAG_INT2; // 2 byte signed integers
                 break;
        case 2 : *piVTFlags |= VTFLAG_INT4; // 4 byte signed integers
                 break;
        case 3 : *piVTFlags |= VTFLAG_FLOAT4; // 4 byte floats
                 break;
        case 4 : *piVTFlags |= VTFLAG_FLOAT8; // 8 byte floats
                 break;
        default: break;  // ??
      }
   }
  return iTokenIndex >= 0;
}

//---------------------------------------------------------------------------
BOOL VFIO_StringToAudioFileFormatAndVTFlags(
        char **ppcSource,             // [in,incremented] : "Ogg", "OggVorbis", "Uncompressed" [,"int8" / "int16" / .. ]
        int *piAudioFileFormat,       // [in,out]: AUDIO_FILE_FORMAT_.. (preset with the default by caller)
        int *piVTFlags )              // [in,out]: VTFLAG_INT1/2/4 , VTFLAG_FLOAT4/8 (for non-compressed streams; also preset by caller)
  // Used, for example, in AudioStreamOut_GUI.CPP to parse the configuration file.
{
  char *cp = *ppcSource;
  int iAudioFileFormat = AUDIO_FILE_FORMAT_OGG;
  int iVTFlags = VTFLAG_INT2;
  int iTokenIndex;
  BOOL fOk = FALSE;
  if( piAudioFileFormat != NULL )
   { if (*piAudioFileFormat >= 0 )
      {    iAudioFileFormat = *piAudioFileFormat;
      }
   }
  if( piVTFlags != NULL )
   { if (*piVTFlags >= 0 )
      {    iVTFlags = *piVTFlags;
      }
   }
  // First part of the string: "ogg","vorbis", ..
  iTokenIndex = QFile_SkipStringFromList( &cp, "ogg\0vorbis\0uncompressed\0VT_BLOCKs\0\0" );
  switch( iTokenIndex )
   { case 0:
     case 1:   iAudioFileFormat = AUDIO_FILE_FORMAT_OGG;
               break;
     case 2:   iAudioFileFormat = AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS;
               break;
     case 3:   iAudioFileFormat = AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS;
               // Uncompressed stream from VLF-RX-tools *without* T_StreamHeaders but VT_BLOCKs.
               // Added here 2020-10-11, but not really supported yet .
               break;
     default:
               break;
   }
  // The second, optional, part of the string is only important for non-compressed streams:
  //     "int8", "int16", "int32", "float32", "float64" .
  if( QFile_SkipChar( &cp, ',' ) )
   {
   }
  if( piAudioFileFormat != NULL )
   { *piAudioFileFormat = iAudioFileFormat;
   }
  if( piVTFlags != NULL )
   { *piVTFlags = iVTFlags;
   }
  *ppcSource = cp;
  return fOk;
} // end VFIO_StringToAudioFileFormatAndVTFlags()


//---------------------------------------------------------------------------
T_VorbisFileHelper *VorbisFileHelper_Alloc( class C_VorbisFileIO *pThis )
  // Allocates a 'vorbis helper structure' .
  // Contains everything we don't want to expose in VorbisFileIO.h .
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper *)UTL_NamedMalloc( // 8-byte aligned..
           "VorbisIO", sizeof(T_VorbisFileHelper) );
  // Despite everyone talking about malloc() ensures 8-byte alignment,
  //   DON'T TAKE THAT FOR GRANTED. Borland C++Builder V6's malloc does NOT.
  //   Before replacing malloc() by UTL_NamedMalloc(), got here with
  //   pvfh = 0x13A3801C, which is only 4-byte ("DWORD") aligned.
  //
  if( pvfh != NULL )
   {    memset( pvfh, 0, sizeof(T_VorbisFileHelper) ); // clear old garbage, including pvfh->stream_header_1
        pvfh->pThis = pThis;  // address of the class instance which 'owns' this helper-struct
#      if (VORBIS_USE_QFILE) // use "QFile" or standard file I/O from the RTL ?
        pvfh->m_QFile.iHandle = -1;
#      else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
        pvfh->m_FileHandle    = -1;
#      endif
        pvfh->fIsInternetStream = FALSE;
        pvfh->fStoreSamplesForMultipleReaders = FALSE;
        pvfh->iStreamFormat = STREAM_FORMAT_UNKNOWN; // not sure if the stream(!) contains OggVorbis or non-compressed audio
        pvfh->fStreamFormatDetectedFromData = FALSE; // not sure about iStreamFormat yet
        pvfh->dwBufferUsage = pvfh->dwBufferIndex = 0;
        ChunkInfo_Init( &pvfh->chunk_info );
        pvfh->dblStartTime  = 0.0;
        pvfh->iVTFlags = VTFLAG_INT2;          // if uncompressed stream w/o headers, assume 16-bit signed integer samples
        pvfh->nChannelsPerSampleInStream = 1;  // initial guess, also important for uncompressed streams !

        // > It is safe to call ogg_stream_clear on the same structure more than once.
        // Ok, so let's make liberal use of it, despite the 'memset' further above:
        ogg_stream_clear( &pvfh->osv );  // Ogg stream for Vorbis (audio)
        ogg_stream_clear( &pvfh->ost );  // Ogg stream for timestamps


        INET_InitClient( &pvfh->InetClient,
              VorbisFileIO_InetCallback, // [in, optional] progress indicator callback
              (void *)pvfh );            // [in, optional] user callback data
        return pvfh;
   }
  return NULL; // no luck, the 'pool' of T_VorbisFileHelper instances is exhausted
} // end VorbisFileHelper_Alloc()


//---------------------------------------------------------------------------
void VorbisFileHelper_Free( T_VorbisFileHelper *pvfh )
{
  if( pvfh != NULL )
   {
#      if (VORBIS_USE_QFILE)
        if( pvfh->m_QFile.iHandle > 0 )
         { QFile_Close( &pvfh->m_QFile ); // should never be necessary to close it here, but ....
           pvfh->m_QFile.iHandle = -1;
         }
        if( pvfh->qfStreamLog.fOpened )
         { QFile_Close( &pvfh->qfStreamLog );
         }
#      else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
        if( pvfh->m_FileHandle > 0 )
         { _rtl_close(pvfh->m_FileHandle); // ... avoid a file handle leak !
           // On this occasion: What's wrong with the STANDARD file I/O functions, Borland ?!
           // Why must we use this _rtl_close-stuff instead of just 'close' or 'fclose' ?
#          error "We don't support this _rtl_-nonsense yet !"
           pvfh->m_FileHandle = -1;
         }
#      endif
        UTL_free( pvfh );  // not 'free()' since we used UTL_NamedMalloc(), not malloc() !
   }
} // end VorbisFileHelper_Free()

//---------------------------------------------------------------------------
static int CallBack(T_VorbisFileHelper *pvfh, // "calls the callback" ...
               int iEventType, // one of the VORBIS_IO_EVENT_TYPE_.. - values
               char *pszInfo, // string parameter: Info, Error message, or Server response
               int iParam1 )  // integer parameter: progress percentage or error code
{
  int iResult = 0;
  if( pvfh->pInfoCallback != NULL )
   { iResult = pvfh->pInfoCallback(
                    pvfh->pvUserCallbackData, // [in] address of any user-speficed data
                    iEventType, // one of the following (or maybe a bit combination?)
                           //  AUDIO_FILE_IO_EVENT_TYPE_INFO  (0)
                           //  AUDIO_FILE_IO_EVENT_TYPE_ERROR (1)
                           //  AUDIO_FILE_IO_EVENT_TYPE_PARSE_RESPONSE (2)
                           //  AUDIO_FILE_IO_EVENT_FLAG_SHOW_TIMESTAMP (8)
                    pszInfo,   // info string or HTTP response
                    iParam1);  // progress percentage or numeric error code
     // In Spectrum Lab, the above call takes us from
     //   StartAudioStream() -> C_VorbisFileIO::OutOpen() -> CallBack_Info() -> CallBack()
     //     to AudioStreamOut_GUI.cpp::MyVorbisIOCallback(), where the strings
     //     passed this way will be appended to the connection log .
   }
  return iResult;
} // end CallBack()

//---------------------------------------------------------------------------
static void CallBack_Info( T_VorbisFileHelper *pvfh,
                           char *pszFormat, ... )
  // Text messages 'emitted' this way will appear on SL's
  //      "Analyse / play audio stream" control panel,
  //      through a long-and-winding (but thread-safe) road
  //      explained in CallBack_Error() further below .
{
  char sz255[256], *cp;
  va_list arglist;
  va_start(arglist, pszFormat);
  UTL_FormatDateAndTime("hh:mm:ss.s ", TIM_GetCurrentUTC(), sz255 );
  cp = sz255+strlen(sz255);
  vsnprintf(cp, 255-(cp-sz255), pszFormat, arglist);
  va_end(arglist);
  CallBack( pvfh, AUDIO_FILE_IO_EVENT_TYPE_INFO, sz255, 0 );
} // end CallBack_Info()

//---------------------------------------------------------------------------
static void CallBack_Error( T_VorbisFileHelper *pvfh,
                            double dblUnixTime, // use 0 for "current" time
                            char *pszFormat, ... )
  // Error messages 'emitted' this way will appear on SL's
  //      "Analyse / play audio stream" control panel,
  //      through a long-and-winding (but thread-safe) road
  //      explained further below .

{
  char sz255[256], *cp;
  va_list arglist;
  va_start(arglist, pszFormat);
  if( dblUnixTime <= 0.0 )
   {  dblUnixTime = TIM_GetCurrentUTC();
   }
  UTL_FormatDateAndTime("hh:mm:ss.s ", dblUnixTime, sz255 );
  cp = sz255+strlen(sz255);
  vsnprintf(cp, 255-(cp-sz255), pszFormat, arglist);
  va_end(arglist);
  CallBack( pvfh, AUDIO_FILE_IO_EVENT_TYPE_ERROR, sz255, 0 );
  // In Spectrum Lab, the above call is actually made from -> to :
  //  TSoundThread::Execute() -> C_AnyAudioFileIO::ReadSampleBlocks()
  //   -> C_VorbisFileIO::ReadSampleBlocks() -> ReadAndDecodeNonCompressed()
  //       -> ProcessHeaderFromNonCompressedStream() -> tq_add_vtblock()
  //           -> tq_add( < detected a timestamp problem / failed plausibility test > )
  //               -> CallBack_Error( "tq_add: timestamps not plausible ... " )
  //                   -> SL_WebstreamInfoCallback( event type = AUDIO_FILE_IO_EVENT_TYPE_ERROR )
  //                       -> DEBUG_EnterErrorHistory() [which adds ITS OWN timestamp !]
  //                           -> OutputDebugString(),    [optional, prints to the debugger's message list WHEN ATTACHED TO A DEBUGGER]
  //                              UTL_WriteRunLogEntry(), [optional, see SpecMain.cpp]
  //                              DEBUG_ErrorHistoryBuffer[] [feeds the visual 'Error History' in the main task]
} // end CallBack_Error()


//---------------------------------------------------------------------------
// Vorbis callback (and file I/O) functions.
//  As their names suggest, they are expected to work in exactly
//  the same way as normal c io functions (fread, fclose etc.).
//  Its up to us to return the information that the libs need to parse
//  the file from memory, stream, 'real files', ROM, or whatever .
// > The function prototypes for the callbacks are basically the same as for
// > the stdio functions fread, fseek, fclose, ftell.
// > The one difference is that the FILE* arguments have been replaced with
// > a void* - this is to be used as a pointer to whatever internal data these
// > functions might need.
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Function: VorbisRead
// Purpose: Callback for the Vorbis read function. Also suited for NON-COMPRESSED streams and files.
//            For STREAMS, waits until the requested number of bytes was delivered
//            or a timeout occurred.
//            The return value indicates how many samples have actually been returned.
//---------------------------------------------------------------------------
static size_t VorbisRead(  // .. a possibly BLOCKING CALL (for webstreams) !
        void *ptr,         // ptr to the data that the vorbis files need
        size_t byteSize,   // how big a byte is  ;)   [usually one]
        size_t sizeToRead, // How much we can read
        void *datasource,  // a pointer to the data we passed into ov_open_callbacks
        int  *piNumMillisecondsWaited ) // [out,optional] : number of milliseconds spent WAITING for input; "accumulated"
                                        // (important to find out the 'pace setter' : Network stream or audio-OUTPUT ?)
{
  // ex:
  // size_t nBytesRead = fread( ptr, byteSize, sizeToRead, (FILE*)datasource );
  // if( nBytesRead>0 )
  //  { g_file_current_position += nBytesRead;
  //  }
  // return nBytesRead;  // returns the number of bytes(?) actually read .
  size_t nBytesRead = 0;
  int n;
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)datasource;
  if( pvfh )
   {
#   if (VORBIS_USE_QFILE)
     if( pvfh->m_QFile.iHandle > 0 )
      {
        nBytesRead = QFile_Read( &pvfh->m_QFile,  ptr, sizeToRead*byteSize );
      }
#   else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
     if( pvfh->m_FileHandle > 0 )
      {
        nBytesRead = _rtl_read(pvfh->m_FileHandle, ptr, sizeToRead*byteSize );
      }
#   endif
     else if( pvfh->fIsInternetStream )
      {
        n = INET_ReadDataFromServer( // "continues" reading if necessary
              &pvfh->InetClient,     // [in,out] T_INET_Client *pInetClient; a web client from YHF_Inet.c
              pvfh->inetTimeout_ms,  // [in] max. timeout in milliseconds
              pvfh->inetReadOptions, // [in] options : INET_RD_OPTION_WAIT_FOR_ALL_DATA
              (BYTE*)ptr, sizeToRead*byteSize, // [in,out] destination buffer for the "contents"
              piNumMillisecondsWaited ); // [out,optional] number of milliseconds spent WAITING here (for network data); "accumulated"
        if( n < 0 )
         { // Something went wrong (connection lost, etc etc etc).
           //  In this case, -n is one of the error codes defined in ErrorCodes.h !
           nBytesRead = 0;
           CallBack_Error( pvfh, 0, "Stream read error: %s", ErrorCodeToString(-n) );
         }
        else
         { nBytesRead = (size_t)n;
           if( nBytesRead > 0 )
            {
              pvfh->nBitrateCounter += 8 * nBytesRead;

              if( pvfh->fWriteLogfile && pvfh->qfStreamLog.fOpened )
               { if( ! QFile_Write( &pvfh->qfStreamLog, (BYTE*)ptr, nBytesRead ) )
                  { QFile_Close( &pvfh->qfStreamLog );
                    pvfh->fWriteLogfile = FALSE; // something fishy with the log, don't try to write to it again
                  }
               }
            }
         }
      } // end if < file or stream > ?

     if( nBytesRead > 0 )  // keep current file position and -size up to date:
      { pvfh->pThis->m_dwCurrFilePos += (DWORD)nBytesRead;
        if( pvfh->pThis->m_dwCurrFilePos > pvfh->pThis->m_dwFileSizeInBytes )
         {  pvfh->pThis->m_dwFileSizeInBytes = pvfh->pThis->m_dwCurrFilePos;
         }
      }

   } // end if ( pvfh != NULL )
  return nBytesRead;
} // end VorbisRead()

//----------------------------------------------------------------------------
static int VorbisSeek(
   void *datasource,   // [in] pointer to the data we passed into ov_open_callbacks
   ogg_int64_t offset, // [in] offset from the point we wish to seek to
   int  whence )       // [in] where we want to seek to (whence = "from where")
  // Callback for the Vorbis seek function
  //  Sets the file pointer associated with stream to a new position
  //  that is offset bytes from the file location given by whence.
  //  fseek (!) returns 0 if the pointer is successfully moved
  //  and nonzero on failure.
  //   ( Hmmm. Not many streams, especially not AUDIO STREAMS, are SEEKABLE !
  //     In Vorbis parlance, "stream" often means "file" .
  //     In SpecLab parlance, "stream" is an ENDLESS stream (never seekable)
  //               and "file" is a file on a local storage medium (seekable).
{
  // ex: return -1; // Let vorbisfile.c know WE CANNOT SEEK :
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)datasource;
  int vorbis_error_code = -1;
  int fromwhere;
  long i32NewAbsFilePosition;
  if( pvfh )
   {
#   if (VORBIS_USE_QFILE)
     if( pvfh->m_QFile.iHandle > 0 )
      {
        // Note: MOST LIKELY (but not guaranteed), SEEK_SET == QFILE_SEEK_SET, etc..
        switch( whence )  // standard file stuff to QFile's seek-command:
         { case SEEK_SET :
              fromwhere = QFILE_SEEK_SET; /* Positionierung vom Dateianfang aus */
              break;
           case SEEK_CUR :
              fromwhere = QFILE_SEEK_CUR; /* Positionierung von der aktuellen Position aus */
              break;
           case SEEK_END :
              fromwhere = QFILE_SEEK_END; /* Positionierung vom Dateiende aus */
           default:
              return -1; // too exotic :)
         } // end switch( whence )
        i32NewAbsFilePosition = QFile_Seek( &pvfh->m_QFile, (long)offset, fromwhere);
        if( i32NewAbsFilePosition >= 0 )
         { vorbis_error_code = 0;
           pvfh->pThis->m_dwCurrFilePos = i32NewAbsFilePosition;
           if( pvfh->pThis->m_dwCurrFilePos > pvfh->pThis->m_dwFileSizeInBytes )
            {  pvfh->pThis->m_dwFileSizeInBytes = pvfh->pThis->m_dwCurrFilePos;
            }
         }
      }
#   else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
     if( pvfh->m_FileHandle > 0 )
      { _rtl_close(pvfh->m_FileHandle);
        pvfh->m_FileHandle = -1;
      }
#   endif
   } // end if ( pvfh != NULL )
  return vorbis_error_code; // hopefully 0 = "no error"

} // end VorbisSeek()


//---------------------------------------------------------------------------
static int VorbisClose(
   void *datasource) // pointer to the data we passed into ov_open_callbacks
  // Callback for the Vorbis close function
  //    Like fclose(), returns 0 on success (!) .
{
  // ex: return fclose( (FILE*)datasource );
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)datasource;
  if( pvfh )
   {
     if( pvfh->fIsInternetStream )
      { pvfh->fIsInternetStream = FALSE;
        INET_CloseClient( &pvfh->InetClient, NULL/*pszWhy*/ );
      }

#   if (VORBIS_USE_QFILE)
     if( pvfh->m_QFile.iHandle > 0 )
      { QFile_Close( &pvfh->m_QFile );
        pvfh->m_QFile.iHandle = -1;
      }
#   else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
     if( pvfh->m_FileHandle > 0 )
      { _rtl_close(pvfh->m_FileHandle);
        pvfh->m_FileHandle = -1;
      }
#   endif

   } // end if ( pvfh != NULL )
  return 0;  // no error
} // end VorbisClose()

//---------------------------------------------------------------------------------
static long VorbisTell(
   void *datasource) // pointer to the data we passed into ov_open_callbacks
  //  Callback for the Vorbis tell function
  //  Returns the current file pointer for stream.
  //  The offset is measured in bytes from the beginning of the file
  //   (btw the file IS ALWAYS binary). The value returned by ftell
  //   can be used in a subsequent call to fseek.
  // NOTE (WB): The vorbisfile API seems to use seek & tell
  //               to determine the TOTAL LENGTH of the file,
  //               making it useless to process REAL AUDIO STREAMS
  //               (with a length unknown in advance) .
  //            But it only does this when VorbisSeek(), called 1st,
  //            indicates by a non-negative result that the 'stream' (file)
  //            is seekable. Found this in the caller of VorbisTell() :
  //     /* we can seek, so set out learning all about this file */
  // and /* If seek_func is implemented, tell_func must also be implemented */
  //     In other words: we don't really need VorbisSeek & VorbisTell
  //     seeking just to play files (without 'hopping around').
  //     Only Spectrum Lab needs seeking, if an audio file is to be played
  //     in an 'endless loop' .  For (endless) streams, this makes no sense anyway.
{
  long i32Result = -1;
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)datasource;
  if( pvfh )
   {
#   if (VORBIS_USE_QFILE)
     if( pvfh->m_QFile.iHandle > 0 )
      { i32Result = QFile_Seek( &pvfh->m_QFile, 0/*offset*/, QFILE_SEEK_CUR/*fromwhere*/ );
        // QFile_Seek( pqFile, 0, QFILE_SEEK_CUR )  returns the current file position .
      }
#   endif // (VORBIS_USE_QFILE)
   } // end if ( pvfh != NULL )
  return i32Result;
} // end VorbisTell()

/****************************************************************************
	End of Vorbis callback functions
****************************************************************************/

//---------------------------------------------------------------------------
extern "C" int VorbisFileIO_InetCallback(
          void *pvUserCallbackData,
          struct tInetClient *pInetClient,
          int iEventType,     // INET_EVENT_TYPE_.. or AUDIO_FILE_IO_EVENT_TYPE_.. (must be compatible)
          char *pszInfoOrResponse,
          int  iProgressPercent)
  // Callback function for the internet client (DL4YHF's T_INET_Client) .
  // Once used as a progress indicator AND to analyse (parse) the HTTP GET RESPONSE
  //     (which may tell us a bit more about the type of the file) .
  // Later (2011-12) also used when, for example, the remote server
  //    asks for a password, or similar stuff which cannot be handled
  //    inside YHF_Inet.c itself (because it's too application-specific) .
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)pvUserCallbackData;
  char *cp = pszInfoOrResponse;

  switch( iEventType & (~INET_EVENT_FLAG_SHOW_TIMESTAMP) )
   { case INET_EVENT_TYPE_ERROR :
        // DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, UTL_USE_CURRENT_TIME, pszInfo );
        CallBack_Error( pvfh, 0, pszInfoOrResponse );
        break;
     case INET_EVENT_TYPE_INFO :
        CallBack( pvfh, iEventType, pszInfoOrResponse, iProgressPercent );
        break;
     case INET_EVENT_TYPE_PARSE_RESPONSE :
        // Opportunity for the application to parse the received HTTP GET RESPONSE
        // before 'downloading' (reading) the rest of the file from the server.
        // In this case, pszInfo is the zero-terminated HTTP GET RESPONSE,
        // without(!) the 'file data'.
        // Example (in pszInfo, in this case the 'response headers' without 'file data',
        //     received after connecting to http://67.207.143.181:80/vlf1 ) :
        // "HTTP/1.0 200 OK\r\nContent-Type: audio/x-mpegurl\r\n\r\n\0"
        CallBack( pvfh, iEventType/*ex:AUDIO_FILE_IO_EVENT_TYPE_PARSE_RESPONSE*/, pszInfoOrResponse, iProgressPercent );
        break;
     default:  // INET_EVENT_TYPE_INFO, etc ...
        break;
   }

  return NO_ERRORS;
} // end VorbisFileIO_InetCallback()


/***************************************************************************/
static void tq_add(
              T_VorbisFileHelper *pvfh, // [in,out] ogg/vorbis-related states
              struct VT_OGG_BLK *blk )  // [in] timestamp, sample counter ("posn"), calibration, GPS;
                                        // compatible with Paul's VLF tools.
  // Adds a timestamp to an internal queue.
  //  Source is usually from an OGG STREAM, but it may also be a NON-COMPRESSED stream.
  // Caller in SpecLab when using NON-COMPRESSED input:
  //  TSoundThread::Execute() -> C_AnyAudioFileIO::ReadSampleBlocks()
  //   -> C_VorbisFileIO::ReadSampleBlocks() -> ReadAndDecodeNonCompressed()
  //    -> ProcessHeaderFromNonCompressedStream() -> tq_add_vtblock() -> tq_add() .
{
  int i;
  long double ldblUnixTime;
  double d1, d2;
  char sz80[84];

   if( blk->magic != BLK_MAGIC)
    { // VT_bailout( "input stream bad blk");
      return;
    }

  // if( pvfh->iTimestampQueueLength >= MAXTQ)
  //  { // VT_bailout( "timing queue full");
  //    return;
  //  }
  // Removed by WB 2011-12-18.  We always keep the last <MAXTQ> entries
  // in memory, regardless of tq_get() being called or not.
  //   (Reason: the calling interval of tq_get depends on the blocksize,
  //    which is only known when ReadSampleBlocks() is called .
  //    We need to be able to deliver these timestamps at ANY rate)
  // Thus the principle of buffering timestamps has been completely modified:
  //  - tq_add always keeps a history of the most recent < MAXTQ > timestamps;
  //  - tq_get doesn't REMOVE anything from the queue; instead it searches
  //    for the two closest VT_OGG_BLOCK entries in the history,
  //    and interpolates or extrapolates if necessary .
   if( pvfh->iTimestampQueueLength >= MAXTQ )  // timestamp-queue already full ?
    { // -> remove the oldest entry (which is at the begin of the list)
      for(i=0; i<MAXTQ-1; ++i )
       { pvfh->tq[i] = pvfh->tq[i+1]; // fancy circular buffer doesn't pay off here
       }
      pvfh->iTimestampQueueLength = MAXTQ-1;
    }

   pvfh->tq[ pvfh->iTimestampQueueLength++ ] = *blk;

   ldblUnixTime = (long double)blk->secs + (long double)blk->nsec * 1e-9; // [Unix seconds]
   if( pvfh->i64TimestampCounter == 0 )
    { pvfh->dblStartTime = ldblUnixTime; // use this timestamp instead of the FILE DATE
    }
   if(blk->posn > pvfh->i64PrevTstampSamplePosn )
    { d1 = ldblUnixTime - pvfh->ldblPrevTstampUnixTime; // -> diff [seconds] from timestamps
      d2 = (double)(blk->posn - pvfh->i64PrevTstampSamplePosn) // -> diff [seconds] from sample counter. BUT WHAT IS THE UNIT OF "posn" ? IT DOESN'T SEEM TO BE "SAMPLE POINTS" aka "FRAMES" ?!
         / (blk->srcal * pvfh->dblNominalSampleRate);
      // d1 and d2, both measured in seconds, shouldn't differ by more
      // than a half sample interval (~ 15 us). Looking at the soundcard's drift rate,
      // this is realistic and not 'very tough'. Despite that, this test often failed. Hold your breath:
      if( d1>(d2+15e-6) || d1<(d2-15e-6) )
       { sprintf( sz80, "tq_add: timestamps not plausible: d1=%lf, d2=%lf [sec]",
                                (double)d1, (double)d2 );
         VFIO_DebuggerBreak( pvfh, sz80, __LINE__ );
         CallBack_Error( pvfh, 0,  sz80 );
       } // end if < too large difference between 'd1' and 'd2 [in seconds] >
    }
   else // blk->posn <= pvfh->i64PrevTstampSamplePosn ...
    { if( blk->posn < pvfh->i64PrevTstampSamplePosn )
       { VFIO_DebuggerBreak( pvfh, "Unexpected 'posn' in timestamp", __LINE__ );
       }
    }
   pvfh->i64PrevTstampSamplePosn = blk->posn;
   pvfh->ldblPrevTstampUnixTime  = ldblUnixTime;
   pvfh->dblTimestampLatency = TIM_GetCurrentUTC() - ldblUnixTime; // this difference should be "slightly POSITIVE".

   ++pvfh->i64TimestampCounter; // here: count RECEIVED timestamps, in tq_add()


} // end tq_add()

/***************************************************************************/
static void tq_add_vtblock(
              T_VorbisFileHelper *pvfh, // [in,out] ogg/vorbis-related states, including pvfh->i64NumSamplesDecoded (for the timestamp-plausibility-test)
              T_VT_Block *pVTblk )      // [in] address of a VT_BLOCK struct;
                                        // compatible with Paul's VLF tools (V0.7) .
  // Adds a timestamp to an internal queue. Source is from an UNCOMPRESSED STREAM .
{
  // For simplicity, the entries from the VT_BLOCK are converted into
  //  a VT_OGG_BLK, and then added to the queue with the already existing functions.
  //  Implemented 2014-04-23 to support UNCOMPRESSED STREAMS
  //  without having to modify the application (SL is already too complex) .
  struct VT_OGG_BLK oblk;
  memset(&oblk, 0, sizeof(oblk) );
  oblk.magic  = BLK_MAGIC; // magic number indicating a VT_OGG_BLK (not a VT_BLOCK!)
  oblk.tbreak = FALSE;     // no Timestamp discontinuity detected
  oblk.posn   = pvfh->i64NumSamplesDecoded;  // [in] for non-compressed input, samples COUNTED in
     // VT_OGG_BLK.posn = Frame count (sample index) to which the rest of this data applies .
     // Important to INTERPOLATE timestamps (and related values) later !
     // Since there is no equivalent for 'posn' in VT_BLOCK,
     // use the LOCALLY COUNTED number of samples instead.
     // When tested with the uncompressed test stream provided by Paul
     //  ( see email 2014-04-21, rawtcp://67.207.139.49:9876 ),
     // got here with pvfh->i64NumSamplesDecoded = 0, 0(!), 2048, 4096, ... ;
     //  pVTblk->bsize  = 2048 = Number of frames per block,
     //  pVTblk->frames = 2048 = Number of frames actually contained. (ok)
  oblk.spare1 = 0;
  oblk.spare2 = 0;
  oblk.secs   = pVTblk->secs;     // Timestamp, seconds
  oblk.nsec   = pVTblk->nsec;     // and nanoseconds
  oblk.srcal  = pVTblk->srcal;    // Sample rate calibration factor (ideally 1.0)
  oblk.glat   = 0.0;        // Geographic coordinates don't exist in a VT_BLOCK
  oblk.glong  = 0.0;
  oblk.gasl   = 0.0;
  tq_add( pvfh, &oblk );
} // end tq_add_vtblock()


/***************************************************************************/
static BOOL tq_get(   // get a timestamp from a queue
              T_VorbisFileHelper *pvfh, // [in,out] ogg/vorbis-related states
              uint64_t u64SampleIndex,  // [in] decoder sample index for which to retrieve data
              long double *timestamp,   // [out] timestamp (in Unix format, fractional)
                  // (Caution, unlike Borland and GCC, some stupid Micro$oft compiler
                  //           treats 'long double' just the same as 'double' !
                  //   Borland's 'long double' is accurate for 19 to 20 decimals.
                  //   Micro$oft's 'long double' is only good for 15 to 16 decimals.
                  //   This is NOT ENOUGH for a nanosecond resolution
                  //   in a 'Unix timestamp' = number of seconds elapsed
                  //   since 1970-01-01 00:00:00  ) .
                  //   To compile this module with Micro$oft compilers, their
                  //   'extended' type would possibly do the job. Good luck.
              double *srcal,  // [out] sample rate calibration factor, near 1.0
              int *tbreak ,   // [out] flag 'Timestamp discontinuity detected'
              struct VT_OGG_BLK **ppTS, // [out, optional] address of a VT_OGG_BLK
              long *pi32TimestampIndex) // [out, optional] index of the returned timestamp
  //  Called to produce timestamp and sample rate calibration for the specified
  //  output frame count (u64SampleIndex, passed as argument).
  //  We interpolate to the current position from the oldest queued
  //  VT_OGG_BLK timestamp record.
  //  Unlike in Paul's vtvorbis.c, the result is placed in a T_ChunkInfo struct.
  //
  // Note: In the timestamped Ogg/Vorbis streams,
  //       the VT_OGG_BLK (timestamp struct) is always sent *before*
  //       the corresponding audio .
  //       This is important for the calling sequence of tq_get()
  //       in ReadSampleBlocks() .
{
  int i;
  struct VT_OGG_BLK *tp1, *tp2;  // two neighbours for interpolation
  long double ndiff, ts1, ts2;
  long i32TimestampIndex = 0;

   if( pvfh->iTimestampQueueLength <= 0 )
    { // VT_bailout( "timing queue empty");
      return FALSE;
    }

   // Search the history of 'most recent' entries in the timestamp history
   // for the closest one (for the requested sample index) .
   // The OLDEST entry, at pvfh->tq[0] should have a sample index ("tq[0].posn")
   //     lower than the requested one; otherwise MAXTQ is too low.
   tp1 = &pvfh->tq[0];  // don't have two neighbours for interpolation ..
   tp2 = tp1;
   for(i=1/*!!*/; i < pvfh->iTimestampQueueLength; ++i)
    {
      if( pvfh->tq[i].posn >= u64SampleIndex )
       { // Arrived here ? The desired SAMPLE_INDEX (u64SampleIndex)
         // must be somewhere between pvfh->tq[i] and pvfh->tq[i-1] .
         tp1 = &pvfh->tq[i-1];  // left neighbour (with lower sample index & timestamp)
         tp2 = &pvfh->tq[i];    // right neighbour (with higher sample index & timestamp)
         i32TimestampIndex = pvfh->i64TimestampCounter - i;
         break;
       }
    } // end for

   if( tp2->posn > tp1->posn )   // valid 'neighbour timestamps' ?
    { ndiff = (long double)(u64SampleIndex - tp1->posn) / (long double)(tp2->posn - tp1->posn);
       // normalized difference for linear interpolation :
       // ndiff = 0 means 'it's the left neighbour we're looking for'  (tp1)
       // ndiff = 1 means 'it's the right neighbour we're looking for' (tp2)
       // ndiff between 0 and 1 are realistic,
       // everything outside this means EXTRAPOLATE (may happen occasionally) .
      if( ndiff<0.0 || ndiff>1.0 )
       { ndiff = ndiff;
       }
      ts1 = (long double)tp1->secs + (long double)tp1->nsec * 1e-9; // [seconds]
      ts2 = (long double)tp2->secs + (long double)tp2->nsec * 1e-9; // [seconds]
      *timestamp = ts1 * (1.0 - ndiff)  +   ts2 * ndiff;
      // Assume the soundcard's sampling rate drifts 'slowly and LINEAR' ..
      *srcal  = tp1->srcal * (1.0 - ndiff)  +   tp2->srcal * ndiff;
      *tbreak = tp1->tbreak || tp2->tbreak; // not sure about WHERE EXACTLY the break was ?
    } // end if( tp2->posn > tp1->posn )
   else  // don't INTERPOLATE between the two timestamps (and SR-calibrations):
    {
      // Arrived here: Nothing found, or there is only ONE table entry.
      // Don't interpolate between TWO entries but extrapolate from the one we have.
      *timestamp = tp1->secs + tp1->nsec*1e-9 +
                   (u64SampleIndex - tp1->posn) / (tp1->srcal * pvfh->dblNominalSampleRate);
      *srcal  = tp1->srcal;
      *tbreak = tp1->tbreak;
    }
   if( ppTS != NULL )
    { *ppTS = tp1;    // necessary for the caller to retrieve GPS data
    }
   if( pi32TimestampIndex != NULL )
    { *pi32TimestampIndex = i32TimestampIndex;
    }

   return TRUE;
} // end tq_get()

/***************************************************************************/
static int handle_vorbis_packet(
              T_VorbisFileHelper *pvfh, // [in,out] ogg/vorbis-related states
              ogg_packet *op)           // [in] Ogg packet with Vorbis content
  //  Attempt to decode a vorbis packet, returns zero if failed.
  //  In SpectrumLab, called through
  //   TSoundThread::Execute() -> C_AnyAudioFileIO::ReadSampleBlocks()
  //    -> C_VorbisFileIO::ReadSampleBlocks() -> ReadAndDecodePages()
  //          -> DecodeOggPage() -> handle_vorbis_packet()
  //               -> vorbis_synthesis(), vorbis_synthesis_blockin(),
  //                  implemented in a DLL so cannot step into them from here.
  //
{
  int nSamplePoints, nSamplesConsumed, nChannelsPerSample, vorbis_error_code;
  float **ppfltPCM;
  float *pfltDest;
  BOOL  recognized = FALSE;
  // long double timestamp;
  // double srcal;
  // int tbreak;


  // > Decoding workflow (using Libvorbis, *WITHOUT* vorbisfile.c ) :
  // > 1. When reading the header packets of an Ogg stream, you can use
  // >    vorbis_synthesis_idheader to check whether a stream might be Vorbis.
  // > 2. Initialize a vorbis_info and a vorbis_comment structure using
  // >    the appropriate vorbis_*_init functions, then pass the first
  // >    three packets from the stream (the Vorbis stream header packets)
  // >    to vorbis_synthesis_headerin in order. At this point,
  // >    you can see the comments and basic parameters of the Vorbis stream.
  // > 3. Initialize a vorbis_dsp_state for decoding based on
  // >    the parameters in the vorbis_info by using vorbis_synthesis_init.
  // > 4. Initialize a vorbis_block structure using vorbis_block_init.
  // > 5. While there are more packets to decode:
  // >    5.1. Decode the next packet into a block using vorbis_synthesis.
  // >    5.2. Submit the block to the reassembly layer using vorbis_synthesis_blockin.
  // >    5.3. Obtain some decoded audio using vorbis_synthesis_pcmout
  // >         and vorbis_synthesis_read. Any audio data returned
  // >         but not marked as consumed using vorbis_synthesis_read
  // >         carries over to the next call to vorbis_synthesis_pcmout.
  // > 6. Destroy the structures using the appropriate vorbis_*_clear functions.
  //



  if( pvfh->nRcvdVorbisHeaders <= 0)    // First time through?
   {
      vorbis_info_init( &pvfh->vinfo);  //  (2)
      vorbis_comment_init( &pvfh->vcomment );
      pvfh->nRcvdVorbisHeaders = 0;
   }

  if( pvfh->nRcvdVorbisHeaders < 3)     // Collect 3 vorbis headers
   {
     // > vorbis_synthesis_headerin() decodes a header packet from a Vorbis stream
     // > and applies the contents to the given
     // > vorbis_info structure (to provide codec parameters to the decoder) and
     // > vorbis_comment structure (to provide access to the embedded Vorbis comments).
     // > Once the three Vorbis header packets (info, comments, and codebooks,
     // > in that order) have been passed to this function,
     // > the vorbis_info structure is ready to be used in a call to vorbis_synthesis_init.
     vorbis_error_code = vorbis_synthesis_headerin(
            &pvfh->vinfo/*out*/, &pvfh->vcomment/*out*/, op/*in: ogg_packet to decode*/);
     if( vorbis_error_code < 0)
      {
        if( pvfh->nRcvdVorbisHeaders==0 )
         {
           // 2012-02-17: Got here when streaming from rawtcp://67.207.139.49:4416  .
           //             OV_ENOTVORBIS = "not a Vorbis header packet";
           //             it actually was a  VT_OGG_HDR as detected below:
           if( (size_t)op->bytes == sizeof( struct VT_OGG_HDR) )
            { struct VT_OGG_HDR *pVT_ogg_hdr = (VT_OGG_HDR*)op->packet;
              if( pVT_ogg_hdr->magic == HDR_MAGIC/*0x1996c3a3*/ )
               { CallBack_Error( pvfh, 0/*time*/, "handle_vorbis_packet: Early VT_OGG_HDR" );
                 recognized = TRUE; // "dropped" packet but recognized
               }
            }
           if( !recognized )
            { CallBack_Error( pvfh, 0/*time*/, "handle_vorbis_packet: header #%d bad; \"%s\" .",
                           (int)(pvfh->nRcvdVorbisHeaders+1),
                          (char*)VorbisErrorCodeToString(vorbis_error_code) );
            }
           return 0;
         }
      }
      // VT_report( 2, "got vorbis header %d", nRcvdVorbisHeaders + 1);

      if( ++pvfh->nRcvdVorbisHeaders == 3)  // All mandatory headers received ?
      {
         // Initialise vorbis decoder
         vorbis_synthesis_init( &pvfh->vdsp, &pvfh->vinfo);        // (3)
          // |__ initializes a vorbis_dsp_state structure for decoding
          //     and allocates internal storage for it.
         pvfh->fMustClearDSP = TRUE;  // reminder for cleanup
         vorbis_block_init( &pvfh->vdsp, &pvfh->vblock);           // (4)
          // |__ initializes a vorbis_block structure and allocates
          //     its internal storage. A vorbis_block is used to represent
          //     a particular block of input audio
          //     which can be analyzed and coded as a unit.
         pvfh->fMustClearBlock = TRUE;  // reminder for cleanup

         // Save all the application needs to know in our own struct:
         pvfh->dblNominalSampleRate  = pvfh->vinfo.rate;
         pvfh->nChannelsPerSampleInStream = pvfh->vinfo.channels;
         pvfh->fGotAllHeaders = TRUE;  // ready for business from VORBIS' point of view
      }
   } // end if( pvfh->nRcvdVorbisHeaders < 3)
  else   // Deal with vorbis audio data
   {
     if( (pvfh->fPlainOggVorbis )  && !pvfh->fGotAllHeaders )
      {
        //   VT_report( 0, "no timestamp stream, maybe -p option is required");
        //   VT_bailout( "missing timestamp stream");
      }

     if( vorbis_synthesis( &pvfh->vblock, op) == 0)            //   (5.1)
      { // |_ decodes a Vorbis packet into a block of data.
        //    The vorbis_block should then be submitted to the vorbis_dsp_state
        //    for the decoder instance using vorbis_synthesis_blockin
        //    to be assembled into the final decoded audio
        vorbis_synthesis_blockin( &pvfh->vdsp, &pvfh->vblock); //   (5.2)
        // |_ submits a vorbis_block for assembly into the final decoded audio.
        //    After this, decoded audio can be obtained with vorbis_synthesis_pcmout
      }

     // About decoding Vorbis audio into PCM :
     // > vorbis_synthesis_pcmout retrieves buffers containing decoded audio samples.
     // > The application is not required to make use of all of the samples
     // > made available to it by one call to this function
     // > before it continues to decode. Use vorbis_synthesis_read to inform
     // > the decoder of how many samples were actually used.
     // > Any unused samples will be included in the buffers output
     // > by the next call to this function.
     nSamplesConsumed = 0;
     while( (nSamplePoints=vorbis_synthesis_pcmout( &pvfh->vdsp, &ppfltPCM)) > 0)
      {
        int i, j;

        // Copy as many samples into pvfh->fltDecoderOutput[] as fit in there.
        // Note that the *CHANNELS* in pvfh->fltDecoderOutput[] are interleaved;
        // i.e. pvfh->fltDecoderOutput[0] = left channel, first sample,
        //      pvfh->fltDecoderOutput[1] = right channel, first sample,
        //      pvfh->fltDecoderOutput[0] = left channel, second sample,
        //      pvfh->fltDecoderOutput[1] = right channel, second sample;  etc.
        nChannelsPerSample     = pvfh->nChannelsPerSampleRequested; // <- may be ZERO..
        if( nChannelsPerSample > pvfh->nChannelsPerSampleInStream )
         {  nChannelsPerSample = pvfh->nChannelsPerSampleInStream;
         }
        if( nChannelsPerSample < 1 )
         {  nChannelsPerSample = 1;  // avoid div-by-zero further below
         }
        // May have to limit the number of sample points which can be 'consumed' here:
        nSamplesConsumed = ( MAX_DECODER_OUTPUT_SIZE - (int)pvfh->nDecoderOutputBufferUsage) / nChannelsPerSample;
        // 2021-05-21 : Exception from 'divide by zero' here . nChannelsPerSample = 0 ?
        if( nSamplesConsumed > nSamplePoints )
         {  nSamplesConsumed = nSamplePoints;
         }
        else if( nSamplePoints > nSamplesConsumed )
         { VFIO_DebuggerBreak( pvfh, "vorbis_synthesis_pcmout() delivered more samples than we can handle", __LINE__ );
           // Anyway, we'll inform Vorbis that we consumed less samples than expected !
         }
        pfltDest = pvfh->fltDecoderOutput + pvfh->nDecoderOutputBufferUsage;
        pvfh->nDecoderOutputBufferUsage += nSamplesConsumed * nChannelsPerSample;
        if( pvfh->nDecoderOutputBufferUsage > pvfh->nDecoderOutputBufferPeakUsage )
         {  pvfh->nDecoderOutputBufferPeakUsage = pvfh->nDecoderOutputBufferUsage;
         }
        for( j=0; j<nSamplesConsumed; j++)
         {
    //     if( pvfh->nDecoderOutputBufferUsage == 0) // FIRST sample for the output-block ?
    //      {
    //        if( !pvfh->fPlainOggVorbis )
    //         {  tq_get( pvfh, &timestamp, &srcal, &tbreak);
    //         }
    //        else
    //         {  // Plain vorbis stream, make dummy timestamp
    //            srcal = 1.0;
    //            tbreak = 0;
    //            timestamp = pvfh->dblStartTime + pvfh->i64NumSamplesDecoded / pvfh->dblSampleRate;
    //         }
    //        // VT_set_timebase( vtoutfile, timestamp, srcal);
    //      }

           for( i = 0; i < nChannelsPerSample; i++)
            { // frame[i] = ppfltPCM[i][j];
              *pfltDest++ = ppfltPCM[i][j];
            }
            // VT_insert_frame( vtoutfile, frame);

           // ex:   pvfh->i64NumSamplesDecoded++;  // here in handle_vorbis_packet()
         }
        if( pvfh->i64NumSamplesDecoded==0  )
         { CallBack_Info( pvfh, "Vorbis decoded first %ld PCM samples, %ld timestamp(s).",
                          (long)nSamplesConsumed, (long)pvfh->i64TimestampCounter );
         }

        pvfh->i64NumSamplesDecoded += nSamplesConsumed;  // important for processing received timestamps


        // inform the decoder of how many samples were actually used :
        vorbis_synthesis_read( &pvfh->vdsp, nSamplesConsumed );
        if( nSamplesConsumed < nSamplePoints )  // cannot accept more samples
         {  break;
         }
      } // end while vorbis_synthesis_pcmout(..)
   } // end else < Vorbis samples >

  return 1;
} // end handle_vorbis_packet()

/***************************************************************************/
static int handle_tstamp_packet(
              T_VorbisFileHelper *pvfh, // [in,out] ogg/vorbis-related states
              ogg_packet *op)           // [in] Ogg packet with non-Vorbis content
  // Attempt to decode a timestamp packet, returns zero if failed.
  // Originally only called from DecodeOggPage() if there was no timestamp received yet.
  // Since 2018-11-10, also called from handle_vorbis_packet() because in a
  //       timestamped stream received from DK7FC, the initial timestamp
  //       was received BEFORE the receiver "got all headers".
{
  struct VT_OGG_BLK *pVlftoolsOggBlock;

#if(0)
  if( !pvfh->fGotAllHeaders )  // Still waiting for initial VT_OGG_HDR ?
   {  // Note that this is *NOT* a VT_OGG_BLK with timestamp data !
      if( (size_t)op->bytes != sizeof( struct VT_OGG_HDR))
       { return 0;
       }

      struct VT_OGG_HDR vt_ogg_hdr;
      memcpy( &vt_ogg_hdr, op->packet, sizeof( struct VT_OGG_HDR));
      if( vt_ogg_hdr.magic != HDR_MAGIC)
       { return 0;
       }
      // ex: init_output( pvfh, vt_ogg_hdr.chans, vt_ogg_hdr.rate);
      //
   }
  else  // Must be a VT_OGG_BLK incoming
#endif // (0)
   {
      if( (size_t)op->bytes == sizeof( struct VT_OGG_BLK))
       { pVlftoolsOggBlock = (struct VT_OGG_BLK *)op->packet;
         switch( pVlftoolsOggBlock->magic )
          { case BLK_MAGIC:
               tq_add( pvfh, pVlftoolsOggBlock );
               // > The VT_OGG_BLK always is sent before the corresponding audio .
               break;
            default: // no other "magic block codes" supported here yet
               break;
          }
       }
   }

  return 1;
} // end handle_tstamp_packet()

/***************************************************************************/
static int handle_dummy_packet(
              T_VorbisFileHelper *pvfh, // [in,out] ogg/vorbis-related states
              ogg_packet *op  )
  // Dummy handler for ignoring unrecognised logical streams in the ogg stream
{
   return 1;
} // end handle_dummy_packet()

/***************************************************************************/
static BOOL DecodeOggPage(
              T_VorbisFileHelper *pvfh, // [in,out] ogg/vorbis-related states
              ogg_page *opage )         // [in] received Ogg page
{
  int i;
  int oldest_active_demux, oldest_activity_detector; // added 2013-07-05
  struct tStreamDemux *dm;
  ogg_packet opack; // because the stupid debugger couldn't find the type definition:

                    // It's in c:\cbproj\ogg_vorbis\libogg_1_2_0\include\ogg\ogg.h !

                    // Note: "opack.packet" is just a BYTE POINTER (aka "unsigned char").

  BOOL fResult = FALSE;

  BOOL fNewSerial = FALSE;

  BOOL recognized;


  // 2020-10-11: Crashed somewhere inside DecodeOggPage(), or one of the

  //  DLL functions invoked from there, with a 'NonDelphiException',

  //  when fed with data originating from a raw stream with VT_BLOCK structs only,

  //  i.e. a stream that did NOT contain any Ogg/Vorbis pages at all.

  //  Tried to make this thing a bit more robust then. Grep for date 2020-10-11 .

  //  Used DK7FC's test stream at tcp://129.206.29.82:49335 which had been

  //  identified as AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS

  //  (audio file/stream types defined in c:\cbproj\SoundUtl\AudioFileDefs.h).

  // Call stack (shortly before the suspected crash) :

  //  C_AnyAudioFileIO::InOpen() -> C_VorbisFileIO::InOpen() -> C_VorbisFileIO::InOpenInternal()

  //   -> ReadAndDecodePages() -> DecodeOggPage( opage->header = NULL, header_len=0, body=NULL, body_len=0 )

  if( opage->header == NULL ) // avoid crashing in ogg_page_serialno() [details below]

   { return FALSE;  // doesn't look like a valid ogg_page - ogg_sync_pageout() not successfully called ?

   }

  int serial = ogg_page_serialno( opage );
  // ogg_page_serialno(ogg_page *og) :
  // > Returns the unique serial number for the logical bitstream of this page.
  // > Each page contains the serial number for the logical bitstream that it belongs to.
  // If the stupid IDE (Borland) cannot find the implementation of "ogg_page_serialno":
  // It's in C:\cbproj\ogg_vorbis\libogg_1_2_0\src\framing.c , which -for SpecLab-
  // has been compiled into a DLL ( cbproj/SpecLab/OggVorbis4SL.dll ) because
  // Borland's linker could manage such 'incredibly large object files' anymore.
  // The same DLL also contains many other ogg_page_xyz() functions .
  // ogg_page_serialno() and many other Ogg-helper-functions in the DLL
  // don't perform any error checking at all. Instead, when fed with garbage,
  // they simply crash. Here's an example why :
  // > OV_API int ogg_page_serialno(const ogg_page *og)
  // > {
  // >   return(og->header[14] |
  // >         (og->header[15]<<8) |
  // >         (og->header[16]<<16) |
  // >         (og->header[17]<<24));
  // > }
  // Guess what happens when opage.header is NULL .. (WB, 2020-10-11) .


  // Find the demux entry for this serial, or allocate if not seen before
  dm = pvfh->demux;       // in T_VorbisFileHelper :  demux[MAX_DEMUX];  MAX_DEMUS used to be 10 ..
  for( i = 0; i<pvfh->ndemux && dm->serial != serial; i++, dm++)
   {
   }
  if( i == pvfh->ndemux)  // First time this serial has been seen?
   {
     // 2013-07-05 : Repeatedly got here after playing the 'Palaoa' audio stream
     //       for a minute, with the following .....
     //   serial = 12316 (etc, the INITIAL stream serial was 27529 in that case)
     //   pvfh->ndemux = 1
     //   pvfh->demux->serial = -1
     // An excerpt from SL's "web stream control panel" showed:
     // > 08:25:32.1 Found vorbis stream, serial 13977
     // > 08:25:32.2 Vorbis decoded first 2048 PCM samples, 0 timestamp(s).
     // > 08:26:26.2 Found vorbis stream, serial 31673
     // > 08:27:27.8 Found vorbis stream, serial 5021
     // > 08:28:28.0 Found vorbis stream, serial 26924
     // > 08:29:27.5 Found vorbis stream, serial 6270
     // > 08:30:25.9 Found vorbis stream, serial 26777
     // > 08:31:26.9 Found vorbis stream, serial 5097
     // Looks like the AWI stream server uses a new 'serial' for the Vorbis-encoded
     // audio stream once a minute. SL would run out of 'demultiplexers' after ten minutes
     // without the following modifications...
     pvfh->ndemux++;
     if( pvfh->ndemux >= MAX_DEMUX )
      { // ex: (before 2013-07-05): return FALSE;  // Ignore stream  (and the entire Ogg page)
        // 2013-07-05: tStreamDemux.serial=-1 now marks an UNUSED entry,
        //             and tStreamDemux.activity_detector will be used
        //             to recycle the OLDEST (orphaned) entry :
        for( i = 0; i<MAX_DEMUX; i++ )
         { dm = &pvfh->demux[i]; // in T_VorbisFileHelper :  demux[MAX_DEMUX];  MAX_DEMUS used to be 10 ..
           if( dm->serial < 0 ) // this entry has been FREED so it can be 'recycled' now !
            { break;
            }
         }
        if( i<MAX_DEMUX ) // still no luck.. 'recycle' the OLDEST (orphaned) entry
         { oldest_active_demux = 0;
           oldest_activity_detector = pvfh->demux[oldest_active_demux].activity_detector;
           for( i = 1; i<MAX_DEMUX; i++ )
            { if( pvfh->demux[i].activity_detector < oldest_activity_detector )
               { oldest_activity_detector = pvfh->demux[i].activity_detector;
                 oldest_active_demux = i;
               }
            }
           dm = &pvfh->demux[oldest_active_demux];  // recycle the 'oldest unused' demultiplexer, regardless of its type
         }
        if( dm->serial > 0 )  // delete the orphaned demultiplexer / ogg_stream_state struct ?
         { // About ogg_stream_destroy():
          // > This function frees the internal memory used by the
          // > ogg_stream_state struct as well as the structure itself.
          // > This should be called when you are done working with an ogg stream.
          // > It can also be called to make sure that the struct does not exist.
          // > It calls free() on its argument, so if the ogg_stream_state
          // > is not malloc()'d or will otherwise be freed by your own code,
          // > use ogg_stream_clear instead.
          ogg_stream_clear( &dm->os );
         }
      } // end if( pvfh->ndemux >= MAX_DEMUX )
     if( dm==NULL )
      { return FALSE;
      }
     ogg_stream_init( &dm->os, serial);
     dm->handler = NULL;
     dm->serial = serial;
     // 2013-07-05 : A cure for the problem with the 'changing serials'
     //   in the Palaoa streams could be to set pvfh->have_vorbis = FALSE here,
     //   to allow calling handle_vorbis_packet() further below.
     //   But that would possibly break Paul Nicholson's TIMESTAMPED VLF STREAMS,
     //   this the following was tried:
     fNewSerial = TRUE;  // added 2013-07-05
   } // end if( i == pvfh->ndemux), i.e. "first time this serial has been seen"
  else // this serial has been seen before (which is the NORMAL case) ...
   {
   }

  // Added 2013-07-05: Detect activity for THIS serial's multiplexer,
  //                   and decrement the activitiy-detector for all others
  //                   to find those which are not alive anymore (when needed):
  for( i = 1; i<MAX_DEMUX; i++ )
   { if( pvfh->demux[i].serial == serial )
      {  pvfh->demux[i].activity_detector = 10000;
      }
     else
     if( pvfh->demux[i].activity_detector > 0 )
      { --pvfh->demux[i].activity_detector;
      }
     // When running out of 'demultiplexers', the one with the LOWEST
     // activity-detector-value will be considered as 'orphaned' (not active anymore),
     // and recycled for a new 'serial' (ID). This hopefully cures the problem
     // with the 'PALAOA Neumayer Satellite Stream', which use a new stream ID each minute.
     // 2013-07-05: It did; now http://icecast.awi.de:8000/PALAOA1.OGG
     //             kept running for over 10 minutes.
   } // end for( i = 1; i<MAX_DEMUX; i++ )

  // ogg_stream_pagein : adds a complete page to the bitstream.
  // > In a typical decoding situation, this function would be called
  // > after using ogg_sync_pageout to create a valid ogg_page struct.
  // > Internally, this function breaks the page into packet segments
  // > in preparation for outputting a valid packet to the codec decoding layer.
  if( ogg_stream_pagein( &dm->os, opage) < 0)
   { // VT_bailout( "Error reading bitstream data");
     return FALSE;
   }

  // Read all available packets in this page
  //  (processing a COMPLETE page was fine in Paul's vtvorbis.c
  //   because it could push as many decompressed PCM audio samples
  //   into the output as it wanted.
  //   But here, in VorbisFileIO.cpp, there's only a limited buffer.
  //   The maximum number of audio samples decoded from a single Ogg page
  //   needs yet to be found out ! )
  while( ogg_stream_packetout( &dm->os, &opack) == 1)
   {
     // Pass the packet to the appropriate handler, if we have one,
     // otherwise try each of our decoders .
     recognized = FALSE;
     // ToDo: Get rid of these confusing states, and clean up this mess.
     //       Properly process any type of packet, regardless of "dm->handler".
     //       Begin with what cannot be a "normal" Ogg/Vorbis audio packet:
     if( opack.packet != NULL )   // don't trust anyone.. especially not "C" pointers
      {
        if( (!recognized) && (size_t)opack.bytes == sizeof( struct VT_OGG_HDR)/*16*/ )
         { // Received a "VT_OGG_HDR" (not a "VT_OGG_BLK") .. be prepared for this ANY TIME !
           struct VT_OGG_HDR *pVlftoolsOggHeader = (struct VT_OGG_HDR *)opack.packet;
           switch( pVlftoolsOggHeader->magic )
            { case HDR_MAGIC/*0x1996c3a3*/ :
                 // Receiving such a "VT_OGG_HDR" (not a "VT_OGG_BLK") when there
                 // have been other valid packets may indicate a problem with
                 // the link (broke down, server decided to "restart", etc).
                 // Not sure how to treat this properly (WB 2018-11-10) .
                 // But surely this packet was recognized, so don't pass it
                 // to any other "handler" below:
                 recognized = TRUE;
                 // What's in a "VT_OGG_HDR" ? Only the number of channels,
                 // the nominal(!) sampling rate, and a single 32-bit "spare" value.
                 CallBack_Info( pvfh, "Received VT_OGG_HDR, serial %d, %d chans, %d S/s",
                      (int)serial, (int)pVlftoolsOggHeader->chans,
                                   (int)pVlftoolsOggHeader->rate );
                 break;
              default: // no other "magic block codes" supported here yet
                 break;
            }
         } // end if < check for VT_OGG_HDR >


        if( (!recognized) && (size_t)opack.bytes == sizeof( struct VT_OGG_BLK)/*64*/ )
         { struct VT_OGG_BLK *pVlftoolsOggBlock =(struct VT_OGG_BLK *)opack.packet;
           // Unlike a "VT_OGG_HDR", this is "the stuff with the timestamp" !
           switch( pVlftoolsOggBlock->magic )
            { case BLK_MAGIC/*0x4f60f817*/ :
                 recognized = TRUE;
                 if( pvfh->i64TimestampCounter==0 )
                  { CallBack_Info( pvfh, "Received 1st VT_OGG_BLK, serial %d", serial);
                  }
                 tq_add( pvfh, pVlftoolsOggBlock );
                 // > The VT_OGG_BLK always is sent before the corresponding audio .
                 // Typical debug-output seen when receiving a timestamped stream:
                 // > 22:58:05.7 Received VT_OGG_HDR, serial 1, 1 chans, 48000 S/s
                 // > 22:58:05.7 Found vorbis stream, serial 0  (!!)
                 // > 22:58:05.7 Received 1st VT_OGG_BLK, serial 1
                 // > 22:58:06.8 Vorbis decoded first 128 PCM samples, 1 timestamp(s).
                 break;
              default: // no other "magic block codes" supported here yet
                 break;
            }
         } // end if < check for VT_OGG_BLK >
      } // end if( opack.packet != NULL )


     if( (!recognized) && (dm->handler!=NULL) )
      {  dm->handler( pvfh, &opack);
         recognized = TRUE;
      }
     // ex: if( !pvfh->have_vorbis && handle_vorbis_packet( pvfh, &opack))
     if( (!recognized) &&
       ( ( (!pvfh->have_vorbis) || fNewSerial) && handle_vorbis_packet( pvfh, &opack) )
       )
      {
        if( pvfh->have_vorbis )   // added 2013-07-05 for the Palaoa stream...
         { // OOPS...  had a different VORBIS serial before !?
         }
        pvfh->have_vorbis = TRUE;
        dm->handler = handle_vorbis_packet;
        CallBack_Info( pvfh, "Found vorbis stream, serial %d", serial);
        recognized = TRUE;
      }
     if( (!recognized) && (!pvfh->have_tstamp) && handle_tstamp_packet( pvfh, &opack))
      { // received the (?FIRST?) timestamp-packet,
        //    or at least a VT_OGG_HDR without timestamp,
        //    possibly *before* any audio sample..
        pvfh->have_tstamp = TRUE;  // have a timestamp-STREAM but not necessarily a timestamp itself !
        CallBack_Info( pvfh, "Found timestamp stream, serial %d", serial);
        if( pvfh->fPlainOggVorbis )
         {
           // VT_report( 0, "ignoring timestamp stream, -p option");
           dm->handler = handle_dummy_packet;
         }
        else
         { dm->handler = handle_tstamp_packet;
         }
        recognized = TRUE;
      }
     if( ! recognized )
      {
        // Not a stream we want to handle, dummy handler to ignore it
        dm->handler = handle_dummy_packet;
        CallBack_Info( pvfh, "Ignoring unrecognised stream, serial %d", serial);
        // 2013-07-05 : Always got here after playing the 'Palaoa' audio stream
        //       for a few seconds, with the following .....
        //  serial = 31673
        //  i=1, pvfh->ndemux=2
        //  dm->handler = NULL
        //
        //
      }
   } // end while( ogg_stream_packetout( ... )

  return fResult;

} // end DecodePage()

//---------------------------------------------------------------------------
int CheckBufferForUncompressedStreamHeader(
              BYTE *pbRxBuffer, DWORD nBytes )  // [in] received data
  // Analyses the contents of a 'receive buffer'
  //          for a possible T_StreamHeader (as defined in VorbisFileIO.h) .
  // Called internally from VorbisFileIO.cpp,
  // but also from SL to examine the initial bytes received from a remote server !
  // (for example, to find out if the delivered bytes are just an 'm3u redirector';
  //  see c:\cbproj\SpecLab\SpecMain.cpp :: OpenInputFileAnalysis() )
  // Return value :
  //   0..nBytes-1  =  index into pdwRxBuffer[] at which the next T_StreamHeader was found;
  //                    caller may use this offset to cast pbRxBuffer+<returnValue>
  //                    into a T_StreamHdrUnion* (see details in VorbisFileIO.h) .
  //                    Example (to identify the format shortly after
  //                    opening a non-compressed stream) in
  //                    TSpectrumLab::OpenInputFileAnalysis() [also used for AUDIO STREAMS].
  //   < 0          =  no valid T_StreamHeader was found in the buffer .
  // See also (similar purpose) :
  // * CheckBufferForUncompressedStreamHeader() : searches for T_StreamHeader,
  // * CheckBufferForVlfRxToolsBlock() : checks for T_VT_Block *without* a T_StreamHeader,
  // * C:\cbproj\YHF_Tools\YHF_Inet.c :: INET_GuessAudioFileTypeFromData() .
{
  int i,n;
  T_StreamHeader *pSH;
  n = (nBytes - sizeof(T_StreamHeader)) / 4;  // need at least(!) a complete T_StreamHeader in memory !
  for(i=0; i<n; i+=4/*!*/ )
   { pSH = (T_StreamHeader*)(&pbRxBuffer[i]);
     if(  (pSH->dwPattern8000 == 0x80008000UL) // 1st criterion for a header structure
       &&( pSH->dwStructID == STREAM_STRUCT_ID_VT_BLOCK) // 2nd criterion (a few more ORED here in future)
       )
      { return i;  // returns the (byte-)offset into the caller's buffer  .
        // 2014-04-22 : Got here with i=0 when receiving Paul's "test stream"
        //              ( rawtcp://67.207.139.49:9876 ) - details in the email archive.
        //              *pSH = { 2147516416, 64, 0, 1 }. Should be followed by a VT_BLOCK struct.
        // 2016-01-19 : Got here with i=0 when receiving Jacek's test stream
        //              (with problematic timestamps), also with 0x00 0x80 0x00 0x80 (0x80008000)
        //              at the BEGINNING of SQ5BPF's 48 kSample, 16 bit, TWO channel stream.
        //
      }
   } // end for <all possible 32-bit aligned positions within the buffer>


  // Arrived here: It's NOT a T_StreamHeader (alias struct STREAM_HDR_1 in vtsl.c)
  //               but maybe the steam/file begins DIRECTLY with a 'VT_BLOCK' ?
  //               Hold on, details below .
  // Besides the T_StreamHeader with its 0x80008000-pattern,
  // Paul Nicholson's VLFRX-tools also send streams beginning with
  // D3 6C = 0x6CD3 = decimal 27859 = 'MAGIC_BLK' in vtlib.c .
  // But, as it turned out in 10/2020, those haven't got anything to do
  // with T_StreamHeader.  Instead, a 'MAGIC_BLK' signature identifies
  // a VT_BLOCK struct in VLF-RX-tools, alias T_VT_Block in Spectrum Lab,
  // defined along with VT_MAGIC_BLK (27859) in c:\cbproj\SoundUtl\VorbisFileIO.h .
  //    Because a T_StreamHeader isn't a T_VT_Block, it's useless to check for
  //    other 'stream headers' in CheckBufferForUncompressedStreamHeader() !
  //    To check for VT_BLOCK (alias T_VT_Block) structures in a stream,
  //    use CheckBufferForVlfRxToolsBlock() instead.
  return -1;  // no T_StreamHeader found in the buffer

} // end CheckBufferForUncompressedStreamHeader()


//---------------------------------------------------------------------------
int CheckBufferForVlfRxToolsBlock(
              BYTE *pbRxBuffer, DWORD nBytes )  // [in] received data
  // Analyses the contents of a 'receive buffer'
  //          for a possible T_VT_Block
  //          (as defined in VorbisFileIO.h for historic reasons) .
  // Return value :
  //   0..nBytes-1  =  index into pdwRxBuffer[] at which the next T_VT_Block was found;
  //                    caller may use this offset to cast pbRxBuffer+<returnValue>
  //                    into a T_VT_Block* (see details in VorbisFileIO.h) .
  //                    Example (to identify the format shortly after
  //                    opening a non-compressed stream) in
  //                    TSpectrumLab::OpenInputFileAnalysis() [also used for AUDIO STREAMS].
  //   < 0          =  no valid T_StreamHeader was found in the buffer .
  //
  // See also (similar purpose) :
  // * CheckBufferForUncompressedStreamHeader() : searches for T_StreamHeader,
  // * CheckBufferForVlfRxToolsBlock() : checks for T_VT_Block *without* a T_StreamHeader,
  // * C:\cbproj\YHF_Tools\YHF_Inet.c :: INET_GuessAudioFileTypeFromData() .
{
  int i,n;
  T_VT_Block *pVTblk;
  n = (nBytes - sizeof(T_StreamHeader)) / 4;  // need at least(!) a complete T_StreamHeader in memory !
  for(i=0; i<n; i+=4/*!*/ )
   { pVTblk = (T_VT_Block*)(&pbRxBuffer[i]);
     if( pVTblk->magic == VT_MAGIC_BLK ) // it's a VT_BLOCK with a chance of 65535 in 65536 ...
      { if( i==0 )  // ... immediately where we expected it ? That's ok ..
         { return i;
         }
        // Arrived here ? Saw the 16-bit magic value VT_MAGIC_BLK, but not
        // in the first two bytes of the stream (-buffer). If the stream is
        // in fact headerless, raw, non-compressed audio samples,
        // we've got a chance of 1 in 65536 to get it WRONG here !
        // To reduce that chance a bit, check THIS:
        if( pVTblk->valid == 1 )  // > Set to one if block is valid
         { return i; // accept this buffer location as a valid T_VT_Block
         }
      }
   } // end for <all possible 32-bit aligned positions within the buffer>

  return -1; // there doesn't seem to be a T_VT_Block in the buffer !

} // end CheckBufferForVlfRxToolsBlock()

/***************************************************************************/
static int ReadAndDecodePages(  // usually a "blocking call" (but with LIMITED TIME spent here)
              T_VorbisFileHelper *pvfh, // [in,out] ogg/vorbis-related states
              BOOL fReadSamples, // TRUE: read and decode samples,
                                 // FALSE: only wait for the headers
              int  *piNumMillisecondsWaited ) // [out,optional] : number of milliseconds spent WAITING for input
              // (important to find out the 'pace setter' : Network stream or audio-OUTPUT ?)
  // Main part of the OGG(!) DECODER. Does what the name says, until ...
  //    with fReadSamples = FALSE  :  pvfh->fGotAllHeaders == TRUE ,  or
  //    with fReadSamples = TRUE   :
  // Details in the function body. Call stack: See handle_vorbis_packet() .
  // Return value: number of pages actually read.
  //               Zero    : nothing read but no error.
  //               Negative: one of the error codes defined in ogg.h -> codec.h,
  //                         e.g. OV_ENOTVORBIS, OV_EBADHEADER, OV_EBADPACKET .
  // Because this function is also called from C_VorbisFileIO::InOpen()
  //  to detect the stream format, it must be protected with a reliable
  //  timeout monitor to avoid getting stuck when trying to read/decode
  //  Ogg pages from a stream that only contains T_VT_Blocks (alias VT_BLOCK)
  //  as used within the VLF-RX-tools ecosystem.
{
  int  nPagesRead=0, iOffset;
  int  iWatchdogCount=2000;   // shouldn't be necessary but prevents endless loops if anything else fails
  int  ov_result; // vorbis result code (when negative, it's an error code)
  size_t nBytes;
  ogg_page opage; // ogg_page op only contains a POINTER to the page's netto data,
                  // it's a small object so it's ok to use a local stack var here.
  char *buffer;
  DWORD *pdw;
  // BOOL found_uncompressed_header;  // future plan: also allow UNCOMPRESSED "auxiliary" channels

  memset( &opage, 0, sizeof(opage) ); // don't leave anything to fate - clear garbage

  if( (pvfh->iStreamFormat == STREAM_FORMAT_NON_COMPRESSED ) // oops... wrong decoder !
    ||(pvfh->iStreamFormat == STREAM_FORMAT_VLF_RX_BLOCKS) ) // also the wrong decoder !
   { return 0;
   }

  while( ((iWatchdogCount--) > 0)  && (nPagesRead<10) )
   {
     // We want to obtain a complete page of Ogg data.
     // This is held in an ogg_page structure.
     // The process of obtaining this structure is to do the following steps:
     // 1. Call ogg_sync_pageout. This will take any data currently stored
     //    in the ogg_sync_state object and store it in the ogg_page.
     //    It will return a result indicating when the entire pages data
     //    has been read and the ogg_page can be used.
     //    It needs to be called first to initialize buffers.
     //    It gets called repeatedly as we read data from the file.
     // 2. Call ogg_sync_buffer to obtain an area of memory we can
     //    reading data from the file into. We pass the size of the buffer.
     //    This buffer is reused on each call and will be resized if needed
     //    if a larger buffer size is asked for later.
     // 3. Read data from the file into the buffer obtained above.
     // 4. Call ogg_sync_wrote to tell libogg how much data
     //    we copied into the buffer.
     // 5. Resume from the first step, calling ogg_sync_buffer.
     //    This will copy the data from the buffer into the page,
     //    and return '1' if a full page of data is available.

     // About ogg_sync_pageout() :
     //  > This function takes the data stored in the buffer of the
     //  > ogg_sync_state struct and inserts them into an ogg_page.
     //  > Caution: This function should be called BEFORE(!!!) reading
     //  >          into the buffer to ensure that data does not remain
     //  >          in the ogg_sync_state struct.
     //  >          Failing to do so may result in a memory leak.
     //  >          See the example code below for details.
     //  > Return values :
     //  > -1 : stream has not yet captured sync (bytes were skipped).
     //  >  0 : more data needed or an internal error occurred.
     //  >  1 : indicated a page was synced and returned.
     //  > Example Usage
     //  >  if (ogg_sync_pageout(&oy, &og) != 1)
     //  >   {	buffer = ogg_sync_buffer(&oy, 8192);
     //  >  	bytes = fread(buffer, 1, 8192, stdin);
     //  >  	ogg_sync_wrote(&oy, bytes);
     //  >   }
     //
     // Repeat until ogg_sync_pageout indicates a page was synced and returned .
     // Note that ogg_sync_pageout() reads blocks of data from the incoming stream,
     //           which is a bit surprising for a function named '.._pageout' !
     // Call stack: C_VorbisFileIO::ReadSampleBlocks()
     //               -> ReadAndDecodePages()
     //                   -> VorbisFileIO.cpp :: VorbisRead()
     //                       -> YHF_Inet.c :: INET_ReadDataFromServer()
     //                           -> winsock :: recv()
     // Note: Neither 'ogg_sync_pageout' nor 'ogg_sync_buffer' call any file-I/O-function !
     //       It's our own VorbisRead() that, well, READS from files or streams !
     while( (ogg_sync_pageout(&pvfh->oy/*"state"*/, &opage) != 1)
          &&( (iWatchdogCount--) > 0 )   )
      { //
        //  Call ogg_sync_buffer to obtain an area of memory we can
        //    read data from the file into. We pass the size of the buffer.
        //    This buffer is reused on each call and will be resized if needed
        //    if a larger buffer size is asked for later.
        buffer = ogg_sync_buffer(&pvfh->oy, 4096);
        if( buffer==NULL )
         { return 0;
         }
        // Read data from the file into the buffer obtained above.
        pvfh->inetReadOptions = INET_RD_OPTION_WAIT_FOR_ALL_DATA; // for VorbisRead() : don't return until the requested amount of bytes was received
        if( !pvfh->fGotAllHeaders )
         { // Modified 2013-07-05, because certain streams are 'slow STARTERS':
           if( pvfh->inetTimeout_ms < 3000 ) // here in ReadAndDecodePages() ..
            { pvfh->inetTimeout_ms = 3000;   // 2000 ms often did the trick; but add some margin
            }
         }
        else // stream already running, and "got all headers" : Only wait for as many samples as requested
         {   // in C_VorbisFileIO::ReadSampleBlocks() ... unless THE APPLICATION doesn't want to 'wait for data' at all:
           if( pvfh->iAudioFileOptions & AUDIO_FILE_OPTION_NON_BLOCKING_CALLS ) // application doesn't want to wait at all :
            { pvfh->inetTimeout_ms = 0;   // don't WAIT for data in VorbisRead() -> INET_ReadDataFromServer() at all !
              // (only read those bytes already present in the 'network buffer')
            }
           else // application wants ReadSampleBlocks() to WAIT for the arrival of data :
            { pvfh->inetTimeout_ms = 1000;
            }
         }
        nBytes = VorbisRead( // << this may already take awfully long, see YHF_Inet.c::INET_ReadDataFromServer( .., pvfh->inetTimeout_ms ) !
           buffer,          // ptr to the data that the vorbis files need
           1,               // how big a byte is  ;)   [usually one]
           4096,            // How much we can read
           (void *)pvfh,    // from where to read
           piNumMillisecondsWaited ); // [out,optional] : number of milliseconds spent WAITING for input

        if (nBytes <= 0)
         { // End of file, or (if it's a stream) must wait a bit longer :
           return nPagesRead;
         }

        // Added 2014-04-21: Save the initially received bytes in a buffer.
        //  If after the reception (and processing) of < BUFFER_MAX_DWORDS >
        //  doublewords the Ogg/Vorbis decoder didn't "get all headers"
        //  (i.e. pvfh->fGotAllHeaders still FALSE), we'll analyse
        //  pvfh->bBuffer[] and look for the pattern which indicate
        //  an 'uncompressed stream header' (T_StreamHeader defined in VorbisFileIO).
        if( (!pvfh->fGotAllHeaders) && ((pvfh->dwBufferUsage+nBytes) < BUFFER_MAX_BYTES) )
         { pdw = (DWORD*)buffer; // enough data to tell an OggVorbis stream from an uncompressed one ?
           memcpy( &pvfh->bBuffer[pvfh->dwBufferUsage]/*dst*/, buffer/*src*/, nBytes);
           pvfh->dwBufferUsage += nBytes;
         }

        // Call ogg_sync_wrote to tell libogg how much data we copied into the buffer.
        ov_result = ogg_sync_wrote(&pvfh->oy, nBytes);
        if( ov_result < 0 )
         { return ov_result;
         }

        // Reliably limit the time spent in ReadSampleBlocks() -> .. -> ReadAndDecodePages() : 
        if( TIM_ReadStopwatch_msec( &pvfh->stopwatch ) > pvfh->iMaxTimeToSpendReadingSamples_ms )
         { // leave this INNER loop; too much time spent waiting for data !
           if( pvfh->dwBufferUsage >= 64/*bytes*/ )
            { break; // let CheckBufferForUncompressedStreamHeader()
                     //  or CheckBufferForVlfRxToolsBlock() make a guess ..
            }
           else  // less than 64 bytes received -> give up
            { return OV_ERR_TIMEOUT;
            }
         }

      } // end while  < push more received data (from file or stream) into the Ogg-buffer >

     // Arrived here: there should be at least one valid Ogg page (opage),
     //               usually NOT MORE than one page - but who knows .
     // Decode those pages, until ogg_sync_pageout says we got them all .
     //  Not sure if there will be more than one page pouring out of the buffer ("state")
     //  here, but doing this repeatedly cannot hurt :
     do    //         < pull more pages out of the Ogg-internal buffer > ...
      { // Added for safety to avoid crashing in DecodeOggPage() :
        if( opage.header==NULL )
         { break; // doesn't look like a valid ogg_page - ogg_sync_pageout() failed ?
         }
        ++nPagesRead;
        // DecodeOggPage will let us know when the sample-output-buffer is too full
        // to read more input. Will break from the outer loop in that case.
        DecodeOggPage( pvfh, &opage );
      } while( (ogg_sync_pageout(&pvfh->oy, &opage) == 1 )
                && ((iWatchdogCount--) > 0) );

     // Read in more pages (because we still didn't get what the caller asked for)
     //  or return from the page-decoding-loop now ?
     if( fReadSamples )   // called to 'read audio samples' : Do we have some ?
      { // fltDecoderOutput[] should have been filled in handle_vorbis_packet():
       if( pvfh->nDecoderOutputBufferUsage > 0 ) // number of entries READ INTO fltDecoderOutput[]
         { break;   // return from this loop; there are PCM samples ready for processing
         }
      }
     else           // called from C_VorbisFileIO::InOpen : Got all headers ?
      { if( pvfh->fGotAllHeaders )
         { // yes, got all required Vorbis headers, so
           // pvfh->dblSampleRate and pvfh->nChannelsPerSampleInStream are valid, and...
           pvfh->iStreamFormat = STREAM_FORMAT_OGG_VORBIS; // we know it's NOT an uncompressed audio stream !
           break;
         }
      }

     // Reliably limit the time spent in ReadSampleBlocks() -> .. -> ReadAndDecodePages() :
     if( TIM_ReadStopwatch_msec( &pvfh->stopwatch ) > pvfh->iMaxTimeToSpendReadingSamples_ms )
      { // leave this OUTER loop; too much time spent waiting for data !
        if( pvfh->dwBufferUsage >= 64/*bytes*/ )
         { break; // let CheckBufferForUncompressedStreamHeader()
                  //  or CheckBufferForVlfRxToolsBlock() make a guess ..
         }
        else  // less than 64 bytes received -> give up
         { return OV_ERR_TIMEOUT;
         }
      }

   } // end while(1)  < push more data, and pull more pages from Ogg  >

  if( (!pvfh->fGotAllHeaders) && (pvfh->dwBufferUsage >= 64/*bytes*/) )
   { // Read and analysed a 'fair amount' of bytes, but didn't get all OggVorbis headers yet:
     // Is it an UNCOMPRESSED stream ? (just a fall-back if pvfh->iStreamFormat was WRONG)
     if( (iOffset=CheckBufferForUncompressedStreamHeader( pvfh->bBuffer, pvfh->dwBufferUsage) ) >= 0 )
      { // Indeed, it seems to be an UNCOMPRESSED ("non-Ogg/Vorbis") stream !
        pvfh->iStreamFormat=STREAM_FORMAT_NON_COMPRESSED;
      }
     else if( (iOffset=CheckBufferForVlfRxToolsBlock( pvfh->bBuffer, pvfh->dwBufferUsage) ) >= 0 )
      { pvfh->iStreamFormat=STREAM_FORMAT_VLF_RX_BLOCKS;
        // 2020-10-11 : First successfull test with DK7FC's tcp://129.206.29.82:49335 .
        // But if the correct 'format' is set BEFORE trying to open the stream,
        // we'll never get here (and never try to decode what's been received
        // as 'Ogg Pages'), but call ReadAndDecodeNonCompressed()
        // instead of ReadAndDecodePages() !  What a mess.
      }
   }
  return nPagesRead;
} // end ReadAndDecodePages()


//---------------------------------------------------------------------------
static BOOL ProcessVlfRxToolBlockFromNonCompressedStream(
                T_VorbisFileHelper *pvfh,
                T_VT_Block *pVTblk )
  // Called from ReadAndDecodeNonCompressed(),
  //  or possibly from ProcessHeaderFromNonCompressedStream()
  //  if the received block SEEMS TO contain a 'T_VT_Block' (VT_BLOCK in vlfrx-tools).
  //  [in]  pVTblk at pvfh->bBuffer[pvfh->dwBufferIndex++] ,
  //          with at least sizeof(T_StreamHdrUnion) more bytes
  //          available, which is LARGER THAN sizeof(struct VT_BLOCK) .
  //  [out] pvfh->chunk_info (*if* a valid T_VT_Block was recognized)
  //  [out] pvfh->dwBufferIndex, pvfh->i64NumTotalBytesProcessed : incremented by the number of bytes 'consumed' here
  //
{
  if( pVTblk->magic == VT_MAGIC_BLK ) // double-check this (27859), even if already checked by caller
   {
     // Convert as much as possible from Paul's VT_BLOCK struct
     //            into DL4YHF's "T_ChunkInfo" struct :
     pvfh->iVTFlags = pVTblk->flags;  // e.g. VTFLAG_INT2 (=6)
     switch( pvfh->iVTFlags & VTFLAG_FMTMASK )
      {
        case VTFLAG_FLOAT4 : // 4 byte floats
             pvfh->pThis->m_iSizeOfDataType = 4;
             pvfh->pThis->m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_FLOAT;
             break;
        case VTFLAG_FLOAT8 : // 8 byte floats
             // (that's what we got from tcp://129.206.29.82:49335
             //  instead of the expected 16-bit-INTEGER samples, in 2020-10-11)
             pvfh->pThis->m_iSizeOfDataType = 8;
             pvfh->pThis->m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_FLOAT;
             break;
        case VTFLAG_INT1   : // 1 byte signed integers
             // (not 0..255 as in ancient wave files, thus 0x80 can be avoided.. fb)
             pvfh->pThis->m_iSizeOfDataType = 1;
             pvfh->pThis->m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
             break;
        case VTFLAG_INT2   : // 2 byte signed integers
             pvfh->pThis->m_iSizeOfDataType = 2;
             pvfh->pThis->m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
             // 2016-01-19 : Got here with Jacek's VLF test stream
             //   (which didn't pass SL's "timestamp plausibility test") .
             //   At THIS point: pSHU->u.vtblock.chans = 2  ("stereo")
             //                  pSHU->u.vtblock.bsize = 8192 ("frames per block") .
             //   Jacek's stream delivered TWO channels with SIXTEEN BIT INTEGERS,
             //   48 kSamples / second -> 1500 kByte per second !
             //   But timestamps were signalled [ by calling tq_add() ]
             //
             break;
        case VTFLAG_INT4   : // 4 byte signed integers
             pvfh->pThis->m_iSizeOfDataType = 4;
             pvfh->pThis->m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
             break;
        default : // no info from VT_BLOCK.flags
             break;
      }
     pvfh->nChannelsPerSampleInStream = pVTblk->chans;
     pvfh->nSamplesPerHeaderInStream  = pVTblk->bsize; // Jacek's test: 8192 samples per header (and thus "per block")
       // VT_BLOCK.bsize = > Number of frames per block.
       //         (guess such a 'frame' is a group of samples
       //          in a single 'sample point', possible containing
       //          multiple CHANNELS but not multiple SAMPLE POINTS in time.
       //          In Paul's vtlib.c, bsize depends on the sampling rate,
       //          and may be a multiple or fraction of 8192 - see choose_bsize().
       //          Expect a MAXIMUM 'bsize' of 262144 [sample points], thus
       //          in a single call of ReadAndDecodeNonCompressed(), we can only
       //          process a SMALL PART of the samples following a single VT_BLOCK.
       // VT_BLOCK.frames= > Number of frames actually contained.
       //         (contained in what, the current block ? No, that's 'bsize'.)
     if( pVTblk->frames > 0 ) // so far "frames" was always the same as "bsize" !
      { pvfh->iSampleCountdownBetweenHeaders = pVTblk->frames; // "Number of frames actually contained."
      }
     else
      { pvfh->iSampleCountdownBetweenHeaders = pVTblk->bsize;  // "Number of frames per block."
      }
     pvfh->dblNominalSampleRate = pVTblk->sample_rate; // Nominal sample rate
     pvfh->chunk_info.dblPrecSamplingRate = pVTblk->sample_rate;
     pvfh->chunk_info.ldblUnixDateAndTime = (long double)pVTblk->secs + (long double)pVTblk->nsec * 1e-9;
     if( pVTblk->valid ) // > Set to one if block is valid
      { pvfh->iStreamFormat = STREAM_FORMAT_VLF_RX_BLOCKS; // assume there will be MORE OF THESE BLOCKS later
        pvfh->chunk_info.dwValidityFlags |=
                 CHUNK_INFO_TIMESTAMPS_VALID
               | CHUNK_INFO_SAMPLE_RATE_VALID;
        if( pVTblk->srcal >= 0.9 && pVTblk->srcal <= 1.1 )
         {  pvfh->chunk_info.dblPrecSamplingRate = (double)pVTblk->sample_rate * pVTblk->srcal;
            // Jacek's test: dblPrecSamplingRate = 48000.335849 . Ok.
            pvfh->chunk_info.dwValidityFlags |= CHUNK_INFO_SAMPLE_RATE_CALIBRATED;
         }
        if( pvfh->chunk_info.ldblUnixDateAndTime > 0.0 )
         {
            pvfh->chunk_info.dwValidityFlags |=
                 CHUNK_INFO_TIMESTAMPS_VALID    /* applies to .rtRecTime, .ldblUnixDateAndTime */
               | CHUNK_INFO_TIMESTAMPS_PRECISE; /* assume .ldblUnixDateAndTime "is precise" to ONE AUDIO SAMPLE */
         }
      }
     else // !  pVTblk->valid  ...
      { pvfh->chunk_info.dwValidityFlags &= ~CHUNK_INFO_TIMESTAMPS_VALID;
      }
     // Add this block to the internal timestamp queue, because
     // the application generally reads samples in different chunk sizes
     // than contained in the stream.
     // Thus we need to interpolate later (in C_VorbisFileIO::ReadSampleBlocks).
     tq_add_vtblock( pvfh, pVTblk );  // -> tq_add(), same as for ogg/vorbis
     pvfh->fGotAllHeaders = TRUE;
     // Similar as (before 2020-10) in ProcessHeaderFromNonCompressedStream(),
     // skip the T_VT_Header in the input buffer. If (since 2020-10)
     // ProcessVlfRxToolBlockFromNonCompressedStream() is CALLED FROM THERE,
     // the following buffer-index-increment will have no effect :
     pvfh->dwBufferIndex             += sizeof( T_VT_Block );
     pvfh->i64NumTotalBytesProcessed += sizeof( T_VT_Block );
     return TRUE;
   }
  else // whatever it is, it doesn't have VT_MAGIC_BLK (27859) ...
   { return FALSE;
   }

} // end ProcessVlfRxToolBlockFromNonCompressedStream()


//---------------------------------------------------------------------------
static BOOL ProcessHeaderFromNonCompressedStream(T_VorbisFileHelper *pvfh)
  // Called from ReadAndDecodeNonCompressed() on detection of pattern 0x80008000.
  //  [in]  pvfh->bBuffer[pvfh->dwBufferIndex++] ,
  //          with at least sizeof(T_StreamHdrUnion) more bytes
  //          available, which is LARGER THAN sizeof(struct VT_BLOCK) .
  //  [out] pvfh->chunk_info (*if* a valid header was found)
{
  BOOL fResult = FALSE;
  DWORD    dwOldBufferIndex = pvfh->dwBufferIndex;
  LONGLONG i64OldTotalBytesProcessed = pvfh->i64NumTotalBytesProcessed;

  // On entry, pvfh->dwBufferIndex must be a multiple of four, because
  //   the following struct (a 'stream header') must be on a DWORD-boundary:
  T_StreamHdrUnion *pSHU = (T_StreamHdrUnion*)&pvfh->bBuffer[pvfh->dwBufferIndex];

  // Check the SECOND criterion for any valid 'header' in an uncompressed stream:
  // The size of any header (as well as any block of samples) must be a multiple of FOUR (bytes).
  if( (pSHU->header.nBytes & 0x00000003) == 0 ) // additional criterion for a valid header struct
   { // 2016-01-20 : Got here with Jacek's *mono* 16-bit-per-sample stream with
     //   (1st) pvfh->dwBufferIndex = 0x00000, pSHU->header.nBytes = 64, pSHU->header.dwStructID = 1  (ok).
     //
     switch( pSHU->header.dwStructID ) // use or ignore this 'header' ?
      { case STREAM_STRUCT_ID_NONE    :  // ignore (just a "test dummy")
        default :
           break;
        case STREAM_STRUCT_ID_VT_BLOCK:  // it's a VT_BLOCK as defined in Paul's vlfrx-tools
           if( pSHU->u.vtblock.magic == VT_MAGIC_BLK ) // should contain VT_MAGIC_BLK = 27859 = Data block header
            { fResult = ProcessVlfRxToolBlockFromNonCompressedStream( pvfh, &pSHU->u.vtblock );
              // Beware: ProcessVlfRxToolBlockFromNonCompressedStream()
              //         will increment pvfh->dwBufferIndex and pvfh->i64NumTotalBytesProcessed
              //                  by sizeof( T_VT_Block ) !
              // Thus the kludgy use of dwOldBufferIndex further below,
              //      instead of "pvfh->dwBufferIndex += pSHU->header.nBytes" .
            } // end if < correct 'magic value' in VT_BLOCK
           break; // end case STREAM_STRUCT_ID_VT_BLOCK
      } // end switch( pvfh->next_header.dwStructID )
     // Even if we don't know "what to do with the header" here,
     // SKIP IT from the stream (to make future additions in streams painless).
     // ex: pvfh->dwBufferIndex += pSHU->header.nBytes; // index into pvfh->bBuffer[] (head) - usually incremented by 64 here
     pvfh->dwBufferIndex = dwOldBufferIndex + pSHU->header.nBytes; // index into pvfh->bBuffer[] (head) - usually incremented by 64 here
     pvfh->i64NumTotalBytesProcessed = i64OldTotalBytesProcessed + pSHU->header.nBytes;
   } // end if < valid criteria for a 'header' >
  return fResult;
} // end ProcessHeaderFromNonCompressedStream()


//---------------------------------------------------------------------------
int VFIO_ChunkInfoToStreamHeader(
        T_VorbisFileHelper *pvfh, // [in] pvfh->iVTFlags, pvfh->nChannelsPerSampleInStream,
                                  //      pvfh->nSamplesPerHeaderInStream, ...
        T_ChunkInfo *pChunkInfo,  // [in]  chunk info with timestamp, etc etc etc
        int  iSampleOffset,       // [in] sample offset, required to adjust the timestamp
        T_StreamHdrUnion *pSHU)   // [out] stream header + VT_BLOCK
  // Returns the size of the 'filled out header' in bytes.
  //  Depending on the contents of the chunk info, the caller may have to write
  //  LESS than sizeof(T_StreamHdrUnion) !
{
  int nBytes = sizeof(T_StreamHeader) + sizeof(T_VT_Block);
  long double ldbl;

  memset( pSHU, 0, sizeof(T_StreamHdrUnion) );

  // Fill out the common header :
  pSHU->header.dwPattern8000 = STREAM_HDR_PATTERN; // unique pattern to identify this header
  pSHU->header.nBytes        = nBytes;
  pSHU->header.dwStructID    = STREAM_STRUCT_ID_VT_BLOCK;


  // Fill out the VT_BLOCK, compatible with Paul Nicholson's VLFRX tools:
  pSHU->u.vtblock.magic = VT_MAGIC_BLK;
  pSHU->u.vtblock.flags = pvfh->iVTFlags; // VTFLAG_INT1/2/4, etc
  pSHU->u.vtblock.bsize = pvfh->nSamplesPerHeaderInStream; // Number of frames per block
  pSHU->u.vtblock.frames= pvfh->nSamplesPerHeaderInStream; // Number of frames actually contained
  pSHU->u.vtblock.chans = pvfh->nChannelsPerSampleInStream; // Number of channels per frame
  pSHU->u.vtblock.sample_rate = pvfh->dblNominalSampleRate; // Nominal sample rate
  ldbl = pChunkInfo->ldblUnixDateAndTime
       + (long double)iSampleOffset / pChunkInfo->dblPrecSamplingRate;
  pSHU->u.vtblock.secs  = (uint32_t)ldbl; // Timestamp, seconds part
  ldbl = 1e9 * (ldbl - (long double)pSHU->u.vtblock.secs); // -> fractional part in ns
  pSHU->u.vtblock.nsec  = (uint32_t)ldbl; // Timestamp, nanoseconds part
  pSHU->u.vtblock.valid = 1;     // Set to one if block is valid
  pSHU->u.vtblock.spare = 0;
  pSHU->u.vtblock.srcal = 1.0; // Actual rate is sample_rate * srcal (8 byte IEEE double precision float)
  return nBytes;
} // end VFIO_ChunkInfoToStreamHeader()


//---------------------------------------------------------------------------
int GetBytesPerSampleInUncompressedStream(T_VorbisFileHelper *pvfh)
{
  switch( pvfh->iVTFlags & VTFLAG_FMTMASK )
   {
     case VTFLAG_FLOAT4 : // 4 byte floats
        return 4 * pvfh->nChannelsPerSampleInStream;
     case VTFLAG_FLOAT8 : // 8 byte floats
        return 8 * pvfh->nChannelsPerSampleInStream;
     case VTFLAG_INT1   : // 1 byte signed integers
        return pvfh->nChannelsPerSampleInStream;
     case VTFLAG_INT2   : // 2 byte signed integers
        return 2 * pvfh->nChannelsPerSampleInStream;
     case VTFLAG_INT4   : // 4 byte signed integers
        return 4 * pvfh->nChannelsPerSampleInStream;
     default : // unsupported format !
        return 0;
   } // end switch < data type flags from the VT_BLOCK header >
} // end GetBytesPerSampleInUncompressedStream()

//---------------------------------------------------------------------------
static int ReadAndDecodeNonCompressed( // also turned into a 'monster' over the years...
              T_VorbisFileHelper *pvfh,  // [in,out] instance data
              int nDestMaxSamplePoints,  // [in] maximum(!) number of SAMPLE POINTS to read into the destination
                                         //      MAY BE ZERO IF THE CALLER DOESN'T WANT TO PROCESS ANY SAMPLES YET,
                                         //      but find out the 'properties' of the freshly opened stream or file.
              int nDestChannelsPerSample,// [in] number of channels PER SAMPLE POINT for the destination
                                         //      MAY BE ZERO IF THE CALLER DOESN'T WANT TO PROCESS ANY SAMPLES YET,
                                         //      but find out the 'properties' of the freshly opened stream or file.
              float * pfltDest,          // [out] float *pfltDest
                                         //      MAY BE NULL IF THE CALLER DOESN'T WANT TO PROCESS ANY SAMPLES YET,
                                         //      but find out the 'properties' of the freshly opened stream or file.
              int  *piNumMillisecondsWaited ) // [out,optional] : number of milliseconds spent WAITING for input; "accumulating".
                                         // (important to find out the 'pace setter' : Network stream or audio-OUTPUT ?)
  //  [in] pvfh->inetTimeout_ms : MAXIMUM number of milliseconds to spend here.
  //
  //
  // Similar as ReadAndDecodePages(), but for NON-COMPRESSED streams (several formats).
  // Since 2020-10, supports STREAM_FORMAT_VLF_RX_BLOCKS  (streams with T_VT_Blocks but no T_StreamHeaders)
  //                 besides STREAM_FORMAT_NON_COMPRESSED (streams with everything wrapped in T_StreamHeaders).
  // Called from C_VorbisFileIO::ReadSampleBlocks(),
  // but also from C_VorbisFileIO::InOpenInternal() [once] to read/analyse
  //               the initial 'stream header' or 'VT_Block' !
  // Return value: number of SAMPLE POINTS (with 1 or 2 channels each) actually read.
  //               Zero    : nothing read but no error.
  //               Negative: one of the error codes defined in ogg.h
{
  int  iWatchdogCount=2000;   // shouldn't be necessary but prevents endless loops if anything else fails
  int  crud, nBytesToProcess, nBytesToRead, nBytesReadFromFile=0, nSamplePointsRead=0;
  int  nBytesReadInThisCallOfReadAndDecode = 0;
  int  nNumMillisecondsWaited = 0;
  int  iChannel, nCommonChannelsPerSample, nSourceBytesPerSample, nDestUnusedChannelsPerSample;
  T_StreamPtrUnion pSrc;
  BOOL fProcessSamplesNow; // TRUE=yes, FALSE=no ...
  BOOL fAnythingHappenedInThisLoop; // WHAT AN UGLY HACK ! But this stopped the watchdog from barking .
  T_TIM_Stopwatch myStopwatch;
  char sz511[512], *cp;
  __int64 i64PrevNumSamplesDecoded = pvfh->i64NumSamplesDecoded;

  static __int64 nTotalSampleCounter = 0;    // only for TESTING
  static __int64 iSampleCountAtLastHeader=0; //   "   "   "
  static long nCalls = 0;                    //   "   "   "

  ++nCalls;
  if(nCalls==5)
   { nCalls=nCalls;  // << conditional breakpoint,
     // to find the reason for the wrong "nSamplesBetweenHeaders" .
     // Details in C:\cbproj\SpecLab\test_notes\2014_04_26_bugs_in_VorbisFileIO.txt .
   }

  TIM_StartStopwatch( &myStopwatch );  // extra timeout-monitor

  // Place the non-compressed samples ("pcm") in the same destination
  //      as if handle_vorbis_packet() would do (for real Ogg/Vorbis) :
  // Note that the *CHANNELS* in pvfh->fltDecoderOutput[] are interleaved;
  // i.e. pvfh->fltDecoderOutput[0] = left channel, first sample,
  //      pvfh->fltDecoderOutput[1] = right channel, first sample,
  //      pvfh->fltDecoderOutput[0] = left channel, second sample,
  //      pvfh->fltDecoderOutput[1] = right channel, second sample;  etc.
  nSourceBytesPerSample = GetBytesPerSampleInUncompressedStream(pvfh); // f(data type, number of channels)
  // 2020-10-11 : The above may be 'just a guess' if we haven't found out
  //              if the format is STREAM_FORMAT_VLF_RX_BLOCKS
  //                            or STREAM_FORMAT_NON_COMPRESSED yet !
  //
  while( (iWatchdogCount--) > 0)  // receive more bytes, look for 'headers' and/or VT_BLOCKs, convert samples...
   { nBytesReadFromFile = 0; // .. in THIS LOOP ..
     fAnythingHappenedInThisLoop = FALSE;
     if( nTotalSampleCounter == 8192 )
      { nTotalSampleCounter = nTotalSampleCounter; // <-- place for a conditional breakpoint
        // (after 8192 samples, we often expect the next VT_BLOCK
        //  in a VLF-RX-tools-compatible audio stream, but ymmv.
        //  At this point, pvfh->iSampleCountdownBetweenHeaders should be 0 .
        //  But why was pvfh->dwBufferIndex 48 bytes larger (65584)
        //         than pvfh->dwBufferUsage (65536 = max = BUFFER_MAX_BYTES) ?
      } // end if < "conditional breakpoint" for hardcore debugging >

     // Read more input from the file/stream into bBuffer[] ?
     nBytesToProcess = pvfh->dwBufferUsage - pvfh->dwBufferIndex; // -> number of bytes still buffered but not processed yet
           // (example: dwBufferUsage=1, dwBufferInex=0 -> nBytesToProcess=1, ONE byte left to process)
     nBytesToRead = (BUFFER_MAX_BYTES - nBytesToProcess); // -> APPROXIMATE number of bytes which may be read into the buffer now
     if( (nBytesToRead >= (BUFFER_MAX_BYTES/4) )  // don't call the blocking VorbisRead() unnecessarily often ..
      ||((nBytesToRead > 0) && (!pvfh->fIsInternetStream) ) ) // ... unless the source is a REAL FILE (then read "all we can")
      {
        if( nBytesToProcess > 0 )
         { // If there are a few bytes remaining AT THE END OF THE BUFFER which have not been processed yet,
           //     memmove (not memcopy!) them to the BEGIN of the buffer :
           // ex: memmove( &pvfh->bBuffer[0]/*dst*/, &pvfh->bBuffer[pvfh->dwBufferIndex]/*src*/, nBytesToProcess);
           //     pvfh->dwBufferIndex = 0;  // index for the next byte to process (further below)
           //     pvfh->dwBufferUsage = (DWORD)nBytesToProcess; // number of bytes now in the buffer
           // BUT The network socket API may return any number of bytes from the network buffer,
           //     including 'zero' and 'one' byte.
           //     Also, dwBufferIndex may not be a multiple of four -> MISALIGNED data,
           //     because even though bBuffer[] is an array of BYTES, the entries
           //     in it must be aligned for 64-bit integers and floats,
           //     relative to the very first byte received from the stream !
           //     Thus the address offset (distance between 'source' and 'destination' for memmove)
           //     must always be a MULTIPLE OF EIGHT .
           // Principle illustrated in  C:\cbproj\SpecLab\test_notes\2014_04_26_bugs_in_VorbisFileIO.txt .
           // Solution (to keep pvfh->bBuffer[0] aligned to N*8 bytes since the begin of the stream):
           crud = pvfh->dwBufferIndex & 7;  // -> number of 'crud' bytes required at the NEW begin of the buffer (see principle)
           memmove( &pvfh->bBuffer[0], // destination, including 0..7 'crud' bytes for alignment
                    &pvfh->bBuffer[pvfh->dwBufferIndex-crud], // source, also 8-byte-aligned
                    nBytesToProcess+crud); // number of bytes remaining in the buffer, including 0..7 crud bytes for alignment
           // Note that the difference between source and destination is always N*8 bytes,
           // thus the data in the buffer will always remain 8-byte-aligned .
           pvfh->dwBufferIndex = crud;  // continue processing AFTER the 'crud' (see principle)
           pvfh->dwBufferUsage = (DWORD)(crud+nBytesToProcess); // number of bytes now in the buffer,
               // for example 2+1022 = 1024 bytes
         }
        else // buffer was completely empty (BufferIndex==BufferUsage), but 'beware of crud' ?
         { crud = pvfh->dwBufferUsage & 7;
           pvfh->dwBufferIndex = crud;
           pvfh->dwBufferUsage = crud;
         }
        nBytesToRead = (BUFFER_MAX_BYTES - pvfh->dwBufferUsage); // -> number of bytes which may be appended to dwBuffer+dwBufferIndex
        // When reading samples from the buffer in 'frame' sizes notequal to N * 8 (bytes),
        //      the DWORD-aligned test for the 4-byte-pattern ( (pvfh->dwBufferIndex & 3) == 0 )
        //      must not fail. So TRY TO read data from the input in chunks of N * 8 bytes, too.
        // DON'T ASSUME ANYTHING about the number of samples requested PER BLOCK by the caller,
        // and expect the number of bytes per sample point to be anything, even or odd, etc.
        nBytesToRead &= 0x7FFFFF8;  // <- usually "does nothing"
            // because the buffer-index is aligned to EIGHT-byte-boundaries since 10/2020)

        // VorbisRead() may already take awfully long, depending on the timeout
        //  in YHF_Inet.c::INET_ReadDataFromServer(), but WB didn't want all the fuss with multithreading here.
        if( pvfh->i64NumSamplesDecoded <= 0 )
         { pvfh->inetReadOptions = INET_RD_OPTION_WAIT_FOR_ALL_DATA; // for VorbisRead() : don't return until the requested amount of bytes was received
           pvfh->inetTimeout_ms  = 3000;
         }
        else // If the stream is already RUNNING, don't wait long for reception of data, just return "all we have at the moment"
         { pvfh->inetReadOptions = INET_RD_OPTION_NORMAL; // for VorbisRead() : no need to wait until the buffer is COMPLETELY FULL - just return those bytes from the network buffer
           pvfh->inetTimeout_ms  = 300;
           // Similar as in ReadAndDecodePages() [for COMPRESSED streams], prepare pvfh->inetTimeout_ms depending on AUDIO_FILE_OPTION_NON_BLOCKING_CALLS :
           if( pvfh->iAudioFileOptions & AUDIO_FILE_OPTION_NON_BLOCKING_CALLS ) // application doesn't want to wait at all :
            { pvfh->inetTimeout_ms = 0;   // don't WAIT for data in VorbisRead() -> INET_ReadDataFromServer() at all !
              // (only read those bytes already present in the 'network buffer')
            }
           else // the application wants ReadSampleBlocks() to WAIT for the arrival of data :
            { // [in] pvfh->stopwatch, pvfh->iMaxTimeToSpendReadingSamples_ms : prepared in ReadSampleBlocks()
              pvfh->inetTimeout_ms = pvfh->iMaxTimeToSpendReadingSamples_ms - TIM_ReadStopwatch_msec( &pvfh->stopwatch );
              //  ,---'
              //  '--> used as input in VorbisRead() -> INET_ReadDataFromServer() .
            }
         }
        nBytesReadFromFile = VorbisRead(   // <- reads from file or stream, but doesn't convert or decode anything
           &pvfh->bBuffer[pvfh->dwBufferUsage], // destination : AFTER the not-yet-processed bytes which must remain in the buffer
           sizeof(BYTE),    // size of a single buffer item
           nBytesToRead,    // max. number of buffer items TO READ
           (void *)pvfh,    // a pointer to the data which were once passed into ov_open_callbacks
           &nNumMillisecondsWaited); // [in,out] number of milliseconds spent WAITING for input; "accumulated".
           // (important to find out the 'pace setter' in the processing thread: Network stream or audio-OUTPUT ?)

        if( nBytesReadFromFile > 0 )
         { pvfh->dwBufferUsage += nBytesReadFromFile;
           nBytesReadInThisCallOfReadAndDecode += nBytesReadFromFile;
           fAnythingHappenedInThisLoop = TRUE; // something happened : successfully READ SOMETHING from file/stream
           if( pvfh->dwBufferUsage > BUFFER_MAX_BYTES ) // something went wrong ("assertion failed") ..
            {  VFIO_DebuggerBreak( pvfh, "VorbisRead() running wild !", __LINE__ );
               pvfh->dwBufferUsage = BUFFER_MAX_BYTES;
            }
         }
        nBytesToProcess = pvfh->dwBufferUsage - pvfh->dwBufferIndex; // -> number of bytes buffered but not processed yet
      } // end if < read more data into the buffer > ?


     // Process more samples from bBuffer[] ?  Not yet.. first look for a HEADER and/or VT_BLOCK !
     // ProcessHeaderFromNonCompressedStream() may only be called from here
     //   with at least sizeof(T_StreamHdrUnion) more bytes
     //   available in the buffer. Similar for streams with only VT_BLOCKs
     //   but no T_StreamHeaders (since 2020-10) .
     //   Note: sizeof(T_StreamHdrUnion) is a bit LARGER than sizeof(struct VT_BLOCK) !
     if( pvfh->iSampleCountdownBetweenHeaders <= 0 )  // only "expecting" stream-headers or VT_BLOCKs now (not SAMPLES) ?
      {
        if( (nBytesToProcess >= sizeof(T_StreamHdrUnion) ) // enough bytes for a 'stream header' ...
          ||(nBytesToProcess >= sizeof(T_VT_Block) ) // ... or for a 'VT_BLOCK' ?
          )
         {
           // If the current buffer index is a multiple of four ("DWORD-aligned"),
           //  check for a HEADER (structure) in the stream (which is 'completely in the buffer',
           //  because there will always be at least 64 bytes remaining in the buffer (see above).
           // (For a stream with one channel and 16 bits per channel, we will get HERE
           //  with (pvfh->dwBufferIndex & 3)  == 0,2,0,2,0,2... ! )
           if( (pvfh->dwBufferIndex & 3)==0 )   // ok, we're on a DWORD-boundary ..
            {
              if( (*(DWORD*)(&pvfh->bBuffer[pvfh->dwBufferIndex]) == 0x80008000UL ) // << 1st criterion for a header structure
               && (nBytesToProcess >= sizeof(T_StreamHdrUnion) ) // .. only if enough data for a 'stream header' already...
                )
               {
                 // Should get here with pvfh->dwBufferIndex = 0, because the file or stream *should* begin with a header.
                 // But don't assume anything. 2016-01-20 : When analysing Jacek's test stream,
                 //   got here with (1st) pvfh->dwBufferIndex = 0x00000 (ok),
                 //                 (2nd) pvfh->dwBufferIndex = 0x04040 (ok),
                 //                 (3rd) pvfh->dwBufferIndex = 0x08080 (ok),
                 //
                 if( ProcessHeaderFromNonCompressedStream(pvfh) ) // [in]  pvfh->dwBuffer[pvfh->dwBufferIndex++] ->
                  {
                    // Update a few parameters which MAY depend on the new header.
                    // Note: pvfh->dwBufferIndex may have been incremented by a few dozen bytes,
                    //       which means 'the buffer may be too empty now' for reading the next frame (=sample point) !
                    //       For example, when reading a stream (or file) with a VT_BLOCK
                    //       at the begin, pvfh->dwBufferIndex will be 64 at this point.
                    nSourceBytesPerSample = GetBytesPerSampleInUncompressedStream(pvfh); // f(data type, number of channels)
                    // Added for testing (2014-04-25) : How many samples have REALLY been received
                    //                                  between TWO HEADERS ?
                    int nSamplesBetweenHeaders = nTotalSampleCounter - iSampleCountAtLastHeader;
                    iSampleCountAtLastHeader = nTotalSampleCounter;
                    if( (nSamplesBetweenHeaders!=0) && (nSamplesBetweenHeaders!=pvfh->nSamplesPerHeaderInStream ) )
                     { nSamplesBetweenHeaders = nSamplesBetweenHeaders; // << place for a breakpoint
                       // 2014-04-26: Got here .. details in C:\cbproj\SpecLab\test_notes\2014_04_26_bugs_in_VorbisFileIO.txt .
                       // The '0x80008000'-patterns were at file were ok, at pos. 0x0000, 0x1040, 0x2080, 0x30C0,
                     }
                    fAnythingHappenedInThisLoop = TRUE; // something happened : processed a HEADER and drained it from the buffer
                  }
                 else // Suspicious but not necessarily a bug: saw the pattern for a header struct (0x80008000),
                  {   // but ProcessHeaderFromNonCompressedStream() could not handle it
                    pvfh->dwBufferIndex = pvfh->dwBufferIndex; // << place for a breakpoint
                    VFIO_DebuggerBreak( pvfh, "ReadAndDecodeNonCompressed: unrecognized header", __LINE__ );
                  }
                 nBytesToProcess = pvfh->dwBufferUsage - pvfh->dwBufferIndex; // -> number of bytes buffered but not processed yet,
                 //  "revised" because ProcessHeaderFromNonCompressedStream() MAY have
                 //  incremented pvfh->dwBufferIndex, even if it didn't know how to process the header
                 //  (and just SKIPPED it, and returned FALSE for that reason).
                 // 2016-01-20 : In the FIRST call, got here with nBytesToProcess = 65472 = 65536-64. Ok.
               } // end if < 1st criterion for a header structure (T_StreamHeader) fulfilled >
              else // Not a T_StreamHeader with the 0x80008000 pattern .. what else ?
              if( (  ( (pvfh->dwBufferIndex == 0 ) // only at the FIRST byte in the FIRST buffer...
                    && (pvfh->i64NumTotalBytesProcessed == 0) ) // ... and nothing processed in a PREVIOUS call...
                   ||(  pvfh->iStreamFormat==STREAM_FORMAT_VLF_RX_BLOCKS) ) // ... or if we already know there are VT_BLOCKs in here..
               && (nBytesToProcess >= sizeof(T_VT_Block) ) // ... and enough bytes for a T_VT_Block (48) ...
               && (*(WORD*)(&pvfh->bBuffer[pvfh->dwBufferIndex]) == VT_MAGIC_BLK ) // it's a VT_BLOCK with a chance of 65535 in 65536 ...
                ) // << SOME criteria for a T_VT_Block struct, but there's more ...
               { if( ProcessVlfRxToolBlockFromNonCompressedStream( pvfh,
                      (T_VT_Block*)&pvfh->bBuffer[pvfh->dwBufferIndex] ) )
                  { // 2020-10-11 : First successfull test with DK7FC's tcp://129.206.29.82:49335 .
                    // In older versions, T_VT_Blocks were only accepted when wrapped
                    // in a T_StreamHeader (the thing with the magic 0x80008000 pattern).
                    nBytesToProcess = pvfh->dwBufferUsage - pvfh->dwBufferIndex; // -> number of bytes buffered but not processed yet,
                    //  "revised" because ProcessVlfRxToolBlockFromNonCompressedStream()
                    //  MAY(!) also have incremented pvfh->dwBufferIndex .
                    //  Otherwise, the bytes belonging to this T_VT_Block
                    //  may be 'misinterpreted' as AUDIO SAMPLES further below.
                    // Typically, after reading 65536 bytes into our buffer,
                    // nBytesToProcess=65488 remained to process 'as samples'
                    // (further below), or simply left in our network-buffer
                    // FOR LATER (if the caller doesn't want any samples yet).
                    fAnythingHappenedInThisLoop = TRUE; // something happened : processed a VT_BLOCK and drained it from the buffer
                  } // end if( ProcessVlfRxToolBlockFromNonCompressedStream() .. )
               } // end if < 1st criterion for a VLF-RX-Tool-"Block" (T_VT_Block without T_StreamHeader) fulfilled >
              else // neither recognized a T_StreamHeader nor a T_VT_Block .. is this acceptable ?
               { switch( pvfh->iStreamFormat )
                  { case STREAM_FORMAT_VLF_RX_BLOCKS : // getting here is an ERROR ...
                       if( ! pvfh->fMissingVTBlock )
                        { pvfh->fMissingVTBlock = TRUE; // avoid flooding the error log with these...
                          // %lld for 64-bit integer didn't seem to work with Borland C++Builder V6,
                          // so use %Ld instead for what they call "__int64" :
                          CallBack_Info( pvfh, "Missing VT_BLOCK at byte index %Ld",
                              (__int64)pvfh->i64NumTotalBytesProcessed );
                        }
                       break;
                    default:
                       break;
                  } // end switch( pvfh->iStreamFormat )
               } // end else < neither recognized a T_StreamHeader nor a T_VT_Block >
            } // end if < buffer index on a DOUBLEWORD boundary ? >
         }  // end if( nBytesToProcess >= sizeof(T_StreamHdrUnion) )
      } // end if( pvfh->iSampleCountdownBetweenHeaders <= 0 )

     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // above : check for HEADERS in the non-compressed input stream.
     // below : process SAMPLES (if still enough bytes are in the buffer)
     //         Beware, the buffer may not contain ALL samples as specified
     //         in the T_VT_Block or T_StreamHeader yet.
     //         For example, from 65536 bytes originally in pvfh->bBuffer[]
     //         only nBytesToProcess = 65488 MAY (or may not!) contain
     //         SAMPLES.
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     while( (nBytesToProcess >= nSourceBytesPerSample )    // e.g. here with 65488 bytes of "samples" to process, pvfh->dwBufferIndex=48=sizeof(VT_BLOCK), pvfh->iSampleCountdownBetweenHeaders=8192
         && (nSamplePointsRead < nDestMaxSamplePoints ) )  // e.g. with nDestMaxSamplePoints = 16384 (typical chunk size used by Spectrum Lab) -> need MULTIPLE READS a 65536 bytes !
      { fProcessSamplesNow = FALSE;
        if( pvfh->iSampleCountdownBetweenHeaders > 0 ) // definitely "expecting" SAMPLES, not stream-headers or VT_BLOCKs now ->
         { fProcessSamplesNow = TRUE;
         }
        else // iSampleCountdownBetweenHeaders <= 0 .. expect SAMPLE POINTS anyway ?
         { switch( pvfh->iStreamFormat )
            { case STREAM_FORMAT_VLF_RX_BLOCKS : // stream with samples 'counted' in VT_BLOCKs but without T_StreamHeaders ->
                 // Don't set fProcessSamplesNow in this case !
                 fProcessSamplesNow = FALSE;
                 // If, for example, the loop for VTFLAG_FLOAT8 (further below)
                 // encounters an 'invalid' 8-byte floating point number
                 // (even with the source address properly aligned to 8 bytes),
                 // Mr. FPU will raise an exception with "floating point overflow" !
                 break;
              case STREAM_FORMAT_NON_COMPRESSED: // Non-compressed, but usually with T_StreamHeader blocks
                 // This format was initially used by DL4YHF to send 'uncounted' samples,
                 // e.g. very simple streams without 'stream headers' at all,
                 // or using T_StreamHeaders with a 'magic value' (0x80008000)
                 // that would never occurr in digitzed 16- or 32-bit integer
                 // samples at all (due to the limitation to +/-32767, not -32768).
                 if(  ((pvfh->dwBufferIndex & 3) == 0 ) // on a DWORD-boundary, AND..
                   && (*(DWORD*)(&pvfh->bBuffer[pvfh->dwBufferIndex]) == 0x80008000UL )
                   )
                  { // This is not a digitzed sample but another T_StreamHeader !
                    fProcessSamplesNow = FALSE;
                  }
                 else // assume a stream with non-COUNTED samples (no VT_BLOCKs, no T_StreamHeaders)
                  { fProcessSamplesNow = TRUE;
                  }
                 break;
            } // end switch( pvfh->iStreamFormat )
         } // end else < iSampleCountdownBetweenHeaders <= 0 >
        if( fProcessSamplesNow )
         {
           if( pvfh->nChannelsPerSampleInStream >= nDestChannelsPerSample )
            {  nCommonChannelsPerSample = nDestChannelsPerSample;
               nDestUnusedChannelsPerSample = 0;
            }
           else // pvfh->nChannelsPerSampleInStream < nDestChannelsPerSample :
            {  nCommonChannelsPerSample = pvfh->nChannelsPerSampleInStream;
               nDestUnusedChannelsPerSample = nDestChannelsPerSample - pvfh->nChannelsPerSampleInStream;
            }
           // Copy/convert the uncompressed audio samples for a single SAMPLE POINT (in time).
           // VT_BLOCK supports 1/2/4-byte integers, and 4/8-byte floats - so do we.
           // ALL channels in a stream (or file) use the same data type.
           // [in]  nChannelsPerSample : "number of channels which the caller WANTS TO READ" .
           //       pvfh->nChannelsPerSampleInStream : "what is actually IN THE STREAM" (!)
           iChannel = 0;
           pSrc.pU8 = pvfh->bBuffer + pvfh->dwBufferIndex;
           switch( pvfh->iVTFlags & VTFLAG_FMTMASK )
            {
              case VTFLAG_FLOAT4 : // 4 byte floats (IEEE single precision)
                 DESHOBI(pSrc.u32); // SOURCE address in the internal buffer.
                 while( iChannel<nCommonChannelsPerSample )
                  { *pfltDest++ = pSrc.pF32[iChannel++];
                  }
                 DOBINI(); // survived the FLOAT4 copying loop
                 break;
              case VTFLAG_FLOAT8 : // 8 byte floats (IEEE double precision)
                 DESHOBI(pSrc.u32); // SOURCE address in the internal buffer.
                 if( (pSrc.u32 & 7) == 0)  // properly aligned for 'double' ?
                  { // 2020-10-11: pSrc.u32 was NOT properly aligned
                    //             to 8-byte-boundaries. Fixed by replacing
                    //             malloc() by UTL_NamedMalloc() somewhere.
                    //
                    while( iChannel<nCommonChannelsPerSample )
                     { *pfltDest++ = pSrc.pF64[iChannel++];
                       // Disassembly of the above C instruction:
                       //  mov  edx,[ebp-0x18]         ; -> edx = 0x13BD8050
                       //  fld  qword ptr [edx+eax*8]  ; 1st loop with eax=0, ok, the FPU's "ST0" gets valid, val=-0.01431
                       //  fstp dword ptr [esi]        ; ok, the FPU's "ST0" gets empty again, 8-byte value stored in *esi = 0x13BE97E8
                       //  inc  eax                    ; EAX = 'iChannel', steps by ONE, ok
                       //  add  esi,0x04               ; ESI = pfltDest, steps by FOUR, ok
                       // Tried to INTERCEPT the "floating point overflow"
                       //  - see looong explanation near the END of this file .
                       //
                       // 2020-10-12: Debugger-Exception: "floating point overflow"
                       //  when testing this with DK7FC's tcp://129.206.29.82:49336 ( 8 kSamples/sec)
                       //     but also (less often) with  tcp://129.206.29.82:49336 (48 kSamples/sec).
                       // 2020-10-17: Similar trouble again after stopping
                       //       and almost immediately re-starting the 'fast' stream
                       //       at tcp://129.206.29.82:49335 .  BCB stopped on the
                       //       line above ( *pfltDest++ = pSrc.pF64[iChannel++]; )
                       //       ReadAndDecodeNonCompressed() was NOT listed
                       //       in the 'Call Stack'-display but
                       //  C_VorbisFileIO::ReadSampleBlocks() -> ntdll.dll -> ntdll.dll -> System::NotifyNonDelphiException()
                       //     -> C:\Windows\SysWOW64\KERNELBASE.dll , blah, blubb...
                       //       As usual, none of the local variables could be
                       //       inspected (after an exception). Only the bavarian
                       //       debug macros could confirm we were 'really here',
                       //       with pSrc.u32 = 0x13E30038 (not inspectable but captured in 'DESHOBI()')
                       //       SOUND_AudioFileInput.VorbisIO.m_pVorbisFileHelper = 0x13E30010 (global var, inspectable AFTER an exception)
                       //       ((T_VorbisFileHelper*)SOUND_AudioFileInput.VorbisIO.m_pVorbisFileHelper)->bBuffer = 0x13E30020
                       //     Thus, when the exception occurred, pSrc was VALID,
                       //     and pointed into pvfh->bBuffer with a BYTE-OFFSET
                       //     of 0x13E30038 - 0x13E30010 = 0x28 = 40 ~~ dwBufferIndex=10, but (inspected) :
                       //       ((T_VorbisFileHelper*)SOUND_AudioFileInput.VorbisIO.m_pVorbisFileHelper)->dwBufferIndex = 24 (VT_Block not skipped, WHY ?)
                       //       ((T_VorbisFileHelper*)SOUND_AudioFileInput.VorbisIO.m_pVorbisFileHelper)->dwBufferUsage = 65536
                       //       ((T_VorbisFileHelper*)SOUND_AudioFileInput.VorbisIO.m_pVorbisFileHelper)->i64NumSamplesDecoded = 3
                       //       ((T_VorbisFileHelper*)SOUND_AudioFileInput.VorbisIO.m_pVorbisFileHelper)->iSampleCountdownBetweenHeaders = 8198
                       //       Oops. i64NumSamplesDecoded = 3 ? In that case,
                       //       pvfh->bBuffer[0..47] should still contain the
                       //       initial VT_Block, so let's try to inspect that
                       //       with the Borland debugger and this monster expression:
                       //       *(T_VT_Block*)(((T_VorbisFileHelper*)SOUND_AudioFileInput.VorbisIO.m_pVorbisFileHelper)->bBuffer)
                       //   ->  { 27859, 0,   8192,  1,   48000, 1602950013, 341333333, 1, 8192,    0,   1 }
                       //          |    |      |    |     |             |      |        |   |       |    |
                       //         magic flags bsize chans sample_rate secs    nsec  valid frames spare srcal -> OK !
                       //
                       // Tried to WATCH the following in Borland's debugger:
                       //       pSrc.pF64  =  0x13BD8050 (ok, that's properly aligned)
                       //       pSrc.pF64[0] = -0.014313 (ok, sounds reasonable)
                       //       pSrc.pF64[1] =  0.065706 (ok, ... )
                       //       pSrc.pF64[2] = -0.016541 (ok)
                       //       nSourceBytesPerSample = 8 (ok)
                     }
                    DOBINI(); // survived the FLOAT8 copying loop
                  } // end if < source-address properly aligned to 8-byte boundary >
                 break;
              case VTFLAG_INT1   : // 1 byte signed integers
                 DESHOBI(pSrc.u32); // SOURCE address in the internal buffer.
                 while( iChannel<nCommonChannelsPerSample )
                  { *pfltDest++ = (1.0/256.0) * (float)(pSrc.pI8[iChannel++]);
                  }
                 DOBINI(); // survived the INT1 (8-bit signed integer) conversion loop
                 break;
              case VTFLAG_INT2   : // 2 byte signed integers, little endian
                 DESHOBI(pSrc.u32); // SOURCE address in the internal buffer.
                 while( iChannel<nCommonChannelsPerSample )
                  {
#                  if(1) // (1)=normal compilation, (0)=test with a sinewave signal
                    *pfltDest++ = (1.0/32768.0) * (float)(pSrc.pI16[iChannel++]);
#                  else  // test: produce a 'coherent' sine wave with an 11-sample period
                    *pfltDest++ = sin( (double)(nTotalSampleCounter % 11) * (2.0 * 3.1415926 / 11.0) );
                    ++iChannel;
                    // Test result (2014-04-25) : Perfect pure tone, no phase jumps or lost samples.
#                  endif
                  }
                 DOBINI(); // survived the INT2 (16-bit signed integer) conversion loop
                 break;
              case VTFLAG_INT4   : // 4 byte signed integers, little endian
                 DESHOBI(pSrc.u32); // SOURCE address in the internal buffer.
                 while( iChannel<nCommonChannelsPerSample )
                  { *pfltDest++ = (1.0/2147483648.0) * (float)(pSrc.pI32[iChannel++]);
                  }
                 DOBINI(); // survived the INT4 (32-bit signed integer) conversion loop
                 break;
              default:
                 DOBINI();  // oops.. wosbini ? woarum ? desderfnetsei !
                 break;

            } // end switch < data type flags from the VT_BLOCK header >

           // If the DESTINATION has more channels than contained in the stream,
           // fill those unused channels with 'silence' :
           while( iChannel<nDestChannelsPerSample )
            { *pfltDest++ = 0.0;
              ++iChannel;
            }
           if( nCommonChannelsPerSample > 0 )
            { fAnythingHappenedInThisLoop = TRUE; // something happened: processed a SAMPLE POINT, and drained it from the buffer
            }
           ++nSamplePointsRead; // 'sample point' = a group of channels sampled AT THE SAME TIME.
           ++nTotalSampleCounter;
           if(  pvfh->iSampleCountdownBetweenHeaders > 0 )
            { --pvfh->iSampleCountdownBetweenHeaders; // ...3,2,1,0 -> "expect the next VT_BLOCK or similar" !
            }
           pvfh->dwBufferIndex += nSourceBytesPerSample; // index into pvfh->bBuffer[] (head)
           nBytesToProcess = pvfh->dwBufferUsage - pvfh->dwBufferIndex; // update the REMAINING number of bytes to process
           pvfh->i64NumTotalBytesProcessed += nSourceBytesPerSample;
           ++pvfh->i64NumSamplesDecoded;  // keep track of the TOTAL number of samples received, also used in ProcessHeaderFromNonCompressedStream() to check timestamps for plausibility
           // When processing the FIRST sample (after a 64-byte header),
           //      pvfh->dwBufferIndex was incremented from 64 to 66 here.
         } // end if( fProcessSamplesNow )
        else // ! fProcessSamplesNow ->
         { break; // break from the sample-processing-loop
         }
      } // end if/while( (nBytesToProcess >= nSourceBytesPerSample ) && ... )

     if( nSamplePointsRead >= nDestMaxSamplePoints )
      { break; // got enough sample points read for THIS call of ReadAndDecodeNonCompressed() now -> break from loop
        // (must break out of the loop AFTER checking for headers, because
        //  ReadAndDecodeNonCompressed() may have been called with nDestMaxSamplePoints = 0
        //  to read the HEADER ONLY !    The same applies to FILES as well as STREAMS .
      }
     if( !fAnythingHappenedInThisLoop ) // "nothing" really happened in this loop
      { break; // "see you later", in the next call of ReadAndDecodeNonCompressed()
      }

     if( TIM_ReadStopwatch_msec( &myStopwatch ) > pvfh->inetTimeout_ms )
      { break; // chunks of data only seem to 'dripple' into VorbisRead() .. give up IN THIS CALL
      }
   } // end while(1)

  if( piNumMillisecondsWaited != NULL ) // optional output : number of milliseconds spent WAITING for input, accumulated
   { *piNumMillisecondsWaited += nNumMillisecondsWaited;
   }

  if( (nSamplePointsRead>0) && ( (pvfh->nSuccessfullCallsOfReadAndDecode<5) || (i64PrevNumSamplesDecoded==0) ) )
   { ++pvfh->nSuccessfullCallsOfReadAndDecode;
     CallBack_Info( pvfh, "ReadAndDecode #%d: %ld samples, %ld timestamp(s), flags:%s, %ld bytes read in THIS call, took %ld ms, %d ms spent WAITING for input .",
                     (int)pvfh->nSuccessfullCallsOfReadAndDecode,
                    (long)nSamplePointsRead,
                    (long)pvfh->i64TimestampCounter,
                   (char*)VFIO_VTFlagsToString(pvfh->iVTFlags),
                    (long)nBytesReadInThisCallOfReadAndDecode,
                    (long)TIM_ReadStopwatch_msec( &myStopwatch ),
                     (int)nNumMillisecondsWaited );
     // Test result from the above, when compiled with L_DRAIN_NETWORK_BUFFER = 0 :
     // 21:04:55.3 Waiting for samples from tcp://f6etu.duckdns.org:4420 .
     // 21:04:52.2 Trying to connect host f6etu.duckdns.org
     // 21:04:52.2 Connected to f6etu.duckdns.org within 72 ms .
     // 21:04:52.2 OpenAndRead("tcp://f6etu.duckdns.org:4420") successful
     // 21:04:55.3 Received first 6160 bytes after 3007 ms .
     // 21:04:55.3 Format=VLF-RX, dtype=float64, chns=1, SR=250 .
     // 21:04:59.8 ReadAndDecode #1: 64 samples, 1 timestamp(s), flags:float64, 9520 bytes read in THIS call, took 3011 ms .
     // 21:04:59.8 Time local,remote: 21:04:59.8 21:04:44.9
     // 21:04:59.9 ReadAndDecode #2: 64 samples, 2 timestamp(s), flags:float64, 560 bytes read in THIS call, took 0 ms .
     // 21:04:59.9 Time local,remote: 21:04:59.9 21:04:45.1
     // 21:05:00.1 ReadAndDecode #3: 64 samples, 3 timestamp(s), flags:float64, 560 bytes read in THIS call, took 0 ms .
     // 21:05:00.1 Time local,remote: 21:05:00.1 21:04:45.4
     // 21:05:00.4 ReadAndDecode #4: 64 samples, 4 timestamp(s), flags:float64, 560 bytes read in THIS call, took 0 ms .
     // 21:05:00.4 Time local,remote: 21:05:00.4 21:04:45.6
     // 21:05:00.6 ReadAndDecode #5: 64 samples, 5 timestamp(s), flags:float64, 560 bytes read in THIS call, took 0 ms .
     // 21:05:00.6 Time local,remote: 21:05:00.6 21:04:45.9
     // Test with the same stream, but compiled with L_DRAIN_NETWORK_BUFFER = 1 :
     // 21:40:24.9 Trying to detect format of 'tcp://f6etu.duckdns.org:4420'...
     // 21:40:26.1 Trying to connect host f6etu.duckdns.org
     // 21:40:26.2 Connected to f6etu.duckdns.org within 71 ms .
     // 21:40:26.2 OpenAndRead("tcp://f6etu.duckdns.org:4420") successful
     // 21:40:26.6 Received first 560 bytes after 298 ms .
     // 21:40:26.6 Closing socket connection after analysing the format
     // 21:40:26.6 Analysing first 560 bytes from remote server ..
     //   Format seems to be 'audio/vlfrx', dtype=float64, chns=1, SR=250 .
     // 21:40:26.6 Opening stream, without logfile .
     //  250.000000000 samples/s
     // 21:43:22.6 Waiting for samples from tcp://f6etu.duckdns.org:4420 .
     // 21:43:19.5 Trying to connect host f6etu.duckdns.org
     // 21:43:19.6 Connected to f6etu.duckdns.org within 69 ms .
     // 21:43:19.6 OpenAndRead("tcp://f6etu.duckdns.org:4420") successful
     // 21:43:22.6 Received first 5600 bytes after 3008 ms .
     // 21:43:22.6 Format=VLF-RX, dtype=float64, chns=1, SR=250 .
     // 21:43:26.1 ReadAndDecode #1: 64 samples, 1 timestamp(s), flags:float64, 8400 bytes read in THIS call, took 3012 ms, 3012 ms spent WAITING for input .
     // 21:43:26.1 Time local,remote: 21:43:26.1 21:43:15.5
     // 21:43:26.4 ReadAndDecode #2: 64 samples, 2 timestamp(s), flags:float64, 0 bytes read in THIS call, took 301 ms, 301 ms spent WAITING for input .
     // 21:43:26.4 Time local,remote: 21:43:26.4 21:43:15.8
     // 21:43:26.9 ReadAndDecode #3: 128 samples, 4 timestamp(s), flags:float64, 1120 bytes read in THIS call, took 351 ms, 351 ms spent WAITING for input .
     // 21:43:26.9 Time local,remote: 21:43:26.9 21:43:16.3
     // 21:43:27.4 ReadAndDecode #4: 192 samples, 7 timestamp(s), flags:float64, 1120 bytes read in THIS call, took 498 ms, 497 ms spent WAITING for input .
     //  ,-------------------------------------------------------------------------------------------------------------------'
     //  '--> that's MUCH TOO LONG, no suprise to have 'stuttering audio' !
     //       Added AUDIO_FILE_OPTION_NON_BLOCKING_CALLS to find if that made
     //       a significant difference (2022-03-21).
     //
     // 21:43:27.4 Time local,remote: 21:43:27.4 21:43:17.1
     // 21:43:27.9 ReadAndDecode #5: 128 samples, 9 timestamp(s), flags:float64, 1120 bytes read in THIS call, took 339 ms, 339 ms spent WAITING for input .
     // 21:43:27.9 Time local,remote: 21:43:27.9 21:43:17.6
     if( pvfh->i64TimestampCounter > 0 )
      { // Show the first RECEIVED timestamps versus LOCAL time, to debug the problem
        // diagnosed in C:\cbproj\SpecLab\bugfixes_and_notes\renato_problem_with_timestamped_VLF_streams.txt :
        cp = sz511;
        strcpy(cp,"Time local,remote: ");
        SL_SkipToEndOfString( &cp );
        UTL_FormatDateAndTime("hh:mm:ss.s ", TIM_GetCurrentUnixDateAndTime(), cp );
        SL_SkipToEndOfString( &cp );
        UTL_FormatDateAndTime("hh:mm:ss.s ", pvfh->ldblPrevTstampUnixTime, cp );
        CallBack_Info( pvfh, sz511 );
        // Result seen when testing tcp://f6etu.duckdns.org:4420 :
        // > 22:30:27.3 Time local,remote: 22:30:27.3 22:30:08.7
        //    ,----------------------------|___________________|
        //    '--> Sounds plausible. The first timestamp was
        //         18.6 seconds old when it arrived from the
        //         Lot Et Garonne stream (F6ETU) .
      } // end if( pvfh->i64TimestampCounter > 0 )
   } // end if < one of the first successful calls WITH SAMPLES RETRIEVED >
  if( nSamplePointsRead != nDestMaxSamplePoints )
   { nSamplePointsRead = nSamplePointsRead;  // << place for a 'conditional breakpoint'
                // (should only get here at the end of a file, and NEVER with a stream)
   }
  return nSamplePointsRead;  // .. number of sample-points read in THIS call
  // (may be ZERO when called from C_VorbisFileIO::InOpenInternal() to detect the STREAM FORMAT)
} // end ReadAndDecodeNonCompressed()



extern "C" void vorbis_registry_dummy(void);

/*------------ Implementation of the class C_VorbisFileIO -----------------*/
// (no templates, no inheriting, no dynamic whatsoever, just a SIMPLE CLASS)

/***************************************************************************/
C_VorbisFileIO::C_VorbisFileIO()
  /* Constructor of the Wave-IO Class.
   * Initializes all properties etc.
   */
{
   // To avoid having to include the Ogg- and Vorbis header files
   // in VorbisFileIO.H, we use this extremely ugly KLUDGE :
   m_pVorbisFileHelper = NULL; // haven't reserved on of these 'helpers' yet

   m_cErrorString[0]  = '\0';

   m_dbl_SampleRate = 11025.0; // initial guess, only important for RAW files
   m_iSizeOfDataType = 1;      // just a GUESS for reading RAW files,
   m_nChannelsPerSampleInFile = 1;  // should be properly set before opening such files
   m_dwCurrentSampleIndex  = 0;
   m_dwFileLengthInSamples = 0;
   m_dwCurrFilePos = m_dwFileSizeInBytes = 0;

} // end of C_VorbisFileIO's constructor

/***************************************************************************/
C_VorbisFileIO::~C_VorbisFileIO()
  /* Destructor of the Wave-IO Class.
   *  - Flushes all output files of this class (if any)
   *  - Closes all handles (if still open)
   *  - Frees all other resources that have been allocated.
   *
   * Note: To avoid the following in the 'debug run log' ...
   *   > 21:38:28.7 WARNING: Application didn't free 1 memory block  when quitting.
   *   > 21:38:28.7 WARNING: Name of 1st non-freed block = "VorbisIO" .
   *  ... call C_VorbisFileIO::Exit() **before** UTL_ExitFunction() .
   */
{
   if( m_pVorbisFileHelper != NULL )
    { Exit();  // <- method, also "user-callable", to avoid mis-diagnosed memory leaks
      // Note: The C runtime seems to call C++ destructors *AFTER*
      //       the old-fashioned 'atexit'-handler (here: UTL_ExitFunction() ),
      //       so when trying to UTL_free() something in the destructor,
      //       UTL_fInitialized has already been cleared by UTL_ExitFunction(),
      //       thus UTL_free() will not even call the original free() anymore !
    }
} // end C_VorbisFileIO's destructor

/***************************************************************************/
void C_VorbisFileIO::Exit(void)
  /* "Semi-Destructor" of the Wave-IO Class.
   * Can be called BY THE APPLICATION to prevent (false) warnings
   * about non-freed memory blocks - see C_VorbisFileIO::~C_VorbisFileIO() .
   *  - Flushes all output files of this class (if any)
   *  - Closes all handles (if still open)
   *  - Frees all other resources that have been allocated.
   */
{
   if( m_pVorbisFileHelper != NULL )
    {
      // free resources, and give one of these 'helpers' back to the pool :
      VorbisFileHelper_Free( (tVorbisFileHelper*)m_pVorbisFileHelper );
      m_pVorbisFileHelper = NULL;  // not in use anymore !
    }
} // end C_VorbisFileIO::Exit()


/***************************************************************************/
BOOL C_VorbisFileIO::SetInfoCallback(
     T_AudioFileInfoCallback pInfoCallback, // [in, optional] user callback function
     void *pvUserCallbackData)              // [in, optional] user callback data
  // Only used for debugging, and for display in SL's web stream panel.
  // Type and parameters of the user's "info callback" is defined
  //      in c:\cbproj\SoundUtl\AudioFileDefs.h .
  // Used, for example, in C:\cbproj\SpecLab\AudioStreamServer_GUI.cpp
  //      to display certain audio-stream-related events on a control panel.
{
 T_VorbisFileHelper *pvfh; // contains everything which shouldn't go into VorbisFileIO.h
  if( m_pVorbisFileHelper == NULL )
   { // allocate one of these 'helpers' from a pool (in fact a simple array):
     m_pVorbisFileHelper = VorbisFileHelper_Alloc( this );
   }
  pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { pvfh->pInfoCallback      = pInfoCallback;
     pvfh->pvUserCallbackData = pvUserCallbackData;
     return TRUE;
   }
  return FALSE;

} // end C_VorbisFileIO::SetInfoCallback()


/***************************************************************************/
BOOL C_VorbisFileIO::SetStreamLogfile( char *pszLogfileName )
  // Prepares writing an optional logfile with compressed audio data.
  //  [in] : pszLogfileName = filename of an optional (audio-)logfile,
  //                          or NULL when *no* logfile shall be written.
  //
  // This method must be called BEFORE opening a webstream,
  // because otherwise we'd miss the signature ("OggS") in the log,
  // and the result wouldn't be a valid Ogg/Vorbis file anymore !
{
  T_VorbisFileHelper *pvfh; // contains everything which shouldn't go into VorbisFileIO.h
  if( m_pVorbisFileHelper == NULL )
   { // allocate one of these 'helpers' from a pool (in fact a simple array):
     m_pVorbisFileHelper = VorbisFileHelper_Alloc( this );
   }
  pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { if( pszLogfileName == NULL )
      { pvfh->fWriteLogfile = FALSE;
        if( pvfh->qfStreamLog.fOpened )
         { QFile_Close( &pvfh->qfStreamLog );
         }
      }
     else
      { pvfh->fWriteLogfile = TRUE;
        strncpy( pvfh->sz255StreamLogfile, pszLogfileName, 255 );
        // Note: We will NOT write (create) a new Ogg Vorbis logfile here yet !
        //  (that follows later, AFTER successfully connecting to a remote server
        //   in C_VorbisFileIO::InOpenInternal() )
      }
     return TRUE;
   }
  return FALSE;
} // end C_VorbisFileIO::SetStreamLogfile()

/***************************************************************************/
BOOL C_VorbisFileIO::CloseLogfile( void )
  // Closes the *LOGFILE*, without closing the *STREAM* itself.
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh != NULL )
   {
   }
  return FALSE;
} // end C_VorbisFileIO::CloseLogfile()


/***************************************************************************/
BOOL C_VorbisFileIO::GetLogfileInfo(
       T_StreamLogInfo *pStreamLogInfo) // [out] 'audio stream log info', defined in c:\cbproj\SoundUtl\AudioFileDefs.h
  // Retrieves info about the current STREAM LOGFILE (not the stream itself).
  // Returns TRUE if the info in the T_StreamLogInfo is *valid*,
  //         which doesn't mean the stream is CURRENTLY BEING LOGGED !
  // ( check flag T_StreamLogInfo.iLogging, as specified in AudioFileDefs.h )
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( (pvfh!=NULL) && (pStreamLogInfo!= NULL) )
   { memset( pStreamLogInfo, 0, sizeof(T_StreamLogInfo) );  // <- this also terminates clipped STRING !
     pStreamLogInfo->iLogging = pvfh->qfStreamLog.fOpened;
     // In THIS case (Vorbis), the LOG FILE FORMAT should be the same as the STREAM FORMAT:
     pStreamLogInfo->iLogFileFormat = GetParameterByID( AUDIO_FILE_PARAM__FILE_FORMAT );
     pStreamLogInfo->iLogFileSize_kByte = QFile_Seek(  &pvfh->qfStreamLog, 0, SEEK_CUR ) / 1024;
     strncpy(pStreamLogInfo->sz255LogFileName, pvfh->qfStreamLog.sz512PathAndName, 255 );
     strncpy(pStreamLogInfo->sz255StreamURL,   m_sz255FileName, 255 );
     return TRUE;
   }
  return FALSE;

} // end C_VorbisFileIO::GetLogfileInfo()


/************************************************************************/
BOOL C_VorbisFileIO::SetServerPassword(char *pszPassword)
  /* Sets the password for certain web streams; call BEFORE opening a stream.
   */
{
 T_VorbisFileHelper *pvfh; // contains everything which shouldn't go into VorbisFileIO.h
  if( m_pVorbisFileHelper == NULL )
   { // allocate one of these 'helpers' from a pool (in fact a simple array):
     m_pVorbisFileHelper = VorbisFileHelper_Alloc(this);
   }
  pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { if( pszPassword != NULL )
          strncpy(pvfh->sz80ServerPassword, pszPassword, 80 );
     else pvfh->sz80ServerPassword[0] = '\0';
     return TRUE;
   }
  return FALSE;
} // end C_VorbisFileIO::SetServerPassword()


/***************************************************************************/
BOOL C_VorbisFileIO::InOpenInternal( char * pszFilename,
                                     int options, // AUDIO_FILE_OPTION_TIMESTAMPS, AUDIO_FILE_OPTION_NON_BLOCKING_CALLS, etc [-> AudioFileDefs.h]
                                     BOOL fReOpenForWriting )
  /* Opens a file (or stream) for READING, or opens an existing file for WRITING / APPENDING .
   * Returns TRUE on success and FALSE on any error .
   * Notes:
   *   - if a NON-COMPRESSED file or stream shall be opened (not an Ogg/Vorbis compressed one),
   *     use  SetParameterByID( AUDIO_FILE_PARAM__FILE_FORMAT, STREAM_FORMAT_NON_COMPRESSED ) *before* trying to open,
   *     because neither the stream address ("rawtcp://..")
   *     nor the file extension ("*.dat") can provide that important info !
   */
{
 struct ftime f_time;
 char *cp;
 T_VorbisFileHelper *pvfh; // contains everything which shouldn't go into VorbisFileIO.h
 int  ov_result; // vorbis result code (when negative, it's an error code)
 int  n, nMillisecondsWaited=0;
 BOOL fExtensionIsOgg;
 BOOL fSubtractRecLengthFromTime = FALSE;
 BOOL ok = TRUE;

  CloseFile( "to re-open"/*pszWhy*/ );   // Close "old" file if it was open
  m_EndOfOutput = FALSE;  // "end of output" (from ov_read) NOT reached yet
  m_dwCurrentSampleIndex  = 0;
  m_dwFileLengthInSamples = 0;
  m_cErrorString[0] = 0;  // forget the old error string

  // Look at the file extension..
  cp = strrchr(pszFilename, '.');
  if(cp)
       fExtensionIsOgg = (strnicmp(cp,".ogg",4)==0);
  else fExtensionIsOgg = FALSE;
  // ex: m_iFileFormat = AUDIO_FILE_FORMAT_UNKNOWN;  // not sure whether RAW, RIFF, or SETI ...

  // removed from here: m_dbl_SampleRate = 0.0;  // still unknown (!)
  if( m_pVorbisFileHelper == NULL )
   { // allocate one of these 'helpers' from a pool (in fact a simple array):
     m_pVorbisFileHelper = VorbisFileHelper_Alloc(this);
   }
  pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh==NULL ) // bad luck; all 'vorbis file helpers' already in use ?!
   {
     strcpy(m_cErrorString,"Could not allocate vorbis 'file helper'");
     return FALSE;
   }
  pvfh->inetReadOptions = INET_RD_OPTION_WAIT_FOR_ALL_DATA; // for the *INITIAL CALL* of VorbisRead() : don't return until the requested amount of bytes was received
  pvfh->inetTimeout_ms  = AUDIO_FILE_IO_MAX_TIMEOUT_MS; // initial timeout value for VorbisRead() when used with 'internet streams'
                        // (subsequent calls of VorbisRead() may use different timeout values,
                        //  or not wait for anything at all.
                        //  See AUDIO_FILE_OPTION_NON_BLOCKING_CALLS in ReadAndDecodePages() for details.)
  pvfh->fConnecting     = FALSE;
  pvfh->fOpenedForReading = FALSE;
  pvfh->fOpenedForWriting = FALSE;
  pvfh->fPlainOggVorbis = FALSE;  // PFLAG from Paul's vtvorbis
  pvfh->fMustClearVorbisInfo = FALSE;  // not yet.. flag polled in CloseFile()
  pvfh->fMustClearDSP   = FALSE;  // safe to do here because we called CloseFile() !
  pvfh->fMustClearBlock = FALSE;
  pvfh->fMustClearComment=FALSE;
  pvfh->iSampleCountdownBetweenHeaders = 0; // here in InOpen() : expect the initial header or VT_BLOCK a.s.a.p. !
  pvfh->fGotStartTimeFromTimestamp = FALSE; // prepare an awful lot of 'flags'..
  pvfh->have_vorbis = FALSE;
  pvfh->want_tstamp = FALSE;
  pvfh->have_tstamp = FALSE;
  pvfh->fGotAllHeaders = FALSE;
  pvfh->fStreamFormatDetectedFromData = FALSE; // not sure about iStreamFormat yet
  pvfh->fMissingVTBlock = FALSE;
  pvfh->dwBufferUsage = pvfh->dwBufferIndex = 0; // make bBuffer[] invalid
  pvfh->nDecoderOutputBufferUsage = pvfh->nDecoderOutputBufferPeakUsage = pvfh->iDecoderOutputReadIndex = 0; // make fltDecoderOutput[] invalid
  pvfh->iTimestampQueueLength   = 0;
  pvfh->i64TimestampCounter     = 0;
  pvfh->i64NumSamplesDecoded    = 0;
  pvfh->i64NumSamplesReadByApp  = 0;
  pvfh->i64PrevTstampSamplePosn = 0;
  pvfh->ldblPrevTstampUnixTime  = 0.0;

  pvfh->dblStartTime            = 0.0;

  pvfh->nRcvdVorbisHeaders      = 0;

  pvfh->vt_packetno             = 0;

  pvfh->nSuccessfullCallsOfReadAndDecode = 0;

  pvfh->iAudioFileOptions = options; // bitwise combineable flags like AUDIO_FILE_OPTION_NON_BLOCKING_CALLS, etc (see AudioFileDefs.h)

  if( options & AUDIO_FILE_OPTION_TIMESTAMPS )
   {  pvfh->want_tstamp = TRUE;  // WANT timestamps, but haven't got a valid one for the encoder yet !
   }

  pvfh->ndemux = 0;

  for(n=0;n<MAX_DEMUX;++n)

   { pvfh->demux[n].serial = -1;  // tStreamDemux.serial=-1 marks an UNUSED entry
     pvfh->demux[n].activity_detector = 0; // added 2013-07-05 to recycle the OLDEST (orphaned) entry
   }

  m_dwFileLengthInSamples = 1024;   // guesswork / totally useless for STREAMS
  m_ldblPrevCheckedTimestamp = 0.0; // only used for DEBUGGING (plausibility check)
  pvfh->dblTimestampLatency = C_SND_MATH_INVALID_FLOAT_VALUE; // latency between "local" and "remote" TIMESTAMP hasn't been measured yet


  if( fReOpenForWriting )
   {
#    if (VORBIS_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
      ok = QFile_Open(&pvfh->m_QFile, pszFilename, QFILE_O_RDWR );
#    else
      pvfh->m_FileHandle = _rtl_open( pszFilename, O_RDWR );
      ok = (pvfh->m_FileHandle>0);
#    endif  // ! VORBIS_USE_QFILE
   }
  else  // open for READING (only) :
   {
     // Let YHF_Inet.c find out if this is an INTERNET STREAM or a LOCAL FILE:
     if( INET_IsInternetResource( pszFilename ) )
      { // It's an internet stream, not a local file,
        //      for example http://67.207.143.181:80/vlf1  --->
        pvfh->fIsInternetStream = TRUE;
        pvfh->fStoreSamplesForMultipleReaders = FALSE; // here in InOpen..() : only supports ONE (not MULTIPLE) readers !
        pvfh->fConnecting = TRUE;
        n = INET_OpenAndReadFromServer(
             &pvfh->InetClient, // [in,out] T_INET_Client *pInetClient; web client data
              pszFilename,  // [in] filename or URL
              AUDIO_FILE_IO_MAX_TIMEOUT_MS, // [in] max. timeout in milliseconds
              INET_RD_OPTION_RETURN_AFTER_RESPONSE, // [in] options (from YHF_Inet.h)
              NULL, 0 );    // [in,out] destination buffer for the "contents" (don't drain any RECEIVED DATA yet)
        if( n < 0 )
         { ok = FALSE;
           snprintf( m_cErrorString, sizeof(m_cErrorString)-1, "Couldn't open URL: %s", ErrorCodeToString(-n) );
           // Details / reason already reported in YHF_Inet.c . No need for CallBack_Error() here.
         }
        else
         { ok = TRUE;  // successfully opened the connection and read the 'response headers'
                       // (but not the "contents" yet)
           pvfh->fOpenedForReading = TRUE;  // here: internet stream (not a LOCAL FILE, not seekable)
           pvfh->dblStartTime = TIM_GetCurrentUTC(); // hopefully replaced with a timestamp FROM THE REMOTE SERVER soon
           // Write an 'audio log' with the compressed audio (similar to Oddcast's log) ?
           if( pvfh->fWriteLogfile && pvfh->sz255StreamLogfile[0]!=0 )
            { if( ! QFile_Create( &pvfh->qfStreamLog, pvfh->sz255StreamLogfile, 0 ) )
               { strcpy( m_cErrorString, "Couldn't create stream log" );
                 CallBack_Error( pvfh, 0, m_cErrorString );
                 ok = FALSE;
               }
            }
         }
        pvfh->fConnecting = FALSE;
      }
     else
      { // it's not an (internet-) stream but a LOCAL FILE :
#    if (VORBIS_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
        ok = QFile_Open(&pvfh->m_QFile, pszFilename, QFILE_O_RDONLY );
#    else
        pvfh->m_FileHandle = _rtl_open( pszFilename, O_RDONLY);
        ok = (pvfh->m_FileHandle>0);
#    endif  // ! VORBIS_USE_QFILE

        if( ok )   // ok, the (disk-)file has been opened.
         { pvfh->fOpenedForReading = TRUE;  // here: LOCAL FILE (not internet stream)
           // Get the "file time" as a first coarse value when the file has been
           // recorded. Note that "getftime" returns its own time structure
           // which is very different from our internal "time()"-like format.
           // "getftime" returns the date relative to 1980,
           // but SpecLab uses the Unix-style format which is
           //    "Seconds elapsed since 00:00:00 GMT(=UTC), January 1, 1970"
           if( getftime(
#           if (VORBIS_USE_QFILE)
             pvfh->m_QFile.iHandle,
#           else
             pvfh->m_FileHandle,
#           endif
             &f_time)==0) // "0" = "no error"
            {
              pvfh->dblStartTime = UTL_ConvertYMDhmsToUnix(
                 f_time.ft_year+1980,  // int year  (FULL year, like 2002, not crap like "year-1980",
                 f_time.ft_month,      // int month (1..12)
                 f_time.ft_day,        // int day   (1..31)
                 f_time.ft_hour,    // int hour  (0..23) !! not the nerving "AM PM" stuff please... :-)
                 f_time.ft_min,     // int minute (0..59)
                 2*f_time.ft_tsec); // T_Float dblSecond  (0.000...59.999)
              //  If there's a timestamp in the Ogg stream,
              //  pvfh->dblStartTime will be overwritten soon !
            }
           else
            { pvfh->dblStartTime=0; // haven't got a clue about the "start time" (yet)
            }
         } // end if < LOCAL file has been opened >
      } // end if < STREAM or local FILE > ?
     if( ok )  // successfully opened the STREAM or FILE, but that doesn't mean a lot:
      { if((pvfh->iStreamFormat == STREAM_FORMAT_NON_COMPRESSED ) // added 2014-04-21 ...
        || (pvfh->iStreamFormat == STREAM_FORMAT_VLF_RX_BLOCKS) ) // added 2020-10-12 ...
         { // The audio stream turned out to be NON COMPRESSED, but we don't know much more yet.
           // The structure should be defined the structs
           // defined in VorbisFileIO.h : T_StreamHeader, T_VT_Block, combined as T_StreamHeader1.
           ov_result = ReadAndDecodeNonCompressed( pvfh,  // <- initial call from C_VorbisFileIO::InOpenInternal()
                  0/*don't need samples yet, only the header, leave all samples IN THE BUFFER*/,
                  0/*chn*/, NULL/*dest*/, &nMillisecondsWaited );
           // Under ideal conditions, pvfh->fStreamFormatDetectedFromData is TRUE now,
           // and whatever is in a T_VT_Block (VLF-RX-tools: VT_BLOCK) has been
           // extracted by ReadAndDecodeNonCompressed() ->
           if( ov_result >= 0 ) // zero or more samples read, and NO ERROR ...
            { m_dbl_SampleRate = pvfh->dblNominalSampleRate;
              m_nChannelsPerSampleInFile = pvfh->nChannelsPerSampleInStream;
              // Show what we just 'decoded' from the first VT_BLOCK or T_StreamHeader
              // on the 'Analyse & play stream' - control panel :
              CallBack_Info( pvfh, "Format=%s, dtype=%s, chns=%d, SR=%ld .",
                    StreamFormatToString(pvfh->iStreamFormat),
                    AudioSampleDataTypeToString(pvfh->pThis->m_iSizeOfDataType,
                                                pvfh->pThis->m_iDataTypeFlags),
                     (int)pvfh->nChannelsPerSampleInStream,
                    (long)pvfh->dblNominalSampleRate );
            }
           else // if( ov_result < 0) : ERROR
            {
              snprintf( m_cErrorString, sizeof(m_cErrorString)-1, "OV error: %s",
                    (char*)VorbisErrorCodeToString( ov_result ) );
              CallBack_Error( pvfh, 0, m_cErrorString );
              ok = FALSE;   // error; file will be closed before returning
            }
         }
        else // not UNCOMPRESSED but COMPRESSED, i.e. "real Ogg/Vorbis",
         {   // or an attempt(!) to find out the true type in ReadAndDecodePages() :
           //
           // ex: Let vorbisfile.c process the initial headers, etc....
           //     formerly using ov_open_callbacks() .
           // Replaced 2011-12-14 with direct use of the Libvorbis API, because:
           // > Libvorbis is the reference implementation of the Vorbis codec.
           // > It is the lowest-level interface to the Vorbis encoder and decoder,
           // > working with packets directly.
           //
           // The following is loosely based on Paul Nicholson's "vtvorbis.c"
           // but here (under the hood of windoze) we cannot use stdin as the
           // source, thus a bit more red tape was required.
           //
           // Another, nicely explained, example for an Ogg/Vorbis decoder
           // using libogg was found at
           //  www.bluishcoder.co.nz/2009/06/24/reading-ogg-files-using-libogg.html
           //
           // The first thing we need to do when reading an Ogg file is find
           // the first page of data. We use an ogg_sync_state structure
           // to keep track of search for the page data. This needs to be
           // initialized with ogg_sync_init and later cleaned up with ogg_sync_clear:
           if(ogg_sync_init( &pvfh->oy )!=0) // init our instance of 'ogg_sync_state'
            { strcpy( m_cErrorString, "error in ogg_sync_init" );
              CallBack_Error( pvfh, 0, m_cErrorString );
              CloseFile( m_cErrorString/*pszWhy*/ );
              return FALSE;
            }
           ov_result = ReadAndDecodePages( pvfh, FALSE/*don't need samples yet*/, &nMillisecondsWaited );
           if( ov_result < 0)
            {
              snprintf( m_cErrorString, sizeof(m_cErrorString)-1, "OV error: %s",
                    (char*)VorbisErrorCodeToString( ov_result ) );
              CallBack_Error( pvfh, 0, m_cErrorString );
              ok = FALSE;   // error; file will be closed before returning
            }
           if( pvfh->fGotAllHeaders )   // Successfully read/received all required Vorbis headers ?
            {
              m_dbl_SampleRate = pvfh->dblNominalSampleRate;
              m_nChannelsPerSampleInFile = pvfh->nChannelsPerSampleInStream;
            }
           else
            { strcpy( m_cErrorString, "OV error: missing headers" );
              CallBack_Error( pvfh, 0, m_cErrorString );
              ok = FALSE;   // error; file will be closed before returning
            }
         } // end else < "real" Ogg/Vorbis, not an 'uncompressed' stream >
      } // end if < disk file or stream opened successfully >
     else
      { if( m_cErrorString[0]==0 )
         { snprintf( m_cErrorString, sizeof(m_cErrorString)-1,
                     "Couldn't open '%s' for reading", pszFilename );
           CallBack_Error( pvfh, 0, m_cErrorString );
         }
      }
   }  // end else <open the file for READING >
  if(ok)
   {
    strncpy(m_sz255FileName, pszFilename, 255); // here: in InOpenInternal() ..
   } // end if <file opened successfully>
  else
   { if( m_cErrorString[0] == 0 )
      { strcpy(m_cErrorString,"_open failed");
      }
     // ex: ok = FALSE;
     return FALSE;
   }

  if (!ok)
   {
    if(m_cErrorString[0]=='\0')
     { strcpy(m_cErrorString,"other 'open' error");
     }
    CloseFile(m_cErrorString);  // close input file if opened but bad header
    return FALSE;
   }
  else
   {
    return TRUE;
   }
} // C_VorbisFileIO::InOpenInternal(..)

/***************************************************************************/
BOOL C_VorbisFileIO::InOpen(
                char *pszFilename, // [in] filename, URL, pseudo-URL, or COM port name
                int  options )  // AUDIO_FILE_OPTION_TIMESTAMPS , ... (bitflags)
  /* Opens a wave file (or an internet audio stream, or serial port) for READING.
   * Returns TRUE on success and FALSE on any error .
   */
{
  return InOpenInternal( pszFilename, options, FALSE/*!fReOpenForWriting*/ );
} // C_VorbisFileIO::InOpen(..)


/************************************************************************/
BOOL C_VorbisFileIO::IsOpenedForReading(void)
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh != NULL )
   { return pvfh->fOpenedForReading; // applies to files AND streams !
   }
  return FALSE;
}

/************************************************************************/
BOOL C_VorbisFileIO::IsOpenedForWriting(void)
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh )
   { return pvfh->fOpenedForWriting && (!pvfh->InetClient.fConnectionLost);
   }
  return FALSE;
  // 2016-05-18 : returned FALSE when the audio-stream-output was 'passive',
  //              caller = SoundThd_WaveSaveProcess() ,
  //              because the 'vorbis file helper' wasn't allocated .
}

/************************************************************************/
BOOL C_VorbisFileIO::IsStreamButNoFile(void) // -> TRUE=stream, FALSE=local file
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh )
   { return pvfh->fIsInternetStream;
   }
  return FALSE;
} // end C_VorbisFileIO::IsStreamButNoFile()


/************************************************************************/
long C_VorbisFileIO::GetParameterByID( int iAudioFileParam )
  // iAudioFileParam: one of the AUDIO_FILE_PARAM_..values from AudioFileDefs.h .
  // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
  // for example GetVTFlags() -> GetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS) .
{
  T_VorbisFileHelper *pvfh;
  if( m_pVorbisFileHelper == NULL )
   { // allocate one of these 'helpers' from a pool (in fact a simple array):
     m_pVorbisFileHelper = VorbisFileHelper_Alloc( this );
   }
  pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
      {
        case AUDIO_FILE_PARAM__FILE_FORMAT :
           // Replacement for   int GetFileFormat(void) .
           //  The parameter (iAudioFileFormat) must be one of the
           //  AUDIO_FILE_FORMAT_...-constants defined in AudioFileDefs.
           return GetFileFormat();

        case AUDIO_FILE_PARAM__FILE_SIZE_KBYTE :
           // Returns the current file size (if there is an opened file) in KBYTE.
           // Useful especially when logging audio samples  to keep track of the used disk space.
           // Returns a negative value on any error.
           // ex: if (m_OpenedForWriting || m_OpenedForReading) .. modified 2016, we want to know the filesize AFTER CLOSING !
           return m_dwCurrFilePos  / 1024;

        case AUDIO_FILE_PARAM__NUM_CHANNELS_PER_SAMPLE_POINT :
           // Retrieve (or, sometimes, modify) the NUMBER OF CHANNELS PER SAMPLE (!).
           return m_nChannelsPerSampleInFile;

        case AUDIO_FILE_PARAM__CURRENT_SAMPLE_INDEX :
           // Read the current sample index that will be used on the next
           // READ- or WRITE- access to the audio samples in the opened file.
           // Returns a negative value on any error.
           // Return value : sample index; 0xFFFFFFFF = ERROR ("4 Gig minus one")
           return m_dwCurrentSampleIndex;

        case AUDIO_FILE_PARAM__TOTAL_COUNT_OF_SAMPLES :
           // Returns the total number of sampling points in the file.
           // Replacement for xyz.GetTotalCountOfSamples() since 2015-02-12 .
           return m_dwFileLengthInSamples;

        case AUDIO_FILE_PARAM__BYTES_PER_SAMPLE_POINT :
           // A "sample point" contains all channels, sampled at the very same time.
           // Example: Stereo Recording, 16 bits per (single) sample
           //          ->  GetBytesPerSampleGroup = 4 (not 2 !)
           // Used to convert a "sample index" into a "file offset" .
           return m_iSizeOfDataType * m_nChannelsPerSampleInFile;

        case AUDIO_FILE_PARAM__CONNECTION_LOST        :
           // Returns TRUE when a previously opened network connection was lost,
           //  and the application should take action (like wait a few seconds,
           //  and then try to reconnect if configured that way) .
           // Used to check the "connected" status for web streams.
           // In a few (rare) cases, can also be used to SET the "connect"-status
           // ( used as a kludge, and for testing, in AudioStreamOut_GUI.cpp )
           return pvfh->InetClient.fConnectionLost;

        case AUDIO_FILE_PARAM__NUM_TIMESTAMPS         :
           // Returns the number of timestamps received (or sent), since the stream was started
           return (long)pvfh->i64TimestampCounter;

        case AUDIO_FILE_PARAM__VLF_TOOL_FLAGS         :
           // VTFLAG_INT1/2/4 , VTFLAG_FLOAT4/8, for non-compressed streams,
           //   compatible with Paul Nicholson's VLFRX tools.
           //   The VT-Flags themselves are defined in vlfrx-tools/vtlib.h .
           if( m_iDataTypeFlags & AUDIO_FILE_DATA_TYPE_FLOAT )
            { switch( m_iSizeOfDataType )
               { case 4:
                    return VTFLAG_FLOAT4;
                 case 8:
                    return VTFLAG_FLOAT8;
                 default:
                    break;
               }
            }
           else // not FLOAT but INTEGER :
            { switch( m_iSizeOfDataType )
               { case 1:
                    return VTFLAG_INT1;
                 case 2:
                    return VTFLAG_INT2;
                 case 4:
                    return VTFLAG_INT4;
                 default:
                    break;
               }
            }
           return 0; // ?

        case AUDIO_FILE_PARAM_WAITED_FOR_INPUT_MS: return pvfh->iWaitedForInput_ms;

        default:
           break;
      } // end switch( iAudioFileParam )
   } // end if( pvfh )

  // Arrived here ? The requested parameter is not supported.
  // KISS, return a negative value, because ZERO is usually a valid value for future parameters.
  return -1;

} // end C_VorbisFileIO::GetParameterByID()

//---------------------------------------------------------------------------
double C_VorbisFileIO::GetParameterByID_Double( int iAudioFileParam )
  // iAudioFileParam: one of the AUDIO_FILE_PARAM_..-numbers for which INTEGER
  //         values are not sufficient, e.g. AUDIO_FILE_PARAM_LAST_TIMESTAMP .
  // May return C_SND_MATH_INVALID_FLOAT_VALUE for unsupported parameters .
  // Full specs in cbproj/SoundUtl/AudioFileIO.cpp : GetParameterByID_Double().
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  double d;

  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM_LAST_TIMESTAMP :
        // Last timestamp (Unix Second) REALLY RECEIVED from a timestamped stream.
        // Value doesn't change unless NEW timestamps arrive !
        break;
     case AUDIO_FILE_PARAM_TIMESTAMP_LATENCY :
        // Stream latency, determined by comparing the local time and the received one,
        // calculated difference stored at the time-of-arrival of the last REALLY RECEIVED timestamp.
        if( pvfh != NULL )
         { return pvfh->dblTimestampLatency;
         }
        break;
     case AUDIO_FILE_PARAM_RX_BUFFER_USAGE_SECONDS: // Stream-RX-buffer-usage in *seconds*:
        if( pvfh != NULL )
         { // Number of samples still waiting in fltDecoderOutput[]:
           d = pvfh->nDecoderOutputBufferUsage - pvfh->iDecoderOutputReadIndex;
           if( (d>0.0) && (pvfh->nChannelsPerSampleInStream>0) && (pvfh->dblNominalSampleRate>0) )
            { d /= (pvfh->nChannelsPerSampleInStream * pvfh->dblNominalSampleRate);
            }
           else
            { d = 0.0;
            }
           return d;
         }
        break;
     default:
        break;
   }
  return C_SND_MATH_INVALID_FLOAT_VALUE; // parameter not supported by this incarnation of GetParameterByID_Double() !

} // end C_VorbisFileIO::GetParameterByID_Double()


/************************************************************************/
BOOL C_VorbisFileIO::SetParameterByID( int iAudioFileParam, long i32NewValue )
  // Returns TRUE if the new value could be set, otherwise FALSE.
  // Replaces a bundle of older, rarely used, "Set"-functions since 02/2015,
  // for example SetVTFlags(iVTFlags) -> SetParameterByID(AUDIO_FILE_PARAM__VLF_TOOL_FLAGS,..) .
{
  T_VorbisFileHelper *pvfh;
  DWORD dwNewSampleIndex, dwNewFilePos;
  BOOL  fOk = FALSE;

  if( m_pVorbisFileHelper == NULL )
   { // allocate one of these 'helpers' from a pool (in fact a simple array):
     m_pVorbisFileHelper = VorbisFileHelper_Alloc( this );
   }
  pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )


  if( pvfh )
   { switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
      {
        case AUDIO_FILE_PARAM__FILE_FORMAT :
           // Replacement for  BOOL SetFileFormat(int iAudioFileFormat) .
           //  The parameter (i32NewValue = iAudioFileFormat) must be one of the
           //  AUDIO_FILE_FORMAT_...-constants defined in AudioFileDefs.
           switch( i32NewValue )
            { case AUDIO_FILE_FORMAT_OGG  :
                 pvfh->iStreamFormat = STREAM_FORMAT_OGG_VORBIS;
                 return TRUE;
              case AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS :
                 pvfh->iStreamFormat = STREAM_FORMAT_NON_COMPRESSED;
                 return TRUE;
              case AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS:
                 // Uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
                 //  but VT_BLOCKs.  Added here 2020-10-11 but not really supported yet.
                 pvfh->iStreamFormat = STREAM_FORMAT_VLF_RX_BLOCKS;
                 return TRUE;
              default:
                 break;
            }
           return FALSE;

        case AUDIO_FILE_PARAM__FILE_SIZE_KBYTE :
           // Current file size (if there is an opened file) in KBYTE.
           return FALSE;  // this parameter is READ ONLY (at least here) !

        case AUDIO_FILE_PARAM__NUM_CHANNELS_PER_SAMPLE_POINT :
           // Retrieve (or, sometimes, modify) the NUMBER OF CHANNELS PER SAMPLE (!).
           return FALSE;  // this parameter is READ ONLY (at least here) !


        case AUDIO_FILE_PARAM__CURRENT_SAMPLE_INDEX :
           dwNewSampleIndex = (DWORD)i32NewValue;
           // Sets the current sample index that will be used on the next
           //  READ- or WRITE- access to the audio samples an opened file.
           //  Returns TRUE if ok ,  otherwise FALSE.
           //
           //  Note that 'winding to' a certain position ("seek")
           //  is only possible if the source is a local disk file,
           //  but not if the input is an endless stream received via TCP/IP (etc).
           if( pvfh->fIsInternetStream )
            {  // cannot "rewind" an internet stream (without an awful lot of work)
            }
           else // not an "internet stream" but a local file.
            {   // So far, due to the compression or unknown amount of 'headers',
                // the only 'position' (sample index) which can easily be reached
                // is the BEGIN of the file:
              dwNewFilePos = 0;
              if( dwNewSampleIndex > 0 ) // it's getting tricky.. tricky part not implemented yet :-)
               {
               }
#            if (VORBIS_USE_QFILE)
              if( pvfh->m_QFile.iHandle > 0 )
               { fOk = (QFile_Seek( &pvfh->m_QFile, dwNewFilePos , SEEK_SET ) > 0);
               }
#            else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
              if( pvfh->m_FileHandle > 0 )
               { fOk = ( lseek( m_FileHandle, dwNewFilePos, SEEK_SET ) > 0);
               }
#            endif
              if( fOk ) // successfully modified the file position..
               { // mark the internal buffers as 'empty' to force re-loading :
                 pvfh->dwBufferUsage = pvfh->dwBufferIndex = 0;
                 m_dwCurrFilePos = dwNewFilePos;
                 if( m_dwCurrFilePos > m_dwFileSizeInBytes )
                  {  m_dwFileSizeInBytes = m_dwCurrFilePos;
                  }
               }
            }
           return fOk; // TRUE=yes(ok), FALSE=no(error)

        case AUDIO_FILE_PARAM__TOTAL_COUNT_OF_SAMPLES :
           // The total number of sampling points in the file.
           // Replacement for xyz.GetTotalCountOfSamples() since 2015-02-12 .
           return FALSE; // "parameter is not writeable"

        case AUDIO_FILE_PARAM__BYTES_PER_SAMPLE_POINT :
           // A "sample point" contains all channels, sampled at the very same time.
           // Example: Stereo Recording, 16 bits per (single) sample
           //          ->  GetBytesPerSampleGroup = 4 (not 2 !)
           // Used to convert a "sample index" into a "file offset" .
           return FALSE; // "parameter is not writeable"

        case AUDIO_FILE_PARAM__CONNECTION_LOST        :
           // Used to check the "connected" status for web streams.
           // In a few (rare) cases, can also be used to SET the "connect"-status
           // ( used as a kludge, and for testing, in AudioStreamOut_GUI.cpp )
           pvfh->InetClient.fConnectionLost = (i32NewValue!=0);  // flag for the application
           return TRUE;  // "parameter successfully set"

        case AUDIO_FILE_PARAM__NUM_TIMESTAMPS         :
           // Returns the number of timestamps received, since stream started
           return FALSE; // "parameter is not writeable"

        case AUDIO_FILE_PARAM__VLF_TOOL_FLAGS         :
           // VTFLAG_INT1/2/4 , VTFLAG_FLOAT4/8, for non-compressed streams,
           //   compatible with Paul Nicholson's VLFRX tools.
           //   The VT-Flags themselves are defined in vlfrx-tools/vtlib.h .
           pvfh->iVTFlags = i32NewValue;
           fOk = TRUE;
           switch( pvfh->iVTFlags & VTFLAG_FMTMASK )
            {
              case VTFLAG_FLOAT4 : // 4 byte floats
                 m_iSizeOfDataType = 4;
                 m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_FLOAT;
                 break;
              case VTFLAG_FLOAT8 : // 8 byte floats
                 m_iSizeOfDataType = 8;
                 m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_FLOAT;
                 break;
              case VTFLAG_INT1   : // 1 byte signed integers
                 // (not 0..255 as in ancient wave files, thus 0x80 can be avoided.. fb)
                 m_iSizeOfDataType = 1;
                 m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
                 break;
              case VTFLAG_INT2   : // 2 byte signed integers
                 m_iSizeOfDataType = 2;
                 m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
                 break;
              case VTFLAG_INT4   : // 4 byte signed integers
                 m_iSizeOfDataType = 4;
                 m_iDataTypeFlags  = AUDIO_FILE_DATA_TYPE_SIGNED_I;
                 break;
              default :
                 fOk = FALSE;
                 break;
            }
           return fOk; // TRUE:format supported,  FALSE:not supported

        default:
           break;
      } // end switch( iAudioFileParam )
   } // end if( pvfh )

  // Arrived here ? The requested parameter is not supported. Life can be so simple.
  return FALSE;

} // end C_VorbisFileIO::SetParameterByID()

//---------------------------------------------------------------------------
BOOL C_VorbisFileIO::SetParameterByID_Double( int iAudioFileParam, double dblNewValue )
  // Almost the same as SetParameterByID(), but for 64-bit "double".
  // Returns TRUE if the specified parameter could be set, otherwise FALSE .
  // Full specs in cbproj/SoundUtl/AudioFileIO.cpp : SetParameterByID_Double().
{
  switch( iAudioFileParam ) // possible values defined in c:\cbproj\SoundUtl\AudioFileDefs.h
   {
     case AUDIO_FILE_PARAM_LAST_TIMESTAMP :
        // Last timestamp (Unix Second) REALLY RECEIVED from a timestamped stream.
        // Value doesn't change unless NEW timestamps arrive !
        break;
     case AUDIO_FILE_PARAM_TIMESTAMP_LATENCY :
        // Stream latency, determined by comparing the local time and the received one,
        // calculated difference stored at the time-of-arrival of the last REALLY RECEIVED timestamp.
        break;
     default:
        break;
   }
  return FALSE; // parameter not supported by this incarnation of GetParameterByID_Double() !

} // end C_VorbisFileIO::SetParameterByID_Double()


/***************************************************************************/
LONG C_VorbisFileIO::ReadSampleBlocks(
     int channel_nr,     // channel_nr for 1st destination block
     int nSamplesWanted, // length of destination blocks:
                         //    pFltDest1[0..nSamplesWanted-1]
                         //    pFltDest2[0..nSamplesWanted-1]
     T_Float *pFltDest1, // 1st destination block (~"LEFT", or Inphase component)
     T_Float *pFltDest2, // 2nd destination block (~"RIGHT"), may be NULL (!)
     T_ChunkInfo *pChunkInfo) // additional results, see ChunkInfo.h, may be a NULL pointer (!)
  /* Reads some samples from an Ogg/Vorbis file (or stream) .
   * Input parameters:
   *   channel_nr: 0= begin with the first channel (LEFT if stereo recording)
   *               1= begin with the second channel (RIGHT if stereo, or Q if complex)
   *     The UPPER BYTE of 'channel_nr' can optionally define
   *         HOW MANY CHANNELS SHALL BE PLACED IN ONE DESTINATION BLOCK:
   *    (channel_nr>>8) : 0x00 or 0x01 = only ONE channel per destination block
   *                      0x02         = TWO channels per destination block
   *                                    (often used for complex values aka I/Q)
   *
   *   nSamplesWanted: Max number of FLOATs which can be stored in T_Float dest[0..nSamplesWanted-1]
   *               (this is NOT always the same as the number of sampling points,
   *                for example if the output are I/Q samples, all placed in
   *                pFltDest1, there are only nSamplesWanted/2 complex sampling points).
   *   *pFltDestX = Pointers to destination arrays (one block per channel, sometimes "paired")
   * Return value:
   *      the NUMBER OF FLOATING-POINT VALUES placed in each destination block
   *                 (not necessarily the number of sample points!) ,
   *      or a negative value if an error occurred.
   * Note that returning less samples than you may expect is not an error
   * but indicates reaching the file's end, or a currently empty network buffer.
   *   ( This convention later caused trouble when the input is not a FILE
   *     but a STREAM, see WPH_IO_Plugin.cpp ) .
   * (*) T_Float may be either SINGLE- or DOUBLE-precision float,
   *     defined in some header file, depending on SWI_FLOAT_PRECISION .
   *
   * Depending on C_VorbisFileIO.m_iAudioFileOptions & AUDIO_FILE_OPTION_NON_BLOCKING_CALLS,
   *   if the "file" is in fact a STREAM, this function MAY or MAY NOT wait
   *   until a sufficient amount of data (or 'Ogg pages') has arrived from
   *   the network to deliver the requested number of samples.
   *   If the function is NOT allowed to block (wait for something),
   *   expect C_VorbisFileIO::ReadSampleBlocks() to return LESS SAMPLES than
   *   requested - the caller must have to cope with anything from ZERO samples
   *   to nSamplesWanted !
   *
   * In Spectrum Lab, called as follows :
   *   TSoundThread::Execute() -> C_AnyAudioFileIO::ReadSampleBlocks()
   *                             -> C_VorbisFileIO::ReadSampleBlocks() .
   *
   *
   */
{
  int  n, n2, nBufferedChannelsPerSample, nSamplePointsReturned;
  // long nBytesReadFromDecoder;
  // int  current_section;
  int  ov_result; // vorbis result code (when negative, it's an error code)
  int  iWatchdogCount=2000;   // shouldn't be necessary (but prevents endless loops)
  T_VorbisFileHelper* pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  struct VT_OGG_BLK* pTS;
  long double ldblTimestamp;  // beware of stupid Microsoft compilers:
                              // we need at least 80 bit precision here !
  double dblSrCal; int tbreak; // other results from tq_get()
  double d;
  long i32Counter;
  char sz80[84];
  // LONGLONG i64TimeToLeave, i64Tnow, i64Frequency; // removed because this turned debugging into a nightmare

  if( pvfh==NULL ) // no valid 'file helper' allocated !
   { return -1;
   }

  // QueryPerformanceCounter( (LARGE_INTEGER*)&i64Tnow );
  // QueryPerformanceFrequency( (LARGE_INTEGER*)&i64Frequency );
  // i64TimeToLeave = i64Tnow + 5 * i64Frequency;  // stay here for a MAXIMUM of 5 seconds
  pvfh->iWaitedForInput_ms = -1;  // didn't spend any time WAITING for data
     // received from the network in ReadSampleBlocks() yet.
     // This may be used later to identify the "pace setter" in SoundThd.cpp .
     // To retrieve the number of milliseconds spent WAITING in ReadSampleBlocks(),
     // use GetParameterByID(AUDIO_FILE_PARAM_WAITED_FOR_INPUT_MS) after the call.
  pvfh->nChannelsPerSampleRequested = 1;
  if( pFltDest2 != NULL )
   { pvfh->nChannelsPerSampleRequested = 2;
   }
  nBufferedChannelsPerSample     = pvfh->nChannelsPerSampleRequested;
  if( nBufferedChannelsPerSample > pvfh->nChannelsPerSampleInStream )
   {  nBufferedChannelsPerSample = pvfh->nChannelsPerSampleInStream;
   }
  if( nBufferedChannelsPerSample <= 0 )
   { return -2;  // bail out to prevent div-by-zero
   }

  if( pChunkInfo != NULL )  // additional output with timestamp, calibration, etc ?
   { pChunkInfo->dblSampleValueRange = 1.0;   // the Vorbis decoder outputs PCM value range +/- 1.0
     pChunkInfo->i64TotalSampleCounter = pvfh->i64NumSamplesReadByApp;
     pChunkInfo->nChannelsPerSample  = nBufferedChannelsPerSample;
     pChunkInfo->dwValidityFlags     = pvfh->chunk_info.dwValidityFlags;
     pChunkInfo->dwInfoFlags         = pvfh->chunk_info.dwInfoFlags;
     if( pvfh->chunk_info.dblPrecSamplingRate > 0.0 )
      { pChunkInfo->dblPrecSamplingRate = pvfh->chunk_info.dblPrecSamplingRate;
      }
     else
      { pChunkInfo->dblPrecSamplingRate = GetSampleRate();
      }
   }

  nSamplePointsReturned = 0;
  VFIO_i64InspectMeAfterDebuggerException = pvfh->i64NumSamplesReadByApp; // 2020-10-12


  TIM_StartStopwatch( &pvfh->stopwatch ); // .. to measure and limit the time spent in ReadSampleBlocks() [and anything called from there], WAITING for input from the network
  if( m_dbl_SampleRate > 0 )
   { pvfh->iMaxTimeToSpendReadingSamples_ms = (1000 * nSamplesWanted) / m_dbl_SampleRate;
     // 2022-03 : Got here with iMaxTimeToSpendReadingSamples_ms = 1000*64/250 = 256 [ms],
     //           when trying to PLAY BACK the (stuttering) F6ETU stream.
   }
  else
   { pvfh->iMaxTimeToSpendReadingSamples_ms = 100;
   }

  // Ex: Repeat until a sufficient number of samples has been read .
  //     (in older versions, before 2022-03-12, pvfh->fltDecoderOutput[] had to
  //      be EMPTY before the next Ogg page could be decoded.)
  //
  // Since 2022-03-12: Try to read as many samples as pvfh->fltDecoderOutput[] permits
  //                   - more than SL requested when calling SOUND_AudioFileInput.ReadSampleBlocks() from TSoundThread::Execute() -
  //                to drain the 'network buffer' as far as possible so we can
  //                detect if a backlash is building up (because the application
  //                doesn't CONSUME the samples at the rate the other end
  //                PRODUCES them). This new 'principle' can be selected by the
  //                following macro:
# define L_DRAIN_NETWORK_BUFFER 1 /* 1='new principle' since 2022-03-12; 0=old, without draining the network buffer */

  while( (nSamplePointsReturned<nSamplesWanted)
        && (!m_EndOfOutput) && (iWatchdogCount>0)
        && (pvfh->nChannelsPerSampleInStream>0)  ) // <- added 2022-03-12 to prevent div-by-zero somewhere
   {
     // As long as there are still some samples waiting in fltDecoderOutput[],
     //  take the samples from there, instead of reading the next Ogg page.
     if ( pvfh->iDecoderOutputReadIndex < pvfh->nDecoderOutputBufferUsage )
      { // there's at least ONE sample remaining in pvfh->fltDecoderOutput[]..
        if( (pChunkInfo!=NULL) && (pvfh->iDecoderOutputReadIndex==0) )
         {  // Fill out a T_ChunkInfo structure with a timestamp for the
            // FIRST PCM sample in this block. Usually, we'll have to interpolate
            // because Paul's VT_OGG_BLK structure will occurr less than once
            // per <nSamplesWanted>  .
            // ---> retrieval of a timestamp for the FIRST sample in the block
            //      is postponed until we processed more input from Vorbis !
         }
        DESHOBI( (long)pFltDest1 );
        while( ( nSamplePointsReturned < nSamplesWanted )
            && ( pvfh->iDecoderOutputReadIndex < pvfh->nDecoderOutputBufferUsage )
            && ( pvfh->iDecoderOutputReadIndex < MAX_DECODER_OUTPUT_SIZE ) )
         { *pFltDest1++ = pvfh->fltDecoderOutput[ pvfh->iDecoderOutputReadIndex ];
           //
           // If the caller expects to read a 2nd channel : ...
           if( pFltDest2 != NULL )
            { if( nBufferedChannelsPerSample >= 2 )
               { *pFltDest2++ = pvfh->fltDecoderOutput[ pvfh->iDecoderOutputReadIndex+1 ];
               }
              else
               { *pFltDest2++ = 0.0;
               }
            }
           pvfh->iDecoderOutputReadIndex += nBufferedChannelsPerSample;
           ++nSamplePointsReturned;
         }
      } // end if(pvfh->iDecoderOutputReadIndex<pvfh->nDecoderOutputBufferUsage)

     // Request new PCM samples from the vorbis decoder
     //           (or read them from an UNCOMPRESSED stream)
     // if our internal buffer (pvfh->fltDecoderOutput[]) is SUFFICIENTLY EMPTY
     //           (not "completely empty", as it used to be before 2022-03-12).
     // Note that ReadAndDecodeNonCompressed() must be capable of processing
     //           just a SMALL PART of the samples following a single VT_BLOCK,
     //           thus it should be ok to call ReadAndDecodeNonCompressed()
     //           with any (small but nonzero) number of SAMPLE POINTS (below: 'n').
#   if( ! L_DRAIN_NETWORK_BUFFER ) // OLD stuff (only called ReadAndDecodeNonCompressed(etc) when pvfh->fltDecoderOutput[] was COMPLETELY empty) :
     if( (pvfh->iDecoderOutputReadIndex >= pvfh->nDecoderOutputBufferUsage ) // pvfh->fltDecoderOutput[] COMPLETELY empty ?
      && (nSamplePointsReturned<nSamplesWanted) ) // << added 2014-04
      { // 'Want' more samples, and pvfh->fltDecoderOutput[] is COMPLETELY empty .
        // Time to read the next block of samples from the file or stream :
        pvfh->iDecoderOutputReadIndex   = 0;
        pvfh->nDecoderOutputBufferUsage = 0;

#    else  // L_DRAIN_NETWORK_BUFFER : Already call ReadAndDecodeNonCompressed(etc) when pvfh->fltDecoderOutput[] is SUFFICIENTLY empty :
     n = pvfh->nDecoderOutputBufferUsage - pvfh->iDecoderOutputReadIndex; // -> number of single floats STILL WAITING in pvfh->fltDecoderOutput[]
     n2 = (MAX_DECODER_OUTPUT_SIZE-n) / pvfh->nChannelsPerSampleInStream; // -> #sample points FREE in pvfh->fltDecoderOutput[]
     if( n2 >= 16 )
      { // Can 'accept' at least 16 more SAMPLE POINTS, i.e. pvfh->fltDecoderOutput[] is 'sufficiently empty' ->
        // Try to read the next block of samples from the file or stream to keep the 'network buffer' usage as low as possible.
        if( n <= 0 ) // pvfh->fltDecoderOutput[] COMPLETELY empty ?
         {
           pvfh->iDecoderOutputReadIndex   = 0;
           pvfh->nDecoderOutputBufferUsage = 0;
         }
        else  // some sample-points remaining in pvfh->fltDecoderOutput[] ->
         { // Scroll up the remaining buffer content so the OLDEST sample
           // will be at pvfh->fltDecoderOutput[0] :
           memmove( &pvfh->fltDecoderOutput[0], // destination
                    &pvfh->fltDecoderOutput[pvfh->iDecoderOutputReadIndex], // source
                    n * sizeof(float) );        // number of BYTES(!) to copy
           pvfh->iDecoderOutputReadIndex   = 0;
           pvfh->nDecoderOutputBufferUsage = n; // number of SINGLE FLOATS (not "sample points") remaining in pvfh->fltDecoderOutput[]
         }
#    endif // L_DRAIN_NETWORK_BUFFER ?
        if((pvfh->iStreamFormat == STREAM_FORMAT_NON_COMPRESSED ) // added 2014-04-21 :
        || (pvfh->iStreamFormat == STREAM_FORMAT_VLF_RX_BLOCKS) ) // added 2020-10-11 :
         { // ------------------ read NON-COMPRESSED streams --------------->
           // The audio stream turned out to be NON COMPRESSED,
           // and the structure should be defined by 'injected' structures
           // defined in VorbisFileIO.h : T_StreamHeader, T_VT_Block, combined as T_StreamHeader1.
#         if( ! L_DRAIN_NETWORK_BUFFER ) // OLD stuff : only read as many sample as 'wanted' by the caller..
           n = nSamplesWanted - nSamplePointsReturned;
#         else  // L_DRAIN_NETWORK_BUFFER :
           n = n2;  // retrieve as many samples as can be appended to pvfh->fltDecoderOutput[]
           // 2022-03 : Got here with ...
           //    1st call : n = 262144 -> ReadAndDecodeNonCompressed() returned 64 sample points only,
           //                             new pvfh->nDecoderOutputBufferUsage =
           //  (long gap while paused in the debugger; "network buffer" filled up a lot..)
           //    2nd call : n = 262144 -> ReadAndDecodeNonCompressed() returned 13824 sample points,
           //                             new pvfh->nDecoderOutputBufferUsage =
           //    3rd call : n = 248384 -> ReadAndDecodeNonCompressed() returned 13824 sample points,
#         endif // L_DRAIN_NETWORK_BUFFER ?
           if( n > (MAX_DECODER_OUTPUT_SIZE / nBufferedChannelsPerSample) )
            {  n =  MAX_DECODER_OUTPUT_SIZE / nBufferedChannelsPerSample;
            }
           if( n>0 )
            { ov_result = ReadAndDecodeNonCompressed( pvfh, // here: called from C_VorbisFileIO::ReadSampleBlocks() -> ReadAndDecodeNonCompressed() -> VorbisRead() -> INET_ReadDataFromServer() [blocking call with timeout]
                          n,                          // [in] iMaxSamplePoints
                          nBufferedChannelsPerSample, // [in] number of channels PER SAMPLE POINT
                          &pvfh->fltDecoderOutput[pvfh->nDecoderOutputBufferUsage], // [out] float *pfltDest
                          &pvfh->iWaitedForInput_ms); // [out] number of milliseconds spent here WAITING for data
              // > ReadAndDecodeNonCompressed() returns...
              // >  the number of SAMPLE POINTS (with nChannelsPerSample channels each) actually read.
              // >  Zero    : nothing read but no error.
              // >  Negative: one of the error codes defined in ogg.h
              // Note: Samples don't "pour in" in complete VT_BLOCK fragments here.
              //       Due to the limited network buffer (65546 bytes ?),
              //       expect to have only about 2000 samples delivered
              //       per call of ReadAndDecodeNonCompressed(), depending
              //       on the data type used 'on the network' (e.g. 8-byte float).
              if( ov_result > 0 )  // successfully read a few samples :
               {  pvfh->nDecoderOutputBufferUsage +=  // > number of single floats(!) READ INTO fltDecoderOutput[]
                            ov_result * nBufferedChannelsPerSample;
                  // with tcp://129.206.29.82:49335 : 1st and 2nd ov_result = 2000 [sample points] at 48kSamples/sec
                  // with tcp://129.206.29.82:49336 : 1st and 2nd ov_result = 2000 [sample points] at 8 kSamples/sec

               }
            }
           else
            { ov_result = 0;  // no reason to call ReadAndDecodeNonCompressed() now
            }
         }
        else  // -------------- read COMPRESSED Ogg/Vorbis stream ---------->
         { ov_result = ReadAndDecodePages( pvfh, TRUE/*read more samples if possible*/,
                       &pvfh->iWaitedForInput_ms); // [out] number of milliseconds spent here WAITING for data
         }
        if( ov_result <= 0 )
         { // "error" or "nothing available at the moment" ?
           if( pvfh->fIsInternetStream )
            { // Since even 'endless' streams tend to break down sometimes,
              // added a crude timeout monitor in the outer loop..
              Sleep(10);
              if( pvfh->iWaitedForInput_ms < 0 )
               { pvfh->iWaitedForInput_ms = 10;
               }
              else
               { pvfh->iWaitedForInput_ms += 10;
               }
            }
           else  // not a stream but a local disk file :   END of file !
            { m_EndOfOutput = TRUE;
            }
           break;
         }
      } // end  if( m_dwDecoderOutputByteIndex >= m_dwDecoderOutputNumBytes )
     else  // there are still some bytes waiting in m_c4kDecoderOutput[]
      {    // which have not been processed yet : leave this loop !
        break;
      }
     --iWatchdogCount;
     // QueryPerformanceCounter( (LARGE_INTEGER*)&i64Tnow );
     // if( i64Tnow > i64TimeToLeave )
     //  { break;
     //  }
   } // end   while(nBytesRemainingToRead>0) && ...

  if( pChunkInfo != NULL )  // additional output with timestamp, calibration, etc ?
   { // Update these flags *AFTER* extracting more 'chunk info' from VT_BLOCKs (or whatever) :
     pChunkInfo->dwValidityFlags = pvfh->chunk_info.dwValidityFlags;
     pChunkInfo->dwInfoFlags     = pvfh->chunk_info.dwInfoFlags;
     // Fill out some other fields in the T_ChunkInfo structure,
     //      for the first returned sample in this block,
     //      using the timestamps in Paul's VT_OGG_BLK structure.
     // For that purpose, the most recent timestamp-packets were queued up,
     // so we can interpolate (or extrapolate) the most important parameters from the entry
     // BEFORE, and the entry AFTER the current output-sample-index (i64NumSamplesReadByApp).
     // Note that pvfh->i64NumSamplesReadByApp is incremented AFTER retrieving the timestamp.
     if( tq_get(  // TRY TO get a timestamp from the queue
           pvfh,  // [in,out] T_VorbisFileHelper *pvfh, ogg/vorbis-related states
           pvfh->i64NumSamplesReadByApp, // [in] decoder sample index for which to retrieve data
           &ldblTimestamp,               // [out] timestamp (in Unix format, fractional)
           &dblSrCal,                    // [out] sample rate calibration factor, near 1.0
           &tbreak,                      // [out] flag 'Timestamp discontinuity detected'
           &pTS,                         // [out, optional] address of a VT_OGG_BLK
           &i32Counter )  )              // [out, optional] index of the returned timestamp
      {
        if( ! pvfh->fGotStartTimeFromTimestamp )
         { pvfh->dblStartTime = ldblTimestamp;
           pvfh->fGotStartTimeFromTimestamp = TRUE;
         }
        pChunkInfo->rtRecTime = ldblTimestamp - pvfh->dblStartTime;
        pChunkInfo->ldblUnixDateAndTime= ldblTimestamp;
        pChunkInfo->dwValidityFlags |= CHUNK_INFO_SAMPLE_RATE_VALID | CHUNK_INFO_TIMESTAMPS_VALID;
        if( pvfh->dblNominalSampleRate>0 &&  dblSrCal>0.9 && dblSrCal<1.1 )
         { pChunkInfo->dblPrecSamplingRate = pvfh->dblNominalSampleRate * dblSrCal;
           pChunkInfo->dwValidityFlags |= CHUNK_INFO_SAMPLE_RATE_CALIBRATED;
         }
        pChunkInfo->gps.iUpdateCounter = i32Counter;
        pChunkInfo->gps.dblLat_deg = pTS->glat;
        pChunkInfo->gps.dblLon_deg = pTS->glong;
        pChunkInfo->gps.dbl_mASL   = pTS->gasl;
        pChunkInfo->gps.dblVel_kmh = 0.0; // not supported by VT_OGG_BLK
        pChunkInfo->gps.dblDateAndTime = ldblTimestamp;
#     if(1)   // plausibility check for the timestamps (read from a file) ?
        if( m_ldblPrevCheckedTimestamp > 0 )
         { ldblTimestamp = m_ldblPrevCheckedTimestamp
              + (long double)nSamplePointsReturned / pChunkInfo->dblPrecSamplingRate;
           d = ldblTimestamp - pChunkInfo->ldblUnixDateAndTime;
           if( (d<-8e-6) || (d>8e-6) )
            { sprintf(sz80, "ReadSamples: TS gap, dt=%.3lf sec", (double)d );
              VFIO_DebuggerBreak( pvfh, sz80, __LINE__ ); // <<< set breakpoint here .. <<<
              // 2012-01-08 : Detected two consecutive breaks with d = 0.256 seconds here.
              //              That's 8192 / 32000 Hz - suspicious !
              //    Even more suspicious: This only happens every 5 minutes.
              //    Logged as 2012_01_08_1730_stream_log_out_with_512ms_breaks.ogg .
              //  A similar plausibility test was then added
              //    in C_VorbisFileIO::WriteSamples_Float() .
            }
         }
        m_ldblPrevCheckedTimestamp = pChunkInfo->ldblUnixDateAndTime;
#     endif   // plausibility check for the timestamps ?
      }
     else
      { // failed to retrieve (or interpolate) the timestamp (etc) ->
        pChunkInfo->dwValidityFlags &=
            ~ (  CHUNK_INFO_SAMPLE_RATE_CALIBRATED
               | CHUNK_INFO_TIMESTAMPS_VALID
               | CHUNK_INFO_TIMESTAMPS_PRECISE );
        // In this case (no valid timestamp), the CALLER should use
        // his own timestamp - but just for sanity, we provide our own here,
        // entirely based on the SAMPLE COUNT :
        pChunkInfo->ldblUnixDateAndTime = pvfh->dblStartTime;  // may be ZERO .. sigh !
        if( pvfh->dblNominalSampleRate > 0.0 )
         { pChunkInfo->ldblUnixDateAndTime +=
               (double)pvfh->i64NumSamplesReadByApp
                     / pvfh->dblNominalSampleRate; // 2020-10 : again, div-by-zero.. sigh ! !
         }
      }
   } // end if( pChunkInfo != NULL )

  if( nSamplePointsReturned > 0 )
   { pvfh->i64NumSamplesReadByApp += nSamplePointsReturned;  // AFTER retrieving the timestamp !
     m_dwCurrentSampleIndex += nSamplePointsReturned;

     // Finish the current bitrate-measurement-interval ?
     if( pChunkInfo != NULL )
      { d = pChunkInfo->ldblUnixDateAndTime - pvfh->dblStartTimeOfBitrateMeasurement;
        if( d<0 || d>10 )
         {
           if( d>1 )
            { pvfh->nBitsPerSecond = pvfh->nBitrateCounter / d;
            }
           pvfh->dblStartTimeOfBitrateMeasurement = pChunkInfo->ldblUnixDateAndTime;
           pvfh->nBitrateCounter = 0;
         }
      }

   } // end if( nSamplePointsReturned > 0 )

  if( nSamplePointsReturned != nSamplesWanted )
   {  nSamplePointsReturned = nSamplePointsReturned; // << place for a 'conditional breakpoint'
      // Details : C:\cbproj\SpecLab\test_notes\2014_04_26_bugs_in_VorbisFileIO.txt
   }

  // Return THE NUMBER OF SAMPLE-POINTS read into the caller's buffer(s) :
  return nSamplePointsReturned;

} // C_VorbisFileIO::ReadSampleBlocks(..)





/*****************************************************************************/
/*  Encoder                                                                  */
/*****************************************************************************/

/***************************************************************************/
BOOL WriteOggPage( T_VorbisFileHelper *pvfh, ogg_page *pOggPage )
  // Writes an Ogg page (usually with Vorbis data) to a file or stream .
  //
  //  Calling sequence (when writing a local disk file) :
  //   1.) WriteSamples_Float() -> EncoderInit2() -> WriteOggPage() [postponed init]
  //         pOggPage->header="OggS", header_len=28, body="\x01vorbis..", body_len=30
  //   2.) WriteSamples_Float() -> EncoderInit2() -> WriteOggPage() [postponed init]
  //         pOggPage->header="OggS", header_len=43, body="\x03vorbis*.", body_len=3857
  //   3.) WriteSamples_Float() -> WriteOggPage()
  //         pOggPage->header="OggS", header_len=57, body=<binary stuff>, body_len=4308
{
  BOOL fResult = FALSE;
  int i,j, nBytesForHeaderAndBody;
  DWORD dw;
  if( pvfh!=NULL && pOggPage!=NULL )
   { nBytesForHeaderAndBody = pOggPage->header_len + pOggPage->body_len;

     pvfh->nBitrateCounter += 8 * nBytesForHeaderAndBody;
     ++pvfh->dwPageIndex;   // unique serial number, here incremented in WriteOggPage()

#   if (VORBIS_USE_QFILE)
     if( pvfh->m_QFile.iHandle > 0 )
      {
        if( ( QFile_Write( &pvfh->m_QFile, pOggPage->header, pOggPage->header_len ) )
          &&( QFile_Write( &pvfh->m_QFile, pOggPage->body,   pOggPage->body_len ) ) )
         { fResult = TRUE;
           pvfh->pThis->m_dwCurrFilePos += (DWORD)(pOggPage->header_len+pOggPage->body_len);
           if( pvfh->pThis->m_dwCurrFilePos > pvfh->pThis->m_dwFileSizeInBytes ) // keep track of file size !
            {  pvfh->pThis->m_dwFileSizeInBytes = pvfh->pThis->m_dwCurrFilePos;
            }
         }
      }
#   else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
#     error "option not supported yet . Add support for YOUR file system here !"
#   endif
     else if( pvfh->fIsInternetStream )
      {
        // Save the outbound Ogg-stream to a local file ("audio logfile") ?
        if( pvfh->fWriteLogfile && pvfh->qfStreamLog.fOpened )
         {
           if( ( QFile_Write( &pvfh->qfStreamLog, pOggPage->header, pOggPage->header_len ) )
             &&( QFile_Write( &pvfh->qfStreamLog, pOggPage->body,   pOggPage->body_len ) ) )
            { // fResult = TRUE;
            }
           else // something fishy with the log, don't try to write to it again
            {
              CallBack_Error( pvfh, 0/*time*/, "Failed to write %s", pvfh->sz255StreamLogfile );
              QFile_Close( &pvfh->qfStreamLog );
              pvfh->fWriteLogfile = FALSE;
            }
         }

        // Are we actively PUSHING the stream (as a client) to a single remote receiver ?
        if( pvfh->InetClient.sock!=INVALID_SOCKET )
         {
           // Send the Ogg page to the stream itself :
           i = INET_Write(  // implemented in C:\cbproj\YHF_Tools\YHF_Inet.c
              &pvfh->InetClient,  // [in,out] T_INET_Client *pInetClient; a web client from YHF_Inet.c
              0,                  // [in] max. timeout in milliseconds, here 0 = non-blocking
              INET_WR_OPTION_DONT_FLUSH, // [in] options : DO NOT SEND YET but assemble the page in a buffer
              (BYTE*)pOggPage->header,  pOggPage->header_len);
           j = INET_Write(
              &pvfh->InetClient,  // [in,out] T_INET_Client *pInetClient; a web client from YHF_Inet.c
              0,                  // [in] max. timeout in milliseconds, here 0 = non-blocking
              INET_WR_OPTION_DONT_FLUSH, // [in] options
              (BYTE*)pOggPage->body,   pOggPage->body_len );
           //  If the internet socket isn't able to send the data within
           //  the specified timeout value, YHF_Inet.c will append those data
           //  into an internal buffer, which it will hopefully send later.
           //   Thus if i or j are negative, there's something fishy, but not
           //   necessarily an ERROR .
           if( j<i ) i=j;
           if( i<0 )
            { // Something went wrong (connection lost, etc etc etc).
              if( pvfh->nWriteErrors==0 )
               {
                 CallBack_Error( pvfh, 0/*time*/, "Failed to send: %s", ErrorCodeToString( -i ) );
               } // end if( pvfh->first_write_error ) ?
              ++pvfh->nWriteErrors;
            } // end if < error trying to push the stream into the internet-client >

         } // end if < actively PUSH the stream (if this side is the CLIENT) >
        else // not "actively pushing" ->
         { // Most likely, acting as a SERVER. In that case, remote clients
           // may pull audio data from SL's integrated web server (via HTTP).
           // That happens in GetNextPagesForStreamOutputServer() [not HERE].
         }
      } // end if <internet stream>

     // store this Ogg-page in the buffer for the 'initial header' sent to each new remote client ?
     if( pvfh->fStoreInitialHeaders )  // flag set in EncoderInit2() to store the initial headers for multiple remote receivers
      {
        if( (pvfh->nInitialHeaderBufferedBytes + nBytesForHeaderAndBody ) < INITIAL_HEADER_BUFFER_SIZE )
         { dw = pvfh->nInitialHeaderBufferedBytes;
           memcpy( pvfh->bInitialHeaderBuffer + dw, pOggPage->header, pOggPage->header_len );
           dw += pOggPage->header_len;
           memcpy( pvfh->bInitialHeaderBuffer + dw, pOggPage->body,   pOggPage->body_len );
           dw += pOggPage->body_len;
           pvfh->nInitialHeaderBufferedBytes = dw;  // THIS will make the "initial header" available...
           // ... for the 'multiple readers', which will possibly access the buffer IN ANOTHER THREAD(!) .
         }
      } // end if < store initial headers in bInitialHeaderBuffer[] >
     else // the 'initial headers' are placed in a persistent buffer, but what about the 'AUDIO SAMPLES' ?
     if( pvfh->fStoreSamplesForMultipleReaders ) // flag set in OutOpen(), used for OUTPUT STREAM SERVER :
      { // Store the last few seconds of compressed audio in a circular buffer (FIFO-like),
        // from where they can be "pulled out" by MULTIPLE READERS .
        // Details about the one-writer, multiple-reader buffer in C_VorbisFileIO::GetNextPagesForStreamOutputServer() :
        // >            _____________ ___________ ________ _______ ________
        // > bBuffer : | Ogg-page[n] | page[n-4] | [n-3]  | [n-1] | unused |
        // >           |_____________|___________|________|_______|________|
        // >                         |           |        |       |
        // >                     dwBufferIndex   |        |    dwBufferUsage  (from writer)
        // >          /|\                        |         to be read from the buffer
        // >           |           circular wrap                  |
        // >            --------------<<<<<-----------------------
        // >
        // Because especially for protocols like UDP (but also for TCP),
        // a single Ogg page shall be send 'in a single over' (if possible),
        // thus the FIFO buffer must be aware of Ogg page boundaries,
        // and an Ogg page must not cross the wrapping-point (at the upper end of the buffer).
        // So will the new Ogg page fit in the buffer without wrapping ?
        if( (pvfh->dwBufferIndex + nBytesForHeaderAndBody ) >= BUFFER_MAX_BYTES ) // circular buffer wrap ?
         { // pvfh->dwBufferIndex + nBytesForHeaderAndBody would exceed bBuffer[BUFFER_MAX_BYTES] :
           // buffer-head-index wraps around, but dwBufferUsage indicates the "current end" for all readers:
           pvfh->dwBufferUsage = pvfh->dwBufferIndex;
           pvfh->dwBufferIndex = 0;  // index into pvfh->bBuffer[] (head)
           // pvfh->iBufferPagesHead = the index into BufferPages[] wraps around, too !
           // Make the end of the 'Ogg page directory' invalid (=the "unused" rest in the diagram above):
           pvfh->nBufferPagesUsed = pvfh->iBufferPagesHead; // only BufferPages[0..nBufferPagesUsed-1] may be read in GetNextPagesForStreamOutputServer()
           pvfh->iBufferPagesHead = 0;
         } // end if < circular buffer wrap > ?

        if( pvfh->iBufferPagesHead >= BUFFER_MAX_PAGES ) // for safety only:
         { // (this will rarely ever happen because BUFFER_MAX_BYTES(64 kByte?) isn't large enough
           //  to keep more than a few dozen Ogg pages in pvfh->bBuffer[],
           //  and BUFFER_MAX_PAGES (64?) is much larger than that)
           pvfh->nBufferPagesUsed = pvfh->iBufferPagesHead; // only BufferPages[0..nBufferPagesUsed-1] may be read in GetNextPagesForStreamOutputServer()
           pvfh->iBufferPagesHead = 0;
         }

        // For safety, make sure bBuffer[0..BUFFER_MAX_BYTES] is not exceeded:
        if( (pvfh->dwBufferIndex + nBytesForHeaderAndBody ) < BUFFER_MAX_BYTES )
         { dw = pvfh->dwBufferIndex;
           memcpy( pvfh->bBuffer + dw, pOggPage->header,  pOggPage->header_len );
           dw += pOggPage->header_len;
           memcpy( pvfh->bBuffer + dw, pOggPage->body,    pOggPage->body_len );
           dw += pOggPage->body_len;
           pvfh->i64NumTotalBytesProcessed += (pOggPage->header_len + pOggPage->body_len);
           // Also update the "directory" of the buffered Ogg pages:
           pvfh->BufferPages[ pvfh->iBufferPagesHead ].iBufIndex = pvfh->dwBufferIndex; // index into bBuffer[] where this page begins
           pvfh->BufferPages[ pvfh->iBufferPagesHead ].iSize_Byte= nBytesForHeaderAndBody;
           pvfh->BufferPages[ pvfh->iBufferPagesHead ].dwPageIndex = pvfh->dwPageIndex; // unique index (or "counter") for this Ogg page
           ++pvfh->iBufferPagesHead;  // new 'head index' for the Ogg page directory (pvfh->BufferPages[])
           // >  BufferPages[] shall only contain info about those Ogg pages
           // >  which are still available in pvfh->bBuffer[] !
           if( pvfh->nBufferPagesUsed < pvfh->iBufferPagesHead )
            {  pvfh->nBufferPagesUsed = pvfh->iBufferPagesHead;
            }
           pvfh->dwBufferIndex = dw;  // THIS will make the new Ogg page available...
           // ...for the multiple readers, which will possibly read the FIFO
           // in ANOTHER TREAD. No problem because the buffer is large enough
           // so none of the multiple readers will still need the data
           // which have just been overwritten by the memcpy()s above.
           // When a NEW READER attaches, VorbisFileIO will set 'his' stream-sample-position
           // to the NEWEST (! - not the OLDEST - !) part of the buffer for that reason.
           // See details in GetNextPagesForStreamOutputServer() .
         }

      } // end else < ! pvfh->fStoreInitialHeaders >

   } // end if ( pvfh != NULL )

  return fResult;
} // end WriteOggPage()  (no class member!)

/***************************************************************************/
BOOL VFIO_Write( T_VorbisFileHelper *pvfh,
                 BYTE *pbSource, int nBytesToWrite,
                 int iOptions ) // [in] INET_WR_OPTION_DONT_FLUSH, etc etc
  // Writes a couple of bytes to a local file, or sends them to a network stream.
  // Not used by Ogg/Vorbis encoded files/streams (they use WriteOggPage)
  // but for writing/sending non-compressed files/streams .
  // Actually based on WriteOggPage.
{
  BOOL fResult = FALSE;
  int i;
  if( pvfh!=NULL && nBytesToWrite>0 )
   {
#   if (VORBIS_USE_QFILE)
     if( pvfh->m_QFile.iHandle > 0 )
      {
        if( QFile_Write( &pvfh->m_QFile, pbSource, nBytesToWrite ) )
         { fResult = TRUE;
           pvfh->pThis->m_dwCurrFilePos += (DWORD)nBytesToWrite;
           if( pvfh->pThis->m_dwCurrFilePos > pvfh->pThis->m_dwFileSizeInBytes ) // keep track of file size !
            {  pvfh->pThis->m_dwFileSizeInBytes = pvfh->pThis->m_dwCurrFilePos;
            }
         }
      }
#   else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
#     error "option not supported yet . Add support for YOUR file system here !"
#   endif
     else if( pvfh->fIsInternetStream )
      {
        // Save the block to a local file ("audio logfile") ?
        if( pvfh->fWriteLogfile && pvfh->qfStreamLog.fOpened )
         {
           if( QFile_Write( &pvfh->qfStreamLog, pbSource, nBytesToWrite ) )
            { // fResult = TRUE;
            }
           else // something fishy with the log, don't try to write to it again
            {
              CallBack_Error( pvfh, 0/*time*/, "Failed to write %s", pvfh->sz255StreamLogfile );
              QFile_Close( &pvfh->qfStreamLog );
              pvfh->fWriteLogfile = FALSE;
            }
         }

        // Send the block to the stream itself :
        i = INET_Write(  // implemented in C:\cbproj\YHF_Tools\YHF_Inet.c
              &pvfh->InetClient,  // [in,out] T_INET_Client *pInetClient; a web client from YHF_Inet.c
              0,                  // [in] max. timeout in milliseconds, here 0 = non-blocking
              iOptions,           // [in] options : DO NOT SEND YET but assemble the page in a buffer
              (BYTE*)pbSource, nBytesToWrite );
        // Note: If the internet socket isn't able to send the data now,
        //       YHF_Inet.c will append those data to an internal buffer,
        //       which it will hopefully send later.
        //   Thus if i is negative, there's something fishy, but not
        //   necessarily an ERROR .  DON'T PANIC .
        pvfh->nBitrateCounter += 8 * ( nBytesToWrite );
        if( i<0 )
         { // Something went wrong (connection lost, etc etc etc).
           // ToDo: dump the error code in some kind of error-log ?
           if( pvfh->nWriteErrors==0 )
            {
              CallBack_Error( pvfh, 0/*time*/, "Failed to send: %s", ErrorCodeToString( -i ) );
            } // end if( pvfh->first_write_error ) ?
           ++pvfh->nWriteErrors;
         }
        else // no error: assume ALL bytes are on their way, or at least in the tx-buffer
         { fResult = TRUE;
         }
      }
   } // end if ( pvfh != NULL )

  return fResult;

} // end VFIO_Write()


/***************************************************************************/
static BOOL EncoderInit2( // "postponed" (*) part of the encoder initialisation
              T_VorbisFileHelper *pvfh,
              T_ChunkInfo *pChunkInfo,  // [in] precise sample rate and (GPS-) timestamp
              ogg_page    *pOggPage )   // [out] Ogg page (kind of container)
   // (*) This is now postponed until the timestamps are valid,
   //     to send them to Pauls's VLF server .
{
  int n, v_result;
  struct VT_OGG_HDR sr_ogg_hdr;
  ogg_packet sr_header;

  pvfh->fStoreInitialHeaders = TRUE;  // store the initial headers for remote clients which may connect the server LATER
  pvfh->nInitialHeaderBufferedBytes = 0;  // nothing cached yet

  // Setup one or two stream encoders, stream 0 for vorbis, stream 1 for timestamps
  if( (v_result=ogg_stream_init( &pvfh->osv, 0/*serialno*/) ) < 0)
   { CallBack_Error( pvfh, 0/*time*/, "Can't init ogg-stream #1 (audio): %s",
                    (char*)VorbisErrorCodeToString( v_result ) );
     return FALSE;
   }
  if( pvfh->want_tstamp )
   { if( (v_result=ogg_stream_init( &pvfh->ost, 1/*serialno*/) ) < 0)
      { CallBack_Error( pvfh, 0/*time*/, "Can't init ogg-stream #2 (timestamps): %s",
                       (char*)VorbisErrorCodeToString( v_result ) );
        return FALSE;
      }
     //  Construct and send the VT_OGG_HDR on stream 'ost' = TIMESTAMPS .
     //   (don't modify this - it is used in Paul's stream server !
     //    This doesn't seem to be the first 'real' timestamp, but just another header)
     sr_ogg_hdr.magic = HDR_MAGIC;
     sr_ogg_hdr.chans = pvfh->nChannelsPerSampleInStream;
     sr_ogg_hdr.rate  = pvfh->dblNominalSampleRate; // must not be zero in this 'initial' VT_OGG_HDR !
     sr_ogg_hdr.spare = 0;

     sr_header.packet = (unsigned char *) &sr_ogg_hdr;
     sr_header.bytes = sizeof( struct VT_OGG_HDR);
     sr_header.b_o_s = 1;
     sr_header.e_o_s = 0;
     sr_header.granulepos = 0;
     sr_header.packetno = pvfh->vt_packetno++;
     ogg_stream_packetin( &pvfh->ost, &sr_header);

     // From Paul:
     // > Make sure you are sending a VT_OGG_BLK timestamp packet
     // > before you send the corresponding audio.
     // The 'while' loop forces ogg to send this packet immediately.
     while( ogg_stream_pageout( &pvfh->ost, pOggPage ) ||
            ogg_stream_flush( &pvfh->ost,   pOggPage ) )
      { WriteOggPage( pvfh, pOggPage );  // not really 'write and flush' yet..
        // only prepare it in a network buffer
        // to avoid calling the SOCKET interface with very small chunks .
        // 2014-05: For the 'stream server' (with ONE WRITER, MULTIPLE READERS),
        //
      }
   } // end if( pvfh->want_tstamp )
  else
   { ogg_stream_clear( &pvfh->ost );
     // > It is safe to call ogg_stream_clear on the same structure more than once.
   }




  // The following code was originally based on
  //     libvorbisXYZ/examples/encoder_example.c ,
  //     similar fragments are in Paul's vtvorbis.c :
  // > Add a (vorbis-) comment :
  vorbis_comment_init(&pvfh->vcomment);
  vorbis_comment_add_tag(&pvfh->vcomment,"ENCODER","VorbisFileIO.cpp");

  // > set up the analysis state and auxiliary encoding storage
  vorbis_analysis_init(&pvfh->vdsp,&pvfh->vinfo);
  pvfh->fMustClearDSP   = TRUE;
  vorbis_block_init(&pvfh->vdsp,   &pvfh->vblock);
  pvfh->fMustClearBlock = TRUE;


  // > Vorbis streams begin with three headers; the initial header (with
  // > most of the codec setup parameters) which is mandated by the Ogg
  // > bitstream spec.  The second header holds any comment fields.  The
  // > third header holds the bitstream codebook.  We merely need to
  // > make the headers, then pass them to libvorbis one at a time;
  // > libvorbis handles the additional Ogg bitstream constraints .
  // About vorbis_analysis_headerout :
  // > This function creates and returns the three header packets
  // > needed to configure a decoder to accept compressed data.
  // > Should be called after all encoder initialization and configuration is complete.
  // > The output packets should be placed in order at the start of the
  // > compressed vorbis stream, prior to the first data packet.
  vorbis_analysis_headerout(
        &pvfh->vdsp,    // [in] vorbis_dsp_state with the encoder configuration
        &pvfh->vcomment,// [in] vorbis_comment structure with the metadata
                        //      associated with the stream being encoded
        &pvfh->header,      // [out] ogg_packet to be filled out with the stream ID header
        &pvfh->header_comm, // [out] ogg_packet to be filled out with the serialied vorbis_comment data
        &pvfh->header_code);// [out] ogg_packet to be filled out with the codebooks,
                        //       mode descriptions, etc.
                        //       which will be used encoding the stream
  ogg_stream_packetin(&pvfh->osv,&pvfh->header); /* automatically placed in its own page */
  ogg_stream_packetin(&pvfh->osv,&pvfh->header_comm);
  ogg_stream_packetin(&pvfh->osv,&pvfh->header_code);
  // About ogg_stream_packetin() :
  // > This function submits a packet to the bitstream for page encapsulation.
  // > After this is called, more packets can be submitted, or pages can be written out.
  // > In a typical encoding situation, this should be used after filling
  // > a packet with data. The data in the packet is copied into the internal
  // > storage managed by the ogg_stream_state, so the caller is free
  // > to alter the contents of op after this call has returned. (..)

  // > ogg_stream_flush() checks for remaining packets inside the stream
  // > and forces remaining packets into a page, regardless of the size of the page.
  // > ! This should only be used when you want to flush an undersized page
  // > ! from the middle of the stream.
  // ( well, not really. In ??/libvorbis_1_3_1/examples/encoder_example.c ,
  //   ogg_stream_flush() is also called to flush the 'three headers' !
  while(1)
   { v_result = ogg_stream_flush( &pvfh->osv/*in*/, pOggPage/*out*/);
     if( v_result==0 ) // no more 'flushed' pages to write at the moment..
      { break;
      }
     WriteOggPage( pvfh, pOggPage ); // append to data in network buffer, but no FLUSH yet !
   }

  pvfh->fStoreInitialHeaders = FALSE;  // finished storing the initial headers for remote clients (in WriteOggPage)

  // After assembling Ogg pages in a buffer, flush the output to the "web" (socket)..
  //    .. but only if the encoder is already connected to anything out there.
  //       For SL's built-in audio server, the compressed data may be collected
  //       in a large FIFO (one writer, multiple readers) BEFORE any remote client
  //       has connected (the HTTP server). In that case, don't call INET_Write() from here.
  if( pvfh->fIsInternetStream && ( pvfh->InetClient.sock!=INVALID_SOCKET ) )
   { n = INET_Write(  //  FLUSH OUT the network buffer to the web
              &pvfh->InetClient,  // [in,out] T_INET_Client *pInetClient; a web client from YHF_Inet.c
              0,                  // [in] max. timeout in milliseconds, here 0 = non-blocking
              INET_WR_OPTION_FLUSH_NOW, // [in] options : flush out all data ("send now IF POSSIBLE")
              NULL, 0 );          // nothing more to add now
     if( n<0 )
      { CallBack_Error( pvfh, 0/*time*/, "Failed to send headers : %s", ErrorCodeToString( -n ) );
      }
   }

  // In ??/libvorbis_1_3_1/examples/encoder_example.c , the next action
  // would be calling vorbis_analysis_buffer() to fill in PCM samples .

  pvfh->fEncoderInit2 = FALSE;  // 'done' (no need to call InitEncoder2 again)

  return TRUE;

} // end EncoderInit2()


/***************************************************************************/
static BOOL EncodeTimestamp( // Converts DL4YHF's T_ChunkInfo to Paul's VT_OGG_BLK and adds it to the output
              T_VorbisFileHelper *pvfh,
              T_ChunkInfo *pChunkInfo,  // [in] precise sample rate and (GPS-) timestamp
              ogg_page    *pOggPage )   // [out] Ogg page (kind of container)
{
  struct VT_OGG_BLK vt_ogg_blk;
  ogg_packet opack;
  memset( &opack, 0, sizeof(opack) ); // don't leave anything to fate - clear garbage
  memset( &vt_ogg_blk, 0, sizeof(vt_ogg_blk) );

  pvfh->nSamplesOutputAtLastEncoderTimestamp = 0;
  pvfh->dblUnixTimeAtLastEncoderTimestamp = 0.0;



     // Retrieve timestamp, sample rate calibration, etc,
     // and put the info into a VT_OGG_BLK
     vt_ogg_blk.magic = BLK_MAGIC;
     vt_ogg_blk.secs = (int32_t)pChunkInfo->ldblUnixDateAndTime;
     vt_ogg_blk.nsec = (int32_t)(1e9*fmodl(pChunkInfo->ldblUnixDateAndTime, 1.0) );
     vt_ogg_blk.srcal = pChunkInfo->dblPrecSamplingRate / pvfh->dblNominalSampleRate;
     vt_ogg_blk.tbreak = 0;
     vt_ogg_blk.posn   = pvfh->i64NumSamplesWritten;
     vt_ogg_blk.spare1 = 0;
     vt_ogg_blk.spare2 = 0;
     vt_ogg_blk.glat  = pChunkInfo->gps.dblLat_deg;  // geographic latitude in degrees, positive = EAST, <-999 = INVALID
     vt_ogg_blk.glong = pChunkInfo->gps.dblLon_deg;  // geographic longitude in degrees, positive = NORTH
     vt_ogg_blk.gasl  = pChunkInfo->gps.dbl_mASL;    // meters above sea level (from GPS receiver)

     // Send the VT_OGG_BLK out on the timestamp stream
     opack.packet = (unsigned char *) &vt_ogg_blk;
     opack.bytes = sizeof( struct VT_OGG_BLK);
     opack.b_o_s = 0;
     opack.e_o_s = 0;
     opack.granulepos = pvfh->i64NumSamplesWritten;
     opack.packetno   = pvfh->vt_packetno++;
     ogg_stream_packetin( &pvfh->ost, &opack );
     ++pvfh->i64TimestampCounter; // here: count ENCODED, WRITTEN, or SENT timestamps, in EncodeTimestamp()

     // Send all available pages, this ensures that the VT_OGG_BLK gets
     // to the other end ahead of the vorbis data it refers to
     // From Paul:
     // > Make sure you are sending a VT_OGG_BLK timestamp packet
     // > before you send the corresponding audio.
     // The 'while' loop forces ogg to send this packet immediately.
     while( ogg_stream_pageout( &pvfh->ost, pOggPage) ||
            ogg_stream_flush( &pvfh->ost, pOggPage) )
      { WriteOggPage( pvfh, pOggPage );  // not really 'write and flush' yet..
            // we only prepage it in a network buffer
            // to avoid calling the SOCKET interface with very small chunks .
      }



  return TRUE;
} // end EncodeTimestamp()


/***************************************************************************/
BOOL C_VorbisFileIO::WriteHeader(
           T_ChunkInfo *pChunkInfo) // << precise sample rate, 'radio frequency', date and time, etc (must not be NULL)
  /* Writes the 'header' for the audio file (usually a RIFF WAVE file).
   *  This was once done immediately in OutOpen(), but since 2011-02
   *  writing the header is postponed until the first chunk of samples
   *  is available, and a PRECISE(!) timestamp is present in pChunkInfo .
   *  All relevant information has already been set in OutOpen(),
   *  but not written to the file yet (by the time we get here).
   */
{
   return TRUE;
} // end C_VorbisFileIO::WriteHeader()

/***************************************************************************/
BOOL C_VorbisFileIO::CloseAndReturnWithError(  char *pszFormat, ... )
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  va_list arglist;
  va_start(arglist, pszFormat);
  vsnprintf(m_cErrorString, sizeof(m_cErrorString)-1, pszFormat, arglist);
  va_end(arglist);
  if( pvfh!=NULL ) // bad luck; all 'vorbis file helpers' already in use ?!
   { CallBack( pvfh, AUDIO_FILE_IO_EVENT_TYPE_ERROR, m_cErrorString, 0 );
   }
  CloseFile( m_cErrorString/*pszWhy*/ );
  return FALSE;
} // end C_VorbisFileIO::CloseAndReturnWithError()


/***************************************************************************/
BOOL C_VorbisFileIO::OutOpen( // Opens a FILE, STREAM or just a buffer(*) for output / SENDING
                char* pszFilename,  // name of a local file, or URL of a remote server (signal sink) (*)
                int  options,       // AUDIO_FILE_OPTION_TIMESTAMPS , ... (bitflags)
                int channels,       // number of audio channels PER SAMPLE POINT
                int nominal_sample_rate, // [in] 'nominal' sample rate (44100, 48000, etc)
                float base_quality, // Desired quality level, currently from -0.1 to 1.0 (lo to hi).
                double dblStartTime )
  /* Creates and Opens an audio file (or, maybe, a stream) for WRITING .
   *         Returns TRUE on success and FALSE on any error.
   *         By default using Ogg/Vorbis format, otherwise (for non-compressed streams)
   *         call   SetParameterByID( AUDIO_FILE_PARAM__FILE_FORMAT,AUDIO_FILE_FORMAT_UNCOMPRESSED_STREAM)
   *            and SetVTFlags(VTFLAG_INT2) *BEFORE* OutOpen() .
   *
   * Input parameters..
   *   pszFilename:      USUALLY the name of a file, URL, or pseudo-URL.
   *                (*)  With pszFilename=NULL, the Vorbis compressor
   *                     doesn't send or store the data anywhere,
   *                     but keeps them in a FIFO from where
   *                       MULTIPLE READERS (usually via SL's HTTP server)
   *                     can 'pull them out'. Example:
   *                     SOUND_VorbisStreamOutputServer, configured and opened for output
   *                     in cbproj/SpecLab/AudioStreamServer_GUI.cpp : StartAudioWebstreamServer() .
   *
   *   base_quality:     Desired quality level, currently from -0.1 to 1.0
   *                     (lo to hi). A good starting point seems to be 0.5  .
   *   channels:         1=mono, 2=stereo recording etc
   *   sample_rate:      number of samples per second.
   *                     Beware that Ogg Vorbis only supports a limited set
   *                     of sample rates !
   */
{
  // For details, see the Ogg Vorbis / "libvorbisenc" documentation .
  // Don't miss http://www.xiph.org/vorbis/doc/vorbisenc/examples.html
  //      and   http://www.xiph.org/vorbis/doc/vorbisenc/overview.html
  //
  // Some experiences by Paul Nicholson, 2010-06-02 :
  // > PS, I switched to using variable bitrate encoding which
  // > appears to remove the artifacts I noticed, and also reduces
  // > the CPU load considerably - now about the same as libmp3lame.
  // >
  // > Was calling:
  // >    vorbis_encode_init( &vi, chans, sample_rate, -1, bps, -1)
  // >
  // > Now calls instead:
  // >    Q = 0.5; // quality factor, -0.1 (low) to 1.0 (high)
  // >    vorbis_encode_init_vbr( &vi, chans, sample_rate, Q)
  // >
  // > See http://www.xiph.org/vorbis/doc/vorbisenc/vorbis_encode_init_vbr.html
  // >
  // > Seems like the encoder had to work extra hard to produce the
  // > constant bitrate stream. With reasonable amount of buffer,
  // > VBR should be no problem at all, since we don't have to run
  // > in 'real time' or worry about latency.
  // >
  // > Currently encoding a 5-channel stream at 32k frames/sec,
  // > uses 13% of 2GHz CPU.
  // ex: vorbis_info vi;  // moved into a static structure (see notes below)
  int v_result;    // vorbis result code / error code; 0 = "no error"
  int n;  // number of bytes, or result code used in YHF_inet.h (def'd in c:\cbproj\SoundUtl\ErrorCodes.h)
  int coupling_mode;
  BOOL ok = FALSE;
  T_VorbisFileHelper *pvfh; // contains everything which shouldn't go into VorbisFileIO.h
  // ogg_page opage; // ogg_page op only contains a POINTER to the page's netto data,
                  // it's a small object so it's ok to use a local stack var here.

  CloseFile( "to re-open output"/*pszWhy*/ ); // Close "old" file if it was open
  m_cErrorString[0] = 0;                      // forget the old error string

  if( m_pVorbisFileHelper == NULL )
   { // allocate one of these 'helpers' from a pool (in fact a simple array):
     m_pVorbisFileHelper = VorbisFileHelper_Alloc(this);
   }
  pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh==NULL ) // bad luck; all 'vorbis file helpers' already in use ?!
   { return CloseAndReturnWithError( "Could not allocate vorbis 'file helper'" );
   }
  pvfh->dblStartTime = dblStartTime;
  pvfh->nChannelsPerSampleInStream = channels; // save for later ..
  pvfh->dblNominalSampleRate = nominal_sample_rate;
  pvfh->fPlainOggVorbis = FALSE;
  pvfh->nSamplesOutputAtLastEncoderTimestamp = 0;
  pvfh->dblUnixTimeAtLastEncoderTimestamp = 0.0;
  pvfh->nRcvdVorbisHeaders   = 0;
  pvfh->nSamplesPerHeaderInStream = 0; // use the same size as the feeding-chunk-size, as soon as we know it
  pvfh->inetReadOptions = INET_RD_OPTION_WAIT_FOR_ALL_DATA; // for VorbisRead() : don't return until the requested amount of bytes was received
  pvfh->inetTimeout_ms  = 3000;   // initial timeout value for VorbisRead() when used with 'internet streams'
  pvfh->iSampleCountdownBetweenHeaders = 0; // here in OutOpen() : emit the initial header or VT_BLOCK a.s.a.p. !
  pvfh->vt_packetno          = 0;

  pvfh->i64NumSamplesWritten = 0;

  pvfh->i64NumSamplesWrittenAtLastTimestamp = 0;

  pvfh->nBitsPerSecond = pvfh->nBitrateCounter = 0;

  pvfh->dblStartTimeOfBitrateMeasurement = 0.0;
  pvfh->fOpenedForReading = FALSE;

  pvfh->fOpenedForWriting = FALSE;
  pvfh->fMustClearVorbisInfo = FALSE;  // not yet.. flag polled in CloseFile()
  pvfh->fMustClearDSP   = FALSE;  // safe to do here because we called CloseFile() !
  pvfh->fMustClearBlock = FALSE;
  pvfh->fMustClearComment=FALSE;
  pvfh->have_vorbis = FALSE;
  pvfh->want_tstamp = FALSE;
  pvfh->have_tstamp = FALSE;
  if( options & AUDIO_FILE_OPTION_TIMESTAMPS )
   {  pvfh->want_tstamp = TRUE;  // WANT timestamps, but haven't got a valid one for the encoder yet !
   }

  m_ldblPrevCheckedTimestamp = 0.0; // only used for DEBUGGING (plausibility check)
  pvfh->i64TimestampCounter  = 0;
  m_dwCurrFilePos = m_dwFileSizeInBytes = 0;


  // Create the output file,  or open a connection to the remote stream server
  //     (with THIS SIDE acting as the client),
  // or just init the buffer/compressor for "one writer, multiple readers"
  //     (with THIS SIDE being used for a server) ?
  pvfh->fStoreSamplesForMultipleReaders = FALSE;
  if( pszFilename==NULL ) // don't write a file or establish a connection, but..
   { pvfh->fIsInternetStream = TRUE; // it's not a disk file but a "stream" (encoded IN MEMORY only)
     pvfh->fStoreSamplesForMultipleReaders = TRUE;  // here: flag set in OutOpen() with pszFilename==NULL
     pvfh->fOpenedForWriting = TRUE; // say 'open for business' even without any READER
     ok = TRUE;
   }
  else // Let YHF_Inet.c find out if this is an INTERNET STREAM or a LOCAL FILE:
  if( INET_IsInternetResource( pszFilename ) )
   { // It's an internet stream, not a local file,
     //      for example http://67.207.143.181:80/vlf1  --->
     pvfh->fIsInternetStream = TRUE; // it's not a disk file but a "stream" (encoded IN MEMORY only)
     CallBack_Info( pvfh, "Trying to connect %s", pszFilename );
     n = INET_OpenAndReadFromServer( // here: client side, going to "push" audio to a remote server (!)
             &pvfh->InetClient, // [in,out] T_INET_Client *pInetClient; web client data
              pszFilename,  // [in] filename, URL, or pseudo-URL a la Spectrum Lab ("rawtcp:\\blabla:port")
              5000,         // [in] max. timeout in milliseconds
              INET_RD_OPTION_RETURN_AFTER_RESPONSE, // [in] options (from YHF_Inet.h)
              NULL, 0 );    // [in,out] destination buffer (but nothing to READ here)
     if( n < 0 )
      { return CloseAndReturnWithError( "Couldn't open URL: %s", ErrorCodeToString(-n) );
      }
     else
      { CallBack_Info( pvfh, "Connected to %s", pszFilename );
        ok = TRUE;  // successfully opened the connection and read the 'response headers'
                    // (but not the "contents" yet)
        pvfh->fOpenedForWriting = TRUE;
      }
   }
  else
   { // it's not an (internet-) stream but a LOCAL FILE :
#    if (VORBIS_USE_QFILE)    // use "QFile" or standard file I/O from the RTL ?
        ok = QFile_Create(&pvfh->m_QFile, pszFilename, 0 );
#    else
#       error "other file access methods are not supported yet !"
#    endif  // ! VORBIS_USE_QFILE
        if( ok )   // ok, the (disk-)file has been opened.
         { pvfh->fOpenedForWriting = TRUE;  // here: LOCAL FILE (not internet stream)
         } // end if < LOCAL file has been opened >
   } // end if < STREAM or local FILE > ?

  if( pvfh->fOpenedForWriting )
   { // Write an 'audio log' with the compressed audio (similar to Oddcast's log) ?
    if( pvfh->fWriteLogfile && pvfh->sz255StreamLogfile[0]!=0 )
     { if( ! QFile_Create( &pvfh->qfStreamLog, pvfh->sz255StreamLogfile, 0 ) )
        { strcpy( m_cErrorString, "Couldn't create stream log" );
          CallBack_Error( pvfh, 0/*time*/, m_cErrorString );
          ok = FALSE;  // don't bail out in this case..
        }
       else
        { CallBack_Info( pvfh, "Created audio logfile"  );
        }
     }
  }


  if( ! ok )
   { return FALSE;
   }
  if( pszFilename != NULL )
   {  strncpy(m_sz255FileName, pszFilename, 255);   // this may be a pseudo-URL like "rawtcp://127.0.0.1:12345" !
   }
  else
   {  m_sz255FileName[0] = '\0';  // here: no file but an endless stream (encoded chunk-wise in memory, for the HTTP server)
   }

  // The 'stream format' should have already been set
  //     (by the caller, calling C_VorbisFileIO::SetParameterByID( AUDIO_FILE_PARAM__FILE_FORMAT,) );
  // if not, set the 'stream format' here:
  if( pvfh->iStreamFormat==STREAM_FORMAT_UNKNOWN )
   {  pvfh->iStreamFormat=STREAM_FORMAT_OGG_VORBIS; // "default if not yet speficied"
   }


  switch( pvfh->iStreamFormat )
   { case STREAM_FORMAT_OGG_VORBIS:   // start writing/sending/buffering Ogg/Vorbis ...
     default:
        vorbis_info_init(&pvfh->vinfo);

        // Workflow (using libvorbisenc, quoted from doc/vorbisenc/overview.html
        // > Libvorbisenc is used only during encoder setup; its function is
        // > to automate initialization of a multitude of settings in a
        // > vorbis_info structure which libvorbis then uses as a reference
        // > during the encoding process.
        // > Libvorbisenc plays no part in the encoding process after setup.
        // >
        // > Encode setup using libvorbisenc consists of three steps:
        // >
        // >  1. high-level initialization of a vorbis_info structure
        // >     by calling one of vorbis_encode_setup_vbr() [....]
        // >  2. optional adjustment of the basic setup defaults
        // >     using vorbis_encode_ctl()
        // >  3. calling vorbis_encode_setup_init() to finalize the high-level setup
        // >     into the detailed low-level reference values needed by libvorbis
        // >     to encode audio. The vorbis_info structure is then ready to use
        // >     for encoding by libvorbis.
        // (so better do NOT use a local 'stack' variable for vorbis_info - WB)
        // > These three steps can be collapsed into a single call by using
        // > vorbis_encode_init_vbr  to set up a quality-based VBR stream [....] .
        // The latter is used here, for an easy start :

        // Prepare encoding using a VBR quality mode.  The usable range is -.1
        // (lowest quality, smallest file) to 1.0 (highest quality, largest file).
        // Example quality mode .4: 44kHz stereo coupled, roughly 128kbps VBR
        // EX: v_result = vorbis_encode_init_vbr( &pvfh->vinfo, // [in] vorbis_info *vi,
        //                    channels, nominal_sample_rate, base_quality);
        //  Removed 2011-12-21 because vorbis_encode_init_vbr() internally calls
        //          vorbis_encode_setup_init(), and after that,
        //          vorbis_encode_ctl() is impossible. Use ..._.._setup_vbr() instead:
        v_result = vorbis_encode_setup_vbr( &pvfh->vinfo, // [in] vorbis_info *vi,
                        channels, nominal_sample_rate, base_quality);

        // Do not continue if setup failed; this can happen if we ask for a
        // mode that libVorbis does not support (eg, too low a quality mode, etc,
        // will return 'OV_EIMPL')
        if( v_result != 0 )
         { return CloseAndReturnWithError( "Error in vorbis_encode_init: %s",
                          (char*)VorbisErrorCodeToString( v_result ) );
         }

        // Call vorbis_encode_ctl(), as in Paul's vlfrx-tools-0.3e\vtvorbis.c :
        // > This function implements a generic interface to miscellaneous
        // > encoder settings similar to the clasasic UNIX 'ioctl()' system call.
        // With int request = OV_ECTL_COUPLING_SET :
        // >  Argument: int *
        // >  Enables/disables channel coupling in multichannel encoding according to arg.
        // >  *arg of zero disables all channel coupling, nonzero allows the encoder
        // >  to use coupling if a coupled mode is available for the input.
        // >  At present, coupling is available for stereo and 5.1 input modes.
        if( vorbis_encode_ctl( &pvfh->vinfo, OV_ECTL_COUPLING_GET, &coupling_mode) < 0)
         { return CloseAndReturnWithError( "vorbis_encode_ctr(OV_ECTL_COUPLING_GET) failed" );
         }
        if( coupling_mode != 0 )  // we want INDEPENDENT encoding of the channels, so:
         {
            coupling_mode = 0;
            if( vorbis_encode_ctl( &pvfh->vinfo, OV_ECTL_COUPLING_SET, &coupling_mode) < 0)
             { return CloseAndReturnWithError( "OV_ECTL_COUPLING_SET failed" );
               // 2011-12-21: 'OV_ECTL_COUPLING_SET' failed because some odd flag
               //             named 'set_in_stone' was already set in vorbis_encode_ctl().
               //  Reason: vorbis_encode_setup_init() had already been called,
               //          INTERNALLY, camouflaged in vorbis_encode_init_vbr() !
             }
            if( vorbis_encode_ctl( &pvfh->vinfo, OV_ECTL_COUPLING_GET, &coupling_mode) < 0)
             { return CloseAndReturnWithError( "OV_ECTL_COUPLING_GET failed");
             }
            if( coupling_mode != 0)
             { return CloseAndReturnWithError( "failed to set independent encoding");
             }
         }

        // > vorbis_encode_setup_init() : performs the last stage of three-step encoding setup,
        // > as described in the API overview under managed bitrate modes.
        // > Before this function is called, the vorbis_info struct should be initialized
        // > by using vorbis_info_init() from the libvorbis API, (....) .
        // > vorbis_encode_setup_init() finalizes the highlevel encoding structure
        // > into a complete encoding setup after which the application may make no further setup changes.
        // > After encoding, vorbis_info_clear should be called.
        pvfh->fMustClearVorbisInfo = TRUE;  // see comment above
        if( vorbis_encode_setup_init( &pvfh->vinfo) < 0)
         { return CloseAndReturnWithError( "vorbis_encode_setup_init failed");
         }

        pvfh->fEncoderInit2 = TRUE;  // added 2011-12-27 : "postponed" the rest..
        // ex: Setup one or two stream encoders, stream 0 for vorbis, stream 1 for timestamps
        // All this happens 'a bit later' now, after the GPS decoder spit out
        //    the first GPS timestamps. See EncoderInit2() .
        break; // end case  pvfh->iStreamFormat == STREAM_FORMAT_OGG_VORBIS

     case STREAM_FORMAT_NON_COMPRESSED:   // start writing/sending/buffering NON-COMPRESSED file/stream
        // Nothing else to do here, nothing to write/send yet
        // because we don't have a timestamped header (to send in a VT_BLOCK) yet !
        // Headers are assembled and sent in C_VorbisFileIO::WriteSamples_Float().
        // Or (with pszFilename=NULL), if VorbisFileIO is used for BUFFERING
        // audio as the SERVER for MULTIPLE CLIENTS, nothing else will happen
        // until the first client connects our server, and asks for samples
        // via ____ .
        // 
        break; // end case  pvfh->iStreamFormat == STREAM_FORMAT_NON_COMPRESSED

   } // end switch < 'stream format' >


  return pvfh->fOpenedForWriting;

} // C_VorbisFileIO::OutOpen(..)


/***************************************************************************/
LONG C_VorbisFileIO::WriteSamples_Float( // app-layer output, using NORMALIZED floats
       T_Float *pfltSrc,     // [in] caller's source buffer, 32-bit floats, possibly interleaved
       T_ChunkInfo *pChunkInfo, // [in] number of samples, channels, precise sample rate,
                                //      scaling range, 'radio frequency', date and time, etc
          // btw, T_ChunkInfo is defined in c:\cbproj\SoundUtl\ChunkInfo.h .
       char * pszExtraInfo ) // [in] extra 'info' (usually 'data') added to info chunk
  /* Writes some samples in the 'normalized' floating point format (-1.0 ... +1.0)
   *       to an audio file which has been opened for WRITING;
   *       or to an OUTGOING audio stream .
   * Returns the number of SAMPLE POINTS written,
   *       or a NEGATIVE value if there was an error.
   */
{
  T_VorbisFileHelper *pvfh;
  int v_result;    // vorbis result code / error code; 0 = "no error"
  int i, n;
  int iChannel, nSamplePoints;
  int nSrcChannelsPerSample, nDstChannelsPerSample, nDstBytesPerSample, nCommonChannelsPerSample;
  int iSample, nSamplesInVorbisBlock;
  double d;
  char sz80[84];
  T_Float fltScalingFactor;
  T_Float *pfltSource2;
  float **ppfltBuffer;  // not T_Float
  T_StreamPtrUnion pDst;
  LONG nSamplesWritten = 0;
  ogg_page opage; // ogg_page only contains a POINTER to the page's netto data,
                  // it's a small object so it's ok to use a local stack var here.
  ogg_packet opack;
  memset( &opage, 0, sizeof(opage) ); // don't leave anything to fate - clear garbage
  memset( &opack, 0, sizeof(opack) );

  if( pChunkInfo == NULL )  // T_ChunkInfo is MANDATORY now !
   { return -1;
   }
  pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh==NULL )
   { return -2;
   }

  if( pvfh->fIsInternetStream && pvfh->InetClient.fConnectionLost ) // connection lost !
   { // Don't try to push more samples into Ogg/Vorbis !
     return CloseAndReturnWithError( "Stream connection lost !" );
   }

  nSamplePoints         = (int)pChunkInfo->dwNrOfSamplePoints;
  nSrcChannelsPerSample = pChunkInfo->nChannelsPerSample;
  nDstChannelsPerSample = pvfh->nChannelsPerSampleInStream;
  nCommonChannelsPerSample = nSrcChannelsPerSample; // who knows.. caller may deliver less than initially expected
  if( nCommonChannelsPerSample > nDstChannelsPerSample )
   {  nCommonChannelsPerSample = nDstChannelsPerSample;
   }
  if(  pChunkInfo->dblSampleValueRange > 0.0)
        fltScalingFactor = 1.0 / pChunkInfo->dblSampleValueRange;
  else  fltScalingFactor = 1.0;  // wild guess (!) .. see notes in \cbproj\SoundUtl\ChunkInfo.h

#if(1)   // plausibility check for the timestamps (sent to file or stream) ?
         // Modified 2013-11-24: If there's no GPS receiver at all,
         //  the timestamps don't need to be "previse",
         //  and the sampling rate doesn't need to be "calibrated" :
  if( ( (pChunkInfo->dwValidityFlags &
             ( CHUNK_INFO_SAMPLE_RATE_VALID     /*0x02*/
              |CHUNK_INFO_SAMPLE_RATE_CALIBRATED/*0x04*/
              |CHUNK_INFO_TIMESTAMPS_VALID      /*0x08*/
              |CHUNK_INFO_TIMESTAMPS_PRECISE    /*0x10*/) )
          == ( CHUNK_INFO_SAMPLE_RATE_VALID|CHUNK_INFO_SAMPLE_RATE_CALIBRATED
              |CHUNK_INFO_TIMESTAMPS_VALID |CHUNK_INFO_TIMESTAMPS_PRECISE) )
    ||( (pChunkInfo->dwValidityFlags & CHUNK_INFO_PRECISION_NOT_AVAILABLE/*0x20*/ ) != 0 ) )
         // .. will NEVER be 'CALIBRATED'/'PRECISE' because, for example, because no GPS is connected at all
   {
     if(   (m_ldblPrevCheckedTimestamp > 0 )
        &&((pChunkInfo->dwValidityFlags & CHUNK_INFO_PRECISION_NOT_AVAILABLE/*0x20*/ ) == 0 ) )
      { d = pChunkInfo->ldblUnixDateAndTime - m_ldblPrevCheckedTimestamp;
        if( (d< -20e-6) || (d> 20e-6) )
         {
           sprintf(sz80, "WriteSamples: TS gap, dt=%.3lf sec", (double)d );
           CallBack_Error( pvfh, 0/*not pChunkInfo->ldblUnixDateAndTime*/, sz80 );
           // The error emitted via CallBack_Error() should eventually appear on the 'Event Log'
           //    (for the 'pushing' CLIENT) in c:\cbproj\SpecLab\AudioStreamOut_GUI.cpp,
           // or (for the SERVER/provider)  in c:\cbproj\SpecLab\AudioStreamServer_GUI.cpp.
           VFIO_DebuggerBreak( pvfh, sz80, __LINE__ );    // <<< set breakpoint here <<<
              // added 2012-01-08, after observing 'jumps' in the streamed timestamps
              //                   by 512 ms every 5 minutes.
              //    Multi-threading issue / using a T_ChunkInfo in SoundDec.c
              //    which is currently being overwritten ?
              //    More in C:\CBproj\SoundUtl\SoundDec.cpp .
              //
         }
      }
    m_ldblPrevCheckedTimestamp = pChunkInfo->ldblUnixDateAndTime
      + (long double)pChunkInfo->dwNrOfSamplePoints / pChunkInfo->dblPrecSamplingRate;
   }
  else // timestamps declared 'invalid', don't check them :
   { m_ldblPrevCheckedTimestamp = 0;
   }
#endif   // plausibility check for the timestamps ?


  if( pvfh->want_tstamp )  // WANT to send timestamps ?
   { // Do we HAVE valid timestamps now (passed in pChunkInfo) ?
     if( pChunkInfo->dwValidityFlags & CHUNK_INFO_TIMESTAMPS_VALID )
      { pvfh->have_tstamp = TRUE; // ok, ready for the timestamped stream.
        // Will call the postponed 'EncoderInit2' now, send the Vorbis header,
        // and also send the first timestamp-data a bit further below.
      }
   }


  // Call the 'postponed' part of the Ogg stream initialisation now ?
  if( pvfh->iStreamFormat==STREAM_FORMAT_OGG_VORBIS)
   { if( pvfh->fEncoderInit2 )
      { if(  ( pvfh->want_tstamp && pvfh->have_tstamp )
          || (!pvfh->want_tstamp) )
         { EncoderInit2( pvfh, pChunkInfo/*in*/, &opage/*out*/ );  // also SENDS the first data !
         }
      }
     if( pvfh->fEncoderInit2 ) // still waiting for the 1st timestamp,
      { strcpy( m_cErrorString, "Cannot send/write, invalid timestamp" );
        CallBack_Error( pvfh, pChunkInfo->ldblUnixDateAndTime, m_cErrorString );
        return 0;              // cannot emit audio samples yet !
        // (to avoid this error message, THE CALLER shouldn't call us if he
        //  "wants" timestamps but has none yet. See cbproj/SpecLab/SoundThd.cpp .
      }
     n = (int)(0.5+ pvfh->dblNominalSampleRate);
     if( n<16384 ) n = 16384;
     if( pvfh->want_tstamp     // WANT to send timestamps ?
       &&(   (pvfh->i64NumSamplesWritten == 0) // 'very first' timestamp ?
           ||(pvfh->i64NumSamplesWritten >= pvfh->i64NumSamplesWrittenAtLastTimestamp + n)  // or 'time for another timestamp' ?
           ||(pvfh->i64NumSamplesWritten <  pvfh->i64NumSamplesWrittenAtLastTimestamp) ) )
      { EncodeTimestamp( pvfh, pChunkInfo/*in*/, &opage/*out*/ );
        pvfh->i64NumSamplesWrittenAtLastTimestamp = pvfh->i64NumSamplesWritten;
      }

   } // end if( pvfh->iStreamFormat==STREAM_FORMAT_OGG_VORBIS)


  // Would it be possible to write / send  audio samples now ?
  if (  (!pvfh->fOpenedForWriting )
     || (nSamplePoints <= 0 )
     || (nCommonChannelsPerSample <= 0)
     || (pChunkInfo->dblSampleValueRange <= 0.0)  )
   { strcpy( m_cErrorString, "Cannot send, invalid params" );
     CallBack_Error( pvfh, pChunkInfo->ldblUnixDateAndTime, m_cErrorString );
     return -3;
   }



#if(0)
  // Also emit a new GPS position to the file ? (since 2010-07-07)
  if(  (m_iPosUpdateCounter_WR != pChunkInfo->gps.iUpdateCounter )
        &&( m_dblInfoWriteInterval_sec != 0 )
        &&( qfAuxFile.fOpened )    )
   {
     WriteChunkInfo( m_dwSamplePointsWritten/*index*/, pChunkInfo, pszExtraInfo );
     m_iPosUpdateCounter_WR = pChunkInfo->gps.iUpdateCounter;
   } // end if < periodically emit 'position info' (with GPS data, etc) ?
#endif


  if( pvfh->iStreamFormat==STREAM_FORMAT_OGG_VORBIS)
   {
     // Push the samples into Vorbis' own buffer (later used as input for encoding)
     while( nSamplePoints > 0 )
      { nSamplesInVorbisBlock = (nSamplePoints>1024) ? 1024 : nSamplePoints;
        nSamplePoints -= nSamplesInVorbisBlock;

        ppfltBuffer = vorbis_analysis_buffer( &pvfh->vdsp, nSamplesInVorbisBlock );
        // About vorbis_analysis_buffer() :
        // > requests a buffer array for delivering audio to the encoder for compression.
        // > The Vorbis encoder expects the caller to write audio data
        // > as non-interleaved floating point samples into its internal buffers.
        // > The general procedure is to call this function with
        // > the number of samples you have available ("1024 is a reasonable choice").
        // > The encoder will arrange for that much internal storage
        // > and return an array of buffer pointers, one for each channel of audio.
        // > The caller must then write the audio samples into those buffers,
        // > as float values, and finally call vorbis_analysis_wrote()
        // > to tell the encoder the data is available for analysis.
        // > Returns a **float where the first index is the channel,
        // >                    and the second is the sample index.
        iChannel=0;
        while( iChannel<nCommonChannelsPerSample )
         { if( (pDst.pF32 = ppfltBuffer[iChannel]) != NULL )
            { pfltSource2 = pfltSrc + iChannel;  // source: INTERLEAVED, float or double
              for( iSample=0; iSample<nSamplesInVorbisBlock; ++iSample )
               { *pDst.pF32++ = *pfltSource2;  // destination: non interleaved
                 pfltSource2 += nSrcChannelsPerSample;
               }
            }
           ++iChannel;
         }
        // If the caller now delivers less channels than specified in OutOpen(),
        //  fill the remaining channels with silence :
        while( iChannel<nDstChannelsPerSample )
         { if( (pDst.pF32=ppfltBuffer[iChannel]) != NULL )
            { for( iSample=0; iSample<nSamplesInVorbisBlock; ++iSample )
               { *pDst.pF32++ = 0.0;
               }
            }
           ++iChannel;
         }
        // Inform the Vorbis encoder that we have finished writing into the buffer:
        v_result = vorbis_analysis_wrote( &pvfh->vdsp, nSamplesInVorbisBlock );
        if( v_result<0 ) // failure !
         { nSamplesWritten = 0;
           snprintf( m_cErrorString, sizeof(m_cErrorString)-1,
                    "Cannot send: %s", (char*)VorbisErrorCodeToString( v_result ) );
           CallBack_Error( pvfh, pChunkInfo->ldblUnixDateAndTime, m_cErrorString );
           break;
         }
        else // ok, the Vorbis encoder has 'accepted' the buffered data
         { nSamplesWritten += nSamplesInVorbisBlock;
           pvfh->i64NumSamplesWritten += nSamplesInVorbisBlock;
         }

        // prepare the (interleaved) source-pointer for the next loop:
        pfltSrc += nSamplesInVorbisBlock * nSrcChannelsPerSample;
      } // end while( nSamplePoints > 0 )

     // Encode what we can (in the Vorbis audio file or stream) .
     //    About vorbis_analysis_blockout :
     // > examines the available uncompressed data and tries to break it
     // > into appropriate sized blocks. It should be called in a loop
     // > after adding new data with vorbis_analysis_buffer()/vorbis_analysis_wrote()
     // > until it returns zero (need more data) or an negative value (error).
     // > Each block returned should be passed to vorbis_analysis() for transform and coding.
     // > Returns 1 for "success, more blocks are available",
     // >         0 for "success, this is the last block from the current input",
     // >         negative values for failure  (but nobody seems to care)
     while( vorbis_analysis_blockout( &pvfh->vdsp, &pvfh->vblock) == 1)
      {
        v_result = vorbis_analysis( &pvfh->vblock, NULL);  // 2011-12-22: Crashed with access violation (in analysis.c -> psy.c; floor[0] corrupted)
        // -> Once the uncompressed audio data has been divided into blocks,
        //  > vorbis_analysis() is called on each block. It looks up
        //  > the encoding mode and dispatches the block to the
        //  > forward transform provided by that mode. (..)
        v_result = vorbis_bitrate_addblock( &pvfh->vblock);
        // -> ..addblock() submits a transformed block to the bitrate management engine
        //  > for final encoding. Packets are buffered and the packet boundaries
        //  > adjusted and padded to meet the target bitrate, if any.
        //  > After calling vorbis_bitrate_addblock(), the passed vorbis_block
        //  > structure can be reused in another call to vorbis_analysis_blockout().
        //  > Call vorbis_bitrate_flushpacket() to obtain the final compressed data.
        while( ( v_result = vorbis_bitrate_flushpacket( &pvfh->vdsp, &opack )) > 0 )
         {
           // -> ..flushpacket() returns the next available completed packet from the
           //  > bitrate management engine. It should be called in a loop
           //  > after any call to vorbis_bitrate_addblock() until it returns
           //  > either 0 (more data needed) or a negative value (error).
           //  > The data returned in the ogg_packet structure can be copied
           //  > to the final compressed output stream.

           ogg_stream_packetin( &pvfh->osv, &opack);
           // About ogg_stream_packetin() :
           // > This function submits a packet to the bitstream for page encapsulation.
           // > After this is called, more packets can be submitted, or pages can be written out.
           // > In a typical encoding situation, this should be used after filling
           // > a packet with data. The data in the packet is copied into the internal
           // > storage managed by the ogg_stream_state, so the caller is free
           // > to alter the contents of op after this call has returned. (..)

           // write out pages (if any)
           while( ogg_stream_pageout( &pvfh->osv, &opage) )
            { WriteOggPage( pvfh, &opage );
            }

         } // end while( .. vorbis_bitrate_flushpacket() .. "should be called in a loop".... )
      }
     // Now, after assembling a couple of Ogg pages in a buffer (in YHF_Inet),
     //      FLUSH the TCP-buffer to send that buffer to the network layer ?
     // ( see YHF_Inet.h : char cTxBuffer[INET_TX_BUFFER_SIZE] )
     if(   pvfh->fIsInternetStream
       &&( (pvfh->InetClient.nBytesInTxBuffer >= INET_TX_BUFFER_SIZE/2) // <<< mod. 2012-04-24: Send LARGE BLOCKS if possible !
         ||(pvfh->InetClient.nBytesInTxBuffer >= 8192 ) )
       )
      { n = INET_Write(  // FLUSH OUT the network buffer....
              &pvfh->InetClient,  // [in,out] T_INET_Client *pInetClient; a web client from YHF_Inet.c
              0,                  // [in] max. timeout in milliseconds, here: 0 = "don't block" (*)
              INET_WR_OPTION_FLUSH_NOW, // [in] options : FLUSH/actually send the internal 'network' buffer
              NULL, 0 );          // nothing more to add now
        // (*) with non-blocking call, YHF_Inet.c push new data into a large buffer,
        //     until send() is ready to send the next block.
        // If the connection is slow, that block will be VERY large -> high latency.
        // If the connection is fast, each send() success, and many SMALL blocks are sent.
        // The effect (output buffer usage in percent) can be seen in SL's AudioStreamOut_GUI .
        if( n<0 )
         { CallBack_Error( pvfh, pChunkInfo->ldblUnixDateAndTime, "Failed to send samples : %s", ErrorCodeToString( -n ) );
         }
      }
   } // end if( pvfh->iStreamFormat==STREAM_FORMAT_OGG_VORBIS)
  else  // added 2014-04-27 : Send a NON-COMPRESSED stream, with 'headers' to indicate the format ?
  if( pvfh->iStreamFormat==STREAM_FORMAT_NON_COMPRESSED)
   { // Encode samples into INT8/16/32 or FLOAT32/64 for the output,
     //   and insert headers as necessary :
     nDstBytesPerSample = GetBytesPerSampleInUncompressedStream(pvfh); // f(data type, number of channels)
     if( pvfh->nSamplesPerHeaderInStream < 16 )
      {  pvfh->nSamplesPerHeaderInStream = pChunkInfo->dwNrOfSamplePoints;
      }
     for(iSample=0; iSample<nSamplePoints; ++iSample )
      {
        // Time to send the next header now ?
        --pvfh->iSampleCountdownBetweenHeaders;
        if( pvfh->iSampleCountdownBetweenHeaders <= 0 )
         { pvfh->iSampleCountdownBetweenHeaders = pvfh->nSamplesPerHeaderInStream;
           // Flush the previous data (if any),
           //   so for UDP, the next header is at the begin of a datagram.
           if( pvfh->dwBufferIndex > 0 )
            { if( ! VFIO_Write( pvfh, pvfh->bBuffer, pvfh->dwBufferIndex, INET_WR_OPTION_FLUSH_NOW) )
               { // CallBack_Error() : already called by VFIO_Write() .
                 break;
               }
            }
           pvfh->dwBufferIndex = VFIO_ChunkInfoToStreamHeader(
                            pvfh, // [in] pvfh->iVTFlags, pvfh->nChannelsPerSampleInStream,
                                  //      pvfh->nSamplesPerHeaderInStream, ...
                      pChunkInfo, // [in]  chunk info with timestamp, etc etc etc
                         iSample, // [in] sample offset, required to adjust the timestamp
            (T_StreamHdrUnion*)pvfh->bBuffer); // [out] stream header + VT_BLOCK
           pvfh->i64NumSamplesWrittenAtLastTimestamp = pvfh->i64NumSamplesWritten;

         } // end if( pvfh->iSampleCountdownBetweenHeaders <= 0 )

        // If the remaining space in the output buffer is too low for appending another sample,
        //  flush it to the file/stream BEFORE a complete block of samples is finished:
        if( (pvfh->dwBufferIndex + nDstBytesPerSample) > BUFFER_MAX_BYTES )
         { if( ! VFIO_Write( pvfh, pvfh->bBuffer, pvfh->dwBufferIndex, INET_RD_OPTION_NORMAL) )
            { // already called in VFIO_Write: CallBack_Error()
              break;
            }
           pvfh->dwBufferIndex = 0;
         }

        // Copy (or convert) the next sample-point (with N channels) into the transmit-buffer:
        iChannel=0;
        pDst.pU8 = pvfh->bBuffer + pvfh->dwBufferIndex;
        switch( pvfh->iVTFlags & VTFLAG_FMTMASK )
         {
           case VTFLAG_FLOAT4 : // 4 byte floats (IEEE single precision)
              while( iChannel<nCommonChannelsPerSample )
               { pDst.pF32[iChannel] = pfltSrc[iChannel];
                 iChannel++;
               }
              // Fill the remaining output channels with silence (if any)
              while( iChannel<nDstChannelsPerSample )
               { pDst.pF32[iChannel++] = 0.0;
               }
              break;
           case VTFLAG_FLOAT8 : // 8 byte floats (IEEE double precision)
              while( iChannel<nCommonChannelsPerSample )
               { pDst.pF64[iChannel] = pfltSrc[iChannel];
                 iChannel++;
               }
              while( iChannel<nDstChannelsPerSample )
               { pDst.pF64[iChannel++] = 0.0;
               }
              break;
           case VTFLAG_INT1   : // 1 byte signed int, range +/- 127 (not -128..+127!)
              while( iChannel<nCommonChannelsPerSample )
               { i = (int)( 127.0 * pfltSrc[iChannel] );
                 if( i>127 ) i=127;
                 if( i<-127) i=-127;
                 pDst.pI8[iChannel] = (signed char)i;
                 iChannel++;
               }
              while( iChannel<nDstChannelsPerSample )
               { pDst.pI8[iChannel++] = 0;
               }
              break;
           case VTFLAG_INT2   : // 2 byte signed integers, little endian
              while( iChannel<nCommonChannelsPerSample )
               { i = (int)( 32767.0 * pfltSrc[iChannel] );
                 if( i>32767 ) i= 32767;
                 if( i<-32768) i=-32767;
                 pDst.pI16[iChannel] = (short int)i;
                 iChannel++;
               }
              while( iChannel<nDstChannelsPerSample )
               { pDst.pI16[iChannel++] = 0;
               }
              break;
           case VTFLAG_INT4   : // 4 byte signed integers, little endian
              while( iChannel<nCommonChannelsPerSample )
               { pDst.pI32[iChannel] = (long)( 2147483648.0 * pfltSrc[iChannel] );
                 iChannel++;
               }
              while( iChannel<nDstChannelsPerSample )
               { pDst.pI32[iChannel++] = 0;
               }
              break;
         } // end switch < data type flags from the VT_BLOCK header >
        pvfh->dwBufferIndex += nDstBytesPerSample;
        pfltSrc += nSrcChannelsPerSample;
        ++nSamplesWritten;
        ++pvfh->i64NumSamplesWritten;
      } // end for(iSample=0; iSample<nSamplePoints; ++iSample )


   } // end if( pvfh->iStreamFormat==STREAM_FORMAT_NON_COMPRESSED)

  // Finish the current bitrate-measurement-interval ?
  d = pChunkInfo->ldblUnixDateAndTime - pvfh->dblStartTimeOfBitrateMeasurement;
  if( d<0 || d>10 )
   {
     if( d>1 )
      { pvfh->nBitsPerSecond = pvfh->nBitrateCounter / d;
      }
     pvfh->dblStartTimeOfBitrateMeasurement = pChunkInfo->ldblUnixDateAndTime;
     pvfh->nBitrateCounter = 0;
   }

  return nSamplesWritten;

} // end C_VorbisFileIO::WriteSamples_Float(...)


/***************************************************************************/
int C_VorbisFileIO::GetInitialHeadersForStreamOutputServer(
              BYTE *pbDest,   int iMaxDestLength, // [out] network buffer, with limited length
              DWORD *pdwPageIndex )  // [out] indicator for the next call of GetNextPagesForStreamOutputServer()
  // Used to implement a multi-stream-server in Spectrum Lab.
  // "Compress once" (via WriteSamples_Float),
  // "ready multiple" (via GetInitialHeadersForStreamOutputServer + GetNextPagesForStreamOutputServer).
  //
  //   Called by the HTTP server for each new remote client
  //   which attaches to the outbound audio stream:
  //   TSpectrumLab::MyWindowProc()
  //    -> HttpSrv_HandleAsyncMsg() [message actually sent from the WINSOCK API]
  //       -> HttpSrv_OnRead()
  //          -> HttpSrv_ParseRequest()
  //             -> HttpSrv_SendMsgToApplication( HTTP_MSG_FILE_REQUEST )
  //                -> .... lots of windows-specific mumbo-jumbo ....
  //                   -> TSpectrumLab::MyWindowProc(case WM_WINSOCK_HTTP_APPMSG)
  //                      -> TSpectrumLab::OnHttpFileRequestOrCompletion( "_audiostream.ogg" )  [or similar]
  //                         -> TSpectrumLab::OnHttpAudioStreamRequest()
  //                            -> C_VorbisFileIO::GetInitialHeadersForStreamOutputServer()
  //   and, a few ms later:     -> C_VorbisFileIO::GetNextPagesForStreamOutputServer()
  // Returns the number of bytes actually placed in pbDest .
  //
  // Details: Added 2014-04 to support MULTIPLE STREAM CLIENTS ('receivers')
  //               fed by a SINGLE STREAM SERVER ('transmitter')
  //               which only holds the latest few seconds (actually, Ogg pages)
  //               in memory, and a buffer with the three mandatory headers
  //               which will be sent first when a client connects
  //               to SL's built-in HTTP server. For more details, see :
  //  * c:\cbproj\SpecLab\SpecHttpSrv.cpp
  //     ( uses a single C_VorbisFileIO instance, SOUND_VorbisStreamOutputServer,
  //       which provides the compressed data for MULTIPLE remote clients)
  //  * c:\cbproj\SpecLab\http_server_sourcecode\http_intf.c
  //     ( "interface" for the HTTP server, without application specific stuff)
  //  * C:\cbproj\SpecLab\http_server_sourcecode\httpa.c
  //     (implements a subset of HTTP, required for audio streaming)
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;

  if( pvfh!=NULL )
   {
     if( pdwPageIndex != NULL )
      { *pdwPageIndex = pvfh->dwPageIndex; // begin reading samples at the NEXT OGG PAGE which becomes available
        // (don't try to send the backlog to a freshly connected client!)
      }
     // The following header-block has already been prepared ONCE,
     // when the Ogg/Vorbis encoder was initialized (OutOpen)
     // and all 'initial' headers have been produced (EncoderInit2) :
     if(   pvfh->nInitialHeaderBufferedBytes > 0
       &&  pvfh->nInitialHeaderBufferedBytes <= iMaxDestLength )
      { memcpy( pbDest, pvfh->bInitialHeaderBuffer, pvfh->nInitialHeaderBufferedBytes );
        return pvfh->nInitialHeaderBufferedBytes; // returns the number of bytes actually placed in pbDest
      }
     // If pvfh->nInitialHeaderBufferedBytes remains ZERO all the time,
     //
   }
  return 0;  // "nothing copied"
} // end C_VorbisFileIO::GetInitialHeadersForStreamOutputServer()

/***************************************************************************/
int C_VorbisFileIO::GetNumPagesAvailableForStreamOutput(
              DWORD dwPageIndex) // [in] indicator for the next call of GetNextPagesForStreamOutputServer()
  // Implemented for the 'audio stream server statistics' in AudioStreamServer_GUI.h .
{
  int iPage, nPages = 0;
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   {
     // Similar loop as in GetNextPagesForStreamOutputServer() :
     for(iPage=0; iPage<pvfh->nBufferPagesUsed; ++iPage)
      { if( (pvfh->BufferPages[iPage].dwPageIndex >= dwPageIndex ) // bingo, the requested page is still available (in the buffer) ..
         && (pvfh->BufferPages[iPage].iSize_Byte > 0 ) )
         { ++nPages;  // count the number of pages which THIS reader would be able to send now
         }
      }   
   }
  return nPages;
} // end C_VorbisFileIO::GetNumPagesAvailableForStreamOutput()


/***************************************************************************/
int C_VorbisFileIO::GetNextPagesForStreamOutputServer(
              BYTE *pbDest,   int iMaxDestLength,
              DWORD *pdwPageIndex) // [in,out] indicator for the next call of GetNextPagesForStreamOutputServer()
  // Retrieves as many samples as possible (limited to complete Ogg pages)
  //           for one of many possible STREAM READERS .
  // Used to implement a multi-stream-server in Spectrum Lab. Principle:
  //    "one writer/compressor"  (in WriteSamples_Float),
  //    "multiple readers" (GetInitialHeadersForStreamOutputServer + GetNextPagesForStreamOutputServer).
  //
  // Periodically called by the HTTP server to send more samples to a remote client.
  // Called by the HTTP server for each remote client attached to the audio stream:
  // (*complete* call stack in c:\cbproj\SpecLab\SpecHttpSrv.cpp::SpecLab_HttpAudioStreamOutCallback)
  //   WinMain() -> lots of red tape -> TSpectrumLab::Timer1Timer()
  //    -> HttpSrv_OnTimerEvent() -> HttpSrv_OnContinueWrite() ->
  //        -> HttpSrv_SendResponseHeaderAndFileContents()
  //            -> HttpSrv_ReadFromDataBlock() [because this stream is encoded IN MEMORY]
  //                -> SpecLab_HttpAudioStreamOutCallback()
  //                    -> C_VorbisFileIO::GetNextPagesForStreamOutputServer()
  //
  // Returns the number of bytes actually placed in pbDest .
  //
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  DWORD dwPageIndex;
  int   iPage,page_size,page_buf_idx, nBytesRead=0, nPagesRead=0;
  BOOL  page_ok;

  if( pvfh!=NULL )
   {
     if( pdwPageIndex != NULL )
      { dwPageIndex = *pdwPageIndex; // for Ogg/Vorbis encoded streams, this is actually the Ogg Page counter
                                     // which started at zero when the 'COMPRESSOR' was initialized .
      }
     else // Ogg page index not specified -> wait for the next page which becomes available
      { dwPageIndex = pvfh->dwPageIndex;
      }

     if( pvfh->fStoreSamplesForMultipleReaders )  // can we pull samples out of the page-buffer ?
      { // Is the reader's "sample position" (= the FIFO TAIL INDEX) still valid,
        // i.e. does dwPageIndex still point to a valid Ogg page boundary
        // (or to the header of an un-compressed stream) ?
        page_ok = FALSE;
        for(iPage=0; iPage<pvfh->nBufferPagesUsed; ++iPage)
         { if( (pvfh->BufferPages[iPage].dwPageIndex == dwPageIndex ) // bingo, the requested page is still available (in the buffer) ..
            && (pvfh->BufferPages[iPage].iSize_Byte > 0 ) )
            { page_ok = TRUE;
              break;  // which means all 'later' pages (if any, in the buffer) are also still valid
            }
         }
        if( !page_ok )
         { //  Most likely reason: the reader (web server?) didn't call
           //  GetNextPagesForStreamOutputServer() frequently enough.
           //  'His' buffer-tail-index will be reset to the new buffer-head-index.
           if( dwPageIndex == (pvfh->dwPageIndex+1) )
            {   // reader must wait for the NEXT OGG PAGE. Leave dwPageIndex unchanged.
            }
           else // something wrong, discard this caller's Ogg page counter (he will lose a few pages!)
            { dwPageIndex = pvfh->dwPageIndex;  // <<< set breakpoint here
            }
         }
        else // page_ok :
         {   // pvfh->BufferPages[iPage] is the next page to be copied to pbDest,
             // and there is AT LEAST ONE more complete Ogg page available for this reader.
             // pvfh->dwBufferIndex is the index of the next byte WRITTEN INTO the buffer (in WriteOggPage),
             // it also marks the CURRENT END-STOP for all readers.
             //  Example : THREE Ogg-pages currently available, each 4000 bytes long; [in reality, an Ogg page was 4..5 kByte large]
             //            reader starts with pdwPageIndex = 0
             //
             //     pdwPageIndex=0  ..4000    ..8000     pvfh->dwBufferIndex=1200
             //           |         |         |         |
             //           |         |         |         |
             //          \|/_______\|/_______\|/_______\|/_______________
             // .bBuffer | page[0] | page[1] | page[2] |not available yet|
             //          |_________|_________|_________|_________________|
             //                                         filled in future calls
             //                                         of WriteOggPage
             //
           // Repeat this for as many Ogg pages as possible :
           while( (page_size=pvfh->BufferPages[iPage].iSize_Byte) <= iMaxDestLength)
            { page_buf_idx = pvfh->BufferPages[iPage].iBufIndex;
              if( page_size<=0 || page_buf_idx<0 )  // page invalid ?
               { break;
               }
              if( pvfh->BufferPages[iPage].dwPageIndex!=dwPageIndex ) // wrong page (doesn't belong to the sequence,
               { break;       // most likely it's an OLDER page in the circular FIFO)
               }
              memcpy( pbDest, &pvfh->bBuffer[page_buf_idx], page_size );
              pbDest      += page_size;
              iMaxDestLength -= page_size;
              nBytesRead  += page_size;
              ++iPage;
              if( iPage >= pvfh->nBufferPagesUsed )
               {  iPage = 0;
               }
              ++nPagesRead;
              ++dwPageIndex;  // the caller's (reader's) Ogg page counter
            }
         } // end else < pdwPageIndex points to the begin of a valid page in the buffer >
      } // if( pvfh->fStoreSamplesForMultipleReaders )
   }
  if( pdwPageIndex != NULL )
   { *pdwPageIndex = dwPageIndex;
   }
  return nBytesRead;
} // end C_VorbisFileIO::GetNextPagesForStreamOutputServer()


/************************************************************************/
int C_VorbisFileIO::GetState(void) // ->
  // Only used to simplify the 'activity display' in the GUI - nothing else !
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh==NULL )
   { return AUDIO_FILE_IO_STATE_PASSIVE;
   }
  if( pvfh->fConnecting )
   { return AUDIO_FILE_IO_STATE_WAIT_CONN;
   }
  if( GetParameterByID( AUDIO_FILE_PARAM__CONNECTION_LOST ) )
   { return AUDIO_FILE_IO_STATE_LOST_CONN;
   }
  if( pvfh->fOpenedForReading )
   { return AUDIO_FILE_IO_STATE_RECEIVING;
   } // end if( pvfh->fOpenedForReading )
  if( pvfh->fOpenedForWriting ) // also applies to WEB-STREAMS ready for "sending"
   { if( pvfh->want_tstamp && (!pvfh->have_tstamp) )
      { return AUDIO_FILE_IO_STATE_NEED_TIMESTAMP;
      }
     return AUDIO_FILE_IO_STATE_SENDING;
   } // end if( pvfh->fOpenedForWriting )

  return AUDIO_FILE_IO_STATE_PASSIVE;
} // end C_VorbisFileIO::GetState()


/************************************************************************/
  // Some 'Get' - and 'Set' - functions for the AUDIO FILE class ..


int C_VorbisFileIO::GetFileFormat(void)  // -> AUDIO_FILE_FORMAT_OGG, AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS, and maybe more
{
  T_VorbisFileHelper *pvfh; // contains everything which shouldn't go into VorbisFileIO.h
  if( m_pVorbisFileHelper != NULL )
   { pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
     switch( pvfh->iStreamFormat )
      { case STREAM_FORMAT_OGG_VORBIS:
           return AUDIO_FILE_FORMAT_OGG;
        case STREAM_FORMAT_NON_COMPRESSED:
           return AUDIO_FILE_FORMAT_STREAM_WITH_HEADERS;
        case STREAM_FORMAT_VLF_RX_BLOCKS :
           // Uncompressed stream from VLF-RX-tools *without* T_StreamHeaders
           //  but VT_BLOCKs.  Added here 2020-10-11 but not really supported yet.
           return AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS;
        default:
           break;
      }
   }
  return AUDIO_FILE_FORMAT_UNKNOWN;
}

double C_VorbisFileIO::GetSampleRate(void)  // this is the FILE's sampling rate !
{ return m_dbl_SampleRate;
}
void   C_VorbisFileIO::SetSampleRate(double dblSampleRate)
{ m_dbl_SampleRate = dblSampleRate;
}

//---------------------------------------------------------------------------
int    C_VorbisFileIO::GetNrChannels(void)
{ return m_nChannelsPerSampleInFile;
}

//---------------------------------------------------------------------------
void   C_VorbisFileIO::SetNrChannels(int iNrChannels)
{
  // Setting the "number of channels" is only legal BEFORE WRITING a file !
  if( ! IsOpenedForWriting() && (iNrChannels>0) && (iNrChannels<=8) )
   { m_nChannelsPerSampleInFile = iNrChannels;
   }
}

//---------------------------------------------------------------------------
void C_VorbisFileIO::GetDataType(
        int *piSizeOfDataType, // [out] 0=unknown, 1=BYTEs, 2=short INTs, 4=floats, 8=double's
        int *piDataTypeFlags)  // [out] data type flags, see AudioFileDefs.h : AUDIO_FILE_DATA_TYPE_SIGNED_I / ..UNSIGNED / ..FLOAT / ..MOTOROLA
{
  if( piSizeOfDataType != NULL )
   { *piSizeOfDataType = m_iSizeOfDataType;
   }
  if( piDataTypeFlags  != NULL )
   { *piDataTypeFlags  = m_iDataTypeFlags;
   }
} // end C_VorbisFileIO::GetDataType()

//---------------------------------------------------------------------------
int C_VorbisFileIO::GetBitsPerSample(void) // ex: 16 (fixed value)
{ return m_iSizeOfDataType * 8;
} // end C_VorbisFileIO::GetBitsPerSample()


//---------------------------------------------------------------------------
double C_VorbisFileIO::GetFrequencyOffset(void)  // "radio-" minus "baseband"-frequency
{ // Details in C_WaveIO::GetFrequencyOffset() !
  return m_dblFrequencyOffset;
} // end C_VorbisFileIO::GetFrequencyOffset()

BOOL   C_VorbisFileIO::SetFrequencyOffset(double dblFrequencyOffset)
{ // Details in C_WaveIO::GetFrequencyOffset() !
  m_dblFrequencyOffset = dblFrequencyOffset;
  return TRUE;
} // end C_VorbisFileIO::SetFrequencyOffset()

//---------------------------------------------------------------------------
LONGLONG C_VorbisFileIO::GetNumSamplesWritten(void)
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { return pvfh->i64NumSamplesWritten;
     // Note: For the Vorbis-Stream-SERVER integrated in SpecLab,
     //       GetNumSamplesWritten() returns the number of samples
     //       "written" into the common FIFO which may feed MULTIPLE clients !
     //       To determine the number of audio samples sent to a certain client,
     //       use HttpAudioStreamServer_GetStatistics() instead !
   }
  return 0;
} // end C_VorbisFileIO::GetNumSamplesWritten()

//---------------------------------------------------------------------------
long C_VorbisFileIO::GetCurrentBitrate(void)
  // measures the average data rate in Bit/second
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { return pvfh->nBitsPerSecond;
   }
  return 0;
} // end C_VorbisFileIO::GetCurrentBitrate()


//---------------------------------------------------------------------------
int  C_VorbisFileIO::GetTxBufferUsage_kByte(void)
{
  int result=0;
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { result = INET_GetTxBufferUsage_kByte( &pvfh->InetClient );
   }
  return result;
} // end C_VorbisFileIO::GetTxBufferUsage_kByte()


//---------------------------------------------------------------------------
int  C_VorbisFileIO::GetInetPeakWaitTime_ms(BOOL fReset)
{
  int result=0;
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { result = pvfh->InetClient.nMillisecondsWaited_peak;
     if( fReset )
      { pvfh->InetClient.nMillisecondsWaited_peak = 0;
      }
   }
  return result;
} // end C_VorbisFileIO::GetInetPeakWaitTime_ms()

//---------------------------------------------------------------------------
int  C_VorbisFileIO::GetInetCurrWaitTime_ms(void)
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { return pvfh->InetClient.nMillisecondsWaited;
   }
  return 0;
} // end C_VorbisFileIO::GetInetCurrWaitTime_ms()


//---------------------------------------------------------------------------
double C_VorbisFileIO::GetCurrentRecordingTime(void) // see specification in c:\cbproj\SoundUtl\AudioFileIO.h (t=0 is the RECORDING START TIME, aka "track time")
{
  return 0.0;    // **NOT** "+ Start Time" ! ! !
}

//---------------------------------------------------------------------------
double C_VorbisFileIO::GetRecordingStartTime(void)
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   {
     return pvfh->dblStartTime; // seconds elapsed since 00:00:00 GMT, January 1, 1970.
   }
  else
   { return 0.0;
   }
}

//---------------------------------------------------------------------------
void   C_VorbisFileIO::SetRecordingStartTime(double dblStartTime)
{
  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh!=NULL )
   { pvfh->dblStartTime = dblStartTime;
   }
}

//---------------------------------------------------------------------------
void C_VorbisFileIO::GetFileName(char *pszDest, int iMaxLen)
{ strncpy( pszDest, m_sz255FileName, iMaxLen );
}

//---------------------------------------------------------------------------
char *C_VorbisFileIO::GetLastErrorAsString(void) // for diagnostic purposes
{
  return m_cErrorString;
}


//---------------------------------------------------------------------------
BOOL C_VorbisFileIO::CloseFile(
        char *pszWhy ) // [in, optional] reason for closing, may be NULL.
        // Added for the display on the 'Analyse / play audio stream' panel,
        // which often contained 'closing internet socket' but didn't say WHY.
  /* Closes the audio-file or -stream, when opened.
   * It doesn't hurt to call this method when NOT open.
   * Returns TRUE when a file HAS BEEN CLOSED; FALSE on error or 'nothing to close'.
   */
{
  BOOL fClosedSomething = FALSE;

  T_VorbisFileHelper *pvfh = (T_VorbisFileHelper*)m_pVorbisFileHelper;
  if( pvfh )
   {
     if( pvfh->fMustClearBlock )       // last initialized, first cleared
      { pvfh->fMustClearBlock = FALSE;
        vorbis_block_clear( &pvfh->vblock ); // cleanup for vorbis_block_init()
      }
     if( pvfh->fMustClearDSP )         // not sure about the '_clear' sequence !
      {  pvfh->fMustClearDSP = FALSE;
        vorbis_dsp_clear( &pvfh->vdsp );     // cleanup for vorbis_synthesis_init() [?]
      }
     if( pvfh->fMustClearVorbisInfo )  // first initialized, last cleared
      {  pvfh->fMustClearVorbisInfo = FALSE;
        vorbis_info_clear( &pvfh->vinfo );   // cleanup for vorbis_encode_setup_init()
      }
#   if (VORBIS_USE_QFILE)
     if( pvfh->m_QFile.iHandle > 0 )
      { QFile_Close( &pvfh->m_QFile ); // should never be necessary to close it here, but ....
        pvfh->m_QFile.iHandle = -1;
        fClosedSomething = TRUE;
      }
#   else  // don't use WB's QFile but some bastardized 'standard' file I/O by Borland:
     if( pvfh->m_FileHandle > 0 )
      { _rtl_close(pvfh->m_FileHandle); // ... avoid a file handle leak !
        pvfh->m_FileHandle = -1;
        fClosedSomething = TRUE;
      }
#   endif
     if( pvfh->fIsInternetStream )
      { pvfh->fIsInternetStream = FALSE;
        INET_CloseClient( &pvfh->InetClient, pszWhy );
        fClosedSomething = TRUE;
      }
     if( pvfh->qfStreamLog.fOpened )
      { QFile_Close( &pvfh->qfStreamLog );
        fClosedSomething = TRUE;
      }
     pvfh->fOpenedForReading = FALSE;
     pvfh->fOpenedForWriting = FALSE;
   }
  return fClosedSomething;
} // C_VorbisFileIO::CloseFile(..)

// About floating point exceptions... WB, 2020-10-11 :
// From http://www.virtualdub.org/blog/pivot/entry.php?id=53 :
//
// > In general, you don't want to be tripping floating-point exceptions,
// > even if they are masked. The reason is that when the FPU hits one,
// > the fast hardware can't handle it and punts to the microcode, which then
// > takes about twenty times longer. This is especially bad with NaNs since
// > any operation with a NaN produces another NaN, causing them to spread
// > throughout your calculations (NaN disease) and slow down everything massively.
// > You can even crash due to NaNs blowing past clamp expressions,
// > since any comparison with a NaN is false and converting one to integer
// > form results in integer indefinite (0x80000000).
// > Despite the erroneous results, though, NaNs can appear sporadically
// > in a large Win32 program (such as ours, WB..) without anyone knowing,
// > and may go unnoticed in a code base for years.
// >
// > How Borland C/C++ factors into the picture
// >
// > The Borland DLL run-time library, as it turns out, enables floating-point
// > exceptions on initialization. This happens even if you simply load the DLL!
// > Because Windows programs generally don't touch the floating-point control word,
// > the effects of this can persist long after the DLL has been unloaded.
// > For instance, you could:
// >   * launch an Open file dialog in a program and hover over an AVI file in the list,
// >     ...thus causing Explorer to load its shell media extension
// >        to retrieve the file's properties for the tooltip,
// >     ...causing a codec search for the video stream in the file,
// >     ...thus loading and unloading the Borland-compiled DLL.
// > It is possible to disable this behavior of the Borland run-time library
// > and avoid this problem, but most people aren't aware of it,
// > and unintentionally release DLLs that cause this issue. (....)
// > The best way to fix it is to not modify the control word, but I don't know
// > if that is possible; barring that, a usable workaround is to remask
// > the exceptions with _controlfp(), as noted at
// >   http://homepages.borland.com/ccalvert/TechPapers/FloatingPoint.html .
//     (of course that page doesn't exist anymore, but hold on..)
// > Sadly, strictly speaking you cannot simply recover by remasking
// > the FPU exceptions and restarting the faulting instruction.
// > The problem is that the x87 FPU doesn't signal the interrupt
// > until the next floating-point instruction, at which point
// > necessary information to retry the instruction is irretrievably lost.
// > Take this instruction sequence for example:
// >   FDIV DWORD PTR [EAX]
// >   XOR  EAX, EAX
// >   FSTP DWORD PTR [EDX]
// > A divide-by-zero error here will actually result in the FSTP instruction
// > faulting, not the FDIV.
// WB: That's the reason for not being able to find the reason for the
//     "crash" (exception) in ReadAndDecodeNonCompressed() ...
//     the error most likely occurred in "fld" or "fstp" - we just don't now !

/* EOF <VorbisFileIO.cpp> */



