//---------------------------------------------------------------------------
// File  :  SpecEventLog.h
// Date  :  2005-08-13   (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:  "Spectrum Event Logger" .
//     Implementation of a class for logging spectrum 'events'.
//     Principle: User notices something interesting in the spectrogram,
//                clicks on it, enters a short description (text)
//                which will then be saved in a text file (along with
//                precise timestamp and frequency).
//             The events may be marked in the spectrogram as small circles
//             (or similar), when the user clicks on one of them (later),
//             the remark will be displayed again.
//                This is not the same as the small "text lines" which may be
//                inserted in the SPECTRUM BUFFER ! In fact, the "events"
//                contained in this file will appear in any spectrogram,
//                no matter from which buffer-file the SPECTROGRAM was loaded.
//
// Revision history (YYYY-MM-DD):
//   2005-08-13  Written for SpecLab.
//               DOES NOT USE VCL STUFF, ONLY CLEAN WIN API CALLS !
//               (to have a compact DLL without VCL dependencies)
//---------------------------------------------------------------------------

#ifndef YHF_File_H
  // #include "YHF_File.h"      // DL4YHF's little FILE helper routines
  // (only required for the IMPLEMENTATION so far !)
#endif

//---------------------------------------------------------------------------
#ifndef SpecEventLogH
#define SpecEventLogH


//----------------------- Constants ---------------------------------------


//----------------------- Data Types --------------------------------------
typedef struct _SpecLogEntry // T_SpecLogEntry
{
  _SpecLogEntry *pPrev, *pNext;  // pointers for double-linked list
  double dblTime;   // UNIX time (seconds passed since 1970-01-01 00:00:00 UTC)
  double dblFreq;   // frequency of interest
  int iFlags;       // flags (importance, priority, how to mark in spectrogram, etc)
  char *pszText;        // text (multiple lines separated by newline characters),
       // dynamically allocated: pszText[iAllocdTextLength-1] plus trailing zero
  int  iAllocdTextSize; // length of the allocated text buffer

} T_SpecLogEntry;


//------------------------ Global Variables -------------------------------



//**************************************************************************
//    Prototypes (no class methods)
//**************************************************************************




//***************************************************************************
//  Definition of the  CSpecEventLog   class
//***************************************************************************

class CSpecEventLog
{
public:
   CSpecEventLog();            // constructor without parameters
   virtual ~CSpecEventLog();   // destructor (cleans up)
   void Clear(void);           // clears the list and the "modified"-flag
   BOOL IsModified(void);      // returns state of the "modified"-flag   

   BOOL LoadFile(char *pszFilename);  // loads a logfile with the specified name
   BOOL SaveFile(char *pszFilename);  // saves the (sorted) buffer as a text file


   BOOL AddEntry( double dblTime,  double dblFrequency, int iFlags, char *pszText );
   
   BOOL GetNextEntry( double dblTime_in, // input: time to start seeking from
              double *pdblTime_out,  // output: oldest time found (but YOUNGER than dblTime_in)
              double *pdblFrequency_out, // output: frequency of interest
              int    *piFlags_out,   // output: "flags"
              char *pszText_out, int iMaxLen );  // returned text, max text length

   BOOL GetPrevEntry( double dblTime_in, // input: time to start seeking from
              double *pdblTime_out,  // output: time of preceding entry (but OLDER than dblTime_in)
              double *pdblFrequency_out, // output: frequency of interest
              int    *piFlags_out,   // output: "flags"
              char *pszText_out, int iMaxLen );  // returned text, max text length

   BOOL GetNearestEntry( double dblTime_in, // input: time to seek
              double *pdblTime_out,      // output: time of nearest entry (no matter if older or younger)
              double *pdblFrequency_out, // output: frequency of nearest entry
              int    *piFlags_out,       // output: "flags"  (how to mark this in the spectrogram)
              char *pszText_out, int iMaxLen );  // returned text, max text length

   //-----------------------------------------------------------------------
private:
   T_SpecLogEntry *m_pListStart, *m_pListEnd; // double-linked list of entries, dynamically allocated,
                                              // start is the OLDEST, end is the YOUNGEST entry
   char m_sz255LastFileName[256];
   BOOL m_fModified;              // modified-flag (for caller, to decide wheter to save)
   BOOL AddEntryIntern( double dblTime,  double dblFrequency, int iFlags, char *pszText );
   BOOL ReturnEntry( T_SpecLogEntry *pEntry, // input: pointer into list
                  double *pdblTime_out,      // output: oldest time found (A BIT OLDER than dblTime_in)
                  double *pdblFrequency_out, // output: frequency of interest
                  int    *piFlags_out,   // output: "flags"  (how to mark this in the spectrogram)                  
                  char *pszText_out, int iMaxLen );  // returned text, max text length


}; // end class CSpecEventLog


#endif // SpecEventLog
