//---------------------------------------------------------------------------
// File  :  SpecEventLog.cpp
// Date  :  2005-08-13   (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:  "Spectrum Event Logger" .
//     Implementation of a class for logging spectrum 'events'.
//     Principle: User notices something interesting in the spectrogram,
//                clicks on it, enters a short description (text)
//                which will then be saved in a text file (along with
//                precise timestamp and frequency).
//             The events may be marked in the spectrogram as small circles
//             (or similar), when the user clicks on one of them (later),
//             the remark will be displayed again.
//                This is not the same as the small "text lines" which may be
//                inserted in the SPECTRUM BUFFER ! In fact, the "events"
//                contained in this file will appear in any spectrogram,
//                no matter from which buffer-file the SPECTROGRAM was loaded.
//
// Revision history (YYYY-MM-DD):
//   2005-08-13  Written for SpecLab.
//               DOES NOT USE VCL STUFF, ONLY CLEAN WIN API CALLS !
//               (to have a compact DLL without VCL dependencies)
//---------------------------------------------------------------------------


#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#include <windows.h>
#include <math.h>
#include <string.h>
#include <stdio.h>

#pragma hdrstop        // no precompiled headers after this point

#include "QFile.h"         // DL4YHF's text-file access routines
#include "Utility1.h"      // utility package with time & date conversion stuff
#include "SpecEventLog.h"  // header for this module



//**************************************************************************
//    Helper routines (no class methods)
//**************************************************************************

//--------------------------------------------------------------------------
T_SpecLogEntry *SpecEvtLog_NewEntry( int iAllocdTextSize )
{
 T_SpecLogEntry *pNewEntry = (T_SpecLogEntry *)malloc( sizeof(T_SpecLogEntry) );
 if( !pNewEntry )
     return NULL;
 // Initialize the members of the new entry, and allocate the text buffer
 if(iAllocdTextSize<80) iAllocdTextSize=80; // allow at least 80 characters!
 iAllocdTextSize = (iAllocdTextSize+1+4) & 0xFFFC; // add one for the trailing zero, and round UP to next DWORD block size
 memset( pNewEntry, 0, sizeof(T_SpecLogEntry) );
 pNewEntry->dblTime = pNewEntry->dblTime = 0.0;  // ensure VALID floating-point format
 pNewEntry->iAllocdTextSize = iAllocdTextSize;
 pNewEntry->pszText = (char*)malloc(iAllocdTextSize);
 if( pNewEntry->pszText == NULL )
  {  // oops, not enough memory to allocate the text buffer ?!
    free( pNewEntry );  // very suspicious; I surrender..
    return NULL;
  }
 return pNewEntry;
} // end SpecEvtLog_NewEntry()

//--------------------------------------------------------------------------
void SpecEvtLog_DeleteEntry(T_SpecLogEntry *pEntry)
{
  if( pEntry!=NULL )
   {
     if( pEntry->iAllocdTextSize>0 && pEntry->pszText!=NULL)
      { free( pEntry->pszText ); // release memory for the TEXT BUFFER
      }
     free( pEntry ); // release memory for the T_SpecLogEntry struct itself
   }
} // end SpecEvtLog_DeleteEntry()

//--------------------------------------------------------------------------
BOOL SpecEvtLog_WriteEntryToFile(T_QFile *pQFile, T_SpecLogEntry *pEntry)
{
  char sz255Temp[256]; char *cp;
  BOOL fResult;

  cp = sz255Temp;
  *cp++ = '[';
  UTL_FormatDateAndTime( "YYYY-MM-DD hh:mm:ss.sss", pEntry->dblTime, cp );
  cp += strlen(cp);
  strcpy( cp, "]\r\nfr=" );
  cp += strlen(cp);
  UTL_FloatToFixedLengthString( pEntry->dblFreq, cp, 10/*iNrOfChars*/ );
  cp += strlen(cp);
  sprintf(cp,"\r\nfl=%ld\r\n",(long) pEntry->iFlags );

  fResult = QFile_WriteString( pQFile, sz255Temp );
  if( pEntry->pszText!=NULL )
   { fResult &= QFile_WriteString( pQFile, pEntry->pszText );
   }
  fResult &= QFile_WriteString( pQFile, "\r\n\r\n" );  // empty line for easy readability

  return fResult;

} // end SpecEvtLog_WriteEntryToFile()

//--------------------------------------------------------------------------
void SpecEvtLog_DeleteList( T_SpecLogEntry *pEntries )
{
  T_SpecLogEntry *pNextEntry;
  while( pEntries )
   { pNextEntry = pEntries->pNext;
     SpecEvtLog_DeleteEntry( pEntries );
     pEntries = pNextEntry;
   }
} // end SpecEvtLog_DeleteList()


//--------------------------------------------------------------------------
BOOLEAN SpecEvtLog_IsNumString(char *cp, int nDigits)
   // Checks if a string contains a valid NUMBER with exactly(!) N digits
   // Used to detect new sections in the logfile (which begin with a timestamp)
{
  while(nDigits--)
   { if(*cp<'0' || *cp>'9') return FALSE;
     ++cp;
   }
  return *cp<'0' || *cp>'9';  // the following char must *not* be a digit
} // end SpecEvtLog_IsNumString()


//***************************************************************************
//  Implementation of methods for the    CSpecEventLog   class
//***************************************************************************


//--------------------------------------------------------------------------
CSpecEventLog::CSpecEventLog()     // constructor
{
  m_pListStart = NULL;  // double-linked list of entries, dynamically allocated
  m_pListEnd   = NULL;
  m_fModified  = FALSE;
  m_sz255LastFileName[0] = 0;
} // end CSpecEventLog::CSpecEventLog()


//--------------------------------------------------------------------------
void CSpecEventLog::CloseAll(void) // application-callable replacement for the destructive destructor ;o)
{
  // 2010-06-10 : Beware of a VERY STRANGE sequence of atexit(), VclExit(),
  //     and other 'desctructor' calls - see UTILITY1.CPP::UTL_ExitFunction() !
  // (that's why the following code was moved here, from the destructor .
  //  Ideally, the destructor doesn't need to 'do anything' now;
  //  and the calling sequence of the destructors doesn't matter anymore)
  if( m_pListStart )
   {  SpecEvtLog_DeleteList( m_pListStart );
      m_pListStart = NULL;
   }
} // end CSpecEventLog::CloseAll(void)


//--------------------------------------------------------------------------
CSpecEventLog::~CSpecEventLog()    // destructor, clean up
{
  // 2010-06-10 : Beware of a VERY STRANGE sequence of atexit(), VclExit(),
  //     and other 'desctructor' calls - see UTILITY1.CPP::UTL_ExitFunction() !
  CloseAll();  // ex; if( m_pListStart ) { SpecEvtLog_DeleteList( m_pListStart ); ....
} // end CSpecEventLog::~CSpecEventLog() [destructor]

//--------------------------------------------------------------------------
void CSpecEventLog::Clear(void)
{
  if( m_pListStart )
   {  SpecEvtLog_DeleteList( m_pListStart );
      m_pListStart = NULL;
   }
  m_fModified = FALSE;
} // end CSpecEventLog::Clear()

//--------------------------------------------------------------------------
BOOL CSpecEventLog::IsModified(void)
{    // returns state of the "modified"-flag
  return m_fModified;
}

//--------------------------------------------------------------------------
BOOL CSpecEventLog::LoadFile(char *pszFilename)
   // Opens a logfile with the specified name
   // and "loads" (actually MERGES!) it into the chained list in RAM .
   // To load ONLY (without merging), call the Clear() method before loading.
{
  T_QFile qfile;
  double dblTime, dblFreq;
  char sz255InputLine[256];
  char *sz64kTextBuffer; char *cp;
  BOOL fResult = FALSE;
  int  iSection, iFlags;

  if( pszFilename==NULL || pszFilename[0]=='\0' )
   { pszFilename = m_sz255LastFileName;
   }
  else
   { strncpy( m_sz255LastFileName, pszFilename, 255 );
   }

  sz64kTextBuffer = (char*)malloc(65536);
  if( sz64kTextBuffer==NULL ) return FALSE;
  sz64kTextBuffer[0] = '\0';
  if( pszFilename[0]!='\0' )
   {
     if(  QFile_Open( &qfile, pszFilename, QFILE_O_RDONLY ) )
      { fResult = TRUE;
        if( m_pListStart )  // a list already exists; so we are "MERGING" now
         {  m_fModified = TRUE;
         }
        // Read all lines from the input file,
        //  and add as T_SpecLogEntry structs to the list :
        iSection = 0;  // expecting a TIME SECTION header next..
        iFlags  = 0;
        dblTime = dblFreq = 0.0;
        while(QFile_ReadLine( &qfile, sz255InputLine, 255 )>=0)
         { // Parse the line..
           sz255InputLine[255] = '\0';  // prevent running into the woods..
           cp = sz255InputLine;
           if(   cp[ 0]=='[' && SpecEvtLog_IsNumString(cp+ 1,4) // "[YYYY" [0..4]
              && cp[ 5]=='-' && SpecEvtLog_IsNumString(cp+ 6,2) // "-MM"  [5..7]
              && cp[ 8]=='-' && SpecEvtLog_IsNumString(cp+ 9,2) // "-DD"  [8..10]
              && cp[11]==' ' && SpecEvtLog_IsNumString(cp+12,2) // " HH"  [11..13]
              && cp[14]==':' && SpecEvtLog_IsNumString(cp+15,2) // ":MM"  [14..16]
             ) // looks like a timestamp . Note : Minutes, seconds, milliseconds are OPTIONAL (!)
            { if( iSection>0 && dblTime>0.0 )
               { // must append the previous entry to the chained list :
                 AddEntryIntern( dblTime,  dblFreq, iFlags, sz64kTextBuffer );
               }
              ++cp; // skip the '['-character (which identifies the "section header")
              if( UTL_ParseFormattedDateAndTime( (BYTE*)"YYYY-MM-DD hh:mm:ss.sss", &dblTime, (BYTE**)&cp ) )
               { iSection = 1;
               }
              sz64kTextBuffer[0]=0; dblFreq=0.0;  // nothing else read for this logfile entry yet
            } // end if < found next section ([YYYY-MM-DD hh:mm[:ss.sss])
           else // no "new section", but...
            {
             if( iSection==1 )
              {   // still in the "header section" ..
               if( UTL_CheckAndSkipToken( (BYTE**)&cp, "fr=" ) )
                { // it's the FREQUENCY for this entry :
                  dblFreq = atof( cp );
                }
               else if(UTL_CheckAndSkipToken( (BYTE**)&cp, "fl=" ) )
                { // it's the FLAGS-value of this entry
                  // (which may be used to decide how to mark this event in the spectrogram)
                  iFlags = atoi( cp );
                }
               else // everything else (unknown tokens) mark the END OF THE HEADER SECTION
                {   // for this entry; the line will be handled in the next if-statement !
                  iSection = 2;
                }
              } // end if( iSection==1 )
             if( iSection==2) // no header, but the text itself... append it !
              { if(strlen(sz64kTextBuffer)<60000)
                 { if(sz64kTextBuffer[0] != '\0')
                      strcat(sz64kTextBuffer, "\r\n");
                   strcat(sz64kTextBuffer, sz255InputLine);
                 }
              } // end if( iSection==1 )
            } // end else < no "new section" >
         } // end while < more lines in the input file >
        if( iSection>0 && dblTime>0.0 )
         { // also append the LAST entry after the line-reading loop ?
           AddEntryIntern( dblTime,  dblFreq, iFlags, sz64kTextBuffer );
         }
        QFile_Close( &qfile );
      } // end if < file OPENED FOR READING successfully >
   } // end if < temporary buffer (for ONE entry) successfully allocated ? >
  free ( sz64kTextBuffer );
  return fResult;
} // end CSpecEventLog::LoadFile()


//--------------------------------------------------------------------------
BOOL CSpecEventLog::SaveFile(char *pszFilename) // saves the log-buffer as a text file
{
  T_QFile qfile;
  T_SpecLogEntry *pEntry;
  BOOL fResult = FALSE;

  if( pszFilename==NULL || pszFilename[0]=='\0' )
   { pszFilename = m_sz255LastFileName;
   }
  else
   { strncpy( m_sz255LastFileName, pszFilename, 255 );
   }

  if( m_pListStart && pszFilename[0]!='\0' )
   { pEntry = m_pListStart;

     if( QFile_Create( &qfile, pszFilename, 0 ) )
      { fResult = TRUE;
        while( pEntry && fResult )
         { // Write the contents of a T_SpecLogEntry struct to the text file:
           fResult &= SpecEvtLog_WriteEntryToFile( &qfile, pEntry );
           pEntry = pEntry->pNext;
         } // end while( pEntry )
        QFile_Close( &qfile );
      } // end if < file CREATED successfully >
   } // end if < list not empty >
  if( fResult==TRUE )
   { m_fModified = FALSE;
   }
  return fResult;
} // end CSpecEventLog::SaveFile()

//--------------------------------------------------------------------------
BOOL CSpecEventLog::AddEntryIntern( double dblTime,  double dblFrequency, int iFlags, char* pszText )
   // Adds an entry in the double-linked list,
   //    keeping the list SORTED BY TIME OF ACQUISITION.
{
  T_SpecLogEntry *pEntry, *pNewEntry;
  int iRequiredTextSize;
  char *cp;
  int iLen;
  if( pszText!=NULL )
       iRequiredTextSize = strlen(pszText);
  else iRequiredTextSize = 0;
  if( iRequiredTextSize < 84 )
      iRequiredTextSize = 84;
  pNewEntry = SpecEvtLog_NewEntry( iRequiredTextSize );
  if( pNewEntry==NULL ) // oops... out of memory ?!?
    return FALSE;
  // Fill the members of the "fresh" T_SpecLogEntry structure :
  pNewEntry->dblTime = dblTime;
  pNewEntry->dblFreq = dblFrequency;
  if( (pNewEntry->pszText!=NULL) && (pszText!=NULL) )
   { strcpy(pNewEntry->pszText, pszText );
     // service for the "load from file"-routine :
     // Remove unnecessary CR + NL characters at the end of the text.
     // They were added to improve the file's readability with a text editor,
     // but we don't want any trailing empty lines in memory.
     cp = pNewEntry->pszText;
     iLen = strlen(cp);
     while(iLen>0 && (cp[iLen-1]=='\r' || cp[iLen-1]=='\n') )
      { --iLen;
        cp[iLen-1] = '\0';
      }
   } // end if < text present >

  // Where to insert the new entry in the chain ? (the tricky part)
  if( m_pListStart==NULL )
   {  // this is the first entry in the list...
      m_pListStart = pNewEntry;
      m_pListEnd   = pNewEntry;
      return TRUE;
   }
  else // not the first entry..
   { pEntry = m_pListStart;
     // search the timestamps in the SORTED list:
     while(pEntry)
      { if(pNewEntry->dblTime <= pEntry->dblTime )
         { // bingo; must insert the new entry BEFORE this "older" entry
           pNewEntry->pNext = pEntry;
           pNewEntry->pPrev = pEntry->pPrev;  // this pointer may be NULL !
           pEntry->pPrev = pNewEntry;
           if(pNewEntry->pPrev != NULL)
              pNewEntry->pPrev->pNext = pNewEntry;
           return TRUE;
         }
        if( pEntry->pNext == NULL )
         { // arrived here, the new entry must be APPENDED at the end of the list
           m_pListEnd    = pNewEntry;
           pEntry->pNext = pNewEntry;
           pNewEntry->pPrev = pEntry;
           pNewEntry->pNext = NULL;
           return TRUE;
         }
        pEntry = pEntry->pNext;
      }
   } // end else < not the 1st entry in the linked list >

  return FALSE;   // oops... how did it get here ?
} // end CSpecEventLog::AddEntryIntern()


//--------------------------------------------------------------------------
BOOL CSpecEventLog::AddEntry( double dblTime,  double dblFrequency, int iFlags, char *pszText )
   // Adds an entry in the double-linked list, keeping the list
   // SORTED BY TIME-OF-RECORDING (!)
{
  BOOL fResult = AddEntryIntern( dblTime, dblFrequency, iFlags, pszText );
  m_fModified |= fResult;
  return fResult;
} // end CSpecEventLog::AddEntry()


//--------------------------------------------------------------------------
BOOL CSpecEventLog::ReturnEntry( T_SpecLogEntry *pEntry, // input: pointer into list
                  double *pdblTime_out,      // output: oldest time found (A BIT OLDER than dblTime_in)
                  double *pdblFrequency_out, // output: frequency of interest
                  int    *piFlags_out,   // output: "flags"  (how to mark this in the spectrogram)                  
                  char *pszText_out, int iMaxLen )   // returned text, max text length
{
  int iCopyLen;
  if( pEntry==NULL )  return FALSE;
  if( pdblTime_out )
     *pdblTime_out = pEntry->dblTime;
  if( pdblFrequency_out )
     *pdblFrequency_out = pEntry->dblFreq;
  if( piFlags_out )
     *piFlags_out = pEntry->iFlags;   
  if( pszText_out!=NULL && (iMaxLen>1) )   // caller wants to retrieve the text too..
   { if( (pEntry->pszText!=NULL) && (pEntry->iAllocdTextSize>1) )
      { iCopyLen = pEntry->iAllocdTextSize-1;
        if( iCopyLen>iMaxLen ) iCopyLen=iMaxLen;
        strncpy(pszText_out, pEntry->pszText, iCopyLen );
        pszText_out[iMaxLen-1] = '\0';    // make sure returned string is TERMINATED
      }
     else // caller wants to retrieve text, but there is none:
      { pszText_out[0] = '\0';
      }
   }
  return TRUE;
} // end CSpecEventLog::ReturnEntry()


//--------------------------------------------------------------------------
BOOL CSpecEventLog::GetNextEntry( double dblTime_in, // input: time to seek (excluding THIS EXACT value)
        double *pdblTime_out,  // output: oldest time found (but younger than dblTime_in)
        double *pdblFrequency_out, // output: frequency of interest
        int    *piFlags_out,   // output: "flags"  (how to mark this in the spectrogram)
        char *pszText_out, int iMaxLen )   // returned text, max text length
   // searches for an entry with the next HIGHER timestamp value (in other words, "younger").
{
  T_SpecLogEntry *pEntry = m_pListStart; // start search at the start of the list = OLDEST entry
  double d;
  while( pEntry != NULL )
   { d = pEntry->dblTime - dblTime_in;
     if( d>0.0 ) // ex: pEntry->dblTime >= dblTime, but this is easier for debugging..
      { // bingo ... found what the caller is looking for
        return ReturnEntry( pEntry, pdblTime_out, pdblFrequency_out, piFlags_out, pszText_out, iMaxLen );
      }
     pEntry = pEntry->pNext;
   }
  return FALSE;  // arrived here: not found !
} // end CSpecEventLog::GetNextEntry()

//--------------------------------------------------------------------------
BOOL CSpecEventLog::GetPrevEntry( double dblTime_in, // input: time to seek (excluding THIS EXACT value)
                  double *pdblTime_out,  // output: time of preceding entry (A BIT LATER than dblTime_in)
                  double *pdblFrequency_out, // output: frequency of interest
                  int    *piFlags_out,   // output: "flags"  (how to mark this in the spectrogram)
                  char *pszText_out, int iMaxLen )  // returned text, max text length
{
  T_SpecLogEntry *pEntry = m_pListEnd; // start search at the end(!) of the list = YOUNGEST entry
  double d;
  while( pEntry != NULL )
   { d = pEntry->dblTime - dblTime_in;
     if( d<0.0 ) // ex: pEntry->dblTime < dblTime, but this is easier for debugging..
      { // bingo ... found what the caller is looking for: a YOUNGER ("previous") entry
        return ReturnEntry( pEntry, pdblTime_out, pdblFrequency_out, piFlags_out, pszText_out, iMaxLen );
      }
     pEntry = pEntry->pPrev;
   }
  return FALSE;  // arrived here: not found !
} // end CSpecEventLog::GetPrevEntry()


//--------------------------------------------------------------------------
BOOL CSpecEventLog::GetNearestEntry( double dblTime_in, // input: time to seek
              double *pdblTime_out,      // output: time of nearest entry (no matter if older or younger)
              double *pdblFrequency_out, // output: frequency of nearest entry
              int    *piFlags_out,       // output: "flags"  (how to mark this in the spectrogram)
              char *pszText_out, int iMaxLen )  // returned text, max text length
{
  T_SpecLogEntry *pEntry = m_pListStart; // start search at the start of the list = OLDEST entry
  double d1, d2=-1e6;
  while( pEntry != NULL )
   { d1 = pEntry->dblTime - dblTime_in;
     if( (d1>=0.0) || (pEntry->pNext==NULL) )
      { // may have found what the caller is looking for...
        if( (pEntry->pPrev!=NULL) && (fabs(d2)<fabs(d1)) )
         { // the previous entry was closer to the wanted timestamp than the present:
           pEntry = pEntry->pPrev;
         }
        return ReturnEntry( pEntry, pdblTime_out, pdblFrequency_out, piFlags_out, pszText_out, iMaxLen );
      }
     d2 = d1;
     pEntry = pEntry->pNext;
   }
  return FALSE;  // arrived here: not found !
} // end CSpecEventLog::GetNearestEntry()



// EOF <SpecEventLog.cpp>

