//---------------------------------------------------------------------------
// File  :  c:\cbproj\SoundUtl\SpecBuff.h
// Date  :  2013-05-18     (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Implementation of a class for decimating and buffering audio samples.
//
// Revision history (YYYY-MM-DD):
//   See SpecBuff.cpp !
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#ifndef SpecBuffH
#define SpecBuffH

#include "SoundDec.h"     // using T_SOUND_DECIMATOR here
// Note: SoundDec.h includes an awful lot of other headers, like:
//       SoundTab.h ,  ChunkInfo.h ,   ....

#ifndef TSpectrumH
  #include "TSpectrum.h"  // C:\CBproj\SpecLab\TSpectrum.h
#endif

//----------------------- Constants ---------------------------------------



//----------------------- Data Types --------------------------------------

typedef struct // TFloatArray (dynamic, since 2010-05-17)
{
  T_Float *pfltData;
  DWORD dwMagic27182818;  // first 'magic' number (0x27182818 when valid)
  DWORD dwAllocatedSize;
  DWORD dwMagic31415926;  // second 'magic' number (0x31415926 when valid)
} TFloatArray;


   // Since 2007-12-23, the entries in the buffer file may have individual sizes.
   //  To find a certain entry (identified by the 64-bit "index"),
   //  a look-up table with the following contents was implemented :
typedef struct // T_SpectrumFileLookupTableEntry
{
  LONGLONG i64EntryIndex;  // 64-bit "entry index" for CSpectrumBuffer::GetSpectrum()
  DWORD    dwFilePos;      // file position (offset) for this entry; 0=invalid .

} T_SpectrumFileLookupTableEntry;


// File header for saving a lot of spectra in a disk file .
// Any reader of the file finds all required info in the header.
typedef struct // T_SpectrumFileHeader, will be written to the beginning of the file
{
  char  szFileTextInfo[64];  // "Spectrum Lab Buffer File\r\n\n\0\0\0.."
        //  (will be visible if someone tries to open the file with a viewer or text editor.
        //   The string "Spectrum" will also be checked when opening such a file)

  DWORD cbHeaderStructSize; // may be used for compatibility checks .
                            // Should contain sizeof(T_SpectrumFileHeader) .
  // This is also the offset from the file start to the 1st data entry.
  // Append new components of a structure always at the end !
  //  As long as nothing is changed in the old components,
  //  compatibility can be maintained if the reader evaluates cbStructSize
  //  and ignores unused components.

  DWORD dwSizeOfArrayEntry; // sizeof(T_SPECTRUM), but only the USED PART(!)
  DWORD dwMaxFileSize;      // for finding out when to "wrap" the current position.
        // When opening a spectrum file, it may not have already grown to this size !
        // Maximum file size is 2^31 bytes = approx. 2 GByte (using signed 32 bit ints).

  LONGLONG i64TotalEntryCount;  // The total number of spectra ..
                                // entered since the creation of the file.
   // With a modulo operation, the file position could be calculated like this:
   // 1.) NumberOfEntriesInFile = trunc(Heaer.dwMaxFileSize / Header.dwSizeOfArrayEntry)
   // 2.) FileArrayIndex = NonWrappingIndex % NumberOfEntriesInFile
   //     for latest entry: NonWrappingIndex = (TotalEntryCount-1)
   // 3.) File Pointer = sizeof(T_SpectrumFileHeader)
   //                    + FileArrayIndex * Header.dwSizeOfArrayEntry
   // But, since 2007-12-23, the entries in the buffer file MAY have
   //      different sizes, so Header.dwSizeOfArrayEntry is rather useless now.

  // Added 2007-12-27 (only valid if cbHeaderStructSize == sizeof(T_SpectrumFileHeader) ) :
  DWORD dwNextFilePosForWriting; // file position for next data record (to be written)
  T_SpectrumFileLookupTableEntry LookupTable[100];  // array to speed up searching a certain entry
  #define SPECBUFF_HEADER_LOOKUP_TABLE_SIZE 100

} T_SpectrumFileHeader;



//------------------------ Global Variables -------------------------------



//**************************************************************************
//    Prototypes (no class methods)
//**************************************************************************

extern void MainForm_TermMsgCallback(char *pszWhatsUp); // somewhere in the main form

void SpecBuf_ClearTFloatArray( T_Float *pFltDst, DWORD dwArrayLength );

BOOL TFloatArray_IsValid( TFloatArray *pFltArray );
BOOL TFloatArray_SetSize( TFloatArray *pFltArray, DWORD dwNewSize );
DWORD TFloatArray_GetSize( TFloatArray *pFltArray );


void SpecBuf_InitSpectrumTextMsgStruct( T_SPECTRUM_TEXT_MSG *pSpTextMsg );

BOOL SpecBuf_IsValidSpectrum( T_SPECTRUM *pSpectrum ); // checks if a spectrum struct is "valid" (since 09/2007)
DWORD SpecBuf_GetSizeOfSpectrumInRAM( T_SPECTRUM *ps );
long SpecBuf_GetMaxNrOfFrequencyBins( T_SPECTRUM *pSpectrum );
BOOL SpecBuf_LimitFrequencyBinIndices( T_SPECTRUM *pSpectrum, int *pIndex_From, int *pIndex_To );
double SpecBuf_GetNormalizedPowerFromFrequencyBin( T_SPECTRUM *pSpectrum, int iFreqBinIndex );
double SpecBuf_GetAmplitudeFromFrequencyBin( T_SPECTRUM *pSpectrum, int iFreqBinIndex );
double SpecBuf_GetAmpl_dB_FromFrequencyBin( T_SPECTRUM *pSpectrum, int iFreqBinIndex );

T_SPECTRUM *NewSpectrum( T_SpecDataType nSpectrumDataType, int nCompsPerBin, int iNrOfFrequencyBins ); // allocates and initializes a new spectrum struct
T_SPECTRUM *NewEmptySpectrum(void);
void DeleteSpectrum(T_SPECTRUM **ppSpectrum); // frees an allocated spectrum, and "kills" the pointer to it

BOOL CopySpectrum( T_SPECTRUM *psSrc,  // pointer to source spectrum
                  T_SPECTRUM **ppsDst);// pointer to pointer to destination spectrum






//***************************************************************************
//  Definition of the  CSpectrumBuffer   class
//***************************************************************************

class CSpectrumBuffer
{
public:
   CSpectrumBuffer();            // constructor without parameters
   virtual ~CSpectrumBuffer();   // destructor (cleans up)

   BOOL Init(
      DWORD dwMaxBufSize_RAM,        // max. allowed size of buffer IN RAM [Bytes]
      char *pszFileBufferName,       // name of a BUFFER FILE (NULL= use RAM only)
      T_SpecDataType nInitialDataType, // SPEC_DT_FLOAT_ARRAY, SPEC_DT_RDF_SPECTRUM, ...
      DWORD dwMaxFftBinsPerEntry,    // max. entries in T_SPECTRUM.float fltSpectrum[]
      DWORD dwMaxFileSize ,          // max. acceptable buffer file size in BYTES
      BOOLEAN fCreateNewFile );      // TRUE = create a new buffer file; FALSE = open existing
   BOOL Close(void);
   BOOL Clear(void);    // clears the contents of the spectrogram buffer ( RAM, possibly also FILE)
   void FlushAll(void); // Flushes the buffer to disk, and updates the file header
                        //     *IF* the buffer uses a file at all ...

   BOOL AppendSpectrum( T_SPECTRUM *pSpectrum, double dblAudioCenterFrequency );
         // Appends a SPECTRUM (or, sometimes, a TEXT MESSAGE) to the buffer
# if( SPECDISP_TEXT_MESSAGES_IN_SPECTRUM_BUFFER )
   BOOL AppendTextMsg( T_SPECTRUM_TEXT_MSG *pSpectrumTextMsg );
# endif

   BOOL GetSpectrum( LONGLONG i64EntryIndex, T_SPECTRUM **pSpectrum );
   T_SPECTRUM * GetSpectrumByTimestamp( double dblTimestamp );
   T_SPECTRUM * GetSpectrum2( int iSourceChannel, LONGLONG *pi64TailIndex );


   DWORD    GetMaxFileSize(void);
   LONGLONG GetIndexOfLatestEntry(void);  // (can be a spectrum, OR A TEXT MESSAGE)
   LONGLONG GetIndexOfOldestEntry(void);
   LONGLONG GetIndexOfLatestSpectrum(void);  // notice the difference !
   LONGLONG GetIndexOfOldestSpectrum(void);  // Since 2013-05-18, text messages are in an extra buffer, this there's not such a big difference anymore...
   LONGLONG GetIndexOfEntryByTime(double dblTime);
   long     GetCountOfBufferedLines(void);
   long     GetMaxCountOfBufferedLines(void);
   int      GetNumberOfSourceChannels(void);
   char *   GetErrorString(void);

private:
   void     SetIndexOfLatestSpectrum( LONGLONG i64EntryIndex );

   //-----------------------------------------------------------------------
   // General buffer information  and status
   LONGLONG m_i64TotalEntryCount;  // the total number of spectra OR TEXT MESSAGES
            // (entered since the creation or opening of the file buffer)

   LONGLONG m_i64IndexOfLatestSpectrum;
 //LONGLONG m_i64IndexOfLatestTextMessage;

   DWORD    dwLastError;
   char     sz80LastError[82];
   T_SPECTRUM *m_psTempSpectrum;      // working copy for "shrinking" entries ETC(!)
   int      iLatestSourceChannels[4];


   //-----------------------------------------------------------------------
   // All information about the RAM BUFFER (always used, with and without FILE buffer)
   T_SPECTRUM** m_PtrArray;  // array of POINTERS to dynamically allocated spectra
   int   m_iPtrArrayAllocatedEntries; // allocated number of entries in m_PtrArray[]
   int   m_iPtrArrayUsedEntries;      // currently USED(!) number of entries in m_PtrArray[]
      // m_PtrArray[0]                        = OLDEST entry in the spectrum buffer;
      // m_PtrArray[m_iPtrArrayUsedEntries-1] = NEWEST entry in the spectrum buffer.
   BYTE* m_pbBuffer;     // 'large' allocated block; circular (!)
   DWORD m_dwAllocatedBytesInBuffer;   // allocated number of bytes in m_pbBuffer[]
   DWORD m_dwBufferHeadIndex; // index (into m_pbBuffer) for the next APPENDED entry
   // end of RAM BUFFER info

   void DeleteEntriesForBufferIndices( DWORD dwBufIdxLo, DWORD dwBufIdxHi );


   //-----------------------------------------------------------------------
   // All information about the BUFFER FILE, handle, copy of the header, etc.
   char   m_sz1023FileName[1024];   // file name; required to close and re-open
   HANDLE m_hFile         ;  // File handle, if not used: INVALID_HANDLE_VALUE
   DWORD  m_dwCurrentFilePos; // for traditional access, to minimize SetFilePos()-calls
   DWORD  m_dwCurrentFilePosForWriting;
   DWORD  m_dwCurrentFileSize;
   DWORD  m_dwMaxFileSizeInBytes; // limiter for the file size when writing. Does not get lost when header init'd..
   //DWORD  m_dwSizeOfArrayEntry;   // ex: sizeof(T_SPECTRUM), but only the USED PART(!)

   // A copy of the header. Must be filled before a mapped file exists,
   // so this is real memory !
   T_SpectrumFileHeader  m_FileHeader;
   BOOL   m_fMustFlushHeader;

   // Since 2007-12-23, the entries in the buffer file may have individual sizes.
   //  To find a certain entry (identified by the 64-bit "index"),
   //  the following look-up table was implemented :
   T_SpectrumFileLookupTableEntry *m_pFileLookupTable;
   DWORD m_dwFileLookupTableNAllocatedEntries;  // number of entries allocated in m_pFileLookupTable
   DWORD m_dwFileLookupTableUsedEntries;        // number of entries actually USED (so far)
   DWORD m_dwFileLookupTableEstimatedRequiredEntries; // 'estimated' number of entries, may increase during runtime
   BOOL  m_fFileLookupTableIsUpToDate;
   LONGLONG m_i64FileLookupTableFirstSeekedIndex, m_i64FileLookupTableLastSeekedIndex;
   // end of spectrum FILE info

   BOOL OpenBufferFile(char *pszFileName,BOOLEAN fCreateNewFile);
   BOOL SetFilePosIfNeeded(DWORD dwNewFilePos);
   BOOL WriteFileHeader(void);
   void CreateFileLookupTable( long i32MinNumberOfEntries );
   void DeleteFileLookupTable(void);
   BOOL QSearchInFileLookupTable(LONGLONG i64EntryIndex, DWORD *pdwLookupTableIndex);
   void AddEntryToFileLookupTable( LONGLONG i64EntryIndex, DWORD dwFilePos );
   BOOL ReadHeadersFromFile( LONGLONG i64WantedEntryIndex, DWORD *pdwFilePos, T_SPECTRUM_HDR *pHdr );
   BOOL EntryIndexToFilePos( LONGLONG i64EntryIndex, DWORD *pdwFilePos);
   BOOL WriteSpectrumToFile( T_SPECTRUM *psSrc, double dblCenterFrequency );
   BOOL ReadSpectrumFromFile(LONGLONG i64EntryIndex, T_SPECTRUM **ppsDst );
   void CloseFile(void);


}; // end class CSpectrumBuffer


#endif // SpecBuffH