//---------------------------------------------------------------------------
// File  :  SpecBuff.cpp
// Date  :  2013-05-18     (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Spectrum Buffer functions .
//
// Revision history (YYYY-MM-DD, latest entry first):
//   2013-05-18  The 'spectrum display buffer' is now also accessed by the
//               HTTP buffer, when a remote client (web browser?) requests
//               arbitrary parts from that buffer (arbitrary frequency-
//               and time-ranges).
//   2012-08-26  Max. size of the RAM-BUFFER (array) now expressed in BYTES,
//                 not as a maximum number of spectrogram lines.
//   2010-05-17  Added a new data type with a simple array of T_Float numbers:
//                 TFloatArray .
//   2009-06-06  Added a new data type for buffering cells of the
//                 REASSIGNED SPECTROGRAM (a la Fulop/Fitz; see ra_spectrum.cpp)
//   2007-12-23  File buffer changed: Now supporting different
//               sizes of the entries written. Previous version
//               saved as SpecBuff_071223.* .
//   2005-05-21  Added debugging stuff like 'dwIntegrityCheck123'
//                because there were suspicious crashes in SpecLab
//                when the spectrum buffer was accessed (r: G7IZU).
//                But this built-in "debugger" never trapped ..
//   2004-05-02  Added the possibility to embed TEXT MESSAGES
//                in the buffer. May be used one day to insert LABELS
//                in the spectrogram display .
//   2004-01-19  Added the method FlushAll() to prevent loss of data
//                in case of PC crash. May be periodically called .
//   2002-08-02  Written for SpecLab, with memory mapped file option
//               DOES NOT USE VCL STUFF,
//                  ONLY PLAIN WIN API CALLS & standard data types
//                  to have a compact DLL without VCL dependencies !
//---------------------------------------------------------------------------


#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !



#include <windows.h>
#include <math.h>

#pragma hdrstop        // no precompiled headers after this point

#include "Utility1.h"  // stuff like DEBUG_EnterErrorHistory, LastErrorCodeToString, ..
#include "Timers.h"    // stuff like TIM_GetCurrentUnixDateAndTime(), ...
#include "TSpectrum.h" // definition of T_SPECTRUM in C:\CBproj\SpecLab\TSpectrum.h
#include "SpecBuff.h"  // header for this module

#pragma warn -8017



//**************************************************************************
//    Global & Static (internal) Variables
//**************************************************************************

static nSpectrumCopyErrors = 0;

//**************************************************************************
//    Functions (no class methods; no C++; may compile this with PellesC once)
//**************************************************************************

//---------------------------------------------------------------------------
BOOL TFloatArray_IsValid( TFloatArray *pFltArray )
{
  if( pFltArray != NULL )
   {
     // If the structure is obviously 'valid' already, FREE THE OLD BUFFER :
     if(  (pFltArray->pfltData!=NULL) && (pFltArray->dwAllocatedSize>0)
       && (pFltArray->dwMagic27182818==0x27182818)
       && (pFltArray->dwMagic31415926==0x31415926) )
      {
        return TRUE;
      }
   }
  return FALSE;
} // end TFloatArray_IsValid()

//---------------------------------------------------------------------------
DWORD TFloatArray_GetSize( TFloatArray *pFltArray )
{
  if( TFloatArray_IsValid(pFltArray) )
   { return pFltArray->dwAllocatedSize;
   }
  else
   { return 0;
   }
} // end TFloatArray_GetSize()

//---------------------------------------------------------------------------
BOOL TFloatArray_SetSize( TFloatArray *pFltArray, DWORD dwNewSize )
  // Modifies the size of the array, if necessary.
  // If not necessary (because the size is already sufficient),
  //   this function doesn't consume significant time
  //   so it's safe to call it before any array access (BEFORE a loop).
  // TFloatArray_SetSize( array, 0 )   frees the array .
{
  BOOL fStructValid;
  DWORD dw;
  if( pFltArray != NULL )
   {
     // If the buffer (array) already exists, FREE IT if the size shall be modified:
     fStructValid = TFloatArray_IsValid( pFltArray );
     if( fStructValid && (pFltArray->pfltData!=NULL) )
      { // The structure is already valid -> free the old array,
        // if the size is different from the new size :
        if( (pFltArray->dwAllocatedSize != dwNewSize )
           || (dwNewSize==0) ) // called to FREE this buffer
         { pFltArray->dwAllocatedSize = 0;
           UTL_free( pFltArray->pfltData );
           pFltArray->pfltData = NULL;
         }
      } // end if( fStructValid && (pFltArray->pfltData!=NULL) )
     if( (dwNewSize>0) && (pFltArray->pfltData==NULL) )
      { pFltArray->pfltData = (T_Float*)UTL_NamedMalloc( "TFloatAr", sizeof(T_Float)*dwNewSize );
        if( pFltArray->pfltData != NULL )
         { for(dw=0; dw<dwNewSize; ++dw)
            { pFltArray->pfltData[dw] = 0.0;
            }
           pFltArray->dwAllocatedSize = dwNewSize;
           pFltArray->dwMagic27182818 = 0x27182818;
           pFltArray->dwMagic31415926 = 0x31415926; // now open for business
         }
      } // end if( (dwNewSize>0) && (pFltArray->pfltData==NULL) )
     return ( pFltArray->dwAllocatedSize >= dwNewSize );
   } // end if( pFltArray != NULL )
  return FALSE;
} // end TFloatArray_SetSize()



//---------------------------------------------------------------------------
void SpecBuf_ClearTFloatArray( T_Float *pFltDst, DWORD dwArrayLength )
{
  if( pFltDst )
   { while( dwArrayLength-- )
      { *pFltDst++ = 0.0;
      }
   }
} // end SpecBuf_ClearTFloatArray()


//--------------------------------------------------------------------------
void SpecBuf_InitSpectrumTextMsgStruct( T_SPECTRUM_TEXT_MSG *pSpTextMsg )
{
  memset( pSpTextMsg, 0, sizeof(T_SPECTRUM_TEXT_MSG) );
  pSpTextMsg->dblFrequencyOfInterest = 0.0;
  pSpTextMsg->dblMarkerRectF1 = 0.0;
  pSpTextMsg->dblMarkerRectF2 = 0.0;
  pSpTextMsg->dblMarkerRectT1 = 0.0;
  pSpTextMsg->dblMarkerRectT2 = 0.0;
  pSpTextMsg->dwMaxChars = MAX_CHARS_PER_SPEC_TEXT_MSG;
  pSpTextMsg->sTextOptions.dwFontColor_rgb = 0xFFFFFFFF;  // 'invalid' colour (use configured value)
#if( !SPECDISP_TEXT_MESSAGES_IN_SPECTRUM_BUFFER )
  pSpTextMsg->iRotation = -1;  // text rotation in degrees: -1 = "default, depending on waterfall orientation"
  pSpTextMsg->dblTimestamp = TIM_GetCurrentUnixDateAndTime(); // !
#endif // !SPECDISP_TEXT_MESSAGES_IN_SPECTRUM_BUFFER ?

} // end InitSpectrumTextMsgStruct()

//--------------------------------------------------------------------------
DWORD SpecBuf_GetMinAllocSize(
        T_SpecDataType nDataType,
        int nCompsPerBin,  // usually ZERO for automatic detection, depending on nSpectrumDataType
        int iNrOfFrequencyBins  )
  // See also: SpecBuf_GetMinAllocSize() ,  SpecBuf_GetMaxNrOfFrequencyBins() .
{
  int nMinCompsPerBin;
  DWORD dwResult = sizeof( T_SPECTRUM_HDR )
                 + 4/*safety; struct alignment*/
                 + 4/*32-bit safety-check location AFTER the array*/;

  nMinCompsPerBin = 1;
  if( nDataType == SPEC_DT_COMPLEX_ARRAY )
   { nMinCompsPerBin = 2;
   }
  if( nDataType == SPEC_DT_TRIPLE_ARRAY )
   { nMinCompsPerBin = 3;  // three components per bin for each "triple"
     // three floating point values per f-bin in pNewSpectrum->u.pfltSpectrum[i]
   }
  if( nCompsPerBin < nMinCompsPerBin )
   {  nCompsPerBin = nMinCompsPerBin;
   }

  if(iNrOfFrequencyBins<1)
     iNrOfFrequencyBins=1;  // at least ONE array index (for safety..)
  switch(nDataType)
   {
    case SPEC_DT_RDF_SPECTRUM : // type for the RADIO DIRECTION FINDER
         dwResult += iNrOfFrequencyBins * nCompsPerBin * sizeof(T_RDF_SPECTRUM);
         break;

    case SPEC_DT_RA_SPECTRUM: // type for the REASSIGNED SPECTROGRAM, used in ra_spectrum.cpp
         dwResult += iNrOfFrequencyBins * nCompsPerBin * sizeof(T_RA_SPECTRUM);
         break;

    case SPEC_DT_TEXT_MESSAGE :  // in this case all sizes are FIXED,
                                 // and iNrOfFrequencyBins is meaningless
         dwResult += sizeof(T_SPECTRUM_TEXT_MSG);
         break;

    case SPEC_DT_COMPLEX_ARRAY:  // real part in fltSpectrum[2*i], imaginary part in fltSpectrum[2*i+1]
         dwResult += iNrOfFrequencyBins * nCompsPerBin/*ex:2*/ * sizeof(T_Float);
         break;

    case SPEC_DT_TRIPLE_ARRAY:   // X, Y, and Z in fltSpectrum[3*i .. 3*i+2]
         dwResult += iNrOfFrequencyBins * nCompsPerBin/*ex:3*/ * sizeof(T_Float);
         break;

    default:
    case SPEC_DT_FLOAT_ARRAY  :
         dwResult += iNrOfFrequencyBins * nCompsPerBin/*usually:1*/ * sizeof(T_Float);
         break;
   } // end switch(iDataType)

  return dwResult;
} // end SpecBuf_GetMinAllocSize()


//--------------------------------------------------------------------------
DWORD SpecBuf_GetSizeOfSpectrumInRAM( T_SPECTRUM *ps )
  // Determines the total size of a T_SPECTRUM instance in RAM .
{
  if( SpecBuf_IsValidSpectrum( ps ) )
   {  // there is a valid spectrum in memory, just look at the header...
     return ps->hdr.c.cbAllocatedSize;
   }
  else
   {  // there is no valid spectrum in memory...
     return 0;
   }
} // end SpecBuf_GetSizeOfSpectrumInRAM()



//--------------------------------------------------------------------------
long SpecBuf_GetMaxNrOfFrequencyBins( T_SPECTRUM *pSpectrum )
  // Returns the maximum allowed value for pSpectrum->hdr.v.iNrOfBins,
  // depending on the allocated memory size (pSpectrum->hdr.c.cbAllocatedSize)
  // and the spectrum data type (pSpectrum->hdr.v.nDataType) .
  // Note: The number of frequency bins actually used (pSpectrum->hdr.v.iNrOfBins)
  //   may be LESS than this "maximum allowed value", because we don't
  //   automatically free and re-allocate a T_SPECTRUM in memory
  //   just because the FFT size was reduced a tiny bit (to avoid memory fragmentation).
  // See also: SpecBuf_GetMinAllocSize() ,  SpecBuf_GetMaxNrOfFrequencyBins() .
{
  long i32ArraySizeInByte;
  long iMaxNrOfFftBins;
  if( pSpectrum == NULL )
   { return 0;     // invalid pointer; "no frequency bins" for this !
   }

  if(  (pSpectrum->hdr.c.dwMagic != SPECTRUM_HDR_MAGIC ) // see SpecBuff.h for an explanation..
     ||(pSpectrum->hdr.c.cbHdrStructSize!= sizeof(T_SPECTRUM_HDR) ) )
   { return 0;     // wrong header size indication, no frequency bins in there..
   }
  i32ArraySizeInByte = (long)pSpectrum->hdr.c.cbAllocatedSize
       - ( sizeof( T_SPECTRUM_HDR ) + 4/*safety; struct alignment*/ + 4/*test location AFTER the array*/ );

  switch(pSpectrum->hdr.v.nDataType)
   {
    case SPEC_DT_RDF_SPECTRUM :
         iMaxNrOfFftBins = i32ArraySizeInByte / sizeof(T_RDF_SPECTRUM);
         break;

    case SPEC_DT_RA_SPECTRUM: // type for the REASSIGNED SPECTROGRAM, used in ra_spectrum.cpp
         iMaxNrOfFftBins = i32ArraySizeInByte / sizeof(T_RA_SPECTRUM);
         break;

    case SPEC_DT_TEXT_MESSAGE :  // in this case all sizes are FIXED, but..
         iMaxNrOfFftBins = i32ArraySizeInByte / sizeof(T_SPECTRUM_TEXT_MSG);
         break;

    case SPEC_DT_FLOAT_ARRAY  :
         iMaxNrOfFftBins = i32ArraySizeInByte / sizeof(T_Float);
         break;

    case SPEC_DT_COMPLEX_ARRAY:  // uses T_SPECTRUM.fltSpectrum[] for complex pairs (re,im)
         iMaxNrOfFftBins = i32ArraySizeInByte / ( 2 * sizeof(T_Float) );
         break;

    case SPEC_DT_TRIPLE_ARRAY:   // X, Y, and Z in fltSpectrum[3*i .. 3*i+2]
         iMaxNrOfFftBins = i32ArraySizeInByte / ( 3 * sizeof(T_Float) );
         break;

    default:                   // unknown spectrum data type ?
         iMaxNrOfFftBins = 0;  // say "it's an error" (something missing)
         break;

   } // end switch(iDataType)

  if(iMaxNrOfFftBins > MAX_FREQUENCY_BINS_PER_SPECTRUM)
     iMaxNrOfFftBins = MAX_FREQUENCY_BINS_PER_SPECTRUM;

  return iMaxNrOfFftBins;

} // end SpecBuf_GetMaxNrOfFrequencyBins()

//--------------------------------------------------------------------------
double SpecBuf_GetAmplitudeFromFrequencyBin( T_SPECTRUM *pSpectrum, int iFreqBinIndex )
  // Returns the amplitude (or sometimes power) for a single FFT frequency bin .
  // Does *NOT* convert the unit ! ( uses the same unit as in the spectrum;
  //                                 i.e. pSpectrum->hdr.v.scale_unit )
  // To retrieve the amplitude / "magnitude" in decibel,
  //    use SpecBuf_GetAmpl_dB_FromFrequencyBin() instead .
{
  if( pSpectrum == NULL )
   { return 0.0;   // invalid pointer; "no frequency bins" for this !
   }
  if(  (pSpectrum->hdr.c.dwMagic != SPECTRUM_HDR_MAGIC ) // see SpecBuff.h for an explanation..
     ||(pSpectrum->hdr.c.cbHdrStructSize!= sizeof(T_SPECTRUM_HDR) ) )
   { return 0.0;   // wrong header size indication, no frequency bins in there..
   }
  if( iFreqBinIndex<0 || iFreqBinIndex>=pSpectrum->hdr.v.iNrOfBins )
   { return 0.0;   // illegal bin index
   }

  switch(pSpectrum->hdr.v.nDataType)
   {
     case SPEC_DT_RDF_SPECTRUM :
        return pSpectrum->u.pRdfSpectrum[iFreqBinIndex].fltAmpl;

    case SPEC_DT_RA_SPECTRUM: // type for the REASSIGNED SPECTROGRAM, used in ra_spectrum.cpp
        return pSpectrum->u.pRASpectrum[iFreqBinIndex].fltPower;

     case SPEC_DT_TEXT_MESSAGE :  // in this case all sizes are FIXED, but..
        return 0.0;   // a text message doesn't contain frequency bins !

     case SPEC_DT_FLOAT_ARRAY  :
        return pSpectrum->u.pfltSpectrum[iFreqBinIndex];

     case SPEC_DT_COMPLEX_ARRAY:  // uses T_SPECTRUM.fltSpectrum[] for complex pairs (re,im)
      { int ai = 2*iFreqBinIndex;
        double re = pSpectrum->u.pfltSpectrum[ai];
        double im = pSpectrum->u.pfltSpectrum[ai+1];
        return sqrt(re*re + im*im);  // -> amplitude of complex frequency bin
      }

     case SPEC_DT_TRIPLE_ARRAY:  // uses T_SPECTRUM.fltSpectrum[] for 3D triples (x,y,z)
      { int ai = 3*iFreqBinIndex;
        double x = pSpectrum->u.pfltSpectrum[ai];
        double y = pSpectrum->u.pfltSpectrum[ai+1];
        double z = pSpectrum->u.pfltSpectrum[ai+2];
        return sqrt(x*x + y*y + z*z);  // -> length of the vector (3D)
      }

     default:                     // unknown spectrum data type ?
        return 0.0;
   } // end switch(iDataType)

} // end SpecBuf_GetAmplitudeFromFrequencyBin()

//--------------------------------------------------------------------------
double SpecBuf_GetNormalizedPowerFromFrequencyBin( T_SPECTRUM *pSpectrum, int iFreqBinIndex )
  // Returns the POWER for a single FFT frequency bin, frequently used
  //    by signal averaging functions (adding or averaging POWERS makes
  //    more sense than adding decibels) .
  //    Strictly said, the unit of the returned value is SCALE_UNIT_NORM_PWR;
  //    i.e. "power, normalized to 0...1 for full A/D converter input range" .
  // To retrieve the amplitude / "magnitude" in decibel,
  //    use SpecBuf_GetAmpl_dB_FromFrequencyBin() instead .
{
  double d;

  if( pSpectrum == NULL )
   { return 0.0;   // invalid pointer; "no frequency bins" for this !
   }
  if(  (pSpectrum->hdr.c.dwMagic != SPECTRUM_HDR_MAGIC ) // see SpecBuff.h for an explanation..
     ||(pSpectrum->hdr.c.cbHdrStructSize!= sizeof(T_SPECTRUM_HDR) ) )
   { return 0.0;   // wrong header size indication, no frequency bins in there..
   }
  if( iFreqBinIndex<0 || iFreqBinIndex>=pSpectrum->hdr.v.iNrOfBins )
   { return 0.0;   // illegal bin index
   }

  switch(pSpectrum->hdr.v.nDataType)
   {
     case SPEC_DT_RDF_SPECTRUM :
        d = pSpectrum->u.pRdfSpectrum[iFreqBinIndex].fltAmpl;
        break;

    case SPEC_DT_RA_SPECTRUM: // type for the REASSIGNED SPECTROGRAM, used in ra_spectrum.cpp
        d = pSpectrum->u.pRASpectrum[iFreqBinIndex].fltPower;
        break;

     case SPEC_DT_TEXT_MESSAGE :  // in this case all sizes are FIXED, but..
        d = 0.0;   // a text message doesn't contain frequency bins !
        break;

     case SPEC_DT_FLOAT_ARRAY  :
        d = pSpectrum->u.pfltSpectrum[iFreqBinIndex] * pSpectrum->hdr.v.scale_factor;
        break;

     case SPEC_DT_COMPLEX_ARRAY:  // uses T_SPECTRUM.fltSpectrum[] for complex pairs (re,im)
      { int ai = 2*iFreqBinIndex;
        double re = pSpectrum->u.pfltSpectrum[ai];
        double im = pSpectrum->u.pfltSpectrum[ai+1];
        return re*re + im*im;  // -> power from complex frequency bin
      }

     case SPEC_DT_TRIPLE_ARRAY:  // uses T_SPECTRUM.fltSpectrum[] for 3D triples (x,y,z)
      { int ai = 3*iFreqBinIndex;
        double x = pSpectrum->u.pfltSpectrum[ai];
        double y = pSpectrum->u.pfltSpectrum[ai+1];
        double z = pSpectrum->u.pfltSpectrum[ai+2];
        return x*x + y*y + z*z;  // -> length of the vector (3D)
      }

     default:                     // unknown spectrum data type ?
        d = 0.0;
   } // end switch(iDataType)

  // Arrived here, 'd' contains a voltage, power, dB-value, or whatever .
  // Convert this into a POWER (or 'energy') :
  if( (pSpectrum->hdr.v.scale_unit & SCALE_UNIT_MASK_dB) == SCALE_UNIT_dB)
   {    // 'd' is a value in dB : Convert from dB (usually dBFS) to a POWER
        // Warning: forget "pow10" and "pow10l" here !!!!!
        // Though the LOUSY help system says:
        //    > pow10l is the long double version;
        //    > it takes long double arguments     <<<<<< TERRIBLY WRONG!
        //    > and returns a long double result.
        //  (taken from Borland C++Builder V4.0 online help system),
        // in fact both "pow10" AND "pow10l" both take bloody INTEGER
        // arguments !!!!!!! (Took a few minutes to find this out).
        //   power(rel.) = exp( <dB_value> * ln(10) )
        //   example: exp( -3[db]  *  ln(10)/10 ) = 0.5 (W)
     d = exp( d * 0.2302585093/*ln(10)/10*/ );
   }
  else   // 'd' is not in dB
  switch( pSpectrum->hdr.v.scale_unit )
   {
     case SCALE_UNIT_PERCENT:  // (2) "percentage of full scale", also proportinal to a VOLTAGE
     case SCALE_UNIT_VOLT:     // (3) something proportinal to a VOLTAGE
        // ToDo : multiply 'd' with some scaling factor
     case SCALE_UNIT_NORM_V:   // (1) normalized ADC units (voltage), +/- 1.0  for full A/D converter input range
        d = d*d;               // squared voltage; proportional to a POWER
        break;
     case SCALE_UNIT_WATT:     // (4) power (W), 'calibrated' with UTL_dblAdcMaxInputVoltage + UTL_dblAdcInputImpedance
     case SCALE_UNIT_NORM_PWR: // (5) power, normalized to 0...1 for full A/D converter input range
        break;
   } // end switch( pSpectrum->hdr.v.scale_unit )

  return d;
} // end SpecBuf_GetNormalizedPowerFromFrequencyBin()


//--------------------------------------------------------------------------
double SpecBuf_GetAmpl_dB_FromFrequencyBin( T_SPECTRUM *pSpectrum, int iFreqBinIndex )
  // Returns the amplitude *in decibel* for a single FFT frequency bin .
  // To retrieve the amplitude without unit conversion,
  //    use SpecBuf_GetAmplitudeFromFrequencyBin() instead .
{
  double dblAmpl = SpecBuf_GetAmplitudeFromFrequencyBin(pSpectrum,iFreqBinIndex);
  double d;
  if( pSpectrum != NULL )
   { // What is the physical unit of this thing (defined in UTILITY1.H) :
     if( (pSpectrum->hdr.v.scale_unit & SCALE_UNIT_MASK_dB) != SCALE_UNIT_dB)
      { // The values is *NOT* scaled into decibel yet .
        // Convert them to decibels here:
        if( (pSpectrum->hdr.v.scale_unit == SCALE_UNIT_WATT )    // power (W), 'calibrated' with UTL_dblAdcMaxInputVoltage + UTL_dblAdcInputImpedance
          ||(pSpectrum->hdr.v.scale_unit == SCALE_UNIT_NORM_PWR) // power, normalized to 0...1 for full A/D converter input range
          )
         { d = 10.0;
         }
        else // everything else is considered "proportional to a VOLTAGE" :
         { d = 20.0;
         }
        if(dblAmpl>0) dblAmpl = d * log10( dblAmpl );
                 else dblAmpl = -200; // pretty low value, "almost zero"
      }
   }
  return dblAmpl;
} // end SpecBuf_GetAmpl_dB_FromFrequencyBin()

//--------------------------------------------------------------------------
BOOL SpecBuf_IsValidSpectrum( T_SPECTRUM *pSpectrum )
{
  DWORD dwMinAllocSize;
  DWORD *pdwSafetyCheck; // .. right after the array data

  if( pSpectrum != NULL ) // a NULL pointer can never point to a valid spectrum..
   {
     if(  (pSpectrum->hdr.c.dwMagic == SPECTRUM_HDR_MAGIC ) // see SpecBuff.h for an explanation..
        &&(pSpectrum->hdr.c.cbHdrStructSize== sizeof(T_SPECTRUM_HDR) ) )
      { // header size, and "magic number" ok ..
        dwMinAllocSize  = SpecBuf_GetMinAllocSize(
             pSpectrum->hdr.v.nDataType,
             pSpectrum->hdr.v.iCompsPerBin,
             pSpectrum->hdr.v.iNrOfBins );
        if( pSpectrum->hdr.c.cbAllocatedSize >= dwMinAllocSize )
         {
           // Test the contents of the "safety check location" (after the array data) :
           pdwSafetyCheck = (DWORD*)( (BYTE*)pSpectrum + pSpectrum->hdr.c.cbAllocatedSize - 4);
           if( *pdwSafetyCheck != 0x31415926 ) // magic number (AFTER the array data) still ok ?
            { // ooops... no... someone must have ignored the maximum array size !
              UTL_WriteRunLogEntry( "PANIC: Someone destroyed a T_SPECTRUM structure in memory !" );
              return FALSE;  // TERROR ! PANIC !
            }

           return TRUE;    // all "ok"
         }
      }
   }
  return FALSE;     // Should NEVER arrive here ... set a breakpoint here !
} // end SpecBuf_IsValidSpectrum()


//--------------------------------------------------------------------------
BOOL SpecBuf_LimitFrequencyBinIndices( T_SPECTRUM *pSpectrum, int *pIndex_From, int *pIndex_To )
{ int iSwap;

  if( ! SpecBuf_IsValidSpectrum( pSpectrum ) )
   { *pIndex_From = *pIndex_To = 0;
     return FALSE;
   }
  if( *pIndex_From < 0 )
      *pIndex_From = 0;
  if( *pIndex_To >= pSpectrum->hdr.v.iNrOfBins )
      *pIndex_To = pSpectrum->hdr.v.iNrOfBins-1;
  if( *pIndex_From > *pIndex_To )
   {  iSwap = *pIndex_From;
      *pIndex_From = *pIndex_To;
      *pIndex_To = iSwap;
   }
  return TRUE;

} // end SpecBuff_LimitFrequencyBinIndices()


//--------------------------------------------------------------------------
T_SPECTRUM *NewSpectrum(
        T_SpecDataType nSpectrumDataType,
        int nCompsPerBin,  // usually ZERO for automatic detection, depending on nSpectrumDataType
        int iNrOfFrequencyBins  )
{
  T_SPECTRUM *pNewSpectrum;
  DWORD *pdwSafetyCheck; // .. right after the array data
  DWORD dwAllocSize;
  T_Float *pfltDest;
  int i, nMinCompsPerBin;

  if( iNrOfFrequencyBins < 0 )
      iNrOfFrequencyBins = 0;
  if( iNrOfFrequencyBins > MAX_FREQUENCY_BINS_PER_SPECTRUM )
      iNrOfFrequencyBins = MAX_FREQUENCY_BINS_PER_SPECTRUM;

  nMinCompsPerBin = 1;
  if( nSpectrumDataType == SPEC_DT_COMPLEX_ARRAY )
   { nMinCompsPerBin = 2;
   }
  if( nSpectrumDataType == SPEC_DT_TRIPLE_ARRAY )
   { nMinCompsPerBin = 3;  // three components per bin for each "triple"
     // three floating point values per f-bin in pNewSpectrum->u.pfltSpectrum[i]
   }
  if( nCompsPerBin < nMinCompsPerBin )
   {  nCompsPerBin = nMinCompsPerBin;
   }
  if( nCompsPerBin > 8 )
   {  nCompsPerBin = 8; // should be enough for 4 channels with COMPLEX values each
   }


  // Calculate the size of the T_SPECTRUM struct, consisting of HEADERS+ARRAYS:
  dwAllocSize = SpecBuf_GetMinAllocSize( nSpectrumDataType, nCompsPerBin, iNrOfFrequencyBins );

  // Try to allocated a sufficiently large memory block. THIS MAY(!) FAIL :
  pNewSpectrum = (T_SPECTRUM *) UTL_NamedMalloc( "Spectrum", dwAllocSize);
  if(pNewSpectrum)
   {
     pNewSpectrum->hdr.c.dwMagic = SPECTRUM_HDR_MAGIC; // see SpecBuff.h for an explanation..
     pNewSpectrum->hdr.c.cbHdrStructSize = sizeof(T_SPECTRUM_HDR); // may be used for compatibility checks
     pNewSpectrum->hdr.c.cbAllocatedSize = dwAllocSize;
     pNewSpectrum->hdr.v.nDataType   = nSpectrumDataType;
     pNewSpectrum->hdr.v.dbl_time    = 0.0;
     pNewSpectrum->hdr.v.bin_width_hz= 1.0;
     pNewSpectrum->hdr.v.af_offset_hz = pNewSpectrum->hdr.v.rf_offset_hz = 0.0;
     pNewSpectrum->hdr.v.iNrOfBins   = iNrOfFrequencyBins;
     pNewSpectrum->hdr.v.iCompsPerBin= nCompsPerBin;  // 1 for "normal" spectra, 2 for complex pairs etc
     pNewSpectrum->hdr.v.scale_unit  = 0;
     pNewSpectrum->hdr.v.scale_factor= 0.0;
     pNewSpectrum->hdr.v.scale_offset = 0.0;

     // Set the contents of the "safety check location", right after the array data :
     //  ( the additional 4 bytes have been considered in SpecBuf_GetMinAllocSize )
     pdwSafetyCheck = (DWORD*)( (BYTE*)pNewSpectrum + dwAllocSize - 4);
     *pdwSafetyCheck = 0x31415926; // no magic, just "pi"

     // Now fill the array data, so we have VALID FLOATING POINT NUMBERS everywhere:
     switch(nSpectrumDataType)
      {
       case SPEC_DT_RDF_SPECTRUM :
            for(i=0;i<iNrOfFrequencyBins;++i)
             { pNewSpectrum->u.pRdfSpectrum[i].fltAmpl = SPECTRUM_UNINITIALIZED_ARRAY_VALUE; // ex: 0.0;
               pNewSpectrum->u.pRdfSpectrum[i].i16Azim = 0;
               pNewSpectrum->u.pRdfSpectrum[i].i16Elev = 0;
             }
            break;

       case SPEC_DT_RA_SPECTRUM: // type for the REASSIGNED SPECTROGRAM, used in ra_spectrum.cpp
            for(i=0;i<iNrOfFrequencyBins;++i)
             { pNewSpectrum->u.pRASpectrum[i].fltPower = 0.0;
               pNewSpectrum->u.pRASpectrum[i].fltCIF   = 0.0;
               pNewSpectrum->u.pRASpectrum[i].fltLGD   = 0.0;
             }
            break;

       case SPEC_DT_TEXT_MESSAGE :
            SpecBuf_InitSpectrumTextMsgStruct( &pNewSpectrum->u.TextMsg );
            break;

       case SPEC_DT_FLOAT_ARRAY  :
            for(i=0;i<iNrOfFrequencyBins;++i)
             { pNewSpectrum->u.pfltSpectrum[i] = SPECTRUM_UNINITIALIZED_ARRAY_VALUE; // ex: 0.0;
             }
            break;

       case SPEC_DT_COMPLEX_ARRAY  :
            pfltDest = pNewSpectrum->u.pfltSpectrum;
            for(i=0;i<iNrOfFrequencyBins;++i)
             { *(pfltDest++) = SPECTRUM_UNINITIALIZED_ARRAY_VALUE; // REAL part
               *(pfltDest++) = SPECTRUM_UNINITIALIZED_ARRAY_VALUE; // IMAGINARY part
             }
            break;

       case SPEC_DT_TRIPLE_ARRAY  :
            pfltDest = pNewSpectrum->u.pfltSpectrum;
            for(i=0;i<iNrOfFrequencyBins;++i)
             { *(pfltDest++) = SPECTRUM_UNINITIALIZED_ARRAY_VALUE; // "X" (?)
               *(pfltDest++) = SPECTRUM_UNINITIALIZED_ARRAY_VALUE; // "Y" (?)
               *(pfltDest++) = SPECTRUM_UNINITIALIZED_ARRAY_VALUE; // "Z" (?)
             }
            break;

       default:  // unknown data types will NOT be initialized here (!)
            break;
      } // end switch(iDataType)
   }
  if( ! SpecBuf_IsValidSpectrum( pNewSpectrum ) )
   {  DEBUGGER_BREAK();
      if( nSpectrumCopyErrors < 5 )
       {  nSpectrumCopyErrors++;
          DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, 0.0/*time*/,
           "NewSpectrum() : result invalid !" );
       }
   }

  return pNewSpectrum;
} // end NewSpectrum()


//--------------------------------------------------------------------------
T_SPECTRUM *NewEmptySpectrum(void)
{
  return NewSpectrum(
             SPEC_DT_FLOAT_ARRAY, // nSpectrumDataType,
             0,   // nCompsPerBin, ZERO for automatic detection
             1);  // iNrOfFrequencyBins
} // end NewEmptySpectrum()


//--------------------------------------------------------------------------
void DeleteSpectrum(T_SPECTRUM **ppSpectrum)
{
  DWORD *pdwSafetyCheck;
  BOOL  fHealthy;

  if( ppSpectrum!=NULL )
   {
     if( *ppSpectrum!=NULL )
      {
        // Test the contents of the "safety check location" (after the array data) :
        fHealthy = ((*ppSpectrum)->hdr.c.dwMagic == SPECTRUM_HDR_MAGIC);
        if( fHealthy )
         { pdwSafetyCheck = (DWORD*)( (BYTE*)(*ppSpectrum) + (*ppSpectrum)->hdr.c.cbAllocatedSize - 4);
           fHealthy = (*pdwSafetyCheck==0x31415926); // magic number (AFTER the array data) still ok ?
         }
        if( !fHealthy )
         { // ooops... someone must have ignored the maximum array size ! (expect to crash soon)
           DEBUGGER_BREAK();
           UTL_WriteRunLogEntry( "PANIC: Tried to delete an inconsistent T_SPECTRUM structure in memory !" );
         }

        UTL_free( *ppSpectrum );
        *ppSpectrum = NULL;   // let anyone keep "hands off" now
      }
   }
} // end DeleteSpectrum()


//--------------------------------------------------------------------------
void FillDefaultSpectrumFileHeader( T_SpectrumFileHeader *pHeader,
                            DWORD dwMaxFileSizeInBytes  )
{
  int i;
  memset(pHeader, 0, sizeof(T_SpectrumFileHeader) );

  strcpy(pHeader->szFileTextInfo,   // max 64 characters...
      //  0        1         2         3         4         5         6
      //  123456789012345678901234567890123456789012345678901234567890123
         "Spectrum Lab Buffer File ** Version 1.1 ** DO NOT EDIT **\r\n");
      //  (will be visible if someone tries to open the file with a viewer or text editor.
      //   The string "Spectrum" will also be checked when opening such a file)

  pHeader->i64TotalEntryCount = 0;
  pHeader->dwSizeOfArrayEntry = sizeof( T_SPECTRUM ); // default value; overwritten soon !
  pHeader->dwMaxFileSize      = dwMaxFileSizeInBytes;
  pHeader->cbHeaderStructSize = sizeof( T_SpectrumFileHeader );
  pHeader->dwNextFilePosForWriting = pHeader->cbHeaderStructSize; // (!)
  for(i=0; i<SPECBUFF_HEADER_LOOKUP_TABLE_SIZE; ++i)
   { pHeader->LookupTable[i].i64EntryIndex = 0;
     pHeader->LookupTable[i].dwFilePos = 0;
   }

} // end FillDefaultSpectrumFileHeader( )


//--------------------------------------------------------------------------
int SpecBuf_GetSizeOfFreqEntryPerBin( T_SPECTRUM *ps )
  // Determines the memory size [in bytes] for a single "frequency bin"
  //  in a T_SPECTRUM structure, not including the header size !
{ int iSizeOfEntry;
  switch( ps->hdr.v.nDataType )  // SPEC_DT_FLOAT_ARRAY, SPEC_DT_RDF_SPECTRUM, ...
   {
    case SPEC_DT_FLOAT_ARRAY :  /* use T_SPECTRUM.fltSpectrum[], amplitudes only  */
         iSizeOfEntry = sizeof(float);  // one float entry per frequency bin
         break;
    case SPEC_DT_COMPLEX_ARRAY: /* use T_SPECTRUM.fltSpectrum[], complex values */
         iSizeOfEntry = sizeof(float);  // will be multiplied by TWO below !
         if(ps->hdr.v.iCompsPerBin != 2) // a bug somewhere...
          { DEBUGGER_BREAK(); // set breakpoint here, or use the 'common' breakpoint in DebugU1.cpp ::
          }
         break;
    case SPEC_DT_TRIPLE_ARRAY:  /* use T_SPECTRUM.fltSpectrum[], 3D vectors or "triples" */
         iSizeOfEntry = sizeof(float);  // will be multiplied by THREE below !
         if(ps->hdr.v.iCompsPerBin != 3) // a bug somewhere...
          { DEBUGGER_BREAK(); // set breakpoint here, or use the 'common' breakpoint in DebugU1.cpp
          }
         break;
    case SPEC_DT_RDF_SPECTRUM:  /* use T_SPECTRUM.RdfSpectrum */
         iSizeOfEntry = sizeof( T_RDF_SPECTRUM ); break;
    case SPEC_DT_RA_SPECTRUM: // type for the REASSIGNED SPECTROGRAM
         iSizeOfEntry = sizeof( T_RA_SPECTRUM );  break;
    case SPEC_DT_TEXT_MESSAGE:  /* use T_SPECTRUM.TextMessage : FIXED SIZE, no array ! */
        iSizeOfEntry = 0;  break;
    case SPEC_DT_UNUSED_ENTRY:  /* the T_SPECTRUM is "unused" */
    default:
         iSizeOfEntry = 0; break;  // cannot be handled here
   } // end switch( ps->hdr.v.iDataType )
  if(ps->hdr.v.iCompsPerBin > 1)           // one or two CHANNELS ?
      iSizeOfEntry *= ps->hdr.v.iCompsPerBin;
  return iSizeOfEntry;
} // end SpecBuf_GetSizeOfFreqEntryPerBin()


//--------------------------------------------------------------------------
BOOL CopySpectrum( T_SPECTRUM *psSrc,  // pointer to source spectrum
                  T_SPECTRUM **ppsDst) // pointer to pointer to destination spectrum(*)
{
 // Note: Since all T_SPECTRUM instances are dynamically allocated (09/2007),
 //       this is the ONLY POSSIBILITY to copy the contents from one
 //       T_SPECTRUM into another ! Simple BLOCK-COPY is forbidden,
 //       because that would crash the memory management .
 // (*) need a POINTER TO THE POINTER TO THE DESTINATION here, because
 //     CopySpectrum may have to free and re-allocate the destination
 //     (if it is too small for the data, or incompatible, or whatever).
 //     For this reason, the address of the destination spectrum in memory
 //     *may* change.   Phew.
 int   iNrOfEntries;
 T_SPECTRUM *psDst;


  // 2005-05-26: Crashed here in V2.4 B32 at G7IZU under Win XP "Professional",
  //             when moving the mouse over the WATERFALL .
  //             See C:\Postarchiv\Re_SpecLab\Crash_in_V24_B32_G7IZU.eml .
  if( ppsDst== NULL )
      return FALSE;
  if( psSrc == *ppsDst )
      return TRUE; // no need to copy (would crash if destination was adjusted in size)

  if( ! SpecBuf_IsValidSpectrum( psSrc ) )
   {  DEBUGGER_BREAK();
      if( nSpectrumCopyErrors < 5 )
       {  nSpectrumCopyErrors++;
          DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, 0.0/*time*/,
           "CopySpectrum() : source invalid !" );
       }
      return FALSE; // source spectrum is invalid !
   }

  if( ! SpecBuf_IsValidSpectrum( *ppsDst ) )
   {  DEBUGGER_BREAK();
      if( nSpectrumCopyErrors < 5 )
       {  nSpectrumCopyErrors++;
          DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, 0.0/*time*/,
           "CopySpectrum() : destination invalid !" );
       }
      return FALSE; // destination spectrum is invalid !
      // 2019-02-18 : occasionally got here, called from
      //     ProcessSpectraFromRemoteRig() -> DrawNewWaterfallLine(),
      //     when (*ppsDst)->hdr.v.iNrOfBins = 475 (ok, # frequency bins from IC-7300),
      //          (*ppsDst)->hdr.c.cbHdrStructSize = 168 (ok)
      //          (*ppsDst)->hdr.c.cbAllocatedSize = 2076 = 4*475 + 168 + 8
      //                               (ok, see SpecBuf_GetMinAllocSize() ),
      //     but the magic number (AFTER the array data) was corrupted !
   } // end if < destination NOT valid >

  // If the type of the destination is not the same as the source,
  //  or the allocated size of the destination is too small,
  // then delete and re-allocate the destination (to make it fit):
  if(  (psSrc->hdr.v.nDataType    != (*ppsDst)->hdr.v.nDataType )
    || (psSrc->hdr.v.iNrOfBins    != (*ppsDst)->hdr.v.iNrOfBins )
    || (psSrc->hdr.v.iCompsPerBin != (*ppsDst)->hdr.v.iCompsPerBin) )
   { DeleteSpectrum( ppsDst );  // crashed here for an RDF spectrum (2007-10)
     *ppsDst = NewSpectrum( psSrc->hdr.v.nDataType, psSrc->hdr.v.iCompsPerBin, psSrc->hdr.v.iNrOfBins );
     if( ! SpecBuf_IsValidSpectrum( *ppsDst ) )
         return FALSE;   // re-allocation of destination spectrum failed !!
   }
  psDst = *ppsDst;

  // Copy the "variable part" of the spectrum header from source to destination :
  psDst->hdr.v = psSrc->hdr.v;  // .v = "variable" part (which may be copied 1:1)

  // Now copy the frequency bins.
  // Note: For every bin, there may be more than one component !
  iNrOfEntries = psSrc->hdr.v.iCompsPerBin * psSrc->hdr.v.iNrOfBins;
  // Since 2003-07, several different types of spectra must be distinguished:
  switch(psSrc->hdr.v.nDataType)
   { case SPEC_DT_RDF_SPECTRUM:
       {// a special spectrum type for the Radio Direction Finder..
        T_RDF_SPECTRUM *pRdfSrc= psSrc->u.pRdfSpectrum;  // source pointer for frequency bins
        T_RDF_SPECTRUM *pRdfDst= psDst->u.pRdfSpectrum;  // destination pointer
        while(iNrOfEntries--)
          *pRdfDst++ = *pRdfSrc++;  // crashed HERE 2005-05-18 on G7IZU's XP Pro machine ?
       }break;

     case SPEC_DT_RA_SPECTRUM:
       {// a special spectrum type for the REASSIGNED SPECTROGRAM..
        T_RA_SPECTRUM *pRASrc= psSrc->u.pRASpectrum;  // source pointer for frequency bins
        T_RA_SPECTRUM *pRADst= psDst->u.pRASpectrum;  // destination pointer
        while(iNrOfEntries--)
          *pRADst++ = *pRASrc++;
       }break;

     case  SPEC_DT_TEXT_MESSAGE:
       {// not really a 'spectrum' , but a TEXT MESSAGE in the large spectrum buffer...
        psDst->hdr.v.nDataType = SPEC_DT_TEXT_MESSAGE;
     // if( iNrOfEntries<=0 || iNrOfEntries>MAX_CHARS_PER_SPEC_TEXT_MSG )
     //  {  iNrOfEntries=MAX_CHARS_PER_SPEC_TEXT_MSG;
     //  }
        psDst->hdr = psSrc->hdr;  // <<< this should be a BLOCK MOVE
        psDst->u.TextMsg = psSrc->u.TextMsg;  // <<< and another BLOCK MOVE
       }break;

     case SPEC_DT_FLOAT_ARRAY:    // a "normal" spectrum..  MAX_SAMPLES_PER_SPECTRUM = 32768 (in TSpectrum.h)
     case SPEC_DT_COMPLEX_ARRAY:  // a complex spectrum.. MAX_SAMPLES_PER_SPECTRUM/2 (two elements per bin)
     case SPEC_DT_TRIPLE_ARRAY:   // an array of 3D vectors or "triples" (three elements per bin)
       { T_Float *pfltSrc = psSrc->u.pfltSpectrum;  // source pointer for frequency bins
         T_Float *pfltDst = psDst->u.pfltSpectrum;  // destination pointer
         while(iNrOfEntries--)
          { *pfltDst++ = *pfltSrc++;
          }
       }break;

     default:  // for all other "spectrum" struct types ... SOMETHING MISSING HERE !
        break;
   } // end switch(psSrc->hdr.v.iDataType)


  if( ! SpecBuf_IsValidSpectrum( *ppsDst ) )
   {  DEBUGGER_BREAK();
      if( nSpectrumCopyErrors < 5 )
       {  nSpectrumCopyErrors++;
          DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, 0.0/*time*/,
           "CopySpectrum() : result invalid !" );
       }
   }

  return TRUE;
} // end CopySpectrum







//***************************************************************************
//  Implementation of methods for the    CSpectrumBuffer   class
//***************************************************************************


//--------------------------------------------------------------------------
CSpectrumBuffer::CSpectrumBuffer()     // constructor
{
   // properties of the small RAM buffer..
   m_PtrArray = NULL;
   m_psTempSpectrum = NULL;
   m_iPtrArrayAllocatedEntries = m_iPtrArrayUsedEntries = 0;
   m_pbBuffer = NULL;
   m_dwAllocatedBytesInBuffer = m_dwBufferHeadIndex = 0;

   m_i64IndexOfLatestSpectrum = 0; // here: cleared in the CONSTRUCTOR
 //m_i64IndexOfLatestTextMessage = 0;

   // properties for the optional 'buffer file'..
   m_hFile = INVALID_HANDLE_VALUE; // <- cannot emphasise enough that this isn't ZERO !
                                   // (use the old memset(0), and you'll be in trouble)
   m_sz1023FileName[0] = '\0';     // <- make sure no-one accidentally uses a BUFFER FILE
   m_fMustFlushHeader = FALSE;
   m_pFileLookupTable = NULL;
   m_dwFileLookupTableNAllocatedEntries = m_dwFileLookupTableUsedEntries = 0;
   m_dwFileLookupTableEstimatedRequiredEntries = 0;

   m_fFileLookupTableIsUpToDate = FALSE;
   m_i64FileLookupTableFirstSeekedIndex = m_i64FileLookupTableLastSeekedIndex = -1;


} // end CSpectrumBuffer::CSpectrumBuffer()


//--------------------------------------------------------------------------
CSpectrumBuffer::~CSpectrumBuffer()    // destructor, clean up
{
  Close();  // no problem to call this if nothing opened
} // end CSpectrumBuffer::~CSpectrumBuffer() [destructor]

//--------------------------------------------------------------------------
void CSpectrumBuffer::SetIndexOfLatestSpectrum( LONGLONG i64EntryIndex ) // private !
  // Replaces the simple assignment to m_i64IndexOfLatestSpectrum .
  // Added 2020-06-11 to find the bug described in GetIndexOfOldestSpectrum().
{
  if( i64EntryIndex < m_i64IndexOfLatestSpectrum )
   { DEBUGGER_BREAK();  // warp back ? That's ILLEGAL !
   }
  m_i64IndexOfLatestSpectrum = i64EntryIndex;
  // The above 64-bit assignment was compiled into the following code:
  //     mov ecx,[ebp+0x0c]   // lower 32 bits ...
  //     mov [ebx+0x0c],ecx
  //     mov ecx,[ebp+0x10]   // ... upper 32 bits
  //     mov [ebx+0x10],ecx
  // -> not an 'atomic' instruction, and thus not necessarily thread-safe.
  //    But in this case, BOTH the reader and writer were called from the
  //    same thread (from the VCL / GUI thread: "Forms::TApplication::Run()") .
} // CSpectrumBuffer::SetIndexOfLatestSpectrum()


//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::Init(
      DWORD dwMaxBufSize_RAM,        // max. allowed size of buffer IN RAM [Bytes] (*)
      char *pszFileBufferName,       // name of a BUFFER FILE (NULL= use RAM only)
      T_SpecDataType nInitialDataType, // SPEC_DT_FLOAT_ARRAY, SPEC_DT_RDF_SPECTRUM, ...
      DWORD dwMaxFftBinsPerEntry,    // max. entries in T_SPECTRUM.float fltSpectrum[]
      DWORD dwMaxFileSize ,          // max. acceptable buffer file size in BYTES
      BOOLEAN fCreateNewFile )       // TRUE = create a new buffer file; FALSE = open existing
  // Sets the parameters for the spectrum buffer
  // and opens the shop for business .
  // (*) Prior to 2012-08, the max. buffer size was measured in 'number of entries',
  //     which caused problems after switching to 'very long FFTs'.
  //     Since then, the size-limitation is measured in BYTES. If the FFTs are
  //     'short', many of them will fit in the buffer;
  //     if the FFTs are 'long', this buffer will accept less spectra.
{
  DWORD dwSizePerEntry;
  int   nBufferedLinesInRAM;

  // Determine the size of a single entry (in the spectrum buffer) :
  dwSizePerEntry = SpecBuf_GetMinAllocSize( nInitialDataType, 0/*CompsPerBin:AUTO*/, dwMaxFftBinsPerEntry );
  if(dwSizePerEntry<256)
   { dwSizePerEntry=256;
   }

  // Refuse to use crazy parameters. For a complex 512k FFT, 2*4*512*1024 bytes = approx 4 MByte are required.
  // We use AT LEAST 5 MByte in RAM, regardless of SL's "system" settings !
  // Without this, SL would not be able to paint a waterfall with 'long' FFTs.
  if( dwMaxBufSize_RAM < 5*1024*1024 )   // at least 5 MByte..
   {  dwMaxBufSize_RAM = 5*1024*1024;
   }
  if( dwMaxBufSize_RAM < 2*dwSizePerEntry ) // and at least TWO buffer entries
   {  dwMaxBufSize_RAM = 2*dwSizePerEntry;
   }
  if( dwMaxBufSize_RAM > 1024*1024*1024 ) // but at most 1 GByte [RAM!]
   {  dwMaxBufSize_RAM = 1024*1024*1024;
   }
  if( m_dwAllocatedBytesInBuffer != dwMaxBufSize_RAM )
   {  m_iPtrArrayUsedEntries = 0;  // forget old pointers; they will be invalid now:
      if( m_pbBuffer != NULL )
       { UTL_free(m_pbBuffer);     // fsssh ..
       }
      m_pbBuffer = (BYTE*)UTL_NamedMalloc( "SpecBuff", dwMaxBufSize_RAM );
      if( m_pbBuffer != NULL )
       { m_dwAllocatedBytesInBuffer = dwMaxBufSize_RAM;
       }
   }
  // Determine the size of the POINTER-ARRAY, and if neccessary, allocate it:
  nBufferedLinesInRAM = (dwMaxBufSize_RAM + dwSizePerEntry - 1) / dwSizePerEntry;
  if (nBufferedLinesInRAM<10)   nBufferedLinesInRAM=10;
  if (nBufferedLinesInRAM>4000) nBufferedLinesInRAM=4000;
  if( m_PtrArray )
   { // RAM-Array already allocated (in previous call) ?
     if( m_iPtrArrayAllocatedEntries != (int)nBufferedLinesInRAM )
      { // Size has changed -> get rid of the old array; a new array is allocated further below
        UTL_free(m_PtrArray);  // fsssh ..
        m_PtrArray = NULL;
        m_iPtrArrayUsedEntries = 0; // forget old pointers; they are invalid now
      }
   } // end if < RAM-array already exists ? >
  if( m_PtrArray == NULL )
   {  m_iPtrArrayAllocatedEntries = m_iPtrArrayUsedEntries = 0;
      m_PtrArray = (T_SPECTRUM**)UTL_NamedMalloc( "SpecPtrs", nBufferedLinesInRAM * sizeof(T_SPECTRUM*) );
      // Note: This is just an ARRAY OF POINTERS(!), total size is small !
      if( m_PtrArray != NULL )      // .. but we still expect failed memory allocation..
       { m_iPtrArrayAllocatedEntries = nBufferedLinesInRAM;
         memset( m_PtrArray, 0, nBufferedLinesInRAM * sizeof(T_SPECTRUM*) );
       }
   }


  // Clean up the buffer-file if it has been in use already ...
  if(m_hFile != INVALID_HANDLE_VALUE)
   { CloseFile();
   } // end if < buffer file already opened ? >

  DeleteFileLookupTable();  // force rebuilding the file lookup table
  // ex: m_i64IndexOfLatestSpectrum = 0; // here: cleared again in CSpectrumBuffer::Init()
  SetIndexOfLatestSpectrum( 0 );  // replacement for m_i64IndexOfLatestSpectrum = 0
  m_i64TotalEntryCount = 0;       // here: cleared again in CSpectrumBuffer::Init()


  // Fill the T_SpectrumFileHeader structure 'Header' with an "all-new" header
  // First, set all components to 'zero' because we may not know all
  // components which will be implemented (appended!) in future.
  m_dwMaxFileSizeInBytes = dwMaxFileSize;
  FillDefaultSpectrumFileHeader( &m_FileHeader, m_dwMaxFileSizeInBytes );
  if(pszFileBufferName)  // use a BUFFER FILE...
   {
     strncpy( m_sz1023FileName, pszFileBufferName, 1023);   // file name; required to close and re-open
     OpenBufferFile(m_sz1023FileName,fCreateNewFile);  // TRUE = create a new buffer file; FALSE = open existing
   }
  else
   {
     m_sz1023FileName[0] = 0;  // no file name !
   }

  // Added 2013-05-05 : Guess the 'required' number of entries
  //                    for the file-lookup-table (m_pFileLookupTable),
  //  which (at this point) has not been allocated yet :
  m_dwFileLookupTableEstimatedRequiredEntries = // 'estimated' number of entries, may increase during runtime
    ( m_dwMaxFileSizeInBytes + dwSizePerEntry - 1) / dwSizePerEntry;


  return m_PtrArray!=NULL;
} // end CSpectrumBuffer::Init()


//--------------------------------------------------------------------------
char * CSpectrumBuffer::GetErrorString(void)
{
  return sz80LastError;
} // end CSpectrumBuffer::GetErrorString()


//--------------------------------------------------------------------------
void CSpectrumBuffer::FlushAll(void)
  // Flushes the buffer to disk, and updates the file header
  //  *IF* the buffer uses a file..
{
  if(m_hFile != INVALID_HANDLE_VALUE) // <- added 2020-06-11 to avoid unintentionally use a BUFFER FILE
   {
     CloseFile();   // <-- this includes updating the file header !
     if( m_sz1023FileName[0] != '\0' ) // <- added 2020-06-11 to avoid unintentionally clearing m_i64IndexOfLatestSpectrum
      {  OpenBufferFile(m_sz1023FileName, FALSE);  // TRUE = create a new buffer file; FALSE = open existing
      }
   }
} // end SpectrumBuffer::FlushAll()

//--------------------------------------------------------------------------
void CSpectrumBuffer::CreateFileLookupTable( long i32MinNumberOfEntries )
{
  DWORD dw;
  m_dwFileLookupTableUsedEntries = 0;
  m_fFileLookupTableIsUpToDate   = FALSE;
  m_i64FileLookupTableFirstSeekedIndex = m_i64FileLookupTableLastSeekedIndex = -1;
  if( m_pFileLookupTable==NULL )
   { m_dwFileLookupTableNAllocatedEntries = i32MinNumberOfEntries;
     if( m_dwFileLookupTableNAllocatedEntries < 32768 )
      { m_dwFileLookupTableNAllocatedEntries = 32768;
      }
     m_pFileLookupTable = (T_SpectrumFileLookupTableEntry*)UTL_NamedMalloc(
         "SBufLUT", sizeof(T_SpectrumFileLookupTableEntry) * m_dwFileLookupTableNAllocatedEntries);
     if( m_pFileLookupTable )
      { for(dw=0; dw<m_dwFileLookupTableNAllocatedEntries; ++dw)
         { m_pFileLookupTable[dw].i64EntryIndex = 0;
           m_pFileLookupTable[dw].dwFilePos = 0;
         }
      }
     else // something went wrong with the memory allocation..
      { m_dwFileLookupTableNAllocatedEntries = 0;
      }
   } // end if < file lookup table not allocated yet >

} // end CSpectrumBuffer::CreateFileLookupTable()


//--------------------------------------------------------------------------
void CSpectrumBuffer::DeleteFileLookupTable(void)
{
  if( m_pFileLookupTable!=NULL )
   { UTL_free( (void*)m_pFileLookupTable );
     m_pFileLookupTable = NULL;
   }
  m_dwFileLookupTableNAllocatedEntries = m_dwFileLookupTableUsedEntries = 0;
} // end CSpectrumBuffer::DeleteFileLookupTable()


//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::QSearchInFileLookupTable(LONGLONG i64EntryIndex,  DWORD *pdwLookupTableIndex )
  // Output: *pdwLookupTableIndex = an index into m_pFileLookupTable .
  // Return value :  TRUE = "exactly found",  FALSE = "not found" .
  //   The result is valid even if <i64EntryIndex> is not "exactly" found !
  //     If QSearchInFileLookupTable(i64EntryIndex, &dwTableIndex) returns FALSE,
  //         dwTableIndex is the index where a new entry should be INSERTED,
  //         moving the rest of the table UP .
  //     Also, the caller may start a sequential file search,
  //         beginning at m_pFileLookupTable[dwTableIndex].dwFilePos
  //         to find the wanted spectrum in the file .
{
  long i32Low, i32High, i32Mid;

  *pdwLookupTableIndex = 0;
  if( m_pFileLookupTable==NULL )
   { return FALSE;
   }

  // Use a binary search to find the specified buffer entry .
  // In this "optimized" implementation, we check if the entry was found
  //  *AFTER* the search loop (not *IN* the search loop).
  // See http://en.wikipedia.org/wiki/Binary_search  .
  i32Low  = 0;
  i32High = m_dwFileLookupTableUsedEntries;  // "N", not "N MINUS ONE" ! !
  while( i32Low < i32High )
   {
     /* new search position in the middle between low and high position */
     i32Mid = ( i32High + i32Low ) / 2;

     if( m_pFileLookupTable[i32Mid].i64EntryIndex < i64EntryIndex )
      { // entry in table is too low; replace lower index :
        i32Low = i32Mid+1; // note the "+1" !
      }
     else // entry in table is larger or equal..
      {   // we'll care about "found" / "not found" AFTER the loop !
        i32High = i32Mid;  // this is tricky, despite it's simplicity
      }
   } // end while
  // AFTER the loop, check if the entry was found or not.
  // From the wiki article:
  // > This algorithm has two other advantages. At the end of the loop,
  // > i32Low points to the first entry greater than or equal to <value>,
  // > so a new entry can be inserted if no match is found
  // (this is exactly what we're going to use this "feature" for) .
  // > Moreover, it only requires one comparison (..) .
  *pdwLookupTableIndex = (DWORD)i32Low/*!*/;  // found index, or "insertion" index
  if( ( i32Low < (long)m_dwFileLookupTableUsedEntries)
    &&( m_pFileLookupTable[i32Low].i64EntryIndex == i64EntryIndex ) )
   { return TRUE;
   }
  else
   { return FALSE; // not "found", but *pdwLookupTableIndex contains the index
                   // at which a new entry should be inserted .
   }

} // end CSpectrumBuffer::QSearchInFileLookupTable()


//--------------------------------------------------------------------------
void CSpectrumBuffer::AddEntryToFileLookupTable( LONGLONG i64EntryIndex, DWORD dwFilePos )
  // Adds a new entry to m_pFileLookupTable[], keeping it sorted (!)
{
  DWORD dwTableIndex;
  BOOL found;
  if( m_pFileLookupTable==NULL )
   { CreateFileLookupTable( 65536/*i32MinNumberOfEntries*/ );
   } // end if < file lookup table not allocated yet >
  found = QSearchInFileLookupTable( i64EntryIndex, &dwTableIndex );
  if( found )
   { // Overwrite the already existing entry, because
     //     the FILE POSITION may have been changed..
     m_pFileLookupTable[dwTableIndex].i64EntryIndex = i64EntryIndex;
     m_pFileLookupTable[dwTableIndex].dwFilePos = dwFilePos;
   }
  else // not found: append or insert a new entry in the lookup-table
   { // Even if "not found", the table index returned by QSearchInFileLookupTable()
     // tells us where the new entry shall be inserted ("insert before").
     if( dwTableIndex < m_dwFileLookupTableNAllocatedEntries )
      { if( dwTableIndex < m_dwFileLookupTableUsedEntries )
         { // insert in table, scrolling the rest UP :
           for( DWORD dw=m_dwFileLookupTableUsedEntries; dw>dwTableIndex; --dw )
            { m_pFileLookupTable[dw] = m_pFileLookupTable[dw-1];
            }
         }
        m_pFileLookupTable[dwTableIndex].i64EntryIndex = i64EntryIndex;
        m_pFileLookupTable[dwTableIndex].dwFilePos = dwFilePos;
        if(  m_dwFileLookupTableUsedEntries < m_dwFileLookupTableNAllocatedEntries )
         { ++m_dwFileLookupTableUsedEntries;
         }
      }
   }
} // end AddEntryToFileLookupTable()

//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::ReadHeadersFromFile(
           LONGLONG i64WantedEntryIndex, // [in] max 'buffer entry index' requested by caller
           DWORD *pdwFilePos,          // [in,out] current file position (byte offset)
           T_SPECTRUM_HDR *pHdr )      // [out] buffer which receives the spectrum-HEADER
  // Reads a part of the spectrum buffer file,
  //    and adds entries to the file-lookup-table ( m_pFileLookupTable[] ) .
{
  DWORD dwNumberOfBytesRead;
  BOOL fResult = FALSE;

#ifdef __BORLANDC__
  (void)fResult; // ... "was a assigned a value that is never used" .. oh, shut up !
#endif

  // The following loop SEQUENTIALLY reads from the file (no fseek etc).
  // Since 2010-10, it is also used to fill the file-lookup-table .
  while( 1 )
   {
     if(!SetFilePosIfNeeded(*pdwFilePos) )
      { // Something went wrong when trying to set the file pointer .
        //  Most likely: File is shorter than it should be, possibly header corrupted.
        fResult = FALSE;
        break;
      }
     dwNumberOfBytesRead = 0;
     if( (!ReadFile( m_hFile, (void*)pHdr,
           sizeof(T_SPECTRUM_HDR),  // number of bytes to read (ideally)
           &dwNumberOfBytesRead,    // number of bytes actually read
            NULL) )
        || ( dwNumberOfBytesRead!=sizeof(T_SPECTRUM_HDR) ) )
      { // must be the end of the buffer file
        fResult = FALSE;
        break;
      }
     else   // spectrum-HEADER successfully read.. what's this entry ?
      { // Note: m_dwCurrentFilePosForWriting must not be affected by this !!
        // Arrived here, we *should* have read the HEADER of a T_SPECTRUM structure.
        // But is it valid ?
        if( pHdr->c.dwMagic != SPECTRUM_HDR_MAGIC )
         { fResult = FALSE;
           break;
         }

        // 2010-10-21: Got here with pHdr->v.i64EntryIndex= 157585, 157586, 157587, ..
        //        even though m_FileHeader.i64TotalEntryCount was only 356 ?!
        // Reason: SL had been abnormally terminated with the debugger,
        //         so m_FileHeader had not been 'written back' to the file,
        //         causing m_i64TotalEntryCount to jump-back illegally,
        //         and Spectrum Lab refused to load a part of the buffer .
        //  Too bad !  The safest way would be re-creating the lookup table
        //  when restarting, possibly taking the unique absolute TIMESTAMPS
        //  into account, to fix this 'broken index' problem .
        if( pHdr->v.i64EntryIndex >= m_i64TotalEntryCount )
         { m_i64TotalEntryCount = pHdr->v.i64EntryIndex + 1;  // just a BUGFIX !
         }

        // Overwrite or append the "entry index" and the file position
        // in the file-lookup-table (keeping the table SORTED) .
        // AddEntryToFileLookupTable() may decide NOT to add an entry,
        //    for example if the lookup table is already full,
        //    and there is already an entry close to hdr.v.i64EntryIndex :
        AddEntryToFileLookupTable( pHdr->v.i64EntryIndex, *pdwFilePos );
        // Keep track of the index-range in which the file has been scanned.
        // This avoids unnecessary file-read operations next time we get here.
        if( (pHdr->v.i64EntryIndex < m_i64FileLookupTableFirstSeekedIndex )
         || (m_i64FileLookupTableFirstSeekedIndex < 0 ) )
         {  m_i64FileLookupTableFirstSeekedIndex = pHdr->v.i64EntryIndex;
         }
        if( (pHdr->v.i64EntryIndex > m_i64FileLookupTableLastSeekedIndex )
         || (m_i64FileLookupTableLastSeekedIndex < 0 ) )
         {  m_i64FileLookupTableLastSeekedIndex = pHdr->v.i64EntryIndex;
         }


        // If this is what the caller was looking for, set the result:
        if(  pHdr->v.i64EntryIndex == i64WantedEntryIndex )
         { fResult = TRUE;
           break;
         }

        // Advance the file position to the next array entry .
        // See notes about hdr.c.cbAllocatedSize in \CBproj\SpecLab\TSpectrum.h !
        *pdwFilePos += pHdr->c.cbAllocatedSize; // skip everything until the next entry

      } // end if < ReadFile(HEADER) ok >
   } // end while
  return fResult;
} // end CSpectrumBuffer::ReadHeadersFromFile()


//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::EntryIndexToFilePos( LONGLONG i64EntryIndex, DWORD *pdwFilePos)
{
  // Finds the FILE POSITION (file offset) for a given spectrum buffer entry.
  //
  // Note: 'i64EntryIndex' is not an ARRAY INDEX, but
  //       a unique number which identifies ONE particular entry in the buffer.
  //       This index increases monotoneously; it can be larger
  //       than the buffer capacity (i64EntryIndex doesn't WRAP) .
  //    The 64-bit entry index is assigned to the spectrum whenever a spectrum
  //    is entered in the buffer (see AppendSpectrum; m_i64TotalEntryCount) .
  //
  // Since 2007-12-23, there are individual sizes for each entry
  //                   in the buffer file, so things got more complicated !
  //
  // Call Stack (during normal operation of Spectrum Lab, 2013-05-05) :
  //  TSpectrumLab::Timer1Timer() -> TSpectrumLab::ProcessSpectraFromInputStream()
  //   -> TSpectrumLab::AnalyseSpectraAndUpdateDisplay()
  //       -> TSpectrumLab::DrawNewWaterfallLines( iMaxFFLines=17 for a fast-scrolling spectrogram)
  //          -> SpecDisp_GetSpectrumButNoMsg() [repeatedly for a number of spectra]
  //              -> CSpectrumBuffer::GetSpectrum()
  //                  -> CSpectrumBuffer::ReadSpectrum()
  //                      -> CSpectrumBuffer::ReadSpectrumFromFile()  [!!!! - oh, it shouldn't]
  //                          -> CSpectrumBuffer::EntryIndexToFilePos()
  //  [after A LOT of specta]    -> CSpectrumBuffer::ReadHeadersFromFile() [takes a lot of time]
  BOOL  fResult = FALSE;
  DWORD dwTableIndex, dwFilePos;
  T_SPECTRUM_HDR hdr;

  // Preset the current file position (for the lookup-table-filling-loop)...
  if( m_FileHeader.cbHeaderStructSize != 0 )
        dwFilePos = m_FileHeader.cbHeaderStructSize; // file offset of first T_SPECTRUM entry
   else dwFilePos = sizeof(T_SpectrumFileHeader);


  dwTableIndex = 0;
  if( m_pFileLookupTable!=NULL )
   { // (sorted) table already exists: try to find the entry there ...
     if( QSearchInFileLookupTable( i64EntryIndex, &dwTableIndex ) )
      { // found the entry we were looking for :
        *pdwFilePos = m_pFileLookupTable[dwTableIndex].dwFilePos;
        return TRUE;
      }
     else // didn't find "exactly" what the caller was looking for,
      {   // but we -hopefully- came closer to the required file position.
          // This avoids having to skip HUNDREDS OF MEGABYTES in the file !
        if(    (dwTableIndex>0)
           && ((dwTableIndex>=m_dwFileLookupTableUsedEntries)
             ||(m_pFileLookupTable[dwTableIndex].i64EntryIndex > i64EntryIndex)
              )
          )
         { --dwTableIndex;
         }
        if( dwTableIndex < m_dwFileLookupTableUsedEntries  )
         { // index into the file-lookup-table is valid;
           // begin searching the file from this position:
           dwFilePos = m_pFileLookupTable[dwTableIndex].dwFilePos;
         }
      }
   }
  else  // lookup-table doesn't exist yet:
   { CreateFileLookupTable( 65536/*i32MinNumberOfEntries*/  );
   }

  // Arrived here: The requested entry couldn't be found in the lookup-table .
  //    This does NOT mean it's not in the file,
  //    because the table may only be *partially* filled .
  // So, continue to fill the "file lookup table" .
  // But *where* to continue ?
  if(  (m_i64FileLookupTableFirstSeekedIndex < 0 )
     ||(m_i64FileLookupTableLastSeekedIndex < 0 )
     ||(i64EntryIndex < m_i64FileLookupTableFirstSeekedIndex )
     ||(i64EntryIndex > m_i64FileLookupTableFirstSeekedIndex )
    )
   {
     // ex: while( m_dwFileLookupTableUsedEntries < m_dwFileLookupTableNAllocatedEntries )
     // ex: m_fFileLookupTableIsUpToDate = TRUE;
     if( ! m_fFileLookupTableIsUpToDate )
      { fResult = ReadHeadersFromFile( i64EntryIndex, &dwFilePos, &hdr );
        m_fFileLookupTableIsUpToDate = TRUE; // at least we TRIED ! (don't try again)
      }
     // 2013-05-05 : Got here when SL got 'very slow', as precisely reported by DF6NM,
     //   after  65536 * 5 ms [5ms=scroll rate].  When that happened :
     //   m_dwFileLookupTableUsedEntries       = 65536,
     //   m_i64FileLookupTableFirstSeekedIndex = 129589 ,
     //   m_i64FileLookupTableLastSeekedIndex  = 144674 ,
     //   i64EntryIndex = 195052 ,
   }

  return fResult;

} // end CSpectrumBuffer::EntryIndexToFilePos( )


//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::Close(void)
   // Closes the spectrum buffer (regardless of RAM and/or DISK FILE),
   // and frees all dynamically allocated RAM entries .
   // May be called by application in case of RAM shortage,
   // or when terminating.
{
  CloseFile();   // flush if necessary and close all handles
  DeleteFileLookupTable();

  m_iPtrArrayUsedEntries = m_iPtrArrayAllocatedEntries = 0;
  if(m_PtrArray)
   { MainForm_TermMsgCallback("de-allocating spectrum pointers");
     UTL_free(m_PtrArray);  // fsssh ..
     MainForm_TermMsgCallback("spectrum pointers de-allocated");
     m_PtrArray = NULL;
   }
  m_dwBufferHeadIndex = m_dwAllocatedBytesInBuffer = 0;
  if( m_pbBuffer != NULL )
   { MainForm_TermMsgCallback("de-allocating spectrum buffer");
     UTL_free(m_pbBuffer);     // fsssh ..
     m_pbBuffer = NULL;
   }
  if( m_psTempSpectrum )
   { DeleteSpectrum( &m_psTempSpectrum );
   }

  return TRUE;  // no reason to fail (up to now)
} // end CSpectrumBuffer::Close()


//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::Clear(void)
  // Clears the contents of the spectrogram buffer ( RAM, possibly also FILE)
  // without DELETING the buffer (to delete the buffer in RAM, use Close)
{
  m_i64TotalEntryCount   = 0;   // here: in
  m_iPtrArrayUsedEntries = 0;
  m_dwBufferHeadIndex   = 0;
  m_FileHeader.i64TotalEntryCount = 0;  // assume the file is "empty" without physically deleting it
  m_fMustFlushHeader   = TRUE;

  return TRUE;  // no reason to fail (up to now)
} // end CSpectrumBuffer::Clear()


//--------------------------------------------------------------------------
DWORD CSpectrumBuffer::GetMaxFileSize(void)
{
  return m_FileHeader.dwMaxFileSize;
} // end CSpectrumBuffer::GetMaxFileSize()


//--------------------------------------------------------------------------
LONGLONG CSpectrumBuffer::GetIndexOfLatestEntry(void)
{
  return m_i64TotalEntryCount - 1;
} // end CSpectrumBuffer::GetIndexOfLatestEntry()


//--------------------------------------------------------------------------
LONGLONG CSpectrumBuffer::GetIndexOfLatestSpectrum(void)
{
  return m_i64IndexOfLatestSpectrum;
} // end CSpectrumBuffer::GetIndexOfLatestSpectrum()


//--------------------------------------------------------------------------
LONGLONG CSpectrumBuffer::GetIndexOfOldestSpectrum(void)
{
  LONGLONG i64Result = m_i64IndexOfLatestSpectrum - (m_iPtrArrayUsedEntries-1);
    // Example: m_i64IndexOfLatestSpectrum = 4, m_iPtrArrayUsedEntries = 2
    //     ->   index of OLDEST spectrum (which can be retrieved from the buffer) = 4 - (2-1) = 3 (!)
  if( i64Result > m_i64IndexOfLatestSpectrum )
   {  i64Result = m_i64IndexOfLatestSpectrum;
   }

  if( i64Result < 0 ) // Garbage ! Circular wrap despite 64-bit integer counters ? Uh,uh..
   {  DEBUGGER_BREAK(); // <- set a breakpoint here and find out why we got here :)
      // 2020-06-11 : Hit the breakpoint with
      //                 m_i64IndexOfLatestSpectrum  = 0 (!? .. where did this get set ? )
      //                 m_iPtrArrayUsedEntries      = 530 (and again later with .. = 1057 )
      //                 m_i64TotalEntryCount        = 530 (that's our NON-WRAPPING counter)
      //                 m_iPtrArrayAllocatedEntries = 4000
      //                 i64Result = 0 - (530-1) = -529 .
      //                 m_PtrArray[0]->hdr.v.i64EntryIndex   = 0 (ok)
      //                 m_PtrArray[529]->hdr.v.i64EntryIndex = 529 (ok, THIS is really the latest entry..)
      //                 m_PtrArray[529]->hdr.v.nDataType = SPEC_DT_FLOAT_ARRAY (.. and it's REALLY a spectrum)
      //                 m_PtrArray[530] = NULL (.. ok, and #529 was REALLY the LATEST)
      //       In fact the OLDEST spectrum was still at index ZERO
      //       (since m_iPtrArrayAllocatedEntries wasn't exhausted yet).
      // To debug this, replaced the ASSIGNMENTS to m_i64IndexOfLatestSpectrum
      //                by a 'setter function' (SetIndexOfLatestSpectrum() ).
      //                Indeed m_i64IndexOfLatestSpectrum was set to zero in
      //     TSpectrumLab::Timer1Timer() -> CSpectrumBuffer::FlushAll()
      //      -> CSpectrumBuffer::OpenBufferFile(), even though SL's config
      //         said "NO SPECTRUM BUFFER FILE" .
      i64Result = 0;
   }
  return i64Result;
} // end CSpectrumBuffer::GetIndexOfOldestSpectrum()


//--------------------------------------------------------------------------
LONGLONG CSpectrumBuffer::GetIndexOfOldestEntry(void)
{
  return m_i64TotalEntryCount - m_iPtrArrayUsedEntries; // ??
} // end CSpectrumBuffer::GetIndexOfOldestEntry()


//--------------------------------------------------------------------------
LONGLONG CSpectrumBuffer::GetIndexOfEntryByTime(double dblTime)
  // See also (similar function) : GetSpectrumByTimestamp() [faster, w/o copying]
{
  LONGLONG i64Searchpos;
  BOOLEAN  higher, lower;
  LONGLONG i64High = m_i64IndexOfLatestSpectrum;
  LONGLONG i64Low  = m_i64IndexOfLatestSpectrum - (GetCountOfBufferedLines()-1);
  if( i64Low < 0 )
      i64Low = 0;

  if( m_psTempSpectrum==NULL )
   { m_psTempSpectrum = NewSpectrum( SPEC_DT_FLOAT_ARRAY,  0/*guess comps_per_bin*/,   0 );
   }

  // use a binary search to find the nearest buffer entry with the specified TIME :
  if( i64High <= i64Low )
     return m_i64IndexOfLatestSpectrum;  // entry cannot be found (by binary search)

  do
   {
     /* new search position in the middle between low and high position */
     i64Searchpos = ( i64High + i64Low ) / 2;

     if(! GetSpectrum( i64Searchpos, &m_psTempSpectrum ) )
      { // if a spectrum could not be retrieved, assume it's "too old" :
        higher = FALSE;
        lower  = TRUE;
      }
     else
      { higher = m_psTempSpectrum->hdr.v.dbl_time > dblTime;
        lower  = m_psTempSpectrum->hdr.v.dbl_time < dblTime;
      }

     if ( (! higher) && (! lower) )
        break;   /* found !  (quite unlikely to be "exactly the same" floating-point value) */

     if ( higher )
      {
         if ( i64Searchpos == i64High )  // reached the "other end" ? -> ready
            break;
         else
            i64High = i64Searchpos;
      }
     if ( lower )
      {
         if ( i64Searchpos == i64Low )   // reached the "other end" ? -> ready
             break;
         else
            i64Low = i64Searchpos;
      }
   }  while ( TRUE );

  return i64Searchpos ;

} // end CSpectrumBuffer::GetIndexOfEntryByTime()


//--------------------------------------------------------------------------
long CSpectrumBuffer::GetMaxCountOfBufferedLines(void)
{
  if( (m_hFile!=INVALID_HANDLE_VALUE) && (m_FileHeader.dwSizeOfArrayEntry>0) )
   {
    return m_FileHeader.dwMaxFileSize / m_FileHeader.dwSizeOfArrayEntry;
   }
  else // no BUFFER FILE, just a small RAM ARRAY:
   {
    return m_iPtrArrayAllocatedEntries; // number of spectra which can be stored in RAM
   }
} // end CSpectrumBuffer::GetMaxCountOfBufferedLines()


//--------------------------------------------------------------------------
long CSpectrumBuffer::GetCountOfBufferedLines(void)
{
  long lResult = m_iPtrArrayUsedEntries; // ex: = m_iPtrArrayAllocatedEntries;
  if( (m_hFile!=INVALID_HANDLE_VALUE) && m_FileHeader.dwSizeOfArrayEntry>0)
   {
    DWORD dwNumberOfEntriesInFile =   // .. MAXIMUM, not all of these are USED !
     ( m_FileHeader.dwMaxFileSize - m_FileHeader.cbHeaderStructSize )
     / m_FileHeader.dwSizeOfArrayEntry;
    // Note 1: This calculation ignores the fact that all entries in the file
    //         may have individual sizes !
    // Note 2: A "stero" spectrum occupies TWO buffer-entries in the file !
    //         ( the spectrum buffer class doesn't care for the graphic
    //           representation of the buffered spectra on the screen )
    if( (long)dwNumberOfEntriesInFile > lResult)
       lResult = dwNumberOfEntriesInFile;
   }
  if(lResult>m_i64TotalEntryCount)
     lResult = (long) m_i64TotalEntryCount;
  return lResult;
} // end CSpectrumBuffer::GetCountOfBufferedLines()


//--------------------------------------------------------------------------
int CSpectrumBuffer::GetNumberOfSourceChannels(void)
{
  // Examples for a few possible sequences:
  //   A, A, A, A  : ONE source channel
  //   A, B, A, B  : TWO source channels
  //   A, B, C, D  : FOUR source channels
  if(   (iLatestSourceChannels[0]==iLatestSourceChannels[1])
     && (iLatestSourceChannels[0]==iLatestSourceChannels[2]) )
     return 1;   // only ONE source channel
  if(   (iLatestSourceChannels[0]==iLatestSourceChannels[1])
     || (iLatestSourceChannels[0]==iLatestSourceChannels[2])
     || (iLatestSourceChannels[1]==iLatestSourceChannels[2]) )
     return 2;   // TWO source channels
  else
     return 4;   // FOUR source channels
} // end CSpectrumBuffer::GetNumberOfSourceChannels()

//--------------------------------------------------------------------------
void CSpectrumBuffer::DeleteEntriesForBufferIndices( DWORD dwBufIdxLo, DWORD dwBufIdxHi )
  // Deletes all entries in m_PtrArray[0 ... m_iPtrArrayUsedEntries-1]
  // which point into the specified butter-index-range .
  // These are the OLDEST entries in the buffer, which are
  // overwritten when more spectra are appended in AppendSpectrum() .
{
  int i, j;
  T_SPECTRUM *ps;
  BYTE *pbRangeLo, *pbRangeHi, *pbFirst, *pbLast; // pointers for address-comparison
  if( (m_pbBuffer != NULL ) && (m_PtrArray != NULL) && (m_iPtrArrayUsedEntries>0) )
   { pbRangeLo = m_pbBuffer + dwBufIdxLo;
     pbRangeHi = m_pbBuffer + dwBufIdxHi;
     i=0;
     while( i<m_iPtrArrayUsedEntries ) // delete all entries in m_PtrArray[i] which belong to this range
      { ps = m_PtrArray[i];
        if( ps!=NULL && ps->hdr.c.dwMagic==SPECTRUM_HDR_MAGIC )
         {
           pbFirst = (BYTE*)ps;
           pbLast  = pbFirst + ps->hdr.c.cbAllocatedSize-1;
           // The two ranges overlap if the first byte, or the last byte,
           // or both 'end bytes' are within the deleted range :
           if(   ( pbFirst>=pbRangeLo && pbFirst<=pbRangeHi )
              || ( pbLast >=pbRangeLo && pbLast <=pbRangeHi )  )
            { // This object must be removed from the buffer, because
              // it overlaps the deleted range (anywhere, even ONE BYTE is enough).
              // "Delete" it by scrolling up the rest.
              // Note: We're only "scrolling" POINTERS here, with 4 bytes per entry.
              for(j=i; j<(m_iPtrArrayUsedEntries-1); ++j)
               { m_PtrArray[j] = m_PtrArray[j+1];
               }
              --m_iPtrArrayUsedEntries;
              // There's no need to set m_PtrArray[m_iPtrArrayUsedEntries] NULL,
              // because after the following line, it will be invalid anyway.
              // But for the sake of 'stability' :
              if( m_iPtrArrayUsedEntries >= 0 )
               {  m_PtrArray[m_iPtrArrayUsedEntries] = NULL; // avoid "invalid" pointers in the array
               }
            }
         } // end if < ps points to a VALID T_SPECTRUM >
        ++i; // examine the rest of the array ( m_PtrArray[] )
      } // end while( i<m_iPtrArrayUsedEntries )
   } // end if < buffer and pointer-array VALID > ?

} // end CSpectrumBuffer::DeleteEntriesForBufferIndices()

//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::AppendSpectrum( T_SPECTRUM *pSpectrum,
                  double dblAudioCenterFrequency )
 // Copies a T_Spectrum into the large buffer (logically "at the end", physically not..)
 //   If the total size of spectra stored in RAM(!) gets too large,
 //   old entries will be kicked out.
 // Caller (seen in 2021-10) :
 // TSpectrumLab::AnalyseSpectraAndUpdateDisplay() ["the incredible monster"]
 //  -> TSpectrumLab::AppendSpectrumToBuffer()
 //      -> CSpectrumBuffer::AppendSpectrum()
 // [in] pSpectrum->hdr.v.iSourceChannel : 0 .. CFG_NUM_CHANNELS_PER_SPECTRUM_ANALYSER-1,
 //         should be 0,1,2,3,0,1,2,3,0,1,2,3 in subsequent calls (but don't assume anything).
{
  BOOL fOk = FALSE;
  long i, size, aligned_size;
  T_SPECTRUM *psDest;
  DWORD dwNewBufferHeadIndex;

  size = SpecBuf_GetSizeOfSpectrumInRAM( pSpectrum );
  if( size<=0 || size>(long)m_dwAllocatedBytesInBuffer
     || m_PtrArray==NULL || m_iPtrArrayAllocatedEntries<=0
     || m_pbBuffer==NULL || m_dwAllocatedBytesInBuffer<=0 )
   { return FALSE;   // buffer or source spectrum is invalid; refuse to append it !
   }

  // Align for 8-byte boundaries, for the sake of 64-bit integers and double floats:
  aligned_size = (size + 7) & 0x7FFFFFF8L;

  // Place the unique entry number in the spectrum header.
  //  This is required later to identify this entry in a buffer.
  pSpectrum->hdr.v.i64EntryIndex = m_i64TotalEntryCount; // incremented a bit later (below)

  // Save the "source channel number" to find out how many different channels
  // are placed in the buffer.
  // This may be important later for "mono" / "stereo" waterfalls.
  if( (pSpectrum->hdr.v.nDataType == SPEC_DT_FLOAT_ARRAY)
    ||(pSpectrum->hdr.v.nDataType == SPEC_DT_COMPLEX_ARRAY)
    ||(pSpectrum->hdr.v.nDataType == SPEC_DT_TRIPLE_ARRAY)
    ||(pSpectrum->hdr.v.nDataType == SPEC_DT_RDF_SPECTRUM)
    ||(pSpectrum->hdr.v.nDataType == SPEC_DT_RA_SPECTRUM) )
   { iLatestSourceChannels[ (int)m_i64TotalEntryCount % 3]
          = pSpectrum->hdr.v.iSourceChannel;
   }
  ++m_i64TotalEntryCount;  // Important: increment AFTER writing new entry
  if(m_i64TotalEntryCount<0)
   { m_i64TotalEntryCount=0; // overflow after a million years... ;-)
   }


  // Put the new spectrum into the RAM buffer ......
  // m_PtrArray[0]                        = address of the OLDEST entry in the spectrum buffer
  // m_PtrArray[m_iPtrArrayUsedEntries-1] = address of the NEWEST entry in the spectrum buffer
  // m_iPtrArrayAllocatedEntries = number of entries allocated in m_PtrArray[]
  // m_iPtrArrayUsedEntries      = currently USED(!) number of entries in m_PtrArray[]
  // BYTE *m_pbBuffer            = 'large' allocated block; circular (!)
  // m_dwAllocatedBytesInBuffer = number of bytes allocated in m_pbBuffer[]
  // m_dwBufferHeadIndex        = index into m_pbBuffer for the new APPENDED entry
  dwNewBufferHeadIndex = m_dwBufferHeadIndex + aligned_size;

  if( dwNewBufferHeadIndex >= m_dwAllocatedBytesInBuffer ) // new entry won't fit..
   {  // Circular wrap of index into m_pbBuffer[] !
      // Remove all pointers in m_PtrArray[] which pointed to the 'end' of m_pbBuffer[]:
      DeleteEntriesForBufferIndices( m_dwBufferHeadIndex, m_dwAllocatedBytesInBuffer-1 );
      m_dwBufferHeadIndex  = 0;
      dwNewBufferHeadIndex = aligned_size;
   }
  // Before adding the new entry, delete OLD entries which will possibly be overwritten:
  DeleteEntriesForBufferIndices( m_dwBufferHeadIndex, dwNewBufferHeadIndex-1 );
  psDest = (T_SPECTRUM*) ( (BYTE*)m_pbBuffer+m_dwBufferHeadIndex );
  memcpy( psDest, pSpectrum, size ); // copy the T_Spectrum (including frequency bins)
  m_dwBufferHeadIndex = dwNewBufferHeadIndex;  // new 'head' index, incremented by size

  // Append (!) an entry in the pointer-array. If already full, remove oldest entry [0].
  if(m_iPtrArrayUsedEntries >= m_iPtrArrayAllocatedEntries )
   { m_iPtrArrayUsedEntries =  m_iPtrArrayAllocatedEntries-1; // ONE item appended further below
     for(i=0; i<m_iPtrArrayUsedEntries; ++i)
      { m_PtrArray[i] = m_PtrArray[i+1];         // move the rest up
        // (this is just a movement of POINTERS. No expensive memcopy.)
      }
   }
  m_PtrArray[ m_iPtrArrayUsedEntries++ ] = psDest; // append pointer to new spectrum


  // If possible, also write the spectrum (FFT) into the buffer file
  //   which serves as "scroll-back"-buffer and to share the results
  //   with other applications.
  // Note: The result will not be FALSE if the memory-mapped file fails !
  if(m_hFile!=INVALID_HANDLE_VALUE)
   { // hooray, there is a memory mapped file to write into...
     WriteSpectrumToFile( pSpectrum, dblAudioCenterFrequency );
   }

  // If the entry is a SPECTRUM (and not a TEXT MESSAGE), remember the index.
  if( (pSpectrum->hdr.v.nDataType == SPEC_DT_FLOAT_ARRAY)
    ||(pSpectrum->hdr.v.nDataType == SPEC_DT_COMPLEX_ARRAY)
    ||(pSpectrum->hdr.v.nDataType == SPEC_DT_TRIPLE_ARRAY)
    ||(pSpectrum->hdr.v.nDataType == SPEC_DT_RDF_SPECTRUM)
    ||(pSpectrum->hdr.v.nDataType == SPEC_DT_RA_SPECTRUM) )
   { // ex: m_i64IndexOfLatestSpectrum = pSpectrum->hdr.v.i64EntryIndex;
     SetIndexOfLatestSpectrum( pSpectrum->hdr.v.i64EntryIndex );
   }
  else
  if(  pSpectrum->hdr.v.nDataType == SPEC_DT_TEXT_MESSAGE)
   {
    // m_i64IndexOfLatestTextMessage = pSpectrum->hdr.v.i64EntryIndex;
   }


  return fOk;
} // end CSpectrumBuffer::AppendSpectrum()

#if( SPECDISP_TEXT_MESSAGES_IN_SPECTRUM_BUFFER )
//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::AppendTextMsg( T_SPECTRUM_TEXT_MSG *pSpectrumTextMsg )
{
  BOOL fResult = FALSE;
  T_SPECTRUM *pTempSpectrum = NewSpectrum( SPEC_DT_TEXT_MESSAGE,  0/*comps_per_bin*/,  0 );
  if( pTempSpectrum != NULL )
   { pTempSpectrum->u.TextMsg = *pSpectrumTextMsg;
     fResult = AppendSpectrum( pTempSpectrum, 0.0 );
   }
  DeleteSpectrum( &pTempSpectrum );
  return fResult;
} // end CSpectrumBuffer::AppendTextMsg()
#endif // SPECDISP_TEXT_MESSAGES_IN_SPECTRUM_BUFFER ?


//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::GetSpectrum( LONGLONG i64EntryIndex, // source index
                                 T_SPECTRUM **ppSpectrum ) // destination
 // Reads a spectrum (or, sometimes, a text message) from the large buffer
 //   and copies it into the caller's buffer (unlike GetSpectrum2() ).
 // Returns TRUE if the required buffer entry is available.
 // Note: 'i64EntryIndex' is not an ARRAY INDEX, but
 //       a unique number which identifies ONE particular entry in the buffer.
 //       This index increases monotoneously; it can be larger
 //       than the buffer capacity (and never "wraps around") .
 // Caller (in Spectrum Lab) : TSpectrumLab::ProcessSpectraFromInputStream()
 //        -> TSpectrumLab::AnalyseSpectraAndUpdateDisplay()
 //            -> TSpectrumLab::DrawNewWaterfallLines()
 //                -> SpecDisp_GetMultiChannelSpectraFromBuffer()
 //                    -> SpecDisp_GetSpectrumButNoMsg()
 //                        -> CSpectrumBuffer::GetSpectrum() .
{
  int i;
  // Check if the requested spectrum can be in the buffer at all..
  if( (i64EntryIndex<0) || (i64EntryIndex>m_i64TotalEntryCount) )
     return FALSE;

  // First try to retrieve the requested spectrum from the RAM buffer
  //   to avoid unnecessary file accesses.
  // Note: The entries in the RAM-buffer may be chaotically organized,
  //       we may use it as a cache for entries read from the FILE BUFFER(!)
  //       So we just do a linear search for the requested entry.
  //       If it's not there, we may get it from the FILE later.
  for(i=0; i<m_iPtrArrayUsedEntries; ++i)
   { if( m_PtrArray[i] != NULL )
      {
        if(m_PtrArray[i]->hdr.v.i64EntryIndex == i64EntryIndex)
         {
           return CopySpectrum(
              m_PtrArray[i], // pointer to source spectrum
              ppSpectrum );  // pointer to destination spectrum
         }
      }
   } // end <search loop for RAM array>

  // Arrived here: requested spectrum is not avalilable in the RAM buffer.
  // Try to read it from the BUFFER FILE :
  if(m_hFile!=INVALID_HANDLE_VALUE)
   { // bingo, there is a memory mapped file to read from...
     return ReadSpectrumFromFile( i64EntryIndex, ppSpectrum );
   }
  else
   {
     // There is no file to read from...
   }

  return FALSE;
} // end CSpectrumBuffer::GetSpectrum()


//--------------------------------------------------------------------------
T_SPECTRUM *CSpectrumBuffer::GetSpectrumByTimestamp( double dblTimestamp )
 // Tries to find the spectrum closest to the specified timestamp.
 //       Optimized for SPEED (uses a binary search, and ONLY looks
 //       at the spectra currently buffered in RAM; not those in a FILE).
 //  If the caller needs to find check the NEIGHBOUR spectra (closest to a
 //  given timestamp, e.g. from a TEXT MESSAGE painted into the spectrogram),
 //  the neighbours can be retrieved via GetSpectrum2(), passing in the
 //  T_SPECTRUM.hdr.v.i64EntryIndex PLUS/MINUS ONE from the spectrum returned by
 //  GetSpectrumByTimestamp() .
 //
 //   Unlike GetSpectrum(), GetSpectrumByTimestamp() does not COPY data.
 //   Unlike GetSpectrum(), GetSpectrumByTimestamp() does not read from a BUFFER FILE.
 //
 // Returns NULL if there no spectrum in the buffer (RAM) close enough
 // to the specified timestamp .
 // First used by TSpectrumLab::UpdateWaterfallTextMessages() for a
 //   'pixel-accurate' placement of TEXT MESSAGES on the waterfall,
 //   even if the waterfall's SPECTRUM TIMESTAMPS aren't nicely spaced.
 //   (no constant timestamp differences between adjacent spectra,
 //    which happened when receiving spectra via RigControl.c ) .
 //
 // See also (similar function) : GetIndexOfEntryByTime() .
{
  LONGLONG i64Searchpos;
  BOOLEAN  higher, lower;
  LONGLONG i64High = GetIndexOfLatestSpectrum();
  LONGLONG i64Low  = GetIndexOfOldestSpectrum();
  LONGLONG i64ArrayIndex;
  T_SPECTRUM *pSpectrum;

  if( i64Low < 0 )
      i64Low = 0;

  // use a binary search to find the nearest buffer entry with the specified TIME :
  if( i64High <= i64Low )
     return NULL;  // entry cannot be found by binary search in the ARRAY !

  do
   {
     /* new search position in the middle between low and high position */
     i64Searchpos = ( i64High + i64Low ) / 2;

     // Convert the "unique" 64-bit spectrum index into an ARRAY-index :
     i64ArrayIndex = i64Searchpos - m_PtrArray[0]->hdr.v.i64EntryIndex;
     // Now m_PtrArray[i64ArrayIndex] is our best bet for the 'wanted' entry, if the
     // buffer contains the wanted entry at all. See CSpectrumBuffer::GetSpectrum2().
     if( i64ArrayIndex < 0 ) // spectrum cannot be retrieved, it's "far too old" :
      { higher = FALSE;
        lower  = TRUE;
        pSpectrum = NULL;
      }
     else if( i64ArrayIndex >= m_iPtrArrayUsedEntries ) // spectrum cannot be retrieved, it's "not in the buffer yet" :
      { higher = TRUE;
        lower  = FALSE;
        pSpectrum = NULL;
      }
     else
      { pSpectrum = m_PtrArray[(int)i64ArrayIndex];
        higher = pSpectrum->hdr.v.dbl_time > dblTimestamp;
        lower  = pSpectrum->hdr.v.dbl_time < dblTimestamp;
      }

     if( (! higher) && (! lower) )
      { break;   /* found !  (quite unlikely to be "exactly the same" floating-point value) */
      }

     if ( higher )
      {
         if ( i64Searchpos == i64High )  // reached the "other end" ? -> ready
            break;
         else
            i64High = i64Searchpos;
      }
     if ( lower )
      {
         if ( i64Searchpos == i64Low )   // reached the "other end" ? -> ready
             break;
         else
            i64Low = i64Searchpos;
      }
   }  while ( TRUE );

  return pSpectrum;
} // end CSpectrumBuffer::GetSpectrumByTimestamp()


//--------------------------------------------------------------------------
T_SPECTRUM * CSpectrumBuffer::GetSpectrum2(
                int iSourceChannel, // [in] for multi-channel analysis :
                          //  0 = LEFT audio channel ,   1 = RIGHT audio channel .
                LONGLONG *pi64TailIndex ) // [in,out] tail index for FIFO (in RAM)
 // Retrieves the address of the next spectrum in the FIFO (RAM).
 //   Unlike GetSpectrum(), does not COPY data.
 //   Unlike GetSpectrum(), does not read from the optional BUFFER FILE.
 //   First used by SL's integrated OpenWebRX server .
 // Return value : Address of the next spectrum when a new entry is available,
 //                otherwise NULL .
 // Note: The TAIL INDEX is not strictly an ARRAY INDEX, but
 //       a unique number which identifies ONE particular entry in the buffer.
 //       This index increases monotoneously; it can be larger
 //       than the buffer capacity (and never "wraps around") .
 // GetSpectrum2() is also used to distribute spectra for the 'OpenWebRX'-server,
 //    when called though TSpectrumLab::Timer1Timer()
 //      -> HttpSrv_OnTimerEvent() -> HttpSrv_OnPollForTxData()
 //       -> OnHttpSrvPollForTxData()         [in SpecLab/SpecHttpSrc]
 //        -> OWRX_OnWebSocketPollForTxData() [in OpenWebRX_Server.cpp]
 //         -> OWRX_Server_GetNextSpectrum()
 //          -> OWRX_GetSpectrumFromFIFO()
 //           -> CSpectrumBuffer::GetSpectrum2() .   Phew !
{
  int i;
  LONGLONG i64EntryIndex = *pi64TailIndex;
  LONGLONG i64Latest = GetIndexOfLatestSpectrum();
  LONGLONG i64Oldest = GetIndexOfOldestSpectrum();
  T_SPECTRUM *pResult;

  // Check if the requested spectrum can be in the buffer at all..
  if( i64EntryIndex < i64Oldest )  // the "wanted" spectrum isn't available anymore..
   {  i64EntryIndex = i64Oldest;   // .. return the OLDEST instead
   }

  if( i64EntryIndex > i64Latest )  // entry not available yet ..
   { // If FIFO has been cleared in between, the caller's tail index may be
     // far ahead of i64IndexOfLatestEntry. Avoid a 'frozen' spectrum/waterfall:
     if( i64EntryIndex > (i64Latest+1) )  // too far into the future..
      {  i64EntryIndex =  i64Latest;
         //
      }
     else
      {  return NULL;  // ok, caller just needs to wait
      }
   } // if( i64EntryIndex > i64Latest )

  // Shortcut to avoid the search-loop further below, if the caller wants
  // to retrieve the LATEST (newest) available entry, which is often the case.
  if( (m_iPtrArrayUsedEntries >= 2) && (m_PtrArray[0] != NULL) )
   { i = i64EntryIndex - m_PtrArray[0]->hdr.v.i64EntryIndex;
     // Now m_PtrArray[i] is our best bet for the 'wanted' entry, if the
     // buffer still contains the wanted entry. Simple example:
     //    m_iPtrArrayUsedEntries = 10  (-> only m_PtrArray[0] .. m_PtrArray[9] are possible canditates )
     //    m_PtrArray[0]->hdr.v.i64EntryIndex = 30  (index wrapped around THREE times)
     //    i64EntryIndex = 35                       ("wanted" by caller)
     //    i = 35 - 30   = 5     (-> that's our "best bet", the maximum would be i=9)
     if( i < m_iPtrArrayUsedEntries ) // if 'i' is a valid index into m_PtrArray[]..
      { if( (pResult=m_PtrArray[i]) != NULL )
         { if( pResult->hdr.v.i64EntryIndex == i64EntryIndex )
            { // bingo, all those indices look ok, but is it really a SPECTRUM ?
              switch( pResult->hdr.v.nDataType )
               { case SPEC_DT_FLOAT_ARRAY:   // yes... it's a spectrum, with amplitudes only
                 case SPEC_DT_COMPLEX_ARRAY: // yes... it's a spectrum, with complex frequency bins
                 case SPEC_DT_TRIPLE_ARRAY:  // yes... special spectrum, with three floats per f-bin
                 case SPEC_DT_RDF_SPECTRUM:  // yes... it's a spectrum for radio direction finding
                 case SPEC_DT_RA_SPECTRUM:   // yes... it's a reassigned spectrum
                    *pi64TailIndex = i64EntryIndex + 1; // entry index (~counter) for the NEXT call
                    return pResult;
                 case SPEC_DT_UNUSED_ENTRY:  // index already "occupied" but no valid data yet ?
                    return NULL;   // see you later !
                 default : // anything else (e.g. text message overlaid in the waterfall) -> SKIP !
                    ++i64EntryIndex;
                    *pi64TailIndex = i64EntryIndex; // deliver the NEXT entry instead (maybe in a FUTURE call)
                    break;
               } // switch( (pResult->hdr.v.nDataType )
            }
         }
      }
   }


  // Arrived here ? Bite the bullet and use simple "linear search" :
  for(i=0; i<m_iPtrArrayUsedEntries; ++i)
   { if( (pResult=m_PtrArray[i]) != NULL )
      { if( pResult->hdr.v.i64EntryIndex == i64EntryIndex )
         { // Found the wanted index, but is it really a SPECTRUM ?
           switch( pResult->hdr.v.nDataType )
            { case SPEC_DT_FLOAT_ARRAY:   // yes... it's a spectrum, with amplitudes only
              case SPEC_DT_COMPLEX_ARRAY: // yes... it's a spectrum, with complex frequency bins
              case SPEC_DT_TRIPLE_ARRAY:  // yes... special spectrum, with three floats per f-bin
              case SPEC_DT_RDF_SPECTRUM:  // yes... it's a spectrum for radio direction finding
              case SPEC_DT_RA_SPECTRUM:   // yes... it's a reassigned spectrum
                 *pi64TailIndex = i64EntryIndex + 1; // entry index (~counter) for the NEXT call
                 return pResult;
              case SPEC_DT_UNUSED_ENTRY:  // index already "occupied" but no valid data yet ?
                 return NULL;   // see you later !
              default : // anything else (e.g. text message overlaid in the waterfall) -> SKIP !
                 ++i64EntryIndex;
                 *pi64TailIndex = i64EntryIndex; // deliver the NEXT entry instead (maybe in a FUTURE call)
                 break;
            } // switch( (pResult->hdr.v.nDataType )
         } // end if( pResult->hdr.v.i64EntryIndex == i64EntryIndex )
      }  // end ( m_PtrArray[i] != NULL
   } // end <search loop for RAM array>

  return NULL;
} // end CSpectrumBuffer::GetSpectrum2()

//--------------------------------------------------------------------------
void CSpectrumBuffer::CloseFile(void)
{

   if( m_fMustFlushHeader )
    { // Write the header back to the buffer file.
      // This is important if the old spectra shall become visible again
      // when re-starting the program.
      // 2004-05-06, important:
      WriteFileHeader();
    }

  if(m_hFile != INVALID_HANDLE_VALUE)
   { // To "CloseHandle" or "CloseFile", that's the question !
     // On re-opening the file again with "CreateFile", the
     // API says "already used by another process" though the file
     // had been closed with CloseHandle !!!
     if(!CloseHandle(m_hFile))
       {
        dwLastError = GetLastError();
       }
     m_hFile = INVALID_HANDLE_VALUE;
   }
} // end CSpectrumBuffer::CloseFile()


//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::SetFilePosIfNeeded(DWORD dwNewFilePos)
{
  if(m_dwCurrentFilePos == dwNewFilePos)
    return TRUE;  // no need to call SetFilePointer(), may save some time
  if(dwNewFilePos > m_dwCurrentFileSize)
   {   // this can only go wrong; bail out
    strcpy(sz80LastError,"filepos would exceed size");
    DEBUGGER_BREAK();  // SET BREAKPOINT HERE,  or in DebugU1.cpp !
    return FALSE;
   }
  if(SetFilePointer(m_hFile,dwNewFilePos,NULL,FILE_BEGIN)==0xFFFFFFFF)
   { // error in SetFilePointer....
    strcpy(sz80LastError,"SetFilePointer failed");
    return FALSE;
   }
  else // SetFilePointer successfull...
   {
    m_dwCurrentFilePos = dwNewFilePos;
    return TRUE;
   }
} // end CSpectrumBuffer::SetFilePosIfNeeded()



//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::WriteFileHeader(void)
{
 BOOL fResult;
 long i,j,stepwidth;  // 16 bit integer is not sufficient, so beware !
 DWORD dwNrBytesWritten;

  // Before writing the header to the file, update some parts of it :
  m_FileHeader.i64TotalEntryCount = m_i64TotalEntryCount;

  // Caution: We may "operate" on an old file, where the header was smaller
  //          than it is now. To avoid trashing old files, we cannot
  //          simply set m_FileHeader.cbHeaderStructSize = sizeof(T_SpectrumFileHeader) :
  // Only if the indicated size is "totally wrong", correct it :
  if( m_FileHeader.cbHeaderStructSize < 88/*old header struct size*/ )
   {  m_FileHeader.cbHeaderStructSize = 88; // "old" sizeof(T_SpectrumFileHeader)
   }
  if( m_FileHeader.cbHeaderStructSize > sizeof(T_SpectrumFileHeader) )
   {  m_FileHeader.cbHeaderStructSize = sizeof(T_SpectrumFileHeader);
   }
  // Update the small look-up table in the header .
  //  (the file-lookup-table in RAM is much larger than the "essay" in the header,
  //   but even the small table in the header helps to find certain entries
  //   in the spectrum buffer, because it avoids having to skip many megabytes..)
  stepwidth = m_dwFileLookupTableUsedEntries / SPECBUFF_HEADER_LOOKUP_TABLE_SIZE;
  if( stepwidth < 1 )
      stepwidth = 1;
  j = 0;
  if( m_pFileLookupTable != NULL )
   { for(i=0; i<(long)m_dwFileLookupTableUsedEntries && j<SPECBUFF_HEADER_LOOKUP_TABLE_SIZE; i+=stepwidth)
      { m_FileHeader.LookupTable[j].i64EntryIndex = m_pFileLookupTable[i].i64EntryIndex;
        m_FileHeader.LookupTable[j].dwFilePos     = m_pFileLookupTable[i].dwFilePos;
        ++j;
      }
   }
  while( j<SPECBUFF_HEADER_LOOKUP_TABLE_SIZE )
   { m_FileHeader.LookupTable[j].i64EntryIndex = 0;
     m_FileHeader.LookupTable[j].dwFilePos     = 0;
     ++j;
   }

  // Writing the header should normally NOT affect m_dwCurrentFilePosForWriting .
  // But make sure the next write-access to the data array will not overwrite the header :
  if( ( m_dwCurrentFilePosForWriting < m_FileHeader.cbHeaderStructSize )
    ||( m_dwCurrentFilePosForWriting >= m_FileHeader.dwMaxFileSize) )
   {  m_dwCurrentFilePosForWriting = m_FileHeader.cbHeaderStructSize;
   }
  m_FileHeader.dwNextFilePosForWriting = m_dwCurrentFilePosForWriting;

  if(m_hFile == INVALID_HANDLE_VALUE)
    return FALSE;

  if(SetFilePointer(m_hFile,0,NULL,FILE_BEGIN) == 0xFFFFFFFF)
   { strcpy(sz80LastError,"cannot rewind to header");
     return FALSE;
   }
  m_dwCurrentFilePos = 0;
  m_fMustFlushHeader = FALSE;
  fResult = WriteFile(m_hFile, &m_FileHeader, m_FileHeader.cbHeaderStructSize, &dwNrBytesWritten, NULL);
  if( fResult==TRUE )
   { m_dwCurrentFilePos += dwNrBytesWritten;
   }
  if( m_dwCurrentFileSize < m_FileHeader.cbHeaderStructSize )
   {  m_dwCurrentFileSize = m_FileHeader.cbHeaderStructSize;
   }

  return fResult;

} // end CSpectrumBuffer::WriteFileHeader()


//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::OpenBufferFile(
      char *pszFileName,
      BOOLEAN fCreateNewFile ) // TRUE = create a new buffer file; FALSE = open existing
{
  int i;
  T_SPECTRUM_HDR hdr;
  DWORD dwFilePos;

  // Close the old file, and free temporary memory if needed..
  CloseFile();

  m_dwCurrentFileSize=0;

  // Since 2007-12-23, initialize (but do not fill)
  // the "file lookup table" which we *may* need later
  // to find a certain entry in the buffer file :
  DeleteFileLookupTable();
  SetIndexOfLatestSpectrum( 0 );  // here: cleared before opening the BUFFER FILE
  m_i64FileLookupTableFirstSeekedIndex = -1;
  m_i64FileLookupTableLastSeekedIndex  = -1;

  // First step for a classic access OR memory-mapped file...
  //   Create or open a file which will be mapped into memory later.
  m_hFile = CreateFile(
     pszFileName, // LPCTSTR lpFileName     = pointer to name of the file
     GENERIC_READ | GENERIC_WRITE, // DWORD dwDesiredAccess  = access mode
      0 ,         // DWORD dwShareMode      = share mode
     NULL,        // no pointer to security attributes
     fCreateNewFile ?  // DWORD dwCreationDistribution = how to create..
        CREATE_ALWAYS : OPEN_ALWAYS,
     FILE_ATTRIBUTE_NORMAL, // DWORD dwFlagsAndAttributes =   file attributes
     NULL ); // HANDLE hTemplateFile       = handle to file with attributes to copy
  if(m_hFile == INVALID_HANDLE_VALUE)
    {
     dwLastError = GetLastError();
     strcpy(sz80LastError,"couldn't open buffer file");
     return FALSE;
    }

  // Next step:
  //   If an existing file shall be opened, detect the size.
  //   The SetFilePointer function moves the file pointer of an open file.
  if( ! fCreateNewFile )
   {
     m_dwCurrentFileSize =  SetFilePointer(
                 m_hFile, // handle of file
                     0, // number of bytes to move file pointer
                  NULL, // address of high-order word of distance to move
             FILE_END); // DWORD dwMoveMethod = how to move
     if(m_dwCurrentFileSize==0xFFFFFFFF) // SetFilePointer failed
        m_dwCurrentFileSize = 0;
   }
  else // don't open an existing file, create a new one. The old size doesn't matter !
   {
     m_dwCurrentFileSize = 0;
   }


  BOOL fHeaderOk = FALSE;
#ifdef __BORLANDC__
  (void)fHeaderOk; // ... "was a assigned a value that is never used" .. oh, shut up !
#endif

  if( (fCreateNewFile) || (m_dwCurrentFileSize < sizeof(T_SpectrumFileHeader)) )
   {
     // If the size (dwFilePos) is zero, the file did(!) not exist before
     //  calling CreateFile.   A new header will be written below.
     fHeaderOk = FALSE;
   }
  else // size of the opened file is not zero: Analyse header structure...
   {
     // TRY TO  read and analyse the header of an existing buffer file.
     DWORD dwNumberOfBytesRead;
     SetFilePointer( m_hFile, 0, NULL, FILE_BEGIN);
     m_dwCurrentFilePos = 0;
     dwNumberOfBytesRead = 0;
     if( ReadFile(  // 'ReadFile' failed 2004-01-19 for unknown reasons
                 m_hFile, (void*)&m_FileHeader,
                 sizeof(T_SpectrumFileHeader), // number of bytes to write
                 &dwNumberOfBytesRead, NULL)
         && (dwNumberOfBytesRead==sizeof(T_SpectrumFileHeader)) )
       { // File header read successfully
         m_dwCurrentFilePos += dwNumberOfBytesRead;
         fHeaderOk = (strncmp(m_FileHeader.szFileTextInfo, "Spectrum", 8) == 0);
         fHeaderOk &= (m_FileHeader.dwMaxFileSize  == m_dwMaxFileSizeInBytes);
         fHeaderOk &= (m_FileHeader.dwSizeOfArrayEntry >  sizeof(T_SPECTRUM_HDR));
            // ex:    &&(m_FileHeader.dwSizeOfArrayEntry <= sizeof(T_SPECTRUM) )   );
         // The following members were added 2007-12-27 . Before that date,
         // sizeof(T_SpectrumFileHeader) was 88 bytes :
         if( m_FileHeader.cbHeaderStructSize == 0 )
          { // assume it's a "very old" buffer file (without size indicator)
            m_FileHeader.cbHeaderStructSize = 88; // not sizeof(T_SpectrumFileHeader) !!
          }
         if( m_FileHeader.cbHeaderStructSize==sizeof(T_SpectrumFileHeader) )
          { // must be an up-to-date version of the header structure:
          }
         else
          { // must be an outdated version of the header structure:
            // forget everything which didn't exist in the very first version .
            m_FileHeader.dwNextFilePosForWriting = m_FileHeader.cbHeaderStructSize; // (!)
            for(i=0; i<SPECBUFF_HEADER_LOOKUP_TABLE_SIZE; ++i)
             { m_FileHeader.LookupTable[i].i64EntryIndex = 0;
               m_FileHeader.LookupTable[i].dwFilePos = 0;
             }
          }

         if(fHeaderOk) // IF the buffer-file-header is valid ,
          {  // copy a few important parameters from the header :
           m_dwCurrentFilePosForWriting = m_FileHeader.dwNextFilePosForWriting;
           m_i64TotalEntryCount = m_FileHeader.i64TotalEntryCount;
           SetIndexOfLatestSpectrum( m_FileHeader.i64TotalEntryCount-1 ); // beware...
           for(i=0; i<SPECBUFF_HEADER_LOOKUP_TABLE_SIZE; ++i)
            { if( m_FileHeader.LookupTable[i].dwFilePos > 0 )
               { AddEntryToFileLookupTable(
                     m_FileHeader.LookupTable[i].i64EntryIndex,
                     m_FileHeader.LookupTable[i].dwFilePos );
                 if( m_FileHeader.LookupTable[i].i64EntryIndex > m_i64IndexOfLatestSpectrum )
                   { SetIndexOfLatestSpectrum( m_FileHeader.LookupTable[i].i64EntryIndex );
                   }
               }
            }
           // Added 2010-10-21 : Because buffer-files, and the above lookup-table,
           //  sometimes got corrupted due to abnormal program termination,
           //                                 or version incompatibility,
           //  load the LOOKUP TABLE too, and while doing that,
           //  increment m_i64TotalEntryCount if necessary
           //  (the same 64-bit index must NEVER be used twice,
           //   because m_i64TotalEntryCount is used as the 'running index'
           //   for any T_SPECTRUM entry in the buffer ..
           //   like a database TABLE KEY which must be unique )
           dwFilePos = m_dwCurrentFilePos;
           ReadHeadersFromFile( m_FileHeader.i64TotalEntryCount, &dwFilePos, &hdr );

          } // end if(fHeaderOk)
       }
      else
       { // header could not be read. Forget the old file...
         strcpy(sz80LastError,"couldn't read header of buffer file");
         fHeaderOk = FALSE;  // try to create and write a new header
       }
   } // end if <try to read the header of an existing file>

  // To avoid a catastropy if the file header is corrupted:
  if( m_FileHeader.dwMaxFileSize < sizeof(T_SpectrumFileHeader) )
      fHeaderOk = FALSE;

  // Must create an "all-new" header structure...
  //  because we need a VALID HEADER in memory to calculate the max. size
  //  of the memory mapped file before calling CreateFileMapping.
  if(!fHeaderOk)
   { // fill a "fresh" file header..
     FillDefaultSpectrumFileHeader(&m_FileHeader, m_dwMaxFileSizeInBytes);
     // Writing not yet possible here because no file mapping available
     m_fMustFlushHeader = TRUE;
   } // end if (dwFilePos==0)

  // If allowed and required, write a "clean" header...
  if(m_fMustFlushHeader)
   {
     WriteFileHeader();
   } // end if(!fHeaderOk) ..

  return TRUE;
} // end CSpectrumBuffer::OpenBufferFile()

//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::ReadSpectrumFromFile( LONGLONG i64EntryIndex, T_SPECTRUM **ppsDst )
 // Note: 'i64EntryIndex' is not an ARRAY INDEX, but
 //       a unique number which identifies ONE particular entry in the buffer.
 //       This index increases monotoneously; it can be larger
 //       than the buffer capacity (NO WRAPPING) .
{
 DWORD dwFilePos;
 long  i32NumberOfBytesToRead;
 DWORD dwNumberOfBytesRead;
 long  i32MaxDestCapacityInByte;
 long  i32;
 T_SPECTRUM_HDR hdr;

  if(!ppsDst)
    return FALSE;
  if( ! SpecBuf_IsValidSpectrum( *ppsDst ) )
   {
     if( nSpectrumCopyErrors < 5 )
      {  nSpectrumCopyErrors++;
         DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, 0.0/*time*/,
                 "ReadSpectrumFromFile() : destination invalid !" );
      }
     return FALSE;
   }

  // Check valid array index and calculate the file position...
  if(m_hFile==INVALID_HANDLE_VALUE)
     return FALSE;
  if(! EntryIndexToFilePos( i64EntryIndex, &dwFilePos) )
     return FALSE;
  if( dwFilePos >= m_dwCurrentFileSize )
   { // there must be a bug in the lookup-table. Set a breakpoint here, and try again:
     EntryIndexToFilePos( i64EntryIndex, &dwFilePos) ;
   }

  // Read a spectrum from file, using "traditional" access methods
  if(!SetFilePosIfNeeded(dwFilePos) )
   { // Oops, something utterly wrong here.
     //  Most likely: File is shorter than it should be, possibly header corrupted.
     return FALSE;
   }
  // First read the "spectrum header" only, then calculate the size
  //       of the entire entry and read the "rest" (=the frequency bins) .
  dwNumberOfBytesRead = TRUE;  // did you know ReadFile returns TRUE even though nothing was read ?!
  if( (!ReadFile( m_hFile, (void*)&hdr,
                   sizeof(T_SPECTRUM_HDR),  // number of bytes to read (ideally)
                   &dwNumberOfBytesRead,    // number of bytes actually read
                   NULL) )
        || (dwNumberOfBytesRead!=sizeof(T_SPECTRUM_HDR) )  )
        return FALSE;
      else
       { m_dwCurrentFilePos += dwNumberOfBytesRead;
         // Note: m_dwCurrentFilePosForWriting must not be affected by this !!
         // Arrived here, we have read the HEADER of a T_SPECTRUM structure.
         // Do a quick check on the data in the header :
         if( hdr.c.dwMagic != SPECTRUM_HDR_MAGIC )
          { return FALSE;
          }
         // If the destination (in RAM) is too small to load the T_SPECTRUM from file,
         // free and re-allocate it with the required size. BTW this is the reason
         // why CSpectrumBuffer::ReadSpectrumFromFile() requires a POINTER TO A POINTER
         // for the destination spectrum : it may have to change it's address.
         if(  ( hdr.v.nDataType != (*ppsDst)->hdr.v.nDataType )
           || ( ( hdr.v.nDataType != SPEC_DT_TEXT_MESSAGE)
               && hdr.v.iNrOfBins > (*ppsDst)->hdr.v.iNrOfBins) )
          { DeleteSpectrum( ppsDst );
            *ppsDst = NewSpectrum( hdr.v.nDataType, 0/*guess comps_per_bin*/, hdr.v.iNrOfBins );
            if( *ppsDst==NULL )
               return FALSE;
          }
         (*ppsDst)->hdr.v = hdr.v;  // copy the "variable" part of the header
             // (but not the "constant" part; which remains as set in NewSpectrum() )
         // Arrived here, all ok, so read the rest (ARRAY, union).
         // How many remaining bytes must be read into that part of the struct ?
         //    ( carefully read the notes about T_SPECTRUM.c.cbAllocatedSize
         //      in C:\CBproj\SpecLab\TSpectrum.h ! )
         i32NumberOfBytesToRead = (long)hdr.c.cbAllocatedSize - (long)sizeof(T_SPECTRUM_HDR);
         // Make sure not to exceed the maximum capacity of the destination struct:
         i32MaxDestCapacityInByte= (long)(*ppsDst)->hdr.c.cbAllocatedSize - (long)sizeof(T_SPECTRUM_HDR);
         i32 = i32NumberOfBytesToRead;
         if( i32>i32MaxDestCapacityInByte )
             i32=i32MaxDestCapacityInByte;
         if( ReadFile( m_hFile, (void*)&((*ppsDst)->u),
                 i32,  // number of bytes to read
                 &dwNumberOfBytesRead, NULL) )
          { m_dwCurrentFilePos += dwNumberOfBytesRead;
            if( i32<i32NumberOfBytesToRead ) // must skip some of the file contents ?!
             { SetFilePosIfNeeded( m_dwCurrentFilePos + (i32NumberOfBytesToRead-i32 ) );
             }
            // Update the array of 'latest-source-channels',
            //   to detect the number of channels in the file
            //   for the spectrum-replay-function .
            if( i64EntryIndex > (m_i64TotalEntryCount-8) )
             { // the spectrum just read from the buffer is "one of the newest".
               // Update iLatestSourceChannels[0..3] so the spectrum-replay-function
               // will know if the buffer contains "mono" or "stereo" recordings.
               if( (hdr.v.nDataType == SPEC_DT_FLOAT_ARRAY)
                ||( hdr.v.nDataType == SPEC_DT_COMPLEX_ARRAY)
                ||( hdr.v.nDataType == SPEC_DT_TRIPLE_ARRAY)
                ||( hdr.v.nDataType == SPEC_DT_RDF_SPECTRUM)
                ||( hdr.v.nDataType == SPEC_DT_RA_SPECTRUM) )
                { iLatestSourceChannels[ (int)i64EntryIndex % 3] = hdr.v.iSourceChannel;
                }
             } // end < stuff added 2008-03-02 >

            if( ! SpecBuf_IsValidSpectrum( *ppsDst ) )
             {  DEBUGGER_BREAK();
                if( nSpectrumCopyErrors < 5 )
                 {  nSpectrumCopyErrors++;
                    DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, 0.0/*time*/,
                      "ReadSpectrumFromFile() : result invalid !" );
                 }
             }

            return TRUE;
          }
    } // end else < ReadFile(HEADER) ok >
  return FALSE;

} // end CSpectrumBuffer::ReadSpectrumFromFile()

//--------------------------------------------------------------------------
BOOL CSpectrumBuffer::WriteSpectrumToFile( T_SPECTRUM *psSrc, double dblCenterFrequency )
  // private function; called from CSpectrumBuffer::AppendSpectrum() .
{
 // DWORD dwFilePos;
 // T_SpectrumFileLookupTableEntry sFileEntryInfo;
 DWORD dwAllocSize;
 // int iMaxNrOfBins;
 BOOL ok;
 // double d;

  // Also write the "running number" into the header. Important for other readers.
  // Important: Set flag to FLUSH the header later when closing the file !
  if( m_FileHeader.i64TotalEntryCount != m_i64TotalEntryCount )
     m_fMustFlushHeader = TRUE;
  m_FileHeader.i64TotalEntryCount = m_i64TotalEntryCount;
  m_FileHeader.dwSizeOfArrayEntry = psSrc->hdr.c.cbAllocatedSize;


  if(m_hFile==INVALID_HANDLE_VALUE)
    return FALSE;  // error, no access to a memory mapped file

  // Refuse to write anything into the file which is not a valid T_SPECTRUM:
  if( ! SpecBuf_IsValidSpectrum( psSrc ) )
   { strcpy(sz80LastError,"Refused to write invalid struct");
     return FALSE;  // error, this is not a valid T_SPECTRUM !
   }

  // Check valid array index and calculate the file position...
  if( ( m_dwCurrentFilePosForWriting < m_FileHeader.cbHeaderStructSize )
    ||( (m_dwCurrentFilePosForWriting + psSrc->hdr.c.cbAllocatedSize)
          > m_FileHeader.dwMaxFileSize) )
   { // wrap around (the spectrum buffer file is a CIRCULAR buffer) :
     m_dwCurrentFilePosForWriting = m_FileHeader.cbHeaderStructSize;
   }

  AddEntryToFileLookupTable( psSrc->hdr.v.i64EntryIndex,
                             m_dwCurrentFilePosForWriting );

  // Only call "SetFilePointer" internally if really required :
  if(! SetFilePosIfNeeded( m_dwCurrentFilePosForWriting ) )
     return FALSE;

  // If the size-per-entry in the buffer file shall be smaller
  // than in the passed structure, TEMPORARILY replace the
  // "number of allocated bytes" in the spectrum's header
  dwAllocSize = psSrc->hdr.c.cbAllocatedSize;

  DWORD dwNumberOfBytesWritten=0;
    // Warning: The very first "WriteFile" after BOOTING(!) THE PC
    //          may take very long (many seconds), if the file pos is high;
    //          even if only a few kilobytes are written ! ! ! !
    //   No workaround known yet. The effect cannot be reproduced.
    ok = WriteFile( m_hFile, (void*)psSrc, // source: original or minimized spectrum
             psSrc->hdr.c.cbAllocatedSize, // [in] number of bytes to be written
           &dwNumberOfBytesWritten, NULL); // [out] number of bytes actually written
    psSrc->hdr.c.cbAllocatedSize = dwAllocSize; // restore original header (source)

    if(!ok)
      { // WriteFile() failed ?!  (this happened 2007-12-27) :
        dwLastError = GetLastError();   // 2007-12-27: got error #1784 =
        //  "Der angegebene Benutzerpuffer ist fr den angeforderten Vorgang nicht zulssig."
        // Totally ridiculous. WriteFile should be able to write more than
        // just a crippled amount of 65536 bytes per call (here:  ) !
        DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR, 0, 0.0/*time*/,
           "WriteSpectrumToFile: Windoze moans \"%s\".",
            UTL_LastApiErrorCodeToString(dwLastError) );
        return FALSE;
      }
    else
      { m_dwCurrentFilePosForWriting += dwNumberOfBytesWritten;
        if(m_dwCurrentFilePosForWriting > m_dwCurrentFileSize)
           m_dwCurrentFileSize = m_dwCurrentFilePosForWriting; // keep track of the current file size
        // no return here !
      }

  m_fMustFlushHeader = TRUE;  // ..because m_FileHeader.i64TotalEntryCount has been changed

  return TRUE;
} // end CSpectrumBuffer::WriteSpectrumToFile()


// EOF <SpecBuff.cpp>
