//---------------------------------------------------------------------------
// File  :  SoundUps.h
// Date  :  2004-12-21  (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Interface for the CSoundUpsamplingBuffer class.
//
// Used in the following projects (at least) :
//    - Spectrum Lab         (c:\CBproj\SpecLab\SpecLab)
//    - Sound Output Utility (c:\CBproj\SoundUtl\SndOutpt)
//
// Revision history :  See *.cpp
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#ifndef SoundUpsH
#define SoundUpsH

#include "SoundTab.h"  // frequently used types and tables for audio processing

//----------------------- Constants ---------------------------------------

// Number of cascaded upsampling stages for the input data :
//     8 stages can multiply the sampling rate with 3^8  =  6561
//    12 stages can multiply the sampling rate with 3^12 = 531441
#define SoundUps_MAX_UPSAMPLING_STAGES  12



//----------------------- Data Types --------------------------------------


typedef struct
{
 int    ratio;            // decimation ratio of this stage
 int    count;            // counter for decimation_ratio, 0=output register empty
 T_Complex cplxOutput[3]; // Output buffer. ONE input sample may produce up to THREE output samples.
                          //     TO READ: Use .cplxOutput[ -- count ]  (!!)
 T_Complex queue[SoundTab_DEC_FIR_LENGTH];   // FIR filter input queue....
 T_Complex *inptr;        // pointer for circular queue
 const T_Float *coeffs;    // pointer to a table of coefficients
} T_SOUND_UPSAMPLER;


//------------------------ Global Variables -------------------------------


//**************************************************************************
//    Prototypes (no class methods)
//**************************************************************************



//***************************************************************************
//  Definition of the  CSoundUpsamplingBuffer   class
//***************************************************************************

class CSoundUpsamplingBuffer
{
public:
   CSoundUpsamplingBuffer();            // constructor without parameters
   virtual ~CSoundUpsamplingBuffer();   // destructor (cleans up)

   BOOL Init(  long lInputBufferCapacity, // max count of sample points in the INPUT BUFFER
             T_Float dblOutputSampleRate,  // 'reading' sample rate, higher than input sample rate
               long lSampleRateMultiplier, // powers of two preferred
             T_Float dblMixerFrequency,  // "L.O." frequency in Hertz
               int  iComplexity,        // like COMPLEX_IN_REAL_OUT, etc (defined in SoundTab.h)
                int iNrChannels );      // 1=mono, 2=stereo processing


   long    GetFreeInputBufferSpace(void);     // returns a number of sample points(!)
   long    GetOccupiedInputBufferSpace(void); // returns a number of sample points(!)

   long    GetTotalBufferSpace(void);        // returns a number of sample points(!)

   inline int GetNumberOfChannels(void )   { return m_iNrAudioChannels;    };
   inline int GetDecimationRatio(  void  ) { return m_iUpsamplingFactor;   };
   inline int GetFreqConversionFlag(void ) { return m_fFreqConversion;     };
   inline int GetComplexInputFlag( void )  { return m_fBufferIsComplex;    };
   inline T_Float GetInputSamplingRate(void)   { return m_dblInputSampleRate; };
   inline T_Float GetDecimatedSampleRate(void) { return m_dblInputSampleRate; };
         // For compatibility. Here: decimated sample rate == "input sample rate"

   long   GetInputComponentsPerSample(void);  // -> 1, 2 or 4 "components per sample" for INPUT         


   T_Float GetOutputSamplingRate(void);
   inline T_Float GetNcoFrequency(void)    { return m_dblNcoFrequency;     };
   BOOL    SetNcoFrequency(T_Float dblNcoFrequency);


   long    GetTotalSampleCountOut(void);   // returns a number of sample points(!)

   void    SetFastMode(int iFastMode); // 0=normal with proper filtering, 1=ugly sample duplication
   int     GetFastMode(void);

   long EnterRealSamples( T_Float *pdblSource, long lNrSamplePoints, int iNrChannels);
  /* Puts a number of audio samples into a buffer.
   * Purpose: called from the "application", filled with decimated
   * samples.
   * Sometimes called from a different THREAD than GetSamples() .
   * ONE 'Sample Point' may contain TWO channels !
   *
   * Source buffer structure:
   *   Mono input, real values:
   *        pdblSource[0..iNrSamplePoints-1]
   *   Stereo input, real values:
   *        pdblSource[0,2,4, ..2*(iNrSamplePoints-1)]   = left channel
   *        pdblSource[1,3,5, ..2*(iNrSamplePoints-1)+1] = right channel
   *
   * Return value: Count of sample points which could be placed
   *               in the buffer.
   */
   long EnterInt16Samples( SHORT *pi16Source, long lNrSamplePoints, int iNrSrcChannels);



   long    EnterSamples( T_Float *pdblSource, // for REAL OR(!) COMPLEX data
                      long lNrSamplePoints ); // number of SAMPLE POINTS(!)
       // Note: there may be 1,2,or even 4 components per sample point !


   long    GetRealSamples( long lNrSamplePoints,
                            int iNrChannels,
                           BOOL fDontGimmeLess,
                         T_Float *pdblDest  );

   inline int  GetComponentsPerSample( void ) { return m_iInputCompsPerSample; };


private:
   // A chain of decimators (with anti-aliasing lowpass filters) :
   T_SOUND_UPSAMPLER m_Upsampler[2][SoundUps_MAX_UPSAMPLING_STAGES];
                           //   [0]=LEFT audio channel, [1]=RIGHT audio channel
   T_Float  m_dblOutputSampleRate;
   T_Float  m_dblInputSampleRate;
   int      m_iUpsamplingFactor; // alias "decimation ratio" in SoundDec

   // Properties of a circular buffer where the INPUT samples are placed.
   long    m_lBufferSize;     // number of "float64" values, NOT SAMPLE POINTS
   T_Float *m_dblBuffer;
   int     m_iNrAudioChannels;
   int     m_iInputCompsPerSample;  // 1, 2, or 4 components; depends on channels+complex/real input
   int     m_iOutputCompsPerSample; // 1, 2, or 4 components; depends on channels+complex/real output
   long    m_lBufferIndexIn;     // buffer index of next COMPONENT to be entered ("head")
   long    m_lBufferIndexOut;    // buffer index of next COMPONENT to be taken out ("tail")
   bool    m_fBufferIsComplex;   // TRUE: the buffer uses complex values (= I/Q samples)
   bool    m_fFreqConversion;    // TRUE: frequency conversion enabled


   T_Float m_dblNcoFrequency;
   T_Float m_dblNcoPhase;

   // diagnostic stuff
   LONGLONG m_i64SampleCountOut; // counts the total amount of sample points READ FROM the buffer


}; // end class CSoundUpsamplingBuffer


#endif // SoundUpsH
