//---------------------------------------------------------------------------
// File  :  SoundUps.cpp ("Sound Upsampler")
// Date  :  2011-02-05  (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Implementation of a class for upsampling and buffering audio samples.
//
// Used in the following projects (at least) :
//    - Spectrum Lab         (c:\CBproj\SpecLab\SpecLab)
//    - Sound Output Utility (c:\CBproj\SoundUtl\SndOutpt)
//
//
// Revision history (YYYY-MM-DD):
//   2002-05-12  Written for the "Audio Output Tool" in c:\CBProj\SoundUtl\..
//   2004-12-21  Minor changes when used in SpecLab to read "AUD"-files
//   2011-02-05  Modified to support precise timestamp and sampling rate (info)
//---------------------------------------------------------------------------

#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#include <windows.h>
#include <math.h>

#pragma hdrstop   // no precompiled headers after this point

#include "utility1.h"  // uses UTL_NamedMalloc instead of malloc (for debugging)
#include "SoundUps.h"

// #pragma warn -8017
#pragma warn -8004  // .. is assigned a value that is never used (well..)
#pragma warn -8080  // .. is declared but never used (.. so what ?)



//**************************************************************************
//    Tables used in this module
//**************************************************************************



//**************************************************************************
//    Global Variables  (mostly for testing purposes)
//**************************************************************************

int SndUps_iFastAndUgly = 0;   // 0=crude duplication, 1=proper filtering


//**************************************************************************
//    Routines (no class methods)
//**************************************************************************


/***************************************************************************/
int  SndUps_InitUpsamplingChain(
      T_SOUND_UPSAMPLER *upsampler,
                    int nr_stages,
                    int upsampling_ratio
                         )
   /* Programs a chain (here:array) of 'upsampling' stages for the desired
    * upsampling ratio.
    * Note: not all ratios are possible, because every stage
    *       can multiply the input sample rate either by 1,2 or 3 !
    *       Sorry for the 'pure C' implementation of this...
    * The returned value is the ratio which could be REALIZED.
    */
{
  int i,j;
  int realized_ratio;
  long n2,n3, n2_ratio, n3_ratio, best_ratio,best_n2,best_n3;

  // 'default settings' for all stages:
  for(i=0;i<nr_stages;++i)
   {
    // Initialize all decimator stages, clear the circular delay line.
    for(j=0;j<SoundTab_DEC_FIR_LENGTH;++j)
     {
      upsampler[i].queue[j].re = 0.0;
      upsampler[i].queue[j].re = 0.0;  // also good for COMPLEX processing !
     }
    upsampler[i].ratio = 1;  // output/input ratio of this stage 
    upsampler[i].count = 0;  // no output samples available from this stage
    upsampler[i].inptr = upsampler[i].queue;
    upsampler[i].coeffs= SoundTab_Dec2FilterCoeffs;
   }


  // How can the desired output/input ratio be realized ?
  // Simply try all combinations of 2^(n2) *  3^(n3)
  //  and use the one which works best.
  best_ratio=1; best_n2=0; best_n3=0;
  n2_ratio = 1;
  for(n2=0; n2<=nr_stages; ++n2)
   {
     n3_ratio = 1;
     for(n3=0; n3+n2<=nr_stages; ++n3)
      { // Note: n2+n3 must never exceed 'nr_stages' !
        realized_ratio = n2_ratio * n3_ratio;
        if(   abs(realized_ratio - upsampling_ratio)
            < abs(best_ratio - upsampling_ratio) )
         { // found a better combination to realize the decimation ratio :
           best_ratio = realized_ratio;
           best_n2 = n2;
           best_n3 = n3;
         }
        n3_ratio *= 3;  // next power of three for next 'n3' loop
      } // end for (n3)
     n2_ratio *= 2;  // next power of two for next 'n2' loop
   } // end for (n2)
  // arrived here, we know how many 'mult 2 times' and 'mult 3 times' - stages
  // are required.
  // Example: looking for an output/input ratio of 100, realizable: 2*2 * 3*3*3 = 108
  //          -> best_n2 = 2,   best_n3 = 3
  // Setup the upsampler stages ...
  i = 0;
  while(i<best_n2)
        upsampler[i++].ratio = 2;  // from example: i=0,1
  while(i<best_n2+best_n3 && i<nr_stages)
        upsampler[i++].ratio = 3;  // from example: i=2,3,4
  while(i<nr_stages)
        upsampler[i++].ratio = 1;  // all the rest is 'unused'
  // arrived here: ratios for all upsampler stages are set.


  // Set the filter coefficients for all stages
  //  and calculate the resulting 'total decimation ratio':
  realized_ratio = 1;
  for(i=0;i<nr_stages;++i)
   {
    switch(upsampler[i].ratio)
     {
       case 2:
          upsampler[i].ratio = 2;
          upsampler[i].coeffs= SoundTab_Dec2FilterCoeffs;
          break;
       case 3:
          upsampler[i].ratio = 3;
          upsampler[i].coeffs= SoundTab_Dec3FilterCoeffs;
          break;
       default:
          upsampler[i].ratio = 1;
          upsampler[i].coeffs= SoundTab_Dec2FilterCoeffs;
          break;
     }
    realized_ratio *= upsampler[i].ratio;
   }
  return realized_ratio;

} // end SndUps_InitUpsamplingChain(..)


/***************************************************************************/
BOOL RunComplexUpsamplingStage(
      T_SOUND_UPSAMPLER *pUpsampler, // pointer to FIR coeffs and queue
             T_Complex cplxInputSample) // input sample to be processed
   // Caution: this routine must only be called if the output buffer
   //          is completely EMPTY (i.e. uUpsampler->count = 0) .
   // Every 'Run' produces 2 or 3 output sample pairs !
{

 // Use local copies for better speed...
 const T_Float *coeffs;               // pointer to a table of coefficients
 T_Complex *inptr = pUpsampler->inptr;  // input position in circular delay line
 T_Complex *qptr  = pUpsampler->queue;  // first element of circular delay line
 int     qlen  = SoundTab_DEC_FIR_LENGTH;
 int    iRatio = pUpsampler->ratio;

 if(iRatio>3)
    iRatio = 3; // '3' is the maximum output/input ratio we can handle
 if(iRatio <=1)
  {  // This stage looks like a "pass-through" stage. Not much to do:
     pUpsampler->cplxOutput[0] = cplxInputSample;
     pUpsampler->count = 1;
     return TRUE;
  }
 else
 for(int iUpsLoop=0; iUpsLoop<iRatio; ++iUpsLoop)
  {
   T_Complex  acc; // complex "accumulator" for output from digital filter

    if(SndUps_iFastAndUgly)  // TEST: how much CPU power consumed by the digital filter ? here NO filtering:
     {
      acc.re = cplxInputSample.re;  // pass real part from input to output (two or three times THE SAME sample)
      acc.im = cplxInputSample.im;  // pass imaginary part "   "   "
     }
    else   // normal operation with digital filtering:
     {
      inptr--;
      if(inptr < qptr)        // deal with FIR pointer wrap around
         inptr = qptr+qlen-1; // (2 indices per COMPLEX sample!)
      inptr->re = cplxInputSample.re;  // place real part in circular Queue (two or three times THE SAME sample)
      inptr->im = cplxInputSample.im;  // place imaginary part "   "   "

      // Run interpolating filter (an ordinary FIR lowpass, 1/2 or 1/3 band)
      //      Prepare complex MAC's for FIR lowpass
      coeffs = pUpsampler->coeffs;   // pointer to filter coeffs
      acc.re = 0.0;                // clear output accumulator
      acc.im = 0.0;
      T_Complex *firptr = inptr;
      for(int j=0; j<qlen; ++j )   // do the complex MAC's
       {
         acc.re += ( (firptr->re)*(*coeffs) );
         acc.im += ( (firptr->im)*(*coeffs++) );
         if( (++firptr) >= qptr+qlen ) //deal with wraparound
                firptr  =  qptr;
       }
     } // end if <normal operation with digital filtering>

    // filter output for the next stage now in acc . Save it in 'output' .
    // For ratio=2, the 1st sample to process is cplxOutput[1] (count=2) ;
    //              the 2nd sample to process is cplxOutput[0] (count=1) !
    pUpsampler->cplxOutput[ iRatio - 1 - iUpsLoop ] = acc;

  } // end for(iUpsLoop=0; ... )

 // save position in circular delay line
 pUpsampler->inptr = inptr;


 pUpsampler->count = iRatio;  // 'output[]' now completely filled with 2 or 3 pairs

 return TRUE;

} // end RunComplexUpsamplingStage( )

/***************************************************************************/
BOOL RunRealUpsamplingStage(
      T_SOUND_UPSAMPLER *pUpsampler, // pointer to FIR coeffs and queue
             T_Float fltInputSample) // input sample to be processed
   // Caution: this routine must only be called if the output buffer
   //          is completely EMPTY (i.e. uUpsampler->count = 0) .
   // Every 'Run' produces 2 or 3 output sample pairs !
{

 // Use local copies for better speed...
 const T_Float *coeffs;               // pointer to a table of coefficients
 T_Complex *inptr = pUpsampler->inptr;  // input position in circular delay line
 T_Complex *qptr  = pUpsampler->queue;  // first element of circular delay line
 int     qlen  = SoundTab_DEC_FIR_LENGTH;
 int    iRatio = pUpsampler->ratio;

 if(iRatio>3)
    iRatio = 3; // '3' is the maximum output/input ratio we can handle
 if(iRatio <=1)
  {  // This stage looks like a "pass-through" stage. Not much to do:
     pUpsampler->cplxOutput[0].re = fltInputSample;
     pUpsampler->count = 1;
     return TRUE;
  }
 else
 for(int iUpsLoop=0; iUpsLoop<iRatio; ++iUpsLoop)
  {
   T_Float  acc; // "accumulator" for output from digital filter

    if(SndUps_iFastAndUgly)  // TEST: how much CPU power consumed by the digital filter ? here NO filtering:
     {
      acc = fltInputSample;  // pass sample from input to output (two or three times THE SAME sample)
     }
    else   // normal operation with digital filtering:
     {
      inptr--;
      if(inptr < qptr)        // deal with FIR pointer wrap around
         inptr = qptr+qlen-1; // (2 indices per COMPLEX sample!)
      inptr->re = fltInputSample;  // place sample in circular Queue (two or three times THE SAME sample)

      // Run interpolating filter (an ordinary FIR lowpass, 1/2 or 1/3 band)
      //      Prepare complex MAC's for FIR lowpass
      coeffs = pUpsampler->coeffs;   // pointer to filter coeffs
      acc    = 0.0;                  // clear output accumulator
      T_Complex *firptr = inptr;
      for(int j=0; j<qlen; ++j )   // do the MAC's
       {
         acc += ( (firptr->re)*(*coeffs++) );
         if( (++firptr) >= qptr+qlen ) //deal with wraparound
                firptr  =  qptr;
       }
     } // end if <normal operation with digital filtering>

    // filter output for the next stage now in acc . Save it in 'output' .
    // For ratio=2, the 1st sample to process is cplxOutput[1] (count=2) ;
    //              the 2nd sample to process is cplxOutput[0] (count=1) !
    pUpsampler->cplxOutput[ iRatio - 1 - iUpsLoop ].re = acc;

  } // end for(iUpsLoop=0; ... )

 // save position in circular delay line
 pUpsampler->inptr = inptr;


 pUpsampler->count = iRatio;  // 'output[]' now completely filled with 2 or 3 pairs

 return TRUE;

} // end RunRealUpsamplingStage( )




//***************************************************************************
//  Implementation of methods for the CSoundUpsamplingBuffer   class
//***************************************************************************


//***************************************************************************
CSoundUpsamplingBuffer::CSoundUpsamplingBuffer()     // constructor
{
  m_dblInputSampleRate= m_dblOutputSampleRate = 0;
  m_dblNcoFrequency = 0;
  m_iUpsamplingFactor = 1;
  m_lBufferIndexIn = m_lBufferIndexOut = 0; // buffer is completely empty now
  m_fBufferIsComplex = FALSE;
  m_fFreqConversion  = FALSE;
  m_iInputCompsPerSample = m_iOutputCompsPerSample = 0;
  m_iNrAudioChannels  = 0;
  m_lBufferSize = 0;
  m_dblBuffer   = NULL;
  ChunkInfo_Init( &m_LatestChunkInfo ); // avoid garbage in the audio-chunk-info
} // end CSoundUpsamplingBuffer::CSoundUpsamplingBuffer()

void CSoundUpsamplingBuffer::Close(void) // clean up (for static instances)
{
  if(m_dblBuffer)
   { UTL_free(m_dblBuffer);  // fsssh
     m_dblBuffer = NULL;
   }
}

CSoundUpsamplingBuffer::~CSoundUpsamplingBuffer() // destructor, clean up
{
  Close();
} // end CSoundUpsamplingBuffer::~CSoundUpsamplingBuffer() [destructor]



//***************************************************************************
CSoundUpsamplingBuffer::Init(
       long lInputBufferCapacity, // max count of sample "points", "pairs", or "quads" (!)
     T_Float dblOutputSampleRate, // 'reading' sample rate, higher than input sample rate
       long lSampleRateMultiplier, // only powers of two are allowed
     T_Float fltNcoFrequency ,  // "L.O." frequency in Hertz
        int iComplexity,         // like COMPLEX_IN_REAL_OUT, etc (defined in SoundTab.h)
        int iNrChannels )        // 1=mono, 2=stereo processing
   // Prepares "upsampling" of formerly decimated audio samples.
   //       Also features a buffer where "writer" and "reader" may be different threads.
   // Note: Not all ratios are possible, because every decimation stage
   //       can multiply the input sampling rate either by 1,2 or 3 !
   //       m_iUpsamplingFactor  is the decimation factor
   //            (or the divisor for a simple integrate-and-dump filter)
   //       which COULD BE REALIZED ( n^2 * m^3,  n*m <= 12 )
   //
{
 long lNewBufferSize;

  ChunkInfo_Init( &m_LatestChunkInfo ); // avoid garbage in the audio-chunk-info
  m_lBufferSize  = 0;   // not open for business now (for multi-threaded appl.)

  // Delete the old contents of the 'big buffer'
  // (because with new sample rate, the old data are useless)
  m_lBufferIndexIn = m_lBufferIndexOut = 0; // buffer is completely empty now

  if(iNrChannels<1)
     iNrChannels=1;
  if(iNrChannels>2)
     iNrChannels=2;

  m_fFreqConversion = (fltNcoFrequency!=0);  // frequency conversion enabled ?
  m_dblNcoFrequency = fltNcoFrequency;
  m_iUpsamplingFactor = lSampleRateMultiplier;
  if(m_iUpsamplingFactor < 1)
     m_iUpsamplingFactor = 1;

  m_iUpsamplingFactor = SndUps_InitUpsamplingChain(
      m_Upsampler[0]/*left chn*/, SoundUps_MAX_UPSAMPLING_STAGES,
      m_iUpsamplingFactor );
  if( m_iUpsamplingFactor < 1 )
      m_iUpsamplingFactor = 1;
  SndUps_InitUpsamplingChain(m_Upsampler[1]/*right chn*/,
      SoundUps_MAX_UPSAMPLING_STAGES, m_iUpsamplingFactor );

  m_dblOutputSampleRate= dblOutputSampleRate;
  m_dblInputSampleRate = dblOutputSampleRate / (T_Float)m_iUpsamplingFactor;
  m_LatestChunkInfo.dblPrecSamplingRate = m_dblInputSampleRate;
  m_LatestChunkInfo.dblRadioFrequency   = fltNcoFrequency;

  // memorize the type of samples contained in the buffer.
  // Remember: UPSAMPLING takes place on READ, not on WRITING into the buffer
  switch(iComplexity)
   { case REAL_IN_REAL_OUT      :
          m_fBufferIsComplex = false;
          m_iInputCompsPerSample  = iNrChannels;
          m_iOutputCompsPerSample = iNrChannels;
          break;
     case REAL_IN_COMLEX_OUT    :
          m_fBufferIsComplex = false;
          m_iInputCompsPerSample  = iNrChannels;
          m_iOutputCompsPerSample = 2 * iNrChannels;
          break;
     case COMPLEX_IN_REAL_OUT   :
          m_fBufferIsComplex = true ;
          m_iInputCompsPerSample  = 2 * iNrChannels;
          m_iOutputCompsPerSample = iNrChannels;
          break;
     case COMPLEX_IN_COMLEX_OUT :
          m_fBufferIsComplex = true ;
          m_iInputCompsPerSample  = 2 * iNrChannels;
          m_iOutputCompsPerSample = 2 * iNrChannels;
          break;
     default:
          m_fBufferIsComplex = false;
          m_iInputCompsPerSample  = iNrChannels;
          m_iOutputCompsPerSample = iNrChannels;
          break;
   }
  m_LatestChunkInfo.nComponentsPerSamplePoint = m_iInputCompsPerSample;

  // Which buffer structure ?  1, 2, or 4 components per "sample pair" ,
  //  depends on channels+complex/real output..
  m_iNrAudioChannels = iNrChannels;


  // Allocate a ring-buffer where the DECIMATED samples are collected.
  // (lInputBufferCapacity+1) because ...
  //    In this type of circular buffer, ONE BUFFER LOCATION always remains empty:
  //    The buffer is COMPLETELY FULL, if
  //     ( (IndexIn + 1) MODULO SIZE ) == IndexOut .
  //    Buffer[IndexIn] is always an EMPTY LOCATION !
  // Note: we'll buffer the DECIMATED INPUT SAMPLES to save a lot of space .
  lNewBufferSize = (lInputBufferCapacity+1) * (long)m_iInputCompsPerSample;
  if( m_dblBuffer!=NULL )
   { // must free the old buffer and allocate a new because size may have changed
     UTL_free(m_dblBuffer);  // fsssh
   }
  m_dblBuffer = (T_Float*)UTL_NamedMalloc( "SoundUps", lNewBufferSize * sizeof(T_Float) );
  m_lBufferSize = lNewBufferSize;   // open for business now

  return (m_dblBuffer!=NULL);
} // end CSoundUpsamplingBuffer::Init()




/***************************************************************************/
//T_Float CSoundUpsamplingBuffer::GetNcoFrequency(void)
//{
//  return m_dblNcoFrequency;
//} // end GetNcoFrequency()
// "inline" since 2003-07


/***************************************************************************/
BOOL   CSoundUpsamplingBuffer::SetNcoFrequency(T_Float dblNcoFrequency)
{
  m_dblNcoFrequency = dblNcoFrequency;
  return TRUE;
} // end SetNcoFrequency()


/***************************************************************************/
T_Float CSoundUpsamplingBuffer::GetOutputSamplingRate(void)
{
  int i,j;
  T_Float d;

  d = m_dblInputSampleRate;
  for( i=0; i<SoundUps_MAX_UPSAMPLING_STAGES; ++i)
   {
     j = m_Upsampler[0][i].ratio;
     if (j>1)
       d /= (T_Float)j;
   }         // Looking at the 1st audio channel is enough,
  return d;  // lets assume both channels use the same decimation parameters !
} // end CSoundUpsamplingBuffer::GetOutputSamplingRate()



/***************************************************************************/
long CSoundUpsamplingBuffer::GetTotalBufferSpace(void)
  /* Returns the total count of SAMPLE PAIRS which can be placed in the buffer .
   */
{
  if(m_iInputCompsPerSample<=0)
     return 0; // no "zero-length" components are allowed in my pretty buffer ;-)
  // In this type of circular buffer, ONE BUFFER LOCATION always remains empty !
  return m_lBufferSize / m_iInputCompsPerSample - 1;
} // GetTotalBufferSpace()


/***************************************************************************/
long CSoundUpsamplingBuffer::GetOccupiedInputBufferSpace(void)
  /* Calculates the count of SAMPLE PAIRS still waiting in the INPUT buffer .
   *      Originally used for debugging and to detect how much buffer
   *      was really required (depends a bit on the CPU speed).
   */
{
 long l;

 if( (m_iInputCompsPerSample<=0) || (m_lBufferSize<=0) || (m_dblBuffer==NULL) )
    return 0;

 // How many SAMPLE PAIRS are presently in the buffer (for caller's index?)
 // NOTE: In SoundUps, 'm_iInputCompsPerSample' is used  (UPSAMPLE on READ)
 //       In SoundDec, 'm_iNrOutputComponents' is used (DECIMATE on ENTER)
 l = m_lBufferIndexIn - m_lBufferIndexOut;
 if (l<0) l += m_lBufferSize;
 return l / m_iInputCompsPerSample;
} // end CSoundUpsamplingBuffer::GetOccupiedInputBufferSpace()


/***************************************************************************/
long CSoundUpsamplingBuffer::GetFreeInputBufferSpace() // returns a number of sample points(!)
  // ...which may be entered in the (input-)buffer before an overflow would occurr.
  // It's important to distinguish between "number of sample points" (used here)
  // and the "buffer size" !   A single "sample point" may contain the values
  // for more than one channel, sampled at the same time .
{
 long l;

 if( (m_iInputCompsPerSample<=0) || (m_lBufferSize<=0) || (m_dblBuffer==NULL) )
    return 0;

 // How many SAMPLE PAIRS are presently in the buffer (for caller's index?)
 // NOTE: In SoundUps, 'm_iInputCompsPerSample' is used (UPSAMPLE on READ)
 //       In SoundDec, 'm_iNrOutputComponents' is used (DECIMATE on ENTER)
 // + In this type of circular buffer, ONE BUFFER LOCATION always remains empty !
 // example: m_lBufferIndexIn=1, m_lBufferIndexOut=0, m_lBufferSize=4
 //          ->
 l = m_lBufferIndexOut - m_lBufferIndexIn - 1;
 if (l<0) l += m_lBufferSize;
 return l / m_iInputCompsPerSample;
} // end CSoundUpsamplingBuffer::GetFreeInputBufferSpace()


/***************************************************************************/
long CSoundUpsamplingBuffer::GetInputComponentsPerSample(void)
  // Return values :
  //   REAL-valued input,  mono  :  1
  //   REAL-valued input,  stereo:  2
  //   COMPLEX input,      mono  :  2
  //   COMPLEX input,      stereo:  4
{
 return m_iInputCompsPerSample;
} // end CSoundUpsamplingBuffer::GetInputComponentsPerSample()



/***************************************************************************/
long CSoundUpsamplingBuffer::EnterRealSamples(
                  T_Float *pdblSource,
                    long lNrSamplePoints,
                     int iNrChannels,
               T_ChunkInfo *pChunkInfo ) // [in] see c:\cbproj\SoundUtl\ChunkInfo.h.
                                         //  Optional, may be NULL when 'not applicable' !!
  /* Puts a number of audio samples into a buffer.
   * Purpose: called from the "application", filled with decimated
   * samples.
   * Sometimes called from a different THREAD than GetSamples() .
   * ONE 'Sample Point' may contain TWO channels !
   *
   * Source buffer structure:
   *   Mono input, real values   (iNrChannels=1) :
   *        pdblSource[0..iNrSamplePoints-1]
   *   Stereo input, real values (iNrChannels=2) :
   *        pdblSource[0,2,4, ..2*(iNrSamplePoints-1)]   = left channel
   *        pdblSource[1,3,5, ..2*(iNrSamplePoints-1)+1] = right channel
   *
   * Return value: Count of sample points which could be placed
   *               in the buffer.
   */
{
 long l;

  if( pChunkInfo != NULL )
   { m_LatestChunkInfo = *pChunkInfo;
     // Note: Some members in T_ChunkInfo will be overwritten later,
     //       in GetRealSamples() !
   }

  if( (lNrSamplePoints<=0) || (m_iInputCompsPerSample<=0) )
     return 0;


 if(m_fBufferIsComplex)   // entering REAL-VALUED samples into a COMPLEX buffer ?
  {
    l = lNrSamplePoints * m_iNrAudioChannels;
    while(l--)     // put REAL values into a COMPLEX buffer:
     {
       if(m_lBufferIndexIn >= (m_lBufferSize-1) )
          m_lBufferIndexIn = 0;  // wrap around (without splitting RE + IM)
       m_dblBuffer[m_lBufferIndexIn++] = *pdblSource++;
       m_dblBuffer[m_lBufferIndexIn++] = 0;     // set imaginary part to zero
     }
  }
 else // !  (m_fBufferIsComplex) :
  {  // entering REAL-VALUED samples into a REAL-VALUED buffer:
    l = lNrSamplePoints * m_iNrAudioChannels;
    while(l--)
     {
       if(m_lBufferIndexIn >= m_lBufferSize)
          m_lBufferIndexIn = 0;  // wrap around
       m_dblBuffer[m_lBufferIndexIn++] = *pdblSource++;
     }
  } // end else < ! m_fBufferIsComplex >

 if(m_lBufferIndexIn >= m_lBufferSize)
    m_lBufferIndexIn = 0;  // one more wrap to keep things tidy

 return lNrSamplePoints;
} // end CSoundUpsamplingBuffer::EnterRealSamples()

/***************************************************************************/
long CSoundUpsamplingBuffer::EnterInt16Samples(
                   SHORT *pi16Source,
                    long lNrSamplePoints,
                     int iNrSrcChannels,
               T_ChunkInfo *pChunkInfo ) // [in] see c:\cbproj\SoundUtl\ChunkInfo.h.
                                         //  Optional, may be NULL when 'not applicable' !!
   /* Puts a number of audio samples into a buffer.
   * Purpose: called from the "application", filled with decimated
   * samples.
   * Sometimes called from a different THREAD than GetSamples() .
   * ONE 'Sample Point' may contain TWO channels !
   *
   * Source buffer structure:
   *   Mono input, real values   (iNrChannels=1) :
   *        pi16Source[0..iNrSamplePoints-1]
   *   Stereo input, real values (iNrChannels=2) :
   *        pi16Source[0,2,4, ..2*(iNrSamplePoints-1)]   = left channel
   *        pi16Source[1,3,5, ..2*(iNrSamplePoints-1)+1] = right channel
   *
   * Return value: Count of sample points which could be placed
   *               in the buffer.
   */
{
 long i32;
 int  iDestChannel;

  if( (lNrSamplePoints<=0) || (m_iInputCompsPerSample<=0) )
    return 0;
  if( iNrSrcChannels > m_iNrAudioChannels )
    return 0;  // bug !
  m_i64InputSampleCounter += lNrSamplePoints;
  if( pChunkInfo != NULL ) // Note: the INPUT chunk-info is optional (it may be NULL) ....
   { m_LatestChunkInfo = *pChunkInfo;
   }
  else // caller didn't pass a T_ChunkInfo, so build our own:
   {
     m_LatestChunkInfo.i64TotalSampleCounter = m_i64InputSampleCounter;
     // not here: m_LatestChunkInfo->dblPrecSamplingRate = m_dblInputSampleRate;
     if( m_dblInputSampleRate > 0.0 )
      { double dblDeltaT = (double)lNrSamplePoints / m_dblInputSampleRate;
      }
   }

 if(m_fBufferIsComplex)   // entering REAL-VALUED samples into a COMPLEX buffer ?
  {
    i32 = lNrSamplePoints * m_iNrAudioChannels;
    while(i32--)     // put REAL values into a COMPLEX buffer:
     {
       if(m_lBufferIndexIn >= (m_lBufferSize-1) )
          m_lBufferIndexIn = 0;  // wrap around (without splitting RE + IM)
       m_dblBuffer[m_lBufferIndexIn++] = *pi16Source++;
       m_dblBuffer[m_lBufferIndexIn++] = 0;     // set imaginary part to zero
     }
  }
 else // !  (m_fBufferIsComplex) :
  {  // entering REAL-VALUED samples into a REAL-VALUED buffer:
    if( iNrSrcChannels == m_iNrAudioChannels )
     {
       i32 = lNrSamplePoints * m_iNrAudioChannels;
       while(i32--)
        {
          if(m_lBufferIndexIn >= m_lBufferSize)
             m_lBufferIndexIn = 0;  // wrap around
          m_dblBuffer[m_lBufferIndexIn++] = *pi16Source++;
        }
     }
    else // the number of "source channels" is different from the buffer's number of channels :
     { // use a slower, but universal copying loop in this case:
       for(i32=0; i32<lNrSamplePoints; ++i32)
        { for(iDestChannel=0; iDestChannel<m_iNrAudioChannels && iDestChannel<iNrSrcChannels;
            ++iDestChannel )
           { if(m_lBufferIndexIn >= m_lBufferSize)
                m_lBufferIndexIn = 0;  // wrap around
             m_dblBuffer[m_lBufferIndexIn++] = pi16Source[iDestChannel];
           }
          while(iDestChannel<m_iNrAudioChannels )  // fill all other destination channels with SILENCE(!)
           { if(m_lBufferIndexIn >= m_lBufferSize)
                m_lBufferIndexIn = 0;  // wrap around
             m_dblBuffer[m_lBufferIndexIn++] = 0;
             ++iDestChannel;
           }
         pi16Source += iNrSrcChannels;  // advance source pointer to next SAMPLE POINT (with n channels)
        }
     }
  } // end else < ! m_fBufferIsComplex >

 if(m_lBufferIndexIn >= m_lBufferSize)
    m_lBufferIndexIn = 0;  // one more wrap to keep things tidy

 return lNrSamplePoints;
} // end CSoundUpsamplingBuffer::EnterInt16Samples()


/***************************************************************************/
long CSoundUpsamplingBuffer::EnterSamples(
        T_Float *pdblSource,    // [in] block of audio samples, floating point
        long lNrSamplePoints,   // [in] number of samples (or sample pairs)
        int  iNrCompsPerSample, // [in] number of channels, multiplied by one (real) or two (complex)
        T_ChunkInfo *pChunkInfo ) // [in] see c:\cbproj\SoundUtl\ChunkInfo.h.
                                  //  Optional, may be NULL when 'not applicable' !!
  /* Puts a number of audio samples (either REAL or COMPLEX) into a buffer.
   * Purpose: called from the "application", filled with decimated
   * samples.
   * Sometimes called from a different THREAD than GetSamples() .
   * ONE 'Sample Point' may contain TWO channels !
   *
   * Source buffer structure (must match the params from ..Init() ):
   *   Mono input, complex values:
   *        pdblSource[0]=real part, pdblSource[1]=imaginary part,
   *      ..pdblSource[2*lNrSamplePoints-2]=real part of last sample,
   *        pdblSource[2*lNrSamplePoints-1]=imaginary part of last sample.
   *   Stereo input, complex values:
   *        pdblSource[0]=real part, pdblSource[1]=imaginary part, LEFT channel
   *        pdblSource[2]=real part, pdblSource[3]=imaginary part, RIGHT channel
   *      ..pdblSource[4*lNrSamplePoints-1]=imaginary of last sample, last channel.
   *
   * Return value: Count of sample points which could be placed
   *               in the buffer.
   */
{
 long i32,lBufIndex;
 long i32DstNumFloats;
 long i32SrcNumFloats;
 int iDestChannel;

  if( pChunkInfo != NULL )
   { m_LatestChunkInfo = *pChunkInfo;
   }

  if( (lNrSamplePoints<=0) || (m_iInputCompsPerSample<=0) )
    return 0;

  // how many single floating point values ?
  i32SrcNumFloats = lNrSamplePoints * iNrCompsPerSample;
  i32DstNumFloats = lNrSamplePoints * m_iNrAudioChannels * m_iInputCompsPerSample;

  if( i32SrcNumFloats == i32DstNumFloats )
   {
     for(i32=0; i32<i32SrcNumFloats; ++i32)
      {
       if(m_lBufferIndexIn>=m_lBufferSize)
          m_lBufferIndexIn = 0;    // wrap around
       m_dblBuffer[m_lBufferIndexIn++] = *pdblSource++;
      }
   }
  else // the number of "source channels" is different from the buffer's number of channels :
   { // use a slower, but universal copying loop in this case:
     for(i32=0; i32<lNrSamplePoints; ++i32)
      { for(iDestChannel=0; iDestChannel<m_iNrAudioChannels && iDestChannel<iNrCompsPerSample;
          ++iDestChannel )
         { if(m_lBufferIndexIn >= m_lBufferSize)
              m_lBufferIndexIn = 0;  // wrap around
           m_dblBuffer[m_lBufferIndexIn++] = pdblSource[iDestChannel];
         }
        while(iDestChannel<iNrCompsPerSample )  // fill all other destination channels with SILENCE(!)
         { if(m_lBufferIndexIn >= m_lBufferSize)
              m_lBufferIndexIn = 0;  // wrap around
           m_dblBuffer[m_lBufferIndexIn++] = 0.0;
           ++iDestChannel;
         }
       pdblSource += iNrCompsPerSample; // advance source pointer to next SAMPLE POINT (with n channels or components)
      }
   }

 if(m_lBufferIndexIn >= m_lBufferSize)
    m_lBufferIndexIn = 0;  // one more wrap to keep things tidy

 return lNrSamplePoints;
} // end CSoundUpsamplingBuffer::EnterSamples()


/***************************************************************************/
long CSoundUpsamplingBuffer::GetNumberOfSamplesInInputBuffer(void)
{
  long i32SamplesInInputBuffer;

  // How many SAMPLE PAIRS are waiting in the output-buffer at the moment ?
  i32SamplesInInputBuffer = m_lBufferIndexIn - m_lBufferIndexOut;
  if (i32SamplesInInputBuffer < 0)
      i32SamplesInInputBuffer += m_lBufferSize;
  i32SamplesInInputBuffer = i32SamplesInInputBuffer * m_iUpsamplingFactor;
  if( m_iInputCompsPerSample > 1 )
      i32SamplesInInputBuffer /= m_iInputCompsPerSample;
  if( i32SamplesInInputBuffer > m_lBufferSize )
      i32SamplesInInputBuffer = m_lBufferSize;
  return i32SamplesInInputBuffer;
} // end CSoundUpsamplingBuffer::GetNumberOfSamplesInInputBuffer()


/***************************************************************************/
long CSoundUpsamplingBuffer::GetRealSamples(
         long lNrSamplePoints, // count of output samples (maybe pairs or quads)
          int iNrChannels,
         BOOL fDontGimmeLess,  // flag if caller can accept LESS SAMPLES than required or not
       T_Float *pdblDest,
       T_ChunkInfo *pOutChunkInfo) // [out] optional, since 2011-02
  /* Copies some audio samples from an internal buffer into the caller's buffer.
   * The maximum number of samples copied this way should be  less or equal the
   * the "internal" buffer size (which has been defined upon creation).
   *
   * Parameters:
   *   lNrSamplePoints: Count of samples(!maybe PAIRS or QUADS!) the caller wants to get
   *                    from the buffer as A MAXIMUM COUNT
   *      (The caller must be able to handle less than this size,
   *       otherwise use GetOccupiedBufferSpace() before calling this routine).
   *   pdblDest[0...lNrSamplePoints]: Pointer to destination buffer for REAL samples
   *   Return value:  <0 : error, something wrong in initialization etc
   *                   0 : no samples available
   *                  >0 : success, count of SAMPLE POINTS placed in destination.
   * Notes:
   *  -  If the buffer holds COMPLEX values (I/Q samples),
   *     2 * <iNrSamplePoints> 64-bit floating point values are copied !
   *     Buffer format for complex samples:
   *             dst[2*k]   = real part of complex value[k]
   *             dst[2*k+1] = imaginary part,  with 0 <= k < iNrSamplePoints .
   *  -  If the buffer is configured for STEREO MODE, with complex (I/Q-)output,
   *     4 * <iNrSamplePoints> 64-bit floating point values are copied !
   */
{
 long l;
 LONGLONG i64;
 static T_Float dblVcoPhase=0;
 T_Float phzinc;
 long lBufferSizeAsSamplePairs;
 long i32SamplesInInputBuffer = GetNumberOfSamplesInInputBuffer();
 int  iDestSampleIndex = 0;

 // Are we able to deliver any samples ?
 if(   (lNrSamplePoints<=0) || (m_iInputCompsPerSample<=0)
     ||(i32SamplesInInputBuffer<=0) || (m_dblBuffer==NULL) )
    return 0;

 if(fDontGimmeLess)
  {
     if(i32SamplesInInputBuffer < lNrSamplePoints)
       return 0;  // sorry, not enough samples available !
  }
 else // "may return less samples than the caller wants (ideally)" :
  {   //  (because some callers can deal with any number of samples,
      //   others, like the FFT, cannot) ...
    if(  lNrSamplePoints/*wanted*/ > i32SamplesInInputBuffer/*available*/ )
         lNrSamplePoints = i32SamplesInInputBuffer;
  }

 if( pOutChunkInfo != NULL )
  {
    // ex: GetCurrentOutputTimestamps( // get the timestamp for the 1st sample in the returned block..
    //      &pOutChunkInfo->dblUnixDateAndTime, // [out] current date+timestamp in UNIX format
    //      &pOutChunkInfo->dblSecondsOfDay );  // [out] 'SECONDS' of the current day, for more resolution.
    // Since 2011-02, ChunkInfo.c::ChunkInfo_AdjustInfoForSampleIndexDiff()
    //      will prepare the 'output' chunk info :
    double dblSampleIndexDiff = m_i64SampleCountOut - m_i64SampleCountOutOfLastTimestampSync;
    ChunkInfo_AdjustInfoForSampleIndexDiff(
       &m_LatestChunkInfo, // [in] 'stored' chunk info
       dblSampleIndexDiff, // [in] sample index difference
                           // between the index requested by the caller
                           //        (for the first sample he wants to retrieve)
                           //   minus the older *stored* chunk-info-block .
       pOutChunkInfo);     // [out] 'adjusted' chunk info (especially the timestamp)
     // The returned chunk-info shall contain the number of returned samples now,
     // not the number of samples which has been written into the buffer. So :
     pOutChunkInfo->dwNrOfSamplePoints = (DWORD)lNrSamplePoints;
  } // end if( pOutChunkInfo != NULL )

 if( ! m_fBufferIsComplex )
   { // No COMPLEX, but a REAL-VALUE input AND OUTPUT (means: no I/Q-mixing):
     // If required, do the "upampling" here to save buffer space.
    if( m_iUpsamplingFactor <= 1)
     { // NO upsampling to a higher sample rate for the DAC...
      l = lNrSamplePoints * m_iNrAudioChannels;
      while(l--)
        { // Copy the next sample from the input chunk to the buffer
          if (m_lBufferIndexOut >= m_lBufferSize)
              m_lBufferIndexOut = 0;       // buffer wrap
          *pdblDest++ = m_dblBuffer[m_lBufferIndexOut++];
        } // end for(i..)
      return lNrSamplePoints;
     }
    else // Do some UPSAMPLING without frequency conversion when copying the chunk into the buffer:
     {
      T_Float flt;
      l = lNrSamplePoints;
      while(l--)
       {
        // Run the samples through some upsampling stages
        //     with decent low-pass filtering to avoid aliasing .
        // Principle:
        //     Beginning with the input stage,
        //     feed every upsampling stage with empty(*) output buffers
        //     with ONE sample from a previous stage.
        //        [ (*)only stages with empty output buffers can be fed ]
        //     After one loop through all stages, we can be SURE
        //     that there will be at least one sample available
        //     at the output.
        //  The first stage is something special, because it receives
        //     the "input" from the buffer (and not from an upsampling stage):
        if( (m_Upsampler[0][0].count==0) && (i32SamplesInInputBuffer>0) )
         { // the first upsampling stage (stage=0) has an EMPTY output buffer:
           --i32SamplesInInputBuffer;
           for(int i=0; i<m_iNrAudioChannels; ++i)
            {
             flt = m_dblBuffer[m_lBufferIndexOut++];
             if (m_lBufferIndexOut >= m_lBufferSize)
                 m_lBufferIndexOut = 0;       // wrap circular input buffer
             RunRealUpsamplingStage( &m_Upsampler[i][0], flt );
             // now m_Upsampler[i][0].count must be > 0 !
            }
         } // end if <first upsampling stage has an empty output buffer>
        int iOutputStage = 0;
        for(int stage=1; (stage<SoundUps_MAX_UPSAMPLING_STAGES); ++stage )
         {
          if(m_Upsampler[0][stage].ratio > 1)
           {
            iOutputStage = stage;  // after the stage-loop we'll know where the output comes from
            int iPrevCount = m_Upsampler[0][stage-1].count;
            if( (m_Upsampler[0][stage].count==0) && (iPrevCount>0) )
              { // this upsampling stage has empty output, the previous NOT...
                // Take ONE input sample, lower input rate, from the previous stage
                //      and put it into the stage we are looking at right now.
                --iPrevCount; // took one sample out of here
                for(int i=0; i<m_iNrAudioChannels; ++i)
                 {
                  RunRealUpsamplingStage( &m_Upsampler[i][stage],
                     m_Upsampler[i][stage-1].cplxOutput[iPrevCount].re );
                  m_Upsampler[i][stage-1].count = iPrevCount;
                 }
              } // end if (fFilterOutputReady && ..Ratio.. > 1) )
           }
          else  // .ratio <= 1 , means all 'active' stages done
           {
             break;
           }
         } // end for <all active upsampling stages>
        if (m_Upsampler[0][iOutputStage].count>0)
         { // upsampled 'real' sample ready ...
           pdblDest[iDestSampleIndex++]
               = m_Upsampler[0][iOutputStage].cplxOutput  // get output from upsampling chain, last active stage
                  [--m_Upsampler[0][iOutputStage].count].re;

           if(m_iNrAudioChannels>1)
            { // The same also for the SECOND audio channel ("right").
              // Here, use a constant index for m_Upsampler[i] to save a few microseconds
              pdblDest[iDestSampleIndex++]
                   = m_Upsampler[1][iOutputStage].cplxOutput
                     [--m_Upsampler[1][iOutputStage].count].re;
            }

         } // end if <output from the last stage of the upsampling chain ready>
        else
         { // No output from the last stage of the chain ?!
           // Must have ran out of input samples, or the chain screwed up :o)
           break;
         }
       } // end while ( .. more REAL samples .. )
     } // end if <put sampled data into the ring buffer WITH DOWNSAMPLING>
   } // end if (!complex_input)
  else  // buffered samples COMPLEX ("I/Q"), output samples COMPLEX or REAL  ..
   {
     // Principle: Complex upsamling (both I+Q),
     //            then multiply the COMPLEX samples with the COMPLEX output
     //            of a 'numerical controlled oscillator',
     //            then convert to real values by (??)adding(??) I+Q .
     double dblNcoI, dblNcoQ;
     int    iCosTableIndex, iSineTableOffset;
     double dblPhzInc;
     SoundTab_GetNcoParams( // prepare the "numeric controlled oscillator" :
        m_dblNcoFrequency/m_dblOutputSampleRate, // input : NCO frequency divided by sampling rate
        &iSineTableOffset,  // output: offset from cosine- to sine table index
        &dblPhzInc );       // output: phase increment value (floating point!!)
     if(m_dblNcoPhase<0)
        m_dblNcoPhase=0;

     lNrSamplePoints *= m_iOutputCompsPerSample;  // -> total count of delivered 'components'
     while(iDestSampleIndex<lNrSamplePoints)
      {
        // Run the I/Q samples through some upsampling stages
        //     with decent low-pass filtering to avoid aliasing .
        // Principle:
        //     Beginning with the input stage,
        //     feed every upsampling stage with empty(*) output buffers
        //     with ONE sample from a previous stage.
        //        [ (*)only stages with empty output buffers can be fed ]
        //     After one loop through all stages, we can be SURE
        //     that there will be at least one sample available
        //     at the output.
        //  The first stage is something special, because it receives
        //     the "input" from the buffer (and not from an upsampling stage):
        T_Complex  cplx;
        if( (m_Upsampler[0][0].count==0) && (i32SamplesInInputBuffer>0) )
         { // the first upsampling stage (stage=0) has an EMPTY output buffer:
           --i32SamplesInInputBuffer;
           for(int i=0; i<m_iNrAudioChannels; ++i)
            {
             cplx.re = m_dblBuffer[m_lBufferIndexOut++];
             cplx.im = m_dblBuffer[m_lBufferIndexOut++];
             if (m_lBufferIndexOut >= m_lBufferSize)
                 m_lBufferIndexOut = 0;       // wrap circular input buffer
             RunComplexUpsamplingStage( &m_Upsampler[i][0], cplx );
             // now m_Upsampler[i][0].count must be > 0 !
            }
         } // end if <first upsampling stage has an empty output buffer>
        int iOutputStage = 0;
        for(int stage=1; (stage<SoundUps_MAX_UPSAMPLING_STAGES); ++stage )
         {
          if(m_Upsampler[0][stage].ratio > 1)
           {
            iOutputStage = stage;  // after the stage-loop we'll know where the output comes from
            int iPrevCount = m_Upsampler[0][stage-1].count;
            if( (m_Upsampler[0][stage].count==0) && (iPrevCount>0) )
              { // this upsampling stage has empty output, the previous NOT...
                // Take ONE input sample, lower input rate, from the previous stage
                //      and put it into the stage we are looking at right now.
                --iPrevCount; // took one sample out of here
                for(int i=0; i<m_iNrAudioChannels; ++i)
                 {
                  RunComplexUpsamplingStage( &m_Upsampler[i][stage],
                     m_Upsampler[i][stage-1].cplxOutput[iPrevCount]);
                  m_Upsampler[i][stage-1].count = iPrevCount;
                 }
              } // end if (fFilterOutputReady && ..Ratio.. > 1) )
           }
          else  // .ratio <= 1 , means all 'active' stages done
           {
             break;
           }
         } // end for <all active upsampling stages>
        if (m_Upsampler[0][iOutputStage].count>0)
         { // upsampled complex sample ready ...

           // Let the NCO (numerical controlled oscillator) produce quadrature-phase signals :
           // Increment the NCO phase, and calculate an index into the cosine table:
           m_dblNcoPhase += dblPhzInc;
           while(m_dblNcoPhase >=(T_Float)SOUND_COS_TABLE_LEN) // "while", not "if" !!
                 m_dblNcoPhase -=(T_Float)SOUND_COS_TABLE_LEN; // table index wrap
           iCosTableIndex = (int)m_dblNcoPhase;
           dblNcoI = SoundTab_fltCosTable[iCosTableIndex];
           dblNcoQ = SoundTab_fltCosTable[(iCosTableIndex+iSineTableOffset) % SOUND_COS_TABLE_LEN];


           cplx = m_Upsampler[0][iOutputStage].cplxOutput  // get output from upsampling chain, last active stage
                  [--m_Upsampler[0][iOutputStage].count];
           // Turn the COMPLEX into a REAL-valued sample ?
           // Multiply the COMPLEX sample (from the buffer) with the NCO output,
           // and add both I- and Q- channels for "REAL-valued" output.
#if(0) // TEST .. to check the NCO's spectral purity
           pdblDest[iDestSampleIndex++] = 8192*dblNcoI  +  8192*dblNcoQ ;
#else  // normal operation:
           pdblDest[iDestSampleIndex++] = cplx.re * dblNcoI  +  cplx.im * dblNcoQ ;
#endif

           if(m_iNrAudioChannels>1)
            { // The same also for the SECOND audio channel ("right").
              // Here, use a constant index for m_Upsampler[i] to save a few microseconds
              cplx = m_Upsampler[1][iOutputStage].cplxOutput
                     [--m_Upsampler[1][iOutputStage].count];
#if(0) // TEST .. to check the NCO's spectral purity
              pdblDest[iDestSampleIndex++] = 8192*dblNcoI  +  8192*dblNcoQ ;
#else  // normal operation:
              pdblDest[iDestSampleIndex++] = cplx.re * dblNcoI  +  cplx.im * dblNcoQ ;
#endif
            }

         } // end if <output from the last stage of the upsampling chain ready>
        else
         { // No output from the last stage of the chain ?!
           // Must have ran out of input samples, or the chain screwed up :o)
           break;
         }
      } // end while ( .. more samples .. )
   } // end else (complex_input)

 return iDestSampleIndex;   // returns the count of delivered samples (not single comp's)
} // end   CSoundUpsamplingBuffer::GetSamples()



/***************************************************************************/
long CSoundUpsamplingBuffer::GetTotalSampleCountOut(void)
    // returns a number of sample points(!)
{
  return (long)m_i64SampleCountOut;
} // end CSoundUpsamplingBuffer::GetTotalSampleCountOut()

/***************************************************************************/
void CSoundUpsamplingBuffer::SetCurrentInputTimestamps(
        double dblUnixDateAndTime, // [in] current date+timestamp in UNIX format
        double dblSecondsOfDay )   // [in] 'SECONDS' of the current day, more resolution.
  // Sets the "time of recording" (absolute) for the next audio sample
  //   which will be retrieved from the buffer .
  // Call this shortly ***before*** entering new INPUT samples .
{
  long i32SamplesInInputBuffer, i32SamplesInInputBuffer2;
  double dblTimestampOffsetInSeconds;

  do   // may have to repeat this when "interrupted" by another thread !
   {   // ( which is very unlikely to happen, but... )
     i32SamplesInInputBuffer = GetNumberOfSamplesInInputBuffer();
     dblTimestampOffsetInSeconds = (double)i32SamplesInInputBuffer * m_dblInputSampleRate;
     // Note: the above doesn't take the additional delay from the FIR filters
     //       into account.  May have to modify this one day !
     m_dblLastTimestamp_UnixDateAndTime = dblUnixDateAndTime + dblTimestampOffsetInSeconds;
     m_dblLastTimestamp_dblSecondsOfDay = dblSecondsOfDay + dblTimestampOffsetInSeconds;
     m_i64SampleCountOutOfLastTimestampSync = m_i64SampleCountOut;
     i32SamplesInInputBuffer2 = GetNumberOfSamplesInInputBuffer();
   }while(i32SamplesInInputBuffer!=i32SamplesInInputBuffer2);

} // end CSoundUpsamplingBuffer::SetCurrentInputTimestamps()


/***************************************************************************/
void CSoundUpsamplingBuffer::GetCurrentOutputTimestamps(
        double *pdblUnixDateAndTime, // [out] current date+timestamp in UNIX format
        double *pdblSecondsOfDay )   // [out] 'SECONDS' of the current day, more resolution.
  // Returns the time of recording   for the next audio sample
  //   which will be retrieved from the buffer (related to output) .
  // Call this function shortly ***before*** reading output samples .
{
  double dblTimestampOffsetInSeconds;
  LONGLONG i64OutputSamplesSinceLastTimestampSync
     = m_i64SampleCountOut - m_i64SampleCountOutOfLastTimestampSync;
  if( m_dblOutputSampleRate > 0 )
     dblTimestampOffsetInSeconds = (double)i64OutputSamplesSinceLastTimestampSync / m_dblOutputSampleRate;
  else // oops !
     dblTimestampOffsetInSeconds = 0.0;
  if( pdblUnixDateAndTime )
   { *pdblUnixDateAndTime = m_dblLastTimestamp_UnixDateAndTime + dblTimestampOffsetInSeconds;
   }
  if( pdblSecondsOfDay )  // similar as above, but with more resolution..
   { *pdblSecondsOfDay = m_dblLastTimestamp_dblSecondsOfDay + dblTimestampOffsetInSeconds;
   }
} // end CSoundUpsamplingBuffer::GetCurrentOutputTimestamps()

/***************************************************************************/
void CSoundUpsamplingBuffer::SetFastMode(int iFastMode)
{
  SndUps_iFastAndUgly = iFastMode;
} // end CSoundUpsamplingBuffer::SetFastMode()

/***************************************************************************/
int CSoundUpsamplingBuffer::GetFastMode(void)
{
  return SndUps_iFastAndUgly;
} // end CSoundUpsamplingBuffer::GetFastMode()
