/***************************************************************************/
/* SoundThd.cpp =  Sound Input Thread.                                     */
/*   - Receives continuous input from the soundcard                        */
/*       and places the sound samples in two alternating buffers.          */
/*   - requires "SOUND.CPP" ( CSound class )  to access the audio device   */
/*   - by DL4YHF,  September 2000  -  December 2001                        */
/*   - VCL dependencies removed in May 2002 to keep the code compact (!)   */
/***************************************************************************/

//---------------------------------------------------------------------------
#include <windows.h>   // dont use the VCL here !!
#include <stdio.h>
#include <stdlib.h>

#pragma hdrstop


#define _I_AM_SOUND_THREAD_ 1  // for "single-source" variables of this module
#include "SoundThd.h"

#pragma warn -8071

/*-------------- constants, tables ---------------------------------------*/
#define SNDTHD_EXIT_OK  1
#define SNDTHD_EXIT_BAD 2

/*-------------- initialized(!) Variables --------------------------------*/

double SOUND_dblAudioPeakValue = 0; // crude peak detector for display purposes

// A large "decimating" buffer where the samples from the soundcard are stored
CSoundDecimatingBuffer DecimatingAudioBuffer;



/*------------- internal(!) variables ------------------------------------*/
BOOL   SndThd_fTerminateRequest;
HANDLE SndThd_hWorkerThread;    // Win32 API handle for the worker thread
DWORD  SndThd_dwWorkerThreadId; // Identifier for the worker thread

//------------- Implementation of routines and classes ----------------------

//---------------------------------------------------------------------------
void SOUND_SetDefaultParameters(void)
  // Prepare some soundcard settings (WAVEFORMATEX) required later.
  // Note: some of these settings may be overridden by the command line parser.
{
  // Preset some soundcard settings (WAVEFORMATEX) required for opening...
  SOUND_WFXSettings.nChannels = 2;
  SOUND_WFXSettings.wBitsPerSample = 16;  // ALWAYS 16 BIT, EVEN WHEN EXPORTING 8-BIT-VALUES(!!)
  SOUND_WFXSettings.nSamplesPerSec = 11025;

  // Prepare some soundcard settings (WAVEFORMATEX) required for opening.
  // Note: The commented settings have already been set in SetDefaultParameters():
  // SOUND_WFXSettings.nChannels = 2;
  // SOUND_WFXSettings.wBitsPerSample = 16;
  // SOUND_WFXSettings.nSamplesPerSec = 11025;
  SOUND_WFXSettings.wFormatTag = WAVE_FORMAT_PCM;
  SOUND_WFXSettings.nBlockAlign =
      SOUND_WFXSettings.nChannels *(SOUND_WFXSettings.wBitsPerSample/8);
  SOUND_WFXSettings.nAvgBytesPerSec =
      SOUND_WFXSettings.nSamplesPerSec * SOUND_WFXSettings.nBlockAlign;
  SOUND_WFXSettings.cbSize = 0;
} // end SOUND_SetDefaultParameters()

//---------------------------------------------------------------------------
char *SOUND_ErrorCodeToString(int error_code )
   // Most error codes are defined in ErrorCodes.h,
   //      some are application specific, some originate from Win32 API calls.
{
 static char unknown_msg[81];

  switch(error_code)
   {
     case NO_ERROR            :   return("no error");

     // general "wave" errors.  See "ErrorCodes.h" and <MMSYSTEM.H>:
     case WAVERR_BADFORMAT    :   return("unsupported wave format");
     case WAVERR_STILLPLAYING :   return("still something playing");
     case WAVERR_UNPREPARED   :   return("header not prepared");
     case WAVERR_SYNC         :   return("device is synchronous");

     // input wave file error codes
     case WAVIN_ERR_OPEN     :    return("can't open wave file for input");
     case WAVIN_ERR_NOTWAVE  :    return("file is not a RIFF wave type");
     case WAVIN_ERR_INVALID  :    return("invalid wave file");
     case WAVIN_ERR_NODATA   :    return("no data in file");
     case WAVIN_ERR_NOTSUPPORTED: return("not a supported data type");
     case WAVIN_ERR_READING :     return("error reading data from file");
     case WAVIN_ERR_NOTOPEN :     return("tried to read and file is not open");

     // output wave file error codes
     case WAVOUT_ERR_OPEN   :     return("can't open wave file for output");
     case WAVOUT_ERR_WRITING:     return("error writing to wave file");
     case WAVOUT_ERR_NOTOPEN:     return("tried to write and file is not open");

     // Soundcard input error codes
     case SOUNDIN_ERR_NOTOPEN:    return("tried to read and soundcard is not open");
     case SOUNDIN_ERR_OVERFLOW:   return("input buffers overflowed");
     case SOUNDIN_ERR_TIMEOUT :   return("timed out waiting for input buffers");

     // Soundcard output error codes
     case SOUNDOUT_ERR_NOTOPEN:   return("tried to write and soundcard is not open");
     case SOUNDOUT_ERR_UNDERFLOW: return("output buffers underflowed");
     case SOUNDOUT_ERR_TIMEOUT:   return("timed out waiting for output buffers");

     // general multimedia error values as defined in mmsystem.h
     case MMSYSERR_ERROR:         return("unspecified MMSYS error");
     case MMSYSERR_BADDEVICEID:   return("device ID out of range");
     case MMSYSERR_NOTENABLED:    return("driver failed enable");
     case MMSYSERR_ALLOCATED :    return("device already allocated");
     case MMSYSERR_INVALHANDLE:   return("device handle is invalid");
     case MMSYSERR_NODRIVER   :   return("no device driver present");
     case MMSYSERR_NOMEM      :   return("memory allocation error");
     case MMSYSERR_NOTSUPPORTED:  return("function isn't supported");
     case MMSYSERR_BADERRNUM   :  return("error value out of range");
     case MMSYSERR_INVALFLAG   :  return("invalid flag passed");
     case MMSYSERR_INVALPARAM  :  return("invalid parameter passed");
     case MMSYSERR_HANDLEBUSY  :  return("handle being used");
     case MMSYSERR_NODRIVERCB  :  return("driver does not call DriverCallback");

     // Other errors...
     case SOUNDERR_WORD_AT_ODD_ADDRESS: return ("word at odd address");
     case SOUNDERR_THREAD_FAILED: return ("worker thread failed");
     case SOUNDERR_PIPE_FAILED  : return ("communication pipe failed");
     case SOUNDERR_UNKNOWN_CMD_ARGUMENT:  return ("unknown command argument");
     case SOUNDERR_ILLEGAL_CMD_PARAMETER: return ("illegal parameter value");
     case SOUNDERR_CMD_COMBI_UNSUPPORTED:    return ("command combination not supported");
     case SOUNDERR_DATA_TYPE_NOT_SUPPORTED:  return ("data type not supported");
     case SOUNDERR_OUTPUT_FILE_NOT_CONSUMED: return ("output file not consumed");

     case MEMORY_ERROR         :  return("Memory error");

     default:  // unknown MMSYSTEM error code:
          sprintf(unknown_msg, "unknown error #%d", error_code);
          return unknown_msg;
   }
} // end SOUND_ErrorCodeToString()









/*------------ Implementation of the Sound Thread -------------------------*/

// Sample buffers used to read mono- or stereo-samples from the input
//  NOTE: The data for left+right channel are NOT separated
//        (in contrast to Spectrum Lab !) so [2*x] elements.
double SndThd_dblInputChunk[  2/*!!*/ * SOUND_MAX_CHUNK_SIZE];



//---------------------------------------------------------------------------
DWORD WINAPI SoundThdFunc( LPVOID )
  /* Executes the sound analyzer thread.
   *   Was once the "Execute" method of a VCL Thread object,
   *   but this unit does not use the VCL any longer since May 2002.
   */
{
 INT i;
 INT error_code;
 double d;

   int iSoundInputDeviceID  = -1;  // -1 means "the default audio input device"
   int iSoundOutputDeviceID = -1;

  if(SoundDev==NULL)
     SoundDev = new CSound();
  if(SoundDev==NULL)
     ExitThread( SNDTHD_EXIT_BAD ); // exit code for this thread

  // Open Sound Device for input
  error_code=SoundDev->InOpen(  iSoundInputDeviceID, // often "-1"
                &SOUND_WFXSettings,
                SOUND_MAX_CHUNK_SIZE,
                0, // sample limit, 0 signifies continuous input stream
                FALSE/*don't start immediately*/   );
  if( error_code != NO_ERROR )
   {
     SOUND_iErrorCode = error_code;
     ExitThread( SNDTHD_EXIT_BAD ); // exit code for this thread
   }


  // Open Sound Device for output if necessary.
  //  Note: The same WFX-Settings must be used for InOpen() and OutOpen() !
/*
  error_code = SoundDev->OutOpen( m_iSoundOutputDeviceID,
                &SOUND_WFXSettings,SOUND_MAX_CHUNK_SIZE,0,
                FALSE ); // don't start immediately
  if( error_code != NO_ERROR )
      SOUND_iErrorCode = error_code;
/**/


  SoundDev->Start();   // start input & output synchronously...



  // SOUND_thd_error_buf_in = SOUND_thd_error_buf_out = 0; // NO !!
  SOUND_exited_from_thread_loop = false;


  /* Almost endless thread loop begins HERE................................*/
  while( !SndThd_fTerminateRequest && (SoundDev!=NULL) )
   {
     if( SoundDev->IsInputOpen() )
      {
       // Get input data from soundcard object.
       // NOTE: THE SECOND PARAMETER FOR InReadDouble
       //      IS THE COUNT OF SAMPLE PAIRS, NOT SIMPLE NUMBERS.
       //      Therefore, the "chunk" must have 2*SOUND_MAX_CHUNK_SIZE elements.
       // If SOUND_MAX_CHUNK_SIZE is 4096, and the soundcard runs in stereo mode,
       // the following routine places 8192 values in  _dblInputChunk !
       if( SoundDev->InReadDouble( SndThd_dblInputChunk,
             SOUND_MAX_CHUNK_SIZE ) != SOUND_MAX_CHUNK_SIZE )
        {
         error_code = SoundDev->m_ErrorCodeIn;
         if (error_code!=NO_ERRORS)
             SOUND_iErrorCode = error_code;

         switch(error_code)
          {
          /* waveform audio error return values from MMSYSTEM:    */
          case WAVERR_BADFORMAT   :   /* unsupported wave format  */
          case WAVERR_STILLPLAYING:   /* still something playing  */
          case WAVERR_UNPREPARED  :   /* header not prepared      */
          case WAVERR_SYNC        :   /* device is synchronous    */
            //SoundDev->InClose();  // what an ugly solution: close & open again..
            //error_code=SoundDev->InOpen(..
            break;
          /* error codes from ErrorCodes.H :                      */
          case NO_ERRORS:
            break;     // hmmm... so why did InRead() fail ??
          case SOUNDIN_ERR_NOTOPEN:
             // someone must have closed the soundcard's input. ?!?!?
            SoundDev->InOpen(  iSoundInputDeviceID, // often "-1"
                &SOUND_WFXSettings,
                SOUND_MAX_CHUNK_SIZE,
                0, // sample limit, 0 signifies continuous input stream
                TRUE); // start immediately
            break;
          case SOUNDIN_ERR_OVERFLOW:
          case SOUNDIN_ERR_TIMEOUT:
            //cpu couldn't keep up so try again: Close and re-open Sound Device.
            SoundDev->InClose();
            SoundDev->InOpen(  iSoundInputDeviceID, // often "-1"
                &SOUND_WFXSettings,
                SOUND_MAX_CHUNK_SIZE,
                0, // sample limit, 0 signifies continuous input stream
                TRUE); // start immediately

            break;
          default:
            // don't know how to handle this error, set breakpoint here:
            break;
          } // end switch(error_code)
        } /* end if (..InRead(..) != ..BUF_SIZE) */
       else
        { /* everything looks beautiful :-) */
         error_code = SoundDev->m_ErrorCodeIn; // hmm..
         if (error_code!=NO_ERRORS) // reading ok, but something is wrong...
          {
           SOUND_iErrorCode = error_code;
           SoundDev->m_ErrorCodeIn = NO_ERRORS;  // ouch !!!!!!!!
          }
        } /* end if <sampling of input data ok> */
      } // end if( SoundDev->IsInputOpen() )
     else
      {  // Re-Open Sound Device for input
        error_code=SoundDev->InOpen(  iSoundInputDeviceID, // often "-1"
                &SOUND_WFXSettings,
                SOUND_MAX_CHUNK_SIZE,
                0, // sample limit, 0 signifies continuous input stream
                TRUE/* start immediately*/   );
        if( error_code != NO_ERROR )
         {
           SOUND_iErrorCode = error_code;
           ExitThread( SNDTHD_EXIT_BAD ); // exit code for this thread
         }
      } // end if <sound device NOT open for input (?!)


    // crude peak detector for display purposes..
    d = 0;
    for(i=0; i<SOUND_MAX_CHUNK_SIZE; ++i)
     if(d<SndThd_dblInputChunk[i])
        d=SndThd_dblInputChunk[i];
    SOUND_dblAudioPeakValue = d;


    /* Put the data into the decimating 'output'-buffer:           */
    DecimatingAudioBuffer.EnterRealSamples(
                SndThd_dblInputChunk,   // source buffer
               SOUND_MAX_CHUNK_SIZE);   // count of source SAMPLE PAIRS(!!)

/*
     if(  SOUND_dac_output_active && SoundDev->IsOutputOpen() )
      {

       // Send output samples to sound device.
       //  Note that the thread will usually ***NOT*** wait HERE,
       //  because the output buffer should be almost empty now
       //  (as long as the sound-input is active).
       if( SoundDev->OutWriteStereo(
            m_i16_OutputChunk[0], m_i16_OutputChunk[1],
            SOUND_MAX_CHUNK_SIZE, &have_waited_here) != SOUND_MAX_CHUNK_SIZE)
        {
         SOUND_InternalComponentStates.cc[CIRCUIT_PARAM_AUDIO_OUTPUT1].iState=CIRCUIT_STATE_ERROR;
         SOUND_output_healthy = FALSE;
         ++m_OutputErrorCounter;  // set C++Builder breakpoint HERE !!
         error_code = SoundDev->m_ErrorCodeOut;
         // after maximizing a window, often got error_code =
         //      SOUNDIN_ERR_TIMEOUT (yes, SOUNDIN_.. ! )
         //   or SOUNDOUT_ERR_UNDERFLOW.
         // See ErrorCodes.h in this directory !
         //
         if( error_code != NO_ERROR )
          {
            SOUND_PutErrorMessage(error_code," in SoundOutWrite", 0);
          }
         else
          {
            SOUND_PutErrorString("sound output failed but no MMSYS error ?!?" );
          }

         switch(error_code) // Only handle errors resulting from the OUTPUT
          {
           case SOUNDOUT_ERR_NOTOPEN:
             // someone must have closed the soundcard's output. ?!?!?
             Synchronize( (TThreadMethod)&UpdateAudioSettings);
             break;

           case SOUNDOUT_ERR_UNDERFLOW:
             // ignore, dont close & reopen
             break;  // end case < OUTPUT UNDERFLOW

           case SOUNDOUT_ERR_TIMEOUT:
             // cpu couldn't keep up.. better luck next time
             break; //  end case < OUTPUT UNDERFLOW or OUTPUT TIMEOUT >

           default: // all other sound errors (like INPUT errors):
             break; // IGNORE input errors, don't close the output !!!
          } // end switch(error_code)
        } // end if <OutWrite .. != BUF_SIZE
       else
        { // everything looks beautiful :-)
         SOUND_InternalComponentStates.cc[CIRCUIT_PARAM_AUDIO_OUTPUT1].iState=CIRCUIT_STATE_ACTIVE;
         SOUND_output_healthy = TRUE;
         // if (have_waited_here)
            waited_somewhere = true;  // this thread has been WAITING here !
         ++m_TxBufferCounter;
        }
      } // end  if(  SOUND_dac_output_active )
/**/
   }  /* end <almost endless thread loop> */

  /* if the program ever gets here, the thread is quite "polite"... */
  /*                    or someone has pulled the emergency brake ! */
  SOUND_exited_from_thread_loop = true;

  ExitThread( SNDTHD_EXIT_OK ); // exit code for this thread
     // Note: The code STILL_ACTIVE must not be used as parameter for
     // ExitThread(), but what else ? See Win32 API help !
     // in winbase.h:
     //    #define STILL_ACTIVE STATUS_PENDING
     // in winnt.h:
     //    #define STATUS_PENDING ((DWORD   )0x00000103L)
  return SNDTHD_EXIT_OK; // will this ever be reached after "ExitThread" ?
} // end SoundThdFunc()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
BOOL SoundThd_LaunchWorkerThread(void)
   // Kick the real-time sound processing thread to life,
   //  without the help of Borland's Visual Component Library (VCL).
{

  // Make sure the cosine table exists, the frequency converter (NCO) may need it
  SOUND_InitCosTable();

  // NO-NO: SOUND_SetDefaultParameters();  // called done by application before evaluating command line
  // Prepare some 'derived' soundcard settings (WAVEFORMATEX) required for opening.
  // Note: The commented settings have already been set in SetDefaultParameters():
  // SOUND_WFXSettings.nChannels = 2;
  // SOUND_WFXSettings.wBitsPerSample = 16;
  // SOUND_WFXSettings.nSamplesPerSec = 11025;
  SOUND_WFXSettings.wFormatTag = WAVE_FORMAT_PCM;
  SOUND_WFXSettings.nBlockAlign =
      SOUND_WFXSettings.nChannels *(SOUND_WFXSettings.wBitsPerSample/8);
  SOUND_WFXSettings.nAvgBytesPerSec =
      SOUND_WFXSettings.nSamplesPerSec * SOUND_WFXSettings.nBlockAlign;
  SOUND_WFXSettings.cbSize = 0;


  // Initialize a 'decimating' buffer which is used to pass the samples
  // from the audio device to the application,
  // with the option to decimate the sample rate (et al) .
  DecimatingAudioBuffer.Init(
     65536L,       // lBufferLength, max count of sample points (!)
     SOUND_WFXSettings.nSamplesPerSec, // dblInputSampleRate, often 11025, 22050 or 44100
     1,           // lSampleRateDivisor, only powers of two are allowed
     0.0,         // dblMixerFrequency,  "NCO" frequency in Hertz
     FALSE,       // fComplex:  TRUE = with frequency shift + complex output
     SOUND_WFXSettings.nChannels ); // iNrChannels: 1=mono, 2=stereo processing


  // Now create the real-time worker thread
  SndThd_fTerminateRequest = FALSE;
  SndThd_hWorkerThread = CreateThread(
     NULL,    // LPSECURITY_ATTRIBUTES lpThreadAttributes = pointer to thread security attributes
     0,       // DWORD dwStackSize  = initial thread stack size, in bytes
     SoundThdFunc, // LPTHREAD_START_ROUTINE lpStartAddress = pointer to thread function
     0,       // LPVOID lpParameter = argument for new thread
     0,       // DWORD dwCreationFlags = creation flags
              // zero -> the thread runs immediately after creation
     &SndThd_dwWorkerThreadId // LPDWORD lpThreadId = pointer to returned thread id
    );
  // The thread object remains in the system until the thread has terminated
  // and all handles to it have been closed through a call to CloseHandle.

  if(!SndThd_hWorkerThread)
   { // CreateThread returned NULL, thread could not be created
     return FALSE;
   }

  // Define the Sound Thread's priority as required.
  SetThreadPriority(
    SndThd_hWorkerThread,          // handle to the thread
    THREAD_PRIORITY_ABOVE_NORMAL); // thread priority level
    
  return TRUE;
} // end SoundThd_LaunchWorkerThread()


/***************************************************************************/
BOOL SoundThd_TerminateAndDeleteThread(void)
   /* Terminates the sound thread, cleans up
    * and deletes(!) the sound thread.
    */
{
  int i;
  DWORD dwExitCode;
  BOOL  fOk = TRUE;

  if(SndThd_hWorkerThread)
   {
     // First, politely try to let the sound thread terminate itself
     //        allowing it to clean up everything.
     // About GetExitCodeThread(HANDLE hThread,LPDWORD lpExitCode):
     //       lpExitCode: Points to a 32-bit variable
     //          to receive the thread termination status.
     //       If the function succeeds, the return value is nonzero.
     //       If the specified thread has not terminated,
     //          the termination status returned is STILL_ACTIVE.
     dwExitCode=STILL_ACTIVE;
     SndThd_fTerminateRequest = TRUE;
     for(i=0;
        (i<20) && (GetExitCodeThread(SndThd_hWorkerThread,&dwExitCode));
         ++i)
      {
        // MessageBeep(-1);  // drums please
        Sleep(50);  // wait a little longer for the thread to terminate itself
        if(dwExitCode==SNDTHD_EXIT_OK)
           break;
      }

     // Before deleting the sound thread...
     if(SoundDev!=NULL)
      {
        if(SoundDev->IsInputOpen())
           SoundDev->InClose();
        if(SoundDev->IsOutputOpen())
           SoundDev->OutClose();
      } // end if <SoundDev>

     if(dwExitCode==STILL_ACTIVE)
      { // dangerous termination of the sound thread:
        TerminateThread(SndThd_hWorkerThread, // handle to the thread
                        SNDTHD_EXIT_BAD);     // exit code for the thread
      }

     CloseHandle(SndThd_hWorkerThread);
     SndThd_hWorkerThread = NULL;
     fOk &= (dwExitCode==SNDTHD_EXIT_OK);  // TRUE = polite ending,   FALSE = something went wrong.
   }

  return fOk;
} // end SOUND_TerminateAndDeleteThread()

