//---------------------------------------------------------------------------
// File  :  C:\CBproj\SoundUtl\SoundDec.h
// Date  :  2004-01-29   (ISO 8601, YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Interface for the CSoundDecimatingBuffer class.
//
// Used by:  SerInput, SndInput, SpecLab, PSK Detector/Demodulator,  ...
// Revision history :  See *.cpp
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#ifndef SoundDecH
#define SoundDecH

#include "SoundTab.h"

//----------------------- Constants ---------------------------------------

// Number of cascaded decimation stages for the FFT input (in the "Big Buffer")
//     8 decimation stages may divide by 3^8  =  6561
//    12 decimation stages may divide by 3^12 = 531441
#define SoundDec_MAX_DECIMATION_STAGES  12

// Other 'often needed' stuff..
#ifndef  C_PI
 #define C_PI  (3.1415926535897932384626)
#endif


//----------------------- Data Types --------------------------------------


#ifndef _T_SOUND_DECIMATOR_
#define _T_SOUND_DECIMATOR_
typedef struct
{
 int    ratio;                     // decimation ratio of this stage
 int    count;                     // counter for decimation_ratio
 T_Complex queue[SoundTab_DEC_FIR_LENGTH];
 T_Complex *inptr;                 // pointer for circular queue(s)
 const T_Float *coeffs;             // pointer to a table of coefficients
} T_SOUND_DECIMATOR;
#endif // _T_SOUND_DECIMATOR_


//------------------------ Global Variables -------------------------------



//**************************************************************************
//    Prototypes (no class methods)
//**************************************************************************



//***************************************************************************
//  Definition of the  CSoundDecimatingBuffer   class
//***************************************************************************

class CSoundDecimatingBuffer
{
public:
   CSoundDecimatingBuffer();            // constructor without parameters
   virtual ~CSoundDecimatingBuffer();   // destructor (cleans up)

   BOOL Init(  long lBufferLength,      // max count of sample points for OPTIONAL buffering
                                        //  (use 0 if you need no buffered processing)
             T_Float fltInputSampleRate, // often 11025, 22050 or 44100
               long lSampleRateDivisor, // powers of two are preferred !
             T_Float fltNcoFrequency,   // "L.O." frequency in Hertz
               bool fComplex, // TRUE  = complex output
                              // FALSE = real output values (regardless of f-conversion!)
                int iNrChannels );      // 1=mono, 2=stereo processing

   // Non-buffered processing: Results will overwrite the input data block !
   int ProcessWithoutBuffering(
       T_Float *pfltData,     // pointer to LEFT  DATABLOCK (or 'paired' data)
       T_Float *pfltData2,    // pointer to RIGHT DATABLOCK (NULL for 'paired' data)
         int iNrSamplePairs); // count of samples/pairs to be processed

   // Non-buffered processing, WITHOUT overwriting the input block:
   int ProcessAndCopy(
       T_Float *pfltData,     // pointer to LEFT  DATABLOCK (or 'paired' data)
       T_Float *pfltData2,    // pointer to RIGHT DATABLOCK (NULL for 'paired' data)
         int iNrSamplePairs,  // count of samples(/pairs?) to be processed
       T_Float *pfltDest,     // pointer to LEFT DESTINATION BLOCK (or 'paired' data)
       T_Float *pfltDest2,    // pointer to RIGHT SOURCE CHANNEL (NULL for 'paired' data)
         int iMaxDestSize);   // max count of T_Floats in each of the destination blocks


   // Buffered processing: decimated samples will be placed in a buffer..
   BOOL EnterSamples(
       T_Float *pfltSource,   // pointer to LEFT SOURCE CHANNEL (or 'paired' data)
       T_Float *pfltSource2,  // pointer to RIGHT SOURCE CHANNEL (NULL for 'paired' data)
         int iNrSamplePairs);

   // Read a few decimated samples from the internal buffer. May be called from a different thread.
   long    GetSamplePoints( int iNrSamplePoints,
               LONGLONG *pi64UniqueIndex, // pair index of 1st sample to be read, will be updated for next call
               T_Float *pfltDest,  // points to the caller's destination buffer, IF FLOATING POINT values
               SHORT  *pi16Dest); // points to the caller's destination buffer, IF 16-BIT INTEGERS

   // A few "Get"- and "Set"-routines to query and modify some parameters on the fly:
   T_Float GetNcoFrequency(void);
   BOOL    SetNcoFrequency(T_Float fltNewFreq);

   T_Float GetDecimatedSampleRate(void);  // here: "buffer output rate"

   void SetFastMode(int iFastMode); // 0=normal with good anti-alias filtering, 1=ugly but quick decimation
   int  GetFastMode(void);

   long    GetOccupiedBufferSpace( LONGLONG i64UniqueIndex);
   long    GetTotalBufferSpace(void);

   inline LONGLONG GetLatestBufferIndex( void ) { return m_i64TotalSampleCount; };
   inline int      GetComponentsPerPair( void ) { return m_iNrOutputComponents; };
   inline int      GetNumberOfChannels( void  ) { return m_iNrChannels;         };
   inline int      GetDecimationRatio(  void  ) { return m_iDecimationRatio;    };
   inline int      GetFreqConversionFlag(void ) { return m_fFreqConversion;     };
   inline int      GetComplexOutputFlag( void ) { return m_fComplex;            };


private:
   // To process data with or without buffering :
   int  ProcessSamples(  // returns the count of DECIMATED sample points
       T_Float *pfltSource,  // pointer to LEFT SOURCE CHANNEL (or 'paired' data)
       T_Float *pfltSource2, // pointer to RIGHT SOURCE CHANNEL (NULL for 'paired' data)
         int iNrSamplePairs, // count of sample points (not single floats) to process
       T_Float *pfltDest,    // pointer to LEFT DESTINATION BLOCK (or 'paired' data)
       T_Float *pfltDest2,   // pointer to RIGHT SOURCE CHANNEL (NULL for 'paired' data)
         int iDestIndex, int iDestSize ); // index and length of destination buffer (may be CIRCULAR)

   // A chain of decimators (with anti-aliasing lowpass filters) :
   T_SOUND_DECIMATOR m_Decimator[2][SoundDec_MAX_DECIMATION_STAGES];
   T_SOUND_DECIMATOR m_Complex2RealDecimator[2]; // for Complex->Real conversion
   int     m_iComplex2RealMixerPhase;  // four-phase signal for Complex->Real conversion
   int     m_iUsedDecimatorStages; // count of stages actually used
   T_Float m_fltInputSampleRate;
   int     m_iDecimationRatio;
   int     m_iFastAndUgly;

   // Properties of a "big" ring-buffer where the input samples
   //    for the FFT's are be collected.
   long     m_lBufferSize;         // max index: m_fltBuffer[m_lBufferSize-1] (!)
   T_Float *m_pfltBuffer;
   LONGLONG m_i64TotalSampleCount; // used both as COUNTER and ARRAY INDEX (modulo size)

   int     m_iNrChannels;    // up to now, only 1 or 2 INPUT channels
   int     m_iNrOutputComponents; // 1, 2, or 4 components; depends on channels+complex/real output

   bool    m_fComplex;             // TRUE: the buffer uses complex values (= I/Q samples)
   bool    m_fFreqConversion;      // TRUE: frequency conversion enabled
   T_Float m_fltNcoFrequency;
   T_Float m_fltNcoPhase;
   bool    m_fRealOutputSign;      // to convert a complex output stream into "real" values

}; // end class CSoundDecimatingBuffer


#endif // SoundDecH
