//-----------------------------------------------------------------------
// C:\cbproj\SoundUtl\Sound.cpp : implementation of the CSound class.
//
// This class implements an object that reads from / writes to  a soundcard.
//      Usually uses the multimedia system API (MMSYSTEM),
//      but in a few cases this module MAY work with ASIO drivers too.
//
// Author: Wolfgang Buescher (DL4YHF) 2000..2015
//         Parts for MMSYSTEM taken from "WinPSK31 by Moe Wheatley, AE4JY" .
//         ASIO support based on a sample from the Steinberg ASIO SDK .
// Used in:  * Spectrum Lab               (project: C:\cbproj\SpecLab)
//           * The 'Sound Input Utility'  (project: C:\cbproj\SoundUtl)
//
//
//  Contents :
//   - ASIO input, lowest level: CSound_AsioCallback()
//-----------------------------------------------------------------------


/* Revision History (lastest entry first):
 *
 *  2022-10-16 : The SAMPLE VALUE RANGE in floating point arrays is now
 *        +/- 1.0 (aka "normalized"), unless specified in a "chunk info block".
 *
 *  2015-10-14 : Added direct support for a few other sample formats received
 *        from an external ADC via the serial port (USB adapter, actually).
 *        Had to de-activate the stupid 'Serial Microsoft BallPoint' in the
 *        system control again (now under Win8.1), because the stupid hardware
 *        detection thought the digitized noise were mouse events,
 *        and played havoc with the GUI.
 *
 *  2015-02-01 : Experienced a very strange bug with the E-MU 0202 @ 192 kHz:
 *      - When scrolling in a web browser (V35.0.1 until tomorrow..),
 *        the audio input timed out (noticed in CSound::WaitForInputData),
 *        but even with a ridiculously long timeout, no more input from the E-MU.
 *      - Tried with an without CSOUND_USE_CRITICAL_SECTION : No difference.
 *      - Tried an old version from 2009 (V2.73 b1) : No problem there.
 *      - Turned off all DSP functions in V2.81 b10 which were not available 2009:
 *           Input Preprocessor (GPS pulse det + resampler) off,
 *           Phase/Amplitude meters off, Decimated Wave Save off:
 *           Problem disappeared ! (CPU load dropped to 38 % for a 'single core')
 *      - Input Preprocessor with GPS pulse det AND resampling to nominal SR
 *           on again, total CPU load 70 % : Rapid scrolling stopped input again.
 *           Soundcard buffer: 5 buffers with 16384 samples each .
 *      - Changed the priority of SL's *AUDIO PROCESSING THREAD* from 0 to 1:
 *           SEEMED TO (!) reduce the problem. Story continues in SoundThd.cpp .
 *
 *
 *  2012-08-14 :
 *      - Added support for Winrad-compatible ExtIO-DLLs, by virtue of
 *        DL4YHF's "Audio-I/O-Host" (AudioIO.c) which can optionally load
 *        such DLLs dynamically, in addition to the normal Audio-I/O-DLLs .
 *        Because many ExtIO-DLLs also control the VFO (HF tuning frequency),
 *        methods were added to the CSound class which can pass on this info.
 *        New: CSound::HasVFO(), GetVFOFrequency(), SetVFOFrequency(), etc.
 *
 *  2012-07-06 :
 *      - Eliminated the use of AUDIO DEVICE NUMBERS within Spectrum Lab,
 *        because the triple-stupid windows juggles around with these numbers
 *        whenever an USB audio device is connected/disconnected, which includes
 *        stupid 'dummy' audio devices like bluetooth audio, etc etc.
 *        Thus the beloved E-MU 0202 may have been device number 3 yesterday,
 *        but number 4 today. No thanks, we don't want this kind of bullshit.
 *        SL now saves the device's "Product Name" in its configuration,
 *        which is part of microsoft's WAVEINCAPS structure.
 *        Because the windows multimedia API still uses device NUMBERS,
 *        not NAMES, Sound_GetAudioDeviceNameWithoutPrefix() can optionally
 *        determine a NUMERIC "device index" for any given audio device name
 *        (for the ancient multimedia system).
 *
 *  2012-03-25 :
 *       - Timestamp in T_ChunkInfo.ldblUnixDateAndTime is now set as close as possible
 *          to the 'real time' in CSound::GetDateAndTimeForInput(),
 *          using the system's high-resolution timer ("QueryPerformanceCounter").
 *
 *  2008-10-19 :
 *       - Eliminated WaitForSingleObject() because with certain kind of input
 *          devices, it's difficult to mimick the strange Windows MM API...
 *
 *  2008-03-29 :
 *       - Added support for the new "driver-less AUDIO-I/O" system,
 *          which is basically a DLL with circular buffer in shared memory.
 *          Can be used to send the output stream "directly" to Winamp,
 *          without a virtual audio cable or other crap in between .
 *          Modified parts can be found by searching SWI_AUDIO_IO_SUPPORTED .
 *
 *  2007-07-13 :
 *       - Output routine now accepts ARBITRARY block lengths,
 *          which was necessary when the RESAMPLING option was added in SpecLab.
 *
 *
 *  2006-04-14 :
 *       -  2 * 16-bit ASIO works, surprisingly even while the output is sent
 *          to an MMSYSTEM device using the same soundcard at the same time.
 *          Two different ASIO drivers (one for input, a different one for
 *          output) does *NOT* work, because the ASIO-wrapper (+"thiscall" fix)
 *          can only load one ASIO driver at a time .
 *
 *  2006-03-23 :
 *       -  Checked the possibility to support ASIO drivers in this wrapper
 *          as well as the standard multimedia system, since several users
 *          reported that the 24-bit mode didn't work with their soundcards.
 *          Furthermore, Johan Bodin /SM6LKM asked for it ;-)
 *
 *  2005-04-23 :
 *       -  SOUND.CPP now used in the experimental WOLF GUI, too
 *           ( WOLF = Weak-signal Operation on Low Frequency, by KK7KA).
 *       -  Fixed a problem in InClose() / MMSYSTEM error "Still Playing" (!)
 *
 *  2004-05-27 :
 *       -  Changed the data type in InReadStereo() from SHORT to T_Float,
 *          making it easier to handle different ADC- and DAC- resolutions
 *          internally.  EVERYTHING WAS(!) SCALED TO +/-32k REGARDLESS
 *          of 8, 16, or 24 bits per sample. No problem with 32-bit floats.
 *          (One could also normalize everything to -1.0...+1.0 as well,
 *           but this would mean a lot of modifications in Spectrum Lab).
 *          This renders the CSound class incompatible with earlier versions !
 *       -  First successful experiment to read 24-bit-samples
 *          from a Soundblaster Audigy 2 ZS . Amazingly the standard multi-
 *          media system (with waveInOpen etc) also supports sampling at
 *          96 kHz with 24 bits resolution - no need to play with ASIO drivers,
 *          Direct X, and other stuff yet (other cards require ASIO for this, though)
 *       -  To achieve 24 bit resolution also for the OUTPUT (!),
 *          it was necessary to use the WAVEFORMATEXTENSIBLE structure.
 *           (but many years later, with an E-MU 0202 on Win8.1 64bit,
 *            output with 24 bit per sample didn't work anymore : 2016-10-18 )
 *
 *  2004-05-11 :
 *       -  Modified buffers, now dynamically allocated.
 *          Reason: There's a big difference between 96kHz, 2 channels, 24 bit
 *                  and 11kHz, 1 channel, 16 bit on a notebook with tiny RAM !
 *
 *  2001-11-24 (YYYY-MM-DD):
 *       -  Optional STEREO processing implemented,
 *          depending on an option defined in the file SWITCHES.H .
 *
 *  2000-09-02 :
 *       -  Changed the error codes in ErrorCodes.h.
 *       -  Splitted m_ErrorCode into m_ErrorCodeIn and m_ErrorCodeOut
 *          because simultaneous input and output caused trouble
 *          before this.
 *   June 10, 2000: taken from WinPSK31 V1.2 "special edition"
 *                   "Originally written by Moe Wheatley, AE4JY"
 *                   "Modified and extended by Dave Knight, KA1DT"
 *       -  Adapted for C++Builder 4 by Wolfgang Buescher, DL4YHF
 *       -  All TAB characters from the source file removed.
 *          (never use TABS in source files !! tnx.)
 *       -  Defined a global variable of type CSound
 *          that will be used by the GENERATOR and ANALYSER
 *          (simultaneously)
 *       -  This unit will always remain "public domain"
 *          like the original work by AE4JY and KA1DT.
 *
 */


#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !


#pragma alignment /* should report the alignment (if the compiler is kind enough) */
   // 2012-03-10 : BCB6 said "Die Ausrichtung betrgt 8 Byte, die enum-Gre betrgt 1 Byte"
   // This was, most likely, the reason for the malfunction of the goddamned WAVEFORMATEX .


#include <windows.h>  // try NOT to use any Borland-specific stuff here (no VCL)
#include <math.h>     // to generate test signals only (sin)
#include <stdio.h>    // no standard-IO, but sprintf used here
#pragma hdrstop

#include "Sound.h"        // Once just the soundcard interface, now multi-purpose
#include "ErrorCodes.h"
#include "VorbisFileIO.h" // here: using T_StreamHdrUnion for uncomressed audio via COM-port


#ifndef  SWI_UTILITY1_INCLUDED
# define SWI_UTILITY1_INCLUDED 1
#endif
#ifndef  SWI_UTILITY2_INCLUDED
# define SWI_UTILITY2_INCLUDED (!SWI_UTILITY1_INCLUDED)
#endif

#if( SWI_UTILITY1_INCLUDED || SWI_UTILITY2_INCLUDED) /* "utility"-package #1 or #2 included ? */
  // > SoundUtl/utility2.cpp is the small brother of utility1; both use THE SAME header file:
  #include "utility1.h"  // UTL_WriteRunLogEntry(), UTL_NamedMalloc(), etc, *GUARANTEE* 8-byte alignment
#endif

#ifndef  SWI_AUDIO_IO_SUPPORTED
# define SWI_AUDIO_IO_SUPPORTED 0
#endif
#if( SWI_AUDIO_IO_SUPPORTED )
# include "AudioIO.h"   // located in \cbproj\AudioIO\*.*
#endif // SWI_AUDIO_IO_SUPPORTED


// local defines
#define TIMELIMIT 1000  // Time limit to WAIT on a new soundcard buffer
                        // to become ready in milliSeconds.
                        // 2004-05-10: Changed from 2000 to 1000,
                        //   because a too long delay causes a ping-pong-type
                        //   of error when running in full-duplex mode.
                        //   ( especially bad when tested with Audigy 2
                        //     and 96 kHz sampling rate, but small buffers )
                        //   On the other hand, 500ms was too short .

#define SOUND_MAGIC_BYTE_AT_END_OF_BUFFER 0xCB /* for a primitive 'buffer integrity test' */

static DWORD Sound_dwDebuggerDummy;

typedef struct tSoundTokenTableEntry
{ char *pszToken;
  int  iSoundSampleFormat;
} T_SoundTokenTableEntry;

static T_SoundTokenTableEntry SoundSampleFormats[] =
{ { "unknown", SOUND_SAMPLE_FMT_UNKNOWN },
  { "U8",      SOUND_SAMPLE_FMT_U8      }, // 8 bit per channel (in each sample point), UNSIGNED integer
  { "S8",      SOUND_SAMPLE_FMT_S8      }, // 8 bit per channel (in each sample point), UNSIGNED integer
  { "U16",     SOUND_SAMPLE_FMT_U16_LE  }, // 16 bit per channel (in each sample point), UNSIGNED integer, Little Endian
  { "S16",     SOUND_SAMPLE_FMT_S16_LE  }, // 16 bit per channel (in each sample point), UNSIGNED integer, Little Endian
  { "U16_BE",  SOUND_SAMPLE_FMT_U16_BE  }, // 16 bit per channel (in each sample point), UNSIGNED integer, Big Endian
  { "S16_BE",  SOUND_SAMPLE_FMT_S16_BE  }, // 16 bit per channel (in each sample point), UNSIGNED integer, Big Endian
  { "U24",     SOUND_SAMPLE_FMT_U24_LE  }, // 24 bit per channel (in each sample point), UNSIGNED integer, Little Endian
  { "S24",     SOUND_SAMPLE_FMT_S24_LE  }, // 24 bit per channel (in each sample point), UNSIGNED integer, Little Endian
  { "U24_BE",  SOUND_SAMPLE_FMT_U24_BE  }, // 24 bit per channel (in each sample point), UNSIGNED integer, Big Endian
  { "S24_BE",  SOUND_SAMPLE_FMT_S24_BE  }, // 24 bit per channel (in each sample point), UNSIGNED integer, Big Endian
  { "U32",     SOUND_SAMPLE_FMT_U32_LE  }, // 32 bit per channel (in each sample point), UNSIGNED integer, Little Endian
  { "S32",     SOUND_SAMPLE_FMT_S32_LE  }, // 32 bit per channel (in each sample point), UNSIGNED integer, Little Endian
  { "U32_BE",  SOUND_SAMPLE_FMT_U32_BE  }, // 32 bit per channel (in each sample point), UNSIGNED integer, Big Endian
  { "S32_BE",  SOUND_SAMPLE_FMT_S32_BE  }, // 32 bit per channel (in each sample point), UNSIGNED integer, Big Endian
  { "F32",     SOUND_SAMPLE_FMT_F32_LE  }, // 32 bit per channel (in   each sample point), FLOATING POINT, for Intel CPU+FPU
  { "F32_BE",  SOUND_SAMPLE_FMT_F32_BE  }, // 32 bit per channel (in each sample point), FLOATING POINT, for we-don't-know

  { "IQ12",    SOUND_SAMPLE_FMT_IQ12    }, // header byte + 2 * 12 bit unsigned, I/Q format used by DL4YHF's PIC12F675 ADC for the serial port
  { "GPSDO3",  SOUND_SAMPLE_FMT_GPSDO3  }, // header byte + 2 or 4 bytes, real or I/Q format used by DL4YHF's PIC-based GPSDO for the serial port
  { "Ogg",     SOUND_SAMPLE_FMT_OGG     }, // Ogg containers, hopefully with Vorbis-compressed audio
  { NULL,      0 } // << marker for the end of the list
}; // end SoundSampleFormats[]

// A "GUID" for working with the WAVEFORMATEXTENSIBLE struct (for PCM audio stream)
static const GUID KSDATAFORMAT_SUBTYPE_PCM = {  // here: "PCM" means "integer samples"
      0x1,0x0000,0x0010,{0x80,0x00,0x00,0xaa,0x00,0x38,0x9b,0x71} };
 // Found the following GUID in an 'existing' wave file (pls55.wav), here as a BYTE array:
 // const BYTE WaveIO_b16GUID_PCM[16] = /* taken from an 'existing' wave file */
 //  [0]  [1]  [2]  [3]  [4]  [5]  [6]  [7]  [8]  [9]  [10] [11] [12] [13] [14] [15]
 // { 0x01,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x80,0x00,0x00,0xaa,0x00,0x38,0x9b,0x71 };
 // The MS site says it should be this "GUID" :
 //  00000001 - 0000 - 0010 - 8000 - 00aa00389b71
 // Note the strange different byte orders in bytes[6/7] and [8/9] !


#if(0)
// Another "GUID" for working with the WAVEFORMATEXTENSIBLE struct (for FLOATING POINT samples)
static const GUID KSDATAFORMAT_SUBTYPE_IEEE_FLOAT = { // also PCM, but here for FlOATING POINT
      0x3,0x0000,0x0010,{0x80,0x00,0x00,0xaa,0x00,0x38,0x9b,0x71} };
#endif

// removed: WAVEFORMATEX SOUND_WFXSettings;  // input- and output audio device settings

// MMSYSTEM callback functions for use in soundcard input/output
void CALLBACK WaveInCallback( HWAVEIN, UINT, CSound*, DWORD, DWORD );
void CALLBACK WaveOutCallback( HWAVEOUT, UINT, CSound*, DWORD, DWORD );

#if( SWI_ASIO_SUPPORTED )      // Support ASIO audio drivers too ?
 void CSound_AsioCallback( T_AsioWrap_DriverInfo *pDriverInfo, long index, DWORD dwInstance );
#endif

static int CountBits( DWORD dwMask )  // internal helper function
{ int n=0;
  for(int i=0; i<32; ++i)
   { if( (dwMask&1) != 0)
       ++n;
     dwMask>>=1;
   }
  return n;
}

extern BOOL SndThd_fRequestToTerminate; // Notbremse fr 'gecrashte' Threads

//**************************************************************************
//    Debugging
//**************************************************************************
extern BOOL APPL_fMayUseASIO; // Since 2011-08-09 ASIO may be disabled via command line switch .
       // Located (for SpecLab) in C:\cbproj\SpecLab\SplashScreen.cpp
       // or (for the 'WOLF-GUI') in c:\cbproj\WOLF\WOLF_GUI1.cpp .
       // or (for the 'SndInput' utility) in c:\cbproj\SoundUtl\SndInMain.cpp .
BOOL SOUND_fDumpEnumeratedDeviceNames = FALSE;  // flag for Sound_InputDeviceNameToDeviceID()


#if (SWI_ALLOW_TESTING) // unexplainable crash with corrupted "call stack" ?
 int  Sound_iSourceCodeLine = 0;  // debugging stuff: last source code line
 int  Sound_iMMSourceCodeLine=0;  // similar, for callback from the 'driver'
 int  Sound_iAsioCbkSourceCodeLine=0; // .. and yet another (incompatible) callback
 int  Sound_iUnknownWaveInCallbackMsg = -1;
 int  Sound_iUnknownWaveOutCallbackMsg = -1;
 #define DOBINI() Sound_iSourceCodeLine=__LINE__
 #define DOBINI_MM_CALLBACK() Sound_iMMSourceCodeLine=__LINE__
 #define DOBINI_ASIO_CALLBACK() Sound_iAsioCbkSourceCodeLine=__LINE__
  // Examine the most recent source code line in the debugger's "watch"-window .
  // Details about "DOBINI()" (Bavarian "here I am")  in SoundThd.cpp  !
 static void LocalBreakpoint(void)
  {
  }  // <<< set a breakpoint HERE (when running under debugger control) !
 #define LOCAL_BREAK() LocalBreakpoint()
#else
 #define DOBINI()
 #define DOBINI_MM_CALLBACK()
 #define LOCAL_BREAK()
#endif


//--------------------------------------------------------------------------
// Helper functions (there's no need to turn everything into a shiny CLASS!)
//--------------------------------------------------------------------------

//---------------------------------------------------------------------
long double SOUND_GetSystemDateAndTimeAsUnixSeconds(void)
  // Retrieves the PC's 'local system time' as number of seconds
  //           elapsed since 1970-01-01 00:00:00 (UTC).
  // Since the SI unit for time is SECONDS, we use seconds, not micro, not milli.. !
  // A 'long double' has a mantissa of approximately 64 bits which is precise enough.
{
  FILETIME ft; // date and time in a crazy windows specific format
  unsigned __int64 u64StrangeMsTime;
  long double ldblUnixSeconds;

  GetSystemTimeAsFileTime(&ft);  // this is a crazy Windows API function:
     // > The GetSystemTimeAsFileTime function obtains the current system date and time.
     // > The information is in Coordinated Universal Time (UTC) format.
     // > The FILETIME structure is a 64-bit value representing
     // > the number of 100-nanosecond intervals since January 1, 1601.
     // > typedef struct _FILETIME {
     // >     DWORD dwLowDateTime;   // low-order 32 bits of the 'file' time
     // >     DWORD dwHighDateTime;  // high-order 32 bits of the 'file' time
     // > } FILETIME;
  u64StrangeMsTime = ((unsigned __int64)ft.dwHighDateTime << 32)
                   | ((unsigned __int64)ft.dwLowDateTime) ;
  // Convert microsoft's strange "file time" to Unix seconds :
  ldblUnixSeconds = (long double)u64StrangeMsTime * (long double)100e-9 - 11644473600.0;
  return ldblUnixSeconds;
} // end SOUND_GetSystemDateAndTimeAsUnixSeconds()


/***************************************************************************/
long SOUND_GetClosestNominalStandardSamplingRate( double dblCalibratedSamplingRate )
   // Decides to which "nominal" soundcard sampling rate should be used
   //  for a certain 'calibrated' soundcard sampling rate.
{
  if( dblCalibratedSamplingRate < 5800 )
   { return 5512;
   }
  if( dblCalibratedSamplingRate < 7000 )
   { return 6000;
   }
  if( dblCalibratedSamplingRate < 9000 )
   { return 8000;
   }
  if( dblCalibratedSamplingRate < 11500 )
   { return 11025;
   }
  if( dblCalibratedSamplingRate < 14000 )
   { return 12000;           // 48k / 4
   }
  if( dblCalibratedSamplingRate < 20000 )
   { return 16000;           // 2 * 8 kHz
   }
  if( dblCalibratedSamplingRate < 25000 )
   { return 22050;           // 44k1 / 2
   }
  if( dblCalibratedSamplingRate < 38000 )
   { return 32000;           // also a standard !  4 * 8 kHz
   }
  if( dblCalibratedSamplingRate < 46000 )
   { return 44100;           // "CD"
   }
  if( dblCalibratedSamplingRate < 55000 )
   { return 48000;
   }
  if( dblCalibratedSamplingRate < 70000 )
   { return 64000;           // 8 * 8 kHz (really a standard ?)
   }
  if( dblCalibratedSamplingRate>=86000 && dblCalibratedSamplingRate<90000 )
   { return 88200; // "Sampling rate used by some professional recording equipment when the destination is CD (multiples of 44,100 Hz)."
   }
  if( dblCalibratedSamplingRate>=94000 && dblCalibratedSamplingRate<=98000 )
   { return 96000; // DVD-Audio, 2 * 48 kHz
   }
  if( dblCalibratedSamplingRate>=174000 && dblCalibratedSamplingRate<=178000 )
   { return 176400; // "used by professional applications for CD production"
   }
  if( dblCalibratedSamplingRate>=190000 && dblCalibratedSamplingRate<=194000 )
   { return 192000; // "High-Definition DVD" (and supported by E-MU 0202)
   }
  // Arrived here: No joy, simply return the ROUNDED sampling rate
  return (long)(dblCalibratedSamplingRate+0.5);
} // end Config_NominalSampleRateToCalibTableIndex()


/***************************************************************************/
char *Sound_GetAudioDeviceNameWithoutPrefix(
           char *pszNameOfAudioDeviceOrDriver,  // [in] for example "4 E-MU 0202 | USB"
           int  *piDeviceIndex ) // [out] for example -1 or 4 (parsed from the name!)
  // Use with MConfig.sz255AudioInputDeviceOrDriver !
  // The result (example: "E-MU 0202 | USB" is only valid as long as the input (string) exists !
  //
  // Even though the input is a STRING, it may be a decimal string
  // representing one of the ancient 'multimedia audio devices' in Windoze,
  // for example pszNameOfAudioDeviceOrDriver = "-1" -> *piDeviceIndex = -1
  //                  -> "use the STANDARD audio device for in- or output".
{
  int idx=0, sign=1;
  char *cp;
  BOOL got_a_digit = FALSE;

  cp = strrchr(pszNameOfAudioDeviceOrDriver, '.' );
  if( (cp!=NULL) && stricmp( cp, ".dll") == 0 )
   { // It's the filename of a DLL - this cannot be a normal "audio device" !
     if( piDeviceIndex!=NULL )
      { *piDeviceIndex = C_SOUND_DEVICE_AUDIO_IO;
      }
     return pszNameOfAudioDeviceOrDriver;
   }

  cp = pszNameOfAudioDeviceOrDriver;
  if( strncmp( cp, "A: ", 3) == 0 ) // must be an ASIO DEVICE !
   { if( piDeviceIndex!=NULL )
      { *piDeviceIndex = C_SOUND_DEVICE_ASIO;
      }
     return cp+3;
   }
  // Try a few 'fixed device names' which need extra treatment :
  //    Extrawrste fr SDR-IQ und Perseus.. vgl. Definitionen in CONFIG.H !
  if( strcmp( cp, "SDR-IQ / SDR-14" )==0 )
   { if( piDeviceIndex!=NULL )
      { *piDeviceIndex = C_SOUND_DEVICE_SDR_IQ;
      }
     return cp;
   }
  if( strcmp( cp, "Perseus" )==0 )
   { if( piDeviceIndex!=NULL )
      { *piDeviceIndex = C_SOUND_DEVICE_PERSEUS;
      }
     return cp;
   }
  if( (strncmp( cp, "COM", 3)==0) && (cp[3]>='0') && (cp[3]<='9') )
   { if( piDeviceIndex!=NULL )
      { *piDeviceIndex = C_SOUND_DEVICE_COM_PORT; // audio via serial port
      }
     return cp;
   }
  if( *cp=='-' )                // Skip the optional sign (e.g. "-1" = default audio device)
   { sign = -1;
     ++cp;
   }
  while( *cp>='0' && *cp<='9' ) // skip DIGITS
   { idx = 10*idx + (*cp-'0');
     ++cp;
     got_a_digit = TRUE;
   }
  idx *= sign;
  // Beware, with braindead names like "2- USB Audio CODEC" (for the audio from
  // an Icom IC-7300) we would get here with cp = "- USB Audio CODEC",
  // but the "minus after the digits" certainly does NOT belong to the
  // "constant" part of the device name (e.g. "USB Audio CODEC"). Thus:
  if( got_a_digit ) // take care for some special characters immediately AFTER the digit(s):
   { if( *cp=='-' ) // Ooops ... what LOOKED like a prefix isn't OURS (*) !
      { cp = pszNameOfAudioDeviceOrDriver;
        idx = 0;
        // (*) OUR numeric prefixes, used in the drop-down list of audio-
        //       and other devices in Spectrum lab, are separated by SPACES
        //       but no bloody trailing minus-characters.
        // Thus we "undo" skipping the "2" in "2- USB Audio CODEC" at this point,
        // and hope that Sound_InputDeviceNameToDeviceID() will find a UNIQUE
        // match for the remaining name ("2- USB Audio CODEC", not "- USB Audio CODEC")
        // in the crippled 'waveIn'-device-names. Phew.
      }
   }
  if( piDeviceIndex!=NULL )
   { *piDeviceIndex = idx;
   }
  while( *cp==' ' ) // skip separator between numeric prefix (audio device index) and device name
   { ++cp;
   }
  return cp;  // result: Pointer to the DEVICE NAME or the end of the string (trailing zero)
} // end Sound_GetAudioDeviceNameWithoutPrefix()


//--------------------------------------------------------------------------
int Sound_InputDeviceNameToDeviceID( char* pszNameOfAudioDevice )
  // Added 2012-07-06 for Spectrum Lab. Reasons below,
  //  as well as in C:\cbproj\SpecLab\Config.cpp .
  // Return value from Sound_InputDeviceNameToDeviceID() :
  //  <positive> : "device ID" of a standard (WMM, Windows Multi Media) audio device,
  //  <negative> : one of the following constants :
  //       C_SOUND_DEVICE_SDR_IQ, C_SOUND_DEVICE_PERSEUS: SDRs with built-in support
  //       C_SOUND_DEVICE_ASIO    : ASIO device, identified by NAME
  //       C_SOUND_DEVICE_AUDIO_IO: Name of an Audio-I/O-DLL, or possibly ExtIO;
  //                                 in fact, any FILE+PATH which looks like a DLL.
  //       C_SOUND_DEVICE_STEAM  : Dummy for web audio streams (name beginning with
  //                                 "tcp://", "http://", "udp://", "raw://" ).
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Since 2012-07, Spectrum Lab doesn't rely on those stupid AUDIO DEVICE NUMBERS
  // anymore because the triple-stupid windows juggles around with these numbers
  // whenever an USB audio device is connected/disconnected, which includes
  // stupid 'dummy' audio devices like 'bluetooth audio', etc.
  // Thus the beloved E-MU 0202 may have been device number 3 yesterday,
  //  but number 4 today. No thanks, we don't want this kind of bullshit.
  //  SL now saves the device's "Product Name" in its configuration,
  //  which is part of microsoft's WAVEINCAPS structure.
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Since 2020-09, Sound_InputDeviceNameToDeviceID() tries even harder
  //  to make sense from pszNameOfAudioDevice when comparing that name
  //  with the garbage delivered by waveInGetDevCaps(), especially since
  //  everything with an analog/digital converter seems to be called a
  //  'Microphone' (or 'Mikrofon' on a german PC) these days now.
  //  Under Windows 8, audio devices could be renamed somewhere in the
  //  'classic' audio control panel ('Aufnahmegerte'..'Eigenschaften'..'Allgemein').
  //  For example, a HP notebook's onboard microphone was renamed from
  //  'Mikrofon' to 'Onboard-Microphone' (which is all we can modify,
  //  the 'Realtek High Definition Audio' cannot be edited).
  //  The WAVEINCAPS.szPname returned by waveInGetDevCaps() was
  //      NOT "Onboard-Microphone (Realtek High Definition Audio)" then,
  //      but "Onboard-Microphone (Realtek Hig" due to the stupid truncation .
  //           |________________|  |__________..
  //               |                  |
  //               |                 Often useless due to TRUNCATION !
  //               |
  //               Sound_InputDeviceNameToDeviceID() will also find a match
  //               for THIS (renameable) part of the audio defice name now !

{
 int i, search_pattern_length, n_devs, iDeviceID, iBestDeviceID, iSecondBestDeviceID;
 WAVEINCAPS  my_waveincaps;
 char *cp1, *cp2, *pszAudioDeviceNameWithoutPrefix;
 BOOL found_perfect_match = FALSE;

  pszAudioDeviceNameWithoutPrefix = Sound_GetAudioDeviceNameWithoutPrefix(
                                         pszNameOfAudioDevice, &iDeviceID );
  search_pattern_length = strlen(pszAudioDeviceNameWithoutPrefix);
  // Note: Sound_GetAudioDeviceNameWithoutPrefix() onlx removes OUR prefix.
  //       It would not skip the "2-" in "2- USB Audio CODEC" (stupid, and
  //       ever-changing name of the IC-7300's built-in audio input device).
  if(    iDeviceID==C_SOUND_DEVICE_ASIO
      || iDeviceID==C_SOUND_DEVICE_SDR_IQ
      || iDeviceID==C_SOUND_DEVICE_PERSEUS
      || iDeviceID==C_SOUND_DEVICE_AUDIO_IO /* aka "driver DLL" like in_AudioIO.dll */
    )
   { return iDeviceID;    // it's a DUMMY ID, not one of the standard multimedia device IDs !
     // (don't waste time to compare the name against the enumerated devices below)
   }
  i = strlen(pszNameOfAudioDevice);
  if( i>4 && strnicmp(pszNameOfAudioDevice+i-4,".dll",4)==0 )
   { return C_SOUND_DEVICE_AUDIO_IO; /* pseudo "device number" for DL4YHF's AudioIO-DLLs & I2PHD's ExtIO-DLLs */
   }

  //--------- check all available AUDIO INPUT DEVICES .... -----------------
  // Since 2014-12-27, Sound_InputDeviceNameToDeviceID() is a bit more tolerant
  //  than it used to be in older versions because certain AUDIO DEVICE NAMES
  //  end with a TRAILING SPACE CHARACTER, for whatever triple-stupid reason.
  //  From an email by Mike, KD7TS :
  // > There does seem to be some issue with the device name/ID
  // > here is a snippet from USBview
  // > It says:
  // > [Loc21]Device connected: USB Composite Device
  //   (...)
  // > idVendor: 0x08BB (Burr-Brown Japan, Ltd.)
  //   (...)
  // > iProduct: 0x02
  // > 0x0409: "USB Audio CODEC "
  // >  This goes on and on about endpoints etc.
  // > But notice the description "USB Audio CODEC " includes white space at the end.
  //
  iBestDeviceID = iSecondBestDeviceID = -1;  // Use the 'default' device if none of the device names matches..
  n_devs = waveInGetNumDevs();
  if( SOUND_fDumpEnumeratedDeviceNames )
   {
#   if( SWI_UTILITY1_INCLUDED )
     UTL_WriteRunLogEntry( "InputDeviceNameToDeviceID('%s') enumerating %d wave-input devices:",
                           (char*)pszNameOfAudioDevice,   (int)n_devs );
#   endif
   }
  for(i=0; i<n_devs; ++i)
   {
    if( waveInGetDevCaps(     i, // UINT uDeviceID
                 &my_waveincaps, // LPWAVEINCAPS pwic,
             sizeof(WAVEINCAPS)) // UINT cbwic
       == MMSYSERR_NOERROR)
     { // The WAVEINCAPS structure describes the capabilities
       // of a waveform-audio input device.
       // Drivers for the standard multi-media system are identified by NUMBERS.
      if( (my_waveincaps.dwFormats &
           (  WAVE_FORMAT_4M16  // must support 44.1 kHz, mono, 16-bit, or ...
            | WAVE_FORMAT_4S16  // 44.1 kHz, stereo, 16 bit, or...
            | WAVE_FORMAT_48M08 // 48 kHz, Mono, 8-bit
            | WAVE_FORMAT_48S08 // 48 kHz, Stereo, 8-bit
            | WAVE_FORMAT_48M16 // 48 kHz, Mono, 16-bit
            | WAVE_FORMAT_48S16 // 48 kHz, Stereo, 16-bit
            | WAVE_FORMAT_96M08 // 96 kHz, Mono, 8-bit
            | WAVE_FORMAT_96S08 // 96 kHz, Stereo, 8-bit
            | WAVE_FORMAT_96M16 // 96 kHz, Mono, 16-bit
            | WAVE_FORMAT_96S16 // 96 kHz, Stereo, 16-bit
            | WAVE_FORMAT_1M16 // 11.025 kHz, mono, 16 bit
            | WAVE_FORMAT_1M08 // 11.025 kHz, mono, 8 bit (lower end of the scale)
           )  )
       // Note: the ugly "WiNRADIO Virtual Soundcard" thingy reported dwFormats=0,
       //       so accept the device also if the "number of channels" is non-zero:
          || (my_waveincaps.wChannels > 0 )  // added 2007-03-06 for "VSC" which doesn't report ANY "supported format" (!)
        )
       {
        if( SOUND_fDumpEnumeratedDeviceNames )
         {
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "  Device #%d is '%s', formats=0x%08lX, ok",
             (int)i, (char*)my_waveincaps.szPname, (long)my_waveincaps.dwFormats );
#         endif
         }
        // Seen in my_waveincaps.szPname :
        //   "Stereomix (Realtek High Definit" (note the lousy truncation !),
        //   "CABLE output (VB-Audio Virtual " (note the lousy trunction),
        //   "Mikrofon (Realtek High Definiti" (note the.. oh well... )
        //   "Mikrofon (2- USB Audio CODEC )"  (note the stupid trailing space)
        if( strcmp( my_waveincaps.szPname, pszAudioDeviceNameWithoutPrefix ) == 0 )
         { // Bingo, this is the matching audio device name (here: INPUT) ..
           // but is it UNIQUE (same device "ID" alias device index) ?
           if( i == iDeviceID )
            { // ex: return i;
              iBestDeviceID = i;
              found_perfect_match = TRUE;
            }
           else
            { if( !found_perfect_match )
               { iBestDeviceID = i;   // may have to use this 'second best' choice later
               }
            }
         }
        else // my_waveincaps.szPname != pszAudioDeviceNameWithoutPrefix
         { // 2014-12-27 : Bei einigen Gerten muss der String-Vergleich sehr "tolerant" sein,
           //   z.B. wegen dmlicher endstndiger Leerzeichen(!) im Device-Namen.
           //   Vergleichen wir (fr den "zweitbesten Match") also nochmal,
           //   OHNE ENDSTNDIGE LEERZEICHEN, und bei der Gelegenheit auch
           //   ohne Beachtung von Gross/Kleinbuchstaben im Device-Namen.
           //   Dieselbe Abscheulichkeit muss unter Anderem (!)
           //     sowohl in c:\cbproj\SoundUtl\Sound.cpp :: Sound_InputDeviceNameToDeviceID()
           //   als auch in c:\cbproj\CfgDlgU.cpp :: CheckAndAddNameToInputDevices()
           //   bercksichtigt werden.
           if( !found_perfect_match )
            { cp1 = my_waveincaps.szPname;
              // Who knows.. maybe one day they will use LEADING spaces, too ?
              while( *cp1==' ' )                // skip stupid leading spaces
               { ++cp1;
               }
              cp2 = cp1 + strlen(cp1) - 1;
              while( (cp2>cp1) && (*cp2==' ') )  // skip stupid trailing spaces
               { --cp2;
               }
              if( strnicmp( cp1, pszAudioDeviceNameWithoutPrefix,
                           search_pattern_length/* number of chars to compare */ ) == 0 )
               { // Our "pattern" matches so far, but check the remaining stuff
                 // in my_waveincaps.szPname. Accept the pattern only if it's
                 // one of the following delimiters:
                 cp1 += search_pattern_length; // skip the compared pattern to check the REST..
                 if( (*cp1=='\0') || (*cp1=='(') || (*cp1==' ') )
                  { iSecondBestDeviceID = i; // may have to use this 'second best' choice later
                  }
               }
            } // end if( !found_perfect_match )

           // Das dmliche "Windows 7" meldet frecherweise
           // nicht einfach nur den Namen des Gertes (hier z.B. "FiFiSDR Soundcard"),
           // sondern garniert den Namen mit irrefhrendem Schrott,
           // so dass unter Windows 7 in den Auswahlboxen
           //  statt (bei XP) "FiFiSDR Soundcard" der folgende SCHWACHSINN angezeigt wird:
           //       "Mikrofon (FiFiSDR Soundcard)" .
           // H ? ? Armes FiFi-SDR, hat man dich doch glatt zu einem dmlichen
           //  'Mikrofon' degradiert. Um diesen Windoofs-typischen Schwachsinn
           //  zumindest teilweise zu umgehen, sucht Spectrum Lab (d.h. sound.cpp)
           //  nun auch nach einem AUSDRUCK IN KLAMMERN, und vergleicht diesen
           //  mit dem gesuchten Gertenamen.
           // Wegen der schwachsinnigerweise abgeschnittenen Namen in my_waveincaps.szPname
           // (z.B. "Stereomix (Realtek High Definit" statt "Stereomix (Realtek High Definition Audio)")
           // knnte die schliessende Klammer allerdings auch fehlen. AU BACKE !
           cp1 = strchr(  my_waveincaps.szPname, '(' );
           cp2 = strrchr( my_waveincaps.szPname, ')' );
           if( cp1!=NULL && cp2!=NULL && cp2>cp1 ) // we're lucky because the name in parentheses is NOT truncated...
            { // The enumerated name could indeed be one of those stupidly 'decorated' names:
              ++cp1;        // skip the opening parenthesis
              *cp2 = '\0';  // isolate the real device name (strip the parenthesis)
              if( strncmp( cp1, pszAudioDeviceNameWithoutPrefix, search_pattern_length ) == 0 )
               { // We only compared <search_pattern_length> characters to ignore
                 // the stupid TRAILING SPACE (after the device name, before the closing parenthesis).
                 // What remains (after the matching pattern) in cp1 is either
                 // '\0', ' ', or ')' - anything else is NOT considered a match:
                 cp1 += search_pattern_length;
                 if( (*cp1=='\0') || (*cp1==' ') || (*cp1==')') )
                  { if( ! found_perfect_match )
                     { iBestDeviceID = i; // use this 'second best' choice later
                       // (i.e. accept "Mikrofon (FiFiSDR Soundcard)" from Vista/Win7
                       //   as well as "FiFiSDR Soundcard" from good old XP;
                       // or "2- USB Audio CODEC" in "Mikrofon (2- USB Audio CODEC)",
                       //    "Mikrofon (2- USB Audio CODEC )"  (note the stupid space), etc.
                       // 2020-09-26 : With the above modification, the stoneage
                       //    'Sound-Input-Utility' finally identified an Icom
                       //    IC-9700 (with stupid device name "2- USB Audio CODEC"
                       //    as numeric wave-audio-device-ID 3 (!), which confirms
                       //    that the "2-" in the device name doesn't have
                       //    got anything to do with the enumeration of audio
                       //    devices. The "2-" may have been added by the USB
                       //    driver itself, to distinguish the IC-7300 from the IC-9700.
                     }
                  }
               }
            } // end if < windoze-7 specific decoration with a stupid prefix, and parenthesis >
           else if( cp1!=NULL && cp2==NULL ) // stupidly truncated name (missing closing parenthesis)
            { // The enumerated name could contain A REST of the stupidly 'decorated' name,
              // e.g. "Stereomix (Realtek High Definit" may be the "Realtek High Definition Audio".
              // Thus this workaround: If there only is an OPENING parenthesis (at cp1),
              // compare the REST of the string, as many characters as we have *after* cp1,
              // and ignore the excessive characters in pszAudioDeviceNameWithoutPrefix :
              ++cp1; // skip the opening parenthesis, leaving only "Realtek High Definit" behind
                     // (truncated by the stupid operating system)
              if( strncmp( cp1, pszAudioDeviceNameWithoutPrefix, strlen(cp1) ) == 0 )
               { iSecondBestDeviceID = i; // use this 'second best' choice later
                    // (i.e. accept "Faselblah (Realtek High Definit" from Win8/10
                    //   as well as "Realtek High Definition Audio" from good old XP )
               }
            } // end if < windoze-7/8/10 specific decoration with a stupid prefix, and parenthesis >
         }
       }
      else
       {
        if( SOUND_fDumpEnumeratedDeviceNames )
         {
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "  Device #%d is '%s', formats=0x%08lX, NOT SL-compatible !",
             (int)i, (char*)my_waveincaps.szPname, (long)my_waveincaps.dwFormats );
#         endif
         }
       }
     }
    else  // waveInGetDevCaps failed on this device :
     {
        if( SOUND_fDumpEnumeratedDeviceNames )
         {
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "InputDeviceNameToDeviceID: Failed to retrieve 'capabilities' for device #%d !",
             (int)i );
#         endif
         }
     }
   } // end for < all windows multimedia audio INPUT devices >

 if( iBestDeviceID == -1 )  // no match found yet ... does the name begin with a PROTOCOL (pseudo-URL) ?
  { if( (strnicmp(pszNameOfAudioDevice,"http://", 7 ) == 0 )
      ||(strnicmp(pszNameOfAudioDevice, "tcp://", 6 ) == 0 )
      ||(strnicmp(pszNameOfAudioDevice, "udp://", 6 ) == 0 )
      ||(strnicmp(pszNameOfAudioDevice, "raw://", 6 ) == 0 ) )
     { return C_SOUND_DEVICE_STEAM;
       // This will cause SL's TConfigDlg::Btn_AudioInControlClick()
       //      to call InputWebstreamPanel_Open() ["Analyse/play audio stream"]
       //      instead of the windows 'Sound' dialog .
     }
  }


 if( SOUND_fDumpEnumeratedDeviceNames )
  {
#  if( SWI_UTILITY1_INCLUDED )
    if( found_perfect_match )
     { UTL_WriteRunLogEntry( "InputDeviceNameToDeviceID: Perfect match for device #%d = '%s' .",
                             (int)iBestDeviceID, (char*)pszAudioDeviceNameWithoutPrefix );
     }
    else
     { UTL_WriteRunLogEntry( "InputDeviceNameToDeviceID: No perfect match; best bet is device #%d = '%s' .",
                             (int)iBestDeviceID, (char*)pszAudioDeviceNameWithoutPrefix );
     }
#  endif // SWI_UTILITY1_INCLUDED ?
  }


  SOUND_fDumpEnumeratedDeviceNames = FALSE; // no need to dump those 'device capabilities' in the debug-run-log again


  if( iBestDeviceID >= 0 )
       return iBestDeviceID;
  else return iSecondBestDeviceID;


} // end Sound_InputDeviceNameToDeviceNumber()


//--------------------------------------------------------------------------
int Sound_OutputDeviceNameToDeviceID( char *pszNameOfAudioDevice )
  // Added 2012-07-06 for Spectrum Lab.
  // For details, see Sound_InputDeviceNameToDeviceID() .
{
 int i, n_devs, iDeviceID, iBestDeviceID;
 BOOL found_perfect_match = FALSE;
 WAVEOUTCAPS my_waveoutcaps;
 char *pszAudioDeviceNameWithoutPrefix;

  pszAudioDeviceNameWithoutPrefix = Sound_GetAudioDeviceNameWithoutPrefix(
                                         pszNameOfAudioDevice, &iDeviceID );

  if(    iDeviceID==C_SOUND_DEVICE_ASIO
      || iDeviceID==C_SOUND_DEVICE_SDR_IQ
      || iDeviceID==C_SOUND_DEVICE_PERSEUS
      || iDeviceID==C_SOUND_DEVICE_AUDIO_IO /* aka "driver DLL" like in_AudioIO.dll */
    )
   { return iDeviceID;    // it's a DUMMY ID, not one of the standard multimedia device IDs !
     // (don't waste time to compare the name against the enumerated devices below)
   }
  i = strlen(pszNameOfAudioDevice);
  if( i>4 && strnicmp(pszNameOfAudioDevice+i-4,".dll",4)==0 )
   { return C_SOUND_DEVICE_AUDIO_IO; /* pseudo "device number" for DL4YHF's AudioIO-DLLs & I2PHD's ExtIO-DLLs */
   }

  //--------- check all available AUDIO OUTPUT DEVICES .... -----------------
  iBestDeviceID = -1;
  n_devs = waveOutGetNumDevs();
  if( SOUND_fDumpEnumeratedDeviceNames )
   {
#   if( SWI_UTILITY1_INCLUDED )
     UTL_WriteRunLogEntry( "OutputDeviceNameToDeviceID('%s') enumerating %d wave-output devices:",
                           (char*)pszNameOfAudioDevice,  (int)n_devs );
#   endif                           
   }
  for( i=0; i<n_devs; ++i)
   {
    if( waveOutGetDevCaps(    i, // UINT uDeviceID
                &my_waveoutcaps, // LPWAVEOUTCAPS pwoc,
            sizeof(WAVEOUTCAPS)) // UINT cbwoc
       == MMSYSERR_NOERROR)
     { // The WAVEOUTCAPS structure describes the capabilities
       // of a waveform-audio input device.
       //

      if( (my_waveoutcaps.dwFormats  // must support at least one of the following:
           &(  WAVE_FORMAT_4M16 // must support 44.1 kHz, mono, 16-bit, or ...
             | WAVE_FORMAT_4S16 // 44.1 kHz, stereo, 16 bit, or...
#ifdef WAVE_FORMAT_96S16
             | WAVE_FORMAT_96S16 // 96 kHz, stereo, 16 bit (upper end of the scale)
             | WAVE_FORMAT_48S16
#else
#       error "Your windoze header files are severely outdated !"
#endif
             | WAVE_FORMAT_1M16 // 11.025 kHz, mono, 16 bit
             | WAVE_FORMAT_1M08 // 11.025 kHz, mono, 8 bit (lower end of the scale)
           ) )
       // Note: the ugly "WiNRADIO Virtual Soundcard" thingy reported dwFormats=0,
       //       so accept the device also if the "number of channels" is non-zero:
         || (my_waveoutcaps.wChannels > 0 )  // added 2007-03-06 for "VSC"
        )
       {
        if( SOUND_fDumpEnumeratedDeviceNames )
         {
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "  Device #%d is '%s', formats=0x%08lX, ok",
             (int)i, (char*)my_waveoutcaps.szPname, (long)my_waveoutcaps.dwFormats );
#         endif             
         }


        if( strcmp( my_waveoutcaps.szPname, pszAudioDeviceNameWithoutPrefix ) == 0 )
         { // Bingo, this is the matching audio device name (here: INPUT) ..
           // but is it UNIQUE (same device "ID" alias device index) ?
           if( i == iDeviceID )
            { iBestDeviceID = i;
              found_perfect_match = TRUE;
            }
           else
            { if( ! found_perfect_match )
               { iBestDeviceID = i;
               }
            }
         }
       }
      else
       {
        if( SOUND_fDumpEnumeratedDeviceNames )
         {
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "  Device #%d is '%s', formats=0x%08lX, NOT SL-compatible !",
             (int)i, (char*)my_waveoutcaps.szPname, (long)my_waveoutcaps.dwFormats );
#         endif             
         }
       }
     }
    else  // waveOutGetDevCaps failed on this device :
     {
     }
   } // end for < all windows multimedia audio OUTPUT devices >

  if( SOUND_fDumpEnumeratedDeviceNames )
   {
#   if( SWI_UTILITY1_INCLUDED )
     if( found_perfect_match )
      { UTL_WriteRunLogEntry( "OutputDeviceNameToDeviceID: Perfect match for device #%d = '%s' .",
                              (int)iBestDeviceID, (char*)pszAudioDeviceNameWithoutPrefix );
      }
     else
      { UTL_WriteRunLogEntry( "OutputDeviceNameToDeviceID: No perfect match; best bet is device #%d = '%s' .",
                              (int)iBestDeviceID, (char*)pszAudioDeviceNameWithoutPrefix );
      }
#   endif      
   }

  SOUND_fDumpEnumeratedDeviceNames = FALSE; // no need to dump those 'device capabilities' in the debug-run-log again

  return iBestDeviceID;

} // end Sound_OutputDeviceNameToDeviceID()

//--------------------------------------------------------------------------
BOOL Sound_OpenSoundcardInputControlPanel(
         char *pszNameOfAudioDevice ) // [in] reserved for future use
{
  int iResult;
  if( UTL_iWindowsMajorVersion >= 6 )  // verdammt... es ist "Vista" oder spteres Zeug...
   { // Under "Vista" (eeek) and Windows 7, SNDVOL32.EXE is not available.
     // Thanks Jurgen B. for pointing out the following trick for Windows 7:
     // "C:\windows\system32\rundll32.exe" Shell32.dll,Control_RunDLL mmsys.cpl,,1
     DOBINI();  // 2008-05-21: Got stuck here when the "phantom breakpoint" occurred again
     //  DON'T TRY THIS WHEN RUNNING UNDER CONTROL OF BORLAND'S DEBUGGER !
     //  It will frequently kill both the application *AND* the IDE,
     //  Borland C++ Builder will usually crash with the
     //  old familiar 'EEFFACE' bug (debugger exeption)
     //  because the debugger ITSELF has various bugs !
     iResult = (int)ShellExecute(  // See Win32 Programmer's Reference
        APPL_hMainWindow, // HWND hwnd = handle to parent window
              NULL,    // LPCTSTR lpOperation = pointer to string that specifies operation to perform
       "rundll32.exe", // LPCTSTR lpFile = pointer to filename or folder name string
       "Shell32.dll,Control_RunDLL mmsys.cpl,,1", // LPCTSTR lpParameters = pointer to string that specifies parameters
              NULL,    // LPCTSTR lpDirectory = pointer to string that specifies default directory
         SW_SHOWNORMAL); // .. whether file is shown when opened
        DOBINI();
     // When trying this under Borland C++ Builder Debugger Control,
     // the program sometimes went CRASH, BANG, BOOM with yet another cryptic
     // error message, something like "External Exception EEFFACE" . WTF ??!?
     // It's a Delphi / VCL thing, so the SL author decided not to waste time on this.

   }
  else
   { // Under Windows 98, NT, 2000, XP, the trusty old SNDVOL32.EXE did the job,
     //  with the command line parameter "-r" for RECORD:
     DOBINI();  // 2008-05-21: Got stuck here when the "phantom breakpoint" occurred again
     iResult = (int)ShellExecute(  // See Win32 Programmer's Reference
        APPL_hMainWindow, // HWND hwnd = handle to parent window
              NULL,    // LPCTSTR lpOperation = pointer to string that specifies operation to perform
       "sndvol32.exe", // LPCTSTR lpFile = pointer to filename or folder name string
              "/r",    // LPCTSTR lpParameters = pointer to string that specifies parameters
              NULL,    // LPCTSTR lpDirectory = pointer to string that specifies default directory
         SW_SHOWNORMAL); // .. whether file is shown when opened
     DOBINI();
   }
  // About ShellExecute():
  // > If the function fails, the return value is an error value
  // > that is less than or equal to 32.
  return (iResult>32);
} // end Sound_OpenSoundcardInputControlPanel()

//--------------------------------------------------------------------------
BOOL Sound_OpenSoundcardOutputControlPanel(
         char *pszNameOfAudioDevice ) // [in] reserved for future use
{
  int iResult;
  if( UTL_iWindowsMajorVersion >= 6 )  // verdammt... es ist "Vista" oder spteres Zeug...
   { // Under "Vista" (eeek) and Windows 7, SNDVOL32.EXE is not available.
     DOBINI();
     iResult = (int)ShellExecute(  // See Win32 Programmer's Reference
       APPL_hMainWindow, // HWND hwnd = handle to parent window
              NULL,    // LPCTSTR lpOperation = pointer to string that specifies operation to perform
         "sndvol.exe", // LPCTSTR lpFile = pointer to filename or folder name string
                "",    // LPCTSTR lpParameters = pointer to string that specifies parameters
              NULL,    // LPCTSTR lpDirectory = pointer to string that specifies default directory
       SW_SHOWNORMAL); // .. whether file is shown when opened
     DOBINI();
   }
  else  // Win98, ME, NT, XP :
   {
      DOBINI();
      iResult = (int)ShellExecute(  // See Win32 Programmer's Reference
       APPL_hMainWindow, // HWND hwnd = handle to parent window
              NULL,    // LPCTSTR lpOperation = pointer to string that specifies operation to perform
       "sndvol32.exe", // LPCTSTR lpFile = pointer to filename or folder name string
                "",    // LPCTSTR lpParameters = pointer to string that specifies parameters
              NULL,    // LPCTSTR lpDirectory = pointer to string that specifies default directory
       SW_SHOWNORMAL); // .. whether file is shown when opened
     DOBINI();
   }
  return (iResult>32);
} // end Sound_OpenSoundcardOutputControlPanel()


//---------------------------------------------------------------------------
int Sound_ParseSampleFormatSpecifier( BYTE **ppbSource )
  // Checks for any of the "sample format specifiers" (like "U8") in the source string,
  // skips it, and returns the equivalent code ( SOUND_SAMPLE_FMT_ ... )  .
  // Called, for example, from CSound::InOpen(),
  //        if the input device appears to be a "COM port"
  //        (e.g. UART, RS232, or maybe USB/VCP) .
{
  BYTE *pb = *ppbSource;  // don't use "char" because the default 'char' type is signed, which completely sucks.
  BYTE *pb2;
  int i=0,len=0;
  int iResult = SOUND_SAMPLE_FMT_UNKNOWN;

  UTL_SkipSpaces( &pb );
  while( (pb[len]>='A' && pb[len]<='Z') || (pb[len]>='a' && pb[len]<='z') || pb[len]=='_' || (pb[len]>='0' && pb[len]<='9') )
   { ++len;  // count the number of characters in the token (only letters and digits, nothing else)
   }
  while( SoundSampleFormats[i].pszToken != NULL )
   { if( strncmp( SoundSampleFormats[i].pszToken, (char*)pb, len) == 0 )
      { iResult = SoundSampleFormats[i].iSoundSampleFormat;  // success
        // Since 2015-10-15, the optional 'frame sync' option (,SYNC)
        // is considered part of the 'sample format specifier' so parse it HERE:
        pb2 = pb+len;
        if( UTL_SkipSpaces( &pb2 )==',' )
         { ++pb2;
           UTL_SkipSpaces( &pb2 );
           if( UTL_CheckAndSkipToken( &pb2, "SYNC") )
            { iResult |= SOUND_SAMPLE_FMT_SYNC_PATTERN;
              pb = pb2;
            }
         }
        *ppbSource = pb;
        break;
      }
     ++i;
   }


  // *ppbSource = pb;  // skip the unknown token ? better don't !
  return iResult;
} // end Sound_ParseSampleFormatSpecifier()


#if( SWI_AUDIO_IO_SUPPORTED )  // only if the 'audio-I/O-host' is supported..
//--------------------------------------------------------------------------
int ConvertErrorCode_AIO_to_SOUND( int iAIOErrorCode )
  // Converts an error code from DL4YHF's Audio-I/O-Library (see AudioIO.h)
  // into the closest match for the CSound class, defined in ?\SoundUtl\ErrorCodes.h .
{
  switch( iAIOErrorCode )
   { // "Audio-I/O" error codes from \cbproj\AudioIO\AudioIO.h :
     case AIO_GENERAL_ERROR            : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_DLL_NOT_LOADED     : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_DLL_NOT_COMPATIBLE : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_NOT_IMPLEMENTED    : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_INVALID_PARAMETER  : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_OUT_OF_RESOURCES   : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_FORMAT_NOT_SUPPORTED : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_SAMPRATE_NOT_SUPPORTED: return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_FILE_NOT_FOUND     : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_INPUT_NOT_OPEN     : return SOUNDIN_ERR_NOTOPEN;
     case AIO_ERROR_OUTPUT_NOT_OPEN    : return SOUNDOUT_ERR_NOTOPEN;
     case AIO_INVALID_HANDLE           : return ERROR_ILLEGAL_CMD_PARAMETER;
     case AIO_ERROR_CANCELLED          : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_CANNOT_WRITE_YET   : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_CANNOT_READ_YET    : return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_NOT_INITIALIZED    : return ERROR_FROM_AUDIO_IO_DLL;
     default: if( iAIOErrorCode<0 )      return ERROR_FROM_AUDIO_IO_DLL;
              else                       return NO_ERRORS;
   }
} // end ConvertErrorCode_AIO_to_SOUND()
#endif // ( SWI_AUDIO_IO_SUPPORTED )


//--------------------------------------------------------------------------
void Sound_InterleaveTwoBlocks( T_Float *pfltChannelA, T_Float *pfltChannelB,
                                int     nSamplePoints,
                                T_Float *pfltInterleavedOutput )
{
  while( nSamplePoints-- )
   { *pfltInterleavedOutput++  = *pfltChannelA++;
     *pfltInterleavedOutput++  = *pfltChannelB++;
   }
} // end Sound_InterleaveTwoBlocks()

#if( SWI_AUDIO_IO_SUPPORTED )
int Sound_AudioIOErrorToMyErrorCode( int iAudioIOErrorCode )
{
  switch(iAudioIOErrorCode)
   { case AIO_GENERAL_ERROR:  /* use this error code if you don't find anything better below */
     case AIO_ERROR_DLL_NOT_LOADED    :
     case AIO_ERROR_DLL_NOT_COMPATIBLE:
     case AIO_ERROR_NOT_IMPLEMENTED   :
     case AIO_ERROR_INVALID_PARAMETER :
     case AIO_ERROR_OUT_OF_RESOURCES  :
        return ERROR_FROM_AUDIO_IO_DLL;
     case AIO_ERROR_FORMAT_NOT_SUPPORTED  :
     case AIO_ERROR_SAMPRATE_NOT_SUPPORTED:
        return WAVIN_ERR_NOTSUPPORTED;
     case AIO_ERROR_FILE_NOT_FOUND    :
     case AIO_ERROR_INPUT_NOT_OPEN    : return SOUNDIN_ERR_NOTOPEN;
     case AIO_ERROR_OUTPUT_NOT_OPEN   : return SOUNDOUT_ERR_NOTOPEN;
     case AIO_INVALID_HANDLE          :
     case AIO_ERROR_CANCELLED         :
     case AIO_ERROR_CANNOT_WRITE_YET  : return SOUNDOUT_ERR_NOTOPEN;
     case AIO_ERROR_CANNOT_READ_YET   : return SOUNDIN_ERR_NOTOPEN;
     default :  return ERROR_FROM_AUDIO_IO_DLL;
   }
} // end Sound_AudioIOErrorToMyErrorCode()
#endif // ( SWI_AUDIO_IO_SUPPORTED )


// ex: CSound *SoundDev;  /* global instance for both GENERATOR and ANALYSER */

//--------------------------------------------------------------------------
// Construction/Destruction
//--------------------------------------------------------------------------
CSound::CSound()  // constructor
{
  DOBINI();  // -> Sound_iSourceCodeLine
  m_iUsingASIO = 0;  // not using ASIO driver yet
#if( SWI_ASIO_SUPPORTED )
  m_hAsio = C_ASIOWRAP_ILLEGAL_HANDLE;
#endif
#if( SWI_AUDIO_IO_SUPPORTED )
  AIO_Host_InitInstanceData( &aio_In );
  AIO_Host_InitInstanceData( &aio_Out );
#endif

  m_pfltInterleaveBuffer = NULL;  // nothing allocated yet..
  m_i32InterleaveBufferSize = 0;


  m_iInputTimeout_ms = 500; // max. number of milliseconds to wait for input data
  // (once fixed to 500 ms; user-defineable because the ASIO driver for 'Dante' seemed to require more)

  m_InputOpen = m_OutputOpen = m_fWaveInStarted = FALSE;
  m_hwvin     = NULL;
  m_hwvout    = NULL;
  m_hComPort  = INVALID_HANDLE_VALUE;  // not NULL !
  m_InEventHandle  = NULL;             // not INVALID_HANDLE_VALUE !
  m_OutEventHandle = NULL;
  m_ErrorCodeIn  = 0;
  m_ErrorCodeOut = 0;
  m_fFirstCallOfWaveInAddBuffer = TRUE; // only for debugging
  m_sz255LastErrorStringIn[0] = 0;
  m_sz255LastErrorStringOut[0]= 0;
  m_LostBuffersIn = 0;    // nothing "lost" yet
  m_LostBuffersOut= 0;
  m_nTimingBreaks = 0;
  m_NumInputBuffers= 0;   // explained in C:\cbproj\SoundUtl\Sound.h
  m_pbInputBuffer  = NULL;      // no input buffer(s) allocated yet
  m_pbOutputBuffer = NULL;      // no output buffer(s) allocated yet
  m_dblOutputLatency_sec = 0.0;
  m_dwInputSelector  = 0x00003; // channel selection matrix for INPUT (ASIO only)
  m_dwOutputSelector = 0x00003; // channel selection matrix for OUTPUT (ASIO only)

  memset(&m_OutFormat, 0, sizeof(WAVEFORMATEX) );
  memset(&m_InFormat , 0, sizeof(WAVEFORMATEX) );

# if(sizeof(WAVEFORMATEX) != 18 )
#   error "This stupid compiler seems to have a problem with struct alignment"
  // For obscure reasons, this check failed when compiling the same project
  // on a 'new' PC, with a fresh installed C++ Builder V6 .
  // Cure: Use a MODIFIED ?\cbuilder6\include\mmsystem.h  for Borland C++ ,
  //       with something like the following before the definition of tWAVEFORMATEX
  // force 2-byte alignment, the "Borland" way :
  //  #ifdef __BORLANDC__
  //  # pragma pack(push,2)
  //  #endif // __BORLANDC__
# endif // (sizeof(WAVEFORMATEX) != 18 ) ?


#if( CSOUND_USE_CRITICAL_SECTION )
  memset(&m_CriticalSection, 0 , sizeof(m_CriticalSection) ); // what's this "LockCount"-thingy,
    // and why does it reach incredibly large values (like 0x81D4609C) ?
  InitializeCriticalSection(&m_CriticalSection);
  // Note: A critical section is one of the fastest synchronisation objects,
  //       because it doesn't require a kernel mode switch .
  //       ( -> http://msdn.microsoft.com/en-us/library/ms810428.aspx )
#endif // CSOUND_USE_CRITICAL_SECTION

  DOBINI();  // -> Sound_iSourceCodeLine
}

CSound::~CSound()  // destructor ..  clean up POLITELY (if possible)
{
  DOBINI();  // -> Sound_iSourceCodeLine

  // Some caution must be taken if the buffers are still in use at this moment:
  if(m_InputOpen)
   { InClose();  // <- WAITS for a few milliseconds, if the buffers are still in use in another thread !
   }
  if(m_OutputOpen)
   { OutClose();
   }
#if( SWI_ASIO_SUPPORTED )
  if( m_hAsio != C_ASIOWRAP_ILLEGAL_HANDLE )
   { AsioWrap_Close( m_hAsio );
   }
#endif // SWI_ASIO_SUPPORTED
#if( SWI_AUDIO_IO_SUPPORTED )
  AIO_Host_FreeDLL( &aio_In );
  AIO_Host_FreeDLL( &aio_Out );
#endif // SWI_AUDIO_IO_SUPPORTED
  if(m_pbInputBuffer)      // free buffers ...
   { m_NumInputBuffers = m_InBufferSize = 0;
     UTL_free(m_pbInputBuffer);
     m_pbInputBuffer = NULL;
   }
  if(m_pbOutputBuffer)
   { m_NumOutputBuffers = m_OutBufferSize = 0;
     UTL_free(m_pbOutputBuffer);
     m_pbOutputBuffer = NULL;
   }
  m_i32InterleaveBufferSize = 0;
  if(m_pfltInterleaveBuffer)
   { UTL_free(m_pfltInterleaveBuffer);
     m_pfltInterleaveBuffer = NULL;
   }

#if( SWI_AUDIO_IO_SUPPORTED )
  AIO_Host_DeleteInstanceData( &aio_In ); // counterpart to AIO_Host_InitInstanceData()
  AIO_Host_DeleteInstanceData( &aio_Out);
#endif

  if(m_InEventHandle != NULL)
   { CloseHandle(m_InEventHandle);
     m_InEventHandle = NULL;    // only deleted here in the destructor !
   }
  if(m_OutEventHandle != NULL)
   { CloseHandle(m_OutEventHandle);
     m_OutEventHandle = NULL;   // only deleted here in the destructor !
   }


#if( CSOUND_USE_CRITICAL_SECTION )
  DeleteCriticalSection(&m_CriticalSection);
#endif // ( CSOUND_USE_CRITICAL_SECTION )
  DOBINI();  // -> Sound_iSourceCodeLine
} // end CSound::~CSound() [Dr. Destructo]

//--------------------------------------------------------------------------
void CSound::SetInputSelector( DWORD dwInputSelector )
  // Only works for ASIO drivers.
  // Example: dwInputChannelMask=0x0003 selects the first and the second
  //          channel for input.
{
   m_dwInputSelector = dwInputSelector;
} // end CSound::SetInputSelector()

//--------------------------------------------------------------------------
void CSound::SetOutputSelector( DWORD dwOutputSelector )
{ // similar as above, but for OUTPUT channels
   m_dwOutputSelector = dwOutputSelector;
}

//--------------------------------------------------------------------------
void CSound::SetInputTimeout( int iInputTimeout_ms )
  // Sets the max. number of milliseconds to wait for input data
  //  This was once fixed to 500 ms; now user-defineable because
  //  the ASIO driver for 'Dante' seemed to require more than that.
  // 2015-01-29 : Still, the E-MU 0202 occassionally 'timed out on input'
  //              when running at 192 kS/s, and 8 buffers * 16384 bytes.
{
  if( iInputTimeout_ms < 100 )
   {  iInputTimeout_ms = 100;
   }
  if( iInputTimeout_ms > 5000 )
   {  iInputTimeout_ms = 5000;
   }
  m_iInputTimeout_ms = iInputTimeout_ms;
}

//--------------------------------------------------------------------------
// VFO control (added HERE 2012-08-14, because some ExtIO-DLLs need this):
//--------------------------------------------------------------------------
BOOL CSound::HasVFO(void) // -> TRUE=VFO supported, FALSE=no. Used for ExtIO-DLLs.
{
#if( SWI_AUDIO_IO_SUPPORTED )
  return AIO_Host_HasVFO( &aio_In );
#else
  return FALSE;
#endif // ( SWI_AUDIO_IO_SUPPORTED ) ?
}

//--------------------------------------------------------------------------
long CSound::GetVFOFrequency(void) // -> current VFO tuning frequency in Hertz.
{
#if( SWI_AUDIO_IO_SUPPORTED )
  return AIO_Host_GetVFOFrequency( &aio_In );
#else
  return 0;
#endif // ( SWI_AUDIO_IO_SUPPORTED ) ?

}

//--------------------------------------------------------------------------
long CSound::SetVFOFrequency( // Return value explained in AudioIO.c (!)
              long i32VFOFrequency ) // [in] new VFO frequency in Hertz
{
#if( SWI_AUDIO_IO_SUPPORTED )
  m_i32VFOFrequency = i32VFOFrequency;  // may be required when "Starting" !
  return AIO_Host_SetVFOFrequency( &aio_In, i32VFOFrequency );
#else
  return 0;
#endif // ( SWI_AUDIO_IO_SUPPORTED ) ?
}

#if( SWI_ASIO_SUPPORTED )
//--------------------------------------------------------------------------
AsioHandle CSound::GetAsioHandle(void)
{ // Just a kludge to retrieve more info about "driver-in-use" .
  // See configuration dialog in Spectrum Lab for a very ugly example.
  return m_hAsio;   // handle for DL4YHF's ASIO wrapper
}
#endif // SWI_ASIO_SUPPORTED

//--------------------------------------------------------------------------
UINT CSound::InOpen(  // In Spectrum Lab, called from SoundThd_InitializeAudioDevices() ..
           char * pszAudioInputDeviceOrDriver, // [in] name of an audio device, COM port, or DL4YHF's "Audio I/O driver DLL" (full path)
           char * pszAudioInputDriverParamStr, // [in] extra parameter string for driver or Audio-I/O / ExtIO-DLL
                                               //      optional : may be NULL.
           char * pszAudioStreamID,   // [in,optional] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE
           char * pszDescription,     // [in,optional a descriptive name which identifies
           // the audio sink ("audio reader", or "consumer"). This name may appear
           // on the "patch panel" on the configuration screen of the Audio-I/O DLL .
           // For example, the 1st running instance of Spectrum Lab will
           //     identify itself as "SpectrumLab1", the 2nd as "SpectrumLab2", etc .
           long i32SamplesPerSecond, // [in] "wanted" nominal sample rate - sometimes ignored !
           int  iBitsPerSample,     // ..per channel, usually 16 (also for stereo, etc)
           int  iNumberOfChannels,  // 1 or 2 channels (mono/stereo)
           DWORD dwMinBufSize,      // number of SAMPLES per buffer expected by the CALLER .
                 // (must be at least as large as the 'processing chunk size';
                 //  see caller in SoundThd.cpp : SoundThd_InitializeAudioDevices .
                 //  Internally, the total length to communicate with the audio driver
                 //  will always be "a bit larger" to reduce the risk of missing samples)
           DWORD SampleLimit,       // use 0 (zero) for a continuous stream
           BOOL  start)
//  Opens a Soundcard for input, using MMSYSTEM or ASIO. Sample size can be
//  1,2 or 4 bytes long depending on bits per sample and number of channels.
//( ie. a 1000 sample buffer for 16 bit stereo will be a 4000 byte buffer)
//   Sampling begins immediately so the thread must start calling
//   one of the "Read"-functions to prevent buffer overflow.
//
//  Parameters:
//      pszAudioInputDeviceOrDriver = name of an AUDIO- or ASIO-device,
//                  or AUDIO-I/O-driver.  See examples further below.
//                  Also see notes in Sound_GetAudioDeviceNameWithoutPrefix().
//      i32SamplesPerSecond = sampling rate in Hertz .
//                  this parameter is sometimes ignored, for example
//                  if the 'input device or driver' is actually an ExtIO-DLL
//                  connected to a certain hardware, which only supports a fixed
//                  sampling rate (which may have got nothing in common with the
//                  standard soundcard sampling rates). To be prepared for such cases,
//                  the application should call CSound::InGetNominalSamplesPerSecond()
//                  after (!) CSound::InOpen() .
//      dwMinBufSize   = DWORD specifies the soundcard buffer size number
//                  in number of samples to buffer.
//                  If this value is Zero, the soundcard is opened and
//                  then closed. This is useful   for checking the
//                  soundcard.
//      SampleLimit = limit on number of samples to be read. 0 signifies
//                  continuous input stream.
//      start:  FALSE=only open input, but don't start
//              TRUE =open input and start sampling immediately.
//   Return value:
//        0           if opened OK
//        ErrorCode   if not
//
// Examples for pszAudioInputDeviceOrDriver :
//   "4 E-MU 0202 | USB"  (device ID and name of a windows multimedia device)
//
// IMPORTANT: For the ExtIO-DLLs, the radio's VFO FREQUENCY(!) must be set
//  BEFORE opening the "driver" for input, because for some reason,
//  the VFO frequency must be passed as a function argument in "StartHW" !
//  (failing to do so caused ExtIO_RTL.DLL to open a stupid message box
//   telling something about "PLL not locked", instead of silently returning
//   with an error code, and leaving the user interaction to the HOST) .
// Thus, call CSound::SetVFOFrequency() with a VALID tuning frequency,
//   to prevent the annoying message box (from the DLL) from popping up.
//   Details in AudioIO.c :: AIO_Host_OpenAudioInput() .
{
 int i;
 int newInBufferSize/*per "part"*/, newNumInputBuffers;
 long i32TotalBufSizeInByte;
 DWORD dw;
 char sz255[256]; //, *cp;
 char *pszAudioDeviceName = pszAudioInputDeviceOrDriver;
 char *pszAudioDeviceNameWithoutPrefix;
 BYTE *pb;
#if( SWI_ASIO_SUPPORTED )      // Support ASIO audio drivers too ?
  ASIOChannelInfo *pAsioChannelInfo;
#endif
  int iErrorCode;     // error code, ususally from calling the dreadful 'wave' API
#if( SWI_AUDIO_IO_SUPPORTED )
  int iAIOErrorCode;  // negative error codes from AudioIO.h (for audio-I/O-DLLs)
#endif
  DCB dcb; // esoteric, and most likely Windoze-specific way to configure a serial port
  int iBitsPerSecond, iUartDataBits=8, iParity=-1, iStopbits=1;

  double dblRealizedSampleRate;

  DOBINI();  // -> Sound_iSourceCodeLine
  if(m_InputOpen)
   {  InClose();
   }
  if( m_hComPort != INVALID_HANDLE_VALUE )
   { CloseHandle( m_hComPort );
     m_hComPort  =  INVALID_HANDLE_VALUE;
   }
  m_hwvin = NULL;    // forget handle for MMSYSTEM "wave-input"
  if( m_InputOpen && m_InWaiting ) // oops.. buffers may still be occupied...
   { m_InputOpen = FALSE;  // politely ask whoever-uses-this to release the buffer(s)
     for( i=0; i<5; ++i )
      { Sleep(20);
        if( ! m_InWaiting ) // "the other guy" has returned, and released the buffer(s)
         { break;
         }
        if( SndThd_fCloseDevicesDuringSleep ) // received WM_POWERBROADCAST / PBT_APMSUSPEND somewhere in the GUI !
         { // instead of fooling around with "WaitForMultipleObjects" and other windows-specific crap,
           // bail out of this busy-spinning loop (and many others)
           // as soon as recognizing the above flag
           break;
         }
      }
   } // end if < input-buffer(s) POSSIBLY occupied (in another thread) >
  m_InputOpen  = FALSE;  // since 2015-10, this is also a FLAG to break from waiting-loops (in other threads) before resizing buffers, etc
  m_InWaiting  = FALSE;  // since 2015-10, this flag is used for a kind-of 'handshake' in combination with m_InputOpen
  m_fFirstCallOfWaveInAddBuffer = TRUE; // only for debugging
  if( m_InEventHandle != NULL )
    { ResetEvent(m_InEventHandle); // let WaitForInputData() wait when it needs to (!)
    }
  m_InOverflow = FALSE;
  m_iInHeadIndex  = 0;  // index for ENTERING new samples into buffer's "head"
  m_iInTailIndex  = 0;  // index for TAKING OUT samples from the buffer's "tail"
#if( SWI_AUDIO_IO_SUPPORTED )
  m_ExtIONeedsSoundcardForInput = FALSE;
#endif
  m_InBufPosition = 0;
  memset( m_InBufInfo, 0, sizeof(m_InBufInfo) );
  for(i=0; i<SOUND_MAX_INPUT_BUFFERS; ++i)
   { m_InBufInfo[i].iBufferIndex = i;
   }

  m_i64NumSamplesRead = 0;   // number of samples read by the application
  m_ldblSampleBasedInputTimestamp = 0.0; // input timestamp still unknown
  m_dblNominalInputSampleRate  // initial values for the sampling rate..
    = dblRealizedSampleRate = i32SamplesPerSecond; // initial value for the 'measured' SR (MUST ALWAYS BE VALID)

  // ex: m_InFormat = *pWFX; // copy a lot of params from WAVEFORMATEX struct
  // WoBu didn't want to have this damned "waveformatex"-stuff in the interface,
  // so got rid of this micro$oft-specific stuff, and only use it internally :
  m_InFormat.wFormatTag = WAVE_FORMAT_PCM;
     // Since 2003-12-14, the "number of channels" for the ADC is not necessarily
     //       the same as the number of channels in the internal process !
  m_InFormat.nChannels = (WORD)iNumberOfChannels;   // only 1 or 2 so far, regardless of interleaved/non-interleaved output later
  m_InFormat.wBitsPerSample = (WORD)iBitsPerSample;  // ex 16
  m_InFormat.nSamplesPerSec = i32SamplesPerSecond;  // <- "uncalibrated", "nominal" sampling rate !
     // note that the WAVEFORMATEX-thingy require the "uncalibrated" sample rate !!
  m_InFormat.nBlockAlign = (WORD)( m_InFormat.nChannels *(m_InFormat.wBitsPerSample/8) );
     // In a comment in MMREG.H, Microsoft says 'nBlockAlign' is the "block size of data" .
     // What a stupidly ambiguous comment. What they mean is "bytes per sample block",
     // for example nBlockAlign = 6 [byte] for a stereo stream with 24 "bits per single-channel sample".
  m_InFormat.nAvgBytesPerSec = m_InFormat.nSamplesPerSec * m_InFormat.nBlockAlign;
  m_InFormat.cbSize = 0/*!*/; // no "extra format information" appended to the end of the WAVEFORMATEX structure
  m_InBytesPerSamplePoint = (m_InFormat.wBitsPerSample/8)*m_InFormat.nChannels;  // for ASIO, this is a WISH (!)

  m_sz255LastErrorStringIn[0] = 0;

  if( CountBits(m_dwInputSelector) < m_InFormat.nChannels )
   { m_dwInputSelector = ( 0xFFFFFFFF >> (32-m_InFormat.nChannels) );
   }

  pszAudioDeviceNameWithoutPrefix = Sound_GetAudioDeviceNameWithoutPrefix(
                             pszAudioInputDeviceOrDriver, &m_iInputDeviceID );
  // Sound_GetAudioDeviceNameWithoutPrefix() also recognizes 'special names'
  //       like serial ports (COM1, COM2, COM3), and in such cases sets
  //       m_iInputDeviceID to something like  C_SOUND_DEVICE_COM_PORT .
  // m_sz255AudioInputDeviceOrDriver will later be the name of the REALLY OPENED
  // audio device.  The following is just an "initial guess" :
  strncpy( m_sz255AudioInputDeviceOrDriver, pszAudioDeviceNameWithoutPrefix, 255 );
  // Note: At this point, m_iInputDeviceID is simply the decimal number
  //       parsed from the begin of the device name in Sound_GetAudioDeviceNameWithoutPrefix() .
  //       Sound_InputDeviceNameToDeviceID() does a more thorough job:
  //       It ENUMERATES the available wave input devices (using a windows API function),
  //       and compares the *DEVICES NAMES* with the audio device name:
  SOUND_fDumpEnumeratedDeviceNames = TRUE;  // flag for Sound_InputDeviceNameToDeviceID()
  // later: iSoundInputDeviceID = Sound_InputDeviceNameToDeviceID( pszAudioInputDeviceOrDriver );



#if( SWI_AUDIO_IO_SUPPORTED )
  // Use DL4YHF's  "Audio-IO" for input ?
  if( m_iInputDeviceID == C_SOUND_DEVICE_AUDIO_IO )
   {
#if( SWI_AUDIO_IO_SUPPORTED )
     if( aio_In.h_AudioIoDLL == NULL )
      {
#      if( SWI_UTILITY1_INCLUDED )
        UTL_WriteRunLogEntry( "CSound::InOpen: Trying to load DLL \"%s\"",
                              (char*)pszAudioInputDeviceOrDriver );
#      endif                              
        iAIOErrorCode = AIO_Host_LoadDLL( &aio_In, pszAudioInputDeviceOrDriver );
        if( iAIOErrorCode < AIO_NO_ERROR ) // error codes from AudioIO.h are NEGATIVE !
         {
           SetInputError( ConvertErrorCode_AIO_to_SOUND(iAIOErrorCode) ); // -> m_ErrorCodeIn
           sprintf( m_sz255LastErrorStringIn, "Failed to load %s", pszAudioInputDeviceOrDriver );
           DOBINI();
           return m_ErrorCodeIn;
         }
       }
      // Problem with some ExtIO-DLLs (in AIO_Host_OpenAudioInput) :
      //  Some of these drivers need to send the VFO TUNING FREQUENCY
      //  to the 'radio' (RTL-SDR) before starting [in StartHW()], so that info must be
      //  passed to the ExtIO-DLL-host *before* calling AIO_Host_OpenAudioInput().
      // For other kinds of 'radio control drivers', it works the other way round:
      //  FIRST open the driver / "start the hardware", THEN set the VFO frequeny. Holy shit.
      AIO_Host_SetVFOFrequency( &aio_In, // set VFO tuning frequency in Hertz.
                m_i32VFOFrequency); // must have been already set because of trouble with StartHW() in certain ExtIO-DLLs !
      if( AIO_Host_OpenAudioInput( // implemented in C:\cbproj\AudioIO\AudioIO.c
                &aio_In,           // [in,out] DLL-host instance data for the INPUT
                pszAudioStreamID,  // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE (here: optional)
                pszDescription,    // [in] a descriptive name which identifies
                i32SamplesPerSecond,     // [in(!)] "wanted" sample rate, often IGNORED (reasons in AudioIO.c..)
                &dblRealizedSampleRate,  // [out(!)] realized samples per second, dictated by hardware or driver
                iNumberOfChannels,   // [in] number of channels IN EACH SAMPLE-POINT
                200,                 // [in] iTimeout_ms, max timeout in milliseconds for THIS call
                     // (may have to wait here for a short time, for example when...
                     //   - the I/O lib needs to communicate with an external device (to open it)
                     //   - the I/O lib needs to wait until the 'writer' (=the other side)
                     //     has called AIO_Host_OpenAudioOutput() and set the sampling rate)
                AIO_OPEN_OPTION_NORMAL, // [in] bit combination of AIO_OPEN_OPTION_... flags
                NULL // [out,optional] T_ChunkInfo *pOutChunkInfo, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL (!)
            ) < 0 )
       {   // failed to open the AUDIO-I/O-DLL for "input" :
           SetInputError( SOUNDIN_ERR_NOTOPEN ); // -> m_ErrorCodeIn
           strcpy( m_sz255LastErrorStringIn, "Couldn't open AUDIO_IO for input" );
           DOBINI();
           return m_ErrorCodeIn;
       }
      else // successfully opened the AUDIO-I/O-DLL for "input"... but that's not all:
       {
           // 2016-05-27 : With ExtIO_perseus.dll, got here with
           //              dblRealizedSampleRate = 125000.0, even though we "wanted" 250 kS/sec .
           //        and   aio_In.i32ExtIOHWtype = 5 = EXTIO_HWT_USB_Int24 .
           if( dblRealizedSampleRate > 0.0 )
            {  m_InFormat.nSamplesPerSec = (int)(dblRealizedSampleRate+0.49); // kludge for InGetNominalSamplesPerSecond()
            }
           m_InputOpen = TRUE;  // Note: m_InputOpen only becomes true AFTER buffer allocation was ok
           DOBINI();  // -> Sound_iSourceCodeLine
           // Added 2012-10-28 to support FiFi-SDR: Some ExtIO-DLLs (ExtIO_Si570.dll)
           //  rely on the ordinary soundcard as an input devices. Check this:
           if( aio_In.i32ExtIOHWtype == EXTIO_HWT_Soundcard /*4*/ )
            { // > Theaudiodataarereturnedviathesoundcardmanagedbythe host.
              // > TheexternalhardwarejustcontrolstheLO,
              // > andpossiblyapreselector,underDLLcontrol.
              // In this case, Sound.cpp + AudioIO.c are the 'host' (for the DLL),
              // so the ugly soundcard processing shall be performed here.
              // In Spectrum Lab, there is already a software module for the
              // soundcard-input processing (Sound.cpp) so we keep the soundcard-stuff
              // out of the 'DLL host'. Instead, the soundcard processing remains where it was:
              // in C:\cbproj\SoundUtl\Sound.cpp : CSound::InOpen() .
              m_ExtIONeedsSoundcardForInput = TRUE;  // using ExtIO to control the radio, but the soundcard to acquire samples
              // Because in this case, 'pszAudioInputDeviceOrDriver' is occupied
              // by the NAME OF THE ExtIO-DLL, it cannot be used (further below)
              // as the NAME OF THE AUDIO INPUT DEVICE aka "soundcard" device.
              // Kludge: Pass the name of the soundcard in the 'extra parameter string':
              pszAudioDeviceName = NULL;
              if( pszAudioInputDriverParamStr != NULL )
               { if( pszAudioInputDriverParamStr[0] != '\0' )
                  { pszAudioDeviceName = pszAudioInputDriverParamStr;
#                  if( SWI_UTILITY1_INCLUDED )
                    UTL_WriteRunLogEntry( "CSound::InOpen: Using ExtIO with soundcard, device name: %s",
                                          (char*)pszAudioDeviceName );
#                  endif
                  }
               }
              if( pszAudioDeviceName==NULL )
               { // Name of the SOUNDCARD device not specified (after the name of the ExtIO-DLL) .
                 // Make an educated guess :
                 if( stricmp( aio_In.extHWname, "SoftRock Si570") == 0 )  // ExtIO used for the FiFi-SDR = "ExtIO_Si570.dll"
                  { pszAudioDeviceName = "FiFiSDR Soundcard"; // name of FiFi-SDR's internal(!) soundcard
                    // Called much further below, Sound_InputDeviceNameToDeviceID()
                    // will find the windoze multimedia / "wave audio" device ID
                    // for this name. Storing the number anywhere is stupid,
                    // because it changes from day to day, depending on the USB
                    // port to which the FiFi-SDR is plugged.
                  }
                 else
                  { pszAudioDeviceName = "-1";  // use the standard 'wave input' device
                  }
#               if( SWI_UTILITY1_INCLUDED )
                 UTL_WriteRunLogEntry( "CSound::InOpen: Using ExtIO with soundcard, guessed device name: %s",
                              (char*)pszAudioDeviceName );
#               endif                              
               }
              pszAudioDeviceNameWithoutPrefix = pszAudioDeviceName;
              // NO 'return' in this case !
            } // end if( pInstData->i32ExtIOHWtype == EXTIO_HWT_Soundcard /*4*/ )
           else // In all other cases, the Audio-I/O or ExtIO-DLL provides the samples:
            { m_ExtIONeedsSoundcardForInput = FALSE;
              return 0;  // "no error" (from CSound::InOpen)
              // In THIS case, do NOT enter the soundcard-part further below
            }
       }
#endif  // SWI_AUDIO_IO_SUPPORTED
   } // end if( m_iInputDeviceID == C_SOUND_DEVICE_AUDIO_IO )
  else // do NOT use the "audio I/O DLL" for input :
   { AIO_Host_FreeDLL( &aio_In ); // ... so free it, if loaded earlier
   }
#endif // SWI_AUDIO_IO_SUPPORTED


  // Use a communications port (serial port, "COM port", virtual COM port, etc) for input ?
  if( m_iInputDeviceID==C_SOUND_DEVICE_COM_PORT )  // here in CSound::InOpen() ...
   { // Parse the COM port number from the device name :
     i = atoi( pszAudioInputDeviceOrDriver + 3/*strlen("COM")*/  );
     if( i<1 || i>255 ) // only COM1...COM255 supported here
      {
        SetInputError( WAVIN_ERR_OPEN ); // -> m_ErrorCodeIn
        sprintf( m_sz255LastErrorStringIn, "Invalid COM port number (%d)", (int) i );
#      if( SWI_UTILITY1_INCLUDED )
        DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR,  // -> also calls UTL_WriteRunLogEntry() !
            0, UTL_USE_CURRENT_TIME,
            "CSound::InOpen: %s", m_sz255LastErrorStringIn );
#      endif            
        return m_ErrorCodeIn;
      }

     // Since 2015-10, the serial baudrate, UART frame format, and sample-block-format
     //  are parsed from the 'params' string as specified in
     //  SpecLab/html/settings.htm#adc_on_serial_port .  Example:
     //    115200,8-N-1,IQ12
     //    |____| |___| |___|
     //      |      |     |
     //      |      |     Audio sample stream format
     //      |      UART "frame" format
     //     serial bitrate ("baudrate")
     pb = (BYTE*)pszAudioInputDriverParamStr;
     iBitsPerSecond = UTL_ParseInteger( &pb, 8/*max digits*/ );
     if( iBitsPerSecond <= 0 )
      {
#      if( SWI_UTILITY1_INCLUDED )
        DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR,  // -> also calls UTL_WriteRunLogEntry() !
            0, UTL_USE_CURRENT_TIME,
            "CSound::InOpen: Bad serial bitrate in parameter string '%s'", pszAudioInputDriverParamStr );
#      endif // SWI_UTILITY1_INCLUDED ?
        SetInputError( ERROR_ILLEGAL_CMD_PARAMETER ); // -> m_ErrorCodeIn
        return m_ErrorCodeIn;
      }
     if( UTL_SkipSpaces( &pb ) == ',' ) // more arguments in the "param"-string ?
      { ++pb;  // skip the comma.
        // Next: UART frame format in the usual notation ("8N1" or "8-N-1")
        iUartDataBits = UTL_ParseInteger( &pb, 8 );
        if( *pb=='-' )
         { ++pb;
         }
        // Next: parity bit ? (N)one, (E)ven, (O)dd ?
        switch( *pb )
         { case 'N' :
           case 'n' :
              iParity = NOPARITY;
              ++pb;
              break;
           case 'E' :
           case 'e' :
              iParity = EVENPARITY;
              ++pb;
              break;
           case 'O' :
           case 'o' :
              iParity = ODDPARITY;
              ++pb;
              break;
           case 'M':
           case 'm':
              iParity = MARKPARITY;
              ++pb;
              break;
           default:
#            if( SWI_UTILITY1_INCLUDED )
              DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR,  // -> also calls UTL_WriteRunLogEntry() !
                 0, UTL_USE_CURRENT_TIME,
                 "CSound::InOpen: Unknown parity bit option in '%s'", pszAudioInputDriverParamStr );
#            endif // SWI_UTILITY1_INCLUDED ?
              SetInputError( ERROR_ILLEGAL_CMD_PARAMETER ); // -> m_ErrorCodeIn
              return m_ErrorCodeIn;
         }
        if( *pb=='-' )
         { ++pb;
         }
        // Next: number of stopbits ? Almost ever, ONE stop-bit
        switch( *pb )
         { case '1' :
              if( UTL_CheckAndSkipToken( &pb, "1.5" ) )
               { iStopbits = ONE5STOPBITS;  // how bizarre.. one AND A HALF stop-bits !
               }
              else
               { iStopbits = ONESTOPBIT;
                 ++pb;
               }
              break;
           case '2' :
              iStopbits    = TWOSTOPBITS;
              ++pb;
              break;
           default:
#            if( SWI_UTILITY1_INCLUDED )
              DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR,  // -> also calls UTL_WriteRunLogEntry() !
                 0, UTL_USE_CURRENT_TIME,
                 "CSound::InOpen: Bad number of stopbits in '%s'", pszAudioInputDriverParamStr );
#            endif                 
              SetInputError( ERROR_ILLEGAL_CMD_PARAMETER ); // -> m_ErrorCodeIn
              return m_ErrorCodeIn;
         }
      } // end if < UART - frame-format-specification in the usual "8N1" notation >

     // Next : specification of the SAMPLE BLOCK FORMAT. Drums please !
     if( UTL_SkipSpaces( &pb ) == ',' ) // more arguments in the "param"-string ?
      { ++pb;  // skip the comma before the optional SAMPLE FORMAT specification
        m_iSoundInputSampleFormat = Sound_ParseSampleFormatSpecifier( &pb );  // "U8", "I16", "IQ12", etc -> SOUND_SAMPLE_FMT_U8, etc
        if( UTL_SkipSpaces( &pb ) == ',' ) // even more arguments in the "param"-string ?
         { ++pb;  // skip the comma before the optional NUMBER OF CHANNELS in EACH SAMPLE-POINT
           m_InFormat.nChannels = UTL_ParseInteger( &pb, 8/*max digits*/ );
         }
        else  // if not explicitly specified, assume the stream contains as many channels
         {    // as the application "wants to read" (beware, such assumptions are evil)
           m_InFormat.nChannels = iNumberOfChannels;
           switch( m_iSoundInputSampleFormat & SOUND_SAMPLE_FMT_MASK )
            { case SOUND_SAMPLE_FMT_IQ12 : // always TWO channels (in the serial stream) ...
                 m_InFormat.nChannels = 2; // 1 channel for 'I' + 1 channel for 'Q'
                 break;
              case SOUND_SAMPLE_FMT_GPSDO3: // header byte with the MSBIT always set; all other bytes in the same have their MSBytes *cleared*
                 m_InFormat.nChannels = 1; // First used by DL4YHF's PIC-based GPSDO for the serial port (2016-03), with only ONE input channel from a PIC16F1783 ...
                 break;

              default:  // for all other sample formats, show a WARNING if the number of channels is not specified:
                 DEBUG_EnterErrorHistory( DEBUG_LEVEL_WARNING,0,UTL_USE_CURRENT_TIME,
                    "CSound::InOpen: Number of channels not specified in '%s'", pszAudioInputDriverParamStr );
                 break;
            }
         }
        m_InBytesPerSamplePoint = 0;  // not set yet .
        // Do we support the specified sample format here (via serial port) ?
        switch( m_iSoundInputSampleFormat & SOUND_SAMPLE_FMT_MASK )
         { // On this occasion, also set m_InFormat.nChannels (and a few other fields in m_InFormat)
           // because the old 'Get'-functions like CSound::InGetNumberOfChannels()
           // assume the ancient 'Waveformatex'-thing is valid even though we don't use it here.
           case SOUND_SAMPLE_FMT_U8      : /* 8 bit per channel (in each sample point), UNSIGNED integer */
           case SOUND_SAMPLE_FMT_S8      : /* 8 bit per channel (in each sample point), UNSIGNED integer */
              m_InFormat.wBitsPerSample = 8;     /* in the WAVEFORMATEX : "number of bits per sample of mono data" */
              break;
           case SOUND_SAMPLE_FMT_U16_LE  : /* 16 bit per channel (in each sample point), UNSIGNED integer, Little Endian (LSByte first) */
           case SOUND_SAMPLE_FMT_S16_LE  : /* 16 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
           case SOUND_SAMPLE_FMT_U16_BE  : /* 16 bit per channel (in each sample point), UNSIGNED integer, Big Endian    (MSByte first) */
           case SOUND_SAMPLE_FMT_S16_BE  : /* 16 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
              m_InFormat.wBitsPerSample = 16;    /* in the WAVEFORMATEX : "number of bits per sample of mono data" */
              break;

           case SOUND_SAMPLE_FMT_U24_LE  : /* 24 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
           case SOUND_SAMPLE_FMT_S24_LE  : /* 24 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
           case SOUND_SAMPLE_FMT_U24_BE  : /* 24 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
           case SOUND_SAMPLE_FMT_S24_BE  : /* 24 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
              m_InFormat.wBitsPerSample = 24;     /* in the WAVEFORMATEX : "number of bits per sample of mono data" */
              break;

           case SOUND_SAMPLE_FMT_U32_LE  : /* 32 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
           case SOUND_SAMPLE_FMT_S32_LE  : /* 32 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
           case SOUND_SAMPLE_FMT_U32_BE  : /* 32 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
           case SOUND_SAMPLE_FMT_S32_BE  : /* 32 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
              m_InFormat.wBitsPerSample = 32;     /* in the WAVEFORMATEX : "number of bits per sample of mono data" */
              break;

           case SOUND_SAMPLE_FMT_F32_LE  : /* 32 bit per channel (in each sample point), FLOATING POINT, for Intel CPU+FPU*/
           case SOUND_SAMPLE_FMT_F32_BE  : /* 32 bit per channel (in each sample point), FLOATING POINT, for we-don't-know*/
              m_InFormat.wBitsPerSample = 32;     /* in the WAVEFORMATEX : "number of bits per sample of mono data" */
              break;

           case SOUND_SAMPLE_FMT_OGG     : /* Ogg containers, hopefully with Vorbis-compressed audio */
              m_InBytesPerSamplePoint = 64; // need at least 64 bytes in the buffer to find out what the stream actually contains
              break;

           case SOUND_SAMPLE_FMT_IQ12    : /* header byte + 2 * 12 bit unsigned, I/Q format used by DL4YHF's PIC12F675 ADC for the serial port */
              m_InBytesPerSamplePoint = 4;
              break;
           case SOUND_SAMPLE_FMT_GPSDO3  : /* header byte with the MSBIT always set; all other bytes in the same have their MSBytes *cleared* */
              // First used by DL4YHF's PIC-based GPSDO for the serial port (2016-03) .
              // We will know the number of 'bytes per frame' (and thus the number of channels)
              // AFTER the reception of the first complete frame ! The following is just an ASSUMPTION,
              // based on the optional NUMBER OF CHANNELS in the 'parameter string', after "GPSDO3" :
              m_InBytesPerSamplePoint = 1 + 2 * m_InFormat.nChannels;
              break;

           case SOUND_SAMPLE_FMT_UNKNOWN :
           default:
              DEBUG_EnterErrorHistory( DEBUG_LEVEL_FATAL,0,UTL_USE_CURRENT_TIME,
                "CSound::InOpen: Unknown sample format specifier in '%s'", pszAudioInputDriverParamStr );
              SetInputError( ERROR_ILLEGAL_CMD_PARAMETER ); // -> m_ErrorCodeIn
              return m_ErrorCodeIn;
         } // end switch( m_iSoundInputSampleFormat )

        m_InFormat.nBlockAlign = (WORD)(m_InFormat.nChannels * (m_InFormat.wBitsPerSample/8) );  /* in the WAVEFORMATEX : "block size of data" */
        if( m_InBytesPerSamplePoint <= 0 )  // not set yet (in the above switch/case) ?
         {  m_InBytesPerSamplePoint = m_InFormat.nBlockAlign;  // << this will be ok for the "trivial" cases (without extra bytes in each sample-point)
         }

      } // end if < SAMPLE BLOCK FORMAT specified (to receive audio FROM A SERIAL PORT) >


     // For obscure reasons, the following cryptic device name must be used
     // to successfully open COM ports with numbers above 9 using CreateFile():
     sprintf( sz255, "\\\\.\\COM%d", (int) i );
#   if( SWI_UTILITY1_INCLUDED )
     UTL_WriteRunLogEntry( "CSound::InOpen: trying to open %s", sz255 );
#   endif     
     m_hComPort = CreateFile( sz255,   // try to open a SERIAL PORT (not a file)
         GENERIC_READ | GENERIC_WRITE, // access (read-write) mode
                                    0, // share mode
                                 NULL, // pointer to security attributes
                        OPEN_EXISTING, // how to create
                FILE_ATTRIBUTE_NORMAL, // file flags and attributes
                                NULL); // handle to file with attributes to copy
     // Note: If the 'COM port' is in fact a VIRTUAL com port, e.g. STM32F407 with USB,
     //       it was not necessary to set any 'communication parameters'
     //       for the serial port at all. The data sent by the STM32F4 via USB
     //       simply start pouring in, without filling out a bulky struct ("DCB")
     //       and passing that DCB to "SetCommState" !
     if( m_hComPort == INVALID_HANDLE_VALUE )
      { dw = GetLastError();  // "is the port in use" / "sharing error" ?
        SetInputError( WAVIN_ERR_OPEN ); // -> m_ErrorCodeIn : can't open wave "file" (here: COM port) for input
        sprintf( m_sz255LastErrorStringIn, "Could not open COM%d : %s",
                 (int)i, UTL_LastApiErrorCodeToString(dw) );
        DEBUG_EnterErrorHistory( DEBUG_LEVEL_FATAL,0,UTL_USE_CURRENT_TIME,
                 "CSound::CreateFile failed: %s", m_sz255LastErrorStringIn );
        return m_ErrorCodeIn;
      }
     else // the COM port was successfully opened ...
      {

        // Buffer size guesstimation ...
        // Not sure what the remote ADC will send yet (won't WAIT FOR RECEPTION here) !
        // Neither the sample rate, number of channels, not bits per sample are known yet;
        // thus "expect the worse" (with a VIRTUAL COM PORT) as guesstimated below:
        //   200 kSamples/second (for external ADC circuitry, for example STM32F407)
        //     3 A/D converters running simultaneously (in the STM32F407)
        //    16 bit/sample
        //   200 milliseconds between two calls of CSound::InReadxyz() ->
        // 200 * 1000 * 3 * 2 * 0.2 = ca. 240 kByte, let's say 256 kByte buffer.
        //
        // Prepare the input buffer for the reception of data from the COM port:
        if(m_pbInputBuffer)
         { // free old buffer (with different SIZE) :
           m_InBufferSize = m_NumInputBuffers = 0; // existing buffers are now INVALID
           Sleep(10); // give other threads the chance to stop using the buffer (kludge)
           UTL_free(m_pbInputBuffer);
         }
        i = 256 * 1024; // allocate ONE buffer with 256 kByte (guesstimated above)
        m_pbInputBuffer = (BYTE*)UTL_NamedMalloc( "TSoundIn", i + 1/*reserve for 'integrity test'*/ );
        if( m_pbInputBuffer != NULL )
         {
            m_InBufferSize    = i/*BYTES*/;
            m_NumInputBuffers = 1;
            m_pbInputBuffer[i] = SOUND_MAGIC_BYTE_AT_END_OF_BUFFER;
         }
        else // oops... failed to allocate input-buffers for "audio via COM" ?!
         {
            DEBUG_EnterErrorHistory( DEBUG_LEVEL_FATAL,0,UTL_USE_CURRENT_TIME,
               "CSound: Failed to allocate buffers for 'Audio via COM'." );
            m_InBufferSize    = 0;
            m_NumInputBuffers = 0;
         }



        // Adjust the COM port's RX- and TX- buffer size.
        //    You guessed it, there's yet another API function for this .
        //    Beware (about SetupComm) :
        // > If the device driver determines that the recommended buffer sizes
        // > involve transfers beyond its ability to handle, the function can fail.
        // Gosh. "Involve something beyond its ability to handle". We don't want that.
        // By the way, without SetupComm(), the default size seems to be 4065 (bytes)
        // which appeared a bit low, considering the following:
        // - The audio processing thread reads a block of samples approximately
        //   every 400 milliseconds (11025 samples/second, 4096 samples/block).
        // - Each sample (sent over the serial port) consists of 2 bytes
        //    -> at least 4096 [samples/block] * 2 [bytes/sample] * 2 [safety]
        //         = 16384 bytes/block .
        // - That size would be sufficient for baudrates up to
        //       16384 [bytes] * 10 [bits/serial "byte"] / 0.4 [sec] = 410 kBit/sec,
        //       which is more than the usual RS-232, and more than a cheap
        //       ASK- or FSK-radio-link can handle anyway.
        // > If SetupComm fails, the return value is zero.
        if( 0==SetupComm( m_hComPort,
                 m_InBufferSize,  // [in] DWORD dwInQueue,  size of input buffer in bytes
                 m_InBufferSize)) // [in] DWORD dwOutQueue, size of output buffer in bytes
         { DEBUG_EnterErrorHistory( DEBUG_LEVEL_WARNING,0,UTL_USE_CURRENT_TIME,
            "CSound: SetupComm failed to set buffer sizes to %d bytes.",
                     (int)m_InBufferSize );
         }

        // Configure the serial bitrate and other UART-related stuff ?
        //    First retrieve all CURRENT settings of the serial port.
        //    Further below, we MAY modify some of those (but not all).
        dcb.DCBlength = sizeof( DCB );
        GetCommState( m_hComPort, &dcb );

        if( iBitsPerSecond > 0 )
         { dcb.BaudRate = iBitsPerSecond;     // current baud rate (bits per second or some esoteric macro, which we don't use here)
         }
        dcb.fBinary  = TRUE;      // binary mode, no EOF check
        dcb.fOutxCtsFlow = FALSE; // CTS output flow control
        dcb.fOutxDsrFlow = FALSE; // DSR output flow control
        // Many "simple" CI-V interface draw the positive supply from the
        // DTR line (pin 4 of 9pins, for example G3VGR circuit) .
        dcb.fDtrControl = DTR_CONTROL_ENABLE;  // DTR flow control type
        dcb.fDsrSensitivity = FALSE;   // DSR sensitivity (FALSE=ignore state of DSR)
        dcb.fTXContinueOnXoff = FALSE; // XOFF continues Tx ?
        dcb.fOutX      = FALSE;    // XON/XOFF out flow control
        dcb.fInX       = FALSE;    // XON/XOFF in flow control
        dcb.fErrorChar = FALSE;    // enable error replacement ? (no)
        dcb.fNull      = FALSE;    // enable null stripping ? (no)
        dcb.fRtsControl= FALSE;    // RTS flow control
        dcb.fAbortOnError = FALSE; // abort reads/writes on error

        if( iUartDataBits > 0 )
         { dcb.ByteSize = iUartDataBits; // number of bits/byte, 4-8
         }
        dcb.fParity  = FALSE;      // enable parity checking ON RECEIVE ?
        if( iParity>=0 )
         { dcb.Parity= (BYTE)iParity;    // 0-4=no,odd,even,mark,space
         }
        if( iStopbits>=0 )
         { dcb.StopBits = (BYTE)iStopbits;  // 0,1,2 = 1, 1.5, 2
         }
        // dcb.XonChar = 17;            // Tx and Rx XON character
        // dcb.XoffChar= 19;            // Tx and Rx XOFF character
        // dcb.ErrorChar= 0;            // error replacement character
        // dcb.EofChar = 0;             // end of input character
        // dcb.EvtChar = 0;             // received event character
        SetCommState( m_hComPort, &dcb );


        // Configure the COM port timeouts, so that ReadFile() won't block for more than a few dozen milliseconds:
        COMMTIMEOUTS timeouts;
        GetCommTimeouts( m_hComPort, &timeouts );
        // About COMMTIMEOUTS.ReadIntervalTimeout :
        // > "maximum time, in ms, allowed to elapse between the arrival of two characters" . (..)
        // > A value of MAXDWORD, combined with zero values for both the
        // > ReadTotalTimeoutConstant and ReadTotalTimeoutMultiplier members,
        // > specifies that the read operation is to return immediately
        // > with the characters that have already been received, even if
        // > no characters have been received.
        // The above is what we use here, to make "ReadFile" non-blocking.
        timeouts.ReadIntervalTimeout = 0xFFFFFFFF; // "MAXDWORD"
        timeouts.ReadTotalTimeoutMultiplier = 0;
        timeouts.ReadTotalTimeoutConstant   = 0;
        timeouts.WriteTotalTimeoutMultiplier= 0;
        timeouts.WriteTotalTimeoutConstant  = 0;
        SetCommTimeouts( m_hComPort, &timeouts );

        m_InputOpen = TRUE; // here: SERIAL PORT (COM-port) open for audio-input
        return 0;  // "no error" (from CSound::InOpen, if the input device is a SERIAL PORT)
      }

   } // end if( m_iInputDeviceID==C_SOUND_DEVICE_COM_PORT )

  // use MMSYSTEM or ASIO driver for input ?
  m_iUsingASIO &= ~1;  // up to now, NOT using ASIO for input....
  if(  (m_iInputDeviceID==C_SOUND_DEVICE_ASIO) // pszAudioDriverName = name of an ASIO driver
    && APPL_fMayUseASIO )     // Since 2011-08-09, use of ASIO may be disabled through the command line
   {
#if( SWI_ASIO_SUPPORTED )     // Support ASIO audio drivers too ?
     if((m_iUsingASIO&1)==0 ) // only if ASIO driver not already in use for INPUT..
      {
       if( !AsioWrap_fInitialized )
        {  AsioWrap_Init();     // init DL4YHF's ASIO wrapper if necessary
        }
       // Prepare the most important ASIO settings :
       AsioSettings MyAsioSettings;
       AsioWrap_InitSettings( &MyAsioSettings,
                              iNumberOfChannels,
                              i32SamplesPerSecond,
                              iBitsPerSample );
       if( m_hAsio == C_ASIOWRAP_ILLEGAL_HANDLE )
        {  // Only open the ASIO driver if not open already. Reason:
           // An ASIO device always acts as IN- AND(!) OUTPUT simultaneously.
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "CSound::InOpen: Trying to open ASIO-driver \"%s\"",
                              (char*)pszAudioInputDeviceOrDriver );
#         endif
           m_hAsio = AsioWrap_Open( pszAudioDeviceNameWithoutPrefix, &MyAsioSettings, CSound_AsioCallback, (DWORD)this );
             // Note: AsioWrap_Open() also creates the 'buffers' in the driver !
           if( (m_hAsio==C_ASIOWRAP_ILLEGAL_HANDLE) && (AsioWrap_sz255LastErrorString[0]!=0) )
            { // immediately save error string for later:
              strncpy( m_sz255LastErrorStringIn, AsioWrap_sz255LastErrorString, 79 );
              m_sz255LastErrorStringIn[79] = '\0';
            }
        }
       if( m_hAsio != C_ASIOWRAP_ILLEGAL_HANDLE ) // ASIO wrapper willing to co-operate ?
        { m_iUsingASIO |= 1;    // now using ASIO for input...
           // BEFORE allocating the buffers, check if the ASIO driver will be kind enough
           // to deliver the samples in the format *WE* want .
           // ( some stubborn drivers don't, for example Creative's )
           pAsioChannelInfo = AsioWrap_GetChannelInfo( m_hAsio, 0/*iChnIndex*/, 1/*iForInput*/ );
           if( pAsioChannelInfo!=NULL )
            {
              snprintf( m_sz255AudioInputDeviceOrDriver, 255, // << "really used input device" (here: including the channel name, reported by ASIO driver)
                        "%s:%s (ASIO chn)",
                        pszAudioInputDeviceOrDriver,
                        pAsioChannelInfo->name, 255 );
              switch(pAsioChannelInfo->type)
               { // SOO many types, so hope for the best, but expect the worst .....  see ASIO.H !
                 case ASIOSTInt16MSB   :
                      m_InFormat.wBitsPerSample = 16;
                      break;
                 case ASIOSTInt24MSB   : // used for 20 bits as well
                      m_InFormat.wBitsPerSample = 24;
                      break;
                 case ASIOSTInt32MSB   :
                      m_InFormat.wBitsPerSample = 32;
                      break;
                 case ASIOSTFloat32MSB : // IEEE 754 32 bit float
                      m_InFormat.wBitsPerSample = 32;
                      break;
                 case ASIOSTFloat64MSB : // IEEE 754 64 bit double float
                      m_InFormat.wBitsPerSample = 64;
                      break;

                 // these are used for 32 bit data buffer, with different alignment of the data inside
                 // 32 bit PCI bus systems can be more easily used with these
                 case ASIOSTInt32MSB16 : // 32 bit data with 16 bit alignment
                      m_InFormat.wBitsPerSample = 32;
                      break;
                 case ASIOSTInt32MSB18 : // 32 bit data with 18 bit alignment
                      m_InFormat.wBitsPerSample = 32;
                      break;
                 case ASIOSTInt32MSB20 : // 32 bit data with 20 bit alignment
                      if( m_InFormat.wBitsPerSample != 32 )
                       { DEBUG_EnterErrorHistory( DEBUG_LEVEL_WARNING, 0, UTL_USE_CURRENT_TIME,
                            "ASIO-IN : Asked for %d bits/sample but driver delivers 'Int32MSB20' .",
                            (int)m_InFormat.wBitsPerSample );
                       }
                      m_InFormat.wBitsPerSample = 32;
                      break;
                 case ASIOSTInt32MSB24 : // 32 bit data with 24 bit alignment
                      if( m_InFormat.wBitsPerSample != 32 )
                       { DEBUG_EnterErrorHistory( DEBUG_LEVEL_WARNING, 0, UTL_USE_CURRENT_TIME,
                            "ASIO-IN : Asked for %d bits/sample but driver delivers 'Int32MSB24' .",
                            (int)m_InFormat.wBitsPerSample );
                       }
                      m_InFormat.wBitsPerSample = 32;
                      break;
                 case ASIOSTInt16LSB   : // 16 bit, used by the 'ASIO Sample' driver !
                      if( m_InFormat.wBitsPerSample != 16 )
                       { DEBUG_EnterErrorHistory( DEBUG_LEVEL_WARNING, 0, UTL_USE_CURRENT_TIME,
                            "ASIO-IN : Asked for %d bits/sample but driver delivers 'Int16LSB' .",
                            (int)m_InFormat.wBitsPerSample );
                       }
                      m_InFormat.wBitsPerSample = 16;
                      break;
                 case ASIOSTInt24LSB   : // used for 20 bits as well
                      if( m_InFormat.wBitsPerSample != 24 )
                       { DEBUG_EnterErrorHistory( DEBUG_LEVEL_WARNING, 0, UTL_USE_CURRENT_TIME,
                            "ASIO-IN : Asked for %d bits/sample but driver delivers 'Int24LSB' .",
                            (int)m_InFormat.wBitsPerSample );
                       }
                      m_InFormat.wBitsPerSample = 24;
                      // 2015-07: With an E-MU 0202 under Windows XP,
                      // got here (with ASIOSTInt24LSB) even though
                      // THE APPLICATION (SpecLab) wanted to receive only 16 bits per sample !
                      // Yet another example of a stubborn driver - see notes further above .
                      break;
                 case ASIOSTInt32LSB   : // This is what Creative's "updated" driver seems to use,
                      m_InFormat.wBitsPerSample = 32; // interestingly NOT one of those many 24-bit types.
                      break;
                 case ASIOSTFloat32LSB : // IEEE 754 32 bit float, as found on Intel x86 architecture
                      m_InFormat.wBitsPerSample = 32;
                      break;
                 case ASIOSTFloat64LSB : // IEEE 754 64 bit double float, as found on Intel x86 architecture
                      m_InFormat.wBitsPerSample = 64;
                      break;

                 // these are used for 32 bit data buffer, with different alignment of the data inside
                 // 32 bit PCI bus systems can more easily used with these
                 case ASIOSTInt32LSB16 : // 32 bit data with 18 bit alignment
                 case ASIOSTInt32LSB18 : // 32 bit data with 18 bit alignment
                 case ASIOSTInt32LSB20 : // 32 bit data with 20 bit alignment
                 case ASIOSTInt32LSB24 : // 32 bit data with 24 bit alignment
                      m_InFormat.wBitsPerSample = 32;
                      break;

                 // ASIO DSD format.
                 case ASIOSTDSDInt8LSB1: // DSD 1 bit data, 8 samples per byte. First sample in Least significant bit.
                 case ASIOSTDSDInt8MSB1: // DSD 1 bit data, 8 samples per byte. First sample in Most significant bit.
                 case ASIOSTDSDInt8NER8: // DSD 8 bit data, 1 sample per byte. No Endianness required.
                      m_InFormat.wBitsPerSample = 8;
                      break;
                 default:
                      strcpy( m_sz255LastErrorStringIn, "ASIO data type not supported" );
                      break;
                } // end switch(pAsioChannelInfo->type)
               // Revise the "number of bytes per sample" based on the above types:
               m_InBytesPerSamplePoint = (m_InFormat.wBitsPerSample/8) * m_InFormat.nChannels;
            } // end if < got info about this ASIO input channel >
           else
            { // got no "info" about this ASIO channel -> ignore it
            }
         } // end if < successfully opened the ASIO driver >
        else  // bad luck (happens !) :  ASIO driver could not be loaded
         { SetInputError( ERROR_WITH_ASIO_DRIVER ); // -> m_ErrorCodeIn
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "CSound::InOpen: Failed to open ASIO-driver !" );
#         endif           
           return m_ErrorCodeIn;
        }
      } // end if < ASIO driver not loaded yet >
#endif // SWI_ASIO_SUPPORTED
   } // end if < use ASIO driver instead of MMSYSTEM ? >

  // Allocate and prepare buffers for INPUT (from soundcard to application)
  if( dwMinBufSize > 0 )  // only if not just a soundcard check ...
   {
     // How many buffers, and how large should each part be ?
     //    Ideally enough for 0.2 to 0.5 seconds;
     //    but for some applications we need the lowest possible latency.
     //    In those cases, only ONE of these buffers will be waiting to be filled.
     //    Spectrum Lab will ask for 'dwMinBufSize' as large as the PROCESSING CHUNKS
     //    (details on that in SoundThd.cpp :: SoundThd_InitializeAudioDevices ),
     //    Example:  96 kHz, approx. 0.5 second chunks -> dwMinBufSize=65536 [SAMPLES].
     // 2015-02-01 : Be generous with the NUMBER OF BUFFERS (but not the size PER BUFFER) !
     //              In case of doubt, spend a few hundred kBytes more than necessary,
     //              to reduce the risk of missing samples in case of high CPU load :
     dwMinBufSize += m_InFormat.nSamplesPerSec/10;  // at least 100 ms of audio (additionally)
     // Some authors (also plaged by the obfuscated 'wave audio' API) noted that
     //    the size of a single buffer passed to waveOutWrite / waveInAddBuffer
     //    must not exceed 64 kBytes (here: m_InBufferSize) .   See
     // http://v3.juhara.com/en/articles/multimedia-programming/17-sound-playback-with-wave-api :
     // > We can send all WAV data to device driver by calling waveOutWrite once,
     // > but please note that, on some soundcards (especially the old ones),
     // > maximum buffer size can be processed is 64 KB, so if you have waveform data
     // > bigger than 64 KB, waveOutWrite must be called more than once with smaller data blocks.
     // Furthermore, to keep the buffer contents aligned to "sample point boundaries"
     // (where a "sample point" is a point in time, with N channels per point),
     // the buffer size (in bytes) must be a multiple of Microsoft calls "BlockAlign".
     i32TotalBufSizeInByte = (long)(2*dwMinBufSize) * m_InBytesPerSamplePoint; // may be as much as a HALF MEGABYTE !
     i = (i32TotalBufSizeInByte+3) / 4/*parts*/; // -> approx size of each buffer-part, in BYTES.
     if(i<1024)  i=1024;    // at least 512 bytes(!) per buffer-part
     if(i>65536) i=65536;   // HISTORIC limit (for stoneage audio drivers): maximum  64 kByte per buffer-part
     newNumInputBuffers = (i32TotalBufSizeInByte + i - 1) / i;
     if( newNumInputBuffers < 4)  // should use at least FOUR buffers, and..
         newNumInputBuffers = 4;
     if( newNumInputBuffers > SOUND_MAX_INPUT_BUFFERS ) // a maximum of SIXTEEN buffers..
         newNumInputBuffers = SOUND_MAX_INPUT_BUFFERS;
     // Now, after limiting the NUMBER OF BUFFERS, how large should each buffer-part be ?
     // For example, the caller (Spectrum Lab : SoundThd.cpp, SoundThd_InitializeAudioDevices)
     // may want to read or write samples in blocks of 65536 samples each (=dwMinBufSize) .
     // The TOTAL internal buffer size must be at least twice that value,
     // because one of those 4..16 buffer-parts is always occupied by the multimedia driver,
     // while the other parts are being filled, or already filled and waiting for in/output.
     i = (i32TotalBufSizeInByte + newNumInputBuffers/*round up*/ - 1)  / newNumInputBuffers;
     // For what it's worth, the size of each 'buffer part' should be a power of two,
     //     but at last 256  [samples bytes per 'buffer part']
     newInBufferSize = 256 * m_InBytesPerSamplePoint;
     while( (newInBufferSize < i) && (newInBufferSize < 32768) )
      { newInBufferSize <<= 1;
      }
     // EXAMPLES:
     //   96kHz, with 500 milliseconds per chunk: newInBufferSize=65536, newNumInputBuffers=8
     //
     DOBINI();  // -> Sound_iSourceCodeLine
     if(m_pbInputBuffer)
      { // free old buffer (with possibly different SIZE) ?
        if( (m_InBufferSize!=newInBufferSize) || (m_NumInputBuffers!=newNumInputBuffers) )
         { m_InBufferSize = m_NumInputBuffers = 0; // existing buffers are now INVALID
           // Sleep(10); // give other threads the chance to stop using the buffer (kludge.. now unnecessary)
           UTL_free(m_pbInputBuffer);
           m_pbInputBuffer = NULL;
         }
      }
     if( m_pbInputBuffer==NULL )
      {
         m_pbInputBuffer = (BYTE*)UTL_NamedMalloc( "TSoundIn", newNumInputBuffers * newInBufferSize);
         m_InBufferSize    = newInBufferSize/*per "part", measures in BYTES */;
         m_NumInputBuffers = newNumInputBuffers;
      }
     DOBINI();  // -> Sound_iSourceCodeLine
     if ( m_pbInputBuffer==NULL )
      { // bad luck, the requested memory could not be allocated !
       SetInputError( MEMORY_ERROR ); // -> m_ErrorCodeIn
       return m_ErrorCodeIn;
      }
    } // end if < not just a soundcard test >

   m_InSampleLimit = SampleLimit;
   DOBINI();

   // event for callback function to notify new buffer available
   if( m_InEventHandle == NULL )
    {  m_InEventHandle = CreateEvent(NULL, FALSE,FALSE,NULL);
    }

  DOBINI();

  // Open sound card input and get handle(m_hwvin) to device,
  // depends on whether MMSYSTEM or ASIO driver shall be used :
  m_fWaveInStarted = FALSE;   // waveInStart() has not been called yet, thus don"undone" via waveInReset(). What an intuitive API...
  if( m_iUsingASIO & 1 )   // using ASIO for input :
   {
#if( SWI_ASIO_SUPPORTED )      // Support ASIO audio drivers too ?
      // Already opened the driver above, but did NOT start it then,
      // because the buffers were not allocated yet.
#endif // SWI_ASIO_SUPPORTED
   }
  else // not using ASIO driver for input but MMSYSTEM :
   { int iSoundInputDeviceID;
     //  At this point, m_iInputDeviceID is simply the decimal number
     //       parsed from the begin of the device name in Sound_GetAudioDeviceNameWithoutPrefix() .
     //       Sound_InputDeviceNameToDeviceID() does a more thorough job:
     //       It ENUMERATES the available wave input devices (using a windows API function),
     //       and compares the *DEVICES NAMES* with the audio device name:
     SOUND_fDumpEnumeratedDeviceNames = TRUE;  // flag for Sound_InputDeviceNameToDeviceID()
     iSoundInputDeviceID = Sound_InputDeviceNameToDeviceID( pszAudioDeviceName );
         // Beware: If Sound_InputDeviceNameToDeviceID() cannot make sense out of
         //         pszAudioDeviceName, it returns iSoundInputDeviceID = -1
         //         to use the 'default' audio device. To find out the proper
         //         name of a certain audio device, DON'T RELY ON THE FOOLISH
         //         WINDOZE SYSTEM CONTROL / classic audio control panel /
         //         "Recording" / "Aufnahme" ! For example, with an IC-7300
         //         connected, the braindead thing showed "2- USB Audio CODEC",
         //         but the "2 minus" in there doesn't have ANYTHING to do
         //         with the numeric identifier that we need to pass to
         //         waveInOpen() a few lines further below !

     if (iSoundInputDeviceID < 0)
      {  iSoundInputDeviceID = WAVE_MAPPER;
      }
#  if( 1 )  // (1)=normal compilation, (0)=TEST to force using *ONLY* the WAVEFORMATEXTENSIBLE [which sometimes failed - details below]
#   if( SWI_UTILITY1_INCLUDED )
     UTL_WriteRunLogEntry( "CSound::InOpen: Trying to open \"%s\", ID #%d, %d Hz, %d bit, %d channel(s), for input.",
                              (char*)pszAudioDeviceNameWithoutPrefix,
                              (int)iSoundInputDeviceID,
                              (int)m_InFormat.nSamplesPerSec,
                              (int)m_InFormat.wBitsPerSample,
                              (int)m_InFormat.nChannels );
#   endif                              
     // Note: It is not clear why sometimes, deep inside waveInOpen(),
     //       the Borland debugger stops on 'ntdll.DbgBreakPoint'  .
     if( (iErrorCode = waveInOpen(
            &m_hwvin,             // [out] handle to identify the device
            iSoundInputDeviceID,  // [in]  Identifier of the waveform-audio input device to open
            &m_InFormat,          // [in]  Pointer to a WAVEFORMATEX structure that identifies the desired format for recording
            (DWORD)WaveInCallback,// [in]  Pointer to a fixed callback function
            (DWORD)this,          // [in]  User-instance data passed to the callback mechanism
            CALLBACK_FUNCTION     // [in]  "fdwOpen", Flags for opening the device
             | WAVE_FORMAT_DIRECT // discovered 2015-07-08 : WAVE_FORMAT_DIRECT :
                                  // "If this flag is specified, the ACM driver does not perform conversions on the audio data"
         ) ) != MMSYSERR_NOERROR )
#endif  // (1)=normal compilation, (0)=TEST
      { // Could not open the WAVE INPUT DEVICE with the old-fashioned way.
        // SO FAR, THIS IS NO REASON TO THROW AN ERROR YET !
        //    Try something new, using a WAVEFORMATEXTENSIBLE structure ...
        // > Formats that support more than two channels
        // > or sample sizes of more than 16 bits can be described
        // > in a WAVEFORMATEXTENSIBLE structure, which includes the WAVEFORMAT structure.
        WAVEFORMATEXTENSIBLE wfmextensible;
#      if(sizeof(WAVEFORMATEXTENSIBLE) != 40 )
#        error "This stupid compiler seems to have a problem with struct alignment"
#      endif
#      if( SWI_UTILITY1_INCLUDED )
        UTL_WriteRunLogEntry( "CSound::InOpen: waveInOpen failed. Trying again to open audio device for input, now using WAVEFORMATEXTENSIBLE" );
#      endif        
        // don't know what may be in it, so clear everything (also unknown components)
        memset( &wfmextensible, 0, sizeof(WAVEFORMATEXTENSIBLE) );
        // Now set all required(?) members. Based on info found from various sources.
        // Note that the new WAVEFORMATEXTENSIBLE includes the old WAVEFORMATEX .
        wfmextensible.Format = m_InFormat;  // COPY the contents of the WAVEFORMATEX .
            // wfmextensible.Format.nChannels, .wBitsPerSample, .nBlockAlign,
            //                     .nSamplesPerSec, .nAvgBytesPerSec already set !
        wfmextensible.Format.cbSize = sizeof(WAVEFORMATEXTENSIBLE);
            // > WAVEFORMATEX.cbSize
            // >   Size, in bytes, of extra format information appended to the end
            // >   of the WAVEFORMATEX structure. This information can be used
            // >   by non-PCM formats to store extra attributes for the wFormatTag.
            // >   If no extra information is required by the wFormatTag,
            // >   this member must be set to 0.
            // >  For WAVE_FORMAT_PCM formats (and only WAVE_FORMAT_PCM formats),
            // >  this member is ignored.
            // >  When this structure is included in a WAVEFORMATEXTENSIBLE structure,
            // >  this value must be at least 22.
            // Why ? From msdn.microsoft.com/en-us/library/windows/hardware/dn653308(v=vs.85).aspx :
            // > cbSize is at Least 22
            // >  For WAVEFORMATEXTENSIBLE, cbSize must always be set to at least 22.
            // >  This is the sum of the sizes of the Samples union (2),
            // >  DWORD dwChannelMask (4), and GUID guidSubFormat (16).
            // >  This is appended to the initial WAVEFORMATEX Format (size 18),
            // >  so a WAVEFORMATPCMEX and WAVEFORMATIEEEFLOATEX structure is 64-bit aligned.
            // Holy shit. "How to make simple things as complicated as possible".
            // By the way, more information about 24-bit-per-sample in WAVE FILES
            // is in c:\cproj\SoundUtl\CWaveIO.cpp : C_WaveIO::OutOpen() !
        wfmextensible.Format.wFormatTag = WAVE_FORMAT_EXTENSIBLE;
        wfmextensible.Samples.wValidBitsPerSample = wfmextensible.Format.wBitsPerSample;
        wfmextensible.SubFormat = KSDATAFORMAT_SUBTYPE_PCM;
        // Now make a guess for the speaker configuration :
        wfmextensible.dwChannelMask = SPEAKER_FRONT_LEFT;   // MONO ...
        if(wfmextensible.Format.nChannels >= 2)             // STEREO..
           wfmextensible.dwChannelMask |= SPEAKER_FRONT_RIGHT;
        if(wfmextensible.Format.nChannels >= 3)             // ... and more ...
           wfmextensible.dwChannelMask |= SPEAKER_FRONT_CENTER;
        if(wfmextensible.Format.nChannels >= 4)
           wfmextensible.dwChannelMask |= SPEAKER_LOW_FREQUENCY;
        if(wfmextensible.Format.nChannels >= 5)
           wfmextensible.dwChannelMask |= SPEAKER_BACK_LEFT ;
        if(wfmextensible.Format.nChannels >= 6)             // typical "5.1" system
           wfmextensible.dwChannelMask |= SPEAKER_BACK_RIGHT;

        // Try again to open the audio input device, this time passing a pointer
        //  to a WAVEFORMATEXTENSIBLE struct (instead of the old WAVEFORMATEX ) :
        if( (iErrorCode = waveInOpen(
            &m_hwvin,             // [out] handle to identify the device
            iSoundInputDeviceID,  // [in]  Identifier of the waveform-audio input device to open
            (WAVEFORMATEX*)&wfmextensible, // [in]  Pointer to a WAVEFORMATEX structure that identifies the desired format for recording
            (DWORD)WaveInCallback,// [in]  Pointer to a fixed callback function
            (DWORD)this,          // [in]  User-instance data passed to the callback mechanism
            CALLBACK_FUNCTION     // [in]  "fdwOpen", Flags for opening the device
             | WAVE_FORMAT_DIRECT // discovered 2015-07-08 : WAVE_FORMAT_DIRECT :
                                  // "If this flag is specified, the ACM driver does not perform conversions on the audio data"
          ) ) != MMSYSERR_NOERROR )
         {
           SetInputError( iErrorCode );  // -> m_ErrorCodeIn
           // Arrived here ? waveInOpen() worked neither the OLD STYLE nor the NEW WAY.
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "CSound::InOpen: Failed to open multimedia audio device for input" );
#         endif           

           InClose();   // m_ErrorCode = MMSYSERR_xxx
           // 2015-09-19: Got here when trying to open 'Virtual Audio Cable V3.12'
           //             by E. Muzychenko for input, with m_ErrorCodeIn = 32 .
           //  Error message:  "unsupported wave format in SoundInOpen" .
           //             Sounds familiar - see 2012-03-10 .
           //             Obviously VAC V3.12 is unable to support the shiny "new"
           //             'WAVEFORMATEXTENSIBLE', so WB had to put the ancient
           //             'WAVEFORMATEX' back into service.
           // 2012-03-10: Got here when trying to open an E-MU 0202 for input,
           //             with 24 bits / sample, 1 channel, 11025 samples/second.
           //  Error message:  "unsupported wave format in SoundInOpen" .
           //  In an old version (V2.77 b03 from 2011_10_05), it still worked.
           //  But after the same old version was compiled with Borland C++ Builder V6,
           //  the same error ("unsupported wave format") also appeared;
           //  so obviously Borland made some changes in their "mmsystem.h"
           //  causing a serious compatibility problem ?
           //  At least, the problem could be fixed by modifying "mmsystem.h"
           //  by inserting the following before typedef struct tWAVEFORMATEX ...
           //
           // #ifdef __BORLANDC__
           // # pragma pack(push,2)
           // #endif // __BORLANDC__
           //
           //  .. and the following AFTER the structure definition:
           //
           // #ifdef __BORLANDC__
           // # pragma pack(pop)
           // #endif // __BORLANDC__
           //
           return m_ErrorCodeIn;
         }
      } // end if <could not open the WAVE INPUT DEVICE with the old-fashioned way>
     // Added 2014-12 : Save the name of the "really used" device,
     //                 in a similar notation as in SL's input device combo:
     snprintf( m_sz255AudioInputDeviceOrDriver, 255, // << "really used input device" (here: including the REALLY USED "Wave-In-Device-ID")
               "%d %s (wave-In)", (int)iSoundInputDeviceID,  pszAudioDeviceNameWithoutPrefix );
     // The result may be funny, e.g. "3 2- USB Audio CODEC (wave-In)"
     //                                | |                  |_______|
     //                                | |                     |
     //                                | |                    "our" suffic (to tell WAVE-audio from ASIO, etc etc)
     //                                | "the driver's own prefix" (2= second instance of an "USB Audio CODEC" ?)
     //                 "our" prefix (3)
   } // end else < use MMSYSTEM, not ASIO >


  if( dwMinBufSize == 0 )  // see if just a soundcard check
   {
     InClose();   // if so close the soundcard input
     return 0;
   }
#if( SWI_UTILITY1_INCLUDED )
  UTL_WriteRunLogEntry( "CSound::InOpen: Creating %d input buffers for 'wave' input",
                        (int)m_NumInputBuffers );
#endif
  for(i=0; i<m_NumInputBuffers; i++ )
   {
     // initialize WAVEHDR structures (that's the buffer-stuff, NOT the sample format description)
     m_InputWaveHdr[i].dwBufferLength = m_InBufferSize; // number of BYTES (!!!), not samples
     m_InputWaveHdr[i].dwFlags = 0;
     m_InputWaveHdr[i].dwUser = 0; // ex NULL (but this is NO pointer)
     m_InputWaveHdr[i].dwBytesRecorded = 0; // ex NULL (this is NO POINTER!)
     m_InputWaveHdr[i].lpData = (LPSTR) m_pbInputBuffer + i*m_InBufferSize;
     if( m_hwvin != NULL )
      { // only if MMSYSTEM used (but not for ASIO or input from COM port, etc)
        DOBINI();
        if( ( iErrorCode = waveInPrepareHeader(m_hwvin, &m_InputWaveHdr[i],
                             sizeof(WAVEHDR))) != MMSYSERR_NOERROR )
         { SetInputError( iErrorCode ); // -> m_ErrorCodeIn, with the chance for a SINGLE breakpoint to catch ALL errors
           InClose( );   // m_ErrorCodeIn = MMSYSERR_xxx
           return m_ErrorCodeIn;
         }
        // The waveInAddBuffer function sends an input buffer to the given
        // waveform-audio input device.
        // When the buffer is filled, the application is notified.
        DOBINI();
        if( (iErrorCode = waveInAddBuffer(m_hwvin, &m_InputWaveHdr[i],
                             sizeof(WAVEHDR)) )!= MMSYSERR_NOERROR )
         { SetInputError( iErrorCode ); // -> m_ErrorCodeIn, with the chance for a SINGLE breakpoint to catch ALL errors
           InClose();   // m_ErrorCodeIn = MMSYSERR_xxx
           return m_ErrorCodeIn;
         }
      } // end if( m_hwvin != NULL )

    } // end for <all input-buffers>

  DOBINI();

  /* start input capturing to buffer (except for waveInStart ? ) */
  if(start)
   {
     if( m_iUsingASIO & 1 )   // using ASIO for input :
      { // Start the ASIO driver (for BOTH INPUT AND OUTPUT !)
#if( SWI_ASIO_SUPPORTED )      // Support ASIO audio drivers too ?
#      if( SWI_UTILITY1_INCLUDED )
        UTL_WriteRunLogEntry( "CSound::InOpen: Starting input from ASIO driver" );
#      endif        
        if( ! AsioWrap_StartOrStop( m_hAsio, TRUE/*fStart*/ ) )
         { if( m_sz255LastErrorStringIn[0]==0 && AsioWrap_sz255LastErrorString[0]!=0 )
            { // save error string for later:
              strncpy( m_sz255LastErrorStringIn, AsioWrap_sz255LastErrorString, 79 );
              m_sz255LastErrorStringIn[79] = '\0';
            }
           SetInputError( ERROR_WITH_ASIO_DRIVER ); // -> m_ErrorCodeIn, with the chance for a SINGLE breakpoint to catch ALL errors
           InClose();   // m_ErrorCodeIn = MMSYSERR_xxx
           return m_ErrorCodeIn;
         }
#endif // ( SWI_ASIO_SUPPORTED )
      }
     else // not using ASIO driver for input but MMSYSTEM :
     if( m_hwvin != NULL )
      { // only if MMSYSTEM used :
#      if( ! MOD_2015_11_25 )   // modified 2015-11 : Now waiting until the first call of InRead(), before calling waveInStart() !
        DEBUG_EnterErrorHistory( DEBUG_LEVEL_INFO,0,UTL_USE_CURRENT_TIME,
           "CSound: Calling waveInStart()" );
        if( (iErrorCode = waveInStart(m_hwvin) )!= MMSYSERR_NOERROR )
         { SetInputError( iErrorCode ); // -> m_ErrorCodeIn, with the chance for a SINGLE breakpoint to catch ALL errors
           InClose();   // m_ErrorCodeIn = MMSYSERR_xxx
           return m_ErrorCodeIn;
         }
        else
         { m_fWaveInStarted = TRUE;
         }
#      endif // (0)
      }
   } // end if(start)
#if( SWI_UTILITY1_INCLUDED )
  UTL_WriteRunLogEntry( "CSound::InOpen: Successfully opened audio input" );
#endif  
  m_InputOpen = TRUE;  // Note: m_InputOpen only becomes true AFTER buffer allocation was ok
  DOBINI();  // -> Sound_iSourceCodeLine
  return 0;  // "no error" (from CSound::InOpen)
} // end CSound::InOpen()


//--------------------------------------------------------------------------
BOOL CSound::IsInputOpen(void)   // checks if the sound input is opened
{
  return m_InputOpen;
}

#define VERSION_WAIT_FOR_INPUT_DATA 0
  // 0 = use original implementation of CSound::WaitForInputData(), < 2015-02-01
  // 1 = use an overly simplified test from 2015-02-01
  // 2 = use even somthing else
#if( VERSION_WAIT_FOR_INPUT_DATA == 0 )
//--------------------------------------------------------------------------
BOOL CSound::WaitForInputData(int iTimeout_ms)
  // Does what the name says, but only waits if the input buffers are ALL empty.
  // Return value:  TRUE = "there's at least ONE buffer filled with samples
  //                        received from the soundcard";
  //                FALSE= "sorry, have been waiting for data
  //                        but nothing arrived from the soundcard" .
  //
{
  LONGLONG i64TimeToLeave, i64Tnow, i64TimerFrequency;  // added 2011-03-13
  int i;

  QueryPerformanceCounter( (LARGE_INTEGER*)&i64Tnow );
  QueryPerformanceFrequency( (LARGE_INTEGER*)&i64TimerFrequency );
  i64TimeToLeave = i64Tnow + ((LONGLONG)iTimeout_ms * i64TimerFrequency) / 1000;  // -> max. 500 (?) milliseconds to spend here

#if( CSOUND_USE_CRITICAL_SECTION )
  EnterCriticalSection(&m_CriticalSection);
#endif
  if( m_iInTailIndex == m_iInHeadIndex )  // if no buffer is filled...
   {
      // wait for mmsystem (or ASIO) to fill up a new buffer
      m_InWaiting = TRUE;
#    if( CSOUND_USE_CRITICAL_SECTION )
      LeaveCriticalSection(&m_CriticalSection);
#    endif
      DOBINI();  // -> Sound_iSourceCodeLine
      i = (iTimeout_ms+49/*ms*/) / 50;
      while( (i--)>0 )
       {
         if( SndThd_fCloseDevicesDuringSleep ) // received WM_POWERBROADCAST / PBT_APMSUSPEND somewhere in the GUI !
          { // instead of fooling around with "WaitForMultipleObjects" and other windows-specific crap,
            // bail out of this "tamed busy-spinning" loop as soon as recognizing this "going-to-sleep"-flag :
            break;
          }
         DOBINI();   // -> Sound_iSourceCodeLine
         Sleep(50);  // wait a little longer for the wave-in-callback to fill another buffer
                     // Caution: Sleep() may actually take LONGER than expected !
         DOBINI();
         if( m_iInTailIndex != m_iInHeadIndex )
          { break;   // bingo; WaveInCallback() has sent us another buffer
          }
         if( !m_InputOpen ) // return -1 if no inputs are active
          {
#          if( SWI_UTILITY1_INCLUDED )
            UTL_WriteRunLogEntry( "SOUND: Warning; input closed while waiting in 'InRead()', line %d.",(int)__LINE__ );
#          endif            
            return FALSE;
          }
         QueryPerformanceCounter( (LARGE_INTEGER*)&i64Tnow );
         if( i64Tnow > i64TimeToLeave )  // because of the stupid Sleep()-behaviour
          { DOBINI(); // -> Sound_iSourceCodeLine
            i = 0;
            // 2015-02-01 : Frequently got here with an E-MU 0202 when
            //              SCROLLING RAPIDLY IN FIREFOX ?!
            break;    // added 2011-03 because of the crippled Sleep() function
          }
       } // end for
      DOBINI();   // -> Sound_iSourceCodeLine
      if( i<=0 )
       { // been waiting too long, error..
         DOBINI();   // -> Sound_iSourceCodeLine
#       if( CSOUND_USE_CRITICAL_SECTION )
         EnterCriticalSection(&m_CriticalSection);
#       endif
          {
            DEBUGGER_BREAK(); // set breakpoint here, or use the 'common' breakpoint in C:\cbproj\SpecLab\DebugU1.cpp : DEBUG_Breakpoint()
            // 2015-01-28 : Got here when the CPU was heavily loaded (by scrolling in Firefox).
            //              Tried a much downsized variant of CSound::WaitForInputData()
            //              to find out if the output from the E-MU 0202 was really DEAD,
            //              or the problem was in fact caused by the lazy implementation
            //              (and one should better use WaitForSingleObject() as in an MS demo).
            // 2020-09-26 : Got here in the 'Sound-Input-Utility' when called from
            //               SoundThdFunc() -> CSound::InReadStereo()
            //                -> CSound::WaitForInputData( iTimeout_ms = 50 ) .
            //              Obviously, WaveInCallback() now has a MUCH larger
            //              latency than it used to have in the good old days .
            //   Increasing the timeout value from 50 to 300 milliseconds
            //       (in C:\cbproj\SoundUtl\SndInThd.cpp : SoundThdFunc() ) .
            SetInputError(  SOUNDIN_ERR_TIMEOUT );
            // ex: InClose();  // removed 2014-05-03 ... the APPLICATION should take care of this !
            // Also, we don't want to close the input just because of a temporarily delay (by other apps, etc)
            if( m_sz255LastErrorStringIn[0]==0 )
             { strcpy(m_sz255LastErrorStringIn,"Timed out waiting for input");
             }
          }
#       if( CSOUND_USE_CRITICAL_SECTION )
         LeaveCriticalSection(&m_CriticalSection);
#       endif
         DOBINI();  // -> Sound_iSourceCodeLine
       }
    }
   else
    {
#    if( CSOUND_USE_CRITICAL_SECTION )
      LeaveCriticalSection(&m_CriticalSection);
#    endif
    }
  return ( m_iInTailIndex!=m_iInHeadIndex );
} // end CSound::WaitForInputData()
#endif // ( VERSION_WAIT_FOR_INPUT_DATA == 0 ) ?

#if( VERSION_WAIT_FOR_INPUT_DATA == 1 )
//--------------------------------------------------------------------------
BOOL CSound::WaitForInputData(int iTimeout_ms)
  // Simplified variant, but also didn't fix the problem of
  //  "stopped audio input when scrolling in Firefox" !
{
  int i = 2000 + iTimeout_ms / 50;
  // 2015-02-01 : Even with 2000 loops a 50 ms (during which 'Sleep' really slept),
  //              m_iInHeadIndex wasn't incremented after scrolling in the browser
  //              so looks like the problem isn't caused directly by WaitForInputData().
  //              Next test: Back to the original code, but now
  //              with the option CSOUND_USE_CRITICAL_SECTION = 1 (in Sound.h) .

  if( m_iInTailIndex == m_iInHeadIndex )  // if no buffer is filled...
   {
      // wait for mmsystem (or ASIO) to fill up a new buffer
      m_InWaiting = TRUE;
      DOBINI();  // -> Sound_iSourceCodeLine
      while( (i--)>0 )
       {
         if( m_iInTailIndex != m_iInHeadIndex )
          { break; // bingo; WaveInCallback() has sent us another buffer
          }
         if( SndThd_fCloseDevicesDuringSleep ) // received WM_POWERBROADCAST / PBT_APMSUSPEND somewhere in the GUI !
          { // instead of fooling around with "WaitForMultipleObjects" and other windows-specific crap,
            // bail out of this "tamed busy-spinning" loop (and many others)
            // as soon as recognizing this "going-to-sleep"-flag :
            break;
          }
         Sleep(50);  // wait a little longer for the wave-in-callback to fill another buffer
                     // Caution: Sleep() may actually take LONGER than expected !
         if( !m_InputOpen ) // return -1 if no inputs are active
          {
#          if( SWI_UTILITY1_INCLUDED )
            UTL_WriteRunLogEntry( "SOUND: Warning; input closed while waiting in 'InRead()', line %d.",(int)__LINE__ );
#          endif            
            break;
          }
       } // end for
      DOBINI();   // -> Sound_iSourceCodeLine
      if( i<=0 )
       { // been waiting too long, error..
         SetInputError(  SOUNDIN_ERR_TIMEOUT );  // YHF: code (2)113
         if( m_sz255LastErrorStringIn[0]==0 )
          { strcpy(m_sz255LastErrorStringIn,"Timed out waiting for input");
          }
       }
   }
  return ( m_iInTailIndex!=m_iInHeadIndex );
} // end CSound::WaitForInputData()
#endif // ( VERSION_WAIT_FOR_INPUT_DATA == 1 ) ?


//--------------------------------------------------------------------------
BYTE * CSound::WrapInBufPositionAndGetPointerToNextFragment( // .. for AUDIO INPUT
     int iOptions, // [in] one of the following:
       #define CSOUND_WRAPBUF_OPT_DONT_WAIT     0
       #define CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA 1
     T_SoundBufferInfo **ppSoundBufferInfo ) // [out] address of buffer info
{
  MMRESULT result;
  if( m_InBufPosition >= m_InBufferSize )  // finished with an INPUT-buffer
   { // so add it back in to the *INPUT*-Queue (for the MMSYSTEM API)
     if( (m_iInTailIndex<0) || (m_iInTailIndex >= SOUND_MAX_INPUT_BUFFERS)
                          || (m_iInTailIndex >= m_NumInputBuffers)
       )
      { // added 2011-03-13 for safety ...
        m_iInTailIndex = 0;   // possibly an UNHANDLED wrap-around ?
      }
     m_InputWaveHdr[m_iInTailIndex].dwBytesRecorded = 0;
     if( m_hwvin != NULL ) // only if MMSYSTEM used (not required for ASIO) :
      { DOBINI();   // -> Sound_iSourceCodeLine
        result = waveInAddBuffer(m_hwvin, &m_InputWaveHdr[m_iInTailIndex], sizeof(WAVEHDR));
        if( m_fFirstCallOfWaveInAddBuffer )
         {
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "First call of waveInAddBuffer() during audio input; result=%d", (int)result );
#         endif
           m_fFirstCallOfWaveInAddBuffer = FALSE; // don't flood the log with these messages !
         }
        DOBINI();   // -> Sound_iSourceCodeLine
      }
     m_InBufPosition = 0;
     if( ++m_iInTailIndex >= m_NumInputBuffers)   // wrap around (circular buffer)
      {  m_iInTailIndex = 0;
      }
   }
  else // ( m_InBufPosition < m_InBufferSize ) :
   {   // not finished with the previous *INPUT*-buffer yet !
     // No-No: return m_pbInputBuffer + m_iInTailIndex*m_InBufferSize + m_InBufPosition;
     if( ppSoundBufferInfo != NULL )
      { *ppSoundBufferInfo = &m_InBufInfo[m_iInTailIndex];
      }
     return m_pbInputBuffer + m_iInTailIndex*m_InBufferSize;
   }
  // Arrived here: there are no bytes remaining from the previous 'fragment'...
  if( m_iInTailIndex != m_iInHeadIndex )  // ok, we have AT LEAST one filled buffer !
   { if( ppSoundBufferInfo != NULL )
      { *ppSoundBufferInfo = &m_InBufInfo[m_iInTailIndex];
      }
     return m_pbInputBuffer + m_iInTailIndex*m_InBufferSize;
   }
  // Arrived here: No filled buffer available.. WAIT for the next one ?
  if( iOptions & CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA )
   { if( WaitForInputData(m_iInputTimeout_ms) )
      { // arrived here: hooray, we've got at least one new fragment ("WAVEHDR") of samples..
        if( ppSoundBufferInfo != NULL )
         { *ppSoundBufferInfo = &m_InBufInfo[m_iInTailIndex];
         }
        return m_pbInputBuffer + m_iInTailIndex*m_InBufferSize;
      }
   }
  // Arrived here: No filled buffer, caller doesn't want to wait, or wait timed out:
  return NULL;
} // end CSound::WrapInBufPositionAndGetPointerToNextFragment()


//--------------------------------------------------------------------------
void CSound::GetDateAndTimeForInput(
           T_SoundBufferInfo *pSoundBufferInfo, // [in]  pSoundBufferInfo->i64HRTimerValue, etc
           T_ChunkInfo *pOutChunkInfo )         // [out] pOutChunkInfo->ldblUnixDateAndTime
  // Periodically(!!) called from the 'input reading function' to retrieve
  //  and absolute timestamp for the first sample in the current sample block.
  // The calling frequency depends on the APPLICATION's "audio chunk size",
  // which (in Spectrum Lab) is typically 4096 samples (for 11025 or 12000 samples/seconds);
  // the 'decision' is made in SoundThd_GetBestChunkSizeAndAllocateBuffers() .
{
  double t1, abs_t1;
  int    iNumSamplesPerBuffer;
  long double ldblUnixDateAndTime;

  iNumSamplesPerBuffer = GetNumSamplesPerInputBuffer();
  if( (pSoundBufferInfo != NULL ) && (pOutChunkInfo != NULL) && iNumSamplesPerBuffer>0 )
   { // Retrieve the timestamp from the audio-input-buffer (beware, awful jitter!) :
     ldblUnixDateAndTime = pSoundBufferInfo->ldblUnixDateAndTime; // time source: WaveInCallback(),
                           // may have been returned by SOUND_GetSystemDateAndTimeAsUnixSeconds(),
                           // thus this timestamp may be subject to extreme jitter !
                           // ASIO drivers may have provided their own (less jittering) time,
                           // converted into the Unix date-and-time format in

     // If there is NO TIMESTAMP delivered by the "low-level" driver,
     //    provide an 'idealized' timestamp here (locally) :
     if( ldblUnixDateAndTime <= 0.0 )
      {  ldblUnixDateAndTime = SOUND_GetSystemDateAndTimeAsUnixSeconds();
      }
     else  // use timestamp from WaveInCallback() or CSound_AsioCallback() :
      {
        // Subtract the length of ONE AUDIO BUFFER (in seconds) from the buffer's timestamp because
        // the 'timer' was queried in WaveInCallback() when the block was COMPLETE,
        // but ldblUnixDateAndTime shall apply to the FIRST sample in the block.
        if( m_dblNominalInputSampleRate > 0 ) // avoid div-by-zero, as usual..
         { ldblUnixDateAndTime -= (double)iNumSamplesPerBuffer / m_dblNominalInputSampleRate;
         }
        if( m_ldblSampleBasedInputTimestamp <= 0 )
         {  m_ldblSampleBasedInputTimestamp = ldblUnixDateAndTime; // set INITIAL value (only once, after starting)
         }
      } // end else < use timestamp from WaveInCallback() > ?


     // To avoid jitter in the timestamp(*), an AUDIO-SAMPLE-BASED time should be used.
     //  When checked on a windows XP system, there was a lot of jitter (*)
     //  in 'ldblUnixDateAndTime', probably caused by task switching
     //  and non-constant latencies, approximately 10..100 ms.
     //  If the timestamps are 'improved later' (using a GPS sync signal),
     //  it's sufficent to keep T_ChunkInfo.ldblUnixDateAndTime
     //  within +/- 499 milliseconds of the 'correct' timestamp.
     // (*) Jitter in the timestamps must be avoided because since 2012,
     //     the timestamps are used instead of the 'sample counter'
     //     in many of Spectrum Lab's signal processing modules .
     t1 = ldblUnixDateAndTime - m_ldblSampleBasedInputTimestamp; // -> [s]
     abs_t1 = fabs(t1);
     if( abs_t1<0.25 )
      { // Sample-based timestamp is reasonably close to the system-time:
        // Use the SAMPLE-BASED timestamp, which is almost jitter-free .
        // Slowly 'pull in' the sample-based timestamp,
        // with a maximum slew rate of 1 millisecond per call:
        t1 *= 0.05;  // only remove 1/20 th of the timestamp difference..
        if( t1<-1e-3 ) t1=-1e-3;
        if( t1> 1e-3 ) t1= 1e-3;
        // If m_ldblSampleBasedInputTimestamp lags ldblUnixDateAndTime,
        // m_ldblSampleBasedInputTimestamp is "too low" and d is positive, thus:
        m_ldblSampleBasedInputTimestamp += t1; // apply CORRECTION
        // Use this 'mostly sample-counter-based' time for the chunk-info.
        // Note: As long as the flag 'CHUNK_INFO_TIMESTAMPS_PRECISE'
        //       is NOT set in T_ChunkInfo.dwValidityFlags,
        //       none of the signal-processing modules in Spectrum Lab
        //       will use the timestamp as argument (angle) to generate
        //       sinewaves, oscillator signals for frequency converters, etc.
        //   Therefore, a small amount of jitter doesn't hurt much.
        ldblUnixDateAndTime = m_ldblSampleBasedInputTimestamp;
      }
     else // difference too large -> Let's do the time-warp again..
      { m_ldblSampleBasedInputTimestamp = ldblUnixDateAndTime;
        // (this may happen periodically, if the nominal sampling rate
        //  is 'totally off' the real sampling rate,
        //  which can only be measured COARSELY here
        //  due to the timestamp jitter)
        ++m_nTimingBreaks;
      }
     pOutChunkInfo->ldblUnixDateAndTime = ldblUnixDateAndTime;
     // At this point, we KNOW that the internally measured sampling rate,
     //   as well as the absolute timestamp set above are
     //   **NOT PRECISE** ("calibrated") but at least they are VALID now.
     // Spectrum Lab's GPS-based timing reference will do much better than this !
     // Thus, at THIS point: Declare the timestamp (in the T_ChunkInfo) as VALID,
     //                      but not as PRECISE :
     pOutChunkInfo->dwValidityFlags = // see c:\cbproj\SoundUtl\ChunkInfo.h ...
       ( pOutChunkInfo->dwValidityFlags & ~(CHUNK_INFO_TIMESTAMPS_PRECISE)  )
                                          | CHUNK_INFO_TIMESTAMPS_VALID;
     // Note: the SAMPLING RATE for the chunk-info is set somewhere else !
  }
} // end CSound::GetDateAndTimeForInput()


// Not used in SpectrumLab anymore, but still used in the 'WOLF GUI' (2013) :
//---------------------------------------------------------------------------
LONG CSound::InReadInt16( SHORT* pData, int Length)  // DEPRECATED (but still used)
  // Reads 'Length' samples of signed 16-bit integer data
  // from the opened soundcard input.
  //   returns:
  //   Length   if 'Length' samples were succesfully read
  //       0 =  if reaches the specified sample limit
  //      -1 =  if there is an error ( use GetError() to retrieve error )
{
 int i;
 BYTE *pb;
 union uBSHORT{    // byte-adressable 16-bit integer
      BYTE b[2];   // [0]=LSB, [1]=MSB (INTEL byte order)
      short i16;
   }bsTemp;
 T_SoundBufferInfo *pSoundBufferInfo;
 MMRESULT result;


   if( !m_InputOpen )      // return -1 if no inputs are active
    {
      SetInputError(  SOUNDIN_ERR_NOTOPEN );
      return -1;
    }
   if( m_InOverflow )    // has been set by WaveInCallback()
    {
      SetInputError(  SOUNDIN_ERR_OVERFLOW );
#if(0)
      InClose();         // happens quite often, really have to CLOSE now ??
      return -1;         // return -1 if overflow
#else
      m_InOverflow = false;
#endif
    }

  DOBINI();  // -> Sound_iSourceCodeLine


  if( m_iInTailIndex == m_iInHeadIndex )  // if no buffer is filled ...
   { if( ! WaitForInputData(m_iInputTimeout_ms) ) // ... then wait, but only for a limited time
      { DOBINI();  // -> Sound_iSourceCodeLine
        return -1;
      }
   }
  // here if there is data to retrieve
  DOBINI();  // -> Sound_iSourceCodeLine
  // ex: pb = m_pbInputBuffer + m_iInTailIndex*m_InBufferSize;  // YHF: use local pointer for speed !
  pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_DONT_WAIT, &pSoundBufferInfo);
  if( pb==NULL ) // oops... no filled 'input' data available ?!
   { DOBINI();   // -> Sound_iSourceCodeLine
     return -1;
   }
  if( m_InFormat.wBitsPerSample == 16 )
   {
     for( i=0; i < (Length*m_InFormat.nChannels); i++ )
      {
        if (m_InBufPosition & 1)
         { // very suspicious..
           SetInputError(  ERROR_WORD_AT_ODD_ADDRESS ); // -> m_ErrorCodeIn + "chance for a COMMON breakpoint"
         }
        bsTemp.b[0] = pb[m_InBufPosition++];
        bsTemp.b[1] = pb[m_InBufPosition++];
        *(pData + i) = bsTemp.i16;
        if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
         {                                       // so use the next (*if* there's one)
           pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
           if( pb==NULL ) // oops... no more filled 'input' data available ?!
            { DOBINI();   // -> Sound_iSourceCodeLine
              return -1;
            }
         }
      }
   }
  else // not 16 bits per sample, but most likely 8 bits per sample:
  if( m_InFormat.wBitsPerSample == 8 )
   {
     bsTemp.i16 = 0;
     for( i=0; i < (Length*m_InFormat.nChannels); i++ )
      { bsTemp.b[1] = (BYTE)( pb[m_InBufPosition++] - 128 );
        *(pData + i) = bsTemp.i16;
      }
   }
  if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
   {                                       // so add it back in to the Queue
     m_InputWaveHdr[m_iInTailIndex].dwBytesRecorded = 0;
     if( m_hwvin != NULL )
      { // only if MMSYSTEM used : call windows API to let this buffer be filled by input device
        DOBINI();  // -> Sound_iSourceCodeLine
        result = waveInAddBuffer(m_hwvin, &m_InputWaveHdr[m_iInTailIndex], sizeof(WAVEHDR));
        if( m_fFirstCallOfWaveInAddBuffer )
         {
#         if( SWI_UTILITY1_INCLUDED )
           UTL_WriteRunLogEntry( "First call of waveInAddBuffer() during audio input; result=%d", (int)result );
#         endif
           m_fFirstCallOfWaveInAddBuffer = FALSE; // don't flood the log with these messages !
         }
        DOBINI();
      }
     m_InBufPosition = 0;
     if( ++m_iInTailIndex >= m_NumInputBuffers)   // handle wrap around
           m_iInTailIndex = 0;
    }
  m_i64NumSamplesRead += Length;
  if( (m_InSampleLimit != 0) && (m_i64NumSamplesRead >= m_InSampleLimit) )
   {
     InClose();
     return 0;
   }
  DOBINI();
  return Length;
} // end CSound::InReadInt16()


//---------------------------------------------------------------------------
int Sound_ConvSamples_I8toFloat( // converts a block of 8-bit-samples into floating point numbers
        int iSoundInputSampleFormat, // [in] here: only EIGHT BIT INTEGER types (SOUND_SAMPLE_FMT_U8,SOUND_SAMPLE_FMT_S8)
                                 //                with or without bitflag SOUND_SAMPLE_FMT_SYNC_PATTERN .
        BYTE    * pbSource,      // [in] source block (don't assume anything about alignment!)
        int       nBytesInSource,// [in] number of BYTES in the above source block
        int nSourceChannels,     // [in] number of CHANNELS (in the SOURCE data)
        int iMaxSamplesPerChannel, // [in] maximum number of samples for each of the blocks below
        T_Float * pLeftData,     // [out] samples for 'left' audio channel (or "I")
        T_Float * pRightData,    // [out] samples for 'right' audio channel (or "Q")
        int     * piNumSamplesEmitted) // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
  // First used (in CSound::InReadStereo) when the source was a SERIAL PORT .
  // Returns THE NUMBER OF BYTES ACTUALLY CONSUMED (from *pbSource) .
  // The SOURCE BLOCK (an array of bytes) cannot be a circular FIFO.
  //  (if it is in fact a circular FIFO, the CALLER must call this function twice:
  //   once for the data BEFORE the circular index wrap,
  //   then -if necessary- shuffle the data around to achive a 'linear' block,
  //   then call Sound_ConvSamples_XYZtoFloat() a second time for the remaining samples).
  // The same applies to an awful lot of other Sound_ConvSamples_XYZtoFloat()-functions.
{ T_Float fltOffset, fltFactor;
  int nDestChannels   = (pRightData != NULL) ? 2 : 1;
  int nBytesPerSamplePoint = 1 * nSourceChannels;
  int nSamplesEmitted = 0;
  BYTE *pb     = pbSource;
  BYTE *pbLast = pbSource + nBytesInSource - nBytesPerSamplePoint;
        // Example: 2 bytes in source, two bytes per sample point -> pbLast = pb = only ONE POSSIBLE SOURCE-INDEX !
  BYTE bSyncPattern[1];

  if( nDestChannels > nSourceChannels )
   {  nDestChannels = nSourceChannels;
   }
  switch( iSoundInputSampleFormat & SOUND_SAMPLE_FMT_MASK )  // prepare scaling factors, etc..
   { // conversion forumla (principle) : output = (float)input * factor + offset;
     //             output value range : -1.0 ... +1.0  (floating point)
     case SOUND_SAMPLE_FMT_U8 :
        fltOffset = -1.0;          // 0  .. +255 -> -1.0 .. +1.0
        fltFactor = 1.0 / 128.0;
        bSyncPattern[0] = 0xFF;    // sync pattern = UPPER ENDSTOP (255 must be excluded from the sample values)
        break;
     case SOUND_SAMPLE_FMT_S8 :
        fltOffset = 0.0;           // -127..+127 -> -1.0 .. +1.0
        fltFactor = 1.0 / 128.0;
        bSyncPattern[0] = 0x80;    // sync pattern = NEGATIVE ENDSTOP (-128 must be excluded from the sample values)
        break;
     default :    // any other input type is NOT supported by this function
        return 0;
   } // end switch( iSoundInputSampleFormat & .. )

  if( iSoundInputSampleFormat & SOUND_SAMPLE_FMT_SYNC_PATTERN )
   {
     if( nDestChannels == 1 )
      {  while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
          {
            if( pb[0] == bSyncPattern[0] )
             { // skip the sync pattern, do NOT emit a sample
             }
            else
             { *pLeftData++ = (T_Float)pb[0] * fltFactor + fltOffset;
               nSamplesEmitted++;
             }
            pb++;
          } // end while
      }
     else // two channels, with sync pattern :
      {  while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
          {
            if( pb[0] == bSyncPattern[0] )
             { pb++;        // skip the one-byte sync pattern
             }
            else if( pb[1] == bSyncPattern[0] )
             { pb += 2;     // skip one byte of garbage PLUS the sync-pattern
             }
            else
             { *pLeftData++  = (T_Float)pb[0] * fltFactor + fltOffset;
               *pRightData++ = (T_Float)pb[1] * fltFactor + fltOffset;
               nSamplesEmitted++;
               pb += 2;
             }
          } // end while
      }   // end else < two channels, 8 bit, with sync pattern >
   } // end if < look for the SYNC PATTERN ? >
  else // do NOT look for the SYNC PATTERN :
   {
     while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
      {
        *pLeftData++     = (T_Float)pb[0] * fltFactor + fltOffset;
        if( nDestChannels > 1 )
         { *pRightData++ = (T_Float)pb[1] * fltFactor + fltOffset;
         }
        pb  += nBytesPerSamplePoint;
        nSamplesEmitted++;
      } // end while
   } // end else < no sync pattern >

  if( piNumSamplesEmitted != NULL )
   { *piNumSamplesEmitted = nSamplesEmitted;
   }

  return pb - pbSource; // > Returns THE NUMBER OF BYTES ACTUALLY CONSUMED (from *pbSource)

} // end Sound_ConvSamples_I8toFloat()

//---------------------------------------------------------------------------
int Sound_ConvSamples_I16toFloat( // converts a block of 16-bit-samples into floating point numbers
        int iSoundInputSampleFormat, // [in] here: only 16-BIT INTEGER types (SOUND_SAMPLE_FMT_U16,SOUND_SAMPLE_FMT_S16, .._LE, .._BE)
                                 //                with or without bitflag SOUND_SAMPLE_FMT_SYNC_PATTERN .
        BYTE    * pbSource,      // [in] source block (don't assume anything about alignment!)
        int       nBytesInSource,// [in] number of BYTES in the above source block
        int nSourceChannels,     // [in] number of CHANNELS (in the SOURCE data)
        int iMaxSamplesPerChannel, // [in] maximum number of samples for each of the blocks below
        T_Float * pLeftData,     // [out] samples for 'left' audio channel (or "I")
        T_Float * pRightData,    // [out] samples for 'right' audio channel (or "Q")
        int     * piNumSamplesEmitted) // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
  // Details : See Sound_ConvSamples_I8ToFloat() .
{ T_Float fltOffset, fltFactor;
  int nDestChannels = (pRightData != NULL) ? 2 : 1;
  int nBytesPerSamplePoint = 2 * nSourceChannels;
  int nSamplesEmitted = 0;
  int iSampleFmt2 =  iSoundInputSampleFormat & SOUND_SAMPLE_FMT_MASK;
  BOOL fBigEndian = (iSampleFmt2==SOUND_SAMPLE_FMT_U16_BE) || (iSampleFmt2==SOUND_SAMPLE_FMT_S16_BE);
  BYTE *pb     = pbSource;
  BYTE *pbLast = pbSource + nBytesInSource - nBytesPerSamplePoint;
        // Example: 2 bytes in source, two bytes per sample point -> pbLast = pb = only ONE POSSIBLE SOURCE-INDEX !
  BYTE bSyncPattern[2];
  int  iValue, iSignBits;
  if( nDestChannels > nSourceChannels )
   {  nDestChannels = nSourceChannels;
   }
  if( fBigEndian ) // most significant byte first :
   { bSyncPattern[0] = 0x80;
     bSyncPattern[1] = 0x00;
   }
  else             // least significant byte first :
   { bSyncPattern[1] = 0x80;
     bSyncPattern[0] = 0x00;
   }
  switch( iSampleFmt2 )  // prepare scaling factors, etc..
   { // conversion forumla (principle) : output = (float)input * factor + offset;
     //             output value range : -1.0 ... +1.0  (floating point)
     case SOUND_SAMPLE_FMT_U16_LE :
     case SOUND_SAMPLE_FMT_U16_BE :
        fltOffset = -1.0;
        fltFactor = 1.0 / 32768.0;
        iSignBits = 0x00000000;
        bSyncPattern[0] = bSyncPattern[1] = 0xFF;
        break;
     case SOUND_SAMPLE_FMT_S16_LE :
     case SOUND_SAMPLE_FMT_S16_BE :
        fltOffset = 0.0;
        fltFactor = 1.0 / 32768.0;
        iSignBits = 0xFFFF8000;
        break;
     default :    // any other input type is NOT supported by this function
        return 0;
   } // end switch ( sample format )

  while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
   {

     if( iSoundInputSampleFormat & SOUND_SAMPLE_FMT_SYNC_PATTERN )
      { // Check for sync pattern in the next <nBytesPerSamplePoint> bytes .
        // Since the source may not be word-aligned, we're doomed to do this byte-wise  :
        if( (pb[0]==bSyncPattern[0]) && (pb[1]==bSyncPattern[1]) )
         { pb += 2;   // skip the sync-pattern
           continue;  // .. with the next 'while' loop
         }
        if( (pb+1)<=pbLast )  // look for sync-pattern, offset by ONE BYTE
         { if( (pb[1]==bSyncPattern[0]) && (pb[2]==bSyncPattern[1]) )
            { pb += 3; // skip the sync-pattern plus ONE byte of garbage
              continue;
            }
         }
        if( nDestChannels>1 ) // look for sync-pattern, offset by TWO BYTES
         { if( (pb[2]==bSyncPattern[0]) && (pb[3]==bSyncPattern[1]) )
            { pb += 4; // skip the sync-pattern plus TWO bytes of garbage
              continue;
            }
           if( (pb+3)<=pbLast ) // look for sync-pattern, offset by THREE BYTES
            { if( (pb[3]==bSyncPattern[0]) && (pb[4]==bSyncPattern[1]) )
               { pb += 5; // skip the sync-pattern plus THREE bytes of garbage
                 continue;
               }
            }
         }
      } // end if( iSoundInputSampleFormat & SOUND_SAMPLE_FMT_SYNC_PATTERN )


     if( fBigEndian )
      { iValue = (int)pb[1] | ((int)pb[0]<<8);
      }
     else // not big endian but little endian (LSByte first, which is the default for Intel CPUs) :
      { iValue = (int)pb[0] | ((int)pb[1]<<8);
      }
     if( iValue &  iSignBits )
      {  iValue |= iSignBits;
      }
     *pLeftData++ = (T_Float)iValue * fltFactor + fltOffset;
     if( nDestChannels > 1 )
      {
        if( fBigEndian )
         { iValue = (int)pb[3] | ((int)pb[2]<<8);
         }
        else // not big endian but little endian (LSByte first, which is the default for Intel CPUs) :
         { iValue = (int)pb[2] | ((int)pb[3]<<8);
         }
        if( iValue &  iSignBits )
         {  iValue |= iSignBits;
         }
        *pRightData++ = (T_Float)iValue * fltFactor + fltOffset;
      }
     pb += nBytesPerSamplePoint;
     nSamplesEmitted++;
   } // end while

  if( piNumSamplesEmitted != NULL )
   { *piNumSamplesEmitted = nSamplesEmitted;
   }

  return pb - pbSource; // > Returns THE NUMBER OF BYTES ACTUALLY CONSUMED (from *pbSource)

} // end Sound_ConvSamples_I16toFloat()

//---------------------------------------------------------------------------
int Sound_ConvSamples_I24toFloat( // converts a block of 24-bit-samples into floating point numbers
        int iSoundInputSampleFormat, // [in] here: only 24-BIT INTEGER types (SOUND_SAMPLE_FMT_U24,SOUND_SAMPLE_FMT_S24, .._LE, .._BE)
                                 //                with or without bitflag SOUND_SAMPLE_FMT_SYNC_PATTERN .
        BYTE    * pbSource,      // [in] source block (don't assume anything about alignment!)
        int       nBytesInSource,// [in] number of BYTES in the above source block
        int nSourceChannels,     // [in] number of CHANNELS (in the SOURCE data)
        int iMaxSamplesPerChannel, // [in] maximum number of samples for each of the blocks below
        T_Float * pLeftData,     // [out] samples for 'left' audio channel (or "I")
        T_Float * pRightData,    // [out] samples for 'right' audio channel (or "Q")
        int     * piNumSamplesEmitted) // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
  // Details : See Sound_ConvSamples_I8ToFloat() .
{ T_Float fltOffset, fltFactor;
  int nDestChannels   = (pRightData != NULL) ? 2 : 1;
  int nBytesPerSamplePoint = 3 * nSourceChannels;
  int nSamplesEmitted = 0;
  int iSampleFmt2 =  iSoundInputSampleFormat & SOUND_SAMPLE_FMT_MASK;
  BOOL fBigEndian = (iSampleFmt2==SOUND_SAMPLE_FMT_U24_BE) || (iSampleFmt2==SOUND_SAMPLE_FMT_S24_BE);
  BYTE *pb     = pbSource;
  BYTE *pbLast = pbSource + nBytesInSource - nBytesPerSamplePoint;
        // Example: 3 bytes in source, 3 bytes per sample point -> pbLast = pb = only ONE POSSIBLE SOURCE-INDEX !
  BYTE bSyncPattern[3];
  int  iValue, iSignBits;
  if( nDestChannels > nSourceChannels )
   {  nDestChannels = nSourceChannels;
   }

  if( fBigEndian ) // most significant byte first :
   { bSyncPattern[0] = 0x80;
     bSyncPattern[1] = 0x00;
     bSyncPattern[2] = 0x00;
   }
  else             // least significant byte first :
   { bSyncPattern[2] = 0x80;
     bSyncPattern[1] = 0x00;
     bSyncPattern[0] = 0x00;
   }

  switch( iSampleFmt2 )  // prepare scaling factors, etc..
   { // conversion forumla (principle) : output = (float)input * factor + offset;
     //             output value range : -1.0 ... +1.0  (floating point)
     case SOUND_SAMPLE_FMT_U24_LE :
     case SOUND_SAMPLE_FMT_U24_BE :
        fltOffset = -1.0;
        fltFactor = 1.0 / 8388608.0;  // 1 / 2^23
        iSignBits = 0x00000000;
        break;
     case SOUND_SAMPLE_FMT_S24_LE :
     case SOUND_SAMPLE_FMT_S24_BE :
        fltOffset = 0.0;
        fltFactor = 1.0 / 8388608.0;  // 1 / 2^23
        iSignBits = 0xFF800000;
        break;
     default :    // any other input type is NOT supported by this function
        return 0;
   } // end switch( sample format )

  while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
   {
     if( fBigEndian ) // most significant byte first :
      { iValue = (int)pb[2] | ((int)pb[1]<<8) | ((int)pb[0]<<16);
      }
     else // not big endian but little endian (LSByte first, which is the default for Intel CPUs) :
      { iValue = (int)pb[0] | ((int)pb[1]<<8) | ((int)pb[2]<<16);
      }
     if( iValue &  iSignBits )
      {  iValue |= iSignBits;
      }
     *pLeftData++ = (T_Float)iValue * fltFactor + fltOffset;
     if( nDestChannels > 1 )
      {
        if( fBigEndian ) // most significant byte first :
         { iValue = (int)pb[5] | ((int)pb[4]<<8) | ((int)pb[3]<<16);
         }
        else // not big endian but little endian (LSByte first, which is the default for Intel CPUs) :
         { iValue = (int)pb[3] | ((int)pb[4]<<8) | ((int)pb[5]<<16);
         }
        if( iValue &  iSignBits )
         {  iValue |= iSignBits;
         }
        *pRightData++ = (T_Float)iValue * fltFactor + fltOffset;
      }
     pb += nBytesPerSamplePoint;
     nSamplesEmitted++;
   } // end while

  if( piNumSamplesEmitted != NULL )
   { *piNumSamplesEmitted = nSamplesEmitted;
   }

  return pb - pbSource; // > Returns THE NUMBER OF BYTES ACTUALLY CONSUMED (from *pbSource)

} // end Sound_ConvSamples_I24toFloat()

//---------------------------------------------------------------------------
int Sound_ConvSamples_I32toFloat( // converts a block of 32-bit-samples into floating point numbers
        int iSoundInputSampleFormat, // [in] here: only 32-BIT INTEGER types (SOUND_SAMPLE_FMT_U32,SOUND_SAMPLE_FMT_S32, .._LE, .._BE)
                                 //                with or without bitflag SOUND_SAMPLE_FMT_SYNC_PATTERN .
        BYTE    * pbSource,      // [in] source block (don't assume anything about alignment!)
        int       nBytesInSource,// [in] number of BYTES in the above source block
        int nSourceChannels,     // [in] number of CHANNELS (in the SOURCE data)
        int iMaxSamplesPerChannel, // [in] maximum number of samples for each of the blocks below
        T_Float * pLeftData,     // [out] samples for 'left' audio channel (or "I")
        T_Float * pRightData,    // [out] samples for 'right' audio channel (or "Q")
        int     * piNumSamplesEmitted) // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
  // Details : See Sound_ConvSamples_I8ToFloat() .
{ T_Float fltOffset, fltFactor;
  int nDestChannels   = (pRightData != NULL) ? 2 : 1;
  int nBytesPerSamplePoint = 4 * nSourceChannels;
  int nSamplesEmitted = 0;
  int iSampleFmt2 =  iSoundInputSampleFormat & SOUND_SAMPLE_FMT_MASK;
  BOOL fBigEndian = (iSampleFmt2==SOUND_SAMPLE_FMT_U32_BE) || (iSampleFmt2==SOUND_SAMPLE_FMT_S32_BE);
  BYTE *pb     = pbSource;
  BYTE *pbLast = pbSource + nBytesInSource - nBytesPerSamplePoint;
        // Example: 4 bytes in source, 4 bytes per sample point -> pbLast = pb = only ONE POSSIBLE SOURCE-INDEX !
  BYTE bSyncPattern[4];
  int  iValue;
  if( nDestChannels > nSourceChannels )
   {  nDestChannels = nSourceChannels;
   }

  if( fBigEndian ) // most significant byte first :
   { bSyncPattern[0] = 0x80;
     bSyncPattern[1] = 0x00;
     bSyncPattern[2] = 0x00;
     bSyncPattern[3] = 0x00;
   }
  else             // least significant byte first :
   { bSyncPattern[3] = 0x80;
     bSyncPattern[2] = 0x00;
     bSyncPattern[1] = 0x00;
     bSyncPattern[0] = 0x00;
   }

  switch( iSoundInputSampleFormat )  // prepare scaling factors, etc..
   { // conversion forumla (principle) : output = (float)input * factor + offset;
     //             output value range : -1.0 ... +1.0  (floating point)
     case SOUND_SAMPLE_FMT_U32_LE :
     case SOUND_SAMPLE_FMT_U32_BE :
        fltOffset = -1.0;
        fltFactor = 1.0 / (32768.0 * 65536.0);
        break;
     case SOUND_SAMPLE_FMT_S32_LE :
     case SOUND_SAMPLE_FMT_S32_BE :
        fltOffset = 0.0;
        fltFactor = 1.0 / (32768.0 * 65536.0);
        break;
     default :    // any other input type is NOT supported by this function
        return 0;
   } // end switch( sample format )

  while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
   {
     if( fBigEndian ) // most significant byte first :
      { iValue = (int)pb[3] | ((int)pb[2]<<8) | ((int)pb[1]<<16) | ((int)pb[0]<<24);
      }
     else // not big endian but little endian (LSByte first, which is the default for Intel CPUs) :
      { iValue = (int)pb[0] | ((int)pb[1]<<8) | ((int)pb[2]<<16) | ((int)pb[3]<<24);
      }
     *pLeftData++ = (T_Float)iValue * fltFactor + fltOffset;
     if( nDestChannels > 1 )
      {
        if( fBigEndian ) // most significant byte first :
         { iValue = (int)pb[7] | ((int)pb[6]<<8) | ((int)pb[5]<<16) | ((int)pb[4]<<24);
         }
        else // not big endian but little endian (LSByte first, which is the default for Intel CPUs) :
         { iValue = (int)pb[4] | ((int)pb[5]<<8) | ((int)pb[6]<<16) | ((int)pb[7]<<24);
         }
        *pRightData++ = (T_Float)iValue * fltFactor + fltOffset;
      }
     pb += nBytesPerSamplePoint;
     nSamplesEmitted++;
   } // end while

  if( piNumSamplesEmitted != NULL )
   { *piNumSamplesEmitted = nSamplesEmitted;
   }

  return pb - pbSource; // > Returns THE NUMBER OF BYTES ACTUALLY CONSUMED (from *pbSource)

} // end Sound_ConvSamples_I32toFloat()

//---------------------------------------------------------------------------
int Sound_ConvSamples_F32toFloat( // converts a block of 32-bit-FLOATING POINT samples into the host's native format
        int iSoundInputSampleFormat, // [in] here: only 32-BIT INTEGER types (SOUND_SAMPLE_FMT_F32_LE, SOUND_SAMPLE_FMT_F32_BE)
                                 //                with or without bitflag SOUND_SAMPLE_FMT_SYNC_PATTERN .
        BYTE    * pbSource,      // [in] source block (don't assume anything about alignment!)
        int       nBytesInSource,// [in] number of BYTES in the above source block
        int nSourceChannels,     // [in] number of CHANNELS (in the SOURCE data)
        int iMaxSamplesPerChannel, // [in] maximum number of samples for each of the blocks below
        T_Float * pLeftData,     // [out] samples for 'left' audio channel (or "I")
        T_Float * pRightData,    // [out] samples for 'right' audio channel (or "Q")
        int     * piNumSamplesEmitted) // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
  // Details : See Sound_ConvSamples_I8ToFloat() .
{
  int nDestChannels   = (pRightData != NULL) ? 2 : 1;
  int nBytesPerSamplePoint = 4 * nSourceChannels;
  int nSamplesEmitted = 0;
  int iSampleFmt2 =  iSoundInputSampleFormat & SOUND_SAMPLE_FMT_MASK;
  BOOL fBigEndian = (iSampleFmt2==SOUND_SAMPLE_FMT_F32_BE) || (iSampleFmt2==SOUND_SAMPLE_FMT_F32_BE);
  BYTE *pb     = pbSource;
  BYTE *pbLast = pbSource + nBytesInSource - nBytesPerSamplePoint;
        // Example: 4 bytes in source, 4 bytes per sample point -> pbLast = pb = only ONE POSSIBLE SOURCE-INDEX !
  BYTE bSyncPattern[4];
  union { float f; BYTE b[4]; } value;

  if( nDestChannels > nSourceChannels )
   {  nDestChannels = nSourceChannels;
   }
  if( fBigEndian ) // most significant byte first :
   { bSyncPattern[0] = 0x80;
     bSyncPattern[1] = 0x00;
     bSyncPattern[2] = 0x00;
     bSyncPattern[3] = 0x00;
   }
  else             // least significant byte first :
   { bSyncPattern[3] = 0x80;
     bSyncPattern[2] = 0x00;
     bSyncPattern[1] = 0x00;
     bSyncPattern[0] = 0x00;
   }

  while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
   {
     if( fBigEndian ) // most significant byte first :
      { value.b[0] = pb[3];
        value.b[1] = pb[2];
        value.b[2] = pb[1];
        value.b[3] = pb[0];
      }
     else // not big endian but little endian (LSByte first, which is the default for Intel CPUs) :
      { value.b[0] = pb[0];
        value.b[1] = pb[1];
        value.b[2] = pb[2];
        value.b[3] = pb[3];
      }
     *pLeftData++ = value.f;
     if( nDestChannels > 1 )
      {
        if( fBigEndian ) // most significant byte first :
         { value.b[0] = pb[7];
           value.b[1] = pb[6];
           value.b[2] = pb[5];
           value.b[3] = pb[4];
         }
        else // not big endian but little endian (LSByte first, which is the default for Intel CPUs) :
         { value.b[0] = pb[4];
           value.b[1] = pb[5];
           value.b[2] = pb[6];
           value.b[3] = pb[7];
         }
        *pRightData++ = value.f;
      }
     pb += nBytesPerSamplePoint;
     nSamplesEmitted++;
   } // end while

  if( piNumSamplesEmitted != NULL )
   { *piNumSamplesEmitted = nSamplesEmitted;
   }

  return pb - pbSource; // > Returns THE NUMBER OF BYTES ACTUALLY CONSUMED (from *pbSource)

} // end Sound_ConvSamples_F32toFloat()

//---------------------------------------------------------------------------
int Sound_ConvSamples_IQ12toFloat( // converts a block of 2 * 12-bit unsigned integers into floating point numbers
        BYTE    * pbSource,      // [in] source block (don't assume anything about alignment!)
        int       nBytesInSource,// [in] number of BYTES in the above source block
        int iMaxSamplesPerChannel, // [in] maximum number of samples for each of the blocks below
        T_Float * pLeftData,     // [out] samples for 'left' audio channel (or "I")
        T_Float * pRightData,    // [out] samples for 'right' audio channel (or "Q")
        int     * piNumSamplesEmitted) // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
  // Internals : See Sound_ConvSamples_I8ToFloat() .
  // Specification of DL4YHF's "IQ12"-format, as used in the PIC12F675 firmware:
  //     http://www.qsl.net/dl4yhf/soundutl/serpicad.htm  (partly quoted below).
{
  int nDestChannels   = (pRightData != NULL) ? 2 : 1;
  int nSamplesEmitted = 0;
  int nBytesPerSamplePoint = 4;
  BYTE *pb     = pbSource;
  BYTE *pbLast = pbSource + nBytesInSource - nBytesPerSamplePoint;
        // Example: 4 bytes in source, 4 bytes per sample point -> pbLast = pb = only ONE POSSIBLE SOURCE-INDEX !

  union { WORD w[2]; BYTE b[4]; } sample_point;
  T_Float fltOffset = -1.0;
  T_Float fltFactor = 1.0 / 2048.0;


  while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
   {
     // "Packet" structure, as used in DL4YHF's simple PIC-based ADC for the serial port:
     // > Byte[0] = Sync/Status
     // >   Usually 0xFF which looks like a "STOP BIT" to the receiver. Used for frame
     // >   sync, but also for BYTE sync if receiver was turned on too late. (...)
     // >   In future firmware versions, the PIC will be react on commands (which are
     // >   sent from PC to PIC), then the "Sync/Status" byte will carry a response code
     // >   like 0xFE. In that case, byte[1] .. byte[3] may have different meanings.
     // > Byte[1] = I-channel, LSB
     // >   Least significant bits of 1st analog input channel (bits I7..I0).
     // > Byte[2] = Q-channel, LSB
     // >   Least significant bits of 2nd analog input channel (bits Q7..Q0).
     // > Byte[3] : Most significant bits, combined.
     // >   Bits 3..0 = Most significant bits of 1st channel (I11..I8),
     // >   Bits 7..4 = Most significant bits of 2nd hannel (Q11..Q8).
     // After some bit-fiddling, both 'I' and 'Q' are UNSIGNED 12-bit integers.
     //
     // Because the ADC may have been running much longer than SL,
     //  the first byte received here may be anything - *not* just the header !
     //
     if( pb[0] != 0xFF ) // out of sync .. ignore anything until the next header !
      { ++pb;  // (note: pb[1..3] may contain 0xFF, so it may take some time to sync-in)
      }
     else // ok, pb[0] contains 0xFF which is the header of the 4-byte sample group...
      { sample_point.b[0] = pb[1];               // 8 LSBits of "I"
        sample_point.b[1] = pb[3] & 0x0F;        // 4 MSBits of "I"
        sample_point.b[2] = pb[2];               // 8 LSBits of "Q"
        sample_point.b[3] =(pb[3] >> 4) & 0x0F;  // 4 MSBits of "Q"
        // At this point, if the ADC is properly biased (and DC-free),
        //    both sample_point.w[0] and sample_point.w[1]
        //    should contain approximately 2048 without an AC input signal.
        //    With DL4YHF's PIC12F675-based ADC, they did,
        //    and the visual noise level in 1 Hz bandwidth was about -96 dBfs .
        *pLeftData++ = (T_Float)sample_point.w[0] * fltFactor + fltOffset;
        if( nDestChannels > 1 )
         { *pRightData++ = (T_Float)sample_point.w[1] * fltFactor + fltOffset;
         }
        pb += nBytesPerSamplePoint;
        nSamplesEmitted++;
      }
   } // end while

  if( piNumSamplesEmitted != NULL )
   { *piNumSamplesEmitted = nSamplesEmitted;
   }

  return pb - pbSource; // > Returns THE NUMBER OF BYTES ACTUALLY CONSUMED (from *pbSource)

} // end Sound_ConvSamples_IQ12toFloat()

//---------------------------------------------------------------------------
int Sound_ConvSamples_GPSDO3toFloat( // converts a block of 'GPSDO3' frames into floating point samples
        BYTE    * pbSource,      // [in] source block (don't assume anything about alignment!)
        int       nBytesInSource,// [in] number of BYTES in the above source block
        int iMaxSamplesPerChannel, // [in] maximum number of samples for each of the blocks below
        T_Float * pLeftData,     // [out] samples for 'left' audio channel (or "I")
        T_Float * pRightData,    // [out] samples for 'right' audio channel (or "Q")
        int     * piNumSamplesEmitted) // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
  // Specification of the "GPSDO3" sample format, first used in the PIC16F1783 firmware,
  //               C:\pic\GPSDO\gpsdo_pic_main.c, also in  C:\cbproj\SoundUtl\Sound.cpp,
  //     published also in http://qsl.net/dl4yhf/gpsdo/html/gpsdo_pic.htm :
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Short form :  3- or 5-byte frame ........                   OPTIONAL: quadrature component or 2nd channel
  //        Byte[0] = Header   Byte[1]="I",LSB  Byte[2]="I",MSB  Byte[3]="Q",LSB  Byte[4]="Q",MSB
  //  bits:   7 6 5 4 3 2 1 0  7 6 5 4 3 2 1 0  7 6 5 4 3 2 1 0  7 6 5 4 3 2 1 0  7 6 5 4 3 2 1 0
  //          | |___| :.: |_|  | |___________|  | |___________|  | |___________|  | |___________|
  //          |  AUX  (Q) "I"  | "I" bits 8..2  | "I" bits 15..9 | "Q" bits 8..2  | "Q" bits 15..9
  //          |        LSBits  |                |                |                |
  //        always "1" (sync)  always "0"       always "0"       always "0"       always "0"
  //
  // "AUX": Auxiliary data, multiplexed, may extend over many consecutive frames.
  //      3-bit binary codes ...
  //        111 = "idle" (must preceed any of the codes below, otherwise they are DATA, not OPCODES)
  //        110 = GPS pulse (after '111')
  //        101 = begin of a 'multiplexed' packet, variable-length opcode and data
  //                in byte[0], bit 6..4 of the FOLLOWING frames
  //              (used in the PIC16F1783 firmware because the 'serial channel bandwidth'
  //               and the lack of an interrupt-driven transmit buffer didn't allow
  //               to insert EXTRA BYTES in the sample-frame shown above)
  //        100, 011, 010, 001, 000 = reserved opcodes (for future use)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
{

  int nDestChannels   = (pRightData != NULL) ? 2 : 1;
  int nSamplesEmitted = 0;
  int nBytesPerSamplePoint = 5; // actually, the "number of bytes per sample point" may be 3 or 5 (depending on real or I/Q data)
  BYTE *pb     = pbSource;
  BYTE *pbLast = pbSource + nBytesInSource - nBytesPerSamplePoint;
        // Example: 5 bytes in source, 5 bytes per sample point -> pbLast = pb = only ONE POSSIBLE SOURCE-INDEX !

  union { short i16[2]; BYTE b[4]; } sample_point;
  T_Float fltFactor = 1.0 / 32767.0;


  while( (pb<=pbLast) && (nSamplesEmitted<iMaxSamplesPerChannel) )
   {
     //
     // Because the ADC may have been running before SL was started,
     //  the first byte received here may be anything - *not* just the header !
     //
     if( ( (pb[0] & 0x80) == 0 ) // out of sync .. ignore anything until the next header !
       ||( (pb[1] & 0x80) != 0 )
       ||( (pb[2] & 0x80) != 0 ) ) // MSBits in the first 3 bytes of a 3- or 5-byte-frame ok ?
      { ++pb;  // <-- this should only happen AFTER STARTING. If it happens later,
               //     an unknown number of samples has been lost !
      }
     else // ok, pb[0] contains 0xFF which is the header of the 4-byte sample group...
      { sample_point.b[0] = (  pb[0] & 0x03)     // 2 least significant bits of "I"
                          | ( (pb[1] & 0x3F)<<2);// 6 low   significant bits of "I"
        sample_point.b[1] = ( (pb[1] & 0x40)>>6) // bit 8 of "I"
                          | ( (pb[2] & 0x7F)<<1);// 7 most significant bits  of "I"
        // If the MSBits of the next TWO bytes are also zero, it's a 5-byte frame :
        if( ( (pb[2] & 0x80) == 0 ) && ( (pb[3] & 0x80) == 0 ) )
         { sample_point.b[2] =((pb[0] & 0x0C)>>2)  // 2 least significant bits of "Q"
                             |((pb[3] & 0x3F)<<2); // 6 low   significant bits of "Q"
           sample_point.b[3]= ((pb[3] & 0x40)>>6)  // bit 8 of "Q"
                             |((pb[4] & 0x7F)<<1); // 7 most significant bits  of "Q"
           nBytesPerSamplePoint = 5;
         }
        else  // not a 5- but a 3-byte frame (not "I/Q" but real input from the PIC16F1783 variant)
         { sample_point.b[2]    = 0;
           sample_point.b[3]    = 0;
           nBytesPerSamplePoint = 3;
         }
        // At this point, if the ADC is properly biased (and DC-free),
        //    both sample_point.i16[0] and sample_point.i16[1]
        //    should contain approximately ZERO without an AC input signal.
        *pLeftData++ = (T_Float)sample_point.i16[0] * fltFactor;
        if( nDestChannels > 1 )
         { *pRightData++ = (T_Float)sample_point.i16[1] * fltFactor;
         }
        pb += nBytesPerSamplePoint;
        nSamplesEmitted++;
      }
   } // end while

  if( piNumSamplesEmitted != NULL )
   { *piNumSamplesEmitted = nSamplesEmitted;
   }

  return pb - pbSource; // > Returns THE NUMBER OF BYTES ACTUALLY CONSUMED (from *pbSource)

} // end Sound_ConvSamples_GPSDO3toFloat()

//---------------------------------------------------------------------------
int CSound::InReadMultiChannel(  // <- used by Spectrum Lab since 2021-10-02, when adding support for multi-channel devices like Behringer UMC404HD .
        T_Float **ppFltDestBlocks, // [in] pointer to a short ARRAY of pointers to the channel blocks
        int   nDestBlocks,       // [in] number of destination blocks (1 for mono, 2 for stereo, etc);
                                 //      may be LESS than the number of OPENED channels so beware !
                                 // [out] T_Float* ppFltDestBlocks[0] : samples for 'left' audio channel (or "I")
                                 // [out] T_Float* ppFltDestBlocks[1] : samples for 'right' audio channel (or "Q")
                                 //       ....
                                 // [out] T_Float* ppFltDestBlocks[nDestBlocks-1] : samples for the LAST audio channel
        int iNumberOfSamples,    // [in] number of samples for each of the above blocks
        int iTimeout_ms,         // [in] max timeout in milliseconds, 0=non-blocking
        int *piHaveWaited_ms,    // [out,optional] "have been waiting here for XX milliseconds"
        T_ChunkInfo *pOutChunkInfo) // [out,optional,(*)] chunk info with timestamp, GPS, calib;
                                  //       see c:\cbproj\SoundUtl\ChunkInfo.h
  // Reads 'iNumberOfSamples' samples of floating-point samples
  // from the opened soundcard input (or audio I/O DLL, etc),
  // and if necessary splits the data into separated blocks for LEFT + RIGHT channel
  // (i.e. non-interlaced output for two separate channels).
  //   Returns:
  //   iNumberOfSamples   if 'iNumberOfSamples' samples were succesfully read
  //       0 =  if reaches the specified sample limit
  //      -1 =  if there is an error ( use GetError() to retrieve error )
  // Notes:
  //  -  'iNumberOfSamples' must not exceed the size of the internal buffers !
  //  -  If an audio source doesn't support the 'chunk info'
  //      (for example, the soundcard), the contents of *pOutChunkInfo
  //      will not be touched by CSound::InReadStereo() .
  //      Thus, the caller (like SL) may provide a 'meaningful default' there.
  // (*) Some of the fields in pOutChunkInfo
  //      may have been filled in by the CALLER already,
  //      indicated through pOutChunkInfo->dwValidityFlags:
  //  CHUNK_INFO_SAMPLE_RATE_VALID : pOutChunkInfo->dblPrecSamplingRate is set and VALID;
  //  CHUNK_INFO_SAMPLE_RATE_CALIBRATED : pOutChunkInfo->dblPrecSamplingRate is known to be CALIBRATED.
{
 int i,nChannels,iChannel;
 BYTE *pb;
 float *pflt;
 float *pfltDst[SOUND_MAX_CHANNELS];
 T_Float fltSample, fltScalingFactor; 
 LONG i32nSamplesRead = -1;
 T_SoundBufferInfo *pSoundBufferInfo;
 BOOL set_sr;
 int  waited_ms = 0;

  union uBSHORT  // byte-adressable 16-bit integer
   {  BYTE b[2];  // [0]=LSB, [1]=MSB (INTEL byte order)
      short i16;
   }bsTemp;
  union uBLONG  // byte-adressable 32-bit-integer
    { BYTE  b[4];  // b[0] = bits 7..0,  b[3] = bits 31..24 (INTEL)
      long  i32;
    }blTemp;

  // The caller MAY read less channels than the opened device permits...
  if( nDestBlocks > m_InFormat.nChannels ) // .. but not MORE channels than the device was opened for
   { nChannels = m_InFormat.nChannels;
   }
  else  // ok, caller wants samples from all opened channels in SEPARATE blocks
   { nChannels = nDestBlocks;
   }
  if( nChannels>SOUND_MAX_CHANNELS )
   {  nChannels=SOUND_MAX_CHANNELS;
   }
  if( nChannels>1 ) // prevent writing to NULL-pointers further below
   { // (a caller may have passed in nDestBlocks = 4
     //           but ppFltDestBlocks = { pLeftData, pLeftData, NULL, NULL }.
     //           In that case, only emit TWO channels here, not FOUR. )
     while( (nChannels>1) && (ppFltDestBlocks[ nChannels-1 ] == NULL ) )
      { --nChannels;
      }
   } // end if( nChannels>1 )
  // Copy the destination pointers.
  //    pfltDst[chn] may be modified (incremented) when copying,
  //    ppFltDestBlocks[chn] MUST NOT be modified because it's owned by the caller.
  for( i=0; i<SOUND_MAX_CHANNELS; ++i )
   { pfltDst[i] = (i<nChannels) ? ppFltDestBlocks[i] : NULL;
   }

  if( !m_InputOpen ) // return -1 if no inputs are active
   {
     DOBINI();  // -> Sound_iSourceCodeLine
     SetInputError( SOUNDIN_ERR_NOTOPEN ); // -> m_ErrorCodeIn + "chance for a COMMON breakpoint"
     return -1;
   }

#if ( MOD_2015_11_25 )
  // Modified 2015-11-25 : Too much time elapsed between waveInStart() and the first call of InReadStereo(),
  //                       so waveInStart() is now called from HERE, not from THERE. What a mess.
  // But for certain soundcards (like the E-Mu 0202 @ 192 kHz), this seemed to improve stability.
  if(m_InputOpen && (m_hwvin != NULL) )
   {
     if( ! m_fWaveInStarted )
      { m_fWaveInStarted = TRUE;
        DEBUG_EnterErrorHistory( DEBUG_LEVEL_INFO,0,UTL_USE_CURRENT_TIME,
           "Calling waveInStart()" );
        waveInStart( m_hwvin );
        // About waveInStart() :
        // > The waveInStart function starts input on the given waveform-audio input device.
        // > Calling this function when input is already started has no effect, and the function returns zero.
        // Despite that, since 2015-11-25, waveInStart() is only called when necessary !
      }
   }
#endif // MOD_2015_11_25 ?


  if( pOutChunkInfo != NULL )
   {
     set_sr = TRUE;
     // Does the CALLER know more about the inut device's sampling rate than CSound ?
     //      Some of the fields in pOutChunkInfo
     //      may have been filled in by the CALLER already,
     //      indicated through pOutChunkInfo->dwValidityFlags:
     if( pOutChunkInfo->dwValidityFlags & CHUNK_INFO_SAMPLE_RATE_VALID )
      { // pOutChunkInfo->dblPrecSamplingRate has already been set by caller and is VALID..
        // .. is it REALLY ?
        if( (pOutChunkInfo->dblPrecSamplingRate > (0.9 * m_dblNominalInputSampleRate) )
          &&(pOutChunkInfo->dblPrecSamplingRate < (1.1 * m_dblNominalInputSampleRate) ) )
         { // Ok, the indicated sampling rate looks plausible.. USE IT !
           m_dblNominalInputSampleRate = pOutChunkInfo->dblPrecSamplingRate;
           if( pOutChunkInfo->dwValidityFlags & CHUNK_INFO_SAMPLE_RATE_CALIBRATED )
            { // The sampling rate is not just 'valid', it's also 'calibrated'
              //  (for example, using a GPS sync signal as reference, etc..)
              // In this case, DO NOT OVERWRITE THE SAMPLING RATE IN THE T_ChunkInfo
              // with the 'default' or the 'measured' sampling rate !
              set_sr = FALSE;
            }
         } // end if pOutChunkInfo->dblPrecSamplingRate looks plausible
        else
         { // Ignore the *INPUT* sampling rate in T_ChunkInfo->dblPrecSamplingRate;
           // instead use the 'measured' (or, initially, the "nominal") SR :
           // pOutChunkInfo->dblPrecSamplingRate = m_dblNominalInputSampleRate;
         }
      } // end if( pOutChunkInfo->dwValidityFlags & CHUNK_INFO_SAMPLE_RATE_VALID )
     else  // The CALLER didn't fill in a valid sampling rate -> do it here:
      { // pOutChunkInfo->dblPrecSamplingRate = m_dblNominalInputSampleRate;
      }
     if( set_sr && ( ( pOutChunkInfo->dwValidityFlags & CHUNK_INFO_SAMPLE_RATE_VALID )==0 ) )
      { pOutChunkInfo->dblPrecSamplingRate = m_dblNominalInputSampleRate;
        pOutChunkInfo->dwValidityFlags |= CHUNK_INFO_SAMPLE_RATE_VALID;
      }
   } // end if( pOutChunkInfo != NULL )



   // Totally different handling (internally) if an "Audio I/O Driver DLL" is used:
#if( SWI_AUDIO_IO_SUPPORTED )
   if( (aio_In.h_AudioIoDLL != NULL)  // using an Audio-I/O or ExtIO-DLL for *INPUT* ...
     &&( !m_ExtIONeedsSoundcardForInput ) ) // ...except for those which use the SOUNDCARD:
    { // Because the audio-I/O-DLL uses multiple channels in a SINGLE SAMPLE POINT,
      // the data must first be read into a temporary array, and then re-arranged:
      float *pfltTemp = (float*)malloc( sizeof(float) * iNumberOfSamples * nChannels ); // float, not T_Float !
      if( pfltTemp != NULL )
       {
         i32nSamplesRead = AIO_Host_ReadInputSamplePoints( // returns the number of sample-POINTS, or a negative error code
           &aio_In,  // [in,out] T_AIO_DLLHostInstanceData *pInstData, DLL-host instance data
           pfltTemp, // [out] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
           iNumberOfSamples, // [in] iNumSamplePoints; number of SAMPLE POINTS(!) to read
           nChannels, // [in] nChannelsPerSample; number of samples PER SAMPLE POINT
           iTimeout_ms, // [in] max timeout in milliseconds, 0 would be 'non-blocking'
                  // (must use a blocking call here, because the audio-source
                  //  usually 'sets the pace' for the entire processing chain in SL.
                  // 200 ms are sufficient for 8000 samples/second and 1024 samples per chunk)
           pOutChunkInfo,// [out,optional] T_ChunkInfo *pOutChunkInfo, see c:\cbproj\SoundUtl\ChunkInfo.h
           piHaveWaited_ms);    // [out,optional] INT32 *piHaveWaited_ms; "have been waiting here for XX milliseconds"
         if( i32nSamplesRead < 0 )
          { SetInputError(  ConvertErrorCode_AIO_to_SOUND( i32nSamplesRead ) );
          }
         else // no error -> split the sample-points into separate blocks
          { if(i32nSamplesRead>iNumberOfSamples )
             { i32nSamplesRead=iNumberOfSamples;  // should never happen, but expect the worst
             }
            pflt = pfltTemp;
            for(i=0; i<i32nSamplesRead; ++i)
             { if( pfltDst[0] != NULL )
                  *pfltDst[0]++ = pflt[0];
               if( pfltDst[1] != NULL )
                  *pfltDst[1]++ = pflt[1];
               if( pfltDst[2] != NULL )
                  *pfltDst[2]++ = pflt[2];
               if( pfltDst[3] != NULL )
                  *pfltDst[3]++ = pflt[3];
               pflt += nChannels;
             }
          }
         free(pfltTemp);
       }
     if( ( pOutChunkInfo != NULL ) && ( i32nSamplesRead>0 ) )
      {  pOutChunkInfo->nChannelsPerSample = nChannels;
         pOutChunkInfo->dwNrOfSamplePoints = i32nSamplesRead;
      }

     return i32nSamplesRead;
   }  // end if( aio_In.h_AudioIoDLL != NULL )
#endif // SWI_AUDIO_IO_SUPPORTED ?

  // Use a communications port (serial port, "COM port", virtual COM port, etc) for input ?
  if( m_iInputDeviceID==C_SOUND_DEVICE_COM_PORT )  // here in CSound::InRead() ...
   { if( m_hComPort != INVALID_HANDLE_VALUE )
      { // Read as many BYTES (not samples) as possible into the input buffer,
        //  which - in THIS case - is treated as a SINGLE circular FIFO:
        // BYTE *m_pbInputBuffer;  dynamically allocated for [m_NumInputBuffers * m_InBufferSize];
        // m_iInHeadIndex and  m_iInTailIndex are byte-indices into the ENTIRE buffer here !
        int iTotalBufSize = m_NumInputBuffers/*should be ONE*/ * m_InBufferSize;
        int nBytesInBuffer, nBytesToRead, nBytesConsumed, nSamplesEmitted;
        int nLoops = 100;      // who knows.. the samples may arrive in "very small chunks" ! With an FTDI adapter, only a few hundred bytes per ReadFile() despite a high sampling rate.
        DWORD dwNumBytesRead;
        i32nSamplesRead = 0;
        if( m_InputOpen && (iTotalBufSize>0) && (m_pbInputBuffer!=NULL) ) // safety first..
         {
           m_InWaiting = TRUE;  // here: "waiting for the arrival of samples FROM THE SERIAL PORT"

           // Added 2015-10 : Is "our" buffer still intact ?
           if( (m_NumInputBuffers==1) && (m_InBufferSize>0) && (m_pbInputBuffer!=NULL) )
            { if( m_pbInputBuffer[m_InBufferSize] != SOUND_MAGIC_BYTE_AT_END_OF_BUFFER )
               {  DEBUG_EnterErrorHistory( DEBUG_LEVEL_FATAL,0,UTL_USE_CURRENT_TIME,
                       "CSound: Input-buffer corrupted (end-marker overwritten)" );
                  m_pbInputBuffer[m_InBufferSize] = SOUND_MAGIC_BYTE_AT_END_OF_BUFFER; // prevent same error-msg again
               }
            }
           // Read bytes from the serial port and convert them into samples ...
           while( iNumberOfSamples > 0 ) // may have to repeat this until enough samples were read
            {
              m_iInHeadIndex %= iTotalBufSize; // indices may run from zero...
              m_iInTailIndex %= iTotalBufSize; // to 'iTotalBufSize' MINUS ONE
              nBytesInBuffer = m_iInHeadIndex - m_iInTailIndex;
              if( nBytesInBuffer < 0 )         // circular FIFO wrap-around..
               {  nBytesInBuffer += iTotalBufSize;
               }
              // If there are "old" data in the buffer (which were not be processed in a previous call),
              //    copy them TO THE BEGIN of the buffer
              //    because the Sound_ConvSamples_XYZtoFloat()-functions called further below
              //    don't support circular index wrapping .
              //             __________________________________________________
              // before..:  |_____67890________________________________________|
              //                  |    |
              //                  tail head  (tail~"consume", head~"produce"-index)
              //              ____|
              //             |
              //            \|/ block-copy, beware, source+dest may OVERLAP !
              //             __________________________________________________
              //            |67890_____________________________________________|
              //             |    |
              //             tail head
              //                  |____________________________________________|
              //                  |<--free space, destination for ReadFile()-->|
              //             __________________________________________________
              // after Read:|67890ABCDEFGHIJKLMNO______________________________|
              //             |                   |
              //             tail                head
              //                  |______________|
              //                  |<--new data-->| (actually read from ReadFile)
              //
              if( m_iInTailIndex > 0 )  // scroll up (block-move) to avoid circular-buffer-index-fiddling further below !
               { if( nBytesInBuffer > 0 ) // old data to be moved to the start of the buffer ?
                  { memmove( m_pbInputBuffer/*dst*/, m_pbInputBuffer+m_iInTailIndex/*src*/, nBytesInBuffer ); // memmove, not memcpy !
                    // 2016-02-28: When reading samples from the PIC-based GPSDO (used as A/D converter) at 500 kBit/sec, and 12500 samples/sec,
                    //             got here with nBytesInBuffer==1 which is *very* suspicious.
                    //  Monitored the number of bytes received in each call of ReadFile(), result further below.
                    m_iInTailIndex = 0;
                    m_iInHeadIndex = nBytesInBuffer;
                  }
                 else // nothing in the buffer yet -> set all indices to the start !
                  { m_iInHeadIndex = m_iInTailIndex = nBytesInBuffer = 0;
                  }
               } // end if( m_iInTailIndex > 0 )
              nBytesToRead  = (iTotalBufSize-1) - nBytesInBuffer;  // max. capacity = ONE BYTE LESS than the allocated size
              nBytesToRead &= 0x7FFFFFFC;         // read from the 'file' in multiples of FOUR bytes only !
              dwNumBytesRead = 0;
              i = iTotalBufSize - m_iInHeadIndex; // example: iTotalBufSize=256, HeadIndex=255 : can only read ONE (!) byte this time
              if( nBytesToRead > i )
               {  nBytesToRead = i;
               }
              if( nBytesToRead > 0 )  // Can we 'stuff more bytes' into the buffer ?
               { // Note: Due to the 'COM timeouts', ReadFile will not WAIT
                 //       for the reception of data. If there's nothing
                 //       in the receive-buffer, ReadFile should return immediately
                 //       with 'zero bytes read' (past tense of 'read') .
                 if( /*BOOL*/ReadFile( m_hComPort,  // read from "COM"-port (usually a fast USB<->UART adapter, e.g. FTDI 232R) ...
                           m_pbInputBuffer+m_iInHeadIndex, // destination
                           nBytesToRead, &dwNumBytesRead,
                           NULL/*no 'overlap'-thing*/ ) )
                  { if( dwNumBytesRead > 0 ) // just for debugger-friendlyness.. set breakpoint ON RECEPTION below
                     { nBytesInBuffer += dwNumBytesRead;
                       m_iInHeadIndex  = ( m_iInHeadIndex + (int)dwNumBytesRead) % iTotalBufSize; // here: with bytes received from a 'COM port'
                       // 2016-02-28: When reading samples from the PIC16F1783-based GPSDO
                       // (used as A/D converter with at 500 kBit/sec, 12500 samples/sec,
                       //  iNumberOfSamples 'wanted by caller' = 4096 ),
                       // got here with dwNumBytesRead = 781, 781, 781, ... ?!?
                       //
                       // Using the PIC12F675-based ADC (with 12-bit I/Q output at 2500 samples / sec),
                       // got here with dwNumBytesRead = 309, 307, 304, ...
                       //   Only after pausing the debugger for a few seconds,
                       //   dwNumBytesRead almost reached the max buffer size
                       //   ..at least with a 'Belkin / Prolific' USB adapter.
                     }
                  } // end if < ReadFile successfull >
                 else
                  { dwNumBytesRead = 0;
                  }

                 // Added 2015-10 : Is "our" buffer still intact ?
                 if( (m_NumInputBuffers==1) && (m_InBufferSize>0) && (m_pbInputBuffer!=NULL) )
                  { if( m_pbInputBuffer[m_InBufferSize] != SOUND_MAGIC_BYTE_AT_END_OF_BUFFER )
                     {  DEBUG_EnterErrorHistory( DEBUG_LEVEL_FATAL,0,UTL_USE_CURRENT_TIME,
                             "CSound: Input-buffer corrupted (end-marker overwritten)" );
                        m_pbInputBuffer[m_InBufferSize] = SOUND_MAGIC_BYTE_AT_END_OF_BUFFER; // prevent same error-msg again
                     }
                  }
                 else
                  { DEBUGGER_BREAK(); // << set breakpoint here, or use the 'common' breakpoint in DebugU1.cpp
                  } // end < 'buffer-integrity-test' > [2015-10-14]

               } // end if( nBytesToRead > 0 )

              if( nBytesInBuffer > 4 )
               { // Try to process as many samples from the buffer as requested by caller.
                 nBytesConsumed = nSamplesEmitted = 0;
                 pb = &m_pbInputBuffer[ m_iInTailIndex ]; // Note: the buffer's "tail" (aka "consume"-) index should be ZERO, reasons explained above

                 switch( m_iSoundInputSampleFormat & SOUND_SAMPLE_FMT_MASK ) // here: after reading a bunch of bytes from the SERIAL PORT ...
                  { case SOUND_SAMPLE_FMT_U8: /* 8 bit per channel (in each sample point), UNSIGNED integer */
                    case SOUND_SAMPLE_FMT_S8: /* 8 bit per channel (in each sample point), UNSIGNED integer */
                       nBytesConsumed = Sound_ConvSamples_I8toFloat( // converts a block of EIGHT-bit-samples into floating point numbers
                          m_iSoundInputSampleFormat, // [in] iSoundInputSampleFormat, here: only EIGHT BIT INTEGER types with or without sync pattern
                          pb,                        // [in] BYTE * pbSource, source block (don't assume anything about alignment!)
                          nBytesInBuffer,            // [in] number of BYTES in the above source block
                          m_InFormat.nChannels,      // [in] nSourceChannels, number of CHANNELS (in the SOURCE data)
                          iNumberOfSamples,          // [in] iMaxSamplesPerChannel, MAXIMUM number of samples for each of the destination blocks below
                          pfltDst[0],                // [out] T_Float *, samples for 'left' audio channel (or "I")
                          pfltDst[1],                // [out] T_Float *, samples for 'right' audio channel (or "Q")
                          &nSamplesEmitted );        // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
                       break;

                    case SOUND_SAMPLE_FMT_U16_LE: /* 16 bit per channel (in each sample point), UNSIGNED integer, Little Endian (LSByte first) */
                    case SOUND_SAMPLE_FMT_S16_LE: /* 16 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
                    case SOUND_SAMPLE_FMT_U16_BE: /* 16 bit per channel (in each sample point), UNSIGNED integer, Big Endian    (MSByte first) */
                    case SOUND_SAMPLE_FMT_S16_BE: /* 16 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
                       nBytesConsumed = Sound_ConvSamples_I16toFloat( // converts a block of SIXTEEN-bit-samples into floating point numbers
                          m_iSoundInputSampleFormat, // [in] iSoundInputSampleFormat, here: only 16 BIT INTEGER types with or without sync pattern
                          pb,                        // [in] BYTE * pbSource, source block (don't assume anything about alignment!)
                          nBytesInBuffer,            // [in] number of BYTES in the above source block
                          m_InFormat.nChannels,      // [in] nSourceChannels, number of CHANNELS (in the SOURCE data)
                          iNumberOfSamples,          // [in] iMaxSamplesPerChannel, MAXIMUM number of samples for each of the destination blocks below
                          pfltDst[0],                // [out] T_Float *,  samples for 'left' audio channel (or "I")
                          pfltDst[1],                // [out] T_Float *, samples for 'right' audio channel (or "Q")
                          &nSamplesEmitted );        // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
                       break;

                    case SOUND_SAMPLE_FMT_U24_LE: /* 24 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
                    case SOUND_SAMPLE_FMT_S24_LE: /* 24 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
                    case SOUND_SAMPLE_FMT_U24_BE: /* 24 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
                    case SOUND_SAMPLE_FMT_S24_BE: /* 24 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
                       nBytesConsumed = Sound_ConvSamples_I24toFloat( // converts a block of TWENTYFOUR-bit-samples into floating point numbers
                          m_iSoundInputSampleFormat, // [in] iSoundInputSampleFormat, here: only 24 BIT INTEGER types with or without sync pattern
                          pb,                        // [in] BYTE * pbSource, source block (don't assume anything about alignment!)
                          nBytesInBuffer,            // [in] number of BYTES in the above source block
                          m_InFormat.nChannels,      // [in] nSourceChannels, number of CHANNELS (in the SOURCE data)
                          iNumberOfSamples,          // [in] iMaxSamplesPerChannel, MAXIMUM number of samples for each of the destination blocks below
                          pfltDst[0],                // [out] T_Float *,  samples for 'left' audio channel (or "I")
                          pfltDst[1],                // [out] T_Float *, samples for 'right' audio channel (or "Q")
                          &nSamplesEmitted );        // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
                       break;

                    case SOUND_SAMPLE_FMT_U32_LE: /* 32 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
                    case SOUND_SAMPLE_FMT_S32_LE: /* 32 bit per channel (in each sample point), UNSIGNED integer, Little Endian */
                    case SOUND_SAMPLE_FMT_U32_BE: /* 32 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
                    case SOUND_SAMPLE_FMT_S32_BE: /* 32 bit per channel (in each sample point), UNSIGNED integer, Big Endian    */
                       nBytesConsumed = Sound_ConvSamples_I32toFloat( // converts a block of 32-bit-samples (integers) into floating point numbers
                          m_iSoundInputSampleFormat, // [in] iSoundInputSampleFormat, here: only 32 BIT INTEGER types with or without sync pattern
                          pb,                        // [in] BYTE * pbSource, source block (don't assume anything about alignment!)
                          nBytesInBuffer,            // [in] number of BYTES in the above source block
                          m_InFormat.nChannels,      // [in] nSourceChannels, number of CHANNELS (in the SOURCE data)
                          iNumberOfSamples,          // [in] iMaxSamplesPerChannel, MAXIMUM number of samples for each of the destination blocks below
                          pfltDst[0],                // [out] T_Float *,  samples for 'left' audio channel (or "I")
                          pfltDst[1],                // [out] T_Float *, samples for 'right' audio channel (or "Q")
                          &nSamplesEmitted );        // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
                       break;

                    case SOUND_SAMPLE_FMT_F32_LE: /* 32 bit per channel (in each sample point), FLOATING POINT, for Intel CPU+FPU*/
                    case SOUND_SAMPLE_FMT_F32_BE: /* 32 bit per channel (in each sample point), FLOATING POINT, for we-don't-know*/
                       nBytesConsumed = Sound_ConvSamples_F32toFloat( // converts a block of 32-bit-FLOATS into the host's floating point number format
                          m_iSoundInputSampleFormat, // [in] iSoundInputSampleFormat, here: only 32 BIT FLOAT types, with or without sync pattern
                          pb,                        // [in] BYTE * pbSource, source block (don't assume anything about alignment!)
                          nBytesInBuffer,            // [in] number of BYTES in the above source block
                          m_InFormat.nChannels,      // [in] nSourceChannels, number of CHANNELS (in the SOURCE data)
                          iNumberOfSamples,          // [in] iMaxSamplesPerChannel, MAXIMUM number of samples for each of the destination blocks below
                          pfltDst[0],                // [out] T_Float *,  samples for 'left' audio channel (or "I")
                          pfltDst[1],                // [out] T_Float *, samples for 'right' audio channel (or "Q")
                          &nSamplesEmitted );        // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
                       break;

                    case SOUND_SAMPLE_FMT_IQ12: /* header byte + 2 * 12 bit unsigned, I/Q format used by DL4YHF's PIC12F675 ADC for the serial port */
                       nBytesConsumed = Sound_ConvSamples_IQ12toFloat( // converts a block of 2 * 12-bit unsigned integers into floating point
                          pb,                        // [in] BYTE * pbSource, source block (don't assume anything about alignment!)
                          nBytesInBuffer,            // [in] number of BYTES in the above source block
                          iNumberOfSamples,          // [in] iMaxSamplesPerChannel, MAXIMUM number of samples for each of the destination blocks below
                          pfltDst[0],                // [out] T_Float *, samples for 'left' audio channel (or "I")
                          pfltDst[1],                // [out] T_Float *, samples for 'right' audio channel (or "Q")
                          &nSamplesEmitted );        // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
                       break;

                    case SOUND_SAMPLE_FMT_GPSDO3: // header byte with the MSBIT always set; all other bytes in the same have their MSBytes *cleared*
                       nBytesConsumed = Sound_ConvSamples_GPSDO3toFloat( // converts a block of 'GPSDO3' frames into floating point samples
                          pb,                        // [in] BYTE * pbSource, source block (don't assume anything about alignment!)
                          nBytesInBuffer,            // [in] number of BYTES in the above source block
                          iNumberOfSamples,          // [in] iMaxSamplesPerChannel, MAXIMUM number of samples for each of the destination blocks below
                          pfltDst[0],                // [out] T_Float *, samples for 'left' audio channel (or "I")
                          pfltDst[1],                // [out] T_Float *, samples for 'right' audio channel (or "Q")
                          &nSamplesEmitted );        // [in] NUMBER OF SAMPLES placed in EACH of the (one or two) output blocks
                       break;



                    case SOUND_SAMPLE_FMT_OGG: /* Ogg containers, hopefully with Vorbis-compressed audio */
                       // Watch for VLF-Tools compatible 'stream headers'
                       // as defined in c:\cbproj\SoundUtl\VorbisFileIO.h (!) .
                       // The pattern defined as STREAM_HDR_PATTERN (0x80008000) there
                       // should only appear at DOUBLEWORD (32-bit) borders in the stream,
                       // but since the COM port may actually be a homebuilt one-way radio link,
                       // expect that SINGLE BYTES can get lost at any time,
                       // and the receiver (which is implemented HERE) must be able
                       // to synchronize on the received stream again without big hassle.
                       // Principle (so far just a FUTURE PLAN) :
                       //    Keep at least 64 bytes remaining in the FIFO.
                       //    If the pattern (0x80008000) is seen at the FIFO tail index,
                       //    there will be enough bytes left in the FIFO to process
                       //    a complete T_StreamHdrUnion (defined in VorbisFileIO.h) .
                       //    Note: The pattern (0x80008000) itself belongs to the T_StreamHdrUnion.
                       if( nBytesInBuffer >= sizeof(T_StreamHdrUnion) )
                        {
                        } // end if( nBytesInBuffer >= sizeof(T_StreamHdrUnion) )
                       if(dwNumBytesRead==0) // nothing read -> break the loop, or wait and try again ?
                        { if( iTimeout_ms > 0 ) // caller said it's ok to block (wait) here for a few ms..
                           { Sleep(20);
                             iTimeout_ms -= 20;
                           }
                          else
                          { break;
                          }
                        } // end if(dwNumBytesRead==0)
                       break; // end case SOUND_SAMPLE_FMT_OGG


                    case SOUND_SAMPLE_FMT_UNKNOWN :
                    default:
                       SetInputError(  ERROR_DATA_TYPE_NOT_SUPPORTED );
                       return m_ErrorCodeIn;
                  } // end switch( sample format )
                 if( nBytesConsumed > 0 )
                  { m_iInTailIndex = (m_iInTailIndex+nBytesConsumed) % iTotalBufSize;
                    // Note: Some of the "converters" called above may actually
                    //       have CONSUMED NOTHING because they need more bytes
                    //       before being able to 'work' (for example, Ogg/Vorbit) !
                  } // end if( nBytesConsumed > 0 )
                 if( nSamplesEmitted > 0 )
                  { if( pfltDst[0] != NULL )
                     {  pfltDst[0] += nSamplesEmitted;
                     }
                    if( pfltDst[1] != NULL )
                     {  pfltDst[1] += nSamplesEmitted;
                     }
                    if( pfltDst[2] != NULL )
                     {  pfltDst[2] += nSamplesEmitted;
                     }
                    if( pfltDst[3] != NULL )
                     {  pfltDst[3] += nSamplesEmitted;
                     }
                    i32nSamplesRead  += nSamplesEmitted;
                    iNumberOfSamples -= nSamplesEmitted;
                  } // if( nSamplesEmitted > 0 )
               } // end if( nBytesInBuffer > 4 )

              if( ! m_InputOpen ) // seems another thread is about to CLOSE the audio input..
               { break;           // -> immediately break from the waiting loop
               }
              if( (nLoops--) <= 0 )
               { break;
               }
              if( iNumberOfSamples <= 0 )
               { break;    // got enough samples -> DON'T WAIT FOR "MORE" !
               }
              // Only repeat the loop if the caller allows to 'wait' here.
              if( iTimeout_ms > 0 )  // [in] max timeout in milliseconds, 0=non-blocking
               {
                 Sleep(20);  // wait for 20 ms (let other threads run), then try again to read bytes from the serial port
                 iTimeout_ms -= 20;
                 waited_ms   += 20;
               }
              else // don't wait any longer .. return all the samples we have at the moment
               { break;
               }

            } // end while < more samples requested by caller >
           m_InWaiting = FALSE;  // here: "not waiting for the arrival of samples FROM THE SERIAL PORT anymore"
         } // end if( (iTotalBufSize>0) && (m_pbInputBuffer!=NULL) )
      } // end if( m_hComPort != INVALID_HANDLE_VALUE )
     else // COM port handle is NOT valid :
      {
      } // end else <  m_hComPort != INVALID_HANDLE_VALUE >

     if( piHaveWaited_ms != NULL )
      { *piHaveWaited_ms = waited_ms;
      }

     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // For input from a serial port, do NOT enter the 'soundcard' code further below !
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     return i32nSamplesRead;
   } // end if( m_iInputDeviceID==C_SOUND_DEVICE_COM_PORT )

  if( m_InOverflow )    // has been set by WaveInCallback()
   {
     DOBINI();  // -> Sound_iSourceCodeLine
     SetInputError(  SOUNDIN_ERR_OVERFLOW  );
     m_InOverflow = FALSE;
   }

  DOBINI();  // -> Sound_iSourceCodeLine

  if( m_iInTailIndex == m_iInHeadIndex )  // if no buffer is filled ...
   { if( ! WaitForInputData(iTimeout_ms) ) // ... then wait, but only for a limited time
      { DOBINI();  // -> Sound_iSourceCodeLine
        return -1;
      }
   }
  // here if there is data to retrieve
  DOBINI();  // -> Sound_iSourceCodeLine
  // ex: pb = m_pbInputBuffer + m_iInTailIndex*m_InBufferSize;
  pb = WrapInBufPositionAndGetPointerToNextFragment(
           CSOUND_WRAPBUF_OPT_DONT_WAIT,
           &pSoundBufferInfo );
  if( pb==NULL ) // oops... no filled 'input' data available ?!
   { DOBINI();  // -> Sound_iSourceCodeLine
     return -1;
   }
  if( pOutChunkInfo != NULL )
   { if( m_InFormat.nChannels>0 )
      { pOutChunkInfo->nChannelsPerSample = m_InFormat.nChannels; // added 2011-12-27
        // ( Even if the application specified TWO blocks ( pfltDst[0] + pfltDst[1] ),
        //   the T_ChunkInfo struct contains the number of "inputs" from InOpen() )
      }
#   if( SWI_AUDIO_IO_SUPPORTED )  // only if the 'audio-I/O-host' is supported..
     if( aio_In.h_AudioIoDLL != NULL)  // using an ExtIO-DLL, but the SOUNDCARD for *INPUT* ...
      { // In this case, a few fields in c:\cbproj\SoundUtl\ChunkInfo.h :: T_ChunkInfo
        //    are copied from AudioIO.h::T_AIO_DLLHostInstanceData :
        pOutChunkInfo->dblRadioFrequency = aio_In.i32ExternalVfoFrequency;
      }
#   endif // SWI_AUDIO_IO_SUPPORTED ?

     GetDateAndTimeForInput(  // 2015-07-13 : This FAILED with ASIO but was ok with "MMSYSTEM" .
              pSoundBufferInfo, // [in]  T_SoundBufferInfo, contains .ldblUnixDateAndTime
              pOutChunkInfo );  // [out] pOutChunkInfo->ldblUnixDateAndTime
   }
  if( m_InFormat.wBitsPerSample == 16 )
   { // usually 16 BITS PER SAMPLE ...
     // Since 2022-10-16, ALWAYS use a sample value range of +/- 1.0
     if( pOutChunkInfo != NULL )  // if the caller is aware of the value range,
      { pOutChunkInfo->dblSampleValueRange = 1.0;  // scale everything to +/- 1.0
      }
     fltScalingFactor = 1.0 / 32768.0; // ALWAYS normalize the floating-point sample value range to +/- 1.0,
     if (m_InBufPosition & 1)
      { // very suspicious..
        DOBINI();  // -> Sound_iSourceCodeLine
        SetInputError(  ERROR_WORD_AT_ODD_ADDRESS  );
      }
     if( (m_InFormat.nChannels==1 ) // 16 bit, Mono input
       ||(pfltDst[1] == NULL) )    // or caller only wants ONE channel
      { int iBufIdxInc = 2*m_InFormat.nChannels;
        if( pfltDst[1] != NULL )  // mono input, but two *output* channels :
         { DOBINI();  // -> Sound_iSourceCodeLine
           for( i=0; i<iNumberOfSamples; i++ )
            {
              bsTemp.b[0] = pb[m_InBufPosition];   // LSB
              bsTemp.b[1] = pb[m_InBufPosition+1]; // MSB
              fltSample   = (T_Float)bsTemp.i16 * fltScalingFactor;
              pfltDst[0][i] = fltSample;
              if( pfltDst[1] != NULL )
               {  pfltDst[1][i] = fltSample;
               }
              m_InBufPosition += iBufIdxInc;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }
            } // end for <iNumberOfSamples>
         }
        else          // read 16 bit per (single) sample, only ONE destination channel :
         { DOBINI();  // -> Sound_iSourceCodeLine
           for( i=0; i<iNumberOfSamples; i++ )
            {
              bsTemp.b[0] = pb[m_InBufPosition];
              bsTemp.b[1] = pb[m_InBufPosition+1];
              fltSample = (T_Float)bsTemp.i16 * fltScalingFactor;
              pfltDst[0][i] = fltSample;
              if( pfltDst[1] != NULL )
               {  pfltDst[1][i] = fltSample;
               }
              if( pfltDst[2] != NULL )
               {  pfltDst[2][i] = fltSample;
               }
              if( pfltDst[3] != NULL )
               {  pfltDst[3][i] = fltSample;
               }
              m_InBufPosition += iBufIdxInc;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }
            } // end for
         }
       } // end if <16 bit mono>
     else // 16-bit samples, m_InFormat.nChannels>=2 :
      { DOBINI();  // -> Sound_iSourceCodeLine
        for( i=0; i<iNumberOfSamples; i++ )
         {
           for( iChannel=0; iChannel<m_InFormat.nChannels; ++iChannel )
            { bsTemp.b[0] = pb[m_InBufPosition++];
              bsTemp.b[1] = pb[m_InBufPosition++];
              fltSample = (T_Float)bsTemp.i16 * fltScalingFactor;
              if( pfltDst[iChannel] != NULL )
               {  pfltDst[iChannel][i] = fltSample;
               }
            }
           // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
           if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
            {                                       // so use the next (*if* there's one)
              pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
              if( pb==NULL ) // oops... no more filled 'input' data available ?!
               { DOBINI();   // -> Sound_iSourceCodeLine
                 return -1;
               }
            }
         } // end for
      } // end if <16 bit stereo>
   }
  else  // don't read 16 bits per sample, but maybe 8 bits per sample...
  if( m_InFormat.wBitsPerSample == 8 )
    {
      if( pOutChunkInfo != NULL )  // if the caller is aware of the value range,
       { pOutChunkInfo->dblSampleValueRange = 1.0;  // scale everything to +/- 1.0
       }
      fltScalingFactor = 1.0 / 128.0;  // normalize from 0..255 to +/- 1.0 (as floating point value)
      bsTemp.i16 = 0;               // clear unused bits before the loop
      if( (m_InFormat.nChannels==1) // 8 bit, Mono input
        ||(pfltDst[1] == NULL) )    // or caller only wants ONE channel
       { int iBufIdxInc = m_InFormat.nChannels;
        if( pfltDst[1] != NULL )  // two *output* channels :
         { DOBINI();  // -> Sound_iSourceCodeLine
           for( i=0; i < iNumberOfSamples; i++ )
            {
              bsTemp.b[1] = (BYTE)( pb[m_InBufPosition] - 128); // write to UPPER byte (!)
              pfltDst[0][i] = pfltDst[1][i] = (T_Float)bsTemp.i16 * fltScalingFactor;
              m_InBufPosition += iBufIdxInc;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }

            }
         }
        else          // read 8 bit per (single) sample, only ONE destination channel :
         { DOBINI();  // -> Sound_iSourceCodeLine
           for( i=0; i < iNumberOfSamples; i++ )
            {
              bsTemp.b[1] = (BYTE)( pb[m_InBufPosition] - 128);  // high byte only
              pfltDst[0][i] = (T_Float)bsTemp.i16 * fltScalingFactor;
              m_InBufPosition += iBufIdxInc;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }
            }
         }
       } // end if <8 bit mono>
      else
      if( (m_InFormat.nChannels==2) && (pfltDst[1]!=NULL) ) // 8 bit, stereo input
       { DOBINI();  // -> Sound_iSourceCodeLine
        for( i=0; i < iNumberOfSamples; i++ )
         {
           bsTemp.b[1]  = (BYTE)( pb[m_InBufPosition++] - 128 );
           pfltDst[0][i] = (T_Float)bsTemp.i16 * fltScalingFactor;
           bsTemp.b[1]  = (BYTE)( pb[m_InBufPosition++] - 128 );
           pfltDst[1][i]= (T_Float)bsTemp.i16 * fltScalingFactor;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }
         }
       } // end if <8 bit stereo>
   } // end if < 8 bits per sample >
  else
  if( m_InFormat.wBitsPerSample == 24 )
   {  // may get here with ASIO, too... even if only 16 bit/sample were "wanted" !!
      if( pOutChunkInfo != NULL )  // if the caller is aware of the value range,
       { pOutChunkInfo->dblSampleValueRange = 1.0;  // scale everything to +/- 1.0
       }
      fltScalingFactor = 1.0/8388608.0;  // multiply by 1 / 2^23 to "normalize" the input

      // Strange: With an E-MU 0202, levels in 24 bit mode were approx. 5 dB larger than in 16 bit mode ?!
      blTemp.b[0] = 0;              // least significant 8 bits in 32-bit value are always zero
      if( (m_InFormat.nChannels==1) // 24 bit, Mono input
        ||(pfltDst[1] == NULL) )    // or caller only wants ONE channel
       { int iBufIdxInc = 3 * m_InFormat.nChannels;
        if( pfltDst[1] != NULL )  // mono input, but two *output* channels :
         { DOBINI();  // -> Sound_iSourceCodeLine
           for( i=0; i<iNumberOfSamples; i++ )
            {
              blTemp.b[1] = pb[m_InBufPosition];   // copy bits  7..0 of 24 bits  into bits 15..8 of 32 bits
              blTemp.b[2] = pb[m_InBufPosition+1]; // copy bits 15..8 of 24 bits  into bits 23..16 of 32 bits
              blTemp.b[3] = pb[m_InBufPosition+2]; // copy bits 23..16 of 24 bits into bits 31..24 of 32 bits
              pfltDst[0][i] = pfltDst[1][i] =  (T_Float)blTemp.i32 * fltScalingFactor;
              m_InBufPosition += iBufIdxInc;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }
            } // end for
         }
        else  // 24 bit per (single) sample,  only ONE destination channel :
         {
           for( i=0; i<iNumberOfSamples; i++ )
            {
              blTemp.b[1] = pb[m_InBufPosition];   // copy bits  7..0 of 24 bits  into bits 15..8 of 32 bits
              blTemp.b[2] = pb[m_InBufPosition+1]; // copy bits 15..8 of 24 bits  into bits 23..16 of 32 bits
              blTemp.b[3] = pb[m_InBufPosition+2]; // copy bits 23..16 of 24 bits into bits 31..24 of 32 bits
              pfltDst[0][i]= (T_Float)blTemp.i32 * fltScalingFactor;
              m_InBufPosition += iBufIdxInc;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }
            } // end for
         }
       } // end if <24 bit mono>
      else
      if(m_InFormat.nChannels==2) // 24 bit, Stereo
       { DOBINI();  // -> Sound_iSourceCodeLine
        // 2015-07-08 : The "24 bit" input from "MMSYSTEM" seemed to be ONLY NOISE :
        // *pb = 0x04, 0x00, 0x73,  0xCB, 0x0D, 0x40,   (<- first sampling point)
        //       0xFA, 0xFF, 0x97,  0x51, 0x07, 0x94,   (<- second sampling point)
        for( i=0; i<iNumberOfSamples; i++ )
         {
           blTemp.b[1] = pb[m_InBufPosition++]; // copy bits  7..0 of 24 bits  into bits 15..8 of 32 bits
           blTemp.b[2] = pb[m_InBufPosition++]; // copy bits 15..8 of 24 bits  into bits 23..16 of 32 bits
           blTemp.b[3] = pb[m_InBufPosition++]; // copy bits 23..16 of 24 bits into bits 31..24 of 32 bits
           pfltDst[0][i] =  (T_Float)blTemp.i32 * fltScalingFactor;

           blTemp.b[1] = pb[m_InBufPosition++];
           blTemp.b[2] = pb[m_InBufPosition++];
           blTemp.b[3] = pb[m_InBufPosition++];
           if( pfltDst[1] != NULL )
            {  pfltDst[1][i] =  (T_Float)blTemp.i32 * fltScalingFactor;
            }
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
           if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
            {                                       // so use the next (*if* there's one)
              pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
              if( pb==NULL ) // oops... no more filled 'input' data available ?!
               { DOBINI();   // -> Sound_iSourceCodeLine
                 return -1;
               }
            }

         } // end for
       } // end if <24 bit stereo>
   } // end if < 24 bits per sample >
  else
  if( m_InFormat.wBitsPerSample == 32 )
   {
      int iBufIdxInc = 4 * m_InFormat.nChannels;
      if( pOutChunkInfo != NULL )  // if the caller wants to know the value range...
       { pOutChunkInfo->dblSampleValueRange = 1.0;  // .. let him know we scale everything to +/- 1.0
       }
      fltScalingFactor = 1.0/(32768.0*65536.0); // 32 bit (signed) ->  +/- 1.0
      if( (m_InFormat.nChannels==1) // 32 bit, Mono input
        ||(pfltDst[1] == NULL) )    // or caller only wants ONE channel
       {
        if( pfltDst[1] != NULL )  // two *output* channels :
         { DOBINI();  // -> Sound_iSourceCodeLine
           for( i=0; i<iNumberOfSamples; i++ )
            { blTemp.i32 = *(long*)(pb+m_InBufPosition);
              pfltDst[0][i] = pfltDst[1][i] = (T_Float)blTemp.i32 * fltScalingFactor;
              m_InBufPosition += iBufIdxInc;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }
            }
         }
        else          // only ONE destination channel :
         { DOBINI();  // -> Sound_iSourceCodeLine
           for( i=0; i<iNumberOfSamples; i++ )
            { blTemp.i32 = *(long*)(pb+m_InBufPosition);
              pfltDst[0][i]= (T_Float)blTemp.i32 * fltScalingFactor;
              m_InBufPosition += iBufIdxInc;
              // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
              if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
               {                                       // so use the next (*if* there's one)
                 pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
                 if( pb==NULL ) // oops... no more filled 'input' data available ?!
                  { DOBINI();   // -> Sound_iSourceCodeLine
                    return -1;
                  }
               }
            } // end for
         }
       } // end if <32 bit mono>
      else
      if(m_InFormat.nChannels>=2) // 32 bit, multiple channels
       { DOBINI();  // -> Sound_iSourceCodeLine
         for( i=0; i<iNumberOfSamples; i++ )
          { for( iChannel=0; iChannel<m_InFormat.nChannels; ++iChannel )
             { blTemp.i32 = *(long*)(pb+m_InBufPosition);
               m_InBufPosition += 4;
               if( pfltDst[iChannel] != NULL )
                {  pfltDst[iChannel][i] = (T_Float)blTemp.i32 * fltScalingFactor;
                }
             }
            // Added 2011-03-13 for callers which need to read HUGE BLOCKS in one over:
            if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
             {                                       // so use the next (*if* there's one)
               pb = WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_WAIT_FOR_DATA, &pSoundBufferInfo);
               if( pb==NULL ) // oops... no more filled 'input' data available ?!
                { DOBINI();   // -> Sound_iSourceCodeLine
                  return -1;
                }
            }
          } // end for( i=0; i<iNumberOfSamples; i++ )
       } // end if <32 bit stereo>
    } // end if < 32 bits per sample >

  DOBINI();  // -> Sound_iSourceCodeLine

  if( m_InBufPosition >= m_InBufferSize )  // finished with this buffer
   {                                       // so add it back in to the Queue
     WrapInBufPositionAndGetPointerToNextFragment(CSOUND_WRAPBUF_OPT_DONT_WAIT, &pSoundBufferInfo);
   }
  m_i64NumSamplesRead += iNumberOfSamples;
  if( m_dblNominalInputSampleRate > 0 )
   { m_ldblSampleBasedInputTimestamp += (long double)iNumberOfSamples / m_dblNominalInputSampleRate;
     // (timestamp in UNIX format; used to provide a jitter-free timestamp
     //  in GetDateAndTimeForInput)
   }
  else if( m_dblNominalInputSampleRate > 0 )  // fall back to second best alternative..
   { m_ldblSampleBasedInputTimestamp += (long double)iNumberOfSamples / m_dblNominalInputSampleRate;
     // Note: this timestamp can only be accurate to a few ten milliseconds.
     //  If the application (spectrum lab) requires better timestamps, use GPS.
   }

  if( (m_InSampleLimit != 0) && (m_i64NumSamplesRead >= m_InSampleLimit) )
   {
     InClose();
     return 0;
   }
  DOBINI();   // -> Sound_iSourceCodeLine
  if( ( pOutChunkInfo != NULL ) && ( iNumberOfSamples>0 ) )
   {  pOutChunkInfo->nChannelsPerSample = nChannels;
      pOutChunkInfo->dwNrOfSamplePoints = iNumberOfSamples;
   }
  return iNumberOfSamples;
} // end CSound::InReadMultiChannel()

//---------------------------------------------------------------------------
int CSound::InReadStereo(   // used by Spectrum Lab ... until 2021-10-02, when adding support for 4 channels
        T_Float * pLeftData,     // [out] samples for 'left' audio channel (or "I")
        T_Float * pRightData,    // [out] samples for 'right' audio channel (or "Q")
        int iNumberOfSamples,    // [in] number of samples for each of the above blocks
        int iTimeout_ms,         // [in] max timeout in milliseconds, 0=non-blocking
        int *piHaveWaited_ms,    // [out,optional] "have been waiting here for XX milliseconds"
      T_ChunkInfo *pOutChunkInfo) // [out,optional,(*)] chunk info with timestamp, GPS, calib;
                                  //       see c:\cbproj\SoundUtl\ChunkInfo.h
  // Reads 'iNumberOfSamples' samples of floating-point samples
  // from the opened soundcard input (or audio I/O DLL, etc),
  // and if necessary splits the data into separated blocks for LEFT + RIGHT channel
  // (i.e. non-interlaced output for two separate channels).
  //   Returns:
  //   iNumberOfSamples   if 'iNumberOfSamples' samples were succesfully read
  //       0 =  if reaches the specified sample limit
  //      -1 =  if there is an error ( use GetError() to retrieve error )
  // For details and notes, see InReadMultiChannel()
{
  T_Float *pFltDestBlocks[2]; // pointer to a short ARRAY of pointers to the channel blocks
  int nDestBlocks = 1;         // number of destination blocks (1 for mono, 2 for stereo)
  if( pRightData != NULL )
   {  nDestBlocks = 2;
   }
  pFltDestBlocks[0] = pLeftData;
  pFltDestBlocks[1] = pRightData;
  return InReadMultiChannel( pFltDestBlocks, nDestBlocks, iNumberOfSamples,
                             iTimeout_ms, piHaveWaited_ms, pOutChunkInfo );
} // end CSound::InReadStereo()


#if(0)
//----------------------------------------------------------------------
int CSound::ReadInputSamplePoints( // preferred method since 2011-06
        float *pfltDest,            // [out] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
        int   iNumSamplePoints,     // [in] number of SAMPLE POINTS(!) to read
        int   nChannelsPerSample,   // [in] number of samples PER SAMPLE POINT
        int   iTimeout_ms ,         // [in] max timeout in milliseconds, 0=non-blocking
        int   *piHaveWaited_ms,     // [out,optional] "have been waiting here for XX milliseconds"
        T_ChunkInfo *pOutChunkInfo) // [out,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL(!)
  // Returns the number of sample-POINTS, or a negative error code .
{
  return AIO_Host_ReadInputSamplePoints(
        &aio_In,                 // [in,out] DLL-host instance data
        pfltDest,iNumSamplePoints,nChannelsPerSample,iTimeout_ms,
        pOutChunkInfo,           // [out,optional] chunk info
        NULL ); // [in,optional] address of a callback function; not used here (NULL)
} // end CSound::AIO_ReadInputSamplePoints()
#endif


//---------------------------------------------------------------------------
void CSound::InClose()
 //  Closes the Soundcard Input if open and free up resources.
{
 int i,j;
 int result;

  DOBINI();

  if( m_InputOpen && m_InWaiting ) // oops.. buffers may still be occupied...
   { m_InputOpen = FALSE;  // politely ask whoever-uses-this to release the buffer(s)
     for( i=0; i<5; ++i )
      { Sleep(20);
        if(! m_InWaiting ) // "the other guy" has returned, and released the buffer(s)
         { break;
         }
        if( SndThd_fCloseDevicesDuringSleep ) // received WM_POWERBROADCAST / PBT_APMSUSPEND somewhere in the GUI !
         { // instead of fooling around with "WaitForMultipleObjects" and other windows-specific crap,
           // bail out of this "tamed busy-spinning" loop (and many others)
           // as soon as recognizing this "going-to-sleep"-flag :
           break;
         }
      }
   } // end if < input-buffer(s) POSSIBLY occupied (in another thread) >
  m_InputOpen = FALSE;  // prevent other threads / callers from accessing buffers
    // (2008-10-19: this should also prevent "starving"  in ..InRead..()
  if ( m_hwvin != NULL) // used MMSYSTEM for input ?
    {
  //  if( m_fWaveInStarted )
  //   {  m_fWaveInStarted = FALSE;
  //      waveInStop    // counterpart to waveInStart() ?  There's no waveInStop() but waveInReset() :
  //      // > The waveInReset function stops input on the given waveform-audio input device
  //      // > and resets the current position to zero. All pending buffers
  //      // >  are marked as done and returned to the application.
  //   }
      if( (result=waveInReset(m_hwvin)) != MMSYSERR_NOERROR)
       { // waveInReset() went wrong... wonder why, only windooze knows ;-)
         SetInputError( result );  // -> m_ErrorCodeIn + chance for a COMMON BREAKPOINT
       }
#    if ( MOD_2015_11_25 )
      m_fWaveInStarted = FALSE;   // waveInStart() "undone" via waveInReset(). What an intuitive API...
#    endif // MOD_2015_11_25 ?
      for( i=0; i<m_NumInputBuffers; i++ )
       {
        if( m_InputWaveHdr[i].dwFlags & WHDR_PREPARED )
         {  DOBINI();
          if( (result=waveInUnprepareHeader(m_hwvin, &m_InputWaveHdr[i],
                 sizeof (WAVEHDR)) ) != MMSYSERR_NOERROR)
           { // 'UnprepareHeader' can go wrong if MMSYSTEM still playing
             //  (yes, "playing", thats what the doc says..)
             DOBINI();
             // 2005-04-23: Got here for the WOLF GUI with result=33
             //   which means "WAVERR_STILLPLAYING" (?\cbuilder6\include\mmsystem.h).
             //   A-ha . Who's "still playing", any why ?!
             //   Win32 programmer's reference about waveInUnprepareHeader() :
             // > WAVERR_STILLPLAYING :
             // >   The buffer pointed to by the pwh parameter is still in the queue.
             // A crude fix (not even a kludge) ...
             j = 10;  // wait up to 10 * 50ms 'til MMSYSTEM finished with this buffer:
             while( (j>0) && result==WAVERR_STILLPLAYING )
              { Sleep(50);
                result=waveInUnprepareHeader(m_hwvin, &m_InputWaveHdr[i], sizeof(WAVEHDR) );
                --j;
                if( SndThd_fCloseDevicesDuringSleep ) // received WM_POWERBROADCAST / PBT_APMSUSPEND somewhere in the GUI !
                 { // Got NO TIME to spend here (shortly before SUSPEND), so
                   // bail out of this "tamed busy-spinning" loop (and many others)
                   // as soon as recognizing this "going-to-sleep"-flag :
                   break;
                 }
              } // end while
             if( result!=MMSYSERR_NOERROR)
              { SetInputError( result );  // -> m_ErrorCodeIn + chance for a COMMON BREAKPOINT
              }
           }
         }
       } // end for(i..)
      DOBINI();
      if( (result=waveInClose(m_hwvin)) != MMSYSERR_NOERROR)
       { // a CLOSE-action failed... very mysterious but IT DOES HAPPEN !
         // (last occurred with result = 5 which means:  ???  )
         SetInputError( result );  // -> m_ErrorCodeIn + chance for a COMMON BREAKPOINT
       }
      m_hwvin = NULL;
    }

  DOBINI();

  if(m_iUsingASIO & 1)    // was using ASIO for input ?
   { m_iUsingASIO &= ~1;  // clear "using-ASIO-for-input"-flag
     if( m_iUsingASIO==0 )     // ASIO no longer in use now ?
      {
#if( SWI_ASIO_SUPPORTED )      // ASIO really support (through DL4YHF's wrapper)
        if( m_hAsio != C_ASIOWRAP_ILLEGAL_HANDLE )
         {
           DOBINI();  // -> Sound_iSourceCodeLine
           AsioWrap_Close( m_hAsio );
           DOBINI();  // -> Sound_iSourceCodeLine
           m_hAsio = C_ASIOWRAP_ILLEGAL_HANDLE;
         }
#endif // SWI_ASIO_SUPPORTED
      }
   } // end if(m_iUsingASIO & 1)


#if( SWI_AUDIO_IO_SUPPORTED )
  AIO_Host_CloseAudioInput( &aio_In );
  // 2012-10-26: Crashed above (in C:\RTL-SDR\RTLSDRGUI\ExtIO_RTLSDR.dll) ??
  // 2012-11-18: Same shit again (in C:\RTL-SDR\RTLSDRGUI\ExtIO_RTLSDR.dll) .
#endif // SWI_AUDIO_IO_SUPPORTED

  if( m_hComPort != INVALID_HANDLE_VALUE )
   { CloseHandle( m_hComPort );
     m_hComPort  =  INVALID_HANDLE_VALUE;
   }

  DOBINI();  // -> Sound_iSourceCodeLine
} // end InClose()

//---------------------------------------------------------------------------
long CSound::InGetNominalSamplesPerSecond(void)  // -> NOMINAL value (as set in InOpen)
{ return m_InFormat.nSamplesPerSec;
}

//---------------------------------------------------------------------------
long CSound::OutGetNominalSamplesPerSecond(void) // -> NOMINAL value (as set in OutOpen)
{ return m_OutFormat.nSamplesPerSec;
}


//---------------------------------------------------------------------------
int CSound::InGetNumberOfChannels(void)
{
  return m_InFormat.nChannels;
}

//---------------------------------------------------------------------------
int CSound::InGetBitsPerSample(void)
{
  return m_InFormat.wBitsPerSample;
}


//---------------------------------------------------------------------------
void CSound::SetInputError( int iErrorCode ) // -> m_ErrorCodeIn, plus debugger-breakpoint
{
  if( iErrorCode != NO_ERROR )
   { m_ErrorCodeIn = iErrorCode;
     DEBUGGER_BREAK(); // << set a breakpoint here, or use the 'common' breakpoint in DebugU1.cpp
   }
} // end CSound::SetInputError()

//---------------------------------------------------------------------------
void CSound::SetOutputError( int iErrorCode ) // -> m_ErrorCodeOut, plus debugger-breakpoint
{
  if( iErrorCode != NO_ERROR )
   { m_ErrorCodeOut = iErrorCode;
     DEBUGGER_BREAK(); // << set a breakpoint here, or use the 'common' breakpoint in DebugU1.cpp
   }
} // end CSound::SetOutputError()


//---------------------------------------------------------------------------
UINT CSound::OutOpen(
    //ex:int iOutputDeviceID,     // Identifies a soundcard, or a replacement like C_SOUND_DEVICE_AUDIO_IO
          // negative: one of the C_SOUND_DEVICE_... values defined in C:\cbproj\SoundUtl\Sound.h
          //  ( C_SOUND_DEVICE_DEFAULT_WAVE_INPUT_DEVICE = -1 is dictated by Windows;
          //    C_SOUND_DEVICE_SDR_IQ,   C_SOUND_DEVICE_PERSEUS,
          //    C_SOUND_DEVICE_AUDIO_IO, C_SOUND_DEVICE_ASIO  all added by DL4YHF )
        char * pszAudioOutputDeviceOrDriver, // [in] name of an audio device or DL4YHF's "Audio I/O driver DLL" (full path)
        char * pszAudioStreamID,  // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE (here: mandatory)
        char * pszDescription,    // [in] a descriptive name which identifies
           // the audio source ("audio producer"). This name may appear
           // on a kind of "patch panel" on the configuration screen of the Audio-I/O DLL .
           // For example, Spectrum Lab will identify itself as "SpectrumLab1"
           // for the first running instance, "SpectrumLab2" for the 2nd, etc .
        long i32SamplesPerSecond, // nominal sampling rate (not the "true, calibrated value"!)
        int  iBitsPerSample,      // ..per channel, usually 16 (also for stereo, etc)
        int  iNumberOfChannels,   // 1 or 2 channels (mono/stereo)
        DWORD dwMinBufSize,       // min output buffer size; measured in SAMPLE POINTS .
                 // (must be at least as large as the 'processing chunk size';
                 //  see caller in SoundThd.cpp : SoundThd_InitializeAudioDevices )
        DWORD SampleLimit ) // usually 0 = "continuous stream"
  // Opens a Soundcard, or many other kinds of 'audio output device' for writing.
  //
  // Sample size can be 1,2 or 4 bytes long depending on bits per sample
  // and number of channels.( ie. a 1000 sample buffer for 16 bit stereo
  // will be a 4000 byte buffer)
  //  Output does not start until at least half the buffers are filled
  //  or m_OutWrite() is called with a length of '0' to flush all buffers.
  //    parameters:
  //      iOutputDeviceID = device ID for waveOutOpen.
  //                  See Win32 Programmer's Reference for details.
  //      pWFX    = WAVEFORMATEX structure with desired soundcard settings
  //      dwMinBufSize = DWORD specifies the soundcard buffer size number
  //                  in number of samples to buffer.
  //                 If this value is Zero, the soundcard is opened and
  //                  then closed. This is useful   for checking the
  //                  soundcard.
  //      SampleLimit = limit on number of samples to be written. 0 signifies
  //                  continuous output stream.
  //    returns:
  //        0         if opened OK
  //      ErrorCode   if not
{
 int i,j,iErrorCode;
 int newOutBufferSize/*per "part"*/, newNumOutputBuffers;
 long i32TotalBufSizeInByte;
 char *pszAudioDeviceNameWithoutPrefix;
 BYTE *pb;
#if( SWI_ASIO_SUPPORTED )      // Support ASIO audio drivers too ?
 ASIOChannelInfo *pAsioChannelInfo;
#endif

  DOBINI();

   if(m_OutputOpen)
    { OutClose();     // -> m_hwvout=NULL
    }
   m_OutputOpen = FALSE;
   m_OutWaiting = FALSE;
   m_OutUnderflow = FALSE;
   if(m_OutEventHandle != NULL)
    { ResetEvent(m_OutEventHandle); // let OutFlushBuffer() wait when it needs to (!)
      // (without this, it didn't WAIT anymore after RE-opening the output,
      //  causing the output buffers to overflow )
    }
   strncpy( m_sz255AudioOutputDeviceOrDriver, pszAudioOutputDeviceOrDriver, 255 );
   m_sz255AudioOutputDeviceOrDriver[255] = '\0';

   m_sz255LastErrorStringOut[0] = 0;
   m_iOutHeadIndex = m_iOutTailIndex = 0; // output buffer EMPTY now (explained in OutFlushBuffer)
   m_OutSampleLimit = SampleLimit;  // if only a limited number of samples is to be played
   m_OutBufPosition = 0;
   // ex: m_OutFormat = *pWFX;    // copy a lot of params from WAVEFORMATEX struct
   // WoBu didn't want to have this damned "waveformatex"-stuff in the interface,
   // so got rid of this micro$oft-specific stuff, and use it only internally :
     // Prepare some soundcard settings (WAVEFORMATEX) required for opening,
     //  ALSO FOR THE AUDIO FILE READER / WRITER (!)
   m_OutFormat.wFormatTag = WAVE_FORMAT_PCM;
     // Since 2003-12-14, the "number of channels" for the ADC is not necessarily
     //       the same as the number of channels in the internal process !
   m_OutFormat.nChannels = (WORD)iNumberOfChannels;   // only 1 or 2 so far..
   m_OutFormat.wBitsPerSample = (WORD)iBitsPerSample;  // ex 16
   m_OutFormat.nSamplesPerSec = i32SamplesPerSecond; /* ex 11025 */
     // note that the WFXSettings require the "uncalibrated" sample rate !!
   m_OutFormat.nBlockAlign = (WORD)( m_OutFormat.nChannels *(m_OutFormat.wBitsPerSample/8) );
     // In a comment in MMREG.H, Microsoft says 'nBlockAlign' is the "block size of data" .
     // What a stupidly ambiguous comment. What they mean is "bytes per sample block",
     // for example nBlockAlign = 6 [byte] for a stereo stream with 24 "bits per single-channel sample".
   m_OutFormat.nAvgBytesPerSec = m_OutFormat.nSamplesPerSec * m_OutFormat.nBlockAlign;
   m_OutFormat.cbSize = 0/*!*/; // no "extra format information" appended to the end of the WAVEFORMATEX structure
   m_OutBytesPerSamplePoint = (m_OutFormat.wBitsPerSample/8)*m_OutFormat.nChannels;

  if( CountBits(m_dwOutputSelector) < m_OutFormat.nChannels )
   { m_dwOutputSelector = ( 0xFFFFFFFF >> (32-m_OutFormat.nChannels) );
   }

  pszAudioDeviceNameWithoutPrefix = Sound_GetAudioDeviceNameWithoutPrefix(
                           pszAudioOutputDeviceOrDriver, &m_iOutputDeviceID );


  // Use DL4YHF's  "Audio-IO" for output ?  ( something like in_AudioIO.DLL, see ?\AudioIO\*.* )
  if( m_iOutputDeviceID == C_SOUND_DEVICE_AUDIO_IO )
   {
#if( SWI_AUDIO_IO_SUPPORTED ) // host for AUDIO-I/O-DLLs, later also for Winrad ExtIO :

#else // !SWI_AUDIO_IO_SUPPORTED )
     SetOutputError( SOUNDIN_ERR_NOTOPEN );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
     return m_ErrorCodeOut;
#endif // SWI_AUDIO_IO_SUPPORTED
   } // end if( m_iOutputDeviceID == C_SOUND_DEVICE_AUDIO_IO )
  else // not AUDIO_IO, but what ? ...
   { // ....
   }


  // use MMSYSTEM or ASIO driver for OUTput ?
  if( (m_iOutputDeviceID==C_SOUND_DEVICE_ASIO)
     && APPL_fMayUseASIO // Since 2011-08-09, use of ASIO may be disabled through the command line
     )
   {
#if( SWI_ASIO_SUPPORTED )      // Support ASIO audio drivers too ?
      if((m_iUsingASIO&2)==0 ) // only if ASIO driver not already in use for OUTPUT..
       {
        if( !AsioWrap_fInitialized )
         {  AsioWrap_Init();     // init DL4YHF's ASIO wrapper if necessary
         }
        // Prepare the most important ASIO settings :
        AsioSettings MyAsioSettings;
        DOBINI();
        AsioWrap_InitSettings( &MyAsioSettings,
                               iNumberOfChannels,
                               i32SamplesPerSecond,
                               iBitsPerSample );
        DOBINI();
        MyAsioSettings.i32InputChannelBits  = m_dwInputSelector;
        MyAsioSettings.i32OutputChannelBits = m_dwOutputSelector;
        if( m_hAsio == C_ASIOWRAP_ILLEGAL_HANDLE )
         { // Only open the ASIO driver if not open already. Why ?
           // because an ASIO device always acts as IN- AND(!) OUTPUT simultaneously.
           m_hAsio = AsioWrap_Open( pszAudioDeviceNameWithoutPrefix, &MyAsioSettings, CSound_AsioCallback, (DWORD)this );
           if( (m_hAsio==C_ASIOWRAP_ILLEGAL_HANDLE) && (AsioWrap_sz255LastErrorString[0]!=0) )
            { // immediately save error string for later:
              strncpy( m_sz255LastErrorStringOut, AsioWrap_sz255LastErrorString, 79 );
              m_sz255LastErrorStringOut[79] = '\0';
            }
         }
        DOBINI();
        if( m_hAsio != C_ASIOWRAP_ILLEGAL_HANDLE ) // ASIO wrapper willing to co-operate ?
         { m_iUsingASIO |= 2;    // now using ASIO for OUTput...
           // BEFORE allocating the buffers, check if the ASIO driver will be kind enough
           // to accept the samples in the format *WE* send it ( some stubborn drivers don't...)
           pAsioChannelInfo = AsioWrap_GetChannelInfo( m_hAsio, 0/*iChnIndex*/, 0/*0= for OUTPUT*/ );
           if( pAsioChannelInfo!=NULL )
            { switch(pAsioChannelInfo->type)
               { // SOO many types, so hope for the best, but expect the worst .....  see ASIO.H !
                 case ASIOSTInt16LSB   : // the most usual format on an INTEL CPU (LSB first)
                      m_OutFormat.wBitsPerSample = 16;
                      break;
                 case ASIOSTInt24LSB   : // may be used too (but not by Creative)
                      m_OutFormat.wBitsPerSample = 24;
                      break;
                 case ASIOSTInt32LSB   : // Creative's "updated" driver usese
                                         // this format for 24(!)-bit output !
                      m_OutFormat.wBitsPerSample = 32;
                      break;

                 // a dirty dozen other formats are not supported here ! !
                 default:
                      strcpy( m_sz255LastErrorStringOut, "ASIO data type not supported" );
                      break;

                } // end switch(pAsioChannelInfo->type)
               // Revise the "number of bytes per sample" based on the above types:
               m_OutBytesPerSamplePoint = (m_OutFormat.wBitsPerSample/8)*m_OutFormat.nChannels;
            } // end if < got info about this ASIO OUTput channel >
         } // end if < successfully opened the ASIO driver >
        else  // bad luck (happens !) :  ASIO driver could not be loaded
         { SetOutputError( ERROR_WITH_ASIO_DRIVER );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
           if( m_sz255LastErrorStringOut[0] == 0 )
            { strcpy( m_sz255LastErrorStringOut, "Could not load ASIO driver" );
            }
           DOBINI();
           return m_ErrorCodeOut;
         }
       } // end if < ASIO driver not loaded yet >
#else  // name of ASIO driver specified, but ASIO not supported in compilation:
     SetOutputError( ERROR_WITH_ASIO_DRIVER );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
     if( m_sz255LastErrorStringOut[0] == 0 )
      { strcpy( m_sz255LastErrorStringOut, "Could not load ASIO driver" );
      }
     DOBINI();
     return m_ErrorCodeIn;
#endif // SWI_ASIO_SUPPORTED
   } // end if < use ASIO driver instead of MMSYSTEM ? >

  // 2015-02-01 : Be generous with the NUMBER OF BUFFERS (but not the size PER BUFFER) !
  //              In case of doubt, spend a few hundred kBytes more than necessary,
  //              to reduce the risk of missing samples in case of high CPU load :
  dwMinBufSize += m_OutFormat.nSamplesPerSec/10;  // at least 100 ms of audio (additionally)

  // Note: m_OutBytesPerSamplePoint may have been changed depending on ASIO driver .
  //  The following output-buffer-estimation applies to ASIO and MMSYSTEM :
  i32TotalBufSizeInByte = (long)(2*dwMinBufSize) * m_OutBytesPerSamplePoint; // may be as much as a HALF MEGABYTE !
  DOBINI();

  // Event for callback function to signal next buffer is free.
  // This Event will be Set in the output callback to wake up the 'writer' .
  if( m_OutEventHandle==NULL )
   {  m_OutEventHandle = CreateEvent(
           NULL,   // pointer to security attributes
           FALSE,  // flag for manual-reset event. If FALSE, Windows automatically resets the state to nonsignaled
                   //  after a single waiting thread has been released.
           FALSE,  // flag for initial state.  FALSE = "not signalled"
           NULL);  // pointer to event-object name
   }

  // open sound card output, depending on whether MMSYSTEM,
  //  ASIO driver,  or DL4YHF's AudioIO system   shall be used :
  if( m_iUsingASIO & 2 )   // using ASIO for input :
   {
#if( SWI_ASIO_SUPPORTED )      // Support ASIO audio drivers too ?
      // Already opened the driver above, but did NOT start it then,
      // because the buffers were not allocated yet.
#endif // SWI_ASIO_SUPPORTED
    }
   else if( m_iOutputDeviceID==C_SOUND_DEVICE_AUDIO_IO )
    {
#if( SWI_AUDIO_IO_SUPPORTED )
      if( aio_Out.h_AudioIoDLL == NULL )
       { // In this case, pszAudioDriverName should contain THE FULL PATH to the DLL
         // because SL must load this DLL from the very same place as Winamp...
         // ... which is, of course, Winamp's "plugins" folder !
         if( AIO_Host_LoadDLL( &aio_Out, pszAudioOutputDeviceOrDriver ) < 0 )
          {
           SetOutputError( SOUNDOUT_ERR_NOTOPEN );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
           strcpy( m_sz255LastErrorStringOut, "Couldn't load AUDIO_IO DLL" );
           DOBINI();
           return m_ErrorCodeOut;
          }
       }
      if( AIO_Host_OpenAudioOutput(
                &aio_Out,           // [in,out] T_AIO_DLLHostInstanceData *pInstData, DLL-host instance data
                pszAudioStreamID,   // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE (here: mandatory)
                "SpectrumLab",      // [in] a descriptive name which identifies
                    // the audio source ("audio producer"). This name may appear
                    // on the "patch panel" on the configuration screen of the Audio-I/O DLL .
                i32SamplesPerSecond, // [in] samples per second, "precise" if possible
                iNumberOfChannels,   // [in] 1=mono, 2=stereo, 4=quadrophonic
                200,                 // [in] max timeout in milliseconds for THIS call
                AIO_OPEN_OPTION_NORMAL // [in] bit combination of AIO_OPEN_OPTION_... flags
           ) < 0 )
       {   // failed to open the AUDIO-I/O-DLL for "output" :
           SetOutputError( SOUNDOUT_ERR_NOTOPEN );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
           strcpy( m_sz255LastErrorStringOut, "Couldn't open AUDIO_IO for output" );
           DOBINI();
           return m_ErrorCodeOut;
       }
      else // successfully opened the AUDIO-I/O-DLL for "output"
       {
       }
#endif  // SWI_AUDIO_IO_SUPPORTED
    }
   else // neither ASIO driver nor AudioIO used for output, but the ancient MMSYSTEM :
    { int iSoundOutputDeviceID;

      SOUND_fDumpEnumeratedDeviceNames = TRUE;  // flag for Sound_OutputDeviceNameToDeviceID()
      iSoundOutputDeviceID = Sound_OutputDeviceNameToDeviceID( pszAudioOutputDeviceOrDriver );


     if ( iSoundOutputDeviceID < 0)
          iSoundOutputDeviceID = WAVE_MAPPER;

#   if( SWI_UTILITY1_INCLUDED )
     UTL_WriteRunLogEntry( "CSound::OutOpen: Trying to open \"%s\", ID #%d, %d Hz, %d bit, %d channel(s), BlockAlign=%d, for output.",
                              (char*)pszAudioDeviceNameWithoutPrefix,
                              (int)iSoundOutputDeviceID,
                              (int)m_OutFormat.nSamplesPerSec,
                              (int)m_OutFormat.wBitsPerSample,
                              (int)m_OutFormat.nChannels,
                              (int)m_OutFormat.nBlockAlign );
#   endif // SWI_UTILITY1_INCLUDED ?


     // 2008-05-04: BCB triggered a strange breakpoint deep inside waveOutOpen, why ??
     if( (iErrorCode = waveOutOpen( &m_hwvout,
            iSoundOutputDeviceID,
            &m_OutFormat,   // [in] WAVEFORMATEX (not WAVEFORMATEXTENSIBLE)
            (DWORD)WaveOutCallback, (DWORD)this, CALLBACK_FUNCTION ) )
         != MMSYSERR_NOERROR )
      { // Could not open the WAVE OUTPUT DEVICE with the old-fashioned way.
        //    Try something new, using a WAVEFORMATEXTENSIBLE structure ...
        WAVEFORMATEXTENSIBLE wfmextensible;

        // not yet : SetOutputError( iErrorCode );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
        DEBUG_EnterErrorHistory( DEBUG_LEVEL_WARNING,0,UTL_USE_CURRENT_TIME,
           "CSound::OutOpen: waveOutOpen failed with WAVEFORMATEX, now trying WAVEFORMATEXTENSIBLE..." );

        // don't know what may be in it, so clear everything (also unknown components)
        memset( &wfmextensible, 0, sizeof(WAVEFORMATEXTENSIBLE) );
        // Now set all required(?) members. Based on info found from various sources.
        // Note that the new WAVEFORMATEXTENSIBLE includes the old WAVEFORMATEX .
        wfmextensible.Format = m_OutFormat; // COPY the contents of the WAVEFORMATEX(!)
        wfmextensible.Format.cbSize          = sizeof(WAVEFORMATEXTENSIBLE);
        wfmextensible.Format.wFormatTag      = WAVE_FORMAT_EXTENSIBLE;
        // wfmextensible.Format.nChannels, .wBitsPerSample, .nBlockAlign,
        //                     .nSamplesPerSec, .nAvgBytesPerSec already set !
        wfmextensible.Samples.wValidBitsPerSample = wfmextensible.Format.wBitsPerSample;
        wfmextensible.SubFormat = KSDATAFORMAT_SUBTYPE_PCM;
        // Now make a guess for the speaker configuration :
        wfmextensible.dwChannelMask = SPEAKER_FRONT_LEFT;   // MONO ...
        if(wfmextensible.Format.nChannels >= 2)             // STEREO..
           wfmextensible.dwChannelMask |= SPEAKER_FRONT_RIGHT;
        if(wfmextensible.Format.nChannels >= 3)             // ... and more ...
           wfmextensible.dwChannelMask |= SPEAKER_FRONT_CENTER;
        if(wfmextensible.Format.nChannels >= 4)
           wfmextensible.dwChannelMask |= SPEAKER_LOW_FREQUENCY;
        if(wfmextensible.Format.nChannels >= 5)
           wfmextensible.dwChannelMask |= SPEAKER_BACK_LEFT ;
        if(wfmextensible.Format.nChannels >= 6)             // typical "5.1" system
           wfmextensible.dwChannelMask |= SPEAKER_BACK_RIGHT;

        // Try again to open the audio output device, this time passing a pointer
        //  to a WAVEFORMATEXTENSIBLE struct (instead of the old WAVEFORMATEX ) :
        DOBINI();
        if( (iErrorCode = waveOutOpen( &m_hwvout, iSoundOutputDeviceID,
               (WAVEFORMATEX*)&wfmextensible ,
               (DWORD)WaveOutCallback, (DWORD)this, CALLBACK_FUNCTION ) )
           != MMSYSERR_NOERROR )
         {
           // Arrived here ? waveOutOpen() worked neither the OLD STYLE nor the NEW WAY.
           // 2015-10-20: Shit happens, over and over again. Got this "unspecified MMSYS error"
           //             on a Win8.1 system after the audio OUTPUT (to device 'MAGIC LCD 190 (Intel(R) Display')
           //             had been running for hours. SpecLab did't manage to kick the damned thing alive
           //             without yet another modification in Sound.cpp  .
           //   The error seemed to happen quite often after being paused
           //   via debugger !
           SetOutputError( iErrorCode );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
           OutClose();   // m_ErrorCodeOut = MMSYSERR_xxx
           DOBINI();
           DEBUG_EnterErrorHistory( DEBUG_LEVEL_ERROR,0,UTL_USE_CURRENT_TIME,
              "CSound::OutOpen: waveOutOpen failed with WAVEFORMATEX *and* WAVEFORMATEXTENSIBLE... \"%s\" .. I surrender !",
              ErrorCodeToString( iErrorCode ) );
           return m_ErrorCodeOut;
         }
      } // end if <could not open the WAVE INPUT DEVICE with the old-fashioned way>
    } // end else < not using ASIO driver for output but MMSYSTEM >

   if( dwMinBufSize == 0 )  // see if just a soundcard check
    {
      OutClose();      // if so close the soundcard OUTput
      DOBINI();
      return 0;
    }

  // Start out paused so don't let output begin until some buffers are filled.
  //   ( m_fOutputHoldoff will be SET somewhere below for this purpose )
  if( m_hwvout != NULL )
   {  // only if mmsystem used for output ..
     DOBINI();
     if( (iErrorCode = waveOutPause( m_hwvout ))!= MMSYSERR_NOERROR )
      {
        SetOutputError( iErrorCode );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
        OutClose();   // m_ErrorCodeOut = MMSYSERR_xxx
        DOBINI();
        return m_ErrorCodeOut;
      }
   } // end if( m_hwvout != NULL )

  DOBINI();

  // Allocate and prepare all the output buffers ....
  // How many OUTPUT buffers ? Should be enough for 0.2 to 0.5 seconds :
  // Allocate and prepare all the output buffers .... same principle as in InOpen:
  i = i32TotalBufSizeInByte / 4/*parts*/; // -> approx size of each buffer-part, in BYTES.
  if(i<1024)  i=1024;    // at least 512 bytes(!) per buffer-part
  if(i>65536) i=65536;   // maximum  64 kByte per buffer-part
  newNumOutputBuffers = (i32TotalBufSizeInByte+i-1) / i;
  ++newNumOutputBuffers; // ..because one of the N buffer-parts must remain empty
  if( newNumOutputBuffers < 4)  // should use at least FOUR buffers, and..
      newNumOutputBuffers = 4;
  if( newNumOutputBuffers > SOUND_MAX_OUTPUT_BUFFERS ) // a maximum of SIXTEEN buffers..
      newNumOutputBuffers = SOUND_MAX_OUTPUT_BUFFERS;
  // How large should each buffer-part be ?
  i = (i32TotalBufSizeInByte + newNumOutputBuffers/*round up*/ - 1)  / newNumOutputBuffers;
  // Traditionally, the size of each 'buffer part' was be a power of two.
  // But for 24 bit/sample, that's totally unsuited because the buffer size
  // must always be a multiple of what Microsoft calls 'nBlockAlign' in their 'WAVEFORMATEX' !
  newOutBufferSize = 64 * m_OutFormat.nBlockAlign;  // ex: at last 512  [bytes per 'buffer part']
  while( (newOutBufferSize < i) && (newOutBufferSize < 65536) )
   { newOutBufferSize <<= 1;
   }
  DOBINI();  // -> Sound_iSourceCodeLine
  if(m_pbOutputBuffer != NULL )
   { // free old buffer (with possibly different SIZE) ?
     if( (m_OutBufferSize!=newOutBufferSize) || (m_NumOutputBuffers!=newNumOutputBuffers) )
      { m_OutBufferSize = m_NumOutputBuffers = 0; // existing buffers are now INVALID
        UTL_free( m_pbOutputBuffer );
        m_pbOutputBuffer = NULL;
      }
   }
  if( m_pbOutputBuffer==NULL )
   {
     m_pbOutputBuffer = (BYTE*)UTL_NamedMalloc( "TSoundO", newNumOutputBuffers * newOutBufferSize);
     m_OutBufferSize    = newOutBufferSize/*per "part"*/;
     m_NumOutputBuffers = newNumOutputBuffers;
   }
  if ( m_pbOutputBuffer==NULL )
   { // could not allocate the sound buffer .. sh..
     m_OutBufferSize = m_NumOutputBuffers = 0; // existing buffers are INVALID
     OutClose();
     SetOutputError( MEMORY_ERROR );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
     DOBINI();
     return m_ErrorCodeOut;
   }

  // initialize output data buffer (which is in fact a SINGLE CONTIGUOUS BLOCK)
  pb = m_pbOutputBuffer;
  j  = m_OutBufferSize/*per "part"*/  *  m_NumOutputBuffers;
  while( (j--) > 0 ) // clear this sound buffer
   { *pb++ = 0;
   }

  // Prepare those strange 'wave headers' (windows multimedia API) ..
  for(i=0; i<m_NumOutputBuffers; i++ )
   {
     // initialize output data headers
     //   (they should ALL be unprepared, and NOT IN USE in this moment,
     //    which can be seen by the 'dwFlags' member of each block :
     if(  m_OutputWaveHdr[i].dwFlags  &
            (   WHDR_INQUEUE/*16*/ /* "Set by Windows to indicate that the buffer is queued for playback" */
             |  WHDR_PREPARED/*2*/ /* "... the buffer has been prepared with waveOutPrepareHeader" */
       )    )
      { LOCAL_BREAK(); // << set breakpoint HERE (or in "LocalBreakpoint" )
      }
     m_OutputWaveHdr[i].dwBufferLength = m_OutBufferSize; // .. for a SINGLE BUFFER (there will be multiple)
     m_OutputWaveHdr[i].dwBytesRecorded= 0; // here: means "number of bytes PLAYED" from this buffer
     m_OutputWaveHdr[i].dwFlags = 0;
     m_OutputWaveHdr[i].dwUser  = 0; // ex: = NULL; but Dev-C++ complained
     m_OutputWaveHdr[i].dwLoops = 0;
     m_OutputWaveHdr[i].lpData  = (LPSTR)m_pbOutputBuffer + i*m_OutBufferSize;

     // prepare output data headers (only if MMSYSTEM is used here) :
     DOBINI();
     if( m_hwvout != NULL )
      { if(  ( iErrorCode = waveOutPrepareHeader(m_hwvout, &m_OutputWaveHdr[i],
              sizeof(WAVEHDR)) ) != MMSYSERR_NOERROR )
         { SetOutputError( iErrorCode ); // -> m_ErrorCodeOut, plus debugger-breakpoint
           OutClose();   // m_ErrorCodeOut = MMSYSERR_xxx
           DOBINI();
           return m_ErrorCodeOut;
         }
      } // end if( m_hwvout != NULL )
    } // end for <all output buffers>
  DOBINI();
  m_OutputOpen = TRUE;
  m_fOutputHoldoff = TRUE;  // output starts when enough samples are collected
  DOBINI();
  return(0);  // no error
}  // end OutOpen()

//--------------------------------------------------------------------------
BOOL CSound::IsOutputOpen(void)   // checks if the sound output is opened
{
  return m_OutputOpen;
}

//--------------------------------------------------------------------------
char *CSound::GetInputDeviceName(void)
{ return m_sz255AudioInputDeviceOrDriver;
}

//--------------------------------------------------------------------------
char *CSound::GetOutputDeviceName(void)
{ return m_sz255AudioOutputDeviceOrDriver;
}

//--------------------------------------------------------------------------
int  CSound::GetNumInputBuffers(void)
{ return m_NumInputBuffers;  // number of audio buffers used  for INPUT (from the soundcard)
}

//--------------------------------------------------------------------------
int  CSound::GetNumOutputBuffers(void)
{ return m_NumOutputBuffers;  // number of audio buffers used for OUTPUT (to the soundcard)
}

//--------------------------------------------------------------------------
int  CSound::GetNumSamplesPerInputBuffer(void)
{
  if( m_InBytesPerSamplePoint >= 1 )
      return m_InBufferSize / m_InBytesPerSamplePoint;
  else
      return m_InBufferSize;
}

//--------------------------------------------------------------------------
int  CSound::GetNumSamplesPerOutputBuffer(void)
{
  if( m_OutBytesPerSamplePoint >= 1 )
      return m_OutBufferSize / m_OutBytesPerSamplePoint;
  else
      return m_OutBufferSize;
}


//--------------------------------------------------------------------------
int  CSound::GetInputBufferUsagePercent(void)
{ // Checks the current 'buffer usage' of the sound input in percent.
  //   0 means all buffers empty (0..30% is OK for an input buffer),
  // 100 means all buffers are occupied (80..100% is BAD for an input,
  //                  because there's the risk to lose input samples)
 int occupied_buffers;

  if(m_NumInputBuffers<=0)   // no output buffers allocated ?
     return 0;
  if( m_InWaiting ) // waiting for an input buffer to become available ?
     return 0;      //  -> all input buffers are empty ! (which is GOOD)

  occupied_buffers = m_iInHeadIndex - m_iInTailIndex;

  if(occupied_buffers<0)  // circular buffer-index wrap
     occupied_buffers+=m_NumInputBuffers;

  return (100*occupied_buffers) / m_NumInputBuffers;
}


//--------------------------------------------------------------------------
int  CSound::GetNumOccupiedOutputBuffers(void)
  // Returns the number of output buffers (0...m_NumOutputBuffers)
  // which are still "occupied", i.e. waiting to be written to the output.
  // Formerly a part of GetOutputBufferUsagePercent(), but separated in
  // and extra function for use with GetOutputLatency() .   [2015-01-25]
{ int occupied_buffers;

  if( m_OutWaiting )  // when WAITING FOR OUTPUT ...
   { return m_NumOutputBuffers;  // .. then *ALL* output buffers are occupied !
     // (and thus the output latency is at the maximum)
   }

  // EnterCriticalSection(&m_CriticalSection);
  occupied_buffers = m_iOutHeadIndex - m_iOutTailIndex;
  // LeaveCriticalSection(&m_CriticalSection);

  if(occupied_buffers<0)  // ring-buffer wrap
     occupied_buffers+=m_NumOutputBuffers;
  if(occupied_buffers>0)  // subtract one because ONE of the buffers..
   --occupied_buffers;    // .. must remain empty, like a classic circular, lock-free, FIFO
  // (without this, dropouts were audible though occupied_buffers was ONE)

  return occupied_buffers;
} // end GetNumOccupiedOutputBuffers()

//--------------------------------------------------------------------------
int  CSound::GetOutputBufferUsagePercent(void)
{ // Checks the current 'buffer usage' of the sound output in percent.
  //   0 means all buffers empty (which is BAD for an output buffer),
  // 100 means all buffers are occupied (50..100% is GOOD for an output).
 int occupied_buffers = GetNumOccupiedOutputBuffers();

  if(m_NumOutputBuffers<=0)   // no output buffers allocated ?
     return 0;
  else
     return (100*occupied_buffers) / m_NumOutputBuffers;
} // end GetOutputBufferUsagePercent()


//--------------------------------------------------------------------------
double CSound::GetOutputTimePerBuffer_sec(void)
   // [in] m_OutBufferSize = "size of a SINGLE output buffer in BYTES"
   //      + various other ugly stuff (m_OutFormat, etc) required to convert
   //      the above 'buffer size' into a TIME IN SECONDS.
   // [return] number of seconds for a single buffer (in the output queue).
   //        Used by GetOutputLatency() .
{
  double d = OutGetNominalSamplesPerSecond();  // -> NOMINAL value (as set in OutOpen)
  if( d > 0.0 )
   { return GetNumSamplesPerOutputBuffer() / d;
   }
  else
   { return 0;
   }
} // end GetOutputTimePerBuffer_sec()

//--------------------------------------------------------------------------
double CSound::GetOutputLatency(void)
  // Returns the expected output latency in seconds,
  //   guesstimated from the buffer usage
  //   WHEN THE LAST BLOCK OF SAMPLES WAS WRITTEN TO THE OUTPUT .
{
  return m_dblOutputLatency_sec;
}

//--------------------------------------------------------------------------
BOOL CSound::GetInputProperties(T_SoundSampleProperties *pProperties)
{
  if(pProperties)
   { pProperties->wChannels       = m_InFormat.nChannels;
     pProperties->dwSamplesPerSec = m_InFormat.nSamplesPerSec;
     pProperties->wBitsResolution = m_InFormat.wBitsPerSample;
     return TRUE;
   }
  else
     return FALSE;
} // end GetInputProperties()

//--------------------------------------------------------------------------
BOOL CSound::GetOutputProperties(T_SoundSampleProperties *pProperties)
{
  if(pProperties)
   { pProperties->wChannels       = m_OutFormat.nChannels;
     pProperties->dwSamplesPerSec = m_OutFormat.nSamplesPerSec;
     pProperties->wBitsResolution = m_OutFormat.wBitsPerSample;
     return TRUE;
   }
  else
     return FALSE;
} // end GetOutputProperties()

//---------------------------------------------------------------------
T_Float* CSound::ResizeInterleaveBuffer( int iNumSamplePoints )
   // Returns a pointer to the 'sufficiently resized' interleave buffer.
   // NULL means "out of memory" which is unlikely to happen.
{
  if( iNumSamplePoints > m_i32InterleaveBufferSize )
   { // The interleave/de-interleave buffer is too small, or doesn't exist.
     if( m_pfltInterleaveBuffer != NULL )
      { UTL_free( m_pfltInterleaveBuffer );
      }
     m_pfltInterleaveBuffer = (T_Float*)UTL_NamedMalloc( "IleavBuf", iNumSamplePoints * 2 * sizeof( T_Float ) );
     if( m_pfltInterleaveBuffer != NULL )
      { m_i32InterleaveBufferSize = iNumSamplePoints;
      }
     else
      { m_i32InterleaveBufferSize = 0;
      }
   }
  return m_pfltInterleaveBuffer;
} // end CSound::ResizeInterleaveBuffer()


//---------------------------------------------------------------------
int CSound::GetNumberOfFreeOutputBuffers(void)
   // Result:  0 ... m_NumOutputBuffers MINUS ONE (!)
{
   int i = m_iOutHeadIndex - m_iOutTailIndex; // -> number of OCCUPIED buffers
   if( i<0 )  i += m_NumOutputBuffers;    // .. circular unwrapped..
   i = m_NumOutputBuffers - 1 - i;        // -> number of FREE buffers (available for filling)

   // Note: like in a classic circular, lock-free FIFO,
   //       ONE of the <m_NumOutputBuffers> parts of the buffer
   //       is always empty.
   // Thus, the difference between the above two buffer indices (wrapped around)
   //       tells us (here for FOUR buffers, head minus tail modulo 4 ):
   //   difference = 0   :  all four buffers ('part') are EMPTY, and THREE are free for filling
   //   difference = 1   :  one of the four buffers is FILLED,   and TWO are free for filling
   //   difference = 2   :  two buffers are filled,              and ONE is free for filling
   //   difference = 3   :  three buffers are filled, WHICH IS THE MAXIMUM (!)
   //   difference = 4   :  impossible due to the modulo-operation

  return i;
} // end CSound::GetNumberOfFreeOutputBuffers()

//---------------------------------------------------------------------
int CSound::OutFlushBuffer(int iTimeout_ms) // flushes m_OutputWaveHdr[m_iOutHeadIndex] to the output device; and WAITS if necessary
   // Called INTERNALLY if( m_OutBufPosition >= m_OutBufferSize ) ,
   //   i.e. when another part of the output buffer is full enough
   //   to be sent to the soundcard .
   // Returns the number of MILLISECONDS spent here waiting for the soundcard
   //  (giving the CPU to other threads, which may be important for the caller to know),
   //  or a NEGATIVE result if the caller cannot add more data because
   //  all output buffers are FULL .
   //   Return value 0 means "didn't wait, and there's a free buffer now" .
   //
   //
   // Since 2013-02-17, m_iOutHeadIndex and m_iOutTailIndex are treated like
   //                   a classic, lock-free, circular FIFO :
   // * m_iOutHeadIndex==m_iOutTailIndex means NONE of the 'output buffers'
   //                    is filled, i.e. "completely empty", which indicates a problem
   //                    (output-buffer not filled quick enough -> audio drop-out) .
   // * ( (m_iOutHeadIndex+1) % m_NumOutputBuffers) == m_iOutTailIndex
   //                    means ALL buffers are full, and the caller must wait
   //                    (or be blocked) until more samples can be filled in.
   // * It's impossible to have ALL 'm_NumOutputBuffers' occupied and queued up for output,
   //                    because (for example with FOUR buffers)
   //                    if the HEAD-INDEX would be FOUR STEPS AHEAD of the
   //                    tail index, the situation (with FOUR FULL BUFFERS) would be:
   //           m_iOutHeadIndex==(m_iOutTailIndex+4) modulo 4 == m_iOutTailIndex;
   //                    i.e. same situation as for "completly empty" buffers !
   //
   // Example: Typical situation when OutFlushBuffer() is CALLED .
   //          When RETURNING from OutFlushBuffer(), the caller expects to be able
   //          to add more samples into the next buffer (at the 'HEAD' index).
   //          The audio output driver's 'output completion callback'
   //          will (later) increment the TAIL index (m_iOutTailIndex) .
   //
   //   BUFFER INPUT SIDE:                       BUFFER OUTPUT SIDE:
   //     "Head Index"                             "Tail Index"
   //                          ____________
   // m_iOutHeadIndex --->    |Buffer[0] : | (this buffer is currently being filled
   //   |                     | filling    |       in CSound::OutWriteStereo,
   //   |                     |____________|       then passed to waveOutWrite()
   //   | (when this part of the buffer
   //   |       is COMPLETELY filled,
   //   |  OutFlushBuffer waits
   //   |    until Buffer[new_head_index]
   //   |    is not 'playing' anymore,
   //   |  then the new (incremented)
   //   |  HEAD INDEX is set in OutFlushBuffer )
   //   |
   //  \|/                      ____________
   //   - -new_head_index- -> |Buffer[1] : | <--- m_iOutTailIndex (don't touch this buffer yet
   //                         | PLAYING    |        |              because it's used by mmsystem.
   //                         |____________|        |  The TAIL index will be incremented in WaveOutCallback(),
   //                                               |  when output is "done" by the soundcard driver / MMSYSTEM !
   //                                               |
   //                          ____________        \|/
   //                         |Buffer[2] : | <- - - - - - -    (waiting to be played)
   //                         |  queued up |        .
   //                         |____________|        .
   //                                               .
   //                                               .
   //                          ____________        \./
   //                         |Buffer[3] : | <- - - - - - -    (waiting to be played)
   //                         |  queued up |
   //                         |____________|
   //
   //
   //
   //
{
  LONGLONG t1, t2;
  MMRESULT result;
  int  i, new_head_index, iWaited_ms = 0; // , iWaited_for_single_object = 0;
  DWORD dw, /*dwFlags1,*/ dwFlags2, dwFlags3, dwFlags4;
  BOOL  must_wait = FALSE;
  double d;

  if(  (m_NumOutputBuffers<=0) || (m_OutEventHandle==NULL) )
   { if( m_ErrorCodeOut == 0 )
      { SetOutputError( SOUNDOUT_ERR_NOTOPEN );  // -> m_ErrorCodeOut + chance for a COMMON BREAKPOINT
      }
     return -1;   // definitely 'not open for business'
   }


  // dwFlags1 = m_OutputWaveHdr[m_iOutHeadIndex].dwFlags; // for diagnostics..
  dwFlags2 = dwFlags3 = dwFlags4 = 0;
#ifdef __BORLANDC__
  (void)dwFlags2;
#endif


  // When m_OutBufPosition reaches m_OutBufferSize, it's time to switch
  //      to the next buffer part. If no 'free' buffer is availble, wait .
  if( m_OutBufPosition >= m_OutBufferSize/*measured in BYTES, not SAMPLES*/ )
   {
     m_OutBufPosition = 0;  // sample index to fill the NEW output block (one part of the buffer)

     i = GetNumberOfFreeOutputBuffers(); // -> 0 ... m_NumOutputBuffers MINUS ONE (!)

     // If all buffers are full then need to wait.  See ASCII graphics above !
     if( i<=0 )  // according to the FIFO indices, all buffer-parts are full !
      { must_wait = TRUE;  // this is the 'normal' way to detect 'no available buffer' ..
      }
     new_head_index = (m_iOutHeadIndex+1) % m_NumOutputBuffers;
     if( m_hwvout != NULL ) // only for the standard windoze multimedia API ..
      {
        dwFlags2 = m_OutputWaveHdr[new_head_index].dwFlags; // 2nd check for a 'free' buffer:
        if( (dwFlags2 & WHDR_INQUEUE ) != 0 )  // the 'new' buffer is still PLAYING..
         { must_wait = TRUE;
           if( i>0 ) // Ooops.. buffer NOT FULL ?! ?!
            { // THIS SHOULD NOT HAPPEN .
#            ifdef __BORLANDC__
              (void)i;
#            else
              i = i;
#            endif
            }
         } // end if < the 'new' (next-to-be-filled) buffer is STILL BEING PLAYED ? >
      } // end if < using the 'waveOut' API >

     if( must_wait ) // -> WAIT until a buffer gets available, or bail out (if caller doesn't want to be blocked)
      {
        if( iTimeout_ms > 0 )   // CALLER said it's ok to wait ..
         { m_OutWaiting = TRUE; // here set in OutFlushBuffer(), cleared in audio callback
           //
           // Wait for mmsystem to free up a new buffer, i.e. 'finish using' it.
           //
           // ex: m_OutWaiting = TRUE; // set in OutFlushBuffer(), cleared in audio callback
           //     2013-02-10 : m_OutWaiting now set BEFORE calling waveOutWrite() !
           // Wait until the output callback function
           //      calls  SetEvent( pCSound->m_OutEventHandle) ..
           // 2008-10-19 : Strange things happened when WaitForSingleObject
           //     was called from a thread with THREAD_PRIORITY_ABOVE_NORMAL !
           // 2013-02-17 : Does SetEvent *queue up multiple events* for the
           //     same 'consumer' (thread which calls WaitForSingleObject) ?
           //     WaitForSingleObject seemed to return immediately, even though
           //     m_OutputWaveHdr[m_iOutHeadIndex] was 'still playing' ....
           DOBINI();
           QueryPerformanceCounter( (LARGE_INTEGER*)&t1 );

           if( (dw=WaitForSingleObject( m_OutEventHandle, iTimeout_ms ) )
                                     != WAIT_OBJECT_0 )
            { DOBINI();
              if( dw == WAIT_FAILED ) // not just a timeout, but a real problem with WaitForSingleObject
               {  dw = GetLastError(); // is GetLastError aware of the calling thread ? mystery..
#                ifdef __BORLANDC__
                  (void)dw;
#                else
                  dw = dw;
#                endif
               }
              SetOutputError(  SOUNDOUT_ERR_TIMEOUT );
              OutClose();
              DOBINI();
              return -1;   // ERROR from OutFlushBuffer() : "failed to wait"
            }
           else
            { // waited successfully... (?)

              // How long did WaitForSingleObject() REALLY wait ?
              QueryPerformanceCounter( (LARGE_INTEGER*)&t2 );
              t2 -= t1;
              QueryPerformanceFrequency( (LARGE_INTEGER*)&t1 ); // -> frequency in Hertz
              if( t1>0 )
               { iWaited_ms = (1000 * t2) / t1/*Hz*/;
               }
              else
               { iWaited_ms = 0;
               }
              // iWaited_for_single_object = iWaited_ms; // for diagnostics, further below

              // After 'waiting successfully', the buffer should NOT be full now anymore .
              //  Check again if there really is an empty buffer entry now :
              i = GetNumberOfFreeOutputBuffers(); // -> 0 ... m_NumOutputBuffers MINUS ONE (!)
              if( i<=0 )  // still no joy... try something different (a KLUDGE, "second chance") :
               { i=i; // <<< set a breakpoint here !
                 // Since WaitForSingleObject() played a dirty trick, try something else :
                 while( iWaited_ms < iTimeout_ms )
                  { Sleep(20);
                    iWaited_ms += 20;
                    i = GetNumberOfFreeOutputBuffers();  // hope WaveOutCallback() was called during Sleep() ...
                    if( i>0 )                            // obviously YES, because the buffer is free now !
                     { break;  // "success on the second attempt", using Sleep() instead of WaitForSingleObject() !
                       // 2013-02-17 : This kludge seemed to work !
                     }
                    if( SndThd_fCloseDevicesDuringSleep ) // received WM_POWERBROADCAST / PBT_APMSUSPEND somewhere in the GUI !
                     { // Got NO TIME to spend here (shortly before SUSPEND), so
                       // bail out of this "tamed busy-spinning" loop (and many others)
                       // as soon as recognizing this "going-to-sleep"-flag :
                       break;
                     }
                  } // end while
                 // 2013-02-17 : Got here with i=1 (i.e. "late success") quite often. WHY ?
                 //              total time spent waiting  = 220 ms ( iWaited_ms )
                 //              iWaited_for_single_object =   0 ms ( why ? )
                 // Calling 'ResetEvent' before 'waveOutWrite' cured this .
                 // But this "second chance" was left in the code, for what it's worth. "Defensive programming" :)
                 if( i<=0 )    // completely out of luck ?
                  { LOCAL_BREAK();
                    return -1; // no success waiting for a free output-buffer !
                  }
               } // end if < NO EMPTY BUFFER AFTER WAITING ? >

              // remove certain old errors:
              DOBINI();
              if(m_ErrorCodeOut == SOUNDOUT_ERR_UNDERFLOW)
               { m_ErrorCodeOut = NO_ERRORS;
               }
            } // end if < waited SUCCESSFULLY >
         }
        else // iTimeout_ms<=0
         { // ALL output-buffers are full, but the caller does NOT want to wait for output..
           // .. most likely, because Spectrum Lab has already been waiting for INPUT.
           return -1;  // let the caller know he cannot append more samples now !
         } // end else( iTimeout_ms > 0 )
      } // end if( must_wait )

     // Arrived here: There's at least one more buffer which may be filled,
     //        and there's one more buffer which can be queued for output.
     //        (the latter because m_OutBufPosition was >= m_OutBufferSize)
     //
     // Regardless of HOW MANY buffers are already filled,
     //   START PLAYING this part of the buffer by calling waveOutWrite() .
     //   At this point, there's at least one full buffer,
     //   and waveOutWrite should be called even if there are
     //   are other buffers already queued up for output .
     // Note:  m_iOutHeadIndex is incremented in OutFlushBuffer();
     //        m_iOutTailIndex is incremented in WaveOutCallback() !
#   if( SWI_AUDIO_IO_SUPPORTED )
     if( aio_Out.h_AudioIoDLL != NULL )
      { // send the audio stream to an AUDIO-I/O-LIBRARY (DLL);
        // using the audio-IO-DLL client (host) from C:\cbproj\AudioIO\AudioIO.c .
        // In this case, Sound.cpp calls AIO_Host_WriteOutputSamplePoints()
        // somewhere else, and does NOT perform any buffering of its own !
        // return AIO_Host_WriteOutputSamplePoints( .. ) ;
      }
#   endif // SWI_AUDIO_IO_SUPPORTED
     if( m_hwvout != NULL ) // use the standard windoze multimedia API
      { DOBINI();
        if( iTimeout_ms>0 )
         { m_OutWaiting = TRUE; // here set in OutFlushBuffer(), cleared in WaveOutCallback()
         }
        ResetEvent( m_OutEventHandle );  // added 2013-02-17: convince WaitForSingleObject() to REALLY wait
        dwFlags3 = m_OutputWaveHdr[m_iOutHeadIndex].dwFlags;  // initially 2 = only WHDR_PREPARED, not DONE; later 3=PREPARED+DONE
        result = waveOutWrite( m_hwvout,&m_OutputWaveHdr[m_iOutHeadIndex], sizeof (WAVEHDR)/*!!!*/ );
        dwFlags4 = m_OutputWaveHdr[m_iOutHeadIndex].dwFlags;  // usually 18 = WHDR_PREPARED(2) + WHDR_INQUEUE(16)
        // > The waveOutWrite function sends a data block to the given waveform-audio output device.
        // > Returns MMSYSERR_NOERROR if successful or an error otherwise (..)
        // > When the buffer is finished, the WHDR_DONE bit is set
        // > in the dwFlags member of the WAVEHDR structure.
        //    ( in fact, multiple blocks should be queued up for output,
        //      to minimize the risk of audio drop-outs.
        //      The waveOut-API calls us back (WaveOutCallback) at an UNPREDICTABLE time,
        //      and WaveOutCallback will increment m_iOutTailIndex also at an UNPREDICTABLE time.
        //      Thus m_iOutTailIndex may ONLY be set in the callback;
        //      outside the callback it is 'read-only' ! )
        //
        if( (dwFlags3!=2 && dwFlags3!=3) || (dwFlags4!=18) )
         { LOCAL_BREAK(); // << set breakpoint HERE (or in "LocalBreakpoint" )
           // 2013-02-17 : Got here with dwFlags3=18, which means 'ALREADY IN QUEUE' (application error) .
           //       dwFlags1= ,  dwFlags2= ,  dwFlags3= ,   dwFlags4=   ,
           //       m_iOutHeadIndex=2 ,  new_head_index = 3, m_iOutTailIndex = 2 ,
           //       result = 33 = 'still playing' .
           //
         }
        DOBINI();
        if( result != MMSYSERR_NOERROR/*0*/ ) // oops.. what's wrong with waveOutWrite() ?
         { LOCAL_BREAK(); // << set breakpoint HERE (or in "LocalBreakpoint" )
           SetOutputError( result );
           // 2013-02-10: Occasionally got here with result=33 . WTF .. ?!
           // > error code 33 = ERROR_LOCK_VIOLATION
           // > The process cannot access the file because another process
           // > has locked a portion of the file.
           // After that happened: dwFlags1 = dwFlags2 = 18
           //                = WHDR_INQUEUE (16) + WHDR_PREPARED (2) ! !
           // ( constants found in C:\Programme\CBuilder6\Include\mmsystem.h )
           // Not good ! Sounds like m_iOutHeadIndex was incremented
           // too early, so that m_OutputWaveHdr[m_iOutHeadIndex] was STILL IN USE !
           //
           //
           iWaited_ms = -1;  // return value for 'no succcess'
         }
        else // waveOutWrite successfull ...
         {

           // Time to switch to the next buffer for 'filling' (left side of the ASCII graphics).
           new_head_index = (m_iOutHeadIndex+1) % m_NumOutputBuffers;
           m_iOutHeadIndex = new_head_index; // set the new (incremented) head index; here in OutFlushBuffer() AND NOWHERE ELSE
           // Note: Since the time when WaveOutCallback() is called is UNPREDICTABLE,
           //       WaveOutCallback() must only increment m_iOutTailIndex,
           //       but it must not look at m_iOutHeadIndex because the latter
           //       may, or may not, be incremented at that time .
           //

           // Guesstimate for the momentary output latency:
           d = GetOutputTimePerBuffer_sec(); // E-MU 0202 at 192kS/s : d = 85.3 ms (per buffer)
           i = GetNumOccupiedOutputBuffers();
           if( i > 0 )
            { --i;                           // << modified 2015-01-25
            }
           m_dblOutputLatency_sec = d * (double)i;
           // When setting m_dblOutputLatency_sec *here*, and SUBTRACTING ONE (above),
           //  the RS-232 keying output lagged the audible Morse tone by
           //  30 milliseconds (measured with an E-MU 0202, only the DAC(!) enabled,
           //  time of a single buffer (d) = 85.3 ms . Confirms there was NO additional
           //  buffer in the output queue.
           // See also: Guesstimation of T_ChunkInfo.dblOutputLatency_sec in SoundThd.cpp .

         } // end else < waveOutWrite successfull >

        if( m_fOutputHoldoff )   // holdoff logic doesn't start sound
         {                       // until half the buffers are full
           if( m_iOutHeadIndex >= m_NumOutputBuffers/2 )
            {
              m_fOutputHoldoff = FALSE;
              result = waveOutRestart( m_hwvout ); // buffer full enough, start output
              // > The waveOutRestart function resumes playback
              // > on a paused waveform-audio output device.
              // ("Restart" doesn't mean "re-start from the beginning" here)
#            ifdef __BORLANDC__
              (void)result;
#            endif
            }
         }
      } // end if < use 'wave out' API >
#   if( SWI_ASIO_SUPPORTED )
     else // m_hwvout==NULL  ->  send it to ASIO driver instead ?
      {
        m_OutputWaveHdr[m_iOutHeadIndex].dwBytesRecorded = 0; // for ASIO callback !
        // here: 'dwBytesRecorded' actually means "number of bytes PLAYED" from this buffer.
        // The ASIO callback function will soon notice that
        // m_iOutHeadIndex is NOT equal to m_iOutTailIndex ,
        // and increment m_iOutTailIndex when done(!) . See CSound_AsioCallback() .
      }
#    endif //  SWI_ASIO_SUPPORTED

   } // end if( m_OutBufPosition >= m_OutBufferSize ) -> "ready to flush"

  return iWaited_ms;  // the return value may be NEGATIVE when non-blocking, and all buffers are in use !
} // end CSound::OutFlushBuffer()


//---------------------------------------------------------------------
void FloatTo24BitInteger( // converts floating point into 24-bit integer
        T_Float flt,    // [in] -8388607.0 ... +-8388607.0 (floating point, already scaled for 24-bit signed integer)
        BYTE *pbDest )  // [out] three-byte-buffer
{
  union uBLONG   // byte-adressable 32-bit-integer
   { BYTE  b[4];
     long  i32;
   } blTemp;

  if(flt>8388607.0 )   // max 2^23 - 1
     flt=8388607.0;
  if(flt<-8388607.0 )   // min - (2^23 - 1)
     flt=-8388607.0;
  blTemp.i32 = (long)flt;
  pbDest[0]  = blTemp.b[0];   // bits  7..0  of the 24-bit integer
  pbDest[1]  = blTemp.b[1];   // bits 15..8  of the 24-bit integer
  pbDest[2]  = blTemp.b[2];   // bits 23..16  of the 24-bit integer
} // end FloatTo24BitInteger()


//---------------------------------------------------------------------
int CSound::OutWriteInt16(  // OUTDATED !  Not used by Spectrum Lab but the "WOLF GUI"..
      SHORT *pi16Data,      // [in] points to 1st sample or SAMPLE PAIR
      int Length,           // [in] number of SAMPLING POINTS or PAIRS(!) to write
      int iTimeout_ms,      // [in] max timeout in milliseconds, 0=non-blocking
      int *piHaveWaited_ms) // [out,optional] "have been waiting here for XX milliseconds"
  // Writes 'Length' audio samples of type 'SHORT' = 16-bit integer
  // to the soundcard output.
  // If the soundcard runs in stereo mode,
  //     [Length] PAIRS(!!) are read in the caller's buffer,
  //         pi16Data[0]=left channel,  pi16Data[1]=right channel,
  //      .. pi16Data[2*Length-2]=left, pi16Data[2*Length-1]=right channel .
  //
  // Returns:
  //  'Length' if data was succesfully placed in the output buffers.
  //         0 if output has finished( reached the specified sample limit )
  //        -1 if error ( use GetError() to retrieve error )
  // CAUTION: <Length> must be an integer fraction of <dwMinBufSize>
  //          passed to OutOpen() !
{
 int i,result;
 BYTE *pb;
 int  iWaited_ms = 0;

 union uBSHORT{    // byte-adressable 16-bit integer
      BYTE b[2];
      short i16;
      WORD  u16;
   }bsTemp;

  if( piHaveWaited_ms != NULL ) // [out,optional] "have been waiting here for XX milliseconds"
   { *piHaveWaited_ms = 0;
   }

  if( !m_OutputOpen )      // return -1 if output not running.
   {
     SetOutputError( SOUNDOUT_ERR_NOTOPEN );
     return -1;
   }

  DOBINI();
  if( Length == 0 )   // need to flush partially filled buffer and exit
   {
     OutClose();
     DOBINI();
     if(m_ErrorCodeOut == NO_ERRORS)
        return Length;
     else
        return -1;
   }
  else   // here to add new data to soundcard buffer queue
   { DOBINI();

     if( m_OutBufPosition >= m_OutBufferSize )  // flush buffer if already full
      { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
        if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
         { iWaited_ms +=  result;
         }
        else // Negative return value from OutFlushBuffer() -> ERROR !
         { return -1; // all output buffers are still completely filled, cannot add more samples !
         }
        DOBINI();
      }
     pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // local pointer for speed
     i = Length * m_OutFormat.nChannels;
     if(m_OutFormat.wBitsPerSample == 16)
      {
        while(i--)
         {
           bsTemp.i16 = (SHORT)*pi16Data++;
           *pb++ = bsTemp.b[0]; // lsb
           *pb++ = bsTemp.b[1]; // msb
           m_OutBufPosition += 2;
           if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
            { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
              if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
               { iWaited_ms +=  result;
               }
              else // Negative return value from OutFlushBuffer() -> ERROR !
               { return -1; // all output buffers are still completely filled, cannot add more samples !
               }
              DOBINI();
              pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
            }
         }
      }
     else  // not 16 bits per sample
      {    //  (must 8 bit, or what ?)
        while(i--)
         {
           bsTemp.u16 = (WORD)( (*pi16Data++) + 32767 );
           *pb++ = bsTemp.b[1];  // msb only
           ++m_OutBufPosition;
           if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
            { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
              if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
               { iWaited_ms +=  result;
               }
              else // Negative return value from OutFlushBuffer() -> ERROR !
               { return -1; // all output buffers are still completely filled, cannot add more samples !
               }
              DOBINI();
              pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
            }
         }
      } // end else <not 16 bits per sample>

     // arrived here: audio samples have been written into the output buffer..
     m_OutSamplesWritten += Length;  // number of "pairs" (not multiplied with the number of channels in each pair)
     if( piHaveWaited_ms != NULL ) // [out,optional] "have been waiting here for XX milliseconds"
      { *piHaveWaited_ms = iWaited_ms;
      }

     // only if no ENDLESS playing:
     if( (m_OutSampleLimit != 0) && (m_OutSamplesWritten >= m_OutSampleLimit) )
      {  // reached the number of samples to be played. Close output.
         OutClose();
         if(m_ErrorCodeOut == NO_ERRORS )
            return 0;
         else
            return -1;
      }
     return Length;   // return number Samples accepted OK
   }
}  // end OutWriteInt16()

//---------------------------------------------------------------------
int CSound::OutWriteFloat(  // Not used by Spectrum Lab !
        T_Float* pFltSamplePoints, // [in] audio samples, single precision float,
                                  //       value range +/- 1.0 ("normalized") . 
        int iNumberOfSamplePairs, // [in] number of SAMPLE POINTS(!) to write or send
        int iTimeout_ms,          // [in] max timeout in milliseconds, 0=non-blocking
        int *piHaveWaited_ms,     // [out,optional] "have been waiting here for XX milliseconds"
        T_ChunkInfo *pInChunkInfo) // [in,optional] chunk info with scaling info;
                                   //       see c:\cbproj\SoundUtl\ChunkInfo.h
  // Writes 'Length' audio samples (from a floating-point array)
  // to the soundcard output.
  //    parameters:
  //      pLeftData = pointer to block of 'Length' float's to output.
  //      Length   = Number of sample-points ("pairs") to write to the output.
  //         If the soundcard runs in stereo mode,
  //             2 * iNumberOfSamplePairs "single floating point numbers"
  //            are taken from the caller's buffer :
  //         pFltSamplePoints[0]=left channel,  pFltSamplePoints[1]=right channel,
  //      .. pFltSamplePoints[2*Length-2]=left, pFltSamplePoints[2*Length-1]=right channel .
  //      If Length is zero then the sound output is flushed and closed .
  //
  // Returns:
  //  'Length' if data was succesfully placed in the output buffers.
  //         0 if output has finished( reached the specified sample limit )
  //        -1 if error ( use GetError() to retrieve error )
{
 int i,nSingleValues;
 int result;
 BYTE *pb;
 T_Float fltTemp, fltScalingFactor = 1.0;
 int iWaited_ms = 0;


 union uBSHORT  // byte-adressable 16-bit integer
   { BYTE b[2];
     short i16;
     WORD  u16;
   } bsTemp;
 union uBLONG   // byte-adressable 32-bit-integer
   { BYTE  b[4];
     long  i32;
   } blTemp;

   if( pInChunkInfo != NULL )  // if the caller provides the value range, use it
    { if( pInChunkInfo->dblSampleValueRange > 0.0 )
       { fltScalingFactor = 1.0 / pInChunkInfo->dblSampleValueRange;
       }
    }


   if( piHaveWaited_ms != NULL ) // [out,optional] "have been waiting here for XX milliseconds"
    { *piHaveWaited_ms = 0;
    }
   if( !m_OutputOpen )      // return -1 if output not running.
    {
      SetOutputError(  SOUNDOUT_ERR_NOTOPEN  );
      return -1;
    }

   if( iNumberOfSamplePairs <= 0 )   // need to flush partially filled buffer and exit
    {
      OutClose();
      if(m_ErrorCodeOut == NO_ERRORS)
         return 0;
      else
         return -1;
    }
   else   // here to add new data to soundcard buffer queue
    {
     DOBINI();
     pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // local pointer for speed
     nSingleValues = iNumberOfSamplePairs * m_OutFormat.nChannels;
     if(m_OutFormat.wBitsPerSample == 16)
      { fltScalingFactor *= 32767.0; // scale from +/-1.0 (or whatever the caller is using)
                                     //   to +/- 32767 for signed 16-bit integer
       for( i=0; i < nSingleValues; i++ )
        {
            fltTemp = *pFltSamplePoints++ * fltScalingFactor;
            if(fltTemp>32767.0)  fltTemp=32767.0;
            if(fltTemp<-32768.0) fltTemp=-32768.0;
            bsTemp.i16 = (SHORT)fltTemp; // typecast because Dev-C++ complained..
            pb[m_OutBufPosition++] = bsTemp.b[0];
            pb[m_OutBufPosition++] = bsTemp.b[1];
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
             }
        } // end for <all 16-bit output samples>
      }
     else  // not 16 bits per sample ?! ->
     if(m_OutFormat.wBitsPerSample == 8)  // write output as 8-bit samples (0..255 for historic reasons)
      { fltScalingFactor *= 127.0; // scale from +/-1.0 (or whatever the caller is using)
        //   to 0..255 for UNSIGNED 8-bit integer (don't ask why.. "historic")
        for( i=0; i < nSingleValues; i++ )
         {
            fltTemp = *pFltSamplePoints++ * fltScalingFactor + 127.0;
            if(fltTemp>255.0)  fltTemp=255.0;
            if(fltTemp<0.0)    fltTemp=0.0;
            pb[m_OutBufPosition++] = (BYTE)(fltTemp);  // convert to 8-bit unsigned int
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;
             }
         }
      } // end if < 8 bits per sample>
     else
     if(m_OutFormat.wBitsPerSample == 24)
      {
        // 24 bits per sample.
        // Note that the input samples are FLOATING POINT NUMBERS,
        //  formerly scaled to +/- 32k for historic reasons,
        //  but (since 2022-10-16), default range is +/-1.0 ("normalized").
        fltScalingFactor *= 8388608.0;  // multiply float by 2^23 before converting to int

        blTemp.i32 = 0;
        for( i=0; i < nSingleValues; i++ )
         {
            fltTemp = *pFltSamplePoints++ * fltScalingFactor;
            if(fltTemp>8388607.0)  fltTemp=8388607.0;  // limit to 2^23 - 1
            if(fltTemp<-8388607.0) fltTemp=-8388607.0;
            blTemp.i32 = (long)(fltTemp);
            pb[m_OutBufPosition++] = blTemp.b[0];
            pb[m_OutBufPosition++] = blTemp.b[1];
            pb[m_OutBufPosition++] = blTemp.b[2];
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
             }
         } // end for <all 24-bit output samples>
      } // end if < 24 bits per sample >
     else
     if(m_OutFormat.wBitsPerSample == 32)
      {
        // 32 bits per sample.
        // Note that the input samples are FLOATING POINT NUMBERS,
        //  and (since 2022-10-16) per default, their range is +/- 1.0 ("normalized").
        fltScalingFactor *= (32768.0*65536.0);  // multiply float by 2^31 before converting to int
        blTemp.i32 = 0;
        pb += m_OutBufPosition;  // added 2007-07-18
        for( i=0; i < nSingleValues; i++ )
          {
            fltTemp = *pFltSamplePoints++;
            if(fltTemp>(32768.0*65536.0-1.0))  fltTemp=(32768.0*65536.0-1.0);
            if(fltTemp<-(32768.0*65536.0-1.0)) fltTemp=-(32768.0*65536.0-1.0);
            *((long*)pb) = (long)( fltTemp );  // -> +/- 2^31
            pb+=4;  m_OutBufPosition+=4;
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;
             }
          } // end for <all 32-bit output samples>
      } // end if < 32 bits per sample >

     DOBINI();
     m_OutSamplesWritten += iNumberOfSamplePairs; // Length; ?
     if( piHaveWaited_ms != NULL ) // [out,optional] "have been waiting here for XX milliseconds"
      { *piHaveWaited_ms = iWaited_ms;
      }

     // only if no ENDLESS playing:
     if( (m_OutSampleLimit != 0) && (m_OutSamplesWritten >= m_OutSampleLimit) )
      { // reached the number of samples to be played. Close output.
        OutClose();
        DOBINI();
        if(m_ErrorCodeOut == NO_ERRORS )
           return 0;
        else
           return -1;
      }
     DOBINI();
     return iNumberOfSamplePairs;   // return number of sample-pairs accepted
   } // end if( iNumberOfSamplePairs > 0 )

}  // end OutWriteFloat()



//---------------------------------------------------------------------
int CSound::OutWriteStereo(  // Sends audio samples to the OUTPUT. Used by SpecLab.
        T_Float *pLeftData,   // [in] audio samples, 32-bit FLOATING POINT, range +/- 1.0, left channel or "I"-component
        T_Float *pRightData,  // [in] audio samples, right channel or "Q"-component
        int iNumberOfSamples, // [in] number of samples in each of the above blocks
        int iTimeout_ms,      // [in] max timeout in milliseconds, 0=non-blocking
        int *piHaveWaited_ms, // [out,optional] "have been waiting here for XX milliseconds"
      T_ChunkInfo *pInChunkInfo) // [in,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h
  // Used by Spectrum Lab.  Will be outdated in future versions with MORE THAN TWO CHANNELS !
  // Writes 'Length' audio samples (from one or two floating-point arrays)
  // to the soundcard output.   This is the only wave-output function called from SpecLab.
  //    parameters:
  //      pLeftData, pRightData = pointer to block of 'Length' float's to output.
  //      iNumberOfSamples   = Number of samples to write from each pXXData-Block.
  //               If is zero then the sound output is flushed and closed.
  //      iTimeout_ms = maximum number of milliseconds to 'wait' here.
  //                    0 for 'non-blocking' (which is used by Spectrum Lab
  //                    if the audio-processing thread already waited
  //                    when READING samples from a soundcard.
  //                    See  C:\cbproj\SpecLab\SoundThd.cpp  .
  // Returns:
  //  'iNumberOfSamples' if data was succesfully placed in the output buffers.
  //         0 if output has finished( reached the specified sample limit )
  //        -1 if error ( use GetError() to retrieve error )
  //
  // Caller (in Spectrum Lab) :  TSoundThread::Execute() -> CSound::OutWriteStereo() .
  //
{
 int i, result, nChannelsPerSample;
 BYTE *pb;
 T_Float fltTemp, fltFactor, *pfltInterleaveBuffer, *pfltSource;
 int iWaited_ms = 0;


 union uBSHORT  // byte-adressable 16-bit integer
   { BYTE b[2];
     short i16;
     WORD  u16;
   } bsTemp;
 union uBLONG   // byte-adressable 32-bit-integer
   { BYTE  b[4];
     long  i32;
   } blTemp;

   if( piHaveWaited_ms != NULL ) // [out,optional] "have been waiting here for XX milliseconds"
    { *piHaveWaited_ms = 0; // set this to ZERO if we didn't wait (further below)
    }
   if( !m_OutputOpen )      // return -1 if output not running.
    {
      m_ErrorCodeOut = SOUNDOUT_ERR_NOTOPEN;
      return -1;
    }

   if( iNumberOfSamples == 0 )   // need to flush partially filled buffer and exit
    {
      OutClose();
      if(m_ErrorCodeOut == NO_ERRORS)
         return iNumberOfSamples;
      else
         return -1;
    }
   else   // here to add new data to soundcard buffer queue
    {
#  if( SWI_AUDIO_IO_SUPPORTED )
      if( aio_Out.h_AudioIoDLL != NULL )
       { // send the audio stream to an AUDIO-I/O-LIBRARY (DLL) :
         pfltSource = pLeftData;
         nChannelsPerSample = 1;
         // Because AIO_Host_WriteOutputSamplePoints() expects
         // interleaved channels (only ONE source block, with N channels per point)
         // the separated (non-interleaved) source channels may have to be
         // interleaved here:
         if( pRightData != NULL ) // multiple input channels in separate blocks ?
          { if( (pfltInterleaveBuffer = ResizeInterleaveBuffer( iNumberOfSamples )) != NULL )
             { Sound_InterleaveTwoBlocks( pLeftData, pRightData, iNumberOfSamples, pfltInterleaveBuffer );
               pfltSource = pfltInterleaveBuffer;
               nChannelsPerSample = 2;
             }
          }
#       if( SWI_FLOAT_PRECISION == 1 )  // single or double precision ?
         i = AIO_Host_WriteOutputSamplePoints(  // -> C:\cbproj\AudioIO\AudioIO.c
#       else                            // similar for double precision:
         i = AIO_Host_WriteOutputSamplePoints_Double(
#       endif
              &aio_Out,          // [in] DLL-host instance data
              pfltSource,        // [in] audio samples, FLOATING POINT, interleaved as "sample points"
              iNumberOfSamples,  // [in] number of SAMPLE POINTS(!) to write or send
              nChannelsPerSample,// [in] number of samples PER SAMPLE POINT
              iTimeout_ms ,      // [in] max timeout in milliseconds, 0=non-blocking
              pInChunkInfo,      // [in,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL !
              piHaveWaited_ms);  // [out,optional] "have been waiting here for XX milliseconds"
         if( i>0 )   // SUCCESS (and the number of samples still unused in the recipient's buffer)
          {
            return iNumberOfSamples;  // return number Samples successfully accepted
          }
         else        // i<0 : AIO_Host_WriteOutputSamplePoints() failed,
          {          //       and 'i' is actually one of the error codes defined in AudioIO.h
            strncpy( m_sz255LastErrorStringOut, AIO_Host_ErrorCodeToString( &aio_Out, i ), 80 );
            SetOutputError( Sound_AudioIOErrorToMyErrorCode( i ) );
            return -1;
          }
       }
#  endif // SWI_AUDIO_IO_SUPPORTED ?
     // Arrived here: use the standard 'multimedia API' for audio output .
     if( (m_pbOutputBuffer==0) || (m_NumOutputBuffers<=0) || (!m_OutputOpen) )
      { return 0;   // something very wrong; bail out !
      }

     // If the caller does NOT want to be blocked, make sure the output-buffer
     // is empty enough to accept the caller's number of samples (to be written):
     if( iTimeout_ms <= 0 )  // Caller does NOT want to be blocked at all :
      { // Bail out (with an error) if the output-buffer is completely full:
        if( ( (m_iOutHeadIndex+1) % m_NumOutputBuffers) == m_iOutTailIndex )
         {  // ALL of the output buffers are completely filled ->
            return 0; // bail out (caller needs to be blocked but he doesn't want to..)
            // 2013-02-09 : Occasionally got here, when the sampling rate
            //              was deliberately increased 'too much' for the output.
            //       But dropping a few output chunks is much better (faster)
            //       then periodically closing and re-opening the audio device !
         }
      } // end if( iTimeout_ms <= 0 )
     pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // local pointer for speed
     if(m_OutFormat.wBitsPerSample == 16)
      {
        fltFactor = 32767.0;  // scaling factor IF the input range was +/- 1.0
        if( pInChunkInfo != NULL )
         { if( pInChunkInfo->dblSampleValueRange > 0.0 )
            { // Scale from the caller's value range to +/-32767 for 16-bit.
              // The T_ChunkInfo struct from c:\cbproj\SoundUtl\ChunkInfo.h
              // contains the scaling range :
              fltFactor = 32767.0 / pInChunkInfo->dblSampleValueRange;
            }
         }

       if(m_OutFormat.nChannels==1) // 16 bit, Mono output
        {
         for( i=0; i < iNumberOfSamples; i++ )
          {
            fltTemp = pLeftData[i] * fltFactor;
                 // 2009-08-06: access violation HERE,
                 // after switching the output device from the "Audio-I/O DLL"
                 // to a normal soundcard.
            if(fltTemp>32767.0)  fltTemp=32767.0;
            if(fltTemp<-32768.0) fltTemp=-32768.0;
            bsTemp.i16 = (SHORT)fltTemp; // typecast because Dev-C++ complained..
            pb[m_OutBufPosition++] = bsTemp.b[0];
            pb[m_OutBufPosition++] = bsTemp.b[1];
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
             }
          } // end for <all 16-bit MONO output samples>
        }
       else
       if(m_OutFormat.nChannels==2) // 16 bit, Stereo output
        {
         for( i=0; i < iNumberOfSamples; i++ )
          {
            fltTemp = pLeftData[i] * fltFactor;
            // 2012-03-26 : Crashed here when changing audio params on-the-fly ?
            // 2012-11-18 : Crashed here when switching the configuration
            //              while acquiring samples from an SDR-IQ .
            if(fltTemp>32767.0)  fltTemp=32767.0;
            if(fltTemp<-32768.0) fltTemp=-32768.0;
            bsTemp.i16 = (SHORT)fltTemp;
            pb[m_OutBufPosition++] = bsTemp.b[0];
            pb[m_OutBufPosition++] = bsTemp.b[1];
            if( pRightData )
             { fltTemp = pRightData[i] * fltFactor;
               if(fltTemp>32767.0)  fltTemp=32767.0;
               if(fltTemp<-32768.0) fltTemp=-32768.0;
             }
            bsTemp.i16 = (SHORT)fltTemp;
            pb[m_OutBufPosition++] = bsTemp.b[0];
            pb[m_OutBufPosition++] = bsTemp.b[1];
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() ->
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
             }
          } // end for <all 16-bit STEREO output samples>
         DOBINI();
        }
      }
     else  // not 16 bits per sample
     if(m_OutFormat.wBitsPerSample == 8)  // write output as 8-bit samples (0..255 for historic reasons)
      { // Must convert/scale to 0..255 for UNSIGNED 8-bit integer (don't ask why.. "historic")
        if( pInChunkInfo->dblSampleValueRange > 0.0 )
         { // The T_ChunkInfo struct from c:\cbproj\SoundUtl\ChunkInfo.h
           // contains the scaling range :
           fltFactor = 127.0 / pInChunkInfo->dblSampleValueRange;
         }
        else
         { fltFactor = 127.0; // scale from +/-1.0 (or whatever the caller is using)
         }

       if(m_OutFormat.nChannels==1) // 8 bit, Mono output
        {
         for( i=0; i < iNumberOfSamples; i++ )
           {
            fltTemp = pLeftData[i] * fltFactor + 127.0;
            if(fltTemp>255.0)  fltTemp=255.0;
            if(fltTemp<0.0)    fltTemp=0.0;
            pb[m_OutBufPosition++] = (BYTE)(fltTemp);
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;
             }
          }
        }
       else
       if(m_OutFormat.nChannels==2) // 8 bit, Stereo output
        {
         for( i=0; i < iNumberOfSamples; i++ )
           {
            fltTemp = pLeftData[i] * fltFactor + 127.0;
            if(fltTemp>255.0)  fltTemp=255.0;
            if(fltTemp<0.0)    fltTemp=0.0;
            pb[m_OutBufPosition++] = (BYTE)(fltTemp);

            if( pRightData )
             { fltTemp = pRightData[i] * fltFactor + 127.0;
               if(fltTemp>255.0)  fltTemp=255.0;
               if(fltTemp<0.0)    fltTemp=0.0;
             }
            pb[m_OutBufPosition++] = (BYTE)(fltTemp);
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
             }
          }
        }
      } // end if < 8 bits per sample>
     else
     if(m_OutFormat.wBitsPerSample == 24)
      {
        // 24 bits per sample. Modified 2016-10-18, because 24bit/sample didn't work for OUTPUT to an E-MU 0202 !
        //  The input samples are FLOATING POINT NUMBERS, typically ranging from -1.0 to +1.0,
        //  but the caller may use "anything he wants" as long as he tells us in the "chunk info" .
        fltFactor = 8388607.0;  // <- scaling factor for 24 bit (signed integer : +/- (2^23 - 1) ) IF the input was scaled from -1 to +1
        if( pInChunkInfo != NULL )
         { if( pInChunkInfo->dblSampleValueRange > 0.0 )
            { fltFactor /= pInChunkInfo->dblSampleValueRange;
            }
         }

        blTemp.i32 = 0;
        if(m_OutFormat.nChannels==1) // 24 bit, Mono output
         {
           if( (m_OutBufferSize % 3) != 0 ) // unsuited buffer size (sample would cross boundary in the loop below)
            { return -1;
            }
           for( i=0; i < iNumberOfSamples; i++ )
            {
              FloatTo24BitInteger( pLeftData[i] * fltFactor, &pb[m_OutBufPosition] );
              m_OutBufPosition += 3;  // 3 bytes per sample in a single channel
              if( m_OutBufPosition >= m_OutBufferSize ) // send it if full
               { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
                 if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                  { iWaited_ms +=  result;
                  }
                 else // Negative return value from OutFlushBuffer() -> ERROR !
                  { return -1; // all output buffers are still completely filled, cannot add more samples !
                  }
                 DOBINI();

                 pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
               }
            } // end for <all 24-bit MONO output samples>
         }
        else
        if(m_OutFormat.nChannels==2) // 24 bit, Stereo output
         {
           if( (m_OutBufferSize % 6) != 0 ) // unsuited buffer size (sample would cross boundary in the loop below)
            { return -1;
            }

           for( i=0; i < iNumberOfSamples; i++ )
            { if( pLeftData )  // caller provides data for the 1st channel ->
               { fltTemp = pLeftData[i] * fltFactor;
               }
              else             // fill the unused output channel with silence
               { fltTemp = 0.0;
               }
              FloatTo24BitInteger( fltTemp, &pb[m_OutBufPosition] ); // emit 24 bit integer for "left" channel
              m_OutBufPosition += 3;  // 3 bytes per sample in a single channel

              if( pRightData ) // caller provides data for the 2nd channel ->
               { fltTemp =  pRightData[i] * fltFactor;
               }
              else             // fill the unused output channel with silence
               { fltTemp = 0.0;
               }
              FloatTo24BitInteger( fltTemp, &pb[m_OutBufPosition] ); // emit 24 bit integer for "right" channel
              m_OutBufPosition += 3;  // 3 bytes per sample in a single channel

              if( m_OutBufPosition >= m_OutBufferSize )  // send buffer if full
               { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
                 if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                  { iWaited_ms +=  result;
                  }
                 else // Negative return value from OutFlushBuffer() -> ERROR !
                  { return -1; // all output buffers are still completely filled, cannot add more samples !
                  }
                 DOBINI();
                 pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
              }
           } // end for <all 24-bit STEREO output samples>
        }
      } // end if < 24 bits per sample >
     else
     if(m_OutFormat.wBitsPerSample == 32)
      {
       // 32 bits per sample.
       // Note that the input samples are FLOATING POINT NUMBERS,
       //  but may still be scaled to +/- 32k for historic reasons !
       fltFactor = 1.0;  // scaling factor IF the input was scaled from -32767..+32767
       if( pInChunkInfo != NULL )
        { if( pInChunkInfo->dblSampleValueRange > 0.0 )
           { fltFactor = 32767.0 / pInChunkInfo->dblSampleValueRange;
           }
        }

       blTemp.i32 = 0;
       if(m_OutFormat.nChannels==1) // 24 bit, Mono output
        { pb += m_OutBufPosition;  // added 2007-07-18
         for( i=0; i < iNumberOfSamples; i++ )
          {
            fltTemp = pLeftData[i] * fltFactor;
            if(fltTemp>32767.0)  fltTemp=32767.0;
            if(fltTemp<-32768.0) fltTemp=-32768.0;
            *((long*)pb) = (long)( fltTemp * 65536.0 );  // -> +/- 2^31
            pb+=4;  m_OutBufPosition+=4;
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;
             }
          } // end for <all 32-bit MONO output samples>
        }
       else
       if(m_OutFormat.nChannels==2) // 32 bit, Stereo output
        { pb += m_OutBufPosition;  // added 2007-07-18
         for( i=0; i < iNumberOfSamples; i++ )
          {
            fltTemp = pLeftData[i] * fltFactor;
            if(fltTemp>32767.0)  fltTemp=32767.0;
            if(fltTemp<-32768.0) fltTemp=-32768.0;
            *((long*)pb) = (long)( fltTemp * 65536.0 );  // -> +/- 2^31
            pb+=4;  m_OutBufPosition+=4;

            if( pRightData )
             { fltTemp = pRightData[i] * fltFactor;
               if(fltTemp>32767.0)  fltTemp=32767.0;
               if(fltTemp<-32768.0) fltTemp=-32768.0;
             }
            *((long*)pb) = (long)( fltTemp * 65536.0 );
            pb+=4;  m_OutBufPosition+=4;
            if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
             { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
               if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                { iWaited_ms +=  result;
                }
               else // Negative return value from OutFlushBuffer() -> ERROR !
                { return -1; // all output buffers are still completely filled, cannot add more samples !
                }
               DOBINI();
               pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;
             }
          } // end for <all 32-bit STEREO output samples>
        }
      } // end if < 32 bits per sample >

     DOBINI();
     m_OutSamplesWritten += iNumberOfSamples;
     if( piHaveWaited_ms != NULL ) // [out,optional] "have been waiting here for XX milliseconds"
      { *piHaveWaited_ms = iWaited_ms;
      }

     // only if no ENDLESS playing:
     if( (m_OutSampleLimit != 0) && (m_OutSamplesWritten >= m_OutSampleLimit) )
      { // reached the number of samples to be played. Close output.
        OutClose();
        DOBINI();
        if(m_ErrorCodeOut == NO_ERRORS )
           return 0;
        else
           return -1;
      }
     DOBINI();
     return iNumberOfSamples;   // return number Samples accepted OK
   }
}  // end OutWriteStereo()



//---------------------------------------------------------------------
int CSound::WriteOutputSamplePoints(  // NOT used by Spectrum Lab ! !
        float *pfltSource,        // [in] audio samples, as 32-bit FLOATING POINT numbers,
                                  //  grouped as "sample points" (N channels PER POINT),
                                  //  value range normalized to +/- 1.0 for full ADC range
        int iNumSamplePoints,     // [in] number of SAMPLE POINTS(!) to write or send
        int nChannelsPerSample,   // [in] number of samples PER SAMPLE POINT
        int iTimeout_ms,          // [in] max timeout in milliseconds, 0=non-blocking
        int *piHaveWaited_ms,     // [out,optional] "have been waiting here for XX milliseconds"
        T_ChunkInfo *pInChunkInfo) // [in,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL !
  // Return : >= 0 on success (the number indicates HOW MANY SAMPLES can be
  //                           placed in the buffer if we'd immediately
  //                           call WriteOutputSamplePoints() again ) .
  //          < 0  : one of the NEGATIVE error codes defined in AudioIO.H .
  // Unlike the older (and now discarded) 'audio output' methods, this one
  //  supports the 'T_ChunkInfo' struct defined in c:\cbproj\SoundUtl\ChunkInfo.h,
  //  including timestamp, calibrated sample rate, GPS date+time+position, etc.
  //  It also supports multi-channel audio (grouped with MULTIPLE CHANNELS per "sample point") .
{
 int i,result,chn;
 BYTE *pb;
 float fltFactor,fltTemp;
 long  i32Result = 0;
 int   iWaited_ms = 0;


 union uBSHORT  // byte-adressable 16-bit integer
   { BYTE b[2];
     short i16;
     WORD  u16;
   } bsTemp;
 union uBLONG   // byte-adressable 32-bit-integer
   { BYTE  b[4];
     long  i32;
   } blTemp;

   if(piHaveWaited_ms != NULL)
     *piHaveWaited_ms = 0;
   if( !m_OutputOpen )      // return -1 if output not running.
    {
      SetOutputError(  SOUNDOUT_ERR_NOTOPEN  );
      return -1;
    }

   blTemp.i32 = 0;
   if( iNumSamplePoints == 0 )   // need to flush partially filled buffer and exit
    {
      OutClose();
      if(m_ErrorCodeOut == NO_ERRORS)
         return 0;
      else
         return -1;
    }
#if( SWI_AUDIO_IO_SUPPORTED )
   if( aio_Out.h_AudioIoDLL != NULL )
    { // send the audio stream to an AUDIO-I/O-LIBRARY (DLL) :
      return AIO_Host_WriteOutputSamplePoints(
              &aio_Out,          // [in] DLL-host instance data
              pfltSource,        // [in] audio samples, as 32-bit FLOATING POINT numbers, grouped as "sample points"
              iNumSamplePoints,  // [in] number of SAMPLE POINTS(!) to write or send
              nChannelsPerSample,// [in] number of samples PER SAMPLE POINT
              iTimeout_ms ,      // [in] max timeout in milliseconds, 0=non-blocking
              pInChunkInfo,      // [in,optional] chunk info, see c:\cbproj\SoundUtl\ChunkInfo.h, MAY BE NULL !
              piHaveWaited_ms);  // [out,optional] "have been waiting here for XX milliseconds"
    }
#endif // SWI_AUDIO_IO_SUPPORTED
   else   // here to add new data to *SOUNDCARD* buffer queue
    {
     pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // local pointer for speed
     if(m_OutFormat.wBitsPerSample == 16)
      {
        fltFactor = 32767.0;  // scaling factor IF the input was normalized ( +/- 1.0 )
        if( pInChunkInfo != NULL )
         { if( pInChunkInfo->dblSampleValueRange > 0.0 )
            { // Originally the sample value range was +/-32767.0 ,
              // but since 2011-07, the preferred(!) range is +/- 1.0 ,
              // i.e. the sample blocks use floating point values normalized to -1.0 ... +1.0 .
              // Caution, some older modules may still use +/- 32k value range !
              // The T_ChunkInfo struct from c:\cbproj\SoundUtl\ChunkInfo.h
              // contains the scaling range :
              fltFactor /= pInChunkInfo->dblSampleValueRange;
            }
         }
        for( i=0; i < iNumSamplePoints; i++ )
         {
           for(chn=0; chn<m_OutFormat.nChannels; ++chn)
            { if( chn<nChannelsPerSample )
               { fltTemp = pfltSource[chn];
               }
              else
               { fltTemp = 0.0;  // put "silence" into all unused channels
               }
              fltTemp *= fltFactor;        // scale input to 16-bit signed integer
              if( fltTemp < -32767.0 )
               {  fltTemp = -32767.0;
               }
              if( fltTemp >  32767.0 )
               {  fltTemp =  32767.0;
               }
              bsTemp.i16 = (SHORT)fltTemp; // typecast because Dev-C++ complained..
              pb[m_OutBufPosition++] = bsTemp.b[0];
              pb[m_OutBufPosition++] = bsTemp.b[1];
              if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
               { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
                 if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                  { iWaited_ms +=  result;
                  }
                 else // Negative return value from OutFlushBuffer() -> ERROR !
                  { return -1; // all output buffers are still completely filled, cannot add more samples !
                  }
                 DOBINI();
                 pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
               }
              pfltSource += nChannelsPerSample;
            } // end for < all CHANNELS within the current sample point >
         } // end for <all 16-bit output samples>
      }
     else  // not 16 bits per sample
     if(m_OutFormat.wBitsPerSample == 8)
      {
        fltFactor = 127.0;  // scaling factor IF the input was normalized ( +/- 1.0 )
        if( pInChunkInfo != NULL )
         { if( pInChunkInfo->dblSampleValueRange > 0.0 )
            { fltFactor /= pInChunkInfo->dblSampleValueRange;
            }
         }
        for( i=0; i < iNumSamplePoints; i++ )
         {
           for(chn=0; chn<m_OutFormat.nChannels; ++chn)
            { if( chn<nChannelsPerSample )
               { fltTemp = pfltSource[chn];
               }
              else
               { fltTemp = 0.0;  // put "silence" into all unused channels
               }
              fltTemp = 127.0 + (fltTemp*fltFactor);  // scale from +/- X  to  8-bit UNsigned integer (0..255)
              pb[m_OutBufPosition++] = (BYTE)(int)fltTemp;
              if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
               { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
                 if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                  { iWaited_ms +=  result;
                  }
                 else // Negative return value from OutFlushBuffer() -> ERROR !
                  { return -1; // all output buffers are still completely filled, cannot add more samples !
                  }
                 DOBINI();
                 pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;
               }
              pfltSource += nChannelsPerSample;
            } // end for < all CHANNELS within the current sample point >
         } // end for <all 8-bit output samples>
      } // end if < 8 bits per sample>
     else
     if(m_OutFormat.wBitsPerSample == 24)
      {
        // 24 bits per sample.
        // Note that the input samples are FLOATING POINT NUMBERS,
        //  in most cases normalized to +/- 1.0  (since 2011-07) !
        fltFactor = 8388607.0; // factor: 2^23 - 1   IF the input was normalized
        if( pInChunkInfo != NULL )
         { if( pInChunkInfo->dblSampleValueRange > 0.0 )
            { fltFactor /= pInChunkInfo->dblSampleValueRange;
            }
         }
        for( i=0; i < iNumSamplePoints; i++ )
         {
           for(chn=0; chn<m_OutFormat.nChannels; ++chn)
            { if( chn<nChannelsPerSample )
               { fltTemp = pfltSource[chn];
               }
              else
               { fltTemp = 0.0;  // put "silence" into all unused channels
               }
              blTemp.i32 = (long)( fltTemp * fltFactor );
              pb[m_OutBufPosition++] = blTemp.b[0];
              pb[m_OutBufPosition++] = blTemp.b[1];
              pb[m_OutBufPosition++] = blTemp.b[2];
              if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
               { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
                 if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                  { iWaited_ms +=  result;
                  }
                 else // Negative return value from OutFlushBuffer() -> ERROR !
                  { return -1; // all output buffers are still completely filled, cannot add more samples !
                  }
                 DOBINI();
                 pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;  // new output block, new local pointer !
               }
              pfltSource += nChannelsPerSample;
            } // end for < all CHANNELS within the current sample point >
         } // end for <all 24-bit output samples>
      } // end if < 24 bits per sample >
     else
     if(m_OutFormat.wBitsPerSample == 32)
      {
        // 32 bits per sample.
        // Note that the input samples are FLOATING POINT NUMBERS,
        //  in this case normalized to +/- 1.0  (since 2011-07) !
        // Note that the input samples are FLOATING POINT NUMBERS,
        //  in this case normalized to +/- 1.0  (since 2011-07) !
        for( i=0; i < iNumSamplePoints; i++ )
         {
           for(chn=0; chn<m_OutFormat.nChannels; ++chn)
            { if( chn<nChannelsPerSample )
               { fltTemp = pfltSource[chn];
               }
              else
               { fltTemp = 0.0;  // put "silence" into all unused channels
               }
              if(fltTemp >  1.0) fltTemp = 1.0; // clip the input (just a SAFETY PRECAUTION)
              if(fltTemp < -1.0) fltTemp =-1.0;
              // scale from +/- 1.0  to  32-bit signed integer .
              // Scaling factor is 2147483647 = 2^31 - 1 .
              *((long*)pb) = (long)( fltTemp * 2147483647.0 );  // -> +/- 2^31
              pb+=4;  m_OutBufPosition+=4;
              if( m_OutBufPosition >= m_OutBufferSize )  // send it if full
               { result = OutFlushBuffer(iTimeout_ms);  // (due to arbitrary blocksize, this may happen "anytime" !)
                 if( result>=0 ) // successfully flushed, or empty output buffers available -> fill next buffer
                  { iWaited_ms +=  result;
                  }
                 else // Negative return value from OutFlushBuffer() -> ERROR !
                  { return -1; // all output buffers are still completely filled, cannot add more samples !
                  }
                 DOBINI();
                 pb = m_pbOutputBuffer + m_iOutHeadIndex*m_OutBufferSize;
               }
              pfltSource += nChannelsPerSample;
            } // end for < all CHANNELS within the current sample point >
         } // end for <all 32-bit output samples>
      } // end if < 32 bits per sample >

     DOBINI();
     m_OutSamplesWritten += iNumSamplePoints;  // number of "sample points" (not multiplied with the number of channels in each point)
     if(piHaveWaited_ms != NULL)
       *piHaveWaited_ms = iWaited_ms;


     // only if no ENDLESS playing:
     if( (m_OutSampleLimit != 0) && (m_OutSamplesWritten >= m_OutSampleLimit) )
      { // reached the number of samples to be played. Close output.
        OutClose();
        DOBINI();
        if(m_ErrorCodeOut == NO_ERRORS )
           return 0;
        else
           return -1;
      }
     DOBINI();
   }
  return i32Result;  // returns the remaining 'free' output buffer space,
          // measured in sample points (not "bytes" or other stupidities)
} // end CSound::WriteOutputSamplePoints()


//---------------------------------------------------------------------
void CSound::OutClose()
   //  Closes the Soundcard (or similar) output if open.
{
  int i,j,result;

   m_OutputOpen = FALSE;
   if(m_hwvout != NULL)
    {
     DOBINI();
     waveOutReset(m_hwvout);  // stop and release buffers
        // > This function stops playback on a specified waveform output device
        // > and resets the current position to 0. All pending playback buffers
        // > are marked as done and returned to the application.
     DOBINI();
     for(i=0; i<m_NumOutputBuffers; i++ )
      {
        // only have to UNPREPARE sound output HEADERS if they are static,
        //           but MUST NOT 'free' them here !
        if( m_OutputWaveHdr[i].dwFlags & WHDR_PREPARED )
         { DOBINI();
           result = waveOutUnprepareHeader(m_hwvout, &m_OutputWaveHdr[i],sizeof(WAVEHDR));
           // > This function cleans up the preparation performed by waveOutPrepareHeader.
           // > The function must be called after the device driver is finished with a data block.
           // > This function complements waveOutPrepareHeader.
           // > You must call this function before freeing the buffer.
           // > After passing a buffer to the device driver with the waveOutWrite function,
           // > you must wait until the driver is finished with the buffer
           // > before calling waveOutUnprepareHeader.
           // (Naja, warum einfach wenn's auch KOMPLIZIERT geht.. ? )
           //
           // Similar as for the input, an output header' may also be
           // still in use when getting here (due to multithreading);
           // in that case result may be '33' which means "WAVERR_STILLPLAYING":
           // > WAVERR_STILLPLAYING :
           // >   The buffer pointed to by the pwh parameter is still in the queue.
           // A crude fix (similar as in InClose) :
           j = 10;  // wait up to 10 * 50ms 'til MMSYSTEM finished with this buffer:
           while( (j>0) && result==WAVERR_STILLPLAYING )
            { Sleep(50);
              result = waveOutUnprepareHeader(m_hwvout, &m_OutputWaveHdr[i],sizeof(WAVEHDR));
              --j;
              if( SndThd_fCloseDevicesDuringSleep ) // received WM_POWERBROADCAST / PBT_APMSUSPEND somewhere in the GUI !
               { // Got NO TIME to spend here (shortly before SUSPEND), so
                 // bail out of this "tamed busy-spinning" loop (and many others)
                 // as soon as recognizing this "going-to-sleep"-flag :
                 break;
               }
            }
           if( result==MMSYSERR_NOERROR)
            { // Successfully 'unprepared' the wave-header-thing .
              // At this point, only 'WHDR_DONE' was set in dwFlags .
              // Clear that flag, too; for what it's worth..
              m_OutputWaveHdr[i].dwFlags = 0; // back to the 'initial state' for OutOpen() .
            }
           else
            { if( m_ErrorCodeOut==0)
               { SetOutputError( result );  // -> m_ErrorCodeOut
               }
            }
         }
      }
      DOBINI();
      waveOutClose(m_hwvout);
      m_hwvout = NULL;
    } // end if(m_hwvout != NULL)

#if( SWI_AUDIO_IO_SUPPORTED )
  DOBINI();
  AIO_Host_CloseAudioOutput( &aio_Out );
#endif // SWI_AUDIO_IO_SUPPORTED

  DOBINI();
  if(m_iUsingASIO & 0x0002)    // was using ASIO for output ?
   { m_iUsingASIO &= ~0x0002;  // clear "using-ASIO-for-output"-flag
     if( m_iUsingASIO==0 )     // ASIO no longer in use now ?
      {
#if( SWI_ASIO_SUPPORTED )      // ASIO supported ? (through DL4YHF's wrapper)
        if( m_hAsio != C_ASIOWRAP_ILLEGAL_HANDLE )
         { AsioWrap_Close( m_hAsio );
           m_hAsio = C_ASIOWRAP_ILLEGAL_HANDLE;
         }
#endif // SWI_ASIO_SUPPORTED
      }
   } // end if(m_iUsingASIO & 0x0002)

  DOBINI();
} // end OutClose()


void CSound::Start(void)  // starts input and output devices (must be OPENED)
  // Note: Start() and Stop() seem to be MANDATORY for the ExtIO-DLLs !
{
  DOBINI();

#if ( SWI_AUDIO_IO_SUPPORTED )
  if( m_iInputDeviceID==C_SOUND_DEVICE_AUDIO_IO )  // Audio-I/O or "ExtIO"-DLL ...
   { // Up to 2012-08, there was no extra starting. It's already complicated enough !
     // Input starts when opening the device for input,
     // output start when opening the device for output.  Basta.
     // See AudioIO.c :: AIO_Host_OpenAudioInput()  !
     // But for the ExtIO-DLLs, an extra "start" seems necessary, thus:
     AIO_Host_Start( &aio_In );
     // Note: The 'start' for ExtIO requires the EXTERNAL VFO FREQUENCY (!)
     //       already set, through CSound::SetVFOFrequency() [nnnngrrrr]
     // Because AIO_Host_Start() may take some time (during which a WAVE INPUT BUFFER
     //       did overflow; grep 2012-10-29), the 'audio-I/O' & ExtIO-host
     //       now starts the external device BEFORE starting the input from
     //       the soundcard (especially for FiFi-SDR) .
   }
#endif // SWI_AUDIO_IO_SUPPORTED ?

#if ( ! MOD_2015_11_25 )
  // Modified 2015-11-25 : Too much time elapsed between waveInStart() and the first call of InReadStereo(),
  //                       so waveInStart() is now called from THERE, not from HERE. What a mess.
  if(m_InputOpen && (m_hwvin != NULL) )
   {
     if( ! m_fWaveInStarted )
      { m_fWaveInStarted = TRUE;
        DEBUG_EnterErrorHistory( DEBUG_LEVEL_WARNING,0,UTL_USE_CURRENT_TIME,
           "Calling waveInStart()" );
        waveInStart( m_hwvin );
        // About waveInStart() :
        // > The waveInStart function starts input on the given waveform-audio input device.
        // > Calling this function when input is already started has no effect, and the function returns zero.
        // Despite that, since 2015-11-25, waveInStart() is only called when necessary !
      }
   }
#endif // ! MOD_2015_11_25 ?

#if(0) // removed 2007-11; output only starts if buffer is full enough
  if(m_OutputOpen && (m_hwvout != NULL) && (!m_fOutputHoldoff) )
   {
     m_fOutputHoldoff = FALSE;
     waveOutRestart( m_hwvout );
   }
#endif // REMOVED 2007-11

#if( SWI_ASIO_SUPPORTED ) // ASIO supported ? (through DL4YHF's wrapper)
  if( m_iUsingASIO!=0 )   // using ASIO for in- or/and output :
   { // Start the ASIO driver (for BOTH INPUT AND OUTPUT !)
     AsioWrap_StartOrStop( m_hAsio, TRUE/*fStart*/ );
   }
#endif // SWI_ASIO_SUPPORTED


  DOBINI();
} // end CSound::Start()

void CSound::Stop(void)  // stops input and output (must be OPENED)
  // Note: Start() and Stop() seem to be MANDATORY for the ExtIO-DLLs !
{
#if ( SWI_AUDIO_IO_SUPPORTED )
  if( m_iInputDeviceID==C_SOUND_DEVICE_AUDIO_IO )
   { // Up to 2012-08, there was no extra starting / stopping.
     // But for the ExtIO-DLLs, an extra "stop" seems necessary, thus:
     AIO_Host_Stop( &aio_In );
     // 2012-11-19 : Crashed in AIO_Host_Stop() when called through
     //  SndThd_On50msTimer() -> CSound::Stop() , because the critical section
     //  was still occupied (from another thread), and some goddamned function
     //  in the ExtIO-DLL did never return. This happened over and over again.
   }
#endif // SWI_AUDIO_IO_SUPPORTED ?
} // end CSound::Stop()



//---------------------------------------------------------------------------
void CALLBACK WaveInCallback(
        HWAVEIN m_hwvin,
        UINT uMsg,
        CSound* pCSound,
        DWORD dwParam1,  // is this the same as the "LPARAM"-thingy ?
        DWORD dwParam2 )
  //  Callback Routine called by mmsystem when a buffer becomes full.
  // > Remarks
  // > Applications should not call any system-defined functions from inside
  // > a callback function, except for EnterCriticalSection, LeaveCriticalSection,
  // > midiOutLongMsg, midiOutShortMsg, OutputDebugString, PostMessage,
  // > PostThreadMessage, SetEvent, timeGetSystemTime, timeGetTime,
  // > timeKillEvent, and timeSetEvent.
  // > Calling other wave functions will cause deadlock.
{
  long double ldblUnixDateAndTime;
  WAVEHDR *pWaveHdr;
#define L_TEST_OVERLOAD 0
#if(L_TEST_OVERLOAD)
  static bool already_here = false;
   if(already_here)
      return;        // Set breakpoint HERE ! (should never trap)
   already_here=true;
#endif
  DOBINI_MM_CALLBACK();
  // LOCAL_BREAK(); // 2008-10-19 : Called from KERNEL32.DLL -> WINMM.DLL -> WaveInCallback(),
                    //  while InReadStereo() was waiting in the call after line "809" .

  switch( uMsg )
   {
     case WIM_OPEN:
     case WIM_CLOSE:
        DOBINI_MM_CALLBACK();
        break;           // simply ignore these messages

     case WIM_DATA:
        // > Sent .. when waveform-audio data is present in the input buffer
        // > and the buffer is being returned to the application.
        // > The message can be sent either when the buffer
        // > is full or after the waveInReset function is called.
        // > WIM_DATA Parameters:
        //    WIM_DATA :
        //        dwParam1 = (DWORD) lpwvhdr
        //      ( Pointer to a WAVEHDR structure that identifies the buffer
        //        containing the data )
        //        dwParam2 = reserved
        // > Remarks
        // >   The returned buffer might not be full. Use the dwBytesRecorded
        // >   member of the WAVEHDR structure specified by lParam to determine
        // >   the number of bytes recorded into the returned buffer.
        pWaveHdr = (WAVEHDR*)dwParam1;  // added 2015-07-08 for the sake of EXAMINING the "WAVEHDR":
        pWaveHdr = pWaveHdr;
        // 2015-07-08 : With the non-functional 24-bit-per-sample input from an E-MU 0202,
        //              got here with ...
        Sound_dwDebuggerDummy = (DWORD)pWaveHdr->lpData;   // pointer to locked data buffer
        Sound_dwDebuggerDummy = pWaveHdr->dwBufferLength;  // length of data buffer : 65536
        Sound_dwDebuggerDummy = pWaveHdr->dwBytesRecorded; // used for input only   : 65536
        Sound_dwDebuggerDummy = pWaveHdr->dwUser;          // for client's use      : 0
        Sound_dwDebuggerDummy = pWaveHdr->dwFlags;         // assorted flags : 3 = "DONE" and "PREPARED" (explained below)
        Sound_dwDebuggerDummy = pWaveHdr->dwLoops;         // loop control counter
           // WAVEHDR.dwFlags (defined in mmsystem.h) :
           // #define WHDR_DONE       0x00000001  /* done bit */
           // #define WHDR_PREPARED   0x00000002  /* set if this header has been prepared */
           // #define WHDR_BEGINLOOP  0x00000004  /* loop start block */
           // #define WHDR_ENDLOOP    0x00000008  /* loop end block */
           // #define WHDR_INQUEUE    0x00000010  /* reserved for driver */


        if( (pCSound != NULL) && (pCSound->m_NumInputBuffers>0) )
         {
           ldblUnixDateAndTime = SOUND_GetSystemDateAndTimeAsUnixSeconds(); // <- very unreliable and "jittering" !
              // (but at least, the system-date-and-time doesn't "drift away"
              //                like an audio-sample-counter-based timestamp .. )
           DOBINI_MM_CALLBACK();
#   if( CSOUND_USE_CRITICAL_SECTION )
           EnterCriticalSection(&pCSound->m_CriticalSection);
#   endif
            { DOBINI_MM_CALLBACK();

              pCSound->m_InBufInfo[pCSound->m_iInHeadIndex].ldblUnixDateAndTime = ldblUnixDateAndTime;

              // Pass this "filled" input buffer to the application (here: CSound)
              // through the internal queue of audio input buffers:
              if( ++pCSound->m_iInHeadIndex >= pCSound->m_NumInputBuffers) // inc ptr (here in WaveInCallback)
               {  pCSound->m_iInHeadIndex = 0;   // handle wrap around
               }
              if( pCSound->m_iInHeadIndex == pCSound->m_iInTailIndex )   // looks like an overflow
               {
                 pCSound->m_InOverflow = TRUE; // here in WaveInCallback() [windows mmsystem]
                 ++pCSound->m_LostBuffersIn;
               }
              if(pCSound->m_InWaiting)      // if user thread is waiting for buffer
               {
                 pCSound->m_InWaiting = FALSE;
                 if( pCSound->m_InEventHandle != NULL )
                  { DOBINI_MM_CALLBACK();
                    SetEvent( pCSound->m_InEventHandle); // signal it (here in MMSYSTEM callback)
                  }
               }
            } // end of critical(?) section
#   if( CSOUND_USE_CRITICAL_SECTION )
           LeaveCriticalSection(&pCSound->m_CriticalSection);
#   endif
           DOBINI_MM_CALLBACK();
         }
        else
         {
           DOBINI_MM_CALLBACK();
         }
        break; // end case MM_WIM_DATA
     default:
#     if (SWI_ALLOW_TESTING)
        Sound_iUnknownWaveInCallbackMsg = uMsg;
        DOBINI_MM_CALLBACK();
#     endif
        break;
   } // end switch( uMsg )

#if(L_TEST_OVERLOAD)
  already_here=false;
#endif

} // end WaveInCallback()


//---------------------------------------------------------------------------
void CALLBACK WaveOutCallback( HWAVEOUT m_hwvout, UINT uMsg, CSound* pCSound,
                     DWORD dwParam1, DWORD dwParam2 )
  // Callback function for the windoze 'multimedia' API.
  //    Called by mmsystem when one of the previously 'written' buffers
  //    becomes empty.  For details, see OutFlushBuffer() .
  //    No OS functions except SetEvent may be called from here.
  //
  // Call Tree (usually) :
  //    kernel32.dll -> wdmaud.drv -> .. -> WINMM.DLL -> msacm32.drv (sometimes)
  //        -> WaveOutCallback() .
  // Note: Sometimes, a "phantom breakpoint" fired from WDMAUD.D??  ?!
{
  int buffer_tail_index;
  DOBINI_MM_CALLBACK();
  switch( uMsg )
   {
     case WOM_DONE :  // ( 957 ?) "buffer empty" message
     #if( CSOUND_USE_CRITICAL_SECTION )
        EnterCriticalSection(&pCSound->m_CriticalSection);
     #endif
        buffer_tail_index = pCSound->m_iOutTailIndex + 1;
        if( buffer_tail_index >= pCSound->m_NumOutputBuffers )
         {  buffer_tail_index = 0;
         }
        pCSound->m_iOutTailIndex = buffer_tail_index;   // set the new (incremented) buffer TAIL index, here in WaveOutCallback()
        // Note: The only WRITER for m_iOutTailIndex is here, in the callback,
        //       thus there are no conflicts due to multitasking.
        //       By using a local variable to increment + wrap the index,
        //       the value of pCSound->m_iOutTailIndex is always VALID,
        //       and can be safely read anywhere .
        if(pCSound->m_OutWaiting) // if user thread is waiting for buffer
         {
           pCSound->m_OutWaiting = FALSE;
           DOBINI_MM_CALLBACK();
           SetEvent( pCSound->m_OutEventHandle);
           // 2013-20: what happens with SetEvent when there's nobody really waiting ?
           // > The SetEvent function sets the state of the specified event object to signaled.
           // > The state of an auto-reset event object remains signaled
           // > until a single waiting thread is released, at which time the system
           // > automatically sets the state to nonsignaled. If no threads are waiting,
           // > the event object's state remains signaled.
           //
         }
#      if( CSOUND_USE_CRITICAL_SECTION )
        LeaveCriticalSection(&pCSound->m_CriticalSection);
#      endif
        break; // end case WOM_DONE

     case WOM_OPEN : // ignore
     case WOM_CLOSE: // ignore
        break;
     default:
#      if (SWI_ALLOW_TESTING)
        Sound_iUnknownWaveOutCallbackMsg = uMsg;  // what's this ? set breakpoint here...
        DOBINI_MM_CALLBACK();
#      endif
        break;
   } // end switch uMsg
  DOBINI_MM_CALLBACK();
} // end WaveOutCallback()

WAVEHDR * CSound::GetInputBufHdr(int iBufNrOffset)  // used in ASIO callback
   // iBufNrOffset is the number of buffers added as 'offset'
   // to the current wave-buffer-"HEAD"-index ("m_iInHeadIndex") .
{
  if( m_NumInputBuffers>0 )  // avoid div by zero
   { iBufNrOffset = ( iBufNrOffset + m_iInHeadIndex ) % m_NumInputBuffers;
     return &m_InputWaveHdr[iBufNrOffset];
   }
  else
   { return &m_InputWaveHdr[0];
   }
}

WAVEHDR * CSound::GetOutputBufHdr(int iBufNrOffset)  // used in ASIO callback
   // iBufNrOffset is the number of buffers added as 'offset'
   // to the current wave-buffer-"HEAD"-index ("m_iOutTailIndex") .
{
  if( m_NumOutputBuffers>0 ) // avoid div by zero
   { iBufNrOffset = ( iBufNrOffset + m_iOutTailIndex ) % m_NumOutputBuffers;
     return &m_OutputWaveHdr[iBufNrOffset];
   }
  else
   { return &m_InputWaveHdr[0];
   }

}


//----------------------------------------------------------------------------------
// Callback function to process the data for ASIO driver system (in- AND output)
//   Call stack (example):
//     C:\WINDOWS\system32\emasio.dll -> AsioWrap_BufferSwitchTimeInfo0()
//          -> AsioWrap_BufferSwitchTimeInfo() -> CSound_AsioCallback()  .
#if( SWI_ASIO_SUPPORTED )
void CSound_AsioCallback(
       T_AsioWrap_DriverInfo *pDriverInfo,
            long bufferIndex,  // [in] 'index' from Steinberg's bufferSwitch(),
                               //  but INDEX OF WHAT ? .. such a stupid name.
                               //  Someone called it bufferIndex to make this clear.
                               // It's definitely NOT a SAMPLE-INDEX.
           DWORD dwInstance )
{  // The actual processing callback. Must match 'AsioDataProc' in asiowrapper.h .
   // Beware that this is normally in a seperate thread, hence be sure that you take care
   // about thread synchronization. Not required here thanks to lock-free circular FIFOs.
  union { long i32; WORD w[2]; BYTE b[4]; } blong;  // byte-addressable 'long'. For INTEL, b[0]=LSB
  CSound* pCSound = (CSound*)dwInstance; // pointer to "this" CSound instance
  WAVEHDR *pInBufHdr, *pOutBufHdr;
  BYTE  *pbDst, *pbSrc;
  DWORD dwBytesRecorded;    // .. in CSound's waveIn-like buffer
  DWORD dwBytesPlayed;      // .. in CSound's waveOut-like buffer
  DWORD dwInDstBytesPerSingleSample;    //  (BitsPerSample+7)/8 for Audio-INPUT(destination buffer format)
  DWORD dwInDstBytesPerSamplePoint;     // ((BitsPerSample+7)/8) * NumberOfChannels (!)
  DWORD dwOutSrcBytesPerSingleSample;   //  (BitsPerSample+7)/8 for Audio-OUTPUT(source buffer format)
  DWORD dwOutSrcBytesPerSamplePoint;    // ((BitsPerSample+7)/8) * NumberOfChannels
  DWORD dwInputChannelMask, dwOutputChannelMask;
  int  nInputChannelsRead, nOutputChannelsWritten;
  int  nInBuffersReady = 0;  // number of CSound input buffers finished in this call
  int  nOutBuffersReady= 0;  // number of CSound output buffers finished in this call

  // Buffer size in samples. This is the 3rd parameter passed
  //  to ASIOCreateBuffers(), not necessarily the "preferred" size !
  long buffSize = pDriverInfo->dwUsedBufferSize; // this is the NUMBER OF SAMPLES, not BYTES !
  // Beware: 'buffSize' can be anything; see asiowrapper.c for details .
  //          It has nothing to do with CSound's m_InBufferSize member !
  // In fact, the author's Audigy 2 ZS wished a buffer size of 4800 samples ,
  // while the CSound class always used  a power of two  for convenience .

  if(pCSound == NULL)
   { // oops .. this should not happen !
     DOBINI_ASIO_CALLBACK();  // -> Sound_iAsioCbkSourceCodeLine
     return;
   }

  if(! pCSound->m_iUsingASIO)
   { // oops .. this should also not happen !
     DOBINI_ASIO_CALLBACK();  // -> Sound_iAsioCbkSourceCodeLine
     return;
   }


     DOBINI_ASIO_CALLBACK();  // -> Sound_iAsioCbkSourceCodeLine (?)
     nInputChannelsRead     = 0;  // no input channel read yet
     nOutputChannelsWritten = 0;  // no output channel written yet
     dwInputChannelMask = dwOutputChannelMask = 1;
     dwInDstBytesPerSingleSample= (pCSound->m_InFormat.wBitsPerSample+7)/8;
     dwInDstBytesPerSamplePoint = pCSound->m_InFormat.nChannels * dwInDstBytesPerSingleSample;
     dwOutSrcBytesPerSingleSample= (pCSound->m_OutFormat.wBitsPerSample+7)/8;
     dwOutSrcBytesPerSamplePoint = pCSound->m_OutFormat.nChannels * dwOutSrcBytesPerSingleSample;

     dwBytesRecorded = dwBytesPlayed = 0; // to see if loop 'did nothing'
     pInBufHdr = pOutBufHdr = NULL;

     // Loop for all ASIO-buffers begins here >>>>>>>>>>>>>>>>
     for (int i = 0; i < pDriverInfo->nInputBuffers + pDriverInfo->nOutputBuffers; i++)
      {
        if(   (pDriverInfo->bufferInfos[i].isInput )
           && (pCSound->m_InputOpen ) // <<< added 2011-03-13 for safety
          )
         { // ASIO INPUT (from ASIO to application) ---->>>
           if(   (pCSound->m_NumInputBuffers>0) && ((pCSound->m_iUsingASIO&1)!=0)
              && ((pCSound->m_dwInputSelector & dwInputChannelMask)!=0)
              && (nInputChannelsRead < pCSound->m_InFormat.nChannels )
             )
            { // It's an input (from ASIO to application),
              // and the CSound class wants to "have it" .
              DOBINI_ASIO_CALLBACK(); // -> Sound_iAsioCbkSourceCodeLine
              // Determine the DESTINATION ADDRESS for the new input data :
              nInBuffersReady = 0;   // must be cleared again for EACH ASIO-BUFFER-LOOP !
              pInBufHdr = pCSound->GetInputBufHdr(nInBuffersReady);
              dwBytesRecorded = pInBufHdr->dwBytesRecorded; // number of bytes already in buffer
              pbDst = (BYTE*)pInBufHdr->lpData + dwBytesRecorded
                           + nInputChannelsRead*dwInDstBytesPerSingleSample;
              // Copy (and most likely convert) the data from the ASIO buffer
              // into CSound's buffer .
              // Notes:
              //  - the buffer sizes may be very different (ASIO can be EXTEMELY strange !)
              //  - the DATA TYPES may also be very strange. Here, only the most
              //    "typical" cases (for a windows PC) will be handled !
              //  - We may fill NONE, ONE, or MANY CSound buffers here,
              //    depending on the relationship between buffSize (ASIO, number of SAMPLES)
              //    and pInBufHdr->dwBufferLength (due to MMSYSTEM/WAVEHDR, number of BYTES) !
              //  - The driver tells US which format WE must cope with,
              //    instead of *us* telling the driver what we want from it !
              //  - pDriverInfo->channelInfos[i].type is set in
              //    c:\cbproj\SoundUtl\myasio\asiowrapper.cpp ;
              //    called through CSound::InOpen() -> AsioWrap_Open()
              //      -> AsioWrap_CreateBuffers() -> ASIOGetChannelInfo() .
              //  - Even if the soundcard should have been opened with SIXTEEN
              //      bits per samples (E-MU0202),
              //    got here with pDriverInfo->channelInfos[i].type = 17 = ASIOSTInt24LSB,
              //    and dwInDstBytesPerSamplePoint = 6 (!) .
              //
              switch (pDriverInfo->channelInfos[i].type)
               {
                 case ASIOSTInt16LSB: // 16-bit signed integer, LSB first ("Intel format")
                  { SHORT *pi16 = (SHORT*)pDriverInfo->bufferInfos[i].buffers[bufferIndex]; // SOURCE : 16 bit per sample
                    if( pCSound->m_InFormat.wBitsPerSample==16 )
                     { // no need to convert here,
                       //  it's our "native" 16-bit format so just COPY 16-bit wise:
                       for(int sample=0; sample<buffSize; ++sample)
                        { *((WORD*)pbDst) = *pi16++;  // source & dest: 16 bit, LSB first
                          pbDst += dwInDstBytesPerSamplePoint;
                          dwBytesRecorded += dwInDstBytesPerSamplePoint;
                          if( dwBytesRecorded >= pInBufHdr->dwBufferLength )
                           { // next buffer please.. but don't "emit" this one yet, because
                             // other channels may follow which will be copied into the same
                             // interlaced buffer (so we may have to access it again).
                             // NOT YET: pInBufHdr->dwBytesRecorded = pInBufHdr->dwBufferLength;
                             ++nInBuffersReady;
                             pInBufHdr = pCSound->GetInputBufHdr(nInBuffersReady); // pointer to "next" buffer !
                             dwBytesRecorded = pInBufHdr->dwBytesRecorded = 0;
                             pbDst = (BYTE*)pInBufHdr->lpData + nInputChannelsRead*dwInDstBytesPerSingleSample;
                           }
                        }
                     }
                    else  // pCSound->m_InFormat.wBitsPerSample!=16  ->  must convert...
                     { for(int sample=0; sample<buffSize; ++sample)
                        { blong.i32 = *pi16++;  // source: 16 bit, LSB first ("Intel")
                        }
                     }
                   } break; // end case ASIOSTInt16LSB
                 case ASIOSTInt24LSB: // (code 17) used for 20 bits as well !
                    // Note YHF: This does NOT seem to be the usual type for 24-bit output !!!
                   {// should be the "first" channel if we can trust the spec...
                    BYTE *pb = (BYTE*)pDriverInfo->bufferInfos[i].buffers[bufferIndex]; // SOURCE : 24 bit per sample
                    if( pCSound->m_InFormat.wBitsPerSample==24 )
                     { // no need to convert here,
                       //  it's our "native" 24-bit format so just COPY 24-bit wise:
                       for(int sample=0; sample<buffSize; ++sample)
                        {
                          pbDst[0] = *pb++;  // LSB first
                          pbDst[1] = *pb++;  // mid byte
                          pbDst[2] = *pb++;  // high byte (of 24-bit quantity)
                          pbDst           += dwInDstBytesPerSamplePoint;
                          dwBytesRecorded += dwInDstBytesPerSamplePoint;
                          if( dwBytesRecorded >= pInBufHdr->dwBufferLength )
                           { // next buffer please.. but don't "emit" this one yet, because
                             // other channels may follow which will be copied into the same
                             // interlaced buffer (so we may have to access it again).
                             // (remember, ASIO can be ultimately strange, passing each channel
                             //  in its own stupid buffer) .
                             // Got here with an E-MU 0202, which ALWAYS seemed to deliver
                             // 24 bits per sample (regardless of what the application
                             // was asking for), with buffSize = 9600 [samples per channel],
                             // and pInBufHdr->dwBufferLength = 49152 [bytes per buffer] .
                             // NOT YET: pInBufHdr->dwBytesRecorded = pInBufHdr->dwBufferLength;
                             ++nInBuffersReady;
                             pInBufHdr = pCSound->GetInputBufHdr(nInBuffersReady); // pointer to "next" buffer !
                             dwBytesRecorded = pInBufHdr->dwBytesRecorded = 0; // number of bytes recorded into THE NEW BUFFER
                             pbDst = (BYTE*)pInBufHdr->lpData + nInputChannelsRead*dwInDstBytesPerSingleSample;
                           }
                        }
                     } // end if( pCSound->m_InFormat.wBitsPerSample==24 )
                   } break; // end case ASIOSTInt24LSB
                 case ASIOSTInt32LSB: // (code 18)
                  { // Note: This is what Creative's "updated" AUDIGY driver seemed to use,
                    //       interestingly NOT one of those many 24-bit types .
                    long *pi32 = (long*)pDriverInfo->bufferInfos[i].buffers[bufferIndex];
                    if( pCSound->m_InFormat.wBitsPerSample==32 )
                     { // no need to convert a lot here, it's our "native" 32-bit format
                       for(int sample=0; sample<buffSize; ++sample)
                        { *((long*)pbDst) = *pi32++;  // source & dest: 32 bit, LSB first
                          pbDst += dwInDstBytesPerSamplePoint;
                          dwBytesRecorded += dwInDstBytesPerSamplePoint; // 4 or 8
                          if( dwBytesRecorded >= pInBufHdr->dwBufferLength )
                           { ++nInBuffersReady;   // next destination buffer ...
                             pInBufHdr = pCSound->GetInputBufHdr(nInBuffersReady); // pointer to "next" buffer !
                             dwBytesRecorded = pInBufHdr->dwBytesRecorded = 0;
                             pbDst = (BYTE*)pInBufHdr->lpData + nInputChannelsRead*dwInDstBytesPerSingleSample;
                           }
                        }
                     }
                    else  // must convert...
                    if( pCSound->m_InFormat.wBitsPerSample==16 )  // ... from 32 to 16 bit
                     { for(int sample=0; sample<buffSize; ++sample)
                        { blong.i32 = *pi32++;          // source: 32 bit, LSB first, left-aligned
                          *((WORD*)pbDst) = blong.w[0]; // dest: 16 bit (upper 16 bit from source)
                          pbDst += dwInDstBytesPerSamplePoint;
                          dwBytesRecorded += dwInDstBytesPerSamplePoint;
                          if( dwBytesRecorded >= pInBufHdr->dwBufferLength )
                           { ++nInBuffersReady;   // next destination buffer ...
                             pInBufHdr = pCSound->GetInputBufHdr(nInBuffersReady); // pointer to "next" buffer !
                             dwBytesRecorded = pInBufHdr->dwBytesRecorded = 0;
                             pbDst = (BYTE*)pInBufHdr->lpData + nInputChannelsRead*dwInDstBytesPerSingleSample;
                           }
                        }
                     }
                  } break;
                 case ASIOSTFloat32LSB: // IEEE 754 32 bit float, as found on Intel x86 architecture
                    break;
                 case ASIOSTFloat64LSB: // IEEE 754 64 bit double float, as found on Intel x86 architecture
                    break;

                    // these are used for 32 bit data buffer, with different alignment of the data inside
                    // 32 bit PCI bus systems can more easily used with these .
                    // YHF: NNNNNGRRRRRR... Can we yet some more useless types please ?!?
                 case ASIOSTInt32LSB16: // 32 bit data with 18 bit alignment
                 case ASIOSTInt32LSB18: // 32 bit data with 18 bit alignment
                 case ASIOSTInt32LSB20: // 32 bit data with 20 bit alignment
                 case ASIOSTInt32LSB24: // 32 bit data with 24 bit alignment
                    break;

                 case ASIOSTInt16MSB:
                    break;
                 case ASIOSTInt24MSB: // used for 20 bits as well
                    break;
                 case ASIOSTInt32MSB:
                    break;
                 case ASIOSTFloat32MSB: // IEEE 754 32 bit float, as found on Intel x86 architecture
                    break;
                 case ASIOSTFloat64MSB: // IEEE 754 64 bit double float, as found on Intel x86 architecture
                    break;

                 // these are used for 32 bit data buffer, with different alignment of the data inside
                 // 32 bit PCI bus systems can more easily used with these
                 case ASIOSTInt32MSB16: // 32 bit data with 18 bit alignment
                 case ASIOSTInt32MSB18: // 32 bit data with 18 bit alignment
                 case ASIOSTInt32MSB20: // 32 bit data with 20 bit alignment
                 case ASIOSTInt32MSB24: // 32 bit data with 24 bit alignment
                    break;
               }
              // not yet: pInBufHdr->dwBytesRecorded = dwBytesRecorded; !
              // may have to write into the same interlaced buffer for the next channel,
              // so write 'dwBytesRecorded' back to the CSound buffer AFTER the channel-loop.
              ++nInputChannelsRead;    // count the number of PROCESSED input channels
              DOBINI_ASIO_CALLBACK(); // -> Sound_iAsioCbkSourceCodeLine
            } // end if < want this input chanel >
           else
            { // Application not ready to 'accept' this input buffer ?
              DOBINI_ASIO_CALLBACK(); // -> Sound_iAsioCbkSourceCodeLine
            }
           dwInputChannelMask <<= 1;
         } // end if < valid INPUT >
        else  // cannot accept the input at the moment, for whatever reason
         { DOBINI_ASIO_CALLBACK(); // -> Sound_iAsioCbkSourceCodeLine
         }
        if( (pDriverInfo->bufferInfos[i].isInput == false) // ASIO OUTPUT ---->>>
           && (pCSound->m_InputOpen ) // <<< added 2011-03-13 for safety
          )
         { // OK do processing for the outputs
           if(   (pCSound->m_NumOutputBuffers>0) && ((pCSound->m_iUsingASIO&2)!=0)
              && ((pCSound->m_dwOutputSelector & dwOutputChannelMask)!=0)
              && (nOutputChannelsWritten < pCSound->m_OutFormat.nChannels )
             )
            { // It's an output channel (from the application to ASIO driver ),
              // and the CSound class wants to "fill it" .
              // Determine the DESTINATION ADDRESS for the new input data :
              nOutBuffersReady = 0;   // must be cleared again for EACH ASIO-BUFFER-LOOP !
              pOutBufHdr = pCSound->GetOutputBufHdr(nOutBuffersReady);
              dwBytesPlayed = pOutBufHdr->dwBytesRecorded; // number of bytes already PLAYED (!)
              if( dwBytesPlayed>= pOutBufHdr->dwBufferLength )
               { dwBytesPlayed = 0;   // should never happen, but avoid access violation !
               }
              pbSrc = (BYTE*)pOutBufHdr->lpData + dwBytesPlayed   // source block for output
                    + nOutputChannelsWritten*dwOutSrcBytesPerSingleSample;
              switch (pDriverInfo->channelInfos[i].type)   // how to convert the OUTPUT...
               {
                 case ASIOSTInt16LSB: // into 16-bit signed integer, LSB first ("Intel format")
                  { SHORT *pi16 = (SHORT*)pDriverInfo->bufferInfos[i].buffers[bufferIndex];
                    if( pCSound->m_OutFormat.wBitsPerSample==16 )
                     { // no need to convert a lot here, it's our "native" 16-bit format:
                       for(int sample=0; sample<buffSize; ++sample)
                        {
                          *pi16++ = *((WORD*)pbSrc);
                          pbSrc += dwOutSrcBytesPerSamplePoint;
                          dwBytesPlayed += dwOutSrcBytesPerSamplePoint;
                          if( dwBytesPlayed >= pOutBufHdr->dwBufferLength )
                           { // next buffer please.. but don't increment m_iOutTailIndex now, because
                             // other channels may follow which will be filled from the same
                             // interlaced buffer (so we may have to access it again).
                             ++nOutBuffersReady;
                             pOutBufHdr = pCSound->GetOutputBufHdr(nOutBuffersReady);
                             dwBytesPlayed = pOutBufHdr->dwBytesRecorded = 0;
                             pbSrc = (BYTE*)pOutBufHdr->lpData + nOutputChannelsWritten*dwOutSrcBytesPerSingleSample;
                           }
                        } // end for
                     } // end if < buffer contains 16 bits per single sample too >
                    else
                     {
                     }
                  } break; // end case ASIOSTInt16LSB  ( on  OUTPUT )

                 case ASIOSTInt24LSB: // used for 20 bits as well
                  { // Note: This does NOT seem to be the usual type for 24-bit output
                    // (at least not for Creative's Audigy 2 ZS), but other cards
                    // may use it  - - - so support it too :
                    BYTE *pb = (BYTE*)pDriverInfo->bufferInfos[i].buffers[bufferIndex];
                    if( pCSound->m_OutFormat.wBitsPerSample==24 )
                     { // copy 24-bit-wise :
                       for(int sample=0; sample<buffSize; ++sample)
                        {
                          *pb++ = pbSrc[0];   // bits 23..16 (MSB)
                          *pb++ = pbSrc[1];   // bits 15..8
                          *pb++ = pbSrc[2];   // bits  7..0  (LSB)
                          pbSrc += dwOutSrcBytesPerSamplePoint; // increment by 3 or 6
                          dwBytesPlayed += dwOutSrcBytesPerSamplePoint;
                          if( dwBytesPlayed >= pOutBufHdr->dwBufferLength )
                           { ++nOutBuffersReady;  // next buffer please
                             pOutBufHdr = pCSound->GetOutputBufHdr(nOutBuffersReady);
                             dwBytesPlayed = pOutBufHdr->dwBytesRecorded = 0;
                             pbSrc = (BYTE*)pOutBufHdr->lpData + nOutputChannelsWritten*dwOutSrcBytesPerSingleSample;
                           }
                        } // end for
                     }
                    else // ASIO wants 24 bits per sample for output, but the source is different:
                    if( pCSound->m_OutFormat.wBitsPerSample==16 )
                     { // copy 16-bit samples into 24-bit destination :
                       for(int sample=0; sample<buffSize; ++sample)
                        {
                          *pb++ = pbSrc[0];   // bits 23..16 (MSB)
                          *pb++ = pbSrc[1];   // bits 15..8
                          *pb++ = 0;          // leave bits 7..0 zero
                          pbSrc += dwOutSrcBytesPerSamplePoint; // increment by 2 or 4
                          dwBytesPlayed += dwOutSrcBytesPerSamplePoint;
                          if( dwBytesPlayed >= pOutBufHdr->dwBufferLength )
                           { ++nOutBuffersReady;  // next buffer please
                             pOutBufHdr = pCSound->GetOutputBufHdr(nOutBuffersReady);
                             dwBytesPlayed = pOutBufHdr->dwBytesRecorded = 0;
                             pbSrc = (BYTE*)pOutBufHdr->lpData + nOutputChannelsWritten*dwOutSrcBytesPerSingleSample;
                           }
                        } // end for
                     }
                  } break; // end case ASIOSTInt24LSB  ( on  OUTPUT )

                 case ASIOSTInt32LSB:
                  { // This is what Creative's "updated" driver seems to use,
                    // interestingly they don't use one of those many 24-bit types .
                    if( pCSound->m_OutFormat.wBitsPerSample==32 )
                     { // no need to convert a lot here, it's our "native" 32-bit format:
                       long *pi32 = (long*)pDriverInfo->bufferInfos[i].buffers[bufferIndex];
                       for(int sample=0; sample<buffSize; ++sample)
                        {
                          *pi32++ = *((long*)pbSrc);
                          pbSrc += dwOutSrcBytesPerSamplePoint; // 4 or 8 bytes per "point"
                          dwBytesPlayed += dwOutSrcBytesPerSamplePoint;
                          if( dwBytesPlayed >= pOutBufHdr->dwBufferLength )
                           { ++nOutBuffersReady;  // next buffer please
                             pOutBufHdr = pCSound->GetOutputBufHdr(nOutBuffersReady);
                             dwBytesPlayed = pOutBufHdr->dwBytesRecorded = 0;
                             pbSrc = (BYTE*)pOutBufHdr->lpData + nOutputChannelsWritten*dwOutSrcBytesPerSingleSample;
                           }
                        } // end for
                     } // end if < buffer contains 32 bits per sample too >
                    else // ASIO wants 32 bits per sample for output, but the source is different:
                    if( pCSound->m_OutFormat.wBitsPerSample==16 )
                     { // convert from 16 bits/sample [source]   to 32 bits/sample [destination] for output:
                       BYTE *pb = (BYTE*)pDriverInfo->bufferInfos[i].buffers[bufferIndex];
                       for(int sample=0; sample<buffSize; ++sample)
                        {
                          *pb++ = pbSrc[0]; // bits 31..24 (MSB)
                          *pb++ = pbSrc[1]; // bits 23..16
                          *pb++ = 0;        // leave bits 15..8 zero
                          *pb++ = 0;        // leave bits  7..0 zero
                          pbSrc += dwOutSrcBytesPerSamplePoint; // increment by 2 or 4
                          dwBytesPlayed += dwOutSrcBytesPerSamplePoint;
                          if( dwBytesPlayed >= pOutBufHdr->dwBufferLength )
                           { ++nOutBuffersReady;  // next buffer please
                             pOutBufHdr = pCSound->GetOutputBufHdr(nOutBuffersReady);
                             dwBytesPlayed = pOutBufHdr->dwBytesRecorded = 0;
                             pbSrc = (BYTE*)pOutBufHdr->lpData + nOutputChannelsWritten*dwOutSrcBytesPerSingleSample;
                           }
                        } // end for
                     }
                  } break; // end case ASIOSTInt32LSB  ( on  OUTPUT )

                 case ASIOSTFloat32LSB: // IEEE 754 32 bit float, as found on Intel x86 architecture
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 4);
                    break;
                 case ASIOSTFloat64LSB: // IEEE 754 64 bit double float, as found on Intel x86 architecture
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 8);
                    break;

                    // these are used for 32 bit data buffer, with different alignment of the data inside
                    // 32 bit PCI bus systems can more easily used with these .
                    // YHF: NNNNNGRRRRRR... Is it really necessary to support these ?
                 case ASIOSTInt32LSB16: // 32 bit data with 18 bit alignment
                 case ASIOSTInt32LSB18: // 32 bit data with 18 bit alignment
                 case ASIOSTInt32LSB20: // 32 bit data with 20 bit alignment
                 case ASIOSTInt32LSB24: // 32 bit data with 24 bit alignment
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 4);
                    break;

                 case ASIOSTInt16MSB:
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 2);
                    break;
                 case ASIOSTInt24MSB:  // used for 20 bits as well
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 3);
                    break;
                 case ASIOSTInt32MSB:
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 4);
                    break;
                 case ASIOSTFloat32MSB: // IEEE 754 32 bit float, as found on Intel x86 architecture
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 4);
                    break;
                 case ASIOSTFloat64MSB: // IEEE 754 64 bit double float, as found on Intel x86 architecture
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 8);
                    break;

                 // these are used for 32 bit data buffer, with different alignment of the data inside
                 // 32 bit PCI bus systems can more easily used with these
                 case ASIOSTInt32MSB16: // 32 bit data with 18 bit alignment
                 case ASIOSTInt32MSB18: // 32 bit data with 18 bit alignment
                 case ASIOSTInt32MSB20: // 32 bit data with 20 bit alignment
                 case ASIOSTInt32MSB24: // 32 bit data with 24 bit alignment
                    memset (pDriverInfo->bufferInfos[i].buffers[bufferIndex], 0, buffSize * 4);
                    break;
               }
              ++nOutputChannelsWritten;  // count the number of FILLED output channels
              // not yet: pOutBufHdr->dwBytesRecorded = dwBytesPlayed; !
              // may have to write into the same interlaced buffer for the next channel,
              // so write 'dwBytesRecorded' back to the CSound buffer AFTER the channel-loop.
            } // end if < want to fill this OUTPUT-channel ? >
           else // don't want to fill this output channel ...
            {
              // The question is these "unused output channels" must be
              //  "filled with silence" (like in Steinberg's demo from the ASIO SDK)
              // or if it's ok to leave these channels untouched .
            } // end if < want this input chanel >
           dwOutputChannelMask <<= 1;
         } // end if < valid OUTPUT >
      } // end for < all buffers the ASIO driver wants to exchange with us during this call >

     if(pInBufHdr!=NULL)
      { // AFTER processing the last input channel, set the new number of bytes
        // which have been 'recorded' in CSound's currently used input-buffer.
        //  Note: if nInBuffersReady>0, this is already the "next" buffer.
        pInBufHdr->dwBytesRecorded = dwBytesRecorded;
      }
     // Emit the new filled buffers *AFTER* the ASIO-buffer-loop.
     // Note that pCSound->m_iInHeadIndex was not modified in the above loop !
     if(nInBuffersReady>0)    // note: in rare cases, there may be MORE THAN ONE buffer filled now !
      {
#     if( CSOUND_USE_CRITICAL_SECTION )
       EnterCriticalSection(&pCSound->m_CriticalSection);  // try to live without this, it took quite long !
#     endif       
       // (remember, this ASIO callback may be called 500 times a second.. no time to waste)
       for( int i=0; i<nInBuffersReady; ++i )
        { // mark the next input buffer as "filled" ..
          pInBufHdr = &pCSound->m_InputWaveHdr[pCSound->m_iInHeadIndex];
          pInBufHdr->dwBytesRecorded = pInBufHdr->dwBufferLength;
          if( ++pCSound->m_iInHeadIndex >= pCSound->m_NumInputBuffers) // inc ptr (here in CSound_AsioCallback)
                pCSound->m_iInHeadIndex = 0;   // handle wrap around
          if( pCSound->m_iInHeadIndex == pCSound->m_iInTailIndex )     // chk for overflow
           {
             pCSound->m_InOverflow = TRUE;  // here in CSound_AsioCallback() [for the ASIO driver]
             ++pCSound->m_LostBuffersIn;
             // 2021-10-02: History repeating. Occasionally got here with a Behringer UMC404HD,
             //      with pCSound->m_NumInputBuffers = 16,
             //           pCSound->m_InFormat.wBitsPerSample = 32(!) because
             //           pDriverInfo->channelInfos[0].type = 18 = ASIOSTInt32LSB,
             //           pCSound->m_InFormat.nChannels = 4,
             // 2010-04-22: Occasionally got here at 192 kSamples/second, E-MU 0202,
             //      with pCSound->m_NumInputBuffers = 4 ,
             //           pCSound->m_InFormat.wBitsPerSample=32 ,
             //           pCSound->m_InFormat.nChannels = 1 ,
             //      and  dwBufferLength = 131072 bytes = 32768 samples;
             //      i.e. despite 0.68 seconds of (total) buffer size,
             //      samples got lost .. even though the CPU performance window
             //      in SpecLab, and the task manager, showed a CPU load of ~60 % .
           }
        } // end for
       if(pCSound->m_InWaiting)      // if user thread is waiting for buffer
        {
          pCSound->m_InWaiting = FALSE;
          if( pCSound->m_InEventHandle != NULL )
           { DOBINI_ASIO_CALLBACK();
             SetEvent( pCSound->m_InEventHandle); // signal it (here in ASIO callback)
           }
        }
#     if( CSOUND_USE_CRITICAL_SECTION )
       LeaveCriticalSection(&pCSound->m_CriticalSection);
#     endif       
      } // end if(nInBuffersReady>0)

     if(pOutBufHdr!=NULL)
      { // AFTER sending the last output channel, set the new number of bytes
        // which have been 'played' from CSound's current active output-buffer.
        //   Note: if nOutBuffersReady>0, this is already the "next" buffer.
        // The component 'dwBytesRecorded' is from a WAVEHDR struct....
        // The Win32 programmer's reference only mentions its meaning on
        //  INPUT-buffers :
        // > When the header is used in input, this member specifies how much data
        // > is in the buffer.
        // Here:
        // < The header is used for output; this member specifies how much data is already PLAYED
        // < from the buffer.
        pOutBufHdr->dwBytesRecorded/*"!"*/ = dwBytesPlayed/*!*/;
      }
     // Signal CSound if finished with another output buffer *AFTER* the ASIO-buffer-loop.
     // Note that pCSound->m_iOutTailIndex was not modified in the above loop !
     if(nOutBuffersReady>0)    // note: in rare cases, there may be MORE THAN ONE buffer filled now !
      {
#      if( CSOUND_USE_CRITICAL_SECTION )
        EnterCriticalSection(&pCSound->m_CriticalSection);  // try to save this time ?
#      endif
        for( int i=0; i<nOutBuffersReady; ++i )
         { // mark another output buffer as "sent" ..
           if( ++pCSound->m_iOutTailIndex >= pCSound->m_NumOutputBuffers) // inc ptr, here for ASIO (not MMSYS)
               pCSound->m_iOutTailIndex = 0;   // handle wrap around
           if( pCSound->m_iOutHeadIndex == pCSound->m_iOutTailIndex )
            { // chk for underflow
              pCSound->m_OutUnderflow = TRUE;
              ++pCSound->m_LostBuffersOut;  // lost another OUTPUT block (here in ASCI-callback)
              if( (pCSound->m_ErrorCodeOut == NO_ERRORS )
                ||(pCSound->m_ErrorCodeOut == SOUNDOUT_ERR_LOW_WATER) )
               {  pCSound->m_ErrorCodeOut = SOUNDOUT_ERR_UNDERFLOW;
               }
            }
         } // end for
        if(pCSound->m_OutWaiting) // if user thread is waiting for buffer..
         {
           pCSound->m_OutWaiting = FALSE;
           DOBINI_ASIO_CALLBACK();
           SetEvent( pCSound->m_OutEventHandle); // .. then kick it alive again
         }
#      if( CSOUND_USE_CRITICAL_SECTION )
        LeaveCriticalSection(&pCSound->m_CriticalSection);
#      endif
        DOBINI_ASIO_CALLBACK();
      } // end if(nOutBuffersReady>0)
     else // no output buffers ready... we simply may not use the output at the moment !
      { DOBINI_ASIO_CALLBACK(); // -> Sound_iAsioCbkSourceCodeLine
      }

} // end CSound_AsioCallback()
#endif // SWI_ASIO_SUPPORTED



/* EOF < Sound.cpp >  */






