/***************************************************************************/
/* SndOutThd.cpp =  Sound Output Thread.                                   */
/*   - Receives continuous input from the soundcard                        */
/*       and places the sound samples in two alternating buffers.          */
/*   - requires "SOUND.CPP" ( CSound class )  to access the audio device   */
/*   - by DL4YHF,  September 2000  -  October 2022                         */
/*   - VCL dependencies removed in May 2002 to keep the code compact (!)   */
/***************************************************************************/

//---------------------------------------------------------------------------
#include <windows.h>   // dont use the VCL here !!
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#pragma hdrstop


#define _I_AM_SOUND_THREAD_ 1  // for "single-source" variables of this module
#include "SndOutThd.h"

#pragma warn -8071

/*-------------- constants, tables ---------------------------------------*/
#define SNDTHD_EXIT_OK  1
#define SNDTHD_EXIT_BAD 2

/*-------------- Variables -----------------------------------------------*/
LONGLONG SOUND_i64SamplePairIndex=0;  // COUNTER and BUFFER INDEX for audio sample pairs

T_Float SOUND_fltAudioPeakValue[2]={0,0}; // crude peak detector for display purposes
T_Float SOUND_fltTestGeneratorFreq=0; // '0' means generator is disabled


long   SOUND_lOutThreadLoops = 0;   // for debugging purposes only !

// input- and output device settings...
WAVEFORMATEX SOUND_WFXSettings;

// An instance of the CSound class to access the soundcards...
CSound SoundDev;

// A large "upsampling" buffer where the samples for the DAC are taken from..
CSoundUpsamplingBuffer UpsamplingAudioBuffer;

// Sample buffers used to read mono- or stereo-samples from the input
//  NOTE: The data for left+right channel are NOT separated
//        (in contrast to Spectrum Lab !) so [2*x] elements.
T_Float SndThd_fltOutputChunk[  2/*!!*/ * SOUND_MAX_CHUNK_SIZE];


// Some parameters which may be 'overwritten' by command line arguments.
//  Therefore, the names are common for several modules like SndInThd, SerInThd, etc.
int    SOUND_iNrOfAudioChannels;
T_Float SOUND_fltGainFactor   = 1.00;
int    SOUND_iUseSignedValues = 1;
long   SOUND_lDecimationRatio = 1;
int    SOUND_fComplexOutput   = 0;     // 0=real, 1=complex output
T_Float SOUND_fltCenterFrequency = 0.0;
int    SOUND_iDcReject = 0;
int    SndThd_iConnectMeterToOutput = 1;
int    SndThd_iChunkSize=2048; // size of a processing 'chunk' [Samples], limited to SOUND_MAX_CHUNK_SIZE somewhere


/*------------- internal(!) variables ------------------------------------*/
BOOL   SndOutThd_fTerminateRequest;
HANDLE SndOutThd_hWorkerThread;    // Win32 API handle for the worker thread
DWORD  SndOutThd_dwWorkerThreadId; // Identifier for the worker thread

//------------- Implementation of routines and classes ----------------------

//---------------------------------------------------------------------------
void SOUND_SetDefaultParameters(void)
  // Prepare some soundcard settings (WAVEFORMATEX) required later.
  // Note: some of these settings may be overridden by the command line parser.
{
  // Preset some soundcard settings (WAVEFORMATEX) required for opening.
  //  (according to the MANUAL where the default settings are specified !)
  SOUND_WFXSettings.nChannels      = 1;   // default: 1 channel = "mono" 
  SOUND_WFXSettings.wBitsPerSample = 16;  // ALWAYS 16 BIT, EVEN WHEN EXPORTING 8-BIT-VALUES(!!)
  SOUND_WFXSettings.nSamplesPerSec = 11025;

  // Prepare some soundcard settings (WAVEFORMATEX) required for opening.
  // Note: The commented settings have already been set in SetDefaultParameters():
  // SOUND_WFXSettings.nChannels = 2;
  // SOUND_WFXSettings.wBitsPerSample = 16;
  // SOUND_WFXSettings.nSamplesPerSec = 11025;
  SOUND_WFXSettings.wFormatTag = WAVE_FORMAT_PCM;
  SOUND_WFXSettings.nBlockAlign =
      SOUND_WFXSettings.nChannels *(SOUND_WFXSettings.wBitsPerSample/8);
  SOUND_WFXSettings.nAvgBytesPerSec =
      SOUND_WFXSettings.nSamplesPerSec * SOUND_WFXSettings.nBlockAlign;
  SOUND_WFXSettings.cbSize = 0;
} // end SOUND_SetDefaultParameters()



void SOUND_AddSineWave(T_Float *pfltDest, int iLength)
  // produce some crude test data for the DAC:
  // SoundTab_fltCosTable[SOUND_COS_TABLE_LEN];
{
  static T_Float sfltVcoPhase = 0.0;
  int    i,j;
  T_Float d;
  T_Float fltFactor = 0.5/*-6dB*/;  // since 2022-10, *all* floating point streams are normalized to -1 ... +1 !
  T_Float fltPhaseInc = SOUND_fltTestGeneratorFreq * (T_Float)SOUND_COS_TABLE_LEN / (T_Float)SOUND_WFXSettings.nSamplesPerSec;

   for(i=0; i<iLength; ++i)
    {
        sfltVcoPhase += fltPhaseInc;
        if(sfltVcoPhase >= SOUND_COS_TABLE_LEN)
           sfltVcoPhase -= SOUND_COS_TABLE_LEN;
        j = (int)sfltVcoPhase;
        if(j<0)
           j=0;
        if(j>=SOUND_COS_TABLE_LEN)
         { j-=SOUND_COS_TABLE_LEN;
           if(j>=SOUND_COS_TABLE_LEN)
              j = 0; // programming error or illegal frequency increment
         }
        d = fltFactor * SoundTab_fltCosTable[ j ];
        //d = 1024 * (i&4);
        (*pfltDest++) += d;
        if(SOUND_WFXSettings.nChannels>1)
           (*pfltDest++) += d;
    }
} // end SOUND_AddSineWave()


/*------------ Implementation of the Sound Thread -------------------------*/


//---------------------------------------------------------------------------
DWORD WINAPI SoundThdFunc( LPVOID )
  /* Executes the sound output thread.
   *   Was once the "Execute" method of a VCL Thread object,
   *   but this unit does not use the VCL any longer since May 2002.
   */
{
 INT i,j,iChn, iTimeout_ms;
 INT error_code;
 T_Float d,p;

 //  int iSoundInputDeviceID  = -1;  // -1 means "the default audio input device"
   int iSoundOutputDeviceID = -1;

  if( SndThd_iChunkSize > SOUND_MAX_CHUNK_SIZE )
   {  SndThd_iChunkSize = SOUND_MAX_CHUNK_SIZE;
   }

  // Open Sound Device for output
  error_code=SoundDev.OutOpen( //ex:int iOutputDeviceID,     // Identifies a soundcard, or a replacement like C_SOUND_DEVICE_AUDIO_IO
                g_sz255DeviceName, // [in] name of an audio device or DL4YHF's "Audio I/O driver DLL" (full path)
                "", // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE
                "SndOutput",  // [in] pszDescription; a descriptive name which identifies
                   // the audio source ("audio producer"). This name may appear
                   // on a kind of "patch panel" on the configuration screen of the Audio-I/O DLL .
                SOUND_WFXSettings.nSamplesPerSec, // long i32SamplesPerSecond,
                SOUND_WFXSettings.wBitsPerSample, // int  iBitsPerSample ..per channel, usually 16 (also for stereo, etc)
                SOUND_WFXSettings.nChannels,      // int  iNumberOfChannels, 1 or 2 channels (mono/stereo)
                // ex: &SOUND_WFXSettings, // WAVEFORMATEX* pWFX
                SndThd_iChunkSize,  // DWORD BufSize
                0 ); // sample limit, 0 signifies continuous input stream
  if( error_code != NO_ERROR )
   {
     SOUND_iErrorCode = error_code;
     ExitThread( SNDTHD_EXIT_BAD ); // exit code for this thread
   }


  SoundDev.Start();   // start output ...

  // SOUND_thd_error_buf_in = SOUND_thd_error_buf_out = 0; // NO !!
  SOUND_exited_from_thread_loop = false;

  SOUND_lOutThreadLoops = 0;


  /* Almost endless thread loop begins HERE................................*/
  while( !SndOutThd_fTerminateRequest )
   {

    SOUND_lOutThreadLoops++;

    // Get some audio data from the upsampling buffer:
    if( (i=UpsamplingAudioBuffer.GetRealSamples(
              SndThd_iChunkSize,           // long lNrSamplePoints, count of DESTINATION SAMPLE PAIRS(!!)
              SOUND_WFXSettings.nChannels, // int iNrChannels
              TRUE,                        // BOOL fDontGimmeLess, TRUE means "I cannot handle less"
              SndThd_fltOutputChunk,  // destination buffer: SndThd_fltOutputChunk[  2/*!!*/ * SOUND_MAX_CHUNK_SIZE]
              NULL ) ) // [out,optional] : T_ChunkInfo *pOutChunkInfo; since 2011-02
         < SndThd_iChunkSize )
     { // the input buffer has not been filled..
       if(i>=0)
        {
         j = i*SOUND_WFXSettings.nChannels;
         while(i++ < SndThd_iChunkSize)  // Less samples than expected ? Pad with silence !
          {
           for(iChn=0;iChn<SOUND_WFXSettings.nChannels; ++iChn)
             SndThd_fltOutputChunk[j++] = 0.0;
          }
        }
     }


    if(SOUND_fltTestGeneratorFreq>0)
     { // make a test signal (instead of file input)
       SOUND_AddSineWave(SndThd_fltOutputChunk, SndThd_iChunkSize);
     }


    // Optional multiplication with a 'gain' factor..
    if( SOUND_fltGainFactor != 1.00 )
     {
       i= SndThd_iChunkSize*SOUND_WFXSettings.nChannels;
       if(i>SOUND_MAX_CHUNK_SIZE)
          i=SOUND_MAX_CHUNK_SIZE;
       while(i--)
         SndThd_fltOutputChunk[i] *= SOUND_fltGainFactor;
     } // end if <multiply everything with the gain factor ?>


    // Crude peak detector for display purposes..
    for(iChn=0; iChn<=1 && iChn<SOUND_WFXSettings.nChannels; ++iChn)
     {
       p = 0;
       i = ((SndThd_iChunkSize-1)*SOUND_WFXSettings.nChannels)+iChn;
       while( i>=0)
        {
          d = SndThd_fltOutputChunk[i];
          if(d>0) { if(d>p)   p=d;  }  // full wave rectifier + peak detector
            else  { if(d<-p)  p=-d; }
          i -= SOUND_WFXSettings.nChannels;
        }
       SOUND_fltAudioPeakValue[iChn] = p;
     } // end for <all AUDIO OUTPUT channels>


    if( SoundDev.IsOutputOpen() )
     {
       // Send output data to the soundcard object.
       // NOTE: THE SECOND PARAMETER FOR OutWriteFloat
       //      IS THE NUMBER OF SAMPLE POINTS, with 'N' channels per point.
       //      Therefore, the "chunk" must have 2*SndThd_iChunkSize elements.
       iTimeout_ms = 300/*ms*/ + (SndThd_iChunkSize * 1100) / SOUND_WFXSettings.nSamplesPerSec;
       // Example:  SndOutpt.exe /dev="MAGIC LCD 190" /sr=96000 /ch=2 /chunk=65536
       //     ->    SndThd_iChunkSize=65536, SOUND_WFXSettings.nSamplesPerSec=96000,
       //           iTimeout_ms = 300 + 65536 * 1100 / 96000 = 1050 [ms] .
       if( SoundDev.OutWriteFloat(
             SndThd_fltOutputChunk, // source: SndThd_fltOutputChunk[  2/*!!*/ * SOUND_MAX_CHUNK_SIZE]
             SndThd_iChunkSize,     // number of sample-point, or sample-PAIRS(!)
             iTimeout_ms,           // 2020-12-15 : now depends on sampling rate and chunk size
             NULL, // [out,optional] *int : "have been waiting here for XX milliseconds"
             NULL) // [in,optional] chunk info with scaling info;
                   //       see c:\cbproj\SoundUtl\ChunkInfo.h
             != SndThd_iChunkSize )
        {
         error_code = SoundDev.m_ErrorCodeOut;
         // after maximizing a window, often got error_code =
         //      SOUNDIN_ERR_TIMEOUT (yes, SOUNDIN_.. ! )
         //   or SOUNDOUT_ERR_UNDERFLOW.
         // See ErrorCodes.h !
         //
         switch(error_code) // Only handle errors resulting from the OUTPUT
          {
           case SOUNDOUT_ERR_NOTOPEN:
             // someone must have closed the soundcard's output. ?!?!?
             break;

           case SOUNDOUT_ERR_UNDERFLOW:
             // ignore, dont close & reopen
             break;  // end case < OUTPUT UNDERFLOW

           case SOUNDOUT_ERR_TIMEOUT:
             // cpu couldn't keep up.. better luck next time
             break; //  end case < OUTPUT UNDERFLOW or OUTPUT TIMEOUT >

           default: // all other sound errors (like INPUT errors):
             break; // IGNORE input errors, don't close the output !!!
          } // end switch(error_code)
        } /* end if <OutWrite .. != BUF_SIZE */
       else
        { /* everything looks beautiful :-) */
         error_code = SoundDev.m_ErrorCodeOut;
         if (error_code!=NO_ERRORS) // reading ok, but something is wrong...
          {
           SOUND_iErrorCode = error_code;
           SoundDev.m_ErrorCodeIn = NO_ERRORS;  // ouch !!!!!!!!
          }
        } /* end if <sampling of input data ok> */
     } // end if( SoundDev.IsOutputOpen() )
    else
     {  // Re-Open Sound Device
        error_code=SoundDev.OutOpen( //ex:int iOutputDeviceID,     // Identifies a soundcard, or a replacement like C_SOUND_DEVICE_AUDIO_IO
                g_sz255DeviceName, // [in] name of an audio device or DL4YHF's "Audio I/O driver DLL" (full path)
                "", // [in] 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE
                "SndOutput",  // [in] pszDescription; a descriptive name which identifies
                   // the audio source ("audio producer"). This name may appear
                   // on a kind of "patch panel" on the configuration screen of the Audio-I/O DLL .
                SOUND_WFXSettings.nSamplesPerSec, // long i32SamplesPerSecond,
                SOUND_WFXSettings.wBitsPerSample, // int  iBitsPerSample ..per channel, usually 16 (also for stereo, etc)
                SOUND_WFXSettings.nChannels,      // int  iNumberOfChannels, 1 or 2 channels (mono/stereo)
                // ex: &SOUND_WFXSettings, // WAVEFORMATEX* pWFX
                SndThd_iChunkSize,  // DWORD BufSize
                0 ); // sample limit, 0 signifies continuous input stream
        if( error_code != NO_ERROR )
         {
           SOUND_iErrorCode = error_code;
           ExitThread( SNDTHD_EXIT_BAD ); // exit code for this thread
         }
     } // end if <sound device NOT open for input (?!)


   }  /* end <almost endless thread loop> */

  /* if the program ever gets here, the thread is quite "polite"... */
  /*                    or someone has pulled the emergency brake ! */
  SOUND_exited_from_thread_loop = true;

  ExitThread( SNDTHD_EXIT_OK ); // exit code for this thread
     // Note: The code STILL_ACTIVE must not be used as parameter for
     // ExitThread(), but what else ? See Win32 API help !
     // in winbase.h:
     //    #define STILL_ACTIVE STATUS_PENDING
     // in winnt.h:
     //    #define STATUS_PENDING ((DWORD   )0x00000103L)
  return SNDTHD_EXIT_OK; // will this ever be reached after "ExitThread" ?
} // end SoundThdFunc()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
BOOL SoundThd_LaunchWorkerThread(void)
   // Kick the real-time sound processing thread to life,
   //  without the help of Borland's Visual Component Library (VCL).
{
 long l;
 int  iComplexity;



  // Make sure the cosine table exists, the frequency converter (NCO) may need it
  SoundTab_InitCosTable();

  // NO-NO: SOUND_SetDefaultParameters();  // called done by application before evaluating command line
  // Prepare some 'derived' soundcard settings (WAVEFORMATEX) required for opening.
  // Note: The commented settings have already been set in SetDefaultParameters():
  // SOUND_WFXSettings.nChannels = 2;
  // SOUND_WFXSettings.wBitsPerSample = 16;
  // SOUND_WFXSettings.nSamplesPerSec = 11025;
  SOUND_WFXSettings.wFormatTag = WAVE_FORMAT_PCM;
  SOUND_WFXSettings.nBlockAlign =
      SOUND_WFXSettings.nChannels *(SOUND_WFXSettings.wBitsPerSample/8);
  SOUND_WFXSettings.nAvgBytesPerSec =
      SOUND_WFXSettings.nSamplesPerSec * SOUND_WFXSettings.nBlockAlign;
  SOUND_WFXSettings.cbSize = 0;


  // Initialize an 'upsampling' buffer which is used to pass the samples
  //    from the input file to the audio-output-device .
  //    This 'upsampling' process is more or less INVERSE to the decimation
  //    (that takes place in e.g. the "Sound Input" utility, see SndInThd.cpp)
  if(SOUND_lDecimationRatio<1)
     SOUND_lDecimationRatio=1;
  l = SOUND_WFXSettings.nSamplesPerSec // enough buffer for ONE second of audio
     / SOUND_lDecimationRatio;
  if(l<16)l=16;
  if(SOUND_fComplexOutput)  // here: I/Q data IN FILE
     iComplexity = COMPLEX_IN_REAL_OUT;  // must be converted to REAL values for the ADC
  else
     iComplexity = REAL_IN_REAL_OUT;     // both REAL in and REAL out
  UpsamplingAudioBuffer.Init(
     l,           // lBufferLength, max count of sample points (pairs!)
     SOUND_WFXSettings.nSamplesPerSec, // sample rate, often 11025, 22050 or 44100
     SOUND_lDecimationRatio,   // lSampleRateDivisor, only powers of 2 or 3 are allowed
     SOUND_fltCenterFrequency, // fltMixerFrequency,  "NCO" frequency in Hertz
     iComplexity, // conversion from COMPLEX to REAL values needed ?
     SOUND_WFXSettings.nChannels ); // iNrChannels: 1=mono, 2=stereo processing


  // Create the real-time worker thread
  SndOutThd_fTerminateRequest = FALSE;
  SndOutThd_hWorkerThread = CreateThread(
     NULL,    // LPSECURITY_ATTRIBUTES lpThreadAttributes = pointer to thread security attributes
     0,       // DWORD dwStackSize  = initial thread stack size, in bytes
     SoundThdFunc, // LPTHREAD_START_ROUTINE lpStartAddress = pointer to thread function
     0,       // LPVOID lpParameter = argument for new thread
     0,       // DWORD dwCreationFlags = creation flags
              // zero -> the thread runs immediately after creation
     &SndOutThd_dwWorkerThreadId // LPDWORD lpThreadId = pointer to returned thread id
    );
  // The thread object remains in the system until the thread has terminated
  // and all handles to it have been closed through a call to CloseHandle.

  if(!SndOutThd_hWorkerThread)
   { // CreateThread returned NULL, thread could not be created
     return FALSE;
   }

  // Define the Sound Thread's priority as required.
  SetThreadPriority(
    SndOutThd_hWorkerThread,          // handle to the thread
    THREAD_PRIORITY_ABOVE_NORMAL); // thread priority level

  return TRUE;
} // end SoundThd_LaunchWorkerThread()


/***************************************************************************/
BOOL SoundThd_TerminateAndDeleteThread(void)
   /* Terminates the sound thread, cleans up
    * and deletes(!) the sound thread.
    */
{
  int i;
  DWORD dwExitCode;
  BOOL  fOk = TRUE;

  if(SndOutThd_hWorkerThread)
   {
     // First, politely try to let the sound thread terminate itself
     //        allowing it to clean up everything.
     // About GetExitCodeThread(HANDLE hThread,LPDWORD lpExitCode):
     //       lpExitCode: Points to a 32-bit variable
     //          to receive the thread termination status.
     //       If the function succeeds, the return value is nonzero.
     //       If the specified thread has not terminated,
     //          the termination status returned is STILL_ACTIVE.
     dwExitCode=STILL_ACTIVE;
     SndOutThd_fTerminateRequest = TRUE;
     for(i=0;
        (i<20) && (GetExitCodeThread(SndOutThd_hWorkerThread,&dwExitCode));
         ++i)
      {
        // MessageBeep(-1);  // drums please
        Sleep(50);  // wait a little longer for the thread to terminate itself
        if(dwExitCode==SNDTHD_EXIT_OK)
           break;
      }

     // Before deleting the sound thread...
     if(SoundDev.IsInputOpen())
        SoundDev.InClose();
     if(SoundDev.IsOutputOpen())
        SoundDev.OutClose();

     if(dwExitCode==STILL_ACTIVE)
      { // dangerous termination of the sound thread:
        TerminateThread(SndOutThd_hWorkerThread, // handle to the thread
                        SNDTHD_EXIT_BAD);     // exit code for the thread
      }

     CloseHandle(SndOutThd_hWorkerThread);
     SndOutThd_hWorkerThread = NULL;
     fOk &= (dwExitCode==SNDTHD_EXIT_OK);  // TRUE = polite ending,   FALSE = something went wrong.
   }

  return fOk;
} // end SOUND_TerminateAndDeleteThread()

