//---------------------------------------------------------------------------
// File  :  c:\cbproj\SoundUtl\SndOutMain.cpp
// Date  :  2022-10-20     (YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Implementation of the Main Window in a non-VCL application.
//
// Revision history (yyyy-mm-dd):
//   2022-10-20  Replaced WinExec() by CreateProcess(), and re-arranged the
//               calling sequence of ParseCommandLine() and OnWmCreate()
//               to get the "/fs" ("fool shell") option working again.
//               (under Windows 8.1, when e.g. invoking the utility
//                from a BATCH FILE with the following, the old version failed
//                to 'fool the shell' : The batch file's ugly black "console window"
//                remained open until the soundcard utility had TERMINATED ITSELF
//                .... eek)
//   2022-10-17  Corrected a bug in the data type conversion / scaling .
//               Made the window a bit larger again, because Windows.. bleah !
//   2008-03-01  Removed the FlashWindow()-call (caused problems somewhere)
//   2002-05-04  Used in the "Audio Input Tool" in c:\CBProj\SndInput\..
//   2002-05-03  Started as a simple skeleton for a non-VCL application
//               in c:\CBproj\NoVcl\MainWnd.cpp
//---------------------------------------------------------------------------

#include <windows.h>
#include <stdio.h>
#pragma hdrstop

#include "ErrorCodes.h"

#include "SndOutMain.h"
#include "SndOutThd.h"

#include "AudioRdr.h"      // audio file reader
#include "AudioMsgHdlr.h"  // WoBu's proprietary (but simple) "local audio message handler"

#include "Goodies.h"   // Mother's little helpers


//--------------- Defines ---------------------------------------------------

#define L_USE_FlashWindow  0  /* 0 = do NOT use FlashWindow (caused probs) */

 // Menu items for "system menu" alias "window menu" alias "control menu"
 // (the little popup which opens when you click the symbol in the upper left corner)
#define IDM_CONNECT_METER_TO 1
#define IDM_FAST_AND_UGLY    2



//--------------- Global variables -------------------------------------------
char *APPL_szWindowName   = "SND_OUTPUT";
char *APPL_szWindowCaption= "SndOutput"; // ex: "Sound Output" (but that was too long under Windows 8.x, 10, 11)

char g_sz255DeviceName[256] = "-1";    // "-1" is the 'standard audio device' in bad old Windows tradition

BOOL MainWnd_fRunning = TRUE;
int  APPL_iUpdating=0;  // set while some user controls must not react on changes

BOOL SndThd_fCloseDevicesDuringSleep; // added 2019-01, set in the handler for WM_POWERBROADCAST somewhere.
     // When SET, CSound::WaitForInputData() & Co return within 50 milliseconds.

//---- Internal variables for the "user interface", unfortunately GLOBAL -----
HWND MainWnd_hEditNcoFreq;
HWND MainWnd_hEditTestFreq;
HWND MainWnd_hRunStopButton;
HWND MainWnd_hQuitButton;
HWND MainWnd_hHelpButton;
HWND MainWnd_hGainCtrlButton;
#define C_HORZ_BUTTONS 4     // count of control buttons, horizontally aligned
#define C_HORZ_EDITS   2     // count of edit fields, horizontally aligned


RECT MainWnd_rParamDisplay;  // client region for the parameter display
BOOL MainWnd_fMustUpdateOnPaint = TRUE;

int  MainWnd_iTimerTicks;

HINSTANCE SOUND_hInstVolCtrl=NULL; // handle for external volume control program
PROCESS_INFORMATION SoundVol_ProcessInfo;

//**************************************************************************
// application specific stuff
//**************************************************************************
 char     SOUND_sz255InputFile[256] = "audio.dat";
 int      SOUND_iExportDataType = DATA_TYPE_INT16;
 CAudioFileReader     AudioFileReader;      // see AudioRdr.cpp for details
 CAudioMessageHandler AudioMessageHandler;  // see AudioMsgHdlr.cpp ..
 BOOL     g_fUseAudioFile = FALSE; // used in conjunction with /of (output file)
 BOOL     g_fUseAudioMessages = FALSE; // used in conjunction with /sendto
 long     g_lMinTempFileSize = 1024;
 long     g_lMaxTempFileSize = 64*1024;  // '0' means unlimited size
 long     g_lSamplesWaitingInBuffer = 0; // for debugging and display only
 T_Float  SndOut_fltOldCenterFrequency = -1; // used to detect changes
 T_Float  SndOut_fltOldTestGeneratorFrequency=-1;

 BOOL     APPL_fMayUseASIO = FALSE; // ASIO ? Heavens, not. It died long ago.
 


//----------------- For debugging ... ---------------------------------------

//---------------------------------------------------------------------------
CPROT void DEBUG_Breakpoint( void )  // common breakpoint for all "self detected errors"
  // Only called through macro DEBUGGER_BREAK() in 'beta releases' !
  // Replaces the non-functional assert()-stuff .
  // When used for spectrum lab, implemented in C:\cbproj\SpecLab\DebugU1.cpp .
{

} // end DEBUG_Breakpoint()


//---------------------------------------------------------------------------
CPROT void DEBUG_EnterErrorHistory(
     int  debug_level,   // importance of the message (to be entered or not) :
                         //   DEBUG_LEVEL_ERROR, ..WARNING, ..INFO, etc .
     int  options,       // reserved for future extensions
     double time_of_occurrence,  // fractional GMT SECONDS since Jan.1970 ,
                         // may be UTL_USE_CURRENT_TIME or UTL_DONT_SHOW_CURRENT_TIME !
     char *message_fmt,  // pointer to null-terminated format string (source)
                 ... )   // optional argument list (printf-style)
  // CAUTION ! THIS FUNCTION MAY BE CALLED FROM VARIOUS "WORKER THREADS" !
  // (thus it must not use VCL-stuff, because Borland's VCL isn't thread-safe)
  // Looking for DebugPrintf(), ShowError(), or whatever-has-been-used ? It's HERE : DEBUG_EnterErrorHistory() .
  // For Spectrum Lab, implemented in C:\cbproj\SpecLab\DebugU1.cpp .
  // For the 'sound utilities', a stripped-down variant exists in
  //                                  C:\cbproj\SoundUtl\SndInMain.cpp .
{
  va_list parameter;       // argument list for VA_... macros
  char buf[512];
  char *cp = buf;
    va_start( parameter, message_fmt );
    vsprintf( cp, message_fmt, parameter );    // format argument list
    va_end  ( parameter );

    // ex: OutputDebugString( buf );
    
} // end DEBUG_EnterErrorHistory()


//----------------- Analyze the command line --------------------------------
static BOOL fooling_shell = FALSE; // flag set when using the "/fs"-option


//---------------------------------------------------------------------------
void ParseCommandLine(char *pszCmdLine)
  // Some examples (copy & paste to CBuilder IDE: Run..Parameters
  //  /sr=5513 /ch=1 /testsig=1000 /gain=0.1
  //  /sr=44100 /ch=2
  // 
  // Note: Similar (but not identical) incarnations of this function are in:
  //  * C:\cbproj\SoundUtl\SndInMain.cpp
  //  * C:\cbproj\SoundUtl\SndOutMain.cpp
  //  * C:\cbproj\SoundUtl\SndOutMain.cpp
  //  * C:\cbproj\CurveEdit\CurveEdMain.c 
  //  * .. and possibly some more .. grep for "ParseCommandLine" !
{
 char *cp=pszCmdLine;
 T_Float fltParam;
 int  iResult;
 long lParam;
 HWND hwndSender;
 int  iDataType, iErrorCode;
 BOOL fUsingChunkSizeFromCommandline = FALSE;
 char sz255Msg[256];

  while(*cp)
   {
    while(*cp==' ')
         ++cp;  // skip white spaces between arguments
    if(*cp==0)
         break;     

    if(strncmp(cp,"/sh",3)==0)
     { // /sh=X : how to 'show' the window initially..
       cp+=3;
       if(*cp=='o') ++cp; if(*cp=='w') ++cp;
       if(*cp=='=') ++cp;
       lParam = atol(cp);
       switch(lParam)    // second argument for ShowWindow() in WinMain()...
        { case 0:  // Displays as a minimized window.
                   // The active window remains active.
                   APPL_iWinCmdShow = SW_SHOWMINNOACTIVE;
                   break;
          case 1:  // Activates the window
                   // and displays it in its current size and position.
                   APPL_iWinCmdShow = SW_SHOWNORMAL;
                   break;
          case 2:  // Displays the window in its current (predef'd) state.
                   // The active window remains active.
                   APPL_iWinCmdShow = SW_SHOWNA;
                   break;
          default:
                   SOUND_iErrorCode = ERROR_ILLEGAL_CMD_PARAMETER;
                   break;
        } // end switch( "/show=" - parameter )
     } // end "/sh", "/show"
    else if(strncmp(cp,"/sr",3)==0)
     { // /srNNNNN = sample rate
       cp+=3; if(*cp=='=') ++cp;
       lParam = atol(cp);
       if(lParam>=1000 && lParam<=144000) // who knows which sample rates will be supported ?
          SOUND_WFXSettings.nSamplesPerSec = lParam;
        else
          SOUND_iErrorCode = ERROR_ILLEGAL_CMD_PARAMETER;
     }
    else if(strncmp(cp,"/dec",4)==0)
     { // /dec=NN    :  sample rate decimation (here: OF INCOMING DATA)
       cp+=4; if(*cp=='=') ++cp;
       lParam = atol(cp);
       if(lParam>=1 && lParam<=531441/*3 pow 12*/) // who knows what is supported ?
          SOUND_lDecimationRatio = lParam;
        else
          SOUND_iErrorCode = ERROR_ILLEGAL_CMD_PARAMETER;
     }
    else if(strncmp(cp,"/dev",4)==0)
     { // /dev=<string> :  device name (here: OUTPUT device name)
       cp+=4; if(*cp=='=') ++cp;
       GetFileNameFromSource(&cp, g_sz255DeviceName, 255);
       // How to find the goddamned audio device name under windows,
       // quoted from C:\cbproj\SoundUtl\SoundUtilityInfo_01.txt :
       // > 
     }
    else if(strncmp(cp,"/fc",3)==0)
     { // /fc=NN.NNN :  mixer frequency ("center frequency")
       cp+=3; if(*cp=='=') ++cp;
       SOUND_fComplexOutput = 1;    // 0=real, 1=complex output
       SOUND_fltCenterFrequency = strtod( cp, &cp/*char **endptr*/   );
     }
    else if(strncmp(cp,"/chunk",6)==0)
     { // /chunk=NNNN sets the chunk size
       cp+=6; if(*cp=='=') ++cp;
       lParam = atol(cp);
       if( (lParam>=256) && (lParam<=SOUND_MAX_CHUNK_SIZE) )
        { SndThd_iChunkSize = lParam;
          fUsingChunkSizeFromCommandline = TRUE; // no "automatism" further below
        }
     }
    else if(strncmp(cp,"/ch",3)==0)
     { // /chN = set number of channels, 1=mono, 2=stereo
       cp+=3; if(*cp=='=') ++cp;
       lParam = atol(cp);
       if(lParam>=1 && lParam<=2) // so far, we're limited to TWO channels
          SOUND_WFXSettings.nChannels = (WORD)lParam;
        else
          SOUND_iErrorCode = ERROR_ILLEGAL_CMD_PARAMETER;
     }
    else if(strncmp(cp,"/dt",3)==0)
     { // /dtN = set data type for import/export, 0=int8, 1=int16, 2=float32, 3=float64
       cp+=3; if(*cp=='=') ++cp;
       lParam = atol(cp);
       if(lParam>=0 && lParam<=3)
          SOUND_iExportDataType = (WORD)lParam;
        else
          SOUND_iExportDataType = ERROR_ILLEGAL_CMD_PARAMETER;
     }
    else if(strncmp(cp,"/signed",7)==0)
     { // /unsigned = set data type modifier to 'unsigned'
       cp+=7; if(*cp=='=') ++cp;
       SOUND_iUseSignedValues = (int)strtod( cp, &cp/*char **endptr*/   );
     }
    else if(strncmp(cp,"/gain",5)==0)
     { // /gfN = set gain factor  (for software)
       cp+=5; if(*cp=='=') ++cp;
       SOUND_fltGainFactor = strtod( cp, &cp/*char **endptr*/   );
     }
    else if(strncmp(cp,"/testsig",8)==0)
     { // /testsig=N.NNN  : produce sine wave with given frequency as test signal
       cp+=8; if(*cp=='=') ++cp;
       SOUND_fltTestGeneratorFreq = strtod( cp, &cp/*char **endptr*/   );
     }
    else if((strncmp(cp,"/of",3)==0)||(strncmp(cp,"/fn",3)==0))
     { // /of"XXXXXXXX" = define name of output file (default: audio.dat)
       cp+=3; if(*cp=='=') ++cp;  // skip "/of"
       GetFileNameFromSource(&cp, SOUND_sz255InputFile, 255);
       g_fUseAudioFile = TRUE; // used in conjunction with /sendto :
       // If /sendto is specified, but not /of , then do NOT use the
       //               option "use file as audio exchange medium" !
       // Only if both /sendto *AND* /of are specified, use BOTH media .
     }
    else if(strncmp(cp,"/minsize",8)==0)
     { // /minsize=NNN : min size of output file in KByte
       cp += 8;  // skip "/minsize"
       if(*cp=='=') ++cp;
       fltParam = strtod( cp, &cp/*char **endptr*/ );
       if(*cp=='k' || *cp=='K') { ++cp; fltParam*=1024; }
       g_lMinTempFileSize = (long)fltParam;
     }
    else if(strncmp(cp,"/maxsize",8)==0)
     { // /maxsize=NNN : max size of output file in KByte
       cp += 8;  // skip "/maxsize"
       if(*cp=='=') ++cp;
       fltParam = strtod( cp, &cp/*char **endptr*/ );
       if(*cp=='k' || *cp=='K') { ++cp; fltParam*=1024; }
       g_lMaxTempFileSize = (long)fltParam;
     }
    else if(strncmp(cp,"/fs",3)==0)
     { // /fs = Fool Shell.  Pass the arguments to another program and return.
       // Has been used to fool QuickBasic's SHELL command which waits until
       // the started application has returned.
       // Where to find the name of the executable including the full path ?
       // When tested under Windows 8.1 in the author's project directory,
       // GetModuleFileName( NULL, ..) delivered the following :
       //   sz255ChildCmdLine = "C:\\CBproj\\SoundUtl\SndInput.exe" .
       // Thus, with a 256-byte string buffer, enough space remains
       // to append THE REST OF the original command line, *after* the "/fs" token.
       char sz255ChildCmdLine[256];
       if( ! GetModuleFileName(
                          NULL, // handle to module to find filename for
             sz255ChildCmdLine, // pointer to buffer for module path
                       127 ) )  // size of buffer, in characters
         { // replacement if GetModuleFileName failed
           strcpy(sz255ChildCmdLine,"SndInput.exe");
         }
       // Call "myself" with same parameters, but without the /fs switch :
       strcat( sz255ChildCmdLine, cp+3 );
       // 2022-10-20: At this point, sz255ChildCmdLine was ..
       //  "C:\\CBproj\\SoundUtl\SndInput.exe /sh=1 /sr=48000 /ch=2 /dt=2 /dev=\"IN 3-4\" testsig=440"
       // ,----------------------------------'
       // '--> this "separating space" has been copied from the 'original'
#     if(0)   // out of the blue (somewhere between 2000 and 2022), this didn't work anymore:
       WinExec(sz255ChildCmdLine, // address of command line
               SW_SHOWNORMAL  );  // window style for new application
       // 2022-10-20: At this point (when single-stepping) one would expect
       //             to see TWO INSTANCES of SndInput.exe running, e.g. :
       //  explorer.exe
       //    |- bcb.exe  (Borland C++ Builder running as child of 'explorer'.. ok...)
       //    |   |- SndInput.exe   (Sound Input running as child of Borland C++ Builder.. umm..)
       //  NO OTHER INSTANCE of SndInput.exe was seen anywhere in Process Explorer !
       //  Obviously, a lot has been modified under the (windows-) hood
       //  since the "fool shell"-option was last tested (guess that was on Windows XP).
       fooling_shell = TRUE;      // -> let the CALLER call PostQuitMessage( 0 ) and return.
       return;                    // ciao ..
#     else  // try something else instead of the simple (but now non-functional) WinExec() :
       iErrorCode = StartExecutableWithCommandLine(sz255ChildCmdLine,
               SW_SHOWNORMAL  );  // window style for new application
       // With CreateProcess() [called from StartExecutableWithCommandLine()],
       // the new instance of SndInput.exe appeared as a 'child' of the launcher
       // in Process Explorer. Not sure if this is what we want... :
       //  explorer.exe
       //    |- bcb.exe  (Borland C++ Builder running as child of 'explorer'.. ok...)
       //    |   |- SndInput.exe   (Sound Input running as child of Borland C++ Builder..)
       //    |   |   |- SndInput.exe   (Sound Input running as child of Sound Input !)

       if( iErrorCode == NO_ERRORS )
        { fooling_shell = TRUE;      // -> let the CALLER call PostQuitMessage( 0 ) and return.
          return;                    // ciao ..
        }
       else // StartExecutableWithCommandLine() FAILED for some reason ..
        { // ignore the "fool shell" switch and let THIS instance do the job.
          SOUND_iErrorCode = iErrorCode; // let the tiny GUI show the problem somewhere
        }
#     endif
       //
     }
    else if(strncmp(cp,"/rcvfrom",8)==0)
     { // /rcvfrom = request and receive audio data from a certain window .
       //            Implemented 2004-06-13 .
       cp += 8;  // skip "/rcvfrom"
       if(*cp=='=') ++cp; // .. or "/rcvfrom=" <handle>,<datatype>
       // Params: - handle of window to request data from, and
       //         - type of data to be used in communication .
       hwndSender = (HWND)atol(cp);
       iDataType = DATA_TYPE_INT16;  // = 0x02
       while(*cp>='0' && *cp<='9') ++cp;
       if(*cp==',') // data type also specified ?
        { ++cp;
          iDataType = atol(cp);
        }
       if(hwndSender!=NULL)
        { AudioMessageHandler.ConnectSource(
                  hwndSender,     // audio sender's 'window handle'
                  iDataType );    // type of data "I" want to receive from him
          g_fUseAudioMessages = TRUE;
        }
     }
    else if(strncmp(cp,"/quit",5)==0)
     { PostQuitMessage( 0 );      // terminate myself
       return;                    // ciao ciao ..
     }
    else
     { // none of these !
       SOUND_iErrorCode = ERROR_UNKNOWN_CMD_ARGUMENT;
       sprintf(sz255Msg,"Unknown command argument : \"%s\" .",cp);
       MessageBox( APPL_hMainWindow,
                sz255Msg,       // address of text in message box
          APPL_szWindowCaption, // address of title of message box
          MB_OK | MB_ICONEXCLAMATION ); // style of message box
     }

    // skip the token+parameter, advance to next char, but don't eat the terminator :-)
    while(*cp!=' ' && *cp!='\0')
        ++cp;
   } // end while(*cp)

  // If no chunk size has been specified, make a processing chung large enough
  // for a few hundred milliseconds of audio buffering :
  if( ! fUsingChunkSizeFromCommandline )
   { SndThd_iChunkSize = 2048; // at least 2048 samples per chunk
     while( (SndThd_iChunkSize < (SOUND_WFXSettings.nSamplesPerSec/8) ) // less than 1/8 second buffer ?
         && (SndThd_iChunkSize < (2*SOUND_MAX_CHUNK_SIZE) ) )
      { SndThd_iChunkSize *= 2;
      }
   } // end if( ! fUsingChunkSizeFromCommandline )


  // The "max file size" should be larger than the "min file size"...
  if(g_lMaxTempFileSize<g_lMinTempFileSize)
     g_lMaxTempFileSize=g_lMinTempFileSize;

} // end ParseCommandLine()


//---------------------------------------------------------------------------
void UpdateFrequencyDisplay(void)
{
 char sz80Temp[81];
  ++APPL_iUpdating;
   if(SOUND_fComplexOutput)   // 0=real, 1=complex output
      sprintf(sz80Temp,"%5.1f", (float)SOUND_fltCenterFrequency );
   else
      sprintf(sz80Temp,"--off--");
   SetWindowText( MainWnd_hEditNcoFreq, sz80Temp );
   sprintf(sz80Temp,"%5.1f", (float)SOUND_fltTestGeneratorFreq );
   SetWindowText( MainWnd_hEditTestFreq,sz80Temp );
  --APPL_iUpdating;
}

//---------------------------------------------------------------------------
void UpdateMyMenu(void)
{
  HMENU hMenu = GetSystemMenu( APPL_hMainWindow, FALSE/*revert*/ );

  CheckMenuItem( hMenu, IDM_FAST_AND_UGLY, MF_BYCOMMAND |
    ((UpsamplingAudioBuffer.GetFastMode()!=0)? MF_CHECKED : MF_UNCHECKED ) );
  PrintIntoMenuItem( // first 3 params like CheckMenuItem :
                 hMenu, IDM_CONNECT_METER_TO, MF_BYCOMMAND,
                     // parameter 4,5 and following like "wsprintf" :
                 "Meter connected to %s",
                 SndThd_iConnectMeterToOutput?"output":"input" );
}


//---------------------------------------------------------------------------
#define L_CHUNK_SIZE 16384
T_Float  fltChunk[  4/*!!!!*/ *  L_CHUNK_SIZE];
void HandleSamplesFromAudioFileServer(void)  // called every 100ms or so from a TIMER
{    // Similar routines exist in SpecLab.cpp and in SndOutMain.cpp !
 long lNrSamplePoints, lNrSamplesToRead, lNrSamplesRead;
 int  iChannels;
 int  iMaxLoops=5;
 T_Float *pflt;

  // Before we are going to process samples from the RAM buffer,
  //  save the count of OCCUPIED buffer locations for later (display!):
  g_lSamplesWaitingInBuffer = UpsamplingAudioBuffer.GetOccupiedInputBufferSpace();

  // If the input audio file is not open, try to open it.
  // If successfully opened, it will be read and copied into a buffer later.
  // If not successfully opened, try again later. The audio processing thread
  // will send "silence" to the DAC converter if it runs out of data.
  if( ! AudioFileReader.IsOpen() )
   {
     if(! AudioFileReader.OpenFile(
         SOUND_sz255InputFile,// name of input file
           SOUND_iExportDataType, // usually DATA_TYPE_INT16
                       NULL)) // used if no WAVe file header is found
      {
        SOUND_iErrorCode = AudioFileReader.GetLastError();
      }
     else
      {
        if(SOUND_iErrorCode == ERROR_FILE_DOESNT_EXIST)
           SOUND_iErrorCode = NO_ERROR;
      }
   } // end if <audio source file does not exist>



  while(--iMaxLoops>0)
   { // repeat until no more samples can be processed ...


    if( AudioFileReader.IsOpen() )
     {
      // How many SAMPLES can be placed in the buffer for real-time processing ?
      lNrSamplePoints = UpsamplingAudioBuffer.GetFreeInputBufferSpace();
      // Convert the max count of SAMPLE PAIRS into a count of FLOATING POINT VALUES
      if(lNrSamplePoints>L_CHUNK_SIZE)
         lNrSamplePoints=L_CHUNK_SIZE;
      lNrSamplesToRead = lNrSamplePoints * UpsamplingAudioBuffer.GetComponentsPerSample();
      if(lNrSamplesToRead<=0)
        return;   // audio buffer completely full, try again later

      // Read a bunch of audio samples from the input file.
      //   If all works as planned, sample values will range from -1.0 to +1.0,
      //   because in October 2022, the 16-bit-anachronism (value range +/-32677)
      //   has been removed everywhere (hopefully).
      lNrSamplesRead = AudioFileReader.ReadSamplesFromFile(
                                  fltChunk, // pointer to destination block
                         lNrSamplesToRead); // max number of 'float' values (!)

      if(lNrSamplesRead<0)
       { // there was a PROBLEM in ReadSamplesFromFile (not just EOF)
         // why did it fail ?
         SOUND_iErrorCode = AudioFileReader.GetLastError();
       }
      else
      if(lNrSamplesRead>0)
       { // got some new data from the input file -> put them into the buffer
         lNrSamplesRead /= UpsamplingAudioBuffer.GetComponentsPerSample();
         iChannels = UpsamplingAudioBuffer.GetNumberOfChannels();
         if( ! SndThd_iConnectMeterToOutput )
          { T_Float v,p;
            for(int n=0; n<iChannels; ++n)
             { p=0;
              pflt = fltChunk + n;
              for(int i=0; i<=lNrSamplesRead; ++i )
               { v = *pflt;
                 if(v>0) { if(v>p)   p=v;  } // full wave recifier + peak detector
                   else  { if(v<-p)  p=-v; } // note: sample values should range from -1.0 to +1.0,
                                             // thus we could simply SQUARE the input
                 pflt += iChannels;
              }
             SOUND_fltAudioPeakValue[n] = p;
            }
          } // end if( SndThd_iConnectMeterToOutput )

         if( ! UpsamplingAudioBuffer.EnterSamples(
                fltChunk, // T_Float *pFltSource
                lNrSamplesRead, // long lNrSamplePoints
                iChannels, // iNrCompsPerSample, number of channels, multiplied by one (real) or two (complex)
                NULL) // [in,optional]    T_ChunkInfo *pChunkInfo : see c:\cbproj\SoundUtl\ChunkInfo.h.

            )
          { // the 'upsampling buffer' object could not accept the data (?)

          }
       }
      else // looks like end-of-file from the Audio File Reader..
       {   // Close the file and delete it, so the 'audio file producer'
           //  writes a new one which we can 'consume' here a.s.a.p.
           if(! AudioFileReader.CloseFile() )
             { // why did it fail ?
               SOUND_iErrorCode = AudioFileReader.GetLastError();
             }
           else
            {
             // Delete(!) the INPUT(!) file  to let the "producer" emit a new one
             if(! DeleteFile(SOUND_sz255InputFile) )
                 SOUND_iErrorCode = GetLastError();
            }
       } // end else   < end of audio input file >
     } // end if( AudioFileReader.IsOpen() )
    else
     {
       return;
     }

   } // end while(--iMaxLoops>0)

} // end HandleSamplesFromAudioFileServer()




//**************************************************************************
// Handle Button- and other events
//**************************************************************************

//---------------------------------------------------------------------------
void OnQuitButton(void)
{
  // handle button press by quitting
  SendMessage(APPL_hMainWindow, WM_CLOSE, 0,0);
} // end OnQuitButton()

//---------------------------------------------------------------------------
void OnRunStopButton(void)
{
   MainWnd_fRunning = !MainWnd_fRunning;
} // end OnRunStopButton()

//---------------------------------------------------------------------------
void OnHelpButton(void)
{
 char sz4000[4444];
 char *cp;
 char *pszParam;
 float fc,bw;   // center frequency + bandwidth


  strcpy(sz4000, "SoundOutput utility by Wolfgang Buescher (DL4YHF)\r\n");
  strcat(sz4000, "   Reads from disk file and sends to soundcard .\r\n");
  strcat(sz4000, "   Experimental software with no warranty of any kind !\r\n");
  strcat(sz4000, "   Updates were available at www.qsl.net/dl4yhf/snd_utl1.html .\r\n");
  strcat(sz4000, "   For more info, read \"SoundUtilityInfo_*.txt\" .\r\n\n");
  cp=sz4000+strlen(sz4000);
  wsprintf(cp, "\r\nCompiled: %s\r\nCmd line: \"%s\"\r\n", __DATE__, APPL_sz255CmdLine);
  cp+=strlen(cp);
  sprintf(cp, "Window Handle: %ld\r\n", (long)APPL_hMainWindow);
  cp+=strlen(cp);  


  // Describe the data input...
  wsprintf(cp,"\r\nInput file: %s\r\n",SOUND_sz255InputFile);
  cp+=strlen(cp);
  // wsprintf(cp,"Temporary: %s\r\n",g_sz255TempOpFile); cp+=strlen(cp);
  wsprintf(cp,"Format: "); cp+=strlen(cp);
  switch(SOUND_iExportDataType)
   {
     case DATA_TYPE_INT8     : strcpy(cp,"8 bit integer"); break;
     case DATA_TYPE_INT16    : strcpy(cp,"16 bit integer"); break;
     case DATA_TYPE_FLOAT32  : strcpy(cp,"32 bit float"); break;
     case DATA_TYPE_FLOAT64  : strcpy(cp,"64 bit float"); break;
     default: wsprintf(cp,"<unknown, code %d>",(int)SOUND_iExportDataType); break;
   }
  cp+=strlen(cp);
  wsprintf(cp,", %d channel(s)/sample",(int)UpsamplingAudioBuffer.GetComponentsPerSample() );
  cp+=strlen(cp);
  sprintf(cp,", %5.1f samples/sec\r\n",(float)UpsamplingAudioBuffer.GetDecimatedSampleRate() );
  cp+=strlen(cp);

  // Describe how the data are processed...
  fc = UpsamplingAudioBuffer.GetNcoFrequency();  // center frequency (mixer L.O.)
  bw = UpsamplingAudioBuffer.GetDecimatedSampleRate() * 0.5;
  if( UpsamplingAudioBuffer.GetComplexInputFlag() )
   { pszParam = "complex";
     bw = bw * 2;            // doubled bandwidth, but also doubled data volume
   }
  else
   { pszParam = "real";
   }
  sprintf(cp,"\r\nProcess:\t"); cp+=strlen(cp);
  if( UpsamplingAudioBuffer.GetFreqConversionFlag() )
   {
    sprintf(cp,"Upsampling %s data with frequency conversion .",pszParam);
    cp+=strlen(cp);
    sprintf(cp,"\r\n\tCenter frequency (NCO) = %5.1f Hz,",(float)fc );
   }
  else // no frequency conversion, BUT ?
   { fc = UpsamplingAudioBuffer.GetDecimatedSampleRate() * 0.25;
     if( UpsamplingAudioBuffer.GetDecimationRatio() > 1)
        sprintf(cp,"Upsampling without frequency conversion .");
     else
        sprintf(cp,"No conversion, no decimation .");
   }
  cp+=strlen(cp);
  if( UpsamplingAudioBuffer.GetDecimationRatio() > 1)
   { sprintf(cp,"\r\n\tUpsampling factor = %d",(int)UpsamplingAudioBuffer.GetDecimationRatio() );
     cp+=strlen(cp);
     if( UpsamplingAudioBuffer.GetFastMode() != 0 )
        sprintf(cp,", CRUDE but FAST filtering ." );
     else
        sprintf(cp,", reasonable anti-alias filtering ." );
     cp+=strlen(cp);
   }
  sprintf(cp,"\r\n\tInput band translated to: %5.1f Hz .. %5.1f Hz", (float)(fc-bw/2), (float)(fc+bw/2) );
  cp+=strlen(cp);
  sprintf(cp,"\r\n\tInternal buffer uses %d samples/chunk.", (int)SndThd_iChunkSize );
  cp+=strlen(cp);


  // Describe the data output...
  sprintf(cp,"\r\n\nOutput Device: \"%s\".", SoundDev.GetOutputDeviceName() );
  // The result may be somewhat surprising, e.g. "3 2- USB Audio Codec"
  //                                              | |  |_____________|
  //                                              | |   |
  //                                              | |   Funny name for an IC-9700
  //                                              | |
  //                                              | Some other funny index,
  //                                              | possibly added by the
  //                                              | USB audio device driver (?)
  //                                              |
  //                                              Our 'wave out' device INDEX
  cp+=strlen(cp);
  sprintf(cp,"\r\nAudio output: %ld samples/sec, %d channel(s)\r\n",
      (long)SOUND_WFXSettings.nSamplesPerSec,
       (int)SOUND_WFXSettings.nChannels);
  cp+=strlen(cp);


  // Note: We are using the Win32 API function "MessageBox" here,
  //       no fancy VCL stuff, no messing around with resource files.
  MessageBox(
    APPL_hMainWindow,	// handle of owner window
    sz4000,             // address of text in message box
    APPL_szWindowCaption, // address of title of message box
    MB_OK | MB_ICONINFORMATION ); // style of message box
} // end OnHelpButton()


//---------------------------------------------------------------------------
void OnGainCtrlButton(void)
{
#if(0)
   // Any idea how to close a program started with ShellExecute() ?...
   ShellExecute(  // See Win32 Programmer's Reference...
     APPL_hMainWindow,  // HWND hwnd = handle to parent window
           NULL,    // LPCTSTR lpOperation = pointer to string that specifies operation to perform
    "sndvol32.exe", // LPCTSTR lpFile = pointer to filename or folder name string
             "",    // LPCTSTR lpParameters = pointer to string that specifies executable-file parameters
           NULL,    // LPCTSTR lpDirectory = pointer to string that specifies default directory
    SW_SHOWNORMAL); // .. whether file is shown when opened
#endif // (0)

#if(0)
    // Try something different. Maybe this is the way to close the window later..
    STARTUPINFO MyStartupInfo;
    GetStartupInfo( &MyStartupInfo);

    CreateProcess( // See Win32 Programmer's Reference...
          NULL,      // LPCTSTR lpApplicationName = pointer to name of executable module
     "sndvol32.exe", // pointer to command line string
          NULL,      // pointer to process security attributes
          NULL,      // pointer to thread security attributes
          FALSE,     // BOOL bInheritHandles = handle inheritance flag
     NORMAL_PRIORITY_CLASS, // DWORD dwCreationFlags = creation flags
          NULL,      // LPVOID lpEnvironment = pointer to new environment block
          NULL,      // LPCTSTR lpCurrentDirectory = pointer to current directory name
    &MyStartupInfo,  // pointer to STARTUPINFO
    &SoundVol_ProcessInfo); // pointer to PROCESS_INFORMATION
#endif // (0)

    // As usual with Microsoft / Windows re-inventing the wheel
    // every couple of years, the above (using "sndvol32.exe") 
    // didn't work anymore. Solution borrowed from Spectrum Lab :
    Sound_OpenSoundcardOutputControlPanel( g_sz255DeviceName );     
    // > Opens a "volume control panel" for the soundcard, or similar stuff
    // > (OS-specific; "SNDVOL32.EXE", "SNDVOL.EXE", or whatever comes next).


} // end OnGainCtrlButton()


void OnNcoEdit(WORD wNotifyCode) // something happened with the NCO frequency edit field
{ // wNotifyCode has been passed from the edit control, through a WM_COMMAND message.
 char sz80Temp[81];  char *cp;

 if(APPL_iUpdating==0)
  {
   switch(wNotifyCode)
    {
     case EN_CHANGE: // The user has modified text in an edit control
         GetWindowText(MainWnd_hEditNcoFreq,sz80Temp,16);
         cp = sz80Temp;
         SOUND_fltCenterFrequency = strtod( cp, &cp/*char **endptr*/  );
         UpsamplingAudioBuffer.SetNcoFrequency(SOUND_fltCenterFrequency);
         SOUND_fltCenterFrequency = UpsamplingAudioBuffer.GetNcoFrequency();
         SndOut_fltOldCenterFrequency = SOUND_fltCenterFrequency;
         break;
    } // end switch(wNotifyCode)
  } // end if <not "updating">

} // end OnNcoEdit()


void OnTestFreqEdit(WORD wNotifyCode) // something happened with the TEST frequency edit field
{
 char sz80Temp[81];  char *cp;

 if(APPL_iUpdating==0)
  {
   switch(wNotifyCode)
    {
     case EN_CHANGE: // The user has modified text in an edit control
         GetWindowText(MainWnd_hEditTestFreq,sz80Temp,16);
         cp = sz80Temp;
         SOUND_fltTestGeneratorFreq = strtod( cp, &cp/*char **endptr*/  );
         SndOut_fltOldTestGeneratorFrequency = SOUND_fltTestGeneratorFreq;
         break;
    } // end switch(wNotifyCode)
  } // end if <not "updating">
} // end OnTestFreqEdit()




//**************************************************************************
//  Functions called by the application skeleton, optional ...
//**************************************************************************

void OnWmTimer(HWND hWnd, WORD wTimerID ) // called on WM_TIMER
{
  static int s_fWasRunning = -1;
#if( L_USE_FlashWindow )
  static int s_fWasFlashing= 0;
#endif // ( L_USE_FlashWindow )

  if( s_fWasRunning != MainWnd_fRunning )
   { // Change the caption of the "Run/Stop" button if needed...
     s_fWasRunning = MainWnd_fRunning;
     SetWindowText(MainWnd_hRunStopButton,
            s_fWasRunning ? "Stop":"Run");
   }

  if(  (SndOut_fltOldCenterFrequency != SOUND_fltCenterFrequency )
     ||(SndOut_fltOldTestGeneratorFrequency != SOUND_fltTestGeneratorFreq) )
   { // someone changed a frequency (NCO, test generator) ... update the display:
     SndOut_fltOldCenterFrequency = SOUND_fltCenterFrequency;
     SndOut_fltOldTestGeneratorFrequency = SOUND_fltTestGeneratorFreq;
     UpdateFrequencyDisplay();
   }


  if(MainWnd_fRunning)
   {
     ++MainWnd_iTimerTicks;

     // If there are new samples in the buffer, get them and put them
     // into the exported file(s)..
     HandleSamplesFromAudioFileServer();

     // update 'parameter display' in upper window half.. OCCASIONALLY
     if( (MainWnd_iTimerTicks & 1) == 1)
       RedrawWindow(
         hWnd,                   // handle of window
         &MainWnd_rParamDisplay, // address of structure with update rectangle
         NULL,                   // handle of update region
         RDW_INVALIDATE );       // array of redraw flags
   } // end if (MainWnd_fRunning)


#if( L_USE_FlashWindow )  // "FlashWindow()" caused problems with hidden task bar (under Win 2k)
  if( (MainWnd_iTimerTicks & 15) == 0)
   {
     // "FlashWindow" is ugly, so use it sparesly
     //   ( Dodo ugly so Dodo must die ;-)
     if(   (SOUND_iErrorCode != NO_ERROR)
        && (SOUND_iErrorCode != ERROR_FILE_DOESNT_EXIST)
       )
      {
        FlashWindow(APPL_hMainWindow, TRUE); // invert the title bar
        s_fWasFlashing = TRUE;
      }
     else
      {
       if(s_fWasFlashing)
        {
         FlashWindow(APPL_hMainWindow, FALSE); // turn flashing off (?!)
         s_fWasFlashing = FALSE;
        }
      } // end if <no error status at the moment>
   } // end if <every N-th timer tick>
#endif // ( L_USE_FlashWindow )

} // end OnWmTimer()

LONG OnWmCopydata(HWND hMyWnd, HWND hSendingWnd, COPYDATASTRUCT *pCds )
   // "Method" called on WM_COPYDATA.
{
  char sz255NewCommandLine[256];
  int n;

  switch(pCds->dwData)
   { case FOUR_CHARS_TO_DW('c','m','d','0'):
          // Command Line, passed from another instance of this program !
          n = pCds->cbData; if(n>255) n=255;
          if( pCds->lpData != NULL )
           {
             strncpy( sz255NewCommandLine, (char*)pCds->lpData, n);
             sz255NewCommandLine[n] = '\0'; // make sure "C" string is terminated
             ParseCommandLine( sz255NewCommandLine );
             return 1;
           }
          return 0;
     default:
          break;
   }
  return 0; // 0 = "this message could not be handled"

} // end OnWmCopydata()


//**************************************************************************
// Red tape for the windows application, mostly called from ApplMain.cpp
//**************************************************************************

//---------------------------------------------------------------------------
LONG OnWmPaint(HWND hWnd)
{ // reaction to WM_PAINT message
  PAINTSTRUCT ps;
  SIZE text_size;
  HDC hDC = BeginPaint(hWnd,&ps);
  HFONT hFont;
  RECT rect;
  int  x, y, iChn;
  char sz80[81];
  T_Float d;
  long   l;
  static T_Float sfltOldPeakValue[2]={-1,-1};
  COLORREF lColor;


      GetClientRect(hWnd,&rect);
      SetBkColor(hDC, GetSysColor(MY_BACKGROUND_COLOR) );
      y = 4;
      // DONT FORGET TO 'UNSELECT' this font later and DELETE the font object :
      hFont = (HFONT)SelectObject(hDC, CreateFont(
                    14/*nHeight*/,      0/*best width*/,
                     0/*nEscapement*/,  0/*nOrientation*/,
                 FW_NORMAL/*fnWeight*/,
                 FALSE/*fdwItalic*/,    FALSE/*fdwUnderline*/,
                 FALSE/*fdwStrikeOut*/,	ANSI_CHARSET/*fdwCharSet*/,
                 OUT_DEFAULT_PRECIS , // fdwOutputPrecision
                 CLIP_DEFAULT_PRECIS, // fdwClipPrecision
                 DEFAULT_QUALITY,     // fdwQuality
                 DEFAULT_PITCH | FF_DONTCARE, // fdwPitchAndFamily
                 "Arial"));           // pointer to typeface name string
      GetTextExtentPoint32(hDC, "Q", 1, &text_size);
      SetBkMode(hDC, OPAQUE );
      SetTextAlign(hDC,TA_LEFT|TA_TOP);
      // Don't erase the bargraph-area here !
      rect.bottom= MainWnd_rParamDisplay.bottom;
      rect.right = MainWnd_rParamDisplay.right - 16*SOUND_WFXSettings.nChannels;
#if(0) // TEST ONLY:
      FillRect(hDC, &rect, (HBRUSH)COLOR_ACTIVECAPTION );
#else  // normal operation
      FillRect(hDC, &rect, (HBRUSH)(MY_BACKGROUND_COLOR+1) ); // ex: COLOR_WINDOW
      // why add one to the colour value ? just another windoze-annoyance..
#endif

      wsprintf(sz80,"Timer: %ld.%ld sec",
         (long)MainWnd_iTimerTicks/10, (long)MainWnd_iTimerTicks%10 );
      TextOut(hDC,MainWnd_rParamDisplay.left, y, sz80, strlen(sz80) );
      y+=text_size.cy;

      wsprintf(sz80,"Samples: %ld = %ld sec",
         (long)UpsamplingAudioBuffer.GetTotalSampleCountOut(),
         (long)UpsamplingAudioBuffer.GetTotalSampleCountOut()
                        / SOUND_WFXSettings.nSamplesPerSec );
      TextOut(hDC,MainWnd_rParamDisplay.left, y, sz80, strlen(sz80) );   y+=text_size.cy;

#if(0)
      wsprintf(sz80,"TempFile : %ld kByte",AudioFileReader.GetBytesWritten()/1024);
#else // // for debugging purposes :
      wsprintf(sz80,"ThdLoops : %ld",(long)SOUND_lOutThreadLoops);
#endif
      TextOut(hDC,MainWnd_rParamDisplay.left, y, sz80, strlen(sz80) );   y+=text_size.cy;

      strcpy(sz80, ErrorCodeToString(SOUND_iErrorCode) );
      TextOut(hDC,MainWnd_rParamDisplay.left, y, sz80, strlen(sz80) );   y+=text_size.cy;

      // Draw vertical bar(s) for the audio volume indicator(s)
      for(iChn=0; iChn<=1 && iChn<SOUND_WFXSettings.nChannels; ++iChn)
       { // there may be one or two audio channels..
        if( (sfltOldPeakValue[iChn] != SOUND_fltAudioPeakValue[iChn])
           || MainWnd_fMustUpdateOnPaint)
         { sfltOldPeakValue[iChn] = SOUND_fltAudioPeakValue[iChn];
           x = MainWnd_rParamDisplay.right - 16*SOUND_WFXSettings.nChannels
                                           + 16*iChn;
           // Draw the "peak value indicator" at the right window edge...
           DrawVerticalIndicatorBar(  hDC,        // device context for output
             x, MainWnd_rParamDisplay.top,
             x+15,MainWnd_rParamDisplay.bottom,   // drawing area: left,top,right,bottom
             (SOUND_fltAudioPeakValue[iChn]>0.7)? // Color of the indicator (RGB-mix)
                  RGB(0xFF,0x00,0x00) : RGB(0x00,0xFF,0x00),
             RGB(0xFF,0xFF,0xFF),  // Color of the background (RGB-mix)
             SOUND_fltAudioPeakValue[iChn]*100.0, // percentage value (0..100)
             (char*)((SOUND_WFXSettings.nChannels>1)? // caption, here: L,R channel
                      ((iChn==0)?"L":"R") : "" ) );
         } // end if <audio peak value changed>
       } // end for <all audio channels>

      // Draw another vertical bar which shows the RAM buffer usage:
      // (from GetOccupiedBufferSpace(); read shortly before taking samples
      //  out of a buffer which is filled in the real-time worker thread)
      l = UpsamplingAudioBuffer.GetTotalBufferSpace();
      if(l>0)
       d = (T_Float)100.0 * (T_Float)g_lSamplesWaitingInBuffer / (T_Float)l;
      x = MainWnd_rParamDisplay.right - 16*(SOUND_WFXSettings.nChannels+1) - 2;
      lColor = RGB(0xAF,0xAF,0xFF);        // light blue = medium water mark
      if(d<10) lColor=RGB(0xFF,0x00,0x00); // red = too low (for save continuous DAC OPERATION!)
      DrawVerticalIndicatorBar(  hDC,        // device context for output
          x, MainWnd_rParamDisplay.top,
          x+17,MainWnd_rParamDisplay.bottom, // drawing area: left,top,right,bottom
          lColor,    // Color of the indicator (RGB-mix)
          (d<10) ?   // Color of the background (RGB-mix)
                 RGB(0xFF,0xAF,0xAF) : RGB(0xFF,0xFF,0xFF),
          d, "bu" ); // percentage value (0..100) ,  caption, here "b"=buffer

      DeleteObject(SelectObject(hDC,hFont));

  EndPaint(hWnd,&ps);
  MainWnd_fMustUpdateOnPaint = FALSE;  // done
  return 0L;
} // end OnWmPaint



//---------------------------------------------------------------------------
LONG OnWmCommand(HWND hWnd,
           WORD wNotifyCode, // notification code
           WORD wID,         // item, control, or accelerator identifier
           HWND hwndCtl )    // handle of control
{ // reaction to WM_COMMAND message

  // Edit field events...
  if(hwndCtl==MainWnd_hEditNcoFreq)
     {  OnNcoEdit(wNotifyCode);  return 0L;   }
  if(hwndCtl==MainWnd_hEditTestFreq)
     {  OnTestFreqEdit(wNotifyCode); return 0L; }

  // Button events...
  if(hwndCtl==MainWnd_hQuitButton)
     {  OnQuitButton();  return 0L;   }
  if(hwndCtl==MainWnd_hHelpButton)
     {  OnHelpButton();  return 0L;   }
  if(hwndCtl==MainWnd_hRunStopButton)
     {  OnRunStopButton(); return 0L; }
  if(hwndCtl==MainWnd_hGainCtrlButton)
     {  OnGainCtrlButton(); return 0L; }


  return 0L;
} // end OnWmCommand()


//---------------------------------------------------------------------------
LONG OnWmSysCommand(HWND hWnd,
       WORD uCmdType,     // (wParam)         type of system command requested
       WORD xPos,         // (LOWORD(lParam)) horizontal postion, in screen coordinates
       WORD yPos )        // (HIWORD(lParam)) vertical postion, in screen coordinates
{ // reaction to WM_SYSCOMMAND message
  switch(uCmdType)
   {
    case IDM_CONNECT_METER_TO: // where to connect the "volume indicator" ?
         SndThd_iConnectMeterToOutput = !SndThd_iConnectMeterToOutput;
         return 0L;     // .. this message has been handled
    case IDM_FAST_AND_UGLY:  // toggles "fast-and-ugly"-mode (for speed test)
         UpsamplingAudioBuffer.SetFastMode(
           !UpsamplingAudioBuffer.GetFastMode() );
         return 0L;     // .. this message has been handled
    default:
         break;
   } // end switch (wParam) for our own additions to the system menu
  return 1L;  // message has not been handled
} // end OnWmSysCommand()




//---------------------------------------------------------------------------
LONG OnWmCreate(HWND hWnd, LPCREATESTRUCT lpcs)
{  // Handler for WM_CREATE message.
 #define C_EDIT_HEIGHT    20
 #define C_STATIC_TEXT_WIDTH 40 /* enough for strings like "fc/Hz" */
 #define C_BUTTON_HEIGHT  18
 #define C_VERT_SEPARATION 4
 #define C_HORZ_SEPARATION 4

   HMENU hMenu;
   RECT rect;
   int  iCtrlWidth, iCtrlHeight, iCtrlTop;

# if(0) // 2022-10-20 : moved the following to an "earlier place" ...
   // Define initial window 'show' state for this special application
   // See Win32 programmer's manual on ShowWindow()  for details.
   APPL_iWinCmdShow = SW_SHOWMINNOACTIVE;
   SOUND_SetDefaultParameters(); // default settings, some overridden by command line..
   fooling_shell   = FALSE;      // guess we DON'T need to "fool the shell"
   ParseCommandLine(APPL_sz255CmdLine);  // ..affects SOUND_WFXSettings
   if( fooling_shell ) // Call PostQuitMessage( 0 ) and return ?
    { PostQuitMessage( 0 ); // terminate THIS instance (not the new one launched via CreateProcess)
      return;
    }
# endif // set defaults and parse command line in OnWmCreate() ?


   // Append some line to the "window menu" alias "system menue" alias "control menu".
   // This only works properly in non-VCL applications.
   hMenu = GetSystemMenu( hWnd, FALSE/*revert*/ );
   // GetSystemMenu(hWnd, bRevert) :
   //    hWnd Identifies the window that will own a copy of the window menu.
   // bRevert Specifies the action to be taken. If this parameter is FALSE,
   //         GetSystemMenu returns the handle of the copy of the window menu
   //         currently in use. The copy is initially identical
   //         to the window menu, but it can be modified.
   AppendMenu( hMenu, MF_SEPARATOR, 0, NULL);
   AppendMenu( hMenu,          // handle to menu to be changed
          MF_STRING,           // UINT uFlags, menu-item flags
          IDM_CONNECT_METER_TO, // menu-item identifier or handle of drop-down menu or submenu
          "Connect peak meter to.."); // lpNewItem, menu-item content
   AppendMenu( hMenu, MF_STRING, IDM_FAST_AND_UGLY, "Fast && Ugly mode");



   GetClientRect(hWnd,&rect); // retrieve coordinates of a window's client area
   // get the area for the parameter display in the upper part of the window
   MainWnd_rParamDisplay.top = 0;
   MainWnd_rParamDisplay.bottom = rect.bottom
                   - (C_EDIT_HEIGHT + C_BUTTON_HEIGHT + 3*C_VERT_SEPARATION);
   MainWnd_rParamDisplay.left = 5;
   MainWnd_rParamDisplay.right= rect.right-5;

   // create edit fields for NCO- and "test-signal"- frequency :
   iCtrlHeight = C_EDIT_HEIGHT;   // height for EDIT FIELDS on lower part
   iCtrlWidth = (rect.right-C_HORZ_SEPARATION*C_HORZ_EDITS)/C_HORZ_EDITS;
   iCtrlTop   = rect.bottom - C_EDIT_HEIGHT - C_BUTTON_HEIGHT - 2*C_VERT_SEPARATION;
   CreateStaticText(hWnd, "fc/Hz", 0/*style*/, 2,iCtrlTop+2,C_STATIC_TEXT_WIDTH,C_EDIT_HEIGHT);
   MainWnd_hEditNcoFreq = CreateEditField(hWnd, "0", WS_BORDER, // text,style
      2+C_STATIC_TEXT_WIDTH+(C_HORZ_SEPARATION+iCtrlWidth)*0,iCtrlTop, // x,y
      iCtrlWidth-C_STATIC_TEXT_WIDTH, C_EDIT_HEIGHT);    // width, height
   CreateStaticText(hWnd, "ft/Hz", 0/*style*/,
      2+(C_HORZ_SEPARATION+iCtrlWidth)*1,iCtrlTop+2,
      C_STATIC_TEXT_WIDTH,C_EDIT_HEIGHT);
   MainWnd_hEditTestFreq = CreateEditField(hWnd, "0", WS_BORDER, // text,style
      2+C_STATIC_TEXT_WIDTH+(C_HORZ_SEPARATION+iCtrlWidth)*1,iCtrlTop, // x,y
      iCtrlWidth-C_STATIC_TEXT_WIDTH, C_EDIT_HEIGHT);    // width, height

   // create a few buttons at the bottom :
   iCtrlHeight = C_BUTTON_HEIGHT;   // height for BUTTONS
   iCtrlWidth = (rect.right-(C_HORZ_SEPARATION*C_HORZ_BUTTONS))/C_HORZ_BUTTONS;
   iCtrlTop   = rect.bottom - iCtrlHeight - C_VERT_SEPARATION;
   MainWnd_hRunStopButton = CreateButton(hWnd, "Run",
        BS_PUSHBUTTON,   // [in] dwStyleOptions
        CTRL_ID_BTN_RUN, // [in] iControlID
        2, iCtrlTop, iCtrlWidth , iCtrlHeight );  // x,y, width,height
   MainWnd_hHelpButton = CreateButton(hWnd, "Info", BS_PUSHBUTTON,
        CTRL_ID_BTN_HELP, // [in] iControlID
        2+(C_HORZ_SEPARATION+iCtrlWidth)*1,
        iCtrlTop,iCtrlWidth,iCtrlHeight );
   MainWnd_hGainCtrlButton = CreateButton(hWnd, "Vol", BS_PUSHBUTTON,
        CTRL_ID_BTN_GAIN, // [in] iControlID
        2+(C_HORZ_SEPARATION+iCtrlWidth)*2,
        iCtrlTop,iCtrlWidth,iCtrlHeight );
   MainWnd_hQuitButton = CreateButton(hWnd, "Quit", BS_PUSHBUTTON,
        CTRL_ID_BTN_QUIT, // [in] iControlID   
        2+(C_HORZ_SEPARATION+iCtrlWidth)*3,
        iCtrlTop,iCtrlWidth,iCtrlHeight );


   // Initialize application-specific stuff...
   memset(&SoundVol_ProcessInfo, 0, sizeof(SoundVol_ProcessInfo) );
 //  DeleteFile(SOUND_sz255InputFile);     // delete old file


   MainWnd_iTimerTicks = 0;
   SetTimer( // creates a timer with the specified time-out value
         hWnd,    // handle of window for timer messages
         0,       // timer identifier
         100,     // UINT uElapse = time-out value, in milliseconds
         NULL );  // lpTimerFunc  = address of timer procedure

   MainWnd_fRunning = SoundThd_LaunchWorkerThread();

   return 0L;
} // end OnWmCreate()


//---------------------------------------------------------------------------
void OnWmClose(HWND hWnd)
{
 int i;

  // Try to close the sndvol32.exe program.
  // All we know about sndvol32 is this info in a PROCESS_INFORMATION struct:
  if(SoundVol_ProcessInfo.hProcess)
   { // Beware: TerminateProcess() is extremely dangerous !
     // NO NO TerminateProcess(SoundVol_ProcessInfo.hProcess, 0); // too dangerous
     // One fine day we may be able to detect the main window's class name
     //   from the info in the PROCESS_INFORMATION struct, but how...
     //   EnumWindows(
     HWND hWnd=FindWindow(
        "sndvol32", // pointer to class name (not the name of the executable!)
             NULL); // pointer to window name
     if(hWnd!=NULL)
      {
        PostMessage(hWnd, WM_CLOSE, 0,0);
      }
     SoundVol_ProcessInfo.hProcess = 0;  // done.
   }

  // Close the worker thread for audio processing.
  SoundThd_TerminateAndDeleteThread();

  // If the window is not minimized, save the current window position in an
  // old-style INI-file (neither in a damned windoze directory nor in the damned registry)
  if((i=GetWindowPosX1(hWnd))>0)
     WriteIntegerToIniFile( APPL_sz255MyIniFile, "MainWin","x", i );
  if((i=GetWindowPosY1(hWnd))>0)
     WriteIntegerToIniFile( APPL_sz255MyIniFile, "MainWin","y", i );


} // end OnWmClose()


/*==============================================================================
       WndProc
==============================================================================*/
LRESULT CALLBACK _export WndProc(HWND hWnd, UINT message, UINT wParam, LONG lParam)
{
  switch (message)
  {
    case WM_CREATE: // sent when application requests that a window be created
      return OnWmCreate(hWnd, (LPCREATESTRUCT) lParam);
    case WM_PAINT:  // request to paint a portion of an application's window
      return OnWmPaint(hWnd);
    case WM_COMMAND: // handle button press etc etc etc
      return OnWmCommand(hWnd,
              HIWORD(wParam), // wNotifyCode: notification code
              LOWORD(wParam), // wID: item, control, or accelerator identifier
              (HWND) lParam); // hwndCtl:  handle of control
    case WM_SYSCOMMAND: // handle events from the "system menu" (alias "window menu")
      if (!OnWmSysCommand(hWnd,
              wParam,  // uCmdType = type of system command requested
              LOWORD(lParam),  // xPos = horizontal postion, in screen coordinates
              HIWORD(lParam))) // yPos = vertical postion, in screen coordinates
        return 0L;     // .. this message has been handled
      else
        return DefWindowProc (hWnd, message, wParam, lParam) ;
    case WM_INITMENU: // sent when a menu is about to become active (including the "system" menu)
      UpdateMyMenu();
      return 0L;             // .. this message has been handled

    case WM_CLOSE:
      OnWmClose(hWnd);
      DestroyWindow(hWnd); // perform wm_destroy
      return 0L;
    case WM_DESTROY:
      PostQuitMessage( 0 );
      return 0L;

    //-------------- optional message handlers ----------------------------
    case WM_TIMER:
      OnWmTimer(hWnd, (WORD)wParam/*wTimerID*/ ); // called on WM_TIMER
      return 0L;
    case WM_COPYDATA:
      return OnWmCopydata( hWnd,  // handle of THIS window
                   (HWND)wParam,  // handle of SENDING window
       (PCOPYDATASTRUCT)lParam);  // pointer to a COPYDATASTRUCT (read-only!)
  }
 return DefWindowProc (hWnd, message, wParam, lParam) ;
} // end WndProc()



//---------------------------------------------------------------------------
BOOL CheckBeforeWndCreation(void)
   // Called by the application skeleton BEFORE any window is created.
   // May be used to detect if another instance is already running.
   // Return value (evaluated somewhere in WinMain() ):
   //    FALSE = exit program, don't create a window, don't call OnWmCreate
   //    TRUE  = let me run, create the main window, then call OnWmCreate
{
 COPYDATASTRUCT cds;

 # if(1) // 2022-10-20 : moved the following HERE (previously in OnWmCreate() ) :
   // Define initial window 'show' state for this special application
   // See Win32 programmer's manual on ShowWindow()  for details.
   APPL_iWinCmdShow = SW_SHOWMINNOACTIVE;

   SOUND_SetDefaultParameters(); // default settings, some overridden by command line..
   g_fUseAudioFile = FALSE;      // used in conjunction with /of (output file)
   g_fUseAudioMessages = FALSE;  // used in conjunction with /sendto
   fooling_shell   = FALSE;      // guess we DON'T need to "fool the shell"
   ParseCommandLine(APPL_sz255CmdLine);  // ..affects SOUND_WFXSettings
   if( fooling_shell ) // Immediately return (instead of letting THIS instance "run") ?
    { // PostQuitMessage( 0 );  // terminate THIS instance (not the new one launched via CreateProcess)
      return FALSE; // exit program, don't create a new window (and program instance), don't call OnWmCreate
    }
# endif // set defaults and parse command line in CheckBeforeWndCreation() ?

  // Check if "this" program is already running somewhere in another instance.
  // If so, DO NOT start a second instance because the soundcard will already
  // be in use !
  HWND hWnd=FindWindow(APPL_szWindowName,NULL);
  if(hWnd!=NULL)
   { // oops.. "I" am already running on this system !
     // PostMessage(hWnd, WM_SETFOCUS, 0,0); // no
     ShowWindow( hWnd, APPL_iWinCmdShow );
     BringWindowToTop(hWnd);

     // Try to send the command line which was directed to THIS instance
     // to the first started instance of this program, hope it can handle it..
     // The WM_COPYDATA message is sent when an application passes data
     // to another application. Fill out a COPYDATASTRUCT ...
     cds.dwData = FOUR_CHARS_TO_DW('c','m','d','0');  // See OnWmCopydata()
     cds.cbData = strlen(APPL_sz255CmdLine);
     cds.lpData = (PVOID)APPL_sz255CmdLine;
     // An application must use the SendMessage function to send this message,
     // not the PostMessage function ...
     SendMessage( hWnd, WM_COPYDATA, 0/*I have no "window"*/,
                         (LPARAM)&cds);
     return FALSE; // don't create a window for this instance, TERMINATE ME PLEASE
   }

  return TRUE;  // ok, create a window and a message loop for this application
} // end CheckBeforeWndCreation()



// EOF <SndOutMain.cpp>



