/***************************************************************************/
/* C:\cbproj\SoundUtl\SndInThd.cpp =  Sound Input Thread.                  */
/*   - Receives continuous input from the soundcard                        */
/*       and places the sound samples in two alternating buffers.          */
/*   - requires "SOUND.CPP" ( CSound class )  to access the audio device   */
/*   - by DL4YHF,  September 2000  -  September 2020                       */
/*   - VCL dependencies removed in May 2002 to keep the code compact (!)   */
/***************************************************************************/

//---------------------------------------------------------------------------
#include <windows.h>   // dont use the VCL here !!
#include <stdio.h>
#include <stdlib.h>

#pragma hdrstop


#define _I_AM_SOUND_THREAD_ 1  // for "single-source" variables of this module
#include "SndInThd.h"

#pragma warn -8071

/*-------------- constants, tables ---------------------------------------*/
#define SNDTHD_EXIT_OK  1
#define SNDTHD_EXIT_BAD 2

/*-------------- Variables -----------------------------------------------*/

// Some parameters which may be 'overwritten' by command line arguments...
int     SOUND_iNrOfAudioChannels= 1;
T_Float SOUND_fltTestGeneratorFreq=0; // '0' means generator is disabled
T_Float SOUND_fltGainFactor     = 1.00;
int     SOUND_iUseSignedValues  = 1;  // 0=no 1=yes
long    SOUND_lDecimationRatio  = 1;
T_Float SOUND_fltCenterFrequency= 0;
int     SOUND_fComplexOutput    = 0;     // 0=real, 1=complex output
T_Float SOUND_fltAudioPeakValue[2] = {0,0}; // crude peak detector for display purposes,
                                            // ranging from zero to one (normalized).

// input- and output device settings...
WAVEFORMATEX SOUND_WFXSettings;  // once located in sound.cpp, now in this worker thread

// An instance of the CSound class to access the soundcards...
CSound SoundDev;

// A large "decimating" buffer where the samples from the soundcard are stored
CSoundDecimatingBuffer DecimatingAudioBuffer;

// Sample buffers used to read mono- or stereo-samples from the input
//  NOTE: Since 2006, data for multiple channels are not interleaved anymore,
//        instead there are two separate chunks .
T_Float SndThd_fltInputChunk[2][SOUND_MAX_CHUNK_SIZE]; // sample blocks, value range +/- 1.0 ("normalized")

// Other stuff...
int SndThd_iConnectMeterToOutput = 0; // 1=meter on output, 0=meter on input
int SndThd_iChunkSize=4096; // size of a processing 'chunk' [Samples], limited to SOUND_MAX_CHUNK_SIZE somewhere
int SOUND_iDcReject   = 1;
T_Float SOUND_fltDcOffset[2] = {0,0};



/*------------- internal(!) variables ------------------------------------*/
BOOL   SndThd_fTerminateRequest;
HANDLE SndThd_hWorkerThread;    // Win32 API handle for the worker thread
DWORD  SndThd_dwWorkerThreadId; // Identifier for the worker thread

//------------- Implementation of routines and classes ----------------------

//---------------------------------------------------------------------------
void SOUND_SetDefaultParameters(void)
  // Prepare some soundcard settings (WAVEFORMATEX) required later.
  // Note: some of these settings may be overridden by the command line parser.
{
  // Preset some soundcard settings (WAVEFORMATEX) required for opening...
  SOUND_WFXSettings.nChannels = SOUND_iNrOfAudioChannels = 1;
  SOUND_WFXSettings.wBitsPerSample = 16;  // ALWAYS 16 BIT, EVEN WHEN EXPORTING 8-BIT-VALUES(!!)
  SOUND_WFXSettings.nSamplesPerSec = 11025;

  SOUND_WFXSettings.wFormatTag = WAVE_FORMAT_PCM;
  SOUND_WFXSettings.nBlockAlign =
      SOUND_WFXSettings.nChannels *(SOUND_WFXSettings.wBitsPerSample/8);
  SOUND_WFXSettings.nAvgBytesPerSec =
      SOUND_WFXSettings.nSamplesPerSec * SOUND_WFXSettings.nBlockAlign;
  SOUND_WFXSettings.cbSize = 0;
} // end SOUND_SetDefaultParameters()

//---------------------------------------------------------------------------
void SOUND_AddSineWave(T_Float *pfltDest, int iLength)
  // Produce some crude test data for the DAC:
  //    [in] SoundTab_fltCosTable[SOUND_COS_TABLE_LEN]
  // Note: In the "Sound Input Utility", pfltDest (for SOUND_AddSineWave)
  //       does not contain MULTIPLE CHANNELS (per "sample point"),
  //       but only a single audio channel. Thus, a single call of
  //       SOUND_AddSineWave() cannot generate a "stereo tone" !
{
  static T_Float sfltVcoPhase = 0.0;
  int    i,j;
  T_Float d;
  T_Float fltFactor = 0.5/*-6dB*/ ;
  T_Float fltPhaseInc = SOUND_fltTestGeneratorFreq * (T_Float)SOUND_COS_TABLE_LEN / (T_Float)SOUND_WFXSettings.nSamplesPerSec;

   for(i=0; i<iLength; ++i)
    {
        sfltVcoPhase += fltPhaseInc;
        if(sfltVcoPhase >= SOUND_COS_TABLE_LEN)
         { sfltVcoPhase -= SOUND_COS_TABLE_LEN;
         }
        j = (int)sfltVcoPhase;
        if(j<0)
         { j=0;
         }
        if(j>=SOUND_COS_TABLE_LEN)
         { j-=SOUND_COS_TABLE_LEN;
         }
        d = fltFactor * SoundTab_fltCosTable[ j ];
        //d = 1024 * (i&4);
        (*pfltDest++) += d;
        // No-No: if(SOUND_WFXSettings.nChannels>1) // removed 2022-10-19
        //         { (*pfltDest++) += d;
        //         }
    }
} // end SOUND_AddSineWave()



/*------------ Implementation of the Sound Thread -------------------------*/


//---------------------------------------------------------------------------
DWORD WINAPI SoundThdFunc( LPVOID )
  /* Executes the sound analyzer thread.
   *   Was once the "Execute" method of a VCL Thread object,
   *   but this unit does not use the VCL any longer since May 2002.
   */
{
 int i,iMax,iChn,error_code,iConversionMode;
 T_Float d,p;
 T_Float fltDcFactor;
 T_ChunkInfo chunk_info;

 // ex: int iSoundInputDeviceID  = -1;  // -1 means "the default audio input device"
 int iSoundOutputDeviceID = -1;

  if( SndThd_iChunkSize > SOUND_MAX_CHUNK_SIZE )
      SndThd_iChunkSize = SOUND_MAX_CHUNK_SIZE;

  ChunkInfo_Init( &chunk_info );

  // Open Sound Device for input
  error_code=SoundDev.InOpen(  // ex: iSoundInputDeviceID, // often "-1"
                g_sz255DeviceName,  // [in] char *pszAudioInputDeviceOrDriver,
                                    // name of an audio device or DL4YHF's "Audio I/O driver DLL" (full path)
                "", // [in] pszAudioInputDriverParamStr; an extra parameter string for driver or Audio-I/O / ExtIO-DLL
                "", // [in] pszAudioStreamID; a 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE (here: optional)
                "SndInput", // [in] pszDescription; a descriptive name which identifies this 'audio sink'
                            // on a kind of "patch panel" on the configuration screen of DL4YHF's Audio-I/O DLL .
                // ex: &SOUND_WFXSettings, // former input for InOpen, windoze-specific crap (WAVEFORMATEX),
                // replaced by i32SamplesPerSecond, iBitsPerSample, iNumberOfChannels :
                SOUND_WFXSettings.nSamplesPerSec, // [in] i32SamplesPerSecond : nominal sampling rate
                SOUND_WFXSettings.wBitsPerSample, // [in] iBitsPerSample ..per channel, usually 16 (also for stereo, etc)
                SOUND_WFXSettings.nChannels, // [in] iNumberOfChannels : here only 1 or 2 channels (mono/stereo)
                SndThd_iChunkSize, // [in] dwMinBufSize : number of SAMPLES per buffer expected by the CALLER
                0, // [in] sample limit : 0 signifies continuous input stream
                FALSE); // [in] BOOL start : FALSE don't start immediately

  if( error_code != NO_ERROR )
   {
     SOUND_iErrorCode = error_code;
     ExitThread( SNDTHD_EXIT_BAD ); // exit code for this thread
   }


  // Open Sound Device for output if necessary.
  //  Note: The same WFX-Settings must be used for InOpen() and OutOpen() !
/*
  error_code = SoundDev.OutOpen( m_iSoundOutputDeviceID,
                &SOUND_WFXSettings,SndThd_iChunkSize,0,
                FALSE ); // don't start immediately
  if( error_code != NO_ERROR )
      SOUND_iErrorCode = error_code;
/**/


  SOUND_fltDcOffset[0]=SOUND_fltDcOffset[1]=0.0;


  SoundDev.Start();   // start input & output synchronously...



  // SOUND_thd_error_buf_in = SOUND_thd_error_buf_out = 0; // NO !!
  SOUND_exited_from_thread_loop = false;


  /* Almost endless thread loop begins HERE................................*/
  while( !SndThd_fTerminateRequest )
   {
     if( SoundDev.IsInputOpen() )
      {
       // Get input data from soundcard object (or whatever imitates a 'soundcard')
       if( SoundDev.InReadStereo(
             SndThd_fltInputChunk[0], // [out] left channel, sample value range +/- 1.0 ("normalized")
             SndThd_fltInputChunk[1], // [out] right channel, sample value range +/- 1.0 ("normalized")
             SndThd_iChunkSize,       // [in]  number of samples per "chunk"
             300/*iTimeout_ms*/,     // 2020-09-26 : increased timeout from 50 to 300 ms.
             NULL/*piHaveWaited_ms*/, NULL/*pOutChunkInfo*/ ) != SndThd_iChunkSize )
        {
         error_code = SoundDev.m_ErrorCodeIn;
         if (error_code!=NO_ERRORS)
             SOUND_iErrorCode = error_code;

         switch(error_code)
          {
          /* waveform audio error return values from MMSYSTEM:    */
          case WAVERR_BADFORMAT   :   /* unsupported wave format  */
          case WAVERR_STILLPLAYING:   /* still something playing  */
          case WAVERR_UNPREPARED  :   /* header not prepared      */
          case WAVERR_SYNC        :   /* device is synchronous    */
            //SoundDev.InClose();  // what an ugly solution: close & open again..
            //error_code=SoundDev.InOpen(..
            break;
          /* error codes from ErrorCodes.H :                      */
          case NO_ERRORS:
            break;     // hmmm... so why did InRead() fail ??
          case SOUNDIN_ERR_NOTOPEN:
             // someone must have closed the soundcard's input ?
             // NO BREAK HERE
          case SOUNDIN_ERR_OVERFLOW:
          case SOUNDIN_ERR_TIMEOUT:
            // cpu couldn't keep up so try again: Close and re-open Sound Device.
            // 2020-09-26 : Periodically got here with error_code = 2114 = SOUNDIN_ERR_TIMEOUT
            //              when the input device was an Icom IC-9700 ("2- USB Audio CODEC").
            //
            //
            if( error_code != SOUNDIN_ERR_NOTOPEN )
             { // for "OVERFLOW" and "TIMEOUT" : try to kick the damned thing
               // alive again, by closing and re-opening the device
               SoundDev.InClose();
             }
            SoundDev.InOpen(
                g_sz255DeviceName,  // [in] char *pszAudioInputDeviceOrDriver,
                                    // name of an audio device or DL4YHF's "Audio I/O driver DLL" (full path)
                "", // [in] pszAudioInputDriverParamStr; an extra parameter string for driver or Audio-I/O / ExtIO-DLL
                "", // [in] pszAudioStreamID; a 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE (here: optional)
                "SndInput", // [in] pszDescription; a descriptive name which identifies this 'audio sink'
                            // on a kind of "patch panel" on the configuration screen of DL4YHF's Audio-I/O DLL .
                // ex: &SOUND_WFXSettings, // former input for InOpen, windoze-specific crap (WAVEFORMATEX),
                // replaced by i32SamplesPerSecond, iBitsPerSample, iNumberOfChannels :
                SOUND_WFXSettings.nSamplesPerSec, // [in] i32SamplesPerSecond : nominal sampling rate
                SOUND_WFXSettings.wBitsPerSample, // [in] iBitsPerSample ..per channel, usually 16 (also for stereo, etc)
                SOUND_WFXSettings.nChannels, // [in] iNumberOfChannels : here only 1 or 2 channels (mono/stereo)
                SndThd_iChunkSize, // [in] dwMinBufSize : number of SAMPLES per buffer expected by the CALLER
                0, // [in] sample limit : 0 signifies continuous input stream
                TRUE); // [in] BOOL start : TRUE to start immediately
            break;
      default:
            // don't know how to handle this error, set breakpoint here:
            break;
          } // end switch(error_code)
        } /* end if (..InRead(..) != ..BUF_SIZE) */
       else
        { /* everything looks beautiful :-) */
         error_code = SoundDev.m_ErrorCodeIn; // hmm..
         if (error_code!=NO_ERRORS) // reading ok, but something is wrong...
          {
           SOUND_iErrorCode = error_code;
           SoundDev.m_ErrorCodeIn = NO_ERRORS;  // ouch !!!!!!!!
          }
        } /* end if <sampling of input data ok> */
      } // end if( SoundDev.IsInputOpen() )
     else    // ! IsInputOpen() ... ->
      {  // Re-Open Sound Device for input
        error_code = SoundDev.InOpen(
                g_sz255DeviceName,  // [in] char *pszAudioInputDeviceOrDriver,
                                    // name of an audio device or DL4YHF's "Audio I/O driver DLL" (full path)
                "", // [in] pszAudioInputDriverParamStr; an extra parameter string for driver or Audio-I/O / ExtIO-DLL
                "", // [in] pszAudioStreamID; a 'Stream ID' for the Audio-I/O-DLL, identifies the AUDIO SOURCE (here: optional)
                "SndInput", // [in] pszDescription; a descriptive name which identifies this 'audio sink'
                            // on a kind of "patch panel" on the configuration screen of DL4YHF's Audio-I/O DLL .
                // ex: &SOUND_WFXSettings, // former input for InOpen, windoze-specific crap (WAVEFORMATEX),
                // replaced by i32SamplesPerSecond, iBitsPerSample, iNumberOfChannels :
                SOUND_WFXSettings.nSamplesPerSec, // [in] i32SamplesPerSecond : nominal sampling rate
                SOUND_WFXSettings.wBitsPerSample, // [in] iBitsPerSample ..per channel, usually 16 (also for stereo, etc)
                SOUND_WFXSettings.nChannels, // [in] iNumberOfChannels : here only 1 or 2 channels (mono/stereo)
                SndThd_iChunkSize, // [in] dwMinBufSize : number of SAMPLES per buffer expected by the CALLER
                0, // [in] sample limit : 0 signifies continuous input stream
                TRUE); // [in] BOOL start : TRUE to start immediately
        if( error_code != NO_ERROR )
         {
           SOUND_iErrorCode = error_code;
           ExitThread( SNDTHD_EXIT_BAD ); // exit code for this thread
         }
      } // end if <sound device NOT open for input (?!)


    if(SOUND_fltTestGeneratorFreq>0)
     { // make a test signal (instead of file input)
       SOUND_AddSineWave(SndThd_fltInputChunk[0], SndThd_iChunkSize);
     }

    // Factor for DC offset integration:
    //  After 1 second, 50% decay of the DC component..
    fltDcFactor = 0.5 / (T_Float)SOUND_WFXSettings.nSamplesPerSec;

    for(iChn=0; iChn<=1 && iChn<SOUND_WFXSettings.nChannels; ++iChn)
     { T_Float *pfltSrc = SndThd_fltInputChunk[iChn]; // [in] samples with value range +/- 1.0 ("normalized")
       p = 0;
       iMax = ((SndThd_iChunkSize-1)*SOUND_WFXSettings.nChannels)+iChn;
       for(i=iChn; i<=iMax; i+=SOUND_WFXSettings.nChannels)
        {
          d = pfltSrc[i];

          // Optional removal of DC components..
          if(SOUND_iDcReject)
           {
             d -= SOUND_fltDcOffset[iChn];
             SOUND_fltDcOffset[iChn] += fltDcFactor * d;
             pfltSrc[i] = d;
           } // end if(SOUND_iDcReject)

          // Crude peak detector for display purposes..
          if(d>0) { if(d>p)   p=d;  }    // full wave recifier + peak detector
            else  { if(d<-p)  p=-d; }
        }
       if(! SndThd_iConnectMeterToOutput )
        { SOUND_fltAudioPeakValue[iChn] = p; // take the rectified "peak value" from the INPUT
          // 2022-10-17 : With a severely overdriven UMC404HD, the "peak value"
          //              was slightly ABOVE ONE !
        }
      } // end for <all AUDIO channels>

     /* Put the data into the decimating output-buffer: */
     chunk_info.dwNrOfSamplePoints = SndThd_iChunkSize;
     chunk_info.nChannelsPerSample = SOUND_WFXSettings.nChannels;
     chunk_info.dblPrecSamplingRate= SOUND_WFXSettings.nSamplesPerSec;
     if( SOUND_fComplexOutput )
      { iConversionMode = SNDDEC_MODE_REAL_IN_COMPLEX_OUT;
      }
     else // ! SOUND_fComplexOutput ...
      { iConversionMode = SNDDEC_MODE_REAL_IN_REAL_OUT;
      }
     DecimatingAudioBuffer.EnterSamples(
         SndThd_fltInputChunk[0], // pointer to LEFT SOURCE CHANNEL
         SndThd_fltInputChunk[1], // pointer to RIGHT SOURCE CHANNEL
         iConversionMode, // [in] SNDDEC_MODE_xxx_IN_yyy_OUT (xxx,yyy = REAL or COMPLEX)
         &chunk_info);    // [in] T_ChunkInfo* with  #samples, precise sample rate, date+time, "radio" frequency, GPS, ...
   }  /* end <almost endless thread loop> */

  /* if the program ever gets here, the thread is quite "polite"... */
  /*                    or someone has pulled the emergency brake ! */
  SOUND_exited_from_thread_loop = true;

  ExitThread( SNDTHD_EXIT_OK ); // exit code for this thread
     // Note: The code STILL_ACTIVE must not be used as parameter for
     // ExitThread(), but what else ? See Win32 API help !
     // in winbase.h:
     //    #define STILL_ACTIVE STATUS_PENDING
     // in winnt.h:
     //    #define STATUS_PENDING ((DWORD   )0x00000103L)
  return SNDTHD_EXIT_OK; // will this ever be reached after "ExitThread" ?
} // end SoundThdFunc()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
BOOL SoundThd_LaunchWorkerThread(void)
   // Kick the real-time sound processing thread to life,
   //  without the help of Borland's Visual Component Library (VCL).
{

  // Make sure the cosine table exists, the frequency converter (NCO) may need it
  SoundTab_InitCosTable();

  // NO-NO: SOUND_SetDefaultParameters();  // called done by application before evaluating command line
  // Prepare some 'derived' soundcard settings (WAVEFORMATEX) required for opening.
  // Note: The commented settings have already been set in SetDefaultParameters():
  // SOUND_WFXSettings.nChannels      = 1;
  // SOUND_WFXSettings.wBitsPerSample = 16;
  // SOUND_WFXSettings.nSamplesPerSec = 11025;
  SOUND_WFXSettings.wFormatTag = WAVE_FORMAT_PCM;
  SOUND_WFXSettings.nBlockAlign =
      SOUND_WFXSettings.nChannels *(SOUND_WFXSettings.wBitsPerSample/8);
  SOUND_WFXSettings.nAvgBytesPerSec =
      SOUND_WFXSettings.nSamplesPerSec * SOUND_WFXSettings.nBlockAlign;
  SOUND_WFXSettings.cbSize = 0;


  // Initialize a 'decimating' buffer which is used to pass the samples
  // from the audio device to the application,
  // with the option to decimate the sample rate (et al) .
  if(SOUND_lDecimationRatio<1)
     SOUND_lDecimationRatio=1;
  DecimatingAudioBuffer.Init(
     SOUND_WFXSettings.nSamplesPerSec  // lBufferLength, max count of sample points (!), here: 1 second of data
          / SOUND_lDecimationRatio,
     SOUND_WFXSettings.nSamplesPerSec, // fltInputSampleRate, often 11025, 22050 or 44100
     SOUND_lDecimationRatio,           // lSampleRateDivisor, powers of two preferred
     SOUND_fltCenterFrequency,         // fltMixerFrequency,  "NCO" frequency in Hertz
     SOUND_fComplexOutput,          // fComplex:  TRUE = with frequency shift + complex output
     SOUND_WFXSettings.nChannels ); // iNrChannels: 1=mono, 2=stereo processing


  // Now create the real-time worker thread
  SndThd_fTerminateRequest = FALSE;
  SndThd_hWorkerThread = CreateThread(
     NULL,    // LPSECURITY_ATTRIBUTES lpThreadAttributes = pointer to thread security attributes
     0,       // DWORD dwStackSize  = initial thread stack size, in bytes
     SoundThdFunc, // LPTHREAD_START_ROUTINE lpStartAddress = pointer to thread function
     0,       // LPVOID lpParameter = argument for new thread
     0,       // DWORD dwCreationFlags = creation flags
              // zero -> the thread runs immediately after creation
     &SndThd_dwWorkerThreadId // LPDWORD lpThreadId = pointer to returned thread id
    );
  // The thread object remains in the system until the thread has terminated
  // and all handles to it have been closed through a call to CloseHandle.

  if(!SndThd_hWorkerThread)
   { // CreateThread returned NULL, thread could not be created
     return FALSE;
   }

  // Define the Sound Thread's priority as required.
  SetThreadPriority(
    SndThd_hWorkerThread,          // handle to the thread
    THREAD_PRIORITY_ABOVE_NORMAL); // thread priority level
    
  return TRUE;
} // end SoundThd_LaunchWorkerThread()


/***************************************************************************/
BOOL SoundThd_TerminateAndDeleteThread(void)
   /* Terminates the sound thread, cleans up
    * and deletes(!) the sound thread.
    */
{
  int i;
  DWORD dwExitCode;
  BOOL  fOk = TRUE;

  if(SndThd_hWorkerThread)
   {
     // First, politely try to let the sound thread terminate itself
     //        allowing it to clean up everything.
     // About GetExitCodeThread(HANDLE hThread,LPDWORD lpExitCode):
     //       lpExitCode: Points to a 32-bit variable
     //          to receive the thread termination status.
     //       If the function succeeds, the return value is nonzero.
     //       If the specified thread has not terminated,
     //          the termination status returned is STILL_ACTIVE.
     dwExitCode=STILL_ACTIVE;
     SndThd_fTerminateRequest = TRUE;
     for(i=0;
        (i<20) && (GetExitCodeThread(SndThd_hWorkerThread,&dwExitCode));
         ++i)
      {
        // MessageBeep(-1);  // drums please
        Sleep(50);  // wait a little longer for the thread to terminate itself
        if(dwExitCode==SNDTHD_EXIT_OK)
           break;
      }

     // Before deleting the sound thread...
     if(SoundDev.IsInputOpen())
        SoundDev.InClose();
     if(SoundDev.IsOutputOpen())
        SoundDev.OutClose();

     if(dwExitCode==STILL_ACTIVE)
      { // dangerous termination of the sound thread:
        TerminateThread(SndThd_hWorkerThread, // handle to the thread
                        SNDTHD_EXIT_BAD);     // exit code for the thread
      }

     CloseHandle(SndThd_hWorkerThread);
     SndThd_hWorkerThread = NULL;
     fOk &= (dwExitCode==SNDTHD_EXIT_OK);  // TRUE = polite ending,   FALSE = something went wrong.
   }

  return fOk;
} // end SOUND_TerminateAndDeleteThread()

