/****************************************************************************/
/* \cbproj\yhf_tools\QFile.h :  "Quick File" helper routines                */
/*                                                                          */
/*               NO APPLICATION-SPECIFIC STUFF IN HERE !                    */
/*               This unit is used in a number of projects (see history).   */
/*                                                                          */
/* Author and Copyright by:                                                 */
/*   Wolfgang Buescher (DL4YHF)                                             */
/*   Use of this sourcecode for commercial purposes strictly forbidden !    */
/*                                                                          */
/* Revision History:                                                        */
/*    see   *.c                                                             */
/*        2006-03-23:  Added the "QFILE_SEEK_xxx"-macros in QFile.h .       */
/*        2007-04-25:  Modified QFile_ReadLine() to support DOS- and UNIX-  */
/*                     textfiles: DOS uses CR+LF, UNIX only LF as line end. */
/*        2007-07-08:  Fixed a bug in QFile_ReadLine(): CR-LF at buffer     */
/*                     boundaries caused reading an additional empty line.  */
/*        2010-06-21:  Checking for the UTF-8 'BOM' if a file is read       */
/*                     using QFile_ReadLine() , i.e. when reading the file  */
/*                     'as a sequence of TEXT LINES'. If the file begins    */
/*                     with the infamous UTF-8 'BOM' (0xEF 0xBB 0xBF),      */
/*                     QFile_ReadLine() will try to replace UTF-8 sequences */
/*                     with simple 8-bit ANSI characters, by calling the    */
/*                     new API function QFile_UnicodeTo8BitANSI() .         */
/*                                                                          */
/****************************************************************************/

#ifndef _QFILE_H_
#define _QFILE_H_

#include <windows.h>    // only required for data types like BYTE, DWORD, etc
#include <io.h>         // some file DEFINITIONS

//---------- Constants ------------------------------------------------------

// Possible values for 'oflags' in QFile_Open .  Must be compatible with fcntl.h !
// Declared here so other modules don't need to include dozens of headers .
#define QFILE_O_RDONLY    0
#define QFILE_O_WRONLY    1
#define QFILE_O_RDWR      2
#define QFILE_O_MASK_RDWR 0x03    // mask to isolate only the above flags
#define QFILE_O_CREATE    0x4000  // IF THE FILE DOESN'T EXIST, create it in QFile_Open()
                                  // (note that "create" isn't the same as "truncate")
#define QFILE_O_SHARED    0x8000  // for QFile_Open(..oflags), allow SHARED access (since 2008-12)
#define QFILE_ATTR_SHARED 0x8000  // for QFile_Create(..attrib), allow SHARED access (since 2008-12)
#define QFILE_ATTR_NORMAL 0x0000

#define QFILE_SEEK_SET  0  /* Positionierung vom Dateianfang aus */
#define QFILE_SEEK_CUR  1  /* Positionierung von der aktuellen Position aus */
#define QFILE_SEEK_END  2  /* Positionierung vom Dateiende aus */


//ex: #define QFILE_BUFFER_SIZE 2048  // must be a power of two !
#define QFILE_BUFFER_SIZE 65536 // must be a power of two !
                 // 2008-03-17 : increased QFILE_BUFFER_SIZE from 2048 to 65536,
                 // to see if this would speed up the analysis of very large wave files
                 // in Spectrum Lab.  Result:  CRASHED !
                 // Rebuilt the entire project -> works (stupid MAKE ? ),
                 //  but not faster than the original code with 2048-byte-buffer.

//---------- Data types (no C++ classes!) -----------------------------------
typedef struct // T_QFile (for "Quick file access")
{
  int   iHandle;
  char  sz512PathAndName[514];
  BYTE  bBuffer[QFILE_BUFFER_SIZE];
  BYTE  bPrevCharFromReadLine; // added 2007-07-08 to detect end of text lines properly
  BOOL  fBufferModified;
  DWORD dwBufferStartPos,dwPrevBufferStartPos; // important for QFile_Seek()
  DWORD dwBufferIndex;   // buffer index for the next read/write access
  DWORD dwNumUsedBytesInBuffer;   // count of "used" bytes in the buffer
  DWORD dwLastError;     // if there was an error, the error code is here
  BOOL  fOpened;
  int   iEncoding;   // basic character encoding, added 2010-06-21, one of these:
#       define QFILE_ENCODING_NONE  0
#       define QFILE_ENCODING_UTF8  8
  int   iUTF8_DecoderState;  // internal, simplistic, UTF-8 decoder state
  char  c4_UTF8_DecoderSequence[4];
} T_QFile;


//------------ Prototypes -------------------------------------------------

#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  /* nicht "cplusplus" */

/***************************************************************************/
CPROT void QFile_LastErrorCodeToString(DWORD dwError, char *dest_str, int maxlen);

/***************************************************************************/
CPROT BYTE QFile_UnicodeTo8BitANSI( DWORD dwUnicode ); // actually to 'ISO Latin-1'

/***************************************************************************/
CPROT BOOL QFile_Open( T_QFile *pqFile, const char *filename, int oflags);
  // Opens a file.  Parameters similar to _rtl_open :
  //   QFILE_O_RDONLY , QFILE_O_WRONLY ,  or   QFILE_O_RDWR  .

/***************************************************************************/
CPROT BOOL QFile_IsOpen( T_QFile *pqFile );

/***************************************************************************/
CPROT void QFile_Close( T_QFile *pqFile );
  // Closes a file.

/***************************************************************************/
CPROT long QFile_Seek( T_QFile *pqFile, long offset, int fromwhere);
  // Works like lseek on a 'buffered' file.
  // QFile_Seek(  pqFile, 0, SEEK_CUR )  returns the current file position .

/***************************************************************************/
CPROT int QFile_Read( T_QFile *pqFile, void *buf, DWORD len);
  // Reads a few bytes from the file, or -if possible- from a buffer

/***************************************************************************/
CPROT int  QFile_ReadLine( T_QFile *pqFile, char *pszDest, int iMaxLen );
  // Reads a single text line from a file,  up to the CR / NL terminator.
  // The [CR /] NL terminator will be skipped but not entered in pszDest .
  // Return value:  > 0 = Count of characters,
  //                  ZERO if an empty line has been read,
  //               or NEGATIVE if there was an access error or end of file.


/***************************************************************************/
CPROT BOOL QFile_Create( T_QFile *pqFile, const char *filename, int attrib);
  // Creates a new file (or truncates existing). Parameters similar to _rtl_creat:
  // attrib=0 is a "normal" file (neither hidden nor something else)

/***************************************************************************/
CPROT BOOL QFile_Write( T_QFile *pqFile, BYTE *pbSrc, long i32CountOfBytes );


/***************************************************************************/
CPROT BOOL QFile_WriteString( T_QFile *pqFile, char *pszSrc );
   // Writes a zero-terminated string.
   // Counts the characters, but DOES NOT APPEND CR/NL by itself .


//--------------------------------------------------------------------------
//  String handling routines .   Some used when reading TEXT FILES .
//--------------------------------------------------------------------------
CPROT char *QFile_GetFilenameWithoutPath( char *pszFilenameWithPath );
CPROT int  QFile_SkipSpaces(char **ppcSource ); // actually skips spaces and TABS
CPROT BOOL QFile_SkipChar(char **ppcSource, char cCharToSkip );
CPROT BOOL QFile_SkipString(char **ppcSource, char *pszStringToSkip);
CPROT int  QFile_SkipStringFromList(char **ppcSource, char *pszzStringList);
CPROT long QFile_ParseInteger(char **ppcSource, int ndigits, int radix, long deflt);
CPROT double QFile_ParseDFloat(char **ppcSource, double dblDefault);
CPROT BOOL QFile_ParseQuotedString(char **ppcSource, char *pszDest, int iMaxLen );
CPROT void QFile_CopyStringUntilEndOfLine(char **ppcSource, char *pszDest, int iMaxLen );

#endif // _QFILE_H_
