/****************************************************************************/
/* \cbproj\yhf_tools\QFile.c:  "Quick File" helper routines                 */
/*                                                                          */
/*               NO APPLICATION-SPECIFIC STUFF IN HERE !                    */
/*               This unit is used in a number of projects (see history).   */
/*                                                                          */
/* Author and Copyright by:                                                 */
/*   Wolfgang Buescher (DL4YHF)                                             */
/*   Use of this sourcecode for commercial purposes strictly forbidden !    */
/*                                                                          */
/* Revision History:                                                        */
/*  V1.0, 2002-03-13:  Created for DL4YHF's "Spectrum Lab".                 */
/*  V1.1, 2003-09-25:  Used on the QRL in the CAN-Logger-Utility .          */
/*  V1.2, 2003-12-15:  Copied ..LastErrorCodeToString() into this module .  */
/*  V1.3, 2004-11-21:  Also used in module "YHF_MultiLang.cpp" now .        */
/*  V1.4, 2005-07-22:  Fixed a bug in QFile_Seek()                          */
/*        2006-03-20:  Module used in project "GPS-Player" now .            */
/*        2006-03-23:  Added the "QFILE_SEEK_xxx"-macros in QFile.h .       */
/*  V1.5, 2006-04-02:  Optionally NOT using the buffer, if the file is read */
/*                     in large chunks (for example the GTOPO30 database) . */
/*  V1.6, 2006-09-04:  Added QFile_ParseInteger(), used in YHF_graf .       */
/*  V1.7, 2007-04-25:  Modified QFile_ReadLine() to support DOS- and UNIX-  */
/*                     textfiles: DOS uses CR+LF, UNIX only LF as line end. */
/*        2007-07-08:  Fixed a bug in QFile_ReadLine(): CR-LF at buffer     */
/*                     boundaries caused reading an additional empty line.  */
/*        2007-11-26:  Fixed another bug in QFile_Read() :                  */
/*                      Unneccessary attempt to read past the end-of-file,  */
/*                      if the file was read exactly in multiples of the    */
/*                      internal buffer size .                              */
/*        2008-03-20:  Fixed a bug in QFile_Flush(), and increased the      */
/*                     internal buffer size (to minimize the number of      */
/*                     OS calls when saving wave files in SpecLab) .        */
/*  V1.8, 2010-06-21:  Checking for the UTF-8 'BOM' if a file is read       */
/*                     using QFile_ReadLine() , i.e. when reading the file  */
/*                     'as a sequence of TEXT LINES'. If the file begins    */
/*                     with the infamous UTF-8 'BOM' (0xEF 0xBB 0xBF),      */
/*                     QFile_ReadLine() will try to replace UTF-8 sequences */
/*                     with simple 8-bit ANSI characters, by calling the    */
/*                     new API function QFile_UnicodeTo8BitANSI() .         */
/****************************************************************************/

#pragma hdrstop  // don't use precompiled header (due to the #ifdef sphagetti)

#include "io.h"      // some strange header for file access functions
#include "fcntl.h"   // O_RDWR and other stuff (which is missing in IO.H etc)
#include "share.h"   // bitmasks for shared files, except S_IREAD + S_IWRITE
#include "sys/stat.h" // even more damned headers for file access

#include "QFile.h"   // header for this module


//-------- Some red tape required in many windoze app's -----------


/***************************************************************************/
void QFile_LastErrorCodeToString(DWORD dwError, char *dest_str, int maxlen)
{
  LPVOID lpMsgBuf;
  char *cp;

  FormatMessage(
    FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
    NULL,
    dwError,
    MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
    (LPTSTR) &lpMsgBuf,
    0,
    NULL      );
  // Note : there is a disgusting NEW LINE character somewhere in the string !

  // Copy the string to the caller's buffer
  strncpy( /*dest:*/dest_str, /*source:*/(char*)lpMsgBuf,  maxlen );

  // Remove that DISGUSTING CR+NL characters if we can find them:
  cp = dest_str+strlen(dest_str); // dangerous !
  if( (cp>dest_str+2) && (cp<dest_str+maxlen) )
   { if(*(cp-1)=='\n') *(cp-1)='\0';   // cut off this trailing CR NL junk !
     if(*(cp-2)=='\r') *(cp-2)='\0';
   }


  // Free the buffer.
  LocalFree( lpMsgBuf );
}



//----------------- File Input Routines  ------------------------------------

BOOL QFile_Flush( T_QFile *pqFile );

/***************************************************************************/
BOOL QFile_Open( T_QFile *pqFile, const char *filename, int oflags)
  // Opens a file.  Parameters similar to _rtl_open (which in fact is used internally):
  //   QFILE_O_RDONLY , QFILE_O_WRONLY ,  or   QFILE_O_RDWR  .
  // Additionally, QFILE_O_CREATE may be speficied (bit-ored) .
{
  int iRtlOFlags = O_RDONLY;
  int access, mode, shflag;

  memset(pqFile, 0, sizeof(T_QFile) ); // also sets encoding to zero = none
  pqFile->dwLastError = 0;       // no error yet
  switch( oflags & QFILE_O_MASK_RDWR )
   { case QFILE_O_RDONLY: iRtlOFlags = O_RDONLY; break;
     case QFILE_O_WRONLY: iRtlOFlags = O_WRONLY; break;
     case QFILE_O_RDWR  : iRtlOFlags = O_RDWR;   break;
     default: break;
   }
#ifdef __BORLANDC__  // Do we really have to use "_rtl_open" instead of "_open",
                     // or is this in fact just another joke by Borland ?
  pqFile->iHandle = _rtl_open( filename, iRtlOFlags ); // this is what Borland wants us to use - see notes
#else
  pqFile->iHandle = _open( filename, iRtlOFlags );  // Borland says "this is obsolete" - harr harr...
#endif
  // Note 1: _rtl_open stupidly FAILS when trying to open a NEW file for writing.
  //       We don't want such nonsense; it's quite clear that if a file
  //       is to be opened for WRITING, and it doesn't exist yet,
  //       it shall be CREATED .
  // Note 2: WxDev-C++ (a MinGW/GCC descendant) didn't know _rtl_open() .
  //       Most likely, _rtl_open() is yet another windoze-specific stupidity,
  //       even though someone said if 'replaces _open, which is obsolete' .
  //       Holy shit. Borland says _rtl_open() should be in "io.h" .
  //       But in MinGW's "io.h" (c:\Dec-Cpp\include\io.h), there's only _open.
  //       The same shit with _rtl_read(), which doesn't exist in MinGW's io.h.
  if( (pqFile->iHandle<0) && ((oflags & QFILE_O_CREATE )!=0) )
   { // only IF THE FILE DOESN'T EXIST, create it in QFile_Open()
     access = iRtlOFlags/*O_RDWR*/ | O_CREAT | O_TRUNC;
     // > If the O_CREA(-T??) flag is used in constructing access, you need to supply
     // > the mode argument to _sopen from the following symbolic constants defined in sys\stat.h.
     // >  Value of mode	Access permission
     // >       S_IWRITE   Permission to write
     // >       S_IREAD    Permission to read
     // >       S_IREAD|S_IWRITE  Permission to read/write
     mode   = S_IREAD|S_IWRITE;
     // > shflag specifies the type of file-sharing allowed on the file path.
     // > Symbolic constants for shflag are defined in share.h (holy shit, yet another includefile)
     // >   Value of shflag     What it does
     // >     SH_COMPAT       Sets compatibility mode.
     // >     SH_DENYRW       Denies read/write access
     // >     SH_DENYWR       Denies write access
     // >     SH_DENYRD       Denies read access
     // >     SH_DENYNONE     Permits read/write access
     // >     SH_DENYNO       Permits read/write access
# ifdef SH_DENYNONE
     shflag = SH_DENYNONE; // defined by Borland, doesn't exist in MinGW's share.h .
         // BULLSHIT AGAIN ! (in MinGW's share.h, there's only
         // this ugly, obscure "DENYNO" which also means "deny NOTHING". What a stupid name)
# else   // MinGW doesn't have SH_DENYNONE, only this ugly truncated short stuff:
     shflag = SH_DENYNO;   // stupidly abbreviated name, but there's no other
# endif
     pqFile->iHandle = _sopen(filename, access, shflag, mode );
   }
  pqFile->dwBufferStartPos = 0;  // disk file position of 1st byte in buffer
  pqFile->dwPrevBufferStartPos = 0;
  if(pqFile->iHandle < 0)
   { pqFile->fOpened = FALSE;
     pqFile->dwLastError = GetLastError();  // added 2003-12-15
   }
  else
   { strncpy( pqFile->sz512PathAndName, filename, 512 );
     pqFile->fOpened = TRUE;
   }
  return pqFile->fOpened;
} // end QFile_Open()

/***************************************************************************/
BOOL QFile_IsOpen( T_QFile *pqFile )
{
  return (pqFile!=NULL) && (pqFile->fOpened);
} // end QFile_IsOpen()


/***************************************************************************/
int QFile_Read( T_QFile *pqFile, void *buf, DWORD len)
  // Reads a few bytes from the file, or -if possible- from a buffer .
  // Return value: number of bytes actually read into buf;
  //               negative = error;   zero = nothing more to read .
  // Note: Unlike QFile_ReadLine(), QFile_Read() doesn't translate anything,
  //       since it was designed to read BINARY FILES (not text files) !
{
 DWORD di = 0;  // destination index like the good old 8086-register
 BYTE *pbSrc;
 BYTE *pbDst = (BYTE*)buf;
 long left_in_buffer = (long)pqFile->dwNumUsedBytesInBuffer - (long)pqFile->dwBufferIndex;

#if(0)  // 2011-02-10 : Removed the following special case - didn't work properly:
  if( len>QFILE_BUFFER_SIZE )  // since 2006-04-02 : forget the buffer if chunk is too large !
   {
     // Though we don't use the buffer, keep track of the current file position..
     pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex = 0;
     left_in_buffer = _rtl_read( pqFile->iHandle, buf, len );
     if(left_in_buffer < 0)
      { return 0;   // no more data in file
      }
     else // successfully read something :
      { // keep "buffer start index" up-to-date ...
        // dwBufferStartPos is the disk file position of the 1st byte in the buffer.
        // Must add the number of bytes read IN THE PREVIOUS _read-call (!!)
        // ex: pqFile->dwBufferStartPos += left_in_buffer; (wrong, dwBufferStartPos must be ZERO after very first read() !)
        pqFile->dwBufferStartPos = pqFile->dwPrevBufferStartPos;
        pqFile->dwPrevBufferStartPos += left_in_buffer;
        return left_in_buffer;
      }
   }
  else // caller seems to read rather small chunks. Minimize OS calls by using own buffer:
#endif // (0)
   {
     do
      {
        if( di>=len )  // added 2011-02-10 to support arbitrarily large blocks:
         { return di;  // enough bytes copied into the caller's buffer; nothing more to read here
         } // (don't read more "in advance", the caller may decide to fseek().. )

        if( left_in_buffer>0 )
         { // no need to call the slow '_rtl_read' ..
           pbSrc = pqFile->bBuffer + pqFile->dwBufferIndex;
           while(di<len && (left_in_buffer>0) )
            { *pbDst++ = *pbSrc++;
              ++di;
              --left_in_buffer;
              ++pqFile->dwBufferIndex;
            }
           if(di==len)
              return len;    // bingo, everything from buffer
         }


        // Arrived here: no more data in the buffer. Perform the next READ :
        pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex = 0;
#      ifdef __BORLANDC__  // Borland says '_read' is OBSOLETE and we should use _rtl_read() instead....
        left_in_buffer = _rtl_read( pqFile->iHandle, (void*)pqFile->bBuffer, QFILE_BUFFER_SIZE );
#      else  // but WxDev-C++ doesn't have this shiny new function, only _read(), so use THIS:
        left_in_buffer = _read( pqFile->iHandle, (void*)pqFile->bBuffer, QFILE_BUFFER_SIZE );
#      endif
        if(left_in_buffer <= 0) // modified 2007-11-26, ex: if(left_in_buffer < 0)
           return di;   // no more data in buffer, return the "rest"
        else // >= 0 ..
         {
           // keep "buffer start index" up-to-date ...
           // dwBufferStartPos is the disk file position of 1st byte in buffer.
           // Must add the number of bytes read IN THE PREVIOUS _read-call (!!)
           // ex: pqFile->dwBufferStartPos += left_in_buffer; (wrong, dwBufferStartPos must be ZERO after very first read() !)
           pqFile->dwBufferStartPos = pqFile->dwPrevBufferStartPos;
           pqFile->dwPrevBufferStartPos += left_in_buffer;
           pqFile->dwNumUsedBytesInBuffer = (DWORD)left_in_buffer;
         }
      }while(left_in_buffer>0);
   } // end else <rather small chunks read>

 return -1;        // most likely : reached end-of-file

} // end QFile_Read()

/***************************************************************************/
BYTE QFile_UnicodeTo8BitANSI( DWORD dwUnicode ) // actually 'ISO Latin 1'
  // Input: the now-famous 32-bit UNICODE, common notation "U+XXXX",
  //        where XXXX is a HEXADECIMAL number. Here, this 'Unicode Number'
  //        is just an index into a poor but simple lookup table .
  // The table itself was copied from http://www.alanwood.net/demos/ansi.html ,
  //  and converted into a switch-case list (in "C") by DL4YHF .
  //  Should be ok for most western languages, including German .
  //  INPUT :  "Unicode Number" (from Alan Wood's web site)
  //  OUTPUT:  "ANSI Number"    (from Alan Wood's web site)
{
  if( dwUnicode <= 127 )
   { return (BYTE)dwUnicode; // characters below 127 ARE ACTUALLY 7-bit ASCII (!)
   }
  switch( dwUnicode )
   { case 8364 :  return 128;  // Euro sign
     case  129 :  return 129;  // not used
     case 8218 :  return 130;  // single low-9 quotation mark
     case  402 :  return 131;  // Latin small letter f with hook
     case 8222 :  return 132;  // double low-9 quotation mark
     case 8230 :  return 133;  // horizontal ellipsis (" three dots ")
     case 8224 :  return 134;  // dagger ("requies in pacem" cross)
     case 8225 :  return 135;  // double dagger
     case  710 :  return 136;  // modifier letter circumflex accent
     case 8240 :  return 137;  // per mille sign
     case  352 :  return 138;  // Latin capital letter S with caron
     case 8249 :  return 139;  // single left-pointing angle quotation mark
     case  338 :  return 140;  // Latin capital ligature OE
     case  141 :  return 141;  // not used
     case  381 :  return 142;  // Latin capital letter Z with caron
     case  143 :  return 143;  // not used
     case  144 :  return 144;  // not used
     case 8216 :  return 145;  // left single quotation mark
     case 8217 :  return 146;  // right single quotation mark
     case 8220 :  return 147;  // left double quotation mark
     case 8221 :  return 148;  // right double quotation mark
     case 8226 :  return 149;  // bullet
     case 8211 :  return 150;  // en dash
     case 8212 :  return 151;  // em dash
     case  732 :  return 152;  // small tilde
     case 8482 :  return 153;  // trade mark sign
     case  353 :  return 154;  // Latin small letter s with caron
     case 8250 :  return 155;  // single right-pointing angle quotation mark
     case  339 :  return 156;  // Latin small ligature oe
     case  157 :  return 157;  // not used
     case  382 :  return 158;  // Latin small letter z with caron
     case  376 :  return 159;  // Latin capital letter Y with diaeresis
     case  160 :  return 160;  // no-break space
     case  161 :  return 161;  // inverted exclamation mark
     case  162 :  return 162;  // cent sign
     case  163 :  return 163;  // pound sign
     case  164 :  return 164;  // currency sign
     case  165 :  return 165;  // yen sign
     case  166 :  return 166;  // broken bar
     case  167 :  return 167;  // section sign
     case  168 :  return 168;  // diaeresis
     case  169 :  return 169;  // copyright sign
     case  170 :  return 170;  // feminine ordinal indicator
     case  171 :  return 171;  // left-pointing double angle quotation mark
     case  172 :  return 172;  // not sign
     case  173 :  return 173;  // soft hyphen
     case  174 :  return 174;  // registered sign
     case  175 :  return 175;  // macron
     case  176 :  return 176;  // degree sign
     case  177 :  return 177;  // plus-minus sign
     case  178 :  return 178;  // superscript two
     case  179 :  return 179;  // superscript three
     case  180 :  return 180;  // acute accent
     case  181 :  return 181;  // micro sign
     case  182 :  return 182;  // pilcrow sign
     case  183 :  return 183;  // middle dot
     case  184 :  return 184;  // cedilla
     case  185 :  return 185;  // superscript one
     case  186 :  return 186;  // masculine ordinal indicator
     case  187 :  return 187;  // right-pointing double angle quotation mark
     case  188 :  return 188;  // vulgar fraction one quarter
     case  189 :  return 189;  // vulgar fraction one half
     case  190 :  return 190;  // vulgar fraction three quarters
     case  191 :  return 191;  // inverted question mark
     case  192 :  return 192;  // Latin capital letter A with grave
     case  193 :  return 193;  // Latin capital letter A with acute
     case  194 :  return 194;  // Latin capital letter A with circumflex
     case  195 :  return 195;  // Latin capital letter A with tilde
     case  196 :  return 196;  // Latin capital letter A with diaeresis ()
     case  197 :  return 197;  // Latin capital letter A with ring above
     case  198 :  return 198;  // Latin capital letter AE
     case  199 :  return 199;  // Latin capital letter C with cedilla
     case  200 :  return 200;  // Latin capital letter E with grave
     case  201 :  return 201;  // Latin capital letter E with acute
     case  202 :  return 202;  // Latin capital letter E with circumflex
     case  203 :  return 203;  // Latin capital letter E with diaeresis
     case  204 :  return 204;  // Latin capital letter I with grave
     case  205 :  return 205;  // Latin capital letter I with acute
     case  206 :  return 206;  // Latin capital letter I with circumflex
     case  207 :  return 207;  // Latin capital letter I with diaeresis
     case  208 :  return 208;  // Latin capital letter Eth ("D" with horz line)
     case  209 :  return 209;  // Latin capital letter N with tilde
     case  210 :  return 210;  // Latin capital letter O with grave
     case  211 :  return 211;  // Latin capital letter O with acute
     case  212 :  return 212;  // Latin capital letter O with circumflex
     case  213 :  return 213;  // Latin capital letter O with tilde
     case  214 :  return 214;  // Latin capital letter O with diaeresis
     case  215 :  return 215;  // multiplication sign (like 'x')
     case  216 :  return 216;  // Latin capital letter O with stroke
     case  217 :  return 217;  // Latin capital letter U with grave
     case  218 :  return 218;  // Latin capital letter U with acute
     case  219 :  return 219;  // Latin capital letter U with circumflex
     case  220 :  return 220;  // Latin capital letter U with diaeresis
     case  221 :  return 221;  // Latin capital letter Y with acute
     case  222 :  return 222;  // Latin capital letter Thorn
     case  223 :  return 223;  // Latin small letter sharp s
     case  224 :  return 224;  // Latin small letter a with grave
     case  225 :  return 225;  // Latin small letter a with acute
     case  226 :  return 226;  // Latin small letter a with circumflex
     case  227 :  return 227;  // Latin small letter a with tilde
     case  228 :  return 228;  // Latin small letter a with diaeresis
     case  229 :  return 229;  // Latin small letter a with ring above
     case  230 :  return 230;  // Latin small letter ae
     case  231 :  return 231;  // Latin small letter c with cedilla
     case  232 :  return 232;  // Latin small letter e with grave
     case  233 :  return 233;  // Latin small letter e with acute
     case  234 :  return 234;  // Latin small letter e with circumflex
     case  235 :  return 235;  // Latin small letter e with diaeresis
     case  236 :  return 236;  // Latin small letter i with grave
     case  237 :  return 237;  // Latin small letter i with acute
     case  238 :  return 238;  // Latin small letter i with circumflex
     case  239 :  return 239;  // Latin small letter i with diaeresis
     case  240 :  return 240;  // Latin small letter eth
     case  241 :  return 241;  // Latin small letter n with tilde
     case  242 :  return 242;  // Latin small letter o with grave
     case  243 :  return 243;  // Latin small letter o with acute
     case  244 :  return 244;  // Latin small letter o with circumflex
     case  245 :  return 245;  // Latin small letter o with tilde
     case  246 :  return 246;  // Latin small letter o with diaeresis
     case  247 :  return 247;  // division sign
     case  248 :  return 248;  // Latin small letter o with stroke
     case  249 :  return 249;  // Latin small letter u with grave
     case  250 :  return 250;  // Latin small letter u with acute
     case  251 :  return 251;  // Latin small letter with circumflex
     case  252 :  return 252;  // Latin small letter u with diaeresis
     case  253 :  return 253;  // Latin small letter y with acute
     case  254 :  return 254;  // Latin small letter thorn
     case  255 :  return 255;  // Latin small letter y with diaeresis

     default   :  // everything not covered by the above list ...
       if( dwUnicode <= 255 )
        { return (BYTE)dwUnicode;
        }
       else
        { return 191;  // 8-bit 'Latin-1' (ANSI) for "inverted question mark"
        }
   } // end switch( dwUnicode )

} // end QFile_UnicodeTo8BitANSI()


/***************************************************************************/
BYTE QFile_DecodeByteFromUTF8Sequence( T_QFile *pqFile, BYTE bInputByte )
{
  BYTE bOutputChar = bInputByte;  // output  :   EIGHT(!)-bit-ANSI

  switch( pqFile->iUTF8_DecoderState )
   { case 0:   // default state, not INSIDE an UTF-8 'multibyte' sequence :
        // > The UTF-8 encoding is variable-width, with each character
        // > represented by 1 to 4 bytes. Each byte has 04 leading
        // > consecutive '1' bits followed by a '0' bit to indicate its type.
        // > 2 or more '1' bits indicates the first byte in a sequence
        // > of that many bytes. The scalar value of the Unicode code point
        // > is the concatenation of the non-control bits.
        // > In the table at http://en.wikipedia.org/wiki/UTF-8#Description,
        // > zeroes and ones in black represent control bits, each x
        // > represents one of the lowest 8 bits of the Unicode value,
        // > y  represents the next higher 8 bits, and z represents
        // > the bits higher than that.
        pqFile->c4_UTF8_DecoderSequence[0] = bInputByte; // save 1st byte of the sequence for later
        if( bInputByte>=0xC2 && bInputByte<=0xDF )
         { // Start of a 'valid' UTF-8 TWO-BYTE sequence :
           bOutputChar = 0;  // don't emit a character now (wait for the next byte) !
           pqFile->iUTF8_DecoderState = 1;  // new state: waiting for the 2nd of a TWO-BYTE sequence
         } // end if < start of an UTF-8 TWO-BYTE sequence > ?
        else if( bInputByte>=0xE0 && bInputByte<=0xEF )
         { // Start of a 'valid' UTF-8 THREE-BYTE sequence :
           bOutputChar = 0;  // don't emit a character now (wait for the next byte) !
           pqFile->iUTF8_DecoderState = 2;  // new state: waiting for the 2nd of a THREE-BYTE sequence
         } // end if < start of an UTF-8 THREE-BYTE sequence > ?
        else if( bInputByte>=0xF0 && bInputByte<=0xF4 )
         { // Start of a 'valid' UTF-8 FOUR-BYTE sequence :
           bOutputChar = 0;  // don't emit a character now (wait for the next byte) !
           pqFile->iUTF8_DecoderState = 4;  // new state: waiting for the 2nd of a FOUR-BYTE sequence
         } // end if < start of an UTF-8 THREE-BYTE sequence > ?
        break; // end case iUTF8_DecoderState == 0
     case 1 :  // been waiting for the 2nd of a TWO-BYTE sequence (which now arrived):
        // the 1st received byte (pqFile->c4_UTF8_DecoderSequence[0])
        //     contains a FIVE-BIT CONTROL CODE and three bits of the UNICODE value,
        // the 2nd received byte (bInputByte)
        //     contains a TWO-BIT CONTROL CODE (10bin) and SIX bits of the UNICODE value.
        // See two-byte example at http://en.wikipedia.org/wiki/UTF-8#Description .
        if( (bInputByte & 0xC0) == 0x80 ) // two-bit control code in 2nd byte ok ?
         {
           bOutputChar = QFile_UnicodeTo8BitANSI(
             ((DWORD)(pqFile->c4_UTF8_DecoderSequence[0] & 0x1F) << 6/*!*/ ) // FIVE unicode-bits in 1st byte, plus...
             |(DWORD)(bInputByte & 0x3F ) ); // SIX unicode-bits in 2nd byte .
         }
        else   // UTF-8 encoding error, emit this inverted questionmark
         { // here:  "a start byte not followed by enough continuation bytes"
           bOutputChar = 191;  // 8-bit 'Latin-1' (ANSI) for "inverted question mark"
         }
        pqFile->iUTF8_DecoderState = 0;
        break; // end case iUTF8_DecoderState == 1
     case 2 :  // been waiting for the 2nd of a THREE-BYTE sequence
        pqFile->c4_UTF8_DecoderSequence[1] = bInputByte; // save 2nd of 3 bytes
        bOutputChar = 0;  // don't emit a character now (wait for the next byte) !
        pqFile->iUTF8_DecoderState = 3;
        break; // end case iUTF8_DecoderState == 2
     case 3 :  // been waiting for the 3rd of a THREE-BYTE sequence (which is now complete) .
        // See three-byte example at http://en.wikipedia.org/wiki/UTF-8#Description .
        if(  ( (pqFile->c4_UTF8_DecoderSequence[0] & 0xF0) == 0xE0 ) // four-bit control code in 1st byte ok ?
          && ( (pqFile->c4_UTF8_DecoderSequence[1] & 0xC0) == 0x80 ) // two-bit control code in 2nd byte ok ?
          && ( (bInputByte & 0xC0) == 0x80 ) ) // two-bit control code in 3nd byte ok ?
         {
           bOutputChar = QFile_UnicodeTo8BitANSI(
             ((DWORD)(pqFile->c4_UTF8_DecoderSequence[0] & 0x0F) << 12) // FOUR unicode-bits in 1st byte, plus...
            |((DWORD)(pqFile->c4_UTF8_DecoderSequence[1] & 0x3F) <<  6) // SIX  unicode-bits in 2nd byte, and...
            | (DWORD)(bInputByte & 0x3F ) ) ; // SIX unicode-bits in 3rd byte .
         }
        else   // UTF-8 encoding error, emit this inverted questionmark
         { // here:  "a start byte not followed by enough continuation bytes"
           bOutputChar = 191;  // 8-bit 'Latin-1' (ANSI) for "inverted question mark"
         }
        pqFile->iUTF8_DecoderState = 0;
        break; // end case iUTF8_DecoderState == 3
     case 4 :  // been  waiting for the 2nd of a FOUR-BYTE sequence
        pqFile->c4_UTF8_DecoderSequence[1] = bInputByte; // save 2nd of 3 bytes
        bOutputChar = 0;  // don't emit a character now (wait for the next byte) !
        pqFile->iUTF8_DecoderState = 4;
        break; // end case iUTF8_DecoderState == 4
     case 5 :  // been  waiting for the 3rd of a FOUR-BYTE sequence
        pqFile->c4_UTF8_DecoderSequence[2] = bInputByte; // save 2nd of 3 bytes
        bOutputChar = 0;  // don't emit a character now (wait for the next byte) !
        pqFile->iUTF8_DecoderState = 5;
        break; // end case iUTF8_DecoderState == 5
     case 6 :  // been  waiting for the 4th of a FOUR-BYTE sequence
        // See four-byte example at http://en.wikipedia.org/wiki/UTF-8#Description .
        if(  ( (pqFile->c4_UTF8_DecoderSequence[0] & 0xF8) == 0xF0 ) // five-bit control code in 1st byte ok ?
          && ( (pqFile->c4_UTF8_DecoderSequence[1] & 0xC0) == 0x80 ) // two-bit control code in 2nd byte ok ?
          && ( (pqFile->c4_UTF8_DecoderSequence[2] & 0xC0) == 0x80 ) // two-bit control code in 3rd byte ok ?
          && ( (bInputByte & 0xC0) == 0x80 ) ) // two-bit control code in 4th byte ok ?
         {
           bOutputChar = QFile_UnicodeTo8BitANSI(
             ((DWORD)(pqFile->c4_UTF8_DecoderSequence[0] & 0x07) << 18) // THREE unicode-bits in 1st byte, plus...
            |((DWORD)(pqFile->c4_UTF8_DecoderSequence[1] & 0x3F) << 12) // SIX  unicode-bits in 2nd byte, and...
            |((DWORD)(pqFile->c4_UTF8_DecoderSequence[1] & 0x3F) <<  6) // SIX  unicode-bits in 3rd byte, and...
            | (DWORD)(bInputByte & 0x3F ) ); // SIX unicode-bits in 4th byte .
         }
        else   // UTF-8 encoding error, emit this inverted questionmark
         { // here:  "a start byte not followed by enough continuation bytes"
           bOutputChar = 191;  // 8-bit 'Latin-1' (ANSI) for "inverted question mark"
         }
        pqFile->iUTF8_DecoderState = 0;
        break; // end case iUTF8_DecoderState == 6
    } // end switch( pqFile->iUTF8_DecoderState )
  return bOutputChar;
} // end QFile_DecodeByteFromUTF8Sequence()



/***************************************************************************/
int QFile_ReadLine( T_QFile *pqFile, char *pszDest, int iMaxLen )
  // Reads a single text line from a file,  up to the CR / NL terminator.
  // The [CR /] NL terminator will be skipped but not entered in pszDest .
  //   Since 2007-04-25, works with DOS- *and* UNIX- textfiles !
  //   Since 2010-06-21, removes the stupid UTF-8 "BOM",
  //                     and returns only PLAIN TEXT without UTF-8
  //                     control characters.
  // Return value:  > 0 = Count of characters,
  //                  ZERO if an empty line has been read,
  //               or NEGATIVE if there was an access error or end of file.
{

 BYTE  bCurrChar, bPrevChar;
 int  di = 0;  // destination index like the good old 8086-register
 BYTE *pbSrc;
 BYTE *pbDst = (BYTE*)pszDest;
 long left_in_buffer = (long)pqFile->dwNumUsedBytesInBuffer - (long)pqFile->dwBufferIndex;

 if(iMaxLen>0)
    *pszDest = '\0';  // return empty string if no characters available

 do
  {

   if( left_in_buffer>0 )
    { // no need to call the slow '_rtl_read' ..
      pbSrc = pqFile->bBuffer + pqFile->dwBufferIndex;
      // Added 2010-06-21: Skip the UTF-8 "BOM"-nonsense (if it's nonsense).
      //  > A Byte Order Marker (BOM for short) is a hex value at the very beginning
      //  > of a file that is used as a "flag" or "signature" for the encoding
      //  > and/or hex byte order that should be used for the file.
      //  > With UTF-8 encoded data, this is normally hex bytes EF BB BF.
      //  > The BOM also tells the editor whether the Unicode data is in big endian
      //  > or little endian format. Big endian Unicode data simply means that
      //  > the most significant hex byte is stored in your computer's memory first,
      //  > while little endian stores this in memory last.
      //  > BOMs are not always essential for displaying Unicode data,
      //  > but they can save developers headaches when writing and building applications.
      if( pqFile->dwBufferIndex==0  && pqFile->dwBufferStartPos==0  // "first byte" read from the file ?
        && (left_in_buffer>=3) )  // at least 3 bytes left in the buffer ?
       { // Check for this 'UTF-8 BOM thingy', and skip it :
         if( (pbSrc[0]==0xEF) && (pbSrc[1]==0xBB) && (pbSrc[2]==0xBF) )
          { // this is obviously an UTF-8 'BOM' (inserted by Notepad),
            // skip it, but remember the basic encoding:
            pqFile->iEncoding = QFILE_ENCODING_UTF8;  // beware of codes > 127 !
            pbSrc += 3;
            pqFile->dwBufferIndex = 3;
            left_in_buffer -= 3;
          }
       } // end if < do the UTF-8 "BOM" test ? >
      while(di<iMaxLen && (left_in_buffer>0) )
       { bCurrChar = *pbSrc++;
         bPrevChar = pqFile->bPrevCharFromReadLine;
         pqFile->bPrevCharFromReadLine = bCurrChar;
         ++pqFile->dwBufferIndex;
         --left_in_buffer;
         if( (pqFile->iUTF8_DecoderState==0) && (bCurrChar<=(BYTE)13) && (bCurrChar!='\t') )
          { // Looks like a control character, may be the end of the line.
            // The CR and/or NL characters are *NOT* returned to the caller,
            // instead the C-string will be zero-terminated.
            // Note:
            // - Under UNIX, a text line is terminated with Linefeed ONLY (LF, #10).
            // - Under DOS, a text line is terminated with CR (#13) + LF (#10).
            // - Here, we accept ANY of the following combinations as line separator:
            //   CR+LF, only LF, only CR, and LF+CR (quite exotic but who knows..)
            if( (bCurrChar==(BYTE)10) && (bPrevChar==(BYTE)13) )
             { // the current character is the 2nd part of a DOS-line-end .
               // ( special case added 2007-07-08,
               //  to make this work independent from buffer boundaries)
               bCurrChar = 0;  // don't append this character to the string,
               // because it's the "rest" of a CR-LF sequence which began in the previous buffer part.
             }
            else
             {
               *pbDst = 0;              // terminate "C" string at the 1st ctrl char
               if( left_in_buffer < 0 ) // must read next buffer part to check NL after CR :
                { pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex = 0;
#      ifdef __BORLANDC__  // Borland says '_read' is OBSOLETE and we should use _rtl_read() instead....
                  left_in_buffer = _rtl_read( pqFile->iHandle, (void*)pqFile->bBuffer, QFILE_BUFFER_SIZE );
#      else       // but WxDev-C++ doesn't have this shiny new function, only _read(), so use THIS:
                  left_in_buffer = _read( pqFile->iHandle, (void*)pqFile->bBuffer, QFILE_BUFFER_SIZE );
#      endif  // BORLAND or WxDev-C++/MinGW/GNU ?
                  pbSrc = pqFile->bBuffer;
                  if(left_in_buffer < 0)
                     return di;   // no more data in buffer
                  else
                   { // Keep "buffer start index" up-to-date ...
                     // ex: pqFile->dwBufferStartPos += left_in_buffer;  (WRONG !)
                     pqFile->dwBufferStartPos = pqFile->dwPrevBufferStartPos;
                     pqFile->dwPrevBufferStartPos += left_in_buffer;
                   }
                }
               // If the NEXT character is also a control char, skip it as well...
               // ... but ONLY if the 2nd control char is different from the first,
               // because otherwise we would treat TWO EMPTY "UNIX-Textlines" as ONE .
               // 2007-07-08: this is IMPOSSIBLE if we just reached the end
               //             of the buffer (that's why bPrevCharFromReadLine was added !)
               if( (*pbSrc<13) && (*pbSrc!='\t') && (*pbSrc!=bCurrChar) && (left_in_buffer>0) )
                 { ++pqFile->dwBufferIndex;
                 }
              return di;        // return value = string length
             } // end of line detected !
          } // end if (bCurrChar<=(BYTE)13)
         else // character codes above 13 ... could it be 'something special' ?
         if( (pqFile->iUTF8_DecoderState>0) || (bCurrChar > 127) )
          {
            switch( pqFile->iEncoding )
             { case QFILE_ENCODING_UTF8:
                 bCurrChar = QFile_DecodeByteFromUTF8Sequence( pqFile, bCurrChar );
                 // Note: for UTF-8 sequences with 2 or more bytes,
                 //       the decoder will emit ZERO BYTES !  Example:
                 // bCurrChar = 0xC2 = start of a 2-byte sequence;
                 //     will be 'emitted' as 0x00 here (until the next char was received).
                 break;
             } // end switch( pqFile->iEncoding )
          } // end if( bCurrChar > 127 )
         if( bCurrChar )
          { *pbDst++ = bCurrChar;
            ++di;
          }
       }
      if(di==iMaxLen)
        return di;    // caller's maximum string length exceeded
    } // end if( left_in_buffer>0 )

   // Arrived here: no more data in the buffer. Must do the next READ :
   pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex = 0;
# ifdef __BORLANDC__
   left_in_buffer = _rtl_read( pqFile->iHandle, (void*)pqFile->bBuffer, QFILE_BUFFER_SIZE );
# else
   left_in_buffer = _read( pqFile->iHandle, (void*)pqFile->bBuffer, QFILE_BUFFER_SIZE );
# endif
   if(left_in_buffer <= 0)
     { // no more data in buffer
       return (di>0)?di:-1;    // maybe END-OF-FILE (important!)
     }
   else // >= 0 ..
     { pqFile->dwNumUsedBytesInBuffer = (DWORD)left_in_buffer;
       // keep "buffer start index" up-to-date ....
       // ex: pqFile->dwBufferStartPos += left_in_buffer;  (WRONG !)
       pqFile->dwBufferStartPos = pqFile->dwPrevBufferStartPos;
       pqFile->dwPrevBufferStartPos += left_in_buffer;
     }


  }while(left_in_buffer>0);

  return (di>0)?di:-1;
} // end QFile_ReadLine()


/***************************************************************************/
long QFile_Seek( T_QFile *pqFile, long offset, int fromwhere)
   // Works like lseek on a 'buffered' file.
   // Avoids unnecessary calls to library/OS functions !
   // QFile_Seek(  pqFile, 0, SEEK_CUR )  returns the current file position .
{
  long lResult;
  DWORD dwNewAbsFilePosition;
  DWORD dwOldAbsFilePosition = pqFile->dwBufferStartPos + pqFile->dwBufferIndex;
  switch( fromwhere )
   {
     case QFILE_SEEK_SET: /* Positionierung vom Dateianfang aus */
        dwNewAbsFilePosition = (DWORD)offset;
        break;
     case QFILE_SEEK_CUR: /* Positionierung von der aktuellen Position aus */
        dwNewAbsFilePosition = (DWORD)( (long)dwOldAbsFilePosition + offset );
        break;
     case QFILE_SEEK_END: /* Positionierung vom Dateiende aus */
        if(pqFile->fBufferModified)
         {
           QFile_Flush(pqFile);
           pqFile->dwBufferIndex = pqFile->dwNumUsedBytesInBuffer = 0;
         }
        lResult = lseek( pqFile->iHandle, offset, SEEK_END );
        if( lResult != -1)
         { dwNewAbsFilePosition = (DWORD)lResult;
           pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex = 0;
           pqFile->dwBufferStartPos = dwNewAbsFilePosition;
           pqFile->dwPrevBufferStartPos = dwNewAbsFilePosition;
           return lResult;  // finished here ("special case")
         }
        break;
     default :
        return -1L;  // error  (like lseek, but no fooling around with the global "errno")
   }

  // Is this just a QUERY or a SET-command ?
  if( (fromwhere==SEEK_CUR) && (offset==0) )
   { // it's just a QUERY for the current file position :
     return dwOldAbsFilePosition;
   }

  // Added 2008-04-26 :
  // It may be unnecessary to call the OS to "really" change the
  // file position, for example if the new position is already in the buffer:
  if( ( dwNewAbsFilePosition >= pqFile->dwBufferStartPos )
    &&( dwNewAbsFilePosition < (pqFile->dwBufferStartPos+pqFile->dwNumUsedBytesInBuffer) ) )
   { // the "new" file position is covered by the current buffer contents.
     // No need to call a time-consuming OS function !
     pqFile->dwBufferIndex = dwNewAbsFilePosition - pqFile->dwBufferStartPos;
     return (long)dwNewAbsFilePosition;
   }

  // Arrived here, we know the seek-action will MODIFY the real file pointer.
  // If we're about to change the file position, and the file is WRITEABLE,
  //   we MAY have to flush the buffer back into the real file :
  if(pqFile->fBufferModified)
   {
    QFile_Flush(pqFile);
    pqFile->dwBufferIndex = pqFile->dwNumUsedBytesInBuffer = 0;
   }

  // Change the physical file position and forget everything about the buffer
  lResult = lseek( pqFile->iHandle, dwNewAbsFilePosition, SEEK_SET/*!*/ );
    // > lseek liefert bei fehlerfreier Ausfhrung die neue Zeigerposition,
    // > gemessen in Bytes vom Dateianfang, zurck .

  // Because the current file position may have changed,
  //  the buffer is no longer valid :
  pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex = 0;
  if(lResult>=0)
         pqFile->dwBufferStartPos = lResult;
   else  pqFile->dwBufferStartPos = 0;
  pqFile->dwPrevBufferStartPos = pqFile->dwBufferStartPos; // added 2005-07-22

  return lResult;
} // end QFile_Seek()



//----------------- File Output Routines  -----------------------------------


/***************************************************************************/
BOOL QFile_Create( T_QFile *pqFile,
                   const char *filename,
                   int attrib )
  // Creates a new file (or truncates existing). Parameters similar to _rtl_creat .
{
  int access,shflag,mode;
  memset(pqFile, 0, sizeof(T_QFile) );
  pqFile->dwLastError = 0;       // no error yet
  if( (attrib & QFILE_ATTR_SHARED) == 0)
   { // don't create a "shared", but a "normal" (non-shared) file :
#  ifdef __BORLANDC__  // Borland says '_creat' is OBSOLETE and we should use _rtl_creat() instead....
     // (hey, if they are inventing new names why not avoid this stupid abbreviation 'creat' for 'create' ?!)
     pqFile->iHandle = _rtl_creat(filename, attrib & 0x00FF);   // from BORLAND's io.h
#  else       // but WxDev-C++ doesn't have this shiny new function, only _read(), so use THIS:
     pqFile->iHandle = _creat(filename, attrib & 0x00FF);       // from MingGW's io.h
#  endif  // Borland or something else ?
   }
  else
   { // How to create files with SHARED read/write ?
     // You won't find anything about that topic in Borland's help system on _rtl_creat() !
     // Later, WB discovered _sopen(), which can also be used to *CREATE* new files.
     // So let's use _sopen to create SHARED files, since 2008-12-21 :
     // > For _sopen, access is constructed by ORing flags bitwise from the following lists:
     // > Read/write flags
     // >   You can use only one of the following flags:
     // >      O_RDONLY	Open for reading only.
     // >      O_WRONLY	Open for writing only.
     // >      O_RDWR	Open for reading and writing.
     // > Other access flags
     // >   You can use any logical combination of the following flags:
     // >      O_NDELAY	Not used; for UNIX compatibility.
     // >      O_APPEND If set, the file pointer is set to the end of the file prior to each write.
     // >      O_CREA (??) If the file exists, this flag has no effect.
     // >               If the file does not exist, the file is created,
     // >               and the bits of mode are used to set
     // >               the file attribute bits as in chmod.
     //              WB: "O_CREA" doesn't seem to exist, even though the
     //                  Borland manual mentions it. HOLY SHIT.
     // >      O_TRUNC If the file exists, its length is truncated to 0.
     // >               The file attributes remain unchanged.
     // access = O_RDWR | O_CREA | O_TRUNC;  // what the fuck is "O_CREA" ?!
     access = O_RDWR | O_CREAT | O_TRUNC;
     // > If the O_CREA(-T??) flag is used in constructing access, you need to supply
     // > the mode argument to _sopen from the following symbolic constants defined in sys\stat.h.
     // >  Value of mode	Access permission
     // >       S_IWRITE   Permission to write
     // >       S_IREAD    Permission to read
     // >       S_IREAD|S_IWRITE  Permission to read/write
     mode   = S_IREAD|S_IWRITE;
     // > shflag specifies the type of file-sharing allowed on the file path.
     // > Symbolic constants for shflag are defined in share.h (holy shit, yet another includefile)
     // >   Value of shflag     What it does
     // >     SH_COMPAT       Sets compatibility mode.
     // >     SH_DENYRW       Denies read/write access
     // >     SH_DENYWR       Denies write access
     // >     SH_DENYRD       Denies read access
     // >     SH_DENYNONE     Permits read/write access
     // >     SH_DENYNO       Permits read/write access
# ifdef SH_DENYNONE
     shflag = SH_DENYNONE; // defined by Borland, doesn't exist in MinGW's share.h .
# else   // MinGW doesn't have SH_DENYNONE, only this ugly truncated short stuff:
     shflag = SH_DENYNO;
# endif
     pqFile->iHandle = _sopen(filename,  // from io.h (like _creat)
                              access, shflag, mode );
   }
  pqFile->dwBufferStartPos = 0;
  pqFile->dwPrevBufferStartPos = 0;
  if(pqFile->iHandle < 0)
   { pqFile->fOpened = FALSE;
     pqFile->dwLastError = GetLastError();  // added 2003-12-15
   }
  else
   { strncpy( pqFile->sz512PathAndName, filename, 512 );
     pqFile->fOpened = TRUE;
   }
  return pqFile->fOpened;
} // end QFile_Create()

/***************************************************************************/
BOOL QFile_Flush( T_QFile *pqFile )
{
 int nWritten = 0;
  if(pqFile->fBufferModified && pqFile->fOpened && pqFile->dwNumUsedBytesInBuffer>0 )
   {
#  ifdef __BORLANDC__  // Borland says '_write' is OBSOLETE and we should use _rtl_write() instead....
     nWritten = _rtl_write( pqFile->iHandle, pqFile->bBuffer, pqFile->dwNumUsedBytesInBuffer);
#  else                // stupid, isn't it ? Everyone else happily keeps on using _write() ;-)
     nWritten = _write( pqFile->iHandle, pqFile->bBuffer, pqFile->dwNumUsedBytesInBuffer);
#  endif
     pqFile->fBufferModified = FALSE;
     if(nWritten>0)
      { pqFile->dwBufferStartPos += nWritten;  // keep "buffer start index" up-to-date
        pqFile->dwPrevBufferStartPos = pqFile->dwBufferStartPos;
      }
     else
      {
        pqFile->dwLastError = GetLastError();  // added 2003-12-15
      }
     return (nWritten>0);
   }
  return TRUE;
} // end QFile_Flush()

/***************************************************************************/
void QFile_Close( T_QFile *pqFile )
  // Closes a file.
{
  if(pqFile->iHandle >= 0)
   { if(pqFile->fBufferModified && pqFile->fOpened && pqFile->dwNumUsedBytesInBuffer>0 )
      {
       QFile_Flush( pqFile );   // flush the very last part into the file
      }
#  ifdef __BORLANDC__           // Borland against ... - guess the rest:
     _rtl_close( pqFile->iHandle );
#  else
     _close( pqFile->iHandle ); // Borland says "this is obsolete". WTF ?
#  endif
     pqFile->iHandle = -1;
   }
  pqFile->fOpened = FALSE;
} // end QFile_Close()


/***************************************************************************/
BOOL QFile_Write( T_QFile *pqFile, BYTE *pbSrc, long i32CountOfBytes )
{
 BOOL fOk = pqFile->fOpened;

  if( (pbSrc) && (i32CountOfBytes>0) )
   {
    // once here: pqFile->fBufferModified
    while(i32CountOfBytes)
     { --i32CountOfBytes;
      if(pqFile->dwBufferIndex >= QFILE_BUFFER_SIZE)
       { if(pqFile->dwNumUsedBytesInBuffer < pqFile->dwBufferIndex)
            pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex;
         pqFile->fBufferModified=TRUE;   // 2008-03-23: moved HERE
         fOk &= QFile_Flush( pqFile );
         pqFile->dwBufferIndex = pqFile->dwNumUsedBytesInBuffer = 0;
       }
      pqFile->bBuffer[ pqFile->dwBufferIndex++ ] = *pbSrc++;
     } // end while <more characters from input string>
    pqFile->fBufferModified=TRUE;  // maybe ONE character modified !

    // Update the "buffer usage" indicator; there may be something left
    //  in the buffer which must be flushed on CLOSING the file..
    if(pqFile->dwNumUsedBytesInBuffer < pqFile->dwBufferIndex)
       pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex;

   } // end if < valid data >

  return fOk;
} // end QFile_Write()


/***************************************************************************/
BOOL QFile_WriteString( T_QFile *pqFile, char * pszSrc )
   // Writes a zero-terminated string.
   // Counts the characters, but DOES NOT APPEND CR/NL by itself .
{
 BOOL fOk = pqFile->fOpened;

  // One could detect the string length first, and *then* call QFile_Write(),
  //  but doing everything here in a single loop is a bit FASTER ...
  if(pszSrc)
   {
    pqFile->fBufferModified=TRUE;
    while(*pszSrc)
     {
      if(pqFile->dwBufferIndex >= QFILE_BUFFER_SIZE)
       { if(pqFile->dwNumUsedBytesInBuffer < pqFile->dwBufferIndex)
            pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex;
         fOk &= QFile_Flush( pqFile );
         pqFile->dwBufferIndex = pqFile->dwNumUsedBytesInBuffer = 0;
       }
      pqFile->bBuffer[ pqFile->dwBufferIndex++ ] = *pszSrc++;
     } // end while <more characters from input string>
    pqFile->fBufferModified=TRUE;  // maybe ONE character modified !

    // Update the "buffer usage" indicator; there may be something left
    //  in the buffer which must be flushed on CLOSING the file..
    if(pqFile->dwNumUsedBytesInBuffer < pqFile->dwBufferIndex)
       pqFile->dwNumUsedBytesInBuffer = pqFile->dwBufferIndex;

   } // end if(pszSrc)

  return fOk;
} // end QFile_WriteString()


//----------------- Universal parsing routines ------------------------------
// (often required when text files are read into structures)

/**************************************************************************/
CPROT char *QFile_GetFilenameWithoutPath( char *pszFilenameWithPath )
{
  char *pszNameWithoutPath = strrchr(pszFilenameWithPath, '\\' );
     if( pszNameWithoutPath == NULL )
      { pszNameWithoutPath = strrchr(pszFilenameWithPath, '/' );
      }
     if( pszNameWithoutPath != NULL )
      { ++pszNameWithoutPath;
      }
     else
      { pszNameWithoutPath = pszFilenameWithPath;
      }
  return pszNameWithoutPath;
} // end QFile_GetFilenameWithoutPath()

/**************************************************************************/
CPROT int QFile_SkipSpaces(char **ppcSource )
  // Actually skips spaces and TABS.
  // Returns THE NUMBER OF SPACES (and tabs) if something was skipped,
  //  otherwise zero .
{
  char *cp = *ppcSource;
  int nCharsSkipped = 0;
  while(*cp==' ' || *cp=='\t') // skip SPACES and TABS (for reading "text data files")
   { ++cp;
     ++nCharsSkipped;
   }
  *ppcSource = (char*)cp; // skip the "expected" character
  return nCharsSkipped;
} // end QFile_SkipSpaces()

/**************************************************************************/
CPROT BOOL QFile_SkipChar(char **ppcSource, char cCharToSkip )
  // String handling routine: Checks for a certain character,
  //   and skips it from the "sourcecode" if found .
  // LEADING SPACES are skipped (they usually have no syntactic meaning) .
{
 char *cp = *ppcSource;
  QFile_SkipSpaces( &cp );
  if( *cp == cCharToSkip )
   { *ppcSource = (char*)cp+1; // skip the "expected" character
     return TRUE;
   }
  else
   { return FALSE;
   }
} // end QFile_SkipChar()

/**************************************************************************/
BOOL QFile_SkipString(char **ppcSource, char *pszStringToSkip)
  // String handling routine: Checks for a certain string,
  //   and skips it from the "sourcecode" if found there.
  //   Leading spaces and tab characters are SKIPPED !
  // Returns TRUE when found, otherwise FALSE .
  // Often used when reading configuration files, as a replacement
  // for the terribly slow windows INI-files .
{
  char *cp = *ppcSource;
  while(*cp==' ' || *cp=='\t') // skip SPACES and TABS (for reading "text data files")
   { ++cp;
   }
  while( (*pszStringToSkip!='\0') && (*pszStringToSkip==*cp) )
   { ++pszStringToSkip;
     ++cp;
   }
  if( *pszStringToSkip == '\0' )  // bingo, reached the end of the string-to-skip
   { *ppcSource = (char*)cp;
     return TRUE;
   }
  else
   { return FALSE;
   }
} // end QFile_SkipString()


/***************************************************************************/
int  QFile_SkipStringFromList(char **ppcSource, // [in,out] source pointer
                         char *pszzStringList)  // [in] zero delimited list,
                             // last item terminated with a DOUBLE zero byte
  // Returns the zero-based index of the string in the list (if found),
  //    otherwise a negative value.
  //    The source pointer (*ppcSource) will only be incremented
  //    when a match was found; see QFile_SkipString (called from here).
  // This function is often used to read simple configuration files,
  // using the returned value in a switch-case-list .
  // Beware, this function doesn't care for the character which follows
  // after the string in the source (**ppcSource), so place LONGER names
  // in the list before SHORTER names, like "test123\0test\0\0" .
{
  int n = 0;
  if( (ppcSource!=NULL) && (*ppcSource!=NULL) && (pszzStringList!=NULL) )
   { while( (**ppcSource!='\0') && (*pszzStringList != '\0') )
      { if( QFile_SkipString( ppcSource, pszzStringList ) )
         { // bingo...
           return n;
         }
        else
         { pszzStringList += (strlen(pszzStringList) + 1);
         }
        ++n;
      }
   }
  return -1;  // none of the strings in the list
} // end QFile_SkipStringFromList()


/***************************************************************************/
long QFile_ParseInteger(char **ppcSource, int maxdigits, int radix, long deflt)
  // String handling routine: Parses an integer number from any sourcecode.
  // Also good for HEXADECIMAL NUMBERS WITH '0x' PREFIX since 2006-01 !
  // If the sourcecode doesn't contain a valid number,
  // the source pointer will not be moved, and the default value will be returned .
{
 long ret=0;
 int  neg=0;
 BOOL valid=FALSE;
 BYTE *bp = (BYTE*)*ppcSource;
 BYTE c;
  while(*bp==' ' || *bp=='\t')   // skip SPACES and TABS (for reading "text data files")
    { ++bp;
    }
  if(*bp=='-')
    { ++bp; neg=1; }
  else
  if(*bp=='+')
    { ++bp; }
  if( bp[0]=='0' && bp[1]=='x' ) // hexadecimal (C-style) ?
   { bp += 2;  // skip hex prefix
     radix = 16;
   }
  if( radix == 16 )
   { while(maxdigits>0)
      {
        c=*bp;
        if( c>='0' && c<='9' )
         { ++bp;
           --maxdigits;
           valid = TRUE;
           ret = 16*ret + (c-'0');
         }
        else if(c>='a' && c<='f')
         { ++bp;
           --maxdigits;
           valid = TRUE;
           ret = 160*ret + (c-'a'+10);
         }
        else if(c>='A' && c<='F')
         { ++bp;
           --maxdigits;
           valid = TRUE;
           ret = 16*ret + (c-'A'+10);
         }
        else
           break;
      }
   }
  else // not hexadecimal but decimal :
   { while( (c=*bp)>='0' && (c<='9') && (maxdigits>0) )
      { ++bp;
        --maxdigits;
        valid = TRUE;
        ret = 10*ret + (c-'0');
      }
   }
  *ppcSource = (char*)bp;
  if( valid )
       return neg ? -ret : ret;
  else return deflt;
} // end QFile_ParseInteger()

/***************************************************************************/
double QFile_ParseDFloat(char **ppcSource, double dblDefault)
{
 double dblResult=0.0;
 double dblFractFactor=0.1;
 BOOL neg=FALSE;
 BOOL valid=FALSE;
 BYTE *bp = (BYTE*)*ppcSource;
 BYTE c;
  while(*bp==' ' || *bp=='\t')   // skip SPACES and TABS (for reading "text data files")
    { ++bp;
    }
  if(*bp=='-')
    { ++bp;
      neg=TRUE;
    }
  else
  if(*bp=='+')
    { ++bp;
    }
  // Parse the INTEGER part (here: only decimal, nothing else)
  while( (c=*bp)>='0' && (c<='9') )
   { ++bp;
     valid = TRUE;
     dblResult = 10.0*dblResult + (double)(c-'0');
   }
  if( *bp=='.' ) // parse the FRACTIONAL part, too :
   { ++bp;
     while( (c=*bp)>='0' && (c<='9') )
      { ++bp;
        dblResult += dblFractFactor * (double)(c-'0');
        dblFractFactor *= 0.1;
      }
   }
  *ppcSource = (char*)bp;
  if( valid )
       return neg ? -dblResult : dblResult;
  else return dblDefault;
} // end QFile_ParseDFloat()

/***************************************************************************/
BOOL QFile_ParseQuotedString(char **ppcSource, char *pszDest, int iMaxLen )
{
  char c;
  char *cp = *ppcSource;

  if( pszDest != NULL )
   {
     while(*cp==' ' || *cp=='\t') // skip SPACES and TABS (for reading "text data files")
      { ++cp;
      }
     c = *cp;
     if(c=='"')
      {
        ++cp;   // note that the 'source pointer' was incremented here,
                // so it would be safe to access cp[-1] below .
        while(1)
         {
           c=*cp++;
           if (c=='"')
            { break;  // 'good' end of the double-quoted string
            }
           else if(c=='\0')
            { --cp;   // last increment went too far (yucc)
              break;
            }
           if ( (iMaxLen--) > 0 )
            { *pszDest++ = c;
            }
         } // end while
        *pszDest = '\0';  // null-terminate destination string
        *ppcSource = cp;  // skip the "expected" character only if the syntax is ok
        return TRUE;
      } // end if < source begins with " >
     else // If there's no double-quoted string as espected, DO NOT SKIP ANYTHING
      {   // (besides the leading spaces & tabs which have already been skipped)
          // If the caller wants to accept optionally NON-QUOTED strings,
          // he could use QFile_CopyStringUntilEndOfLine() - see below.
      }
   } // end if( pszDest != NULL )
  return FALSE;
} // end QFile_ParseQuotedString()

/***************************************************************************/
void QFile_CopyStringUntilEndOfLine(char **ppcSource, char *pszDest, int iMaxLen )
{
  char c;
  char *cp = *ppcSource;
  while( ((unsigned char)(c=*cp++)>=32) )
   { if( iMaxLen>1)
      { *pszDest++ = c;
        --iMaxLen;
      }
   }
  if( iMaxLen>0 )
   { *pszDest = '\0';  // null-terminate destination string
   }

  *ppcSource = cp;  // skip the "expected" character only if the syntax is ok

} // end QFile_CopyStringUntilEndOfLine()


/* EOF < YHF_Tools\QFile.c >.  Leave an empty line after this for certain compilers! */



